// Copyright 2014 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef CONTENT_PUBLIC_RENDERER_REQUEST_PEER_H_
#define CONTENT_PUBLIC_RENDERER_REQUEST_PEER_H_

#include <stdint.h>

#include <memory>
#include <string>

#include "base/task_runner.h"
#include "content/common/content_export.h"
#include "mojo/public/cpp/base/big_buffer.h"
#include "mojo/public/cpp/system/data_pipe.h"
#include "services/network/public/mojom/url_response_head.mojom-forward.h"

namespace net {
struct RedirectInfo;
}

namespace network {
struct URLLoaderCompletionStatus;
}

namespace content {

// This is implemented by our custom resource loader within content. The Peer
// and it's bridge should have identical lifetimes as they represent each end of
// a communication channel.
//
// These callbacks mirror net::URLRequest::Delegate and the order and
// conditions in which they will be called are identical. See url_request.h
// for more information.
class CONTENT_EXPORT RequestPeer {
 public:
  // Called as upload progress is made.
  // note: only for requests with upload progress enabled.
  virtual void OnUploadProgress(uint64_t position, uint64_t size) = 0;

  // Called when a redirect occurs. The implementation may return false to
  // suppress the redirect. The URLResponseHead provides information about
  // the redirect response and the RedirectInfo includes information about the
  // request to be made if the method returns true. |removed_headers| outputs
  // header field names that need to be removed.
  virtual bool OnReceivedRedirect(
      const net::RedirectInfo& redirect_info,
      network::mojom::URLResponseHeadPtr head,
      std::vector<std::string>* removed_headers) = 0;

  // Called when response headers are available (after all redirects have
  // been followed).
  virtual void OnReceivedResponse(network::mojom::URLResponseHeadPtr head) = 0;

  // Called when the response body becomes available.
  virtual void OnStartLoadingResponseBody(
      mojo::ScopedDataPipeConsumerHandle body) = 0;

  // Called when the transfer size is updated. This method may be called
  // multiple times or not at all. The transfer size is the length of the
  // response (including both headers and the body) over the network.
  // |transfer_size_diff| is the difference from the value previously reported
  // one (including the one in OnReceivedResponse). It must be positive.
  virtual void OnTransferSizeUpdated(int transfer_size_diff) = 0;

  // Called when metadata generated by the renderer is retrieved from the
  // cache. This method may be called zero or one times.
  virtual void OnReceivedCachedMetadata(mojo_base::BigBuffer data) {}

  // Called when the response is complete.  This method signals completion of
  // the resource load.
  virtual void OnCompletedRequest(
      const network::URLLoaderCompletionStatus& status) = 0;

  // Returns the task runner on which this request peer is running.
  virtual scoped_refptr<base::TaskRunner> GetTaskRunner() = 0;

  virtual ~RequestPeer() {}
};

}  // namespace content

#endif  // CONTENT_PUBLIC_RENDERER_REQUEST_PEER_H_
