// Copyright 2018 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/viz/service/display_embedder/skia_output_surface_impl_on_gpu.h"

#include "base/atomic_sequence_num.h"
#include "base/bind.h"
#include "base/bind_helpers.h"
#include "base/callback_helpers.h"
#include "base/optional.h"
#include "base/threading/thread_task_runner_handle.h"
#include "base/trace_event/memory_dump_manager.h"
#include "base/trace_event/trace_event.h"
#include "components/viz/common/frame_sinks/copy_output_request.h"
#include "components/viz/common/frame_sinks/copy_output_util.h"
#include "components/viz/common/resources/resource_format_utils.h"
#include "components/viz/common/skia_helper.h"
#include "components/viz/common/viz_utils.h"
#include "components/viz/service/display/dc_layer_overlay.h"
#include "components/viz/service/display/output_surface_frame.h"
#include "components/viz/service/display/overlay_candidate.h"
#include "components/viz/service/display_embedder/image_context_impl.h"
#include "components/viz/service/display_embedder/output_presenter_gl.h"
#include "components/viz/service/display_embedder/skia_output_device.h"
#include "components/viz/service/display_embedder/skia_output_device_buffer_queue.h"
#include "components/viz/service/display_embedder/skia_output_device_gl.h"
#include "components/viz/service/display_embedder/skia_output_device_offscreen.h"
#include "components/viz/service/display_embedder/skia_output_device_webview.h"
#include "components/viz/service/display_embedder/skia_output_surface_dependency.h"
#include "gpu/command_buffer/common/swap_buffers_complete_params.h"
#include "gpu/command_buffer/service/gr_shader_cache.h"
#include "gpu/command_buffer/service/mailbox_manager.h"
#include "gpu/command_buffer/service/memory_tracking.h"
#include "gpu/command_buffer/service/scheduler.h"
#include "gpu/command_buffer/service/shared_image_representation.h"
#include "gpu/command_buffer/service/skia_utils.h"
#include "gpu/config/gpu_preferences.h"
#include "gpu/ipc/common/command_buffer_id.h"
#include "gpu/ipc/common/gpu_client_ids.h"
#include "gpu/ipc/common/gpu_peak_memory.h"
#include "gpu/ipc/common/gpu_surface_lookup.h"
#include "gpu/vulkan/buildflags.h"
#include "skia/buildflags.h"
#include "third_party/skia/include/core/SkDeferredDisplayList.h"
#include "ui/gfx/color_space.h"
#include "ui/gfx/geometry/rect_conversions.h"
#include "ui/gl/gl_surface.h"

#if BUILDFLAG(ENABLE_VULKAN)
#include "components/viz/service/display_embedder/skia_output_device_vulkan.h"
#include "gpu/vulkan/vulkan_util.h"
#endif

#if (BUILDFLAG(ENABLE_VULKAN) || BUILDFLAG(SKIA_USE_DAWN)) && defined(USE_X11)
#include "components/viz/service/display_embedder/skia_output_device_x11.h"
#endif

#if defined(USE_OZONE)
#include "ui/ozone/public/ozone_platform.h"
#include "ui/ozone/public/platform_window_surface.h"
#include "ui/ozone/public/surface_factory_ozone.h"
#endif

#if BUILDFLAG(SKIA_USE_DAWN)
#include "components/viz/common/gpu/dawn_context_provider.h"
#if defined(OS_WIN)
#include "components/viz/service/display_embedder/skia_output_device_dawn.h"
#endif
#endif

#if defined(USE_OZONE) || defined(USE_X11)
#include "ui/base/ui_base_features.h"
#endif

#if defined(OS_FUCHSIA)
#include "components/viz/service/display_embedder/output_presenter_fuchsia.h"
#endif

namespace viz {

namespace {

template <typename... Args>
void PostAsyncTaskRepeatedly(
    base::WeakPtr<SkiaOutputSurfaceImplOnGpu> impl_on_gpu,
    const base::RepeatingCallback<void(Args...)>& callback,
    Args... args) {
  // Callbacks generated by this function may be executed asynchronously
  // (e.g. by presentation feedback) after |impl_on_gpu| has been destroyed.
  if (impl_on_gpu)
    impl_on_gpu->PostTaskToClientThread(base::BindOnce(callback, args...));
}

template <typename... Args>
base::RepeatingCallback<void(Args...)> CreateSafeRepeatingCallback(
    base::WeakPtr<SkiaOutputSurfaceImplOnGpu> impl_on_gpu,
    const base::RepeatingCallback<void(Args...)>& callback) {
  return base::BindRepeating(&PostAsyncTaskRepeatedly<Args...>, impl_on_gpu,
                             callback);
}

struct ReadPixelsContext {
  ReadPixelsContext(std::unique_ptr<CopyOutputRequest> request,
                    const gfx::Rect& result_rect,
                    const gfx::ColorSpace& color_space,
                    base::WeakPtr<SkiaOutputSurfaceImplOnGpu> impl_on_gpu)
      : request(std::move(request)),
        result_rect(result_rect),
        color_space(color_space),
        impl_on_gpu(impl_on_gpu) {}

  std::unique_ptr<CopyOutputRequest> request;
  gfx::Rect result_rect;
  gfx::ColorSpace color_space;
  base::WeakPtr<SkiaOutputSurfaceImplOnGpu> impl_on_gpu;
};

class CopyOutputResultYUV : public CopyOutputResult {
 public:
  CopyOutputResultYUV(const gfx::Rect& rect,
                      std::unique_ptr<const SkSurface::AsyncReadResult> result)
      : CopyOutputResult(Format::I420_PLANES, rect),
        result_(std::move(result)) {
    DCHECK_EQ(3, result_->count());
    DCHECK_EQ(0, size().width() % 2);
    DCHECK_EQ(0, size().height() % 2);
  }

  // CopyOutputResult implementation.
  bool ReadI420Planes(uint8_t* y_out,
                      int y_out_stride,
                      uint8_t* u_out,
                      int u_out_stride,
                      uint8_t* v_out,
                      int v_out_stride) const override {
    const auto CopyPlane = [](const uint8_t* src, int src_stride, int width,
                              int height, uint8_t* out, int out_stride) {
      for (int i = 0; i < height; ++i, src += src_stride, out += out_stride) {
        memcpy(out, src, width);
      }
    };
    auto* data0 = static_cast<const uint8_t*>(result_->data(0));
    auto* data1 = static_cast<const uint8_t*>(result_->data(1));
    auto* data2 = static_cast<const uint8_t*>(result_->data(2));
    CopyPlane(data0, result_->rowBytes(0), width(0), height(0), y_out,
              y_out_stride);
    CopyPlane(data1, result_->rowBytes(1), width(1), height(1), u_out,
              u_out_stride);
    CopyPlane(data2, result_->rowBytes(2), width(2), height(2), v_out,
              v_out_stride);
    return true;
  }

 private:
  uint32_t width(int plane) const {
    if (plane == 0)
      return size().width();
    else
      return size().width() / 2;
  }

  uint32_t height(int plane) const {
    if (plane == 0)
      return size().height();
    else
      return size().height() / 2;
  }

  std::unique_ptr<const SkSurface::AsyncReadResult> result_;
};

void OnYUVReadbackDone(
    void* c,
    std::unique_ptr<const SkSurface::AsyncReadResult> async_result) {
  std::unique_ptr<ReadPixelsContext> context(
      static_cast<ReadPixelsContext*>(c));
  if (context->impl_on_gpu)
    context->impl_on_gpu->ReadbackDone();

  if (!async_result) {
    // This will automatically send an empty result.
    return;
  }
  std::unique_ptr<CopyOutputResult> result =
      std::make_unique<CopyOutputResultYUV>(context->result_rect,
                                            std::move(async_result));
  context->request->SendResult(std::move(result));
}

void DestroyAsyncReadResult(void* pixels, void* context) {
  const SkSurface::AsyncReadResult* result =
      static_cast<const SkSurface::AsyncReadResult*>(context);
  delete result;
}

void OnRGBAReadbackDone(
    void* c,
    std::unique_ptr<const SkSurface::AsyncReadResult> async_result) {
  std::unique_ptr<ReadPixelsContext> context(
      static_cast<ReadPixelsContext*>(c));
  if (context->impl_on_gpu)
    context->impl_on_gpu->ReadbackDone();

  if (!async_result) {
    // This will automatically send an empty result.
    return;
  }

  DCHECK_EQ(1, async_result->count());

  auto info = SkImageInfo::MakeN32Premul(context->result_rect.width(),
                                         context->result_rect.height(),
                                         context->color_space.ToSkColorSpace());

  // Passing ownership of |async_result| to DestroyAsyncReadResult.
  auto* result_ptr = async_result.release();
  SkBitmap bitmap;
  bitmap.installPixels(info, const_cast<void*>(result_ptr->data(0)),
                       result_ptr->rowBytes(0), DestroyAsyncReadResult,
                       const_cast<SkSurface::AsyncReadResult*>(result_ptr));

  std::unique_ptr<CopyOutputResult> result =
      std::make_unique<CopyOutputSkBitmapResult>(context->result_rect, bitmap);
  context->request->SendResult(std::move(result));
}

}  // namespace

SkiaOutputSurfaceImplOnGpu::PromiseImageAccessHelper::PromiseImageAccessHelper(
    SkiaOutputSurfaceImplOnGpu* impl_on_gpu)
    : impl_on_gpu_(impl_on_gpu) {}

SkiaOutputSurfaceImplOnGpu::PromiseImageAccessHelper::
    ~PromiseImageAccessHelper() {
  DCHECK(image_contexts_.empty() || impl_on_gpu_->was_context_lost());
}

void SkiaOutputSurfaceImplOnGpu::PromiseImageAccessHelper::BeginAccess(
    std::vector<ImageContextImpl*> image_contexts,
    std::vector<GrBackendSemaphore>* begin_semaphores,
    std::vector<GrBackendSemaphore>* end_semaphores) {
  DCHECK(begin_semaphores);
  DCHECK(end_semaphores);
  begin_semaphores->reserve(image_contexts.size());
  // We may need one more space for the swap buffer semaphore.
  end_semaphores->reserve(image_contexts.size() + 1);
  image_contexts_.reserve(image_contexts.size() + image_contexts_.size());
  image_contexts_.insert(image_contexts.begin(), image_contexts.end());
  impl_on_gpu_->BeginAccessImages(std::move(image_contexts), begin_semaphores,
                                  end_semaphores);
}

void SkiaOutputSurfaceImplOnGpu::PromiseImageAccessHelper::EndAccess() {
  impl_on_gpu_->EndAccessImages(image_contexts_);
  image_contexts_.clear();
}

namespace {

base::AtomicSequenceNumber g_next_command_buffer_id;

scoped_refptr<gpu::SyncPointClientState> CreateSyncPointClientState(
    SkiaOutputSurfaceDependency* deps,
    gpu::SequenceId sequence_id) {
  auto command_buffer_id = gpu::CommandBufferId::FromUnsafeValue(
      g_next_command_buffer_id.GetNext() + 1);
  return deps->GetSyncPointManager()->CreateSyncPointClientState(
      gpu::CommandBufferNamespace::VIZ_SKIA_OUTPUT_SURFACE, command_buffer_id,
      sequence_id);
}

std::unique_ptr<gpu::SharedImageFactory> CreateSharedImageFactory(
    SkiaOutputSurfaceDependency* deps,
    gpu::MemoryTracker* memory_tracker) {
  return std::make_unique<gpu::SharedImageFactory>(
      deps->GetGpuPreferences(), deps->GetGpuDriverBugWorkarounds(),
      deps->GetGpuFeatureInfo(), deps->GetSharedContextState().get(),
      deps->GetMailboxManager(), deps->GetSharedImageManager(),
      deps->GetGpuImageFactory(), memory_tracker,
      true /* enable_wrapped_sk_image */);
}

std::unique_ptr<gpu::SharedImageRepresentationFactory>
CreateSharedImageRepresentationFactory(SkiaOutputSurfaceDependency* deps,
                                       gpu::MemoryTracker* memory_tracker) {
  return std::make_unique<gpu::SharedImageRepresentationFactory>(
      deps->GetSharedImageManager(), memory_tracker);
}

}  // namespace

// Offscreen surfaces for render passes. It can only be accessed on GPU
// thread.
class SkiaOutputSurfaceImplOnGpu::OffscreenSurface {
 public:
  OffscreenSurface() = default;
  OffscreenSurface(const OffscreenSurface& offscreen_surface) = delete;
  OffscreenSurface(OffscreenSurface&& offscreen_surface) = default;
  OffscreenSurface& operator=(const OffscreenSurface& offscreen_surface) =
      delete;
  OffscreenSurface& operator=(OffscreenSurface&& offscreen_surface) = default;
  ~OffscreenSurface() = default;

  SkSurface* surface() { return surface_.get(); }
  void set_surface(sk_sp<SkSurface> surface) {
    surface_ = std::move(surface);
    promise_texture_ = {};
  }

  SkPromiseImageTexture* fulfill() {
    DCHECK(surface_);
    if (!promise_texture_) {
      promise_texture_ =
          SkPromiseImageTexture::Make(surface_->getBackendTexture(
              SkSurface::kFlushRead_BackendHandleAccess));
    }
    return promise_texture_.get();
  }

  sk_sp<SkSurface> TakeSurface() {
    promise_texture_ = {};
    return std::move(surface_);
  }

 private:
  sk_sp<SkSurface> surface_;
  sk_sp<SkPromiseImageTexture> promise_texture_;
};

SkiaOutputSurfaceImplOnGpu::ReleaseCurrent::ReleaseCurrent(
    scoped_refptr<gl::GLSurface> gl_surface,
    scoped_refptr<gpu::SharedContextState> context_state)
    : gl_surface_(gl_surface), context_state_(context_state) {}

SkiaOutputSurfaceImplOnGpu::ReleaseCurrent::~ReleaseCurrent() {
  if (context_state_ && gl_surface_)
    context_state_->ReleaseCurrent(gl_surface_.get());
}

class SkiaOutputSurfaceImplOnGpu::DisplayContext : public gpu::DisplayContext {
 public:
  DisplayContext(SkiaOutputSurfaceDependency* deps,
                 SkiaOutputSurfaceImplOnGpu* owner)
      : dependency_(deps), owner_(owner) {
    dependency_->RegisterDisplayContext(this);
  }
  ~DisplayContext() override { dependency_->UnregisterDisplayContext(this); }

  DisplayContext(const DisplayContext&) = delete;
  DisplayContext& operator=(const DisplayContext&) = delete;

  // gpu::DisplayContext implementation
  void MarkContextLost() override {
    owner_->MarkContextLost(CONTEXT_LOST_UNKNOWN);
  }

 private:
  SkiaOutputSurfaceDependency* const dependency_;
  SkiaOutputSurfaceImplOnGpu* const owner_;
};

// static
std::unique_ptr<SkiaOutputSurfaceImplOnGpu> SkiaOutputSurfaceImplOnGpu::Create(
    SkiaOutputSurfaceDependency* deps,
    const RendererSettings& renderer_settings,
    const gpu::SequenceId sequence_id,
    DidSwapBufferCompleteCallback did_swap_buffer_complete_callback,
    BufferPresentedCallback buffer_presented_callback,
    ContextLostCallback context_lost_callback,
    GpuVSyncCallback gpu_vsync_callback) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::Create");

  auto context_state = deps->GetSharedContextState();
  if (!context_state)
    return nullptr;

  // Even with Vulkan/Dawn compositing, the SharedImageFactory constructor
  // always initializes a GL-backed SharedImage factory to fall back on.
  // Creating the SharedImageBackingFactoryGLTexture invokes GL API calls, so
  // we need to ensure there is a current GL context.
  if (!context_state->MakeCurrent(nullptr, true /* need_gl */)) {
    LOG(ERROR) << "Failed to make current during initialization.";
    return nullptr;
  }
  context_state->set_need_context_state_reset(true);

  auto impl_on_gpu = std::make_unique<SkiaOutputSurfaceImplOnGpu>(
      util::PassKey<SkiaOutputSurfaceImplOnGpu>(), deps,
      context_state->feature_info(), renderer_settings, sequence_id,
      std::move(did_swap_buffer_complete_callback),
      std::move(buffer_presented_callback), std::move(context_lost_callback),
      std::move(gpu_vsync_callback));
  if (!impl_on_gpu->Initialize())
    return nullptr;

  return impl_on_gpu;
}

SkiaOutputSurfaceImplOnGpu::SkiaOutputSurfaceImplOnGpu(
    util::PassKey<SkiaOutputSurfaceImplOnGpu> /* pass_key */,
    SkiaOutputSurfaceDependency* deps,
    scoped_refptr<gpu::gles2::FeatureInfo> feature_info,
    const RendererSettings& renderer_settings,
    const gpu::SequenceId sequence_id,
    DidSwapBufferCompleteCallback did_swap_buffer_complete_callback,
    BufferPresentedCallback buffer_presented_callback,
    ContextLostCallback context_lost_callback,
    GpuVSyncCallback gpu_vsync_callback)
    : dependency_(std::move(deps)),
      feature_info_(std::move(feature_info)),
      sync_point_client_state_(
          CreateSyncPointClientState(dependency_, sequence_id)),
      memory_tracker_(dependency_->GetSharedContextState()->memory_tracker()),
      shared_image_factory_(
          CreateSharedImageFactory(dependency_, memory_tracker_)),
      shared_image_representation_factory_(
          CreateSharedImageRepresentationFactory(dependency_, memory_tracker_)),
      vulkan_context_provider_(dependency_->GetVulkanContextProvider()),
      dawn_context_provider_(dependency_->GetDawnContextProvider()),
      renderer_settings_(renderer_settings),
      did_swap_buffer_complete_callback_(
          std::move(did_swap_buffer_complete_callback)),
      context_lost_callback_(std::move(context_lost_callback)),
      gpu_vsync_callback_(std::move(gpu_vsync_callback)),
      gpu_preferences_(dependency_->GetGpuPreferences()),
      display_context_(std::make_unique<DisplayContext>(deps, this)) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  weak_ptr_ = weak_ptr_factory_.GetWeakPtr();
  buffer_presented_callback_ = CreateSafeRepeatingCallback(
      weak_ptr_, std::move(buffer_presented_callback));
}

SkiaOutputSurfaceImplOnGpu::~SkiaOutputSurfaceImplOnGpu() {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  // |output_device_| may still need |shared_image_factory_|, so release it
  // first.
  output_device_.reset();

  // Since SharedImageFactory also has a reference to ImplOnGpu's member
  // SharedContextState, we need to explicitly invoke the factory's destructor
  // before deleting ImplOnGpu's other member variables.
  shared_image_factory_.reset();
  if (context_state_) {
    context_state_->RemoveContextLostObserver(this);

    // |context_provider_| and clients want either the context to be lost or
    // made current on destruction.
    if (MakeCurrent(false /* need_fbo0 */)) {
      // This ensures any outstanding callbacks for promise images are
      // performed.
      gr_context()->flushAndSubmit();
      release_current_last_.emplace(gl_surface_, context_state_);
    }
  }

  sync_point_client_state_->Destroy();
}

void SkiaOutputSurfaceImplOnGpu::Reshape(const gfx::Size& size,
                                         float device_scale_factor,
                                         const gfx::ColorSpace& color_space,
                                         gfx::BufferFormat format,
                                         bool use_stencil,
                                         gfx::OverlayTransform transform) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::Reshape");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(gr_context());

  if (!MakeCurrent(!dependency_->IsOffscreen() /* need_fbo0 */))
    return;

  size_ = size;
  color_space_ = color_space;
  if (!output_device_->Reshape(size_, device_scale_factor, color_space, format,
                               transform)) {
    MarkContextLost(CONTEXT_LOST_RESHAPE_FAILED);
    return;
  }
}

bool SkiaOutputSurfaceImplOnGpu::FinishPaintCurrentFrame(
    sk_sp<SkDeferredDisplayList> ddl,
    sk_sp<SkDeferredDisplayList> overdraw_ddl,
    std::vector<ImageContextImpl*> image_contexts,
    std::vector<gpu::SyncToken> sync_tokens,
    uint64_t sync_fence_release,
    base::OnceClosure on_finished,
    base::Optional<gfx::Rect> draw_rectangle) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::FinishPaintCurrentFrame");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(!scoped_output_device_paint_);

  bool need_fbo0 = gl_surface_ && !gl_surface_->IsSurfaceless();
  if (!MakeCurrent(need_fbo0))
    return false;

  if (!ddl) {
    MarkContextLost(CONTEXT_LOST_UNKNOWN);
    return false;
  }

  if (draw_rectangle) {
    if (!output_device_->SetDrawRectangle(*draw_rectangle)) {
      MarkContextLost(
          ContextLostReason::CONTEXT_LOST_SET_DRAW_RECTANGLE_FAILED);
      return false;
    }
  }

  // We do not reset scoped_output_device_paint_ after drawing the ddl until
  // SwapBuffers() is called, because we may need access to output_sk_surface()
  // for CopyOutput().
  scoped_output_device_paint_.emplace(output_device_.get());
  DCHECK(output_sk_surface());

  dependency_->ScheduleGrContextCleanup();

  PullTextureUpdates(std::move(sync_tokens));

  {
    base::Optional<gpu::raster::GrShaderCache::ScopedCacheUse> cache_use;
    if (dependency_->GetGrShaderCache()) {
      cache_use.emplace(dependency_->GetGrShaderCache(),
                        gpu::kDisplayCompositorClientId);
    }

    std::vector<GrBackendSemaphore> begin_semaphores;
    std::vector<GrBackendSemaphore> end_semaphores;
    promise_image_access_helper_.BeginAccess(
        std::move(image_contexts), &begin_semaphores, &end_semaphores);
    if (!begin_semaphores.empty()) {
      auto result = output_sk_surface()->wait(
          begin_semaphores.size(), begin_semaphores.data(),
          /*deleteSemaphoresAfterWait=*/false);
      DCHECK(result);
    }

    // Draw will only fail if the SkSurface and SkDDL are incompatible.
    bool draw_success = output_sk_surface()->draw(ddl);
    DCHECK(draw_success);

    destroy_after_swap_.emplace_back(std::move(ddl));

    if (overdraw_ddl) {
      sk_sp<SkSurface> overdraw_surface = SkSurface::MakeRenderTarget(
          gr_context(), overdraw_ddl->characterization(), SkBudgeted::kNo);
      overdraw_surface->draw(overdraw_ddl);
      destroy_after_swap_.emplace_back(std::move(overdraw_ddl));

      SkPaint paint;
      sk_sp<SkImage> overdraw_image = overdraw_surface->makeImageSnapshot();

      sk_sp<SkColorFilter> colorFilter = SkiaHelper::MakeOverdrawColorFilter();
      paint.setColorFilter(colorFilter);
      // TODO(xing.xu): move below to the thread where skia record happens.
      output_sk_surface()->getCanvas()->drawImage(overdraw_image.get(), 0, 0,
                                                  &paint);
    }

    auto end_paint_semaphores =
        scoped_output_device_paint_->TakeEndPaintSemaphores();
    end_semaphores.insert(end_semaphores.end(), end_paint_semaphores.begin(),
                          end_paint_semaphores.end());

    GrFlushInfo flush_info = {
        /*.fNumSemaphores =*/ end_semaphores.size(),
        /*.fSignalSemaphores =*/ end_semaphores.data(),
    };

    gpu::AddVulkanCleanupTaskForSkiaFlush(vulkan_context_provider_,
                                          &flush_info);
    if (on_finished)
      gpu::AddCleanupTaskForSkiaFlush(std::move(on_finished), &flush_info);

    auto result = output_sk_surface()->flush(flush_info);

    if (result != GrSemaphoresSubmitted::kYes &&
        !(begin_semaphores.empty() && end_semaphores.empty())) {
      // TODO(penghuang): handle vulkan device lost.
      DLOG(ERROR) << "output_sk_surface()->flush() failed.";
      return false;
    }
  }
  ReleaseFenceSyncAndPushTextureUpdates(sync_fence_release);
  return true;
}

void SkiaOutputSurfaceImplOnGpu::ScheduleOutputSurfaceAsOverlay(
    const OverlayProcessorInterface::OutputSurfaceOverlayPlane&
        output_surface_plane) {
  DCHECK(!output_surface_plane_);
  output_surface_plane_ = output_surface_plane;
}

void SkiaOutputSurfaceImplOnGpu::SwapBuffers(
    base::TimeTicks post_task_timestamp,
    OutputSurfaceFrame frame,
    base::OnceCallback<bool()> deferred_framebuffer_draw_closure) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::SwapBuffers");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  if (!post_task_timestamp.is_null()) {
    output_device_->SetDrawTimings(post_task_timestamp, base::TimeTicks::Now());
  }

  SwapBuffersInternal(std::move(deferred_framebuffer_draw_closure), &frame);
}

void SkiaOutputSurfaceImplOnGpu::SwapBuffersSkipped(
    base::OnceCallback<bool()> deferred_framebuffer_draw_closure) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  SwapBuffersInternal(std::move(deferred_framebuffer_draw_closure));
}

void SkiaOutputSurfaceImplOnGpu::FinishPaintRenderPass(
    base::TimeTicks post_task_timestamp,
    AggregatedRenderPassId id,
    sk_sp<SkDeferredDisplayList> ddl,
    std::vector<ImageContextImpl*> image_contexts,
    std::vector<gpu::SyncToken> sync_tokens,
    uint64_t sync_fence_release) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::FinishPaintRenderPass");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(ddl);

  if (!post_task_timestamp.is_null()) {
    output_device_->SetDrawTimings(post_task_timestamp, base::TimeTicks::Now());
  }

  if (!MakeCurrent(false /* need_fbo0 */))
    return;

  if (!ddl) {
    MarkContextLost(CONTEXT_LOST_UNKNOWN);
    return;
  }

  PullTextureUpdates(std::move(sync_tokens));

  auto& offscreen = offscreen_surfaces_[id];
  if (!offscreen.surface()) {
    offscreen.set_surface(SkSurface::MakeRenderTarget(
        gr_context(), ddl->characterization(), SkBudgeted::kNo));
    DCHECK(offscreen.surface());
  }

  {
    base::Optional<gpu::raster::GrShaderCache::ScopedCacheUse> cache_use;
    if (dependency_->GetGrShaderCache()) {
      cache_use.emplace(dependency_->GetGrShaderCache(),
                        gpu::kDisplayCompositorClientId);
    }
    std::vector<GrBackendSemaphore> begin_semaphores;
    std::vector<GrBackendSemaphore> end_semaphores;
    promise_image_access_helper_.BeginAccess(
        std::move(image_contexts), &begin_semaphores, &end_semaphores);
    if (!begin_semaphores.empty()) {
      auto result = offscreen.surface()->wait(
          begin_semaphores.size(), begin_semaphores.data(),
          /*deleteSemaphoresAfterWait=*/false);
      DCHECK(result);
    }
    offscreen.surface()->draw(ddl);
    destroy_after_swap_.emplace_back(std::move(ddl));

    GrFlushInfo flush_info = {
        /*.fNumSemaphores =*/ end_semaphores.size(),
        /*.fSignalSemaphores =*/ end_semaphores.data(),
    };

    gpu::AddVulkanCleanupTaskForSkiaFlush(vulkan_context_provider_,
                                          &flush_info);
    auto result = offscreen.surface()->flush(flush_info);
    if (result != GrSemaphoresSubmitted::kYes &&
        !(begin_semaphores.empty() && end_semaphores.empty())) {
      // TODO(penghuang): handle vulkan device lost.
      DLOG(ERROR) << "offscreen.surface()->flush() failed.";
      return;
    }
  }
  ReleaseFenceSyncAndPushTextureUpdates(sync_fence_release);
}

void SkiaOutputSurfaceImplOnGpu::RemoveRenderPassResource(
    std::vector<AggregatedRenderPassId> ids,
    std::vector<std::unique_ptr<ImageContextImpl>> image_contexts) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::RemoveRenderPassResource");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(!ids.empty());

  for (AggregatedRenderPassId id : ids) {
    // It's possible that |offscreen_surfaces_| won't contain an entry for the
    // render pass if draw failed early.
    auto it = offscreen_surfaces_.find(id);
    if (it != offscreen_surfaces_.end()) {
      DeleteSkSurface(context_state_.get(), it->second.TakeSurface());
      offscreen_surfaces_.erase(it);
    }
  }

  // |image_contexts| will go out of scope and be destroyed now.
}

static void PostTaskFromMainToImplThread(
    scoped_refptr<base::SingleThreadTaskRunner> impl_task_runner,
    ReleaseCallback callback,
    const gpu::SyncToken& sync_token,
    bool is_lost) {
  impl_task_runner->PostTask(
      FROM_HERE, base::BindOnce(std::move(callback), sync_token, is_lost));
}

bool SkiaOutputSurfaceImplOnGpu::CopyOutput(
    AggregatedRenderPassId id,
    copy_output::RenderPassGeometry geometry,
    const gfx::ColorSpace& color_space,
    std::unique_ptr<CopyOutputRequest> request,
    base::OnceCallback<bool()> deferred_framebuffer_draw_closure) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::CopyOutput");
  // TODO(crbug.com/898595): Do this on the GPU instead of CPU with Vulkan.
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);

  if (deferred_framebuffer_draw_closure) {
    // returns false if context not set to current, i.e lost
    if (!std::move(deferred_framebuffer_draw_closure).Run())
      return false;
    DCHECK(context_state_->IsCurrent(nullptr /* surface */));
  } else {
    if (!MakeCurrent(true /* need_fbo0 */))
      return false;
  }

  bool from_fbo0 = !id;
  DCHECK(scoped_output_device_paint_ || !from_fbo0);

  DCHECK(from_fbo0 ||
         offscreen_surfaces_.find(id) != offscreen_surfaces_.end());
  auto* surface =
      from_fbo0 ? output_sk_surface() : offscreen_surfaces_[id].surface();

  // If a platform doesn't support RGBX_8888 format, we will use RGBA_8888
  // instead. In this case, we need discard alpha channel (modify the alpha
  // value to 0xff, but keep other channel not changed).
  bool need_discard_alpha = from_fbo0 && (output_device_->is_emulated_rgbx());
  if (need_discard_alpha) {
    base::Optional<gpu::raster::GrShaderCache::ScopedCacheUse> cache_use;
    if (dependency_->GetGrShaderCache()) {
      cache_use.emplace(dependency_->GetGrShaderCache(),
                        gpu::kDisplayCompositorClientId);
    }
    SkPaint paint;
    paint.setColor(SK_ColorBLACK);
    paint.setBlendMode(SkBlendMode::kDstATop);
    surface->getCanvas()->drawPaint(paint);
    surface->flush();
  }

  base::Optional<gpu::raster::GrShaderCache::ScopedCacheUse> cache_use;
  if (dependency_->GetGrShaderCache()) {
    cache_use.emplace(dependency_->GetGrShaderCache(),
                      gpu::kDisplayCompositorClientId);
  }

  // For downscaling, use the GOOD quality setting (appropriate for
  // thumbnailing); and, for upscaling, use the BEST quality.
  bool is_downscale_in_both_dimensions =
      request->scale_to().x() < request->scale_from().x() &&
      request->scale_to().y() < request->scale_from().y();
  SkFilterQuality filter_quality = is_downscale_in_both_dimensions
                                       ? kMedium_SkFilterQuality
                                       : kHigh_SkFilterQuality;

  // Compute |source_selection| as a workaround to support |result_selection|
  // with Skia readback. |result_selection| is a clip rect specified in the
  // destination pixel space. By transforming |result_selection| back to the
  // source pixel space we can compute what rectangle to sample from.
  //
  // This might introduce some rounding error if destination pixel space is
  // scaled up from the source pixel space. When scaling |result_selection| back
  // down it might not be pixel aligned.
  gfx::Rect source_selection = geometry.sampling_bounds;
  if (request->has_result_selection()) {
    gfx::Rect sampling_selection = request->result_selection();
    if (request->is_scaled()) {
      // Invert the scaling.
      sampling_selection = copy_output::ComputeResultRect(
          sampling_selection, request->scale_to(), request->scale_from());
    }
    sampling_selection.Offset(source_selection.OffsetFromOrigin());
    source_selection.Intersect(sampling_selection);
  }

  SkIRect src_rect =
      SkIRect::MakeXYWH(source_selection.x(), source_selection.y(),
                        source_selection.width(), source_selection.height());
  if (request->result_format() ==
      CopyOutputRequest::ResultFormat::I420_PLANES) {
    std::unique_ptr<ReadPixelsContext> context =
        std::make_unique<ReadPixelsContext>(std::move(request),
                                            geometry.result_selection,
                                            color_space, weak_ptr_);
    // Skia readback could be synchronous. Incremement counter in case
    // ReadbackCompleted is called immediately.
    num_readbacks_pending_++;
    surface->asyncRescaleAndReadPixelsYUV420(
        kRec709_SkYUVColorSpace, SkColorSpace::MakeSRGB(), src_rect,
        {geometry.result_selection.width(), geometry.result_selection.height()},
        SkSurface::RescaleGamma::kSrc, filter_quality, &OnYUVReadbackDone,
        context.release());
  } else if (request->result_format() ==
             CopyOutputRequest::ResultFormat::RGBA_BITMAP) {
    // Perform swizzle during readback.
    const bool skbitmap_is_bgra = (kN32_SkColorType == kBGRA_8888_SkColorType);
    SkImageInfo dst_info = SkImageInfo::Make(
        geometry.result_selection.width(), geometry.result_selection.height(),
        skbitmap_is_bgra ? kBGRA_8888_SkColorType : kRGBA_8888_SkColorType,
        kPremul_SkAlphaType);
    std::unique_ptr<ReadPixelsContext> context =
        std::make_unique<ReadPixelsContext>(std::move(request),
                                            geometry.result_selection,
                                            color_space, weak_ptr_);
    // Skia readback could be synchronous. Incremement counter in case
    // ReadbackCompleted is called immediately.
    num_readbacks_pending_++;
    surface->asyncRescaleAndReadPixels(
        dst_info, src_rect, SkSurface::RescaleGamma::kSrc, filter_quality,
        &OnRGBAReadbackDone, context.release());
  } else if (request->result_format() ==
             CopyOutputRequest::ResultFormat::RGBA_TEXTURE) {
    gpu::Mailbox mailbox = gpu::Mailbox::GenerateForSharedImage();
    constexpr auto kUsage = gpu::SHARED_IMAGE_USAGE_GLES2 |
                            gpu::SHARED_IMAGE_USAGE_GLES2_FRAMEBUFFER_HINT |
                            gpu::SHARED_IMAGE_USAGE_RASTER |
                            gpu::SHARED_IMAGE_USAGE_DISPLAY;
    bool result = shared_image_factory_->CreateSharedImage(
        mailbox, ResourceFormat::RGBA_8888,
        gfx::Size(geometry.result_bounds.width(),
                  geometry.result_bounds.height()),
        color_space, kBottomLeft_GrSurfaceOrigin, kUnpremul_SkAlphaType,
        gpu::kNullSurfaceHandle, kUsage);
    if (!result) {
      DLOG(ERROR) << "Failed to create shared image.";
      return false;
    }

    auto representation = dependency_->GetSharedImageManager()->ProduceSkia(
        mailbox, context_state_->memory_type_tracker(), context_state_);
    shared_image_factory_->DestroySharedImage(mailbox);

    SkSurfaceProps surface_props{0, kUnknown_SkPixelGeometry};
    std::vector<GrBackendSemaphore> begin_semaphores;
    std::vector<GrBackendSemaphore> end_semaphores;

    representation->SetCleared();
    auto scoped_write = representation->BeginScopedWriteAccess(
        0 /* final_msaa_count */, surface_props, &begin_semaphores,
        &end_semaphores,
        gpu::SharedImageRepresentation::AllowUnclearedAccess::kYes);
    SkSurface* dest_surface = scoped_write->surface();
    dest_surface->wait(begin_semaphores.size(), begin_semaphores.data());
    SkCanvas* dest_canvas = dest_surface->getCanvas();
    if (request->is_scaled()) {
      dest_canvas->scale(static_cast<SkScalar>(request->scale_to().x()) /
                             request->scale_from().x(),
                         static_cast<SkScalar>(request->scale_to().y()) /
                             request->scale_from().y());
    }

    SkPaint paint;
    paint.setFilterQuality(filter_quality);
    dest_canvas->clipRect(
        SkRect::MakeXYWH(0, 0, src_rect.width(), src_rect.height()));
    surface->draw(dest_canvas, -src_rect.x(), -src_rect.y(), &paint);

    GrFlushInfo flush_info;
    flush_info.fNumSemaphores = end_semaphores.size();
    flush_info.fSignalSemaphores = end_semaphores.data();
    gpu::AddVulkanCleanupTaskForSkiaFlush(vulkan_context_provider_,
                                          &flush_info);
    auto flush_result = dest_surface->flush(
        SkSurface::BackendSurfaceAccess::kNoAccess, flush_info);
    if (flush_result != GrSemaphoresSubmitted::kYes &&
        !(begin_semaphores.empty() && end_semaphores.empty())) {
      // TODO(penghuang): handle vulkan device lost.
      DLOG(ERROR) << "dest_surface->flush() failed.";
      return false;
    }
    auto release_callback = base::BindOnce(
        &SkiaOutputSurfaceImplOnGpu::DestroySharedImageOnImplThread,
        weak_ptr_factory_.GetWeakPtr(), std::move(representation),
        context_state_);
    auto main_callback = SingleReleaseCallback::Create(base::BindOnce(
        &PostTaskFromMainToImplThread, base::ThreadTaskRunnerHandle::Get(),
        std::move(release_callback)));
    request->SendResult(std::make_unique<CopyOutputTextureResult>(
        geometry.result_bounds, mailbox, gpu::SyncToken(), color_space,
        std::move(main_callback)));
  } else {
    NOTREACHED();
  }
  ScheduleCheckReadbackCompletion();
  return true;
}

void SkiaOutputSurfaceImplOnGpu::DestroySharedImageOnImplThread(
    std::unique_ptr<gpu::SharedImageRepresentationSkia> representation,
    scoped_refptr<gpu::SharedContextState> context_state,
    const gpu::SyncToken& sync_token,
    bool is_lost) {
  context_state_->MakeCurrent(nullptr);
  representation.reset();
}

void SkiaOutputSurfaceImplOnGpu::BeginAccessImages(
    const std::vector<ImageContextImpl*>& image_contexts,
    std::vector<GrBackendSemaphore>* begin_semaphores,
    std::vector<GrBackendSemaphore>* end_semaphores) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::BeginAccessImages");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  for (auto* context : image_contexts) {
    // Prepare for accessing render pass.
    if (context->render_pass_id()) {
      // We don't cache promise image for render pass, so the it should always
      // be nullptr.
      auto it = offscreen_surfaces_.find(context->render_pass_id());
      DCHECK(it != offscreen_surfaces_.end());
      context->set_promise_image_texture(sk_ref_sp(it->second.fulfill()));
      if (!context->promise_image_texture()) {
        DLOG(ERROR) << "Failed to fulfill the promise texture created from "
                       "CompositorRenderPassId:"
                    << context->render_pass_id();
      }
    } else {
      context->BeginAccessIfNecessary(
          context_state_.get(), shared_image_representation_factory_.get(),
          dependency_->GetMailboxManager(), begin_semaphores, end_semaphores);
      if (context->end_access_state())
        image_contexts_with_end_access_state_.emplace(context);
    }
  }
}

void SkiaOutputSurfaceImplOnGpu::ResetStateOfImages() {
  for (auto* context : image_contexts_with_end_access_state_) {
    DCHECK(context->end_access_state());
    if (!gr_context()->setBackendTextureState(
            context->promise_image_texture()->backendTexture(),
            *context->end_access_state())) {
      DLOG(ERROR) << "setBackendTextureState() failed.";
    }
  }
  image_contexts_with_end_access_state_.clear();
}

void SkiaOutputSurfaceImplOnGpu::EndAccessImages(
    const base::flat_set<ImageContextImpl*>& image_contexts) {
  TRACE_EVENT0("viz", "SkiaOutputSurfaceImplOnGpu::EndAccessImages");
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(image_contexts_with_end_access_state_.empty());
  for (auto* context : image_contexts)
    context->EndAccessIfNecessary();
}

sk_sp<GrContextThreadSafeProxy>
SkiaOutputSurfaceImplOnGpu::GetGrContextThreadSafeProxy() {
  return gr_context() ? gr_context()->threadSafeProxy() : nullptr;
}

void SkiaOutputSurfaceImplOnGpu::ReleaseImageContexts(
    std::vector<std::unique_ptr<ExternalUseClient::ImageContext>>
        image_contexts,
    uint64_t sync_fence_release) {
  DCHECK(!image_contexts.empty());
  // The window could be destroyed already, and the MakeCurrent will fail with
  // an destroyed window, so MakeCurrent without requiring the fbo0.
  if (!MakeCurrent(false /* need_fbo0 */)) {
    for (const auto& context : image_contexts)
      context->OnContextLost();
  }

  image_contexts.clear();
  ReleaseFenceSyncAndPushTextureUpdates(sync_fence_release);
}

void SkiaOutputSurfaceImplOnGpu::ScheduleOverlays(
    SkiaOutputSurface::OverlayList overlays,
    std::vector<ImageContextImpl*> image_contexts) {
#if defined(OS_APPLE)
  std::vector<GrBackendSemaphore> begin_semaphores;
  std::vector<GrBackendSemaphore> end_semaphores;
  promise_image_access_helper_.BeginAccess(std::move(image_contexts),
                                           &begin_semaphores, &end_semaphores);
  // GL is used on MacOS and GL doesn't need semaphores.
  DCHECK(begin_semaphores.empty());
  DCHECK(end_semaphores.empty());
  for (auto& overlay : overlays) {
    if (!overlay.ddl)
      continue;

    base::Optional<gpu::raster::GrShaderCache::ScopedCacheUse> cache_use;
    if (dependency_->GetGrShaderCache()) {
      cache_use.emplace(dependency_->GetGrShaderCache(),
                        gpu::kDisplayCompositorClientId);
    }

    const auto& characterization = overlay.ddl->characterization();
    auto backing = GetOrCreateRenderPassOverlayBacking(characterization);
    DCHECK(overlay.mailbox.IsZero());
    overlay.mailbox = backing->mailbox();
    auto scoped_access = backing->BeginScopedWriteAccess(
        /*final_msaa_count=*/0, characterization.surfaceProps(),
        /*begin_semaphores=*/nullptr,
        /*end_semaphores=*/nullptr,
        gpu::SharedImageRepresentation::AllowUnclearedAccess::kYes);
    bool result = scoped_access->surface()->draw(overlay.ddl);
    DCHECK(result);
    context_state_->gr_context()->flushAndSubmit();
    scoped_access.reset();
    backing->SetCleared();
    in_flight_render_pass_overlay_backings_.insert(std::move(backing));
  }
  promise_image_access_helper_.EndAccess();
  output_device_->ScheduleOverlays(std::move(overlays));

  // Release any backings which are not reused by the current frame, probably
  // because the properties of render passes are changed or render passes are
  // removed.
  available_render_pass_overlay_backings_.clear();
#else
  DCHECK(image_contexts.empty());
  output_device_->ScheduleOverlays(std::move(overlays));
#endif
}

void SkiaOutputSurfaceImplOnGpu::SetEnableDCLayers(bool enable) {
  if (!MakeCurrent(false /* need_fbo0 */))
    return;
  output_device_->SetEnableDCLayers(enable);
}

void SkiaOutputSurfaceImplOnGpu::SetGpuVSyncEnabled(bool enabled) {
  output_device_->SetGpuVSyncEnabled(enabled);
}

void SkiaOutputSurfaceImplOnGpu::SetFrameRate(float frame_rate) {
  if (gl_surface_)
    gl_surface_->SetFrameRate(frame_rate);
}

void SkiaOutputSurfaceImplOnGpu::SetCapabilitiesForTesting(
    const OutputSurface::Capabilities& capabilities) {
  MakeCurrent(false /* need_fbo0 */);
  // Check that we're using an offscreen surface.
  DCHECK(dependency_->IsOffscreen());
  output_device_ = std::make_unique<SkiaOutputDeviceOffscreen>(
      context_state_, capabilities.output_surface_origin,
      renderer_settings_.requires_alpha_channel, memory_tracker_,
      GetDidSwapBuffersCompleteCallback());
}

bool SkiaOutputSurfaceImplOnGpu::Initialize() {
  TRACE_EVENT1("viz", "SkiaOutputSurfaceImplOnGpu::Initialize",
               "is_using_vulkan", is_using_vulkan());
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
#if defined(USE_OZONE)
  if (features::IsUsingOzonePlatform()) {
    gpu::SurfaceHandle surface_handle = dependency_->GetSurfaceHandle();
    if (surface_handle != gpu::kNullSurfaceHandle) {
      window_surface_ = ui::OzonePlatform::GetInstance()
                            ->GetSurfaceFactoryOzone()
                            ->CreatePlatformWindowSurface(surface_handle);
    }
  }
#endif

  context_state_ = dependency_->GetSharedContextState();
  DCHECK(context_state_);
  if (!context_state_->gr_context()) {
    DLOG(ERROR) << "Failed to create GrContext";
    return false;
  }

  if (is_using_vulkan()) {
    if (!InitializeForVulkan())
      return false;
  } else if (is_using_dawn()) {
    if (!InitializeForDawn())
      return false;
  } else {
    if (!InitializeForGL())
      return false;
  }

  max_resource_cache_bytes_ =
      context_state_->gr_context()->getResourceCacheLimit();
  if (context_state_)
    context_state_->AddContextLostObserver(this);

  return true;
}

bool SkiaOutputSurfaceImplOnGpu::InitializeForGL() {
  gl::GLSurfaceFormat format;
  if (PreferRGB565ResourcesForDisplay() &&
      !renderer_settings_.requires_alpha_channel) {
    format.SetRGB565();
  }

  if (dependency_->IsOffscreen()) {
    gl_surface_ = dependency_->CreateGLSurface(nullptr, format);
    if (!gl_surface_)
      return false;

    output_device_ = std::make_unique<SkiaOutputDeviceOffscreen>(
        context_state_, gfx::SurfaceOrigin::kTopLeft,
        renderer_settings_.requires_alpha_channel, memory_tracker_,
        GetDidSwapBuffersCompleteCallback());
  } else {
    gl_surface_ =
        dependency_->CreateGLSurface(weak_ptr_factory_.GetWeakPtr(), format);

    if (!gl_surface_)
      return false;

    if (MakeCurrent(true /* need_fbo0 */)) {
      if (gl_surface_->IsSurfaceless()) {
#if !defined(OS_WIN)
        output_device_ = std::make_unique<SkiaOutputDeviceBufferQueue>(
            std::make_unique<OutputPresenterGL>(
                gl_surface_, dependency_, shared_image_factory_.get(),
                shared_image_representation_factory_.get()),
            dependency_, shared_image_representation_factory_.get(),
            memory_tracker_, GetDidSwapBuffersCompleteCallback());
#else
        NOTIMPLEMENTED();
#endif
      } else {
        if (dependency_->NeedsSupportForExternalStencil()) {
          output_device_ = std::make_unique<SkiaOutputDeviceWebView>(
              context_state_.get(), gl_surface_, memory_tracker_,
              GetDidSwapBuffersCompleteCallback());
        } else {
          output_device_ = std::make_unique<SkiaOutputDeviceGL>(
              dependency_->GetMailboxManager(),
              shared_image_representation_factory_.get(), context_state_.get(),
              gl_surface_, feature_info_, memory_tracker_,
              GetDidSwapBuffersCompleteCallback());
        }
      }
    } else {
      gl_surface_ = nullptr;
      context_state_ = nullptr;
      LOG(ERROR) << "Failed to make current during initialization.";
      return false;
    }
  }
  DCHECK_EQ(gl_surface_->IsOffscreen(), dependency_->IsOffscreen());
  return true;
}

bool SkiaOutputSurfaceImplOnGpu::InitializeForVulkan() {
#if BUILDFLAG(ENABLE_VULKAN)
  if (dependency_->IsOffscreen()) {
    output_device_ = std::make_unique<SkiaOutputDeviceOffscreen>(
        context_state_, gfx::SurfaceOrigin::kBottomLeft,
        renderer_settings_.requires_alpha_channel, memory_tracker_,
        GetDidSwapBuffersCompleteCallback());
    return true;
  }

#if defined(USE_X11)
  if (!features::IsUsingOzonePlatform()) {
    if (!gpu_preferences_.disable_vulkan_surface) {
      output_device_ = SkiaOutputDeviceVulkan::Create(
          vulkan_context_provider_, dependency_->GetSurfaceHandle(),
          memory_tracker_, GetDidSwapBuffersCompleteCallback());
    }
    if (!output_device_) {
      output_device_ = std::make_unique<SkiaOutputDeviceX11>(
          context_state_, dependency_->GetSurfaceHandle(), memory_tracker_,
          GetDidSwapBuffersCompleteCallback());
    }
    if (output_device_)
      return true;
  }
#endif  // defined(USE_X11)

#if !defined(OS_WIN)
#if defined(OS_FUCHSIA)
  auto output_presenter = OutputPresenterFuchsia::Create(
      window_surface_.get(), dependency_, shared_image_factory_.get(),
      shared_image_representation_factory_.get());
#else   // defined(OS_FUCHSIA)
  auto output_presenter =
      OutputPresenterGL::Create(dependency_, shared_image_factory_.get(),
                                shared_image_representation_factory_.get());
  if (output_presenter) {
    // TODO(https://crbug.com/1012401): don't depend on GL.
    gl_surface_ = output_presenter->gl_surface();
  }
#endif  // !defined(OS_FUCHSIA)
  if (output_presenter) {
    output_device_ = std::make_unique<SkiaOutputDeviceBufferQueue>(
        std::move(output_presenter), dependency_,
        shared_image_representation_factory_.get(), memory_tracker_,
        GetDidSwapBuffersCompleteCallback());
    return true;
  }
#endif  // !defined(OS_WIN)

  auto output_device = SkiaOutputDeviceVulkan::Create(
      vulkan_context_provider_, dependency_->GetSurfaceHandle(),
      memory_tracker_, GetDidSwapBuffersCompleteCallback());
  if (!output_device)
    return false;

#if defined(OS_WIN)
  gpu::SurfaceHandle child_surface = output_device->GetChildSurfaceHandle();
  if (child_surface != gpu::kNullSurfaceHandle) {
    DidCreateAcceleratedSurfaceChildWindow(dependency_->GetSurfaceHandle(),
                                           child_surface);
  }
#endif  // defined(OS_WIN)
  output_device_ = std::move(output_device);
  return true;
#else   // BUILDFLAG(ENABLE_VULKAN)
  return false;
#endif  // !BUILDFLAG(ENABLE_VULKAN)
}

bool SkiaOutputSurfaceImplOnGpu::InitializeForDawn() {
#if BUILDFLAG(SKIA_USE_DAWN)
  if (dependency_->IsOffscreen()) {
    output_device_ = std::make_unique<SkiaOutputDeviceOffscreen>(
        context_state_, gfx::SurfaceOrigin::kBottomLeft,
        renderer_settings_.requires_alpha_channel, memory_tracker_,
        GetDidSwapBuffersCompleteCallback());
  } else {
#if defined(USE_X11)
    // TODO(sgilhuly): Set up a Vulkan swapchain so that Linux can also use
    // SkiaOutputDeviceDawn.
    if (!features::IsUsingOzonePlatform()) {
      output_device_ = std::make_unique<SkiaOutputDeviceX11>(
          context_state_, dependency_->GetSurfaceHandle(), memory_tracker_,
          GetDidSwapBuffersCompleteCallback());
    } else {
      return false;
    }
#elif defined(OS_WIN)
    std::unique_ptr<SkiaOutputDeviceDawn> output_device =
        std::make_unique<SkiaOutputDeviceDawn>(
            dawn_context_provider_, dependency_->GetSurfaceHandle(),
            gfx::SurfaceOrigin::kTopLeft, memory_tracker_,
            GetDidSwapBuffersCompleteCallback());
    const gpu::SurfaceHandle child_surface_handle =
        output_device->GetChildSurfaceHandle();
    DidCreateAcceleratedSurfaceChildWindow(dependency_->GetSurfaceHandle(),
                                           child_surface_handle);
    output_device_ = std::move(output_device);
#else
    NOTREACHED();
    return false;
#endif
  }
#endif
  return true;
}

bool SkiaOutputSurfaceImplOnGpu::MakeCurrent(bool need_fbo0) {
  // need_fbo0 implies need_gl too.
  bool need_gl = need_fbo0;
  // Only make current with |gl_surface_|, if following operations will use
  // fbo0.
  auto* gl_surface = need_fbo0 ? gl_surface_.get() : nullptr;
  if (!context_state_->MakeCurrent(gl_surface, need_gl)) {
    LOG(ERROR) << "Failed to make current.";
    dependency_->DidLoseContext(
        *context_state_->context_lost_reason(),
        GURL("chrome://gpu/SkiaOutputSurfaceImplOnGpu::MakeCurrent"));
    MarkContextLost(GetContextLostReason(
        gpu::error::kLostContext, *context_state_->context_lost_reason()));
    return false;
  }
  context_state_->set_need_context_state_reset(true);
  return true;
}

void SkiaOutputSurfaceImplOnGpu::PullTextureUpdates(
    std::vector<gpu::SyncToken> sync_tokens) {
  // TODO(https://crbug.com/900973): Remove it when MailboxManager is replaced
  // with SharedImage API.
  if (dependency_->GetMailboxManager()->UsesSync()) {
    for (auto& sync_token : sync_tokens)
      dependency_->GetMailboxManager()->PullTextureUpdates(sync_token);
  }
}

void SkiaOutputSurfaceImplOnGpu::ReleaseFenceSyncAndPushTextureUpdates(
    uint64_t sync_fence_release) {
  // TODO(https://crbug.com/900973): Remove it when MailboxManager is replaced
  // with SharedImage API.
  if (dependency_->GetMailboxManager()->UsesSync()) {
    // If MailboxManagerSync is used, we are sharing textures between threads.
    // In this case, sync point can only guarantee GL commands are issued in
    // correct order across threads and GL contexts. However GPU driver may
    // execute GL commands out of the issuing order across GL contexts. So we
    // have to use PushTextureUpdates() and PullTextureUpdates() to ensure the
    // correct GL commands executing order.
    // PushTextureUpdates(token) will insert a GL fence into the current GL
    // context, PullTextureUpdates(token) will wait the GL fence associated with
    // the give token on the current GL context.
    // Reconstruct sync_token from sync_fence_release.
    gpu::SyncToken sync_token(
        gpu::CommandBufferNamespace::VIZ_SKIA_OUTPUT_SURFACE,
        command_buffer_id(), sync_fence_release);
    dependency_->GetMailboxManager()->PushTextureUpdates(sync_token);
  }
  sync_point_client_state_->ReleaseFenceSync(sync_fence_release);
}

void SkiaOutputSurfaceImplOnGpu::SwapBuffersInternal(
    base::OnceCallback<bool()> deferred_framebuffer_draw_closure,
    OutputSurfaceFrame* frame) {
  DCHECK_CALLED_ON_VALID_THREAD(thread_checker_);
  DCHECK(output_device_);

  if (deferred_framebuffer_draw_closure) {
    // Returns false if context not set to current, i.e lost
    if (!std::move(deferred_framebuffer_draw_closure).Run())
      return;
    DCHECK(context_state_->IsCurrent(nullptr /* surface */));
  } else {
    if (!MakeCurrent(!dependency_->IsOffscreen() /* need_fbo0 */))
      return;
  }

  ResetStateOfImages();
  output_device_->PreGrContextSubmit();
  gr_context()->submit();
  promise_image_access_helper_.EndAccess();
  scoped_output_device_paint_.reset();

  if (frame) {
    if (output_surface_plane_)
      DCHECK(output_device_->IsPrimaryPlaneOverlay());
    output_device_->SchedulePrimaryPlane(output_surface_plane_);
    output_surface_plane_.reset();

    if (frame->sub_buffer_rect) {
      if (capabilities().supports_post_sub_buffer) {
        if (capabilities().output_surface_origin ==
            gfx::SurfaceOrigin::kBottomLeft) {
          frame->sub_buffer_rect->set_y(size_.height() -
                                        frame->sub_buffer_rect->y() -
                                        frame->sub_buffer_rect->height());
        }
        output_device_->PostSubBuffer(*frame->sub_buffer_rect,
                                      buffer_presented_callback_,
                                      std::move(frame->latency_info));

      } else if (capabilities().supports_commit_overlay_planes) {
        // CommitOverlayPlanes() can only be used for empty swap.
        DCHECK(frame->sub_buffer_rect->IsEmpty());
        output_device_->CommitOverlayPlanes(buffer_presented_callback_,
                                            std::move(frame->latency_info));
      } else {
        NOTREACHED();
      }
    } else {
      output_device_->SwapBuffers(buffer_presented_callback_,
                                  std::move(frame->latency_info));
    }
  }

  destroy_after_swap_.clear();
  context_state_->UpdateSkiaOwnedMemorySize();
#if BUILDFLAG(ENABLE_VULKAN)
  if (is_using_vulkan())
    gpu::ReportUMAPerSwapBuffers();
#endif
}

bool SkiaOutputSurfaceImplOnGpu::IsDisplayedAsOverlay() {
  return output_device_->IsPrimaryPlaneOverlay();
}

#if defined(OS_WIN)
void SkiaOutputSurfaceImplOnGpu::DidCreateAcceleratedSurfaceChildWindow(
    gpu::SurfaceHandle parent_window,
    gpu::SurfaceHandle child_window) {
  dependency_->DidCreateAcceleratedSurfaceChildWindow(parent_window,
                                                      child_window);
}
#endif

const gpu::gles2::FeatureInfo* SkiaOutputSurfaceImplOnGpu::GetFeatureInfo()
    const {
  return feature_info_.get();
}

const gpu::GpuPreferences& SkiaOutputSurfaceImplOnGpu::GetGpuPreferences()
    const {
  return gpu_preferences_;
}

GpuVSyncCallback SkiaOutputSurfaceImplOnGpu::GetGpuVSyncCallback() {
  return gpu_vsync_callback_;
}

base::TimeDelta SkiaOutputSurfaceImplOnGpu::GetGpuBlockedTimeSinceLastSwap() {
  return dependency_->GetGpuBlockedTimeSinceLastSwap();
}

void SkiaOutputSurfaceImplOnGpu::DidSwapBuffersComplete(
    gpu::SwapBuffersCompleteParams params) {
  // Handled by SkiaOutputDevice already.
}

void SkiaOutputSurfaceImplOnGpu::BufferPresented(
    const gfx::PresentationFeedback& feedback) {
  // Handled by SkiaOutputDevice already.
}

void SkiaOutputSurfaceImplOnGpu::DidSwapBuffersCompleteInternal(
    gpu::SwapBuffersCompleteParams params,
    const gfx::Size& pixel_size) {
  if (params.swap_response.result == gfx::SwapResult::SWAP_FAILED) {
    DLOG(ERROR) << "Context lost on SWAP_FAILED";
    if (!context_state_->IsCurrent(nullptr) ||
        !context_state_->CheckResetStatus(false)) {
      // Mark the context lost if not already lost.
      MarkContextLost(ContextLostReason::CONTEXT_LOST_SWAP_FAILED);
    }
  }

#if defined(OS_APPLE)
  // |available_render_pass_overlay_backings_| are used or released in
  // ScheduleOverlays() for every frames.
  DCHECK(available_render_pass_overlay_backings_.empty());

  // Erase mailboxes of render pass overlays from |params.released_overlays| and
  // move released backings for those render pass overlays from
  // |in_flight_render_pass_overlay_backings_| to
  // |available_render_pass_overlay_backings_| for reusing.
  base::EraseIf(params.released_overlays, [this](const gpu::Mailbox& mailbox) {
    auto it = in_flight_render_pass_overlay_backings_.find(mailbox);
    if (it == in_flight_render_pass_overlay_backings_.end())
      return false;
    available_render_pass_overlay_backings_.push_back(std::move(*it));
    in_flight_render_pass_overlay_backings_.erase(it);
    return true;
  });

#endif

  PostTaskToClientThread(
      base::BindOnce(did_swap_buffer_complete_callback_, params, pixel_size));
}

SkiaOutputSurfaceImplOnGpu::DidSwapBufferCompleteCallback
SkiaOutputSurfaceImplOnGpu::GetDidSwapBuffersCompleteCallback() {
  return base::BindRepeating(
      &SkiaOutputSurfaceImplOnGpu::DidSwapBuffersCompleteInternal, weak_ptr_);
}

void SkiaOutputSurfaceImplOnGpu::OnContextLost() {
  MarkContextLost(ContextLostReason::CONTEXT_LOST_UNKNOWN);
}

void SkiaOutputSurfaceImplOnGpu::MarkContextLost(ContextLostReason reason) {
  // This function potentially can be re-entered during from
  // SharedContextState::MarkContextLost(). This guards against it.
  if (context_is_lost_)
    return;
  context_is_lost_ = true;

  UMA_HISTOGRAM_ENUMERATION("GPU.ContextLost.DisplayCompositor", reason);

  context_state_->MarkContextLost();
  if (context_lost_callback_) {
    PostTaskToClientThread(std::move(context_lost_callback_));
  }
}

void SkiaOutputSurfaceImplOnGpu::ScheduleCheckReadbackCompletion() {
  if (num_readbacks_pending_ > 0 && !readback_poll_pending_) {
    dependency_->ScheduleDelayedGPUTaskFromGPUThread(
        base::BindOnce(&SkiaOutputSurfaceImplOnGpu::CheckReadbackCompletion,
                       weak_ptr_factory_.GetWeakPtr()));
    readback_poll_pending_ = true;
  }
}

void SkiaOutputSurfaceImplOnGpu::CheckReadbackCompletion() {
  readback_poll_pending_ = false;

  // If there are no pending readback requests or we can't make the context
  // current then exit. There is no thing to do here.
  if (num_readbacks_pending_ == 0 || !MakeCurrent(/*need_fbo0=*/false))
    return;

  gr_context()->checkAsyncWorkCompletion();
  ScheduleCheckReadbackCompletion();
}

#if defined(OS_APPLE)
std::unique_ptr<gpu::SharedImageRepresentationSkia>
SkiaOutputSurfaceImplOnGpu::GetOrCreateRenderPassOverlayBacking(
    const SkSurfaceCharacterization& characterization) {
  ResourceFormat resource_format;
  switch (characterization.colorType()) {
    case kRGBA_8888_SkColorType:
      resource_format = ResourceFormat::RGBA_8888;
      break;
    case kBGRA_8888_SkColorType:
      resource_format = ResourceFormat::BGRA_8888;
      break;
    case kRGBA_F16_SkColorType:
      resource_format = ResourceFormat::RGBA_F16;
      break;
    default:
      resource_format = ResourceFormat::RGBA_8888;
      NOTREACHED();
  }

  gfx::Size size(characterization.width(), characterization.height());
  gfx::ColorSpace color_space(*characterization.colorSpace());

  auto it = std::find_if(
      available_render_pass_overlay_backings_.begin(),
      available_render_pass_overlay_backings_.end(),
      [&characterization, &resource_format, &size, &color_space](
          const std::unique_ptr<gpu::SharedImageRepresentationSkia>& backing) {
        return backing->format() == resource_format &&
               backing->size() == size &&
               backing->color_space() == color_space &&
               backing->surface_origin() == characterization.origin() &&
               backing->alpha_type() ==
                   characterization.imageInfo().alphaType();
      });

  if (it != available_render_pass_overlay_backings_.end()) {
    auto backing = std::move(*it);
    available_render_pass_overlay_backings_.erase(it);
    return backing;
  }

  auto mailbox = gpu::Mailbox::GenerateForSharedImage();
  constexpr auto kOverlayUsage = gpu::SHARED_IMAGE_USAGE_SCANOUT |
                                 gpu::SHARED_IMAGE_USAGE_DISPLAY |
                                 gpu::SHARED_IMAGE_USAGE_RASTER;

  bool result = shared_image_factory_->CreateSharedImage(
      mailbox, resource_format, size, color_space, characterization.origin(),
      characterization.imageInfo().alphaType(), gpu::kNullSurfaceHandle,
      kOverlayUsage);
  if (!result) {
    LOG(ERROR) << "CreateSharedImage() failed.";
    return nullptr;
  }

  auto backing = shared_image_representation_factory_->ProduceSkia(
      mailbox, context_state_.get());
  DCHECK(backing);

  // The |backing| will keep a ref on the shared image, so the image will not be
  // released until |backing| is released.
  shared_image_factory_->DestroySharedImage(mailbox);

  return backing;
}
#endif

}  // namespace viz
