/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var bench = require( '@stdlib/bench' );
var randu = require( '@stdlib/random/base/randu' );
var floor = require( '@stdlib/math/base/special/floor' );
var isNonNegativeIntegerArray = require( '@stdlib/assert/is-nonnegative-integer-array' ).primitives;
var shape2strides = require( './../../../base/shape2strides' );
var strides2offset = require( './../../../base/strides2offset' );
var numel = require( './../../../base/numel' );
var pkg = require( './../package.json' ).name;
var ind2sub = require( './../lib' );


// MAIN //

bench( pkg+':mode=throw,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=throw,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::memory_reuse:assign:mode=throw,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var s;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	out = [ 0, 0, 0 ];

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		s = ind2sub.assign( shape, strides, offset, order, idx, 'throw', out );
		if ( s !== out ) {
			b.fail( 'should return output array' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( s ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::memory_reuse:assign:mode=throw,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var s;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	out = [ 0, 0, 0 ];

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		s = ind2sub.assign( shape, strides, offset, order, idx, 'throw', out );
		if ( out !== s ) {
			b.fail( 'should return output array' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( s ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=throw,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=throw,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=throw,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=throw,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*len );
		out = ind2sub( shape, strides, offset, order, idx, 'throw' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=wrap,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=wrap,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=wrap,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=wrap,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=wrap,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=wrap,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len*10) ) - (len*5);
		out = ind2sub( shape, strides, offset, order, idx, 'wrap' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=clamp,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10.0;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+':mode=clamp,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	strides = shape2strides( shape, order );
	offset = strides2offset( shape, strides );
	len = numel( shape );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10.0;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=clamp,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:offset=0,mode=clamp,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = 0;

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=clamp,order=row-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'row-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10.0;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});

bench( pkg+'::negative_strides:mode=clamp,order=column-major', function benchmark( b ) {
	var strides;
	var offset;
	var order;
	var shape;
	var out;
	var len;
	var idx;
	var i;

	shape = [ 10, 10, 10 ];
	order = 'column-major';
	len = numel( shape );
	strides = shape2strides( shape, order );
	for ( i = 0; i < shape.length; i++ ) {
		strides[ i ] *= -1;
	}
	offset = strides2offset( shape, strides );

	b.tic();
	for ( i = 0; i < b.iterations; i++ ) {
		idx = floor( randu()*(len+20) ) - 10.0;
		out = ind2sub( shape, strides, offset, order, idx, 'clamp' );
		if ( out.length !== shape.length ) {
			b.fail( 'should have expected length' );
		}
	}
	b.toc();
	if ( !isNonNegativeIntegerArray( out ) ) {
		b.fail( 'should return an array' );
	}
	b.pass( 'benchmark finished' );
	b.end();
});
