// Copyright (C) 2005 Open Source Telecom Corp.
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software 
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <cc++/process.h>
#include <cc++/slog.h>
#include "server.h"

#ifdef	HAVE_LIBEXEC

#include <sys/wait.h>

#define	LIBEXEC_BUFFER 64

namespace server {
using namespace ost;
using namespace std;

int Libexec::iopair[2] = {-1, -1};
Libexec *Libexec::libexec = NULL;
bool Libexec::exiting = false;

typedef struct
{
	int pid;
	char tsid[16];
}	slot_t;

static slot_t *slot;
static int pid = 0;
static int pri = 1;
static bool err = false;
static int buffers = 1024;
static socklen_t blen = sizeof(buffers);
static const char *libpath;
static int slots;
static int iosock;
static int interval = 5;
static int server_pid = 0;

extern "C" {

RETSIGTYPE timer(int signo)
{
	if(server_pid && kill(server_pid, 0))
	{
		if(err)
			fprintf(stderr, "libexec exiting; server lost\n");
		else
			slog.notice("server lost...");
		::exit(0);
	}
	alarm(interval);
}

RETSIGTYPE child(int signo)
{
	int status;
	unsigned count;
	char buf[65];
	
	while((pid = wait3(&status, WNOHANG, 0)) > 0)
	{
		count = 0;
		while(count < slots)
		{
			if(slot[count].pid == pid)
				break;
			++count;
		}

		if(count == slots)
		{
			if(err)
				fprintf(stderr, "libexec exiting; unknown pid=%d\n", pid);
			continue;
		}

		if(err)
			fprintf(stderr, "libexec exiting; timeslot=%d, pid=%d, status=%d\n",
				count, pid, WEXITSTATUS(status));

		snprintf(buf, sizeof(buf), "%s exit %d\n",
			slot[count].tsid, WEXITSTATUS(status));

		write(iosock, buf, strlen(buf));
		slot[count].tsid[0] = 0;
		slot[count].pid = 0;
	}
}

};

Libexec::Libexec() :
Thread(0)
{
}

Libexec::~Libexec()
{
	exiting = true;
	write(iopair[0], "down\n", 5);
	terminate();
}

void Libexec::run(void)
{
	BayonneSession *s;
	Event event;
	char buf[512];
	char *tok;
	const char *cp, *tsid, *cmd, *var;
	int pid, reason;
	
	for(;;)
	{
		readline(buf, sizeof(buf));
		if(exiting)
			Thread::sync();

		cp = strtok_r(buf, " \t\r\n", &tok);
		if(!cp || !*cp)
			continue;

		// must be a transaction id

		if(!strchr(cp, '+'))
		{
			slog.error("libexec invalid request; tid=%s", cp);
			continue;
		}

		s = getSid(cp);

		if(!s)
		{
			slog.notice("libexec invalid or expired transaction; tid=%s", cp);
			continue;
		}

		tsid = cp;
		cmd = strtok_r(NULL, " \t\r\n", &tok);
		if(!cmd)
		{
			slog.error("libexec; command missing");
			continue;
		}

		if(!stricmp(cmd, "start"))
		{
			event.id = ENTER_LIBEXEC;
			event.libexec.tid = tsid;
			cp = strtok_r(NULL, " \t\r\n", &tok);
			pid = event.libexec.pid = atoi(cp);
			cp = strtok_r(NULL, " \t\r\n", &tok);
			event.libexec.fname = cp;
			if(!s->postEvent(&event))
			{
				slog.error("libexec start failed; pid=%d", pid);
				kill(pid, SIGINT);
			}
			else
				slog.debug("libexec started; pid=%d, tsid=%s", pid, tsid);
		}
		else if(!stricmp(cmd, "hangup"))
		{
                        event.id = DROP_LIBEXEC;
                        event.libexec.tid = tsid;
                        if(!s->postEvent(&event))
				slog.error("libexec hangup unknown transaction; tsid=%s", tsid);
                        else
                                slog.debug("libexec hangup; tsid=%s", tsid);
                }
                else if(!strnicmp(cmd, "stat", 4))
                {
                        event.id = STAT_LIBEXEC;
                        event.libexec.tid = tsid;
                        if(!s->postEvent(&event))
				slog.error("libexec status unknown transaction; tsid=%s", tsid);
                        else
                                slog.debug("libexec status; tsid=%s", tsid);
                }
		else if(!stricmp(cmd, "head"))
		{
			event.id = HEAD_LIBEXEC;
			event.libexec.tid = tsid;
			if(!s->postEvent(&event))
slog.error("libexec header unknown transaction; tsid=%s", tsid);
			else
				slog.debug("libexec header; tsid=%s", tsid);
		}
                else if(!stricmp(cmd, "read"))
                {
                        event.id = GET_LIBEXEC;
                        event.libinput.tid = tsid;
			cp = strtok_r(NULL, " \t\r\n", &tok);
			if(!cp)
				cp = "6";

			event.libinput.timeout = atol(cp);
			if(event.libinput.timeout < 250)
				event.libinput.timeout *= 1000;

			cp = strtok_r(NULL, " \t\r\n", &tok);
			if(cp)
			{
				event.libinput.exit = "#";
				event.libinput.count = atoi(cp);
			}
			else
			{
				event.libinput.count = 1;
				event.libinput.exit = NULL;
			}
                        if(!s->postEvent(&event))
				slog.error("libexec input unknown transaction; tsid=%s", tsid);
                        else
                                slog.debug("libexec input; tsid=%s", tsid);
                }
		else if(!stricmp(cmd, "clr"))
		{
                        var = strtok_r(NULL, " \t\r\n", &tok);
                        event.varexec.value = "";
                        event.varexec.size = 0;
                        goto var;
		}
		else if(!stricmp(cmd, "add"))
		{
                        var = strtok_r(NULL, " \t\r\n", &tok);
                        event.varexec.value = strtok_r(NULL, " \t\r\n", &tok);
                        event.varexec.size = -1;
                        goto var;
		}
		else if(!stricmp(cmd, "set"))
		{
                        var = strtok_r(NULL, " \t\r\n", &tok);
                        event.varexec.value = strtok_r(NULL, " \t\r\n", &tok);
			event.varexec.size = 0;
			goto var;
		}
		else if(!stricmp(cmd, "new"))
		{
			var = strtok_r(NULL, " \t\r\n", &tok);
			cp = strtok_r(NULL, " \t\r\n", &tok);
			if(!cp)
				cp = "64";
			event.varexec.size = atoi(cp);
			event.varexec.value = "";
			goto var;
		}			
		else if(!stricmp(cmd, "get"))
		{
			var = strtok_r(NULL, " \t\r\n", &tok);
			event.varexec.value = NULL;
			event.varexec.size = 0;
var:
			event.id = VAR_LIBEXEC;
			if(!var)
				continue;
			event.varexec.name = var;
			event.varexec.tid = tsid;			
			goto post;
		}
                else if(!stricmp(cmd, "args"))
                {
                        event.id = ARGS_LIBEXEC;
                        event.libexec.tid = tsid;
post:
                        if(!s->postEvent(&event))
				slog.error("libexec args unknown transaction; tsid=%s", tsid);
                        else
                                slog.debug("libexec args; tsid=%s", tsid);
                }
		else if(!stricmp(cmd, "exit"))
		{
			event.id = EXIT_LIBEXEC;
			event.libexec.tid = tsid;
			cp = strtok_r(NULL, " \t\r\n", &tok);
			if(!cp)
				cp = "255";
			reason = event.libexec.result = atoi(cp);
			if(!s->postEvent(&event))
				slog.error("libexec exiting unknown transaction");
			else
				slog.debug("libexec exiting; tsid=%s", tsid);
		}
		else
			slog.error("libexec unknown request; cmd=%s", cmd);
	}
}

void Libexec::startup(void)
{
	libexec = new Libexec();
	libexec->start();
}

void Libexec::execute(unsigned ts, const char *cmd)
{
	int pid, fd;
	char path[256];
	char fname[32];
	char buf[65];
	char *args[2];
	unsigned count = 3;

	snprintf(path, sizeof(path), "%s/%s", libpath, cmd);
	snprintf(fname, sizeof(fname), "%s/.libexec-%d", path_tmp, ts);

	args[0] = path;
	args[1] = NULL;

	pid = fork();
	if(pid)
	{
		slot[ts].pid = pid;

		if(err)
			fprintf(stderr, "libexec starting; timeslot=%d, pid=%d, cmd=%s\n",
				ts, pid, cmd);
		return;
	}

	Process::setPosixSignal(SIGCHLD, SIG_DFL);
	dup2(iopair[0], 1);
	remove(fname);
	mkfifo(fname, 0660);
	fd = open(fname, O_RDWR);
	dup2(fd, 0);

	while(count < 20)
		::close(count++);

	snprintf(buf, sizeof(buf), "%d", ts);
	Process::setEnv("PORT_TIMESLOT", strdup(buf), true);
	Process::setEnv("PORT_TSESSION", slot[ts].tsid, true);

	snprintf(buf, sizeof(buf), "%s start %d %s\n",
		slot[ts].tsid, getpid(), fname);
	write(1, buf, strlen(buf));

	execv(path, args);
	::exit(-1);		
}

void Libexec::readline(char *buf, unsigned max)
{
	unsigned count = 0;
	*buf = 0;

	--max;

	while(count < max)
	{
		if(read(iopair[1], buf + count, 1) < 1)
			break;
		
		if(buf[count] == '\n')
			break;

		++count;
	}
	buf[count] = 0;
}
	
void Libexec::allocate(void)
{
	const char *cp = keyengine.getLast("gateways");
	char buf[LIBEXEC_BUFFER];
	int len;
	int fd = dup(2);
	char *p;
	unsigned ts;
	long opt;
	char path[256];

	slots = atoi(keyserver.getLast("timeslots"));

	libpath = keypaths.getLast("libexec");

	if(cp)
		pri = atoi(cp);

	cp = keyengine.getLast("buffers");
	if(cp)
		buffers = atoi(cp) * 1024;

	if(getppid() > 1)
		err = true;

	if(socketpair(AF_UNIX, SOCK_STREAM, 0, iopair))
	{
		slog.error("libexec: cannot create socket pair");
		return;
	}

	setsockopt(iopair[0], SOL_SOCKET, SO_RCVBUF, &buffers, blen);
	setsockopt(iopair[1], SOL_SOCKET, SO_RCVBUF, &buffers, blen);
	iosock = iopair[0];

	pid = fork();
	if(pid)
	{
		Process::join(pid);
		::close(fd);
		snprintf(buf, sizeof(buf), "serv%d", getpid());
		write(iopair[1], buf, sizeof(buf));
		return;
	}

	Process::detach();
	dup2(fd, 2);
	::close(fd);

	nice(pri);

	Process::setEnv("SERVER_PROMPTS", keypaths.getLast("prompts"), true);
	Process::setEnv("SERVER_SCRIPTS", keypaths.getLast("scripts"), true);
	Process::setEnv("SERVER_LIBEXEC", libpath, true);
	Process::setEnv("SERVER_SOFTWARE", "bayonne", true);
	Process::setEnv("SERVER_VERSION", VERSION, true);
	Process::setEnv("SERVER_PROTOCOL", "4.0", true);
	Process::setEnv("SERVER_TOKEN", " ", true);
	Process::setEnv("SERVER_TMP", path_tmp, true);
	Process::setEnv("SERVER_TMPFS", path_tmpfs, true);
	snprintf(path, sizeof(path), "%s:/bin:/usr/bin:/usr/local/bin", libpath);
	Process::setEnv("PATH", strdup(path), true);

	slog.open("bayonne", Slog::classDaemon);
	slog.level(Slog::levelInfo);

	slog.info("libexec starting; path=%s", libpath);
	if(err)
		fprintf(stderr, "libexec starting; path=%s\n", libpath);

	slot = new slot_t[slots];

	Process::setPosixSignal(SIGCHLD, child);
	Process::setPosixSignal(SIGALRM, timer);

	memset(slot, 0, sizeof(slot_t) * slots);
	alarm(interval);

	for(;;)
	{	
		len = ::read(iopair[0], buf, sizeof(buf));
		if(len < sizeof(buf))
			continue;

		if(!stricmp(buf, "down"))
		{
			if(err)
				fprintf(stderr, "libexec exiting; server down\n");
			else
				slog.notice("libexec exiting...");
			::exit(0);
		}

		if(!strnicmp(buf, "serv", 4))
			server_pid = atoi(buf + 4);

		p = strchr(buf, ' ');
		if(!p)
			continue;

		*(p++) = 0;
		ts = atoi(buf);
		strcpy(slot[ts].tsid, buf);
		slot[ts].pid = 0;
		execute(ts, p);
	}		
}

bool Libexec::create(BayonneSession *s)
{
	char buf[LIBEXEC_BUFFER];

	if(iopair[1] < 0)
		return false;

	s->newTid();
	snprintf(buf, sizeof(buf), "%04d+%08lx %s", 
		s->getSlot(), s->newTid(), s->getValue());
	write(iopair[1], buf, sizeof(buf));
	return true;
}

void Libexec::cleanup(void)
{
	char buf[LIBEXEC_BUFFER];

	setString(buf, sizeof(buf), "down");

	if(iopair[1] > -1)
		write(iopair[1], buf, sizeof(buf));

	if(libexec)
	{
		delete libexec;
		libexec = NULL;
	}
}

} // end namespace

#endif
