//!
//! Example for the `serialize` and `deserialize` functions in the `singleton_map_recursive` module.
//!
//! This example demonstrates the usage of the `serialize` and `deserialize` functions
//! from the `singleton_map_recursive` module to serialize and deserialize a
//! struct field that contains an enum with multiple variants into a YAML map
//! with a single key-value pair, where the key is the enum variant name and the
//! value is the inner value of the enum variant.
//!

use serde::{Deserialize, Serialize};
use serde_yml::with::singleton_map_recursive;

#[derive(Serialize, Deserialize, PartialEq, Debug)]
enum MyEnum {
    Variant1(String),
    Variant2 { field: i32 },
}

#[derive(Serialize, Deserialize, PartialEq, Debug)]
struct MyStruct {
    #[serde(with = "singleton_map_recursive")]
    field: MyEnum,
}

pub(crate) fn main() {
    // Print a message to indicate the file being executed.
    println!("\n❯ Executing examples/with/singleton_map_recursive_serialize_deserialize.rs");

    let input = MyStruct {
        field: MyEnum::Variant2 { field: 42 },
    };

    let yaml = serde_yml::to_string(&input).unwrap();
    println!("\n✅ Serialized YAML:\n{}", yaml);

    let output: MyStruct = serde_yml::from_str(&yaml).unwrap();
    println!("\n✅ Deserialized YAML:\n{:#?}", output);
    assert_eq!(input, output);
}
