/*
 * Copyright (c) 2023, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * @test
 * @bug 8311546
 * @summary Adopt de-facto standards on x509 Name Constraints with leading dot. Certs
 * can be generated by running generate-certs.sh
 * @library /test/lib
 * @modules java.base/sun.security.x509
 */

import java.io.*;
import java.nio.file.*;
import java.util.*;
import java.security.Security;
import java.security.cert.*;

public class LeadingPeriod {

    private static CertPath makeCertPath(String targetCertStr,
        PKIXParameters params) throws CertificateException {
        // generate certificate from cert strings
        CertificateFactory cf = CertificateFactory.getInstance("X.509");

        ByteArrayInputStream is;

        is = new ByteArrayInputStream(targetCertStr.getBytes());
        Certificate targetCert = cf.generateCertificate(is);
        // set validity date so that validation won't fail when cert expires
        params.setDate(((X509Certificate)targetCert).getNotBefore());

        // generate certification path
        List<Certificate> list = List.of(targetCert);

        return cf.generateCertPath(list);
    }

    private static PKIXParameters genParams(String caStr) throws Exception {
        // generate certificate from cert string
        CertificateFactory cf = CertificateFactory.getInstance("X.509");

        ByteArrayInputStream is = new ByteArrayInputStream(caStr.getBytes());
        Certificate selfSignedCert = cf.generateCertificate(is);

        // generate a trust anchor
        TrustAnchor anchor = new TrustAnchor((X509Certificate) selfSignedCert, null);

        Set<TrustAnchor> anchors = Collections.singleton(anchor);

        PKIXParameters params = new PKIXParameters(anchors);

        // disable certificate revocation checking
        params.setRevocationEnabled(false);

        return params;
    }

    public static void main(String[] args) throws Exception {

        CertPathValidator validator = CertPathValidator.getInstance("PKIX");

        // Load certs with a NameConstraint where DNS value does not begin with a period
        Path targetFromCAWithoutPeriodPath = Paths.get(System.getProperty(
            "test.src", "./") + "/certs/withoutLeadingPeriod/leaf.pem");
        String targetFromCAWithoutPeriod = Files.readString(targetFromCAWithoutPeriodPath);

        Path caWithoutLeadingPeriodPath = Paths.get(System.getProperty(
            "test.src", "./") + "/certs/withoutLeadingPeriod/ca.pem");
        String caWithoutLeadingPeriod = Files.readString(caWithoutLeadingPeriodPath);

        PKIXParameters paramsForCAWithoutLeadingPeriod = genParams(caWithoutLeadingPeriod);
        CertPath pathWithoutLeadingPeriod = makeCertPath(
            targetFromCAWithoutPeriod, paramsForCAWithoutLeadingPeriod);

        validator.validate(pathWithoutLeadingPeriod, paramsForCAWithoutLeadingPeriod);

        // Load certificates with a NameConstraint where the DNS value does begin with a period
        Path targetFromCAWithPeriodPath = Paths.get(System.getProperty(
            "test.src", "./") + "/certs/withLeadingPeriod/leaf.pem");
        String targetFromCAWithPeriod = Files.readString(targetFromCAWithPeriodPath);

        Path caWithLeadingPeriodPath = Paths.get(System.getProperty(
            "test.src", "./") + "/certs/withLeadingPeriod/ca.pem");
        String caWithLeadingPeriod = Files.readString(caWithLeadingPeriodPath);

        PKIXParameters paramsForCAWithLeadingPeriod = genParams(caWithLeadingPeriod);
        CertPath pathWithLeadingPeriod = makeCertPath(targetFromCAWithPeriod, paramsForCAWithLeadingPeriod);

        validator.validate(pathWithLeadingPeriod, paramsForCAWithLeadingPeriod);
    }
}
