/* Definitions for the new event model;
   created 16-jul-91 by Jamie Zawinski
   Copyright (C) 1991, 1992, 1993 Free Software Foundation, Inc.
   Copyright (C) 1995, 1996, 2002 Ben Wing.

This file is part of XEmacs.

XEmacs is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

XEmacs is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License
along with XEmacs; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

/* Synched up with: Not in FSF. */

#ifndef INCLUDED_events_h_
#define INCLUDED_events_h_

#include "systime.h"

#ifdef USE_KKCC
#include "opaque.h"
#endif /* USE_KKCC */

/* There is one object called an event_stream.  This object contains
   callback functions for doing the window-system-dependent operations
   that XEmacs requires.

   If XEmacs is compiled with support for X11 and the X Toolkit, then this
   event_stream structure will contain functions that can cope with input
   on XEmacs windows on multiple displays, as well as input from dumb tty
   frames.

   If it is desired to have XEmacs able to open frames on the displays of
   multiple heterogeneous machines, X11 and SunView, or X11 and NeXT, for
   example, then it will be necessary to construct an event_stream structure
   that can cope with the given types.  Currently, the only implemented
   event_streams are for dumb-ttys, and for X11 plus dumb-ttys,
   and for mswindows.

   To implement this for one window system is relatively simple.
   To implement this for multiple window systems is trickier and may
   not be possible in all situations, but it's been done for X and TTY.

   Note that these callbacks are *NOT* console methods; that's because
   the routines are not specific to a particular console type but must
   be able to simultaneously cope with all allowable console types.

  The slots of the event_stream structure:

 next_event_cb		A function which fills in an XEmacs_event structure
			with the next event available.  If there is no event
			available, then this should block.

			IMPORTANT: timer events and especially process
			events *must not* be returned if there are
			events of other types available; otherwise you
			can end up with an infinite loop in Fdiscard_input().

 event_pending_cb	A function which says whether there are events to be
			read.  If called with an argument of 0, then this
			should say whether calling the next_event_cb will
			block.  If called with an argument of 1, then this
			should say whether there are user-generated events
			pending (that is, keypresses or mouse-clicks).  This
			is used for redisplay optimization, among other
			things.  On dumb ttys, these two results are the
			same, but under a window system, they are not.

			If this function is not sure whether there are events
			to be read, it *must* return 0.  Otherwise various
			undesirable effects will occur, such as redisplay
			not occurring until the next event occurs.

 handle_magic_event_cb	XEmacs calls this with an event structure which
  			contains window-system dependent information that
			XEmacs doesn't need to know about, but which must
			happen in order.  If the next_event_cb never returns
			an event of type "magic", this will never be used.

 format_magic_event_cb  Called with a magic event; print a representation of
                        the innards of the event to PSTREAM.
                        
 compare_magic_event_cb Called with two magic events; return non-zero if
                        the innards of the two are equal, zero otherwise.

 hash_magic_event_cb    Called with a magic event; return a hash of the
                        innards of the event.

 add_timeout_cb		Called with an EMACS_TIME, the absolute time at
			which a wakeup event should be generated; and a
			void *, which is an arbitrary value that will be
			returned in the timeout event.  The timeouts
			generated by this function should be one-shots:
			they fire once and then disappear.  This callback
			should return an int id-number which uniquely
			identifies this wakeup.  If an implementation
			doesn't have microseconds or millisecond
			granularity, it should round up to the closest
			value it can deal with.

 remove_timeout_cb	Called with an int, the id number of a wakeup to
 			discard.  This id number must have been returned by
			the add_timeout_cb.  If the given wakeup has
			already expired, this should do nothing.

 select_process_cb	These callbacks tell the underlying implementation to
 unselect_process_cb	add or remove a file descriptor from the list of fds
  			which are polled for inferior-process input.  When
			input becomes available on the given process
			connection, an event of type "process" should be
			generated.

 select_console_cb	These callbacks tell the underlying implementation
 unselect_console_cb	to add or remove a console from the list of consoles
                        which are polled for user-input.

 select_device_cb	These callbacks are used by Unixoid event loops
 unselect_device_cb	(those that use select() and file descriptors and
			have a separate input fd per device).

 create_io_streams_cb   These callbacks are called by process code to
 delete_io_streams_cb   create the input and output lstreams which are used
                        for subprocess I/O.

 quitp_cb		A handler function called from the `QUIT' macro which
			should check whether the quit character has been
			typed.  On systems with SIGIO, this will not be called
			unless the `sigio_happened' flag is true (it is set
			from the SIGIO handler).

 XEmacs has its own event structures, which are distinct from the event
 structures used by X or any other window system.  It is the job of the
 event_stream layer to translate to this format.
*/

/*
  Stream pairs description
  ------------------------

  Since there are many possible processes/event loop combinations, the event code
  is responsible for creating an appropriate lstream type. The process
  implementation does not care about that implementation.

  The Create stream pair function is passed two void* values, which identify
  process-dependent 'handles'. The process implementation uses these handles
  to communicate with child processes. The function must be prepared to receive
  handle types of any process implementation. Since only one process
  implementation exists in a particular XEmacs configuration, preprocessing
  is a means of compiling in the support for the code which deals with particular
  handle types.

  For example, a unixoid type loop, which relies on file descriptors, may be
  asked to create a pair of streams by a unix-style process implementation.
  In this case, the handles passed are unix file descriptors, and the code
  may deal with these directly. Although, the same code may be used on Win32
  system with X-Windows. In this case, Win32 process implementation passes
  handles of type HANDLE, and the create_io_streams function must call
  appropriate function to get file descriptors given HANDLEs, so that these
  descriptors may be passed to XtAddInput.

  The handle given may have special denying value, in which case the
  corresponding lstream should not be created.

  The return value of the function is a unique stream identifier. It is used
  by processes implementation, in its  platform-independent part. There is
  the get_process_from_usid function, which returns process object given its
  USID. The event stream is responsible for converting its internal handle
  type into USID.

  Example is the TTY event stream. When a file descriptor signals input, the
  event loop must determine process to which the input is destined. Thus,
  the implementation uses process input stream file descriptor as USID, by
  simply casting the fd value to USID type.

  There are two special USID values. One, USID_ERROR, indicates that the stream
  pair cannot be created. The second, USID_DONTHASH, indicates that streams are
  created, but the event stream does not wish to be able to find the process
  by its USID. Specifically, if an event stream implementation never calls
  get_process_from_usid, this value should always be returned, to prevent
  accumulating useless information on USID to process relationship.
*/

/* typedef unsigned int USID; in lisp.h */
#define USID_ERROR ((USID)-1)
#define USID_DONTHASH ((USID)0)


struct event_stream
{
  int  (*event_pending_p)	(int);
  void (*next_event_cb)		(Lisp_Event *);
  void (*handle_magic_event_cb)	(Lisp_Event *);
  void (*format_magic_event_cb)	(Lisp_Event *, Lisp_Object pstream);
  int (*compare_magic_event_cb) (Lisp_Event *, Lisp_Event *);
  Hashcode (*hash_magic_event_cb)(Lisp_Event *);
  int  (*add_timeout_cb)	(EMACS_TIME);
  void (*remove_timeout_cb)	(int);
  void (*select_console_cb)	(struct console *);
  void (*unselect_console_cb)	(struct console *);
  void (*select_process_cb)	(Lisp_Process *, int doin, int doerr);
  void (*unselect_process_cb)	(Lisp_Process *, int doin, int doerr);
  void (*quit_p_cb)		(void);
  void (*force_event_pending)	(struct frame* f);
  void (*create_io_streams_cb)  (void* /* inhandle*/, void* /*outhandle*/ ,
				 void * /* errhandle*/,
				 Lisp_Object* /* instream */,
				 Lisp_Object* /* outstream */,
				 Lisp_Object* /* errstream */,
				 USID * /* in_usid */, USID * /* err_usid */,
				 int /* flags */);
  void (*delete_io_streams_cb)  (Lisp_Object /* instream */,
				 Lisp_Object /* outstream */,
				 Lisp_Object /* errstream */,
				 USID * /* in_usid */, USID * /* err_usid */);
  int (*current_event_timestamp_cb) (struct console *);
};

/* Flags for create_io_streams_cb() FLAGS parameter */
#define STREAM_PTY_FLUSHING		0x0001
#define STREAM_NETWORK_CONNECTION	0x0002

extern struct event_stream *event_stream;

#ifdef USE_KKCC
Lisp_Object make_key_data (void);
Lisp_Object make_button_data (void);
Lisp_Object make_motion_data (void);
Lisp_Object make_process_data (void);
Lisp_Object make_timeout_data (void);
Lisp_Object make_magic_data (void);
Lisp_Object make_magic_eval_data (void);
Lisp_Object make_eval_data (void);
Lisp_Object make_misc_user_data (void);
#endif /* USE_KKCC */

typedef enum emacs_event_type
{
  empty_event,
  key_press_event,
  button_press_event,
  button_release_event,
  pointer_motion_event,
  process_event,
  timeout_event,
  magic_event,
  magic_eval_event,
  eval_event,
  misc_user_event,
  dead_event
} emacs_event_type;

#define first_event_type empty_event
#define last_event_type dead_event

#ifdef MULE

enum alternative_key_chars
{
  KEYCHAR_CURRENT_LANGENV,
  KEYCHAR_DEFAULT_USER,
  KEYCHAR_DEFAULT_SYSTEM,
  KEYCHAR_UNDERLYING_VIRTUAL_KEY_CURRENT_LANGENV,
  KEYCHAR_UNDERLYING_VIRTUAL_KEY_DEFAULT_USER,
  KEYCHAR_UNDERLYING_VIRTUAL_KEY_DEFAULT_SYSTEM,
  KEYCHAR_QWERTY,
  KEYCHAR_LAST
};

#endif /* MULE */

#ifdef USE_KKCC
struct Lisp_Key_Data
#else /* not USE_KKCC */
struct key_data
#endif /* not USE_KKCC */
{
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  /* What keysym this is; a character or a symbol. */
  Lisp_Object keysym;
  /* Modifiers held down when key was pressed: control, meta, etc.
     Also includes buttons.  For many keys, Shift is not a bit; that
     is implicit in the keyboard layout. */
  int modifiers;
#ifdef MULE
  /* Alternate character interpretations for this key in different
     keyboard layouts.  This deals with the problem of pressing C-x in
     the Russian layout (the so-called "Russian C-x problem"), for
     example: `x' gets mapped to a Cyrillic character, so what do we
     do?  For that matter, what about `C-x b'?  What we do is look the
     key up in the default locales (current language environment, user
     default, system default), then check to see if the underlying
     virtual key is alphabetic in the same three defaults, then
     finally check US ASCII.  We ignore the underlying virtual key for
     the current layout to avoid the problem of a French speaker
     (AZERTY layout) who temporarily switches to Russian: The virtual
     keys underlying Russian are US-ASCII, so what the French speaker
     things of as C-a (the key just to the right of TAB) appears as
     C-q. (#### We should probably ignore the current char and look
     *ONLY* in alt_keychars for all control keys.  What about the
     English speaker who temporarily switches to the French layout and
     finds C-q mapped to C-a?) */
  Ichar alt_keychars[KEYCHAR_LAST];
#endif /* MULE */
};

#ifdef USE_KKCC
typedef struct Lisp_Key_Data Lisp_Key_Data;

DECLARE_LRECORD (key_data, Lisp_Key_Data);
#define XKEY_DATA(x) XRECORD (x, key_data, Lisp_Key_Data)
#define wrap_key_data(p) wrap_record (p, key_data)
#define KEY_DATAP(x) RECORDP (x, key_data)
#define CHECK_KEY_DATA(x) CHECK_RECORD (x, key_data)
#define CONCHECK_KEY_DATA(x) CONCHECK_RECORD (x, key_data)

#define XKEY_DATA_KEYSYM(d) (XKEY_DATA (d)->keysym)
#define KEY_DATA_KEYSYM(d) ((d)->keysym)
#define XKEY_DATA_MODIFIERS(d) (XKEY_DATA (d)->modifiers)
#define KEY_DATA_MODIFIERS(d) ((d)->modifiers)

#define XSET_KEY_DATA_KEYSYM(d, k) (XKEY_DATA (d)->keysym = (k))
#define SET_KEY_DATA_KEYSYM(d, k) ((d)->keysym = k)
#define XSET_KEY_DATA_MODIFIERS(d, m) (XKEY_DATA (d)->modifiers = m)
#define SET_KEY_DATA_MODIFIERS(d, m) ((d)->modifiers = m)
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Button_Data
#else /* not USE_KKCC */
struct button_data
#endif /* not USE_KKCC */
{
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  /* What button went down or up. */
  int button;
  /* Bucky-bits on that button: shift, control, meta, etc.  Also
     includes other buttons (not the one pressed). */
  int modifiers;
  /*  Where it was at the button-state-change (in pixels). */
  int x, y;
};
#ifdef USE_KKCC
typedef struct Lisp_Button_Data Lisp_Button_Data;

DECLARE_LRECORD (button_data, Lisp_Button_Data);
#define XBUTTON_DATA(x) XRECORD (x, button_data, Lisp_Button_Data)
#define wrap_button_data(p) wrap_record (p, button_data)
#define BUTTON_DATAP(x) RECORDP (x, button_data)
#define CHECK_BUTTON_DATA(x) CHECK_RECORD (x, button_data)
#define CONCHECK_BUTTON_DATA(x) CONCHECK_RECORD (x, button_data)

#define XBUTTON_DATA_BUTTON(d) (XBUTTON_DATA (d)->button)
#define XBUTTON_DATA_MODIFIERS(d) (XBUTTON_DATA (d)->modifiers)
#define XBUTTON_DATA_X(d) (XBUTTON_DATA (d)->x)
#define XBUTTON_DATA_Y(d) (XBUTTON_DATA (d)->y)

#define XSET_BUTTON_DATA_BUTTON(d, b) (XBUTTON_DATA (d)->button = (b))
#define XSET_BUTTON_DATA_MODIFIERS(d, m) (XBUTTON_DATA (d)->modifiers = (m))
#define XSET_BUTTON_DATA_X(d, new_x) (XBUTTON_DATA (d)->x = (new_x))
#define XSET_BUTTON_DATA_Y(d, new_y) (XBUTTON_DATA (d)->y = (new_y))
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Motion_Data
#else /* not USE_KKCC */
struct motion_data
#endif /* not USE_KKCC */
{
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  /* Where it was after it moved (in pixels). */
  int x, y;
  /* Bucky-bits down when the motion was detected. */
  int modifiers;
};
#ifdef USE_KKCC
typedef struct Lisp_Motion_Data Lisp_Motion_Data;

DECLARE_LRECORD (motion_data, Lisp_Motion_Data);
#define XMOTION_DATA(x) XRECORD (x, motion_data, Lisp_Motion_Data)
#define wrap_motion_data(p) wrap_record (p, motion_data)
#define MOTION_DATAP(x) RECORDP (x, motion_data)
#define CHECK_MOTION_DATA(x) CHECK_RECORD (x, motion_data)
#define CONCHECK_MOTION_DATA(x) CONCHECK_RECORD (x, motion_data)

#define XMOTION_DATA_X(d) (XMOTION_DATA (d)->x)
#define XMOTION_DATA_Y(d) (XMOTION_DATA (d)->y)
#define XMOTION_DATA_MODIFIERS(d) (XMOTION_DATA (d)->modifiers)

#define XSET_MOTION_DATA_X(d, new_x) (XMOTION_DATA (d)->x = (new_x))
#define XSET_MOTION_DATA_Y(d, new_y) (XMOTION_DATA (d)->y = (new_y))
#define XSET_MOTION_DATA_MODIFIERS(d, m) (XMOTION_DATA (d)->modifiers = (m))
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Process_Data
#else /* not USE_KKCC */
struct process_data
#endif /* not USE_KKCC */
{
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  /* the XEmacs "process" object in question */
  Lisp_Object process;
};
#ifdef USE_KKCC
typedef struct Lisp_Process_Data Lisp_Process_Data;

DECLARE_LRECORD (process_data, Lisp_Process_Data);
#define XPROCESS_DATA(x) XRECORD (x, process_data, Lisp_Process_Data)
#define wrap_process_data(p) wrap_record (p, process_data)
#define PROCESS_DATAP(x) RECORDP (x, process_data)
#define CHECK_PROCESS_DATA(x) CHECK_RECORD (x, process_data)
#define CONCHECK_PROCESS_DATA(x) CONCHECK_RECORD (x, process_data)

#define XPROCESS_DATA_PROCESS(d) (XPROCESS_DATA (d)->process)
#define XSET_PROCESS_DATA_PROCESS(d, p) (XPROCESS_DATA (d)->process = (p))
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Timeout_Data
#else /* not USE_KKCC */
struct timeout_data
#endif /* not USE_KKCC */
{
/*
    interval_id		The ID returned when the associated call to
			add_timeout_cb() was made
	------ the rest of the fields are filled in by XEmacs -----
    id_number		The XEmacs timeout ID for this timeout (more
			than one timeout event can have the same value
			here, since XEmacs timeouts, as opposed to
			add_timeout_cb() timeouts, can resignal
			themselves)
    function		An elisp function to call when this timeout is
			processed.
    object		The object passed to that function.
*/
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  int interval_id;
  int id_number;
  Lisp_Object function;
  Lisp_Object object;
};
#ifdef USE_KKCC
typedef struct Lisp_Timeout_Data Lisp_Timeout_Data;

DECLARE_LRECORD (timeout_data, Lisp_Timeout_Data);
#define XTIMEOUT_DATA(x) XRECORD (x, timeout_data, Lisp_Timeout_Data)
#define wrap_timeout_data(p) wrap_record(p, timeout_data)
#define TIMEOUT_DATAP(x) RECORDP (x, timeout_data)
#define CHECK_TIMEOUT_DATA(x) CHECK_RECORD (x, timeout_data)
#define CONCHECK_TIMEOUT_DATA(x) CONCHECK_RECORD (x, timeout_data)

#define XTIMEOUT_DATA_INTERVAL_ID(d) XTIMEOUT_DATA (d)->interval_id
#define XTIMEOUT_DATA_ID_NUMBER(d) XTIMEOUT_DATA (d)->id_number
#define XTIMEOUT_DATA_FUNCTION(d) XTIMEOUT_DATA (d)->function
#define XTIMEOUT_DATA_OBJECT(d) XTIMEOUT_DATA (d)->object

#define XSET_TIMEOUT_DATA_INTERVAL_ID(d, i) XTIMEOUT_DATA (d)->interval_id = (i)
#define XSET_TIMEOUT_DATA_ID_NUMBER(d, n) XTIMEOUT_DATA (d)->id_number = (n)
#define XSET_TIMEOUT_DATA_FUNCTION(d, f) XTIMEOUT_DATA (d)->function = f
#define XSET_TIMEOUT_DATA_OBJECT(d, o) XTIMEOUT_DATA (d)->object = o
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Eval_Data
#else /* not USE_KKCC */
struct eval_data
#endif /* not USE_KKCC */
{
/* This kind of event is used internally; sometimes the window system
   interface would like to inform XEmacs of some user action (such as
   focusing on another frame) but needs that to happen synchronously
   with the other user input, like keypresses.  This is useful when
   events are reported through callbacks rather than in the standard
   event stream.

    function		An elisp function to call with this event object.
    object		Argument of function.
*/
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  Lisp_Object function;
  Lisp_Object object;
};
#ifdef USE_KKCC
typedef struct Lisp_Eval_Data Lisp_Eval_Data;

DECLARE_LRECORD (eval_data, Lisp_Eval_Data);
#define XEVAL_DATA(x) XRECORD (x, eval_data, Lisp_Eval_Data)
#define wrap_eval_data(p) wrap_record(p, eval_data)
#define EVAL_DATAP(x) RECORDP (x, eval_data)
#define CHECK_EVAL_DATA(x) CHECK_RECORD (x, eval_data)
#define CONCHECK_EVAL_DATA(x) CONCHECK_RECORD (x, eval_data)

#define XEVAL_DATA_FUNCTION(d) (XEVAL_DATA (d)->function)
#define XEVAL_DATA_OBJECT(d) (XEVAL_DATA (d)->object)

#define XSET_EVAL_DATA_FUNCTION(d, f) (XEVAL_DATA (d)->function = f)
#define XSET_EVAL_DATA_OBJECT(d, o) (XEVAL_DATA (d)->object = o)
#endif /* USE_KKCC */

#ifdef USE_KKCC
struct Lisp_Misc_User_Data
#else /* not USE_KKCC */
struct misc_user_data
#endif /* not USE_KKCC */
{
/* #### The misc-user type is serious junk.  It should be separated
   out into different events.  There's no reason to create
   sub-subtypes of events.

    function		An elisp function to call with this event object.
    object		Argument of function.
    button		What button went down or up.
    modifiers		Bucky-bits on that button: shift, control, meta, etc.
    x, y		Where it was at the button-state-change (in pixels).
			This is similar to an eval_event, except that it is
			generated by user actions: selections in the
			menubar, scrollbar actions, or drag and drop actions.
			It is a "command" event, like key and mouse presses
			(and unlike mouse motion, process output, and enter
			and leave window hooks).  In many ways, eval_events
			are not the same as keypresses or misc_user_events.
			The button, modifiers, x, and y parts are only used
			by the XEmacs Drag'n'Drop system. Don't depend on their
			values for other types of misc_user_events.
*/
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  Lisp_Object function;
  Lisp_Object object;
  int button;
  int modifiers;
  int x, y;
};
#ifdef USE_KKCC
typedef struct Lisp_Misc_User_Data Lisp_Misc_User_Data;

DECLARE_LRECORD (misc_user_data, Lisp_Misc_User_Data);
#define XMISC_USER_DATA(x) XRECORD (x, misc_user_data, Lisp_Misc_User_Data)
#define wrap_misc_user_data(p) wrap_record(p, misc_user_data)
#define MISC_USER_DATAP(x) RECORDP (x, misc_user_data)
#define CHECK_MISC_USER_DATA(x) CHECK_RECORD (x, misc_user_data)
#define CONCHECK_MISC_USER_DATA(x) CONCHECK_RECORD (x, misc_user_data)

#define XMISC_USER_DATA_FUNCTION(d) (XMISC_USER_DATA (d)->function)
#define XMISC_USER_DATA_OBJECT(d) (XMISC_USER_DATA (d)->object)
#define XMISC_USER_DATA_BUTTON(d) (XMISC_USER_DATA (d)->button)
#define XMISC_USER_DATA_MODIFIERS(d) (XMISC_USER_DATA (d)->modifiers)
#define XMISC_USER_DATA_X(d) (XMISC_USER_DATA (d)->x)
#define XMISC_USER_DATA_Y(d) (XMISC_USER_DATA (d)->y)

#define XSET_MISC_USER_DATA_FUNCTION(d, f) (XMISC_USER_DATA (d)->function = (f))
#define XSET_MISC_USER_DATA_OBJECT(d, o) (XMISC_USER_DATA (d)->object = (o))
#define XSET_MISC_USER_DATA_BUTTON(d, b) (XMISC_USER_DATA (d)->button = (b))
#define XSET_MISC_USER_DATA_MODIFIERS(d, m) (XMISC_USER_DATA (d)->modifiers = (m))
#define XSET_MISC_USER_DATA_X(d, new_x) (XMISC_USER_DATA (d)->x = (new_x))
#define XSET_MISC_USER_DATA_Y(d, new_y) (XMISC_USER_DATA (d)->y = (new_y))
#endif /* USE_KKCC */

#ifdef USE_KKCC
typedef void (*lisp_obj_arg_fun) (Lisp_Object);


struct Lisp_Magic_Eval_Data
#else /* not USE_KKCC */
struct magic_eval_data
#endif /* not USE_KKCC */
{
/* This is like an eval event but its contents are not
   Lisp-accessible.  This allows for "internal eval events" that call
   non-Lisp-accessible functions.  Externally, a magic_eval_event just
   appears as a magic_event; the Lisp programmer need not know
   anything more.

    internal_function	An unexported function to call with this event
			object.  This allows eval events to call internal
			functions.  For a normal eval event, this field
			will always be 0.
    object		Argument of function.

*/
#ifdef USE_KKCC
  struct lrecord_header lheader;
#endif /* USE_KKCC */
  void (*internal_function) (Lisp_Object);
  Lisp_Object object;
};
#ifdef USE_KKCC
typedef struct Lisp_Magic_Eval_Data Lisp_Magic_Eval_Data;

DECLARE_LRECORD (magic_eval_data, Lisp_Magic_Eval_Data);
#define XMAGIC_EVAL_DATA(x) XRECORD (x, magic_eval_data, Lisp_Magic_Eval_Data)
#define wrap_magic_eval_data(p) wrap_record(p, magic_eval_data)
#define MAGIC_EVAL_DATAP(x) RECORDP (x, magic_eval_data)
#define CHECK_MAGIC_EVAL_DATA(x) CHECK_RECORD (x, magic_eval_data)
#define CONCHECK_MAGIC_EVAL_DATA(x) CONCHECK_RECORD (x, magic_eval_data)

#define XMAGIC_EVAL_DATA_INTERNAL_FUNCTION(d) \
  XMAGIC_EVAL_DATA (d)->internal_function
#define XMAGIC_EVAL_DATA_INTERNAL_FUNOBJ(d) (XMAGIC_EVAL_DATA (d)->internal_function)
#define XMAGIC_EVAL_DATA_OBJECT(d) (XMAGIC_EVAL_DATA (d)->object)

#define XSET_MAGIC_EVAL_DATA_INTERNAL_FUNCTION(d, f) \
  (XMAGIC_EVAL_DATA (d)->internal_function = f)
#define XSET_MAGIC_EVAL_DATA_INTERNAL_FUNOBJ(d, f) (XMAGIC_EVAL_DATA (d)->internal_function = (f))
#define XSET_MAGIC_EVAL_DATA_OBJECT(d, o) (XMAGIC_EVAL_DATA (d)->object = (o))
#endif /* USE_KKCC */

#if defined (HAVE_X_WINDOWS) && defined(emacs)
# include <X11/Xlib.h>
#endif

#ifdef HAVE_GTK
#include <gdk/gdk.h>
#endif


#ifdef USE_KKCC
struct Lisp_Magic_Data
{
  struct lrecord_header lheader;

  union {
#ifdef HAVE_GTK
    GdkEvent          gdk_event;
#endif
#ifdef HAVE_X_WINDOWS
    XEvent            x_event;
#endif
#ifdef HAVE_MS_WINDOWS
    int               mswindows_event;
#endif
  } underlying;
};

typedef struct Lisp_Magic_Data Lisp_Magic_Data;

DECLARE_LRECORD (magic_data, Lisp_Magic_Data);
#define XMAGIC_DATA(x) XRECORD (x, magic_data, Lisp_Magic_Data)
#define wrap_magic_data(p) wrap_record(p, magic_data)
#define MAGIC_DATAP(x) RECORDP (x, magic_data)
#define CHECK_MAGIC_DATA(x) CHECK_RECORD (x, magic_data)
#define CONCHECK_MAGIC_DATA(x) CONCHECK_RECORD (x, magic_data)

#define XMAGIC_DATA_UNDERLYING(d) (XMAGIC_DATA (d)->underlying)
#define XSET_MAGIC_DATA_UNDERLYING(d, u) (XMAGIC_DATA (d)->underlying = (u))

#ifdef HAVE_GTK
#define XMAGIC_DATA_GDK_EVENT(d) (XMAGIC_DATA (d)->underlying.gdk_event)
#define XSET_MAGIC_DATA_GDK_EVENT(d, e) (XMAGIC_DATA (d)->underlying.gdk_event = (e))
#endif /*HAVE_GTK*/

#ifdef HAVE_X_WINDOWS
#define XMAGIC_DATA_X_EVENT(d) (XMAGIC_DATA (d)->underlying.x_event)
#define XSET_MAGIC_DATA_X_EVENT(d, e) (XMAGIC_DATA (d)->underlying.x_event = (e))
#endif

#ifdef HAVE_MS_WINDOWS
#define XMAGIC_DATA_MSWINDOWS_EVENT(d) (XMAGIC_DATA (d)->underlying.mswindows_event)
#define XSET_MAGIC_DATA_MSWINDOWS_EVENT(d, e) (XMAGIC_DATA (d)->underlying.mswindows_event = (e))
#endif

#else /* not USE_KKCC */
union magic_data
{
/* No user-serviceable parts within.  This is for things like
   KeymapNotify and ExposeRegion events and so on that XEmacs itself
   doesn't care about, but which it must do something with for proper
   interaction with the window system.

   Magic_events are handled somewhat asynchronously, just like
   subprocess filters.  However, occasionally a magic_event needs to
   be handled synchronously; in that case, the asynchronous handling
   of the magic_event will push an eval_event back onto the queue,
   which will be handled synchronously later.  This is one of the
   reasons why eval_events exist; I'm not entirely happy with this
   aspect of this event model.
*/

#ifdef HAVE_GTK
  GdkEvent          underlying_gdk_event;
#endif
#ifdef HAVE_X_WINDOWS
  XEvent            underlying_x_event;
#endif
#ifdef HAVE_MS_WINDOWS
  int               underlying_mswindows_event;
#endif
};
#endif /* not USE_KKCC */


struct Lisp_Timeout
{
  struct lcrecord_header header;
  int id; /* Id we use to identify the timeout over its lifetime */
  int interval_id; /* Id for this particular interval; this may
                      be different each time the timeout is
                      signalled.*/
  Lisp_Object function, object; /* Function and object associated
                                   with timeout. */
  EMACS_TIME next_signal_time;  /* Absolute time when the timeout
                                   is next going to be signalled. */
  unsigned int resignal_msecs;  /* How far after the next timeout
                                   should the one after that
                                   occur? */
};
typedef struct Lisp_Timeout Lisp_Timeout;

DECLARE_LRECORD (timeout, Lisp_Timeout);
#define XTIMEOUT(x) XRECORD (x, timeout, Lisp_Timeout)
#define wrap_timeout(p) wrap_record (p, timeout)
#define TIMEOUTP(x) RECORDP (x, timeout)
#define CHECK_TIMEOUT(x) CHECK_RECORD (x, timeout)
#define CONCHECK_TIMEOUT(x) CONCHECK_RECORD (x, timeout)

struct Lisp_Event
{
  /* header->next (aka XEVENT_NEXT ()) is used as follows:
     - For dead events, this is the next dead one.
     - For events on the command_event_queue, the next one on the queue.
     - Likewise for events chained in the command builder.
     - Otherwise it's Qnil.
   */
  struct lrecord_header lheader;
  Lisp_Object           next;
  emacs_event_type      event_type;

  /* Where this event occurred on.  This will be a frame, device,
     console, or nil, depending on the event type.  It is important
     that an object of a more specific type than is actually generated
     is not substituted -- e.g. there should not be a frame inserted
     when a key-press event occurs, because events on dead channels
     are automatically ignored.

     Specifically:

     -- for button and mouse-motion events, channel will be a
     frame. (The translation to a window occurs later.)

     -- for keyboard events, channel will be a console.  Note that
     fake keyboard events (generated by `character-to-event' or
     something that calls this, such as macros) need to have the
     selected console stored into them when the event is created.
     This is so that the correct console-local variables (e.g. the
     command builder) will get affected.

     -- for timer, process, magic-eval, and eval events, channel will
     be nil.

     -- for misc-user events, channel will be a frame.

     -- for magic events, channel will be a frame (usually) or a
     device. */
  Lisp_Object           channel;

  /* When this event occurred -- if not known, this is made up. ####
     All timestamps should be measured as milliseconds since XEmacs
     started.  Currently they are raw server timestamps. (The X
     protocol doesn't provide any easy way of translating between
     server time and real process time; yuck.) */
  unsigned int          timestamp;

#ifdef USE_KKCC
  Lisp_Object event_data;
#else /* not USE_KKCC */
  union
    {
      struct key_data           key;
      struct button_data        button;
      struct motion_data        motion;
      struct process_data       process;
      struct timeout_data       timeout;
      struct eval_data          eval;   /* misc_user_event no longer uses this */
      struct misc_user_data     misc;   /* because it needs position information */
      union magic_data          magic;
      struct magic_eval_data    magic_eval;
    } event;
#endif /* not USE_KKCC */
};

DECLARE_LRECORD (event, Lisp_Event);
#define XEVENT(x) XRECORD (x, event, Lisp_Event)
#define wrap_event(p) wrap_record (p, event)
#define EVENTP(x) RECORDP (x, event)
#define CHECK_EVENT(x) CHECK_RECORD (x, event)
#define CONCHECK_EVENT(x) CONCHECK_RECORD (x, event)

DECLARE_LRECORD (command_builder, struct command_builder);

#define EVENT_CHANNEL(a) ((a)->channel)
#define EVENT_TYPE(a) ((a)->event_type)
#define XEVENT_TYPE(a) (XEVENT (a)->event_type)
#define EVENT_NEXT(a) ((a)->next)
#define XEVENT_NEXT(e) (XEVENT (e)->next)

#ifdef USE_KKCC
#define XEVENT_DATA(ev) (XEVENT (ev)->event_data)
#define EVENT_DATA(ev) ((ev)->event_data)
#define XEVENT_CHANNEL(ev) (XEVENT (ev)->channel)
#define EVENT_TIMESTAMP(ev)                                     \
  ((ev)->timestamp)
#define XEVENT_TIMESTAMP(ev) EVENT_TIMESTAMP (XEVENT (ev))

#define SET_EVENT_TIMESTAMP_ZERO(ev) \
  ((ev)->timestamp = 0)
#define SET_EVENT_TIMESTAMP(ev, t)                      \
  (ev)->timestamp = (t)
#define XSET_EVENT_TIMESTAMP(ev, t) SET_EVENT_TIMESTAMP (XEVENT (ev), t)


#define SET_EVENT_CHANNEL(ev, c)                        \
do {                                                    \
  Lisp_Event *mac_event = (ev);                         \
  mac_event->channel = (c);                             \
} while (0)
#define XSET_EVENT_CHANNEL(ev, c) SET_EVENT_CHANNEL (XEVENT (ev), c) 

#define SET_EVENT_DATA(ev, d)                           \
do {                                                    \
  Lisp_Event *mac_event = (ev);                         \
  mac_event->event_data = (d);                          \
} while (0)
#define XSET_EVENT_DATA(ev, d) SET_EVENT_DATA (XEVENT (ev), d)

INLINE_HEADER  void set_event_type(struct Lisp_Event *event, emacs_event_type t);
INLINE_HEADER  void
set_event_type(struct Lisp_Event *event, emacs_event_type t) 
{
  event->event_type = t;

  switch (t) {
  case key_press_event:
    event->event_data = make_key_data ();
    break;
  case button_press_event:
  case button_release_event:
    event->event_data = make_button_data ();
    break;
  case pointer_motion_event:
    event->event_data = make_motion_data ();
    break;
  case process_event:
    event->event_data = make_process_data ();
    break;
  case timeout_event:
    event->event_data = make_timeout_data ();
    break;
  case magic_event:
    event->event_data = make_magic_data ();
    break;
  case magic_eval_event:
    event->event_data = make_magic_eval_data ();
    break;
  case eval_event:
    event->event_data = make_eval_data ();
    break;
  case misc_user_event:
    event->event_data = make_misc_user_data ();
    break;
  default:
    break;
  }
}
#define XSET_EVENT_TYPE(ev, t) set_event_type (XEVENT (ev), t)
#define SET_EVENT_TYPE(ev, t) set_event_type (ev, t)


#define SET_EVENT_NEXT(ev, n)                           \
do {                                                    \
  Lisp_Event *mac_event = (ev);                         \
  mac_event->next = (n);                                \
} while (0)
#define XSET_EVENT_NEXT(ev, n) SET_EVENT_NEXT (XEVENT (ev), n)

#else /* NOT USE_KKCC */
#define XSET_EVENT_NEXT(e, n) do { (XEVENT (e)->next = (n)); } while (0)
#endif

#define EVENT_CHAIN_LOOP(event, chain) \
  for (event = chain; !NILP (event); event = XEVENT_NEXT (event))

#define EVENT_LIVE_P(a) (EVENT_TYPE (a) != dead_event)

#define CHECK_LIVE_EVENT(x) do {                        \
  CHECK_EVENT (x);                                      \
  if (! EVENT_LIVE_P (XEVENT (x)))                      \
    dead_wrong_type_argument (Qevent_live_p, (x));      \
} while (0)
#define CONCHECK_LIVE_EVENT(x) do {                     \
  CONCHECK_EVENT (x);                                   \
  if (! EVENT_LIVE_P (XEVENT (x)))                      \
    x = wrong_type_argument (Qevent_live_p, (x));       \
} while (0)


EXFUN (Fcharacter_to_event, 4);
EXFUN (Fdeallocate_event, 1);
EXFUN (Fevent_glyph_extent, 1);
EXFUN (Fevent_modeline_position, 1);
EXFUN (Fevent_over_modeline_p, 1);
EXFUN (Fevent_over_toolbar_p, 1);
EXFUN (Fevent_over_vertical_divider_p, 1);
EXFUN (Fevent_point, 1);
EXFUN (Fevent_window, 1);
EXFUN (Fmake_event, 2);

extern Lisp_Object QKbackspace, QKdelete, QKescape, QKlinefeed, QKreturn;
extern Lisp_Object QKspace, QKtab, Qmouse_event_p, Vcharacter_set_property;
extern Lisp_Object Qcancel_mode_internal;
extern Lisp_Object Vmodifier_keys_sticky_time;

/* The modifiers XEmacs knows about; these appear in key and button events. */

#define XEMACS_MOD_CONTROL      (1<<0)
#define XEMACS_MOD_META         (1<<1)
#define XEMACS_MOD_SUPER        (1<<2)
#define XEMACS_MOD_HYPER        (1<<3)
#define XEMACS_MOD_ALT          (1<<4)
#define XEMACS_MOD_SHIFT        (1<<5)  /* not used for dual-case characters */
#define XEMACS_MOD_BUTTON1      (1<<6)
#define XEMACS_MOD_BUTTON2      (1<<7)
#define XEMACS_MOD_BUTTON3      (1<<8)
#define XEMACS_MOD_BUTTON4      (1<<9)
#define XEMACS_MOD_BUTTON5      (1<<10)

/* Note: under X Windows, XEMACS_MOD_ALT is generated by the Alt key
   if there are both Alt and Meta keys.  If there are no Meta keys,
   then Alt generates XEMACS_MOD_META instead.
 */

/* Maybe this should be trickier */
#define KEYSYM(x) (intern (x))

/* from events.c */
#ifdef USE_KKCC
void format_event_object (Eistring *buf, Lisp_Object event, int brief);
#else /* not USE_KKCC */
void format_event_object (Eistring *buf, Lisp_Event *event, int brief);
#endif /* not USE_KKCC */
#ifdef USE_KKCC
//void format_event_data_object (Eistring *buf, Lisp_Object data, int brief);
void copy_event_data (Lisp_Object dest, Lisp_Object src);
#endif /* USE_KKCC */
void character_to_event (Ichar c, Lisp_Event *event,
                         struct console *con,
                         int use_console_meta_flag,
                         int do_backspace_mapping);
void zero_event (Lisp_Event *e);
void deallocate_event_chain (Lisp_Object event);
Lisp_Object event_chain_tail (Lisp_Object event);
void enqueue_event (Lisp_Object event, Lisp_Object *head, Lisp_Object *tail);
Lisp_Object dequeue_event (Lisp_Object *head, Lisp_Object *tail);
void enqueue_event_chain (Lisp_Object event_chain, Lisp_Object *head,
                          Lisp_Object *tail);
int event_chain_count (Lisp_Object event_chain);
Lisp_Object transfer_event_chain_pointer (Lisp_Object pointer,
					  Lisp_Object old_chain,
					  Lisp_Object new_chain);
void nth_of_key_sequence_as_event (Lisp_Object seq, int n, Lisp_Object event);
Lisp_Object key_sequence_to_event_chain (Lisp_Object seq);
Lisp_Object event_chain_find_previous (Lisp_Object event_chain,
                                       Lisp_Object event);
Lisp_Object event_chain_nth (Lisp_Object event_chain, int n);
Lisp_Object copy_event_chain (Lisp_Object event_chain);
/* True if this is a non-internal event
   (keyboard press, menu, scrollbar, mouse button) */
int command_event_p (Lisp_Object event);
void define_self_inserting_symbol (Lisp_Object, Lisp_Object);
Ichar event_to_character (Lisp_Event *, int, int, int);
struct console *event_console_or_selected (Lisp_Object event);

/* from event-stream.c */
Lisp_Object allocate_command_builder (Lisp_Object console, int with_echo_buf);
void enqueue_magic_eval_event (void (*fun) (Lisp_Object), Lisp_Object object);
void event_stream_handle_magic_event (Lisp_Event *event);
void event_stream_format_magic_event (Lisp_Event *event, Lisp_Object pstream);
int event_stream_compare_magic_event (Lisp_Event *e1, Lisp_Event *e2);
Hashcode event_stream_hash_magic_event (Lisp_Event *e);
void event_stream_select_console   (struct console *con);
void event_stream_unselect_console (struct console *con);
void event_stream_select_process   (Lisp_Process *proc, int doin, int doerr);
void event_stream_unselect_process (Lisp_Process *proc, int doin, int doerr);
void event_stream_create_io_streams (void* inhandle, void* outhandle,
				     void *errhandle, Lisp_Object* instream,
				     Lisp_Object* outstream,
				     Lisp_Object* errstream,
				     USID* in_usid,
				     USID* err_usid,
				     int flags);
void event_stream_delete_io_streams (Lisp_Object instream,
				     Lisp_Object outstream,
				     Lisp_Object errstream,
				     USID* in_usid,
				     USID* err_usid);
void event_stream_quit_p (void);
void run_pre_idle_hook (void);

struct low_level_timeout
{
  int id;
  EMACS_TIME time;
  struct low_level_timeout *next;
};

int add_low_level_timeout (struct low_level_timeout **timeout_list,
                           EMACS_TIME thyme);
void remove_low_level_timeout (struct low_level_timeout **timeout_list,
                               int id);
int get_low_level_timeout_interval (struct low_level_timeout *
                                    timeout_list, EMACS_TIME *interval);
int pop_low_level_timeout (struct low_level_timeout **timeout_list,
                           EMACS_TIME *time_out);
int event_stream_generate_wakeup (unsigned int milliseconds,
                                  unsigned int vanilliseconds,
                                  Lisp_Object function,
                                  Lisp_Object object,
                                  int async_p);
int event_stream_resignal_wakeup (int interval_id, int async_p,
                                  Lisp_Object *function, Lisp_Object *object);
void event_stream_disable_wakeup (int id, int async_p);

/* from signal.c */
int signal_add_async_interval_timeout (EMACS_TIME thyme);
void signal_remove_async_interval_timeout (int id);

/* from event-stream.c -- focus sanity */
extern int focus_follows_mouse;
void investigate_frame_change (void);

void emacs_handle_focus_change_preliminary (Lisp_Object frame_inp_and_dev);
void emacs_handle_focus_change_final (Lisp_Object frame_inp_and_dev);

Lisp_Object extract_this_command_keys_nth_mouse_event (int n);
Lisp_Object extract_vector_nth_mouse_event (Lisp_Object vector, int n);

void single_console_state (void);
void any_console_state (void);
int in_single_console_state (void);

extern int emacs_is_blocking;

extern volatile int sigint_happened;

#ifdef HAVE_UNIXOID_EVENT_LOOP
/* from event-unixoid.c */

/* Ceci n'est pas un pipe. */
extern int signal_event_pipe[];

void signal_fake_event (void);
void drain_signal_event_pipe (void);

extern int fake_event_occurred;

int event_stream_unixoid_select_console   (struct console *con);
int event_stream_unixoid_unselect_console (struct console *con);
void event_stream_unixoid_select_process (Lisp_Process *proc, int doin,
					  int doerr, int *infd, int *errfd);
void event_stream_unixoid_unselect_process (Lisp_Process *proc, int doin,
					    int doerr, int *infd, int *errfd);
int read_event_from_tty_or_stream_desc (Lisp_Event *event,
					struct console *con);
void event_stream_unixoid_create_io_streams (void* inhandle, void* outhandle,
					     void *errhandle,
					     Lisp_Object* instream,
					     Lisp_Object* outstream,
					     Lisp_Object* errstream,
					     USID* in_usid,
					     USID* err_usid,
					     int flags);
void event_stream_unixoid_delete_io_streams (Lisp_Object instream,
					     Lisp_Object outstream,
					     Lisp_Object errstream,
					     USID* in_usid,
					     USID* err_usid);

/* Beware: this evil macro evaluates its arg many times */
#define FD_TO_USID(fd) ((fd)==0 ? (USID)999999 : ((fd)<0 ? USID_DONTHASH : (USID)(fd)))

#endif /* HAVE_UNIXOID_EVENT_LOOP */

/* Define this if you want the tty event stream to be used when the
   first console is tty, even if HAVE_X_WINDOWS is defined */
/* #define DEBUG_TTY_EVENT_STREAM */


/* #### a hack, until accelerator shit is cleaned up */

/* This structure is what we use to encapsulate the state of a command sequence
   being composed; key events are executed by adding themselves to the command
   builder; if the command builder is then complete (does not still represent
   a prefix key sequence) it executes the corresponding command.
 */
struct command_builder
{
  struct lcrecord_header header;
  Lisp_Object console; /* back pointer to the console this command
                          builder is for */
#if 0
  /* #### Not implemented: nil, or an event representing the first
     event read after the last command completed.  Threaded. */
  Lisp_Object prefix_events;
#endif /* 0 */
  /* nil, or an event chain representing the events in the current
     keymap-lookup sequence.  NOTE: All events in the chain MUST be
     freshly allocated, with no pointers to them elsewhere. */
  Lisp_Object current_events;
  /* Last elt of current_events */
  Lisp_Object most_current_event;
  /* Last elt before function map code took over.  What this means is:
     All prefixes up to (but not including) this event have non-nil
     bindings, but the prefix including this event has a nil binding.
     Any events in the chain after this one were read solely because
     we're part of a possible function key.  If we end up with
     something that's not part of a possible function key, we have to
     unread all of those events. */
  Lisp_Object last_non_munged_event;
  /* One set of values for function-key-map, one for key-translation-map */
  struct munging_key_translation
  {
    /* First event that can begin a possible function key sequence
       (to be translated according to function-key-map).  Normally
       this is the first event in the chain.  However, once we've
       translated a sequence through function-key-map, this will point
       to the first event after the translated sequence: we don't ever
       want to translate any events twice through function-key-map, or
       things could get really screwed up (e.g. if the user created a
       translation loop).  If this is nil, then the next-read event is
       the first that can begin a function key sequence. */
    Lisp_Object first_mungeable_event;
  } munge_me[2];
  Ibyte *echo_buf;

  Bytecount echo_buf_length;          /* size of echo_buf */
  Bytecount echo_buf_index;           /* index into echo_buf
                                       * -1 before doing echoing for new cmd */
  /* Self-insert-command is magic in that it doesn't always push an undo-
     boundary: up to 20 consecutive self-inserts can happen before an undo-
     boundary is pushed.  This variable is that counter.
     */
  int self_insert_countdown;
};

#endif /* INCLUDED_events_h_ */
