;;; xwem-battery.el --- Dockapp APM battery monitor for XWEM.

;; Copyright (C) 2004 by Free Software Foundation, Inc.

;; Author: Zajcev Evgeny <zevlg@yandex.ru>
;;         Steve Youngs  <steve@youngs.au.com>
;; Created: Thu Sep  2 01:14:36 GMT 2004
;; Keywords: xwem
;; X-CVS: $Id: xwem-battery.el,v 1.1 2005/01/01 04:42:51 youngs Exp $

;; This file is part of XWEM.

;; XWEM is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XWEM is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
;; or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
;; License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; APM battery status monitor dockapp for use under XWEM.

;; It looks like:

;;   normal    charching
;;
;;    ****        ****  
;;  ********    *******/
;;  *      *    *     //
;;  *      *    *    //*
;;  *      *    *   // *
;;  ********    ***//***
;;  *------*    *-//---*
;;  *------*    *//----*
;;  *------*    //-----*
;;   ******      ****** 

;; To start using it, just add:

;;   (load-module <path-to-apm-battery-ell>)
;;   (add-hook 'xwem-after-init-hook 'xwem-battery)

;; to your xwemrc.el.

;;; Code:

(eval-when-compile
  (require 'cl)
  (autoload 'apm-battery "battery.ell" "Return current battery status."))

(require 'xlib-xlib)
(require 'xlib-xshape)

(require 'xwem-load)

;;; Customisation
(defgroup xwem-batt nil
  "Group to customise APM battery monitor."
  :prefix "xwem-batt-"
  :group 'xwem)

(defcustom xwem-batt-update-interval 5
  "*Apm battery dockapp update interval in seconds."
  :type 'number
  :group 'xwem-batt)

(defcustom xwem-batt-height 24
  "*Height of apm battery dockapp in pixels."
  :type 'number
  :group 'xwem-batt)

(defcustom xwem-batt-width 10
  "*Width of apm battery dockapp in pixels."
  :type 'number
  :group 'xwem-batt)

(defcustom xwem-batt-percentage-colors
  '((20 . "red3")
    (30 . "red2")
    (50 . "orange")
    (60 . "yellow2")
    (70 . "yellow3")
    (80 . "green3")
    (100 . "green2"))
  "*Table to translate percentage to color."
  :type '(repeat (cons (number :tag "Percents")
                       (color :tag "Color")))
  :group 'xwem-batt)

(defcustom xwem-batt-ac-line-width 4
  "*Width of ac-line."
  :type 'number
  :group 'xwem-batt)

(defcustom xwem-batt-ac-line-color "blue"
  "*Color used to display ac-line."
  :type 'color
  :group 'xwem-batt)

;;; Internal variables


(defun xwem-batt-init (xdpy)
  "On display XDPY create and return APM battery monitor window."
  (let (mgc tgc xwin xmask)
    (setq xwin (XCreateWindow xdpy (XDefaultRootWindow xdpy)
			      0 0 xwem-batt-width xwem-batt-height 0
			      nil nil nil
			      (make-X-Attr :backing-store X-WhenMapped
					   :override-redirect t)))

    ;; Create mask pixmap for xwin
    (setq xmask (XCreatePixmap xdpy (make-X-Pixmap :dpy xdpy
						   :id (X-Dpy-get-id xdpy))
			       xwin 1 xwem-batt-width xwem-batt-height))
    (setq mgc (XCreateGC xdpy xmask
			 (make-X-Gc :dpy xdpy :id (X-Dpy-get-id xdpy)
				    :graphics-exposures X-False
				    :foreground 0.0
				    :background 1.0)))
    (setq tgc (XCreateGC xdpy xmask
			 (make-X-Gc :dpy xdpy :id (X-Dpy-get-id xdpy)
				    :graphics-exposures X-False
				    :foreground 1.0
				    :background 0.0)))

    ;; XXX Draw mask
    (XFillRectangle xdpy xmask mgc 0 0 xwem-batt-width xwem-batt-height)
    (XFillRectangle xdpy xmask tgc 0 2 xwem-batt-width (- xwem-batt-height 3))
    (XDrawSegments xdpy xmask tgc
		   (list (cons (cons 3 0) (cons (- xwem-batt-width 4) 0))
			 (cons (cons 1 1) (cons (- xwem-batt-width 2) 1))
			 (cons (cons 1 (- xwem-batt-height 1))
			       (cons (- xwem-batt-width 2) (- xwem-batt-height 1)))))

    (XFreeGC xdpy mgc)
    (XFreeGC xdpy tgc)

    ;; Set mask
    (X-XShapeMask xdpy xwin X-XShape-Bounding X-XShapeSet 0 0 xmask)
    (X-Win-put-prop xwin 'xwem-batt-xmask xmask)

    xwin))

(defface xwem-batt-tmp-face
  `((t (:foreground "black")))
  "Temporary face used by apm battery dockapp.")

(defun xwem-batt-win-update (xwin &optional force)
  "Update contents of XWIN to reflect current APM battery state."
  ;; TODO: write me
  (let* ((xdpy (X-Win-dpy xwin))
	 (as (apm-battery))
	 (ac-line-p (car as))
	 (cperc (caddr as))
	 (perc-cols xwem-batt-percentage-colors)
         dheight)

    (when (> cperc 100)
      (setq  cperc 100))

    ;; Calculate displayed height
    (setq dheight (round (/ (* cperc (- xwem-batt-height 5)) 100.0)))

    (when (or force (not (eq dheight (X-Win-get-prop xwin 'old-dheight)))
              (not (eq ac-line-p (X-Win-get-prop xwin 'old-ac-line-p))))
      (XClearArea xdpy xwin 0 0 xwem-batt-width xwem-batt-height nil)
      ;; Outline battery
      (XFillRectangle xdpy xwin (XDefaultGC xdpy)
                      0 0 xwem-batt-width xwem-batt-height)
      (XDrawRectangle xdpy xwin (xwem-face-get-gc 'xwem-face-black)
                      1 2 (- xwem-batt-width 3) (- xwem-batt-height 4))
      (XDrawLine xdpy xwin (xwem-face-get-gc 'xwem-face-black)
                 3 1 (- xwem-batt-width 4) 1)
      (setq force t))

    ;; Maybe redraw percentage
    (when (or force (not (eq dheight (X-Win-get-prop xwin 'old-dheight))))
      ;; Find appopriate color
      (while (and perc-cols (> cperc (caar perc-cols)))
        (setq perc-cols (cdr perc-cols)))
      (setq perc-cols (cdar perc-cols))

      (xwem-set-face-foreground 'xwem-batt-tmp-face perc-cols)
      (XFillRectangle xdpy xwin (xwem-face-get-gc 'xwem-batt-tmp-face)
                      2 (- xwem-batt-height 2 dheight)
                      (- xwem-batt-width 4) dheight)
      (when (< dheight (- xwem-batt-height 5))
        (XDrawLine xdpy xwin (xwem-face-get-gc 'xwem-face-black)
                   2 (- xwem-batt-height 2 dheight)
                   (- xwem-batt-width 2) (- xwem-batt-height 2 dheight)))

      ;; Save DHEIGHT
      (X-Win-put-prop xwin 'old-dheight dheight))

    ;; Maybe redraw ac-line status
    (when (or force (not (eq ac-line-p (X-Win-get-prop xwin 'old-ac-line-p))))
      (when ac-line-p
        (xwem-set-face-foreground 'xwem-batt-tmp-face xwem-batt-ac-line-color)
        (let ((acgc (xwem-face-get-gc 'xwem-batt-tmp-face)))
          (setf (X-Gc-line-width acgc) xwem-batt-ac-line-width)
          (XChangeGC xdpy acgc)
          (XDrawLine xdpy xwin acgc
                     xwem-batt-width xwem-batt-ac-line-width
                     0 (- xwem-batt-height xwem-batt-ac-line-width))
          (setf (X-Gc-line-width acgc) 0)
          (XChangeGC xdpy acgc)))
      (X-Win-put-prop xwin 'old-ac-line-p ac-line-p))
    ))

(defun xwem-batt-event-handler (xdpy win xev)
  "Event handler for xwem battery monitor."
  (X-Event-CASE xev
    (:X-Expose
     (xwem-batt-win-update win t))
    (:X-DestroyNotify
     (delete-itimer (X-Win-get-prop win 'xwem-batt-timer))
     (XFreePixmap xdpy (X-Win-get-prop win 'xwem-batt-xmask))
     (X-Win-rem-prop win 'xwem-batt-timer)
     (X-Win-rem-prop win 'xwem-batt-xmask)

     (X-Win-rem-prop win 'old-ac-line-p)
     (X-Win-rem-prop win 'old-dheight))
    (:X-ButtonPress
     (multiple-value-bind (ac-line status perc)
         (values-list (apm-battery))
       (xwem-message 'info "APM Battery: AC-line: %s, Status: %S, Percentage: %d%%"
		     (if ac-line "on" "off") status perc)))
    ))

;;;###autoload
(defun xwem-battery (&optional dockip dockgroup dockalign)
  "Start xwem apm battery monitor in system tray."
  (unless (fboundp 'apm-battery)
    (error "APM Battery module not loaded"))

  (let ((bxwin (xwem-batt-init (xwem-dpy))))
    (XSelectInput (xwem-dpy) bxwin
		  (Xmask-or XM-Exposure XM-StructureNotify
                            XM-ButtonPress XM-ButtonRelease))
    (X-Win-EventHandler-add bxwin 'xwem-batt-event-handler nil
			    (list X-Expose X-DestroyNotify
                                  X-ButtonPress X-ButtonRelease))

    (xwem-XTrayInit (xwem-dpy) bxwin dockip dockgroup dockalign)

    (X-Win-put-prop bxwin 'xwem-batt-timer
                    (start-itimer "xwem-batt"
                                  `(lambda () (xwem-batt-win-update ,bxwin))
                                  xwem-batt-update-interval xwem-batt-update-interval))
    'started))


;;;; In case there is no battery.ell
(unless (fboundp 'apm-battery)
  (defvar apm-program "apm")
  (defvar apm-state-percent-arguments "-bl")
  (defvar apm-status-alist
    '((0 . high) (1 . low) (2 . critical) (3 . charging)))

  (defun apm-battery ()
    "Return battery status."
    (let (state percents)
      (with-temp-buffer
        (call-process apm-program nil (current-buffer)
                      nil apm-state-percent-arguments)
        (goto-char (point-min))
        (setq state (cdr (assq (string-to-int
                                (buffer-substring (point-at-bol)
                                                  (point-at-eol)))
                               apm-status-alist)))
        (forward-line)
        (setq percents (string-to-int
                        (buffer-substring (point-at-bol)
                                          (point-at-eol)))))
      (list (eq state 'charging) state percents))))


(provide 'xwem-battery)

;;; xwem-battery.el ends here
