/* CSL - Common Sound Layer
 * Copyright (C) 2000-2001 Stefan Westerfeld and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef __CSL_DEFS_H__
#define __CSL_DEFS_H__

#include <csl/cslconfig.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 * @file
 * This file defines some common definitions that can be used in CSL
 * for convenience and portability.
 */

/* --- compat defines --- */

/**
 * Portable value for null pointer, if not already defined.
 * @short null pointer.
 */
#ifndef NULL
#define NULL ((void*) 0)
#endif

/**
 * Portable value for boolean true, if not already defined.
 * @short Boolean true.
 */
#ifndef TRUE
#define TRUE	(!FALSE)
#endif

/**
 * Portable value for boolean false, if not already defined.
 * @short Boolean false.
 */
#ifndef FALSE
#define FALSE	(0)
#endif

/**
 * Return the maximum of two values.
 * @param a First value.
 * @param b Second value.
 * @return The larger of a and b
 * @short Return maximum.
 */
#ifndef MAX
#define MAX(a, b)	((a) > (b) ? (a) : (b))
#endif

/**
 * Return the minimum of two values.
 * @param a First value.
 * @param b Second value.
 * @return The lesser of a and b
 * @short Return minimum.
 */
#ifndef MIN
#define MIN(a, b)	((a) > (b) ? (b) : (a))
#endif

/**
 * Clamp a value between the values low and high, i.e. if value is
 * larger than high set it to high.  If it is lower than low, set it
 * to low, otherwise leave it unchanged. Works with any data type.
 * @param x data value
 * @param low lower limit
 * @param high higher limit
 * @return the clamped value of x
 * @short Clamp data within a range.
*/
#ifndef	CLAMP
#define CLAMP(x, low, high)  (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))
#endif

/**
 * Raise a breakpoint exception in a portable way. If running under a
 * debugger this will cause the application to stop and allow
 * debugging. If running outside a debugger it will raise a SIGTRAP
 * exception which by default will terminate the program.
 * @short Breakpoint exception.
 */
#define CSL_BREAKPOINT()   ((void)  raise (5 /* SIGTRAP */))

/* --- typedefs --- */

/**
 * A portable data type for Boolean values.
 * @short Boolean type.
 */
typedef int		          CslBool;

/**
 * A portable data type for byte values.
 * @short Byte-sized type.
 */
typedef int		          CslByte;

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __CSL_DEFS_H__ */
