/* CSL - Common Sound Layer
 * Copyright (C) 2000-2001 Stefan Westerfeld and Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General
 * Public License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include	"cslpcm.h"

#include	"cslutils.h"
#include	"cslprivate.h"
#include	<string.h>


/* --- macros --- */
#define	DRIVER_LOCK(d)		{ (d)->mutex->lock ((d)->mutex); }
#define	DRIVER_UNLOCK(d)	{ (d)->mutex->unlock ((d)->mutex); }


/* --- functions --- */
static CslErrorType
pcm_open_internal (CslDriver       *driver,
		   const char      *role,
		   unsigned int     rate,
		   unsigned int     n_channels,
		   CslPcmFormatType format,
		   CslBool          readable,
		   CslBool          writable,
		   CslPcmStream   **stream_p)
{
  CslErrorType error;
  CslPcmStream *stream;

  csl_return_val_if_fail (driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (driver->pcm_vtable != NULL, CSL_EINTERN);
  csl_return_val_if_fail (role != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream_p != NULL, CSL_EINTERN);

  DRIVER_LOCK (driver);

  error = driver->pcm_vtable->stream_init (driver,
					   role,
					   rate,
					   n_channels,
					   format,
					   readable, writable,
					   &stream);
  if (!error && !stream->active)
    {
      error = driver->pcm_vtable->activate (stream);
      if (error)
	driver->pcm_vtable->stream_destroy (stream);
    }
  if (error)
    *stream_p = NULL;
  else
    *stream_p = stream;

  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_open_output (CslDriver       *driver,
		     const char      *role,
		     unsigned int     rate,
		     unsigned int     n_channels,
		     CslPcmFormatType format,
		     CslPcmStream   **stream_p)
{
  return pcm_open_internal (driver,
			    role,
			    rate,
			    n_channels,
			    format,
			    FALSE, TRUE,
			    stream_p);
}

CslErrorType
csl_pcm_open_input (CslDriver       *driver,
		    const char      *role,
		    unsigned int     rate,
		    unsigned int     n_channels,
		    CslPcmFormatType format,
		    CslPcmStream   **stream_p)
{
  return pcm_open_internal (driver,
			    role,
			    rate,
			    n_channels,
			    format,
			    TRUE, FALSE,
			    stream_p);
}

CslPcmFormatType
csl_pcm_get_format (CslPcmStream *stream)
{
  csl_return_val_if_fail (stream != NULL, 0);
  csl_return_val_if_fail (stream->driver != NULL, 0);

  return stream->format;
}

void
csl_pcm_close (CslPcmStream *stream)
{
  CslDriver *driver;

  csl_return_if_fail (stream != NULL);
  csl_return_if_fail (stream->driver != NULL);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  driver->pcm_vtable->stream_destroy (stream);
  DRIVER_UNLOCK (driver);
}

int
csl_pcm_read (CslPcmStream    *stream,
	      unsigned int     n_bytes,
	      void            *bytes)
{
  CslDriver *driver;
  CslErrorType error;
  
  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->readable == TRUE, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  if (n_bytes)
    {
      csl_return_val_if_fail (bytes != NULL, CSL_EINTERN);
      
      driver = stream->driver;
      DRIVER_LOCK (driver);
      error = driver->pcm_vtable->read (stream, n_bytes, bytes);
      DRIVER_UNLOCK (driver);
    }
  else
    error = CSL_ENONE;

  return error;
}

int
csl_pcm_write (CslPcmStream    *stream,
	       unsigned int     n_bytes,
	       void            *bytes)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->writable == TRUE, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  if (n_bytes)
    {
      driver = stream->driver;
      DRIVER_LOCK (driver);
      error = driver->pcm_vtable->write (stream, n_bytes, bytes);
      DRIVER_UNLOCK (driver);
    }
  else
    error = CSL_ENONE;

  return error;
}

CslErrorType
csl_pcm_get_status (CslPcmStream *stream,
		    CslPcmStatus *status)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (status != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  error = driver->pcm_vtable->update_status (stream);
  /* FIXME: status->... setup */
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_flush (CslPcmStream *stream)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  error = driver->pcm_vtable->flush (stream);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_sink (CslPcmStream *stream)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  error = driver->pcm_vtable->sync (stream);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_activate (CslPcmStream *stream)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == FALSE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  error = driver->pcm_vtable->activate (stream);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_suspend (CslPcmStream *stream)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == TRUE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  error = driver->pcm_vtable->suspend (stream);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_set_title (CslPcmStream *stream,
		   const char   *title)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (title != NULL, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  if (!stream->title || strcmp (stream->title, title) != 0)
    error = driver->pcm_vtable->set_title (stream, title);
  else
    error = CSL_ENONE;
  DRIVER_UNLOCK (driver);

  return error;
}

char*
csl_pcm_dup_title (CslPcmStream *stream)
{
  CslDriver *driver;
  char *title;

  csl_return_val_if_fail (stream != NULL, NULL);
  csl_return_val_if_fail (stream->driver != NULL, NULL);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  title = csl_strdup (stream->title ? stream->title : stream->role);
  DRIVER_UNLOCK (driver);

  return title;
}

CslErrorType
csl_pcm_set_stream_mode (CslPcmStream *stream,
			 unsigned int  buffer_size)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == FALSE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* backend needs to make sure buffer_size is set correctly */
  error = driver->pcm_vtable->set_stream_mode (stream, buffer_size);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_set_packet_mode (CslPcmStream *stream,
			 unsigned int  n_packets,
			 unsigned int  packet_size)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == FALSE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* backend needs to make sure n_packets && packet_size are set correctly */
  error = driver->pcm_vtable->set_packet_mode (stream, n_packets, packet_size);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_set_stream_watermark (CslPcmStream   *stream,
			      unsigned int    n_bytes)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == FALSE, CSL_EINTERN);
  csl_return_val_if_fail (stream->stream_mode == TRUE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* FIXME: should have buffer_size and watermark in stream struct, then assert them */
  error = driver->pcm_vtable->set_stream_watermark (stream, n_bytes);
  DRIVER_UNLOCK (driver);

  return error;
}

CslErrorType
csl_pcm_set_packet_watermark (CslPcmStream   *stream,
			      unsigned int    n_packets)
{
  CslDriver *driver;
  CslErrorType error;

  csl_return_val_if_fail (stream != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->driver != NULL, CSL_EINTERN);
  csl_return_val_if_fail (stream->packet_mode == TRUE, CSL_EINTERN);
  csl_return_val_if_fail (stream->active == FALSE, CSL_EINTERN);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* FIXME: should have n_packets, packet_size and watermark in stream struct, then assert them */
  error = driver->pcm_vtable->set_packet_watermark (stream, n_packets);
  DRIVER_UNLOCK (driver);

  return error;
}

void
csl_pcm_get_stream_settings (CslPcmStream   *stream,
			     unsigned int   *buffer_size_p,
			     unsigned int   *byte_watermark_p)
{
  CslDriver *driver;

  csl_return_if_fail (stream != NULL);
  csl_return_if_fail (stream->driver != NULL);
  csl_return_if_fail (stream->stream_mode == TRUE);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* FIXME: need buffer_size and watermark in stream struct */
  if (buffer_size_p)
    *buffer_size_p = 0;
  if (byte_watermark_p)
    *byte_watermark_p = 0;
  DRIVER_UNLOCK (driver);
}

void
csl_pcm_get_packet_settings (CslPcmStream   *stream,
			     unsigned int   *n_packets_p,
			     unsigned int   *packet_size_p,
			     unsigned int   *packet_watermark_p)
{
  CslDriver *driver;

  csl_return_if_fail (stream != NULL);
  csl_return_if_fail (stream->driver != NULL);
  csl_return_if_fail (stream->packet_mode == TRUE);

  driver = stream->driver;
  DRIVER_LOCK (driver);
  /* FIXME: need n_packets, packet_size and watermark in stream struct */
  if (n_packets_p)
    *n_packets_p = stream->packet.n_total_packets;
  if (packet_size_p)
    *packet_size_p = stream->packet.packet_size;
  if (packet_watermark_p)
    *packet_watermark_p = stream->packet.packet_watermark;
  DRIVER_UNLOCK (driver);
}

char**
csl_pcm_list_channel_mappings (CslDriver    *driver,
			       unsigned int *n_maps_p)
{
  csl_return_val_if_fail (n_maps_p != NULL, NULL);

  *n_maps_p = 0;
  /* FIXME */
  return NULL;
}

CslErrorType
csl_pcm_set_channel_mapping (CslPcmStream *stream,
			     unsigned int  channel,
			     const char   *mapping)
{
  /* FIXME */
  return CSL_EINTERN;
}

char*
csl_pcm_dup_channel_mapping (CslPcmStream *stream,
			     unsigned int  channel)
{
  /* FIXME */
  return NULL;
}
