%module "Math::GSL::Chebyshev"
%include "typemaps.i"
%include "gsl_typemaps.i"
%{
    #include "gsl/gsl_chebyshev.h"
    #include "gsl/gsl_math.h"
    #include "gsl/gsl_mode.h"
%}
%include "gsl/gsl_chebyshev.h"
%include "gsl/gsl_math.h"
%include "gsl/gsl_mode.h"


%perlcode %{
@EXPORT_OK = qw/
               gsl_cheb_alloc 
               gsl_cheb_free 
               gsl_cheb_init 
               gsl_cheb_eval 
               gsl_cheb_eval_err 
               gsl_cheb_eval_n 
               gsl_cheb_eval_n_err 
               gsl_cheb_eval_mode 
               gsl_cheb_eval_mode_e 
               gsl_cheb_calc_deriv 
               gsl_cheb_calc_integ 
             /;
%EXPORT_TAGS = ( all => [ @EXPORT_OK ] );

__END__

=head1 NAME

Math::GSL::Chebyshev - Univariate Chebyshev Series Approximation

=head1 SYNOPSIS

    use Math::GSL::Chebyshev qw /:all/;

    my $cheb             = gsl_cheb_alloc(40);
    my $function         = sub { sin(cos($_[0])) };

    gsl_cheb_init($cheb, $function, 0, 10);

    my $x                = gsl_cheb_eval($cheb, 5.5 );
    my ($status,$y,$err) = gsl_cheb_eval_err($cheb, 7.5 );
    gsl_cheb_free($cheb);

=head1 DESCRIPTION

Here is a list of all the functions in this module :

=over

=item * C<gsl_cheb_alloc($size)>

    my $cheb = gsl_cheb_alloc(50);

Allocates a new Chebyshev object with $size sample points.

=item * C<gsl_cheb_free($cheb)>

Deallocates memory associated to $cheb. Returns void.

=item * C<gsl_cheb_init($cheb,$function, $lower, $upper)>

    gsl_cheb_init($cheb, sub { sin(cos($_[0])) }, 0, 10 );

Initiate a Chebyshev object with a function and upper and lower bounds.
Returns void.

=item * C<gsl_cheb_eval($function, $value)>

    my $evaluated = gsl_cheb_eval($cheb, 5 );

Returns a Perl scalar of the Chebyshev object $cheb evaluated at $value.

=item * C<gsl_cheb_eval_err($cheb, $value)>

    my ($status,$evaluated,$err) = gsl_cheb_eval($cheb, 5 );

Returns a list consisting of a GSL status code, the evaluate value and
the estimated error of the evaluation.

=item * C<gsl_cheb_eval_n >

=item * C<gsl_cheb_eval_n_err >

=item * C<gsl_cheb_eval_mode >

=item * C<gsl_cheb_eval_mode_e >

=item * C<gsl_cheb_calc_deriv($deriv,$cheb) >

   my $status = gsl_cheb_calc_deriv( $deriv, $cheb ); 

This will calculate the derivative of $cheb and stores it
in $deriv, which must be pre-allocated. Returns a GSL status code.

=item * C<gsl_cheb_calc_integ($integ,$cheb) >

   my $status = gsl_cheb_calc_integ( $deriv, $cheb ); 

This will calculate the derivative of $cheb and stores it
in $deriv, which must be pre-allocated. Returns a GSL status code.

=back

For more informations on the functions, we refer you to the GSL offcial
documentation: L<http://www.gnu.org/software/gsl/manual/html_node/>

 Tip : search on google: site:http://www.gnu.org/software/gsl/manual/html_node/ name_of_the_function_you_want


=head1 AUTHORS

Jonathan Leto <jonathan@leto.net> and Thierry Moisan <thierry.moisan@gmail.com>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2008 Jonathan Leto and Thierry Moisan

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

%}
