// This may look like C code, but it is really -*- C++ -*-
//
// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//

// tcsock.h - TECC Socket class for use with winsock.dll / zApp
//
// $Id: tcsock.h,v 1.1 1993/09/02 12:33:58 pete Exp pete $
//
#ifndef _tcsock_h_
#define _tcsock_h_

// Constants for debug message level
enum tcLogLevel {
	TC_DEBUG,				// Real low-level stuff
	TC_TRACE,				// General trace stuff
    TC_OPER,				// Start of an operation
	TC_INFO,				// Informational
	TC_STATE,				// Application state
    TC_WARNING,				// Warning information
	TC_ERROR,				// Real errors
};

// Class for logging error messages

class tcLogger {
  protected:
	static const char *logLevelName[];
  public:
	void			logf(tcLogLevel lev, const char *fmt, ...);
	virtual void	log(tcLogLevel lev, const char *string)			=0;
};
// Use the pre-processor to avoid overhead if we will never want to debug
// If the following symbols are not set appropriately, then the tcSocket
// classes will never log TC_DEBUG or TC_TRACE messages, which means that
// they cannot by logged to file etc even if the tcLogger is capable of
// that.  Normally, one would only want to do this if speed really
// mattered...
#define TCSOCK_TRACE	1		// 1 to generate socket trace messages
#define TCSOCK_DEBUG	1		// 1 to generate socket debug messages



// Currently arbitrary Windows message number to use for socket notification.
// This should really be done via an atom...
#define WM_SOCKET	(WM_USER + 111)

// Arbitrary limit on the number of tcSockets the socket manager can
// handle
const SOCKET_MAX = 16;

//
// zEvent derived class for decomposing Winsock notification messages
//
class sockEvent : public zEvent {
  public:
	SOCKET			sock()	{ return parm1();	};
	unsigned int	event()	{ return loParam();	};
    unsigned int	error()	{ return hiParam();	};
};

//
// The actual socket base class
//
class tcSocket {
	struct errDesc {						// Error message descriptor
		int		num;						// Error number
		char	*string;					// Error description
	};
	friend class tcSockManager;
  protected:
  	static int				instCount;		// Number of sockets that exist
  	static tcSockManager	*tcMgr;			// Global manager
	static errDesc			errString[];	// Error strings
	int						evMask;			// Events we are interested in
	SOCKET					sock;			// The actual socket
	int						lastErr;		// Last socket error
	tcLogger				*logger;		// Event logger
	void					updateMask() {	// Update the event mask
		int rc = WSAAsyncSelect(sock, app->rootWindow()->sysId(),
								WM_SOCKET, evMask);
		if (rc != 0) {
			lastErr = WSAGetLastError();
			logger->logf(TC_ERROR, "Select error: %d", lastErr);
		}
	}
	SOCKET					winSock()	{ return sock;		};
	void					Init(SOCKET s);
  public:
	tcSocket(tcLogger *l, int type =SOCK_STREAM);
	tcSocket(tcLogger *l, SOCKET sock);
	~tcSocket();
	void	addMask(int m) {
		evMask |= m;
		updateMask();
	}
	void	delMask(int m) {
		evMask &= ~m;
		updateMask();
    }
	int				Connect(const char *host, int port);	// Connect to server
	int				Connect(const char *host, const char *serv,
							const char *prot ="tcp");
	int				Listen(int port);				// Listen for client
	tcSocket		*Accept();						// Accept new client
	void			Close() {
		if (sock != INVALID_SOCKET) {
			evMask = 0;
			updateMask();
			closesocket(sock);
			sock = INVALID_SOCKET;
		}
	}
	virtual void	Connected(int error);
	virtual void	Accepted(int error);
	virtual void	Closed(int error);
	virtual void	Readable(int error);
	virtual void	Writable(int error);
	virtual void	OOBData(int error);
	const char		*sockErrMessage(int err);
	int				lastError()			{ return lastErr;					};
	const char		*lastErrorMessage()	{ return sockErrMessage(lastErr);	};
	void			clearError()		{ lastErr = 0;						};
};

/////////////////////////////////////////////////////////////////////////////
// Derived sockets with some better buffer handling
//

class tcBufSocket : public tcSocket {
  protected:
	int		rbSize, wbSize;		// Buffer sizes
	int		nrBuf, nwBuf;		// Number of characters buffered
	char	*rBuf;				// Read buffer
	char	*wBuf;

	void	pullUp(char *buf, int &cnt, int amt) {	// Re-align a buffer
		cnt -= amt;									// Recalculate the counter
		if (cnt > 0) {								// Is there anything to do?
			memmove(buf, buf + amt, cnt);			// Copy remaining data
		}
	}
	void	rdPullUp(int amt)	{ pullUp(rBuf, nrBuf, amt);		};
	void	wrPullUp(int amt)	{ pullUp(wBuf, nwBuf, amt);		};
	void	Init(int rbw, int wbs);					// Set up the buffers
	
  public:
	tcBufSocket(tcLogger *l, int rbs =1024, int wbs =1024);
	tcBufSocket(tcLogger *l, SOCKET s, int rbs =1024, int wbs =1024);
	~tcBufSocket();
	int				numRbuf()		{ return nrBuf;				};
	int				numWbuf()		{ return nwBuf;				};
	int				rdSpace()		{ return (rbSize - nrBuf);	};
	int				wrSpace()		{ return (wbSize - nwBuf);	};
	int				send(const char *data, int amt =-1);
	int				recv(char *dest, int size);

	void			Readable(int error);		// Callback from tcSocket
	void			Writable(int error);		// Callback from tcSocket

	virtual void	dataRead()		=0;			// Data available to read
	virtual void	dataWritten()	=0;			// Write buffer empty
	virtual void	readError(int error);		// Read error callback
	virtual void	writeError(int error);		// Write error callback
};

/////////////////////////////////////////////////////////////////////////////
// Trivial line-at-a-time layer on top of tcBufSocket
//

class tcLineBufSocket : public tcBufSocket {
  protected:
  public:
	tcLineBufSocket(tcLogger *l, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, rbs, wbs) { };
	tcLineBufSocket(tcLogger *l, SOCKET s, int rbs =1024, int wbs =1024)
	: tcBufSocket(l, s, rbs, wbs) { };

	int				readLine(char *buffer, int len); // Read a line
	int				writeLine(const char *line) {	 // For consistent naming
		int rc = send(line);
		if (rc > 0) send("\n", 1);
        return rc;
	}

	void			dataRead();					// Callback from tcBuffSocket

	virtual void	lineRead()		=0;			// A whole line available
};

/////////////////////////////////////////////////////////////////////////////
// tcSockManager.  Class to dispatch messages to tcSockets
//

class tcSockManager : public zEvH {
	friend class tcSocket;
  protected:							// Everything is protected...
	WSADATA			wsData;				// Socket library data
	tcSocket		*socklist[SOCKET_MAX];
	tcSocket		*findSock(SOCKET s);
	BOOL			addSock(tcSocket *s);
	void			delSock(tcSocket *s);
	tcLogger		*logger;

  	tcSockManager(tcLogger *l);
	~tcSockManager();
	int				sockHandler(sockEvent *ev);
};

#endif // _tcsock_h_
