// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
#include <zapp.hpp>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>
#include <errno.h>
#include "winsock.h"
#include "tcsock.h"

static char RCSid[] = "$Id: tcsock.cpp,v 1.1 1993/09/02 12:33:58 pete Exp pete $";
/////////////////////////////////////////////////////////////////////////////
// tcLogger member functions.  Should go elsewhere...
//
const char *tcLogger::logLevelName[] = {
	"Debug",
	"Trace",
    "Start operation",
	"Information",
	"Enter state",
	"WARNING",
	"ERROR",
};

void tcLogger::logf(tcLogLevel lev, const char *fmt, ...) {
	char	buffer[256];
	va_list	args;

	va_start(args, fmt);
	vsprintf(buffer, fmt, args);
	va_end(args);
	log(lev, buffer);
}


/////////////////////////////////////////////////////////////////////////////
// tcSocket class variables and member functions
//
int 			tcSocket::instCount = 0;	// Instance count
tcSockManager	*tcSocket::tcMgr = NULL;	// Global manager object
tcSocket::errDesc tcSocket::errString[] = {
/*
 * Windows Sockets definitions of regular Microsoft C error constants
 */
	WSAEINTR,			"WSAEINTR: Interrupted 'system call'",
	WSAEBADF,			"WSAEBADF: Bad file descriptor",
	WSAEACCES,			"WSAEACCES: Permission denied",
	WSAEFAULT,			"WSAEFAULT: Bad address",
	WSAEINVAL,			"WSAEINVAL: Invalid argument",
	WSAEMFILE,			"WSAEMFILE: Too many open files",
/*
 * Windows Sockets definitions of regular Berkeley error constants
 */
	WSAEWOULDBLOCK,		"WSAEWOULDBLOCK: Would block",
	WSAEINPROGRESS,		"WSAEINPROGRESS: Blocking call already in progress",
	WSAEALREADY,		"WSAEALREADY: Operation already in progress",
	WSAENOTSOCK,		"WSAENOTSOCK: Not a socket",
	WSAEDESTADDRREQ,	"WSAEDESTADDRREQ: Destination address required",
	WSAEMSGSIZE,		"WSAEMSGSIZE: Message too long",
	WSAEPROTOTYPE,		"WSAEPROTOTYPE: Wrong protocol for socket",
	WSAENOPROTOOPT,		"WSAENOPROTOOPT: Protocol option not available",
	WSAEPROTONOSUPPORT,	"WSAEPROTONOSUPPORT: Protocol not suported",
	WSAESOCKTNOSUPPORT,	"WSAESOCKTNOSUPPORT: Socket type not supported",
	WSAEOPNOTSUPP,		"WSAEOPNOTSUPP: Operation not supported on socket",
	WSAEPFNOSUPPORT,	"WSAEPFNOSUPPORT: Protocol family not supported",
	WSAEAFNOSUPPORT,	"WSAEAFNOSUPPORT: Address familt not supported",
	WSAEADDRINUSE,		"WSAEADDRINUSE: Address already in use",
	WSAEADDRNOTAVAIL,	"WSAEADDRNOTAVAIL: Can't assign requested address",
	WSAENETDOWN,		"WSAENETDOWN: Network is down",
	WSAENETUNREACH,		"WSAENETUNREACH: Network is unreachable",
	WSAENETRESET,		"WSAENETRESET: Network dropped connection on reset",
	WSAECONNABORTED,	"WSAECONNABORTED: Software caused connection abort",
	WSAECONNRESET,		"WSAECONNRESET: Connection reset by peer",
	WSAENOBUFS,			"WSAENOBUFS: No buffer space available",
	WSAEISCONN,			"WSAEISCONN: Socket is already connected",
	WSAENOTCONN,		"WSAENOTCONN: Socket is not connected",
	WSAESHUTDOWN,		"WSAESHUTDOWN: Can't send after socket shutdown",
	WSAETOOMANYREFS,	"WSAETOOMANYREFS: Too many references: can't splice",
	WSAETIMEDOUT,		"WSAETIMEDOUT: Connection timed out",
	WSAECONNREFUSED,	"WSAECONNREFUSED: Connection refused",
	WSAELOOP,			"WSAELOOP: Too many levels of symbolic link (!)",
	WSAENAMETOOLONG,	"WSAENAMETOOLONG: File name too long (!)",
	WSAEHOSTDOWN,		"WSAEHOSTDOWN: Host is down",
	WSAEHOSTUNREACH,	"WSAEHOSTUNREACH: No route to host",
	WSAENOTEMPTY,		"WSAENOTEMPTY: Directory not empty (!)",
	WSAEPROCLIM,		"WSAEPROCLIM: Too many processes (!)",
	WSAEUSERS,			"WSAEUSERS: Too many users (!)",
	WSAEDQUOT,			"WSAEDQUOT: Disk quota exceeded (!)",
	WSAESTALE,			"WSAESTALE: Stale NFS handle (!)",
	WSAEREMOTE,			"WSAEREMOTE: Too many levels of remote in path (!)",
/*
 * Extended Windows Sockets error constant definitions
 */
	WSASYSNOTREADY,		"WSASYSNOTREADY: System not ready",
	WSAVERNOTSUPPORTED,	"WSAVERNOTSUPPORTED: Version not supported",
	WSANOTINITIALISED,	"WSANOTINITIALISED: WinSock not initialised",
/*
 * Resolver errors
 */
	WSAHOST_NOT_FOUND,	"WSAHOST_NOT_FOUND: Host not found (authoritative)",
	WSATRY_AGAIN,		"WSATRY_AGAIN: Host not found (non-authoritative)",
	WSANO_RECOVERY,		"WSANO_RECOVERY: Nameserver error",
	WSANO_DATA,			"WSANO_DATA: No host data record of requested type",
	0,					NULL,
};

/////////////////////////////////////////////////////////////////////////////
// Contstructor & destructor
//
tcSocket::tcSocket(tcLogger *l, int type) {

	logger = l;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcSocket, type=%d, this=%p", type, this);
#endif
	
	if (instCount++ == 0) {			// First instance
#if TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "Creating tcSocketManager");
#endif
		// Crap design choice.  The tcSocketManager will get the logger
		// from whatever socket creates it.  This will not be such a problem
		// once tcSockets do their own message handling.
		tcMgr = new tcSockManager(logger);	// Create a manager object
	}
	
	sock = socket(AF_INET, type, 0);	// Get a SOCKET from Winsock
#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcSocket: SOCKET=%d", sock);
#endif
	Init(sock);
}


tcSocket::tcSocket(tcLogger *l, SOCKET s) {

	logger = l;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcSocket, SOCKET=%d, this=%p", s, this);
#endif
	if (instCount++ == 0) {			// First instance
		// Crap design choice.  The tcSocketManager will get the logger
		// from whatever socket creates it.  This will not be such a problem
		// once tcSockets do their own message handling.
		tcMgr = new tcSockManager(logger);	// Create a manager object
	}
	
	sock = s;
	Init(sock);
}

void tcSocket::Init(SOCKET sock) {

	lastErr = 0;
	if (sock == INVALID_SOCKET) {
		lastErr = WSAGetLastError();
    	logger->logf(TC_ERROR, "Socket creation failure: %s",
					 lastErrorMessage());
	}
	else {
        tcMgr->addSock(this);			// Set up in manager list
		evMask = 0;  					// Clear event mask
        updateMask();
	}
}

tcSocket::~tcSocket() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Destroying tcSocket: %p", this);
#endif
	tcMgr->delSock(this);				// Remove entry from manager
	Close();

	if (--instCount == 0) {				// Last instance
		delete tcMgr;					// Delete the manager
		tcMgr = NULL;
	}
}


/////////////////////////////////////////////////////////////////////////////
// Connect: Connect to a remote server
//

int tcSocket::Connect(const char *host, int port) {
	sockaddr_in	addr;
	hostent     *hent;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Connect(%s, %u)", this, host, port);
#endif
	addr.sin_family = AF_INET;
	addr.sin_port = htons(port);

	if (isdigit(host[0])) {
		addr.sin_addr.s_addr = inet_addr(host);
		if (addr.sin_addr.s_addr == NULL) {
			lastErr = WSAEINVAL;
			logger->logf(TC_ERROR, "Invalid numeric IP address: \"%s\"", host);
			return SOCKET_ERROR;
        }
	}
    else {					// Try and look up in DNS
		hent = gethostbyname(host);
	    if (hent == NULL) {
			lastErr = WSAGetLastError();
			logger->logf(TC_ERROR, "Host name lookup failure for \"%s\": %s\n",
						 host, lastErrorMessage());
			return SOCKET_ERROR;
    	}
		addr.sin_addr.s_addr = *((long *) hent->h_addr);
    }
#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "IP Address = %s", inet_ntoa(addr.sin_addr));
#endif

	addMask(FD_CONNECT);				// Get connection notifications
	logger->logf(TC_INFO, "tcSocket calling connect(%d, %s, %d)",
				 sock, inet_ntoa(addr.sin_addr), port);

	int rc = connect(sock, (sockaddr *) &addr, sizeof(addr));
	logger->logf(TC_INFO, "Connect returned %d", rc);
	if (rc == SOCKET_ERROR) {			// Connection not yet complete
		lastErr = WSAGetLastError();
		if (lastErr != WSAEWOULDBLOCK) {
			logger->logf(TC_ERROR, "tcSocket(%p): Connect error: %s",
						 this, lastErrorMessage());
			Close();
			return SOCKET_ERROR;
		}
		logger->logf(TC_DEBUG, "Current event mask: %04x", evMask);
		logger->logf(TC_INFO, "tcSocket(%p): Connection in progress...",
					this);
	}
	else {								// Connection completed OK
		logger->logf(TC_INFO, "Connect() completed synchronously(!)");
        Connected(0);					// Invoke callback
	}
	return 0;
}

int tcSocket::Connect(const char *host, const char *serv, const char *prot) {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Connect(%s, %s, %s)",
				 this, host, serv, prot);
#endif
	servent *ent = getservbyname(serv, prot);
	if (ent == NULL) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "Unable to find service name \"%s\": %s",
					 serv, lastErrorMessage());
		return SOCKET_ERROR;
	}
	return Connect(host, ntohs(ent->s_port));
}

/////////////////////////////////////////////////////////////////////////////
// Listen().  Listen for incoming connections
//
int tcSocket::Listen(int port) {
	sockaddr_in	addr;

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Listen(%d)", this, port);
#endif
	addr.sin_family = AF_INET;
    addr.sin_addr.s_addr = htonl(INADDR_ANY);
    addr.sin_port  = htons(port);
	
	if (bind(sock, (sockaddr *) &addr, sizeof(addr)) == SOCKET_ERROR) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Bind failed: %s",
					 lastErrorMessage());
		Close();
        return SOCKET_ERROR;
	}
	if (listen(sock, 5) == SOCKET_ERROR) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Bind failed: %s",
					 lastErrorMessage());
		Close();
        return SOCKET_ERROR;
    }
	addMask(FD_ACCEPT);			// Get notification when connections arrive

	logger->logf(TC_INFO, "tcSocket(%p): Listening on port %u", this, port);
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// Accept().  Accept a new connection
//
tcSocket *tcSocket::Accept() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "tcSocket(%p)::Accept()", this);
#endif
	SOCKET ns = accept(sock, NULL, NULL);
	if (ns == INVALID_SOCKET) {
		lastErr = WSAGetLastError();
		logger->logf(TC_ERROR, "tcSocket(%p): Accept failed: %s",
					 this, lastErrorMessage());
		return NULL;
	}
	return new tcSocket(logger, ns);			// Create new socket
}


/////////////////////////////////////////////////////////////////////////////
// Default action functions for socket events
//
void tcSocket::Connected(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Connected: error %d", this, error);
}

void tcSocket::Accepted(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Accepted: error %d", this, error);
}

void tcSocket::Closed(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Closed: error %d", this, error);
}

void tcSocket::Readable(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Readable: error %d", this, error);
}

void tcSocket::Writable(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): Writable: error %d", this, error);
}

void tcSocket::OOBData(int error) {
	logger->logf(TC_INFO, "tcSocket(%p): OOBData: error %d", this, error);
}

/////////////////////////////////////////////////////////////////////////////
//
const char *tcSocket::sockErrMessage(int err) {
	static char fakebuf[80];
	
	if (err == 0) return "No WinSock error";

	for (errDesc *cur = errString; cur->num != 0; cur++) {
		if (cur->num == err) return cur->string;
	}
	sprintf(fakebuf, "Unrecognised error number %d", err);
	return fakebuf;
}

/////////////////////////////////////////////////////////////////////////////
// tcBufSocket member functions
//

tcBufSocket::tcBufSocket(tcLogger *l, int rbs, int wbs)
: tcSocket(l, SOCK_STREAM) {			// Only makes sense with TCP sockets

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcBufSocket, this=%p", this);
#endif
	Init(rbs, wbs);
}

tcBufSocket::tcBufSocket(tcLogger *l, SOCKET s, int rbs, int wbs) : tcSocket(l, s) {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Creating tcBufSocket, this=%p", this);
#endif
	Init(rbs, wbs);
}

tcBufSocket::~tcBufSocket() {

#if TCSOCK_TRACE
	logger->logf(TC_TRACE, "Destroying tcBufSocket, this=%p", this);
#endif
	if (rBuf != NULL) delete rBuf;
	if (wBuf != NULL) delete wBuf;
}

void tcBufSocket::Init(int rbs, int wbs) {

	rbSize = wbSize = nrBuf = nwBuf = 0;	// Initialise
	if ((rBuf = new char[rbs]) != NULL) rbSize = rbs;
	if ((wBuf = new char[wbs]) != NULL) wbSize = wbs;
}


int tcBufSocket::send(const char *data, int amt) {

	if (amt == -1) amt = strlen(data);		// Default amount
#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): send(%p, %u)", this, data, amt);
#endif
	if (amt > wrSpace()) {
		lastErr = WSAENOBUFS;
		logger->logf(TC_ERROR, "tcBufSocket(%p): send(): Out of buffer space",
					 this);
		return SOCKET_ERROR;
	}
	if (amt > 0)  {
		memcpy(wBuf + nwBuf, data, amt);	// Copy the data in
		nwBuf += amt;
    }
	addMask(FD_WRITE);						// Get message when writable
	return wrSpace();
}

int tcBufSocket::recv(char *buf, int amt) {

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): recv(%p, %u)", this, buf, amt);
#endif
	if (amt > nrBuf) amt = nrBuf;			// Limit to amount buffered
	if (amt != 0) {
		memcpy(buf, rBuf, amt);				// Give data to user
		rdPullUp(amt);						// Re-align the data 
		if (rdSpace() > 0) addMask(FD_READ); // Ask for more
	}
	return amt;
}

void tcBufSocket::Readable(int error) {		// Called by socket manager

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): Readable(%d)", this, error);
#endif
	if ((error == 0) && (rdSpace() != 0)) {	// Try to fill buffer
		int amt = ::recv(sock, rBuf + nrBuf, rdSpace(), 0);
		if (amt < 0) {
			error = WSAGetLastError();		// Save error number
		}
		else if (amt > 0) {
			nrBuf += amt;					// Update count
			if (rdSpace() == 0)				// No space for more data!
				delMask(FD_READ);
			dataRead();						// Call callback
		}
	}
	if ((error != 0) && (error != WSAEWOULDBLOCK)) {
		readError(error);					// Pass on to error handler
	}
}

void tcBufSocket::Writable(int error) {		// Called by socket manager

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcBufSocket(%p): Writable(%d)", this, error);
#endif
	if (error != 0) logger->logf(TC_ERROR, "tcBufSocket::Writable called with error %d", error);
	while ((error == 0) && (nwBuf != 0)) {		// Try to write data
		int amt = ::send(sock, wBuf, nwBuf, 0);
#ifdef TCSOCK_DEBUG
		logger->logf(TC_DEBUG, "::send(%p, %d) returned %d", wBuf,
					nwBuf, amt);
#endif
		if (amt < 0) {
			error = WSAGetLastError();		// Save error value
		}
		else if (amt > 0) {					// Actually wrote something
			wrPullUp(amt);
		}
	}
	if ((error != 0) && (error != WSAEWOULDBLOCK)) {
		writeError(error);					// Pass on to error handler
	}
	if (nwBuf == 0) {
		delMask(FD_WRITE);					// Nothing more to write
		dataWritten();						// Call callback
	}
}

void tcBufSocket::readError(int error) {

	lastErr = error;
	logger->logf(TC_ERROR, "tcBufSocket(%p): Read error: %s", this,
				 sockErrMessage(error));
}

void tcBufSocket::writeError(int error) {

	lastErr = error;
	logger->logf(TC_ERROR, "tcBufSocket(%p): Write error: %s", this,
				 sockErrMessage(error));
}


/////////////////////////////////////////////////////////////////////////////
// Line buffered sockets...
//

// Read a line of data out of the buffer
int tcLineBufSocket::readLine(char *buffer, int len) {

#if TCSOCK_DEBUG
	logger->logf(TC_DEBUG, "tcLineBufSocket(%p): readLine(%p, %u)", this,
				 buffer, len);
#endif
	if (nrBuf == 0) {					// No data
		lastErr = EIO;
		logger->logf(TC_ERROR, "No data available for readLine");
		return SOCKET_ERROR;
	}
	char *s = (char *) memchr(rBuf, '\n', nrBuf);	// Find end of line
	if (s == NULL) {
		lastErr = EIO;
		logger->logf(TC_ERROR, "Incomplete line data");
		return SOCKET_ERROR;
	}
	int linelen = (s - rBuf) + 1;		// Space needed to hold line
	if (linelen > len) {				// Too big...
		lastErr = WSAENOBUFS;
		logger->log(TC_ERROR, "readLine: User's buffer is too small");
		return SOCKET_ERROR;			
	}
	if (recv(buffer, linelen) != linelen) {
		lastErr = WSAEFAULT;			// Ok, so that's *really* horrible!
		logger->logf(TC_ERROR, "Internal error in readLine()!");
		return SOCKET_ERROR;
	}
	linelen--;							// Get rid of the \n
	buffer[linelen] = '\0';
	return (linelen);
}

void tcLineBufSocket::dataRead() {		// Called by tcBuffSocket

	if (memchr(rBuf, '\n', nrBuf) != NULL) {	// Got a line?
		while (memchr(rBuf, '\n', nrBuf) != NULL) lineRead();
	}
	else {								// Test for overflow
		if (rdSpace() == 0) {			// TODO: Finish this
			logger->logf(TC_ERROR, "Warning! Read buffer overflow!");
		}
	}
}



/////////////////////////////////////////////////////////////////////////////
// tcSockManager member functions
//
/////////////////////////////////////////////////////////////////////////////
// Constructor & Destructor
//

tcSockManager::tcSockManager(tcLogger *l) {

	logger = l;
	
	// Tell zApp to pass all WM_SOCKET messages send to the application's
	// root window to our sockHandler() member (q.v.)
	app->setHandler(this, (NotifyProc) &tcSockManager::sockHandler,	WM_SOCKET);

	// Start Winsock.  TODO: handle version numbers properly.
	logger->logf(TC_INFO, "Initialising WinSock library");
	int rc = WSAStartup(0x0101, &wsData);
	if (rc != 0) {
		logger->logf(TC_ERROR, "Socket startup failure: %d",
					 WSAGetLastError());
	}

	// Zero out the list of sockets
	for (int i = 0; i < SOCKET_MAX; i++) {
		socklist[i] = NULL;
	}
}


tcSockManager::~tcSockManager() {

	// Tell zApp we don't want no more messages
	app->removeHandler(this, (NotifyProc) &tcSockManager::sockHandler,
						WM_SOCKET);

	WSACleanup();					// Close down Winsock
}

/////////////////////////////////////////////////////////////////////////////
// sockHandler().  This function gets called whenever the winsock DLL
// dispatches a WM_SOCKET message to the application.  This function
// uses the SOCKET handle to find a tcSocket, then invokes an appropriate
// member function in that object

int	tcSockManager::sockHandler(sockEvent *ev) {

	tcSocket *s = findSock(ev->sock());		// Find the correct tcSocket
	if (s == NULL) {
// This is not worth logging.  If a socket with FD_CLOSE set is
// closed from the Winsock end at the same time as the remote end,
// then an FD_CLOSE may well be delivered after the socket has gone...
//		statusLine("Message received for unknown socket %u", ev->sock());
		return 0;
	}
	else {						// Got one.  Pick a member function to call
		switch (ev->event()) {
			case FD_READ:		s->Readable(ev->error()); break;
			case FD_WRITE:		s->Writable(ev->error()); break;
			case FD_OOB:		s->OOBData(ev->error()); break;
			case FD_ACCEPT:		s->Accepted(ev->error()); break;
			case FD_CONNECT:	s->Connected(ev->error()); break;
			case FD_CLOSE:		s->Closed(ev->error()); break;
		}
		return 1;
	}
}


/////////////////////////////////////////////////////////////////////////////
// Utility functions for managing the list of sockets.
// Currently these use a linear search down an array of tcSockets, which
// is nearly optimal if only one (or maybe two) tcSockets exist,
// but is cr*p if there are lots.

// Find a socket in the list
tcSocket *tcSockManager::findSock(SOCKET s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if ((socklist[i] != NULL) && (socklist[i]->winSock() == s)) {
			return socklist[i];
		}
	}
	return NULL;
}

// Add a socket to the list
BOOL tcSockManager::addSock(tcSocket *s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if (socklist[i] == NULL) {
			socklist[i] = s;
			return TRUE;
		}
	}
	return FALSE;
}

// Remove a socket from the list
void tcSockManager::delSock(tcSocket *s) {
	for (int i = 0; i < SOCKET_MAX; i++) {
		if (socklist[i] == s){
			socklist[i] = NULL;
		}
	}
}

