// This may look like C code, but it is really -*- C++ -*-
//
// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
// ffClient.h - various classes and structs needed for a FlexFax client
//
// $Id: ffclient.h,v 1.1 1993/09/02 12:33:58 pete Exp pete $
//

// Hack to get output
int statusLine(const char *fmt, ...);

// Forward references
class ffSocket;

// Fascist arbitrary limits
const ffMaxDest = 16;				// Max. destinations for a fax
const ffMaxFile = 16;		   		// Max. files in a fax


/////////////////////////////////////////////////////////////////////////////
// ffSettings
// A struct holding the user configurable stuff (host to use, username etc)
//
// Knows how to store and retrieve itself from a .INI file (defaults to
// WIN.INI)
//

struct ffSettings {
	zString			faxHost;		// Where to submit faxes to
	zString			faxUser;		// User name to use on host
	zString			faxFullName;	// User's full name
	zString			faxEmail;		// Fax user's email address
	zString			coverTemplate;	// Template file for covers
	zString			logFile;		// File to log information to
	int				logToFile;
	int				logDebug, logTrace, logInfo, logOper, logState;
	int				logWarn, logError;

	ffSettings() {				// Zero out all the strings
		faxHost = "";
		faxUser = "";
		faxFullName = "";
		faxEmail = "";
		coverTemplate = "";
		logFile = "";
		logToFile = 0;
		logDebug = 0; logTrace = 0; logInfo = 0; logOper = 0; logState = 0;
		logWarn = 1; logError = 1;

	};
	~ffSettings() { };

	void readFromIni(char *s ="WinFlex", char *iFile = "WIN.INI") {
		char	buffer[128];
		GetPrivateProfileString(s, "FaxHost", "", buffer, 127, iFile);
		faxHost = buffer;
		GetPrivateProfileString(s, "FaxUser", "", buffer, 127, iFile);
		faxUser = buffer;
		GetPrivateProfileString(s, "FullName", "", buffer, 127, iFile);
		faxFullName = buffer;
		GetPrivateProfileString(s, "Email", "", buffer, 127, iFile);
		faxEmail = buffer;
		GetPrivateProfileString(s, "CoverTemplate", "", buffer, 127, iFile);
		coverTemplate = buffer;
		GetPrivateProfileString(s, "LogFile", "", buffer, 127, iFile);
		logFile = buffer;
		logToFile = GetPrivateProfileInt(s, "LogToFile", 0, iFile);
		logDebug = GetPrivateProfileInt(s, "LogDebug", 0, iFile);
		logTrace = GetPrivateProfileInt(s, "LogTrace", 0, iFile);
		logInfo = GetPrivateProfileInt(s, "LogInfo", 0, iFile);
		logOper = GetPrivateProfileInt(s, "LogOperation", 0, iFile);
		logState = GetPrivateProfileInt(s, "LogState", 0, iFile);
		logWarn = GetPrivateProfileInt(s, "LogWarning", 1, iFile);
		logError = GetPrivateProfileInt(s, "LogError", 1, iFile);
	}
	void writeToIni(char *s ="WinFlex", char *iFile = "WIN.INI") {
		WritePrivateProfileString(s, "FaxHost", faxHost, iFile);
		WritePrivateProfileString(s, "FaxUser", faxUser, iFile);
		WritePrivateProfileString(s, "FullName", faxFullName, iFile);
		WritePrivateProfileString(s, "Email", faxEmail, iFile);
		WritePrivateProfileString(s, "CoverTemplate", coverTemplate, iFile);
		WritePrivateProfileString(s, "LogFile", logFile, iFile);
		WritePrivateProfileString(s, "LogToFile", logToFile ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogDebug", logDebug ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogTrace", logDebug ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogInfo", logError ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogOperation", logOper ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogState", logState ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogWarning", logWarn ? "1" : "0", iFile);
		WritePrivateProfileString(s, "LogError", logError ? "1" : "0", iFile);
	}
};

/////////////////////////////////////////////////////////////////////////////
// ffFile.  Abstract base class describing a file to be faxed as part of
// a job.  Derived classes know what type of file they are (ie what tag
// to send to the FlexFax daemon
//

class ffFile {				// Information describing a file to be faxed
  protected:
	enum FType {
		TIFF,				// G3 TIFF File
		PS,					// PostScript
	};
    char	*fileName;		// Name of file to send
	FType	type;			// Type of file

	// Protected constructor to prevent instances of the base class
	// being created
	ffFile(FType t, const char *name) {
		type = t;
		fileName = strpdup(name);
	}

  public:
	~ffFile() {
		if (fileName != NULL) delete fileName;
	};
	const char		*name() const 		{ return fileName;	}
	const char		*typeName() const	{
		return (type == PS) ? "postscript" : "TIFF";
	}
	virtual int		numPages() const		=0;
};


// ffPsFile.  A PostScript ffFile
class ffPsFile : public ffFile {			// PostScript file
  public:
	ffPsFile(const char *name) : ffFile(PS, name) { };
	int				numPages() const;
};

/////////////////////////////////////////////////////////////////////////////
// ffDest.  Destination for a fax.  Hold telephone number, fax parameters
// and things like company, recipient name (for use in generating cover
// sheets.

struct ffDest {						// Description of fax destination
	zString		teleNum;			// Telephone number
	zString		recipName;			// Recipient name
	zString		recipCompany;		// Recipient's company
    zString		regarding;			// Subject line
	zString		comments;			// Comment text
	zString		sendTime;			// Time to send
	zString		killTime;			// Time to kill job
	zString		maxDial;			// Max. number of dial attempts
	long		vRes;				// Vertical resolution
	long		pWidth;				// Page width
	long		pLen;				// Page length
	BOOL		doCover;			// Add a cover sheet?

	// Constructor.  Currently sets default values for non-configurable
	// fields. TODO: Make these configurable.
	ffDest() {
		killTime = "now + 1 day";
		regarding = "";
	}
};

/////////////////////////////////////////////////////////////////////////////
// ffJob.  Class to hold the files, destinations and settings which
// make up a job to be submitted to FlexFax.
//

class ffJobBase {
  protected:
	enum ffJobType {
		SubmitFax, SendStatus, RecvStatus
	};
	ffJobType		type;
	ffSettings		*settings;			// WinFlex settings
	ffFile			*file[ffMaxFile];	// Files to send
	ffDest			*dest[ffMaxDest];	// Destinations for files
	int				nFile, nDest;		// No. of files, destinations

	ffJobBase(ffJobType ty, ffFile *f, ffDest *d, ffSettings *s);

  public:
	virtual				~ffJobBase();
	BOOL				isSubmit()		{ return (type == SubmitFax);	};
	BOOL				isSendStatus()	{ return (type == SendStatus);	};
	BOOL				isRecvStatus()	{ return (type == RecvStatus);	};
	const char			*host()	{ return (char *) settings->faxHost;	};
	const char			*user() { return (char *) settings->faxUser;	};
	const char			*sender() { return (char *) settings->faxFullName;};
	const char			*mailaddr() { return (char *) settings->faxEmail;};
	const char			*coverTemplate() {
		return (char *) settings->coverTemplate;
	};
	int					numFiles() { return nFile;						};
	int					numDest() { return nDest;						};
	const ffFile		*getFile(int n) {
		return ((n < 0) || (n >= nFile)) ? NULL : file[n];
	}
	const ffDest		*getDest(int n) {
		return ((n < 0) || (n >= nDest)) ? NULL : dest[n];
	}
    const ffSettings	*getSettings() { return settings;	};
};

class ffSendJob : public ffJobBase {		// A FlexFax job for submission
  public:
	ffSendJob(ffFile *f, ffDest *d, ffSettings *s)
	: ffJobBase(SubmitFax, f, d, s) { };
};

class ffSendStatusJob : public ffJobBase {
  public:
	ffSendStatusJob(ffSettings *s)
	: ffJobBase(SendStatus, NULL, NULL, s) { };
};

class ffRecvStatusJob : public ffJobBase {
  public:
	ffRecvStatusJob(ffSettings *s)
	: ffJobBase(RecvStatus, NULL, NULL, s) { };
};
/////////////////////////////////////////////////////////////////////////////
// Copy of tcForm (handles dialogue centering).
//
class tcForm : public zFormDialog {
  public:
	tcForm(zWindow *w, zResId r) : zFormDialog(w, r) {
		centreForm();
	};
	int centreForm();
};


/////////////////////////////////////////////////////////////////////////////
// ffJobDialogue.  Class used by a ffSocket for displaying status info.
// as it progresses.  Also holds the CANCEL button so the user can abort
// it all.
//
// TODO: Design virtual methods so this can be used as a base class
//

class ffJobDialogue : public tcForm, public tcLogger {
  protected:
	zStaticText		*status[3];
	zDefPushButton	*canButton;
	ffSocket		*faxSock;
	ffJobBase		*job;
	FILE			*fp_log;

  public:
	ffJobDialogue(zWindow *parent, ffJobBase *j);
	~ffJobDialogue();
	void		setStatus(int line, const char *txt) {
		status[line]->text(txt);
	}
	int				endCancel(zEvent *);
	int				sendAborted(const char *reason);		// ffSocket callbacks
	int				sendComplete();
	// tcLogger abstract function override:-
	void	log(tcLogLevel lev, const char *str);
};

/////////////////////////////////////////////////////////////////////////////
// ffSocket.  The socket class which does all the work.  Created with
// an ffJob to submit and a ffJobDialogue to display status on, it
// connects itself to the FlexFax daemon on the host from the job settings,
// and then uses a state machine driven by asynch. events from the winsock
// DLL to transmit the job to the server.
//
//

//
// Socket class for querying server and queue status

class ffSocket : public tcLineBufSocket, public zEvH {
  protected:
	enum ffState {				// States we can be in
		INIT,					// Initialisation
		CONNECT,				// Just connected
		PERMWAIT,				// Waiting for reply to checkPerm
		FILESEND,				// Sending data files
		DESTSEND,				// Sending destinations
		COVERPROLOG,			// Send PostScript prologue for cover
		COVERTEMPLATE,			// Send template file
		JOBIDWAIT,				// Waiting for job ID from server
		STATRECV,				// Receiving status info
		FINISHED,
	};
	ffState			state;		// Current state
	ffJobDialogue	*dlg;		// Parent dialogue
	ffJobBase		*job;		// Job being submitted

	int				nPages;		// Number of pages in fax

	// Timer for connection timeouts
	zTimer			*connTimer;
	int				connTickCount;

	// Variables used whilst transferring files
	int				fd_file;	// File descriptor to current file
	int				currfile;	// Number of current file in job
	unsigned long	file_size;	// Size of file being sent
	unsigned long	file_count;	// Number of bytes sent
	unsigned long	file_msgcnt;// Count when last message was printed

	// Variables used when transferring destinations
	int				currdest;	// Current destination
	const ffDest	*dp;		// ffDest corresponding to currdest

	// Variables used for receiving status info from the server
	int				nServLine;	// Number of server lines printed
	int				nSendLine;	// Number of send lines printed
	int				nRecvLine;	// Number of recv lines printed

	// Variables used for sending cover sheets
	static char		*coverPLine[];	// Cover sheet prologue lines
	char			**ccovLine;		// Current line being sent
	FILE			*fp_cov;		// Cover template file pointer

  public:
	ffSocket(ffJobDialogue *d, ffJobBase *j);
	~ffSocket()  {
		if (connTimer != NULL) {
			delete connTimer;
			connTimer = NULL;
        }
	};
	
	void	lineRead();					// tcSocket callbacks
	void	dataWritten();
	void	Connected(int error);
	void	Closed(int error);
	void	connTimeout();				// Called if the connection times out

	void	writeLine(const char *tag, const char *data);
	void	writeLine(const char *tag, long value);

	void	Abort();				// Can be called from parent

	void	giveUp(const char *reason);		// Called internally
	void	badResponse(const char *resp);	// Unexpected response
	
	int		isTag(char *buf, const char *match);
	int		isTagData(const char *buf, const char *match);
	const char	*tagData(const char *buf);

	void	startFiles();					// Start sending files
	void	fileSend();						// Send more file data
	void	nextFile();						// Go on to next file

	void	startDest();					// Start sending destinations
	void	destSend();						// Send another destination

	void	startCover();					// Start sending cover sheet
	void	coverSendPrologue();			// Send cover postscript prologue
	void	coverSendTemplate();			// Send template file
	void	coverSendLine(char *line);		// Expand & send a cover line
	void	coverSendComments();			// Split & send comment def'ns
	char	*coverPrintf(char *buf, char *fmt, ...);	// Format cover
	void	coverDef(char *var, char *contents);	// Send a def'n

	void	printServStat(char *s);			// Server status line
	void	printSendStat(char *s);			// Print send status line
	void	printRecvStat(char *s);			// Print recv status line

//	void	printStatus(const char *txt) {	// Put status msgs in dialogue
//		dlg->setStatus(1, txt);
//	}
};

