// Copyright 1993 by Peter Bentley <pete@tecc.co.uk>
//
// Permission to use, copy, modify, distribute, and sell this software and its
// documentation for any purpose is hereby granted without fee, provided that
// the above copyright notice appear in all copies and that both that
// copyright notice and this permission notice appear in supporting
// documentation, and that the name of Peter Bentley not be used in
// advertising or publicity pertaining to distribution of the software without
// specific, written prior permission.  Peter Bentley makes no representations
// about the suitability of this software for any purpose.  It is provided
// "as is" without express or implied warranty.
//
// PETER BENTLEY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
// INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
// EVENT SHALL PETER BENTLEY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
// CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
// DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
// TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
// PERFORMANCE OF THIS SOFTWARE.
//
#include <zapp.hpp>
#include <string.h>
#include <stdarg.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#include "pstring.h"					// Inline string functions
#include "winsock.h"
#include "tcSock.h"
#include "ffClient.h"					// FlexFax client classes
#pragma hdrstop
#include "WinFlex.h"

static char RCSid[] = "$Id: ffclient.cpp,v 1.1 1993/09/02 12:33:58 pete Exp pete $";

/////////////////////////////////////////////////////////////////////////////
// ffFile functions
//
int ffPsFile::numPages() const {
	char buffer[1024];

	FILE *fp = fopen(name(), "r");
	if (fp == NULL) return 0;
    char *s;
	int npagecom = 0;							// %%Page comments
	int npages = 0;								// Value from %%Pages'
	if (fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
		char *s = buffer;
		while (*s == '\x04') s++;				// Skip ^D characters
		if ((s[0] == '%') && (s[1] == '!')) {	// Structured PostScript?
			while (fgets(buffer, sizeof(buffer) - 1, fp) != NULL) {
				int n;
				if (!strncmp(buffer, "%%Page:", 7)) {
					npagecom++;
				}
				else if (sscanf(buffer, "%%%%Pages: %u", &n) == 1) {
					npages += n;
				}
			} 
        }
		fclose(fp);
	}
	return (npages > 0) ? npages : npagecom;
}

/////////////////////////////////////////////////////////////////////////////
// ffJobBase member functions (simple constructor & destructor)

ffJobBase::ffJobBase(ffJobType ty, ffFile *f, ffDest *d, ffSettings *s) {

	type = ty;
	nFile = 0;
	nDest = 0;

	file[0] = f;
	if (file[0] != NULL) nFile = 1;
	dest[0] = d;
	if (dest[0] != NULL) nDest = 1;
	settings = s;
}

ffJobBase::~ffJobBase() {
	for (int i = 0; i < nFile; i++) {
		if (file[i] != NULL) delete file[i];
	}
	for (i = 0; i < nDest; i++) {
		if (dest[i] != NULL) delete dest[i];
	}
}

/////////////////////////////////////////////////////////////////////////////
// ffSocket member functions
//
/////////////////////////////////////////////////////////////////////////////
// Constructor & Destructor
//


ffSocket::ffSocket(ffJobDialogue *d, ffJobBase *j)
: tcLineBufSocket(d, 4096, 4096) {

	dlg = d;
	job = j;
	state = INIT;
	
	connTimer = new zTimer(this, (TimerProc) &ffSocket::connTimeout,
									1000L);		// 1 second timer
	connTickCount = 0;							// No time elapsed yet

	logger->logf(TC_INFO, "Connecting to '%s'", job->host());
	if (Connect(job->host(), "fax") == SOCKET_ERROR) {
		logger->logf(TC_WARNING, "Connect(\"%s\", \"fax\") failed.",
					job->host());
		logger->logf(TC_INFO, "Trying Connect(\"%s\", 4557)",
					job->host());
		if (Connect(job->host(), 4557) == SOCKET_ERROR) {
			logger->logf(TC_ERROR, "Connection failed: %s", lastErrorMessage());
			dlg->sendAborted("Connection Failed");
		}
    }
}

/////////////////////////////////////////////////////////////////////////////
// Members called from the tcSocket classes
//
/////////////////////////////////////////////////////////////////////////////
// Closed().  Called when the remote end closes the TCP connection
//

void ffSocket::Closed(int error) {

	if (job->isSubmit()) {		// Error.  We should have closed first
		logger->logf(TC_ERROR, "Remote closed connection: %s",
					sockErrMessage(error));
		dlg->sendAborted("Remote closed connection");
	}
	else {						// For status, we wait for EOF
		dlg->sendComplete();
	}
}

/////////////////////////////////////////////////////////////////////////////
// Connected().  Called when connect() call completes
//

void ffSocket::Connected(int error) {

	if (connTimer != NULL) {
		delete connTimer;
		connTimer = NULL;
	}
	logger->logf(TC_TRACE, "Socket connected.  Current error code: %d", error);
	if (error == 0) {					// Successful connection
		state = CONNECT;				// Next state
		logger->logf(TC_STATE, "Connected to server");
		logger->logf(TC_INFO, "Sending version information");
		delMask(FD_CONNECT);			// No more connect messages
		addMask(FD_READ);				// But get all read messages
		addMask(FD_CLOSE);				// And suss if the server closes

		writeLine("version", 1);		// Send initial stuff to server
		writeLine("userID", job->user());

		if (job->isSubmit()) {			// Get permission to send file
			writeLine("checkPerm", "send");
		}
		else if (job->isSendStatus()) {
			writeLine("serverStatus:", "");
			writeLine("allStatus", "");
			tcLineBufSocket::writeLine(".");		// Send end of job string
		}
		else if (job->isRecvStatus()) {
			writeLine("serverStatus:", "");
			writeLine("recvStatus", "");
			tcLineBufSocket::writeLine(".");		// Send end of job string
		}
		else {
			logger->logf(TC_ERROR, "Don't know whether to come or go...");
		}
		// dataWritten() callback will be called when all data sent
	}
	else {								// Error during connect
		logger->logf(TC_ERROR, "Connection error: %d", error);
		dlg->sendAborted("Connection failed");
	}
}

void ffSocket::connTimeout() {

	if (connTimer == NULL) return;
    ++connTickCount;
	logger->logf(TC_TRACE, "Time: %d", connTickCount);

	if (connTickCount > 10) {				// 10 seconds elapsed?
		if (state == INIT) {				// Not connected yet
			giveUp("Connection timed out");
		}
		delete connTimer;
		connTimer = NULL;
	}
}
/////////////////////////////////////////////////////////////////////////////
// lineRead().  Called when a complete line of text has been received
// from the server

void ffSocket::lineRead() {		// Data available
	char buffer[255];
	int len = readLine(buffer, 255);		// Copy it out

	if (len == -1) {						// Cock up
		giveUp("Internal read error (lineRead())");
		return;
	}
	char *s = strchr(buffer, '\r');			// Nuke any CRs
	if (s != NULL) *s = '\0';
//	statusLine("Read line %d '%s'", strlen(buffer), buffer);

	switch (state) {
	  case PERMWAIT:						// We're waiting for permission
		if (isTag(buffer, "permission")) {
			if (!isTagData(buffer, "granted")) {
				giveUp("Access denied by server");
			}
			else {
				logger->logf(TC_INFO, "Permission granted by server");
				startFiles();			// Start sending the data
			}
		}
		else {
			badResponse(buffer);
		}
		break;

	  case JOBIDWAIT:					// We're waiting for the job ID
		if (isTag(buffer, "job")) {
			logger->logf(TC_INFO, "Request ID is %s", tagData(buffer));
			dlg->sendComplete();		// Ta Da!
			Close();					// Close the socket
		}
		else if (isTag(buffer, "error")) {
			logger->logf(TC_INFO, "%s", tagData(buffer));
		}
		else {
			badResponse(buffer);
		}
		break;

	  case STATRECV:					// Status info
		if (isTag(buffer, "server")) printServStat((char *)tagData(buffer));
		else if(isTag(buffer, "jobStatus")) printSendStat((char *)tagData(buffer)); 
        else if (isTag(buffer, "recvJob")) printRecvStat((char *)tagData(buffer));
        else badResponse(buffer);
		break;
		
	  default:							// We're not expecting data
		logger->logf(TC_ERROR, "Data read in invalid state");
		dlg->sendAborted("Protocol Error");
		return;
	}
}

/////////////////////////////////////////////////////////////////////////////
// dataWritten().  Called by the socket classes when all queued data has
// been sent.  Switch on the current state to decide what to send next.
//

void ffSocket::dataWritten() {

	switch (state) {
	  case CONNECT:				// Initial negotiation send.  Wait for reply
		if (job->isSubmit()) {	// Send request for permission
			state = PERMWAIT;
			logger->logf(TC_STATE,
						 "Waiting for permission from server to proceed");
		}
		else {					// Assume this is a status (TODO: Fix this)
			state = STATRECV;
			logger->logf(TC_STATE, "Waiting for data from server");
			nServLine = nSendLine = nRecvLine = 0;	// Clear stats
		}
		break;

	  case FILESEND:			// Sending files.  Try and send some more
		fileSend();
		break;

	  case DESTSEND:			// Sending destinations.  Send some more
		destSend();
		break;

	  case COVERPROLOG:			// Sending cover page prolog
		coverSendPrologue();
		break;
		
	  case COVERTEMPLATE:		// Sending cover page template file
		coverSendTemplate();
		break;

	  case JOBIDWAIT:			// Send end of job. Wait for ID
		logger->logf(TC_INFO, "Waiting for job ID from server");
		break;

	  default:					// Unexpected!
		logger->logf(TC_ERROR, "dataWritten() called in invalid state %d",
					 state);
		dlg->sendAborted("Protocol Error");
		break;
	}
}


/////////////////////////////////////////////////////////////////////////////
// giveUp().  Abort the connection and print the reason
//

void ffSocket::giveUp(const char *reason) {

	logger->logf(TC_ERROR, "Giving up: %s", reason);
	Close();								// Close the socket
	if (fd_file >= 0) close(fd_file);		// Close any file we have open
	dlg->sendAborted(reason);				// Tell the status dialogue
}

/////////////////////////////////////////////////////////////////////////////
// badResponse().  Give up because we don't understand the server
//
void ffSocket::badResponse(const char *response) {

	logger->logf(TC_ERROR, response);
	giveUp("Unexpected response from server");
}

/////////////////////////////////////////////////////////////////////////////
// Functions for ripping apart/comparing the tag:value strings
// from the server

int ffSocket::isTag(char *buf, const char *match) {

	char *s = strchr(buf, ':');
	if (s == NULL) return 0;				// Can't be...
	*s = '\0';								// Vape the colon
	int found = !stricmp(buf, match);	// Compare strings
	*s = ':';								// Restore colon
	return found;
}

int ffSocket::isTagData(const char *buf, const char *match) {

	char *s = strchr(buf, ':');
	if (s == NULL) return 0;				// Can't be...
	return !stricmp(s + 1, match);
}

const char *ffSocket::tagData(const char *buf) {

	char *s = strchr(buf, ':');
	if (s == NULL) return "<Invalid>";
	return (s + 1);
}

/////////////////////////////////////////////////////////////////////////////
// File sending functions
//
/////////////////////////////////////////////////////////////////////////////
// startFiles().
// Move into file sending state, kick-start the first file
//

void ffSocket::startFiles() {

	state = FILESEND;
	logger->logf(TC_STATE, "Sending data files");
	currfile = -1;					// Make current file invalid
	fd_file = -1;
    nPages = 0;
	fileSend();						// Send some data
}

/////////////////////////////////////////////////////////////////////////////
// nextFile().
// Start sending the next file, or move onto the next state (destinations)
// NB assumes we are sending the file byte-for-byte the same as it is
// held on the DOS box.  This means that PostScript files will still
// contain CRs when they reach the server, but that shouldn't be a
// problem for any sane PostScript implementation.
//

void ffSocket::nextFile() {

	currfile++;						// Next file
	logger->logf(TC_INFO, "Sending file %d", currfile + 1);
	const ffFile *f = job->getFile(currfile);
	if (f == NULL) {
		logger->logf(TC_INFO, "All files sent");
		startDest();				// Start sending destinations
	}
	else {
		struct stat sbuf;

		nPages += f->numPages();	// Update number of pages sent

		logger->logf(TC_DEBUG, "Open file for send: \"%s\"", f->name());
		fd_file = open(f->name(), O_RDONLY | O_BINARY);
		if (fd_file < 0) {
			zMessage(dlg, "Unable to open file", "Error", MB_ICONHAND);
			giveUp("File open error");
		}
		else if (fstat(fd_file, &sbuf) < 0) {
			zMessage(dlg, "Unable to stat file", "Error", MB_ICONHAND);
			giveUp("File stat() error");
		}
		else {
			file_size = sbuf.st_size;
			file_count = 0L;
			file_msgcnt = 0L;
			writeLine(f->typeName(), file_size);
		}
	}
}   

/////////////////////////////////////////////////////////////////////////////
// fileSend().  Send some more file data.  If there is no file currently
// open, try and move on to the next file.  Otherwise read some data
// from the file and queue it up on the socket.  If we reach EOF then
// close the file and try to move onto the next one.
//

void ffSocket::fileSend() {

	if (fd_file == -1) {			// In between files
		nextFile();
		return;						// dataWritten will call us back
	}
	char buffer[2048];				// Send more of current file
	logger->logf(TC_DEBUG, "Sending more file data");
    while (wrSpace() > 1512) {
		int amt = read(fd_file, buffer, 1512);	// Ethernet MTU...
		if (amt < 0) {					// Error reading file
			logger->logf(TC_ERROR, "File read error: %s", strerror(errno));
			giveUp("File read error");
            break;
		}
		else if (amt == 0) {			// End of file
			close(fd_file);
			fd_file = -1;
			nextFile();					// Go on to next file
            break;
		}
		else if (send(buffer, amt) == SOCKET_ERROR) {	// Socket write problem
			giveUp("Socket write error");
			logger->logf(TC_INFO, "Socker error: %s", lastErrorMessage());
			break;
		}

		// Status output so the user knows what is going on
		file_count += amt;
		if ((file_msgcnt == 0) || ((file_count - file_msgcnt) > 10240)) {
			logger->logf(TC_INFO, "Sent %lu of %lu", file_count, file_size);
			file_msgcnt = file_count;
		}
	}
	logger->logf(TC_DEBUG, "End of file transmit loop");
}

/////////////////////////////////////////////////////////////////////////////
// Destination sending functions
//
/////////////////////////////////////////////////////////////////////////////
// startDest().  Start sending destinations.  Move into the correct state,
// set the current destination to no. 0, then use destSend to start
// sending.

void ffSocket::startDest() {
	state = DESTSEND;
	currdest = 0;					// Next destination to send
	logger->logf(TC_STATE, "Sending Fax recipients");
	destSend();
}

/////////////////////////////////////////////////////////////////////////////
// destSend().  Send another destination.
//

void ffSocket::destSend() {

	dp = job->getDest(currdest);				// Set up pointer
	if (dp == NULL) {							// No more destinations
		logger->logf(TC_INFO, "All destinations transmitted");
		tcLineBufSocket::writeLine(".");		// Send end of job string
		logger->logf(TC_INFO, "Sent end of job marker");
		state = JOBIDWAIT;						// Wait for job ID
		logger->logf(TC_STATE, "Waiting for job ID");
		return;
	}

	// Write the destination tags.  This needs updating *a lot*.
	writeLine("begin", currdest);
	writeLine("killtime", (zString) dp->killTime);
	writeLine("number", (zString) dp->teleNum);
	writeLine("sender", job->sender());
	writeLine("mailaddr", job->mailaddr());
	writeLine("resolution", dp->vRes);
	writeLine("pagewidth", dp->pWidth);
	writeLine("pagelength", dp->pLen);
	if (dp->doCover) {
		startCover();
	}
	else {
		writeLine("end", currdest++);
		logger->logf(TC_INFO, "Recipient number %d", currdest);
	}
}

void ffSocket::startCover() {
	state = COVERPROLOG;
	logger->logf(TC_STATE, "Sending cover sheet prologue");
	if ((fp_cov = fopen(job->coverTemplate(), "r")) == NULL) {
		giveUp("Unable to open cover sheet template");
		return;
	}
	ccovLine = coverPLine;
	writeLine("cover", 1);		// Cover sub-protocol 1
	coverSendPrologue();
}

char *ffSocket::coverPLine[] = {
	"%!PS-Adobe-2.0 EPSF-2.0",
	"%%Creator: WinFlex",
	"%%Title: WinFlex Cover Sheet for FlexFax",
	"%%Creation Date: @T",
	"%%Origin: 0 0",
	"%%BoundingBox: 0 0 @W @L",
	"%%Pages: 1 +1",
	"%%EndComments",
	"%%Begin Prolog",
	"100 dict begin",
	"/to (@t) def",
	"/to-company (@c) def",
	"/to-location (@a) def",
	"/to-voice-number (@v) def",
	"/to-fax-number (@n) def",
	"/pageWidth @w def",
	"/pageWidth @l def",
	"/from (@f) def",
	"/from-fax-number (@N) def",
	"/from-voice-number (@V) def",
	"/from-company (@O) def",
	"/from-location (@A) def",
	"/page-count (@p) def",
	"/todays-date (@T) def",
	"/regarding (@r) def",
	"@C",								// All of comments
	"%%EndProlog",
	"%%Page \"1\" 1",
	NULL,
};

void ffSocket::coverSendPrologue() {

	while (*ccovLine != NULL) {
		if (wrSpace() < 1000) return;	// Allow loads of room
		coverSendLine(*ccovLine);	// Write a line of data
		ccovLine++;					// Move on to the next one
    }
	state = COVERTEMPLATE;				// Next state
	logger->logf(TC_STATE, "Sending cover sheet body");
	coverSendTemplate();
}

char *ffSocket::coverPrintf(char *buf, char *fmt, ...) {
	va_list args;
	char	tmp[2048];

	va_start(args, fmt);
	vsprintf(tmp, fmt, args);
	va_end(args);

	for (char *s = tmp; *s != '\0'; s++) {
		if ((*s == '(') || (*s == ')')) {
        	*buf++ = '\\';					// Escape any ( )'s
		}
        *buf++ = *s;
	}
	return buf;
}

void ffSocket::coverSendLine(char *line) {
	char buffer[2048], *bp = buffer;

	for (; *line != '\0'; line++) {
		if (*line != '@') {
			*bp++ = *line;
			continue;
		}
		line++;						// Skip over the '@'
		switch (*line) {
		  case '@':				// Escaped @
			*bp++ = '@';
			break;
		  case 'T':				// Date & time
			{
            	time_t now;
				time(&now);
				char *s = ctime(&now);
				s[strlen(s) - 1] = '\0';	// Zap the \n
                bp = coverPrintf(bp, "%s", s);
            }
			break;
		  case 'W':				// Width of page in points
          	bp = coverPrintf(bp, "%.0f", (dp->pWidth / 25.4) * 72);  
			break;
		  case 'L':				// Length of page in points
          	bp = coverPrintf(bp, "%.0f", (dp->pLen / 25.4) * 72);
			break;
		  case 't':				// To person
			bp = coverPrintf(bp, "%s", (char *) dp->recipName);
			break;
		  case 'c':				// To company
			bp = coverPrintf(bp, "%s", (char *) dp->recipCompany);
			break;
		  case 'a':				// To location
			break;
		  case 'v':				// To voice number
			break;
		  case 'n':				// To fax number
			bp = coverPrintf(bp, "%s", (char *) dp->teleNum);
			break;
		  case 'w':				// Page width in mm
			bp = coverPrintf(bp, "%lu", (char *) dp->pWidth);
			break;
		  case 'l':				// Page length in mm
			bp = coverPrintf(bp, "%lu", (char *) dp->pLen);
			break;
		  case 'f':				// From person
			bp = coverPrintf(bp, "%s", job->sender());
			break;
		  case 'N':				// From fax number
			break;
		  case 'V':				// From voice number
			break;
		  case 'O':				// From company
			break;
		  case 'A':				// From location
			break;
		  case 'p':				// Page count
			bp = coverPrintf(bp, "%d", nPages);
			break;
		  case 'r':				// Regarding
			bp = coverPrintf(bp, "%s", (char *) dp->regarding);
			break;
		  case 'C':				// Comments defininitions
			coverSendComments();
			break;
		}
	}
	*bp = '\0';
//	statusLine("Cover line '%s'", buffer);
	writeLine("!", buffer);
}

void ffSocket::coverDef(char *var, char *contents) {
	char buffer[2048];
	sprintf(buffer, "/%s (", var);
	char *s = buffer + strlen(buffer);
	for (; *contents != '\0'; contents++) {
		if ((*contents == '(') || (*contents == ')')) {
        	*s++ = '\\';					// Escape any ( )'s
		}
		*s++ = *contents;
	}
	*s = '\0';
	strcat(s, ") def");
    writeLine("!", buffer);
}

void ffSocket::coverSendComments() {
	char tag[32], value[64];

    coverDef("comments", dp->comments);
	char *s = dp->comments;
	int number = 1;
	char *spacep;
	while ((spacep = strchr(s, ' ')) != NULL) {
		int wordlen = (spacep - s) + 1;
		if ((strlen(value) + wordlen) > 35) {		// Flush existing comments
			sprintf(tag, "comment%d", number++);
			coverDef(tag, value);
			strpncpy(value, s, wordlen);
		}
		else {
        	strncat(value, s, wordlen);
		}
		s += wordlen;  
	}
	// Last line
	sprintf(tag, "comment%d", number++);
	coverDef(tag, value);
	// Last word
	sprintf(tag, "comment%d", number++);
	coverDef(tag, s);

}

void ffSocket::coverSendTemplate() {
	char buffer[2048];

    while (wrSpace() > 1000) {						// While we have room
		buffer[0] = '!';
	    buffer[1] = ':';
		if (fgets(buffer + 2, 2047, fp_cov) == NULL) {	// End of file
			fclose(fp_cov);								// Close cover template
			fp_cov = NULL;
			tcLineBufSocket::writeLine("..");			// End of PS file
			writeLine("end", currdest++);				// End of this dest
			state = DESTSEND;							// Next destination
			logger->logf(TC_STATE, "Sending destinations");
			return;
		}
		char *s = strchr(buffer, '\n');					// Kill the NL
		if (s != NULL) *s = '\0';

		tcLineBufSocket::writeLine(buffer);				// Write another line
	}
}


/////////////////////////////////////////////////////////////////////////////
// writeLine overloads for ease of use (and ease of cribbing Sam Leffler's
// code. :-)
//

void ffSocket::writeLine(const char *tag, const char *data) {
	char line[512];

	sprintf(line, "%s:%s", tag, data);
//	logger->logf(TC_INFO, "Send '%s'", line);
	tcLineBufSocket::writeLine(line);
}

void ffSocket::writeLine(const char *tag, long val) {
	char num[16];

	sprintf(num, "%lu", val);
	writeLine(tag, num);
}

void ffSocket::printServStat(char *buffer) {

    char *number, *modem, *status;
	char *s = strchr(buffer, ':');
	if (s != NULL) {
		number = buffer;
		*s++ = '\0';
		char *t = strchr(s, ':');
		if (t == NULL) {
			modem = "";
            status = s;
		}
		else {
			modem = s;
			*t++ = '\0';
            status = t;
		}    	
	}
	else {
		number = buffer;
		modem = "";
		status = "Running";
	}
	nServLine++;
	statusLine("Server on %s:%s for %s: %s", job->host(), modem, number,
				status);
}

void ffSocket::printSendStat(char *buffer) {
	if ((nSendLine == 0) && (nServLine == 0)) {
		statusLine("WARNING: No servers running on host %s.", job->host());
	}
	char *jobname = strtok(buffer, ":");
	char *sender = strtok(NULL, ":");
	char *tts = strtok(NULL, ":");
	char *number = strtok(NULL, ":");
	char *modem = strtok(NULL, ":");
	char *status = strtok(NULL, ":");

	// Cross our fingers that none of those are NULL!

	if (status == NULL) {
		status = (strcmp(tts, "locked")) ? "Waiting" : "Being processed";
	}

	char *fmt = "%-7s %-16s %-10s %-16s %-10s %s";
	if (nSendLine++ == 0) {
    	statusLine(fmt, "Job", "Sender", "Time", "Number", "Modem", "Status");
	}
	statusLine(fmt, jobname, sender, tts, number, modem, status);
}

void ffSocket::printRecvStat(char *buffer) {

	if ((nRecvLine++ == 0) && (nServLine == 0)) {
		statusLine("WARNING: No servers running on host %s.", job->host());
	}
	statusLine("RECV: %s", buffer);
}


/////////////////////////////////////////////////////////////////////////////
// ffJobDialogue member functions.  Update the status box.
//

ffJobDialogue::ffJobDialogue(zWindow *parent, ffJobBase *j)
:tcForm(parent, zResId("WinFlex_Status"))
{
	status[0] = new zStaticText(this, IDD_STATUS1);
	status[1] = new zStaticText(this, IDD_STATUS2);
	status[2] = new zStaticText(this, IDD_STATUS3);
	canButton = new zDefPushButton(this, IDCANCEL);	// Find cancel button

	setStatus(0, "");						// Clear status lines
	setStatus(1, "");
	setStatus(2, "");
	show();									// Get everything on screen

    fp_log = NULL;
	job = j;
	if (job->getSettings()->logToFile) {
		fp_log = fopen((char *) job->getSettings()->logFile, "w");
		if (fp_log == NULL)
			zMessage msg(parent, "WARNING: Unable to open log file", "Error",
						MB_ICONEXCLAMATION);
	}

	setStatus(0, "Creating Socket...");
	faxSock = new ffSocket(this, job);		// Kick socket off
	setStatus(0, "Contacting Fax server...");
	
	modal();								// Make dialogue modal
}

ffJobDialogue::~ffJobDialogue() {

	for (int i = 0; i < 3; i++) {			// Delete attached controls
		if (status[i] != NULL) delete status[i];
	}
	if (canButton != NULL) delete canButton;	
	if (faxSock != NULL) delete faxSock;
	if (fp_log != NULL) fclose(fp_log);
}

/////////////////////////////////////////////////////////////////////////////
//  log: Output a log message
//

void ffJobDialogue::log(tcLogLevel lev, const char *str) {
	int toFile = FALSE;

	switch (lev) {
		case TC_DEBUG:
			toFile = job->getSettings()->logDebug;
			break;			// Ignore

		case TC_TRACE:
			setStatus(2, str);
			toFile = job->getSettings()->logTrace;
			break;

		case TC_OPER:
			setStatus(2, str);
			toFile = job->getSettings()->logOper;
			break;

		case TC_INFO:
			setStatus(1, str);
			setStatus(2, "");
			toFile = job->getSettings()->logInfo;
           break;

		case TC_STATE:
			setStatus(0, str);
			setStatus(1, "");
			setStatus(2, "");
			toFile = job->getSettings()->logState;
			break;

		case TC_WARNING:
			statusLine("%s: %s", logLevelName[lev], str);
			toFile = job->getSettings()->logWarn;
			break;

		case TC_ERROR:
			statusLine("%s: %s", logLevelName[lev], str);
			toFile = job->getSettings()->logError;
			break;

		default:
			statusLine("%s: %s", logLevelName[lev], str);
			toFile = TRUE;
			break;
	}
	if (toFile && (fp_log != NULL)) {
		fprintf(fp_log, "%s: %s\n", logLevelName[lev], str);
		fflush(fp_log);
	}
}
/////////////////////////////////////////////////////////////////////////////
// endCancel.  Called when the cancel button is pressed.
//

int ffJobDialogue::endCancel(zEvent *) {
	setStatus(0, "Cancelled");
	faxSock->giveUp("Cancel button pressed");
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// sendComplete().  Called by the ffSocket when the job has been queued

int ffJobDialogue::sendComplete() {

	if (job->isSubmit()) {
		setStatus(0, "Fax Successfully Queued");
		canButton->text("OK");
	}
	else {
		shutdown();			// Go away quietly...
	}
    return 1;
}

/////////////////////////////////////////////////////////////////////////////
// sendAborted(). Called by the ffSocket if it aborts the job

int ffJobDialogue::sendAborted(const char *reason) {

	setStatus(0, "Transaction Aborted");
	setStatus(1, reason);
	setStatus(2, "");
	canButton->text("OK");
	faxSock->Close();				// Nuke the socket
	return 1;
}

/////////////////////////////////////////////////////////////////////////////
// tcForm centre method.  TODO: Needs a proper home...
//
int tcForm::centreForm()
{
	zPane *dummyPane = new zPane(this, new zSizer());
	zDisplay *dummyCanvas = dummyPane->canvas();

	zDisplayInfo *dispInfo = new zDisplayInfo(dummyCanvas);

	zSizer *Form = sizer();

	int top = (dispInfo->pixHeight() - Form->height())/2;
	int left = (dispInfo->pixWidth() - Form->width())/2;

	if (top < 0) top = 0;
	if (left < 0) left = 0;

	move(left, top, Form->width(), Form->height());

	delete dummyPane;
	delete dispInfo;

	return 1;
}
