# jlibrary.tcl - utility procedures
# 
# Copyright 1992-1994 by Jay Sekora.  All rights reserved, except 
# that this file may be freely redistributed in whole or in part 
# for nonprofit, noncommercial use.
# 
# 
# these procedures are required by
#     browser.tk
#     edit.tk
#     help.tk
#     more.tk
#     people.tk
#     prefs.tk
# they may be located in the file "~/.tk/jlibrary.tcl" (where they will
# be source'd by those applications on startup), or in the site-wide
# tk library directory, where they will be found (and loaded) by the
# default tk  unknown  procedure.
######################################################################

### TO DO
###   j:parse_args -boolean (and maybe -position?)
###   j:fs
###     more error-checking in j:fs
###     fix focus on j:fs
###     option for load/save?
###     mkdir when saving?
###     `default' behaviour needs fixed (do we even need a default now?)
###   documentation
###   improve find panel
###     have find wrap around (if last time didn't match)
###     regex search/replace
###     `find' tags instead of selection (set of buttons)
###     rewrite find routines to pass options instead of using globals
###       e.g. j:find:find_string -case 0 -regex 1 \
###              -backwards 0 -tag found {foo$} .main.t
###   j:buttonbar:button to add a button to the buttonbar
###   j:more:button to add a button to the buttonbar
###   MAKE DIALOGUES NON-MODAL

######################################################################
# j:parse_args arglist - parse arglist in parent procedure
#   arglist is a list of option names (without leading "-");
# j:selection_if_any - return selection if it exists, else {}
# j:no_selection - true if there is no selection
# j:source_config ?options? file - read user configuration from a file
#   option is -directory
# j:read_prefs ?options? defaults - read X defaults from file, set array
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default PREFS)
# <defaults> is a list of two-element sublists.
# j:read_standard_prefs - read standard defaults from ~/.tk/defaults
# j:write_prefs ?options? - write X defaults to file from array
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default PREFS)
# j:default_button button widget... - bind <Return> to default button
# j:cancel_button button widget... - set up bindings for cancel button
# j:tab_ring widget... - bind Tab and Shift-Tab to cycle through widgets
# j:dialogue w - arrange to position window w near ctr of screen
# j:longest_match l - longest common initial string in list l
# j:expand_filename f - expand filename prefix as much as possible
# j:rule parent [args] - returns a rule suitable for parent
# j:filler parent [args] - returns a filler frame suitable for parent
# j:buttonbar w ?options? - make a buttonbar packed in w
#   -default (default none)
#   -padx (default 10)
#   -pady (default 5)
#   -orient (default horizontal)
#   -buttons (default {})
# syntax of button list is {{name text command} ... }
# j:global_pref_panel ?options? - preferences panel for ~/.tk/defaults
#   -title - title of toplevel window
### ADD COLOURS?
# j:colour_chooser w ?options? - create a metawidget for selecting colors
#   -label (default "Colour:")
#   -variable (global variable to set - not really optional)
# j:variable_entry w ?options? - labelled entry for global variable
#   -label (default "Value:")
#   -variable (default value - not really optional)
#   -labelwidth (default 16)
#   -entrywidth (default 40)
#   -promptconfig (default "")
#   -labelconfig (default "")
# j:alert ?options? - alert box
#   -title (default "Alert")
#   -text (default "Alert!" - not really optional)
# j:confirm ?options? - Cancel/OK dialogue box
#   -title (default "Confirm")
#   -text (default "Are you sure?")
#   -priority (default 0)
#   -yesbutton (default "OK")
#   -nobutton (default "Cancel")
# j:more ?options? ?-title hdr? ?-text txt? - displays text in window
#   -title (default "Output")
#   -text (default "" - not really optional)
#   -height (default 24)
#   -width (default 80)
#   -font (default "default")
#   -class (default "More")
# j:more:save w - prompts to save the content of a j:more window
# j:more:print w - prompts to print the content of a j:more window
# j:more:pageup t - scrolls text widget t up
# j:more:pagedown t - scrolls text widget t down
# j:prompt ?options? - prompt the user for information
#   -text (default "Enter a value:"
#   -default (default "")
#   -cancelvalue (default "")
#   -file (default 0)
#   -title (default "Prompt")
# if $files, then the Tab key will do filename completion
# j:prompt_font ?options? - prompt for a font (via xfontsel)
#   -text (default "Font:", but currently ignored)
#   -pattern (default "*")
# j:prompt_tcl - prompt for a tcl command and execute it
# j:prompt_unix - prompt for a unix command and execute it
# j:prompt_colour_name - prompt for a colour name
### PROBLEM: LOCATION OF /usr/lib/X11/rgb.txt IS HARDCODED!
# j:prompt_colour_rgb - prompt for a colour RGB value
# j:configure_font widget fontlist - use font from list, or default
# j:fs ?options? - file selector box
#   -buttons (default {ok cancel home})
#   -prompt (default "Choose a file")
#   -default (default "")
#   -cancelvalue (default "")
#   -fileprompt (default "File:")
#   -title (default "File Selector")
# j:fs:fill_list lb - fill the listbox with files from CWD
# j:find ?options? t - search-and-replace panel for text widget t
#   -replace (default 1)
# j:find:again t - search again for same string
# j:find:find_string string t - find and select string in text widget t
# j:find:replace string t - replace selection in t with string
# j:find:replace_all from to t - change all from to to in widget t
######################################################################

######################################################################
# global variables:
#
global PREFS env
if {! [info exists PREFS(autoposition)]} {set PREFS(autoposition) 0}
if {! [info exists PREFS(confirm)]} {set PREFS(confirm) 1}
#
######################################################################


######################################################################
# j:parse_args arglist - parse arglist in parent procedure
#   arglist is a list of option names (without leading "-");
# this proc puts their values (if any) into variables (named after
#   the option name) in d parent procedure
# any element of arglist can also be a list consisting of an option
#   name and a default value.
######################################################################

proc j:parse_args {arglist} {
  upvar args args

  foreach pair $arglist {
    set option [lindex $pair 0]
    set default [lindex $pair 1]		;# will be null if not supplied
    set index [lsearch -exact $args "-$option"]
    if {$index != -1} {
      set index1 [expr {$index + 1}]
      set value [lindex $args $index1]
      uplevel 1 [list set $option $value]	;# caller's variable "$option"
      set args [lreplace $args $index $index1]
    } else {
      uplevel 1 [list set $option $default]	;# caller's variable "$option"
    }
  }
}

######################################################################
# j:selection_if_any - return selection if it exists, else {}
#   this is from R. James Noble <kjx@comp.vuw.ac.nz>
######################################################################

proc j:selection_if_any {} {
  if {[catch {selection get} s]} {return ""} {return $s}
}

######################################################################
# j:no_selection - true if there is no selection
######################################################################

proc j:no_selection {} {
  if {[catch {selection get} s]} {return 1} {return 0}
}


######################################################################
# j:source_config ?options? file - read user configuration from a file
#   option is -directory
# file is assumed to be in env(HOME)/.tk unless dir is specified
# NOTE: this can also be used just to source an arbitrary Tcl file
######################################################################

proc j:source_config { args } {
  j:parse_args { {directory {} } }

  set file [lindex $args 0]
  global env

  if {$directory == {}} then {
    set directory $env(HOME)/.tk
  }

  if {[file isfile "$directory/$file"]} then {
    uplevel 1 "source $directory/$file"
  }
}

######################################################################
# j:read_prefs ?options? defaults - read X defaults from file, set array
# options are:
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default PREFS)
# <defaults> is a list of two-element sublists.  the first element of
#   each sublist is the name of the default (in the file and in the
#   $array array); the second is the value to use if no such default
#   exists (ie, the hardwired application default)
# If a _default_ is "tk_strictMotif", it sets the element of $array,
#   but also the global tk_strictMotif variable
######################################################################

proc j:read_prefs { args } {
  j:parse_args {
    {array PREFS}
    {directory {} }
    {file defaults}
  }
  set defaults [lindex $args 0]

  global env tk_strictMotif $array

  if {"x$directory" == "x"} then {
    set directory $env(HOME)/.tk	;# NOTE: created if necessary!
  }

  set [format {%s(0)} $array] 1		;# dummy to make sure it's an array

  catch {option readfile $directory/$file userDefault}

  foreach pair $defaults {
    set pref [lindex $pair 0]
    set hard_default [lindex $pair 1]
    
    set value [option get . $pref {}]
    if {"x$value" == "x"} {set value $hard_default}
    set [format {%s(%s)} $array $pref] $value
    
    if {"x$pref" == "xtk_strictMotif"} {
      set tk_strictMotif $value
    }
  }
}

######################################################################
# j:read_standard_prefs - read standard defaults from ~/.tk/defaults
######################################################################

proc j:read_standard_prefs {} {
  global PREFS
  
  j:read_prefs {
    {autoposition 0}
    {bindings basic}
    {confirm 1}
    {printer lp}
    {scrollbarside right}
    {j_fs_fast 0}
    {tk_strictMotif 0}
  }
}

######################################################################
# j:write_prefs ?options? - write X defaults to file from array
# options are:
#   -file (default defaults)
#   -directory (default ~/.tk)
#   -array (default PREFS)
# writes all elements of array $array
######################################################################

proc j:write_prefs { args } {
  j:parse_args { {array PREFS} {directory {} } {file defaults} }
  global env $array
  
  if {"x$directory" == "x"} then {
    set directory $env(HOME)/.tk	;# NOTE: created if necessary!
  }

  if {! [file isdirectory $directory]} {;# make sure directory exists
    exec mkdir $directory
  }
  set f [open $directory/$file {w}]
  
  foreach pref [lsort [array names $array]] {
    set value [set [format {%s(%s)} $array $pref]]
    puts $f "*${pref}:\t${value}"
  }

  close $f
}

######################################################################
# j:default_button button widget... - bind <Return> to default button
#   widget... is one or more widgets that can have the kbd focus
######################################################################

proc j:default_button { button args } {
  foreach w $args {
    bind $w <Return> "$button invoke"
  }
}

######################################################################
# j:cancel_button button widget... - set up bindings for cancel button
#   widget... is one or more widgets that can have the kbd focus
######################################################################

proc j:cancel_button { button args } {
  foreach w $args {
    bind $w <Control-c> "$button invoke"
    bind $w <Control-g> "$button invoke"
    bind $w <Meta-q> "$button invoke"
    bind $w <Meta-period> "$button invoke"
  }
}

######################################################################
# j:tab_ring widget... - bind Tab and Shift-Tab to cycle through widgets
#  widget... is the list of widgets to bind, in order
######################################################################
### It's unfortunate to have to hardwire Shift-Tab to Backtab, but there
### doesn't seem to be a <Backtab> X11 keysym.

proc j:tab_ring {args} {
  # index of last widget
  set last [expr {[llength $args] - 1}]
  
  for {set i 0} {$i < $last} {incr i} {
    set this [lindex $args $i]
    set next [lindex $args [expr {$i + 1}]]
    bind $this <Tab> "focus $next"
    bind $next <Shift-Tab> "focus $this"
  }
  
  # ... and bind last to focus on first:
  set this [lindex $args $last]
  set next [lindex $args 0]
  bind $this <Tab> "focus $next"
  bind $next <Shift-Tab> "focus $this"
}

######################################################################
# j:dialogue w - arrange to position window w near ctr of screen
#   mostly borrowed from /usr/local/lib/tk/dialog.tcl
# does nothing unless $PREFS(autoposition)
######################################################################

proc j:dialogue { w } {
  global PREFS

  if $PREFS(autoposition) {
    # first, display off-screen:
    wm withdraw $w		;# hide the window

    update idletasks		;# force geometry managers to run
    # calculate position:
    set x [expr [winfo screenwidth $w]/2 - [winfo reqwidth $w]/2 \
	    - [winfo vrootx [winfo parent $w]]]
    set y [expr [winfo screenheight $w]/3 - [winfo reqheight $w]/2 \
	    - [winfo vrooty [winfo parent $w]]]
    wm geom $w +$x+$y
    wm deiconify $w

    update idletasks		;# force geometry managers to run
    wm deiconify $w		;# display window
    wm focus $w
  }
}

proc j:dialog [info args j:dialogue] [info body j:dialogue]

######################################################################
# j:longest_match l - longest common initial string in list l
#   used by tab-expansion in filename dialogue box
######################################################################
# this needs commenting desperately

proc j:longest_match { l } {
  case [llength $l] in {
    {0} { return {} }
    {1} { return [lindex $l 0] }
  }
  set first [lindex $l 0]
  set matchto [expr {[string length $first] - 1}]
  for {set i 1} {$i < [llength $l]} {incr i} {
    set current [lindex $l $i]
    # if they don't match up to matchto, find new matchto
    if { [string compare \
           [string range $first 0 $matchto] \
           [string range $current 0 $matchto]] } {
      # loop, decreasing matchto until the strings match that far
      for {} \
          {[string compare \
              [string range $first 0 $matchto] \
              [string range $current 0 $matchto]] } \
          {incr matchto -1 } \
          {}			;# don't need to do anything in body
    } ;# end if they didn't already match up to matchto
  } ;# end for each element in list
  if {$matchto < 0} then {
    return {}
  } else {
    return [string range $first 0 $matchto]
  }
}

######################################################################
# j:expand_filename f - expand filename prefix as much as possible
#       (for use in file dialogue boxes)
######################################################################
# note: if the filename has *, ?, or [...] in it, they will be used
#       as part of the globbing pattern.  i declare this a feature.

proc j:expand_filename { f } {
  set expansion [j:longest_match [glob -nocomplain "${f}*"]]
  if {$expansion == ""} {return $f}
  # make sure it doesn't already end in "/"
  set expansion [string trimright $expansion "/"]
  if [file isdirectory $expansion] {append expansion "/"}
  return $expansion
}

######################################################################
# j:rule parent [args] - returns a rule suitable for parent
#       used as argument to a pack command
######################################################################

proc j:rule { {parent {}} args} {
  global j_rule

  if {$parent == "."} {set parent ""}	;# so "." doesn't give "..rule0"
  
  if {[info exists j_rule(count)]} then {
    set j_rule(count) [expr {$j_rule(count) + 1}]
  } else {
    set j_rule(count) 0
  }

  set rule "$parent.rule$j_rule(count)"
  frame $rule -height 2 -width 2 -borderwidth 1 -relief sunken
  if {$args != ""} {eval $rule configure $args}
  return $rule
}

######################################################################
# j:filler parent [args] - returns a filler frame suitable for parent
#       used as argument to a pack command
######################################################################

proc j:filler { {parent {}} args} {
  global j_filler

  if {$parent == "."} {set parent ""}	;# so "." doesn't give "..filler0"
  
  if {[info exists j_filler(count)]} then {
    set j_filler(count) [expr {$j_filler(count) + 1}]
  } else {
    set j_filler(count) 0
  }

  set filler "$parent.filler$j_filler(count)"
  frame $filler -height 10 -width 10
  if {$args != ""} {eval $filler configure $args}
  return $filler
}

######################################################################
# j:buttonbar w ?options? - make a buttonbar packed in w
# options are:
#   -default (default none)
#   -padx (default 5)
#   -pady (default 5)
#   -orient (default horizontal)
#   -buttons (default {})
# syntax of button list is {{name text command} ... }
######################################################################

proc j:buttonbar {args} {
  j:parse_args {
    {default "(NONE)"}
    {padx 5}
    {pady 5}
    {orient horizontal}
    buttons
  }

  if {[llength $args] != 1} {
    tkerror {Improper arguments}
  }
  
  set newframe [lindex $args 0]
  
  if {$orient == "horizontal"} {
    set side right				;# for packing
  } else {
    set side bottom				;# for packing
  }

  frame $newframe
  if {$padx} {
    pack [j:filler $newframe -width $padx] -in $newframe -side left
    pack [j:filler $newframe -width $padx] -in $newframe -side right
  }
  if {$pady} {
    pack [j:filler $newframe -height $pady] -in $newframe -side top
    pack [j:filler $newframe -height $pady] -in $newframe -side bottom
  }
  
  foreach i $buttons {
    set name [lindex $i 0]
    set text [lindex $i 1]
    set command [lindex $i 2]

    set width [expr {[string length $text] + 1}]    
    if {$width < 8} {set width 8}
    
    set button $newframe.$name
    button $button -width $width -text $text -command $command
    set border $newframe.border_$name
    frame $border -relief flat -borderwidth 1
    raise $button
    pack $button -in $border -padx 2 -pady 2
    pack $border -in $newframe -side $side -padx 2
    if [string match $default $name] {
      $border configure -relief sunken
    }
  }
  return $newframe
}

######################################################################
# j:global_pref_panel ?options? - preferences panel for ~/.tk/defaults
# options include
#   -title - title of toplevel window
### ADD COLOURS?
######################################################################

proc j:global_pref_panel { args } {
  global env tk_strictMotif PREFS
  j:parse_args { {title "Global Preferences"} }
  
  set tl .prefs
  
  toplevel $tl
  wm title $tl $title

  checkbutton $tl.autoposition -relief flat -anchor w \
    -text {Auto-position dialogue boxes} \
    -variable PREFS(autoposition)
  checkbutton $tl.confirm -relief flat -anchor w \
    -text {Confirm actions} \
    -variable PREFS(confirm)
  checkbutton $tl.j_fs_fast -relief flat -anchor w \
    -text {Fast file selector} \
    -variable PREFS(j_fs_fast)
  checkbutton $tl.motif -relief flat -anchor w \
    -text {Strict Motif emulation} \
    -variable PREFS(tk_strictMotif)
  checkbutton $tl.scrollbarside -relief flat -anchor w \
    -text {Scrollbars on left} \
    -variable PREFS(scrollbarside) \
    -onvalue left -offvalue right
  frame $tl.bindings
  radiobutton $tl.bindings.basic -relief flat -anchor w \
    -text {Basic bindings} \
    -variable PREFS(bindings) -value basic
  radiobutton $tl.bindings.vi -relief flat -anchor w \
    -text {vi bindings} \
    -variable PREFS(bindings) -value vi -state disabled
  radiobutton $tl.bindings.emacs -relief flat -anchor w \
    -text {Emacs bindings} \
    -variable PREFS(bindings) -value emacs
  frame $tl.printer
  label $tl.printer.l -text {Printer:}
  entry $tl.printer.e -relief sunken -width 25 \
    -textvariable PREFS(printer)

  j:buttonbar $tl.b -default save -buttons [format {
    {
      save Save {
        j:write_prefs
        destroy %s
      }
    } {
      done Done {
        destroy %s
      }
    }
  } $tl $tl]
  
  pack $tl.bindings.basic \
    $tl.bindings.vi \
    $tl.bindings.emacs \
    -in $tl.bindings -side top -expand yes -fill x
  pack $tl.printer.l -in $tl.printer -side left
  pack $tl.printer.e -in $tl.printer \
    -expand yes -side left -padx 10 -pady 10

  pack $tl.autoposition \
    $tl.confirm \
    $tl.j_fs_fast \
    $tl.motif \
    $tl.scrollbarside \
    $tl.bindings \
    -in $tl -side top -expand yes -fill x
  pack [j:rule $tl] -in $tl -side top -fill x
  pack $tl.printer -in $tl -side top -expand yes -fill x
  pack [j:rule $tl] -in $tl -side top -fill x
  pack $tl.b -in $tl -side top -expand yes -fill x

  j:dialogue $tl		;# position in centre of screen

  focus $tl
  j:default_button $tl.b.save \
    $tl.printer.e \
    $tl

  bind $tl <Key-Tab> {focus $tl.printer.e}
  grab $tl
  tkwait window $tl
}

######################################################################
# j:colour_chooser w ?options? - create a metawidget for selecting colors
# options include
#   -label (default "Colour:")
#   -variable (global variable to set - not really optional)
### SHOULD SUPPORT -padx and -pady
######################################################################

proc j:colour_chooser { w args } {
  j:parse_args { {label "Colour:"} {variable j_prefs_colour} }
  
  set array_or_variable [lindex [split $variable "("] 0]
  
  global $array_or_variable
  
  append $variable {}
  if {[set $variable] == {} } then {set $variable "#a8a8a8"} ;# bisque

  frame $w
  
  label $w.label -anchor e -text $label
  
  button $w.patch -background [set $variable] -width 4
  
  button $w.rgb -width 8 -text "RGB" -command "
    set $variable \[j:prompt_colour_rgb\]
    $w.patch configure -background \[set $variable\]
  "
  button $w.name -width 8 -text "Name" -command "
    set $variable \[j:prompt_colour_name\]
    $w.patch configure -background \[set $variable\]
  "
  
  pack [j:filler $w] -in $w -side top -fill x
  pack [j:filler $w] -in $w -side bottom -fill x
  pack [j:filler $w] $w.name [j:filler $w] $w.rgb [j:filler $w] \
     $w.patch [j:filler $w] \
    -in $w -side right
  pack $w.label -side right -expand yes -fill x
  
  return $w
}

proc j:color_chooser \
  [info args j:colour_chooser] \
  [info body j:colour_chooser]

######################################################################
# j:variable_entry w ?options? - labelled entry for global variable
# options include
#   -label (default "Value:")
#   -variable (default value - not really optional)
#   -labelwidth (default 16)
#   -entrywidth (default 40)
#   -labelconfig (default "")
#   -entryconfig (default "")
# labelconfig and entryconfig are lists of additional configuration
#   options to control the appearance of the label and the entry
# BOGUSNESS:  you need to know that the entry is $newframe.e
#   in order to bind to it!
######################################################################

proc j:variable_entry {args} {
  j:parse_args {
    {label {Value:} }
    {variable value }
    {labelwidth {16} }
    {entrywidth {40} }
    {labelconfig {} }
    {entryconfig {} }
  }
  global $variable

  set newframe [lindex $args 0]
  
  frame $newframe
  set l $newframe.l
  set e $newframe.e
  label $l -text $label -width $labelwidth -anchor e
  entry $e -relief sunken -width $entrywidth -textvariable $variable
  
  if {$labelconfig != ""} {
    eval $l configure $labelconfig
  }
  if {$entryconfig != ""} {
    eval $e configure $entryconfig
  }
  pack $l -in $newframe -side left -fill both
  pack $e -in $newframe -side left -fill both
  
  return $newframe
}

######################################################################
# j:alert ?options? - alert box
# options include
#   -title (default "Alert")
#   -text (default "Alert!" - not really optional)
######################################################################

proc j:alert { args } {
  j:parse_args {
    {title "Alert"}
    {text "Alert!"}
  }
  set old_focus [focus]		;# so we can restore original focus
  toplevel .alert
  wm title .alert $title
  
  message .alert.msg -width 300 -anchor w -text $text
  j:buttonbar .alert.b -default ok -buttons {{ok OK {destroy .alert}}}
  pack append .alert \
    .alert.msg {top fill expand padx 10 pady 10} \
    [j:rule .alert -width 200] {top fillx} \
    .alert.b {bottom fill}

  j:dialogue .alert		;# position in centre of screen

  focus .alert
  j:default_button .alert.b.ok .alert
  grab .alert
  tkwait window .alert
  focus $old_focus
}

######################################################################
# j:confirm ?options? - Cancel/OK dialogue box
# options include
#   -title (default "Confirm")
#   -text (default "Are you sure?")
#   -priority (default 0)
#   -yesbutton (default "OK")
#   -nobutton (default "Cancel")
# returns true (1) on OK; false (0) on Cancel
# if (priority == 0 && $PREFS(confirm) == 0), the dialogue box is
#   not displayed; it always returns 1
######################################################################

proc j:confirm { args } {
  j:parse_args {
    {title Confirm}
    {priority 0}
    {text "Are you sure?"}
    {yesbutton OK}
    {nobutton Cancel}
  }
  
  global confirm_result
  global PREFS			;# for PREFS(confirm)

  if { (! $PREFS(confirm)) && (! $priority) } {
    return 1
  }

  set old_focus [focus]		;# so we can restore original focus

  toplevel .confirm
  wm title .confirm $title
  
  message .confirm.msg -width 300 -anchor w -text $text
  j:buttonbar .confirm.b -default ok -buttons [format {
    {ok %s {set confirm_result 1; destroy .confirm}}
    {cancel %s {set confirm_result 0; destroy .confirm}}
  } $yesbutton $nobutton]
  pack append .confirm  \
    .confirm.msg {top fill expand padx 10 pady 10} \
    [j:rule .confirm -width 200] {top fillx} \
    .confirm.b {bottom fillx}

  j:dialogue .confirm		;# position in centre of screen

  focus .confirm
  j:default_button .confirm.b.ok .confirm
  j:cancel_button .confirm.b.cancel .confirm
  grab .confirm
  tkwait window .confirm
  focus $old_focus
  return $confirm_result
}

######################################################################
# j:more ?options? ?-title hdr? ?-text txt? - displays text in window
# options include
#   -title (default "Output")
#   -text (default "" - not really optional)
#   -height (default 24)
#   -width (default 80)
#   -font (default "default")
#   -class (default "More")
# this unfortunately forces focus-follows-pointer in these windows
######################################################################

proc j:more { args } {
  global PREFS
  if {[lsearch [array names PREFS] {scrollbarside}] == -1} {
    set PREFS(scrollbarside) right ;# make sure it's defined
  }

  j:parse_args {
    {title Output}
    {text {}}
    {wrap char}
    {height 24}
    {width 80}
    {font default}
    {class More}
  }
  
  global j_more

  if {[info exists j_more(count)]} then {
    set j_more(count) [expr {$j_more(count) + 1}]
  } else {
    set j_more(count) 0
  }

  set w ".more$j_more(count)"

  toplevel $w -class $class
  wm title $w $title
  
  # using j:buttonbar for visual consistency, although we can't (easily)
  # set the commands with it (because they depend on the window name):
  
  j:buttonbar $w.b -default ok -buttons {
    {ok Done {}}
    {save Save {}}
    {print Print {}}
    {find {Find . . .} {}}
  }
  $w.b.ok configure -command "destroy $w"
  $w.b.save configure -command "j:more:save $w"
  $w.b.print configure -command "j:more:print $w"
  $w.b.find configure -width 8 -command "j:find -replace 0 $w.t"
  
  scrollbar $w.sb -relief flat -command "$w.t yview"
  text $w.t -yscrollcommand "$w.sb set" -setgrid true -wrap word \
    -height $height -wrap $wrap -width $width
  j:configure_font $w.t $font

  pack append $w \
    $w.b {bottom fillx} \
    [j:rule $w] {bottom fillx} \
    $w.sb "$PREFS(scrollbarside) filly" \
    [j:rule $w] "$PREFS(scrollbarside) filly" \
    $w.t {expand fill}

  $w.t insert end $text
  
  $w.t mark set insert 1.0

  $w.t configure -state disabled ;# prevent its being edited
  
  # FOLLOWING BINDINGS SHOULD BE GENERALISED! and check PREFS(bindings)!
  #
  bind $w.t <Next> "j:more:pageup $w.t"
  bind $w.t <space> "j:more:pageup $w.t"
  bind $w.t <Control-v> "j:more:pageup $w.t"
  
  bind $w.t <Prior> "j:more:pagedown $w.t"
  bind $w.t <b> "j:more:pagedown $w.t"
  bind $w.t <Escape><v> "j:more:pagedown $w.t"
  
  bind $w <Any-Enter> "focus $w.t"
  
  # "cancel" and "ok" amount to the same thing for this window:
  j:default_button $w.b.ok $w.t
  j:cancel_button $w.b.ok $w.t
  
  return $w.t			;# so caller can insert things in it
}

######################################################################
# j:more:save w - prompts to save the content of a j:more window
#   NOTE: this adds a newline!  should check if ends in newline alr.
######################################################################

proc j:more:save { w } {
  set filename [j:fs]
  if {$filename == {}	} {
    return 1
  }
  # should do error checking
  set file [open $filename {w}]
  puts $file [$w.t get 1.0 end]
  close $file
}

######################################################################
# j:more:print w - prompts to print the content of a j:more window
#   command to use should be a preference!
# uses PREFS(printer)
######################################################################

proc j:more:print { w } {
  global env PREFS
  
  append PREFS(printer) {}			;# make sure it's defined
  if {"x$PREFS(printer)" == "x"} then {set PREFS(printer) "lp"}
  
  if [j:confirm -priority 100 -text "Print using lpr to $PREFS(printer)?"] {
    # should do error checking
    set file [open "|lpr -P$PREFS(printer)" {w}]
    puts $file [$w.t get 1.0 end] nonewline
    close $file
  }
}

######################################################################
# j:more:pageup t - scrolls text widget t up
#   requires scrollbar to be sibling named "sb"
#   based on procedure by Paul Raines <raines@bohr.physics.upenn.edu>
######################################################################

proc j:more:pageup { t } {
  set sb "[winfo parent $t].sb"
  $t mark set insert "[lindex [$sb get] 3].0"
  $t yview insert
}

######################################################################
# j:more:pagedown t - scrolls text widget t down
#   requires scrollbar to be sibling named "sb"
#   based on procedure by Paul Raines <raines@bohr.physics.upenn.edu>
######################################################################

proc j:more:pagedown { t } {
  set sb "[winfo parent $t].sb"
  set currentstate [$sb get]
  
  # following is buggy if lines wrap:
  #
  set newlinepos [expr {[lindex $currentstate 2]-[lindex $currentstate 1]}]
  $t mark set insert "$newlinepos.0-2lines"
  $t yview insert
}
  
######################################################################
# j:prompt ?options? - prompt the user for information
# options are:
#   -text (default "Enter a value:"
#   -default (default "")
#   -cancelvalue (default "")
#   -file (default 0)
#   -title (default "Prompt")
# if $file, then the Tab key will do filename completion
######################################################################

proc j:prompt { args } {
  j:parse_args {
    {text "Enter a value:"}
    {default ""}
    {cancelvalue ""}
    {file 0}
    {title Prompt}
  }
  
  global j_prompt

  set old_focus [focus]		;# so we can restore original focus

  toplevel .pr
  wm title .pr $title
  
  message .pr.msg -width 300 -anchor w -text $text
  entry .pr.field -relief sunken -width 40
  j:buttonbar .pr.b -default ok -buttons [format {
    {ok OK {set j_prompt(result) [.pr.field get]; destroy .pr}}
    {cancel Cancel {set j_prompt(result) {%s}; destroy .pr}}
  } $cancelvalue]

  pack append .pr \
    .pr.msg {top fill expand padx 10} \
    .pr.field {top padx 10 pady 10} \
    .pr.b {bottom fillx} \
    [j:rule .pr -width 200] {bottom fillx}

  .pr.field delete 0 end
  .pr.field insert end $default

  j:dialogue .pr			;# position in centre of screen

  if $file {
    bind .pr.field <Tab> {
      set f [%W get]
      %W delete 0 end
      %W insert end [j:expand_filename $f]
    }
  }
  j:default_button .pr.b.ok .pr.field
  j:cancel_button .pr.b.cancel .pr.field

  focus .pr.field
  grab .pr
  tkwait window .pr
  focus $old_focus
  return $j_prompt(result)
}

######################################################################
# j:prompt_font ?options? - prompt for a font (via xfontsel)
# options are:
#   -prompt (default "Font:", but currently ignored)
#   -pattern (default "*")
# usage of xfontsel (`quit' button) not obvious!
######################################################################

proc j:prompt_font { args } {
  j:parse_args {
    {prompt "Font:"}
    {pattern "*"}
  }
  return [exec xfontsel -pattern $pattern -print]
}

######################################################################
# j:prompt_tcl - prompt for a tcl command and execute it
######################################################################

proc j:prompt_tcl {} {
  global j_prompt_tcl
  append j_prompt_tcl(RESULT) {}

  set prompt_result [j:prompt \
    -text "Tcl Command:" -default $j_prompt_tcl(RESULT)]
  if {$prompt_result != {}} then {
    set j_prompt_tcl(RESULT) $prompt_result
    eval $j_prompt_tcl(RESULT)
  }
}

######################################################################
# j:prompt_unix - prompt for a unix command and execute it
######################################################################

proc j:prompt_unix {} {
  global j_prompt_unix
  append j_prompt_unix(RESULT) {}

  set prompt_result [j:prompt \
    -text "Unix Command:" -default $j_prompt_unix(RESULT)]
  if {$prompt_result != {}} then {
    set j_prompt_unix(RESULT) $prompt_result
    set command $prompt_result
    set output [eval exec $command < /dev/null]
    if [string length $output] {
      j:more -title "Output of $command" -text $output
    } else {
      j:alert -text "No output from $command."
    }
  }
}

######################################################################
# j:prompt_colour_name - prompt for a colour name
######################################################################
### PROBLEM: LOCATION OF /usr/lib/X11/rgb.txt IS HARDCODED!
### Also, should open and process without forking off an awk
### Also, getting $w into strings is done in an ugly manner

proc j:prompt_colour_name { args } {
  j:parse_args {
    {prompt "Choose a colour:"}
    {title "Colour Name Selector"}
  }
  global j_prompt
  global PREFS
  if {[lsearch [array names PREFS] {scrollbarside}] == -1} {
    set PREFS(scrollbarside) right ;# make sure it's defined
  }
  
###   if {[info exists j_prompt(count)]} then {
###     set j_prompt(count) [expr {$j_prompt(count) + 1}]
###   } else {
###     set j_prompt(count) 0
###   }
### 
###   set w ".prompt$j_prompt(count)"

  set w ".prompt_colour_name"
  toplevel $w
  wm title $w $title
  
  set rgbfile /usr/lib/X11/rgb.txt
  if [file isfile $rgbfile] {
    set colourlist [lsort [exec awk { NF == 4 { print $4 } } $rgbfile]]
  } else {
    set colourlist {
      aquamarine bisque black blue brown burlywood coral crimson cyan
      firebrick gold goldenrod green grey grey25 grey33 grey50 grey66
      grey75 khaki lavender magenta maroon navy orange orchid pink plum
      purple red salmon tan tomato turquoise white yellow
    }
  }
  
  set j_prompt(colour) {}
  
  label $w.l -text $prompt
  j:buttonbar $w.b -default ok -orient vertical -buttons [list \
    [list ok OK [format {
        catch {set j_prompt(colour) [%s.list get [%s.list curselection]]}
        destroy %s
      } $w $w $w] \
    ] \
  ]
  frame $w.frame -width 100 -height 100 \
    -background bisque -relief raised -borderwidth 2
  frame $w.list
  scrollbar $w.list.sb -relief flat -command "$w.list.lb yview"
  listbox $w.list.lb -yscroll "$w.list.sb set" -relief flat \
    -geometry 20x20 -setgrid true
  
  pack $w.list.sb [j:rule $w.list] \
    -side $PREFS(scrollbarside) -fill y
  pack $w.list.lb -in $w.list -side left -expand yes -fill both
  
  pack $w.l [j:rule $w] -side top -fill x
  pack $w.list [j:rule $w] -side left -expand yes -fill both
  pack $w.frame -side top -fill both -expand yes -padx 10 -pady 10
  pack $w.b -side bottom -fill x
  pack [j:rule $w] -side bottom -fill x
  
  # Fill the listbox with a list of several useful colours:
  
  foreach i $colourlist {
    $w.list.lb insert end $i
  }
  
  # Set up bindings for the browser.
  
  bind $w.list.lb <Control-q> "destroy $w"
  bind $w.list.lb <Control-c> "destroy $w"
  focus $w.list.lb
  bind $w.list.lb <Button-1> "
    $w.list.lb select from \[$w.list.lb nearest %y\]
    catch {set j_prompt(colour) \[$w.list.lb get \[$w.list.lb curselection\]\]}
    $w.frame config -background \$j_prompt(colour)
  "
  bind $w.list.lb <Double-Button-1> "
    $w.b.ok invoke
  "
  
  j:default_button $w.b.ok $w
  focus $w
  j:dialogue $w
  tkwait window $w
  if {$j_prompt(colour) == ""} {set j_prompt(colour) bisque}
  return $j_prompt(colour)
}

proc j:prompt_color_name \
  [info args j:prompt_colour_name] \
  [info body j:prompt_colour_name]

######################################################################
# j:prompt_colour_rgb - prompt for a colour RGB value
#   An eviscerated version of selcol.tcl by Sam Shen <sls@aero.org>,
#   which also let you choose HSV values
######################################################################

proc j:prompt_colour_rgb { args } {
  j:parse_args {
    {prompt "Choose a colour:"}
    {title "RGB Colour Selector"}
  }
  global j_prompt
  
  set j_prompt(red) 255
  set j_prompt(blue) 196
  set j_prompt(green) 228
  set j_prompt(flag) 0
  
  set w .prompt_rgb
  toplevel $w
  wm title $w $title
  wm minsize $w 100 100
  wm maxsize $w [winfo screenwidth $w] [winfo screenheight $w]
  
  label $w.l -text $prompt

  frame $w.patch -width 100 -height 100 -relief raised -borderwidth 2
  entry $w.value -width 12
  bind $w.value <1> {%W select from @0; %W select to end}
  
  frame $w.scales
  set j_prompt(flag) 1
  j:prompt_colour_rgb:make_scale $w $w.scales.red red 255 Red
  j:prompt_colour_rgb:make_scale $w $w.scales.green green 255 Green
  j:prompt_colour_rgb:make_scale $w $w.scales.blue blue 255 Blue
  pack $w.scales.red $w.scales.green $w.scales.blue \
    -side left -fill y -expand yes
  set j_prompt(flag) 0
  
  j:buttonbar $w.b -default ok -orient vertical -buttons {
    {
      ok OK { }
    }
  }
  
  $w.b.ok configure -command "
    set j_prompt(return) \[$w.value get\]
    destroy $w
  "
  
  pack $w.l
  pack [j:rule $w] -fill x
  pack $w.scales -expand yes -side left -fill y
  pack [j:rule $w] -side left -fill y
  pack $w.value -fill both
  pack $w.patch -expand yes -fill both -padx 10 -pady 10
  pack [j:rule $w] -fill x
  pack $w.b -fill x
  
  j:prompt_colour_rgb:update_colour $w red $j_prompt(red)
  
  j:default_button $w.b.ok $w
  focus $w
  j:dialogue $w
  tkwait window $w
  return $j_prompt(return)
}

proc j:prompt_colour_rgb:make_scale {w name var to title} {
  global j_prompt
  
  frame $name
  scale $name.scale -to $to \
    -command "j:prompt_colour_rgb:update_colour $w $var"
  $name.scale set [set j_prompt($var)]
  label $name.label -text $title
  pack $name.label -in $name
  pack $name.scale -in $name -expand yes -fill y
}

proc j:prompt_colour_rgb:update_colour {w var value} {
  global j_prompt

  if {$j_prompt(flag) == 1} {return}
  set j_prompt(flag) 1
  set j_prompt($var) $value
  set colour [format "#%02x%02x%02x" \
    $j_prompt(red) $j_prompt(green) $j_prompt(blue)]
  catch {}
  $w.patch configure -background $colour
  $w.value delete @0 end
  $w.value insert 0 $colour
  set j_prompt(flag) 0
}

######################################################################

proc j:prompt_color_rgb \
  [info args j:prompt_colour_rgb] \
  [info body j:prompt_colour_rgb]

######################################################################
# j:configure_font widget fontlist - use font from list, or default
#   tries to set widget's font to each font in list.
#   if a font is `default', tries to set to X default font.
#   if a font is {}, sets to courier 12-point.
######################################################################

proc j:configure_font {widget fontlist} {
  foreach font $fontlist {
    # try to use each font, until one is successful:
    if {$font == {default}} {
      set font [option get $widget font Font]
      if {$font == {}} {set font {*-courier-medium-r-normal--*-120-*}}
    }
    if {! [catch {$widget configure -font $font}]} {return}
  }
}

######################################################################
# j:fs ?options? - file selector box
# options are:
#   -buttons (default {ok cancel home})
#   -prompt (default "Choose a file")
#   -directory (default ".")
#   -cancelvalue (default "")
#   -fileprompt (default "File:")
#   -title (default "File Selector")
# NOTE: this may do a cd---affects entire app!
######################################################################

proc j:fs { args } {
  j:parse_args {
    {buttons {ok cancel home} }
    {prompt "Choose a file"}
    {directory "."}
    {cancelvalue ""}
    {fileprompt "File:"}
    {title "File Selector"}
  }
    
  global j_fs env PREFS
  global fs_defaultbutton
  set PREFS(0) 1		;# make sure it's intepreted as array
  
  if {[lsearch [array names PREFS] {j_fs_fast}] == -1} {
    set PREFS(j_fs_fast) 0	;# make sure it's defined
  }
  if {[lsearch [array names PREFS] {scrollbarside}] == -1} {
    set PREFS(scrollbarside) right ;# make sure it's defined
  }

  set dir $directory
  set file ""

  if {![file isdirectory $dir]} {
    set dir .
  }

  set fs_defaultbutton [lindex $buttons 0]

  set j_fs(result) $file

  set old_focus [focus]		;# so we can restore original focus

  if [winfo exists .fs] {
    destroy .fs
  }

  cd $dir

  toplevel .fs
  wm title .fs $title
  wm minsize .fs 10 10

  label .fs.prompt -anchor w -text $prompt
  label .fs.cwd -text [pwd]
  frame .fs.list
  listbox .fs.list.lb -yscroll ".fs.list.sb set" -geometry 30x20
  scrollbar .fs.list.sb -relief flat -command ".fs.list.lb yview"
  frame .fs.file
  label .fs.file.l -text $fileprompt -anchor e
  entry .fs.file.e -relief sunken -text $file

  frame .fs.b -width 200
  button .fs.b.ok -width 8 -text {OK} -command {
    set file [.fs.file.e get]
    if {[file isdirectory $file]} {
      cd $file			;# cd into directory, refresh list
      .fs.cwd configure -text [pwd]
      j:fs:fill_list .fs.list.lb
      .fs.file.e delete 0 end	;# clear filename space
    } else {
      set cwd [pwd]
      if {$cwd == "/"} {set cwd ""}
      set file [.fs.file.e get]
      case $file in {
        /*	{set j_fs(result) $file}
        default {set j_fs(result) $cwd/$file}
      }
      destroy .fs
    update
    }
  }
  button .fs.b.gointo -width 8 -text "Go Into" -command {
    set file [.fs.file.e get]
    if {[file isdirectory $file]} {
      cd $file			;# cd into directory, refresh list
      .fs.cwd configure -text [pwd]
      j:fs:fill_list .fs.list.lb
      .fs.file.e delete 0 end	;# clear filename space
    } else {
      j:alert -text "\"$file\" is not a directory."
    }
  }
  button .fs.b.home -width 8 -text {Home} -command {
    cd $env(HOME)
    .fs.cwd configure -text [pwd]
    j:fs:fill_list .fs.list.lb
  }
  button .fs.b.root -width 8 -text {Root} -command {
    cd /
    .fs.cwd configure -text [pwd]
    j:fs:fill_list .fs.list.lb
  }
  button .fs.b.here -width 8 -text {Here} -command {
    set j_fs(result) [pwd]
    destroy .fs
    update
  }
  button .fs.b.cancel -width 8 -text {Cancel} -command "
    set j_fs(result) $cancelvalue
    destroy .fs
    update
  "
  checkbutton .fs.b.fast -text {Fast} -relief flat \
    -variable PREFS(j_fs_fast)

  pack append .fs.list \
    .fs.list.sb "$PREFS(scrollbarside) filly" \
    [j:rule .fs.list] "$PREFS(scrollbarside) filly" \
    .fs.list.lb {left expand fill}
  pack append .fs.file \
    .fs.file.l {left pady 10 padx 10} \
    .fs.file.e {left expand pady 10 fillx padx 10} \
    [j:filler .fs.file] {left}

  # now create the buttons the caller requested:
  #    (NEEDS ERROR CHECKING!)
  pack append .fs.b \
    [j:filler .fs.b] {bottom} \
    .fs.b.fast {top}
  foreach b $buttons {
    set button .fs.b.$b
    set border .fs.b.border_$b
    frame $border -borderwidth 1 -relief flat
    raise $button
    pack $button -in $border -padx 2 -pady 2
    pack $border -in .fs.b -side bottom -padx 10 -pady 4
  }
  # wider border on default button:
  .fs.b.border_$fs_defaultbutton configure -relief sunken

  pack append .fs \
    .fs.prompt {top fill} \
    [j:rule .fs] {top fillx} \
    .fs.cwd {top fill} \
    [j:rule .fs] {top fillx} \
    .fs.file {bottom expand fillx} \
    [j:rule .fs] {bottom fillx} \
    .fs.b {right filly} \
    [j:rule .fs] {right filly} \
    .fs.list {top expand fill}

  j:dialogue .fs		;# position in centre of screen

  .fs.file.e insert end $j_fs(result)

  focus .fs.file.e
  bind .fs.file.e <Key-Return> {
    set file [.fs.file.e get]
    if {$file != {} && [file isdirectory $file]} {
      .fs.b.gointo invoke
    } else {
      .fs.b.$fs_defaultbutton invoke
    }
  }
  bind .fs.file.e <Key-Tab> {	;# expand filename on <Tab>
    set f [%W get]
    %W delete 0 end
    %W insert end [j:expand_filename $f]
  }
  bind .fs.list.lb <Button-1> {	;# select, and insert filename into entry
    %W select from [%W nearest %y]
    set file [lindex [selection get] 0]
    .fs.file.e delete 0 end
    .fs.file.e insert end $file
  }

  bind .fs.list.lb <Double-Button-1> {	;# cd to dir or do default thing
    set file [lindex [j:selection_if_any] 0]
    if [file isdirectory $file] {
      .fs.b.gointo invoke
    } else {
      .fs.b.$fs_defaultbutton invoke
    }
  }
  
  j:cancel_button .fs.b.cancel .fs.file.e

#  grab .fs			;# for some reason this screws up 
				;#   "bind .fs.list.lb <Double-Button-1> ..."

  j:fs:fill_list .fs.list.lb	;# fill the listbox for the first time
  tkwait window .fs
  focus $old_focus
  return $j_fs(result)
}

######################################################################
# j:fs:fill_list lb - fill the listbox with files from CWD
######################################################################

proc j:fs:fill_list {lb} {
  global PREFS
  set PREFS(0) 1
  $lb delete 0 end

  # add ".." to go up a level:
  $lb insert end ".."

  update

  # add all normal (non-dot) files:
  foreach i [lsort [glob -nocomplain *]] {
    if { ! $PREFS(j_fs_fast) } {
      if {[file isdirectory $i]} {
        $lb insert end "$i/"
      } else {
        $lb insert end $i
      }
    } else {
      $lb insert end $i
    }
  }

  # add any dot-files:
  foreach i [lsort [glob -nocomplain .*]] {
    if {$i != "." && $i != ".."} {
      if { ! $PREFS(j_fs_fast) } {
        if {[file isdirectory $i]} {
          $lb insert end "$i/"
        } else {
          $lb insert end $i
        }
      } else {
        $lb insert end $i
      }
    }
  }
}

######################################################################
# j:find ?options? t - search-and-replace panel for text widget t
# option is:
#   -replace (default 1)
# if $replace, the replace box and buttons will be drawn, otherwise
#   it'll only be a search panel
######################################################################

proc j:find {args} {
  j:parse_args {{replace 1}}
  
  set t $args				;# text widget to search in
  global j_find
  set j_find(widget) $t
  append j_find(searchfor) {}		;# make sure it's defined
  append j_find(replacewith) {}		;# make sure it's defined

  if {! [info exists j_find(backwards)]} {
    set j_find(backwards) 0
  }
  if {! [info exists j_find(case)]} {
    set j_find(case) 0
  }

  if [winfo exists .find] {
    wm withdraw .find
    wm deiconify .find			;# just try to make it visible
    focus .find.t.search.e		;# and focus on the search field
    return 0
  }

  toplevel .find
  wm title .find "Find Panel"
  frame .find.t
  j:variable_entry .find.t.search \
    -label "Search for:" -variable j_find(searchfor)
  j:variable_entry .find.t.replace \
    -label "Replace with:" -variable j_find(replacewith)
  frame .find.t.options
  label .find.t.options.filler -text {} -width 16 -anchor e
  checkbutton .find.t.options.backwards -relief flat -anchor w \
    -text {Backwards} -variable j_find(backwards)
  checkbutton .find.t.options.case -relief flat -anchor w \
    -text "Case\255sensitive" -variable j_find(case)
  set buttons(search) {
    search Search
      {
        if {[j:find:find_string $j_find(searchfor) $j_find(widget)] == 0} {
          j:alert -text "Not found."
        }
      }
    }
  set buttons(replace) {
    replace Replace
      {
        j:find:replace $j_find(replacewith) $j_find(widget)
        j:find:find_string $j_find(searchfor) $j_find(widget)
      }
    }
  set buttons(replace_all) {
    replace_all {Replace All}
      {
        j:find:replace_all $j_find(searchfor) $j_find(replacewith) \
          $j_find(widget)
      }
    }
  set buttons(cancel) {cancel Cancel {destroy .find}}

  if $replace {
    j:buttonbar .find.b -default search -buttons [list \
      $buttons(search) \
      $buttons(replace) \
      $buttons(replace_all) \
      $buttons(cancel) \
    ]
  } else {
    j:buttonbar .find.b -default search -buttons [list \
      $buttons(search) \
      $buttons(cancel) \
    ]
  }
  
  j:tab_ring .find.t.search.e .find.t.replace.e
  
  pack append .find.t.options \
    .find.t.options.filler {left fill} \
    .find.t.options.backwards {left fill} \
    .find.t.options.case {left filly padx 20}
  pack append .find.t \
    [j:filler .find.t] {top} \
    .find.t.search {top expand fillx}
  if $replace {
    pack append .find.t \
      [j:filler .find.t] {top} \
      .find.t.replace {top expand fillx}
  }
  pack append .find.t \
    .find.t.options {top expand fill}
  pack append .find \
    .find.t {top fill padx 10 pady 5} \
    [j:rule .find] {top fillx} \
    .find.b {bottom fillx}

  # Meta-g (or Return, below) in either field searches:
  bind .find.t.search.e <Meta-g> \
    {.find.b.search invoke}
  bind .find.t.replace.e <Meta-g> \
    {.find.b.search invoke}

  j:default_button .find.b.search .find.t.search.e .find.t.replace.e
  j:cancel_button .find.b.cancel .find.t.search.e .find.t.replace.e

  focus .find.t.search.e
}

######################################################################
# j:find:again t - search again for same string
######################################################################

proc j:find:again {t} {
  global j_find
  append j_find(searchfor) {}			;# make sure it's defined
  set j_find(widget) $t

  if {$j_find(searchfor) == {}} {
    j:find $t
  } else {
    if {[j:find:find_string $j_find(searchfor) $t] == 0} {
      j:alert -text "Not found."
    }
  }
}

######################################################################
# j:find:find_string string t - find and select string in text widget t
######################################################################
# WARNING: since this takes a copy of the file, it could use a LOT
# of memory!
# should be rewritten to use a different mark than insert.

proc j:find:find_string { string t } {
  global j_find				;# text widget to search in
  set j_find(widget) $t
  append j_find(searchfor) {}		;# make sure it's defined
  append j_find(replacewith) {}		;# make sure it's defined

  if {! [info exists j_find(backwards)]} {
    set j_find(backwards) 0
  }
  if {! [info exists j_find(case)]} {
    set j_find(case) 0
  }

  # don't bother looking for the null string:
  if {$string == {}} {
    return 0				;# return 0 if null string
  }

  if {! [info exists j_find(backwards)]} {
    set j_find(backwards) 0
  }
  if {! [info exists j_find(backwards)]} {
    set j_find(case) 0
  }

  if $j_find(backwards) {
    set lastfirst last
    set textpart [$t get 0.0 {insert -1char}]
    set countfrom 0.0
  } else {
    set lastfirst first
    set textpart [$t get insert end]
    set countfrom insert
  }

  if {!$j_find(case)} {
    set string [string tolower $string]
    set textpart [string tolower $textpart]
  }
  set foundpos [string $lastfirst $string $textpart]
  # find length of selection:
  set lastpos [expr {$foundpos + [string length $string]}]
  
  if {$foundpos == -1} then {
    return 0				;# return 0 if not found
  }
  # deselect any already-selected text:
  catch {$t tag remove sel sel.first sel.last}
  $t tag add sel \
    "$countfrom + $foundpos chars" "$countfrom + $lastpos chars"
  # move insert just after the match (so we can continue from there)
  $t mark set insert "$countfrom + $lastpos chars"
  $t yview -pickplace insert
  return 1				;# return 1 if found
}

######################################################################
# j:find:replace string t - replace selection in t with string
######################################################################
# SHOULD CONFIRM THAT A SELECTION EXISTS!

proc j:find:replace { string t } {
  if [j:no_selection] {
    return 0
  }
  $t insert sel.first $string
  $t mark set insert sel.first
  $t delete sel.first sel.last
  return 1
}

######################################################################
# j:find:replace_all from to t - change all from to to in widget t
### BUG: this and the other routines need to be rewritten not to use
### the "insert" mark
######################################################################

proc j:find:replace_all {from to t} {
  $t mark set insert 0.0
  while {[j:find:find_string $from $t]} {
    j:find:replace $to $t
  }
}
