# NAME

Symbol::Rename - Rename imported symbols within a package

# DESCRIPTION

Sometimes you want to import functions from multiple modules. Sometimes these
functions may have the same name. Not all exporters let you rename the function
when you import it. This module can help!

This module is intended for renaming imported symbols. It is not intended for
renaming other symbols. It will not complain when renaming non-imported
symbols, but be warned that renaming non-imported symbols can cause unexpected
behavior.

# SYNOPSYS

## INSIDE THE PACKAGE DOING THE IMPORTING

    use First::Tool 'widget';
    use Symbol::Rename 'widget' => 'widget_1';

    use Second::Tool 'widget';
    use Symbol::Rename 'widget' => 'widget_2';

    widget_1(...);
    widget_2(...);

## AS A PACKAGE METHOD

    use Import::Into;
    use Symbol::Rename;

    sub bundle_into {
        my $pkg = shift;

        First::Tool->import::into($pkg, 'widget');
        $pkg->Symbol::Rename::rename_symbols('widget' => 'widget_1');

        Second::Tool->import::into($pkg, 'widget');
        $pkg->Symbol::Rename::rename_symbols('widget' => 'widget_2');
    }

# IMPORTING

The primary way to use this tool is to `use` it right after importing symbols
you wish to rename. Since the behavior is kicked off in a `use` statement it
will be evaluated at compile-time, renaming the symbols BEFORE anything can use
them, and before the parser can bind anything. It is rare to need this behavior
at run time.

## IMPORT ARGS

All arguments are expected to be pairs, that is `foo => 'bar', ...`.

### OPTIONS

Any argument prefixed with a '-' is considered a special option. Currently
'-package' is the only valid option:

    use Import::Into -package => 'Foo::Bar', ...;

This option allows you to specify a package name instead of relying on caller.

### SYMBOL RENAMING

All argument pairs where the key does not have a '-' prefix are considered
symbol names to be renamed. The key may include a sigil. If no sigil is listed
then '&' is assumed and it is treated as a subroutine. The new symbol name
**SHOULD NOT** contain a sigil.

    use Import::Into(
        '$foo' => 'new_scalar_name',
        '%foo' => 'new_hash_name',
        '@foo' => 'new_array_name',
        '&foo' => 'new_sub_name',
        'bar'  => 'another_new_sub_name',
        ...
    );

# PACKAGE METHOD

You can use `$pkg->Symbol::Rename::rename_symbols(...)` to rename symbols
without importing this package. The arguments are identical to input, but you
bypass the extra import logic. `$pkg` is the namespace being modified.

This is mainly useful when external tools need to rename symbols in a package
that imports them.

# SOURCE

The source code repository for Symbol-Rename can be found at
`http://github.com/exodist/Symbol-Rename/`.

# MAINTAINERS

- Chad Granum &lt;exodist@cpan.org>

# AUTHORS

- Chad Granum &lt;exodist@cpan.org>

# COPYRIGHT

Copyright 2015 Chad Granum &lt;exodist7@gmail.com>.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

See `http://dev.perl.org/licenses/`
