#!/usr/local/bin/perl -w
use 5.014;
use warnings;

use Test::More tests => 10;

# Before `make install' is performed this script should be runnable with
# `make test'. After `make install' it should work as `perl Quote.t'
# Note:
# This test requires internet connection.
# And it will take several minutes to process.

BEGIN {
	use_ok('Finance::YahooJPN::Quote')
};

#######################################################
my $obj1 = Finance::YahooJPN::Quote->new('1378.t');
isa_ok($obj1, 'Finance::YahooJPN::Quote');

my(@noadjust, @adjust, @TOPIX, @Nikkei, @JASDAQ);
my $file = 'noadjust';
while (<DATA>) {
	my $row = $_;
	$row =~ s/\x0D\x0A|\x0D|\x0A/\n/;
	given ($row) {
		when (/^cut1$/) {
			$file = 'adjust';
		}
		when (/^cut2$/) {
			$file = 'TOPIX';
		}
		when (/^cut3$/) {
			$file = 'Nikkei';
		}
		when (/^cut4$/) {
			$file = 'JASDAQ';
		}
		default {
			given ($file) {
				when ('noadjust') {
					push @noadjust, $row;
				}
				when ('adjust') {
					push @adjust, $row;
				}
				when ('TOPIX') {
					push @TOPIX, $row;
				}
				when ('Nikkei') {
					push @Nikkei, $row;
				}
				when ('JASDAQ') {
					push @JASDAQ, $row;
				}
			}
		}
	}
}
my $noadjust = join '', @noadjust;
my $adjust   = join '', @adjust;
my $TOPIX    = join '', @TOPIX;
my $Nikkei   = join '', @Nikkei;
my $JASDAQ   = join '', @JASDAQ;

$obj1->scan('start' => '2000-03-01');
my @tmp = $obj1->output('noadjust' => 1);
my @quote;
for (my $i = 0; $i <= $#tmp; $i++) {
	push @quote, $tmp[$i], "\n";
	last if $tmp[$i] =~ /2012-05-09/;
}
my $quote = join '', @quote;

is( $quote, $noadjust,
	'run scan() method w/o modification for splits' );

#######################################################
my $obj2 = Finance::YahooJPN::Quote->new('1378.t');
$obj2->scan('start' => '2000-03-01');
@tmp = $obj2->output();
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	push @quote, $tmp[$i], "\n";
	last if $tmp[$i] =~ /2012-05-09/;
}
$quote = join '', @quote;

is( $quote, $adjust,
	'run scan() method w/  modification for splits' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('1378.t', 'Start' => '2000-03-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	push @quote, $tmp[$i], "\n";
	last if $tmp[$i] =~ /2012-05-09/;
}
$quote = join '', @quote;
is( $quote, $adjust,
	'run historical() method (includes new(), scan() and output())' );

#######################################################
my $obj3 = Finance::YahooJPN::Quote->new('9437.t');
$obj3->scan('start' => '2002-03-26', 'last' => '2002-03-26');
@tmp = $obj3->output();
$quote = $tmp[0];
my $expected = '2002-03-26	341000	351000	329000	333000	208021';

is( $quote, $expected,
	'in a rare case split data is at the top row' );

#######################################################
my $obj4 = Finance::YahooJPN::Quote->new('0000.t');
$obj4->scan('start' => '2003-09-01', 'last' => '2003-09-30');
@tmp = $obj4->output();
$quote = join "\n", @tmp;
$expected = '';

is( $quote, $expected,
	'in a case the specified symbol is absent' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('998405', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $TOPIX,
	'run historical() method for TOPIX' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('998407', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $Nikkei,
	'run historical() method for Nikkei' );

#######################################################
@tmp = Finance::YahooJPN::Quote->historical('23337', 'Start' => '2009-01-01');
@quote = ();
for (my $i = 0; $i <= $#tmp; $i++) {
	last if $tmp[$i] =~ /2012-04-25/;
	push @quote, $tmp[$i], "\n";
}
$quote = join '', @quote;
is( $quote, $JASDAQ,
	'run historical() method for JASDAQ' );

__END__
2000-03-01	1170	1170	1100	1140	45000
2000-03-02	1139	1139	959	1000	10000
2000-03-03	997	998	940	941	9000
2000-03-06	941	941	890	890	6000
2000-03-07	790	790	790	790	15000
2000-03-08	770	890	770	890	9000
2000-03-09	890	890	880	880	5000
2000-03-10	780	780	780	780	16000
2000-03-13	750	830	750	800	16000
2000-03-14	801	801	760	760	8000
2000-03-15	800	810	780	780	6000
2000-03-16	800	800	770	800	8000
2000-03-17	799	799	797	797	5000
2000-03-21	797	815	797	805	9000
2000-03-22	750	805	750	805	10000
2000-03-23	804	804	804	804	2000
2000-03-24	800	800	753	760	7000
2000-03-27	770	770	750	770	11000
2000-03-28	770	770	760	760	8000
2000-03-29	750	750	740	740	10000
2000-03-30	740	750	735	750	17000
2000-03-31	745	745	730	745	10000
2000-04-03	740	740	720	740	12000
2000-04-04	740	740	715	715	19000
2000-04-05	698	703	690	690	20000
2000-04-06	685	688	650	688	9000
2000-04-07	678	710	670	700	9000
2000-04-10	700	700	652	700	10000
2000-04-11	700	720	700	720	10000
2000-04-12	770	800	750	800	19000
2000-04-13	835	845	820	820	10000
2000-04-14	770	800	770	800	5000
2000-04-17	790	790	700	700	19000
2000-04-18	721	750	700	750	20000
2000-04-19	750	751	750	751	2000
2000-04-20	752	752	750	750	10000
2000-04-21	750	790	750	780	9000
2000-04-24	780	785	770	770	4000
2000-04-26	750	750	730	730	3000
2000-04-27	680	709	670	700	6000
2000-04-28	701	769	701	769	2000
2000-05-02	759	759	759	759	3000
2000-05-08	800	800	790	790	12000
2000-05-09	730	750	730	750	9000
2000-05-10	739	745	730	730	5000
2000-05-11	750	750	730	730	11000
2000-05-12	731	731	731	731	1000
2000-05-15	731	733	731	733	5000
2000-05-16	750	750	750	750	1000
2000-05-17	740	740	730	730	5000
2000-05-18	731	749	730	739	9000
2000-05-19	741	741	720	720	7000
2000-05-22	750	750	700	700	12000
2000-05-23	700	700	681	681	4000
2000-05-24	680	695	678	690	22000
2000-05-25	690	700	690	700	3000
2000-05-26	700	700	650	700	11000
2000-05-29	724	724	680	690	3000
2000-05-30	680	690	680	690	2000
2000-05-31	690	695	680	680	15000
2000-06-01	690	690	690	690	4000
2000-06-02	680	680	680	680	4000
2000-06-05	700	700	700	700	4000
2000-06-06	680	680	680	680	9000
2000-06-08	680	680	680	680	1000
2000-06-09	680	680	670	670	2000
2000-06-12	670	680	670	680	8000
2000-06-13	690	690	680	690	4000
2000-06-14	690	690	690	690	1000
2000-06-15	680	680	670	670	2000
2000-06-16	690	700	690	700	3000
2000-06-20	700	700	690	690	5000
2000-06-21	689	689	670	670	5000
2000-06-22	670	670	670	670	10000
2000-06-23	670	670	670	670	8000
2000-06-27	671	671	671	671	2000
2000-06-28	671	671	670	670	2000
2000-06-29	670	670	670	670	1000
2000-06-30	690	690	670	670	4000
2000-07-03	699	699	690	699	5000
2000-07-04	759	765	759	765	10000
2000-07-05	764	764	760	763	7000
2000-07-07	759	760	759	760	4000
2000-07-10	759	760	710	714	6000
2000-07-11	763	763	763	763	4000
2000-07-12	715	715	714	714	3000
2000-07-13	760	760	760	760	1000
2000-07-17	718	718	718	718	1000
2000-07-18	719	719	719	719	1000
2000-07-19	720	720	720	720	2000
2000-07-21	722	726	722	726	3000
2000-07-24	701	701	700	700	3000
2000-07-25	700	700	690	690	11000
2000-07-26	700	700	680	680	6000
2000-07-27	680	680	680	680	8000
2000-07-28	700	700	680	680	4000
2000-07-31	680	680	671	671	4000
2000-08-01	700	700	700	700	1000
2000-08-02	680	680	680	680	2000
2000-08-03	650	660	650	660	3000
2000-08-04	660	680	660	680	4000
2000-08-07	700	700	700	700	3000
2000-08-08	680	680	680	680	5000
2000-08-09	680	680	670	670	2000
2000-08-10	660	660	660	660	1000
2000-08-16	700	700	700	700	1000
2000-08-17	690	690	690	690	2000
2000-08-18	688	688	680	680	12000
2000-08-21	700	700	700	700	3000
2000-08-22	680	680	670	670	2000
2000-08-23	660	675	660	675	3000
2000-08-24	668	670	660	660	3000
2000-08-25	660	660	660	660	4000
2000-08-28	660	660	650	650	6000
2000-08-29	661	661	660	660	2000
2000-08-30	660	660	660	660	2000
2000-08-31	670	670	670	670	2000
2000-09-01	695	700	670	670	15000
2000-09-04	680	680	660	670	15000
2000-09-05	680	680	650	650	13000
2000-09-07	680	680	650	650	18000
2000-09-08	650	689	650	689	8000
2000-09-11	700	700	650	650	2000
2000-09-12	651	651	651	651	1000
2000-09-13	691	691	691	691	2000
2000-09-14	651	655	651	655	2000
2000-09-19	685	700	685	700	5000
2000-09-20	700	700	690	690	4000
2000-09-21	690	690	690	690	3000
2000-09-22	685	685	685	685	2000
2000-09-26	659	700	651	700	4000
2000-09-27	660	690	660	690	2000
2000-09-28	690	690	690	690	1000
2000-09-29	660	687	650	687	8000
2000-10-02	690	690	690	690	3000
2000-10-03	690	690	690	690	2000
2000-10-04	669	689	666	666	3000
2000-10-05	690	690	690	690	4000
2000-10-06	690	690	690	690	4000
2000-10-10	655	655	655	655	1000
2000-10-11	660	660	660	660	1000
2000-10-12	652	679	652	679	3000
2000-10-17	670	670	660	660	3000
2000-10-18	660	660	611	611	6000
2000-10-19	620	620	610	610	8000
2000-10-20	612	617	612	616	3000
2000-10-23	621	621	620	620	2000
2000-10-24	620	620	620	620	1000
2000-10-25	644	644	644	644	1000
2000-10-26	643	643	640	640	5000
2000-10-30	640	640	630	640	7000
2000-10-31	640	640	640	640	1000
2000-11-01	640	640	640	640	5000
2000-11-06	640	640	640	640	2000
2000-11-07	640	640	640	640	6000
2000-11-08	620	620	620	620	5000
2000-11-13	600	600	600	600	3000
2000-11-15	620	620	620	620	1000
2000-11-16	610	610	610	610	4000
2000-11-17	639	639	639	639	1000
2000-11-20	649	649	610	610	5000
2000-11-21	601	601	600	601	4000
2000-11-22	600	600	600	600	1000
2000-11-24	610	610	600	600	2000
2000-11-27	600	600	574	574	5000
2000-11-29	576	576	576	576	1000
2000-11-30	590	590	590	590	1000
2000-12-01	600	600	600	600	2000
2000-12-04	600	618	600	618	4000
2000-12-05	619	620	619	620	5000
2000-12-06	620	620	620	620	1000
2000-12-08	610	610	610	610	2000
2000-12-11	600	600	600	600	2000
2000-12-12	600	600	600	600	2000
2000-12-13	570	570	570	570	2000
2000-12-14	571	571	571	571	1000
2000-12-18	620	620	620	620	2000
2000-12-19	571	572	571	571	3000
2000-12-20	600	600	571	571	4000
2000-12-21	521	521	521	521	2000
2000-12-22	570	570	570	570	5000
2000-12-25	570	570	570	570	3000
2000-12-28	570	570	560	560	2000
2001-01-04	600	600	600	600	1000
2001-01-05	610	610	600	600	5000
2001-01-09	540	582	532	582	3000
2001-01-12	530	531	530	530	5000
2001-01-15	575	620	575	620	17000
2001-01-16	640	660	620	660	17000
2001-01-17	675	720	650	720	15000
2001-01-18	720	750	720	720	23000
2001-01-19	747	750	730	750	25000
2001-01-22	850	850	850	850	53000
2001-01-23	870	870	760	770	133000
2001-01-24	760	760	701	719	24000
2001-01-25	700	709	660	680	28000
2001-01-26	709	709	670	704	8000
2001-01-29	706	719	706	719	15000
2001-01-30	739	739	739	739	1000
2001-01-31	739	749	729	749	12000
2001-02-01	750	750	730	730	7000
2001-02-02	730	730	700	720	8000
2001-02-05	700	700	690	700	7000
2001-02-06	672	690	660	660	6000
2001-02-07	680	700	660	685	8000
2001-02-08	685	690	680	680	11000
2001-02-09	680	700	680	690	9000
2001-02-13	700	700	700	700	1000
2001-02-14	700	700	680	680	3000
2001-02-15	699	699	680	680	4000
2001-02-19	679	679	630	630	6000
2001-02-20	699	699	635	635	4000
2001-02-21	651	699	651	690	4000
2001-02-23	699	699	699	699	1000
2001-02-26	644	644	644	644	2000
2001-02-27	650	665	650	665	4000
2001-02-28	665	665	660	660	2000
2001-03-01	645	645	630	630	4000
2001-03-02	671	671	660	660	2000
2001-03-05	660	660	639	639	6000
2001-03-06	630	630	619	619	9000
2001-03-07	630	630	620	620	6000
2001-03-08	620	639	605	605	6000
2001-03-09	639	639	605	629	4000
2001-03-12	639	645	639	645	7000
2001-03-13	640	640	640	640	1000
2001-03-14	620	625	620	620	3000
2001-03-15	610	638	610	638	3000
2001-03-16	648	648	645	645	2000
2001-03-21	640	640	607	607	8000
2001-03-22	617	617	608	610	11000
2001-03-23	617	617	609	609	4000
2001-03-26	610	614	609	614	7000
2001-03-27	611	611	610	611	3000
2001-03-28	610	610	610	610	5000
2001-03-29	619	645	610	645	6000
2001-03-30	639	639	620	620	2000
2001-04-02	629	629	611	611	2000
2001-04-03	612	612	612	612	1000
2001-04-04	614	620	614	620	2000
2001-04-05	630	630	600	604	41000
2001-04-06	606	609	600	600	30000
2001-04-09	599	599	591	591	7000
2001-04-10	591	591	582	582	6000
2001-04-11	590	592	587	592	13000
2001-04-12	591	591	585	585	4000
2001-04-13	585	585	585	585	7000
2001-04-16	579	579	560	560	12000
2001-04-17	564	570	563	570	4000
2001-04-18	589	589	570	570	5000
2001-04-20	630	630	580	580	4000
2001-04-23	600	630	600	630	5000
2001-04-24	615	615	615	615	1000
2001-04-25	619	619	581	581	3000
2001-04-26	619	619	600	600	7000
2001-04-27	600	600	600	600	2000
2001-05-01	600	610	600	601	3000
2001-05-02	619	620	603	603	5000
2001-05-07	611	620	611	617	5000
2001-05-08	620	620	605	605	3000
2001-05-09	601	615	601	615	6000
2001-05-10	602	602	600	600	9000
2001-05-11	600	610	600	600	5000
2001-05-14	610	610	610	610	2000
2001-05-15	610	610	600	610	3000
2001-05-16	612	612	612	612	2000
2001-05-17	601	605	601	605	4000
2001-05-18	615	615	615	615	1000
2001-05-21	615	615	615	615	3000
2001-05-22	610	610	610	610	1000
2001-05-23	605	619	600	619	3000
2001-05-24	600	600	600	600	3000
2001-05-25	615	615	615	615	1000
2001-05-28	635	665	635	640	29000
2001-05-29	640	720	640	720	73000
2001-05-30	710	715	690	690	47000
2001-05-31	681	681	655	679	15000
2001-06-01	679	715	675	700	41000
2001-06-04	700	709	695	709	33000
2001-06-05	690	730	690	730	53000
2001-06-06	740	780	735	768	179000
2001-06-07	768	780	750	750	82000
2001-06-08	749	779	730	770	46000
2001-06-11	767	768	751	751	3000
2001-06-12	750	751	736	751	12000
2001-06-13	755	775	745	775	35000
2001-06-14	765	775	742	751	29000
2001-06-15	743	751	743	750	28000
2001-06-18	755	755	731	745	17000
2001-06-19	745	750	740	750	33000
2001-06-20	740	750	736	750	23000
2001-06-21	745	750	733	750	29000
2001-06-22	750	830	750	812	226000
2001-06-25	806	820	799	812	56000
2001-06-26	802	840	800	812	161000
2001-06-27	820	858	820	850	195000
2001-06-28	851	870	840	847	89000
2001-06-29	845	847	825	825	51000
2001-07-02	855	855	820	837	48000
2001-07-03	840	845	820	825	37000
2001-07-04	805	824	800	820	37000
2001-07-05	800	820	790	791	39000
2001-07-06	796	810	796	809	42000
2001-07-09	810	810	779	790	13000
2001-07-10	790	798	774	798	31000
2001-07-11	779	779	760	760	4000
2001-07-12	740	775	740	740	10000
2001-07-13	768	768	760	765	30000
2001-07-16	760	760	715	730	6000
2001-07-17	730	740	730	740	19000
2001-07-18	745	745	700	740	38000
2001-07-19	730	730	710	710	4000
2001-07-23	709	709	695	695	6000
2001-07-24	692	700	680	699	34000
2001-07-25	690	690	690	690	1000
2001-07-26	690	700	690	700	9000
2001-07-27	690	696	650	694	32000
2001-07-30	700	700	660	660	9000
2001-07-31	699	750	699	720	32000
2001-08-01	730	730	719	719	15000
2001-08-03	670	697	670	697	22000
2001-08-06	697	700	697	700	14000
2001-08-07	690	690	690	690	2000
2001-08-08	689	690	660	660	28000
2001-08-09	660	665	630	660	11000
2001-08-10	660	660	660	660	3000
2001-08-13	660	661	660	661	2000
2001-08-14	661	661	661	661	1000
2001-08-15	660	660	660	660	2000
2001-08-16	630	660	630	660	4000
2001-08-17	660	660	660	660	2000
2001-08-20	675	675	670	675	4000
2001-08-21	660	660	650	650	3000
2001-08-22	650	650	640	640	2000
2001-08-23	640	640	630	630	5000
2001-08-24	630	630	630	630	3000
2001-08-27	615	630	615	615	10000
2001-08-29	610	610	605	610	5000
2001-08-30	600	605	580	585	10000
2001-08-31	590	600	580	600	15000
2001-09-03	600	600	580	580	4000
2001-09-04	582	600	582	594	4000
2001-09-05	624	625	595	604	8000
2001-09-06	624	624	600	600	6000
2001-09-07	600	600	595	600	6000
2001-09-10	599	599	599	599	1000
2001-09-11	580	580	550	560	31000
2001-09-12	510	520	510	520	37000
2001-09-13	510	520	505	515	26000
2001-09-14	530	540	520	535	13000
2001-09-17	555	565	549	549	7000
2001-09-18	579	580	579	580	3000
2001-09-19	590	599	590	590	8000
2001-09-20	585	585	580	580	9000
2001-09-21	580	580	550	550	4000
2001-09-25	589	590	589	589	5000
2001-09-26	570	570	551	551	4000
2001-09-27	560	560	551	551	3000
2001-09-28	550	565	550	565	14000
2001-10-01	560	560	560	560	2000
2001-10-02	580	580	580	580	6000
2001-10-03	570	570	560	560	5000
2001-10-04	561	579	561	579	15000
2001-10-05	589	589	562	562	8000
2001-10-09	575	575	575	575	4000
2001-10-10	570	575	560	560	8000
2001-10-11	590	590	590	590	1000
2001-10-12	575	575	570	570	4000
2001-10-15	590	590	589	589	4000
2001-10-16	570	572	570	570	8000
2001-10-17	570	570	565	565	10000
2001-10-18	585	587	578	587	5000
2001-10-19	590	590	570	570	11000
2001-10-22	570	570	560	561	7000
2001-10-23	562	565	562	565	4000
2001-10-24	575	575	570	570	4000
2001-10-25	570	579	569	578	7000
2001-10-26	581	590	580	589	10000
2001-10-29	585	590	585	590	12000
2001-10-30	590	615	590	605	19000
2001-10-31	602	618	600	618	11000
2001-11-01	630	650	623	645	16000
2001-11-02	640	655	630	640	14000
2001-11-05	641	660	631	650	18000
2001-11-06	650	675	650	675	26000
2001-11-07	675	685	675	685	41000
2001-11-08	680	680	670	680	18000
2001-11-09	680	685	657	670	21000
2001-11-12	670	670	650	650	11000
2001-11-13	630	650	609	645	13000
2001-11-15	640	640	640	640	5000
2001-11-16	620	620	591	601	15000
2001-11-19	611	613	595	599	9000
2001-11-20	599	600	599	600	16000
2001-11-21	590	600	590	600	16000
2001-11-22	600	600	600	600	2000
2001-11-26	612	620	600	600	4000
2001-11-28	595	595	595	595	1000
2001-11-29	600	600	600	600	2000
2001-11-30	610	610	590	590	6000
2001-12-03	590	590	590	590	2000
2001-12-04	590	590	590	590	3000
2001-12-05	600	600	590	590	2000
2001-12-06	590	590	590	590	1000
2001-12-07	590	590	590	590	4000
2001-12-10	590	590	590	590	1000
2001-12-11	580	580	578	578	3000
2001-12-12	571	575	571	575	4000
2001-12-13	575	575	560	562	11000
2001-12-17	551	555	541	547	22000
2001-12-18	542	575	542	575	6000
2001-12-19	555	580	540	540	18000
2001-12-20	539	550	532	535	11000
2001-12-21	530	550	522	522	16000
2001-12-25	522	590	522	590	11000
2001-12-26	590	599	590	599	11000
2001-12-27	609	640	600	600	18000
2001-12-28	592	620	592	610	13000
2002-01-04	580	600	580	600	3000
2002-01-07	610	610	610	610	3000
2002-01-08	600	600	600	600	2000
2002-01-11	590	590	590	590	4000
2002-01-15	590	590	580	580	3000
2002-01-16	555	555	520	550	26000
2002-01-17	555	555	526	545	14000
2002-01-18	579	579	570	570	4000
2002-01-21	580	580	540	540	19000
2002-01-22	540	570	540	570	4000
2002-01-23	553	555	545	545	3000
2002-01-24	545	545	540	541	5000
2002-01-25	541	570	541	570	5000
2002-01-28	564	564	550	550	3000
2002-01-29	551	551	550	550	5000
2002-01-30	550	550	540	540	7000
2002-01-31	556	570	556	570	11000
2002-02-01	570	570	541	541	6000
2002-02-04	543	545	543	545	7000
2002-02-05	565	565	543	543	7000
2002-02-06	541	560	540	543	9000
2002-02-07	543	543	540	540	5000
2002-02-08	560	560	560	560	3000
2002-02-12	559	560	559	560	3000
2002-02-13	560	565	555	565	7000
2002-02-14	579	579	579	579	5000
2002-02-15	570	570	550	550	2000
2002-02-18	570	570	555	560	15000
2002-02-19	579	579	570	570	7000
2002-02-20	556	571	556	571	5000
2002-02-21	575	575	575	575	3000
2002-02-22	575	575	575	575	2000
2002-02-25	570	570	557	560	12000
2002-02-26	556	556	555	555	2000
2002-02-27	560	560	560	560	10000
2002-02-28	561	570	560	570	9000
2002-03-01	570	580	570	580	4000
2002-03-04	580	580	559	574	8000
2002-03-05	580	580	560	580	16000
2002-03-06	564	571	564	571	14000
2002-03-07	565	565	565	565	1000
2002-03-08	565	565	560	560	10000
2002-03-11	561	570	560	560	17000
2002-03-12	560	560	552	552	7000
2002-03-13	556	556	556	556	1000
2002-03-14	556	570	556	570	3000
2002-03-15	575	575	555	555	7000
2002-03-18	570	570	555	556	5000
2002-03-19	556	556	552	552	6000
2002-03-20	564	564	553	554	4000
2002-03-22	560	570	560	570	11000
2002-03-25	577	599	575	599	13000
2002-03-26	599	600	599	599	16000
2002-03-27	598	598	598	598	1000
2002-04-01	585	585	550	550	6000
2002-04-02	570	570	540	540	3000
2002-04-03	546	550	545	550	3000
2002-04-04	548	550	540	550	5000
2002-04-05	574	574	551	551	5000
2002-04-08	552	552	550	550	3000
2002-04-09	545	545	540	545	17000
2002-04-10	545	545	545	545	4000
2002-04-11	560	567	547	547	6000
2002-04-12	550	550	550	550	1000
2002-04-16	560	560	560	560	3000
2002-04-17	560	560	546	546	3000
2002-04-18	570	570	570	570	1000
2002-04-19	564	564	540	543	18000
2002-04-22	551	552	545	545	5000
2002-04-23	545	550	545	550	2000
2002-04-24	560	560	558	558	3000
2002-04-25	558	558	558	558	3000
2002-04-26	555	555	545	545	5000
2002-04-30	545	545	545	545	1000
2002-05-02	552	552	552	552	2000
2002-05-07	556	560	552	560	6000
2002-05-08	570	570	555	555	5000
2002-05-09	555	555	555	555	2000
2002-05-10	550	550	550	550	2000
2002-05-13	551	560	550	560	6000
2002-05-14	550	550	550	550	1000
2002-05-15	550	550	545	545	3000
2002-05-17	555	560	542	560	11000
2002-05-20	565	565	550	550	7000
2002-05-21	560	560	560	560	5000
2002-05-22	555	555	550	550	5000
2002-05-23	555	555	550	550	6000
2002-05-24	563	612	562	612	50000
2002-05-27	629	634	626	628	53000
2002-05-28	634	634	600	618	32000
2002-05-29	601	601	591	591	4000
2002-05-30	605	605	590	590	7000
2002-05-31	606	610	600	600	12000
2002-06-03	601	620	601	620	6200
2002-06-04	615	620	608	620	6100
2002-06-05	619	619	605	605	7100
2002-06-06	605	605	605	605	5000
2002-06-07	605	605	590	590	4200
2002-06-10	593	601	593	601	3100
2002-06-11	593	601	590	590	5900
2002-06-12	580	582	575	575	3000
2002-06-13	596	600	578	600	6300
2002-06-14	581	585	581	585	5800
2002-06-17	585	585	585	585	1300
2002-06-18	595	615	595	610	5900
2002-06-19	611	615	600	610	8800
2002-06-20	610	610	600	600	4100
2002-06-21	600	600	585	585	4700
2002-06-24	600	600	587	600	2600
2002-06-25	600	610	600	610	4100
2002-06-26	610	610	600	600	4400
2002-06-27	585	586	580	580	3000
2002-06-28	600	600	585	585	4100
2002-07-01	610	610	609	609	2600
2002-07-02	615	615	573	573	12600
2002-07-03	598	604	595	604	3000
2002-07-04	604	613	590	590	7300
2002-07-05	605	610	605	610	3100
2002-07-08	613	613	600	600	7400
2002-07-09	601	610	600	600	1900
2002-07-10	613	613	600	600	4000
2002-07-11	610	610	600	600	2500
2002-07-12	600	609	595	600	7100
2002-07-15	592	600	592	600	3800
2002-07-16	615	615	601	615	6300
2002-07-17	609	610	606	606	7000
2002-07-18	600	610	600	610	8200
2002-07-19	610	620	600	620	33900
2002-07-22	614	614	600	600	10500
2002-07-23	601	601	590	600	5200
2002-07-24	590	590	590	590	1900
2002-07-26	596	609	590	609	1100
2002-07-29	614	614	601	601	1400
2002-07-30	608	609	608	608	1200
2002-08-01	601	601	600	600	1400
2002-08-02	581	600	566	590	9200
2002-08-05	600	600	590	590	2600
2002-08-06	585	585	570	580	5800
2002-08-07	580	580	570	580	3000
2002-08-08	580	580	580	580	1800
2002-08-09	580	600	570	600	5100
2002-08-12	600	610	598	610	9100
2002-08-13	605	605	580	580	7600
2002-08-14	600	600	581	581	6400
2002-08-15	595	595	585	585	2500
2002-08-16	595	600	585	600	7000
2002-08-19	600	600	598	599	2900
2002-08-20	600	600	581	599	5200
2002-08-21	582	590	582	589	3900
2002-08-22	581	582	578	578	5000
2002-08-23	591	591	590	591	4400
2002-08-26	592	592	590	590	15800
2002-08-27	590	592	586	592	9200
2002-08-28	592	592	590	590	3100
2002-08-29	592	599	590	590	12200
2002-08-30	599	599	585	585	6100
2002-09-02	585	585	578	578	8000
2002-09-03	589	598	585	585	8800
2002-09-04	577	585	577	582	13500
2002-09-05	590	600	590	600	5500
2002-09-06	586	586	583	583	2900
2002-09-09	599	599	591	591	2200
2002-09-10	590	590	590	590	100
2002-09-11	585	585	585	585	300
2002-09-12	581	581	581	581	1000
2002-09-13	585	590	585	590	1000
2002-09-17	580	580	570	575	4700
2002-09-18	575	575	562	562	2000
2002-09-19	575	575	575	575	300
2002-09-20	590	590	590	590	2600
2002-09-24	580	590	580	590	300
2002-09-25	590	590	585	585	300
2002-09-26	563	566	563	563	3700
2002-09-27	565	565	560	560	7300
2002-09-30	575	575	551	575	3300
2002-10-01	545	560	545	560	2600
2002-10-02	568	568	568	568	400
2002-10-03	568	568	560	560	1400
2002-10-04	561	561	550	560	3400
2002-10-07	567	567	567	567	1400
2002-10-08	510	525	510	525	1300
2002-10-09	525	530	525	530	4600
2002-10-10	520	534	515	515	5500
2002-10-11	555	555	535	535	700
2002-10-15	515	535	515	525	4000
2002-10-16	540	540	525	525	1400
2002-10-17	525	530	520	520	5400
2002-10-18	560	560	520	520	4700
2002-10-21	550	550	545	545	400
2002-10-22	550	550	540	540	2500
2002-10-23	520	524	520	520	6800
2002-10-24	520	520	515	515	5100
2002-10-25	515	540	515	537	3100
2002-10-28	530	531	530	530	2000
2002-10-30	530	530	529	529	4900
2002-10-31	529	540	523	523	6200
2002-11-01	513	525	513	523	4100
2002-11-05	523	524	523	524	4000
2002-11-06	526	526	523	523	800
2002-11-07	525	525	524	524	1900
2002-11-08	489	510	489	510	5900
2002-11-11	510	510	506	509	1100
2002-11-12	509	509	500	500	1400
2002-11-13	500	500	485	485	7600
2002-11-14	485	508	485	505	8700
2002-11-15	486	500	486	500	2200
2002-11-18	498	498	490	498	1800
2002-11-19	498	498	490	490	2900
2002-11-20	500	500	480	480	6500
2002-11-21	481	481	480	480	1000
2002-11-22	480	495	480	490	3600
2002-11-25	500	500	483	494	2200
2002-11-26	499	505	499	505	7000
2002-11-27	495	505	495	505	3200
2002-11-28	500	500	500	500	2000
2002-11-29	500	548	500	548	8100
2002-12-02	548	548	537	537	1300
2002-12-03	525	525	512	520	1500
2002-12-04	525	525	525	525	700
2002-12-05	520	520	500	518	3700
2002-12-06	498	518	490	518	3600
2002-12-09	510	510	495	509	4900
2002-12-10	500	500	500	500	700
2002-12-11	500	500	495	500	2800
2002-12-12	500	500	495	495	2000
2002-12-13	495	495	490	490	4700
2002-12-16	485	494	485	494	2100
2002-12-17	488	490	485	485	8300
2002-12-18	495	495	475	485	5100
2002-12-19	481	494	464	494	9100
2002-12-20	495	495	490	495	6000
2002-12-24	495	495	465	489	4100
2002-12-25	498	498	480	495	9200
2002-12-26	498	498	498	498	100
2002-12-27	498	499	490	490	2200
2002-12-30	500	510	498	510	5100
2003-01-06	520	520	520	520	1500
2003-01-07	530	537	520	520	7800
2003-01-08	515	515	515	515	800
2003-01-09	510	510	510	510	400
2003-01-10	510	510	505	510	3000
2003-01-14	510	510	486	492	6400
2003-01-15	515	515	505	515	2200
2003-01-16	515	515	510	510	2700
2003-01-17	518	520	510	515	5700
2003-01-20	525	530	525	530	3100
2003-01-21	520	520	515	515	2100
2003-01-22	520	520	520	520	300
2003-01-23	520	520	510	510	3500
2003-01-24	520	525	520	525	1700
2003-01-27	524	524	520	524	900
2003-01-28	510	510	510	510	200
2003-01-29	510	510	502	502	1000
2003-01-30	510	520	502	502	1000
2003-01-31	520	520	500	515	600
2003-02-03	520	520	520	520	100
2003-02-04	520	520	500	500	2000
2003-02-05	523	523	520	520	1900
2003-02-06	520	520	510	520	1300
2003-02-07	510	530	510	530	4800
2003-02-10	502	550	502	550	7200
2003-02-12	520	545	520	545	4400
2003-02-13	529	529	520	520	1600
2003-02-14	525	540	525	540	2500
2003-02-17	540	540	540	540	800
2003-02-18	539	539	531	532	700
2003-02-19	540	545	535	545	4800
2003-02-20	550	550	540	549	9000
2003-02-21	530	539	520	530	8000
2003-02-24	521	525	510	525	6600
2003-02-25	525	525	520	520	1100
2003-02-26	520	520	520	520	100
2003-02-27	519	519	510	510	300
2003-02-28	510	510	501	501	3800
2003-03-03	502	502	502	502	900
2003-03-04	501	505	501	501	2000
2003-03-05	510	520	510	520	4500
2003-03-06	510	515	492	515	3200
2003-03-07	500	500	500	500	100
2003-03-10	490	522	490	522	10700
2003-03-11	500	500	482	500	2500
2003-03-12	490	501	490	501	1700
2003-03-13	501	501	490	490	1800
2003-03-14	490	490	480	490	7100
2003-03-17	489	493	489	489	4800
2003-03-18	489	500	489	500	500
2003-03-19	500	500	498	498	300
2003-03-20	500	517	500	517	4700
2003-03-24	510	510	509	509	1000
2003-03-25	505	511	505	511	4700
2003-03-26	510	510	510	510	800
2003-03-27	510	510	510	510	600
2003-03-28	510	510	500	500	3100
2003-03-31	500	500	485	485	2100
2003-04-01	490	495	490	490	2700
2003-04-02	490	490	486	490	1900
2003-04-03	491	499	490	499	2400
2003-04-04	483	498	483	498	1300
2003-04-07	499	499	499	499	1500
2003-04-08	480	497	480	497	2200
2003-04-09	490	490	490	490	300
2003-04-10	490	497	480	495	3900
2003-04-14	499	499	495	495	900
2003-04-15	499	499	480	480	3600
2003-04-16	485	485	480	480	7400
2003-04-17	485	490	480	490	3500
2003-04-18	490	497	490	497	4900
2003-04-21	490	496	485	496	600
2003-04-22	496	497	485	485	4300
2003-04-23	485	490	481	490	1900
2003-04-24	491	495	480	495	8100
2003-04-25	495	495	485	490	3100
2003-04-28	490	490	480	490	1600
2003-04-30	490	495	490	490	800
2003-05-01	480	495	479	494	2500
2003-05-02	494	494	483	493	1000
2003-05-06	494	494	479	491	3300
2003-05-07	482	485	482	485	700
2003-05-08	482	489	482	488	600
2003-05-09	488	488	473	483	11800
2003-05-12	489	489	480	480	900
2003-05-13	482	485	480	480	6200
2003-05-14	480	481	479	479	4500
2003-05-15	478	480	478	480	1600
2003-05-16	488	488	470	473	14100
2003-05-19	473	475	472	472	1100
2003-05-20	479	479	474	474	7100
2003-05-21	475	478	475	478	1100
2003-05-22	476	478	476	478	2800
2003-05-23	479	479	479	479	2500
2003-05-26	501	513	501	505	35100
2003-05-27	505	505	498	498	6900
2003-05-28	496	500	496	500	7200
2003-05-29	496	496	490	490	5500
2003-05-30	497	497	490	490	3900
2003-06-02	500	523	499	523	19400
2003-06-03	530	530	503	503	2800
2003-06-04	513	513	505	512	2700
2003-06-05	512	512	506	506	5300
2003-06-06	512	512	507	510	7900
2003-06-09	512	512	507	507	4300
2003-06-10	492	510	492	510	4000
2003-06-11	506	510	506	510	6100
2003-06-12	510	510	505	505	4700
2003-06-13	505	509	505	509	3100
2003-06-16	510	510	502	502	9400
2003-06-17	508	520	508	520	14600
2003-06-18	523	525	520	520	5100
2003-06-19	530	530	522	522	12100
2003-06-20	510	528	510	525	7300
2003-06-23	528	528	525	525	2100
2003-06-24	521	530	515	515	6500
2003-06-25	518	530	518	530	5400
2003-06-26	523	525	521	521	2700
2003-06-27	528	539	528	539	7600
2003-06-30	539	548	535	548	4900
2003-07-01	548	560	548	558	23900
2003-07-02	565	565	540	550	22700
2003-07-03	549	550	515	530	13700
2003-07-04	529	532	525	527	5000
2003-07-07	550	559	540	559	12500
2003-07-08	560	560	520	540	2500
2003-07-09	540	545	540	545	2400
2003-07-10	546	548	525	531	5300
2003-07-11	530	532	525	532	2600
2003-07-14	533	548	533	548	5600
2003-07-15	548	550	534	534	7500
2003-07-16	530	530	526	527	2500
2003-07-17	520	521	520	521	900
2003-07-18	524	525	524	525	6300
2003-07-22	535	535	511	512	2100
2003-07-23	516	520	510	510	5800
2003-07-24	540	540	515	515	2500
2003-07-25	516	533	510	521	7100
2003-07-28	512	520	512	513	10100
2003-07-29	516	516	512	515	7400
2003-07-30	525	525	515	516	2300
2003-07-31	523	523	510	510	3800
2003-08-01	520	520	510	511	1900
2003-08-04	520	530	515	520	11700
2003-08-05	521	522	515	520	9000
2003-08-06	520	521	520	520	2200
2003-08-07	520	520	508	508	4300
2003-08-08	509	520	509	520	3300
2003-08-11	538	538	530	531	6000
2003-08-12	514	520	514	520	1400
2003-08-13	520	521	520	521	1200
2003-08-14	520	535	520	531	13700
2003-08-15	532	545	528	545	28300
2003-08-18	560	560	540	550	10400
2003-08-19	545	550	538	545	18600
2003-08-20	551	551	540	542	11800
2003-08-21	519	520	502	520	26900
2003-08-22	520	521	518	519	3800
2003-08-25	519	519	515	515	10200
2003-08-26	511	516	510	513	18700
2003-08-27	513	516	513	516	2700
2003-08-28	517	519	512	513	7200
2003-08-29	514	517	513	513	2300
2003-09-01	518	528	515	528	9000
2003-09-02	529	529	520	527	6700
2003-09-03	520	529	520	529	2400
2003-09-04	529	529	528	529	4000
2003-09-05	533	535	529	532	8900
2003-09-08	537	537	532	532	6300
2003-09-09	532	532	530	530	3600
2003-09-10	530	530	520	520	3900
2003-09-11	520	538	520	531	10700
2003-09-12	537	631	536	631	173000
2003-09-16	640	650	626	628	243700
2003-09-17	628	629	605	605	42700
2003-09-18	596	680	595	680	109600
2003-09-19	659	670	632	659	56800
2003-09-22	650	650	625	640	17600
2003-09-24	630	639	625	625	12400
2003-09-25	615	635	596	602	26000
2003-09-26	622	625	604	625	4500
2003-09-29	635	640	625	640	10000
2003-09-30	640	649	619	619	13400
2003-10-01	619	639	613	639	25300
2003-10-02	629	645	623	633	15500
2003-10-03	623	638	623	633	4900
2003-10-06	630	641	615	641	15100
2003-10-07	640	640	625	625	1200
2003-10-08	625	635	610	625	11200
2003-10-09	625	625	617	624	5600
2003-10-10	607	619	598	605	14100
2003-10-14	607	615	600	605	15900
2003-10-15	614	614	601	607	9100
2003-10-16	606	630	605	630	12400
2003-10-17	630	631	613	620	9900
2003-10-20	620	620	601	615	9100
2003-10-21	620	620	605	610	14200
2003-10-22	609	609	602	602	7200
2003-10-23	602	608	595	608	4700
2003-10-24	609	609	596	603	3100
2003-10-27	581	590	560	579	15600
2003-10-28	579	579	550	569	6700
2003-10-29	556	560	556	560	8700
2003-10-30	561	565	561	565	2100
2003-10-31	566	570	563	570	9000
2003-11-04	580	600	575	598	8000
2003-11-05	598	610	580	610	6900
2003-11-06	601	620	580	601	9700
2003-11-07	599	600	590	590	4400
2003-11-10	598	607	595	607	2800
2003-11-11	577	585	572	585	5500
2003-11-12	605	605	590	590	1000
2003-11-13	590	599	580	580	3600
2003-11-14	589	600	589	599	1400
2003-11-17	598	598	551	551	10200
2003-11-18	569	569	520	530	21300
2003-11-19	530	531	510	510	12800
2003-11-20	515	550	500	550	25000
2003-11-21	520	530	500	520	29400
2003-11-25	519	538	519	538	10300
2003-11-26	540	560	495	559	24800
2003-11-27	558	558	517	517	3700
2003-11-28	547	547	520	525	12700
2003-12-01	535	535	520	529	4400
2003-12-02	544	544	522	533	3200
2003-12-03	526	526	515	525	11500
2003-12-04	524	524	515	519	10700
2003-12-05	523	525	507	525	21600
2003-12-08	520	521	490	490	104600
2003-12-09	500	509	500	503	13900
2003-12-10	513	513	503	505	28600
2003-12-11	513	520	507	519	24400
2003-12-12	519	523	510	510	38400
2003-12-15	524	535	522	533	23600
2003-12-16	531	590	530	572	71100
2003-12-17	590	590	510	523	205200
2003-12-18	526	531	521	522	42500
2003-12-19	530	530	523	524	24400
2003-12-22	523	523	501	507	41000
2003-12-24	520	520	504	506	25400
2003-12-25	506	518	503	509	26100
2003-12-26	509	510	499	509	35800
2003-12-29	510	510	504	508	14700
2003-12-30	509	511	509	510	13600
2004-01-05	512	518	512	513	10600
2004-01-06	513	530	513	521	22700
2004-01-07	522	531	521	529	8900
2004-01-08	539	542	532	541	24300
2004-01-09	544	544	532	535	27800
2004-01-13	544	544	532	537	14900
2004-01-14	537	543	535	542	22900
2004-01-15	540	548	537	538	20700
2004-01-16	537	538	532	533	16400
2004-01-19	530	535	521	530	17900
2004-01-20	539	539	526	530	29100
2004-01-21	530	533	525	530	12600
2004-01-22	531	531	524	527	13900
2004-01-23	525	528	525	527	3500
2004-01-26	528	533	525	529	20100
2004-01-27	530	530	525	526	6900
2004-01-28	525	533	520	533	26600
2004-01-29	535	538	530	536	34300
2004-01-30	537	560	532	557	54200
2004-02-02	555	557	541	543	46200
2004-02-03	543	546	521	531	21700
2004-02-04	531	532	525	528	11400
2004-02-05	534	534	528	530	14200
2004-02-06	525	533	524	532	15700
2004-02-09	526	530	525	526	9200
2004-02-10	526	526	521	522	11100
2004-02-12	523	528	521	523	13700
2004-02-13	525	531	525	531	14000
2004-02-16	523	533	521	531	22000
2004-02-17	530	533	528	528	11700
2004-02-18	529	531	525	531	19500
2004-02-19	530	530	525	528	11100
2004-02-20	526	530	525	530	20500
2004-02-23	525	525	522	522	20500
2004-02-24	521	522	516	516	28700
2004-02-25	517	517	514	514	7800
2004-02-26	510	510	506	510	18100
2004-02-27	509	513	509	512	9900
2004-03-01	513	525	513	522	11800
2004-03-02	523	523	520	522	10300
2004-03-03	522	528	521	527	14500
2004-03-04	527	528	525	527	14700
2004-03-05	527	528	525	525	8500
2004-03-08	526	527	525	527	9300
2004-03-09	527	530	527	530	16700
2004-03-10	530	540	530	540	23200
2004-03-11	536	540	535	536	15700
2004-03-12	537	537	529	530	15000
2004-03-15	530	537	529	530	17600
2004-03-16	536	536	529	533	24000
2004-03-17	535	535	530	534	28800
2004-03-18	534	534	529	529	21500
2004-03-19	535	535	530	532	12800
2004-03-22	532	536	532	536	4700
2004-03-23	536	538	535	535	7800
2004-03-24	538	543	536	543	12900
2004-03-25	541	546	540	545	27900
2004-03-26	540	543	533	540	8200
2004-03-29	539	539	532	535	11100
2004-03-30	537	537	533	535	4300
2004-03-31	533	535	533	534	12000
2004-04-01	535	535	533	533	15100
2004-04-02	538	538	531	533	10800
2004-04-05	535	539	534	536	14900
2004-04-06	539	540	533	535	44800
2004-04-07	538	539	535	539	13300
2004-04-08	540	541	538	538	8600
2004-04-09	536	538	531	532	12000
2004-04-12	535	537	533	534	4800
2004-04-13	537	540	536	538	14400
2004-04-14	539	540	538	539	18200
2004-04-15	540	540	537	537	22800
2004-04-16	537	537	535	535	5900
2004-04-19	539	540	537	538	11000
2004-04-20	544	545	540	542	21400
2004-04-21	545	545	540	544	21600
2004-04-22	546	555	545	555	42400
2004-04-23	565	577	560	573	65400
2004-04-26	600	600	581	595	97300
2004-04-27	590	594	573	581	57900
2004-04-28	581	581	571	580	15300
2004-04-30	579	582	575	582	31100
2004-05-06	585	588	580	588	27600
2004-05-07	585	586	570	570	14100
2004-05-10	545	563	535	561	21500
2004-05-11	535	542	535	538	10100
2004-05-12	542	550	541	548	4800
2004-05-13	550	550	545	550	5800
2004-05-14	535	537	535	537	12900
2004-05-17	520	525	500	511	15300
2004-05-18	505	521	503	521	24000
2004-05-19	521	560	521	535	12100
2004-05-20	558	558	550	550	3700
2004-05-21	555	555	530	550	4100
2004-05-24	550	550	536	540	5500
2004-05-25	550	550	536	550	800
2004-05-26	560	560	550	553	13700
2004-05-27	554	554	545	545	4100
2004-05-28	545	546	540	540	6400
2004-05-31	539	543	536	543	7500
2004-06-01	539	543	535	543	1800
2004-06-02	540	540	537	537	2900
2004-06-03	533	534	532	533	6700
2004-06-04	534	534	530	530	10400
2004-06-07	533	545	531	545	7000
2004-06-08	546	547	541	541	2600
2004-06-09	541	542	525	525	14100
2004-06-10	531	540	525	525	15100
2004-06-11	527	538	525	538	13900
2004-06-14	541	541	535	535	4800
2004-06-15	537	540	535	536	4000
2004-06-16	540	540	535	537	8800
2004-06-17	539	540	537	537	10100
2004-06-18	541	547	540	546	24200
2004-06-21	555	573	553	567	35400
2004-06-22	562	568	550	552	23800
2004-06-23	552	555	550	550	4700
2004-06-24	551	559	548	557	8400
2004-06-25	560	568	555	568	26300
2004-06-28	570	578	570	572	19300
2004-06-29	570	573	564	565	7800
2004-06-30	570	570	560	560	7500
2004-07-01	570	570	561	562	9800
2004-07-02	570	570	557	558	15200
2004-07-05	557	557	550	550	6800
2004-07-06	551	557	551	551	4500
2004-07-07	552	566	550	566	6700
2004-07-08	556	565	556	565	3700
2004-07-09	567	567	560	565	5000
2004-07-12	565	567	563	567	8100
2004-07-13	565	565	560	565	9900
2004-07-14	565	610	565	582	56400
2004-07-15	600	605	590	596	67600
2004-07-16	599	632	599	620	191700
2004-07-20	640	679	636	679	434800
2004-07-21	669	779	663	779	1151300
2004-07-22	789	879	781	827	1251600
2004-07-23	807	819	752	785	751000
2004-07-26	800	880	800	880	1382400
2004-07-27	910	918	786	825	1369600
2004-07-28	850	859	785	832	993800
2004-07-29	822	826	763	763	581400
2004-07-30	760	799	740	793	638700
2004-08-02	793	805	762	768	399900
2004-08-03	765	780	738	740	309200
2004-08-04	729	765	722	750	250200
2004-08-05	740	795	731	777	255400
2004-08-06	747	775	746	761	127800
2004-08-09	747	778	746	765	123400
2004-08-10	773	780	760	763	103500
2004-08-11	773	774	764	765	51700
2004-08-12	767	779	765	774	94200
2004-08-13	773	819	765	810	409300
2004-08-16	830	837	819	835	423900
2004-08-17	837	848	815	818	285800
2004-08-18	810	810	780	790	158000
2004-08-19	790	802	782	802	115700
2004-08-20	792	807	783	798	119400
2004-08-23	758	775	749	751	245100
2004-08-24	750	750	711	711	230400
2004-08-25	701	730	670	723	282000
2004-08-26	728	740	706	707	111400
2004-08-27	703	738	703	736	130300
2004-08-30	731	734	720	732	61200
2004-08-31	731	731	720	722	25200
2004-09-01	720	727	720	722	33400
2004-09-02	724	727	721	725	25200
2004-09-03	729	735	726	727	49600
2004-09-06	730	738	730	735	47200
2004-09-07	743	743	716	723	58900
2004-09-08	726	726	711	719	50900
2004-09-09	719	719	705	711	48100
2004-09-10	705	708	685	695	86100
2004-09-13	685	697	677	681	58500
2004-09-14	678	686	672	686	48500
2004-09-15	706	742	700	712	156200
2004-09-16	702	710	701	703	65100
2004-09-17	710	714	703	703	19600
2004-09-21	709	709	702	706	22300
2004-09-22	703	707	703	705	33500
2004-09-24	704	708	700	708	46500
2004-09-27	707	725	703	720	46800
2004-09-28	723	725	707	707	39500
2004-09-29	707	712	692	692	40200
2004-09-30	686	693	685	691	17100
2004-10-01	694	702	694	700	18300
2004-10-04	705	705	696	701	22400
2004-10-05	703	706	701	702	13400
2004-10-06	702	711	702	711	18300
2004-10-07	711	711	703	707	11500
2004-10-08	705	711	701	705	19400
2004-10-12	701	705	687	688	40800
2004-10-13	690	693	685	686	28600
2004-10-14	685	690	680	684	11300
2004-10-15	680	687	678	686	17400
2004-10-18	680	682	678	679	16600
2004-10-19	680	685	676	676	55700
2004-10-20	680	680	670	671	38500
2004-10-21	673	720	673	675	122400
2004-10-22	677	732	677	729	166900
2004-10-25	689	745	681	734	272800
2004-10-26	735	747	722	731	110700
2004-10-27	734	744	718	727	48300
2004-10-28	737	737	724	732	44700
2004-10-29	737	737	718	724	24600
2004-11-01	711	740	711	734	56200
2004-11-02	744	744	730	735	23400
2004-11-04	743	743	733	733	43700
2004-11-05	739	739	725	732	24900
2004-11-08	740	740	725	730	34300
2004-11-09	723	750	723	750	79700
2004-11-10	769	800	748	764	241800
2004-11-11	750	775	750	753	50100
2004-11-12	753	812	753	812	382400
2004-11-15	822	836	801	820	370300
2004-11-16	820	823	785	807	235000
2004-11-17	797	835	797	830	243200
2004-11-18	835	848	806	818	209900
2004-11-19	817	824	788	807	107600
2004-11-22	799	800	768	785	89800
2004-11-24	783	829	778	788	287100
2004-11-25	796	798	778	784	92500
2004-11-26	771	791	760	761	72000
2004-11-29	760	792	760	774	52700
2004-11-30	783	783	775	779	43000
2004-12-01	781	784	771	773	36200
2004-12-02	781	781	764	775	30800
2004-12-03	770	772	765	765	20400
2004-12-06	760	770	760	760	57200
2004-12-07	765	775	756	756	48800
2004-12-08	750	760	750	758	30900
2004-12-09	758	760	730	739	64200
2004-12-10	730	738	720	725	53400
2004-12-13	735	735	701	720	126100
2004-12-14	710	720	703	707	43400
2004-12-15	707	712	698	712	42500
2004-12-16	712	712	700	702	34900
2004-12-17	708	709	702	709	27400
2004-12-20	716	730	711	721	64500
2004-12-21	728	728	713	719	38600
2004-12-22	725	750	718	744	58300
2004-12-24	752	752	726	727	128200
2004-12-27	568	568	530	558	89400
2004-12-28	560	567	559	566	56900
2004-12-29	567	572	558	561	92800
2004-12-30	571	571	561	562	28900
2005-01-04	582	583	567	583	27200
2005-01-05	588	590	579	582	79700
2005-01-06	580	580	569	579	47100
2005-01-07	575	575	570	571	34600
2005-01-11	580	580	570	576	29000
2005-01-12	576	577	572	573	23600
2005-01-13	572	575	571	573	15700
2005-01-14	571	575	571	575	20600
2005-01-17	575	577	571	571	30400
2005-01-18	571	573	570	570	21600
2005-01-19	570	571	565	565	31900
2005-01-20	566	566	556	557	42300
2005-01-21	556	560	552	560	73300
2005-01-24	560	577	556	570	47200
2005-01-25	577	577	563	570	41100
2005-01-26	568	569	556	567	72700
2005-01-27	560	566	558	561	36600
2005-01-28	560	564	556	564	22100
2005-01-31	560	572	560	570	39900
2005-02-01	570	572	565	570	23500
2005-02-02	574	574	565	565	24400
2005-02-03	564	570	558	567	27300
2005-02-04	570	572	558	558	44500
2005-02-07	547	556	531	556	109000
2005-02-08	555	561	551	561	44300
2005-02-09	555	564	555	558	27100
2005-02-10	557	560	557	560	17300
2005-02-14	560	568	559	566	25200
2005-02-15	566	569	560	567	21000
2005-02-16	567	567	558	564	14600
2005-02-17	557	565	557	564	20400
2005-02-18	567	568	560	565	19600
2005-02-21	565	565	559	559	36800
2005-02-22	558	560	550	551	34900
2005-02-23	556	560	549	552	31300
2005-02-24	553	561	547	550	26200
2005-02-25	550	558	547	551	21100
2005-02-28	550	555	550	551	24600
2005-03-01	552	556	552	555	21500
2005-03-02	558	558	550	551	64500
2005-03-03	551	554	545	552	56300
2005-03-04	553	555	548	553	27700
2005-03-07	553	558	550	553	31500
2005-03-08	553	556	546	553	27500
2005-03-09	553	557	552	556	14800
2005-03-10	556	558	555	558	23700
2005-03-11	558	559	556	559	16600
2005-03-14	565	569	561	563	53800
2005-03-15	570	574	565	567	52600
2005-03-16	570	575	568	574	37800
2005-03-17	573	576	571	571	42900
2005-03-18	575	585	573	584	73700
2005-03-22	586	588	584	586	114100
2005-03-23	588	590	586	588	79300
2005-03-24	588	589	583	583	28700
2005-03-25	579	579	569	574	42400
2005-03-28	550	571	550	568	20300
2005-03-29	560	567	555	555	18200
2005-03-30	556	571	556	571	16100
2005-03-31	571	571	554	567	8200
2005-04-01	551	560	551	558	12600
2005-04-04	557	558	551	552	13100
2005-04-05	555	555	551	554	14300
2005-04-06	554	557	550	550	20000
2005-04-07	554	557	553	555	5800
2005-04-08	555	558	554	555	13300
2005-04-11	566	566	555	556	16300
2005-04-12	560	560	555	555	11300
2005-04-13	554	556	553	553	16500
2005-04-14	553	554	545	547	32600
2005-04-15	545	549	541	545	8900
2005-04-18	515	524	510	515	59700
2005-04-19	515	530	515	522	15400
2005-04-20	540	543	530	542	17000
2005-04-21	540	540	522	527	13700
2005-04-22	535	535	521	522	25500
2005-04-25	523	524	520	523	9100
2005-04-26	530	530	520	523	4100
2005-04-27	525	527	522	522	4500
2005-04-28	522	525	521	521	5400
2005-05-02	520	529	520	522	4600
2005-05-06	530	530	515	518	25600
2005-05-09	519	519	500	510	38100
2005-05-10	503	508	502	504	27000
2005-05-11	507	510	503	503	18600
2005-05-12	504	507	504	507	15200
2005-05-13	505	507	505	507	9500
2005-05-16	505	515	503	503	21400
2005-05-17	503	508	501	502	13700
2005-05-18	500	509	500	506	12300
2005-05-19	507	516	507	511	16700
2005-05-20	519	521	512	518	14000
2005-05-23	528	554	528	544	80200
2005-05-24	556	565	555	565	93900
2005-05-25	565	568	560	562	121300
2005-05-26	560	560	530	550	44400
2005-05-27	540	540	533	533	9800
2005-05-30	531	535	526	527	36300
2005-05-31	530	539	530	530	14100
2005-06-01	535	535	530	533	12500
2005-06-02	538	539	532	537	11000
2005-06-03	531	539	531	532	9400
2005-06-06	538	541	535	541	15000
2005-06-07	550	555	538	539	33000
2005-06-08	535	545	535	539	10500
2005-06-09	537	540	535	540	4100
2005-06-10	543	553	543	550	19300
2005-06-13	550	568	549	555	85200
2005-06-14	562	562	551	551	29600
2005-06-15	554	554	550	554	23400
2005-06-16	554	554	550	551	28500
2005-06-17	555	555	549	551	27100
2005-06-20	555	555	551	552	27600
2005-06-21	554	554	550	552	11900
2005-06-22	555	565	554	558	77600
2005-06-23	560	563	555	555	16800
2005-06-24	552	556	551	555	17000
2005-06-27	550	551	545	548	17700
2005-06-28	548	564	548	551	17900
2005-06-29	559	559	550	554	11800
2005-06-30	554	557	550	557	15600
2005-07-01	558	558	550	552	6500
2005-07-04	553	556	552	552	17300
2005-07-05	555	555	552	552	6600
2005-07-06	552	554	551	553	13500
2005-07-07	553	553	550	552	15700
2005-07-08	552	552	550	551	13000
2005-07-11	552	554	544	545	20200
2005-07-12	541	546	540	543	24300
2005-07-13	544	544	541	542	17500
2005-07-14	542	543	540	540	15900
2005-07-15	540	544	540	542	7000
2005-07-19	541	543	541	541	13400
2005-07-20	543	545	541	544	20400
2005-07-21	545	545	541	541	7700
2005-07-22	541	543	540	540	11600
2005-07-25	540	540	521	536	27000
2005-07-26	534	536	528	531	20300
2005-07-27	531	534	531	531	18100
2005-07-28	532	540	532	535	17500
2005-07-29	535	543	535	537	12400
2005-08-01	551	552	540	545	31800
2005-08-02	546	546	539	542	10500
2005-08-03	540	543	538	538	7100
2005-08-04	538	538	529	530	11200
2005-08-05	532	532	517	519	31300
2005-08-08	502	510	491	505	48700
2005-08-09	511	513	505	507	57000
2005-08-10	512	523	512	517	49900
2005-08-11	523	523	515	523	50300
2005-08-12	517	520	515	515	36200
2005-08-15	518	519	515	515	29800
2005-08-16	517	519	513	514	32400
2005-08-17	512	522	512	517	36400
2005-08-18	516	519	516	519	28300
2005-08-19	519	525	517	517	35900
2005-08-22	518	519	516	517	38900
2005-08-23	519	520	517	518	35100
2005-08-24	518	520	518	520	24400
2005-08-25	522	522	520	520	17400
2005-08-26	523	526	521	522	33700
2005-08-29	525	526	523	526	37500
2005-08-30	527	529	525	525	36000
2005-08-31	525	529	525	528	26600
2005-09-01	529	535	528	529	59600
2005-09-02	530	536	530	536	40200
2005-09-05	540	540	535	539	46100
2005-09-06	539	540	536	538	26300
2005-09-07	540	544	538	540	35000
2005-09-08	546	546	539	541	36300
2005-09-09	542	542	539	539	48800
2005-09-12	544	548	542	548	21200
2005-09-13	555	558	552	558	55100
2005-09-14	560	590	560	588	303300
2005-09-15	588	588	561	568	232500
2005-09-16	570	570	556	560	94700
2005-09-20	560	560	550	555	74600
2005-09-21	558	562	556	559	52800
2005-09-22	558	562	555	556	32100
2005-09-26	552	562	552	559	47600
2005-09-27	557	564	556	559	33100
2005-09-28	558	566	557	566	37500
2005-09-29	568	569	563	563	47500
2005-09-30	563	565	559	559	24800
2005-10-03	560	561	555	560	22700
2005-10-04	557	560	555	558	23600
2005-10-05	555	560	553	555	19200
2005-10-06	553	558	551	551	24900
2005-10-07	553	557	550	554	21000
2005-10-11	551	558	550	555	13600
2005-10-12	556	558	555	557	16500
2005-10-13	556	558	555	556	16600
2005-10-14	556	559	555	557	24200
2005-10-17	560	564	557	559	40200
2005-10-18	563	563	559	561	23500
2005-10-19	561	562	559	560	15300
2005-10-20	560	562	559	561	15000
2005-10-21	560	560	557	559	14400
2005-10-24	559	560	557	557	6100
2005-10-25	557	558	557	558	7400
2005-10-26	558	558	551	551	16800
2005-10-27	551	555	551	555	21000
2005-10-28	559	559	554	556	38500
2005-10-31	545	548	535	545	82200
2005-11-01	542	543	541	541	14300
2005-11-02	540	541	538	540	45500
2005-11-04	539	542	538	542	36700
2005-11-07	538	542	538	540	26000
2005-11-08	540	545	540	545	23300
2005-11-09	547	548	546	548	12800
2005-11-10	548	549	543	546	21700
2005-11-11	546	546	541	544	15900
2005-11-14	544	545	541	541	12700
2005-11-15	541	541	538	539	24300
2005-11-16	538	538	535	537	19000
2005-11-17	535	535	525	534	35000
2005-11-18	534	535	530	530	41500
2005-11-21	530	534	530	534	24200
2005-11-22	533	535	531	532	17800
2005-11-24	531	532	530	530	44900
2005-11-25	530	531	530	530	15800
2005-11-28	530	532	528	528	27300
2005-11-29	528	530	525	525	29700
2005-11-30	525	530	525	526	26800
2005-12-01	528	529	525	525	29400
2005-12-02	524	528	524	527	36300
2005-12-05	529	530	527	529	44000
2005-12-06	530	534	530	532	54300
2005-12-07	533	545	532	536	71800
2005-12-08	540	542	537	537	26400
2005-12-09	540	541	536	540	18900
2005-12-12	540	540	536	537	33000
2005-12-13	536	540	533	538	36900
2005-12-14	539	540	533	535	32900
2005-12-15	535	535	531	533	14100
2005-12-16	533	534	530	532	22400
2005-12-19	529	530	528	529	21800
2005-12-20	527	532	525	528	40700
2005-12-21	528	529	525	529	59800
2005-12-22	530	531	526	530	56800
2005-12-26	531	532	528	529	34800
2005-12-27	525	532	525	532	40100
2005-12-28	532	532	530	530	11400
2005-12-29	530	530	528	530	38200
2005-12-30	531	531	528	529	11900
2006-01-04	529	530	528	530	8700
2006-01-05	530	533	529	533	29500
2006-01-06	535	535	533	534	33500
2006-01-10	537	543	537	543	59100
2006-01-11	544	545	533	537	67900
2006-01-12	538	540	536	537	23000
2006-01-13	538	540	535	535	23100
2006-01-16	536	536	532	535	27400
2006-01-17	533	533	528	528	36500
2006-01-18	521	527	507	510	73200
2006-01-19	503	522	503	516	31200
2006-01-20	533	533	525	525	19500
2006-01-23	520	521	510	515	25600
2006-01-24	513	518	512	518	13200
2006-01-25	518	519	515	515	18300
2006-01-26	516	518	515	517	20900
2006-01-27	517	520	514	520	25800
2006-01-30	525	529	520	527	45400
2006-01-31	526	527	525	526	17200
2006-02-01	527	529	520	523	37200
2006-02-02	529	529	523	527	24600
2006-02-03	524	527	522	523	14300
2006-02-06	523	525	522	525	20100
2006-02-07	522	524	522	523	13500
2006-02-08	523	523	521	522	14000
2006-02-09	525	525	523	525	12400
2006-02-10	527	527	523	524	20400
2006-02-13	520	520	514	516	41100
2006-02-14	508	513	500	513	56400
2006-02-15	510	511	508	509	10400
2006-02-16	511	515	503	513	22400
2006-02-17	514	515	501	506	27800
2006-02-20	508	508	499	500	42100
2006-02-21	496	503	494	503	39000
2006-02-22	503	505	496	496	23200
2006-02-23	500	501	499	499	19400
2006-02-24	502	509	500	507	13700
2006-02-27	506	509	505	507	18300
2006-02-28	507	510	506	510	18100
2006-03-01	513	513	509	511	16000
2006-03-02	512	513	504	504	16800
2006-03-03	505	509	501	501	15000
2006-03-06	501	506	501	502	13800
2006-03-07	503	504	501	502	12700
2006-03-08	500	507	500	505	10100
2006-03-09	502	503	501	503	18600
2006-03-10	504	507	501	505	30300
2006-03-13	510	510	503	507	15400
2006-03-14	508	509	503	505	9900
2006-03-15	507	507	504	504	5000
2006-03-16	505	506	505	505	8600
2006-03-17	507	507	504	505	15200
2006-03-20	509	512	509	509	30500
2006-03-22	510	512	510	512	11000
2006-03-23	512	517	510	517	28800
2006-03-24	517	517	512	512	27300
2006-03-27	515	517	512	516	42700
2006-03-28	511	511	504	505	19700
2006-03-29	506	507	501	506	13700
2006-03-30	506	506	504	506	11800
2006-03-31	506	507	503	507	14800
2006-04-03	508	508	505	506	10200
2006-04-04	507	508	506	507	15900
2006-04-05	508	508	506	506	21700
2006-04-06	506	508	502	502	16400
2006-04-07	503	504	503	504	12100
2006-04-10	505	505	503	503	13000
2006-04-11	504	505	504	504	9700
2006-04-12	503	505	501	503	46100
2006-04-13	503	504	502	502	6700
2006-04-14	503	504	502	503	13900
2006-04-17	501	503	500	502	68900
2006-04-18	501	503	500	502	25200
2006-04-19	504	507	503	505	41100
2006-04-20	511	547	511	547	420000
2006-04-21	531	542	521	539	221000
2006-04-24	535	535	511	530	106300
2006-04-25	529	529	513	513	62200
2006-04-26	514	529	514	526	32300
2006-04-27	527	535	524	535	44100
2006-04-28	526	530	521	530	38600
2006-05-01	530	530	524	525	54900
2006-05-02	528	529	521	526	29900
2006-05-08	527	529	522	525	20400
2006-05-09	525	525	523	524	8500
2006-05-10	522	525	521	521	23900
2006-05-11	525	525	517	517	25100
2006-05-12	515	518	506	512	22500
2006-05-15	514	520	513	515	14700
2006-05-16	524	527	510	513	41300
2006-05-17	505	510	497	506	46800
2006-05-18	501	520	501	505	17000
2006-05-19	518	520	510	515	18500
2006-05-22	530	533	515	524	57700
2006-05-23	520	524	510	518	16800
2006-05-24	510	515	508	515	10800
2006-05-25	516	516	511	513	13100
2006-05-26	513	513	500	503	25000
2006-05-29	506	514	505	514	14400
2006-05-30	512	512	503	508	11800
2006-05-31	501	504	500	501	23200
2006-06-01	501	508	495	497	52100
2006-06-02	492	505	465	486	53700
2006-06-05	496	497	486	492	30600
2006-06-06	487	487	470	487	44900
2006-06-07	489	490	480	483	31100
2006-06-08	465	474	460	464	52900
2006-06-09	463	485	460	480	79500
2006-06-12	481	481	465	473	40500
2006-06-13	479	479	462	463	24500
2006-06-14	463	466	460	461	37100
2006-06-15	467	471	461	463	36100
2006-06-16	468	477	465	475	40300
2006-06-19	476	483	473	478	39100
2006-06-20	482	485	473	484	37000
2006-06-21	485	485	477	478	27400
2006-06-22	478	480	475	478	14100
2006-06-23	489	489	480	486	18300
2006-06-26	490	494	489	493	18100
2006-06-27	495	498	490	493	18400
2006-06-28	494	494	488	493	6300
2006-06-29	493	502	490	495	28500
2006-06-30	495	502	495	497	21600
2006-07-03	500	504	499	500	23900
2006-07-04	500	504	500	501	14900
2006-07-05	500	501	498	498	14400
2006-07-06	499	500	497	499	10400
2006-07-07	500	500	495	495	11500
2006-07-10	495	496	491	496	17800
2006-07-11	496	499	493	499	7600
2006-07-12	499	499	495	499	15100
2006-07-13	495	498	486	487	27500
2006-07-14	493	494	482	493	19200
2006-07-18	485	485	472	479	29300
2006-07-19	477	477	455	468	51000
2006-07-20	495	495	481	492	57500
2006-07-21	492	492	486	487	12000
2006-07-24	487	490	483	485	10900
2006-07-25	491	492	484	486	8200
2006-07-26	485	486	480	480	11100
2006-07-27	475	482	475	477	11700
2006-07-28	480	485	477	483	8200
2006-07-31	484	489	482	487	34700
2006-08-01	488	488	481	482	11700
2006-08-02	487	489	480	480	51300
2006-08-03	485	490	484	488	20100
2006-08-04	490	496	486	496	33700
2006-08-07	493	493	480	481	37200
2006-08-08	483	488	481	483	16800
2006-08-09	483	489	483	486	7000
2006-08-10	490	490	486	490	13400
2006-08-11	490	493	486	488	14300
2006-08-14	494	494	489	490	6900
2006-08-15	491	495	491	492	7400
2006-08-16	492	496	492	494	22700
2006-08-17	494	495	491	492	13000
2006-08-18	495	495	490	490	30700
2006-08-21	494	499	490	499	21600
2006-08-22	499	500	492	498	23000
2006-08-23	499	499	496	496	10500
2006-08-24	496	497	495	497	9900
2006-08-25	497	498	495	497	22400
2006-08-28	498	498	497	497	8100
2006-08-29	498	498	495	495	9400
2006-08-30	497	498	495	495	3500
2006-08-31	496	496	493	493	9100
2006-09-01	494	497	494	496	6500
2006-09-04	495	498	494	497	11600
2006-09-05	498	498	495	497	15600
2006-09-06	493	497	492	496	10000
2006-09-07	496	497	494	497	7500
2006-09-08	497	497	491	492	10000
2006-09-11	497	497	493	495	4100
2006-09-12	495	495	492	493	12000
2006-09-13	494	495	490	491	18000
2006-09-14	490	490	486	486	9100
2006-09-15	490	490	487	488	7900
2006-09-19	490	491	488	490	13300
2006-09-20	491	491	489	490	13800
2006-09-21	491	491	489	490	5400
2006-09-22	492	492	488	489	15000
2006-09-25	489	489	486	486	12100
2006-09-26	486	490	479	479	15100
2006-09-27	469	488	469	476	17300
2006-09-28	481	487	480	485	5100
2006-09-29	484	485	483	483	4200
2006-10-02	481	490	481	490	8000
2006-10-03	489	489	483	483	6100
2006-10-04	490	490	481	481	4900
2006-10-05	484	485	481	481	9900
2006-10-06	485	488	481	488	8700
2006-10-10	487	487	481	486	8400
2006-10-11	483	485	477	479	3200
2006-10-12	477	483	475	475	17800
2006-10-13	479	484	477	483	6400
2006-10-16	484	484	480	482	6500
2006-10-17	482	483	477	477	16700
2006-10-18	476	478	475	476	5800
2006-10-19	477	479	475	479	9400
2006-10-20	479	479	477	479	9800
2006-10-23	482	483	478	483	8600
2006-10-24	483	483	480	481	8100
2006-10-25	480	482	477	477	8600
2006-10-26	478	480	477	479	2800
2006-10-27	478	481	476	479	9900
2006-10-30	480	480	471	477	12200
2006-10-31	480	480	474	474	5200
2006-11-01	475	477	474	475	3800
2006-11-02	475	476	473	474	14000
2006-11-06	475	475	470	474	6200
2006-11-07	474	475	472	474	6900
2006-11-08	473	474	470	473	6200
2006-11-09	471	471	470	470	10000
2006-11-10	471	472	469	472	9600
2006-11-13	474	474	469	469	9300
2006-11-14	472	472	469	469	7600
2006-11-15	472	472	468	469	3400
2006-11-16	469	470	465	467	4600
2006-11-17	465	469	456	464	13600
2006-11-20	462	462	455	455	9000
2006-11-21	454	455	451	452	4500
2006-11-22	452	457	450	457	8300
2006-11-24	450	457	450	457	15300
2006-11-27	455	456	453	456	13700
2006-11-28	457	457	455	457	4600
2006-11-29	455	458	455	457	6000
2006-11-30	455	458	451	458	16500
2006-12-01	453	458	452	458	10000
2006-12-04	458	458	455	455	5500
2006-12-05	456	458	455	457	18100
2006-12-06	458	459	450	459	23800
2006-12-07	458	458	454	456	11200
2006-12-08	459	465	456	461	30700
2006-12-11	461	468	460	465	19700
2006-12-12	468	468	465	465	6500
2006-12-13	463	468	460	464	11400
2006-12-14	466	475	464	469	29100
2006-12-15	475	476	466	475	24300
2006-12-18	478	478	468	468	18000
2006-12-19	466	469	462	462	15600
2006-12-20	457	465	457	463	18400
2006-12-21	465	467	461	465	14600
2006-12-22	464	466	460	464	19200
2006-12-25	464	465	460	460	11800
2006-12-26	462	465	460	461	9700
2006-12-27	461	465	461	464	2600
2006-12-28	464	464	463	464	2800
2006-12-29	464	466	464	465	3600
2007-01-04	466	467	465	466	3300
2007-01-05	468	468	462	466	11500
2007-01-09	468	470	467	467	10700
2007-01-10	466	469	466	466	4700
2007-01-11	470	470	467	469	8900
2007-01-12	470	470	466	466	8700
2007-01-15	470	472	466	467	10100
2007-01-16	469	471	468	471	12500
2007-01-17	472	472	468	469	8700
2007-01-18	470	473	469	472	12700
2007-01-19	475	475	470	471	22000
2007-01-22	465	472	464	470	49000
2007-01-23	467	471	467	469	16600
2007-01-24	471	471	465	465	20900
2007-01-25	465	467	465	465	12100
2007-01-26	465	470	464	470	12000
2007-01-29	466	471	464	465	20800
2007-01-30	465	467	465	466	7800
2007-01-31	465	467	463	467	14000
2007-02-01	465	467	464	465	9400
2007-02-02	469	469	465	466	19900
2007-02-05	465	469	465	468	16500
2007-02-06	467	468	466	468	12500
2007-02-07	466	468	466	466	3500
2007-02-08	465	467	465	466	16200
2007-02-09	465	467	464	466	4000
2007-02-13	465	465	464	465	6700
2007-02-14	464	466	463	465	6400
2007-02-15	465	467	464	466	8200
2007-02-16	466	467	464	465	7400
2007-02-19	465	465	463	464	8200
2007-02-20	466	466	460	464	10000
2007-02-21	462	464	461	464	7700
2007-02-22	460	465	460	463	9400
2007-02-23	463	464	462	464	8500
2007-02-26	463	465	463	463	7300
2007-02-27	463	465	462	464	13900
2007-02-28	461	464	455	458	27900
2007-03-01	462	462	458	459	5200
2007-03-02	460	460	459	459	4700
2007-03-05	459	460	456	457	10400
2007-03-06	460	460	454	457	14800
2007-03-07	459	459	453	453	7900
2007-03-08	454	457	453	456	8300
2007-03-09	457	458	456	456	4600
2007-03-12	456	463	456	461	9300
2007-03-13	465	468	460	462	13500
2007-03-14	460	466	459	460	9400
2007-03-15	462	467	461	466	8800
2007-03-16	464	467	460	466	13700
2007-03-19	465	469	463	467	9900
2007-03-20	468	470	466	470	18000
2007-03-22	460	461	457	457	55500
2007-03-23	457	468	457	463	37400
2007-03-26	464	464	460	460	12800
2007-03-27	455	455	453	454	7900
2007-03-28	455	456	450	450	17300
2007-03-29	450	454	448	448	16300
2007-03-30	447	453	447	452	7900
2007-04-02	448	455	448	449	12400
2007-04-03	454	454	446	452	6800
2007-04-04	451	451	449	449	1100
2007-04-05	447	449	445	448	6500
2007-04-06	450	450	441	444	8700
2007-04-09	442	444	438	438	23600
2007-04-10	438	438	429	430	31000
2007-04-11	432	435	430	432	11900
2007-04-12	432	440	430	440	31000
2007-04-13	443	444	442	444	10800
2007-04-16	443	445	443	444	8000
2007-04-17	445	445	435	436	18800
2007-04-18	435	437	429	430	7500
2007-04-19	428	430	423	426	11800
2007-04-20	430	434	425	428	9300
2007-04-23	425	430	421	425	13900
2007-04-24	424	425	424	425	5300
2007-04-25	424	426	424	426	8100
2007-04-26	426	426	419	425	32800
2007-04-27	420	425	420	425	26200
2007-05-01	425	449	425	436	42900
2007-05-02	440	449	433	444	38100
2007-05-07	435	436	431	433	25000
2007-05-08	439	439	430	431	37000
2007-05-09	432	432	430	431	22700
2007-05-10	435	435	431	431	5800
2007-05-11	431	431	427	427	11200
2007-05-14	436	436	429	430	33500
2007-05-15	431	432	428	429	7200
2007-05-16	430	430	423	424	21300
2007-05-17	423	423	420	421	5900
2007-05-18	420	421	420	421	11600
2007-05-21	418	425	418	424	11200
2007-05-22	422	424	418	423	5100
2007-05-23	420	430	420	429	7200
2007-05-24	426	428	426	427	5000
2007-05-25	427	430	425	425	20400
2007-05-28	424	427	423	425	14000
2007-05-29	425	426	424	424	6400
2007-05-30	424	424	423	424	6200
2007-05-31	423	428	423	428	11800
2007-06-01	429	429	424	429	6700
2007-06-04	428	430	427	427	5800
2007-06-05	428	430	424	430	38400
2007-06-06	429	430	423	429	18000
2007-06-07	429	430	426	426	11200
2007-06-08	423	431	423	431	9500
2007-06-11	431	432	430	431	15100
2007-06-12	432	432	429	430	7800
2007-06-13	428	430	428	430	14100
2007-06-14	432	432	427	427	6000
2007-06-15	428	432	428	431	3200
2007-06-18	430	433	430	432	8000
2007-06-19	432	433	430	432	11000
2007-06-20	435	436	433	433	11100
2007-06-21	433	435	432	434	13900
2007-06-22	431	434	430	434	10200
2007-06-25	433	434	430	430	7800
2007-06-26	432	433	429	430	10400
2007-06-27	430	431	430	431	4400
2007-06-28	431	431	430	430	4000
2007-06-29	432	432	425	431	8900
2007-07-02	431	432	429	429	10200
2007-07-03	426	432	426	430	9600
2007-07-04	430	431	429	430	4100
2007-07-05	431	431	429	431	4700
2007-07-06	429	431	429	430	3900
2007-07-09	430	431	429	430	5200
2007-07-10	426	431	426	427	15900
2007-07-11	428	431	428	430	4800
2007-07-12	425	430	425	427	20700
2007-07-13	425	431	425	425	9000
2007-07-17	421	424	421	423	4800
2007-07-18	423	427	423	427	4000
2007-07-19	427	430	425	430	3100
2007-07-20	430	430	428	428	4100
2007-07-23	428	428	425	425	4000
2007-07-24	426	426	424	425	7800
2007-07-25	425	426	425	425	1700
2007-07-26	425	425	423	423	2500
2007-07-27	423	423	420	421	10400
2007-07-30	422	422	396	415	13900
2007-07-31	420	426	415	424	8600
2007-08-01	420	420	411	412	11600
2007-08-02	427	427	417	420	11900
2007-08-03	421	421	413	418	2200
2007-08-06	411	412	410	412	12600
2007-08-07	410	412	410	410	5700
2007-08-08	411	412	410	410	8100
2007-08-09	410	411	401	402	17900
2007-08-10	400	401	400	400	27900
2007-08-13	391	395	386	395	21500
2007-08-14	392	393	385	386	11500
2007-08-15	383	383	380	382	6500
2007-08-16	380	381	375	375	18100
2007-08-17	375	376	368	368	16200
2007-08-20	383	393	378	378	13600
2007-08-21	379	395	377	395	16000
2007-08-22	400	405	400	403	15700
2007-08-23	400	401	384	384	14600
2007-08-24	397	400	384	396	9700
2007-08-27	403	403	387	388	13200
2007-08-28	388	389	370	380	21900
2007-08-29	380	386	380	380	7300
2007-08-30	381	381	380	380	5900
2007-08-31	381	381	380	381	4600
2007-09-03	381	382	380	380	7100
2007-09-04	381	381	380	380	19000
2007-09-05	384	386	368	370	20300
2007-09-06	371	374	368	373	10900
2007-09-07	374	374	370	371	7300
2007-09-10	371	372	370	370	4100
2007-09-11	370	370	367	368	6100
2007-09-12	368	369	368	368	3900
2007-09-13	364	364	358	359	16500
2007-09-14	358	363	338	363	26400
2007-09-18	359	359	347	357	6500
2007-09-19	356	357	355	357	9900
2007-09-20	350	357	345	357	21800
2007-09-21	352	357	339	345	10200
2007-09-25	346	346	336	344	15300
2007-09-26	343	357	343	356	9900
2007-09-27	355	356	355	356	7300
2007-09-28	357	357	355	355	7800
2007-10-01	354	356	354	355	39300
2007-10-02	357	357	345	355	14900
2007-10-03	355	357	348	352	23100
2007-10-04	353	356	350	356	19000
2007-10-05	355	356	351	351	8800
2007-10-09	352	357	352	356	12000
2007-10-10	356	365	356	364	62600
2007-10-11	381	386	371	375	28300
2007-10-12	381	385	376	382	31700
2007-10-15	382	385	380	381	18500
2007-10-16	385	385	379	379	17900
2007-10-17	379	381	378	381	21300
2007-10-18	378	379	378	379	5400
2007-10-19	380	380	378	379	11400
2007-10-22	380	380	361	378	38600
2007-10-23	368	380	367	380	19400
2007-10-24	373	376	371	373	15900
2007-10-25	370	376	364	373	25200
2007-10-26	373	374	373	373	3000
2007-10-29	373	373	371	371	7200
2007-10-30	370	371	368	369	4100
2007-10-31	369	369	365	365	5900
2007-11-01	368	369	362	363	11000
2007-11-02	363	365	358	364	11300
2007-11-05	368	368	345	365	17600
2007-11-06	365	366	365	365	5000
2007-11-07	365	367	362	362	6200
2007-11-08	355	367	352	363	16000
2007-11-09	360	363	360	363	2600
2007-11-12	357	365	351	365	12500
2007-11-13	355	365	355	363	7000
2007-11-14	358	365	358	365	4500
2007-11-15	362	365	362	365	7700
2007-11-16	360	368	360	366	17600
2007-11-19	363	367	363	366	4600
2007-11-20	375	375	362	368	8900
2007-11-21	368	371	365	367	2600
2007-11-22	371	372	369	371	3200
2007-11-26	373	379	370	370	4000
2007-11-27	371	377	371	375	1700
2007-11-28	376	379	375	378	3200
2007-11-29	380	388	378	379	16400
2007-11-30	389	389	380	385	7900
2007-12-03	385	388	384	385	3900
2007-12-04	389	390	385	385	8700
2007-12-05	390	390	387	388	3900
2007-12-06	390	390	385	387	6000
2007-12-07	390	390	385	385	4900
2007-12-10	385	387	382	382	4600
2007-12-11	381	386	375	383	20300
2007-12-12	380	389	377	383	11800
2007-12-13	377	378	360	362	17200
2007-12-14	369	384	369	380	13000
2007-12-17	379	380	376	378	7300
2007-12-18	371	380	371	374	11500
2007-12-19	375	380	372	377	9400
2007-12-20	380	380	370	375	11000
2007-12-21	366	370	355	365	31300
2007-12-25	378	394	378	380	66300
2007-12-26	384	395	384	392	37100
2007-12-27	396	399	391	398	31800
2007-12-28	397	400	392	393	12700
2008-01-04	398	398	391	398	12100
2008-01-07	399	429	399	404	32300
2008-01-08	409	415	409	414	21800
2008-01-09	410	413	404	411	24500
2008-01-10	407	408	403	403	9500
2008-01-11	400	405	390	402	15200
2008-01-15	397	401	385	385	19300
2008-01-16	370	381	350	380	20300
2008-01-17	379	385	367	379	10200
2008-01-18	385	385	377	384	12900
2008-01-21	380	384	370	380	10200
2008-01-22	371	372	363	371	12000
2008-01-23	366	372	360	365	24300
2008-01-24	365	369	364	368	9100
2008-01-25	368	375	368	372	15400
2008-01-28	375	380	374	378	4300
2008-01-29	380	382	380	380	8800
2008-01-30	384	385	377	380	13800
2008-01-31	380	380	376	376	2200
2008-02-01	380	385	375	379	8000
2008-02-04	390	390	378	378	11300
2008-02-05	381	381	376	377	4000
2008-02-06	372	373	365	365	12200
2008-02-07	365	365	360	361	7800
2008-02-08	363	369	360	362	6700
2008-02-12	352	385	350	375	40800
2008-02-13	375	385	365	373	25700
2008-02-14	380	388	373	378	10600
2008-02-15	379	379	369	369	10300
2008-02-18	372	372	366	368	8600
2008-02-19	372	372	368	371	9000
2008-02-20	376	376	363	369	12600
2008-02-21	370	375	368	371	5500
2008-02-22	370	371	366	367	10600
2008-02-25	366	371	364	367	9800
2008-02-26	366	369	365	365	6300
2008-02-27	369	369	365	367	3800
2008-02-28	367	368	363	364	5500
2008-02-29	365	366	362	363	3500
2008-03-03	363	363	358	359	5400
2008-03-04	354	355	351	354	15500
2008-03-05	358	359	352	354	5700
2008-03-06	356	361	355	357	3600
2008-03-07	357	357	351	351	4600
2008-03-10	352	352	350	350	3200
2008-03-11	347	347	335	340	11000
2008-03-12	341	352	341	347	5000
2008-03-13	347	347	340	340	9400
2008-03-14	345	345	337	337	8400
2008-03-17	333	345	331	331	11800
2008-03-18	334	335	331	335	8400
2008-03-19	345	350	330	350	24000
2008-03-21	348	349	342	346	3100
2008-03-24	351	351	347	350	3100
2008-03-25	352	359	349	352	6300
2008-03-26	322	347	320	325	14900
2008-03-27	331	350	331	349	10500
2008-03-28	349	349	339	348	7800
2008-03-31	328	348	328	347	17800
2008-04-01	346	348	336	347	13100
2008-04-02	349	349	340	347	42400
2008-04-03	347	350	345	349	19000
2008-04-04	343	350	327	349	31400
2008-04-07	350	355	346	348	14800
2008-04-08	352	360	351	360	12300
2008-04-09	355	370	350	368	31000
2008-04-10	358	368	356	363	15800
2008-04-11	361	368	361	362	7600
2008-04-14	363	369	357	369	11100
2008-04-15	364	367	354	367	18900
2008-04-16	366	366	360	364	19900
2008-04-17	360	363	355	360	47000
2008-04-18	363	363	358	362	8400
2008-04-21	359	365	355	361	28700
2008-04-22	360	370	356	370	21900
2008-04-23	366	368	360	368	18900
2008-04-24	368	368	361	368	11200
2008-04-25	367	368	365	368	5200
2008-04-28	369	370	364	370	8800
2008-04-30	369	369	364	368	5600
2008-05-01	367	369	360	362	29200
2008-05-02	361	380	357	380	42700
2008-05-07	388	400	383	400	44600
2008-05-08	400	410	396	410	60400
2008-05-09	405	420	386	417	67500
2008-05-12	437	439	409	412	127200
2008-05-13	409	437	402	433	130900
2008-05-14	414	414	364	377	477000
2008-05-15	377	379	369	369	111500
2008-05-16	371	374	367	372	34700
2008-05-19	375	378	373	375	40900
2008-05-20	376	376	362	367	55400
2008-05-21	362	372	362	372	34300
2008-05-22	370	370	363	366	13200
2008-05-23	364	367	363	367	26500
2008-05-26	368	368	364	364	10200
2008-05-27	365	369	362	369	20000
2008-05-28	368	368	354	360	30300
2008-05-29	360	364	357	363	12500
2008-05-30	362	362	354	354	21300
2008-06-02	362	369	354	359	57500
2008-06-03	358	364	356	364	15100
2008-06-04	363	363	359	359	26300
2008-06-05	362	362	360	362	23700
2008-06-06	363	365	361	361	31300
2008-06-09	358	363	357	363	21700
2008-06-10	363	363	359	360	12000
2008-06-11	361	363	360	361	9100
2008-06-12	362	362	356	357	20900
2008-06-13	362	362	356	356	16000
2008-06-16	353	353	346	349	34600
2008-06-17	350	360	349	353	35100
2008-06-18	355	359	353	359	9800
2008-06-19	358	360	355	359	38200
2008-06-20	362	362	356	362	20200
2008-06-23	357	362	356	359	13600
2008-06-24	362	362	359	359	7400
2008-06-25	357	360	355	358	6300
2008-06-26	360	360	358	359	27600
2008-06-27	356	361	356	358	26900
2008-06-30	355	358	355	357	5400
2008-07-01	358	362	356	356	14300
2008-07-02	358	359	352	353	14500
2008-07-03	353	357	351	357	11600
2008-07-04	359	359	355	355	9200
2008-07-07	359	359	354	356	6900
2008-07-08	358	358	350	357	16900
2008-07-09	355	355	352	355	15900
2008-07-10	351	354	351	351	11500
2008-07-11	353	353	351	351	11300
2008-07-14	351	353	351	352	11400
2008-07-15	352	352	350	352	15800
2008-07-16	351	352	350	350	6600
2008-07-17	352	354	352	354	4800
2008-07-18	356	356	345	345	22300
2008-07-22	354	354	350	351	7400
2008-07-23	354	356	353	356	13100
2008-07-24	356	360	355	356	14300
2008-07-25	355	383	355	377	44700
2008-07-28	378	388	376	380	64200
2008-07-29	380	382	370	375	27200
2008-07-30	370	372	367	372	6400
2008-07-31	369	372	368	370	3200
2008-08-01	375	375	367	371	9000
2008-08-04	381	381	365	370	15500
2008-08-05	370	370	363	365	11000
2008-08-06	370	372	365	365	10800
2008-08-07	370	372	360	370	18100
2008-08-08	369	373	364	373	17800
2008-08-11	373	373	367	369	24700
2008-08-12	367	367	360	360	12000
2008-08-13	362	364	359	360	23000
2008-08-14	365	365	357	360	8400
2008-08-15	358	363	355	362	13300
2008-08-18	364	364	355	363	10300
2008-08-19	364	369	362	369	17000
2008-08-20	365	369	364	365	7400
2008-08-21	365	365	361	361	3400
2008-08-22	360	368	359	367	14600
2008-08-25	368	369	364	368	4100
2008-08-26	368	368	362	363	7300
2008-08-27	373	373	368	372	31300
2008-08-28	374	375	369	372	15300
2008-08-29	371	372	365	370	19900
2008-09-01	370	370	365	365	8100
2008-09-02	365	370	365	370	9100
2008-09-03	370	370	368	369	8800
2008-09-04	365	370	356	370	32200
2008-09-05	366	369	356	362	19600
2008-09-08	364	375	362	375	33900
2008-09-09	373	373	355	365	31500
2008-09-10	360	360	355	359	17900
2008-09-11	360	363	355	360	17900
2008-09-12	360	362	355	362	7500
2008-09-16	349	360	342	353	43500
2008-09-17	361	361	353	359	12400
2008-09-18	356	359	354	359	16200
2008-09-19	359	360	356	359	12700
2008-09-22	358	359	345	353	31800
2008-09-24	354	358	351	354	15500
2008-09-25	355	355	351	354	5300
2008-09-26	358	358	352	354	10800
2008-09-29	355	355	349	350	13800
2008-09-30	340	345	333	343	47800
2008-10-01	345	346	340	344	9600
2008-10-02	345	349	339	339	22700
2008-10-03	338	342	335	336	17400
2008-10-06	335	338	330	332	135000
2008-10-07	330	335	320	331	25400
2008-10-08	324	338	320	331	48800
2008-10-09	321	334	319	330	49400
2008-10-10	324	329	305	329	24700
2008-10-14	340	350	340	348	27800
2008-10-15	347	347	344	344	9700
2008-10-16	339	340	333	339	17900
2008-10-17	341	343	339	343	22800
2008-10-20	346	353	340	351	49500
2008-10-21	353	356	348	355	25300
2008-10-22	357	363	355	357	47100
2008-10-23	356	359	354	358	42200
2008-10-24	358	360	357	358	39000
2008-10-27	356	356	349	349	55700
2008-10-28	349	349	346	348	35200
2008-10-29	355	363	349	358	54900
2008-10-30	363	368	357	363	59800
2008-10-31	363	365	355	364	42300
2008-11-04	360	364	355	361	5000
2008-11-05	363	365	360	364	10200
2008-11-06	357	363	357	358	11800
2008-11-07	355	360	354	359	12200
2008-11-10	357	365	350	357	29800
2008-11-11	359	360	355	355	9100
2008-11-12	357	360	352	353	8100
2008-11-13	355	356	351	351	12800
2008-11-14	351	354	350	354	12400
2008-11-17	350	355	350	353	9000
2008-11-18	353	355	352	355	7000
2008-11-19	351	358	351	358	28000
2008-11-20	353	356	340	352	23000
2008-11-21	352	354	345	354	19900
2008-11-25	355	355	347	351	5000
2008-11-26	350	350	335	342	8300
2008-11-27	342	344	340	340	2800
2008-11-28	344	348	342	343	5900
2008-12-01	343	343	340	342	4000
2008-12-02	342	344	339	344	4800
2008-12-03	342	342	338	340	4900
2008-12-04	340	341	339	340	7000
2008-12-05	342	342	337	340	6800
2008-12-08	338	340	335	339	9300
2008-12-09	340	340	336	336	7100
2008-12-10	334	340	334	340	14800
2008-12-11	337	340	335	340	5600
2008-12-12	340	341	336	336	7200
2008-12-15	339	341	338	338	4900
2008-12-16	341	342	339	339	4000
2008-12-17	342	347	337	341	12200
2008-12-18	344	348	343	347	7800
2008-12-19	349	349	343	347	9800
2008-12-22	347	350	344	346	18400
2008-12-24	347	348	345	348	3600
2008-12-25	350	350	345	345	6100
2008-12-26	349	349	345	347	5300
2008-12-29	345	348	344	345	3600
2008-12-30	342	345	342	343	3200
2009-01-05	350	350	343	343	4500
2009-01-06	345	346	342	343	4900
2009-01-07	346	349	341	344	11600
2009-01-08	342	345	340	343	7100
2009-01-09	343	344	341	341	2800
2009-01-13	341	341	340	341	4600
2009-01-14	341	342	340	341	2700
2009-01-15	340	341	339	341	3200
2009-01-16	340	342	335	341	9900
2009-01-19	341	342	339	341	10200
2009-01-20	341	341	336	336	12300
2009-01-21	336	340	335	337	10900
2009-01-22	333	343	333	341	25300
2009-01-23	338	342	338	342	4200
2009-01-26	342	344	338	344	4400
2009-01-27	342	342	340	340	1800
2009-01-28	336	342	336	338	7600
2009-01-29	334	340	334	338	13900
2009-01-30	337	340	335	338	3800
2009-02-02	336	339	336	339	4000
2009-02-03	343	345	340	340	16100
2009-02-04	341	341	340	340	6500
2009-02-05	344	344	340	340	3700
2009-02-06	342	344	340	344	5200
2009-02-09	340	346	339	343	20400
2009-02-10	343	347	343	345	13500
2009-02-12	347	351	347	350	11500
2009-02-13	351	365	351	362	32800
2009-02-16	352	363	352	359	17800
2009-02-17	360	365	359	361	5400
2009-02-18	360	360	359	359	7800
2009-02-19	356	365	356	365	9000
2009-02-20	365	365	359	365	22900
2009-02-23	353	365	350	365	9700
2009-02-24	364	364	356	359	18400
2009-02-25	358	360	352	357	12300
2009-02-26	352	358	352	358	6000
2009-02-27	358	359	355	359	4000
2009-03-02	359	362	355	355	9300
2009-03-03	355	360	355	357	5700
2009-03-04	353	360	353	360	4600
2009-03-05	359	360	359	360	5000
2009-03-06	359	361	359	360	4400
2009-03-09	357	361	356	359	9900
2009-03-10	355	360	355	356	11300
2009-03-11	357	359	355	358	10300
2009-03-12	356	359	355	356	8200
2009-03-13	356	359	356	359	10000
2009-03-16	356	360	356	358	4500
2009-03-17	381	416	381	403	155900
2009-03-18	402	410	398	409	100000
2009-03-19	404	409	400	407	33900
2009-03-23	407	412	400	412	33400
2009-03-24	412	438	412	435	70000
2009-03-25	426	445	426	442	163300
2009-03-26	350	362	350	356	60300
2009-03-27	360	360	340	340	35700
2009-03-30	342	342	323	329	22400
2009-03-31	325	329	324	329	34500
2009-04-01	326	332	326	330	25400
2009-04-02	332	334	325	329	31500
2009-04-03	329	333	326	328	13900
2009-04-06	330	330	326	330	24400
2009-04-07	330	333	329	333	17500
2009-04-08	333	333	330	330	20000
2009-04-09	330	333	327	333	8600
2009-04-10	334	335	331	332	7600
2009-04-13	377	381	355	371	135500
2009-04-14	370	371	366	370	59100
2009-04-15	368	370	366	370	63200
2009-04-16	369	372	366	371	61200
2009-04-17	372	380	372	380	84800
2009-04-20	381	385	380	384	120800
2009-04-21	383	383	379	381	33300
2009-04-22	381	382	380	381	42500
2009-04-23	381	381	379	381	42000
2009-04-24	380	380	377	377	68800
2009-04-27	376	379	375	377	72600
2009-04-28	376	376	361	364	72700
2009-04-30	364	366	363	364	45300
2009-05-01	364	372	364	371	60000
2009-05-07	372	376	372	375	27600
2009-05-08	376	381	376	381	60300
2009-05-11	381	383	380	381	22100
2009-05-12	382	385	381	385	48400
2009-05-13	383	385	378	380	4700
2009-05-14	382	382	371	376	15800
2009-05-15	375	375	364	373	14100
2009-05-18	372	372	370	371	7300
2009-05-19	366	367	362	364	14000
2009-05-20	365	366	354	355	22700
2009-05-21	355	374	355	373	49700
2009-05-22	375	381	375	381	66400
2009-05-25	382	386	377	377	79700
2009-05-26	382	383	378	380	18900
2009-05-27	383	383	376	380	23300
2009-05-28	383	385	380	385	41600
2009-05-29	384	384	379	381	21200
2009-06-01	377	382	373	378	45900
2009-06-02	377	377	369	374	36600
2009-06-03	374	374	369	373	25300
2009-06-04	372	374	369	371	28900
2009-06-05	371	372	370	372	11100
2009-06-08	372	372	371	372	13900
2009-06-09	371	372	370	371	9900
2009-06-10	371	374	371	374	6800
2009-06-11	374	375	372	374	5900
2009-06-12	374	388	373	379	59000
2009-06-15	386	386	380	381	25000
2009-06-16	383	383	379	379	16700
2009-06-17	375	380	375	380	8500
2009-06-18	383	383	376	379	9600
2009-06-19	379	380	375	377	14200
2009-06-22	375	376	369	372	16500
2009-06-23	367	370	366	367	12100
2009-06-24	368	378	368	372	17500
2009-06-25	377	378	371	377	10400
2009-06-26	376	377	375	377	6900
2009-06-29	376	377	375	376	5700
2009-06-30	377	379	375	378	7500
2009-07-01	375	375	367	371	15000
2009-07-02	376	376	372	374	4600
2009-07-03	366	375	364	373	18000
2009-07-06	375	375	370	372	8600
2009-07-07	370	371	367	367	7000
2009-07-08	365	370	355	367	12900
2009-07-09	365	367	358	360	4400
2009-07-10	361	374	361	372	22900
2009-07-13	372	372	355	357	21900
2009-07-14	353	365	353	358	6500
2009-07-15	359	364	355	355	9000
2009-07-16	355	364	355	362	5100
2009-07-17	365	369	362	369	13800
2009-07-21	367	375	365	374	15500
2009-07-22	373	379	371	371	30300
2009-07-23	372	374	372	373	4100
2009-07-24	374	374	370	370	7200
2009-07-27	366	370	366	370	5100
2009-07-28	370	371	367	367	4700
2009-07-29	366	369	366	366	2400
2009-07-30	366	368	366	366	3800
2009-07-31	367	370	367	370	3700
2009-08-03	367	372	367	370	4700
2009-08-04	374	374	368	370	14100
2009-08-05	379	409	378	395	237600
2009-08-06	400	408	400	402	130200
2009-08-07	400	401	393	400	53200
2009-08-10	395	405	394	398	44900
2009-08-11	400	402	397	397	16900
2009-08-12	399	400	396	398	26700
2009-08-13	399	400	398	398	13000
2009-08-14	402	414	398	406	87100
2009-08-17	410	423	408	412	84300
2009-08-18	410	418	410	411	28500
2009-08-19	415	421	413	416	56200
2009-08-20	420	435	420	435	98100
2009-08-21	435	440	420	429	93300
2009-08-24	430	430	421	423	34700
2009-08-25	424	430	420	429	57000
2009-08-26	430	447	430	444	95100
2009-08-27	444	449	435	437	117900
2009-08-28	457	465	451	458	190500
2009-08-31	458	462	445	454	147100
2009-09-01	454	461	450	454	75900
2009-09-02	451	457	448	450	70500
2009-09-03	449	452	440	448	34500
2009-09-04	438	450	423	426	54200
2009-09-07	426	435	410	410	38600
2009-09-08	406	415	402	413	35300
2009-09-09	388	413	388	410	163900
2009-09-10	405	426	405	417	47600
2009-09-11	427	429	421	429	23700
2009-09-14	435	437	421	421	58700
2009-09-15	419	426	411	415	18500
2009-09-16	405	414	402	408	31000
2009-09-17	408	415	404	409	27700
2009-09-18	401	411	398	409	31700
2009-09-24	419	433	416	426	42300
2009-09-25	426	428	418	425	23400
2009-09-28	420	425	415	417	30200
2009-09-29	413	421	411	418	14600
2009-09-30	415	421	409	420	31600
2009-10-01	416	435	411	431	37000
2009-10-02	421	430	416	428	19900
2009-10-05	430	430	417	428	9500
2009-10-06	423	427	419	427	8600
2009-10-07	422	427	422	427	12600
2009-10-08	427	432	421	428	18000
2009-10-09	430	446	429	439	77400
2009-10-13	442	450	442	448	67400
2009-10-14	443	450	437	444	39000
2009-10-15	456	467	454	461	170600
2009-10-16	460	465	452	456	72000
2009-10-19	454	456	440	452	51800
2009-10-20	450	453	448	452	20500
2009-10-21	447	454	447	454	15300
2009-10-22	452	457	449	457	17800
2009-10-23	462	472	456	457	115200
2009-10-26	457	465	457	465	28900
2009-10-27	464	465	458	459	26200
2009-10-28	451	458	450	455	52600
2009-10-29	447	450	446	450	14800
2009-10-30	449	463	445	447	29300
2009-11-02	438	440	426	439	31000
2009-11-04	440	440	427	433	12300
2009-11-05	434	434	421	424	17500
2009-11-06	427	428	412	427	33900
2009-11-09	425	425	411	411	43900
2009-11-10	414	419	412	417	28300
2009-11-11	418	419	415	416	21400
2009-11-12	415	419	410	413	33200
2009-11-13	410	417	409	411	22700
2009-11-16	409	412	399	402	39400
2009-11-17	395	402	393	402	19400
2009-11-18	400	400	381	394	20100
2009-11-19	379	393	374	392	17700
2009-11-20	393	400	389	400	15700
2009-11-24	399	399	395	398	7000
2009-11-25	395	396	391	395	8900
2009-11-26	393	394	392	392	1700
2009-11-27	389	394	389	390	7800
2009-11-30	385	391	385	389	11100
2009-12-01	394	394	380	385	21200
2009-12-02	385	392	382	390	8200
2009-12-03	386	397	386	397	11100
2009-12-04	396	398	392	397	5400
2009-12-07	403	410	400	410	12400
2009-12-08	411	411	405	405	11400
2009-12-09	405	407	405	406	2000
2009-12-10	406	408	396	396	11100
2009-12-11	399	404	399	401	4800
2009-12-14	402	405	400	402	6500
2009-12-15	401	404	400	403	2500
2009-12-16	403	403	398	402	8400
2009-12-17	401	403	401	403	5600
2009-12-18	405	405	400	400	8600
2009-12-21	403	403	397	400	8200
2009-12-22	401	402	397	401	8400
2009-12-24	398	402	396	399	13000
2009-12-25	396	400	395	398	17000
2009-12-28	400	402	399	401	9300
2009-12-29	401	402	400	402	13700
2009-12-30	401	404	399	404	16700
2010-01-04	403	405	400	405	14000
2010-01-05	407	407	403	406	8200
2010-01-06	406	409	404	404	23800
2010-01-07	404	410	404	408	13700
2010-01-08	406	410	406	408	7500
2010-01-12	440	449	420	436	68200
2010-01-13	436	437	430	433	21800
2010-01-14	438	440	433	439	24800
2010-01-15	435	443	435	440	40600
2010-01-18	443	443	440	440	31100
2010-01-19	440	441	437	437	34300
2010-01-20	437	438	435	437	21500
2010-01-21	435	438	435	438	7900
2010-01-22	439	439	435	439	26400
2010-01-25	434	439	433	438	6300
2010-01-26	436	439	436	438	5800
2010-01-27	437	440	434	434	30000
2010-01-28	433	438	433	437	9400
2010-01-29	432	438	430	437	28500
2010-02-01	435	454	435	453	53500
2010-02-02	455	455	433	438	67200
2010-02-03	433	436	432	434	10600
2010-02-04	430	438	427	437	16500
2010-02-05	434	437	431	433	10500
2010-02-08	430	437	429	430	11600
2010-02-09	429	437	429	437	8000
2010-02-10	430	440	430	431	8200
2010-02-12	440	445	435	441	65400
2010-02-15	441	441	436	437	5900
2010-02-16	436	440	432	438	10000
2010-02-17	435	439	434	437	32800
2010-02-18	435	438	430	438	14100
2010-02-19	438	439	433	434	10500
2010-02-22	431	435	430	434	9400
2010-02-23	431	432	431	432	5100
2010-02-24	430	435	430	432	9900
2010-02-25	429	430	428	429	17200
2010-02-26	427	428	427	427	5100
2010-03-01	425	427	423	424	6600
2010-03-02	423	424	422	422	18100
2010-03-03	422	422	419	420	18200
2010-03-04	419	425	419	420	22400
2010-03-05	425	425	421	424	6900
2010-03-08	426	430	424	426	14800
2010-03-09	426	430	426	428	8000
2010-03-10	430	431	427	431	16400
2010-03-11	430	438	430	436	15400
2010-03-12	439	442	439	441	14600
2010-03-15	443	443	439	441	57900
2010-03-16	441	444	441	443	17100
2010-03-17	442	449	442	449	19800
2010-03-18	448	449	446	449	16700
2010-03-19	450	453	448	450	41200
2010-03-23	451	458	451	455	28600
2010-03-24	457	462	456	459	51400
2010-03-25	463	465	460	464	91300
2010-03-26	463	470	458	469	44200
2010-03-29	465	465	457	465	39400
2010-03-30	457	460	452	452	25500
2010-03-31	456	459	454	455	5600
2010-04-01	458	460	452	458	44000
2010-04-02	458	470	458	467	72700
2010-04-05	468	469	462	467	69700
2010-04-06	463	467	463	465	20700
2010-04-07	465	468	464	467	17900
2010-04-08	465	475	465	475	42000
2010-04-09	475	484	473	480	80000
2010-04-12	482	488	482	488	46900
2010-04-13	490	515	490	515	181000
2010-04-14	517	540	515	535	250200
2010-04-15	540	550	534	534	229700
2010-04-16	530	530	517	525	112400
2010-04-19	521	550	521	547	132100
2010-04-20	530	531	520	520	149400
2010-04-21	522	528	512	512	81300
2010-04-22	514	518	500	516	86600
2010-04-23	518	528	518	518	42100
2010-04-26	520	528	519	519	32500
2010-04-27	524	528	520	528	45400
2010-04-28	524	530	520	530	34300
2010-04-30	530	530	524	526	34800
2010-05-06	511	520	511	512	42200
2010-05-07	490	506	488	498	56400
2010-05-10	500	530	500	529	50500
2010-05-11	551	555	541	547	558900
2010-05-12	549	552	546	547	119400
2010-05-13	547	547	537	547	113600
2010-05-14	547	547	540	545	101000
2010-05-17	545	545	540	543	93000
2010-05-18	543	546	534	535	123800
2010-05-19	534	536	523	529	130000
2010-05-20	529	529	521	521	68200
2010-05-21	520	535	516	527	163500
2010-05-24	532	532	517	523	26600
2010-05-25	523	525	513	513	79500
2010-05-26	513	514	508	510	53000
2010-05-27	510	530	504	530	66000
2010-05-28	530	540	530	539	116000
2010-05-31	537	551	532	550	77900
2010-06-01	548	548	534	537	25800
2010-06-02	534	534	528	528	67800
2010-06-03	528	532	527	531	64300
2010-06-04	530	530	523	524	49400
2010-06-07	523	523	516	517	54900
2010-06-08	519	524	517	524	30200
2010-06-09	522	524	519	522	42100
2010-06-10	520	522	519	522	28800
2010-06-11	520	522	518	519	28700
2010-06-14	519	523	519	522	38500
2010-06-15	522	531	519	531	48200
2010-06-16	530	535	530	533	40700
2010-06-17	533	539	533	538	15700
2010-06-18	534	536	533	534	13700
2010-06-21	529	533	520	530	12200
2010-06-22	530	536	529	534	38200
2010-06-23	534	536	529	535	21200
2010-06-24	530	533	523	525	6200
2010-06-25	520	534	517	534	24800
2010-06-28	533	535	530	535	17800
2010-06-29	526	534	525	525	9200
2010-06-30	523	523	505	516	13200
2010-07-01	506	512	505	512	8100
2010-07-02	510	523	510	518	13300
2010-07-05	522	533	522	526	20000
2010-07-06	530	534	526	531	14200
2010-07-07	531	537	528	533	17800
2010-07-08	531	536	530	534	9800
2010-07-09	529	533	529	533	3400
2010-07-12	533	533	525	525	7200
2010-07-13	529	532	525	526	5900
2010-07-14	520	525	510	522	18600
2010-07-15	525	526	520	522	9200
2010-07-16	515	521	511	520	29600
2010-07-20	524	525	520	523	20300
2010-07-21	515	520	513	513	12100
2010-07-22	509	514	507	507	8500
2010-07-23	506	508	501	505	11500
2010-07-26	520	525	515	516	20200
2010-07-27	513	518	510	511	8100
2010-07-28	516	518	512	513	8100
2010-07-29	517	523	516	523	19600
2010-07-30	523	526	519	522	13100
2010-08-02	522	533	520	527	14000
2010-08-03	530	530	518	519	29800
2010-08-04	520	529	519	525	20400
2010-08-05	522	530	522	528	35500
2010-08-06	530	530	525	526	17000
2010-08-09	526	527	525	525	12500
2010-08-10	525	528	525	528	13000
2010-08-11	525	525	524	524	14000
2010-08-12	524	532	524	528	44300
2010-08-13	528	530	523	530	10500
2010-08-16	530	530	527	528	17100
2010-08-17	528	528	527	527	8600
2010-08-18	527	534	527	534	20300
2010-08-19	534	535	532	533	8900
2010-08-20	534	534	515	519	13400
2010-08-23	519	524	519	520	9100
2010-08-24	525	525	519	520	11000
2010-08-25	515	524	512	518	27900
2010-08-26	518	522	513	522	11200
2010-08-27	516	521	516	519	6100
2010-08-30	520	521	515	518	9500
2010-08-31	519	520	518	520	35700
2010-09-01	521	524	503	510	32500
2010-09-02	515	517	507	507	21900
2010-09-03	508	515	507	509	25900
2010-09-06	514	515	511	515	17300
2010-09-07	515	518	514	518	14500
2010-09-08	515	520	515	520	13000
2010-09-09	519	520	517	518	19500
2010-09-10	518	520	511	518	9800
2010-09-13	520	520	514	516	20800
2010-09-14	515	515	505	506	39900
2010-09-15	509	512	500	510	30000
2010-09-16	509	509	507	507	15300
2010-09-17	506	506	503	504	19300
2010-09-21	503	505	500	500	18800
2010-09-22	501	502	500	502	27800
2010-09-24	501	504	501	502	4500
2010-09-27	502	505	500	505	23300
2010-09-28	502	508	502	508	8500
2010-09-29	510	515	509	513	17800
2010-09-30	517	522	514	515	29200
2010-10-01	515	515	510	515	2900
2010-10-04	516	519	515	515	4400
2010-10-05	518	518	510	515	6700
2010-10-06	518	518	511	511	6400
2010-10-07	506	516	505	511	11600
2010-10-08	511	524	508	512	21700
2010-10-12	518	520	509	515	25800
2010-10-13	516	516	508	516	10300
2010-10-14	513	516	510	510	6000
2010-10-15	509	514	507	508	17200
2010-10-18	511	515	510	515	4400
2010-10-19	516	518	513	518	25300
2010-10-20	518	520	512	516	29900
2010-10-21	513	519	513	518	5300
2010-10-22	518	518	511	516	5200
2010-10-25	512	520	512	516	18300
2010-10-26	518	518	515	515	2700
2010-10-27	517	520	515	515	5600
2010-10-28	515	520	510	517	18500
2010-10-29	514	522	514	522	11700
2010-11-01	520	529	519	528	21900
2010-11-02	529	529	517	526	22400
2010-11-04	529	529	520	526	20600
2010-11-05	526	528	524	526	20300
2010-11-08	526	527	523	523	9500
2010-11-09	523	526	521	526	31100
2010-11-10	526	528	520	520	41100
2010-11-11	520	520	518	518	18400
2010-11-12	518	523	518	523	15900
2010-11-15	521	524	517	519	5100
2010-11-16	519	527	519	527	15700
2010-11-17	527	527	522	524	14600
2010-11-18	526	533	526	531	33800
2010-11-19	533	537	531	537	27500
2010-11-22	530	540	530	540	23400
2010-11-24	530	543	530	543	19000
2010-11-25	544	544	535	535	27000
2010-11-26	535	540	535	535	11300
2010-11-29	535	536	530	535	15000
2010-11-30	535	535	532	532	18500
2010-12-01	532	540	532	539	16200
2010-12-02	539	541	530	531	26700
2010-12-03	530	532	529	530	21100
2010-12-06	535	536	533	533	16300
2010-12-07	533	534	527	530	29800
2010-12-08	531	538	531	533	30900
2010-12-09	533	540	533	540	27700
2010-12-10	540	541	531	531	26900
2010-12-13	537	542	534	538	34600
2010-12-14	534	537	532	533	16300
2010-12-15	537	540	534	536	19600
2010-12-16	536	540	536	537	24600
2010-12-17	537	545	537	545	32900
2010-12-20	545	545	538	538	29600
2010-12-21	539	542	538	540	18500
2010-12-22	542	547	540	547	21100
2010-12-24	547	547	536	536	12800
2010-12-27	530	540	530	536	16000
2010-12-28	539	541	538	540	9200
2010-12-29	542	548	542	546	21300
2010-12-30	546	554	544	551	30400
2011-01-04	553	555	551	554	20500
2011-01-05	554	554	550	550	11000
2011-01-06	554	554	550	550	10200
2011-01-07	550	552	549	551	14200
2011-01-11	552	561	551	555	57400
2011-01-12	564	570	559	566	56900
2011-01-13	568	568	562	563	41500
2011-01-14	564	573	564	570	49800
2011-01-17	579	586	572	580	47100
2011-01-18	580	580	567	568	23100
2011-01-19	570	576	570	576	7500
2011-01-20	570	574	564	569	22000
2011-01-21	569	577	569	577	42200
2011-01-24	571	580	570	575	11400
2011-01-25	571	580	571	580	14000
2011-01-26	577	580	570	579	21000
2011-01-27	529	560	524	545	162000
2011-01-28	540	545	536	538	51500
2011-01-31	537	553	537	550	40900
2011-02-01	542	550	532	540	21000
2011-02-02	540	545	538	540	21900
2011-02-03	535	539	535	536	5200
2011-02-04	533	544	533	540	28700
2011-02-07	545	545	537	541	10700
2011-02-08	542	542	533	539	18000
2011-02-09	540	540	535	537	8300
2011-02-10	537	540	535	539	15500
2011-02-14	536	537	532	535	27500
2011-02-15	534	535	528	533	13200
2011-02-16	531	543	531	539	24900
2011-02-17	540	543	539	543	8600
2011-02-18	543	543	540	540	11500
2011-02-21	538	538	532	534	27300
2011-02-22	534	535	533	533	5100
2011-02-23	531	538	530	530	5300
2011-02-24	531	539	531	531	19800
2011-02-25	530	538	530	535	11800
2011-02-28	538	538	532	535	5400
2011-03-01	534	535	530	534	4100
2011-03-02	532	534	530	533	10100
2011-03-03	534	534	526	527	11900
2011-03-04	527	529	520	524	19600
2011-03-07	515	519	511	511	21200
2011-03-08	510	515	510	512	8600
2011-03-09	511	515	510	513	7600
2011-03-10	513	514	511	513	3200
2011-03-11	510	515	510	511	8000
2011-03-14	503	504	457	497	80700
2011-03-15	465	470	417	430	30800
2011-03-16	430	499	430	499	31400
2011-03-17	475	492	470	492	9000
2011-03-18	495	499	490	497	51200
2011-03-22	490	499	472	494	40800
2011-03-23	493	510	490	508	29900
2011-03-24	510	510	497	497	45100
2011-03-25	506	514	503	513	24800
2011-03-28	503	513	502	511	23500
2011-03-29	510	512	500	505	13800
2011-03-30	509	527	509	520	18800
2011-03-31	510	523	510	516	37500
2011-04-01	517	517	505	506	6800
2011-04-04	504	509	490	496	7500
2011-04-05	490	496	485	490	10200
2011-04-06	490	490	478	479	5600
2011-04-07	472	481	461	481	14000
2011-04-08	474	484	466	484	6200
2011-04-11	460	480	460	471	52900
2011-04-12	473	475	465	473	34900
2011-04-13	465	505	465	490	34200
2011-04-14	491	491	484	484	17600
2011-04-15	489	497	485	493	5700
2011-04-18	492	493	490	490	4200
2011-04-19	490	495	489	491	3700
2011-04-20	490	493	489	491	5500
2011-04-21	492	492	489	490	5600
2011-04-22	486	489	486	487	6100
2011-04-25	485	486	483	483	7900
2011-04-26	470	478	469	474	18600
2011-04-27	478	478	457	466	31300
2011-04-28	470	471	465	470	4900
2011-05-02	470	470	465	469	8700
2011-05-06	469	471	465	470	6400
2011-05-09	472	479	471	479	10900
2011-05-10	485	490	465	469	41100
2011-05-11	471	482	470	473	9600
2011-05-12	470	481	470	481	3900
2011-05-13	481	484	466	472	17200
2011-05-16	474	485	469	485	8900
2011-05-17	485	485	480	485	8300
2011-05-18	484	485	469	473	6700
2011-05-19	477	480	474	480	3300
2011-05-20	483	484	479	484	6600
2011-05-23	484	485	480	484	9600
2011-05-24	483	485	475	482	14000
2011-05-25	479	485	473	473	6800
2011-05-26	474	480	474	475	2900
2011-05-27	474	478	470	471	8200
2011-05-30	471	471	470	471	14100
2011-05-31	471	474	470	470	6100
2011-06-01	475	477	474	474	2800
2011-06-02	474	474	463	469	7400
2011-06-03	468	474	465	473	3800
2011-06-06	468	469	465	469	6600
2011-06-07	465	469	465	465	1900
2011-06-08	465	470	465	468	3900
2011-06-09	466	470	465	468	3000
2011-06-10	466	466	462	465	1800
2011-06-13	465	468	465	468	2500
2011-06-14	470	472	462	468	17900
2011-06-15	468	469	465	469	600
2011-06-16	469	473	460	472	18600
2011-06-17	470	470	460	470	22000
2011-06-20	472	472	450	460	16500
2011-06-21	463	469	463	468	3300
2011-06-22	469	472	462	464	11000
2011-06-23	464	464	456	460	18000
2011-06-24	458	465	458	465	1000
2011-06-27	463	463	452	458	16100
2011-06-28	458	464	458	464	5100
2011-06-29	465	469	460	469	5300
2011-06-30	469	469	465	469	1900
2011-07-01	466	468	465	466	2100
2011-07-04	466	468	463	464	3900
2011-07-05	465	465	461	461	5600
2011-07-06	461	464	461	463	8700
2011-07-07	463	463	460	461	5300
2011-07-08	460	460	455	457	13300
2011-07-11	454	455	454	455	4500
2011-07-12	452	452	447	450	12600
2011-07-13	450	450	446	448	13900
2011-07-14	446	449	444	444	8900
2011-07-15	445	445	440	443	6100
2011-07-19	442	446	442	446	11200
2011-07-20	448	448	443	446	11800
2011-07-21	446	446	441	441	3700
2011-07-22	443	444	440	441	6500
2011-07-25	448	448	442	446	11500
2011-07-26	448	449	441	442	7700
2011-07-27	442	443	440	441	9100
2011-07-28	440	442	438	441	5200
2011-07-29	440	441	438	439	2600
2011-08-01	439	439	433	438	23500
2011-08-02	449	449	438	441	20500
2011-08-03	430	430	426	428	12300
2011-08-04	431	432	420	426	9000
2011-08-05	421	425	401	416	13700
2011-08-08	411	414	405	405	11800
2011-08-09	405	410	394	406	27800
2011-08-10	415	415	408	409	7500
2011-08-11	405	410	405	410	8200
2011-08-12	410	412	398	405	22300
2011-08-15	405	410	405	405	4600
2011-08-16	405	405	397	400	25000
2011-08-17	399	403	399	400	9400
2011-08-18	403	403	398	399	5900
2011-08-19	399	399	392	392	11900
2011-08-22	385	390	374	380	29300
2011-08-23	384	384	378	384	14800
2011-08-24	384	386	383	385	8500
2011-08-25	385	387	384	387	6400
2011-08-26	383	386	383	384	4900
2011-08-29	385	385	380	382	6700
2011-08-30	380	385	380	382	14400
2011-08-31	384	384	379	381	15300
2011-09-01	382	384	381	382	9700
2011-09-02	380	382	380	380	9200
2011-09-05	385	385	381	385	15400
2011-09-06	381	381	375	379	34300
2011-09-07	382	382	372	378	14000
2011-09-08	383	383	370	378	15300
2011-09-09	371	379	371	375	6900
2011-09-12	372	378	372	374	5300
2011-09-13	375	375	373	373	6500
2011-09-14	373	376	373	374	23500
2011-09-15	371	373	371	372	17100
2011-09-16	375	379	373	378	14200
2011-09-20	376	383	376	379	12800
2011-09-21	380	383	379	383	12400
2011-09-22	380	383	378	380	8700
2011-09-26	378	379	375	378	10100
2011-09-27	376	378	376	377	13100
2011-09-28	378	381	377	378	5400
2011-09-29	370	382	370	377	8900
2011-09-30	380	380	375	379	2700
2011-10-03	378	379	377	377	5200
2011-10-04	376	376	373	375	7700
2011-10-05	376	376	364	373	9200
2011-10-06	370	374	367	370	17500
2011-10-07	370	372	367	371	14900
2011-10-11	368	372	367	370	10300
2011-10-12	371	371	370	371	5600
2011-10-13	376	380	372	377	23900
2011-10-14	377	377	373	373	7100
2011-10-17	365	369	365	369	42000
2011-10-18	370	370	366	366	13300
2011-10-19	369	370	367	369	5700
2011-10-20	371	371	368	370	12500
2011-10-21	370	371	369	370	4900
2011-10-24	369	371	368	371	8700
2011-10-25	369	370	368	368	3500
2011-10-26	369	376	368	374	20700
2011-10-27	374	374	370	374	14800
2011-10-28	375	375	372	372	4900
2011-10-31	373	375	373	374	1200
2011-11-01	372	372	371	372	3700
2011-11-02	372	373	370	371	8600
2011-11-04	369	371	367	367	13700
2011-11-07	370	370	365	367	11800
2011-11-08	367	367	364	364	20800
2011-11-09	362	365	360	360	46100
2011-11-10	355	357	350	356	24900
2011-11-11	354	358	350	354	5000
2011-11-14	353	355	348	353	12700
2011-11-15	353	353	348	352	15800
2011-11-16	354	354	350	350	5800
2011-11-17	345	346	341	344	20900
2011-11-18	340	346	324	343	60800
2011-11-21	343	346	330	342	36000
2011-11-22	341	352	341	350	67200
2011-11-24	350	350	344	350	16700
2011-11-25	350	350	346	349	2600
2011-11-28	348	349	343	349	7900
2011-11-29	348	353	347	353	41100
2011-11-30	353	355	353	355	24900
2011-12-01	355	359	354	355	34000
2011-12-02	358	360	355	359	30000
2011-12-05	359	363	346	352	53300
2011-12-06	353	355	349	354	27800
2011-12-07	354	354	345	345	50200
2011-12-08	346	350	346	350	27900
2011-12-09	350	350	347	349	28800
2011-12-12	349	349	346	347	38100
2011-12-13	347	347	345	347	34700
2011-12-14	347	349	347	348	37100
2011-12-15	348	348	346	346	31400
2011-12-16	347	350	347	349	30900
2011-12-19	349	351	349	350	34400
2011-12-20	350	350	349	350	31300
2011-12-21	350	353	350	351	24300
2011-12-22	352	354	352	354	45200
2011-12-26	354	354	345	346	18500
2011-12-27	346	346	341	342	21600
2011-12-28	342	342	342	342	3400
2011-12-29	340	340	338	339	5900
2011-12-30	340	340	333	340	6800
2012-01-04	340	345	339	342	33200
2012-01-05	344	345	340	343	34100
2012-01-06	343	344	339	340	45500
2012-01-10	316	335	316	330	86100
2012-01-11	328	329	325	328	55100
2012-01-12	330	335	328	334	8200
2012-01-13	328	333	327	332	13100
2012-01-16	329	333	325	325	22300
2012-01-17	326	331	325	327	12200
2012-01-18	328	330	328	330	4200
2012-01-19	330	334	329	331	24300
2012-01-20	330	335	325	332	32700
2012-01-23	332	333	331	332	21300
2012-01-24	332	333	330	330	50800
2012-01-25	328	330	326	329	59600
2012-01-26	327	330	326	329	15100
2012-01-27	330	330	325	327	11300
2012-01-30	328	330	328	328	5000
2012-01-31	332	334	327	329	14200
2012-02-01	330	330	325	327	17100
2012-02-02	333	333	325	330	28800
2012-02-03	330	330	329	329	6800
2012-02-06	328	329	320	323	68200
2012-02-07	324	327	323	326	43700
2012-02-08	326	328	325	328	13900
2012-02-09	328	330	325	329	16000
2012-02-10	328	328	326	328	10600
2012-02-13	329	333	329	331	39700
2012-02-14	331	333	330	332	25100
2012-02-15	332	332	329	331	16800
2012-02-16	333	335	332	333	31900
2012-02-17	339	339	337	339	37000
2012-02-20	339	341	338	340	84400
2012-02-21	340	342	338	341	26700
2012-02-22	342	343	340	342	30800
2012-02-23	342	342	340	340	10300
2012-02-24	341	341	340	341	22700
2012-02-27	340	345	339	345	45600
2012-02-28	346	363	344	363	74100
2012-02-29	371	383	364	378	113500
2012-03-01	378	380	352	374	54200
2012-03-02	379	380	364	370	28400
2012-03-05	374	374	357	373	24700
2012-03-06	362	373	354	359	28700
2012-03-07	351	352	347	348	16500
2012-03-08	351	351	347	347	7900
2012-03-09	349	350	348	349	8500
2012-03-12	351	356	349	349	16600
2012-03-13	349	355	349	353	7300
2012-03-14	353	355	350	350	11800
2012-03-15	350	352	343	348	23100
2012-03-16	350	350	345	349	8800
2012-03-19	349	349	347	349	29700
2012-03-21	350	353	349	352	36100
2012-03-22	350	351	349	350	10400
2012-03-23	350	353	350	351	28300
2012-03-26	354	354	352	353	13000
2012-03-27	353	354	350	353	16300
2012-03-28	331	346	331	341	17700
2012-03-29	344	344	339	341	8800
2012-03-30	343	343	331	339	19600
2012-04-02	340	353	336	338	43000
2012-04-03	340	340	338	340	14100
2012-04-04	340	340	338	338	10800
2012-04-05	335	335	330	334	22000
2012-04-06	330	335	329	334	7300
2012-04-09	332	338	331	338	6100
2012-04-10	335	336	335	336	2100
2012-04-11	334	334	326	332	13300
2012-04-12	325	330	325	327	7600
2012-04-13	326	332	325	327	7000
2012-04-16	327	331	326	326	4100
2012-04-17	326	327	325	325	4600
2012-04-18	329	329	326	328	2500
2012-04-19	330	332	328	330	4600
2012-04-20	330	330	325	326	11300
2012-04-23	329	329	326	326	1600
2012-04-24	326	328	325	326	1600
2012-04-25	326	329	326	327	3600
2012-04-26	329	329	320	322	23800
2012-04-27	324	324	319	321	6000
2012-05-01	321	321	320	321	4300
2012-05-02	321	324	315	323	11300
2012-05-07	318	319	315	318	14000
2012-05-08	318	319	315	318	3500
2012-05-09	312	317	311	312	10800
cut1
2000-03-01	750	750	705	731	70200
2000-03-02	730	730	615	641	15600
2000-03-03	639	640	602	603	14040
2000-03-06	603	603	571	571	9360
2000-03-07	506	506	506	506	23400
2000-03-08	494	571	494	571	14040
2000-03-09	571	571	564	564	7800
2000-03-10	500	500	500	500	24960
2000-03-13	481	532	481	513	24960
2000-03-14	514	514	487	487	12480
2000-03-15	513	519	500	500	9360
2000-03-16	513	513	494	513	12480
2000-03-17	512	512	511	511	7800
2000-03-21	511	522	511	516	14040
2000-03-22	481	516	481	516	15600
2000-03-23	515	515	515	515	3120
2000-03-24	513	513	483	487	10920
2000-03-27	494	494	481	494	17160
2000-03-28	494	494	487	487	12480
2000-03-29	481	481	475	475	15600
2000-03-30	475	481	472	481	26520
2000-03-31	478	478	468	478	15600
2000-04-03	475	475	462	475	18720
2000-04-04	475	475	458	458	29640
2000-04-05	448	451	442	442	31200
2000-04-06	439	441	417	441	14040
2000-04-07	435	455	429	448	14040
2000-04-10	448	448	418	448	15600
2000-04-11	448	462	448	462	15600
2000-04-12	494	513	481	513	29640
2000-04-13	535	542	525	525	15600
2000-04-14	494	513	494	513	7800
2000-04-17	506	506	448	448	29640
2000-04-18	462	481	448	481	31200
2000-04-19	481	482	481	482	3120
2000-04-20	482	482	481	481	15600
2000-04-21	481	506	481	500	14040
2000-04-24	500	503	494	494	6240
2000-04-26	481	481	468	468	4680
2000-04-27	436	455	429	448	9360
2000-04-28	449	493	449	493	3120
2000-05-02	487	487	487	487	4680
2000-05-08	513	513	506	506	18720
2000-05-09	468	481	468	481	14040
2000-05-10	474	478	468	468	7800
2000-05-11	481	481	468	468	17160
2000-05-12	468	468	468	468	1560
2000-05-15	468	470	468	470	7800
2000-05-16	481	481	481	481	1560
2000-05-17	475	475	468	468	7800
2000-05-18	468	480	468	474	14040
2000-05-19	475	475	462	462	10920
2000-05-22	481	481	448	448	18720
2000-05-23	448	448	437	437	6240
2000-05-24	436	445	435	442	34320
2000-05-25	442	448	442	448	4680
2000-05-26	448	448	417	448	17160
2000-05-29	464	464	436	442	4680
2000-05-30	436	442	436	442	3120
2000-05-31	442	445	436	436	23400
2000-06-01	442	442	442	442	6240
2000-06-02	436	436	436	436	6240
2000-06-05	448	448	448	448	6240
2000-06-06	436	436	436	436	14040
2000-06-08	436	436	436	436	1560
2000-06-09	436	436	429	429	3120
2000-06-12	429	436	429	436	12480
2000-06-13	442	442	436	442	6240
2000-06-14	442	442	442	442	1560
2000-06-15	436	436	429	429	3120
2000-06-16	442	448	442	448	4680
2000-06-20	448	448	442	442	7800
2000-06-21	442	442	429	429	7800
2000-06-22	429	429	429	429	15600
2000-06-23	429	429	429	429	12480
2000-06-27	430	430	430	430	3120
2000-06-28	430	430	429	429	3120
2000-06-29	429	429	429	429	1560
2000-06-30	442	442	429	429	6240
2000-07-03	448	448	442	448	7800
2000-07-04	487	491	487	491	15600
2000-07-05	490	490	487	489	10920
2000-07-07	487	487	487	487	6240
2000-07-10	487	487	455	458	9360
2000-07-11	489	489	489	489	6240
2000-07-12	458	458	458	458	4680
2000-07-13	487	487	487	487	1560
2000-07-17	460	460	460	460	1560
2000-07-18	461	461	461	461	1560
2000-07-19	462	462	462	462	3120
2000-07-21	463	465	463	465	4680
2000-07-24	449	449	448	448	4680
2000-07-25	448	448	442	442	17160
2000-07-26	448	448	436	436	9360
2000-07-27	436	436	436	436	12480
2000-07-28	448	448	436	436	6240
2000-07-31	436	436	430	430	6240
2000-08-01	448	448	448	448	1560
2000-08-02	436	436	436	436	3120
2000-08-03	417	423	417	423	4680
2000-08-04	423	436	423	436	6240
2000-08-07	448	448	448	448	4680
2000-08-08	436	436	436	436	7800
2000-08-09	436	436	429	429	3120
2000-08-10	423	423	423	423	1560
2000-08-16	448	448	448	448	1560
2000-08-17	442	442	442	442	3120
2000-08-18	441	441	436	436	18720
2000-08-21	448	448	448	448	4680
2000-08-22	436	436	429	429	3120
2000-08-23	423	433	423	433	4680
2000-08-24	428	429	423	423	4680
2000-08-25	423	423	423	423	6240
2000-08-28	423	423	417	417	9360
2000-08-29	424	424	423	423	3120
2000-08-30	423	423	423	423	3120
2000-08-31	429	429	429	429	3120
2000-09-01	445	448	429	429	23400
2000-09-04	436	436	423	429	23400
2000-09-05	436	436	417	417	20280
2000-09-07	436	436	417	417	28080
2000-09-08	417	442	417	442	12480
2000-09-11	448	448	417	417	3120
2000-09-12	418	418	418	418	1560
2000-09-13	443	443	443	443	3120
2000-09-14	418	420	418	420	3120
2000-09-19	439	448	439	448	7800
2000-09-20	448	448	442	442	6240
2000-09-21	442	442	442	442	4680
2000-09-22	439	439	439	439	3120
2000-09-26	422	448	418	448	6240
2000-09-27	423	442	423	442	3120
2000-09-28	442	442	442	442	1560
2000-09-29	423	441	417	441	12480
2000-10-02	442	442	442	442	4680
2000-10-03	442	442	442	442	3120
2000-10-04	429	442	427	427	4680
2000-10-05	442	442	442	442	6240
2000-10-06	442	442	442	442	6240
2000-10-10	420	420	420	420	1560
2000-10-11	423	423	423	423	1560
2000-10-12	418	435	418	435	4680
2000-10-17	429	429	423	423	4680
2000-10-18	423	423	392	392	9360
2000-10-19	398	398	391	391	12480
2000-10-20	392	395	392	395	4680
2000-10-23	398	398	398	398	3120
2000-10-24	398	398	398	398	1560
2000-10-25	413	413	413	413	1560
2000-10-26	412	412	410	410	7800
2000-10-30	410	410	404	410	10920
2000-10-31	410	410	410	410	1560
2000-11-01	410	410	410	410	7800
2000-11-06	410	410	410	410	3120
2000-11-07	410	410	410	410	9360
2000-11-08	398	398	398	398	7800
2000-11-13	385	385	385	385	4680
2000-11-15	398	398	398	398	1560
2000-11-16	391	391	391	391	6240
2000-11-17	410	410	410	410	1560
2000-11-20	416	416	391	391	7800
2000-11-21	385	385	385	385	6240
2000-11-22	385	385	385	385	1560
2000-11-24	391	391	385	385	3120
2000-11-27	385	385	368	368	7800
2000-11-29	369	369	369	369	1560
2000-11-30	378	378	378	378	1560
2000-12-01	385	385	385	385	3120
2000-12-04	385	396	385	396	6240
2000-12-05	397	398	397	398	7800
2000-12-06	398	398	398	398	1560
2000-12-08	391	391	391	391	3120
2000-12-11	385	385	385	385	3120
2000-12-12	385	385	385	385	3120
2000-12-13	365	365	365	365	3120
2000-12-14	366	366	366	366	1560
2000-12-18	398	398	398	398	3120
2000-12-19	366	367	366	366	4680
2000-12-20	385	385	366	366	6240
2000-12-21	334	334	334	334	3120
2000-12-22	365	365	365	365	7800
2000-12-25	365	365	365	365	4680
2000-12-28	365	365	359	359	3120
2001-01-04	385	385	385	385	1560
2001-01-05	391	391	385	385	7800
2001-01-09	346	373	341	373	4680
2001-01-12	340	341	340	340	7800
2001-01-15	368	398	368	398	26520
2001-01-16	410	423	398	423	26520
2001-01-17	433	462	417	462	23400
2001-01-18	462	481	462	462	35880
2001-01-19	479	481	468	481	39000
2001-01-22	545	545	545	545	82680
2001-01-23	558	558	487	494	207480
2001-01-24	487	487	449	461	37440
2001-01-25	448	455	423	436	43680
2001-01-26	455	455	429	452	12480
2001-01-29	452	461	452	461	23400
2001-01-30	474	474	474	474	1560
2001-01-31	474	480	468	480	18720
2001-02-01	481	481	468	468	10920
2001-02-02	468	468	448	462	12480
2001-02-05	448	448	442	448	10920
2001-02-06	431	442	423	423	9360
2001-02-07	436	448	423	439	12480
2001-02-08	439	442	436	436	17160
2001-02-09	436	448	436	442	14040
2001-02-13	448	448	448	448	1560
2001-02-14	448	448	436	436	4680
2001-02-15	448	448	436	436	6240
2001-02-19	435	435	404	404	9360
2001-02-20	448	448	407	407	6240
2001-02-21	418	448	418	442	6240
2001-02-23	448	448	448	448	1560
2001-02-26	413	413	413	413	3120
2001-02-27	417	426	417	426	6240
2001-02-28	426	426	423	423	3120
2001-03-01	414	414	404	404	6240
2001-03-02	430	430	423	423	3120
2001-03-05	423	423	410	410	9360
2001-03-06	404	404	397	397	14040
2001-03-07	404	404	398	398	9360
2001-03-08	398	410	388	388	9360
2001-03-09	410	410	388	403	6240
2001-03-12	410	414	410	414	10920
2001-03-13	410	410	410	410	1560
2001-03-14	398	401	398	398	4680
2001-03-15	391	409	391	409	4680
2001-03-16	415	415	414	414	3120
2001-03-21	410	410	389	389	12480
2001-03-22	395	395	390	391	17160
2001-03-23	395	395	391	391	6240
2001-03-26	391	394	391	394	10920
2001-03-27	392	392	391	392	4680
2001-03-28	391	391	391	391	7800
2001-03-29	397	414	391	414	9360
2001-03-30	410	410	398	398	3120
2001-04-02	403	403	392	392	3120
2001-04-03	392	392	392	392	1560
2001-04-04	394	398	394	398	3120
2001-04-05	404	404	385	387	63960
2001-04-06	388	391	385	385	46800
2001-04-09	384	384	379	379	10920
2001-04-10	379	379	373	373	9360
2001-04-11	378	379	376	379	20280
2001-04-12	379	379	375	375	6240
2001-04-13	375	375	375	375	10920
2001-04-16	372	372	359	359	18720
2001-04-17	362	365	361	365	6240
2001-04-18	378	378	365	365	7800
2001-04-20	404	404	372	372	6240
2001-04-23	385	404	385	404	7800
2001-04-24	395	395	395	395	1560
2001-04-25	397	397	372	372	4680
2001-04-26	397	397	385	385	10920
2001-04-27	385	385	385	385	3120
2001-05-01	385	391	385	385	4680
2001-05-02	397	398	387	387	7800
2001-05-07	392	398	392	395	7800
2001-05-08	398	398	388	388	4680
2001-05-09	385	395	385	395	9360
2001-05-10	386	386	385	385	14040
2001-05-11	385	391	385	385	7800
2001-05-14	391	391	391	391	3120
2001-05-15	391	391	385	391	4680
2001-05-16	392	392	392	392	3120
2001-05-17	385	388	385	388	6240
2001-05-18	395	395	395	395	1560
2001-05-21	395	395	395	395	4680
2001-05-22	391	391	391	391	1560
2001-05-23	388	397	385	397	4680
2001-05-24	385	385	385	385	4680
2001-05-25	395	395	395	395	1560
2001-05-28	407	426	407	410	45240
2001-05-29	410	462	410	462	113880
2001-05-30	455	458	442	442	73320
2001-05-31	437	437	420	435	23400
2001-06-01	435	458	433	448	63960
2001-06-04	448	455	445	455	51480
2001-06-05	442	468	442	468	82680
2001-06-06	475	500	472	492	279240
2001-06-07	492	500	481	481	127920
2001-06-08	480	499	468	494	71760
2001-06-11	492	492	482	482	4680
2001-06-12	481	482	472	482	18720
2001-06-13	484	497	478	497	54600
2001-06-14	491	497	475	482	45240
2001-06-15	476	482	476	481	43680
2001-06-18	484	484	468	478	26520
2001-06-19	478	481	475	481	51480
2001-06-20	475	481	472	481	35880
2001-06-21	478	481	470	481	45240
2001-06-22	481	532	481	521	352560
2001-06-25	517	525	512	521	87360
2001-06-26	514	538	513	521	251160
2001-06-27	525	550	525	545	304200
2001-06-28	545	558	538	543	138840
2001-06-29	542	543	529	529	79560
2001-07-02	548	548	525	537	74880
2001-07-03	538	542	525	529	57720
2001-07-04	516	528	513	525	57720
2001-07-05	513	525	506	507	60840
2001-07-06	510	519	510	518	65520
2001-07-09	519	519	499	506	20280
2001-07-10	506	512	496	512	48360
2001-07-11	499	499	487	487	6240
2001-07-12	475	497	475	475	15600
2001-07-13	492	492	487	491	46800
2001-07-16	487	487	458	468	9360
2001-07-17	468	475	468	475	29640
2001-07-18	478	478	448	475	59280
2001-07-19	468	468	455	455	6240
2001-07-23	455	455	445	445	9360
2001-07-24	444	448	436	448	53040
2001-07-25	442	442	442	442	1560
2001-07-26	442	448	442	448	14040
2001-07-27	442	446	417	445	49920
2001-07-30	448	448	423	423	14040
2001-07-31	448	481	448	462	49920
2001-08-01	468	468	461	461	23400
2001-08-03	429	447	429	447	34320
2001-08-06	447	448	447	448	21840
2001-08-07	442	442	442	442	3120
2001-08-08	442	442	423	423	43680
2001-08-09	423	426	404	423	17160
2001-08-10	423	423	423	423	4680
2001-08-13	423	424	423	424	3120
2001-08-14	424	424	424	424	1560
2001-08-15	423	423	423	423	3120
2001-08-16	404	423	404	423	6240
2001-08-17	423	423	423	423	3120
2001-08-20	433	433	429	433	6240
2001-08-21	423	423	417	417	4680
2001-08-22	417	417	410	410	3120
2001-08-23	410	410	404	404	7800
2001-08-24	404	404	404	404	4680
2001-08-27	395	404	395	395	15600
2001-08-29	391	391	388	391	7800
2001-08-30	385	388	372	375	15600
2001-08-31	378	385	372	385	23400
2001-09-03	385	385	372	372	6240
2001-09-04	373	385	373	381	6240
2001-09-05	400	401	382	387	12480
2001-09-06	400	400	385	385	9360
2001-09-07	385	385	382	385	9360
2001-09-10	384	384	384	384	1560
2001-09-11	372	372	352	359	48360
2001-09-12	327	333	327	333	57720
2001-09-13	327	333	324	330	40560
2001-09-14	340	346	333	343	20280
2001-09-17	356	362	352	352	10920
2001-09-18	372	372	372	372	4680
2001-09-19	378	384	378	378	12480
2001-09-20	375	375	372	372	14040
2001-09-21	372	372	352	352	6240
2001-09-25	378	378	378	378	7800
2001-09-26	365	365	353	353	6240
2001-09-27	359	359	353	353	4680
2001-09-28	352	362	352	362	21840
2001-10-01	359	359	359	359	3120
2001-10-02	372	372	372	372	9360
2001-10-03	365	365	359	359	7800
2001-10-04	360	372	360	372	23400
2001-10-05	378	378	360	360	12480
2001-10-09	368	368	368	368	6240
2001-10-10	365	368	359	359	12480
2001-10-11	378	378	378	378	1560
2001-10-12	368	368	365	365	6240
2001-10-15	378	378	378	378	6240
2001-10-16	365	367	365	365	12480
2001-10-17	365	365	362	362	15600
2001-10-18	375	376	371	376	7800
2001-10-19	378	378	365	365	17160
2001-10-22	365	365	359	360	10920
2001-10-23	360	362	360	362	6240
2001-10-24	368	368	365	365	6240
2001-10-25	365	372	365	371	10920
2001-10-26	372	378	372	378	15600
2001-10-29	375	378	375	378	18720
2001-10-30	378	395	378	388	29640
2001-10-31	386	396	385	396	17160
2001-11-01	404	417	399	414	24960
2001-11-02	410	420	404	410	21840
2001-11-05	411	423	405	417	28080
2001-11-06	417	433	417	433	40560
2001-11-07	433	439	433	439	63960
2001-11-08	436	436	429	436	28080
2001-11-09	436	439	422	429	32760
2001-11-12	429	429	417	417	17160
2001-11-13	404	417	391	414	20280
2001-11-15	410	410	410	410	7800
2001-11-16	398	398	379	385	23400
2001-11-19	392	393	382	384	14040
2001-11-20	384	385	384	385	24960
2001-11-21	378	385	378	385	24960
2001-11-22	385	385	385	385	3120
2001-11-26	392	398	385	385	6240
2001-11-28	382	382	382	382	1560
2001-11-29	385	385	385	385	3120
2001-11-30	391	391	378	378	9360
2001-12-03	378	378	378	378	3120
2001-12-04	378	378	378	378	4680
2001-12-05	385	385	378	378	3120
2001-12-06	378	378	378	378	1560
2001-12-07	378	378	378	378	6240
2001-12-10	378	378	378	378	1560
2001-12-11	372	372	371	371	4680
2001-12-12	366	368	366	368	6240
2001-12-13	368	368	359	360	17160
2001-12-17	353	356	347	351	34320
2001-12-18	348	368	348	368	9360
2001-12-19	356	372	346	346	28080
2001-12-20	345	352	341	343	17160
2001-12-21	340	352	335	335	24960
2001-12-25	335	378	335	378	17160
2001-12-26	378	384	378	384	17160
2001-12-27	391	410	385	385	28080
2001-12-28	379	398	379	391	20280
2002-01-04	372	385	372	385	4680
2002-01-07	391	391	391	391	4680
2002-01-08	385	385	385	385	3120
2002-01-11	378	378	378	378	6240
2002-01-15	378	378	372	372	4680
2002-01-16	356	356	333	352	40560
2002-01-17	356	356	337	349	21840
2002-01-18	372	372	365	365	6240
2002-01-21	372	372	346	346	29640
2002-01-22	346	365	346	365	6240
2002-01-23	355	356	349	349	4680
2002-01-24	349	349	346	347	7800
2002-01-25	347	365	347	365	7800
2002-01-28	362	362	352	352	4680
2002-01-29	353	353	352	352	7800
2002-01-30	352	352	346	346	10920
2002-01-31	356	365	356	365	17160
2002-02-01	365	365	347	347	9360
2002-02-04	348	349	348	349	10920
2002-02-05	362	362	348	348	10920
2002-02-06	347	359	346	348	14040
2002-02-07	348	348	346	346	7800
2002-02-08	359	359	359	359	4680
2002-02-12	358	359	358	359	4680
2002-02-13	359	362	356	362	10920
2002-02-14	372	372	372	372	7800
2002-02-15	365	365	352	352	3120
2002-02-18	365	365	356	359	23400
2002-02-19	372	372	365	365	10920
2002-02-20	356	366	356	366	7800
2002-02-21	368	368	368	368	4680
2002-02-22	368	368	368	368	3120
2002-02-25	365	365	357	359	18720
2002-02-26	356	356	356	356	3120
2002-02-27	359	359	359	359	15600
2002-02-28	360	365	359	365	14040
2002-03-01	365	372	365	372	6240
2002-03-04	372	372	358	368	12480
2002-03-05	372	372	359	372	24960
2002-03-06	362	366	362	366	21840
2002-03-07	362	362	362	362	1560
2002-03-08	362	362	359	359	15600
2002-03-11	360	365	359	359	26520
2002-03-12	359	359	354	354	10920
2002-03-13	356	356	356	356	1560
2002-03-14	356	365	356	365	4680
2002-03-15	368	368	356	356	10920
2002-03-18	365	365	356	356	7800
2002-03-19	356	356	354	354	9360
2002-03-20	362	362	355	355	6240
2002-03-22	359	365	359	365	17160
2002-03-25	370	384	368	384	20280
2002-03-26	384	385	384	384	24960
2002-03-27	383	383	383	383	1560
2002-04-01	375	375	352	352	9360
2002-04-02	365	365	346	346	4680
2002-04-03	350	352	349	352	4680
2002-04-04	352	352	346	352	7800
2002-04-05	368	368	353	353	7800
2002-04-08	354	354	352	352	4680
2002-04-09	349	349	346	349	26520
2002-04-10	349	349	349	349	6240
2002-04-11	359	364	351	351	9360
2002-04-12	352	352	352	352	1560
2002-04-16	359	359	359	359	4680
2002-04-17	359	359	350	350	4680
2002-04-18	365	365	365	365	1560
2002-04-19	362	362	346	348	28080
2002-04-22	353	354	349	349	7800
2002-04-23	349	352	349	352	3120
2002-04-24	359	359	358	358	4680
2002-04-25	358	358	358	358	4680
2002-04-26	356	356	349	349	7800
2002-04-30	349	349	349	349	1560
2002-05-02	354	354	354	354	3120
2002-05-07	356	359	354	359	9360
2002-05-08	365	365	356	356	7800
2002-05-09	356	356	356	356	3120
2002-05-10	352	352	352	352	3120
2002-05-13	353	359	352	359	9360
2002-05-14	352	352	352	352	1560
2002-05-15	352	352	349	349	4680
2002-05-17	356	359	348	359	17160
2002-05-20	362	362	352	352	10920
2002-05-21	359	359	359	359	7800
2002-05-22	356	356	352	352	7800
2002-05-23	356	356	352	352	9360
2002-05-24	361	392	360	392	78000
2002-05-27	403	406	402	402	82680
2002-05-28	406	406	385	396	49920
2002-05-29	385	385	379	379	6240
2002-05-30	388	388	378	378	10920
2002-05-31	388	391	385	385	18720
2002-06-03	385	398	385	398	9672
2002-06-04	395	398	390	398	9516
2002-06-05	397	397	388	388	11076
2002-06-06	388	388	388	388	7800
2002-06-07	388	388	378	378	6552
2002-06-10	380	385	380	385	4836
2002-06-11	380	385	378	378	9204
2002-06-12	372	373	368	368	4680
2002-06-13	382	385	371	385	9828
2002-06-14	372	375	372	375	9048
2002-06-17	375	375	375	375	2028
2002-06-18	382	395	382	391	9204
2002-06-19	392	395	385	391	13728
2002-06-20	391	391	385	385	6396
2002-06-21	385	385	375	375	7332
2002-06-24	385	385	376	385	4056
2002-06-25	385	391	385	391	6396
2002-06-26	391	391	385	385	6864
2002-06-27	375	375	372	372	4680
2002-06-28	385	385	375	375	6396
2002-07-01	391	391	391	391	4056
2002-07-02	395	395	368	368	19656
2002-07-03	383	387	382	387	4680
2002-07-04	387	393	378	378	11388
2002-07-05	388	391	388	391	4836
2002-07-08	393	393	385	385	11544
2002-07-09	385	391	385	385	2964
2002-07-10	393	393	385	385	6240
2002-07-11	391	391	385	385	3900
2002-07-12	385	391	382	385	11076
2002-07-15	379	385	379	385	5928
2002-07-16	395	395	385	395	9828
2002-07-17	391	391	388	388	10920
2002-07-18	385	391	385	391	12792
2002-07-19	391	398	385	398	52884
2002-07-22	394	394	385	385	16380
2002-07-23	385	385	378	385	8112
2002-07-24	378	378	378	378	2964
2002-07-26	382	391	378	391	1716
2002-07-29	394	394	385	385	2184
2002-07-30	390	391	390	390	1872
2002-08-01	385	385	385	385	2184
2002-08-02	372	385	363	378	14352
2002-08-05	385	385	378	378	4056
2002-08-06	375	375	365	372	9048
2002-08-07	372	372	365	372	4680
2002-08-08	372	372	372	372	2808
2002-08-09	372	385	365	385	7956
2002-08-12	385	391	383	391	14196
2002-08-13	388	388	372	372	11856
2002-08-14	385	385	372	372	9984
2002-08-15	382	382	375	375	3900
2002-08-16	382	385	375	385	10920
2002-08-19	385	385	383	384	4524
2002-08-20	385	385	372	384	8112
2002-08-21	373	378	373	378	6084
2002-08-22	372	373	371	371	7800
2002-08-23	379	379	378	379	6864
2002-08-26	379	379	378	378	24648
2002-08-27	378	379	375	379	14352
2002-08-28	379	379	378	378	4836
2002-08-29	379	384	378	378	19032
2002-08-30	384	384	375	375	9516
2002-09-02	375	375	371	371	12480
2002-09-03	378	383	375	375	13728
2002-09-04	370	375	370	373	21060
2002-09-05	378	385	378	385	8580
2002-09-06	375	375	374	374	4524
2002-09-09	384	384	379	379	3432
2002-09-10	378	378	378	378	156
2002-09-11	375	375	375	375	468
2002-09-12	372	372	372	372	1560
2002-09-13	375	378	375	378	1560
2002-09-17	372	372	365	368	7332
2002-09-18	368	368	360	360	3120
2002-09-19	368	368	368	368	468
2002-09-20	378	378	378	378	4056
2002-09-24	372	378	372	378	468
2002-09-25	378	378	375	375	468
2002-09-26	361	363	361	361	5772
2002-09-27	362	362	359	359	11388
2002-09-30	368	368	353	368	5148
2002-10-01	349	359	349	359	4056
2002-10-02	364	364	364	364	624
2002-10-03	364	364	359	359	2184
2002-10-04	360	360	352	359	5304
2002-10-07	364	364	364	364	2184
2002-10-08	327	337	327	337	2028
2002-10-09	337	340	337	340	7176
2002-10-10	333	342	330	330	8580
2002-10-11	356	356	343	343	1092
2002-10-15	330	343	330	337	6240
2002-10-16	346	346	337	337	2184
2002-10-17	337	340	333	333	8424
2002-10-18	359	359	333	333	7332
2002-10-21	352	352	349	349	624
2002-10-22	352	352	346	346	3900
2002-10-23	333	336	333	333	10608
2002-10-24	333	333	330	330	7956
2002-10-25	330	346	330	345	4836
2002-10-28	340	341	340	340	3120
2002-10-30	340	340	339	339	7644
2002-10-31	339	346	335	335	9672
2002-11-01	329	337	329	335	6396
2002-11-05	335	336	335	336	6240
2002-11-06	337	337	335	335	1248
2002-11-07	337	337	336	336	2964
2002-11-08	314	327	314	327	9204
2002-11-11	327	327	325	326	1716
2002-11-12	326	326	321	321	2184
2002-11-13	321	321	311	311	11856
2002-11-14	311	325	311	324	13572
2002-11-15	312	321	312	321	3432
2002-11-18	319	319	314	319	2808
2002-11-19	319	319	314	314	4524
2002-11-20	321	321	308	308	10140
2002-11-21	308	308	308	308	1560
2002-11-22	308	318	308	314	5616
2002-11-25	321	321	310	317	3432
2002-11-26	320	324	320	324	10920
2002-11-27	318	324	318	324	4992
2002-11-28	321	321	321	321	3120
2002-11-29	321	352	321	352	12636
2002-12-02	352	352	345	345	2028
2002-12-03	337	337	328	333	2340
2002-12-04	337	337	337	337	1092
2002-12-05	333	333	321	332	5772
2002-12-06	319	332	314	332	5616
2002-12-09	327	327	318	326	7644
2002-12-10	321	321	321	321	1092
2002-12-11	321	321	318	321	4368
2002-12-12	321	321	318	318	3120
2002-12-13	318	318	314	314	7332
2002-12-16	311	317	311	317	3276
2002-12-17	313	314	311	311	12948
2002-12-18	318	318	305	311	7956
2002-12-19	308	317	298	317	14196
2002-12-20	318	318	314	318	9360
2002-12-24	318	318	298	314	6396
2002-12-25	319	319	308	318	14352
2002-12-26	319	319	319	319	156
2002-12-27	319	320	314	314	3432
2002-12-30	321	327	319	327	7956
2003-01-06	333	333	333	333	2340
2003-01-07	340	345	333	333	12168
2003-01-08	330	330	330	330	1248
2003-01-09	327	327	327	327	624
2003-01-10	327	327	324	327	4680
2003-01-14	327	327	312	315	9984
2003-01-15	330	330	324	330	3432
2003-01-16	330	330	327	327	4212
2003-01-17	332	333	327	330	8892
2003-01-20	337	340	337	340	4836
2003-01-21	333	333	330	330	3276
2003-01-22	333	333	333	333	468
2003-01-23	333	333	327	327	5460
2003-01-24	333	337	333	337	2652
2003-01-27	336	336	333	336	1404
2003-01-28	327	327	327	327	312
2003-01-29	327	327	322	322	1560
2003-01-30	327	333	322	322	1560
2003-01-31	333	333	321	330	936
2003-02-03	333	333	333	333	156
2003-02-04	333	333	321	321	3120
2003-02-05	335	335	333	333	2964
2003-02-06	333	333	327	333	2028
2003-02-07	327	340	327	340	7488
2003-02-10	322	352	322	352	11232
2003-02-12	333	349	333	349	6864
2003-02-13	339	339	333	333	2496
2003-02-14	337	346	337	346	3900
2003-02-17	346	346	346	346	1248
2003-02-18	345	345	341	341	1092
2003-02-19	346	349	343	349	7488
2003-02-20	352	352	346	352	14040
2003-02-21	340	345	333	340	12480
2003-02-24	334	337	327	337	10296
2003-02-25	337	337	333	333	1716
2003-02-26	333	333	333	333	156
2003-02-27	333	333	327	327	468
2003-02-28	327	327	322	322	5928
2003-03-03	322	322	322	322	1404
2003-03-04	322	324	322	322	3120
2003-03-05	327	333	327	333	7020
2003-03-06	327	330	315	330	4992
2003-03-07	321	321	321	321	156
2003-03-10	314	335	314	335	16692
2003-03-11	321	321	309	321	3900
2003-03-12	314	322	314	322	2652
2003-03-13	322	322	314	314	2808
2003-03-14	314	314	308	314	11076
2003-03-17	314	316	314	314	7488
2003-03-18	314	321	314	321	780
2003-03-19	321	321	319	319	468
2003-03-20	321	332	321	332	7332
2003-03-24	327	327	326	326	1560
2003-03-25	324	328	324	328	7332
2003-03-26	327	327	327	327	1248
2003-03-27	327	327	327	327	936
2003-03-28	327	327	321	321	4836
2003-03-31	321	321	311	311	3276
2003-04-01	314	318	314	314	4212
2003-04-02	314	314	312	314	2964
2003-04-03	315	320	314	320	3744
2003-04-04	310	319	310	319	2028
2003-04-07	320	320	320	320	2340
2003-04-08	308	318	308	318	3432
2003-04-09	314	314	314	314	468
2003-04-10	314	318	308	318	6084
2003-04-14	320	320	318	318	1404
2003-04-15	320	320	308	308	5616
2003-04-16	311	311	308	308	11544
2003-04-17	311	314	308	314	5460
2003-04-18	314	318	314	318	7644
2003-04-21	314	318	311	318	936
2003-04-22	318	318	311	311	6708
2003-04-23	311	314	308	314	2964
2003-04-24	315	318	308	318	12636
2003-04-25	318	318	311	314	4836
2003-04-28	314	314	308	314	2496
2003-04-30	314	318	314	314	1248
2003-05-01	308	318	307	317	3900
2003-05-02	317	317	310	316	1560
2003-05-06	317	317	307	315	5148
2003-05-07	309	311	309	311	1092
2003-05-08	309	314	309	313	936
2003-05-09	313	313	303	310	18408
2003-05-12	314	314	308	308	1404
2003-05-13	309	311	308	308	9672
2003-05-14	308	308	307	307	7020
2003-05-15	306	308	306	308	2496
2003-05-16	313	313	302	303	21996
2003-05-19	303	305	302	302	1716
2003-05-20	307	307	304	304	11076
2003-05-21	305	306	305	306	1716
2003-05-22	305	306	305	306	4368
2003-05-23	307	307	307	307	3900
2003-05-26	322	329	322	324	54756
2003-05-27	324	324	319	319	10764
2003-05-28	318	321	318	321	11232
2003-05-29	318	318	314	314	8580
2003-05-30	318	318	314	314	6084
2003-06-02	321	335	320	335	30264
2003-06-03	340	340	322	322	4368
2003-06-04	329	329	324	328	4212
2003-06-05	328	328	325	325	8268
2003-06-06	328	328	325	327	12324
2003-06-09	328	328	325	325	6708
2003-06-10	315	327	315	327	6240
2003-06-11	325	327	325	327	9516
2003-06-12	327	327	324	324	7332
2003-06-13	324	326	324	326	4836
2003-06-16	327	327	322	322	14664
2003-06-17	325	333	325	333	22776
2003-06-18	335	337	333	333	7956
2003-06-19	340	340	335	335	18876
2003-06-20	327	338	327	337	11388
2003-06-23	338	338	337	337	3276
2003-06-24	334	340	330	330	10140
2003-06-25	332	340	332	340	8424
2003-06-26	335	337	334	334	4212
2003-06-27	338	345	338	345	11856
2003-06-30	345	352	343	352	7644
2003-07-01	352	359	352	358	37284
2003-07-02	362	362	346	352	35412
2003-07-03	352	352	330	340	21372
2003-07-04	339	341	337	338	7800
2003-07-07	352	358	346	358	19500
2003-07-08	359	359	333	346	3900
2003-07-09	346	349	346	349	3744
2003-07-10	350	352	337	341	8268
2003-07-11	340	341	337	341	4056
2003-07-14	342	352	342	352	8736
2003-07-15	352	352	342	342	11700
2003-07-16	340	340	337	338	3900
2003-07-17	333	334	333	334	1404
2003-07-18	336	337	336	337	9828
2003-07-22	343	343	328	328	3276
2003-07-23	331	333	327	327	9048
2003-07-24	346	346	330	330	3900
2003-07-25	331	342	327	334	11076
2003-07-28	328	333	328	329	15756
2003-07-29	331	331	328	330	11544
2003-07-30	337	337	330	331	3588
2003-07-31	335	335	327	327	5928
2003-08-01	333	333	327	328	2964
2003-08-04	333	340	330	333	18252
2003-08-05	334	335	330	333	14040
2003-08-06	333	334	333	333	3432
2003-08-07	333	333	325	325	6708
2003-08-08	326	333	326	333	5148
2003-08-11	345	345	340	341	9360
2003-08-12	329	333	329	333	2184
2003-08-13	333	334	333	334	1872
2003-08-14	333	343	333	341	21372
2003-08-15	341	349	338	349	44148
2003-08-18	359	359	346	352	16224
2003-08-19	349	352	345	349	29016
2003-08-20	353	353	346	348	18408
2003-08-21	333	333	322	333	41964
2003-08-22	333	334	332	333	5928
2003-08-25	333	333	330	330	15912
2003-08-26	328	331	327	329	29172
2003-08-27	329	331	329	331	4212
2003-08-28	332	333	328	329	11232
2003-08-29	329	332	329	329	3588
2003-09-01	332	338	330	338	14040
2003-09-02	339	339	333	338	10452
2003-09-03	333	339	333	339	3744
2003-09-04	339	339	338	339	6240
2003-09-05	342	343	339	341	13884
2003-09-08	345	345	341	341	9828
2003-09-09	341	341	340	340	5616
2003-09-10	340	340	333	333	6084
2003-09-11	333	345	333	341	16692
2003-09-12	345	405	344	405	269880
2003-09-16	410	417	402	402	380172
2003-09-17	402	403	388	388	66612
2003-09-18	382	436	382	436	170976
2003-09-19	422	429	405	422	88608
2003-09-22	417	417	401	410	27456
2003-09-24	404	410	401	401	19344
2003-09-25	395	407	382	386	40560
2003-09-26	398	401	387	401	7020
2003-09-29	407	410	401	410	15600
2003-09-30	410	416	397	397	20904
2003-10-01	397	410	393	410	39468
2003-10-02	403	414	399	406	24180
2003-10-03	399	409	399	406	7644
2003-10-06	404	411	395	411	23556
2003-10-07	410	410	401	401	1872
2003-10-08	401	407	391	401	17472
2003-10-09	401	401	395	400	8736
2003-10-10	389	397	383	388	21996
2003-10-14	389	395	385	388	24804
2003-10-15	394	394	385	389	14196
2003-10-16	388	404	388	404	19344
2003-10-17	404	405	393	398	15444
2003-10-20	398	398	385	395	14196
2003-10-21	398	398	388	391	22152
2003-10-22	391	391	386	386	11232
2003-10-23	386	390	382	390	7332
2003-10-24	391	391	382	387	4836
2003-10-27	372	378	359	372	24336
2003-10-28	372	372	352	365	10452
2003-10-29	356	359	356	359	13572
2003-10-30	360	362	360	362	3276
2003-10-31	363	365	361	365	14040
2003-11-04	372	385	368	383	12480
2003-11-05	383	391	372	391	10764
2003-11-06	385	398	372	385	15132
2003-11-07	384	385	378	378	6864
2003-11-10	383	389	382	389	4368
2003-11-11	370	375	367	375	8580
2003-11-12	388	388	378	378	1560
2003-11-13	378	384	372	372	5616
2003-11-14	378	385	378	384	2184
2003-11-17	383	383	353	353	15912
2003-11-18	365	365	333	340	33228
2003-11-19	340	341	327	327	19968
2003-11-20	330	352	321	352	39000
2003-11-21	333	340	321	333	45864
2003-11-25	333	345	333	345	16068
2003-11-26	346	359	318	358	38688
2003-11-27	358	358	332	332	5772
2003-11-28	351	351	333	337	19812
2003-12-01	343	343	333	339	6864
2003-12-02	348	348	335	342	4992
2003-12-03	337	337	330	337	17940
2003-12-04	336	336	330	333	16692
2003-12-05	335	337	325	337	33696
2003-12-08	333	334	314	314	163176
2003-12-09	321	326	321	322	21684
2003-12-10	329	329	322	324	44616
2003-12-11	329	333	325	333	38064
2003-12-12	333	335	327	327	59904
2003-12-15	336	343	335	342	36816
2003-12-16	341	378	340	367	110916
2003-12-17	378	378	327	335	320112
2003-12-18	337	341	334	335	66300
2003-12-19	340	340	335	336	38064
2003-12-22	335	335	322	325	63960
2003-12-24	333	333	323	325	39624
2003-12-25	325	332	322	326	40716
2003-12-26	326	327	320	326	55848
2003-12-29	327	327	323	325	22932
2003-12-30	326	328	326	327	21216
2004-01-05	328	332	328	329	16536
2004-01-06	329	340	329	334	35412
2004-01-07	335	341	334	339	13884
2004-01-08	345	348	341	347	37908
2004-01-09	348	348	341	343	43368
2004-01-13	348	348	341	345	23244
2004-01-14	345	348	343	348	35724
2004-01-15	346	352	345	345	32292
2004-01-16	345	345	341	342	25584
2004-01-19	340	343	334	340	27924
2004-01-20	345	345	337	340	45396
2004-01-21	340	342	337	340	19656
2004-01-22	341	341	336	338	21684
2004-01-23	337	338	337	338	5460
2004-01-26	338	342	337	339	31356
2004-01-27	340	340	337	337	10764
2004-01-28	337	342	333	342	41496
2004-01-29	343	345	340	344	53508
2004-01-30	345	359	341	357	84552
2004-02-02	356	357	347	348	72072
2004-02-03	348	350	334	341	33852
2004-02-04	341	341	337	338	17784
2004-02-05	342	342	338	340	22152
2004-02-06	337	342	336	341	24492
2004-02-09	337	340	337	337	14352
2004-02-10	337	337	334	335	17316
2004-02-12	335	338	334	335	21372
2004-02-13	337	341	337	341	21840
2004-02-16	335	342	334	341	34320
2004-02-17	340	342	338	338	18252
2004-02-18	339	341	337	341	30420
2004-02-19	340	340	337	338	17316
2004-02-20	337	340	337	340	31980
2004-02-23	337	337	335	335	31980
2004-02-24	334	335	331	331	44772
2004-02-25	332	332	329	329	12168
2004-02-26	327	327	325	327	28236
2004-02-27	326	329	326	328	15444
2004-03-01	329	337	329	335	18408
2004-03-02	335	335	333	335	16068
2004-03-03	335	338	334	338	22620
2004-03-04	338	338	337	338	22932
2004-03-05	338	338	337	337	13260
2004-03-08	337	338	337	338	14508
2004-03-09	338	340	338	340	26052
2004-03-10	340	346	340	346	36192
2004-03-11	344	346	343	344	24492
2004-03-12	345	345	339	340	23400
2004-03-15	340	345	339	340	27456
2004-03-16	344	344	339	342	37440
2004-03-17	343	343	340	342	44928
2004-03-18	342	342	339	339	33540
2004-03-19	343	343	340	341	19968
2004-03-22	341	344	341	344	7332
2004-03-23	344	345	343	343	12168
2004-03-24	345	348	344	348	20124
2004-03-25	347	350	346	349	43524
2004-03-26	346	348	342	346	12792
2004-03-29	345	345	341	343	17316
2004-03-30	345	345	342	343	6708
2004-03-31	342	343	342	342	18720
2004-04-01	343	343	342	342	23556
2004-04-02	345	345	341	342	16848
2004-04-05	343	345	342	344	23244
2004-04-06	345	346	342	343	69888
2004-04-07	345	345	343	345	20748
2004-04-08	346	347	345	345	13416
2004-04-09	344	345	341	341	18720
2004-04-12	343	345	342	342	7488
2004-04-13	345	346	344	345	22464
2004-04-14	345	346	345	345	28392
2004-04-15	346	346	345	345	35568
2004-04-16	345	345	343	343	9204
2004-04-19	345	346	345	345	17160
2004-04-20	348	349	346	348	33384
2004-04-21	349	349	346	348	33696
2004-04-22	350	356	349	356	66144
2004-04-23	362	370	359	368	102024
2004-04-26	385	385	372	382	151788
2004-04-27	378	381	368	372	90324
2004-04-28	372	372	366	372	23868
2004-04-30	372	373	368	373	48516
2004-05-06	375	377	372	377	43056
2004-05-07	375	375	365	365	21996
2004-05-10	349	361	343	360	33540
2004-05-11	343	348	343	345	15756
2004-05-12	348	352	347	352	7488
2004-05-13	352	352	349	352	9048
2004-05-14	343	345	343	345	20124
2004-05-17	333	337	321	328	23868
2004-05-18	324	334	322	334	37440
2004-05-19	334	359	334	343	18876
2004-05-20	358	358	352	352	5772
2004-05-21	356	356	340	352	6396
2004-05-24	352	352	344	346	8580
2004-05-25	352	352	344	352	1248
2004-05-26	359	359	352	355	21372
2004-05-27	355	355	349	349	6396
2004-05-28	349	350	346	346	9984
2004-05-31	345	348	344	348	11700
2004-06-01	345	348	343	348	2808
2004-06-02	346	346	345	345	4524
2004-06-03	342	342	341	342	10452
2004-06-04	342	342	340	340	16224
2004-06-07	342	349	341	349	10920
2004-06-08	350	351	347	347	4056
2004-06-09	347	348	337	337	21996
2004-06-10	341	346	337	337	23556
2004-06-11	338	345	337	345	21684
2004-06-14	347	347	343	343	7488
2004-06-15	345	346	343	344	6240
2004-06-16	346	346	343	345	13728
2004-06-17	345	346	345	345	15756
2004-06-18	347	351	346	350	37752
2004-06-21	356	368	355	364	55224
2004-06-22	360	364	352	354	37128
2004-06-23	354	356	352	352	7332
2004-06-24	353	358	352	357	13104
2004-06-25	359	364	356	364	41028
2004-06-28	365	371	365	367	30108
2004-06-29	365	368	362	362	12168
2004-06-30	365	365	359	359	11700
2004-07-01	365	365	360	360	15288
2004-07-02	365	365	357	358	23712
2004-07-05	357	357	352	352	10608
2004-07-06	353	357	353	353	7020
2004-07-07	354	363	352	363	10452
2004-07-08	356	362	356	362	5772
2004-07-09	364	364	359	362	7800
2004-07-12	362	364	361	364	12636
2004-07-13	362	362	359	362	15444
2004-07-14	362	391	362	373	87984
2004-07-15	385	388	378	382	105456
2004-07-16	384	405	384	398	299052
2004-07-20	410	435	408	435	678288
2004-07-21	429	499	425	499	1796028
2004-07-22	506	564	501	530	1952496
2004-07-23	518	525	482	503	1171560
2004-07-26	513	564	513	564	2156544
2004-07-27	583	588	504	529	2136576
2004-07-28	545	551	503	533	1550328
2004-07-29	527	529	489	489	906984
2004-07-30	487	512	475	508	996372
2004-08-02	508	516	488	492	623844
2004-08-03	491	500	473	475	482352
2004-08-04	468	491	463	481	390312
2004-08-05	475	510	468	498	398424
2004-08-06	479	497	478	488	199368
2004-08-09	479	498	478	491	192504
2004-08-10	495	500	487	489	161460
2004-08-11	495	496	490	491	80652
2004-08-12	492	499	491	496	146952
2004-08-13	495	525	491	519	638508
2004-08-16	532	537	525	535	661284
2004-08-17	537	544	522	525	445848
2004-08-18	519	519	500	506	246480
2004-08-19	506	514	502	514	180492
2004-08-20	508	518	502	512	186264
2004-08-23	486	497	480	482	382356
2004-08-24	481	481	456	456	359424
2004-08-25	449	468	429	464	439920
2004-08-26	467	475	452	453	173784
2004-08-27	451	473	451	472	203268
2004-08-30	468	471	462	469	95472
2004-08-31	468	468	462	463	39312
2004-09-01	462	466	462	463	52104
2004-09-02	464	466	462	465	39312
2004-09-03	468	472	465	466	77376
2004-09-06	468	473	468	472	73632
2004-09-07	476	476	459	464	91884
2004-09-08	465	465	456	461	79404
2004-09-09	461	461	452	456	75036
2004-09-10	452	454	439	445	134316
2004-09-13	439	447	434	437	91260
2004-09-14	435	440	431	440	75660
2004-09-15	452	475	448	456	243672
2004-09-16	450	455	449	451	101556
2004-09-17	455	458	451	451	30576
2004-09-21	455	455	450	452	34788
2004-09-22	451	453	451	452	52260
2004-09-24	452	454	448	454	72540
2004-09-27	453	465	451	462	73008
2004-09-28	464	465	453	453	61620
2004-09-29	453	456	444	444	62712
2004-09-30	440	445	439	443	26676
2004-10-01	445	450	445	448	28548
2004-10-04	452	452	446	449	34944
2004-10-05	451	452	449	450	20904
2004-10-06	450	456	450	456	28548
2004-10-07	456	456	451	453	17940
2004-10-08	452	456	449	452	30264
2004-10-12	449	452	441	441	63648
2004-10-13	442	445	439	440	44616
2004-10-14	439	442	436	438	17628
2004-10-15	436	441	435	440	27144
2004-10-18	436	437	435	435	25896
2004-10-19	436	439	433	433	86892
2004-10-20	436	436	429	430	60060
2004-10-21	432	462	432	433	190944
2004-10-22	434	469	434	468	260364
2004-10-25	442	478	437	471	425568
2004-10-26	472	479	463	468	172692
2004-10-27	471	477	460	466	75348
2004-10-28	472	472	464	469	69732
2004-10-29	472	472	460	464	38376
2004-11-01	456	475	456	471	87672
2004-11-02	477	477	468	472	36504
2004-11-04	476	476	470	470	68172
2004-11-05	474	474	465	469	38844
2004-11-08	475	475	465	468	53508
2004-11-09	464	481	464	481	124332
2004-11-10	493	513	479	490	377208
2004-11-11	481	497	481	483	78156
2004-11-12	483	521	483	521	596544
2004-11-15	527	536	514	525	577668
2004-11-16	525	528	503	518	366600
2004-11-17	511	535	511	532	379392
2004-11-18	535	544	517	525	327444
2004-11-19	524	528	505	518	167856
2004-11-22	512	513	492	503	140088
2004-11-24	502	532	498	505	447876
2004-11-25	510	512	498	502	144300
2004-11-26	495	507	487	488	112320
2004-11-29	487	508	487	496	82212
2004-11-30	502	502	497	499	67080
2004-12-01	501	502	495	495	56472
2004-12-02	501	501	490	497	48048
2004-12-03	494	495	491	491	31824
2004-12-06	487	494	487	487	89232
2004-12-07	491	497	485	485	76128
2004-12-08	481	487	481	486	48204
2004-12-09	486	487	468	474	100152
2004-12-10	468	473	462	465	83304
2004-12-13	472	472	449	462	196716
2004-12-14	455	462	451	453	67704
2004-12-15	453	456	448	456	66300
2004-12-16	456	456	448	450	54444
2004-12-17	454	455	450	455	42744
2004-12-20	459	468	456	462	100620
2004-12-21	467	467	457	461	60216
2004-12-22	465	481	460	477	90948
2004-12-24	482	482	465	466	199992
2004-12-27	473	473	442	465	107280
2004-12-28	467	473	466	472	68280
2004-12-29	473	477	465	468	111360
2004-12-30	476	476	468	468	34680
2005-01-04	485	486	473	486	32640
2005-01-05	490	492	483	485	95640
2005-01-06	483	483	474	483	56520
2005-01-07	479	479	475	476	41520
2005-01-11	483	483	475	480	34800
2005-01-12	480	481	477	478	28320
2005-01-13	477	479	476	478	18840
2005-01-14	476	479	476	479	24720
2005-01-17	479	481	476	476	36480
2005-01-18	476	478	475	475	25920
2005-01-19	475	476	471	471	38280
2005-01-20	472	472	463	464	50760
2005-01-21	463	467	460	467	87960
2005-01-24	467	481	463	475	56640
2005-01-25	481	481	469	475	49320
2005-01-26	473	474	463	473	87240
2005-01-27	467	472	465	468	43920
2005-01-28	467	470	463	470	26520
2005-01-31	467	477	467	475	47880
2005-02-01	475	477	471	475	28200
2005-02-02	478	478	471	471	29280
2005-02-03	470	475	465	473	32760
2005-02-04	475	477	465	465	53400
2005-02-07	456	463	443	463	130800
2005-02-08	463	468	459	468	53160
2005-02-09	463	470	463	465	32520
2005-02-10	464	467	464	467	20760
2005-02-14	467	473	466	472	30240
2005-02-15	472	474	467	473	25200
2005-02-16	473	473	465	470	17520
2005-02-17	464	471	464	470	24480
2005-02-18	473	473	467	471	23520
2005-02-21	471	471	466	466	44160
2005-02-22	465	467	458	459	41880
2005-02-23	463	467	458	460	37560
2005-02-24	461	468	456	458	31440
2005-02-25	458	465	456	459	25320
2005-02-28	458	463	458	459	29520
2005-03-01	460	463	460	463	25800
2005-03-02	465	465	458	459	77400
2005-03-03	459	462	454	460	67560
2005-03-04	461	463	457	461	33240
2005-03-07	461	465	458	461	37800
2005-03-08	461	463	455	461	33000
2005-03-09	461	464	460	463	17760
2005-03-10	463	465	463	465	28440
2005-03-11	465	466	463	466	19920
2005-03-14	471	474	468	469	64560
2005-03-15	475	478	471	473	63120
2005-03-16	475	479	473	478	45360
2005-03-17	478	480	476	476	51480
2005-03-18	479	488	478	487	88440
2005-03-22	488	490	487	488	136920
2005-03-23	490	492	488	490	95160
2005-03-24	490	491	486	486	34440
2005-03-25	483	483	474	478	50880
2005-03-28	458	476	458	473	24360
2005-03-29	467	473	463	463	21840
2005-03-30	463	476	463	476	19320
2005-03-31	476	476	462	473	9840
2005-04-01	459	467	459	465	15120
2005-04-04	464	465	459	460	15720
2005-04-05	463	463	459	462	17160
2005-04-06	462	464	458	458	24000
2005-04-07	462	464	461	463	6960
2005-04-08	463	465	462	463	15960
2005-04-11	472	472	463	463	19560
2005-04-12	467	467	463	463	13560
2005-04-13	462	463	461	461	19800
2005-04-14	461	462	454	456	39120
2005-04-15	454	458	451	454	10680
2005-04-18	429	437	425	429	71640
2005-04-19	429	442	429	435	18480
2005-04-20	450	453	442	452	20400
2005-04-21	450	450	435	439	16440
2005-04-22	446	446	434	435	30600
2005-04-25	436	437	433	436	10920
2005-04-26	442	442	433	436	4920
2005-04-27	438	439	435	435	5400
2005-04-28	435	438	434	434	6480
2005-05-02	433	441	433	435	5520
2005-05-06	442	442	429	432	30720
2005-05-09	433	433	417	425	45720
2005-05-10	419	423	418	420	32400
2005-05-11	423	425	419	419	22320
2005-05-12	420	423	420	423	18240
2005-05-13	421	423	421	423	11400
2005-05-16	421	429	419	419	25680
2005-05-17	419	423	418	418	16440
2005-05-18	417	424	417	422	14760
2005-05-19	423	430	423	426	20040
2005-05-20	433	434	427	432	16800
2005-05-23	440	462	440	453	96240
2005-05-24	463	471	463	471	112680
2005-05-25	471	473	467	468	145560
2005-05-26	467	467	442	458	53280
2005-05-27	450	450	444	444	11760
2005-05-30	443	446	438	439	43560
2005-05-31	442	449	442	442	16920
2005-06-01	446	446	442	444	15000
2005-06-02	448	449	443	448	13200
2005-06-03	443	449	443	443	11280
2005-06-06	448	451	446	451	18000
2005-06-07	458	463	448	449	39600
2005-06-08	446	454	446	449	12600
2005-06-09	448	450	446	450	4920
2005-06-10	453	461	453	458	23160
2005-06-13	458	473	458	463	102240
2005-06-14	468	468	459	459	35520
2005-06-15	462	462	458	462	28080
2005-06-16	462	462	458	459	34200
2005-06-17	463	463	458	459	32520
2005-06-20	463	463	459	460	33120
2005-06-21	462	462	458	460	14280
2005-06-22	463	471	462	465	93120
2005-06-23	467	469	463	463	20160
2005-06-24	460	463	459	463	20400
2005-06-27	458	459	454	457	21240
2005-06-28	457	470	457	459	21480
2005-06-29	466	466	458	462	14160
2005-06-30	462	464	458	464	18720
2005-07-01	465	465	458	460	7800
2005-07-04	461	463	460	460	20760
2005-07-05	463	463	460	460	7920
2005-07-06	460	462	459	461	16200
2005-07-07	461	461	458	460	18840
2005-07-08	460	460	458	459	15600
2005-07-11	460	462	453	454	24240
2005-07-12	451	455	450	453	29160
2005-07-13	453	453	451	452	21000
2005-07-14	452	453	450	450	19080
2005-07-15	450	453	450	452	8400
2005-07-19	451	453	451	451	16080
2005-07-20	453	454	451	453	24480
2005-07-21	454	454	451	451	9240
2005-07-22	451	453	450	450	13920
2005-07-25	450	450	434	447	32400
2005-07-26	445	447	440	443	24360
2005-07-27	443	445	443	443	21720
2005-07-28	443	450	443	446	21000
2005-07-29	446	453	446	448	14880
2005-08-01	459	460	450	454	38160
2005-08-02	455	455	449	452	12600
2005-08-03	450	453	448	448	8520
2005-08-04	448	448	441	442	13440
2005-08-05	443	443	431	433	37560
2005-08-08	418	425	409	421	58440
2005-08-09	426	428	421	423	68400
2005-08-10	427	436	427	431	59880
2005-08-11	436	436	429	436	60360
2005-08-12	431	433	429	429	43440
2005-08-15	432	433	429	429	35760
2005-08-16	431	433	428	428	38880
2005-08-17	427	435	427	431	43680
2005-08-18	430	433	430	433	33960
2005-08-19	433	438	431	431	43080
2005-08-22	432	433	430	431	46680
2005-08-23	433	433	431	432	42120
2005-08-24	432	433	432	433	29280
2005-08-25	435	435	433	433	20880
2005-08-26	436	438	434	435	40440
2005-08-29	438	438	436	438	45000
2005-08-30	439	441	438	438	43200
2005-08-31	438	441	438	440	31920
2005-09-01	441	446	440	441	71520
2005-09-02	442	447	442	447	48240
2005-09-05	450	450	446	449	55320
2005-09-06	449	450	447	448	31560
2005-09-07	450	453	448	450	42000
2005-09-08	455	455	449	451	43560
2005-09-09	452	452	449	449	58560
2005-09-12	453	457	452	457	25440
2005-09-13	463	465	460	465	66120
2005-09-14	467	492	467	490	363960
2005-09-15	490	490	468	473	279000
2005-09-16	475	475	463	467	113640
2005-09-20	467	467	458	463	89520
2005-09-21	465	468	463	466	63360
2005-09-22	465	468	463	463	38520
2005-09-26	460	468	460	466	57120
2005-09-27	464	470	463	466	39720
2005-09-28	465	472	464	472	45000
2005-09-29	473	474	469	469	57000
2005-09-30	469	471	466	466	29760
2005-10-03	467	468	463	467	27240
2005-10-04	464	467	463	465	28320
2005-10-05	463	467	461	463	23040
2005-10-06	461	465	459	459	29880
2005-10-07	461	464	458	462	25200
2005-10-11	459	465	458	463	16320
2005-10-12	463	465	463	464	19800
2005-10-13	463	465	463	463	19920
2005-10-14	463	466	463	464	29040
2005-10-17	467	470	464	466	48240
2005-10-18	469	469	466	468	28200
2005-10-19	468	468	466	467	18360
2005-10-20	467	468	466	468	18000
2005-10-21	467	467	464	466	17280
2005-10-24	466	467	464	464	7320
2005-10-25	464	465	464	465	8880
2005-10-26	465	465	459	459	20160
2005-10-27	459	463	459	463	25200
2005-10-28	466	466	462	463	46200
2005-10-31	454	457	446	454	98640
2005-11-01	452	453	451	451	17160
2005-11-02	450	451	448	450	54600
2005-11-04	449	452	448	452	44040
2005-11-07	448	452	448	450	31200
2005-11-08	450	454	450	454	27960
2005-11-09	456	457	455	457	15360
2005-11-10	457	458	453	455	26040
2005-11-11	455	455	451	453	19080
2005-11-14	453	454	451	451	15240
2005-11-15	451	451	448	449	29160
2005-11-16	448	448	446	448	22800
2005-11-17	446	446	438	445	42000
2005-11-18	445	446	442	442	49800
2005-11-21	442	445	442	445	29040
2005-11-22	444	446	443	443	21360
2005-11-24	443	443	442	442	53880
2005-11-25	442	443	442	442	18960
2005-11-28	442	443	440	440	32760
2005-11-29	440	442	438	438	35640
2005-11-30	438	442	438	438	32160
2005-12-01	440	441	438	438	35280
2005-12-02	437	440	437	439	43560
2005-12-05	441	442	439	441	52800
2005-12-06	442	445	442	443	65160
2005-12-07	444	454	443	447	86160
2005-12-08	450	452	448	448	31680
2005-12-09	450	451	447	450	22680
2005-12-12	450	450	447	448	39600
2005-12-13	447	450	444	448	44280
2005-12-14	449	450	444	446	39480
2005-12-15	446	446	443	444	16920
2005-12-16	444	445	442	443	26880
2005-12-19	441	442	440	441	26160
2005-12-20	439	443	438	440	48840
2005-12-21	440	441	438	441	71760
2005-12-22	442	443	438	442	68160
2005-12-26	443	443	440	441	41760
2005-12-27	438	443	438	443	48120
2005-12-28	443	443	442	442	13680
2005-12-29	442	442	440	442	45840
2005-12-30	443	443	440	441	14280
2006-01-04	441	442	440	442	10440
2006-01-05	442	444	441	444	35400
2006-01-06	446	446	444	445	40200
2006-01-10	448	453	448	453	70920
2006-01-11	453	454	444	448	81480
2006-01-12	448	450	447	448	27600
2006-01-13	448	450	446	446	27720
2006-01-16	447	447	443	446	32880
2006-01-17	444	444	440	440	43800
2006-01-18	434	439	423	425	87840
2006-01-19	419	435	419	430	37440
2006-01-20	444	444	438	438	23400
2006-01-23	433	434	425	429	30720
2006-01-24	428	432	427	432	15840
2006-01-25	432	433	429	429	21960
2006-01-26	430	432	429	431	25080
2006-01-27	431	433	428	433	30960
2006-01-30	438	441	433	439	54480
2006-01-31	438	439	438	438	20640
2006-02-01	439	441	433	436	44640
2006-02-02	441	441	436	439	29520
2006-02-03	437	439	435	436	17160
2006-02-06	436	438	435	438	24120
2006-02-07	435	437	435	436	16200
2006-02-08	436	436	434	435	16800
2006-02-09	438	438	436	438	14880
2006-02-10	439	439	436	437	24480
2006-02-13	433	433	428	430	49320
2006-02-14	423	428	417	428	67680
2006-02-15	425	426	423	424	12480
2006-02-16	426	429	419	428	26880
2006-02-17	428	429	418	422	33360
2006-02-20	423	423	416	417	50520
2006-02-21	413	419	412	419	46800
2006-02-22	419	421	413	413	27840
2006-02-23	417	418	416	416	23280
2006-02-24	418	424	417	423	16440
2006-02-27	422	424	421	423	21960
2006-02-28	423	425	422	425	21720
2006-03-01	428	428	424	426	19200
2006-03-02	427	428	420	420	20160
2006-03-03	421	424	418	418	18000
2006-03-06	418	422	418	418	16560
2006-03-07	419	420	418	418	15240
2006-03-08	417	423	417	421	12120
2006-03-09	418	419	418	419	22320
2006-03-10	420	423	418	421	36360
2006-03-13	425	425	419	423	18480
2006-03-14	423	424	419	421	11880
2006-03-15	423	423	420	420	6000
2006-03-16	421	422	421	421	10320
2006-03-17	423	423	420	421	18240
2006-03-20	424	427	424	424	36600
2006-03-22	425	427	425	427	13200
2006-03-23	427	431	425	431	34560
2006-03-24	431	431	427	427	32760
2006-03-27	429	431	427	430	51240
2006-03-28	426	426	420	421	23640
2006-03-29	422	423	418	422	16440
2006-03-30	422	422	420	422	14160
2006-03-31	422	423	419	423	17760
2006-04-03	423	423	421	422	12240
2006-04-04	423	423	422	423	19080
2006-04-05	423	423	422	422	26040
2006-04-06	422	423	418	418	19680
2006-04-07	419	420	419	420	14520
2006-04-10	421	421	419	419	15600
2006-04-11	420	421	420	420	11640
2006-04-12	419	421	418	419	55320
2006-04-13	419	420	418	418	8040
2006-04-14	419	420	418	419	16680
2006-04-17	418	419	417	418	82680
2006-04-18	418	419	417	418	30240
2006-04-19	420	423	419	421	49320
2006-04-20	426	456	426	456	504000
2006-04-21	443	452	434	449	265200
2006-04-24	446	446	426	442	127560
2006-04-25	441	441	428	428	74640
2006-04-26	428	441	428	438	38760
2006-04-27	439	446	437	446	52920
2006-04-28	438	442	434	442	46320
2006-05-01	442	442	437	438	65880
2006-05-02	440	441	434	438	35880
2006-05-08	439	441	435	438	24480
2006-05-09	438	438	436	437	10200
2006-05-10	435	438	434	434	28680
2006-05-11	438	438	431	431	30120
2006-05-12	429	432	422	427	27000
2006-05-15	428	433	428	429	17640
2006-05-16	437	439	425	428	49560
2006-05-17	421	425	414	422	56160
2006-05-18	418	433	418	421	20400
2006-05-19	432	433	425	429	22200
2006-05-22	442	444	429	437	69240
2006-05-23	433	437	425	432	20160
2006-05-24	425	429	423	429	12960
2006-05-25	430	430	426	428	15720
2006-05-26	428	428	417	419	30000
2006-05-29	422	428	421	428	17280
2006-05-30	427	427	419	423	14160
2006-05-31	418	420	417	418	27840
2006-06-01	418	423	413	414	62520
2006-06-02	410	421	388	405	64440
2006-06-05	413	414	405	410	36720
2006-06-06	406	406	392	406	53880
2006-06-07	408	408	400	403	37320
2006-06-08	388	395	383	387	63480
2006-06-09	386	404	383	400	95400
2006-06-12	401	401	388	394	48600
2006-06-13	399	399	385	386	29400
2006-06-14	386	388	383	384	44520
2006-06-15	389	393	384	386	43320
2006-06-16	390	398	388	396	48360
2006-06-19	397	403	394	398	46920
2006-06-20	402	404	394	403	44400
2006-06-21	404	404	398	398	32880
2006-06-22	398	400	396	398	16920
2006-06-23	408	408	400	405	21960
2006-06-26	408	412	408	411	21720
2006-06-27	413	415	408	411	22080
2006-06-28	412	412	407	411	7560
2006-06-29	411	418	408	413	34200
2006-06-30	413	418	413	414	25920
2006-07-03	417	420	416	417	28680
2006-07-04	417	420	417	418	17880
2006-07-05	417	418	415	415	17280
2006-07-06	416	417	414	416	12480
2006-07-07	417	417	413	413	13800
2006-07-10	413	413	409	413	21360
2006-07-11	413	416	411	416	9120
2006-07-12	416	416	413	416	18120
2006-07-13	413	415	405	406	33000
2006-07-14	411	412	402	411	23040
2006-07-18	404	404	393	399	35160
2006-07-19	398	398	379	390	61200
2006-07-20	413	413	401	410	69000
2006-07-21	410	410	405	406	14400
2006-07-24	406	408	403	404	13080
2006-07-25	409	410	403	405	9840
2006-07-26	404	405	400	400	13320
2006-07-27	396	402	396	398	14040
2006-07-28	400	404	398	403	9840
2006-07-31	403	408	402	406	41640
2006-08-01	407	407	401	402	14040
2006-08-02	406	408	400	400	61560
2006-08-03	404	408	403	407	24120
2006-08-04	408	413	405	413	40440
2006-08-07	411	411	400	401	44640
2006-08-08	403	407	401	403	20160
2006-08-09	403	408	403	405	8400
2006-08-10	408	408	405	408	16080
2006-08-11	408	411	405	407	17160
2006-08-14	412	412	408	408	8280
2006-08-15	409	413	409	410	8880
2006-08-16	410	413	410	412	27240
2006-08-17	412	413	409	410	15600
2006-08-18	413	413	408	408	36840
2006-08-21	412	416	408	416	25920
2006-08-22	416	417	410	415	27600
2006-08-23	416	416	413	413	12600
2006-08-24	413	414	413	414	11880
2006-08-25	414	415	413	414	26880
2006-08-28	415	415	414	414	9720
2006-08-29	415	415	413	413	11280
2006-08-30	414	415	413	413	4200
2006-08-31	413	413	411	411	10920
2006-09-01	412	414	412	413	7800
2006-09-04	413	415	412	414	13920
2006-09-05	415	415	413	414	18720
2006-09-06	411	414	410	413	12000
2006-09-07	413	414	412	414	9000
2006-09-08	414	414	409	410	12000
2006-09-11	414	414	411	413	4920
2006-09-12	413	413	410	411	14400
2006-09-13	412	413	408	409	21600
2006-09-14	408	408	405	405	10920
2006-09-15	408	408	406	407	9480
2006-09-19	408	409	407	408	15960
2006-09-20	409	409	408	408	16560
2006-09-21	409	409	408	408	6480
2006-09-22	410	410	407	408	18000
2006-09-25	408	408	405	405	14520
2006-09-26	405	408	399	399	18120
2006-09-27	391	407	391	397	20760
2006-09-28	401	406	400	404	6120
2006-09-29	403	404	403	403	5040
2006-10-02	401	408	401	408	9600
2006-10-03	408	408	403	403	7320
2006-10-04	408	408	401	401	5880
2006-10-05	403	404	401	401	11880
2006-10-06	404	407	401	407	10440
2006-10-10	406	406	401	405	10080
2006-10-11	403	404	398	399	3840
2006-10-12	398	403	396	396	21360
2006-10-13	399	403	398	403	7680
2006-10-16	403	403	400	402	7800
2006-10-17	402	403	398	398	20040
2006-10-18	397	398	396	397	6960
2006-10-19	398	399	396	399	11280
2006-10-20	399	399	398	399	11760
2006-10-23	402	403	398	403	10320
2006-10-24	403	403	400	401	9720
2006-10-25	400	402	398	398	10320
2006-10-26	398	400	398	399	3360
2006-10-27	398	401	397	399	11880
2006-10-30	400	400	393	398	14640
2006-10-31	400	400	395	395	6240
2006-11-01	396	398	395	396	4560
2006-11-02	396	397	394	395	16800
2006-11-06	396	396	392	395	7440
2006-11-07	395	396	393	395	8280
2006-11-08	394	395	392	394	7440
2006-11-09	393	393	392	392	12000
2006-11-10	393	393	391	393	11520
2006-11-13	395	395	391	391	11160
2006-11-14	393	393	391	391	9120
2006-11-15	393	393	390	391	4080
2006-11-16	391	392	388	389	5520
2006-11-17	388	391	380	387	16320
2006-11-20	385	385	379	379	10800
2006-11-21	378	379	376	377	5400
2006-11-22	377	381	375	381	9960
2006-11-24	375	381	375	381	18360
2006-11-27	379	380	378	380	16440
2006-11-28	381	381	379	381	5520
2006-11-29	379	382	379	381	7200
2006-11-30	379	382	376	382	19800
2006-12-01	378	382	377	382	12000
2006-12-04	382	382	379	379	6600
2006-12-05	380	382	379	381	21720
2006-12-06	382	383	375	383	28560
2006-12-07	382	382	378	380	13440
2006-12-08	383	388	380	384	36840
2006-12-11	384	390	383	388	23640
2006-12-12	390	390	388	388	7800
2006-12-13	386	390	383	387	13680
2006-12-14	388	396	387	391	34920
2006-12-15	396	397	388	396	29160
2006-12-18	398	398	390	390	21600
2006-12-19	388	391	385	385	18720
2006-12-20	381	388	381	386	22080
2006-12-21	388	389	384	388	17520
2006-12-22	387	388	383	387	23040
2006-12-25	387	388	383	383	14160
2006-12-26	385	388	383	384	11640
2006-12-27	384	388	384	387	3120
2006-12-28	387	387	386	387	3360
2006-12-29	387	388	387	388	4320
2007-01-04	388	389	388	388	3960
2007-01-05	390	390	385	388	13800
2007-01-09	390	392	389	389	12840
2007-01-10	388	391	388	388	5640
2007-01-11	392	392	389	391	10680
2007-01-12	392	392	388	388	10440
2007-01-15	392	393	388	389	12120
2007-01-16	391	393	390	393	15000
2007-01-17	393	393	390	391	10440
2007-01-18	392	394	391	393	15240
2007-01-19	396	396	392	393	26400
2007-01-22	388	393	387	392	58800
2007-01-23	389	393	389	391	19920
2007-01-24	393	393	388	388	25080
2007-01-25	388	389	388	388	14520
2007-01-26	388	392	387	392	14400
2007-01-29	388	393	387	388	24960
2007-01-30	388	389	388	388	9360
2007-01-31	388	389	386	389	16800
2007-02-01	388	389	387	388	11280
2007-02-02	391	391	388	388	23880
2007-02-05	388	391	388	390	19800
2007-02-06	389	390	388	390	15000
2007-02-07	388	390	388	388	4200
2007-02-08	388	389	388	388	19440
2007-02-09	388	389	387	388	4800
2007-02-13	388	388	387	388	8040
2007-02-14	387	388	386	388	7680
2007-02-15	388	389	387	388	9840
2007-02-16	388	389	387	388	8880
2007-02-19	388	388	386	387	9840
2007-02-20	388	388	383	387	12000
2007-02-21	385	387	384	387	9240
2007-02-22	383	388	383	386	11280
2007-02-23	386	387	385	387	10200
2007-02-26	386	388	386	386	8760
2007-02-27	386	388	385	387	16680
2007-02-28	384	387	379	382	33480
2007-03-01	385	385	382	383	6240
2007-03-02	383	383	383	383	5640
2007-03-05	383	383	380	381	12480
2007-03-06	383	383	378	381	17760
2007-03-07	383	383	378	378	9480
2007-03-08	378	381	378	380	9960
2007-03-09	381	382	380	380	5520
2007-03-12	380	386	380	384	11160
2007-03-13	388	390	383	385	16200
2007-03-14	383	388	383	383	11280
2007-03-15	385	389	384	388	10560
2007-03-16	387	389	383	388	16440
2007-03-19	388	391	386	389	11880
2007-03-20	390	392	388	392	21600
2007-03-22	383	384	381	381	66600
2007-03-23	381	390	381	386	44880
2007-03-26	387	387	383	383	15360
2007-03-27	379	379	378	378	9480
2007-03-28	379	380	375	375	20760
2007-03-29	375	378	373	373	19560
2007-03-30	373	378	373	377	9480
2007-04-02	373	379	373	374	14880
2007-04-03	378	378	372	377	8160
2007-04-04	376	376	374	374	1320
2007-04-05	373	374	371	373	7800
2007-04-06	375	375	368	370	10440
2007-04-09	368	370	365	365	28320
2007-04-10	365	365	358	358	37200
2007-04-11	360	363	358	360	14280
2007-04-12	360	367	358	367	37200
2007-04-13	369	370	368	370	12960
2007-04-16	369	371	369	370	9600
2007-04-17	371	371	363	363	22560
2007-04-18	363	364	358	358	9000
2007-04-19	357	358	353	355	14160
2007-04-20	358	362	354	357	11160
2007-04-23	354	358	351	354	16680
2007-04-24	353	354	353	354	6360
2007-04-25	353	355	353	355	9720
2007-04-26	355	355	349	354	39360
2007-04-27	350	354	350	354	31440
2007-05-01	354	374	354	363	51480
2007-05-02	367	374	361	370	45720
2007-05-07	363	363	359	361	30000
2007-05-08	366	366	358	359	44400
2007-05-09	360	360	358	359	27240
2007-05-10	363	363	359	359	6960
2007-05-11	359	359	356	356	13440
2007-05-14	363	363	358	358	40200
2007-05-15	359	360	357	358	8640
2007-05-16	358	358	353	353	25560
2007-05-17	353	353	350	351	7080
2007-05-18	350	351	350	351	13920
2007-05-21	348	354	348	353	13440
2007-05-22	352	353	348	353	6120
2007-05-23	350	358	350	358	8640
2007-05-24	355	357	355	356	6000
2007-05-25	356	358	354	354	24480
2007-05-28	353	356	353	354	16800
2007-05-29	354	355	353	353	7680
2007-05-30	353	353	353	353	7440
2007-05-31	353	357	353	357	14160
2007-06-01	358	358	353	358	8040
2007-06-04	357	358	356	356	6960
2007-06-05	357	358	353	358	46080
2007-06-06	358	358	353	358	21600
2007-06-07	358	358	355	355	13440
2007-06-08	353	359	353	359	11400
2007-06-11	359	360	358	359	18120
2007-06-12	360	360	358	358	9360
2007-06-13	357	358	357	358	16920
2007-06-14	360	360	356	356	7200
2007-06-15	357	360	357	359	3840
2007-06-18	358	361	358	360	9600
2007-06-19	360	361	358	360	13200
2007-06-20	363	363	361	361	13320
2007-06-21	361	363	360	362	16680
2007-06-22	359	362	358	362	12240
2007-06-25	361	362	358	358	9360
2007-06-26	360	361	358	358	12480
2007-06-27	358	359	358	359	5280
2007-06-28	359	359	358	358	4800
2007-06-29	360	360	354	359	10680
2007-07-02	359	360	358	358	12240
2007-07-03	355	360	355	358	11520
2007-07-04	358	359	358	358	4920
2007-07-05	359	359	358	359	5640
2007-07-06	358	359	358	358	4680
2007-07-09	358	359	358	358	6240
2007-07-10	355	359	355	356	19080
2007-07-11	357	359	357	358	5760
2007-07-12	354	358	354	356	24840
2007-07-13	354	359	354	354	10800
2007-07-17	351	353	351	353	5760
2007-07-18	353	356	353	356	4800
2007-07-19	356	358	354	358	3720
2007-07-20	358	358	357	357	4920
2007-07-23	357	357	354	354	4800
2007-07-24	355	355	353	354	9360
2007-07-25	354	355	354	354	2040
2007-07-26	354	354	353	353	3000
2007-07-27	353	353	350	351	12480
2007-07-30	352	352	330	346	16680
2007-07-31	350	355	346	353	10320
2007-08-01	350	350	343	343	13920
2007-08-02	356	356	348	350	14280
2007-08-03	351	351	344	348	2640
2007-08-06	343	343	342	343	15120
2007-08-07	342	343	342	342	6840
2007-08-08	343	343	342	342	9720
2007-08-09	342	343	334	335	21480
2007-08-10	333	334	333	333	33480
2007-08-13	326	329	322	329	25800
2007-08-14	327	328	321	322	13800
2007-08-15	319	319	317	318	7800
2007-08-16	317	318	313	313	21720
2007-08-17	313	313	307	307	19440
2007-08-20	319	328	315	315	16320
2007-08-21	316	329	314	329	19200
2007-08-22	333	338	333	336	18840
2007-08-23	333	334	320	320	17520
2007-08-24	331	333	320	330	11640
2007-08-27	336	336	323	323	15840
2007-08-28	323	324	308	317	26280
2007-08-29	317	322	317	317	8760
2007-08-30	318	318	317	317	7080
2007-08-31	318	318	317	318	5520
2007-09-03	318	318	317	317	8520
2007-09-04	318	318	317	317	22800
2007-09-05	320	322	307	308	24360
2007-09-06	309	312	307	311	13080
2007-09-07	312	312	308	309	8760
2007-09-10	309	310	308	308	4920
2007-09-11	308	308	306	307	7320
2007-09-12	307	308	307	307	4680
2007-09-13	303	303	298	299	19800
2007-09-14	298	303	282	303	31680
2007-09-18	299	299	289	298	7800
2007-09-19	297	298	296	298	11880
2007-09-20	292	298	288	298	26160
2007-09-21	293	298	283	288	12240
2007-09-25	288	288	280	287	18360
2007-09-26	286	298	286	297	11880
2007-09-27	296	297	296	297	8760
2007-09-28	298	298	296	296	9360
2007-10-01	295	297	295	296	47160
2007-10-02	298	298	288	296	17880
2007-10-03	296	298	290	293	27720
2007-10-04	294	297	292	297	22800
2007-10-05	296	297	293	293	10560
2007-10-09	293	298	293	297	14400
2007-10-10	297	304	297	303	75120
2007-10-11	318	322	309	313	33960
2007-10-12	318	321	313	318	38040
2007-10-15	318	321	317	318	22200
2007-10-16	321	321	316	316	21480
2007-10-17	316	318	315	318	25560
2007-10-18	315	316	315	316	6480
2007-10-19	317	317	315	316	13680
2007-10-22	317	317	301	315	46320
2007-10-23	307	317	306	317	23280
2007-10-24	311	313	309	311	19080
2007-10-25	308	313	303	311	30240
2007-10-26	311	312	311	311	3600
2007-10-29	311	311	309	309	8640
2007-10-30	308	309	307	308	4920
2007-10-31	308	308	304	304	7080
2007-11-01	307	308	302	303	13200
2007-11-02	303	304	298	303	13560
2007-11-05	307	307	288	304	21120
2007-11-06	304	305	304	304	6000
2007-11-07	304	306	302	302	7440
2007-11-08	296	306	293	303	19200
2007-11-09	300	303	300	303	3120
2007-11-12	298	304	293	304	15000
2007-11-13	296	304	296	303	8400
2007-11-14	298	304	298	304	5400
2007-11-15	302	304	302	304	9240
2007-11-16	300	307	300	305	21120
2007-11-19	303	306	303	305	5520
2007-11-20	313	313	302	307	10680
2007-11-21	307	309	304	306	3120
2007-11-22	309	310	308	309	3840
2007-11-26	311	316	308	308	4800
2007-11-27	309	314	309	313	2040
2007-11-28	313	316	313	315	3840
2007-11-29	317	323	315	316	19680
2007-11-30	324	324	317	321	9480
2007-12-03	321	323	320	321	4680
2007-12-04	324	325	321	321	10440
2007-12-05	325	325	323	323	4680
2007-12-06	325	325	321	323	7200
2007-12-07	325	325	321	321	5880
2007-12-10	321	323	318	318	5520
2007-12-11	318	322	313	319	24360
2007-12-12	317	324	314	319	14160
2007-12-13	314	315	300	302	20640
2007-12-14	308	320	308	317	15600
2007-12-17	316	317	313	315	8760
2007-12-18	309	317	309	312	13800
2007-12-19	313	317	310	314	11280
2007-12-20	317	317	308	313	13200
2007-12-21	305	308	296	304	37560
2007-12-25	315	328	315	317	79560
2007-12-26	320	329	320	327	44520
2007-12-27	330	333	326	332	38160
2007-12-28	331	333	327	328	15240
2008-01-04	332	332	326	332	14520
2008-01-07	333	358	333	337	38760
2008-01-08	341	346	341	345	26160
2008-01-09	342	344	337	343	29400
2008-01-10	339	340	336	336	11400
2008-01-11	333	338	325	335	18240
2008-01-15	331	334	321	321	23160
2008-01-16	308	318	292	317	24360
2008-01-17	316	321	306	316	12240
2008-01-18	321	321	314	320	15480
2008-01-21	317	320	308	317	12240
2008-01-22	309	310	303	309	14400
2008-01-23	305	310	300	304	29160
2008-01-24	304	308	303	307	10920
2008-01-25	307	313	307	310	18480
2008-01-28	313	317	312	315	5160
2008-01-29	317	318	317	317	10560
2008-01-30	320	321	314	317	16560
2008-01-31	317	317	313	313	2640
2008-02-01	317	321	313	316	9600
2008-02-04	325	325	315	315	13560
2008-02-05	318	318	313	314	4800
2008-02-06	310	311	304	304	14640
2008-02-07	304	304	300	301	9360
2008-02-08	303	308	300	302	8040
2008-02-12	293	321	292	313	48960
2008-02-13	313	321	304	311	30840
2008-02-14	317	323	311	315	12720
2008-02-15	316	316	308	308	12360
2008-02-18	310	310	305	307	10320
2008-02-19	310	310	307	309	10800
2008-02-20	313	313	303	308	15120
2008-02-21	308	313	307	309	6600
2008-02-22	308	309	305	306	12720
2008-02-25	305	309	303	306	11760
2008-02-26	305	308	304	304	7560
2008-02-27	308	308	304	306	4560
2008-02-28	306	307	303	303	6600
2008-02-29	304	305	302	303	4200
2008-03-03	303	303	298	299	6480
2008-03-04	295	296	293	295	18600
2008-03-05	298	299	293	295	6840
2008-03-06	297	301	296	298	4320
2008-03-07	298	298	293	293	5520
2008-03-10	293	293	292	292	3840
2008-03-11	289	289	279	283	13200
2008-03-12	284	293	284	289	6000
2008-03-13	289	289	283	283	11280
2008-03-14	288	288	281	281	10080
2008-03-17	278	288	276	276	14160
2008-03-18	278	279	276	279	10080
2008-03-19	288	292	275	292	28800
2008-03-21	290	291	285	288	3720
2008-03-24	293	293	289	292	3720
2008-03-25	293	299	291	293	7560
2008-03-26	268	289	267	271	17880
2008-03-27	276	292	276	291	12600
2008-03-28	291	291	283	290	9360
2008-03-31	273	290	273	289	21360
2008-04-01	288	290	280	289	15720
2008-04-02	291	291	283	289	50880
2008-04-03	289	292	288	291	22800
2008-04-04	286	292	273	291	37680
2008-04-07	292	296	288	290	17760
2008-04-08	293	300	293	300	14760
2008-04-09	296	308	292	307	37200
2008-04-10	298	307	297	303	18960
2008-04-11	301	307	301	302	9120
2008-04-14	303	308	298	308	13320
2008-04-15	303	306	295	306	22680
2008-04-16	305	305	300	303	23880
2008-04-17	300	303	296	300	56400
2008-04-18	303	303	298	302	10080
2008-04-21	299	304	296	301	34440
2008-04-22	300	308	297	308	26280
2008-04-23	305	307	300	307	22680
2008-04-24	307	307	301	307	13440
2008-04-25	306	307	304	307	6240
2008-04-28	308	308	303	308	10560
2008-04-30	308	308	303	307	6720
2008-05-01	306	308	300	302	35040
2008-05-02	301	317	298	317	51240
2008-05-07	323	333	319	333	53520
2008-05-08	333	342	330	342	72480
2008-05-09	338	350	322	348	81000
2008-05-12	364	366	341	343	152640
2008-05-13	341	364	335	361	157080
2008-05-14	345	345	303	314	572400
2008-05-15	314	316	308	308	133800
2008-05-16	309	312	306	310	41640
2008-05-19	313	315	311	313	49080
2008-05-20	313	313	302	306	66480
2008-05-21	302	310	302	310	41160
2008-05-22	308	308	303	305	15840
2008-05-23	303	306	303	306	31800
2008-05-26	307	307	303	303	12240
2008-05-27	304	308	302	308	24000
2008-05-28	307	307	295	300	36360
2008-05-29	300	303	298	303	15000
2008-05-30	302	302	295	295	25560
2008-06-02	302	308	295	299	69000
2008-06-03	298	303	297	303	18120
2008-06-04	303	303	299	299	31560
2008-06-05	302	302	300	302	28440
2008-06-06	303	304	301	301	37560
2008-06-09	298	303	298	303	26040
2008-06-10	303	303	299	300	14400
2008-06-11	301	303	300	301	10920
2008-06-12	302	302	297	298	25080
2008-06-13	302	302	297	297	19200
2008-06-16	294	294	288	291	41520
2008-06-17	292	300	291	294	42120
2008-06-18	296	299	294	299	11760
2008-06-19	298	300	296	299	45840
2008-06-20	302	302	297	302	24240
2008-06-23	298	302	297	299	16320
2008-06-24	302	302	299	299	8880
2008-06-25	298	300	296	298	7560
2008-06-26	300	300	298	299	33120
2008-06-27	297	301	297	298	32280
2008-06-30	296	298	296	298	6480
2008-07-01	298	302	297	297	17160
2008-07-02	298	299	293	294	17400
2008-07-03	294	298	293	298	13920
2008-07-04	299	299	296	296	11040
2008-07-07	299	299	295	297	8280
2008-07-08	298	298	292	298	20280
2008-07-09	296	296	293	296	19080
2008-07-10	293	295	293	293	13800
2008-07-11	294	294	293	293	13560
2008-07-14	293	294	293	293	13680
2008-07-15	293	293	292	293	18960
2008-07-16	293	293	292	292	7920
2008-07-17	293	295	293	295	5760
2008-07-18	297	297	288	288	26760
2008-07-22	295	295	292	293	8880
2008-07-23	295	297	294	297	15720
2008-07-24	297	300	296	297	17160
2008-07-25	296	319	296	314	53640
2008-07-28	315	323	313	317	77040
2008-07-29	317	318	308	313	32640
2008-07-30	308	310	306	310	7680
2008-07-31	308	310	307	308	3840
2008-08-01	313	313	306	309	10800
2008-08-04	318	318	304	308	18600
2008-08-05	308	308	303	304	13200
2008-08-06	308	310	304	304	12960
2008-08-07	308	310	300	308	21720
2008-08-08	308	311	303	311	21360
2008-08-11	311	311	306	308	29640
2008-08-12	306	306	300	300	14400
2008-08-13	302	303	299	300	27600
2008-08-14	304	304	298	300	10080
2008-08-15	298	303	296	302	15960
2008-08-18	303	303	296	303	12360
2008-08-19	303	308	302	308	20400
2008-08-20	304	308	303	304	8880
2008-08-21	304	304	301	301	4080
2008-08-22	300	307	299	306	17520
2008-08-25	307	308	303	307	4920
2008-08-26	307	307	302	303	8760
2008-08-27	311	311	307	310	37560
2008-08-28	312	313	308	310	18360
2008-08-29	309	310	304	308	23880
2008-09-01	308	308	304	304	9720
2008-09-02	304	308	304	308	10920
2008-09-03	308	308	307	308	10560
2008-09-04	304	308	297	308	38640
2008-09-05	305	308	297	302	23520
2008-09-08	303	313	302	313	40680
2008-09-09	311	311	296	304	37800
2008-09-10	300	300	296	299	21480
2008-09-11	300	303	296	300	21480
2008-09-12	300	302	296	302	9000
2008-09-16	291	300	285	294	52200
2008-09-17	301	301	294	299	14880
2008-09-18	297	299	295	299	19440
2008-09-19	299	300	297	299	15240
2008-09-22	298	299	288	294	38160
2008-09-24	295	298	293	295	18600
2008-09-25	296	296	293	295	6360
2008-09-26	298	298	293	295	12960
2008-09-29	296	296	291	292	16560
2008-09-30	283	288	278	286	57360
2008-10-01	288	288	283	287	11520
2008-10-02	288	291	283	283	27240
2008-10-03	282	285	279	280	20880
2008-10-06	279	282	275	277	162000
2008-10-07	275	279	267	276	30480
2008-10-08	270	282	267	276	58560
2008-10-09	268	278	266	275	59280
2008-10-10	270	274	254	274	29640
2008-10-14	283	292	283	290	33360
2008-10-15	289	289	287	287	11640
2008-10-16	283	283	278	283	21480
2008-10-17	284	286	283	286	27360
2008-10-20	288	294	283	293	59400
2008-10-21	294	297	290	296	30360
2008-10-22	298	303	296	298	56520
2008-10-23	297	299	295	298	50640
2008-10-24	298	300	298	298	46800
2008-10-27	297	297	291	291	66840
2008-10-28	291	291	288	290	42240
2008-10-29	296	303	291	298	65880
2008-10-30	303	307	298	303	71760
2008-10-31	303	304	296	303	50760
2008-11-04	300	303	296	301	6000
2008-11-05	303	304	300	303	12240
2008-11-06	298	303	298	298	14160
2008-11-07	296	300	295	299	14640
2008-11-10	298	304	292	298	35760
2008-11-11	299	300	296	296	10920
2008-11-12	298	300	293	294	9720
2008-11-13	296	297	293	293	15360
2008-11-14	293	295	292	295	14880
2008-11-17	292	296	292	294	10800
2008-11-18	294	296	293	296	8400
2008-11-19	293	298	293	298	33600
2008-11-20	294	297	283	293	27600
2008-11-21	293	295	288	295	23880
2008-11-25	296	296	289	293	6000
2008-11-26	292	292	279	285	9960
2008-11-27	285	287	283	283	3360
2008-11-28	287	290	285	286	7080
2008-12-01	286	286	283	285	4800
2008-12-02	285	287	283	287	5760
2008-12-03	285	285	282	283	5880
2008-12-04	283	284	283	283	8400
2008-12-05	285	285	281	283	8160
2008-12-08	282	283	279	283	11160
2008-12-09	283	283	280	280	8520
2008-12-10	278	283	278	283	17760
2008-12-11	281	283	279	283	6720
2008-12-12	283	284	280	280	8640
2008-12-15	283	284	282	282	5880
2008-12-16	284	285	283	283	4800
2008-12-17	285	289	281	284	14640
2008-12-18	287	290	286	289	9360
2008-12-19	291	291	286	289	11760
2008-12-22	289	292	287	288	22080
2008-12-24	289	290	288	290	4320
2008-12-25	292	292	288	288	7320
2008-12-26	291	291	288	289	6360
2008-12-29	288	290	287	288	4320
2008-12-30	285	288	285	286	3840
2009-01-05	292	292	286	286	5400
2009-01-06	288	288	285	286	5880
2009-01-07	288	291	284	287	13920
2009-01-08	285	288	283	286	8520
2009-01-09	286	287	284	284	3360
2009-01-13	284	284	283	284	5520
2009-01-14	284	285	283	284	3240
2009-01-15	283	284	283	284	3840
2009-01-16	283	285	279	284	11880
2009-01-19	284	285	283	284	12240
2009-01-20	284	284	280	280	14760
2009-01-21	280	283	279	281	13080
2009-01-22	278	286	278	284	30360
2009-01-23	282	285	282	285	5040
2009-01-26	285	287	282	287	5280
2009-01-27	285	285	283	283	2160
2009-01-28	280	285	280	282	9120
2009-01-29	278	283	278	282	16680
2009-01-30	281	283	279	282	4560
2009-02-02	280	283	280	283	4800
2009-02-03	286	288	283	283	19320
2009-02-04	284	284	283	283	7800
2009-02-05	287	287	283	283	4440
2009-02-06	285	287	283	287	6240
2009-02-09	283	288	283	286	24480
2009-02-10	286	289	286	288	16200
2009-02-12	289	293	289	292	13800
2009-02-13	293	304	293	302	39360
2009-02-16	293	303	293	299	21360
2009-02-17	300	304	299	301	6480
2009-02-18	300	300	299	299	9360
2009-02-19	297	304	297	304	10800
2009-02-20	304	304	299	304	27480
2009-02-23	294	304	292	304	11640
2009-02-24	303	303	297	299	22080
2009-02-25	298	300	293	298	14760
2009-02-26	293	298	293	298	7200
2009-02-27	298	299	296	299	4800
2009-03-02	299	302	296	296	11160
2009-03-03	296	300	296	298	6840
2009-03-04	294	300	294	300	5520
2009-03-05	299	300	299	300	6000
2009-03-06	299	301	299	300	5280
2009-03-09	298	301	297	299	11880
2009-03-10	296	300	296	297	13560
2009-03-11	298	299	296	298	12360
2009-03-12	297	299	296	297	9840
2009-03-13	297	299	297	299	12000
2009-03-16	297	300	297	298	5400
2009-03-17	318	347	318	336	187080
2009-03-18	335	342	332	341	120000
2009-03-19	337	341	333	339	40680
2009-03-23	339	343	333	343	40080
2009-03-24	343	365	343	363	84000
2009-03-25	355	371	355	368	195960
2009-03-26	350	362	350	356	60300
2009-03-27	360	360	340	340	35700
2009-03-30	342	342	323	329	22400
2009-03-31	325	329	324	329	34500
2009-04-01	326	332	326	330	25400
2009-04-02	332	334	325	329	31500
2009-04-03	329	333	326	328	13900
2009-04-06	330	330	326	330	24400
2009-04-07	330	333	329	333	17500
2009-04-08	333	333	330	330	20000
2009-04-09	330	333	327	333	8600
2009-04-10	334	335	331	332	7600
2009-04-13	377	381	355	371	135500
2009-04-14	370	371	366	370	59100
2009-04-15	368	370	366	370	63200
2009-04-16	369	372	366	371	61200
2009-04-17	372	380	372	380	84800
2009-04-20	381	385	380	384	120800
2009-04-21	383	383	379	381	33300
2009-04-22	381	382	380	381	42500
2009-04-23	381	381	379	381	42000
2009-04-24	380	380	377	377	68800
2009-04-27	376	379	375	377	72600
2009-04-28	376	376	361	364	72700
2009-04-30	364	366	363	364	45300
2009-05-01	364	372	364	371	60000
2009-05-07	372	376	372	375	27600
2009-05-08	376	381	376	381	60300
2009-05-11	381	383	380	381	22100
2009-05-12	382	385	381	385	48400
2009-05-13	383	385	378	380	4700
2009-05-14	382	382	371	376	15800
2009-05-15	375	375	364	373	14100
2009-05-18	372	372	370	371	7300
2009-05-19	366	367	362	364	14000
2009-05-20	365	366	354	355	22700
2009-05-21	355	374	355	373	49700
2009-05-22	375	381	375	381	66400
2009-05-25	382	386	377	377	79700
2009-05-26	382	383	378	380	18900
2009-05-27	383	383	376	380	23300
2009-05-28	383	385	380	385	41600
2009-05-29	384	384	379	381	21200
2009-06-01	377	382	373	378	45900
2009-06-02	377	377	369	374	36600
2009-06-03	374	374	369	373	25300
2009-06-04	372	374	369	371	28900
2009-06-05	371	372	370	372	11100
2009-06-08	372	372	371	372	13900
2009-06-09	371	372	370	371	9900
2009-06-10	371	374	371	374	6800
2009-06-11	374	375	372	374	5900
2009-06-12	374	388	373	379	59000
2009-06-15	386	386	380	381	25000
2009-06-16	383	383	379	379	16700
2009-06-17	375	380	375	380	8500
2009-06-18	383	383	376	379	9600
2009-06-19	379	380	375	377	14200
2009-06-22	375	376	369	372	16500
2009-06-23	367	370	366	367	12100
2009-06-24	368	378	368	372	17500
2009-06-25	377	378	371	377	10400
2009-06-26	376	377	375	377	6900
2009-06-29	376	377	375	376	5700
2009-06-30	377	379	375	378	7500
2009-07-01	375	375	367	371	15000
2009-07-02	376	376	372	374	4600
2009-07-03	366	375	364	373	18000
2009-07-06	375	375	370	372	8600
2009-07-07	370	371	367	367	7000
2009-07-08	365	370	355	367	12900
2009-07-09	365	367	358	360	4400
2009-07-10	361	374	361	372	22900
2009-07-13	372	372	355	357	21900
2009-07-14	353	365	353	358	6500
2009-07-15	359	364	355	355	9000
2009-07-16	355	364	355	362	5100
2009-07-17	365	369	362	369	13800
2009-07-21	367	375	365	374	15500
2009-07-22	373	379	371	371	30300
2009-07-23	372	374	372	373	4100
2009-07-24	374	374	370	370	7200
2009-07-27	366	370	366	370	5100
2009-07-28	370	371	367	367	4700
2009-07-29	366	369	366	366	2400
2009-07-30	366	368	366	366	3800
2009-07-31	367	370	367	370	3700
2009-08-03	367	372	367	370	4700
2009-08-04	374	374	368	370	14100
2009-08-05	379	409	378	395	237600
2009-08-06	400	408	400	402	130200
2009-08-07	400	401	393	400	53200
2009-08-10	395	405	394	398	44900
2009-08-11	400	402	397	397	16900
2009-08-12	399	400	396	398	26700
2009-08-13	399	400	398	398	13000
2009-08-14	402	414	398	406	87100
2009-08-17	410	423	408	412	84300
2009-08-18	410	418	410	411	28500
2009-08-19	415	421	413	416	56200
2009-08-20	420	435	420	435	98100
2009-08-21	435	440	420	429	93300
2009-08-24	430	430	421	423	34700
2009-08-25	424	430	420	429	57000
2009-08-26	430	447	430	444	95100
2009-08-27	444	449	435	437	117900
2009-08-28	457	465	451	458	190500
2009-08-31	458	462	445	454	147100
2009-09-01	454	461	450	454	75900
2009-09-02	451	457	448	450	70500
2009-09-03	449	452	440	448	34500
2009-09-04	438	450	423	426	54200
2009-09-07	426	435	410	410	38600
2009-09-08	406	415	402	413	35300
2009-09-09	388	413	388	410	163900
2009-09-10	405	426	405	417	47600
2009-09-11	427	429	421	429	23700
2009-09-14	435	437	421	421	58700
2009-09-15	419	426	411	415	18500
2009-09-16	405	414	402	408	31000
2009-09-17	408	415	404	409	27700
2009-09-18	401	411	398	409	31700
2009-09-24	419	433	416	426	42300
2009-09-25	426	428	418	425	23400
2009-09-28	420	425	415	417	30200
2009-09-29	413	421	411	418	14600
2009-09-30	415	421	409	420	31600
2009-10-01	416	435	411	431	37000
2009-10-02	421	430	416	428	19900
2009-10-05	430	430	417	428	9500
2009-10-06	423	427	419	427	8600
2009-10-07	422	427	422	427	12600
2009-10-08	427	432	421	428	18000
2009-10-09	430	446	429	439	77400
2009-10-13	442	450	442	448	67400
2009-10-14	443	450	437	444	39000
2009-10-15	456	467	454	461	170600
2009-10-16	460	465	452	456	72000
2009-10-19	454	456	440	452	51800
2009-10-20	450	453	448	452	20500
2009-10-21	447	454	447	454	15300
2009-10-22	452	457	449	457	17800
2009-10-23	462	472	456	457	115200
2009-10-26	457	465	457	465	28900
2009-10-27	464	465	458	459	26200
2009-10-28	451	458	450	455	52600
2009-10-29	447	450	446	450	14800
2009-10-30	449	463	445	447	29300
2009-11-02	438	440	426	439	31000
2009-11-04	440	440	427	433	12300
2009-11-05	434	434	421	424	17500
2009-11-06	427	428	412	427	33900
2009-11-09	425	425	411	411	43900
2009-11-10	414	419	412	417	28300
2009-11-11	418	419	415	416	21400
2009-11-12	415	419	410	413	33200
2009-11-13	410	417	409	411	22700
2009-11-16	409	412	399	402	39400
2009-11-17	395	402	393	402	19400
2009-11-18	400	400	381	394	20100
2009-11-19	379	393	374	392	17700
2009-11-20	393	400	389	400	15700
2009-11-24	399	399	395	398	7000
2009-11-25	395	396	391	395	8900
2009-11-26	393	394	392	392	1700
2009-11-27	389	394	389	390	7800
2009-11-30	385	391	385	389	11100
2009-12-01	394	394	380	385	21200
2009-12-02	385	392	382	390	8200
2009-12-03	386	397	386	397	11100
2009-12-04	396	398	392	397	5400
2009-12-07	403	410	400	410	12400
2009-12-08	411	411	405	405	11400
2009-12-09	405	407	405	406	2000
2009-12-10	406	408	396	396	11100
2009-12-11	399	404	399	401	4800
2009-12-14	402	405	400	402	6500
2009-12-15	401	404	400	403	2500
2009-12-16	403	403	398	402	8400
2009-12-17	401	403	401	403	5600
2009-12-18	405	405	400	400	8600
2009-12-21	403	403	397	400	8200
2009-12-22	401	402	397	401	8400
2009-12-24	398	402	396	399	13000
2009-12-25	396	400	395	398	17000
2009-12-28	400	402	399	401	9300
2009-12-29	401	402	400	402	13700
2009-12-30	401	404	399	404	16700
2010-01-04	403	405	400	405	14000
2010-01-05	407	407	403	406	8200
2010-01-06	406	409	404	404	23800
2010-01-07	404	410	404	408	13700
2010-01-08	406	410	406	408	7500
2010-01-12	440	449	420	436	68200
2010-01-13	436	437	430	433	21800
2010-01-14	438	440	433	439	24800
2010-01-15	435	443	435	440	40600
2010-01-18	443	443	440	440	31100
2010-01-19	440	441	437	437	34300
2010-01-20	437	438	435	437	21500
2010-01-21	435	438	435	438	7900
2010-01-22	439	439	435	439	26400
2010-01-25	434	439	433	438	6300
2010-01-26	436	439	436	438	5800
2010-01-27	437	440	434	434	30000
2010-01-28	433	438	433	437	9400
2010-01-29	432	438	430	437	28500
2010-02-01	435	454	435	453	53500
2010-02-02	455	455	433	438	67200
2010-02-03	433	436	432	434	10600
2010-02-04	430	438	427	437	16500
2010-02-05	434	437	431	433	10500
2010-02-08	430	437	429	430	11600
2010-02-09	429	437	429	437	8000
2010-02-10	430	440	430	431	8200
2010-02-12	440	445	435	441	65400
2010-02-15	441	441	436	437	5900
2010-02-16	436	440	432	438	10000
2010-02-17	435	439	434	437	32800
2010-02-18	435	438	430	438	14100
2010-02-19	438	439	433	434	10500
2010-02-22	431	435	430	434	9400
2010-02-23	431	432	431	432	5100
2010-02-24	430	435	430	432	9900
2010-02-25	429	430	428	429	17200
2010-02-26	427	428	427	427	5100
2010-03-01	425	427	423	424	6600
2010-03-02	423	424	422	422	18100
2010-03-03	422	422	419	420	18200
2010-03-04	419	425	419	420	22400
2010-03-05	425	425	421	424	6900
2010-03-08	426	430	424	426	14800
2010-03-09	426	430	426	428	8000
2010-03-10	430	431	427	431	16400
2010-03-11	430	438	430	436	15400
2010-03-12	439	442	439	441	14600
2010-03-15	443	443	439	441	57900
2010-03-16	441	444	441	443	17100
2010-03-17	442	449	442	449	19800
2010-03-18	448	449	446	449	16700
2010-03-19	450	453	448	450	41200
2010-03-23	451	458	451	455	28600
2010-03-24	457	462	456	459	51400
2010-03-25	463	465	460	464	91300
2010-03-26	463	470	458	469	44200
2010-03-29	465	465	457	465	39400
2010-03-30	457	460	452	452	25500
2010-03-31	456	459	454	455	5600
2010-04-01	458	460	452	458	44000
2010-04-02	458	470	458	467	72700
2010-04-05	468	469	462	467	69700
2010-04-06	463	467	463	465	20700
2010-04-07	465	468	464	467	17900
2010-04-08	465	475	465	475	42000
2010-04-09	475	484	473	480	80000
2010-04-12	482	488	482	488	46900
2010-04-13	490	515	490	515	181000
2010-04-14	517	540	515	535	250200
2010-04-15	540	550	534	534	229700
2010-04-16	530	530	517	525	112400
2010-04-19	521	550	521	547	132100
2010-04-20	530	531	520	520	149400
2010-04-21	522	528	512	512	81300
2010-04-22	514	518	500	516	86600
2010-04-23	518	528	518	518	42100
2010-04-26	520	528	519	519	32500
2010-04-27	524	528	520	528	45400
2010-04-28	524	530	520	530	34300
2010-04-30	530	530	524	526	34800
2010-05-06	511	520	511	512	42200
2010-05-07	490	506	488	498	56400
2010-05-10	500	530	500	529	50500
2010-05-11	551	555	541	547	558900
2010-05-12	549	552	546	547	119400
2010-05-13	547	547	537	547	113600
2010-05-14	547	547	540	545	101000
2010-05-17	545	545	540	543	93000
2010-05-18	543	546	534	535	123800
2010-05-19	534	536	523	529	130000
2010-05-20	529	529	521	521	68200
2010-05-21	520	535	516	527	163500
2010-05-24	532	532	517	523	26600
2010-05-25	523	525	513	513	79500
2010-05-26	513	514	508	510	53000
2010-05-27	510	530	504	530	66000
2010-05-28	530	540	530	539	116000
2010-05-31	537	551	532	550	77900
2010-06-01	548	548	534	537	25800
2010-06-02	534	534	528	528	67800
2010-06-03	528	532	527	531	64300
2010-06-04	530	530	523	524	49400
2010-06-07	523	523	516	517	54900
2010-06-08	519	524	517	524	30200
2010-06-09	522	524	519	522	42100
2010-06-10	520	522	519	522	28800
2010-06-11	520	522	518	519	28700
2010-06-14	519	523	519	522	38500
2010-06-15	522	531	519	531	48200
2010-06-16	530	535	530	533	40700
2010-06-17	533	539	533	538	15700
2010-06-18	534	536	533	534	13700
2010-06-21	529	533	520	530	12200
2010-06-22	530	536	529	534	38200
2010-06-23	534	536	529	535	21200
2010-06-24	530	533	523	525	6200
2010-06-25	520	534	517	534	24800
2010-06-28	533	535	530	535	17800
2010-06-29	526	534	525	525	9200
2010-06-30	523	523	505	516	13200
2010-07-01	506	512	505	512	8100
2010-07-02	510	523	510	518	13300
2010-07-05	522	533	522	526	20000
2010-07-06	530	534	526	531	14200
2010-07-07	531	537	528	533	17800
2010-07-08	531	536	530	534	9800
2010-07-09	529	533	529	533	3400
2010-07-12	533	533	525	525	7200
2010-07-13	529	532	525	526	5900
2010-07-14	520	525	510	522	18600
2010-07-15	525	526	520	522	9200
2010-07-16	515	521	511	520	29600
2010-07-20	524	525	520	523	20300
2010-07-21	515	520	513	513	12100
2010-07-22	509	514	507	507	8500
2010-07-23	506	508	501	505	11500
2010-07-26	520	525	515	516	20200
2010-07-27	513	518	510	511	8100
2010-07-28	516	518	512	513	8100
2010-07-29	517	523	516	523	19600
2010-07-30	523	526	519	522	13100
2010-08-02	522	533	520	527	14000
2010-08-03	530	530	518	519	29800
2010-08-04	520	529	519	525	20400
2010-08-05	522	530	522	528	35500
2010-08-06	530	530	525	526	17000
2010-08-09	526	527	525	525	12500
2010-08-10	525	528	525	528	13000
2010-08-11	525	525	524	524	14000
2010-08-12	524	532	524	528	44300
2010-08-13	528	530	523	530	10500
2010-08-16	530	530	527	528	17100
2010-08-17	528	528	527	527	8600
2010-08-18	527	534	527	534	20300
2010-08-19	534	535	532	533	8900
2010-08-20	534	534	515	519	13400
2010-08-23	519	524	519	520	9100
2010-08-24	525	525	519	520	11000
2010-08-25	515	524	512	518	27900
2010-08-26	518	522	513	522	11200
2010-08-27	516	521	516	519	6100
2010-08-30	520	521	515	518	9500
2010-08-31	519	520	518	520	35700
2010-09-01	521	524	503	510	32500
2010-09-02	515	517	507	507	21900
2010-09-03	508	515	507	509	25900
2010-09-06	514	515	511	515	17300
2010-09-07	515	518	514	518	14500
2010-09-08	515	520	515	520	13000
2010-09-09	519	520	517	518	19500
2010-09-10	518	520	511	518	9800
2010-09-13	520	520	514	516	20800
2010-09-14	515	515	505	506	39900
2010-09-15	509	512	500	510	30000
2010-09-16	509	509	507	507	15300
2010-09-17	506	506	503	504	19300
2010-09-21	503	505	500	500	18800
2010-09-22	501	502	500	502	27800
2010-09-24	501	504	501	502	4500
2010-09-27	502	505	500	505	23300
2010-09-28	502	508	502	508	8500
2010-09-29	510	515	509	513	17800
2010-09-30	517	522	514	515	29200
2010-10-01	515	515	510	515	2900
2010-10-04	516	519	515	515	4400
2010-10-05	518	518	510	515	6700
2010-10-06	518	518	511	511	6400
2010-10-07	506	516	505	511	11600
2010-10-08	511	524	508	512	21700
2010-10-12	518	520	509	515	25800
2010-10-13	516	516	508	516	10300
2010-10-14	513	516	510	510	6000
2010-10-15	509	514	507	508	17200
2010-10-18	511	515	510	515	4400
2010-10-19	516	518	513	518	25300
2010-10-20	518	520	512	516	29900
2010-10-21	513	519	513	518	5300
2010-10-22	518	518	511	516	5200
2010-10-25	512	520	512	516	18300
2010-10-26	518	518	515	515	2700
2010-10-27	517	520	515	515	5600
2010-10-28	515	520	510	517	18500
2010-10-29	514	522	514	522	11700
2010-11-01	520	529	519	528	21900
2010-11-02	529	529	517	526	22400
2010-11-04	529	529	520	526	20600
2010-11-05	526	528	524	526	20300
2010-11-08	526	527	523	523	9500
2010-11-09	523	526	521	526	31100
2010-11-10	526	528	520	520	41100
2010-11-11	520	520	518	518	18400
2010-11-12	518	523	518	523	15900
2010-11-15	521	524	517	519	5100
2010-11-16	519	527	519	527	15700
2010-11-17	527	527	522	524	14600
2010-11-18	526	533	526	531	33800
2010-11-19	533	537	531	537	27500
2010-11-22	530	540	530	540	23400
2010-11-24	530	543	530	543	19000
2010-11-25	544	544	535	535	27000
2010-11-26	535	540	535	535	11300
2010-11-29	535	536	530	535	15000
2010-11-30	535	535	532	532	18500
2010-12-01	532	540	532	539	16200
2010-12-02	539	541	530	531	26700
2010-12-03	530	532	529	530	21100
2010-12-06	535	536	533	533	16300
2010-12-07	533	534	527	530	29800
2010-12-08	531	538	531	533	30900
2010-12-09	533	540	533	540	27700
2010-12-10	540	541	531	531	26900
2010-12-13	537	542	534	538	34600
2010-12-14	534	537	532	533	16300
2010-12-15	537	540	534	536	19600
2010-12-16	536	540	536	537	24600
2010-12-17	537	545	537	545	32900
2010-12-20	545	545	538	538	29600
2010-12-21	539	542	538	540	18500
2010-12-22	542	547	540	547	21100
2010-12-24	547	547	536	536	12800
2010-12-27	530	540	530	536	16000
2010-12-28	539	541	538	540	9200
2010-12-29	542	548	542	546	21300
2010-12-30	546	554	544	551	30400
2011-01-04	553	555	551	554	20500
2011-01-05	554	554	550	550	11000
2011-01-06	554	554	550	550	10200
2011-01-07	550	552	549	551	14200
2011-01-11	552	561	551	555	57400
2011-01-12	564	570	559	566	56900
2011-01-13	568	568	562	563	41500
2011-01-14	564	573	564	570	49800
2011-01-17	579	586	572	580	47100
2011-01-18	580	580	567	568	23100
2011-01-19	570	576	570	576	7500
2011-01-20	570	574	564	569	22000
2011-01-21	569	577	569	577	42200
2011-01-24	571	580	570	575	11400
2011-01-25	571	580	571	580	14000
2011-01-26	577	580	570	579	21000
2011-01-27	529	560	524	545	162000
2011-01-28	540	545	536	538	51500
2011-01-31	537	553	537	550	40900
2011-02-01	542	550	532	540	21000
2011-02-02	540	545	538	540	21900
2011-02-03	535	539	535	536	5200
2011-02-04	533	544	533	540	28700
2011-02-07	545	545	537	541	10700
2011-02-08	542	542	533	539	18000
2011-02-09	540	540	535	537	8300
2011-02-10	537	540	535	539	15500
2011-02-14	536	537	532	535	27500
2011-02-15	534	535	528	533	13200
2011-02-16	531	543	531	539	24900
2011-02-17	540	543	539	543	8600
2011-02-18	543	543	540	540	11500
2011-02-21	538	538	532	534	27300
2011-02-22	534	535	533	533	5100
2011-02-23	531	538	530	530	5300
2011-02-24	531	539	531	531	19800
2011-02-25	530	538	530	535	11800
2011-02-28	538	538	532	535	5400
2011-03-01	534	535	530	534	4100
2011-03-02	532	534	530	533	10100
2011-03-03	534	534	526	527	11900
2011-03-04	527	529	520	524	19600
2011-03-07	515	519	511	511	21200
2011-03-08	510	515	510	512	8600
2011-03-09	511	515	510	513	7600
2011-03-10	513	514	511	513	3200
2011-03-11	510	515	510	511	8000
2011-03-14	503	504	457	497	80700
2011-03-15	465	470	417	430	30800
2011-03-16	430	499	430	499	31400
2011-03-17	475	492	470	492	9000
2011-03-18	495	499	490	497	51200
2011-03-22	490	499	472	494	40800
2011-03-23	493	510	490	508	29900
2011-03-24	510	510	497	497	45100
2011-03-25	506	514	503	513	24800
2011-03-28	503	513	502	511	23500
2011-03-29	510	512	500	505	13800
2011-03-30	509	527	509	520	18800
2011-03-31	510	523	510	516	37500
2011-04-01	517	517	505	506	6800
2011-04-04	504	509	490	496	7500
2011-04-05	490	496	485	490	10200
2011-04-06	490	490	478	479	5600
2011-04-07	472	481	461	481	14000
2011-04-08	474	484	466	484	6200
2011-04-11	460	480	460	471	52900
2011-04-12	473	475	465	473	34900
2011-04-13	465	505	465	490	34200
2011-04-14	491	491	484	484	17600
2011-04-15	489	497	485	493	5700
2011-04-18	492	493	490	490	4200
2011-04-19	490	495	489	491	3700
2011-04-20	490	493	489	491	5500
2011-04-21	492	492	489	490	5600
2011-04-22	486	489	486	487	6100
2011-04-25	485	486	483	483	7900
2011-04-26	470	478	469	474	18600
2011-04-27	478	478	457	466	31300
2011-04-28	470	471	465	470	4900
2011-05-02	470	470	465	469	8700
2011-05-06	469	471	465	470	6400
2011-05-09	472	479	471	479	10900
2011-05-10	485	490	465	469	41100
2011-05-11	471	482	470	473	9600
2011-05-12	470	481	470	481	3900
2011-05-13	481	484	466	472	17200
2011-05-16	474	485	469	485	8900
2011-05-17	485	485	480	485	8300
2011-05-18	484	485	469	473	6700
2011-05-19	477	480	474	480	3300
2011-05-20	483	484	479	484	6600
2011-05-23	484	485	480	484	9600
2011-05-24	483	485	475	482	14000
2011-05-25	479	485	473	473	6800
2011-05-26	474	480	474	475	2900
2011-05-27	474	478	470	471	8200
2011-05-30	471	471	470	471	14100
2011-05-31	471	474	470	470	6100
2011-06-01	475	477	474	474	2800
2011-06-02	474	474	463	469	7400
2011-06-03	468	474	465	473	3800
2011-06-06	468	469	465	469	6600
2011-06-07	465	469	465	465	1900
2011-06-08	465	470	465	468	3900
2011-06-09	466	470	465	468	3000
2011-06-10	466	466	462	465	1800
2011-06-13	465	468	465	468	2500
2011-06-14	470	472	462	468	17900
2011-06-15	468	469	465	469	600
2011-06-16	469	473	460	472	18600
2011-06-17	470	470	460	470	22000
2011-06-20	472	472	450	460	16500
2011-06-21	463	469	463	468	3300
2011-06-22	469	472	462	464	11000
2011-06-23	464	464	456	460	18000
2011-06-24	458	465	458	465	1000
2011-06-27	463	463	452	458	16100
2011-06-28	458	464	458	464	5100
2011-06-29	465	469	460	469	5300
2011-06-30	469	469	465	469	1900
2011-07-01	466	468	465	466	2100
2011-07-04	466	468	463	464	3900
2011-07-05	465	465	461	461	5600
2011-07-06	461	464	461	463	8700
2011-07-07	463	463	460	461	5300
2011-07-08	460	460	455	457	13300
2011-07-11	454	455	454	455	4500
2011-07-12	452	452	447	450	12600
2011-07-13	450	450	446	448	13900
2011-07-14	446	449	444	444	8900
2011-07-15	445	445	440	443	6100
2011-07-19	442	446	442	446	11200
2011-07-20	448	448	443	446	11800
2011-07-21	446	446	441	441	3700
2011-07-22	443	444	440	441	6500
2011-07-25	448	448	442	446	11500
2011-07-26	448	449	441	442	7700
2011-07-27	442	443	440	441	9100
2011-07-28	440	442	438	441	5200
2011-07-29	440	441	438	439	2600
2011-08-01	439	439	433	438	23500
2011-08-02	449	449	438	441	20500
2011-08-03	430	430	426	428	12300
2011-08-04	431	432	420	426	9000
2011-08-05	421	425	401	416	13700
2011-08-08	411	414	405	405	11800
2011-08-09	405	410	394	406	27800
2011-08-10	415	415	408	409	7500
2011-08-11	405	410	405	410	8200
2011-08-12	410	412	398	405	22300
2011-08-15	405	410	405	405	4600
2011-08-16	405	405	397	400	25000
2011-08-17	399	403	399	400	9400
2011-08-18	403	403	398	399	5900
2011-08-19	399	399	392	392	11900
2011-08-22	385	390	374	380	29300
2011-08-23	384	384	378	384	14800
2011-08-24	384	386	383	385	8500
2011-08-25	385	387	384	387	6400
2011-08-26	383	386	383	384	4900
2011-08-29	385	385	380	382	6700
2011-08-30	380	385	380	382	14400
2011-08-31	384	384	379	381	15300
2011-09-01	382	384	381	382	9700
2011-09-02	380	382	380	380	9200
2011-09-05	385	385	381	385	15400
2011-09-06	381	381	375	379	34300
2011-09-07	382	382	372	378	14000
2011-09-08	383	383	370	378	15300
2011-09-09	371	379	371	375	6900
2011-09-12	372	378	372	374	5300
2011-09-13	375	375	373	373	6500
2011-09-14	373	376	373	374	23500
2011-09-15	371	373	371	372	17100
2011-09-16	375	379	373	378	14200
2011-09-20	376	383	376	379	12800
2011-09-21	380	383	379	383	12400
2011-09-22	380	383	378	380	8700
2011-09-26	378	379	375	378	10100
2011-09-27	376	378	376	377	13100
2011-09-28	378	381	377	378	5400
2011-09-29	370	382	370	377	8900
2011-09-30	380	380	375	379	2700
2011-10-03	378	379	377	377	5200
2011-10-04	376	376	373	375	7700
2011-10-05	376	376	364	373	9200
2011-10-06	370	374	367	370	17500
2011-10-07	370	372	367	371	14900
2011-10-11	368	372	367	370	10300
2011-10-12	371	371	370	371	5600
2011-10-13	376	380	372	377	23900
2011-10-14	377	377	373	373	7100
2011-10-17	365	369	365	369	42000
2011-10-18	370	370	366	366	13300
2011-10-19	369	370	367	369	5700
2011-10-20	371	371	368	370	12500
2011-10-21	370	371	369	370	4900
2011-10-24	369	371	368	371	8700
2011-10-25	369	370	368	368	3500
2011-10-26	369	376	368	374	20700
2011-10-27	374	374	370	374	14800
2011-10-28	375	375	372	372	4900
2011-10-31	373	375	373	374	1200
2011-11-01	372	372	371	372	3700
2011-11-02	372	373	370	371	8600
2011-11-04	369	371	367	367	13700
2011-11-07	370	370	365	367	11800
2011-11-08	367	367	364	364	20800
2011-11-09	362	365	360	360	46100
2011-11-10	355	357	350	356	24900
2011-11-11	354	358	350	354	5000
2011-11-14	353	355	348	353	12700
2011-11-15	353	353	348	352	15800
2011-11-16	354	354	350	350	5800
2011-11-17	345	346	341	344	20900
2011-11-18	340	346	324	343	60800
2011-11-21	343	346	330	342	36000
2011-11-22	341	352	341	350	67200
2011-11-24	350	350	344	350	16700
2011-11-25	350	350	346	349	2600
2011-11-28	348	349	343	349	7900
2011-11-29	348	353	347	353	41100
2011-11-30	353	355	353	355	24900
2011-12-01	355	359	354	355	34000
2011-12-02	358	360	355	359	30000
2011-12-05	359	363	346	352	53300
2011-12-06	353	355	349	354	27800
2011-12-07	354	354	345	345	50200
2011-12-08	346	350	346	350	27900
2011-12-09	350	350	347	349	28800
2011-12-12	349	349	346	347	38100
2011-12-13	347	347	345	347	34700
2011-12-14	347	349	347	348	37100
2011-12-15	348	348	346	346	31400
2011-12-16	347	350	347	349	30900
2011-12-19	349	351	349	350	34400
2011-12-20	350	350	349	350	31300
2011-12-21	350	353	350	351	24300
2011-12-22	352	354	352	354	45200
2011-12-26	354	354	345	346	18500
2011-12-27	346	346	341	342	21600
2011-12-28	342	342	342	342	3400
2011-12-29	340	340	338	339	5900
2011-12-30	340	340	333	340	6800
2012-01-04	340	345	339	342	33200
2012-01-05	344	345	340	343	34100
2012-01-06	343	344	339	340	45500
2012-01-10	316	335	316	330	86100
2012-01-11	328	329	325	328	55100
2012-01-12	330	335	328	334	8200
2012-01-13	328	333	327	332	13100
2012-01-16	329	333	325	325	22300
2012-01-17	326	331	325	327	12200
2012-01-18	328	330	328	330	4200
2012-01-19	330	334	329	331	24300
2012-01-20	330	335	325	332	32700
2012-01-23	332	333	331	332	21300
2012-01-24	332	333	330	330	50800
2012-01-25	328	330	326	329	59600
2012-01-26	327	330	326	329	15100
2012-01-27	330	330	325	327	11300
2012-01-30	328	330	328	328	5000
2012-01-31	332	334	327	329	14200
2012-02-01	330	330	325	327	17100
2012-02-02	333	333	325	330	28800
2012-02-03	330	330	329	329	6800
2012-02-06	328	329	320	323	68200
2012-02-07	324	327	323	326	43700
2012-02-08	326	328	325	328	13900
2012-02-09	328	330	325	329	16000
2012-02-10	328	328	326	328	10600
2012-02-13	329	333	329	331	39700
2012-02-14	331	333	330	332	25100
2012-02-15	332	332	329	331	16800
2012-02-16	333	335	332	333	31900
2012-02-17	339	339	337	339	37000
2012-02-20	339	341	338	340	84400
2012-02-21	340	342	338	341	26700
2012-02-22	342	343	340	342	30800
2012-02-23	342	342	340	340	10300
2012-02-24	341	341	340	341	22700
2012-02-27	340	345	339	345	45600
2012-02-28	346	363	344	363	74100
2012-02-29	371	383	364	378	113500
2012-03-01	378	380	352	374	54200
2012-03-02	379	380	364	370	28400
2012-03-05	374	374	357	373	24700
2012-03-06	362	373	354	359	28700
2012-03-07	351	352	347	348	16500
2012-03-08	351	351	347	347	7900
2012-03-09	349	350	348	349	8500
2012-03-12	351	356	349	349	16600
2012-03-13	349	355	349	353	7300
2012-03-14	353	355	350	350	11800
2012-03-15	350	352	343	348	23100
2012-03-16	350	350	345	349	8800
2012-03-19	349	349	347	349	29700
2012-03-21	350	353	349	352	36100
2012-03-22	350	351	349	350	10400
2012-03-23	350	353	350	351	28300
2012-03-26	354	354	352	353	13000
2012-03-27	353	354	350	353	16300
2012-03-28	331	346	331	341	17700
2012-03-29	344	344	339	341	8800
2012-03-30	343	343	331	339	19600
2012-04-02	340	353	336	338	43000
2012-04-03	340	340	338	340	14100
2012-04-04	340	340	338	338	10800
2012-04-05	335	335	330	334	22000
2012-04-06	330	335	329	334	7300
2012-04-09	332	338	331	338	6100
2012-04-10	335	336	335	336	2100
2012-04-11	334	334	326	332	13300
2012-04-12	325	330	325	327	7600
2012-04-13	326	332	325	327	7000
2012-04-16	327	331	326	326	4100
2012-04-17	326	327	325	325	4600
2012-04-18	329	329	326	328	2500
2012-04-19	330	332	328	330	4600
2012-04-20	330	330	325	326	11300
2012-04-23	329	329	326	326	1600
2012-04-24	326	328	325	326	1600
2012-04-25	326	329	326	327	3600
2012-04-26	329	329	320	322	23800
2012-04-27	324	324	319	321	6000
2012-05-01	321	321	320	321	4300
2012-05-02	321	324	315	323	11300
2012-05-07	318	319	315	318	14000
2012-05-08	318	319	315	318	3500
2012-05-09	312	317	311	312	10800
cut2
2009-01-05	871.01	883.68	870.99	875.91
2009-01-06	883.15	883.15	872.38	876.20
2009-01-07	881.43	896.21	880.26	888.25
2009-01-08	879.64	879.69	860.16	860.89
2009-01-09	863.82	866.15	849.32	855.02
2009-01-13	845.07	845.07	814.12	814.12
2009-01-14	818.21	828.30	813.07	819.39
2009-01-15	807.68	807.68	792.72	795.99
2009-01-16	805.24	820.55	802.75	817.89
2009-01-19	825.18	828.88	815.51	817.73
2009-01-20	812.91	813.46	795.82	805.03
2009-01-21	793.70	795.82	784.50	787.15
2009-01-22	795.05	796.39	781.39	795.91
2009-01-23	789.39	789.39	773.28	773.55
2009-01-26	772.59	777.91	767.82	768.28
2009-01-27	776.84	809.94	776.84	805.49
2009-01-28	802.82	812.32	791.86	804.33
2009-01-29	812.42	822.14	807.50	818.47
2009-01-30	808.58	808.58	787.35	794.03
2009-02-02	785.66	785.66	770.55	777.85
2009-02-03	776.48	796.20	772.02	773.79
2009-02-04	779.56	795.53	776.15	792.78
2009-02-05	790.33	799.56	783.75	786.41
2009-02-06	794.62	803.47	787.54	790.84
2009-02-09	798.85	803.60	778.90	778.90
2009-02-10	787.50	792.24	775.25	778.10
2009-02-12	769.02	770.68	756.98	760.29
2009-02-13	766.19	773.38	759.20	764.59
2009-02-16	763.12	772.45	762.43	770.10
2009-02-17	765.10	765.15	754.89	756.53
2009-02-18	747.84	751.63	744.37	749.26
2009-02-19	753.78	757.28	749.53	751.59
2009-02-20	751.90	752.31	736.21	739.53
2009-02-23	730.60	737.72	721.79	735.28
2009-02-24	725.46	730.28	717.85	730.28
2009-02-25	739.47	746.72	732.99	745.62
2009-02-26	745.11	755.08	739.24	742.53
2009-02-27	744.14	756.76	741.18	756.71
2009-03-02	745.74	745.78	729.86	734.59
2009-03-03	724.04	731.73	714.96	726.80
2009-03-04	720.01	734.54	715.22	732.04
2009-03-05	736.83	751.81	736.52	741.55
2009-03-06	731.93	731.93	720.76	721.39
2009-03-09	721.81	724.26	705.42	710.53
2009-03-10	705.49	708.74	700.83	703.50
2009-03-11	713.91	729.12	713.78	722.28
2009-03-12	717.87	718.03	698.46	700.93
2009-03-13	707.62	726.20	707.35	724.30
2009-03-16	731.61	747.08	731.61	741.69
2009-03-17	747.64	763.29	743.62	760.64
2009-03-18	766.23	772.59	758.36	764.67
2009-03-19	771.25	774.27	762.46	764.77
2009-03-23	766.92	792.13	765.49	791.56
2009-03-24	803.28	815.40	801.84	812.72
2009-03-25	815.77	821.02	809.23	818.49
2009-03-26	813.85	827.38	809.22	826.81
2009-03-27	833.96	845.64	824.53	824.53
2009-03-30	825.06	829.08	789.54	789.54
2009-03-31	785.70	797.26	769.98	773.66
2009-04-01	780.51	795.75	778.21	793.82
2009-04-02	803.57	829.80	803.30	826.69
2009-04-03	835.01	843.46	826.17	831.36
2009-04-06	840.75	847.36	827.22	830.97
2009-04-07	830.93	835.92	827.26	832.60
2009-04-08	824.60	825.64	810.34	815.26
2009-04-09	822.20	842.52	821.70	841.81
2009-04-10	850.64	854.88	836.30	845.97
2009-04-13	844.28	856.08	840.45	848.97
2009-04-14	852.17	852.92	833.73	843.42
2009-04-15	838.53	839.73	825.79	835.25
2009-04-16	843.49	855.19	829.47	832.04
2009-04-17	840.24	848.06	839.40	845.57
2009-04-20	847.00	849.22	838.26	848.30
2009-04-21	838.02	838.02	818.32	830.72
2009-04-22	836.80	838.97	825.26	829.96
2009-04-23	833.28	841.75	823.31	839.50
2009-04-24	839.36	845.06	829.96	830.05
2009-04-27	838.97	843.86	826.70	833.10
2009-04-28	830.07	839.34	811.99	811.99
2009-04-30	822.90	841.67	822.90	837.79
2009-05-01	840.13	849.29	837.68	846.85
2009-05-07	858.93	885.93	858.93	885.93
2009-05-08	883.36	895.35	880.89	895.35
2009-05-11	899.57	906.47	889.91	900.45
2009-05-12	892.54	894.72	885.43	885.43
2009-05-13	886.77	892.04	880.46	888.75
2009-05-14	877.83	878.58	860.44	862.66
2009-05-15	868.78	882.86	867.98	881.65
2009-05-18	872.98	872.98	856.76	859.71
2009-05-19	871.59	883.81	871.46	879.76
2009-05-20	885.62	889.36	881.21	886.30
2009-05-21	880.09	881.44	872.53	881.44
2009-05-22	871.49	879.84	867.65	875.88
2009-05-25	878.48	888.82	878.48	883.00
2009-05-26	887.10	888.20	877.55	883.77
2009-05-27	893.48	898.14	891.46	892.85
2009-05-28	885.76	898.37	885.71	895.59
2009-05-29	897.99	898.34	890.74	897.91
2009-06-01	898.35	914.26	895.56	912.52
2009-06-02	922.92	923.89	913.56	913.56
2009-06-03	914.73	919.71	913.37	914.50
2009-06-04	910.26	918.11	908.36	910.99
2009-06-05	918.59	920.75	913.60	916.56
2009-06-08	923.25	930.23	922.55	926.89
2009-06-09	924.02	927.55	916.90	918.24
2009-06-10	923.33	937.94	922.63	937.01
2009-06-11	937.87	942.62	936.41	940.65
2009-06-12	946.30	954.08	943.66	950.54
2009-06-15	952.88	952.99	944.43	946.82
2009-06-16	936.11	937.24	914.14	914.76
2009-06-17	911.10	924.87	911.10	923.03
2009-06-18	919.54	919.54	906.29	911.21
2009-06-19	917.09	924.84	913.11	918.97
2009-06-22	921.02	928.88	917.35	922.48
2009-06-23	911.60	912.22	897.25	901.69
2009-06-24	905.38	906.78	898.87	902.46
2009-06-25	907.33	926.39	905.18	919.77
2009-06-26	926.65	928.53	920.52	926.80
2009-06-29	925.89	930.91	911.93	915.32
2009-06-30	925.34	932.96	925.11	929.76
2009-07-01	924.18	940.06	922.76	928.30
2009-07-02	933.77	934.55	923.19	924.02
2009-07-03	913.88	920.75	908.77	920.62
2009-07-06	915.32	919.40	910.47	912.42
2009-07-07	914.54	917.25	906.59	909.13
2009-07-08	899.79	899.88	886.64	888.54
2009-07-09	880.63	882.53	873.40	873.91
2009-07-10	877.53	879.94	870.24	872.50
2009-07-13	867.73	878.41	852.11	852.42
2009-07-14	863.04	871.01	862.44	868.57
2009-07-15	872.89	873.79	864.07	866.37
2009-07-16	877.39	886.37	871.12	872.25
2009-07-17	879.15	880.76	874.52	878.29
2009-07-21	889.12	901.55	889.12	901.55
2009-07-22	901.94	909.42	898.48	906.58
2009-07-23	906.14	916.05	905.06	908.69
2009-07-24	918.84	921.65	914.34	920.48
2009-07-27	927.44	936.22	926.39	928.26
2009-07-28	931.12	931.12	925.73	930.13
2009-07-29	925.38	934.81	925.38	930.36
2009-07-30	933.66	938.09	927.67	936.94
2009-07-31	945.55	950.38	943.11	950.26
2009-08-03	953.53	958.45	951.89	957.56
2009-08-04	966.94	970.56	957.05	959.02
2009-08-05	961.61	962.79	949.34	949.58
2009-08-06	950.90	959.88	950.90	957.51
2009-08-07	955.11	956.76	942.72	956.76
2009-08-10	967.02	973.38	965.78	969.24
2009-08-11	968.46	973.83	968.04	973.51
2009-08-12	965.12	966.78	959.87	959.87
2009-08-13	967.19	970.44	965.06	968.41
2009-08-14	969.79	975.88	968.62	973.57
2009-08-17	968.15	968.15	949.23	949.59
2009-08-18	941.78	951.01	941.78	949.66
2009-08-19	950.25	953.77	942.31	943.25
2009-08-20	947.81	961.16	944.29	958.59
2009-08-21	958.16	958.16	938.81	947.34
2009-08-24	960.40	972.31	960.40	970.27
2009-08-25	965.58	969.39	963.05	965.11
2009-08-26	970.27	977.54	968.53	975.59
2009-08-27	970.00	970.73	955.68	964.23
2009-08-28	968.46	970.47	963.76	969.31
2009-08-31	974.87	987.27	959.83	965.73
2009-09-01	962.29	971.91	960.66	968.77
2009-09-02	956.51	956.51	944.18	949.81
2009-09-03	944.74	948.03	940.10	942.77
2009-09-04	944.05	944.16	933.33	935.74
2009-09-07	943.27	945.89	942.21	944.60
2009-09-08	946.41	946.41	940.08	946.40
2009-09-09	943.54	945.39	935.18	939.84
2009-09-10	946.13	959.17	946.13	958.49
2009-09-11	959.94	959.94	947.31	950.41
2009-09-14	946.68	946.68	930.20	934.05
2009-09-15	936.52	940.51	930.97	932.52
2009-09-16	937.75	943.72	930.35	931.43
2009-09-17	939.95	943.45	932.10	939.52
2009-09-18	934.38	939.44	925.50	939.44
2009-09-24	941.79	954.22	940.59	950.20
2009-09-25	936.16	936.18	919.81	922.67
2009-09-28	911.41	912.66	898.08	902.84
2009-09-29	908.02	909.54	901.14	904.00
2009-09-30	905.19	909.84	902.73	909.84
2009-10-01	905.39	906.21	894.08	896.12
2009-10-02	884.19	884.20	871.52	874.67
2009-10-05	875.10	876.65	864.50	867.28
2009-10-06	872.64	873.81	863.78	871.63
2009-10-07	876.59	887.00	875.84	885.69
2009-10-08	886.49	890.98	882.02	887.59
2009-10-09	891.80	897.92	886.08	897.83
2009-10-13	903.29	905.76	900.49	901.40
2009-10-14	901.12	901.12	890.77	894.34
2009-10-15	904.23	909.41	901.54	904.11
2009-10-16	906.23	906.23	896.20	900.95
2009-10-19	895.70	906.30	893.57	905.80
2009-10-20	913.08	914.96	909.75	913.45
2009-10-21	909.21	914.64	908.56	913.70
2009-10-22	907.38	908.60	897.17	908.60
2009-10-23	912.58	912.86	901.40	902.03
2009-10-26	903.50	914.62	902.22	910.72
2009-10-27	904.42	904.95	892.66	895.48
2009-10-28	893.70	896.82	887.00	888.80
2009-10-29	877.94	885.83	874.19	882.26
2009-10-30	891.38	895.61	888.57	894.67
2009-11-02	884.08	884.08	874.66	880.54
2009-11-04	877.69	881.37	875.40	881.27
2009-11-05	879.42	882.07	872.01	874.96
2009-11-06	880.50	880.50	872.29	874.01
2009-11-09	871.86	874.88	866.12	870.67
2009-11-10	877.33	882.21	871.67	872.44
2009-11-11	873.48	879.29	870.94	872.29
2009-11-12	875.79	877.43	865.64	867.70
2009-11-13	866.72	868.76	862.06	866.80
2009-11-16	865.21	865.37	857.67	860.42
2009-11-17	863.16	864.37	854.89	857.00
2009-11-18	853.42	856.17	844.91	850.06
2009-11-19	849.46	849.46	832.09	837.71
2009-11-20	831.03	838.71	829.98	838.71
2009-11-24	839.34	839.40	827.95	829.22
2009-11-25	828.45	834.15	826.36	833.29
2009-11-26	826.97	835.10	824.95	829.56
2009-11-27	819.41	820.40	809.24	811.01
2009-11-30	819.37	839.94	818.86	839.94
2009-12-01	833.52	857.76	829.56	857.76
2009-12-02	854.66	863.14	850.93	858.74
2009-12-03	867.65	888.04	867.57	888.04
2009-12-04	890.62	890.62	883.09	889.58
2009-12-07	898.86	903.65	895.47	898.93
2009-12-08	893.14	898.21	893.12	896.70
2009-12-09	889.00	889.00	883.65	884.94
2009-12-10	882.34	889.02	870.47	873.90
2009-12-11	880.01	888.57	874.62	888.57
2009-12-14	890.48	890.48	877.60	885.08
2009-12-15	882.27	887.52	881.38	884.63
2009-12-16	892.94	903.89	892.94	898.29
2009-12-17	900.87	903.39	896.28	896.28
2009-12-18	892.15	894.80	886.72	893.59
2009-12-21	896.68	897.42	891.48	891.48
2009-12-22	897.58	904.60	895.29	903.06
2009-12-24	907.83	914.73	907.53	913.72
2009-12-25	914.30	914.30	908.95	909.39
2009-12-28	910.91	919.56	910.21	914.78
2009-12-29	913.58	918.58	911.43	915.87
2009-12-30	920.54	920.54	907.59	907.59
2010-01-04	912.48	919.17	912.48	915.75
2010-01-05	926.07	930.51	918.14	919.57
2010-01-06	925.06	932.41	921.12	931.13
2010-01-07	933.84	937.02	929.77	931.85
2010-01-08	937.53	941.81	933.37	941.29
2010-01-12	940.61	955.40	940.61	954.13
2010-01-13	947.66	955.91	943.88	944.02
2010-01-14	948.63	959.01	948.63	959.01
2010-01-15	962.24	966.40	959.04	966.40
2010-01-18	957.72	960.81	953.98	957.55
2010-01-19	957.69	957.69	947.29	949.76
2010-01-20	956.41	956.47	942.99	944.72
2010-01-21	940.92	958.88	937.43	956.03
2010-01-22	942.92	942.95	932.41	940.94
2010-01-25	930.64	938.73	928.87	934.59
2010-01-26	934.01	937.87	916.40	916.40
2010-01-27	917.80	919.66	907.67	907.67
2010-01-28	911.13	920.64	909.59	914.32
2010-01-29	908.73	910.54	901.12	901.12
2010-02-01	900.38	900.38	891.93	898.61
2010-02-02	905.78	913.68	903.73	912.82
2010-02-03	918.20	921.90	915.02	915.68
2010-02-04	918.02	918.55	903.34	911.09
2010-02-05	896.29	896.92	888.73	891.78
2010-02-08	887.90	891.97	882.21	883.01
2010-02-09	877.89	883.58	876.77	881.57
2010-02-10	887.98	890.15	883.50	883.50
2010-02-12	891.80	892.87	887.47	892.16
2010-02-15	892.72	893.85	882.98	883.47
2010-02-16	885.23	887.77	884.03	885.17
2010-02-17	895.87	904.90	895.21	904.63
2010-02-18	907.14	907.36	902.35	904.73
2010-02-19	904.85	906.23	889.08	889.08
2010-02-22	902.76	914.76	902.76	909.75
2010-02-23	906.62	908.42	900.91	907.37
2010-02-24	895.70	897.62	890.86	895.69
2010-02-25	899.59	900.84	889.26	891.41
2010-02-26	890.90	897.13	890.23	894.10
2010-03-01	895.23	902.50	893.81	898.93
2010-03-02	900.50	903.41	896.46	902.71
2010-03-03	900.62	907.11	900.00	905.65
2010-03-04	905.29	906.18	896.68	897.64
2010-03-05	905.11	913.18	905.11	910.81
2010-03-08	923.56	927.31	921.74	927.31
2010-03-09	925.69	927.79	923.60	924.38
2010-03-10	924.10	926.27	921.94	922.44
2010-03-11	928.00	931.16	925.08	930.38
2010-03-12	935.95	937.50	930.94	936.38
2010-03-15	940.81	942.42	935.20	938.91
2010-03-16	936.68	942.33	936.68	938.10
2010-03-17	943.75	949.00	940.84	947.43
2010-03-18	947.83	948.78	940.15	940.79
2010-03-19	944.15	949.01	943.57	948.93
2010-03-23	948.10	951.05	947.13	947.37
2010-03-24	952.53	956.02	948.42	951.97
2010-03-25	955.76	957.26	950.98	952.13
2010-03-26	957.39	966.72	957.30	966.72
2010-03-29	958.19	966.14	957.86	966.13
2010-03-30	971.28	980.33	970.15	979.58
2010-03-31	981.76	984.06	977.39	978.81
2010-04-01	983.99	986.51	978.28	985.26
2010-04-02	991.06	991.06	985.37	989.39
2010-04-05	996.05	996.97	992.41	995.68
2010-04-06	997.11	997.37	986.05	990.79
2010-04-07	991.82	998.97	991.56	995.54
2010-04-08	987.60	992.32	985.80	985.99
2010-04-09	985.62	989.42	983.73	989.42
2010-04-12	996.34	1001.53	994.78	994.78
2010-04-13	993.26	993.56	982.59	988.44
2010-04-14	991.49	995.26	987.40	991.10
2010-04-15	999.54	1001.77	996.97	998.90
2010-04-16	996.55	996.55	987.35	988.84
2010-04-19	974.43	975.87	968.79	970.84
2010-04-20	975.92	980.33	971.41	972.11
2010-04-21	982.98	987.61	982.00	987.07
2010-04-22	981.02	981.02	970.69	978.17
2010-04-23	976.33	979.53	973.74	978.20
2010-04-26	987.35	996.82	987.06	996.71
2010-04-27	992.82	997.63	990.93	997.63
2010-04-28	980.93	981.14	973.57	977.64
2010-04-30	988.94	991.22	984.72	987.04
2010-05-06	970.44	970.46	955.84	956.72
2010-05-07	937.51	937.60	921.36	931.74
2010-05-10	930.54	945.20	927.98	944.64
2010-05-11	954.04	954.34	930.57	932.10
2010-05-12	936.56	940.23	929.39	932.83
2010-05-13	943.48	949.21	941.93	947.90
2010-05-14	936.79	942.89	930.97	936.45
2010-05-17	926.55	928.92	915.23	920.43
2010-05-18	924.23	924.64	910.99	913.91
2010-05-19	902.69	910.64	898.70	910.64
2010-05-20	904.59	908.36	896.01	898.15
2010-05-21	880.90	881.44	872.48	879.69
2010-05-24	878.42	881.73	873.28	880.01
2010-05-25	872.12	872.76	857.00	859.82
2010-05-26	864.18	867.58	854.79	859.00
2010-05-27	851.70	870.26	850.88	869.89
2010-05-28	883.51	884.98	874.41	878.52
2010-05-31	876.60	886.43	876.28	880.46
2010-06-01	879.43	881.55	874.89	880.04
2010-06-02	871.60	882.65	867.20	870.05
2010-06-03	881.66	891.26	881.63	890.64
2010-06-04	891.35	895.09	887.42	890.16
2010-06-07	874.36	874.46	858.90	859.21
2010-06-08	854.33	862.87	853.21	858.33
2010-06-09	854.26	855.11	846.47	850.37
2010-06-10	853.33	857.26	848.87	856.79
2010-06-11	869.60	872.49	865.36	866.44
2010-06-14	875.29	878.56	875.29	878.56
2010-06-15	876.52	881.98	874.91	878.83
2010-06-16	890.53	894.83	889.64	892.38
2010-06-17	889.04	891.62	885.61	887.48
2010-06-18	887.83	887.83	882.34	884.64
2010-06-21	891.86	904.03	891.86	902.49
2010-06-22	895.27	900.39	893.78	894.56
2010-06-23	883.91	884.13	880.33	880.84
2010-06-24	878.91	886.40	878.54	879.77
2010-06-25	870.11	871.63	864.40	867.30
2010-06-28	867.27	867.44	859.30	860.80
2010-06-29	862.55	865.54	850.29	852.19
2010-06-30	839.03	842.19	835.91	841.42
2010-07-01	834.72	834.89	825.06	828.39
2010-07-02	831.30	833.72	825.90	830.98
2010-07-05	833.71	839.30	832.85	836.89
2010-07-06	829.81	847.24	825.26	847.24
2010-07-07	845.69	846.35	837.13	841.51
2010-07-08	856.09	862.73	855.76	861.02
2010-07-09	864.14	864.30	856.99	861.21
2010-07-12	859.21	866.13	857.15	857.70
2010-07-13	861.23	863.78	852.10	854.39
2010-07-14	867.60	874.25	866.18	870.73
2010-07-15	863.75	863.76	856.60	856.60
2010-07-16	850.61	853.21	838.78	840.58
2010-07-20	829.84	837.69	828.73	832.26
2010-07-21	838.64	838.79	827.31	829.35
2010-07-22	824.06	827.43	822.02	825.48
2010-07-23	837.26	844.37	836.08	841.29
2010-07-26	848.79	852.05	845.80	845.88
2010-07-27	846.20	850.12	844.37	846.12
2010-07-28	855.88	866.26	855.88	865.51
2010-07-29	859.39	864.04	859.05	861.27
2010-07-30	858.86	858.86	846.71	849.50
2010-08-02	851.45	858.96	848.82	850.69
2010-08-03	861.82	864.25	855.57	859.18
2010-08-04	854.43	854.71	844.38	845.93
2010-08-05	856.44	859.98	851.31	857.09
2010-08-06	851.83	861.40	851.23	861.17
2010-08-09	853.33	857.62	853.09	857.62
2010-08-10	860.68	862.27	851.22	854.68
2010-08-11	845.54	846.34	833.75	834.45
2010-08-12	820.69	827.78	816.78	827.78
2010-08-13	826.95	832.86	822.92	831.24
2010-08-16	823.05	828.70	820.34	828.63
2010-08-17	820.89	828.17	820.47	826.78
2010-08-18	834.25	838.20	826.99	835.23
2010-08-19	835.41	843.98	835.41	843.98
2010-08-20	833.89	838.20	828.60	829.59
2010-08-23	826.67	828.48	822.45	824.79
2010-08-24	817.23	821.40	814.38	817.73
2010-08-25	809.54	814.31	804.38	807.31
2010-08-26	811.08	811.79	805.18	811.79
2010-08-27	804.44	821.45	804.34	819.62
2010-08-30	830.38	840.68	827.11	829.21
2010-08-31	818.30	818.61	803.78	804.67
2010-09-01	804.25	811.40	800.69	811.40
2010-09-02	822.26	822.98	812.03	819.42
2010-09-03	822.32	825.71	818.95	823.70
2010-09-06	829.48	838.71	827.78	838.71
2010-09-07	833.77	841.17	832.82	834.92
2010-09-08	824.81	825.14	817.21	820.99
2010-09-09	827.95	828.89	823.77	826.84
2010-09-10	830.42	839.75	830.04	833.72
2010-09-13	840.20	843.28	837.14	837.65
2010-09-14	838.07	839.58	833.02	834.87
2010-09-15	832.27	854.11	828.89	848.64
2010-09-16	856.80	856.85	842.59	844.71
2010-09-17	850.03	853.82	848.30	852.09
2010-09-21	858.32	858.77	848.97	849.94
2010-09-22	847.60	851.51	845.38	846.52
2010-09-24	837.79	849.03	835.36	838.41
2010-09-27	844.88	849.30	841.87	849.30
2010-09-28	843.04	845.71	840.69	842.65
2010-09-29	844.06	849.58	841.69	846.97
2010-09-30	846.96	847.92	828.89	829.51
2010-10-01	833.46	833.59	823.90	829.97
2010-10-04	830.35	834.42	821.20	822.74
2010-10-05	820.23	834.30	817.07	832.64
2010-10-06	838.72	844.67	835.37	844.50
2010-10-07	843.11	850.13	842.79	846.06
2010-10-08	844.46	847.23	839.44	839.44
2010-10-12	844.69	844.85	823.32	824.60
2010-10-13	829.74	832.86	821.83	822.65
2010-10-14	830.43	840.11	829.13	836.95
2010-10-15	834.36	834.52	824.31	826.38
2010-10-18	827.21	835.13	826.75	830.52
2010-10-19	830.65	837.74	830.27	833.73
2010-10-20	824.04	824.83	815.67	823.69
2010-10-21	824.11	826.33	815.77	820.40
2010-10-22	821.57	827.69	820.39	824.88
2010-10-25	824.31	826.44	820.14	821.23
2010-10-26	819.30	824.06	817.81	817.94
2010-10-27	821.80	822.79	812.72	817.76
2010-10-28	816.44	819.30	813.35	814.33
2010-10-29	812.98	815.09	803.54	810.91
2010-11-01	806.55	811.18	801.56	803.30
2010-11-02	802.36	803.81	799.64	803.12
2010-11-04	812.97	820.76	812.48	816.33
2010-11-05	827.46	839.56	827.16	834.98
2010-11-08	840.67	842.14	837.08	841.74
2010-11-09	838.18	843.48	836.79	839.95
2010-11-10	843.89	854.15	843.53	852.98
2010-11-11	856.14	858.52	853.43	856.37
2010-11-12	851.99	856.89	846.98	846.98
2010-11-15	851.62	852.26	847.42	851.56
2010-11-16	856.21	857.44	845.51	847.77
2010-11-17	840.22	850.88	840.17	850.30
2010-11-18	851.81	868.81	850.65	868.81
2010-11-19	876.80	877.06	869.13	869.52
2010-11-22	877.04	878.64	873.51	875.48
2010-11-24	862.23	869.96	858.81	866.57
2010-11-25	873.34	874.27	867.56	869.81
2010-11-26	870.10	874.25	866.76	866.81
2010-11-29	868.69	876.43	868.69	874.59
2010-11-30	872.21	874.76	860.94	860.94
2010-12-01	860.70	866.07	858.87	866.07
2010-12-02	878.61	880.72	876.53	877.21
2010-12-03	883.08	884.63	877.37	879.22
2010-12-06	879.49	881.79	878.58	881.41
2010-12-07	880.29	880.47	875.77	879.10
2010-12-08	881.88	887.39	881.48	887.39
2010-12-09	890.08	891.90	887.60	891.60
2010-12-10	896.37	896.37	886.77	888.22
2010-12-13	887.54	897.40	886.38	897.40
2010-12-14	896.40	901.89	896.27	901.89
2010-12-15	904.78	904.99	899.85	902.42
2010-12-16	901.41	907.80	900.54	903.84
2010-12-17	903.78	906.24	901.94	903.14
2010-12-20	903.79	904.95	895.28	898.55
2010-12-21	900.91	908.13	900.68	906.21
2010-12-22	905.43	909.67	903.57	905.78
2010-12-24	901.99	903.96	900.15	901.66
2010-12-27	901.47	906.50	901.47	904.68
2010-12-28	904.31	905.42	902.67	902.83
2010-12-29	902.55	908.01	901.49	908.01
2010-12-30	905.71	906.42	897.74	898.80
2011-01-04	909.44	913.28	906.86	911.80
2011-01-05	912.24	914.29	909.23	911.69
2011-01-06	919.81	924.67	919.41	924.51
2011-01-07	924.69	928.27	923.72	926.42
2011-01-11	923.71	928.70	923.03	926.94
2011-01-12	932.51	935.64	928.78	929.64
2011-01-13	938.00	939.70	934.44	937.74
2011-01-14	933.93	937.66	930.31	930.31
2011-01-17	936.61	936.61	927.09	928.73
2011-01-18	926.59	934.92	925.37	931.58
2011-01-19	935.72	938.03	932.92	936.87
2011-01-20	932.15	933.47	926.32	927.19
2011-01-21	929.01	929.08	909.14	910.85
2011-01-24	914.79	917.54	909.78	917.18
2011-01-25	920.96	931.49	919.04	929.28
2011-01-26	926.27	927.15	922.64	922.64
2011-01-27	925.60	930.50	923.36	929.66
2011-01-28	928.21	929.48	916.95	919.69
2011-01-31	908.30	912.36	904.25	910.08
2011-02-01	914.34	916.41	910.82	913.52
2011-02-02	921.90	931.62	921.47	929.64
2011-02-03	928.42	929.28	925.35	927.57
2011-02-04	935.12	940.80	934.92	935.36
2011-02-07	943.14	943.62	938.57	940.43
2011-02-08	944.76	945.45	942.26	944.00
2011-02-09	948.56	951.28	942.29	944.02
2011-02-10	940.91	948.28	940.49	946.63
2011-02-14	954.24	959.30	953.92	959.19
2011-02-15	960.69	964.31	959.05	962.57
2011-02-16	963.31	971.70	963.27	967.30
2011-02-17	974.33	976.28	971.42	974.14
2011-02-18	973.65	975.69	970.96	973.60
2011-02-21	972.56	974.64	969.27	974.63
2011-02-22	966.87	968.04	955.48	956.70
2011-02-23	947.41	957.25	946.88	946.88
2011-02-24	942.11	942.97	932.35	934.22
2011-02-25	935.21	942.06	934.10	941.93
2011-02-28	941.51	952.56	935.63	951.27
2011-03-01	956.34	964.18	956.19	963.70
2011-03-02	952.05	953.47	942.87	942.87
2011-03-03	945.18	949.43	944.84	948.69
2011-03-04	960.92	962.66	952.90	955.59
2011-03-07	950.22	950.37	938.91	941.63
2011-03-08	941.30	945.50	938.60	939.16
2011-03-09	946.58	951.39	942.75	944.29
2011-03-10	941.31	941.31	928.40	930.84
2011-03-11	920.79	925.41	915.12	915.51
2011-03-14	896.68	896.79	843.86	846.96
2011-03-15	831.52	831.60	725.90	766.73
2011-03-16	779.89	818.94	779.89	817.63
2011-03-17	800.80	821.47	782.19	810.80
2011-03-18	821.27	833.86	820.74	830.39
2011-03-22	848.12	869.95	847.17	868.13
2011-03-23	868.62	868.91	854.91	861.10
2011-03-24	860.78	863.80	853.43	853.95
2011-03-25	865.10	865.16	851.05	857.38
2011-03-28	858.77	858.93	850.72	857.85
2011-03-29	846.22	854.17	839.04	850.21
2011-03-30	852.04	866.09	849.73	866.09
2011-03-31	871.02	871.08	861.76	869.38
2011-04-01	870.96	874.35	862.62	862.62
2011-04-04	867.33	868.72	859.52	859.75
2011-04-05	859.91	860.17	841.76	847.16
2011-04-06	850.00	850.00	837.14	839.61
2011-04-07	845.00	850.02	840.26	841.10
2011-04-08	837.64	855.87	833.06	853.13
2011-04-11	850.75	856.01	849.04	852.34
2011-04-12	842.47	843.52	835.11	838.51
2011-04-13	835.40	845.90	835.27	844.59
2011-04-14	841.02	848.90	838.58	846.72
2011-04-15	845.52	846.77	840.80	841.29
2011-04-18	841.98	843.32	836.34	836.34
2011-04-19	828.11	831.23	824.61	827.56
2011-04-20	833.72	839.13	832.13	837.17
2011-04-21	842.35	844.28	837.53	841.72
2011-04-22	836.92	845.13	834.87	842.18
2011-04-25	842.88	847.84	839.52	840.68
2011-04-26	837.94	838.15	831.80	833.64
2011-04-27	838.42	844.36	836.85	839.87
2011-04-28	844.45	851.85	842.12	851.85
2011-05-02	861.60	866.65	860.49	865.55
2011-05-06	855.16	857.83	851.17	856.50
2011-05-09	859.83	860.43	851.27	853.21
2011-05-10	854.47	859.40	851.09	856.46
2011-05-11	863.01	864.19	856.38	857.62
2011-05-12	852.08	857.64	849.34	849.34
2011-05-13	851.36	851.62	833.19	839.94
2011-05-16	833.98	834.90	829.28	829.55
2011-05-17	827.09	830.42	823.99	828.85
2011-05-18	831.40	839.98	830.30	837.96
2011-05-19	841.70	843.23	830.36	831.89
2011-05-20	830.40	834.05	827.77	827.77
2011-05-23	824.05	824.08	816.43	817.68
2011-05-24	812.88	820.27	812.52	819.16
2011-05-25	821.98	823.53	816.98	817.74
2011-05-26	822.72	827.90	822.27	827.08
2011-05-27	824.70	830.22	821.62	824.90
2011-05-30	823.05	826.56	818.96	823.68
2011-05-31	823.02	838.48	822.70	838.48
2011-06-01	839.41	839.59	834.50	839.41
2011-06-02	826.94	827.22	823.24	825.76
2011-06-03	825.11	828.28	816.34	816.57
2011-06-06	814.57	817.03	805.84	807.99
2011-06-07	807.07	814.60	805.23	813.76
2011-06-08	812.93	815.31	810.12	814.45
2011-06-09	811.00	813.34	807.14	812.95
2011-06-10	818.64	823.79	815.50	817.38
2011-06-13	808.95	813.06	806.99	812.26
2011-06-14	812.13	823.83	811.49	822.86
2011-06-15	826.74	827.35	820.90	824.65
2011-06-16	817.14	819.76	812.41	812.41
2011-06-17	814.71	814.83	801.78	805.34
2011-06-20	807.88	812.20	805.94	806.83
2011-06-21	811.37	815.73	808.38	815.73
2011-06-22	819.51	831.30	819.51	828.99
2011-06-23	823.73	830.85	822.78	825.51
2011-06-24	827.25	834.37	826.90	833.20
2011-06-27	831.21	831.32	824.64	825.64
2011-06-28	832.51	835.51	828.79	830.34
2011-06-29	839.82	844.11	837.99	844.11
2011-06-30	849.14	850.32	844.31	849.22
2011-07-01	854.83	856.79	852.20	853.86
2011-07-04	863.50	866.53	862.00	864.11
2011-07-05	864.12	867.46	863.78	865.18
2011-07-06	865.65	873.51	863.53	873.51
2011-07-07	869.98	873.05	867.87	870.48
2011-07-08	878.39	879.48	873.96	874.34
2011-07-11	869.49	873.01	868.52	870.16
2011-07-12	859.14	861.72	855.33	857.19
2011-07-13	853.48	862.76	853.37	860.53
2011-07-14	858.01	862.06	853.98	856.88
2011-07-15	856.06	860.76	855.85	859.36
2011-07-19	854.81	857.45	852.89	853.75
2011-07-20	862.39	864.91	858.89	860.66
2011-07-21	862.07	862.31	858.13	860.11
2011-07-22	866.95	870.71	864.83	868.81
2011-07-25	866.12	866.71	861.12	861.91
2011-07-26	863.67	869.84	861.95	866.20
2011-07-27	861.69	861.69	854.82	859.11
2011-07-28	850.14	851.88	844.92	848.37
2011-07-29	846.65	848.46	841.37	841.37
2011-08-01	847.12	858.53	847.02	851.70
2011-08-02	845.77	846.66	841.98	843.96
2011-08-03	831.48	831.76	825.18	826.75
2011-08-04	829.07	835.53	824.10	826.36
2011-08-05	809.46	809.72	794.16	800.96
2011-08-08	789.59	791.94	779.87	782.86
2011-08-09	766.94	770.39	744.46	770.39
2011-08-10	783.91	785.29	775.27	776.73
2011-08-11	762.70	770.88	759.78	770.88
2011-08-12	778.78	779.67	765.38	768.19
2011-08-15	778.07	779.72	772.68	777.12
2011-08-16	781.36	783.95	776.36	779.06
2011-08-17	775.38	778.02	771.12	776.65
2011-08-18	775.57	775.66	765.91	767.31
2011-08-19	753.27	756.08	749.45	751.69
2011-08-22	748.51	752.86	742.13	742.84
2011-08-23	747.97	751.85	742.76	750.39
2011-08-24	756.95	759.40	740.51	742.24
2011-08-25	749.93	759.34	749.21	751.82
2011-08-26	750.82	756.35	749.99	756.07
2011-08-29	757.21	765.64	751.53	758.83
2011-08-30	768.00	773.31	766.37	767.30
2011-08-31	767.14	771.18	764.48	770.60
2011-09-01	776.04	780.99	774.52	778.28
2011-09-02	772.42	774.26	766.38	769.78
2011-09-05	760.77	762.15	753.46	755.82
2011-09-06	750.31	750.34	740.39	741.20
2011-09-07	750.87	755.29	748.43	753.63
2011-09-08	761.76	763.08	754.12	757.41
2011-09-09	754.60	760.52	754.60	755.70
2011-09-12	742.61	743.09	738.80	741.26
2011-09-13	745.40	751.75	741.90	749.82
2011-09-14	749.81	753.56	740.58	741.69
2011-09-15	750.53	754.92	749.46	751.76
2011-09-16	760.41	768.42	759.56	768.13
2011-09-20	760.62	760.74	753.37	755.04
2011-09-21	755.50	758.93	754.39	757.13
2011-09-22	749.56	749.77	740.85	744.54
2011-09-26	744.06	744.28	727.33	728.85
2011-09-27	738.64	748.55	737.55	748.55
2011-09-28	746.96	755.60	746.59	754.07
2011-09-29	747.78	762.30	747.26	762.30
2011-09-30	764.06	765.89	755.56	761.17
2011-10-03	748.80	749.67	735.94	747.11
2011-10-04	737.20	737.61	729.80	736.18
2011-10-05	739.36	739.81	724.77	726.25
2011-10-06	732.60	740.28	732.25	736.86
2011-10-07	742.98	748.64	740.95	741.55
2011-10-11	752.38	757.93	751.83	755.00
2011-10-12	750.71	754.92	747.25	753.44
2011-10-13	761.08	762.02	758.00	758.83
2011-10-14	752.81	753.87	748.71	748.81
2011-10-17	759.36	763.85	757.93	761.88
2011-10-18	752.62	754.94	749.64	751.24
2011-10-19	756.34	757.88	749.62	751.49
2011-10-20	749.73	749.88	742.72	746.02
2011-10-21	744.77	746.59	741.87	744.21
2011-10-24	749.25	755.55	749.14	755.44
2011-10-25	757.60	757.85	747.04	747.70
2011-10-26	741.78	751.21	736.50	746.48
2011-10-27	748.11	763.04	744.28	762.79
2011-10-28	774.11	777.39	769.94	771.43
2011-10-31	768.55	779.08	764.06	764.06
2011-11-01	757.77	761.04	753.39	754.50
2011-11-02	745.07	745.21	737.21	738.58
2011-11-04	747.69	752.85	745.38	752.02
2011-11-07	749.96	750.45	747.06	750.45
2011-11-08	747.57	749.93	736.89	738.03
2011-11-09	744.06	749.87	740.90	749.40
2011-11-10	735.64	735.64	727.72	730.30
2011-11-11	731.54	733.19	726.24	729.13
2011-11-14	738.04	739.91	733.93	735.85
2011-11-15	731.82	735.04	729.97	730.91
2011-11-16	730.57	731.61	723.58	724.11
2011-11-17	720.63	728.63	717.78	727.71
2011-11-18	719.57	721.47	717.61	719.98
2011-11-21	716.87	719.19	716.28	717.08
2011-11-22	710.31	719.50	709.86	717.79
2011-11-24	708.05	712.00	704.77	706.08
2011-11-25	704.08	711.36	703.88	706.60
2011-11-28	714.09	719.61	713.98	715.70
2011-11-29	721.94	729.83	719.84	729.68
2011-11-30	724.87	728.60	721.10	728.46
2011-12-01	740.70	746.44	738.76	740.01
2011-12-02	741.83	744.46	739.52	744.14
2011-12-05	748.99	749.39	745.32	748.61
2011-12-06	745.55	747.77	738.01	738.01
2011-12-07	741.36	750.61	739.97	749.63
2011-12-08	745.62	748.39	742.32	745.11
2011-12-09	735.99	741.82	735.99	738.12
2011-12-12	747.73	749.29	745.65	746.69
2011-12-13	737.47	743.74	736.05	740.71
2011-12-14	738.45	739.46	734.90	736.98
2011-12-15	732.17	733.06	724.97	725.02
2011-12-16	726.99	727.85	722.75	723.56
2011-12-19	720.38	720.65	712.27	716.38
2011-12-20	718.13	720.59	717.79	718.49
2011-12-21	727.00	727.60	723.88	725.68
2011-12-22	724.19	725.33	721.77	723.12
2011-12-26	730.21	730.21	725.82	726.44
2011-12-27	723.75	725.38	723.02	724.25
2011-12-28	723.34	725.06	721.04	721.45
2011-12-29	719.24	722.98	716.82	722.12
2011-12-30	724.39	728.62	723.08	728.61
2012-01-04	737.52	744.30	736.60	742.99
2012-01-05	740.70	740.70	736.09	736.28
2012-01-06	735.25	735.82	725.97	729.60
2012-01-10	732.95	736.26	731.37	731.93
2012-01-11	733.45	735.89	731.36	733.47
2012-01-12	732.13	732.21	725.28	727.15
2012-01-13	731.82	735.85	731.82	734.60
2012-01-16	728.54	728.54	722.85	725.24
2012-01-17	727.97	731.61	727.14	731.53
2012-01-18	730.23	740.51	728.66	734.98
2012-01-19	738.68	743.78	738.68	740.68
2012-01-20	750.48	757.10	750.01	755.47
2012-01-23	755.41	758.41	753.83	756.79
2012-01-24	760.25	760.61	755.76	757.40
2012-01-25	761.79	769.36	759.97	767.40
2012-01-26	768.14	768.61	763.28	764.61
2012-01-27	763.95	766.38	759.97	761.13
2012-01-30	758.56	760.96	756.32	757.01
2012-01-31	755.69	759.31	753.84	755.27
2012-02-01	755.10	760.87	754.84	757.96
2012-02-02	762.52	766.13	762.13	762.45
2012-02-03	761.63	764.85	759.58	760.69
2012-02-06	770.01	772.36	768.54	769.85
2012-02-07	768.64	773.17	768.64	772.77
2012-02-08	776.52	782.34	775.75	782.34
2012-02-09	781.34	785.62	777.67	784.49
2012-02-10	785.88	786.05	778.78	779.07
2012-02-13	777.45	784.08	777.42	781.68
2012-02-14	780.25	787.70	778.65	786.80
2012-02-15	791.76	806.63	791.03	802.96
2012-02-16	799.87	806.38	798.40	800.25
2012-02-17	810.69	815.45	809.64	810.45
2012-02-20	821.45	823.48	817.34	819.03
2012-02-21	817.82	821.86	814.61	816.29
2012-02-22	816.99	826.11	815.43	825.40
2012-02-23	825.19	831.37	820.93	829.35
2012-02-24	829.80	835.10	828.67	834.29
2012-02-27	841.45	843.74	834.80	835.25
2012-02-28	828.69	839.20	825.81	838.48
2012-02-29	843.82	847.83	834.21	835.96
2012-03-01	840.23	845.91	827.56	831.54
2012-03-02	839.33	841.00	833.34	837.82
2012-03-05	836.44	840.39	830.90	832.86
2012-03-06	833.14	835.58	824.79	827.35
2012-03-07	815.87	823.42	815.85	822.71
2012-03-08	830.20	836.32	829.82	836.16
2012-03-09	847.26	853.35	842.07	848.71
2012-03-12	854.37	854.37	845.28	845.28
2012-03-13	846.54	854.24	845.33	845.33
2012-03-14	858.64	861.46	857.11	857.11
2012-03-15	862.62	866.42	858.71	863.61
2012-03-16	862.48	866.73	861.81	866.73
2012-03-19	868.38	870.35	866.99	868.35
2012-03-21	864.47	865.42	858.42	858.78
2012-03-22	856.39	863.82	856.21	862.07
2012-03-23	852.77	854.97	852.31	852.53
2012-03-26	854.41	856.03	851.82	851.82
2012-03-27	862.36	872.42	862.30	872.42
2012-03-28	863.02	865.32	859.42	864.43
2012-03-29	860.92	862.15	855.38	857.74
2012-03-30	857.20	857.26	851.63	854.35
2012-04-02	862.07	863.23	855.49	856.05
2012-04-03	854.17	854.33	849.24	851.02
2012-04-04	851.77	853.39	833.97	835.36
2012-04-05	827.07	835.05	823.77	832.57
2012-04-06	826.67	829.92	823.61	825.71
2012-04-09	815.36	819.42	813.69	813.69
2012-04-10	817.28	822.05	811.67	813.43
2012-04-11	802.16	808.50	800.57	805.84
2012-04-12	807.90	811.44	804.30	809.88
2012-04-13	816.40	819.14	813.28	815.48
2012-04-16	806.99	809.16	803.62	803.83
2012-04-17	805.14	807.92	802.72	803.09
2012-04-18	813.55	820.42	813.51	819.27
2012-04-19	814.37	817.08	812.47	814.13
2012-04-20	814.12	814.22	810.08	811.94
2012-04-23	814.41	818.88	808.18	809.54
2012-04-24	802.85	806.78	801.20	803.94
cut3
2009-01-05	8991.21	9127.38	8987.36	9043.12
2009-01-06	9130.01	9171.03	9029.94	9080.84
2009-01-07	9133.80	9325.35	9106.05	9239.24
2009-01-08	9143.21	9148.83	8876.42	8876.42
2009-01-09	8932.71	8956.85	8773.20	8836.80
2009-01-13	8732.63	8732.93	8405.50	8413.91
2009-01-14	8425.75	8516.07	8359.16	8438.45
2009-01-15	8309.38	8309.38	7997.73	8023.31
2009-01-16	8125.20	8283.91	8067.47	8230.15
2009-01-19	8318.26	8351.68	8221.84	8256.85
2009-01-20	8187.14	8190.42	7962.46	8065.79
2009-01-21	7949.96	8009.22	7829.30	7901.64
2009-01-22	7988.30	8051.74	7809.89	8051.74
2009-01-23	7965.41	7965.41	7745.25	7745.25
2009-01-26	7714.26	7807.16	7671.04	7682.14
2009-01-27	7782.90	8115.15	7782.07	8061.07
2009-01-28	8052.25	8171.63	7936.59	8106.29
2009-01-29	8201.16	8305.38	8138.99	8251.24
2009-01-30	8142.88	8142.88	7922.39	7994.05
2009-02-02	7908.51	7955.75	7795.27	7873.98
2009-02-03	7862.95	8084.41	7800.80	7825.51
2009-02-04	7897.24	8084.97	7863.65	8038.94
2009-02-05	7985.53	8093.96	7901.04	7949.65
2009-02-06	8054.27	8169.04	8033.24	8076.62
2009-02-09	8178.07	8257.71	7969.03	7969.03
2009-02-10	8066.94	8124.79	7917.27	7945.94
2009-02-12	7842.53	7862.52	7685.68	7705.36
2009-02-13	7789.35	7887.74	7730.27	7779.40
2009-02-16	7732.68	7804.24	7694.73	7750.17
2009-02-17	7690.13	7710.43	7615.94	7645.51
2009-02-18	7539.96	7565.79	7479.18	7534.44
2009-02-19	7604.22	7642.69	7537.56	7557.65
2009-02-20	7544.07	7554.70	7382.33	7416.38
2009-02-23	7314.30	7417.18	7209.43	7376.16
2009-02-24	7266.68	7270.90	7155.16	7268.56
2009-02-25	7368.44	7471.03	7330.44	7461.22
2009-02-26	7470.60	7599.81	7433.06	7457.93
2009-02-27	7463.42	7589.77	7414.40	7568.42
2009-03-02	7454.28	7454.28	7234.96	7280.15
2009-03-03	7177.79	7288.14	7088.47	7229.72
2009-03-04	7146.71	7320.65	7104.63	7290.96
2009-03-05	7336.02	7532.87	7336.02	7433.49
2009-03-06	7328.29	7328.29	7167.07	7173.10
2009-03-09	7191.13	7241.02	7028.49	7086.03
2009-03-10	7059.77	7100.77	7021.28	7054.98
2009-03-11	7165.39	7393.81	7161.85	7376.12
2009-03-12	7320.45	7345.02	7198.25	7198.25
2009-03-13	7301.12	7571.45	7300.87	7569.28
2009-03-16	7630.20	7754.75	7630.20	7704.15
2009-03-17	7767.34	7967.03	7723.94	7949.13
2009-03-18	8006.86	8054.35	7895.28	7972.17
2009-03-19	8017.93	8034.09	7902.49	7945.96
2009-03-23	7943.14	8229.13	7922.55	8215.53
2009-03-24	8334.68	8504.41	8297.27	8488.30
2009-03-25	8499.69	8553.01	8392.56	8479.99
2009-03-26	8430.22	8640.28	8383.99	8636.33
2009-03-27	8711.72	8843.18	8626.97	8626.97
2009-03-30	8621.85	8651.06	8236.08	8236.08
2009-03-31	8199.43	8383.74	8088.45	8109.53
2009-04-01	8173.36	8351.91	8084.62	8351.91
2009-04-02	8453.73	8741.67	8449.87	8719.78
2009-04-03	8814.10	8884.63	8697.17	8749.84
2009-04-06	8856.84	8992.06	8812.36	8857.93
2009-04-07	8838.66	8884.45	8778.92	8832.85
2009-04-08	8746.73	8765.64	8556.75	8595.01
2009-04-09	8665.16	8920.86	8664.26	8916.06
2009-04-10	9041.23	9068.80	8856.69	8964.11
2009-04-13	8930.35	9024.45	8888.10	8924.43
2009-04-14	8955.90	8961.73	8749.92	8842.68
2009-04-15	8777.68	8800.52	8681.03	8742.96
2009-04-16	8848.43	9030.00	8720.62	8755.26
2009-04-17	8854.33	8953.34	8834.63	8907.58
2009-04-20	8899.59	8933.80	8813.72	8924.75
2009-04-21	8802.09	8802.09	8612.76	8711.33
2009-04-22	8777.53	8802.90	8683.27	8727.30
2009-04-23	8776.94	8860.55	8647.80	8847.01
2009-04-24	8832.10	8852.83	8694.92	8707.99
2009-04-27	8783.34	8840.53	8648.51	8726.34
2009-04-28	8678.28	8808.64	8493.77	8493.77
2009-04-30	8615.45	8844.77	8615.45	8828.26
2009-05-01	8848.84	9017.32	8827.13	8977.37
2009-05-07	9102.35	9385.70	9100.11	9385.70
2009-05-08	9351.40	9464.43	9349.57	9432.83
2009-05-11	9460.72	9503.91	9342.75	9451.98
2009-05-12	9358.25	9389.61	9298.61	9298.61
2009-05-13	9305.79	9379.47	9278.89	9340.49
2009-05-14	9212.30	9223.77	9052.41	9093.73
2009-05-15	9150.21	9272.08	9140.90	9265.02
2009-05-18	9167.05	9167.82	8997.74	9038.69
2009-05-19	9172.56	9326.75	9166.97	9290.29
2009-05-20	9372.72	9399.40	9311.61	9344.64
2009-05-21	9280.35	9286.35	9189.92	9264.15
2009-05-22	9156.14	9267.82	9126.69	9225.81
2009-05-25	9245.95	9402.76	9245.95	9347.00
2009-05-26	9364.22	9379.78	9231.93	9310.81
2009-05-27	9426.62	9491.13	9411.54	9438.77
2009-05-28	9353.33	9492.66	9353.33	9451.39
2009-05-29	9478.21	9522.50	9426.52	9522.50
2009-06-01	9517.49	9691.73	9491.26	9677.75
2009-06-02	9774.55	9793.47	9704.31	9704.31
2009-06-03	9723.82	9774.55	9718.70	9741.67
2009-06-04	9690.93	9743.46	9654.45	9668.96
2009-06-05	9752.51	9774.98	9719.14	9768.01
2009-06-08	9829.28	9914.07	9816.17	9865.63
2009-06-09	9824.17	9855.89	9755.81	9786.82
2009-06-10	9833.64	9991.49	9813.77	9991.49
2009-06-11	9992.98	10022.23	9958.98	9981.33
2009-06-12	10088.21	10170.82	10050.96	10135.82
2009-06-15	10126.55	10126.55	10029.59	10039.67
2009-06-16	9914.40	9941.50	9752.88	9752.88
2009-06-17	9705.73	9845.52	9705.73	9840.85
2009-06-18	9778.35	9778.35	9649.83	9703.72
2009-06-19	9757.85	9822.83	9703.87	9786.26
2009-06-22	9788.07	9887.13	9763.37	9826.27
2009-06-23	9695.76	9712.09	9511.45	9549.61
2009-06-24	9596.78	9624.41	9534.24	9590.32
2009-06-25	9628.18	9871.14	9619.02	9796.08
2009-06-26	9852.92	9893.00	9793.95	9877.39
2009-06-29	9866.28	9943.83	9748.73	9783.47
2009-06-30	9896.56	10000.30	9894.95	9958.44
2009-07-01	9889.34	10086.18	9874.00	9939.93
2009-07-02	9993.77	9993.77	9869.54	9876.15
2009-07-03	9751.69	9816.07	9691.16	9816.07
2009-07-06	9738.49	9771.76	9650.75	9680.87
2009-07-07	9713.62	9734.43	9619.68	9647.79
2009-07-08	9548.81	9557.39	9407.98	9420.75
2009-07-09	9342.33	9384.24	9291.06	9291.06
2009-07-10	9338.76	9370.15	9265.24	9287.28
2009-07-13	9242.13	9343.86	9050.33	9050.33
2009-07-14	9173.14	9284.38	9166.63	9261.81
2009-07-15	9307.45	9333.64	9250.73	9269.25
2009-07-16	9393.98	9489.67	9317.11	9344.16
2009-07-17	9413.86	9420.61	9360.59	9395.32
2009-07-21	9512.52	9652.02	9508.92	9652.02
2009-07-22	9642.79	9750.63	9608.77	9723.16
2009-07-23	9712.10	9861.29	9703.84	9792.94
2009-07-24	9909.40	9950.04	9863.87	9944.55
2009-07-27	10020.87	10179.59	10015.11	10088.66
2009-07-28	10116.89	10116.89	10049.63	10087.26
2009-07-29	10035.91	10161.95	10035.91	10113.24
2009-07-30	10155.71	10169.68	10067.51	10165.21
2009-07-31	10275.14	10359.07	10259.57	10356.83
2009-08-03	10355.23	10380.57	10322.00	10352.47
2009-08-04	10438.40	10479.19	10358.48	10375.01
2009-08-05	10405.05	10406.10	10252.53	10252.53
2009-08-06	10266.01	10426.14	10266.01	10388.09
2009-08-07	10365.71	10412.09	10249.04	10412.09
2009-08-10	10530.62	10585.37	10487.70	10524.26
2009-08-11	10514.74	10587.36	10512.21	10585.46
2009-08-12	10486.36	10526.35	10434.09	10435.00
2009-08-13	10508.09	10556.28	10489.12	10517.19
2009-08-14	10551.33	10630.38	10539.43	10597.33
2009-08-17	10521.02	10521.02	10268.61	10268.61
2009-08-18	10208.12	10325.06	10181.10	10284.96
2009-08-19	10269.97	10327.17	10195.11	10204.00
2009-08-20	10254.53	10407.30	10204.26	10383.41
2009-08-21	10375.22	10375.22	10142.22	10238.20
2009-08-24	10389.57	10591.00	10388.78	10581.05
2009-08-25	10515.57	10566.31	10469.69	10497.36
2009-08-26	10554.44	10668.74	10526.51	10639.71
2009-08-27	10570.78	10575.79	10400.83	10473.97
2009-08-28	10546.88	10560.44	10471.63	10534.14
2009-08-31	10608.16	10767.00	10423.90	10492.53
2009-09-01	10453.37	10577.19	10438.51	10530.06
2009-09-02	10378.08	10378.08	10215.60	10280.46
2009-09-03	10240.02	10290.83	10185.38	10214.64
2009-09-04	10256.49	10256.49	10143.26	10187.11
2009-09-07	10288.56	10326.45	10274.38	10320.94
2009-09-08	10353.51	10393.23	10300.50	10393.23
2009-09-09	10343.80	10390.25	10249.11	10312.14
2009-09-10	10385.35	10521.14	10382.99	10513.67
2009-09-11	10519.33	10522.23	10407.97	10444.33
2009-09-14	10388.32	10388.32	10170.74	10202.06
2009-09-15	10255.73	10292.47	10183.30	10217.62
2009-09-16	10288.72	10394.56	10247.24	10270.77
2009-09-17	10373.25	10443.80	10340.14	10443.80
2009-09-18	10379.21	10399.42	10292.50	10370.54
2009-09-24	10405.53	10566.98	10386.88	10544.22
2009-09-25	10395.45	10395.45	10228.60	10265.98
2009-09-28	10136.24	10151.60	9971.05	10009.52
2009-09-29	10089.11	10125.88	10059.87	10100.20
2009-09-30	10105.17	10138.04	10063.69	10133.23
2009-10-01	10072.64	10077.22	9965.06	9978.64
2009-10-02	9848.07	9852.17	9714.18	9731.87
2009-10-05	9733.07	9751.56	9669.31	9674.49
2009-10-06	9744.42	9744.42	9628.67	9691.80
2009-10-07	9750.24	9821.74	9735.68	9799.60
2009-10-08	9806.03	9863.34	9746.15	9832.47
2009-10-09	9893.63	10016.39	9857.39	10016.39
2009-10-13	10066.85	10116.72	10039.62	10076.56
2009-10-14	10096.01	10096.01	10016.24	10060.21
2009-10-15	10194.91	10272.62	10184.81	10238.65
2009-10-16	10276.40	10290.31	10216.14	10257.56
2009-10-19	10179.46	10264.44	10125.33	10236.51
2009-10-20	10329.30	10357.59	10307.59	10336.84
2009-10-21	10292.39	10350.45	10292.39	10333.39
2009-10-22	10251.09	10274.61	10159.66	10267.17
2009-10-23	10335.34	10364.17	10271.71	10282.99
2009-10-26	10271.84	10397.69	10250.84	10362.62
2009-10-27	10283.46	10290.92	10201.93	10212.46
2009-10-28	10182.14	10209.20	10060.39	10075.05
2009-10-29	9941.57	9945.41	9850.12	9891.10
2009-10-30	10006.84	10049.59	9984.33	10034.74
2009-11-02	9903.77	9904.61	9736.14	9802.95
2009-11-04	9785.28	9845.00	9767.91	9844.31
2009-11-05	9803.50	9825.82	9691.69	9717.44
2009-11-06	9812.73	9849.72	9767.00	9789.35
2009-11-09	9777.64	9845.90	9731.59	9808.99
2009-11-10	9904.68	9979.46	9857.10	9870.73
2009-11-11	9890.18	9949.71	9849.99	9871.68
2009-11-12	9921.45	9943.57	9788.49	9804.49
2009-11-13	9793.19	9805.20	9719.38	9770.31
2009-11-16	9784.14	9802.53	9725.81	9791.18
2009-11-17	9834.58	9847.64	9715.28	9729.93
2009-11-18	9723.23	9788.81	9631.83	9676.80
2009-11-19	9691.86	9691.86	9496.07	9549.47
2009-11-20	9460.21	9513.60	9423.66	9497.68
2009-11-24	9511.00	9511.00	9397.79	9401.58
2009-11-25	9381.90	9453.70	9366.33	9441.64
2009-11-26	9354.80	9457.24	9324.99	9383.24
2009-11-27	9257.20	9257.20	9076.41	9081.52
2009-11-30	9189.17	9353.74	9162.18	9345.55
2009-12-01	9281.82	9572.20	9233.20	9572.20
2009-12-02	9551.79	9643.32	9514.15	9608.94
2009-12-03	9707.02	9977.67	9707.02	9977.67
2009-12-04	10019.24	10022.59	9902.83	10022.59
2009-12-07	10131.90	10204.58	10106.31	10167.60
2009-12-08	10079.81	10149.32	10079.81	10140.47
2009-12-09	10048.94	10048.94	9985.97	10004.72
2009-12-10	9964.05	10035.92	9834.22	9862.82
2009-12-11	9958.00	10107.87	9916.21	10107.87
2009-12-14	10126.61	10126.61	10009.60	10105.68
2009-12-15	10053.45	10111.76	10033.90	10083.48
2009-12-16	10179.33	10222.22	10117.12	10177.41
2009-12-17	10199.92	10260.12	10163.80	10163.80
2009-12-18	10110.64	10157.25	10027.85	10142.05
2009-12-21	10196.71	10215.49	10183.47	10183.47
2009-12-22	10256.00	10378.03	10235.22	10378.03
2009-12-24	10413.37	10558.41	10413.37	10536.92
2009-12-25	10546.97	10546.97	10476.65	10494.71
2009-12-28	10521.81	10652.99	10513.55	10634.23
2009-12-29	10611.85	10683.12	10597.41	10638.06
2009-12-30	10707.51	10707.51	10546.44	10546.44
2010-01-04	10609.34	10694.49	10608.14	10654.79
2010-01-05	10719.44	10791.04	10655.57	10681.83
2010-01-06	10709.55	10768.61	10661.17	10731.45
2010-01-07	10742.75	10774.00	10636.67	10681.66
2010-01-08	10743.30	10816.45	10677.56	10798.32
2010-01-12	10770.35	10905.39	10763.68	10879.14
2010-01-13	10795.48	10866.62	10729.86	10735.03
2010-01-14	10778.07	10909.94	10774.25	10907.68
2010-01-15	10917.41	10982.10	10878.83	10982.10
2010-01-18	10887.61	10895.10	10781.03	10855.08
2010-01-19	10866.83	10866.83	10749.47	10764.90
2010-01-20	10834.91	10860.93	10724.57	10737.52
2010-01-21	10704.79	10886.64	10649.84	10868.41
2010-01-22	10740.21	10768.07	10528.33	10590.55
2010-01-25	10478.31	10557.64	10414.58	10512.69
2010-01-26	10506.15	10566.49	10324.98	10325.28
2010-01-27	10344.07	10373.82	10252.08	10252.08
2010-01-28	10309.73	10462.70	10296.98	10414.29
2010-01-29	10308.05	10324.37	10198.04	10198.04
2010-02-01	10212.36	10224.83	10129.91	10205.02
2010-02-02	10310.98	10396.48	10287.74	10371.09
2010-02-03	10428.12	10436.52	10356.03	10404.33
2010-02-04	10434.52	10438.41	10279.57	10355.98
2010-02-05	10162.34	10166.30	10036.33	10057.09
2010-02-08	10007.47	10063.53	9942.05	9951.82
2010-02-09	9876.61	9956.79	9867.39	9932.90
2010-02-10	10024.26	10049.87	9963.99	9963.99
2010-02-12	10085.35	10099.46	10014.50	10092.19
2010-02-15	10097.82	10119.47	10012.53	10013.30
2010-02-16	10044.53	10062.27	10019.43	10034.25
2010-02-17	10161.72	10306.83	10150.24	10306.83
2010-02-18	10324.98	10340.38	10285.05	10335.69
2010-02-19	10334.40	10354.42	10123.58	10123.58
2010-02-22	10302.07	10449.75	10298.15	10400.47
2010-02-23	10327.64	10375.55	10280.33	10352.10
2010-02-24	10205.15	10243.20	10129.65	10198.83
2010-02-25	10256.10	10267.43	10087.21	10101.96
2010-02-26	10107.08	10171.22	10085.13	10126.03
2010-03-01	10128.73	10215.15	10116.86	10172.06
2010-03-02	10199.19	10238.96	10150.30	10221.84
2010-03-03	10193.95	10274.09	10186.68	10253.14
2010-03-04	10255.69	10263.48	10134.37	10145.72
2010-03-05	10254.61	10376.41	10254.61	10368.96
2010-03-08	10538.12	10586.97	10514.28	10585.92
2010-03-09	10567.32	10593.27	10542.81	10567.65
2010-03-10	10555.85	10588.69	10547.34	10563.92
2010-03-11	10627.99	10664.95	10588.86	10664.95
2010-03-12	10777.49	10777.49	10699.30	10751.26
2010-03-15	10802.11	10808.84	10708.97	10751.98
2010-03-16	10718.63	10776.52	10716.20	10721.71
2010-03-17	10789.24	10864.30	10761.89	10846.98
2010-03-18	10841.71	10845.73	10731.66	10744.03
2010-03-19	10785.88	10826.76	10775.57	10824.72
2010-03-23	10793.05	10817.78	10766.65	10774.15
2010-03-24	10829.36	10880.62	10769.18	10815.03
2010-03-25	10857.81	10872.45	10808.71	10828.85
2010-03-26	10895.29	11001.59	10889.33	10996.37
2010-03-29	10909.98	10990.75	10901.20	10986.47
2010-03-30	11022.20	11108.82	11003.35	11097.14
2010-03-31	11138.72	11147.62	11085.55	11089.94
2010-04-01	11178.92	11272.73	11118.18	11244.40
2010-04-02	11274.18	11313.98	11235.98	11286.09
2010-04-05	11380.63	11408.17	11306.62	11339.30
2010-04-06	11350.70	11358.45	11217.25	11282.32
2010-04-07	11296.31	11350.57	11260.42	11292.83
2010-04-08	11179.67	11232.36	11159.45	11168.20
2010-04-09	11174.62	11204.87	11148.80	11204.34
2010-04-12	11300.79	11351.55	11248.57	11251.90
2010-04-13	11202.92	11212.65	11088.48	11161.23
2010-04-14	11213.46	11270.20	11161.01	11204.90
2010-04-15	11310.07	11316.66	11255.80	11273.79
2010-04-16	11230.53	11230.53	11084.72	11102.18
2010-04-19	10929.64	10953.90	10878.99	10908.77
2010-04-20	10960.11	10999.57	10890.82	10900.68
2010-04-21	11031.79	11093.30	11023.88	11090.05
2010-04-22	10988.01	10989.62	10866.23	10949.09
2010-04-23	10930.90	10967.12	10865.92	10914.46
2010-04-26	11039.59	11165.79	11035.66	11165.79
2010-04-27	11109.26	11213.45	11108.72	11212.66
2010-04-28	11005.51	11008.88	10882.40	10924.79
2010-04-30	11062.12	11092.52	11035.27	11057.40
2010-05-06	10847.90	10847.90	10682.61	10695.69
2010-05-07	10472.70	10472.77	10257.32	10364.59
2010-05-10	10399.05	10540.52	10362.34	10530.70
2010-05-11	10643.24	10643.28	10392.47	10411.10
2010-05-12	10465.03	10501.72	10345.19	10394.03
2010-05-13	10542.77	10638.23	10535.94	10620.55
2010-05-14	10453.40	10551.69	10390.77	10462.51
2010-05-17	10329.37	10331.87	10158.30	10235.76
2010-05-18	10313.14	10336.53	10197.74	10242.64
2010-05-19	10087.33	10186.84	10041.93	10186.84
2010-05-20	10121.85	10161.70	9999.59	10030.31
2010-05-21	9823.84	9829.36	9696.63	9784.54
2010-05-24	9754.99	9793.46	9693.07	9758.40
2010-05-25	9632.14	9642.60	9432.09	9459.89
2010-05-26	9567.35	9622.40	9477.45	9522.66
2010-05-27	9419.30	9642.16	9395.29	9639.72
2010-05-28	9806.06	9824.85	9699.14	9762.98
2010-05-31	9735.47	9831.73	9728.42	9768.70
2010-06-01	9747.26	9747.26	9658.44	9711.83
2010-06-02	9604.45	9763.41	9560.16	9603.24
2010-06-03	9758.82	9920.18	9758.82	9914.19
2010-06-04	9909.24	9962.42	9868.09	9901.19
2010-06-07	9705.36	9714.42	9502.62	9520.80
2010-06-08	9448.79	9587.65	9434.81	9537.94
2010-06-09	9475.57	9487.91	9378.23	9439.13
2010-06-10	9476.78	9544.04	9437.47	9542.65
2010-06-11	9701.32	9764.73	9673.60	9705.25
2010-06-14	9824.93	9882.05	9819.95	9879.85
2010-06-15	9845.65	9911.73	9826.44	9887.89
2010-06-16	10043.10	10109.86	10033.24	10067.15
2010-06-17	10009.25	10053.45	9989.21	9999.40
2010-06-18	10027.98	10029.11	9965.16	9995.02
2010-06-21	10109.27	10251.90	10109.27	10238.01
2010-06-22	10131.91	10183.89	10105.48	10112.89
2010-06-23	9973.00	9975.01	9912.80	9923.70
2010-06-24	9902.62	10009.37	9893.75	9928.34
2010-06-25	9778.70	9794.98	9697.87	9737.48
2010-06-28	9758.91	9762.23	9679.17	9693.94
2010-06-29	9718.40	9760.02	9548.98	9570.67
2010-06-30	9396.45	9420.14	9347.07	9382.64
2010-07-01	9296.86	9297.45	9147.68	9191.60
2010-07-02	9234.56	9259.14	9160.47	9203.71
2010-07-05	9230.00	9282.90	9213.79	9266.78
2010-07-06	9158.28	9351.11	9091.70	9338.04
2010-07-07	9322.56	9329.64	9223.80	9279.65
2010-07-08	9462.82	9545.99	9456.19	9535.74
2010-07-09	9601.62	9610.59	9516.97	9585.32
2010-07-12	9540.47	9632.89	9533.56	9548.11
2010-07-13	9592.87	9629.98	9502.24	9537.23
2010-07-14	9707.50	9807.36	9693.33	9795.24
2010-07-15	9701.14	9726.94	9667.00	9685.53
2010-07-16	9600.88	9636.84	9392.21	9408.36
2010-07-20	9267.24	9378.91	9247.76	9300.46
2010-07-21	9404.07	9404.07	9230.84	9278.83
2010-07-22	9216.28	9254.19	9176.12	9220.88
2010-07-23	9380.12	9474.11	9367.30	9430.96
2010-07-26	9520.73	9561.82	9493.61	9503.66
2010-07-27	9512.67	9541.59	9487.65	9496.85
2010-07-28	9614.47	9760.31	9614.47	9753.27
2010-07-29	9653.51	9732.76	9648.97	9696.02
2010-07-30	9658.61	9658.61	9505.64	9537.30
2010-08-02	9574.64	9676.41	9548.86	9570.31
2010-08-03	9716.57	9750.88	9632.86	9694.01
2010-08-04	9606.82	9613.13	9474.67	9489.34
2010-08-05	9640.34	9688.87	9578.31	9653.92
2010-08-06	9560.65	9658.04	9545.41	9642.12
2010-08-09	9535.51	9572.83	9523.63	9572.49
2010-08-10	9629.97	9667.04	9505.34	9551.05
2010-08-11	9423.03	9445.04	9283.60	9292.85
2010-08-12	9125.49	9212.59	9065.94	9212.59
2010-08-13	9214.81	9278.46	9163.25	9253.46
2010-08-16	9145.22	9210.97	9095.94	9196.67
2010-08-17	9089.85	9188.48	9084.24	9161.68
2010-08-18	9261.50	9280.59	9153.42	9240.54
2010-08-19	9242.84	9362.68	9242.84	9362.68
2010-08-20	9239.36	9287.50	9169.17	9179.38
2010-08-23	9147.24	9171.42	9090.96	9116.69
2010-08-24	9024.60	9069.22	8983.52	8995.14
2010-08-25	8904.44	8949.41	8807.41	8845.39
2010-08-26	8908.02	8910.62	8834.09	8906.48
2010-08-27	8811.47	9021.75	8810.46	8991.06
2010-08-30	9141.45	9280.70	9117.87	9149.26
2010-08-31	8998.80	9008.36	8819.26	8824.06
2010-09-01	8833.32	8932.15	8796.45	8927.02
2010-09-02	9069.62	9083.37	8958.81	9062.84
2010-09-03	9097.59	9141.09	9050.83	9114.13
2010-09-06	9199.99	9303.00	9173.38	9301.32
2010-09-07	9232.02	9311.02	9206.81	9226.00
2010-09-08	9098.86	9105.89	8997.63	9024.60
2010-09-09	9120.54	9136.26	9069.93	9098.39
2010-09-10	9132.25	9291.50	9132.25	9239.17
2010-09-13	9325.64	9390.90	9317.30	9321.82
2010-09-14	9319.50	9330.05	9258.18	9299.31
2010-09-15	9243.48	9578.53	9199.08	9516.56
2010-09-16	9613.89	9620.90	9479.63	9509.50
2010-09-17	9581.65	9643.28	9554.36	9626.09
2010-09-21	9700.88	9704.25	9579.83	9602.11
2010-09-22	9563.20	9625.41	9546.82	9566.32
2010-09-24	9435.04	9601.77	9415.96	9471.67
2010-09-27	9557.37	9613.47	9539.12	9603.14
2010-09-28	9547.03	9553.91	9471.35	9495.76
2010-09-29	9530.05	9600.93	9502.34	9559.38
2010-09-30	9554.83	9566.84	9369.35	9369.35
2010-10-01	9440.52	9470.36	9358.62	9404.23
2010-10-04	9410.92	9508.36	9358.82	9381.06
2010-10-05	9337.43	9538.35	9332.19	9518.76
2010-10-06	9588.47	9693.74	9567.00	9691.43
2010-10-07	9661.09	9716.92	9649.17	9684.81
2010-10-08	9675.59	9675.59	9588.88	9588.88
2010-10-12	9648.59	9650.20	9387.73	9388.64
2010-10-13	9473.78	9510.19	9393.52	9403.51
2010-10-14	9515.17	9614.44	9481.51	9583.51
2010-10-15	9551.21	9553.57	9478.64	9500.25
2010-10-18	9517.68	9564.85	9471.48	9498.49
2010-10-19	9490.09	9575.14	9489.68	9539.45
2010-10-20	9399.65	9402.05	9316.97	9381.60
2010-10-21	9443.95	9479.25	9326.78	9376.48
2010-10-22	9404.09	9457.62	9390.14	9426.71
2010-10-25	9424.44	9460.70	9392.68	9401.16
2010-10-26	9386.55	9455.04	9362.13	9377.38
2010-10-27	9436.01	9453.79	9329.86	9387.03
2010-10-28	9368.61	9413.68	9350.44	9366.03
2010-10-29	9327.25	9337.92	9179.15	9202.45
2010-11-01	9166.85	9258.32	9134.47	9154.72
2010-11-02	9140.28	9172.03	9123.62	9159.98
2010-11-04	9282.79	9377.76	9276.71	9358.78
2010-11-05	9500.60	9680.68	9497.45	9625.99
2010-11-08	9699.79	9737.01	9677.87	9732.92
2010-11-09	9667.44	9726.60	9659.86	9694.49
2010-11-10	9748.61	9842.90	9747.03	9830.52
2010-11-11	9878.11	9885.37	9834.82	9861.46
2010-11-12	9810.30	9851.42	9724.81	9724.81
2010-11-15	9782.11	9830.59	9744.99	9827.51
2010-11-16	9892.82	9908.30	9774.12	9797.10
2010-11-17	9693.21	9817.37	9693.21	9811.66
2010-11-18	9820.60	10013.63	9798.56	10013.63
2010-11-19	10124.84	10130.23	10019.08	10022.39
2010-11-22	10133.48	10157.97	10091.99	10115.19
2010-11-24	9942.54	10064.31	9904.92	10030.11
2010-11-25	10117.85	10123.27	10038.79	10079.76
2010-11-26	10078.09	10134.64	10039.56	10039.56
2010-11-29	10075.66	10149.94	10044.01	10125.99
2010-11-30	10070.31	10101.42	9937.04	9937.04
2010-12-01	9939.80	9988.05	9918.55	9988.05
2010-12-02	10151.15	10187.59	10143.08	10168.52
2010-12-03	10231.78	10254.00	10154.99	10178.32
2010-12-06	10175.23	10179.82	10143.58	10167.23
2010-12-07	10152.92	10167.10	10094.41	10141.10
2010-12-08	10200.16	10258.90	10184.32	10232.33
2010-12-09	10277.87	10298.25	10240.11	10285.88
2010-12-10	10373.70	10373.70	10194.27	10211.95
2010-12-13	10212.52	10293.89	10199.73	10293.89
2010-12-14	10272.70	10319.24	10267.22	10316.77
2010-12-15	10338.14	10340.69	10282.85	10309.78
2010-12-16	10297.55	10347.39	10282.54	10311.29
2010-12-17	10307.17	10327.42	10280.80	10303.83
2010-12-20	10296.55	10301.15	10182.74	10216.41
2010-12-21	10277.14	10372.51	10266.29	10370.53
2010-12-22	10348.48	10394.22	10327.60	10346.48
2010-12-24	10275.97	10290.89	10264.06	10279.19
2010-12-27	10292.84	10376.91	10292.84	10355.99
2010-12-28	10330.01	10340.84	10292.63	10292.63
2010-12-29	10282.84	10344.54	10276.61	10344.54
2010-12-30	10303.13	10315.47	10209.93	10228.92
2011-01-04	10352.19	10409.17	10321.28	10398.10
2011-01-05	10387.95	10413.45	10358.01	10380.77
2011-01-06	10477.52	10530.11	10477.52	10529.76
2011-01-07	10506.72	10550.71	10503.02	10541.04
2011-01-11	10484.62	10538.39	10476.29	10510.68
2011-01-12	10562.28	10576.51	10502.96	10512.80
2011-01-13	10593.46	10620.57	10565.28	10589.76
2011-01-14	10502.77	10579.97	10497.26	10499.04
2011-01-17	10562.09	10562.39	10482.69	10502.86
2011-01-18	10461.04	10548.21	10456.71	10518.98
2011-01-19	10567.88	10580.68	10534.02	10557.10
2011-01-20	10492.32	10505.63	10421.83	10437.31
2011-01-21	10459.04	10461.07	10257.99	10274.52
2011-01-24	10318.60	10347.94	10278.81	10345.11
2011-01-25	10374.64	10480.24	10355.80	10464.42
2011-01-26	10410.29	10440.35	10392.46	10401.90
2011-01-27	10466.34	10496.40	10421.75	10478.66
2011-01-28	10460.29	10480.08	10332.65	10360.34
2011-01-31	10219.98	10265.96	10182.57	10237.92
2011-02-01	10281.55	10299.38	10245.75	10274.50
2011-02-02	10368.48	10479.75	10366.96	10457.36
2011-02-03	10424.68	10450.46	10409.84	10431.36
2011-02-04	10526.39	10580.36	10524.13	10543.52
2011-02-07	10641.30	10644.21	10572.72	10592.04
2011-02-08	10645.03	10648.80	10623.70	10635.98
2011-02-09	10686.72	10701.92	10599.50	10617.83
2011-02-10	10574.80	10631.94	10570.62	10605.65
2011-02-14	10689.65	10725.54	10674.44	10725.54
2011-02-15	10738.81	10760.99	10712.49	10746.67
2011-02-16	10758.86	10842.31	10752.61	10808.29
2011-02-17	10882.91	10891.60	10820.48	10836.64
2011-02-18	10829.15	10862.43	10810.12	10842.80
2011-02-21	10820.78	10859.39	10789.41	10857.53
2011-02-22	10745.52	10761.22	10639.78	10664.70
2011-02-23	10588.82	10666.72	10569.93	10579.10
2011-02-24	10533.20	10542.70	10428.38	10452.71
2011-02-25	10475.78	10532.21	10454.46	10526.76
2011-02-28	10504.54	10628.76	10448.83	10624.09
2011-03-01	10676.24	10754.03	10672.91	10754.03
2011-03-02	10609.28	10629.50	10492.38	10492.38
2011-03-03	10529.77	10593.34	10522.40	10586.02
2011-03-04	10730.91	10768.43	10664.37	10693.66
2011-03-07	10626.97	10626.97	10472.51	10505.02
2011-03-08	10513.08	10565.10	10509.23	10525.19
2011-03-09	10607.23	10662.07	10563.82	10589.50
2011-03-10	10544.13	10549.17	10410.10	10434.38
2011-03-11	10298.64	10378.55	10254.43	10254.43
2011-03-14	10044.17	10049.92	9578.65	9620.49
2011-03-15	9441.66	9441.66	8227.63	8605.15
2011-03-16	8767.20	9168.51	8763.95	9093.72
2011-03-17	8913.35	9093.61	8639.56	8962.67
2011-03-18	9083.95	9275.60	9073.25	9206.75
2011-03-22	9403.05	9625.03	9395.85	9608.32
2011-03-23	9590.38	9592.40	9387.95	9449.47
2011-03-24	9479.77	9509.85	9418.71	9435.01
2011-03-25	9565.49	9569.71	9469.37	9536.13
2011-03-28	9541.76	9541.76	9405.20	9478.53
2011-03-29	9348.60	9502.23	9317.38	9459.08
2011-03-30	9489.51	9708.79	9479.03	9708.79
2011-03-31	9765.28	9765.81	9658.64	9755.10
2011-04-01	9757.28	9822.06	9698.60	9708.39
2011-04-04	9773.91	9808.60	9718.89	9718.89
2011-04-05	9732.26	9732.26	9566.05	9615.55
2011-04-06	9668.11	9671.46	9562.16	9584.37
2011-04-07	9652.65	9687.18	9578.47	9590.93
2011-04-08	9586.17	9804.28	9536.68	9768.08
2011-04-11	9731.32	9775.60	9700.75	9719.70
2011-04-12	9603.50	9603.64	9513.27	9555.26
2011-04-13	9516.50	9655.83	9516.50	9641.18
2011-04-14	9579.79	9676.97	9553.75	9653.92
2011-04-15	9646.22	9651.96	9577.96	9591.52
2011-04-18	9593.41	9608.92	9539.03	9556.65
2011-04-19	9448.65	9478.87	9405.19	9441.03
2011-04-20	9536.79	9631.00	9519.36	9606.82
2011-04-21	9686.27	9725.12	9643.92	9685.77
2011-04-22	9623.00	9732.63	9611.38	9682.21
2011-04-25	9705.90	9748.40	9657.24	9671.96
2011-04-26	9627.43	9627.43	9536.51	9558.69
2011-04-27	9631.53	9728.03	9630.37	9691.84
2011-04-28	9749.36	9849.74	9712.58	9849.74
2011-05-02	9964.39	10017.47	9936.27	10004.20
2011-05-06	9851.15	9876.10	9782.66	9859.20
2011-05-09	9881.67	9897.88	9776.74	9794.38
2011-05-10	9804.27	9853.13	9755.93	9818.76
2011-05-11	9912.48	9929.04	9842.67	9864.26
2011-05-12	9770.71	9836.71	9716.65	9716.65
2011-05-13	9751.45	9751.45	9552.93	9648.77
2011-05-16	9571.13	9598.33	9552.86	9558.30
2011-05-17	9513.50	9593.13	9502.39	9567.02
2011-05-18	9602.88	9693.60	9590.40	9662.08
2011-05-19	9711.32	9731.12	9600.81	9620.82
2011-05-20	9612.87	9670.92	9600.48	9607.08
2011-05-23	9547.86	9547.86	9437.87	9460.63
2011-05-24	9406.04	9490.35	9406.04	9477.17
2011-05-25	9492.43	9503.77	9415.33	9422.88
2011-05-26	9499.45	9567.46	9489.12	9562.05
2011-05-27	9514.16	9588.70	9493.44	9521.94
2011-05-30	9489.78	9540.02	9448.00	9504.97
2011-05-31	9500.60	9697.35	9497.40	9693.73
2011-06-01	9708.05	9720.73	9659.31	9719.61
2011-06-02	9560.63	9575.01	9517.28	9555.04
2011-06-03	9550.59	9603.03	9490.68	9492.21
2011-06-06	9467.37	9490.55	9359.78	9380.35
2011-06-07	9371.21	9454.66	9358.31	9442.95
2011-06-08	9428.62	9458.43	9394.74	9449.46
2011-06-09	9411.71	9472.41	9383.12	9467.15
2011-06-10	9536.84	9613.75	9493.05	9514.44
2011-06-13	9405.16	9466.51	9391.26	9448.21
2011-06-14	9446.84	9565.65	9438.41	9547.79
2011-06-15	9589.54	9599.90	9520.93	9574.32
2011-06-16	9478.63	9497.59	9411.28	9411.28
2011-06-17	9443.32	9447.44	9318.62	9351.40
2011-06-20	9383.24	9421.38	9346.76	9354.32
2011-06-21	9420.43	9459.66	9378.36	9459.66
2011-06-22	9525.25	9658.62	9525.25	9629.43
2011-06-23	9557.72	9652.41	9553.30	9596.74
2011-06-24	9626.43	9695.09	9609.24	9678.71
2011-06-27	9633.92	9638.49	9570.58	9578.31
2011-06-28	9668.01	9703.41	9632.54	9648.98
2011-06-29	9767.91	9797.26	9734.66	9797.26
2011-06-30	9838.09	9849.69	9785.89	9816.09
2011-07-01	9878.69	9900.57	9849.64	9868.07
2011-07-04	9980.16	10005.75	9940.47	9965.09
2011-07-05	9959.39	9995.17	9948.38	9972.46
2011-07-06	9987.25	10082.48	9967.87	10082.48
2011-07-07	10047.15	10102.19	10029.33	10071.14
2011-07-08	10205.71	10207.91	10126.47	10137.73
2011-07-11	10069.01	10109.30	10057.77	10069.53
2011-07-12	9943.32	9971.14	9907.81	9925.92
2011-07-13	9891.11	9979.67	9887.33	9963.14
2011-07-14	9929.18	9998.49	9884.00	9936.12
2011-07-15	9919.59	9985.32	9919.10	9974.47
2011-07-19	9921.50	9945.72	9889.72	9889.72
2011-07-20	10008.00	10042.75	9986.33	10005.90
2011-07-21	10019.94	10026.18	9974.00	10010.39
2011-07-22	10096.89	10149.18	10076.53	10132.11
2011-07-25	10080.82	10091.02	10039.10	10050.01
2011-07-26	10078.48	10130.25	10043.99	10097.72
2011-07-27	10048.34	10053.50	10009.37	10047.19
2011-07-28	9936.97	9945.98	9853.85	9901.35
2011-07-29	9866.82	9914.22	9824.34	9833.03
2011-08-01	9907.04	10040.13	9906.17	9965.01
2011-08-02	9872.17	9873.46	9822.30	9844.59
2011-08-03	9705.02	9705.02	9610.38	9637.14
2011-08-04	9685.50	9768.40	9615.35	9659.18
2011-08-05	9469.16	9469.16	9264.09	9299.88
2011-08-08	9169.67	9215.09	9057.29	9097.56
2011-08-09	8911.80	8949.97	8656.79	8944.48
2011-08-10	9110.81	9144.33	9021.36	9038.74
2011-08-11	8876.54	8983.33	8832.42	8981.94
2011-08-12	9064.17	9070.27	8926.89	8963.72
2011-08-15	9082.53	9117.11	9032.84	9086.41
2011-08-16	9135.06	9150.31	9072.53	9107.43
2011-08-17	9047.89	9081.80	9003.70	9057.26
2011-08-18	9043.18	9043.18	8931.25	8943.76
2011-08-19	8771.70	8796.42	8707.47	8719.24
2011-08-22	8686.91	8756.10	8619.21	8628.13
2011-08-23	8703.31	8747.41	8630.49	8733.01
2011-08-24	8812.16	8825.27	8620.89	8639.61
2011-08-25	8748.01	8849.94	8741.74	8772.36
2011-08-26	8746.77	8805.10	8742.46	8797.78
2011-08-29	8802.25	8926.27	8751.29	8851.35
2011-08-30	8958.50	8992.86	8935.85	8953.90
2011-08-31	8938.31	8967.80	8905.52	8955.20
2011-09-01	9017.01	9098.15	8998.01	9060.80
2011-09-02	8980.56	9014.27	8914.65	8950.74
2011-09-05	8828.46	8842.56	8757.18	8784.46
2011-09-06	8687.11	8709.92	8588.34	8590.57
2011-09-07	8723.05	8773.26	8702.78	8763.41
2011-09-08	8863.24	8876.49	8757.58	8793.12
2011-09-09	8739.49	8803.75	8726.25	8737.66
2011-09-12	8578.71	8584.05	8520.97	8535.67
2011-09-13	8588.74	8635.88	8549.06	8616.55
2011-09-14	8623.55	8671.24	8499.34	8518.57
2011-09-15	8645.38	8695.95	8634.02	8668.86
2011-09-16	8785.28	8864.16	8774.16	8864.16
2011-09-20	8763.61	8771.72	8704.23	8721.24
2011-09-21	8717.00	8771.62	8707.16	8741.16
2011-09-22	8643.02	8643.02	8545.46	8560.26
2011-09-26	8547.62	8556.66	8359.70	8374.13
2011-09-27	8496.33	8609.95	8479.05	8609.95
2011-09-28	8607.92	8665.54	8595.85	8615.65
2011-09-29	8527.64	8706.14	8501.42	8701.23
2011-09-30	8713.91	8756.00	8653.62	8700.29
2011-10-03	8567.98	8577.52	8455.67	8545.48
2011-10-04	8426.39	8470.76	8359.24	8456.12
2011-10-05	8491.43	8501.03	8343.01	8382.98
2011-10-06	8463.83	8545.79	8463.83	8522.02
2011-10-07	8593.22	8663.57	8589.36	8605.62
2011-10-11	8733.54	8806.44	8725.62	8773.68
2011-10-12	8719.67	8763.02	8689.38	8738.90
2011-10-13	8832.81	8854.16	8808.73	8823.25
2011-10-14	8782.91	8787.89	8733.71	8747.96
2011-10-17	8881.44	8911.70	8856.58	8879.60
2011-10-18	8758.74	8773.99	8727.39	8741.91
2011-10-19	8808.25	8831.55	8745.36	8772.54
2011-10-20	8733.60	8733.60	8672.21	8682.15
2011-10-21	8679.04	8700.49	8652.26	8678.89
2011-10-24	8764.92	8844.28	8762.02	8843.98
2011-10-25	8866.43	8870.37	8755.38	8762.31
2011-10-26	8673.85	8798.80	8642.56	8748.47
2011-10-27	8779.40	8926.54	8731.77	8926.54
2011-10-28	9059.04	9086.43	9003.09	9050.47
2011-10-31	9011.18	9152.39	8988.39	8988.39
2011-11-01	8880.75	8946.00	8831.69	8835.52
2011-11-02	8719.92	8719.92	8640.42	8640.42
2011-11-04	8761.27	8814.71	8729.08	8801.40
2011-11-07	8771.82	8779.55	8741.25	8767.09
2011-11-08	8744.36	8772.04	8644.00	8655.51
2011-11-09	8724.60	8762.98	8678.78	8755.44
2011-11-10	8597.28	8600.21	8500.67	8500.80
2011-11-11	8545.92	8566.49	8485.77	8514.47
2011-11-14	8631.28	8655.31	8595.43	8603.70
2011-11-15	8546.99	8584.30	8527.63	8541.93
2011-11-16	8546.75	8567.64	8459.30	8463.16
2011-11-17	8421.88	8491.82	8400.22	8479.63
2011-11-18	8374.16	8398.80	8359.60	8374.91
2011-11-21	8349.05	8375.60	8343.93	8348.27
2011-11-22	8263.86	8350.35	8261.01	8314.74
2011-11-24	8189.42	8212.95	8157.39	8165.18
2011-11-25	8138.19	8199.67	8135.79	8160.01
2011-11-28	8269.91	8322.01	8259.71	8287.49
2011-11-29	8371.01	8477.82	8351.38	8477.82
2011-11-30	8407.63	8435.34	8361.69	8434.61
2011-12-01	8581.20	8653.88	8577.09	8597.38
2011-12-02	8603.23	8643.75	8591.93	8643.75
2011-12-05	8697.78	8704.48	8668.04	8695.98
2011-12-06	8644.97	8671.54	8571.09	8575.16
2011-12-07	8629.19	8729.81	8614.21	8722.17
2011-12-08	8664.35	8688.81	8625.19	8664.58
2011-12-09	8521.33	8576.68	8503.03	8536.46
2011-12-12	8652.98	8682.47	8633.35	8653.82
2011-12-13	8535.52	8594.32	8518.80	8552.81
2011-12-14	8513.77	8540.57	8486.37	8519.13
2011-12-15	8449.05	8455.32	8374.25	8377.37
2011-12-16	8416.21	8433.18	8390.98	8401.72
2011-12-19	8363.39	8364.93	8272.26	8296.12
2011-12-20	8318.03	8354.90	8317.73	8336.48
2011-12-21	8442.74	8471.11	8440.89	8459.98
2011-12-22	8429.54	8436.59	8383.96	8395.16
2011-12-26	8504.46	8516.53	8477.37	8479.34
2011-12-27	8443.00	8458.26	8428.53	8440.56
2011-12-28	8426.09	8456.66	8414.90	8423.62
2011-12-29	8367.71	8404.89	8330.87	8398.89
2011-12-30	8434.90	8455.35	8415.56	8455.35
2012-01-04	8549.54	8581.45	8547.70	8560.11
2012-01-05	8515.66	8519.16	8481.83	8488.71
2012-01-06	8488.98	8488.98	8349.33	8390.35
2012-01-10	8422.99	8450.59	8405.18	8422.26
2012-01-11	8440.96	8463.72	8426.03	8447.88
2012-01-12	8423.10	8426.83	8360.33	8385.59
2012-01-13	8471.10	8509.76	8458.68	8500.02
2012-01-16	8409.79	8409.79	8352.23	8378.36
2012-01-17	8420.12	8475.66	8413.22	8466.40
2012-01-18	8458.29	8595.78	8446.09	8550.58
2012-01-19	8596.68	8668.94	8596.68	8639.68
2012-01-20	8751.18	8791.39	8725.32	8766.36
2012-01-23	8753.91	8795.27	8744.54	8765.90
2012-01-24	8815.36	8825.09	8768.51	8785.33
2012-01-25	8842.01	8911.62	8816.09	8883.69
2012-01-26	8890.49	8894.60	8834.93	8849.47
2012-01-27	8851.02	8886.02	8810.89	8841.22
2012-01-30	8803.79	8832.48	8774.23	8793.05
2012-01-31	8781.44	8836.68	8776.65	8802.51
2012-02-01	8789.06	8830.28	8780.10	8809.79
2012-02-02	8865.28	8893.22	8849.25	8876.82
2012-02-03	8849.17	8877.57	8825.98	8831.93
2012-02-06	8939.99	8949.32	8916.94	8929.20
2012-02-07	8904.16	8928.44	8887.19	8917.52
2012-02-08	8971.88	9015.59	8956.78	9015.59
2012-02-09	8996.13	9018.49	8942.87	9002.24
2012-02-10	9010.53	9016.97	8947.17	8947.17
2012-02-13	8954.03	9023.50	8948.54	8999.18
2012-02-14	8978.72	9072.08	8972.74	9052.07
2012-02-15	9108.84	9314.37	9107.68	9260.34
2012-02-16	9232.38	9308.93	9214.23	9238.10
2012-02-17	9371.12	9435.03	9369.25	9384.17
2012-02-20	9534.03	9549.31	9462.67	9485.09
2012-02-21	9458.68	9517.04	9440.41	9463.02
2012-02-22	9459.74	9564.05	9442.58	9554.00
2012-02-23	9549.77	9609.84	9513.65	9595.57
2012-02-24	9594.85	9647.38	9576.21	9647.38
2012-02-27	9726.20	9736.11	9628.25	9633.93
2012-02-28	9567.12	9722.52	9528.77	9722.52
2012-02-29	9771.62	9866.41	9706.22	9723.24
2012-03-01	9771.34	9865.75	9666.02	9707.37
2012-03-02	9797.05	9803.75	9729.24	9777.03
2012-03-05	9756.98	9792.14	9673.75	9698.59
2012-03-06	9704.61	9732.89	9602.84	9637.63
2012-03-07	9509.10	9603.19	9509.10	9576.06
2012-03-08	9674.94	9768.96	9659.15	9768.96
2012-03-09	9911.07	10007.62	9853.18	9929.74
2012-03-12	10015.92	10021.51	9889.86	9889.86
2012-03-13	9921.25	10011.88	9888.30	9899.08
2012-03-14	10064.12	10115.79	10050.52	10050.52
2012-03-15	10115.40	10158.74	10077.55	10123.28
2012-03-16	10110.57	10148.47	10090.09	10129.83
2012-03-19	10150.85	10172.64	10134.48	10141.99
2012-03-21	10100.87	10132.02	10073.90	10086.49
2012-03-22	10055.16	10136.85	10052.50	10127.08
2012-03-23	10014.87	10031.96	9999.37	10011.47
2012-03-26	10040.10	10056.20	10016.05	10018.24
2012-03-27	10152.90	10255.15	10152.25	10255.15
2012-03-28	10153.51	10197.80	10124.28	10182.57
2012-03-29	10134.78	10147.04	10084.77	10114.79
2012-03-30	10089.10	10111.09	10033.19	10083.56
2012-04-02	10161.72	10190.35	10109.87	10109.87
2012-04-03	10082.21	10084.07	10040.99	10050.39
2012-04-04	10045.88	10059.19	9819.99	9819.99
2012-04-05	9737.50	9806.40	9692.70	9767.61
2012-04-06	9710.25	9739.59	9659.16	9688.45
2012-04-09	9565.31	9618.10	9535.33	9546.26
2012-04-10	9583.36	9648.12	9509.72	9538.02
2012-04-11	9398.60	9487.84	9388.14	9458.74
2012-04-12	9486.19	9541.04	9449.49	9524.79
2012-04-13	9613.19	9690.29	9603.97	9637.99
2012-04-16	9522.70	9532.91	9464.49	9470.64
2012-04-17	9476.15	9519.80	9455.32	9464.71
2012-04-18	9594.30	9682.88	9594.30	9667.26
2012-04-19	9578.68	9617.93	9565.83	9588.38
2012-04-20	9574.99	9580.87	9532.68	9561.36
2012-04-23	9599.20	9643.27	9520.98	9542.17
2012-04-24	9466.08	9510.70	9423.30	9468.04
cut4
2009-01-05	48.49	48.66	48.12	48.19
2009-01-06	48.25	48.87	48.11	48.78
2009-01-07	48.78	48.84	47.06	47.24
2009-01-08	47.15	47.18	46.52	46.60
2009-01-09	46.67	46.86	46.52	46.82
2009-01-13	46.62	46.70	46.23	46.40
2009-01-14	46.45	46.57	45.82	45.82
2009-01-15	45.67	45.67	45.34	45.48
2009-01-16	45.53	45.65	45.46	45.52
2009-01-19	45.65	45.69	45.38	45.52
2009-01-20	45.48	45.53	44.62	44.87
2009-01-21	44.77	44.77	44.55	44.62
2009-01-22	44.61	44.87	44.40	44.87
2009-01-23	44.83	44.93	44.61	44.80
2009-01-26	44.79	44.88	44.60	44.66
2009-01-27	44.75	44.90	44.69	44.88
2009-01-28	44.94	44.94	44.07	44.43
2009-01-29	44.47	44.55	43.71	43.88
2009-01-30	43.87	44.04	43.53	43.95
2009-02-02	43.79	43.83	43.35	43.54
2009-02-03	43.57	43.76	43.40	43.66
2009-02-04	43.54	43.55	43.28	43.48
2009-02-05	43.43	43.55	43.01	43.08
2009-02-06	43.07	43.10	42.52	42.64
2009-02-09	42.66	42.70	42.15	42.21
2009-02-10	42.17	42.19	41.77	42.02
2009-02-12	41.81	41.81	41.26	41.28
2009-02-13	41.35	42.29	41.35	42.27
2009-02-16	42.37	42.57	42.13	42.40
2009-02-17	42.24	42.24	41.69	41.71
2009-02-18	41.60	42.02	41.57	42.01
2009-02-19	41.96	42.12	41.72	41.98
2009-02-20	41.84	41.84	40.49	40.62
2009-02-23	40.39	40.39	39.69	39.84
2009-02-24	39.53	39.72	39.07	39.46
2009-02-25	39.70	40.39	39.70	40.38
2009-02-26	40.42	40.83	40.42	40.82
2009-02-27	40.71	41.79	40.55	41.72
2009-03-02	41.61	41.61	41.20	41.31
2009-03-03	41.12	41.12	39.78	39.91
2009-03-04	39.75	39.81	39.61	39.78
2009-03-05	39.86	40.14	39.77	39.80
2009-03-06	39.62	39.63	39.47	39.51
2009-03-09	39.47	39.54	38.31	38.31
2009-03-10	38.27	38.54	38.27	38.45
2009-03-11	38.67	38.86	38.46	38.49
2009-03-12	38.44	38.44	37.59	37.66
2009-03-13	37.93	38.34	37.93	38.22
2009-03-16	38.34	38.93	38.32	38.75
2009-03-17	38.65	39.04	38.44	39.04
2009-03-18	39.32	39.88	38.63	38.79
2009-03-19	38.88	39.43	38.87	39.43
2009-03-23	39.47	40.47	39.47	40.40
2009-03-24	40.60	40.84	40.50	40.84
2009-03-25	40.82	41.12	40.73	41.12
2009-03-26	40.74	40.91	40.53	40.88
2009-03-27	40.98	41.45	40.98	41.33
2009-03-30	41.38	41.56	41.10	41.14
2009-03-31	41.03	41.15	40.60	41.04
2009-04-01	41.24	41.39	40.59	40.72
2009-04-02	40.79	41.11	40.71	41.08
2009-04-03	41.32	41.33	40.93	41.01
2009-04-06	41.05	41.28	40.89	40.98
2009-04-07	40.96	41.31	40.88	41.01
2009-04-08	40.94	41.29	40.87	41.29
2009-04-09	41.21	41.75	41.20	41.75
2009-04-10	41.88	41.91	41.63	41.72
2009-04-13	41.75	41.77	41.36	41.38
2009-04-14	41.29	41.32	41.05	41.19
2009-04-15	41.13	41.22	40.99	41.19
2009-04-16	41.24	41.42	41.23	41.35
2009-04-17	41.41	41.54	41.34	41.42
2009-04-20	41.44	41.47	41.12	41.24
2009-04-21	41.00	41.00	40.47	40.64
2009-04-22	40.69	41.01	40.69	40.97
2009-04-23	40.92	41.60	40.69	41.49
2009-04-24	41.52	41.59	41.41	41.53
2009-04-27	41.63	41.88	41.63	41.83
2009-04-28	41.81	41.91	41.43	41.47
2009-04-30	41.61	41.74	40.86	40.86
2009-05-01	40.91	40.92	40.38	40.59
2009-05-07	40.73	40.97	40.60	40.67
2009-05-08	40.74	41.51	40.74	41.51
2009-05-11	41.73	41.88	41.58	41.67
2009-05-12	41.64	41.83	41.61	41.68
2009-05-13	41.73	42.01	41.73	42.01
2009-05-14	41.89	42.03	41.80	41.95
2009-05-15	42.05	42.17	42.04	42.17
2009-05-18	42.10	42.11	41.90	41.97
2009-05-19	42.09	42.21	41.98	42.06
2009-05-20	42.12	42.36	42.11	42.36
2009-05-21	42.33	42.50	42.31	42.47
2009-05-22	42.26	42.47	42.18	42.47
2009-05-25	42.57	42.97	42.57	42.97
2009-05-26	42.98	43.00	42.85	42.94
2009-05-27	43.07	43.53	43.07	43.44
2009-05-28	43.35	43.62	43.35	43.62
2009-05-29	43.75	43.90	43.64	43.79
2009-06-01	43.87	43.94	43.67	43.69
2009-06-02	43.84	43.91	43.61	43.78
2009-06-03	43.80	44.11	43.80	44.05
2009-06-04	43.97	44.12	43.89	44.09
2009-06-05	44.25	44.85	44.24	44.77
2009-06-08	44.83	45.00	44.80	44.93
2009-06-09	44.98	45.07	44.66	44.71
2009-06-10	44.77	45.17	44.77	45.14
2009-06-11	45.19	45.75	45.16	45.75
2009-06-12	45.92	47.13	45.90	47.13
2009-06-15	47.36	48.02	47.35	47.82
2009-06-16	47.72	47.77	47.28	47.29
2009-06-17	47.22	47.29	46.88	47.22
2009-06-18	47.29	47.39	47.13	47.38
2009-06-19	47.43	47.90	47.43	47.90
2009-06-22	47.92	47.93	47.61	47.61
2009-06-23	47.37	47.37	46.98	46.98
2009-06-24	47.02	47.23	46.73	46.99
2009-06-25	47.05	47.42	46.89	47.41
2009-06-26	47.55	47.68	47.44	47.68
2009-06-29	47.84	48.10	47.47	47.47
2009-06-30	47.73	48.58	47.73	48.54
2009-07-01	48.60	48.87	48.60	48.86
2009-07-02	48.94	49.02	48.52	48.71
2009-07-03	48.54	48.85	48.46	48.84
2009-07-06	48.94	49.48	48.93	49.48
2009-07-07	49.61	49.81	49.23	49.30
2009-07-08	49.20	49.20	48.60	48.92
2009-07-09	48.71	48.71	48.05	48.09
2009-07-10	48.09	48.19	47.66	47.78
2009-07-13	47.56	47.64	46.14	46.22
2009-07-14	46.40	46.80	46.40	46.80
2009-07-15	46.75	46.94	46.67	46.94
2009-07-16	47.13	47.35	47.03	47.03
2009-07-17	47.17	48.13	47.06	48.11
2009-07-21	48.31	48.62	48.31	48.62
2009-07-22	48.55	48.85	48.53	48.71
2009-07-23	48.75	48.99	48.74	48.88
2009-07-24	49.04	49.28	48.73	49.28
2009-07-27	49.48	49.66	49.44	49.51
2009-07-28	49.57	49.63	49.20	49.60
2009-07-29	49.50	49.89	49.48	49.89
2009-07-30	49.98	49.98	49.46	49.79
2009-07-31	49.90	50.09	49.35	49.44
2009-08-03	49.66	49.82	49.60	49.78
2009-08-04	50.00	50.10	49.79	49.87
2009-08-05	49.89	49.91	49.64	49.68
2009-08-06	49.64	49.81	49.59	49.80
2009-08-07	49.82	49.82	49.52	49.70
2009-08-10	50.01	50.12	49.99	50.08
2009-08-11	50.08	50.34	49.96	50.34
2009-08-12	50.27	50.40	49.97	50.39
2009-08-13	50.48	50.65	50.28	50.37
2009-08-14	50.34	50.47	50.30	50.47
2009-08-17	50.43	50.43	49.93	50.01
2009-08-18	49.85	49.92	49.69	49.87
2009-08-19	49.95	50.11	49.47	49.53
2009-08-20	49.60	49.62	49.48	49.62
2009-08-21	49.66	49.72	49.31	49.39
2009-08-24	49.53	49.55	49.40	49.55
2009-08-25	49.65	49.80	49.63	49.73
2009-08-26	49.72	49.75	49.67	49.75
2009-08-27	49.74	49.79	49.48	49.79
2009-08-28	49.82	50.15	49.80	50.00
2009-08-31	49.99	50.49	49.83	50.49
2009-09-01	50.40	50.92	50.35	50.92
2009-09-02	50.78	50.82	50.65	50.82
2009-09-03	50.77	50.84	50.61	50.69
2009-09-04	50.72	50.77	49.80	49.80
2009-09-07	49.90	50.08	49.68	49.83
2009-09-08	49.77	49.77	49.42	49.56
2009-09-09	49.56	49.63	49.49	49.63
2009-09-10	49.61	50.26	49.61	50.26
2009-09-11	50.33	50.35	50.13	50.32
2009-09-14	50.30	50.30	50.15	50.29
2009-09-15	50.26	50.38	50.10	50.38
2009-09-16	50.29	50.34	50.05	50.24
2009-09-17	50.33	50.33	49.98	50.11
2009-09-18	50.04	50.05	49.65	49.93
2009-09-24	49.99	50.16	49.89	49.96
2009-09-25	49.78	49.80	49.52	49.74
2009-09-28	49.57	49.57	49.24	49.42
2009-09-29	49.47	49.55	49.35	49.46
2009-09-30	49.49	49.72	49.45	49.72
2009-10-01	49.72	49.84	49.46	49.50
2009-10-02	49.06	49.06	48.50	48.61
2009-10-05	48.60	48.60	48.15	48.34
2009-10-06	48.42	48.47	48.08	48.28
2009-10-07	48.33	48.56	48.12	48.56
2009-10-08	48.48	48.48	48.25	48.35
2009-10-09	48.34	49.30	48.34	49.28
2009-10-13	49.36	49.41	49.25	49.29
2009-10-14	49.27	49.27	48.90	48.95
2009-10-15	49.09	49.13	48.86	48.90
2009-10-16	48.87	48.87	48.68	48.79
2009-10-19	48.71	48.97	48.65	48.90
2009-10-20	48.96	49.05	48.78	48.93
2009-10-21	48.96	49.29	48.89	49.25
2009-10-22	49.20	49.32	49.13	49.30
2009-10-23	49.35	49.40	48.98	49.01
2009-10-26	49.03	49.11	49.02	49.07
2009-10-27	49.00	49.08	48.88	48.91
2009-10-28	48.98	48.98	48.17	48.27
2009-10-29	48.05	48.11	47.79	48.10
2009-10-30	48.25	48.59	48.25	48.59
2009-11-02	48.39	48.40	48.06	48.38
2009-11-04	48.43	48.52	48.23	48.33
2009-11-05	48.30	48.30	47.81	47.94
2009-11-06	48.02	48.05	47.60	47.81
2009-11-09	47.71	47.71	47.21	47.31
2009-11-10	47.44	47.66	47.34	47.34
2009-11-11	47.39	47.49	47.11	47.32
2009-11-12	47.35	47.40	47.08	47.25
2009-11-13	47.14	47.20	46.96	47.14
2009-11-16	46.94	47.02	46.05	46.12
2009-11-17	46.12	46.15	45.20	45.24
2009-11-18	45.27	45.34	44.63	44.79
2009-11-19	44.80	44.92	44.60	44.80
2009-11-20	44.67	45.35	44.57	45.35
2009-11-24	45.37	45.46	45.08	45.10
2009-11-25	45.18	45.23	45.01	45.16
2009-11-26	45.10	45.43	45.06	45.33
2009-11-27	45.14	45.14	44.79	45.06
2009-11-30	45.16	45.73	45.16	45.73
2009-12-01	45.70	45.85	45.60	45.85
2009-12-02	45.96	46.08	45.73	45.78
2009-12-03	45.84	46.18	45.83	46.14
2009-12-04	46.20	46.43	46.00	46.39
2009-12-07	46.59	46.76	46.56	46.70
2009-12-08	46.64	47.36	46.64	47.33
2009-12-09	47.21	47.60	47.19	47.50
2009-12-10	47.52	47.67	47.33	47.49
2009-12-11	47.51	47.55	47.20	47.52
2009-12-14	47.48	47.52	47.07	47.08
2009-12-15	47.15	47.37	47.10	47.37
2009-12-16	47.33	47.38	47.23	47.32
2009-12-17	47.39	47.41	47.04	47.17
2009-12-18	47.18	47.37	46.99	47.35
2009-12-21	47.27	47.50	47.26	47.44
2009-12-22	47.55	47.76	47.49	47.69
2009-12-24	47.69	48.22	47.60	48.18
2009-12-25	48.19	48.32	48.01	48.11
2009-12-28	48.01	48.20	47.97	48.14
2009-12-29	48.23	48.51	48.22	48.51
2009-12-30	48.49	48.52	48.26	48.36
2010-01-04	48.54	48.60	48.33	48.47
2010-01-05	48.53	48.76	48.50	48.54
2010-01-06	48.57	48.67	48.49	48.67
2010-01-07	48.64	48.90	48.54	48.59
2010-01-08	48.62	48.82	48.41	48.58
2010-01-12	48.63	48.94	48.63	48.88
2010-01-13	48.83	49.01	48.83	48.90
2010-01-14	48.96	48.97	48.79	48.82
2010-01-15	48.87	49.40	48.83	49.38
2010-01-18	49.29	49.37	49.07	49.37
2010-01-19	49.28	49.31	49.17	49.18
2010-01-20	49.29	49.61	49.20	49.61
2010-01-21	49.50	50.18	49.49	50.18
2010-01-22	49.99	49.99	49.79	49.93
2010-01-25	49.84	50.92	49.84	50.89
2010-01-26	50.87	50.92	50.12	50.14
2010-01-27	50.18	50.38	50.01	50.19
2010-01-28	50.34	51.14	50.34	51.14
2010-01-29	51.29	51.60	51.29	51.52
2010-02-01	51.32	51.35	51.11	51.17
2010-02-02	51.38	51.59	51.08	51.54
2010-02-03	51.65	51.65	51.35	51.45
2010-02-04	51.39	51.45	50.82	50.92
2010-02-05	50.56	50.56	50.12	50.22
2010-02-08	50.15	50.15	49.40	49.42
2010-02-09	49.34	49.66	49.26	49.56
2010-02-10	49.69	49.85	49.69	49.77
2010-02-12	49.91	50.00	49.79	50.00
2010-02-15	49.91	49.96	49.46	49.50
2010-02-16	49.61	50.15	49.59	50.08
2010-02-17	50.15	50.29	50.10	50.19
2010-02-18	50.22	50.27	50.14	50.21
2010-02-19	50.15	50.18	49.93	49.98
2010-02-22	50.01	50.13	49.97	50.08
2010-02-23	50.09	50.26	50.06	50.26
2010-02-24	50.20	50.27	50.13	50.27
2010-02-25	50.38	50.46	50.30	50.36
2010-02-26	50.34	50.50	50.33	50.47
2010-03-01	50.50	51.16	50.48	51.06
2010-03-02	50.85	51.14	50.67	50.81
2010-03-03	50.81	50.86	50.70	50.79
2010-03-04	50.86	50.94	50.84	50.84
2010-03-05	50.95	51.66	50.69	51.49
2010-03-08	51.59	52.01	51.51	51.63
2010-03-09	51.86	52.31	51.84	52.30
2010-03-10	52.03	52.28	51.87	51.92
2010-03-11	51.84	51.88	51.71	51.80
2010-03-12	52.08	52.32	52.01	52.26
2010-03-15	52.29	52.31	52.15	52.26
2010-03-16	52.23	52.38	52.23	52.34
2010-03-17	52.46	52.62	52.44	52.62
2010-03-18	52.66	52.71	52.58	52.64
2010-03-19	52.67	52.88	52.65	52.88
2010-03-23	53.00	53.35	52.99	53.35
2010-03-24	53.46	53.49	53.35	53.48
2010-03-25	53.48	53.59	53.08	53.58
2010-03-26	53.41	53.43	52.96	53.01
2010-03-29	52.60	52.92	52.58	52.88
2010-03-30	52.99	53.47	52.94	53.47
2010-03-31	53.31	53.56	53.13	53.20
2010-04-01	53.25	53.61	53.19	53.52
2010-04-02	53.64	53.82	53.61	53.71
2010-04-05	53.79	53.97	53.79	53.95
2010-04-06	53.99	54.56	53.68	54.48
2010-04-07	54.60	54.64	54.24	54.38
2010-04-08	54.32	54.72	54.31	54.72
2010-04-09	54.74	54.90	54.73	54.90
2010-04-12	55.03	55.09	54.97	55.07
2010-04-13	55.08	55.09	54.44	54.65
2010-04-14	54.58	54.58	54.23	54.30
2010-04-15	54.31	54.55	54.03	54.55
2010-04-16	54.58	54.72	54.51	54.66
2010-04-19	54.12	54.31	53.79	54.28
2010-04-20	54.39	54.48	54.10	54.11
2010-04-21	54.43	54.61	54.27	54.42
2010-04-22	54.49	54.86	54.40	54.85
2010-04-23	54.95	55.71	54.94	55.71
2010-04-26	55.92	56.47	55.92	56.47
2010-04-27	56.52	57.08	56.37	57.06
2010-04-28	56.43	56.43	56.02	56.40
2010-04-30	56.60	56.62	56.04	56.15
2010-05-06	55.62	55.70	55.18	55.29
2010-05-07	54.48	54.48	53.47	53.99
2010-05-10	53.83	54.31	53.82	53.97
2010-05-11	54.28	55.03	54.09	54.09
2010-05-12	54.15	54.34	53.66	53.82
2010-05-13	54.06	54.23	53.97	54.23
2010-05-14	54.03	54.03	53.88	54.02
2010-05-17	53.61	53.61	52.92	53.05
2010-05-18	53.05	53.10	52.41	52.61
2010-05-19	52.14	52.14	51.66	52.05
2010-05-20	51.86	52.01	51.42	51.70
2010-05-21	50.85	50.85	49.80	50.04
2010-05-24	50.14	50.29	49.86	50.11
2010-05-25	50.04	50.04	49.12	49.23
2010-05-26	49.29	49.36	49.01	49.33
2010-05-27	48.93	49.90	48.86	49.90
2010-05-28	50.40	50.62	50.08	50.26
2010-05-31	50.24	50.75	50.23	50.68
2010-06-01	50.93	51.54	50.93	51.48
2010-06-02	51.02	51.06	50.57	50.60
2010-06-03	50.84	50.95	50.79	50.95
2010-06-04	51.20	51.61	51.20	51.42
2010-06-07	51.00	51.00	50.69	50.73
2010-06-08	50.38	51.06	50.32	51.06
2010-06-09	51.07	51.13	50.79	50.91
2010-06-10	50.94	51.21	50.87	51.17
2010-06-11	51.61	51.77	51.33	51.65
2010-06-14	51.69	51.85	51.69	51.85
2010-06-15	51.87	51.88	51.66	51.66
2010-06-16	51.82	51.87	51.72	51.83
2010-06-17	51.76	51.88	51.65	51.79
2010-06-18	51.87	51.87	51.52	51.74
2010-06-21	51.94	52.49	51.93	52.21
2010-06-22	52.15	52.54	52.14	52.46
2010-06-23	52.27	52.27	52.17	52.22
2010-06-24	52.20	52.23	52.13	52.19
2010-06-25	52.08	52.09	51.77	51.97
2010-06-28	51.95	51.96	51.43	51.46
2010-06-29	51.48	51.52	50.95	51.00
2010-06-30	50.63	51.01	50.43	51.01
2010-07-01	50.88	50.90	50.49	50.66
2010-07-02	50.71	50.95	50.66	50.95
2010-07-05	50.98	51.36	50.97	51.16
2010-07-06	51.02	51.67	50.91	51.67
2010-07-07	51.71	51.71	51.42	51.50
2010-07-08	51.70	51.82	51.66	51.75
2010-07-09	51.83	51.96	51.76	51.96
2010-07-12	51.72	51.92	51.44	51.50
2010-07-13	51.54	51.57	51.34	51.42
2010-07-14	51.49	51.60	51.40	51.47
2010-07-15	51.42	51.44	50.72	50.72
2010-07-16	50.71	50.80	50.29	50.40
2010-07-20	50.32	50.32	50.20	50.30
2010-07-21	50.30	50.34	50.07	50.14
2010-07-22	50.01	50.02	49.85	49.99
2010-07-23	50.15	50.30	49.95	50.02
2010-07-26	50.08	50.26	50.08	50.17
2010-07-27	50.42	51.11	50.40	50.93
2010-07-28	50.96	51.08	50.95	51.07
2010-07-29	51.02	51.15	50.80	51.14
2010-07-30	51.13	51.14	50.85	50.96
2010-08-02	50.88	50.98	50.11	50.13
2010-08-03	50.20	50.27	50.02	50.13
2010-08-04	50.06	50.10	49.88	49.96
2010-08-05	50.03	50.17	49.36	49.45
2010-08-06	49.25	49.25	49.01	49.14
2010-08-09	49.03	49.07	48.95	49.07
2010-08-10	49.13	49.14	48.94	49.02
2010-08-11	48.98	48.99	48.74	48.80
2010-08-12	48.23	48.38	47.89	48.11
2010-08-13	48.10	48.44	48.07	48.44
2010-08-16	48.32	48.45	48.24	48.45
2010-08-17	48.13	48.41	48.12	48.38
2010-08-18	48.49	48.51	48.18	48.25
2010-08-19	48.28	48.44	48.24	48.44
2010-08-20	48.35	48.43	48.28	48.29
2010-08-23	48.29	48.32	47.92	48.03
2010-08-24	47.95	48.04	47.78	47.99
2010-08-25	47.83	47.90	47.61	47.71
2010-08-26	47.78	48.35	47.76	48.22
2010-08-27	48.12	48.30	48.09	48.26
2010-08-30	48.41	48.69	48.41	48.50
2010-08-31	48.41	48.41	48.20	48.32
2010-09-01	48.19	48.19	48.04	48.06
2010-09-02	48.14	48.15	47.77	47.98
2010-09-03	47.88	47.96	47.81	47.88
2010-09-06	47.98	48.21	47.89	48.18
2010-09-07	48.13	48.15	48.01	48.10
2010-09-08	47.95	47.95	47.85	47.90
2010-09-09	48.02	48.19	47.72	47.91
2010-09-10	47.92	48.24	47.91	48.19
2010-09-13	48.24	48.42	48.24	48.35
2010-09-14	48.41	48.54	48.38	48.42
2010-09-15	48.35	48.46	48.28	48.37
2010-09-16	48.42	48.46	48.17	48.22
2010-09-17	48.20	48.27	48.16	48.25
2010-09-21	48.39	48.43	48.34	48.40
2010-09-22	48.28	48.45	48.17	48.19
2010-09-24	48.06	48.34	48.05	48.28
2010-09-27	48.35	48.36	47.94	48.21
2010-09-28	48.13	48.20	48.06	48.18
2010-09-29	48.22	48.39	48.20	48.35
2010-09-30	48.23	48.27	47.87	47.87
2010-10-01	47.93	47.93	47.81	47.92
2010-10-04	47.91	47.91	47.62	47.62
2010-10-05	47.56	47.59	47.41	47.49
2010-10-06	47.55	47.67	47.51	47.67
2010-10-07	47.55	47.80	47.55	47.78
2010-10-08	47.76	47.86	47.75	47.81
2010-10-12	48.08	48.11	47.77	47.82
2010-10-13	47.79	47.84	47.30	47.32
2010-10-14	47.49	47.88	47.49	47.70
2010-10-15	47.80	47.99	47.53	47.58
2010-10-18	47.56	47.60	47.51	47.60
2010-10-19	47.51	47.52	47.39	47.46
2010-10-20	47.36	47.36	46.90	46.95
2010-10-21	46.95	46.95	46.66	46.72
2010-10-22	46.65	46.67	46.54	46.64
2010-10-25	46.67	47.09	46.62	46.66
2010-10-26	46.66	46.87	46.64	46.83
2010-10-27	46.89	46.95	46.83	46.91
2010-10-28	46.91	46.92	46.78	46.82
2010-10-29	46.85	46.92	46.73	46.92
2010-11-01	46.87	46.87	46.68	46.75
2010-11-02	46.72	46.74	46.50	46.52
2010-11-04	46.60	46.63	46.35	46.43
2010-11-05	46.46	46.67	46.46	46.67
2010-11-08	46.75	47.02	46.75	47.01
2010-11-09	47.12	47.34	47.12	47.34
2010-11-10	47.27	47.46	47.27	47.37
2010-11-11	47.43	47.51	47.41	47.51
2010-11-12	47.60	47.60	47.40	47.58
2010-11-15	47.61	47.68	47.52	47.68
2010-11-16	47.66	47.66	47.49	47.55
2010-11-17	47.39	47.61	47.37	47.61
2010-11-18	47.67	47.96	47.67	47.96
2010-11-19	48.12	48.27	48.12	48.19
2010-11-22	48.36	48.63	48.35	48.60
2010-11-24	48.36	48.90	48.29	48.89
2010-11-25	49.06	49.31	48.99	49.04
2010-11-26	49.23	49.28	48.85	48.92
2010-11-29	49.15	49.24	48.99	49.06
2010-11-30	49.10	49.16	48.95	49.08
2010-12-01	49.06	49.41	49.05	49.41
2010-12-02	49.74	50.04	49.74	49.91
2010-12-03	50.28	50.55	50.28	50.55
2010-12-06	50.64	50.85	50.64	50.85
2010-12-07	50.95	50.96	50.76	50.87
2010-12-08	50.92	51.10	50.92	51.03
2010-12-09	51.06	51.15	50.89	51.15
2010-12-10	51.13	51.13	51.02	51.06
2010-12-13	51.18	51.49	51.16	51.49
2010-12-14	51.50	51.59	51.29	51.46
2010-12-15	51.59	51.70	51.59	51.67
2010-12-16	51.72	52.10	51.72	51.77
2010-12-17	52.03	52.37	52.03	52.37
2010-12-20	52.47	52.57	52.30	52.36
2010-12-21	52.36	52.42	52.21	52.23
2010-12-22	52.30	52.35	52.00	52.07
2010-12-24	52.09	52.40	52.08	52.40
2010-12-27	52.41	52.51	52.04	52.09
2010-12-28	52.07	52.14	52.00	52.13
2010-12-29	52.13	52.29	52.09	52.29
2010-12-30	52.13	52.30	51.96	52.30
2011-01-04	52.46	52.63	52.46	52.63
2011-01-05	52.57	52.59	52.20	52.20
2011-01-06	52.51	52.68	52.31	52.49
2011-01-07	52.53	52.54	52.40	52.54
2011-01-11	52.57	52.89	52.56	52.89
2011-01-12	53.00	53.12	52.94	53.04
2011-01-13	53.19	53.33	53.13	53.32
2011-01-14	53.31	53.66	53.28	53.33
2011-01-17	53.67	53.70	53.47	53.59
2011-01-18	53.64	53.97	53.62	53.78
2011-01-19	53.89	54.39	53.88	54.39
2011-01-20	54.29	54.30	53.96	54.05
2011-01-21	54.00	54.00	53.07	53.24
2011-01-24	53.26	53.63	53.22	53.63
2011-01-25	53.83	54.01	53.83	54.00
2011-01-26	53.99	54.48	53.92	54.39
2011-01-27	54.44	54.46	54.35	54.46
2011-01-28	54.44	54.44	54.10	54.26
2011-01-31	53.96	54.10	53.85	54.03
2011-02-01	54.14	54.41	54.14	54.41
2011-02-02	54.61	54.91	54.61	54.91
2011-02-03	54.92	54.94	54.23	54.32
2011-02-04	54.31	54.37	54.09	54.18
2011-02-07	54.36	54.61	54.28	54.60
2011-02-08	54.72	54.80	54.62	54.80
2011-02-09	54.83	54.87	54.66	54.68
2011-02-10	54.67	54.92	54.61	54.92
2011-02-14	55.05	55.32	55.05	55.31
2011-02-15	55.12	55.14	54.96	55.14
2011-02-16	55.02	55.08	54.90	55.07
2011-02-17	55.16	55.23	55.09	55.14
2011-02-18	55.17	55.27	55.12	55.27
2011-02-21	55.35	55.54	55.32	55.51
2011-02-22	55.45	55.46	54.96	55.12
2011-02-23	54.75	55.04	54.68	54.85
2011-02-24	54.61	54.61	53.97	54.12
2011-02-25	54.02	54.23	53.96	54.23
2011-02-28	54.47	54.96	54.40	54.96
2011-03-01	55.11	55.38	55.11	55.38
2011-03-02	55.16	55.89	55.09	55.57
2011-03-03	55.74	55.88	55.73	55.83
2011-03-04	56.11	56.21	55.94	56.14
2011-03-07	56.13	56.16	55.82	55.95
2011-03-08	56.09	56.15	55.99	56.04
2011-03-09	56.36	56.42	55.85	55.95
2011-03-10	55.96	55.98	55.31	55.47
2011-03-11	55.02	55.33	54.98	55.15
2011-03-14	54.02	54.02	49.08	49.68
2011-03-15	49.20	49.20	44.28	45.46
2011-03-16	45.50	47.65	45.50	47.42
2011-03-17	46.85	48.07	46.11	47.89
2011-03-18	48.28	49.69	48.28	49.69
2011-03-22	50.24	51.32	50.24	51.32
2011-03-23	51.45	51.83	51.30	51.60
2011-03-24	51.72	51.93	51.33	51.78
2011-03-25	52.03	52.10	51.65	51.76
2011-03-28	51.77	51.77	51.24	51.29
2011-03-29	50.71	50.85	50.45	50.82
2011-03-30	50.91	51.15	50.87	51.13
2011-03-31	51.34	51.40	51.16	51.29
2011-04-01	51.33	51.41	51.11	51.11
2011-04-04	51.22	51.28	51.07	51.10
2011-04-05	51.12	51.12	50.53	50.59
2011-04-06	50.50	50.50	50.00	50.32
2011-04-07	50.34	50.43	50.13	50.38
2011-04-08	50.23	50.56	50.19	50.56
2011-04-11	50.57	50.78	50.56	50.78
2011-04-12	50.61	50.65	49.76	49.84
2011-04-13	49.64	50.17	49.63	50.17
2011-04-14	50.03	50.34	50.03	50.34
2011-04-15	50.37	50.51	50.28	50.47
2011-04-18	50.56	50.63	50.51	50.54
2011-04-19	50.22	50.27	50.08	50.24
2011-04-20	50.39	50.57	50.37	50.54
2011-04-21	50.67	50.93	50.56	50.87
2011-04-22	50.81	51.06	50.76	51.04
2011-04-25	51.13	51.56	51.13	51.50
2011-04-26	51.53	51.57	51.34	51.39
2011-04-27	51.44	51.51	51.28	51.40
2011-04-28	51.47	51.89	51.45	51.89
2011-05-02	51.93	52.32	51.93	52.32
2011-05-06	52.13	52.24	52.05	52.24
2011-05-09	52.33	52.41	52.10	52.18
2011-05-10	52.23	52.31	52.17	52.30
2011-05-11	52.40	52.46	52.32	52.38
2011-05-12	52.15	52.30	52.13	52.21
2011-05-13	52.33	52.34	51.97	52.15
2011-05-16	51.93	51.93	51.12	51.13
2011-05-17	50.95	50.95	50.73	50.84
2011-05-18	50.91	51.23	50.81	51.20
2011-05-19	51.40	51.71	51.38	51.45
2011-05-20	51.46	51.82	51.45	51.82
2011-05-23	51.69	51.70	51.23	51.28
2011-05-24	51.09	51.32	51.07	51.23
2011-05-25	51.33	51.40	51.08	51.11
2011-05-26	51.20	51.38	51.15	51.38
2011-05-27	51.31	51.38	51.22	51.36
2011-05-30	51.40	51.56	51.38	51.54
2011-05-31	51.54	51.89	51.52	51.89
2011-06-01	51.87	51.97	51.83	51.94
2011-06-02	51.65	51.68	51.56	51.66
2011-06-03	51.60	51.86	51.48	51.60
2011-06-06	51.56	51.64	51.39	51.50
2011-06-07	51.46	51.89	51.46	51.89
2011-06-08	51.83	51.86	51.50	51.70
2011-06-09	51.60	51.66	51.48	51.58
2011-06-10	51.63	51.74	51.55	51.74
2011-06-13	51.57	51.73	51.55	51.61
2011-06-14	51.69	51.75	51.54	51.59
2011-06-15	51.67	51.67	51.44	51.54
2011-06-16	51.43	51.54	51.31	51.37
2011-06-17	51.50	51.57	51.41	51.52
2011-06-20	51.58	51.60	51.28	51.28
2011-06-21	51.35	51.61	51.35	51.61
2011-06-22	51.67	51.84	51.67	51.75
2011-06-23	51.67	51.70	51.48	51.54
2011-06-24	51.40	51.42	51.26	51.41
2011-06-27	51.48	51.57	51.47	51.48
2011-06-28	51.57	51.65	51.30	51.39
2011-06-29	51.52	51.97	51.49	51.97
2011-06-30	52.00	52.11	51.89	52.11
2011-07-01	52.11	52.11	51.72	51.85
2011-07-04	52.16	52.18	51.97	52.06
2011-07-05	52.10	52.10	52.01	52.07
2011-07-06	52.12	52.37	52.09	52.37
2011-07-07	52.27	52.41	52.24	52.38
2011-07-08	52.49	52.66	52.49	52.50
2011-07-11	52.48	52.61	52.48	52.53
2011-07-12	52.38	52.38	52.14	52.24
2011-07-13	52.10	52.39	52.10	52.35
2011-07-14	52.44	52.53	52.38	52.53
2011-07-15	52.49	52.50	52.39	52.43
2011-07-19	52.41	52.53	52.38	52.53
2011-07-20	52.84	52.91	52.59	52.66
2011-07-21	52.86	53.12	52.68	53.08
2011-07-22	53.15	53.39	53.14	53.35
2011-07-25	53.41	53.42	53.27	53.32
2011-07-26	53.35	53.36	52.57	52.66
2011-07-27	52.57	52.57	52.40	52.46
2011-07-28	52.30	52.36	52.11	52.12
2011-07-29	52.07	52.13	51.84	51.93
2011-08-01	51.94	52.36	51.94	52.27
2011-08-02	52.18	52.18	51.99	52.10
2011-08-03	51.78	51.79	51.61	51.79
2011-08-04	51.84	51.96	51.77	51.77
2011-08-05	51.21	51.30	50.58	51.30
2011-08-08	51.00	51.01	49.75	50.33
2011-08-09	49.65	49.92	48.58	49.92
2011-08-10	50.28	50.63	49.45	49.47
2011-08-11	49.03	49.48	48.84	49.48
2011-08-12	49.70	49.93	49.58	49.73
2011-08-15	49.88	50.15	49.88	50.13
2011-08-16	50.26	50.35	50.15	50.19
2011-08-17	50.21	50.43	50.21	50.42
2011-08-18	50.45	50.59	50.39	50.41
2011-08-19	50.03	50.18	49.18	49.24
2011-08-22	49.22	49.34	48.53	48.58
2011-08-23	48.59	48.62	48.23	48.38
2011-08-24	48.49	48.52	47.98	48.00
2011-08-25	48.21	48.49	48.20	48.47
2011-08-26	48.51	49.00	48.49	48.80
2011-08-29	49.14	49.43	49.00	49.26
2011-08-30	49.48	49.89	49.48	49.53
2011-08-31	49.86	50.44	49.85	50.44
2011-09-01	50.49	50.64	50.34	50.42
2011-09-02	50.33	50.33	50.08	50.13
2011-09-05	49.85	50.14	49.83	50.14
2011-09-06	49.95	49.97	49.16	49.16
2011-09-07	49.33	49.45	49.27	49.42
2011-09-08	49.73	49.96	49.70	49.91
2011-09-09	49.56	49.99	49.55	49.59
2011-09-12	49.69	49.69	49.45	49.45
2011-09-13	49.39	49.39	49.12	49.16
2011-09-14	49.40	49.47	48.96	48.96
2011-09-15	48.99	49.06	48.87	48.87
2011-09-16	49.10	49.36	49.03	49.33
2011-09-20	49.30	49.34	49.12	49.22
2011-09-21	49.04	49.29	49.04	49.24
2011-09-22	48.77	48.87	48.40	48.47
2011-09-26	48.29	48.29	47.00	47.06
2011-09-27	47.66	48.02	47.58	47.93
2011-09-28	48.20	48.26	48.02	48.08
2011-09-29	48.06	48.59	48.03	48.59
2011-09-30	48.75	48.76	48.57	48.69
2011-10-03	48.59	48.59	48.23	48.32
2011-10-04	48.04	48.34	47.83	48.33
2011-10-05	48.46	48.52	48.12	48.21
2011-10-06	48.27	48.71	48.26	48.45
2011-10-07	48.71	48.99	48.60	48.67
2011-10-11	49.02	49.26	48.92	48.97
2011-10-12	49.23	49.26	49.16	49.20
2011-10-13	49.18	49.40	49.16	49.38
2011-10-14	49.19	49.24	49.07	49.11
2011-10-17	48.86	48.89	48.69	48.69
2011-10-18	48.50	48.55	48.35	48.51
2011-10-19	48.48	48.54	48.34	48.44
2011-10-20	48.38	48.39	48.12	48.17
2011-10-21	48.07	48.10	47.90	48.05
2011-10-24	48.14	48.22	48.10	48.22
2011-10-25	48.48	48.55	48.43	48.43
2011-10-26	48.35	48.37	48.24	48.28
2011-10-27	48.20	48.43	48.20	48.39
2011-10-28	48.58	48.62	48.31	48.33
2011-10-31	48.53	48.74	48.50	48.65
2011-11-01	48.61	48.62	48.38	48.51
2011-11-02	48.35	48.35	48.00	48.07
2011-11-04	48.25	48.42	48.24	48.37
2011-11-07	48.48	48.65	48.34	48.41
2011-11-08	48.34	48.43	48.07	48.13
2011-11-09	47.91	47.98	47.76	47.78
2011-11-10	47.86	47.86	47.67	47.67
2011-11-11	47.58	47.62	47.46	47.49
2011-11-14	47.45	47.69	47.45	47.69
2011-11-15	47.56	47.56	47.27	47.27
2011-11-16	47.29	47.31	46.96	47.01
2011-11-17	46.72	46.81	46.67	46.81
2011-11-18	46.64	46.64	46.48	46.55
2011-11-21	46.56	46.58	46.26	46.26
2011-11-22	46.07	46.30	45.92	46.30
2011-11-24	46.02	46.02	45.88	45.96
2011-11-25	45.72	45.85	45.64	45.64
2011-11-28	45.57	45.81	45.57	45.69
2011-11-29	45.93	46.15	45.93	46.13
2011-11-30	46.42	46.80	46.42	46.65
2011-12-01	47.33	47.55	47.33	47.49
2011-12-02	47.61	47.74	47.56	47.74
2011-12-05	47.95	48.22	47.95	48.22
2011-12-06	48.18	48.21	47.77	47.88
2011-12-07	47.74	47.92	47.69	47.92
2011-12-08	48.11	48.21	47.83	47.93
2011-12-09	47.91	47.95	47.85	47.88
2011-12-12	48.12	48.31	48.12	48.21
2011-12-13	48.11	48.28	48.03	48.28
2011-12-14	48.21	48.23	47.81	47.91
2011-12-15	47.73	47.89	47.69	47.69
2011-12-16	47.63	47.71	47.52	47.61
2011-12-19	47.54	47.56	47.29	47.48
2011-12-20	47.38	47.70	47.38	47.70
2011-12-21	47.77	47.90	47.77	47.88
2011-12-22	47.91	48.05	47.80	47.83
2011-12-26	48.11	48.14	47.89	47.93
2011-12-27	47.85	47.92	47.72	47.72
2011-12-28	47.64	47.90	47.63	47.81
2011-12-29	47.77	47.87	47.72	47.86
2011-12-30	48.02	48.12	47.92	48.12
2012-01-04	48.21	48.33	48.21	48.33
2012-01-05	48.54	48.58	48.36	48.39
2012-01-06	48.28	48.29	47.97	48.08
2012-01-10	48.16	48.20	48.02	48.11
2012-01-11	48.23	48.29	48.15	48.25
2012-01-12	48.12	48.12	47.73	47.82
2012-01-13	47.65	47.79	47.57	47.74
2012-01-16	47.51	47.51	47.34	47.37
2012-01-17	47.34	47.34	47.16	47.29
2012-01-18	47.36	47.69	47.30	47.69
2012-01-19	47.77	47.80	47.66	47.67
2012-01-20	48.03	48.16	47.98	48.14
2012-01-23	48.14	48.25	48.13	48.20
2012-01-24	48.37	48.40	48.21	48.30
2012-01-25	48.34	48.61	48.30	48.46
2012-01-26	48.63	48.65	48.53	48.64
2012-01-27	48.32	48.32	47.79	47.81
2012-01-30	47.87	47.91	47.65	47.74
2012-01-31	47.70	47.89	47.68	47.87
2012-02-01	47.93	48.04	47.86	48.02
2012-02-02	48.24	48.57	48.24	48.52
2012-02-03	48.50	48.62	48.50	48.54
2012-02-06	48.88	48.92	48.75	48.88
2012-02-07	48.83	48.91	48.68	48.91
2012-02-08	49.02	49.19	48.99	49.19
2012-02-09	49.17	49.56	49.17	49.52
2012-02-10	49.63	50.01	49.63	50.01
2012-02-13	49.98	50.14	49.92	50.14
2012-02-14	49.98	50.04	49.80	49.94
2012-02-15	50.16	50.28	50.05	50.28
2012-02-16	49.82	50.00	49.76	49.97
2012-02-17	49.90	49.99	49.64	49.74
2012-02-20	49.77	49.77	49.32	49.43
2012-02-21	49.55	49.88	49.55	49.78
2012-02-22	49.90	50.41	49.90	50.41
2012-02-23	50.50	50.75	50.49	50.75
2012-02-24	50.78	51.03	50.78	51.03
2012-02-27	51.10	51.21	51.05	51.17
2012-02-28	50.96	51.03	50.79	51.02
2012-02-29	51.25	51.37	51.20	51.26
2012-03-01	51.25	51.28	50.84	50.86
2012-03-02	50.83	51.17	50.80	50.92
2012-03-05	51.45	51.70	51.44	51.70
2012-03-06	51.63	51.72	51.54	51.61
2012-03-07	51.16	51.22	51.03	51.20
2012-03-08	51.28	51.58	51.28	51.58
2012-03-09	51.78	51.93	51.74	51.93
2012-03-12	52.04	52.40	52.03	52.14
2012-03-13	52.49	52.74	52.48	52.70
2012-03-14	52.97	53.26	52.91	53.26
2012-03-15	53.37	53.46	53.28	53.34
2012-03-16	53.34	53.53	53.30	53.40
2012-03-19	53.37	53.48	53.37	53.45
2012-03-21	53.51	53.74	53.51	53.74
2012-03-22	53.59	53.68	53.57	53.68
2012-03-23	53.52	53.64	53.45	53.59
2012-03-26	53.63	53.70	53.23	53.25
2012-03-27	53.27	53.31	53.17	53.31
2012-03-28	52.88	53.05	52.84	53.05
2012-03-29	53.08	53.38	52.92	53.11
2012-03-30	53.43	53.68	53.39	53.68
2012-04-02	53.59	54.00	53.59	53.92
2012-04-03	53.90	53.91	53.69	53.77
2012-04-04	53.45	53.60	53.12	53.22
2012-04-05	53.02	53.19	52.88	53.17
2012-04-06	53.14	53.20	53.07	53.19
2012-04-09	53.03	53.07	52.99	53.01
2012-04-10	52.93	53.01	52.75	52.75
2012-04-11	52.57	52.63	52.42	52.63
2012-04-12	52.64	52.74	52.63	52.74
2012-04-13	52.91	53.08	52.90	53.08
2012-04-16	53.08	53.27	53.01	53.16
2012-04-17	53.15	53.30	53.15	53.28
2012-04-18	53.49	53.70	53.46	53.70
2012-04-19	53.49	53.55	53.40	53.48
2012-04-20	53.42	53.45	53.35	53.44
2012-04-23	53.37	53.41	53.18	53.25
2012-04-24	53.24	53.33	53.20	53.29
