package OpenInteract2::SQLInstall::User;

# $Id: User.pm,v 1.4 2003/07/08 12:16:46 lachoy Exp $

use strict;
use base qw( OpenInteract2::SQLInstall );
use Log::Log4perl            qw( get_logger );
use OpenInteract2::Constants qw( :log );
use OpenInteract2::Context   qw( CTX );
use SPOPS::Utility;

my @TABLES = ( 'sys_user.sql' );

my %FILES = (
    oracle => [ 'sys_user_oracle.sql', 'sys_user_sequence.sql' ],
    pg     => [ 'sys_user.sql', 'sys_user_sequence.sql' ],
    ib     => [ 'sys_user_interbase.sql', 'sys_user_generator.sql' ],
);

sub get_structure_set {
    return 'user';
}

sub get_structure_file {
    my ( $self, $set, $type ) = @_;
    if ( $type eq 'Oracle' ) {
        return $FILES{oracle};
    }
    elsif ( $type eq 'Pg' ) {
        return $FILES{pg};
    }
    elsif ( $type eq 'InterBase' ) {
        return $FILES{ib};
    }
    else {
        return [ @TABLES ];
    }
}

sub get_security_file {
    return 'install_security.dat';
}

# Create the admin user and give him (or her) a random password --
# users should change the password using oi2_manage

sub install_data {
    my ( $self ) = @_;
    my $log = get_logger( LOG_INIT );

    my $action_name = 'create administrator';
    my $server_config = CTX->server_config;
    my $email = $server_config->{mail}{admin_email};
    my $id    = $server_config->{default_objects}{superuser};
    my $user = CTX->lookup_object( 'user' )
                  ->new({ email      => $email,
                          login_name => 'superuser',
                          first_name => 'Super',
                          last_name  => 'User',
                          user_id    => $id });
    my $password = SPOPS::Utility->generate_random_code(8);
    if ( $server_config->{login}{crypt_password} ) {
        $user->{password} = SPOPS::Utility->crypt_it( $password );
    }
    eval { $user->save({ is_add        => 1,
                         skip_security => 1,
                         skip_cache    => 1,
                         skip_log      => 1 }) };
    if ( $@ ) {
        $log->error( "Failed to create superuser: $@" );
        $self->_set_state( $action_name,
                           undef,
                           "Failed to create admin user: $@",
                           undef );
    }
    else {
        my $msg_ok = 'Created administrator ok.';
        $self->_set_state( $action_name, 1, $msg_ok, undef );
    }

    # If we needed to process any data files in addition to the
    # above, we could do:
    # $self->SUPER::install_data();
}

1;

__END__

=head1 NAME

OpenInteract2::SQLInstall::User - SQL installer for the base_user package

=head1 SYNOPSIS

 $ oi2_manage install_sql --package=base_user

=head1 DESCRIPTION

This SQLInstall subclass is a little different than many of the other
handlers in OpenInteract -- you B<must> track information that it
generates during the SQL installation process.

We do not want to ship OpenInteract with either a blank or otherwise
known superuser password. And we do not want to force users to type it
in during installation -- doing as much as possible to allow automated
installs is a good thing.

So when we install the SQL we create the superuser with a random
password and echo the password back to you.

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 COPYRIGHT

Copyright (c) 2001-2003 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters E<lt>chris@cwinters.comE<gt>
