/***[cond.c]******************************************************[TAB=4]****\
*                                                                            *
* PHP/FI                                                                     *
*                                                                            *
* Copyright 1995,1996 Rasmus Lerdorf                                         *
*                                                                            *
*  This program is free software; you can redistribute it and/or modify      *
*  it under the terms of the GNU General Public License as published by      *
*  the Free Software Foundation; either version 2 of the License, or         *
*  (at your option) any later version.                                       *
*                                                                            *
*  This program is distributed in the hope that it will be useful,           *
*  but WITHOUT ANY WARRANTY; without even the implied warranty of            *
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
*  GNU General Public License for more details.                              *
*                                                                            *
*  You should have received a copy of the GNU General Public License         *
*  along with this program; if not, write to the Free Software               *
*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                 *
*                                                                            *
\****************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <php.h>
#include <parse.h>

int iflevel=0;
int ifstate=1;

static CondStack *top=NULL;
static MatchStack *mtop=NULL;

void MatchPush(int val) {
	MatchStack *new;

	new = malloc(sizeof(MatchStack));
	new->val = val;
	new->next = mtop;
	mtop = new;	
}

int MatchPop(void) {
	MatchStack *s;
	int ret;

	if(!mtop) {
		Error("if/else/endif error");
		return(1);
	}
	ret = mtop->val;
	s = mtop;
	mtop = mtop->next;
	free(s);
	return(ret);
}	

void ShowCondStack(void) {
	CondStack *s;

	s = top;
	while(s) {
#if DEBUG
		Debug("(%d,%d) ",s->state, s->active);
#endif
		s = s->next;
	}
#if DEBUG
	Debug("\n");
#endif
}

void CondPush(int state, int active) {
	CondStack *new;

	new = malloc(sizeof(CondStack));
	new->state = state;
	new->active = active;
	new->next=top;
	top=new;
}

int CondPop(int *active) {
	int ret=0;
	CondStack *s;

	if(top) ret = top->state;
	else return(ret);
	if(active) *active = top->active;
	s = top;
	top=top->next;
	free(s);
	return(ret);
}

int GetCurrentState(int *active) {
	int ret;

	if(!top) { 
		ret=1; 
		if(active) *active=-1; 
	} else { 
		ret=top->state; 
		if(active) *active=top->active; 
	}
	return(ret);
}

int GetCurrentActive(void) {
	if(!top) return(0);
	else return(top->active);
}

void ClearCondStack(void) {
	CondStack *s;

	s=top;
	while(s) {
		top=s->next;
		free(s);
		s=top;
	}
}

int Logic(int lop) {
	Stack *s, a, b;
	int ret=0;
	char temp[32];

	a.strval=NULL;
	b.strval=NULL;
	s=Pop();		
	if(!s) {
		Error("Stack Error");
		return(0);
	}
	b.type = s->type;
	b.intval = s->intval;
	b.douval = s->douval;
	if(s->strval) b.strval = strdup(s->strval);
 
	s = Pop();
	if(!s) {
		Error("Stack Error");
		return(0);
	}
	a.type = s->type;
	a.intval = s->intval;
	a.douval = s->douval;
	if(s->strval) a.strval = strdup(s->strval);
	
	switch(a.type) {
		case LNUMBER:
			if(lop==LOG_OR) ret = (a.intval || b.intval);
			else ret = (a.intval && b.intval);
			break;

		case DNUMBER:
			if(lop==LOG_OR) ret = (a.douval || b.douval);
			else ret = (a.douval && b.douval);
			break;

		case STRING:
			if(lop==LOG_OR) ret = (strlen(a.strval) || strlen(b.strval));
			else ret = (strlen(a.strval) && strlen(b.strval));
			break;
	}
	if(a.strval) free(a.strval);
	if(b.strval) free(b.strval);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
	return(ret);
}
			
int Compare(int cop) {
	Stack *s, a, b;
	int ret=0;
	char temp[32];
	
	a.strval = NULL;
	b.strval = NULL;
	s=Pop();
	if(!s) {
		Error("Stack Error");
		return(0);
	}
	b.type = s->type;
	b.intval = s->intval;
	b.douval = s->douval;
	if(s->strval) b.strval = strdup(s->strval);
 
	s = Pop();
	if(!s) {
		Error("Stack Error");
		return(0);
	}
	a.type = s->type;
	a.intval = s->intval;
	a.douval = s->douval;
	if(s->strval) a.strval = strdup(s->strval);

	switch(a.type) {
		case LNUMBER:
			switch(cop) {
			case COND_EQ:
				ret = (a.intval == b.intval);
				break;

			case COND_NE:
				ret = (a.intval != b.intval);
				break;
			
			case COND_GT:
				ret = (a.intval > b.intval);
				break;

			case COND_GE:
				ret = (a.intval >= b.intval);
				break;

			case COND_LT:
				ret = (a.intval < b.intval);
				break;

			case COND_LE:
				ret = (a.intval <= b.intval);
				break;
			}
			break;

		case DNUMBER:
			switch(cop) {
			case COND_EQ:
				ret = (a.douval == b.douval);
				break;

			case COND_NE:
				ret = (a.douval != b.douval);
				break;
			
			case COND_GT:
				ret = (a.douval > b.douval);
				break;

			case COND_GE:
				ret = (a.douval >= b.douval);
				break;

			case COND_LT:
				ret = (a.douval < b.douval);
				break;

			case COND_LE:
				ret = (a.douval <= b.douval);
				break;
			}
			break;

		case STRING:
			switch(cop) {
			case COND_EQ:
				ret = (strcmp(a.strval,b.strval) == 0);
				break;

			case COND_NE:
				ret = (strcmp(a.strval,b.strval) != 0);
				break;
			
			case COND_GT:
				ret = (strcmp(a.strval,b.strval) > 0);
				break;

			case COND_GE:
				ret = (strcmp(a.strval,b.strval) >= 0);
				break;

			case COND_LT:
				ret = (strcmp(a.strval,b.strval) < 0);
				break;

			case COND_LE:
				ret = (strcmp(a.strval,b.strval) <= 0);
				break;
			}
			break;
	}	
	if(a.strval) free(a.strval);
	if(b.strval) free(b.strval);
	sprintf(temp,"%d",ret);
	Push(temp,LNUMBER);
	return(ret);
}

void Not(void) {
	Stack *s;
	char temp[128];

	s = Pop();
	if(!s) {
		Error("Stack Error");
		return;
	}
	switch(s->type) {
		case LNUMBER:
			sprintf(temp,"%d",!s->intval);
			break;
		case DNUMBER:
			sprintf(temp,"%d",!s->douval);
			break;
		case STRING:
			sprintf(temp,"%d",!strlen(s->strval));
			break;
	}
	Push(temp,LNUMBER);
}

/* 
 * This returns the conditonal value to be pushed onto the
 * stack by conditional expressions
 */
int CheckCond(Stack *s) {
	if(s->var) {
		switch(s->var->type) {
		case LNUMBER:
		case DNUMBER:
			return(s->intval);
			break;
		case STRING:
			return(strlen(s->strval));
			break;
		}
	} else {
		switch(s->type) {
		case LNUMBER:
		case DNUMBER:
			return(s->intval);
			break;
		case STRING:
			return(strlen(s->strval));
			break;
		}
	}
	return(s->intval);
}

void If(void) {
	Stack *s;
	int c;

	if(!GetCurrentState(NULL)) {
		CondPush(0,0);
		MatchPush(0);
	} else {
		s = Pop();
		if(!s) {
			Error("Stack Error in if statement");
			return;
		}
		c = CheckCond(s);
		CondPush(c,1);
		MatchPush(c);
	}
}

void Else(void) {
	int state;
	int active;
	int match;

	state = GetCurrentState(&active);

	if(active<0) {
		Error("Misplaced else");
		return;
	}
	if(!active) return;
	CondPop(NULL);
	match = MatchPop();
	CondPush(match?0:1,-1);		
	MatchPush(!match);
}

void ElseIf(void) {
	Stack *s;
	int state, active, match, c;

	state = GetCurrentState(&active);
	
	if(active<0) {
		Error("Misplaced elseif");
		return;
	}
	if(!active) return;
	match = MatchPop();
	CondPop(NULL);
	if(!match) {	
		s = Pop();
		if(!s) {
			Error("Stack Error in elseif statement");
			return;
		}
		c = CheckCond(s);
		CondPush(c,1);
		MatchPush(c);
	} else {
		MatchPush(match);
		CondPush(0,1);
	}
}

void EndIf(void) {
	CondPop(NULL);
	MatchPop();
}
