
/*
 * Copyright (c) 1996, 1997, 2000-2002 Michael Shalayeff.
 * 
 * Version 1.89, last modified 19-Sep-99
 *    
 * Copyright Theodore Ts'o, 1994, 1995, 1996, 1997, 1998, 1999.
 * All rights reserved.
 *
 * Copyright 1998 Niels Provos <provos@citi.umich.edu>
 * All rights reserved.
 * Theo de Raadt <deraadt@openbsd.org> came up with the idea of using
 * such a mathematical system to generate more random (yet non-repeating)
 * ids to solve the resolver/named problem.  But Niels designed the
 * actual system based on the constraints.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/smp_lock.h>
#include <linux/random.h>
#include <linux/grsecurity.h>

#define RU_OUT 180
#define RU_MAX 30000
#define RU_GEN 2
#define RU_N 32749
#define RU_AGEN 7
#define RU_M 31104
#define PFAC_N 3
const static __u16 pfacts[PFAC_N] = { 2, 3, 2729 };

static __u16 ru_x;
static __u16 ru_seed, ru_seed2;
static __u16 ru_a, ru_b;
static __u16 ru_g;
static __u16 ru_counter = 0;
static __u16 ru_msb = 0;
static unsigned long ru_reseed = 0;
static __u32 tmp;

#define TCP_RNDISS_ROUNDS	15
#define TCP_RNDISS_OUT		7200
#define TCP_RNDISS_MAX		30000

static __u8 tcp_rndiss_sbox[128];
static __u16 tcp_rndiss_msb;
static __u16 tcp_rndiss_cnt;
static unsigned long tcp_rndiss_reseed;

static __u16 pmod(__u16, __u16, __u16);
static void ip_initid(void);
__u16 ip_randomid(void);

static __u16
pmod(__u16 gen, __u16 exp, __u16 mod)
{
	__u16 s, t, u;

	s = 1;
	t = gen;
	u = exp;

	while (u) {
		if (u & 1)
			s = (s * t) % mod;
		u >>= 1;
		t = (t * t) % mod;
	}
	return (s);
}

static void
ip_initid(void)
{
	__u16 j, i;
	int noprime = 1;

	ru_x = ((tmp = get_random_long()) & 0xFFFF) % RU_M;

	ru_seed = (tmp >> 16) & 0x7FFF;
	ru_seed2 = get_random_long() & 0x7FFF;

	ru_b = ((tmp = get_random_long()) & 0xfffe) | 1;
	ru_a = pmod(RU_AGEN, (tmp >> 16) & 0xfffe, RU_M);
	while (ru_b % 3 == 0)
		ru_b += 2;

	j = (tmp = get_random_long()) % RU_N;
	tmp = tmp >> 16;

	while (noprime) {
		for (i = 0; i < PFAC_N; i++)
			if (j % pfacts[i] == 0)
				break;

		if (i >= PFAC_N)
			noprime = 0;
		else
			j = (j + 1) % RU_N;
	}

	ru_g = pmod(RU_GEN, j, RU_N);
	ru_counter = 0;

	ru_reseed = xtime.tv_sec + RU_OUT;
	ru_msb = ru_msb == 0x8000 ? 0 : 0x8000;
}

__u16
ip_randomid(void)
{
	int i, n;

	if (ru_counter >= RU_MAX || time_after(xtime.tv_sec, ru_reseed))
		ip_initid();

	if (!tmp)
		tmp = get_random_long();

	n = tmp & 0x3;
	tmp = tmp >> 2;
	if (ru_counter + n >= RU_MAX)
		ip_initid();
	for (i = 0; i <= n; i++)
		ru_x = (ru_a * ru_x + ru_b) % RU_M;
	ru_counter += i;

	return ((ru_seed ^ pmod(ru_g, ru_seed2 ^ ru_x, RU_N)) | ru_msb);
}

__u16
tcp_rndiss_encrypt(__u16 val)
{
	__u16 sum = 0, i;

	for (i = 0; i < TCP_RNDISS_ROUNDS; i++) {
		sum += 0x79b9;
		val ^= ((__u16) tcp_rndiss_sbox[(val ^ sum) & 0x7f]) << 7;
		val = ((val & 0xff) << 7) | (val >> 8);
	}

	return val;
}

static void
tcp_rndiss_init(void)
{
	get_random_bytes(tcp_rndiss_sbox, sizeof (tcp_rndiss_sbox));
	tcp_rndiss_reseed = xtime.tv_sec + TCP_RNDISS_OUT;
	tcp_rndiss_msb = tcp_rndiss_msb == 0x8000 ? 0 : 0x8000;
	tcp_rndiss_cnt = 0;
}

__u32
ip_randomisn(void)
{
	if (tcp_rndiss_cnt >= TCP_RNDISS_MAX ||
	    time_after(xtime.tv_sec, tcp_rndiss_reseed))
		tcp_rndiss_init();

	return (((tcp_rndiss_encrypt(tcp_rndiss_cnt++) |
		  tcp_rndiss_msb) << 16) | (get_random_long() & 0x7fff));
}
