/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/*
 * This file is part of the libepubgen project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef INCLUDED_LIBEPUBGEN_EPUBPRESENTATIONGENERATOR_H
#define INCLUDED_LIBEPUBGEN_EPUBPRESENTATIONGENERATOR_H

#include <librevenge/librevenge.h>
#include <librevenge-stream/librevenge-stream.h>

#include "libepubgen-api.h"
#include "libepubgen-decls.h"

namespace libepubgen
{

class EPUBPackage;

/** EPUB generator for presentations.
  */
class EPUBGENAPI EPUBPresentationGenerator: public librevenge::RVNGPresentationInterface
{
  // disable copying
  EPUBPresentationGenerator(const EPUBPresentationGenerator &);
  EPUBPresentationGenerator &operator=(const EPUBPresentationGenerator &);

  class Impl;

public:
  explicit EPUBPresentationGenerator(EPUBPackage *package, EPUBSplitMethod split = EPUB_SPLIT_METHOD_PAGE_BREAK);
  EPUBPresentationGenerator(EPUBEmbeddingContact &contact, EPUBPackage *package);
  virtual ~EPUBPresentationGenerator();

  void setSplitHeadingLevel(unsigned level);
  void setSplitSize(unsigned size);

  /** Register a handler for embedded images.
    *
    * The handler must convert the image to one of the OPS Core Media Types.
    *
    * @param[in] mimeType the MIME type of the image
    * @param[in] imageHandler a function that converts the image to a
    *   suitable format
    *
    * @sa EPUBEmbeddedImage
    */
  void registerEmbeddedImageHandler(const librevenge::RVNGString &mimeType, EPUBEmbeddedImage imageHandler);

  /** Register a handler for embedded objects.
    *
    * @param[in] mimeType the MIME type of the object
    * @param[in] objectHandler a function that generates EPUB content
    *   from an object's data
    *
    * @sa EPUBEmbeddedObject
    */
  void registerEmbeddedObjectHandler(const librevenge::RVNGString &mimeType, EPUBEmbeddedObject objectHandler);

  virtual void startDocument(const librevenge::RVNGPropertyList &propList);

  virtual void endDocument();

  virtual void setDocumentMetaData(const librevenge::RVNGPropertyList &propList);

  virtual void defineEmbeddedFont(const librevenge::RVNGPropertyList &propList);

  virtual void startSlide(const librevenge::RVNGPropertyList &propList);

  virtual void endSlide();

  virtual void startMasterSlide(const librevenge::RVNGPropertyList &propList);

  virtual void endMasterSlide();

  virtual void setStyle(const librevenge::RVNGPropertyList &propList);

  virtual void setSlideTransition(const librevenge::RVNGPropertyList &propList);

  virtual void startLayer(const librevenge::RVNGPropertyList &propList);

  virtual void endLayer();

  virtual void startEmbeddedGraphics(const librevenge::RVNGPropertyList &propList);

  virtual void endEmbeddedGraphics();

  virtual void openGroup(const librevenge::RVNGPropertyList &propList);

  virtual void closeGroup();

  virtual void drawRectangle(const librevenge::RVNGPropertyList &propList);

  virtual void drawEllipse(const librevenge::RVNGPropertyList &propList);

  virtual void drawPolygon(const librevenge::RVNGPropertyList &propList);

  virtual void drawPolyline(const librevenge::RVNGPropertyList &propList);

  virtual void drawPath(const librevenge::RVNGPropertyList &propList);

  virtual void drawGraphicObject(const librevenge::RVNGPropertyList &propList);

  virtual void drawConnector(const librevenge::RVNGPropertyList &propList);

  virtual void startTextObject(const librevenge::RVNGPropertyList &propList);

  virtual void endTextObject();

  virtual void insertTab();

  virtual void insertSpace();

  virtual void insertText(const librevenge::RVNGString &text);

  virtual void insertLineBreak();

  virtual void insertField(const librevenge::RVNGPropertyList &propList);

  virtual void openOrderedListLevel(const librevenge::RVNGPropertyList &propList);

  virtual void openUnorderedListLevel(const librevenge::RVNGPropertyList &propList);

  virtual void closeOrderedListLevel();

  virtual void closeUnorderedListLevel();

  virtual void openListElement(const librevenge::RVNGPropertyList &propList);

  virtual void closeListElement();

  virtual void defineParagraphStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openParagraph(const librevenge::RVNGPropertyList &propList);

  virtual void closeParagraph();

  virtual void defineCharacterStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openSpan(const librevenge::RVNGPropertyList &propList);

  virtual void closeSpan();

  virtual void openLink(const librevenge::RVNGPropertyList &propList);

  virtual void closeLink();

  virtual void startTableObject(const librevenge::RVNGPropertyList &propList);

  virtual void openTableRow(const librevenge::RVNGPropertyList &propList);

  virtual void closeTableRow();

  virtual void openTableCell(const librevenge::RVNGPropertyList &propList);

  virtual void closeTableCell();

  virtual void insertCoveredTableCell(const librevenge::RVNGPropertyList &propList);

  virtual void endTableObject();

  virtual void startComment(const librevenge::RVNGPropertyList &propList);

  virtual void endComment();

  virtual void startNotes(const librevenge::RVNGPropertyList &propList);

  virtual void endNotes();

  virtual void defineChartStyle(const librevenge::RVNGPropertyList &propList);

  virtual void openChart(const librevenge::RVNGPropertyList &propList);

  virtual void closeChart();

  virtual void openChartTextObject(const librevenge::RVNGPropertyList &propList);

  virtual void closeChartTextObject();

  virtual void openChartPlotArea(const librevenge::RVNGPropertyList &propList);

  virtual void closeChartPlotArea();

  virtual void insertChartAxis(const librevenge::RVNGPropertyList &propList);

  virtual void openChartSeries(const librevenge::RVNGPropertyList &propList);

  virtual void closeChartSeries();

  virtual void openAnimationSequence(const librevenge::RVNGPropertyList &propList);

  virtual void closeAnimationSequence();

  virtual void openAnimationGroup(const librevenge::RVNGPropertyList &propList);

  virtual void closeAnimationGroup();

  virtual void openAnimationIteration(const librevenge::RVNGPropertyList &propList);

  virtual void closeAnimationIteration();

  virtual void insertMotionAnimation(const librevenge::RVNGPropertyList &propList);

  virtual void insertColorAnimation(const librevenge::RVNGPropertyList &propList);

  virtual void insertAnimation(const librevenge::RVNGPropertyList &propList);

  virtual void insertEffect(const librevenge::RVNGPropertyList &propList);

private:
  Impl *m_impl;
};

}

#endif // INCLUDED_LIBEPUBGEN_EPUBPRESENTATIONGENERATOR_H

/* vim:set shiftwidth=2 softtabstop=2 expandtab: */
