/*
 * dproc.c - PTX process access functions for lsof
 */


/*
 * Copyright 1995 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1995 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dproc.c,v 1.2 97/04/07 09:27:54 abe Exp $";
#endif

#include "lsof.h"


/*
 * Local static values
 */

static unsigned long Asoplg;		/* kernel's asops_large adddress */
static unsigned long Asopsm;		/* kernel's asops_small adddress */
static long Kp;				/* kernel's process table address */
static int Nf = 0;			/* nominal open files per process */
static struct var Var;			/* kernel's var structure */

#if	PTXV>=400
int Nv = 0;				/* number of Vc[] entries allocated */
struct vnode **Vc = NULL;		/* vnode ponter cache */
int Vu;					/* number of Vc[] entries in use */
#endif	/* PTXV>=400 */


_PROTOTYPE(static void get_kernel_access,(void));

#if	PTXV>=400
_PROTOTYPE(static void process_text,(struct vnode *xv, struct as *as));
_PROTOTYPE(static void store_va,(struct vnode *va));
#endif	/* PTXV>=400 */


/*
 * gather_proc_info() -- gather process information
 */

void
gather_proc_info()
{
	int i;
	struct ofile *ofp;
	static struct ofile_tab *oft = NULL;
	static MALLOC_S oftsz;
	MALLOC_S ofxmax, ofxsz;
	struct ofile *ofx = NULL;
	struct proc *p;
	pid_t pid;
	KA_T pa;
	struct proc ps;
	short pss, sf;
	int px;
	off_t sp;
	uid_t uid;
	caddr_t va;

#if	PTXV<400
	struct swiocread sw;
	static struct user *u;
	static char *ub;
	MALLOC_S ui;
#else	/* PTXV>=400 */
	struct ucred pc;
	struct proc_data pd;
#endif	/* PTXV<400 */

#if	PTXV<400
/*
 * Allocate space for the user structure.
 */
	if (ub == (char *)NULL) {
	    if ((ub = (char *)malloc((MALLOC_S)ctob(UPAGES) + 511)) == NULL) {
		(void) fprintf(stderr,
		    "%s: can't allocate space for user structure\n", Pn);
		Exit(1);
	    }
	    ub = (char *)(((int)ub + 511) & ~511);
	    u = UBTOUSER(ub);
	}
#endif	/* PTXV<400 */

/*
 * Allocate space for the nominal number of open file table structures.
 */
	if (oft == (struct ofile_tab *)NULL) {
	    oftsz = (MALLOC_S)(sizeof(struct ofile_tab)
	          +		   ((Nf - 1) * sizeof(struct ofile)));
	    if ((oft = (struct ofile_tab *)malloc(oftsz)) == NULL) {
		(void) fprintf(stderr,
		    "%s: can't allocate space for ofile structures\n", Pn);
		Exit(1);
	    }
	}

#if     defined(HASNCACHE)
/*
 * Read kernel name cache.
 */
        ncache_load();
#endif  /* defined(HASNCACHE) */

/*
 * Examine proc structures and their associated information.
 */
	for (p = &ps, px = 0; px < Var.v_proc; px++) {

	/*
	 * Read the proc structure.
	 */
		pa = (KA_T)(Kp + (long)(px * sizeof(struct proc)));
		if (kread(pa, (char *)&ps, sizeof(ps)))
			continue;
		if (p->p_stat == 0 || p->p_stat == SZOMB)
			continue;

#if	PTXV<400
	/*
	 * Set UID for PTX < 4.
	 */
		uid = p->p_uid;

#else	/* PTXV>=400 */

	/*
	 * Read credentials for PTX 4 and above to get UID.
	 *
	 * Read proc_data structure to get command name.
	 */
		if (p->p_cred == NULL
		||  kread((KA_T)p->p_cred, (char *)&pc, sizeof(pc))
		||  p->p_data == NULL
		||  kread((KA_T)p->p_data, (char *)&pd, sizeof(pd)))
			continue;
		uid = pc.cr_ruid;
#endif	/* PTXV>=400 */

		if (is_proc_excl(p->p_pid, (int)p->p_pgrp, (UID_ARG)uid,
		    &pss, &sf))
			continue;

#if	PTXV<400
	/*
	 * Read the user area.
	 */
		if ((p->p_flag & SLOAD) == 0) {
			if (Swap < 0)
				continue;
			sw.swr_blkno = (swblk_t)p->p_swaddr;
			sw.swr_len = ctob(UPAGES);
			sw.swr_data = (caddr_t)ub;
			if (ioctl(Swap, SWIOCREAD, &sw) != 0)
				continue;
		} else {
			if (kread((KA_T)p->p_uarea, (char *)u, U_SIZE))
				continue;
		}
		if ((KA_T)u->u_procp != pa)
			continue;
#endif	/* PTXV<400 */

	/*
	 * Allocate a local process structure.
	 */
		if (is_cmd_excl(

#if	PTXV<400
			u->u_comm,
#else	/* PTXV>=400 */
			pd.pd_comm,
#endif	/* PTXV<400 */

			&pss, &sf)
		)
			continue;
		alloc_lproc((int)p->p_pid, (int)p->p_pgrp, (int)p->p_ppid,
			(UID_ARG)uid,

#if	PTXV<400
			u->u_comm,
#else	/* PTXV>=400 */
			pd.pd_comm,
#endif	/* PTXV<400 */
			(int)pss, (int)sf);

		Plf = NULL;
	/*
	 * Save current working directory information.
	 */

#if	PTXV<400
		if ((va = (caddr_t)u->u_cdir) != NULL)
#else	/* PTXV>=400 */
		if ((va = (caddr_t)p->p_cdir) != NULL)
#endif	/* PTXV<400 */
		{

			alloc_lfile(CWD, -1);
			process_node(va);
			if (Lf->sf)
				link_lfile();
		}
	/*
	 * Save root directory information.
	 */

#if	PTXV<400
		if ((va = (caddr_t)u->u_rdir) != NULL)
#else	/* PTXV>=400 */
		if ((va = (caddr_t)p->p_rdir) != NULL)
#endif	/* PTXV<400 */

		{
			alloc_lfile(RTD, -1);
			process_node(va);
			if (Lf->sf)
				link_lfile();
		}

#if	PTXV>=400
	/*
	 * Process PTX 4 and above executable and library text accesses.
	 */
		if (p->p_execvp || p->p_as)
			process_text(p->p_execvp, p->p_as);
#endif	/* PTXV>=400 */

	/*
	 * Read the open file structures.
	 */

#if	PTXV<400
		if (kread((KA_T)u->u_ofile_tab, (char *)oft, oftsz))
			continue;
		if (((long)u->u_ofile_tab + (long)oft->oft_lofile - (long)oft)
		!=  (long)oft->oft_ofile)
#else	/* PTXV>=400 */
		if (kread((KA_T)p->p_ofile_tab, (char *)oft, oftsz))
			continue;
		if (((long)p->p_ofile_tab + (long)oft->oft_lofile - (long)oft)
		!=  (long)oft->oft_ofile)
#endif	/* PTXV<400 */

		{

		/*
		 * Read the open file pointers from an extended area.
		 */
		    ofxsz = (MALLOC_S)(oft->oft_nofile * sizeof(struct ofile));
		    if (ofx == NULL) {
			if ((ofx = (struct ofile *)malloc(ofxsz)) == NULL) {

no_ofx_space:
			    (void) fprintf(stderr,
				"%s: no space for %d open files: PID %d\n",
				Pn, ofxsz, Lp->pid);
			    continue;
			}
			ofxmax = ofxsz;
		    } else if (ofxsz > ofxmax) {
			if ((ofx = (struct ofile *)realloc(ofx, ofxsz))
			== NULL)
			    goto no_ofx_space;
			ofxmax = ofxsz;
		    }
		    if (kread((KA_T)oft->oft_ofile, (char *)ofx, ofxsz))
			continue;
		    ofp = ofx;
		 } else
		    ofp = oft->oft_lofile;

	/*
	 * Save information on file descriptors.
	 */
		for (i = 0; i < oft->oft_nofile; i++, ofp++) {
			if (ofp->of_file) {
				alloc_lfile(NULL, i);
				process_file(ofp->of_file);
				if (Lf->sf)
					link_lfile();
			}
		}
	/*
	 * Examine results.
	 */
		if (examine_lproc())
			return;
	}
}


/*
 * get_kernel_access() - access the required information in the kernel
 */

static void
get_kernel_access()
{
	unsigned long v;

#if	PTXV>=400
	int warn;
#endif	/* PTXV>=400 */

#if	PTXV<400
/*
 * Open swap access.
 */
	if (Memory == NULL || strcmp(Memory, KMEM) == 0) {
	    if ((Swap = open(SWAP, O_RDONLY, 0)) < 0) {
		(void) fprintf(stderr, "%s: %s: %s\n", Pn, SWAP,
		    strerror(errno));
		Exit(1);
	    }
	}
#endif	/* PTXV<400 */

#if	defined(WILLDROPGID)
/*
 * If kernel memory isn't coming from KMEM, drop setgid permission
 * before attempting to open the (Memory) file.
 */
	if (Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the non-KMEM memory file is readable.
 */
	if (Memory && !is_readable(Memory, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Open kernel memory access.
 */
	if ((Kd = open(Memory ? Memory : KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		Memory ? Memory : KMEM, strerror(errno));
	    Exit(1);
	}

#if	defined(WILLDROPGID)
/*
 * Drop setgid permission, if necessary.
 */
	if (!Memory)
		(void) dropgid();
#else	/* !defined(WILLDROPGID) */
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		Exit(1);
#endif	/* defined(WILLDROPGID) */

/*
 * Access kernel symbols.
 */
	(void) build_Nl(Drive_Nl);
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0) {
	    (void) fprintf(stderr, "%s: can't read namelist from %s\n", Pn,
		Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}
	if (get_Nl_value("proc", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Kp, sizeof(Kp)) || !Kp) {
	    (void) fprintf(stderr, "%s: can't read proc table address\n", Pn);
	    Exit(1);
	}
/*
 * Get the kernel's var structure.
 */
	v = (unsigned long)0;
	if (get_Nl_value("var", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Var, sizeof(Var))) {
	    (void) fprintf(stderr, "%s: can't read var structure from %#x\n",
		Pn, v);
	    Exit(1);
	}
	if (Var.v_proc == 0) {
	    (void) fprintf(stderr, "%s: proc table length is zero\n", Pn);
	    Exit(1);
	}
/*
 * Get the per-process file count.
 */
	v = (unsigned long)0;
	if (get_Nl_value("nofile", Drive_Nl, &v) < 0 || !v
	||  kread((KA_T)v, (char *)&Nf, sizeof(Nf)) || Nf < 3) {
	    (void) fprintf(stderr, "%s: bad open file count (%d) from %#x\n",
		Pn, Nf, v);
	    Exit(1);
	}
/*
 * Read clone major device number, if possible.
 */
	if (get_Nl_value("clmaj", Drive_Nl, &v) >= 0 && v
	&&  kread((KA_T)v, (char *)&CloneMajor, sizeof(CloneMajor)) == 0) {
	    HaveCloneMajor = 1;
	    findclones();
	} else
	    HaveCloneMajor = 0;

#if	PTXV>=400
/*
 * Check on asops_* addresses and complain about missing ones.
 */
	warn = 0;
	if (get_Nl_value("asoplg", Drive_Nl, &Asoplg) < 0 || !Asoplg) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: asop_large kernel address is undefined.\n",
		    Pn);
	    warn = 1;
	}
	if (get_Nl_value("asopsm", Drive_Nl, &Asopsm) < 0 || !Asopsm) {
	    if (!Fwarn)
		(void) fprintf(stderr,
		    "%s: WARNING: asops_small kernel address is undefined.\n",
		    Pn);
	    warn = 1;
	}
	if (warn && !Fwarn) {
	    (void) fprintf(stderr,
		"%s: WARNING: may not be able to identify \"txt\" files.\n",
		    Pn);
	}
#endif	/* PTXV>=400 */

}


/*
 * initialize() - perform all initialization
 */

void
initialize()
{
	get_kernel_access();
}


/*
 * kread() - read from kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel memory address */
	char *buf;			/* buffer to receive data */
	READLEN_T len;			/* length to read */
{
	int br;

	if (lseek(Kd, (long)addr, SEEK_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


#if	PTXV>=400
/*
 * process_text() - process text references for PTX 4 and above
 */

static void
process_text(xv, as)
	struct vnode *xv;		/* pointer to executable's vnode */
	struct as *as;			/* address space structure pointer */
{
	struct as a;
	static struct vnode **da = NULL;
	int i, j;
	MALLOC_S len;
	static int nd = 0;
/*
 * Save the address of the executable vnode.
 */
	Vu = 0;
	store_va(xv);
/*
 * If there is an address space map, store the addresses of the vnodes
 * in its mapped vnode lists.  Discard duplicates.
 */
	if (as != (struct as *)NULL
	&&  kread((KA_T)as, (char *)&a, sizeof(a)) == 0) {

	/*
	 * Save the vnodes of the short, static list.
	 */
	    if (Asopsm && Asopsm == (unsigned long)a.as_ops) {
		for (i = j = 0; i < AS_SVSM_MMAP_MAX; i++) {
		    if (a.as_mvp.mvp_small[i] != (struct vnode *)NULL) {
			store_va(a.as_mvp.mvp_small[i]);
			j++;
		    }
		} 
	    }
	/*
	 * If there's a dynamic list, allocate space for it, read it, and save
	 * the addresses of the vnodes it contains.
	 */
	    if (Asoplg && Asoplg == (unsigned long)a.as_ops) {
		if ((i = a.as_mvp.mvp_dynamic_size) > 0 && a.as_mvp.mvp_large) {
		    len = (MALLOC_S)(i * sizeof(struct vnode *));
		    if (da == NULL) {
			if ((da = (struct vnode **)malloc(len)) == NULL) {

no_vm_vp_space:

				(void) fprintf(stderr,
				    "%s: no space for dynamic vnodes: PID %d\n",
				    Pn, Lp->pid);
				Exit(1);
			}
			nd = i;
		    } else if (i >= nd) {
			if ((da = (struct vnode **)realloc(da, len)) == NULL)
			    goto no_vm_vp_space;
			nd = i;
		    }
		    if (kread((KA_T)a.as_mvp.mvp_large, (char *)da,
			(READLEN_T)len)
		    == 0) {
			for (j = 0; j < i; j++) {
			    if (da[j] != (struct vnode *)NULL)
				store_va(da[j]);
			}
		    }
		}
	   }
	}
/*
 * Process the unique vnode addresses accumulated from the executable,
 * short static vnode list, and dynamic vnode list.
 */
	for (i = 0; i < Vu; i++) {
	    alloc_lfile("txt", -1);
	    process_node((caddr_t)Vc[i]);
	    if (Lf->sf)
		link_lfile();
	}
}


/*
 * store_va() - store unique vnode address
 */

void
store_va(va)
	struct vnode *va;		/* vnode address */
{
	int i;
	MALLOC_S len;

	if (va == (struct vnode *)NULL)
		return;
	for (i = 0; i < Vu; i++) {
		if (va == Vc[i])
			return;
	}
	if (Vc == NULL) {
		Nv = AS_SVSM_MMAP_MAX;
		len = (MALLOC_S)(Nv * sizeof(struct vnode *));
		if ((Vc = (struct vnode **)malloc(len)) == NULL) {

no_Vc_space:

			(void) fprintf(stderr,
				"%s: no vnode cache space: PID %d\n",
				Pn, Lp->pid);
			Exit(1);
		}
	} else if (Vu >= Nv) {
		Nv += MVP_DYN_MIN;
		len = (MALLOC_S)(Nv * sizeof(struct vnode *));
		if ((Vc = (struct vnode **)realloc(Vc, len)) == NULL)
			goto no_Vc_space;
	}
	Vc[Vu++] = va;
}
#endif	/* PTXV>=400 */


#if	defined(HASNCACHE) && defined(HASVXFSDNLC)


#include <stddef.h>
#include <sys/dnlc.h>
#include <sys/vx_dnlc.h>


/*
 * Define the local name cache structures.
 */

struct lnch {				/* local name cache structure */
	unsigned long np;		/* node pointer */
	unsigned long dp;		/* directory pointer */
	char *nm;			/* name */
	int nl;				/* name length */
	unsigned char dup;		/* duplicate if 1 */
	unsigned char type;		/* type: 0 = vnode; 1 = VxFS */
	struct lnch *pa;		/* parent address */
	struct lnch *next;		/* link to next same-type structure */
};


/*
 * Local name cache (LNC) definitions, macroes, and static values
 */

static int LNC_asz = 0;			/* local cache allocated size */
static int LNC_csz = 0;			/* local cache current size */
#define	LNCHHLEN	128		/* hash head length (must be a
					 * power of 2) */
#define	LNCINCR		256		/* local size increment */
#define	LNCINIT		1024		/* local initial size */

#define LNC_HASH(np)	((((int)(np)>>2)*31415)&(LNCHHLEN-1))

static struct lnch **LNC_hh = (struct lnch **)NULL;
					/* local hash head pointers */
static struct lnch *LNC_nc = (struct lnch *)NULL;
					/* the linear local cache */


/*
 * Local function prototypes
 */

_PROTOTYPE(static struct lnch *LNC_addr,(unsigned long np, int type));
_PROTOTYPE(static void LNC_enter,(struct lnch *le, char *nm, int nl));
_PROTOTYPE(static int LNC_isroot,(unsigned long np, char *cp, int ty));
_PROTOTYPE(static void LNC_nosp,(int len));
_PROTOTYPE(static void VN_load,());
_PROTOTYPE(static void VXFS_load,());


/*
 * LNC_addr() - look up a node's local address
 */

static struct lnch *
LNC_addr(np, type)
	unsigned long np;		/* vnode or vx_inode pointer */
	int type;			/* type: 0 = vnode; 1 = VxFS */
{
	struct lnch *lc = LNC_hh[LNC_HASH(np)];

	while (lc) {
	    if (lc->np == np && lc->type == type)
		return(lc);
	    lc = lc->next;
	}
	return((struct lnch *)NULL);
}


/*
 * LNC_enter() - make a local name cache entry
 */

static void
LNC_enter(le, nm, nl)
	struct lnch *le;		/* skeleton local entry */
	char *nm;			/* name */
	int nl;				/* name length */
{
	struct lnch *lc;
	int len;

	if (LNC_csz >= LNC_asz) {
	    LNC_asz += LNCINCR;
	    len = (MALLOC_S)(LNC_asz * sizeof(struct lnch));
	    if (!(LNC_nc = (struct lnch *)realloc(LNC_nc, len))) {
		(void) fprintf(stderr,
		    "%s: no more space for %d byte local name cache: %s\n",
		    Pn, len,
		    lc->type ? "VxFS" : "VNODE");
		Exit(1);
	    }
	}
	lc = &LNC_nc[LNC_csz];
/*
 * Allocate space for the name and create the local cache entry.
 */
	if (!(lc->nm = (char *)malloc((MALLOC_S)(nl + 1)))) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes to %s name cache entry\n",
		Pn, nl, lc->type ? "VxFS" : "VNODE");
	    Exit(1);
	}
	lc->dp = le->dp;
	lc->dup = 0;
	lc->nl = nl;
	(void) strncpy(lc->nm, nm, nl);
	lc->nm[nl] = '\0';
	lc->np = le->np;
	lc->next = lc->pa = (struct lnch *)NULL;
	lc->type = le->type;
	LNC_csz++;
}


/*
 * LNC_isroot() - is head of name cache path a file system root?
 */

static int
LNC_isroot(np, cp, ty)
	unsigned long np;			/* kernel node pointer */
	char *cp;				/* partial path */
	int ty;					/* kernel node type */
{
	char buf[MAXPATHLEN];
	int i;
	MALLOC_S len;
	struct mounts *mtp;
	static int rca = 0;
	static int rcn = 0;
	static unsigned long *rnc = (unsigned long *)NULL;
	struct stat sb;
	struct vnode v;
	int vs = 0;

	if (!np)
	    return(0);
/*
 * Search the root vnode cache.
 */
	for (i = 0; i < rcn; i++) {
	    if (np == rnc[i])
		return(1);
	}
/*
 * Get the vn_inode's vnode or the vnode.
 */
	if (ty) {
	    if (read_vxvnode((KA_T)np, &v) == 0)
		vs = 1;
	} else {
	    if (kread((KA_T)np, (char *)&v, sizeof(v)) == 0)
		vs = 1;
	}
/*
 * If there is a vnode, see if it's a VDIR vnode with the VROOT flag set.
 * If it is, then the path is complete.
 *
 * If it isn't, and if the file has an inode number, search the mount table
 * and see if the file system's inode number is known.  If it is, form the
 * possible full path, safely stat() it, and see if it's inode number matches
 * the one we have for this file.  If it does, then the path is complete.
 */
	if (!vs || v.v_type != VDIR || !(v.v_flag & VROOT)) {

	/*
	 * The vnode tests failed.  Try the inode tests.
	 */
	    if (Lf->inp_ty != 1 || !Lf->inode
	    ||  !Lf->fsdir || (len = strlen(Lf->fsdir)) < 1)
		return(0);
	    if ((len + 1 + strlen(cp) + 1) > sizeof(buf))
		return(0);
	    for (mtp = Mtab; mtp; mtp = mtp->next) {
		if (!mtp->dir || !mtp->inode)
		    continue;
		if (strcmp(Lf->fsdir, mtp->dir) == 0)
		    break;
	    }
	    if (!mtp)
		return(0);
	    (void) strcpy(buf, Lf->fsdir);
	    if (buf[len - 1] != '/')
		buf[len++] = '/';
	    (void) strcpy(&buf[len], cp);
	    if (statsafely(buf, &sb) != 0
	    ||  (unsigned long)sb.st_ino != Lf->inode)
		return(0);
	}
/*
 * Add the node address to the root node cache.
 */
	if (rcn >= rca) {
	    if (rca == 0) {
		len = (MALLOC_S)(10 * sizeof(unsigned long));
		rnc = (unsigned long *)malloc(len);
		rca = 10;
	    } else {
		len = (MALLOC_S)((rca + 10) * sizeof(unsigned long));
		rnc = (unsigned long *)realloc(rnc, len);
		rca += 10;
	    }
	    if (!rnc) {
		(void) fprintf(stderr, "%s: no space for root node table\n",
		    Pn);
		Exit(1);
	    }
	}
	rnc[rcn++] = np;
	return(1);
}


/*
 * LNC_nosp() - notify that we're out of space for the local name cache
 */

static void
LNC_nosp(len)
	int len;			/* attempted length */
{
	if (!Fwarn)
	    (void) fprintf(stderr,
		"%s: no space for %d byte local name cache\n",
		Pn, len);
	Exit(1);
}


/*
 * ncache_load() - load the kernel's NFS and DEV name caches
 */

void
ncache_load()
{
	struct lnch *hh, *hl, *hlp, *lc;
	int f, i, j, len;

	if (!Fncache)
	    return;
/*
 * Initialize local name cache, as required.
 */
	if (LNC_asz == 0) {
	    LNC_asz = LNCINIT;
	    len = LNCINIT * sizeof(struct lnch);
	    if (!(LNC_nc = (struct lnch *)malloc((MALLOC_S)len)))
		(void) LNC_nosp(len);
	} else if (LNC_hh) {

	/*
	 * Free the name space previously used by the local cache.
	 */
	    for (i = 0; i < LNCHHLEN; i++) {
		for (lc = LNC_hh[i]; lc; lc = lc->next) {
		    if (lc->nm) {
			(void) free((FREE_P *)lc->nm);
			lc->nm = (char *)NULL;
		    }
		}
	    }
	}
	LNC_csz = 0;

/*
 * Load the vnode cache.
 */
	(void) VN_load();
/*
 * Load the VxFS cache.
 */
	(void) VXFS_load();
/*
 * Reduce local name cache memory usage, as required.
 */
	if (LNC_csz < 1) {
	    LNC_csz = 0;
	    if (!RptTm) {
		(void) free((FREE_P *)LNC_nc);
		LNC_nc = (struct lnch *)NULL;
	    }
	    return;
	}
	if (LNC_csz < LNC_asz && !RptTm) {
	    len = LNC_csz * sizeof(struct lnch);
	    if (!(LNC_nc = (struct lnch *)realloc(LNC_nc, len)))
		(void)LNC_nosp(len);
	    LNC_asz = LNC_csz;
	}
/*
 * Initialize hash head pointers.
 */
	if (!LNC_hh) {
	    LNC_hh = (struct lnch **)calloc(LNCHHLEN, sizeof(struct lnch *));
	    if (!LNC_hh) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for name cache hash table\n",
		    Pn, LNCHHLEN * sizeof(struct lnch *));
		Exit(1);
	    }
	} else
	    (void) zeromem((void *)LNC_hh, (LNCHHLEN * sizeof(struct lnch *)));
/*
 * Enter local name cache pointers in the hash table.  Look for entries with
 * the same identifications that have different names.
 */
	for (i = 0, lc = LNC_nc; i < LNC_csz; i++, lc++) {
	    j = LNC_HASH(lc->np);
	    if (!(hl = LNC_hh[j])) {
		LNC_hh[j] = lc;
		continue;
	    }
	    for (f = 0, hlp = hl; hl; hlp = hl, hl = hl->next) {
		if (lc->np != hl->np || lc->type != hl->type)
		    continue;
		if (strcmp(lc->nm, hl->nm) == 0)
		    f = 1;
		else {
		    f = 2;	/* same identifiers, different names */
		    break;
		}
	    }
	    if (!f)
		hlp->next = lc;
	    else if (f == 2) {

	    /*
	     * Since entries with the same identification but different names
	     * were located, mark entries with the same identification as
	     * duplicates.
	     */
		for (hl = LNC_hh[j]; hl; hl = hl->next) {
		    if (hl->np == lc->np && hl->type == lc->type)
			hl->dup = 1;
		}
	    }
	}
/*
 * Make a final pass through the local name cache and convert parent directory
 * identifications to local name cache pointers. Ignore duplicates.
 */
	for (i = 0, lc = LNC_nc; i < LNC_csz; i++, lc++) {
	    if (lc->dup || !lc->dp)
		continue;
	    lc->pa = LNC_addr(lc->dp, lc->type);
	}
}


/*
 * ncache_lookup() - look up a node's name in the kernel's name caches
 */

char *
ncache_lookup(buf, blen, fp)
	char *buf;			/* receiving name buffer */
	int blen;			/* receiving buffer length */
	int *fp;			/* full path reply */
{
	char *cp = buf;
	struct lnch *lc;
	struct mounts *mtp;
	int nl, rlen;

	*cp = '\0';
	*fp = 0;

# if	defined(HASFSINO)
/*
 * If the entry has an inode number that matches the inode number of the
 * file system mount point, return an empty path reply.  That tells the
 * caller to print the file system mount point name only.
 */
	if (Lf->inp_ty == 1 && Lf->fs_ino && Lf->inode == Lf->fs_ino)
	    return(cp);
# endif	/* defined(HASFSINO) */

/*
 * Look up the name cache entry for the node address.
 */
	if (LNC_nc == 0
	||  !(lc = LNC_addr((unsigned long)Lf->na, (int)Lf->is_vxfs))) {

	/*
	 * If the node has no cache entry, see if it's the mount
	 * point of a known file system.
	 */
	    if (!Lf->fsdir || !Lf->dev_def || Lf->inp_ty != 1)
		return(NULL);
	    for (mtp = Mtab; mtp; mtp = mtp->next) {
		if (!mtp->dir || !mtp->inode)
		    continue;
		if (Lf->dev == mtp->dev
		&&  (unsigned long)mtp->inode == Lf->inode
		&&  strcmp(mtp->dir, Lf->fsdir) == 0)
		    return(cp);
	    }
	    return(NULL);
	}
/*
 * Begin the path assembly.
 */
	if ((nl = lc->nl) > (blen - 1))
	    return(NULL);
	cp = buf + blen - nl - 1;
	rlen = blen - nl - 1;
	(void) strcpy(cp, lc->nm);
/*
 * Look up the name cache entries that are parents of the node address.
 * Quit when:
 *
 *	there's no parent;
 *	the name is too large to fit in the receiving buffer.
 */
	for (;;) {
	    if (!lc->pa) {
		if (LNC_isroot(lc->dp, cp, lc->type))
		    *fp = 1;
		break;
	    }
	    lc = lc->pa;
	    if (((nl = lc->nl) + 1) > rlen)
		break;
	    *(cp - 1) = '/';
	    cp--;
	    rlen--;
	    (void) strncpy((cp - nl), lc->nm, nl);
	    cp -= nl;
	    rlen -= nl;
	}
	return(cp);
}


/*
 * VN_load() - load vnode name cache
 */

static void
VN_load()
{
	struct ncache *cp;
	int i, kcl, kcn, len;
	KA_T ka;
	static struct ncache *kc = (struct ncache *)NULL;
	static KA_T kcp = (KA_T)0;
	static int kcla = 0;
	static KA_T kla = (KA_T)0;
	struct lnch lc;
/*
 * Do first-time setup, as required.
 */
	if (!kc) {

	/*
	 * Get the addresses of the length of the kernel's vnode name cache
	 * and the cache pointer.
	 */
	    if (get_Nl_value(X_NCSIZE, Drive_Nl, (unsigned long *)&kla) < 0
	    ||  !kla)
		return;
	    if (get_Nl_value(X_NCACHE, Drive_Nl, (unsigned long*)&kcp) < 0
	    ||  !kcp)
		return;
	}
/*
 * Get kernel's vnode name cache length.
 */
	if (kread(kla, (char *)&kcn, sizeof(kcn)) || kcn < 1)
	    return;
/*
 * Allocate space for a copy of the kernel's vnode name cache.
 */
	kcl = kcn * sizeof(struct ncache);
	if (!kc) {
	    kc = (struct ncache *)malloc(kcl);
	    kcla = kcl;
	} else if (kcl > kcla) {
	    kc = (struct ncache *)realloc((MALLOC_P *)kc, kcl);
	    kcla = kcl;
	}
	if (!kc) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for kernel vnode cache\n",
		Pn, kcl);
	    Exit(1);
	}
/*
 * Read the kernel's vnode name cache.
 */
	if (!kc || !kcl || !kcp || kread(kcp, (char *)&ka, sizeof(ka)))
	    return;
	if (!ka || kread(ka, (char *)kc, kcl))
	    return;
/*
 * Scan the kernel's vnode cache and create local name cache entries.
 */
	lc.type = 0;
	for (cp = kc, i = 0; i < kcn; cp++, i++) {
	    if (!cp->vp || !cp->name[0] || cp->namlen < 1)
		continue;
	    if (cp->namlen < 3 && cp->name[0] == '.') {
		if (cp->namlen == 1 || (cp->namlen == 2 && cp->name[1] == '.'))
		    continue;
	    }
	    len = (cp->namlen > NC_NAMLEN) ? NC_NAMLEN : cp->namlen;
	    lc.np = (unsigned long)cp->vp;
	    lc.dp = (unsigned long)cp->dp;
	    LNC_enter(&lc, cp->name, len);
	}
/*
 * If not repeating, free space allocated for the copy of the kernel's
 * vnode cache.
 */
	if (kc && !RptTm) {
	    (void) free((FREE_P *)kc);
	    kc = (struct ncache *)NULL;
	    kcla = 0;
	}
}


/*
 * VXFS_load() - load VxFS name cache
 */

static void
VXFS_load()
{
	struct vx_dnlc_hash *ha, *hp;
	int i, len;
	static KA_T ka = (KA_T)0;
	KA_T kt;
	struct lnch lc;
	static char *nm = (char *)NULL;
	static int nma = 0;
	struct vx_dnlc v;
	KA_T vn, vp;
	static struct vx_dnlc_hash *vh = (struct vx_dnlc_hash *)NULL;
	static int vhl = 0;
	static int vhn = 0;
/*
 * Do first-time setup, as required.
 */
	if (!vh) {

	/*
	 * Get the addresses of the length of the kernel's vx_inode name cache
	 * hash table and its entry count.  Read the entry count.
	 */
	    if (get_Nl_value("vxdnlc", Drive_Nl, (unsigned long *)&kt) < 0
	    ||  !kt
	    ||  kread(kt, (char *)&ka, sizeof(ka))
	    ||  !ka)
		return;
	    if (get_Nl_value("vxndnlc", Drive_Nl, (unsigned long *)&kt) < 0
	    ||  !kt
	    ||  kread(kt, (char *)&vhn, sizeof(vhn))
	    ||  vhn < 1)
		return;
	/*
	 * Allocate space for the hash table.
	 */
	    vhl = vhn * sizeof(struct vx_dnlc_hash);
	    if (!(vh = (struct vx_dnlc_hash *)malloc(vhl))) {
		(void) fprintf(stderr,
		    "%s: can't allocate %d bytes for VxFS hash table\n",
		    Pn, len);
		Exit(1);
	    }
	}
/*
 * Read the kernel's VxFS name cache hash structure table.
 */
	if (!ka || !vh || !vhl || kread(ka, (char *)vh, vhl))
	    return;
/*
 * Scan the kernel's VxFS cache hash table and create local name cache entries.
 */
	lc.type = 1;
	for (ha = (struct vx_dnlc_hash *)ka, hp = vh, i = 0;
	     i < vhn;
	     ha++, hp++, i++)
	{
	    if ((vp = (KA_T)hp->hash_next) && vp != (KA_T)ha) {
		do {
		    if (kread(vp, (char *)&v, sizeof(v)))
			break;
		    if (!v.ip || !v.name[0] || v.namelen < 1
		    ||  (v.namelen == 1 && v.name[0] == '.'))
			continue;
		    if (!nm) {
			nma = v.namelen;
			nm = (char *)malloc(nma);
		    } else if ((int)v.namelen > nma) {
			nma = v.namelen;
			nm = (char *)realloc(nm, nma);
		    }
		    if (!nm) {
			(void) fprintf(stderr,
			    "%s: can't allocate %d name bytes\n", Pn, nma);
			Exit(1);
		    }
		    nm[0] = v.name[0];
		    if (v.namelen > 1) {
			vn = vp + (KA_T)(offsetof(struct vx_dnlc, name) + 1);
			if (kread(vn, nm + 1, v.namelen - 1))
			    break;
		    }
		    if ((int)v.namelen == 2 && nm[0] == '.')
			continue;
		    lc.np = (unsigned long)v.ip;
		    lc.dp = (unsigned long)v.dip;
		    LNC_enter(&lc, nm, (int)v.namelen);
		} while ((vp = (KA_T)v.hash_next) && vp != (KA_T)ha
			&& vp != (KA_T)hp->hash_next);
	    }
	}
/*
 * If not repeating, free space allocated for the copy of the kernel's
 * vnode cache.
 */
	if (vh && !RptTm) {
	    (void) free((FREE_P *)vh);
	    vh = (struct vx_dnlc_hash *)NULL;
	    vhl = vhn = 0;
	}
}
#endif	/* defined(HASNCACHE) && defined(HASVXFSDNLC) */
