/*
 * This filter is based on outputFilter.t by Iain Merrick.
 * Modified by Nikos Chantziaras (realnc@hotmail.com).
 * Release date: 4 Sep 2002
 */

outputFilter: function( str )
{
	// All the work is done here. TADS will call this function with a string
	// which is due to be printed. If we return nil, the string will be
	// printed unchanged. If we return another string, TADS will print that
	// string instead. And that's what we're going to do.

	local result = '';  // String we're going to return (initially empty).
	local c;            // Current character.
	local i;            // Loop index.
	local len;          // The string's length.
	local dropNext = 0; // Characters to skip on next loop start.
	local next1;        // Next character (peek).

	// Check too see if we should filter the string.
	if (global.skipFilterOnce) {
		// The string is not intended to be filtered. Activate filter for the
		// next calls and return nil, so the string is printed as is.
		global.skipFilterOnce = nil;
		return nil;
	}

	len = length(str);

	for (i = 1; i <= len; ++i) { // For every character in the string.
		// Has this character already been processed?
		if (dropNext > 0) {
			// Do nothing.
			--dropNext;
			continue;
		}

		// Get the next character.
		c = substr(str, i, 1);

		// Most characters aren't of special meaning to us, so it's much
		// faster if we skip them at the beginning of the loop.
		switch (c) {
			case '<': case '>': case '%': case '\\': case '\'': case '/': case '_': case '-':
			case '^': case '~': case '{': case '}':
				break;
			default:
				result += c;
				continue;
		}

		// We peek here for speed reasons.
		next1 = substr(str, i+1, 1);

		// '<' starts a tag and '>' finishes it; '%' both starts and finishes
		// a format string.
		if (c == '<' and not global.inTag) global.inTag = true;
		if (c == '>' and global.inTag) global.inTag = nil;
		if (c == '%') global.inFmt = not global.inFmt;

		// If we're inside a tag or a format string, we shouldn't modify this
		// character.
		if (global.inTag or global.inFmt) {
			result += c;
			continue;
		}

		// If we find a backslash, print both this character and the next one
		// unchanged (apart from the special case of \').
		if (c == '\\') {
			if ( next1 == '\'' and global.convertApostrophes) {
				// If we find an apostrophe, convert it to '&rsquo;'.
				result += '&rsquo;';
				++dropNext;
				continue;
			}
			// Print the escape sequence unchanged.
			result += c;
			result += next1;
			++dropNext;
			continue;
		}

		// If we find an apostrophe, convert it to '&rsquo;'.
		if (c == '\'' and global.convertApostrophes) {
			result += '&rsquo;';
			continue;
		}

		// If we find two slashes, output an <em> or </em> tag.
		if (c+next1 == '//') {
			if (not global.inEmph) {
				result += '<em>';
				global.inEmph = true;
			} else {
				result += '</em>';
				global.inEmph = nil;
			}
			++dropNext;
			continue;
		}

		// If we find two underscores, output a <b> or </b> tag.
		if (c+next1 == '__') {
			if (not global.inBold) {
				result += '<b>';
				global.inBold = true;
			} else {
				result += '</b>';
				global.inBold = nil;
			}
			++dropNext;
			continue;
		}

		// If we find two opening braces, output a <q> tag.
		if (c+next1 == '{{') {
			result += '<q>';
			++dropNext;
			continue;
		}

		// If we find two closing braces, output a </q> tag.
		if (c+next1 == '}}') {
			result += '</q>';
			++dropNext;
			continue;
		}

		// '--' becomes '&endash' and '---' becomes '&emdash'.
		if (c+next1 == '--') {
			// Peek two chars ahead. It's okay, since we don't intend to win
			// a beauty contest.
			if (substr(str, i+2, 2) == '-') {
				result += '&emdash;';
				dropNext += 2;
				continue;
			}
			result += '&endash;';
			++dropNext;
			continue;
		}

		// '^^' is a line break and '~~' is an indent, used for spacing out
		// paragraphs of text. The precise behaviour is controlled by the
		// global variables 'doublespace' and 'indent'.
		if (c+next1 == '^^') {
			if (global.doublespace) {
				// The <P> tag is better than the \b sequence, since multiple
				// <P>s don't generate multiple line breaks (well, they do,
				// but that's a bug in Tads and likely to be fixed at some
				// point in the future).
				result += '<P>';
			} else {
				result += '\n';
			}
			++dropNext;
			continue;
		}

		if (c+next1 == '~~') {
			if (global.indent) {
				result += '\t';
			}
			++dropNext;
			continue;
		}

		// If we didn't find anything special, just output whatever character
		// we found.
		result += c;

		// Note: a 'switch' statement might be faster than all those 'if's,
		// particularly if you add more modifiers, but I decided that using
		// 'switch' would make the code less readable.
		//
		// Update by Niko: I tried it with switch; no speed difference. Maybe
		// it would be faster than all those "if"s if there were more than
		// 300-400 cases.

		// As it is, this filter doesn't slow things down noticeably on any
		// machine on which I've tried it. Your mileage may vary, however,
		// particularly if you're printing out large blocks of text on a slow
		// machine.
		//
		// Update by Niko: Anything slower than a P60 allows you to go make
		// some coffee before seeing any output. On a 386 you have to pick
		// the coffee from Columbia first. And on a 286 you probably will
		// have to plant the coffee and wait for it to grow. The best way to
		// avoid big delays is to print large amounts of text in smaller
		// chunks (100-150 chars at a time should work fine).

		// If you need an order-of-magnitude speedup, you might want to try
		// locating the special characters with find() or a regexp, instead
		// of just reading the string char by char.
	}
	return result;
}


// Now we add some verbs to control the paragraph spacing. These are lifted
// almost directly from Dave Bagget's 'Colossal Cave Revisited'.
//
// Update from Niko: Not anymore. The grammar is different now (ON and OFF
// are used instead of cycling).

spaceOnVerb: sysverb
	verb = 'space on'
	sdesc = "space on"
	action(actor) =
	{
		if (global.doublespace) {
			"\n[Double-spacing text is already turned on.]\n";
		} else {
			"\n[Double-spacing text is now turned on.]\n";
			global.doublespace = true;
		}
		abort;
	}
;

spaceOffVerb: sysverb
	verb = 'space off'
	sdesc = "space off"
	action(actor) =
	{
		if (not global.doublespace) {
			"\n[Double-spacing text is already turned off.]\n";
		} else {
			"\n[Double-spacing text is now turned off.]\n";
			global.doublespace = nil;
			if (not global.indent) {
				// Disabling both indentation and double-spacing is not
				// allowed.
				indentOnVerb.action(parserGetMe());
			}
		}
		abort;
	}
;

spaceVerb: sysverb
	verb = 'space'
	sdesc = "space"
	action(actor) =
	{
		"\n[Double-spacing text is currently turned ";
		global.doublespace ? "on" : "off";
		".]\n";
		abort;
	}
;

indentOnVerb: sysverb
	verb = 'indent on'
	sdesc = "indent on"
	action(actor) =
	{
		if (global.indent) {
			"\n[Paragraph indentation is already turned on.]\n";
		} else {
			"\n[Paragraph indentation is now turned on.]\n";
			global.indent = true;
		}
		abort;
	}
;

indentOffVerb: sysverb
	verb = 'indent off'
	sdesc = "indent off"
	action(actor) =
	{
		if (not global.indent) {
			"\n[Paragraph indentation is already turned off.]\n";
		} else {
			"\n[Paragraph indentation is now turned off.]\n";
			global.indent = nil;
			if (not global.doublespace) {
				// Disabling both indentation and double-spacing is not
				// allowed.
				spaceOnVerb.action(parserGetMe());
			}
		}
		abort;
	}
;

indentVerb: sysverb
	verb = 'indent'
	sdesc = "indent"
	action(actor) =
	{
		"\n[Paragraph indentation is currently turned ";
		global.indent ? "on" : "off";
		".]\n";
		abort;
	}
;

quotesOnVerb: sysverb
	verb = 'quotes on'
	sdesc = "quotes on"
	action(actor) =
	{
		if (global.convertApostrophes) {
			"\n[Apostrophe to single quote conversion is already turned on.]\n";
		} else {
			global.convertApostrophes = true;
			"\n[Apostrophe to single quote conversion is now turned on.]\n";
		}
		abort;
	}
;

quotesOffVerb: sysverb
	verb = 'quotes off'
	sdesc = "quotes off"
	action(actor) =
	{
		if (not global.convertApostrophes) {
			"\n[Apostrophe to single quote conversion is already turned off.]\n";
		} else {
			global.convertApostrophes = nil;
			"\n[Apostrophe to single quote conversion is now turned off.]\n";
		}
		abort;
	}
;

quotesVerb: sysverb
	verb = 'quotes'
	sdesc = "quotes"
	action(actor) =
	{
		"\n[Apostrophe to single quote conversion is currently turned ";
		global.convertApostrophes ? "on" : "off";
		".]\n";
		abort;
	}
;

// 'global.doublespace', 'global.indent' and the others don't actually exist
// yet, of course, but that's easily fixed:
modify global
	doublespace = nil         // Enable/disable double-spacing.
	indent = true             // Enable/disable indentation.
	convertApostrophes = true // Enable/disable "'" to "&rsquo;" conversion.
	skipFilterOnce = nil      // Disable filter for the next one output.

	// We also need some flags to keep track of what we're doing:
	inFmt = nil   // Are we inside a format string?
	inTag = nil   // Are we inside an HTML tag?
	inEmph = nil  // Are we using //emphasis//?
	inBold = nil  // Are we using __bold__?
;

// Call these to enable/disable the filter.
enableFilter: function()
{
	setOutputFilter(&outputFilter);
}

disableFilter: function()
{
	setOutputFilter(nil);
}


/*
 * Library modifications and replacements follow.
 */

replace showcontcont: function(obj)
{
	if (itemcnt(obj.contents)) {
		if (obj.issurface) {
			if (not obj.isqsurface) {
				"^^~~Sitting on "; obj.thedesc;" is "; listcont(obj);
				". ";
			}
		} else if (obj.contentsVisible and not obj.isqcontainer) {
			"^^~~";
			caps();
			obj.thedesc; " seem";
			if (!obj.isThem) "s";
			" to contain ";
			listcont(obj);
			". ";
		}
	}
	if (obj.contentsVisible and not obj.isqcontainer) {
		listfixedcontcont(obj);
	}
}

modify room
	dispParagraph = "^^~~"
	dispBeginLdesc = "\n~~"

	nrmLkAround(verbosity) =
	{
		local l, cur, i, tot;

		if (verbosity) {
			self.dispBeginLdesc;
			self.ldesc;
			self.dispEndLdesc;

			l = self.contents;
			tot = length(l);
			i = 1;
			while (i <= tot) {
				cur = l[i];
				if (cur.isfixed) {
					// Since this property is never created automaticly by
					// Tads, there's no need to print a paragraph seperator
					// for it.
					cur.heredesc;
				}
				++i;
			}
		}

		if (itemcnt(self.contents)) {
			self.dispParagraph;
			"%You% see%s% "; listcont(self); " here. ";
		}
		listcontcont(self); "\n";

		l = self.contents;
		tot = length(l);
		i = 1;
		while (i <= tot) {
			cur = l[i];
			if (cur.isactor) {
				if (cur <> parserGetMe()) {
					// Don't print a "^^~~" sequence if you provide your own
					// actorDesc property. This is done automaticly here
					// (because this property has a non-empty default value).
					self.dispParagraph;
					cur.actorDesc;
				}
			}
			++i;
		}
		"\n";
	}
;

