/*
 * Copyright (c) 1982, 1990 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	from: @(#)if_le.c	7.6 (Berkeley) 5/8/91
 *	from: if_le.c,v 1.6 1994/06/21 04:02:15 chopps Exp $
 *	Id: if_ae.c,v 1.3 1995/4/23 19:57:53 crest/ill
 *
 *	This is based on the original LANCE files as the PCnet-ISA used on
 *	the Ariadne is a LANCE-descendant optimized for the (guess it?)
 *	PC-ISA bus. This causes some modifications, all data that is to go
 *	into registers or to structures (buffer-describtors, init-block)
 *	has to be byte-swapped. In addition ALL write accesses to the board
 *	have to be WORD or LONG, BYTE-access is prohibited!!
 */

#include "ae.h"
#if NAE > 0

#include "bpfilter.h"

/*
 * AMD 79960 PCnet-ISA
 *
 * This driver will generate and accept tailer encapsulated packets even
 * though it buys us nothing.  The motivation was to avoid incompatibilities
 * with VAXen, SUNs, and others that handle and benefit from them.
 * This reasoning is dubious.
 */
#include <sys/param.h>
#include <sys/systm.h>
#include <sys/mbuf.h>
#include <sys/buf.h>
#include <sys/protosw.h>
#include <sys/socket.h>
#include <sys/syslog.h>
#include <sys/ioctl.h>
#include <sys/errno.h>
#include <sys/device.h>

#include <net/if.h>
#include <net/netisr.h>
#include <net/route.h>

#ifdef INET
#include <netinet/in.h>
#include <netinet/in_systm.h>
#include <netinet/in_var.h>
#include <netinet/ip.h>
#include <netinet/if_ether.h>
#endif

#ifdef NS
#include <netns/ns.h>
#include <netns/ns_if.h>
#endif

#include <machine/cpu.h>
#include <machine/mtpr.h>
#include <amiga/amiga/device.h>
#include <amiga/amiga/isr.h>
#include <amiga/dev/zbusvar.h>
#include <amiga/dev/if_aereg.h>

/*
 * Ethernet software status per interface.
 *
 * Each interface is referenced by a network interface structure,
 * ae_if, which the routing code uses to locate the interface.
 * This structure contains the output queue for the interface, its address, ...
 */
struct	ae_softc {
	struct	isr sc_isr;
	struct	arpcom sc_ac;	/* common Ethernet structures */
#define	sc_if	sc_ac.ac_if	/* network-visible interface */
#define	sc_addr	sc_ac.ac_enaddr	/* hardware Ethernet address */
	void	*sc_base;	/* base address of board */
	struct	aereg1 *sc_r1;	/* LANCE registers */
	struct	aereg2 *sc_r2;	/* dual-port RAM */
	int	sc_rmd;		/* predicted next rmd to process */
	int	sc_tmd;		/* next tmd to use */
	int	sc_no_td;	/* number of tmds in use */
	int	sc_runt;
	int	sc_jab;
	int	sc_merr;
	int	sc_babl;
	int	sc_cerr;
	int	sc_miss;
	int	sc_xint;
	int	sc_xown;
	int	sc_uflo;
	int	sc_rxlen;
	int	sc_rxoff;
	int	sc_txoff;
	int	sc_busy;
	short	sc_iflags;
#if NBPFILTER > 0
	caddr_t sc_bpf;
#endif
} ae_softc[NAE];

#if NBPFILTER > 0
#include <net/bpf.h>
#include <net/bpfdesc.h>
#endif

/* offsets for:	   ID,   REGS,    MEM */
int	aestd[] = { 0, 0x0370, 0x8000 };

/* console error messages */
int	aedebug = 0;

int	aeintr(), aestart(), aeioctl(), ether_output();
void	aeinit();

struct	mbuf *aeget();
extern	struct ifnet loif;

void aeattach __P((struct device *, struct device *, void *));
int aematch __P((struct device *, struct cfdata *, void *args));

struct cfdriver aecd = {
	NULL, "ae", (cfmatch_t)aematch, aeattach, DV_IFNET,
	sizeof(struct ae_softc), NULL, 0};

int
aematch(pdp, cfp, auxp)
	struct device *pdp;
	struct cfdata *cfp;
	void *auxp;
{
	struct zbus_args *zap;

	zap = (struct zbus_args *)auxp;

	/* Ariadne ethernet card */
	if ( zap->manid == 2167 && zap->prodid == 201)
		return(1);

	return (0);
}

/*
 * Interface exists: make available by filling in network interface
 * record.  System will initialize the interface when it is ready
 * to accept packets.
 */
void
aeattach(pdp, dp, auxp)
	struct device *pdp, *dp;
	void *auxp;
{
	register struct aereg0 *aer0;
	register struct aereg2 *aer2;
	struct zbus_args *zap;
	struct aereg2 *aemem = (struct aereg2 *) 0x8000;
	struct ae_softc *ae = &ae_softc[dp->dv_unit];
	struct ifnet *ifp = &ae->sc_if;
	char *cp;
	int i;
	unsigned long ser;
	int s = splhigh ();

	zap =(struct zbus_args *)auxp;

	/*
	 * Make config msgs look nicer.
	 */
	printf("\n");

	aer0 = ae->sc_base = zap->va;
	ae->sc_r1 = (struct aereg1 *)(aestd[1] + (int)zap->va);
	aer2 = ae->sc_r2 = (struct aereg2 *)(aestd[2] + (int)zap->va);

	/*
	 * Serial number for board is used as host ID.
	 */
	ser = (unsigned long) zap->serno;

	/*
	 * Manufacturer decides the 3 first bytes, i.e. ethernet vendor ID.
	 */

	/*
	 * currently borrowed from C= 
	 *
         * the next four lines will soon have to be altered 
         */

	ae->sc_addr[0] = 0x00;
	ae->sc_addr[1] = 0x80;
	ae->sc_addr[2] = 0x10;

	ae->sc_addr[3] = ((ser >> 16) & 0x0f) | 0xf0; /* to diff from A2065 */
	ae->sc_addr[4] = (ser >>  8 ) & 0xff;
	ae->sc_addr[5] = (ser       ) & 0xff;

	printf("ae%d: hardware address %s 32K\n", dp->dv_unit,
		ether_sprintf(ae->sc_addr));

	/*
	 * Setup for transmit/receive
	 */
	aer2->aer2_mode = AE_MODE;

	/* you know: no BYTE access.... */
	aer2->aer2_padr[0] = (ae->sc_addr[0] << 8) | ae->sc_addr[1];
	aer2->aer2_padr[1] = (ae->sc_addr[2] << 8) | ae->sc_addr[3];
	aer2->aer2_padr[2] = (ae->sc_addr[4] << 8) | ae->sc_addr[5];

	aer2->aer2_ladrf0 = 0;
	aer2->aer2_ladrf1 = 0;
	aer2->aer2_rlen = SWAP(AE_RLEN);
	aer2->aer2_rdra = SWAP((int)aemem->aer2_rmd);
	aer2->aer2_tlen = SWAP(AE_TLEN);
	aer2->aer2_tdra = SWAP((int)aemem->aer2_tmd);

	splx (s);

	ifp->if_unit = dp->dv_unit;
	ifp->if_name = "ae";
	ifp->if_mtu = ETHERMTU;
	ifp->if_ioctl = aeioctl;
	ifp->if_output = ether_output;
	ifp->if_start = aestart;
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX;

#if NBPFILTER > 0
	bpfattach(&ae->sc_bpf, ifp, DLT_EN10MB, sizeof(struct ether_header));
#endif
	if_attach(ifp);
	ether_ifattach(ifp);

	ae->sc_isr.isr_intr = aeintr;
	ae->sc_isr.isr_arg = ae;
	ae->sc_isr.isr_ipl = 2;
	add_isr (&ae->sc_isr);
	return;
}

aedrinit(aer2)
	register struct aereg2 *aer2;
{
	register struct aereg2 *aemem = (struct aereg2 *) 0x8000;
	register int i;

	for (i = 0; i < AERBUF; i++) {
		aer2->aer2_rmd[i].rmd0 = SWAP((int)aemem->aer2_rbuf[i]);
		aer2->aer2_rmd[i].rmd1 = AE_OWN;
		aer2->aer2_rmd[i].rmd2 = SWAP(-AEMTU);
		aer2->aer2_rmd[i].rmd3 = 0;
	}

	for (i = 0; i < AETBUF; i++) {
		aer2->aer2_tmd[i].tmd0 = SWAP((int)aemem->aer2_tbuf[i]);
		aer2->aer2_tmd[i].tmd1 = 0;
		aer2->aer2_tmd[i].tmd2 = 0;
		aer2->aer2_tmd[i].tmd3 = 0;
	}
}

void
aereset(unit)
	register int unit;
{
	register struct ae_softc *ae = &ae_softc[unit];
	register struct aereg1 *aer1 = ae->sc_r1;
	/*
	 * This structure is referenced from the CARD's/PCnet-ISA's point
	 * of view, thus the 0x8000 address which is the buffer RAM area
	 * of the Ariadne card. This pointer is manipulated
	 * with the PCnet-ISA's view of memory and NOT the Amiga's. FYI.
	 */
	register struct aereg2 *aemem = (struct aereg2 *) 0x8000;

	register int timo = 100000;
	register int stat;
	volatile register int dummy;

	dummy = aer1->aer1_reset;    /* Reset PCNet-ISA */

#ifdef lint
	stat = unit;
#endif
#if NBPFILTER > 0
	if (ae->sc_if.if_flags & IFF_PROMISC)
		/* set the promiscuous bit */
		ae->sc_r2->aer2_mode = AE_MODE|AE_PROM;
	else
		ae->sc_r2->aer2_mode = AE_MODE;
#endif
	 aer1->aer1_rap =  AE_CSR0;
	 aer1->aer1_rdp =  AE_STOP;

	 aedrinit(ae->sc_r2);

	 ae->sc_rmd = 0;
	 aer1->aer1_rap =  AE_CSR1;
	 aer1->aer1_rdp =  SWAP((int)&aemem->aer2_mode);
	 aer1->aer1_rap =  AE_CSR2;
	 aer1->aer1_rdp =  0;
	 aer1->aer1_rap =  AE_CSR0;
	 aer1->aer1_rdp =  AE_INIT;

	do {
		if (--timo == 0) {
			printf("ae%d: init timeout, stat = 0x%x\n",
			       unit, stat);
			break;
		}
		 stat =  aer1->aer1_rdp;
	} while ((stat & AE_IDON) == 0);

	 aer1->aer1_rdp =  AE_STOP;

/*
 *	re-program LEDs to match meaning used on the Ariadne board
 */
	 aer1->aer1_rap = SWAP(5);
	 aer1->aer1_idp = SWAP(0x90);
	 aer1->aer1_rap = SWAP(6);
	 aer1->aer1_idp = SWAP(0x81);
	 aer1->aer1_rap = SWAP(7);
	 aer1->aer1_idp = SWAP(0x84);

/*
 *	this autoselects between 10Base-T and 10Base2
 *	if an active 10Base-T line is connected then 10Base-T 
 *	is used otherwise 10Base2
 */

	 aer1->aer1_rap = SWAP(2);
	 aer1->aer1_idp = SWAP(0x02);

	 aer1->aer1_rap =  AE_CSR0;
	 aer1->aer1_rdp =  AE_STRT | AE_INEA;

	 ae->sc_if.if_flags &= ~IFF_OACTIVE;

	return;
}

/*
 * Initialization of interface
 */
void
aeinit(unit)
	int unit;
{
	struct ae_softc *ae = &ae_softc[unit];
	register struct ifnet *ifp = &ae->sc_if;
	int s;

	/* not yet, if address still unknown */
	if (ifp->if_addrlist == (struct ifaddr *)0)
		return;

	if ((ifp->if_flags & IFF_RUNNING) == 0) {
		s = splimp();
		ifp->if_flags |= IFF_RUNNING;
		aereset(unit);
	        (void) aestart(ifp);
		splx(s);
	}

	return;
}

#define AENEXTTMP \
	if (++bix == AETBUF) bix = 0, tmd = ae->sc_r2->aer2_tmd; else ++tmd

/*
 * Start output on interface.  Get another datagram to send
 * off of the interface queue, and copy it to the interface
 * before starting the output.
 */
aestart(ifp)
	struct ifnet *ifp;
{
	register struct ae_softc *ae = &ae_softc[ifp->if_unit];
	register int bix;
	register struct aetmd *tmd;
	register struct mbuf *m;
	int len;

	if ((ae->sc_if.if_flags & IFF_RUNNING) == 0)
		return (0);

	bix = ae->sc_tmd;
	tmd = &ae->sc_r2->aer2_tmd[bix];

	for (;;) {
		if (ae->sc_no_td >= AETBUF) {
			ae->sc_if.if_flags |= IFF_OACTIVE;
			break;
		}

		IF_DEQUEUE(&ae->sc_if.if_snd, m);
		if (m == 0)
			break;

		++ae->sc_no_td;

		len = aeput(ae->sc_r2->aer2_tbuf[bix], m);

#if NBPFILTER > 0
		/*
		 * If bpf is listening on this interface, let it
		 * see the packet before we commit it to the wire.
		 */
		if (ae->sc_bpf)
        	        bpf_tap(ae->sc_bpf, ae->sc_r2->aer2_tbuf[bix], len);
#endif

		tmd->tmd3 = 0;
		tmd->tmd2 = SWAP(-len);
		tmd->tmd1 = AE_OWN | AE_STP | AE_ENP;

		AENEXTTMP;
	}

	ae->sc_tmd = bix;

	return (0);
}

aeintr(ae)
	struct ae_softc *ae;
{
#if 0
	register struct ae_softc *ae = &ae_softc[unit];
#else
	int unit = ae->sc_if.if_unit;
#endif
	register struct aereg1 *aer1;
	register int stat;

	/* if not even initialized, don't do anything further.. */
	if (! ae->sc_base)
	  return 0;

	aer1 = ae->sc_r1;
	stat =  aer1->aer1_rdp;

	if (! (stat & AE_INTR))
	  return 0;

	if (stat & AE_SERR) {
		aeerror(unit, stat);
		if (stat & AE_MERR) {
			ae->sc_merr++;
			aereset(unit);
			return(1);
		}
		if (stat & AE_BABL)
			ae->sc_babl++;
		if (stat & AE_CERR)
			ae->sc_cerr++;
		if (stat & AE_MISS)
			ae->sc_miss++;
		 aer1->aer1_rdp =  AE_BABL|AE_CERR|AE_MISS|AE_INEA;
	}
	if ((stat & AE_RXON) == 0) {
		ae->sc_rxoff++;
		aereset(unit);
		return(1);
	}
	if ((stat & AE_TXON) == 0) {
		ae->sc_txoff++;
		aereset(unit);
		return(1);
	}
	if (stat & AE_RINT) {
		/* interrupt is cleared in aerint */
		aerint(unit);
	}
	if (stat & AE_TINT) {
		aer1->aer1_rdp =  AE_TINT|AE_INEA;
		aexint(unit);
	}
	return(1);
}

/*
 * Ethernet interface transmitter interrupt.
 * Start another output if more data to send.
 */
aexint(unit)
	register int unit;
{
	register struct ae_softc *ae = &ae_softc[unit];
	register int bix = (ae->sc_tmd - ae->sc_no_td + AETBUF) % AETBUF;
	register struct aetmd *tmd = &ae->sc_r2->aer2_tmd[bix];

	if ((ae->sc_if.if_flags & IFF_OACTIVE) == 0) {
		ae->sc_xint++;
		return;
	}
	if (tmd->tmd1 & AE_OWN) {
printf("ae%d: extra xint\n", unit);
		ae->sc_xown++;
		return;
	}
	ae->sc_if.if_flags &= ~IFF_OACTIVE;

	do {
		if (ae->sc_no_td <= 0)
			break;
		--ae->sc_no_td;

		if (tmd->tmd1 & AE_ERR) {
err:
printf("ae%d: xint error\n", unit);
			aexerror(unit);
			ae->sc_if.if_oerrors++;
			if (tmd->tmd3 & (AE_TBUFF|AE_UFLO)) {
				ae->sc_uflo++;
				aereset(unit);
				return;
			}
			else if (tmd->tmd3 & AE_LCOL)
				ae->sc_if.if_collisions++;
			else if (tmd->tmd3 & AE_RTRY)
				ae->sc_if.if_collisions += 16;
		}
		else if (tmd->tmd3 & AE_TBUFF)
			/* XXX documentation says BUFF not included in ERR */
			goto err;
		else if (tmd->tmd1 & AE_ONE)
			ae->sc_if.if_collisions++;
		else if (tmd->tmd1 & AE_MORE)
			/* what is the real number? */
			ae->sc_if.if_collisions += 2;
		else
			ae->sc_if.if_opackets++;
		AENEXTTMP;
	} while ((tmd->tmd1 & AE_OWN) == 0);

	(void) aestart(&ae->sc_if);
}

#define	AENEXTRMP \
	if (++bix == AERBUF) bix = 0, rmd = ae->sc_r2->aer2_rmd; else ++rmd

/*
 * Ethernet interface receiver interrupt.
 * If input error just drop packet.
 * Decapsulate packet based on type and pass to type specific
 * higher-level input routine.
 */
aerint(unit)
	int unit;
{
	register struct ae_softc *ae = &ae_softc[unit];
	register int bix = ae->sc_rmd;
	register struct aermd *rmd = &ae->sc_r2->aer2_rmd[bix];

	/*
	 * Out of sync with hardware, should never happen?
	 */
	if (rmd->rmd1 & AE_OWN) {
		 ae->sc_r1->aer1_rdp =  AE_RINT|AE_INEA;
		return;
	}

	/*
	 * Process all buffers with valid data
	 */
	while ((rmd->rmd1 & AE_OWN) == 0) {
		int len = SWAP(rmd->rmd3);

		/* Clear interrupt to avoid race condition */
		 ae->sc_r1->aer1_rdp =  AE_RINT|AE_INEA;

		if (rmd->rmd1 & AE_ERR) {
			ae->sc_rmd = bix;
			aererror(unit, "bad packet");
			ae->sc_if.if_ierrors++;
		} else if ((rmd->rmd1 & (AE_STP|AE_ENP)) != (AE_STP|AE_ENP)) {
			/*
			 * Find the end of the packet so we can see how long
			 * it was.  We still throw it away.
			 */
			do {
			        ae->sc_r1->aer1_rdp = AE_RINT|AE_INEA;
				rmd->rmd3 = 0;
				rmd->rmd1 = AE_OWN;
				AENEXTRMP;
			} while (!(rmd->rmd1 & (AE_OWN|AE_ERR|AE_STP|AE_ENP)));

			ae->sc_rmd = bix;
			aererror(unit, "chained buffer");
			ae->sc_rxlen++;

			/*
			 * If search terminated without successful completion
			 * we reset the hardware (conservative).
			 */
			if ((rmd->rmd1 & (AE_OWN|AE_ERR|AE_STP|AE_ENP)) != AE_ENP) {
				aereset(unit);
				return;
			}
		} else
			aeread(unit, ae->sc_r2->aer2_rbuf[bix], len);

		rmd->rmd3 = 0;
		rmd->rmd1 = AE_OWN;
		AENEXTRMP;

	}

	ae->sc_rmd = bix;
}

aeread(unit, buf, len)
	int unit;
	char *buf;
	int len;
{
	register struct ae_softc *ae = &ae_softc[unit];
	register struct ether_header *et;
    	struct mbuf *m;
	int off, resid;

	ae->sc_if.if_ipackets++;

	et = (struct ether_header *)buf;
	et->ether_type = ntohs((u_short)et->ether_type);

	/* adjust input length to account for header and CRC */
	len = len - sizeof(struct ether_header) - 4;

#define	aedataaddr(et, off, type)	((type)(((caddr_t)((et)+1)+(off))))
	if (et->ether_type >= ETHERTYPE_TRAIL &&
	    et->ether_type < ETHERTYPE_TRAIL+ETHERTYPE_NTRAILER) {
		off = (et->ether_type - ETHERTYPE_TRAIL) * 512;
		if (off >= ETHERMTU)
			return;		/* sanity */
		et->ether_type = ntohs(*aedataaddr(et, off, u_short *));
		resid = ntohs(*(aedataaddr(et, off+2, u_short *)));
		if (off + resid > len)
			return;		/* sanity */
		len = off + resid;
	} else
		off = 0;

	if (len <= 0) {
		if (aedebug)
			log(LOG_WARNING,
			    "ae%d: ierror(runt packet): from %s: len=%d\n",
			    unit, ether_sprintf(et->ether_shost), len);
		ae->sc_runt++;
		ae->sc_if.if_ierrors++;
		return;
	}
#if NBPFILTER > 0
	/*
	 * Check if there's a bpf filter listening on this interface.
	 * If so, hand off the raw packet to bpf, which must deal with
	 * trailers in its own way.
	 */
	if (ae->sc_bpf) {
		bpf_tap(ae->sc_bpf, buf, len + sizeof(struct ether_header));

		/*
		 * Note that the interface cannot be in promiscuous mode if
		 * there are no bpf listeners.  And if we are in promiscuous
		 * mode, we have to check if this packet is really ours.
		 *
		 * XXX This test does not support multicasts.
		 */
		if ((ae->sc_if.if_flags & IFF_PROMISC)
		    && bcmp(et->ether_dhost, ae->sc_addr,
			    sizeof(et->ether_dhost)) != 0
		    && bcmp(et->ether_dhost, etherbroadcastaddr,
			    sizeof(et->ether_dhost)) != 0)
			return;
	}
#endif
	/*
	 * Pull packet off interface.  Off is nonzero if packet
	 * has trailing header; aeget will then force this header
	 * information to be at the front, but we still have to drop
	 * the type and length which are at the front of any trailer data.
	 */
	m = aeget(buf, len, off, &ae->sc_if);
	if (m == 0)
		return;

	ether_input(&ae->sc_if, et, m);
}

/*
 *	here come the two replacements for bcopy() and bzero() as
 *	WORD-access is absolutely required!
 *	They could use some tuning as this is time-critical (copying
 *	packet-data) and should be processed as fast as possible.
 */

static void wcopy(char *a1, char *a2, int length) /* bcopy() word-wise */
{
 ushort i, *b1, *b2;

 b1 = (ushort *)a1;
 b2 = (ushort *)a2;

 i = length / 2;

 while(i--)
  *b2++ = *b1++;

 if (length & 0x0001)
 {
  i = (*b2 & 0x00ff) | (*b1 & 0xff00);
  *b2 = i;
 }
};

static void wzero(char *a1, int length) /* bzero() word-wise */
{
 ushort i, *b1;

 b1 = (ushort *)a1;

 i = length / 2;

 while(i--)
  *b1++ = 0;

 if (length & 0x0001)
  *b1 &= 0x00ff;
};

/*
 * Routine to copy from mbuf chain to transmit
 * buffer in board local memory.
 */
aeput(aebuf, m)
	register char *aebuf;
	register struct mbuf *m;
{
	register struct mbuf *mp;
	register int len, tlen = 0;

	for (mp = m; mp; mp = mp->m_next) {
		len = mp->m_len;
		if (len == 0)
			continue;
		tlen += len;
		wcopy(mtod(mp, char *), aebuf, len);
		aebuf += len;
	}

	m_freem(m);

	if (tlen < AEMINSIZE) {
		wzero(aebuf, AEMINSIZE - tlen);
		tlen = AEMINSIZE;
	}

	return(tlen);
}

/*
 * Routine to copy from board local memory into mbufs.
 */
struct mbuf *
aeget(aebuf, totlen, off0, ifp)
	char *aebuf;
	int totlen, off0;
	struct ifnet *ifp;
{
	register struct mbuf *m;
	struct mbuf *top = 0, **mp = &top;
	register int off = off0, len;
	register char *cp;
	char *epkt;

	aebuf += sizeof (struct ether_header);
	cp = aebuf;
	epkt = cp + totlen;
	if (off) {
		cp += off + 2 * sizeof(u_short);
		totlen -= 2 * sizeof(u_short);
	}

	MGETHDR(m, M_DONTWAIT, MT_DATA);

	if (m == 0)
		return (0);

	m->m_pkthdr.rcvif = ifp;
	m->m_pkthdr.len = totlen;
	m->m_len = MHLEN;

	while (totlen > 0) {

		if (top) {
			MGET(m, M_DONTWAIT, MT_DATA);
			if (m == 0) {
				m_freem(top);
				return (0);
			}
			m->m_len = MLEN;
		}

		len = min(totlen, epkt - cp);
		if (len >= MINCLSIZE) {
			MCLGET(m, M_DONTWAIT);
			if (m->m_flags & M_EXT)
				m->m_len = len = min(len, MCLBYTES);
			else
				len = m->m_len;
		} else {
			/*
			 * Place initial small packet/header at end of mbuf.
			 */
			if (len < m->m_len) {
				if (top == 0 && len + max_linkhdr <= m->m_len)
					m->m_data += max_linkhdr;
				m->m_len = len;
			} else
				len = m->m_len;
		}

		wcopy(cp, mtod(m, caddr_t), (unsigned)len);
		cp += len;
		*mp = m;
		mp = &m->m_next;
		totlen -= len;

		if (cp == epkt)
			cp = aebuf;

	}

	return (top);
}

/*
 * Process an ioctl request.
 */
aeioctl(ifp, cmd, data)
	register struct ifnet *ifp;
	int cmd;
	caddr_t data;
{
	register struct ifaddr *ifa = (struct ifaddr *)data;
	struct ae_softc *ae = &ae_softc[ifp->if_unit];
	struct aereg1 *aer1 = ae->sc_r1;
	int s = splimp(), error = 0;

	switch (cmd) {

	case SIOCSIFADDR:
		ifp->if_flags |= IFF_UP;
		switch (ifa->ifa_addr->sa_family) {
#ifdef INET
		case AF_INET:
			aeinit(ifp->if_unit);	/* before arpwhohas */
			((struct arpcom *)ifp)->ac_ipaddr =
				IA_SIN(ifa)->sin_addr;
			arpwhohas((struct arpcom *)ifp, &IA_SIN(ifa)->sin_addr);
			break;
#endif
#ifdef NS
		case AF_NS:
		    {
			register struct ns_addr *ina = &(IA_SNS(ifa)->sns_addr);

			if (ns_nullhost(*ina))
				ina->x_host = *(union ns_host *)(ae->sc_addr);
			else {
				/*
				 * The manual says we can't change the address
				 * while the receiver is armed,
				 * so reset everything
				 */
				ifp->if_flags &= ~IFF_RUNNING;
				wcopy((caddr_t)ina->x_host.c_host,
				    (caddr_t)ae->sc_addr, sizeof(ae->sc_addr));
			}
			aeinit(ifp->if_unit); /* does ae_setaddr() */
			break;
		    }
#endif
		default:
			aeinit(ifp->if_unit);
			break;
		}
		break;

	case SIOCSIFFLAGS:
		if ((ifp->if_flags & IFF_UP) == 0 &&
		    ifp->if_flags & IFF_RUNNING) {
			 aer1->aer1_rdp =  AE_STOP;
			ifp->if_flags &= ~IFF_RUNNING;
		} else if (ifp->if_flags & IFF_UP && (ifp->if_flags & IFF_RUNNING) == 0)
			aeinit(ifp->if_unit);

		/*
		 * If the state of the promiscuous bit changes, the interface
		 * must be reset to effect the change.
		 */
		if (((ifp->if_flags ^ ae->sc_iflags) & IFF_PROMISC) && (ifp->if_flags & IFF_RUNNING)) {
			ae->sc_iflags = ifp->if_flags;
			aereset(ifp->if_unit);
			(void)aestart(ifp);
		}
		break;

	default:
		error = EINVAL;
	}

	splx(s);
	return (error);
}

aeerror(unit, stat)
	int unit;
	int stat;
{
	if (!aedebug)
		return;

	/*
	 * Not all transceivers implement heartbeat
	 * so we only log CERR once.
	 */
	if ((stat & AE_CERR) && ae_softc[unit].sc_cerr)
		return;

	log(LOG_WARNING,
	    "ae%d: error: stat=%b\n", unit,
	    SWAP(stat),
	    "\20\20ERR\17BABL\16CERR\15MISS\14MERR\13RINT\12TINT\11IDON\10INTR\07INEA\06RXON\05TXON\04TDMD\03STOP\02STRT\01INIT");
}

aererror(unit, msg)
	int unit;
	char *msg;
{
	register struct ae_softc *ae = &ae_softc[unit];
	register struct aermd *rmd;
	int len;

	if (!aedebug)
		return;

	rmd = &ae->sc_r2->aer2_rmd[ae->sc_rmd];
	len = SWAP(rmd->rmd3);

	log(LOG_WARNING,
	    "ae%d: ierror(%s): from %s: buf=%d, len=%d, rmd1=%b\n",
	    unit, msg,
	    len > 11 ? ether_sprintf(&ae->sc_r2->aer2_rbuf[ae->sc_rmd][6]) : "unknown",
	    ae->sc_rmd, len,
	    SWAP(rmd->rmd1),
	    "\20\20OWN\17ERR\16FRAM\15OFLO\14CRC\13RBUF\12STP\11ENP");
}

aexerror(unit)
	int unit;
{
	register struct ae_softc *ae = &ae_softc[unit];
	int bix;
	register struct aetmd *tmd;
	int len;

	if (!aedebug)
		return;

	bix = (ae->sc_tmd - ae->sc_no_td + AETBUF) % AETBUF;
	tmd = &ae->sc_r2->aer2_tmd[bix];
	len = SWAP(-tmd->tmd2);

	log(LOG_WARNING,
	    "ae%d: oerror: to %s: buf=%d, len=%d, tmd1=%b, tmd3=%b\n",
	    unit,
	    len > 5 ? ether_sprintf(&ae->sc_r2->aer2_tbuf[0][0]) : "unknown",
	    0, len,
	    SWAP(tmd->tmd1),
	    "\20\20OWN\17ERR\16RES\15MORE\14ONE\13DEF\12STP\11ENP",
	    SWAP(tmd->tmd3),
	    "\20\20BUFF\17UFLO\16RES\15LCOL\14LCAR\13RTRY");
}

#endif

