/* gtkcombo - combo widget for gtk+
 * Copyright 1997 Paolo Molaro
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include "gtkcombo.h"
#include "gdk/gdkkeysyms.h"

#define COMBO_LIST_MAX_HEIGHT 300

static void gtk_combo_class_init (GtkComboClass *klass);
static void gtk_combo_init (GtkCombo *combo);
static void gtk_combo_destroy(GtkObject* combo);
static int gtk_combo_entry_key_press(GtkEntry * entry, GdkEventKey * event, GtkCombo *combo);
static GtkListItem* gtk_combo_find (GtkCombo* combo);
static gchar* gtk_combo_func (GtkListItem* li);
static gint gtk_combo_focus_idle(GtkCombo* combo);
static gint gtk_combo_entry_focus_out (GtkEntry* entry, GdkEventFocus* event, GtkCombo* combo);
static void gtk_combo_get_pos(GtkCombo* combo, gint* x, gint* y, gint* height, gint* width);
static void gtk_combo_popup_list (GtkButton* button, GtkCombo *combo);
static void gtk_combo_update_entry(GtkList* list, GtkCombo* combo);
static void gtk_combo_update_list(GtkEntry* entry, GtkCombo* combo);
static int gtk_combo_list_key_press(GtkWidget * widget, GdkEventKey * event, GtkCombo *combo);

static GtkHBoxClass* parent_class = NULL;

void
gtk_combo_class_init (GtkComboClass *klass) {
	GtkObjectClass* oclass;

	parent_class = gtk_type_class(gtk_hbox_get_type());
	oclass = (GtkObjectClass*)klass;

	oclass->destroy = gtk_combo_destroy;
}

static void 
gtk_combo_destroy(GtkObject* combo) {

	gtk_object_unref(GTK_OBJECT(GTK_COMBO(combo)->popwin));

	if ( GTK_OBJECT_CLASS(parent_class)->destroy )
		(*GTK_OBJECT_CLASS(parent_class)->destroy)(combo);
}

static int
gtk_combo_entry_key_press(GtkEntry * entry, GdkEventKey * event, GtkCombo *combo) {
	/* completion? */
	/*if ( event->keyval == GDK_Tab ) {
		gtk_signal_emit_stop_by_name (GTK_OBJECT (entry), "key_press_event");
		return TRUE;
	}*/
	return FALSE;
}

static GtkListItem*
gtk_combo_find (GtkCombo* combo) {
	gchar* text;
	gchar* ltext;
	GList* clist;

	text = gtk_entry_get_text(GTK_ENTRY(combo->entry));
	clist = GTK_LIST(combo->list)->children;

	if ( !combo->func )
		return NULL;

	while (clist && clist->data) {
		ltext = (*combo->func)(GTK_WIDGET(clist->data));
		if ( !ltext )
			continue;
		if ( !strcmp(ltext, text) )
			return (GtkListItem*)clist->data;
		clist = clist->next;
	}

	return NULL;
}

static gchar*
gtk_combo_func (GtkListItem* li) {
	GtkWidget* label;
	gchar* ltext = NULL;

	label = GTK_BIN(li)->child;
	if ( !label || !GTK_IS_LABEL(label) )
		return NULL;
	gtk_label_get(GTK_LABEL(label), &ltext);
	return ltext;
}

static gint
gtk_combo_focus_idle(GtkCombo* combo) {
	if ( combo )
		gtk_widget_grab_focus(combo->entry);
	return FALSE;
}

static gint
gtk_combo_entry_focus_out (GtkEntry* entry, GdkEventFocus* event, GtkCombo* combo) {

	if ( combo->flag == GTK_COMBO_IN_LIST_VALUE && !gtk_combo_find(combo) ) {
		/* gdk_beep(); */ /* this can be annoying */
#ifdef TEST
		printf("INVALID ENTRY: `%s'\n", gtk_entry_get_text(entry));
#endif
		gtk_grab_add(GTK_WIDGET(combo));
		gtk_idle_add((GtkFunction)gtk_combo_focus_idle, combo);
		return TRUE;
	}
	return FALSE;
}

static void
gtk_combo_get_pos(GtkCombo* combo, gint* x, gint* y, gint* height, gint* width) {
	GtkAllocation *pos = &(GTK_WIDGET(combo->entry)->allocation);
	GtkRequisition req1;
	GtkRequisition req2;
	GtkRequisition req3;

	gtk_widget_size_request(combo->popup, &req1);
	gtk_widget_size_request(GTK_SCROLLED_WINDOW(combo->popup)->hscrollbar, &req2);
	gtk_widget_size_request(combo->list, &req3);
	*height = req1.height-req2.height+req3.height; /* it's a pain */

	gdk_window_get_origin(GTK_WIDGET(combo->entry)->window, x, y);
	*y += pos->height;

	*height = MIN(COMBO_LIST_MAX_HEIGHT, *height);
	*height = MIN(*height, gdk_screen_height() - *y);
	*width = pos->width;  /* this is wrong when it's resized: who knows why? */
#ifdef TEST
	printf("popup: req_height=(%d, %d, %d) x=%d y=%d width=%d height=%d\n", req1.height, req2.height, req3.height, *x, *y, *width, *height);
#endif
}

static void
gtk_combo_popup_list (GtkButton* button, GtkCombo *combo) {
	gint height, width, x, y;

	gtk_combo_get_pos(combo, &x, &y, &height, &width);

	gtk_widget_set_usize(combo->popwin, width, height);
	gtk_widget_set_uposition(combo->popwin, x, y);
	gtk_widget_show(combo->popwin);
	gtk_widget_grab_focus(combo->popwin);
	gtk_grab_add(combo->popwin);
	gdk_pointer_grab(combo->popwin->window, TRUE, 0, NULL, NULL, GDK_CURRENT_TIME);
}

static void
gtk_combo_update_entry(GtkList* list, GtkCombo* combo) {
	char* text;

	gtk_grab_remove(GTK_WIDGET(combo));
	gtk_signal_handler_block(GTK_OBJECT(list), combo->list_change_id);
	if ( list->selection ) {
		text = (*combo->func)(list->selection->data);
		if ( !text )
			text = "";
		gtk_entry_set_text(GTK_ENTRY(combo->entry), text);
	}
	gtk_widget_hide(combo->popwin);
	gtk_grab_remove(combo->popwin);
	gdk_pointer_ungrab(GDK_CURRENT_TIME);
	gtk_signal_handler_unblock(GTK_OBJECT(list), combo->list_change_id);
}

static void
gtk_combo_update_list(GtkEntry* entry, GtkCombo* combo) {
	GtkList * list = GTK_LIST(combo->list);
	GList* slist = list->selection;
	GtkListItem* li;

	gtk_grab_remove(GTK_WIDGET(combo));

	gtk_signal_handler_block(GTK_OBJECT(entry), combo->entry_change_id);
	if ( slist && slist->data )
		gtk_list_unselect_child(list, GTK_WIDGET(slist->data));
	li = gtk_combo_find(combo);
	if ( li )
		gtk_list_select_child(list, GTK_WIDGET(li));
	gtk_signal_handler_unblock(GTK_OBJECT(entry), combo->entry_change_id);
}

static int
gtk_combo_list_key_press(GtkWidget * widget, GdkEventKey * event, GtkCombo *combo) {
	if ( event->keyval == GDK_Escape ) {
		gtk_widget_hide(combo->popwin);
		gtk_grab_remove(combo->popwin);
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		return TRUE;
	}
	return FALSE;
}

void
gtk_combo_init (GtkCombo *combo) {
	GtkWidget *arrow;

	combo->flag = GTK_COMBO_EVERY_VALUE;
	combo->entry = gtk_entry_new();
	combo->button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_OUT);
	gtk_widget_show(arrow);
	gtk_container_add(GTK_CONTAINER(combo->button), arrow);
	gtk_box_pack_start(GTK_BOX(combo), combo->entry, TRUE, TRUE, 0);
	gtk_box_pack_end(GTK_BOX(combo), combo->button, FALSE, FALSE, 0);
	gtk_widget_show(combo->entry);
	gtk_widget_show(combo->button);
	combo->entry_change_id = gtk_signal_connect(GTK_OBJECT(combo->entry), "changed",
		(GtkSignalFunc)gtk_combo_update_list, combo);
	gtk_signal_connect(GTK_OBJECT(combo->entry), "key_press_event",
		(GtkSignalFunc)gtk_combo_entry_key_press, combo);
	gtk_signal_connect(GTK_OBJECT(combo->entry), "focus_out_event",
		(GtkSignalFunc)gtk_combo_entry_focus_out, combo);
	gtk_signal_connect(GTK_OBJECT(combo->entry), "activate",
		(GtkSignalFunc)gtk_combo_popup_list, combo);
	gtk_signal_connect(GTK_OBJECT(combo->button), "clicked",
		(GtkSignalFunc)gtk_combo_popup_list, combo);

	combo->popwin = gtk_window_new(GTK_WINDOW_POPUP);
	gtk_window_set_policy(GTK_WINDOW(combo->popwin), 1, 1, 0);
	combo->popup = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(combo->popup), 
		GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	combo->list = gtk_list_new();
	/*gtk_list_set_selection_mode(GTK_LIST(combo->list), GTK_SELECTION_SINGLE);*/
	gtk_container_add(GTK_CONTAINER(combo->popwin), combo->popup);
	gtk_container_add(GTK_CONTAINER(combo->popup), combo->list);
	gtk_widget_show(combo->list);
	gtk_widget_show(combo->popup);
	gtk_widget_set_events(combo->popwin, gtk_widget_get_events(combo->popwin) |GDK_KEY_PRESS_MASK);
	combo->list_change_id = gtk_signal_connect(GTK_OBJECT(combo->list), "selection_changed",
		(GtkSignalFunc)gtk_combo_update_entry, combo);
	gtk_signal_connect(GTK_OBJECT(combo->popwin), "key_press_event",
		(GtkSignalFunc)gtk_combo_list_key_press, combo);

	combo->func = (GtkComboFunc)gtk_combo_func;
}

guint
gtk_combo_get_type ()
{
	static guint combo_type = 0;

	if (!combo_type) {
		GtkTypeInfo combo_info = {
			"GtkCombo",
			sizeof(GtkCombo),
			sizeof(GtkComboClass),
			(GtkClassInitFunc) gtk_combo_class_init,
			(GtkObjectInitFunc) gtk_combo_init,
			(GtkArgFunc) NULL,
		};

		combo_type = gtk_type_unique (gtk_hbox_get_type (), &combo_info);
	}
	return combo_type;
}

GtkWidget*
gtk_combo_new ()
{
	return GTK_WIDGET(gtk_type_new(gtk_combo_get_type()));
}

void 
gtk_combo_set_flag (GtkCombo* combo, GtkComboFlag flag) {
	g_return_if_fail(GTK_IS_COMBO(combo));

	combo->flag = flag;

}

void 
gtk_combo_set_func (GtkCombo* combo, GtkComboFunc func) {
	g_return_if_fail(GTK_IS_COMBO(combo));

	combo->func = func;

}

#ifdef TEST

void do_exit() {
	gtk_exit(0);
}

gchar* 
get_string (GtkWidget* w) {
	GtkBox* box;
	GtkLabel* label;
	gchar* text = NULL;

	box = GTK_BOX(GTK_BIN(w)->child); 
	/* get the text of the first column */
	label = GTK_LABEL(((GtkBoxChild*)box->children->data)->widget);
	gtk_label_get(label, &text);

	return text;
}

int main(int argc, char* argv[]) {
	GtkWidget* win;
	GtkWidget* combo;
	GtkWidget* label;
	GtkWidget* vbox;
	GtkWidget* hbox;
	GtkWidget* li;
	GList* list = NULL;
	gint i;
	char * items1[]= {"mele", "pere", "bicocche", "arance", "limoni", 
		"banane", "spierciuli", "noccioline mericane", "noci carniche",
		"ciribattole", "finferli", "catachiurli", "regarnardi"};
	gint nitems1 = 13;
	char * items2[] = {"Rosencrantz", "column2", "Gildenstern", "column2"};
	gint nitems2 = 4;
	char * items3[] = {"A very long label to trigger horizontal scrollbar"};
	gint nitems3 = 1;

	gtk_init(&argc, &argv);
	win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	combo = gtk_combo_new();
	for (i=0; i < nitems1; ++i) {
		li = gtk_list_item_new_with_label(items1[i]);
		gtk_widget_show(li);
		list = g_list_append(list, li);
	}
	gtk_list_append_items(GTK_LIST(GTK_COMBO(combo)->list), list);
	list = NULL;
	gtk_widget_show(combo);
	vbox = gtk_vbox_new(0, 0);
	gtk_widget_show(vbox);
	gtk_container_add(GTK_CONTAINER(win), vbox);
	label = gtk_label_new("Test per GtkCombo");
	gtk_widget_show(label);
	gtk_container_add(GTK_CONTAINER(vbox), label);
	gtk_container_add(GTK_CONTAINER(vbox), combo);
	label = gtk_label_new("Test per GtkCombo n2");
	gtk_widget_show(label);
	gtk_container_add(GTK_CONTAINER(vbox), label);
	combo = gtk_combo_new();
	gtk_combo_set_func(GTK_COMBO(combo), get_string);
	for (i=0; i+1 < nitems2; i+=2) {
		hbox = gtk_hbox_new(0, 2);
		gtk_widget_show(hbox);
		li = gtk_list_item_new();
		gtk_widget_show(li);
		gtk_container_add(GTK_CONTAINER(li), hbox);
		label = gtk_label_new(items2[i]);
		gtk_widget_show(label);
		gtk_box_pack_start(GTK_BOX(hbox), label, 0, 0, 2);
		label = gtk_label_new(items2[i+1]);
		gtk_widget_show(label);
		gtk_box_pack_end(GTK_BOX(hbox), label, 0, 0, 2);
		list = g_list_append(list, li);
	}
	gtk_list_append_items(GTK_LIST(GTK_COMBO(combo)->list), list);
	list = NULL;
	gtk_combo_set_flag(GTK_COMBO(combo), GTK_COMBO_IN_LIST_VALUE);
	gtk_widget_show(combo);
	gtk_container_add(GTK_CONTAINER(vbox), combo);
	combo = gtk_combo_new();
	for (i=0; i < nitems3; ++i) {
		li = gtk_list_item_new_with_label(items3[i]);
		gtk_widget_show(li);
		list = g_list_append(list, li);
	}
	gtk_list_append_items(GTK_LIST(GTK_COMBO(combo)->list), list);
	list = NULL;
	gtk_widget_show(combo);
	gtk_container_add(GTK_CONTAINER(vbox), combo);

	gtk_signal_connect(GTK_OBJECT(win), "destroy", do_exit, NULL);
	gtk_signal_connect(GTK_OBJECT(win), "delete_event", do_exit, NULL);
	gtk_widget_show(win);
	gtk_main();
	return 0;

}

#endif /* TEST */
