#!/usr/bin/perl

#   Copyright (C) 2011 Mauro Carvalho Chehab
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, version 2 of the License.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
# This small script parses register dumps generated by em28xx driver
# with debug options enabled, generating a source code with the results
# of the dump.
#
# It is a variant of parse_em28xx.pl, meant to be used to decode DRX-K
# register read/writes.
#
# It will provide data output like:
#
# write16_flags(state, 0x52, SIO_CC_PWD_MODE__A, 0x0000, 0xc0);
# write16_flags(state, 0x52, SIO_CC_UPDATE__A, 0xfa00ba, 0xc0);
# write16_flags(state, 0x52, SIO_CC_PLL_LOCK__A, 0x0001, 0xc0);
# write32_flags(state, 0x52, SIO_TOP_COMM_KEY__A, 0x0000, 0xc0);
# write16_flags(state, 0x52, SIO_TOP_COMM_KEY__A, 0xfa00ba, 0xc0);
# write32_flags(state, 0x52, SIO_TOP_JTAGID_LO__A, 0x0000, 0xc0);
# read32_flags(state, 0x52, 0x39846380, 0x0000, 0x90);
#

use strict;
use Getopt::Long;

my $show_em28xx = 0;
my $show_other_xfer = 0;
my $show_ac97 = 0;
my $show_other_lines = 0;	# Useful on some cases
my $show_timestamp = 0;
my $show_all = 0;

GetOptions(
	'show_em28xx' => \$show_em28xx,
	'show_other_xfer' => \$show_other_xfer,
	'show_ac97' => \$show_ac97,
	'show_other_lines' => \$show_other_lines,
	'show_timestamp' => \$show_timestamp,
	'show_all' => \$show_all,
) or die "Invalid arguments.\nUse $0 [--show-all] [--show_em28xx] [--show_other_xfer] [--show_ac97] [--show_other_lines] [--show_timestamp]\n";

if ($show_all) {
	$show_em28xx = 1;
	$show_other_xfer = 1;
	$show_ac97 = 1;
	$show_other_lines = 1;
	$show_timestamp = 1;
}

sub add_hex_mark($)
{
	my $data = shift;
	my $out ="{";

	# Check if the string has the correct syntax. If not, just return it as-is
	for (my $i = 0; $i++; $i < length($data)) {
		if ((($i + 1) % 3) == 0) {
			return $data if (substr($data, $i, 1) ne ' ');
		} else {
			return $data if (!(substr($data, $i, 1) =~ /0-9A-Fa-f/));
		}
	}

	while ($data) {
		$out .= "0x" . substr($data, 0, 2) . ", ";
		$data = substr($data, 3);
	}
	$out =~ s/, $//;
	$out .= "}";

	return $out;
}

my %reg_map = (
	"0x00" => "EM28XX_R00_CHIPCFG",
	"0x04" => "EM2880_R04_GPO",
	"0x08" => "EM28XX_R08_GPIO",
	"0x06" => "EM28XX_R06_I2C_CLK",
	"0x0a" => "EM28XX_R0A_CHIPID",
	"0x0c" => "EM28XX_R0C_USBSUSP",
	"0x0e" => "EM28XX_R0E_AUDIOSRC",
	"0x0f" => "EM28XX_R0F_XCLK",
	"0x20" => "EM28XX_XCLK_IR_RC5_MODE",
	"0x10" => "EM28XX_R10_VINMODE",
	"0x11" => "EM28XX_R11_VINCTRL",
	"0x12" => "EM28XX_R12_VINENABLE",
	"0x14" => "EM28XX_R14_GAMMA",
	"0x15" => "EM28XX_R15_RGAIN",
	"0x16" => "EM28XX_R16_GGAIN",
	"0x17" => "EM28XX_R17_BGAIN",
	"0x18" => "EM28XX_R18_ROFFSET",
	"0x19" => "EM28XX_R19_GOFFSET",
	"0x1a" => "EM28XX_R1A_BOFFSET",
	"0x1b" => "EM28XX_R1B_OFLOW",
	"0x1c" => "EM28XX_R1C_HSTART",
	"0x1d" => "EM28XX_R1D_VSTART",
	"0x1e" => "EM28XX_R1E_CWIDTH",
	"0x1f" => "EM28XX_R1F_CHEIGHT",
	"0x20" => "EM28XX_R20_YGAIN",
	"0x21" => "EM28XX_R21_YOFFSET",
	"0x22" => "EM28XX_R22_UVGAIN",
	"0x23" => "EM28XX_R23_UOFFSET",
	"0x24" => "EM28XX_R24_VOFFSET",
	"0x25" => "EM28XX_R25_SHARPNESS",
	"0x26" => "EM28XX_R26_COMPR",
	"0x27" => "EM28XX_R27_OUTFMT",
	"0x28" => "EM28XX_R28_XMIN",
	"0x29" => "EM28XX_R29_XMAX",
	"0x2a" => "EM28XX_R2A_YMIN",
	"0x2b" => "EM28XX_R2B_YMAX",
	"0x30" => "EM28XX_R30_HSCALELOW",
	"0x31" => "EM28XX_R31_HSCALEHIGH",
	"0x32" => "EM28XX_R32_VSCALELOW",
	"0x33" => "EM28XX_R33_VSCALEHIGH",
	"0x34" => "EM28XX_R34_VBI_START_H",
	"0x35" => "EM28XX_R35_VBI_START_V",
	"0x36" => "EM28XX_R36_VBI_WIDTH",
	"0x37" => "EM28XX_R37_VBI_HEIGHT",
	"0x40" => "EM28XX_R40_AC97LSB",
	"0x41" => "EM28XX_R41_AC97MSB",
	"0x42" => "EM28XX_R42_AC97ADDR",
	"0x43" => "EM28XX_R43_AC97BUSY",
	"0x45" => "EM28XX_R45_IR",
	"0x50" => "EM2874_R50_IR_CONFIG",
	"0x51" => "EM2874_R51_IR",
	"0x5f" => "EM2874_R5F_TS_ENABLE",
	"0x80" => "EM2874_R80_GPIO",
);

my %ac97_map = (
	"0x00" => "AC97_RESET",
	"0x02" => "AC97_MASTER_VOL",
	"0x04" => "AC97_LINE_LEVEL_VOL",
	"0x06" => "AC97_MASTER_MONO_VOL",
	"0x0a" => "AC97_PC_BEEP_VOL",
	"0x0c" => "AC97_PHONE_VOL",
	"0x0e" => "AC97_MIC_VOL",
	"0x10" => "AC97_LINEIN_VOL",
	"0x12" => "AC97_CD_VOL",
	"0x14" => "AC97_VIDEO_VOL",
	"0x16" => "AC97_AUX_VOL",
	"0x18" => "AC97_PCM_OUT_VOL",
	"0x1a" => "AC97_RECORD_SELECT",
	"0x1c" => "AC97_RECORD_GAIN",
	"0x20" => "AC97_GENERAL_PURPOSE",
	"0x22" => "AC97_3D_CTRL",
	"0x24" => "AC97_AUD_INT_AND_PAG",
	"0x26" => "AC97_POWER_DOWN_CTRL",
	"0x28" => "AC97_EXT_AUD_ID",
	"0x2a" => "AC97_EXT_AUD_CTRL",
	"0x2c" => "AC97_PCM_OUT_FRONT_SRATE",
	"0x2e" => "AC97_PCM_OUT_SURR_SRATE",
	"0x30" => "AC97_PCM_OUT_LFE_SRATE",
	"0x32" => "AC97_PCM_IN_SRATE",
	"0x36" => "AC97_LFE_MASTER_VOL",
	"0x38" => "AC97_SURR_MASTER_VOL",
	"0x3a" => "AC97_SPDIF_OUT_CTRL",
	"0x7c" => "AC97_VENDOR_ID1",
	"0x7e" => "AC97_VENDOR_ID2",

	# em202 specific AC97 registers

	"0x3e" => "EM202_EXT_MODEM_CTRL",
	"0x4c" => "EM202_GPIO_CONF",
	"0x4e" => "EM202_GPIO_POLARITY",
	"0x50" => "EM202_GPIO_STICKY",
	"0x52" => "EM202_GPIO_MASK",
	"0x54" => "EM202_GPIO_STATUS",
	"0x6a" => "EM202_SPDIF_OUT_SEL",
	"0x72" => "EM202_ANTIPOP",
	"0x74" => "EM202_EAPD_GPIO_ACCESS",
);

my ($r40, $r42, $r43, $dir);

sub output_ac97($)
{
	my $timestamp = shift;

	if (hex($r42) < 0x80) {
		if ($dir < 0) {
			return;
		}
		$r42 = $ac97_map{$r42} if defined($ac97_map{$r42});
		printf "$timestamp " if ($timestamp);
		printf "em28xx_write_ac97(dev, %s, %s);\n",$r42, $r40;
		$r43 = 0;

		return;
	}

	if ($dir > 0) {
		return;
	}
	$r42 = sprintf("0x%02x", hex($r42) - 0x80);
	$r42 = $ac97_map{$r42} if defined($ac97_map{$r42});
	printf "$timestamp " if ($timestamp);
	printf "em28xx_read_ac97(dev, %s);\t/* read 0x%s */\n",$r42, $r40;
	$r43 = 0;
}

# DRX-K registers used at the public driver
# Obtained with this command:
#   cat drivers/media/dvb/frontends/drxj_map.h |perl -ne 'if (/define\s+([^\s]+__A)\s+(.*)/) { printf "\t0x%08x => \"%s\",\n",hex($2), $1; }'
#
my %drxj_map = (
	0xC00000 => "ATV_COMM_EXEC__A",
	0xC00001 => "ATV_COMM_STATE__A",
	0xC00002 => "ATV_COMM_MB__A",
	0xC00003 => "ATV_COMM_INT_REQ__A",
	0xC00005 => "ATV_COMM_INT_STA__A",
	0xC00006 => "ATV_COMM_INT_MSK__A",
	0xC00007 => "ATV_COMM_INT_STM__A",
	0xC0000F => "ATV_COMM_KEY__A",
	0xC10000 => "ATV_TOP_COMM_EXEC__A",
	0xC10001 => "ATV_TOP_COMM_STATE__A",
	0xC10002 => "ATV_TOP_COMM_MB__A",
	0xC10003 => "ATV_TOP_COMM_INT_REQ__A",
	0xC10005 => "ATV_TOP_COMM_INT_STA__A",
	0xC10006 => "ATV_TOP_COMM_INT_MSK__A",
	0xC10007 => "ATV_TOP_COMM_INT_STM__A",
	0xC1000F => "ATV_TOP_COMM_KEY__A",
	0xC10010 => "ATV_TOP_CR_AMP_TH__A",
	0xC10011 => "ATV_TOP_CR_CONT__A",
	0xC10012 => "ATV_TOP_CR_OVM_TH__A",
	0xC10013 => "ATV_TOP_NOISE_TH__A",
	0xC10014 => "ATV_TOP_EQU0__A",
	0xC10015 => "ATV_TOP_EQU1__A",
	0xC10016 => "ATV_TOP_EQU2__A",
	0xC10017 => "ATV_TOP_EQU3__A",
	0xC10018 => "ATV_TOP_ROT_MODE__A",
	0xC10019 => "ATV_TOP_MOD_CONTROL__A",
	0xC1001A => "ATV_TOP_STD__A",
	0xC1001B => "ATV_TOP_VID_AMP__A",
	0xC1001C => "ATV_TOP_VID_PEAK__A",
	0xC1001D => "ATV_TOP_FAGC_TH__A",
	0xC1001E => "ATV_TOP_SYNC_SLICE__A",
	0xC1001F => "ATV_TOP_SIF_GAIN__A",
	0xC10020 => "ATV_TOP_SIF_TP__A",
	0xC10021 => "ATV_TOP_MOD_ACCU__A",
	0xC10022 => "ATV_TOP_CR_FREQ__A",
	0xC10023 => "ATV_TOP_CR_PHAD__A",
	0xC10024 => "ATV_TOP_AF_SIF_ATT__A",
	0xC10025 => "ATV_TOP_STDBY__A",
	0xC10026 => "ATV_TOP_OVERRIDE_SFR__A",
	0xC10027 => "ATV_TOP_SFR_VID_GAIN__A",
	0xC10028 => "ATV_TOP_SFR_AGC_RES__A",
	0xC10029 => "ATV_TOP_OVM_COMP__A",
	0xC1002A => "ATV_TOP_OUT_CONF__A",
	0xFF0000 => "ATV_AFT_COMM_EXEC__A",
	0xFF0010 => "ATV_AFT_TST__A",
	0x1000000 => "AUD_COMM_EXEC__A",
	0x1000002 => "AUD_COMM_MB__A",
	0x1010000 => "AUD_TOP_COMM_EXEC__A",
	0x1010002 => "AUD_TOP_COMM_MB__A",
	0x1010010 => "AUD_TOP_TR_MDE__A",
	0x1010011 => "AUD_TOP_TR_CTR__A",
	0x1010012 => "AUD_TOP_TR_RD_REG__A",
	0x1010013 => "AUD_TOP_TR_TIMER__A",
	0x1010014 => "AUD_TOP_DEMOD_TBO_SEL__A",
	0x1030030 => "AUD_DEM_WR_MODUS__A",
	0x1030020 => "AUD_DEM_WR_STANDARD_SEL__A",
	0x102007E => "AUD_DEM_RD_STANDARD_RES__A",
	0x1020200 => "AUD_DEM_RD_STATUS__A",
	0x102020F => "AUD_DEM_RD_RDS_ARRAY_CNT__A",
	0x1020210 => "AUD_DEM_RD_RDS_DATA__A",
	0x105000E => "AUD_DSP_WR_FM_PRESC__A",
	0x1050010 => "AUD_DSP_WR_NICAM_PRESC__A",
	0x1050000 => "AUD_DSP_WR_VOLUME__A",
	0x1050038 => "AUD_DSP_WR_SRC_I2S_MATR__A",
	0x1050029 => "AUD_DSP_WR_AVC__A",
	0x105000C => "AUD_DSP_WR_QPEAK__A",
	0x1040019 => "AUD_DSP_RD_QPEAK_L__A",
	0x104001A => "AUD_DSP_RD_QPEAK_R__A",
	0x1050014 => "AUD_DSP_WR_BEEPER__A",
	0x1030050 => "AUD_DEM_WR_I2S_CONFIG2__A",
	0x105002A => "AUD_DSP_WR_I2S_OUT_FS__A",
	0x105002B => "AUD_DSP_WR_AV_SYNC__A",
	0x104007B => "AUD_DSP_RD_STATUS2__A",
	0x104001D => "AUD_DSP_RD_XDFP_FW__A",
	0x10404B8 => "AUD_DSP_RD_XFP_FW__A",
	0x103009B => "AUD_DEM_WR_DCO_B_HI__A",
	0x1030093 => "AUD_DEM_WR_DCO_B_LO__A",
	0x10300AB => "AUD_DEM_WR_DCO_A_HI__A",
	0x10300A3 => "AUD_DEM_WR_DCO_A_LO__A",
	0x1030021 => "AUD_DEM_WR_NICAM_THRSHLD__A",
	0x1030022 => "AUD_DEM_WR_A2_THRSHLD__A",
	0x1030023 => "AUD_DEM_WR_BTSC_THRSHLD__A",
	0x1030024 => "AUD_DEM_WR_CM_A_THRSHLD__A",
	0x1030025 => "AUD_DEM_WR_CM_B_THRSHLD__A",
	0x1020023 => "AUD_DEM_RD_NIC_C_AD_BITS__A",
	0x1020038 => "AUD_DEM_RD_NIC_ADD_BITS_HI__A",
	0x1020038 => "AUD_DEM_RD_NIC_CIB__A",
	0x1020057 => "AUD_DEM_RD_NIC_ERROR_RATE__A",
	0x103000F => "AUD_DEM_WR_FM_DEEMPH__A",
	0x103006F => "AUD_DEM_WR_FM_MATRIX__A",
	0x1040018 => "AUD_DSP_RD_FM_IDENT_VALUE__A",
	0x104001B => "AUD_DSP_RD_FM_DC_LEVEL_A__A",
	0x104001C => "AUD_DSP_RD_FM_DC_LEVEL_B__A",
	0x1030017 => "AUD_DEM_WR_FM_DC_NOTCH_SW__A",
	0x1050026 => "AUD_DSP_WR_SYNC_OUT__A",
	0x1060000 => "AUD_XFP_DRAM_1K__A",
	0x1070000 => "AUD_XFP_PRAM_4K__A",
	0x1080000 => "AUD_XDFP_DRAM_1K__A",
	0x1090000 => "AUD_XDFP_PRAM_4K__A",
	0x2400000 => "FEC_COMM_EXEC__A",
	0x2400002 => "FEC_COMM_MB__A",
	0x2400003 => "FEC_COMM_INT_REQ__A",
	0x2400005 => "FEC_COMM_INT_STA__A",
	0x2400006 => "FEC_COMM_INT_MSK__A",
	0x2400007 => "FEC_COMM_INT_STM__A",
	0x2410000 => "FEC_TOP_COMM_EXEC__A",
	0x2410010 => "FEC_TOP_ANNEX__A",
	0x2420000 => "FEC_DI_COMM_EXEC__A",
	0x2420002 => "FEC_DI_COMM_MB__A",
	0x2420003 => "FEC_DI_COMM_INT_REQ__A",
	0x2420005 => "FEC_DI_COMM_INT_STA__A",
	0x2420006 => "FEC_DI_COMM_INT_MSK__A",
	0x2420007 => "FEC_DI_COMM_INT_STM__A",
	0x2420010 => "FEC_DI_STATUS__A",
	0x2420011 => "FEC_DI_MODE__A",
	0x2420012 => "FEC_DI_CONTROL_WORD__A",
	0x2420013 => "FEC_DI_RESTART__A",
	0x2420014 => "FEC_DI_TIMEOUT_LO__A",
	0x2420015 => "FEC_DI_TIMEOUT_HI__A",
	0x2430000 => "FEC_RS_COMM_EXEC__A",
	0x2430002 => "FEC_RS_COMM_MB__A",
	0x2430003 => "FEC_RS_COMM_INT_REQ__A",
	0x2430005 => "FEC_RS_COMM_INT_STA__A",
	0x2430006 => "FEC_RS_COMM_INT_MSK__A",
	0x2430007 => "FEC_RS_COMM_INT_STM__A",
	0x2430010 => "FEC_RS_STATUS__A",
	0x2430011 => "FEC_RS_MODE__A",
	0x2430012 => "FEC_RS_MEASUREMENT_PERIOD__A",
	0x2430013 => "FEC_RS_MEASUREMENT_PRESCALE__A",
	0x2430014 => "FEC_RS_NR_BIT_ERRORS__A",
	0x2430015 => "FEC_RS_NR_SYMBOL_ERRORS__A",
	0x2430016 => "FEC_RS_NR_PACKET_ERRORS__A",
	0x2430017 => "FEC_RS_NR_FAILURES__A",
	0x2440000 => "FEC_OC_COMM_EXEC__A",
	0x2440002 => "FEC_OC_COMM_MB__A",
	0x2440003 => "FEC_OC_COMM_INT_REQ__A",
	0x2440005 => "FEC_OC_COMM_INT_STA__A",
	0x2440006 => "FEC_OC_COMM_INT_MSK__A",
	0x2440007 => "FEC_OC_COMM_INT_STM__A",
	0x2440010 => "FEC_OC_STATUS__A",
	0x2440011 => "FEC_OC_MODE__A",
	0x2440012 => "FEC_OC_DPR_MODE__A",
	0x2440013 => "FEC_OC_DPR_UNLOCK__A",
	0x2440014 => "FEC_OC_DTO_MODE__A",
	0x2440015 => "FEC_OC_DTO_PERIOD__A",
	0x2440016 => "FEC_OC_DTO_RATE_LO__A",
	0x2440017 => "FEC_OC_DTO_RATE_HI__A",
	0x2440018 => "FEC_OC_DTO_BURST_LEN__A",
	0x244001A => "FEC_OC_FCT_MODE__A",
	0x244001B => "FEC_OC_FCT_USAGE__A",
	0x244001C => "FEC_OC_FCT_OCCUPATION__A",
	0x244001E => "FEC_OC_TMD_MODE__A",
	0x244001F => "FEC_OC_TMD_COUNT__A",
	0x2440020 => "FEC_OC_TMD_HI_MARGIN__A",
	0x2440021 => "FEC_OC_TMD_LO_MARGIN__A",
	0x2440022 => "FEC_OC_TMD_CTL_UPD_RATE__A",
	0x2440023 => "FEC_OC_TMD_INT_UPD_RATE__A",
	0x2440026 => "FEC_OC_AVR_PARM_A__A",
	0x2440027 => "FEC_OC_AVR_PARM_B__A",
	0x2440028 => "FEC_OC_AVR_AVG_LO__A",
	0x2440029 => "FEC_OC_AVR_AVG_HI__A",
	0x244002C => "FEC_OC_RCN_MODE__A",
	0x244002D => "FEC_OC_RCN_OCC_SETTLE__A",
	0x244002E => "FEC_OC_RCN_GAIN__A",
	0x2440030 => "FEC_OC_RCN_CTL_RATE_LO__A",
	0x2440031 => "FEC_OC_RCN_CTL_RATE_HI__A",
	0x2440032 => "FEC_OC_RCN_CTL_STEP_LO__A",
	0x2440033 => "FEC_OC_RCN_CTL_STEP_HI__A",
	0x2440034 => "FEC_OC_RCN_DTO_OFS_LO__A",
	0x2440035 => "FEC_OC_RCN_DTO_OFS_HI__A",
	0x2440036 => "FEC_OC_RCN_DTO_RATE_LO__A",
	0x2440037 => "FEC_OC_RCN_DTO_RATE_HI__A",
	0x2440038 => "FEC_OC_RCN_RATE_CLIP_LO__A",
	0x2440039 => "FEC_OC_RCN_RATE_CLIP_HI__A",
	0x244003A => "FEC_OC_RCN_DYN_RATE_LO__A",
	0x244003B => "FEC_OC_RCN_DYN_RATE_HI__A",
	0x2440040 => "FEC_OC_SNC_MODE__A",
	0x2440041 => "FEC_OC_SNC_LWM__A",
	0x2440042 => "FEC_OC_SNC_HWM__A",
	0x2440043 => "FEC_OC_SNC_UNLOCK__A",
	0x2440044 => "FEC_OC_SNC_LOCK_COUNT__A",
	0x2440045 => "FEC_OC_SNC_FAIL_COUNT__A",
	0x2440046 => "FEC_OC_SNC_FAIL_PERIOD__A",
	0x2440047 => "FEC_OC_EMS_MODE__A",
	0x2440048 => "FEC_OC_IPR_MODE__A",
	0x2440049 => "FEC_OC_IPR_INVERT__A",
	0x2440050 => "FEC_OC_OCR_MODE__A",
	0x2440051 => "FEC_OC_OCR_RATE__A",
	0x2440052 => "FEC_OC_OCR_INVERT__A",
	0x2440053 => "FEC_OC_OCR_GRAB_COUNT__A",
	0x2440054 => "FEC_OC_OCR_GRAB_SYNC__A",
	0x2440055 => "FEC_OC_OCR_GRAB_RD0__A",
	0x2440056 => "FEC_OC_OCR_GRAB_RD1__A",
	0x2440057 => "FEC_OC_OCR_GRAB_RD2__A",
	0x2440058 => "FEC_OC_OCR_GRAB_RD3__A",
	0x2440059 => "FEC_OC_OCR_GRAB_RD4__A",
	0x244005A => "FEC_OC_OCR_GRAB_RD5__A",
	0x2450000 => "FEC_DI_RAM__A",
	0x2460000 => "FEC_RS_RAM__A",
	0x2470000 => "FEC_OC_RAM__A",
	0x1800000 => "IQM_COMM_EXEC__A",
	0x1800002 => "IQM_COMM_MB__A",
	0x1800003 => "IQM_COMM_INT_REQ__A",
	0x1800005 => "IQM_COMM_INT_STA__A",
	0x1800006 => "IQM_COMM_INT_MSK__A",
	0x1800007 => "IQM_COMM_INT_STM__A",
	0x1820000 => "IQM_FS_COMM_EXEC__A",
	0x1820002 => "IQM_FS_COMM_MB__A",
	0x1820010 => "IQM_FS_RATE_OFS_LO__A",
	0x1820011 => "IQM_FS_RATE_OFS_HI__A",
	0x1820012 => "IQM_FS_RATE_LO__A",
	0x1820013 => "IQM_FS_RATE_HI__A",
	0x1820014 => "IQM_FS_ADJ_SEL__A",
	0x1830000 => "IQM_FD_COMM_EXEC__A",
	0x1830002 => "IQM_FD_COMM_MB__A",
	0x1840000 => "IQM_RC_COMM_EXEC__A",
	0x1840002 => "IQM_RC_COMM_MB__A",
	0x1840010 => "IQM_RC_RATE_OFS_LO__A",
	0x1840011 => "IQM_RC_RATE_OFS_HI__A",
	0x1840012 => "IQM_RC_RATE_LO__A",
	0x1840013 => "IQM_RC_RATE_HI__A",
	0x1840014 => "IQM_RC_ADJ_SEL__A",
	0x1840015 => "IQM_RC_CROUT_ENA__A",
	0x1840016 => "IQM_RC_STRETCH__A",
	0x1850000 => "IQM_RT_COMM_EXEC__A",
	0x1850002 => "IQM_RT_COMM_MB__A",
	0x1850010 => "IQM_RT_ACTIVE__A",
	0x1850011 => "IQM_RT_LO_INCR__A",
	0x1850012 => "IQM_RT_ROT_BP__A",
	0x1850013 => "IQM_RT_LP_BP__A",
	0x1850014 => "IQM_RT_DELAY__A",
	0x1860000 => "IQM_CF_COMM_EXEC__A",
	0x1860002 => "IQM_CF_COMM_MB__A",
	0x1860003 => "IQM_CF_COMM_INT_REQ__A",
	0x1860005 => "IQM_CF_COMM_INT_STA__A",
	0x1860006 => "IQM_CF_COMM_INT_MSK__A",
	0x1860007 => "IQM_CF_COMM_INT_STM__A",
	0x1860010 => "IQM_CF_SYMMETRIC__A",
	0x1860011 => "IQM_CF_MIDTAP__A",
	0x1860012 => "IQM_CF_OUT_ENA__A",
	0x1860013 => "IQM_CF_ADJ_SEL__A",
	0x1860014 => "IQM_CF_SCALE__A",
	0x1860015 => "IQM_CF_SCALE_SH__A",
	0x1860016 => "IQM_CF_AMP__A",
	0x1860017 => "IQM_CF_POW_MEAS_LEN__A",
	0x1860018 => "IQM_CF_POW__A",
	0x1860020 => "IQM_CF_TAP_RE0__A",
	0x1860021 => "IQM_CF_TAP_RE1__A",
	0x1860022 => "IQM_CF_TAP_RE2__A",
	0x1860023 => "IQM_CF_TAP_RE3__A",
	0x1860024 => "IQM_CF_TAP_RE4__A",
	0x1860025 => "IQM_CF_TAP_RE5__A",
	0x1860026 => "IQM_CF_TAP_RE6__A",
	0x1860027 => "IQM_CF_TAP_RE7__A",
	0x1860028 => "IQM_CF_TAP_RE8__A",
	0x1860029 => "IQM_CF_TAP_RE9__A",
	0x186002A => "IQM_CF_TAP_RE10__A",
	0x186002B => "IQM_CF_TAP_RE11__A",
	0x186002C => "IQM_CF_TAP_RE12__A",
	0x186002D => "IQM_CF_TAP_RE13__A",
	0x186002E => "IQM_CF_TAP_RE14__A",
	0x186002F => "IQM_CF_TAP_RE15__A",
	0x1860030 => "IQM_CF_TAP_RE16__A",
	0x1860031 => "IQM_CF_TAP_RE17__A",
	0x1860032 => "IQM_CF_TAP_RE18__A",
	0x1860033 => "IQM_CF_TAP_RE19__A",
	0x1860034 => "IQM_CF_TAP_RE20__A",
	0x1860035 => "IQM_CF_TAP_RE21__A",
	0x1860036 => "IQM_CF_TAP_RE22__A",
	0x1860037 => "IQM_CF_TAP_RE23__A",
	0x1860038 => "IQM_CF_TAP_RE24__A",
	0x1860039 => "IQM_CF_TAP_RE25__A",
	0x186003A => "IQM_CF_TAP_RE26__A",
	0x186003B => "IQM_CF_TAP_RE27__A",
	0x1860040 => "IQM_CF_TAP_IM0__A",
	0x1860041 => "IQM_CF_TAP_IM1__A",
	0x1860042 => "IQM_CF_TAP_IM2__A",
	0x1860043 => "IQM_CF_TAP_IM3__A",
	0x1860044 => "IQM_CF_TAP_IM4__A",
	0x1860045 => "IQM_CF_TAP_IM5__A",
	0x1860046 => "IQM_CF_TAP_IM6__A",
	0x1860047 => "IQM_CF_TAP_IM7__A",
	0x1860048 => "IQM_CF_TAP_IM8__A",
	0x1860049 => "IQM_CF_TAP_IM9__A",
	0x186004A => "IQM_CF_TAP_IM10__A",
	0x186004B => "IQM_CF_TAP_IM11__A",
	0x186004C => "IQM_CF_TAP_IM12__A",
	0x186004D => "IQM_CF_TAP_IM13__A",
	0x186004E => "IQM_CF_TAP_IM14__A",
	0x186004F => "IQM_CF_TAP_IM15__A",
	0x1860050 => "IQM_CF_TAP_IM16__A",
	0x1860051 => "IQM_CF_TAP_IM17__A",
	0x1860052 => "IQM_CF_TAP_IM18__A",
	0x1860053 => "IQM_CF_TAP_IM19__A",
	0x1860054 => "IQM_CF_TAP_IM20__A",
	0x1860055 => "IQM_CF_TAP_IM21__A",
	0x1860056 => "IQM_CF_TAP_IM22__A",
	0x1860057 => "IQM_CF_TAP_IM23__A",
	0x1860058 => "IQM_CF_TAP_IM24__A",
	0x1860059 => "IQM_CF_TAP_IM25__A",
	0x186005A => "IQM_CF_TAP_IM26__A",
	0x186005B => "IQM_CF_TAP_IM27__A",
	0x1870000 => "IQM_AF_COMM_EXEC__A",
	0x1870002 => "IQM_AF_COMM_MB__A",
	0x1870003 => "IQM_AF_COMM_INT_REQ__A",
	0x1870005 => "IQM_AF_COMM_INT_STA__A",
	0x1870006 => "IQM_AF_COMM_INT_MSK__A",
	0x1870007 => "IQM_AF_COMM_INT_STM__A",
	0x1870010 => "IQM_AF_FDB_SEL__A",
	0x1870011 => "IQM_AF_INVEXT__A",
	0x1870012 => "IQM_AF_CLKNEG__A",
	0x1870013 => "IQM_AF_MON_IN_MUX__A",
	0x1870014 => "IQM_AF_MON_IN5__A",
	0x1870015 => "IQM_AF_MON_IN4__A",
	0x1870016 => "IQM_AF_MON_IN3__A",
	0x1870017 => "IQM_AF_MON_IN2__A",
	0x1870018 => "IQM_AF_MON_IN1__A",
	0x1870019 => "IQM_AF_MON_IN0__A",
	0x187001A => "IQM_AF_MON_IN_VAL__A",
	0x187001B => "IQM_AF_START_LOCK__A",
	0x187001C => "IQM_AF_PHASE0__A",
	0x187001D => "IQM_AF_PHASE1__A",
	0x187001E => "IQM_AF_PHASE2__A",
	0x187001F => "IQM_AF_SCU_PHASE__A",
	0x1870020 => "IQM_AF_SYNC_SEL__A",
	0x1870021 => "IQM_AF_ADC_CONF__A",
	0x1870022 => "IQM_AF_CLP_CLIP__A",
	0x1870023 => "IQM_AF_CLP_LEN__A",
	0x1870024 => "IQM_AF_CLP_TH__A",
	0x1870025 => "IQM_AF_DCF_BYPASS__A",
	0x1870026 => "IQM_AF_SNS_LEN__A",
	0x1870027 => "IQM_AF_SNS_SENSE__A",
	0x1870028 => "IQM_AF_AGC_IF__A",
	0x1870029 => "IQM_AF_AGC_RF__A",
	0x187002A => "IQM_AF_PGA_GAIN__A",
	0x187002B => "IQM_AF_PDREF__A",
	0x187002C => "IQM_AF_STDBY__A",
	0x187002D => "IQM_AF_AMUX__A",
	0x187002E => "IQM_AF_TST_AFEMAIN__A",
	0x1880000 => "IQM_RT_RAM__A",
	0x2000000 => "ORX_COMM_EXEC__A",
	0x2000001 => "ORX_COMM_STATE__A",
	0x2000002 => "ORX_COMM_MB__A",
	0x2000003 => "ORX_COMM_INT_REQ__A",
	0x2000005 => "ORX_COMM_INT_STA__A",
	0x2000006 => "ORX_COMM_INT_MSK__A",
	0x2000007 => "ORX_COMM_INT_STM__A",
	0x2010000 => "ORX_TOP_COMM_EXEC__A",
	0x201000F => "ORX_TOP_COMM_KEY__A",
	0x2010010 => "ORX_TOP_MDE_W__A",
	0x2010011 => "ORX_TOP_AIF_CTRL_W__A",
	0x2020000 => "ORX_FWP_COMM_EXEC__A",
	0x2020002 => "ORX_FWP_COMM_MB__A",
	0x2020010 => "ORX_FWP_AAG_LEN_W__A",
	0x2020011 => "ORX_FWP_AAG_THR_W__A",
	0x2020012 => "ORX_FWP_AAG_THR_CNT_R__A",
	0x2020013 => "ORX_FWP_AAG_SNS_CNT_R__A",
	0x2020014 => "ORX_FWP_PFI_A_W__A",
	0x2020015 => "ORX_FWP_PFI_B_W__A",
	0x2020016 => "ORX_FWP_PFI_C_W__A",
	0x2020017 => "ORX_FWP_KR1_AMP_R__A",
	0x2020018 => "ORX_FWP_KR1_LDT_W__A",
	0x2020019 => "ORX_FWP_SRC_DGN_W__A",
	0x202001A => "ORX_FWP_NYQ_ADR_W__A",
	0x202001B => "ORX_FWP_NYQ_COF_RW__A",
	0x202001C => "ORX_FWP_IQM_FRQ_W__A",
	0x2030000 => "ORX_EQU_COMM_EXEC__A",
	0x2030002 => "ORX_EQU_COMM_MB__A",
	0x2030003 => "ORX_EQU_COMM_INT_REQ__A",
	0x2030005 => "ORX_EQU_COMM_INT_STA__A",
	0x2030006 => "ORX_EQU_COMM_INT_MSK__A",
	0x2030007 => "ORX_EQU_COMM_INT_STM__A",
	0x2030010 => "ORX_EQU_FFF_SCL_W__A",
	0x2030011 => "ORX_EQU_FFF_UPD_W__A",
	0x2030012 => "ORX_EQU_FFF_STP_W__A",
	0x2030013 => "ORX_EQU_FFF_LEA_W__A",
	0x2030014 => "ORX_EQU_FFF_RWT_W__A",
	0x2030015 => "ORX_EQU_FFF_C0RE_RW__A",
	0x2030016 => "ORX_EQU_FFF_C0IM_RW__A",
	0x2030017 => "ORX_EQU_FFF_C1RE_RW__A",
	0x2030018 => "ORX_EQU_FFF_C1IM_RW__A",
	0x2030019 => "ORX_EQU_FFF_C2RE_RW__A",
	0x203001A => "ORX_EQU_FFF_C2IM_RW__A",
	0x203001B => "ORX_EQU_FFF_C3RE_RW__A",
	0x203001C => "ORX_EQU_FFF_C3IM_RW__A",
	0x203001D => "ORX_EQU_FFF_C4RE_RW__A",
	0x203001E => "ORX_EQU_FFF_C4IM_RW__A",
	0x203001F => "ORX_EQU_FFF_C5RE_RW__A",
	0x2030020 => "ORX_EQU_FFF_C5IM_RW__A",
	0x2030021 => "ORX_EQU_FFF_C6RE_RW__A",
	0x2030022 => "ORX_EQU_FFF_C6IM_RW__A",
	0x2030023 => "ORX_EQU_FFF_C7RE_RW__A",
	0x2030024 => "ORX_EQU_FFF_C7IM_RW__A",
	0x2030025 => "ORX_EQU_FFF_C8RE_RW__A",
	0x2030026 => "ORX_EQU_FFF_C8IM_RW__A",
	0x2030027 => "ORX_EQU_FFF_C9RE_RW__A",
	0x2030028 => "ORX_EQU_FFF_C9IM_RW__A",
	0x2030029 => "ORX_EQU_FFF_C10RE_RW__A",
	0x203002A => "ORX_EQU_FFF_C10IM_RW__A",
	0x203002B => "ORX_EQU_MXB_SEL_W__A",
	0x203002C => "ORX_EQU_FBF_UPD_W__A",
	0x203002D => "ORX_EQU_FBF_STP_W__A",
	0x203002E => "ORX_EQU_FBF_LEA_W__A",
	0x203002F => "ORX_EQU_FBF_RWT_W__A",
	0x2030030 => "ORX_EQU_FBF_C0RE_RW__A",
	0x2030031 => "ORX_EQU_FBF_C0IM_RW__A",
	0x2030032 => "ORX_EQU_FBF_C1RE_RW__A",
	0x2030033 => "ORX_EQU_FBF_C1IM_RW__A",
	0x2030034 => "ORX_EQU_FBF_C2RE_RW__A",
	0x2030035 => "ORX_EQU_FBF_C2IM_RW__A",
	0x2030036 => "ORX_EQU_FBF_C3RE_RW__A",
	0x2030037 => "ORX_EQU_FBF_C3IM_RW__A",
	0x2030038 => "ORX_EQU_FBF_C4RE_RW__A",
	0x2030039 => "ORX_EQU_FBF_C4IM_RW__A",
	0x203003A => "ORX_EQU_FBF_C5RE_RW__A",
	0x203003B => "ORX_EQU_FBF_C5IM_RW__A",
	0x203003C => "ORX_EQU_ERR_SEL_W__A",
	0x203003D => "ORX_EQU_ERR_TIS_W__A",
	0x203003E => "ORX_EQU_ERR_EDI_R__A",
	0x203003F => "ORX_EQU_ERR_EDQ_R__A",
	0x2030040 => "ORX_EQU_ERR_ECI_R__A",
	0x2030041 => "ORX_EQU_ERR_ECQ_R__A",
	0x2030042 => "ORX_EQU_MER_MER_R__A",
	0x2030043 => "ORX_EQU_MER_LDT_W__A",
	0x2030044 => "ORX_EQU_SYN_LEN_W__A",
	0x2040000 => "ORX_DDC_COMM_EXEC__A",
	0x2040002 => "ORX_DDC_COMM_MB__A",
	0x2040003 => "ORX_DDC_COMM_INT_REQ__A",
	0x2040005 => "ORX_DDC_COMM_INT_STA__A",
	0x2040006 => "ORX_DDC_COMM_INT_MSK__A",
	0x2040007 => "ORX_DDC_COMM_INT_STM__A",
	0x2040010 => "ORX_DDC_DEC_MAP_W__A",
	0x2040011 => "ORX_DDC_OFO_SET_W__A",
	0x2050000 => "ORX_CON_COMM_EXEC__A",
	0x2050010 => "ORX_CON_LDT_W__A",
	0x2050011 => "ORX_CON_RST_W__A",
	0x2050012 => "ORX_CON_CPH_PHI_R__A",
	0x2050013 => "ORX_CON_CPH_FRQ_R__A",
	0x2050014 => "ORX_CON_CPH_AMP_R__A",
	0x2050015 => "ORX_CON_CPH_KDF_W__A",
	0x2050016 => "ORX_CON_CPH_KPF_W__A",
	0x2050017 => "ORX_CON_CPH_KIF_W__A",
	0x2050018 => "ORX_CON_CPH_APT_W__A",
	0x2050019 => "ORX_CON_CPH_WLC_W__A",
	0x205001A => "ORX_CON_CPH_DLY_W__A",
	0x205001B => "ORX_CON_CPH_TCL_W__A",
	0x205001C => "ORX_CON_KRP_AMP_R__A",
	0x205001D => "ORX_CON_KRN_AMP_R__A",
	0x205001E => "ORX_CON_CTI_DTI_R__A",
	0x205001F => "ORX_CON_CTI_KDT_W__A",
	0x2050020 => "ORX_CON_CTI_KPT_W__A",
	0x2050021 => "ORX_CON_CTI_KIT_W__A",
	0x2050022 => "ORX_CON_CTI_TAT_W__A",
	0x2060000 => "ORX_NSU_COMM_EXEC__A",
	0x2060010 => "ORX_NSU_AOX_STDBY_W__A",
	0x2060011 => "ORX_NSU_AOX_LOFRQ_W__A",
	0x2060012 => "ORX_NSU_AOX_LOMDE_W__A",
	0x2060013 => "ORX_NSU_AOX_LOPOW_W__A",
	0x2060014 => "ORX_NSU_AOX_STHR_W__A",
	0x2060015 => "ORX_NSU_TUN_RFGAIN_W__A",
	0x2060016 => "ORX_NSU_TUN_IFGAIN_W__A",
	0x2060017 => "ORX_NSU_TUN_BPF_W__A",
	0x2060018 => "ORX_NSU_NSS_BITSWAP_W__A",
	0x23F0000 => "ORX_TST_COMM_EXEC__A",
	0x23F0010 => "ORX_TST_AOX_TST_W__A",
	0x1400000 => "QAM_COMM_EXEC__A",
	0x1400002 => "QAM_COMM_MB__A",
	0x1400003 => "QAM_COMM_INT_REQ__A",
	0x1400005 => "QAM_COMM_INT_STA__A",
	0x1400006 => "QAM_COMM_INT_MSK__A",
	0x1400007 => "QAM_COMM_INT_STM__A",
	0x1410000 => "QAM_TOP_COMM_EXEC__A",
	0x1410010 => "QAM_TOP_ANNEX__A",
	0x1410011 => "QAM_TOP_CONSTELLATION__A",
	0x1420000 => "QAM_FQ_COMM_EXEC__A",
	0x1420010 => "QAM_FQ_MODE__A",
	0x1420011 => "QAM_FQ_MU_FACTOR__A",
	0x1420012 => "QAM_FQ_LA_FACTOR__A",
	0x1420016 => "QAM_FQ_CENTTAP_IDX__A",
	0x1420017 => "QAM_FQ_CENTTAP_VALUE__A",
	0x1420020 => "QAM_FQ_TAP_RE_EL0__A",
	0x1420021 => "QAM_FQ_TAP_IM_EL0__A",
	0x1420022 => "QAM_FQ_TAP_RE_EL1__A",
	0x1420023 => "QAM_FQ_TAP_IM_EL1__A",
	0x1420024 => "QAM_FQ_TAP_RE_EL2__A",
	0x1420025 => "QAM_FQ_TAP_IM_EL2__A",
	0x1420026 => "QAM_FQ_TAP_RE_EL3__A",
	0x1420027 => "QAM_FQ_TAP_IM_EL3__A",
	0x1420028 => "QAM_FQ_TAP_RE_EL4__A",
	0x1420029 => "QAM_FQ_TAP_IM_EL4__A",
	0x142002A => "QAM_FQ_TAP_RE_EL5__A",
	0x142002B => "QAM_FQ_TAP_IM_EL5__A",
	0x142002C => "QAM_FQ_TAP_RE_EL6__A",
	0x142002D => "QAM_FQ_TAP_IM_EL6__A",
	0x142002E => "QAM_FQ_TAP_RE_EL7__A",
	0x142002F => "QAM_FQ_TAP_IM_EL7__A",
	0x1420030 => "QAM_FQ_TAP_RE_EL8__A",
	0x1420031 => "QAM_FQ_TAP_IM_EL8__A",
	0x1420032 => "QAM_FQ_TAP_RE_EL9__A",
	0x1420033 => "QAM_FQ_TAP_IM_EL9__A",
	0x1420034 => "QAM_FQ_TAP_RE_EL10__A",
	0x1420035 => "QAM_FQ_TAP_IM_EL10__A",
	0x1420036 => "QAM_FQ_TAP_RE_EL11__A",
	0x1420037 => "QAM_FQ_TAP_IM_EL11__A",
	0x1420038 => "QAM_FQ_TAP_RE_EL12__A",
	0x1420039 => "QAM_FQ_TAP_IM_EL12__A",
	0x142003A => "QAM_FQ_TAP_RE_EL13__A",
	0x142003B => "QAM_FQ_TAP_IM_EL13__A",
	0x142003C => "QAM_FQ_TAP_RE_EL14__A",
	0x142003D => "QAM_FQ_TAP_IM_EL14__A",
	0x142003E => "QAM_FQ_TAP_RE_EL15__A",
	0x142003F => "QAM_FQ_TAP_IM_EL15__A",
	0x1420040 => "QAM_FQ_TAP_RE_EL16__A",
	0x1420041 => "QAM_FQ_TAP_IM_EL16__A",
	0x1420042 => "QAM_FQ_TAP_RE_EL17__A",
	0x1420043 => "QAM_FQ_TAP_IM_EL17__A",
	0x1420044 => "QAM_FQ_TAP_RE_EL18__A",
	0x1420045 => "QAM_FQ_TAP_IM_EL18__A",
	0x1420046 => "QAM_FQ_TAP_RE_EL19__A",
	0x1420047 => "QAM_FQ_TAP_IM_EL19__A",
	0x1420048 => "QAM_FQ_TAP_RE_EL20__A",
	0x1420049 => "QAM_FQ_TAP_IM_EL20__A",
	0x142004A => "QAM_FQ_TAP_RE_EL21__A",
	0x142004B => "QAM_FQ_TAP_IM_EL21__A",
	0x142004C => "QAM_FQ_TAP_RE_EL22__A",
	0x142004D => "QAM_FQ_TAP_IM_EL22__A",
	0x142004E => "QAM_FQ_TAP_RE_EL23__A",
	0x142004F => "QAM_FQ_TAP_IM_EL23__A",
	0x1430000 => "QAM_SL_COMM_EXEC__A",
	0x1430002 => "QAM_SL_COMM_MB__A",
	0x1430003 => "QAM_SL_COMM_INT_REQ__A",
	0x1430005 => "QAM_SL_COMM_INT_STA__A",
	0x1430006 => "QAM_SL_COMM_INT_MSK__A",
	0x1430007 => "QAM_SL_COMM_INT_STM__A",
	0x1430010 => "QAM_SL_MODE__A",
	0x1430011 => "QAM_SL_K_FACTOR__A",
	0x1430012 => "QAM_SL_MEDIAN__A",
	0x1430013 => "QAM_SL_ALPHA__A",
	0x1430014 => "QAM_SL_PHASELIMIT__A",
	0x1430015 => "QAM_SL_MTA_LENGTH__A",
	0x1430016 => "QAM_SL_MEDIAN_ERROR__A",
	0x1430017 => "QAM_SL_ERR_POWER__A",
	0x1440000 => "QAM_DQ_COMM_EXEC__A",
	0x1440010 => "QAM_DQ_MODE__A",
	0x1440011 => "QAM_DQ_MU_FACTOR__A",
	0x1440012 => "QAM_DQ_LA_FACTOR__A",
	0x1440013 => "QAM_DQ_CMA_RATIO__A",
	0x1440014 => "QAM_DQ_QUAL_RADSEL__A",
	0x1440015 => "QAM_DQ_QUAL_ENA__A",
	0x1440018 => "QAM_DQ_QUAL_FUN0__A",
	0x1440019 => "QAM_DQ_QUAL_FUN1__A",
	0x144001A => "QAM_DQ_QUAL_FUN2__A",
	0x144001B => "QAM_DQ_QUAL_FUN3__A",
	0x144001C => "QAM_DQ_QUAL_FUN4__A",
	0x144001D => "QAM_DQ_QUAL_FUN5__A",
	0x144001E => "QAM_DQ_RAW_LIM__A",
	0x1440020 => "QAM_DQ_TAP_RE_EL0__A",
	0x1440021 => "QAM_DQ_TAP_IM_EL0__A",
	0x1440022 => "QAM_DQ_TAP_RE_EL1__A",
	0x1440023 => "QAM_DQ_TAP_IM_EL1__A",
	0x1440024 => "QAM_DQ_TAP_RE_EL2__A",
	0x1440025 => "QAM_DQ_TAP_IM_EL2__A",
	0x1440026 => "QAM_DQ_TAP_RE_EL3__A",
	0x1440027 => "QAM_DQ_TAP_IM_EL3__A",
	0x1440028 => "QAM_DQ_TAP_RE_EL4__A",
	0x1440029 => "QAM_DQ_TAP_IM_EL4__A",
	0x144002A => "QAM_DQ_TAP_RE_EL5__A",
	0x144002B => "QAM_DQ_TAP_IM_EL5__A",
	0x144002C => "QAM_DQ_TAP_RE_EL6__A",
	0x144002D => "QAM_DQ_TAP_IM_EL6__A",
	0x144002E => "QAM_DQ_TAP_RE_EL7__A",
	0x144002F => "QAM_DQ_TAP_IM_EL7__A",
	0x1440030 => "QAM_DQ_TAP_RE_EL8__A",
	0x1440031 => "QAM_DQ_TAP_IM_EL8__A",
	0x1440032 => "QAM_DQ_TAP_RE_EL9__A",
	0x1440033 => "QAM_DQ_TAP_IM_EL9__A",
	0x1440034 => "QAM_DQ_TAP_RE_EL10__A",
	0x1440035 => "QAM_DQ_TAP_IM_EL10__A",
	0x1440036 => "QAM_DQ_TAP_RE_EL11__A",
	0x1440037 => "QAM_DQ_TAP_IM_EL11__A",
	0x1440038 => "QAM_DQ_TAP_RE_EL12__A",
	0x1440039 => "QAM_DQ_TAP_IM_EL12__A",
	0x144003A => "QAM_DQ_TAP_RE_EL13__A",
	0x144003B => "QAM_DQ_TAP_IM_EL13__A",
	0x144003C => "QAM_DQ_TAP_RE_EL14__A",
	0x144003D => "QAM_DQ_TAP_IM_EL14__A",
	0x144003E => "QAM_DQ_TAP_RE_EL15__A",
	0x144003F => "QAM_DQ_TAP_IM_EL15__A",
	0x1440040 => "QAM_DQ_TAP_RE_EL16__A",
	0x1440041 => "QAM_DQ_TAP_IM_EL16__A",
	0x1440042 => "QAM_DQ_TAP_RE_EL17__A",
	0x1440043 => "QAM_DQ_TAP_IM_EL17__A",
	0x1440044 => "QAM_DQ_TAP_RE_EL18__A",
	0x1440045 => "QAM_DQ_TAP_IM_EL18__A",
	0x1440046 => "QAM_DQ_TAP_RE_EL19__A",
	0x1440047 => "QAM_DQ_TAP_IM_EL19__A",
	0x1440048 => "QAM_DQ_TAP_RE_EL20__A",
	0x1440049 => "QAM_DQ_TAP_IM_EL20__A",
	0x144004A => "QAM_DQ_TAP_RE_EL21__A",
	0x144004B => "QAM_DQ_TAP_IM_EL21__A",
	0x144004C => "QAM_DQ_TAP_RE_EL22__A",
	0x144004D => "QAM_DQ_TAP_IM_EL22__A",
	0x144004E => "QAM_DQ_TAP_RE_EL23__A",
	0x144004F => "QAM_DQ_TAP_IM_EL23__A",
	0x1440050 => "QAM_DQ_TAP_RE_EL24__A",
	0x1440051 => "QAM_DQ_TAP_IM_EL24__A",
	0x1440052 => "QAM_DQ_TAP_RE_EL25__A",
	0x1440053 => "QAM_DQ_TAP_IM_EL25__A",
	0x1440054 => "QAM_DQ_TAP_RE_EL26__A",
	0x1440055 => "QAM_DQ_TAP_IM_EL26__A",
	0x1440056 => "QAM_DQ_TAP_RE_EL27__A",
	0x1440057 => "QAM_DQ_TAP_IM_EL27__A",
	0x1450000 => "QAM_LC_COMM_EXEC__A",
	0x1450002 => "QAM_LC_COMM_MB__A",
	0x1450003 => "QAM_LC_COMM_INT_REQ__A",
	0x1450005 => "QAM_LC_COMM_INT_STA__A",
	0x1450006 => "QAM_LC_COMM_INT_MSK__A",
	0x1450007 => "QAM_LC_COMM_INT_STM__A",
	0x1450010 => "QAM_LC_MODE__A",
	0x1450011 => "QAM_LC_CA__A",
	0x1450012 => "QAM_LC_CF__A",
	0x1450013 => "QAM_LC_CF1__A",
	0x1450014 => "QAM_LC_CP__A",
	0x1450015 => "QAM_LC_CI__A",
	0x1450016 => "QAM_LC_EP__A",
	0x1450017 => "QAM_LC_EI__A",
	0x1450018 => "QAM_LC_QUAL_TAB0__A",
	0x1450019 => "QAM_LC_QUAL_TAB1__A",
	0x145001A => "QAM_LC_QUAL_TAB2__A",
	0x145001B => "QAM_LC_QUAL_TAB3__A",
	0x145001C => "QAM_LC_QUAL_TAB4__A",
	0x145001D => "QAM_LC_QUAL_TAB5__A",
	0x145001E => "QAM_LC_QUAL_TAB6__A",
	0x145001F => "QAM_LC_QUAL_TAB8__A",
	0x1450020 => "QAM_LC_QUAL_TAB9__A",
	0x1450021 => "QAM_LC_QUAL_TAB10__A",
	0x1450022 => "QAM_LC_QUAL_TAB12__A",
	0x1450023 => "QAM_LC_QUAL_TAB15__A",
	0x1450024 => "QAM_LC_QUAL_TAB16__A",
	0x1450025 => "QAM_LC_QUAL_TAB20__A",
	0x1450026 => "QAM_LC_QUAL_TAB25__A",
	0x1450027 => "QAM_LC_EQ_TIMING__A",
	0x1450028 => "QAM_LC_LPF_FACTORP__A",
	0x1450029 => "QAM_LC_LPF_FACTORI__A",
	0x145002A => "QAM_LC_RATE_LIMIT__A",
	0x145002B => "QAM_LC_SYMBOL_FREQ__A",
	0x145002C => "QAM_LC_MTA_LENGTH__A",
	0x145002D => "QAM_LC_AMP_ACCU__A",
	0x145002E => "QAM_LC_FREQ_ACCU__A",
	0x145002F => "QAM_LC_RATE_ACCU__A",
	0x1450030 => "QAM_LC_AMPLITUDE__A",
	0x1450031 => "QAM_LC_RAD_ERROR__A",
	0x1450032 => "QAM_LC_FREQ_OFFS__A",
	0x1450033 => "QAM_LC_PHASE_ERROR__A",
	0x1460000 => "QAM_VD_COMM_EXEC__A",
	0x1460002 => "QAM_VD_COMM_MB__A",
	0x1460003 => "QAM_VD_COMM_INT_REQ__A",
	0x1460005 => "QAM_VD_COMM_INT_STA__A",
	0x1460006 => "QAM_VD_COMM_INT_MSK__A",
	0x1460007 => "QAM_VD_COMM_INT_STM__A",
	0x1460010 => "QAM_VD_STATUS__A",
	0x1460011 => "QAM_VD_UNLOCK_CONTROL__A",
	0x1460012 => "QAM_VD_MIN_VOTING_ROUNDS__A",
	0x1460013 => "QAM_VD_MAX_VOTING_ROUNDS__A",
	0x1460014 => "QAM_VD_TRACEBACK_DEPTH__A",
	0x1460015 => "QAM_VD_UNLOCK__A",
	0x1460016 => "QAM_VD_MEASUREMENT_PERIOD__A",
	0x1460017 => "QAM_VD_MEASUREMENT_PRESCALE__A",
	0x1460018 => "QAM_VD_DELTA_PATH_METRIC__A",
	0x1460019 => "QAM_VD_NR_QSYM_ERRORS__A",
	0x146001A => "QAM_VD_NR_SYMBOL_ERRORS__A",
	0x146001B => "QAM_VD_RELOCK_COUNT__A",
	0x1470000 => "QAM_SY_COMM_EXEC__A",
	0x1470002 => "QAM_SY_COMM_MB__A",
	0x1470003 => "QAM_SY_COMM_INT_REQ__A",
	0x1470005 => "QAM_SY_COMM_INT_STA__A",
	0x1470006 => "QAM_SY_COMM_INT_MSK__A",
	0x1470007 => "QAM_SY_COMM_INT_STM__A",
	0x1470010 => "QAM_SY_STATUS__A",
	0x1470011 => "QAM_SY_TIMEOUT__A",
	0x1470012 => "QAM_SY_SYNC_LWM__A",
	0x1470013 => "QAM_SY_SYNC_AWM__A",
	0x1470014 => "QAM_SY_SYNC_HWM__A",
	0x1470015 => "QAM_SY_UNLOCK__A",
	0x1470016 => "QAM_SY_CONTROL_WORD__A",
	0x1480000 => "QAM_VD_ISS_RAM__A",
	0x1490000 => "QAM_VD_QSS_RAM__A",
	0x14A0000 => "QAM_VD_SYM_RAM__A",
	0x800000 => "SCU_COMM_EXEC__A",
	0x800001 => "SCU_COMM_STATE__A",
	0x810000 => "SCU_TOP_COMM_EXEC__A",
	0x810001 => "SCU_TOP_COMM_STATE__A",
	0x810010 => "SCU_TOP_MWAIT_CTR__A",
	0x820000 => "SCU_LOW_RAM__A",
	0x830000 => "SCU_HIGH_RAM__A",
	0x831E96 => "SCU_RAM_AGC_RF_MAX__A",
	0x831E97 => "SCU_RAM_AGC_FAST_SNS_CTRL_DELAY__A",
	0x831E98 => "SCU_RAM_AGC_KI_CYCCNT__A",
	0x831E99 => "SCU_RAM_AGC_KI_CYCLEN__A",
	0x831E9A => "SCU_RAM_AGC_SNS_CYCLEN__A",
	0x831E9B => "SCU_RAM_AGC_RF_SNS_DEV_MAX__A",
	0x831E9C => "SCU_RAM_AGC_RF_SNS_DEV_MIN__A",
	0x831E9D => "SCU_RAM_AGC_KI__A",
	0x831E9E => "SCU_RAM_AGC_KI_RED__A",
	0x831E9F => "SCU_RAM_AGC_KI_INNERGAIN_MIN__A",
	0x831EA0 => "SCU_RAM_AGC_KI_MINGAIN__A",
	0x831EA1 => "SCU_RAM_AGC_KI_MAXGAIN__A",
	0x831EA2 => "SCU_RAM_AGC_KI_MAXMINGAIN_TH__A",
	0x831EA3 => "SCU_RAM_AGC_KI_MIN__A",
	0x831EA4 => "SCU_RAM_AGC_KI_MAX__A",
	0x831EA5 => "SCU_RAM_AGC_CLP_SUM__A",
	0x831EA6 => "SCU_RAM_AGC_CLP_SUM_MIN__A",
	0x831EA7 => "SCU_RAM_AGC_CLP_SUM_MAX__A",
	0x831EA8 => "SCU_RAM_AGC_CLP_CYCLEN__A",
	0x831EA9 => "SCU_RAM_AGC_CLP_CYCCNT__A",
	0x831EAA => "SCU_RAM_AGC_CLP_DIR_TO__A",
	0x831EAB => "SCU_RAM_AGC_CLP_DIR_WD__A",
	0x831EAC => "SCU_RAM_AGC_CLP_DIR_STP__A",
	0x831EAD => "SCU_RAM_AGC_SNS_SUM__A",
	0x831EAE => "SCU_RAM_AGC_SNS_SUM_MIN__A",
	0x831EAF => "SCU_RAM_AGC_SNS_SUM_MAX__A",
	0x831EB0 => "SCU_RAM_AGC_SNS_CYCCNT__A",
	0x831EB1 => "SCU_RAM_AGC_SNS_DIR_TO__A",
	0x831EB2 => "SCU_RAM_AGC_SNS_DIR_WD__A",
	0x831EB3 => "SCU_RAM_AGC_SNS_DIR_STP__A",
	0x831EB4 => "SCU_RAM_AGC_INGAIN__A",
	0x831EB5 => "SCU_RAM_AGC_INGAIN_TGT__A",
	0x831EB6 => "SCU_RAM_AGC_INGAIN_TGT_MIN__A",
	0x831EB7 => "SCU_RAM_AGC_INGAIN_TGT_MAX__A",
	0x831EB8 => "SCU_RAM_AGC_IF_IACCU_HI__A",
	0x831EB9 => "SCU_RAM_AGC_IF_IACCU_LO__A",
	0x831EBA => "SCU_RAM_AGC_IF_IACCU_HI_TGT__A",
	0x831EBB => "SCU_RAM_AGC_IF_IACCU_HI_TGT_MIN__A",
	0x831EBC => "SCU_RAM_AGC_IF_IACCU_HI_TGT_MAX__A",
	0x831EBD => "SCU_RAM_AGC_RF_IACCU_HI__A",
	0x831EBE => "SCU_RAM_AGC_RF_IACCU_LO__A",
	0x831EBF => "SCU_RAM_AGC_RF_IACCU_HI_CO__A",
	0x831EC0 => "SCU_RAM_SP__A",
	0x831EC1 => "SCU_RAM_AGC_FAST_CLP_CTRL_DELAY__A",
	0x831EC2 => "SCU_RAM_AGC_KI_MIN_IFGAIN__A",
	0x831EC3 => "SCU_RAM_AGC_KI_MAX_IFGAIN__A",
	0x831EC4 => "SCU_RAM_FEC_MEAS_COUNT__A",
	0x831EC5 => "SCU_RAM_FEC_ACCUM_CW_CORRECTED_LO__A",
	0x831EC6 => "SCU_RAM_FEC_ACCUM_CW_CORRECTED_HI__A",
	0x831EC7 => "SCU_RAM_GPIO__A",
	0x831EC8 => "SCU_RAM_AGC_CLP_CTRL_MODE__A",
	0x831EC9 => "SCU_RAM_AGC_KI_MIN_RFGAIN__A",
	0x831ECA => "SCU_RAM_AGC_KI_MAX_RFGAIN__A",
	0x831ECB => "SCU_RAM_FEC_ACCUM_PKT_FAILURES__A",
	0x831ECC => "SCU_RAM_INHIBIT_1__A",
	0x831ECD => "SCU_RAM_HTOL_BUF_0__A",
	0x831ECE => "SCU_RAM_HTOL_BUF_1__A",
	0x831ECF => "SCU_RAM_INHIBIT_2__A",
	0x831ED0 => "SCU_RAM_TR_SHORT_BUF_0__A",
	0x831ED1 => "SCU_RAM_TR_SHORT_BUF_1__A",
	0x831ED2 => "SCU_RAM_TR_LONG_BUF_0__A",
	0x831ED3 => "SCU_RAM_TR_LONG_BUF_1__A",
	0x831ED4 => "SCU_RAM_TR_LONG_BUF_2__A",
	0x831ED5 => "SCU_RAM_TR_LONG_BUF_3__A",
	0x831ED6 => "SCU_RAM_TR_LONG_BUF_4__A",
	0x831ED7 => "SCU_RAM_TR_LONG_BUF_5__A",
	0x831ED8 => "SCU_RAM_TR_LONG_BUF_6__A",
	0x831ED9 => "SCU_RAM_TR_LONG_BUF_7__A",
	0x831EDA => "SCU_RAM_TR_LONG_BUF_8__A",
	0x831EDB => "SCU_RAM_TR_LONG_BUF_9__A",
	0x831EDC => "SCU_RAM_TR_LONG_BUF_10__A",
	0x831EDD => "SCU_RAM_TR_LONG_BUF_11__A",
	0x831EDE => "SCU_RAM_TR_LONG_BUF_12__A",
	0x831EDF => "SCU_RAM_TR_LONG_BUF_13__A",
	0x831EE0 => "SCU_RAM_TR_LONG_BUF_14__A",
	0x831EE1 => "SCU_RAM_TR_LONG_BUF_15__A",
	0x831EE2 => "SCU_RAM_TR_LONG_BUF_16__A",
	0x831EE3 => "SCU_RAM_TR_LONG_BUF_17__A",
	0x831EE4 => "SCU_RAM_TR_LONG_BUF_18__A",
	0x831EE5 => "SCU_RAM_TR_LONG_BUF_19__A",
	0x831EE6 => "SCU_RAM_TR_LONG_BUF_20__A",
	0x831EE7 => "SCU_RAM_TR_LONG_BUF_21__A",
	0x831EE8 => "SCU_RAM_TR_LONG_BUF_22__A",
	0x831EE9 => "SCU_RAM_TR_LONG_BUF_23__A",
	0x831EEA => "SCU_RAM_TR_LONG_BUF_24__A",
	0x831EEB => "SCU_RAM_TR_LONG_BUF_25__A",
	0x831EEC => "SCU_RAM_TR_LONG_BUF_26__A",
	0x831EED => "SCU_RAM_TR_LONG_BUF_27__A",
	0x831EEE => "SCU_RAM_TR_LONG_BUF_28__A",
	0x831EEF => "SCU_RAM_TR_LONG_BUF_29__A",
	0x831EF0 => "SCU_RAM_TR_LONG_BUF_30__A",
	0x831EF1 => "SCU_RAM_TR_LONG_BUF_31__A",
	0x831EF2 => "SCU_RAM_ATV_AMS_MAX__A",
	0x831EF3 => "SCU_RAM_ATV_AMS_MIN__A",
	0x831EF4 => "SCU_RAM_ATV_FIELD_CNT__A",
	0x831EF5 => "SCU_RAM_ATV_AAGC_FAST__A",
	0x831EF6 => "SCU_RAM_ATV_AAGC_LP2__A",
	0x831EF7 => "SCU_RAM_ATV_BP_LVL__A",
	0x831EF8 => "SCU_RAM_ATV_BP_RELY__A",
	0x831EF9 => "SCU_RAM_ATV_BP_MTA__A",
	0x831EFA => "SCU_RAM_ATV_BP_REF__A",
	0x831EFB => "SCU_RAM_ATV_BP_REF_MIN__A",
	0x831EFC => "SCU_RAM_ATV_BP_REF_MAX__A",
	0x831EFD => "SCU_RAM_ATV_BP_CNT__A",
	0x831EFE => "SCU_RAM_ATV_BP_XD_CNT__A",
	0x831EFF => "SCU_RAM_ATV_PAGC_KI_MIN__A",
	0x831F00 => "SCU_RAM_ATV_BPC_KI_MIN__A",
	0x831F01 => "SCU_RAM_ORX_RF_RX_FREQUENCY_VALUE__A",
	0x831F02 => "SCU_RAM_ORX_RF_RX_DATA_RATE__A",
	0x831F03 => "SCU_RAM_ORX_SCU_STATE__A",
	0x831F04 => "SCU_RAM_ORX_SCU_LOCK__A",
	0x831F05 => "SCU_RAM_ORX_TARGET_MODE__A",
	0x831F06 => "SCU_RAM_ORX_MER_MIN_DB__A",
	0x831F07 => "SCU_RAM_ORX_RF_GAIN__A",
	0x831F08 => "SCU_RAM_ORX_RF_GAIN_MIN__A",
	0x831F09 => "SCU_RAM_ORX_RF_GAIN_MAX__A",
	0x831F0A => "SCU_RAM_ORX_IF_GAIN__A",
	0x831F0B => "SCU_RAM_ORX_IF_GAIN_MIN__A",
	0x831F0C => "SCU_RAM_ORX_IF_GAIN_MAX__A",
	0x831F0D => "SCU_RAM_ORX_AGN_HEADR__A",
	0x831F0E => "SCU_RAM_ORX_AGN_HEADR_STP__A",
	0x831F0F => "SCU_RAM_ORX_AGN_KI__A",
	0x831F10 => "SCU_RAM_ORX_AGN_LOCK_TH__A",
	0x831F11 => "SCU_RAM_ORX_AGN_LOCK_WD__A",
	0x831F12 => "SCU_RAM_ORX_AGN_ONLOCK_TTH__A",
	0x831F13 => "SCU_RAM_ORX_AGN_UNLOCK_TTH__A",
	0x831F14 => "SCU_RAM_ORX_AGN_LOCK_TOTH__A",
	0x831F15 => "SCU_RAM_ORX_AGN_LOCK_MASK__A",
	0x831F16 => "SCU_RAM_ORX_DGN__A",
	0x831F17 => "SCU_RAM_ORX_DGN_MIN__A",
	0x831F18 => "SCU_RAM_ORX_DGN_MAX__A",
	0x831F19 => "SCU_RAM_ORX_DGN_AMP__A",
	0x831F1A => "SCU_RAM_ORX_DGN_AMPTARGET__A",
	0x831F1B => "SCU_RAM_ORX_DGN_KI__A",
	0x831F1C => "SCU_RAM_ORX_DGN_LOCK_TH__A",
	0x831F1D => "SCU_RAM_ORX_DGN_LOCK_WD__A",
	0x831F1E => "SCU_RAM_ORX_DGN_ONLOCK_TTH__A",
	0x831F1F => "SCU_RAM_ORX_DGN_UNLOCK_TTH__A",
	0x831F20 => "SCU_RAM_ORX_DGN_LOCK_TOTH__A",
	0x831F21 => "SCU_RAM_ORX_DGN_LOCK_MASK__A",
	0x831F22 => "SCU_RAM_ORX_FREQ_GAIN_CORR__A",
	0x831F23 => "SCU_RAM_ORX_FRQ_OFFSET__A",
	0x831F24 => "SCU_RAM_ORX_FRQ_OFFSET_MAX__A",
	0x831F25 => "SCU_RAM_ORX_FRQ_KI__A",
	0x831F26 => "SCU_RAM_ORX_FRQ_DIFF__A",
	0x831F27 => "SCU_RAM_ORX_FRQ_LOCK_TH__A",
	0x831F28 => "SCU_RAM_ORX_FRQ_LOCK_WD__A",
	0x831F29 => "SCU_RAM_ORX_FRQ_ONLOCK_TTH__A",
	0x831F2A => "SCU_RAM_ORX_FRQ_UNLOCK_TTH__A",
	0x831F2B => "SCU_RAM_ORX_FRQ_LOCK_TOTH__A",
	0x831F2C => "SCU_RAM_ORX_FRQ_LOCK_MASK__A",
	0x831F2D => "SCU_RAM_ORX_PHA_DIFF__A",
	0x831F2E => "SCU_RAM_ORX_PHA_LOCK_TH__A",
	0x831F2F => "SCU_RAM_ORX_PHA_LOCK_WD__A",
	0x831F30 => "SCU_RAM_ORX_PHA_ONLOCK_TTH__A",
	0x831F31 => "SCU_RAM_ORX_PHA_UNLOCK_TTH__A",
	0x831F32 => "SCU_RAM_ORX_PHA_LOCK_TOTH__A",
	0x831F33 => "SCU_RAM_ORX_PHA_LOCK_MASK__A",
	0x831F34 => "SCU_RAM_ORX_TIM_OFFSET__A",
	0x831F35 => "SCU_RAM_ORX_TIM_DIFF__A",
	0x831F36 => "SCU_RAM_ORX_TIM_LOCK_TH__A",
	0x831F37 => "SCU_RAM_ORX_TIM_LOCK_WD__A",
	0x831F38 => "SCU_RAM_ORX_TIM_ONLOCK_TTH__A",
	0x831F39 => "SCU_RAM_ORX_TIM_UNLOCK_TTH__A",
	0x831F3A => "SCU_RAM_ORX_TIM_LOCK_TOTH__A",
	0x831F3B => "SCU_RAM_ORX_TIM_LOCK_MASK__A",
	0x831F3C => "SCU_RAM_ORX_EQU_DIFF__A",
	0x831F3D => "SCU_RAM_ORX_EQU_LOCK_TH__A",
	0x831F3E => "SCU_RAM_ORX_EQU_LOCK_WD__A",
	0x831F3F => "SCU_RAM_ORX_EQU_ONLOCK_TTH__A",
	0x831F40 => "SCU_RAM_ORX_EQU_UNLOCK_TTH__A",
	0x831F41 => "SCU_RAM_ORX_EQU_LOCK_TOTH__A",
	0x831F42 => "SCU_RAM_ORX_EQU_LOCK_MASK__A",
	0x831F43 => "SCU_RAM_ORX_FLT_FRQ__A",
	0x831F44 => "SCU_RAM_ORX_RST_CPH__A",
	0x831F45 => "SCU_RAM_ORX_RST_CTI__A",
	0x831F46 => "SCU_RAM_ORX_RST_KRN__A",
	0x831F47 => "SCU_RAM_ORX_RST_KRP__A",
	0x831F48 => "SCU_RAM_ATV_STANDARD__A",
	0x831F49 => "SCU_RAM_ATV_DETECT__A",
	0x831F4A => "SCU_RAM_ATV_DETECT_TH__A",
	0x831F4B => "SCU_RAM_ATV_LOCK__A",
	0x831F4C => "SCU_RAM_ATV_CR_LOCK__A",
	0x831F4D => "SCU_RAM_ATV_AGC_MODE__A",
	0x831F4E => "SCU_RAM_ATV_RSV_01__A",
	0x831F4F => "SCU_RAM_ATV_RSV_02__A",
	0x831F50 => "SCU_RAM_ATV_RSV_03__A",
	0x831F51 => "SCU_RAM_ATV_RSV_04__A",
	0x831F52 => "SCU_RAM_ATV_FAGC_TH_RED__A",
	0x831F53 => "SCU_RAM_ATV_AMS_MAX_REF__A",
	0x831F54 => "SCU_RAM_ATV_ACT_AMX__A",
	0x831F55 => "SCU_RAM_ATV_ACT_AMI__A",
	0x831F56 => "SCU_RAM_ATV_RSV_05__A",
	0x831F57 => "SCU_RAM_ATV_RSV_06__A",
	0x831F58 => "SCU_RAM_ATV_RSV_07__A",
	0x831F59 => "SCU_RAM_ATV_RSV_08__A",
	0x831F5A => "SCU_RAM_ATV_RSV_09__A",
	0x831F5B => "SCU_RAM_ATV_RSV_10__A",
	0x831F5C => "SCU_RAM_ATV_RSV_11__A",
	0x831F5D => "SCU_RAM_ATV_RSV_12__A",
	0x831F5E => "SCU_RAM_ATV_VID_GAIN_HI__A",
	0x831F5F => "SCU_RAM_ATV_VID_GAIN_LO__A",
	0x831F60 => "SCU_RAM_ATV_RSV_13__A",
	0x831F61 => "SCU_RAM_ATV_RSV_14__A",
	0x831F62 => "SCU_RAM_ATV_RSV_15__A",
	0x831F63 => "SCU_RAM_ATV_RSV_16__A",
	0x831F64 => "SCU_RAM_ATV_AAGC_CNT__A",
	0x831F65 => "SCU_RAM_ATV_SIF_GAIN__A",
	0x831F66 => "SCU_RAM_ATV_RSV_17__A",
	0x831F67 => "SCU_RAM_ATV_RSV_18__A",
	0x831F68 => "SCU_RAM_ATV_RATE_OFS__A",
	0x831F69 => "SCU_RAM_ATV_LO_INCR__A",
	0x831F6A => "SCU_RAM_ATV_IIR_CRIT__A",
	0x831F6B => "SCU_RAM_ATV_DEF_RATE_OFS__A",
	0x831F6C => "SCU_RAM_ATV_DEF_LO_INCR__A",
	0x831F6D => "SCU_RAM_ATV_ENABLE_IIR_WA__A",
	0x831F6E => "SCU_RAM_ATV_MOD_CONTROL__A",
	0x831F6F => "SCU_RAM_ATV_PAGC_KI_MAX__A",
	0x831F70 => "SCU_RAM_ATV_BPC_KI_MAX__A",
	0x831F71 => "SCU_RAM_ATV_NAGC_KI_MAX__A",
	0x831F72 => "SCU_RAM_ATV_NAGC_KI_MIN__A",
	0x831F73 => "SCU_RAM_ATV_KI_CHANGE_TH__A",
	0x831F74 => "SCU_RAM_QAM_PARAM_ANNEX__A",
	0x831F75 => "SCU_RAM_QAM_PARAM_CONSTELLATION__A",
	0x831F76 => "SCU_RAM_QAM_PARAM_INTERLEAVE__A",
	0x831F77 => "SCU_RAM_QAM_PARAM_SYM_RCRATE_HI__A",
	0x831F78 => "SCU_RAM_QAM_PARAM_SYM_RCRATE_LO__A",
	0x831F79 => "SCU_RAM_QAM_EQ_CENTERTAP__A",
	0x831F7A => "SCU_RAM_QAM_WR_RSV_0__A",
	0x831F7B => "SCU_RAM_QAM_PARAM_ALT_RCRATE_HI__A",
	0x831F7C => "SCU_RAM_QAM_PARAM_ALT_RCRATE_LO__A",
	0x831F7D => "SCU_RAM_QAM_WR_RSV_5__A",
	0x831F7E => "SCU_RAM_QAM_WR_RSV_6__A",
	0x831F7F => "SCU_RAM_QAM_WR_RSV_7__A",
	0x831F80 => "SCU_RAM_QAM_WR_RSV_8__A",
	0x831F81 => "SCU_RAM_QAM_WR_RSV_9__A",
	0x831F82 => "SCU_RAM_QAM_WR_RSV_10__A",
	0x831F83 => "SCU_RAM_QAM_FSM_FMHUM_TO__A",
	0x831F84 => "SCU_RAM_QAM_FSM_MEDIAN_AV_MULT__A",
	0x831F85 => "SCU_RAM_QAM_FSM_RADIUS_AV_LIMIT__A",
	0x831F86 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET1__A",
	0x831F87 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET2__A",
	0x831F88 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET3__A",
	0x831F89 => "SCU_RAM_QAM_FSM_LCAVG_OFFSET4__A",
	0x831F8A => "SCU_RAM_QAM_FSM_LCAVG_OFFSET5__A",
	0x831F8B => "SCU_RAM_QAM_FSM_STATE_TGT__A",
	0x831F8C => "SCU_RAM_QAM_FSM_LOCK_OVERRIDE__A",
	0x831F8D => "SCU_RAM_QAM_FSM_ATH__A",
	0x831F8E => "SCU_RAM_QAM_FSM_RTH__A",
	0x831F8F => "SCU_RAM_QAM_FSM_FTH__A",
	0x831F90 => "SCU_RAM_QAM_FSM_PTH__A",
	0x831F91 => "SCU_RAM_QAM_FSM_MTH__A",
	0x831F92 => "SCU_RAM_QAM_FSM_CTH__A",
	0x831F93 => "SCU_RAM_QAM_FSM_QTH__A",
	0x831F94 => "SCU_RAM_QAM_FSM_RATE_LIM__A",
	0x831F95 => "SCU_RAM_QAM_FSM_FREQ_LIM__A",
	0x831F96 => "SCU_RAM_QAM_FSM_COUNT_LIM__A",
	0x831F97 => "SCU_RAM_QAM_LC_CA_COARSE__A",
	0x831F98 => "SCU_RAM_QAM_LC_CA_MEDIUM__A",
	0x831F99 => "SCU_RAM_QAM_LC_CA_FINE__A",
	0x831F9A => "SCU_RAM_QAM_LC_CP_COARSE__A",
	0x831F9B => "SCU_RAM_QAM_LC_CP_MEDIUM__A",
	0x831F9C => "SCU_RAM_QAM_LC_CP_FINE__A",
	0x831F9D => "SCU_RAM_QAM_LC_CI_COARSE__A",
	0x831F9E => "SCU_RAM_QAM_LC_CI_MEDIUM__A",
	0x831F9F => "SCU_RAM_QAM_LC_CI_FINE__A",
	0x831FA0 => "SCU_RAM_QAM_LC_EP_COARSE__A",
	0x831FA1 => "SCU_RAM_QAM_LC_EP_MEDIUM__A",
	0x831FA2 => "SCU_RAM_QAM_LC_EP_FINE__A",
	0x831FA3 => "SCU_RAM_QAM_LC_EI_COARSE__A",
	0x831FA4 => "SCU_RAM_QAM_LC_EI_MEDIUM__A",
	0x831FA5 => "SCU_RAM_QAM_LC_EI_FINE__A",
	0x831FA6 => "SCU_RAM_QAM_LC_CF_COARSE__A",
	0x831FA7 => "SCU_RAM_QAM_LC_CF_MEDIUM__A",
	0x831FA8 => "SCU_RAM_QAM_LC_CF_FINE__A",
	0x831FA9 => "SCU_RAM_QAM_LC_CF1_COARSE__A",
	0x831FAA => "SCU_RAM_QAM_LC_CF1_MEDIUM__A",
	0x831FAB => "SCU_RAM_QAM_LC_CF1_FINE__A",
	0x831FAC => "SCU_RAM_QAM_SL_SIG_POWER__A",
	0x831FAD => "SCU_RAM_QAM_EQ_CMA_RAD0__A",
	0x831FAE => "SCU_RAM_QAM_EQ_CMA_RAD1__A",
	0x831FAF => "SCU_RAM_QAM_EQ_CMA_RAD2__A",
	0x831FB0 => "SCU_RAM_QAM_EQ_CMA_RAD3__A",
	0x831FB1 => "SCU_RAM_QAM_EQ_CMA_RAD4__A",
	0x831FB2 => "SCU_RAM_QAM_EQ_CMA_RAD5__A",
	0x831FB3 => "SCU_RAM_QAM_CTL_ENA__A",
	0x831FB4 => "SCU_RAM_QAM_WR_RSV_1__A",
	0x831FB5 => "SCU_RAM_QAM_WR_RSV_2__A",
	0x831FB6 => "SCU_RAM_QAM_WR_RSV_3__A",
	0x831FB7 => "SCU_RAM_QAM_ACTIVE_CONSTELLATION__A",
	0x831FB8 => "SCU_RAM_QAM_ACTIVE_INTERLEAVE__A",
	0x831FB9 => "SCU_RAM_QAM_RD_RSV_4__A",
	0x831FBA => "SCU_RAM_QAM_LOCKED__A",
	0x831FBB => "SCU_RAM_QAM_EVENTS_OCC_HI__A",
	0x831FBC => "SCU_RAM_QAM_EVENTS_OCC_LO__A",
	0x831FBD => "SCU_RAM_QAM_EVENTS_SCHED_HI__A",
	0x831FBE => "SCU_RAM_QAM_EVENTS_SCHED_LO__A",
	0x831FBF => "SCU_RAM_QAM_TASKLETS_SCHED__A",
	0x831FC0 => "SCU_RAM_QAM_TASKLETS_RUN__A",
	0x831FC1 => "SCU_RAM_QAM_ACTIVE_SYM_RCRATE_HI__A",
	0x831FC2 => "SCU_RAM_QAM_ACTIVE_SYM_RCRATE_LO__A",
	0x831FC3 => "SCU_RAM_QAM_RD_RSV_5__A",
	0x831FC4 => "SCU_RAM_QAM_RD_RSV_6__A",
	0x831FC5 => "SCU_RAM_QAM_RD_RSV_7__A",
	0x831FC6 => "SCU_RAM_QAM_RD_RSV_8__A",
	0x831FC7 => "SCU_RAM_QAM_RD_RSV_9__A",
	0x831FC8 => "SCU_RAM_QAM_RD_RSV_10__A",
	0x831FC9 => "SCU_RAM_QAM_AGC_TPOW_OFFS__A",
	0x831FCA => "SCU_RAM_QAM_FSM_STATE__A",
	0x831FCB => "SCU_RAM_QAM_FSM_STATE_NEW__A",
	0x831FCC => "SCU_RAM_QAM_FSM_LOCK_FLAGS__A",
	0x831FCD => "SCU_RAM_QAM_FSM_RATE_VARIATION__A",
	0x831FCE => "SCU_RAM_QAM_FSM_FREQ_VARIATION__A",
	0x831FCF => "SCU_RAM_QAM_ERR_STATE__A",
	0x831FD0 => "SCU_RAM_QAM_ERR_LOCK_FLAGS__A",
	0x831FD1 => "SCU_RAM_QAM_EQ_LOCK__A",
	0x831FD2 => "SCU_RAM_QAM_EQ_STATE__A",
	0x831FD3 => "SCU_RAM_QAM_RD_RSV_0__A",
	0x831FD4 => "SCU_RAM_QAM_RD_RSV_1__A",
	0x831FD5 => "SCU_RAM_QAM_RD_RSV_2__A",
	0x831FD6 => "SCU_RAM_QAM_RD_RSV_3__A",
	0x831FD7 => "SCU_RAM_VSB_CTL_MODE__A",
	0x831FD8 => "SCU_RAM_VSB_NOTCH_THRESHOLD__A",
	0x831FD9 => "SCU_RAM_VSB_RSV_0__A",
	0x831FDA => "SCU_RAM_VSB_RSV_1__A",
	0x831FDB => "SCU_RAM_VSB_RSV_2__A",
	0x831FDC => "SCU_RAM_VSB_RSV_3__A",
	0x831FDD => "SCU_RAM_VSB_RSV_4__A",
	0x831FDE => "SCU_RAM_VSB_RSV_5__A",
	0x831FDF => "SCU_RAM_VSB_RSV_6__A",
	0x831FE0 => "SCU_RAM_VSB_RSV_7__A",
	0x831FE1 => "SCU_RAM_VSB_RSV_8__A",
	0x831FE2 => "SCU_RAM_VSB_RSV_9__A",
	0x831FE3 => "SCU_RAM_VSB_RSV_10__A",
	0x831FE4 => "SCU_RAM_VSB_RSV_11__A",
	0x831FE5 => "SCU_RAM_VSB_RSV_12__A",
	0x831FE6 => "SCU_RAM_VSB_RSV_13__A",
	0x831FE7 => "SCU_RAM_VSB_AGC_POW_TGT__A",
	0x831FE8 => "SCU_RAM_VSB_OUTER_LOOP_CYCLE__A",
	0x831FE9 => "SCU_RAM_VSB_FIELD_NUMBER__A",
	0x831FEA => "SCU_RAM_VSB_SEGMENT_NUMBER__A",
	0x831FEB => "SCU_RAM_DRIVER_VER_HI__A",
	0x831FEC => "SCU_RAM_DRIVER_VER_LO__A",
	0x831FED => "SCU_RAM_PARAM_15__A",
	0x831FEE => "SCU_RAM_PARAM_14__A",
	0x831FEF => "SCU_RAM_PARAM_13__A",
	0x831FF0 => "SCU_RAM_PARAM_12__A",
	0x831FF1 => "SCU_RAM_PARAM_11__A",
	0x831FF2 => "SCU_RAM_PARAM_10__A",
	0x831FF3 => "SCU_RAM_PARAM_9__A",
	0x831FF4 => "SCU_RAM_PARAM_8__A",
	0x831FF5 => "SCU_RAM_PARAM_7__A",
	0x831FF6 => "SCU_RAM_PARAM_6__A",
	0x831FF7 => "SCU_RAM_PARAM_5__A",
	0x831FF8 => "SCU_RAM_PARAM_4__A",
	0x831FF9 => "SCU_RAM_PARAM_3__A",
	0x831FFA => "SCU_RAM_PARAM_2__A",
	0x831FFB => "SCU_RAM_PARAM_1__A",
	0x831FFC => "SCU_RAM_PARAM_0__A",
	0x831FFD => "SCU_RAM_COMMAND__A",
	0x831FFE => "SCU_RAM_VERSION_HI__A",
	0x831FFF => "SCU_RAM_VERSION_LO__A",
	0x400000 => "SIO_COMM_EXEC__A",
	0x400001 => "SIO_COMM_STATE__A",
	0x400002 => "SIO_COMM_MB__A",
	0x400003 => "SIO_COMM_INT_REQ__A",
	0x400005 => "SIO_COMM_INT_STA__A",
	0x400006 => "SIO_COMM_INT_MSK__A",
	0x400007 => "SIO_COMM_INT_STM__A",
	0x410000 => "SIO_TOP_COMM_EXEC__A",
	0x41000F => "SIO_TOP_COMM_KEY__A",
	0x410012 => "SIO_TOP_JTAGID_LO__A",
	0x410013 => "SIO_TOP_JTAGID_HI__A",
	0x420010 => "SIO_HI_RA_RAM_S0_FLG_SMM__A",
	0x420011 => "SIO_HI_RA_RAM_S0_DEV_ID__A",
	0x420012 => "SIO_HI_RA_RAM_S0_FLG_CRC__A",
	0x420013 => "SIO_HI_RA_RAM_S0_FLG_ACC__A",
	0x420014 => "SIO_HI_RA_RAM_S0_STATE__A",
	0x420015 => "SIO_HI_RA_RAM_S0_BLK_BNK__A",
	0x420016 => "SIO_HI_RA_RAM_S0_ADDR__A",
	0x420017 => "SIO_HI_RA_RAM_S0_CRC__A",
	0x420018 => "SIO_HI_RA_RAM_S0_BUFFER__A",
	0x420019 => "SIO_HI_RA_RAM_S0_RMWBUF__A",
	0x42001A => "SIO_HI_RA_RAM_S0_FLG_VB__A",
	0x42001B => "SIO_HI_RA_RAM_S0_TEMP0__A",
	0x42001C => "SIO_HI_RA_RAM_S0_TEMP1__A",
	0x42001D => "SIO_HI_RA_RAM_S0_OFFSET__A",
	0x420020 => "SIO_HI_RA_RAM_S1_FLG_SMM__A",
	0x420021 => "SIO_HI_RA_RAM_S1_DEV_ID__A",
	0x420022 => "SIO_HI_RA_RAM_S1_FLG_CRC__A",
	0x420023 => "SIO_HI_RA_RAM_S1_FLG_ACC__A",
	0x420024 => "SIO_HI_RA_RAM_S1_STATE__A",
	0x420025 => "SIO_HI_RA_RAM_S1_BLK_BNK__A",
	0x420026 => "SIO_HI_RA_RAM_S1_ADDR__A",
	0x420027 => "SIO_HI_RA_RAM_S1_CRC__A",
	0x420028 => "SIO_HI_RA_RAM_S1_BUFFER__A",
	0x420029 => "SIO_HI_RA_RAM_S1_RMWBUF__A",
	0x42002A => "SIO_HI_RA_RAM_S1_FLG_VB__A",
	0x42002B => "SIO_HI_RA_RAM_S1_TEMP0__A",
	0x42002C => "SIO_HI_RA_RAM_S1_TEMP1__A",
	0x42002D => "SIO_HI_RA_RAM_S1_OFFSET__A",
	0x420030 => "SIO_HI_RA_RAM_SEMA__A",
	0x420031 => "SIO_HI_RA_RAM_RES__A",
	0x420032 => "SIO_HI_RA_RAM_CMD__A",
	0x420033 => "SIO_HI_RA_RAM_PAR_1__A",
	0x420034 => "SIO_HI_RA_RAM_PAR_2__A",
	0x420035 => "SIO_HI_RA_RAM_PAR_3__A",
	0x420036 => "SIO_HI_RA_RAM_PAR_4__A",
	0x420037 => "SIO_HI_RA_RAM_PAR_5__A",
	0x420038 => "SIO_HI_RA_RAM_PAR_6__A",
	0x42006E => "SIO_HI_RA_RAM_AB_TEMP__A",
	0x42006F => "SIO_HI_RA_RAM_I2C_CTL__A",
	0x420070 => "SIO_HI_RA_RAM_VB_ENTRY0__A",
	0x420071 => "SIO_HI_RA_RAM_VB_OFFSET0__A",
	0x420072 => "SIO_HI_RA_RAM_VB_ENTRY1__A",
	0x420073 => "SIO_HI_RA_RAM_VB_OFFSET1__A",
	0x420074 => "SIO_HI_RA_RAM_VB_ENTRY2__A",
	0x420075 => "SIO_HI_RA_RAM_VB_OFFSET2__A",
	0x420076 => "SIO_HI_RA_RAM_VB_ENTRY3__A",
	0x420077 => "SIO_HI_RA_RAM_VB_OFFSET3__A",
	0x420078 => "SIO_HI_RA_RAM_VB_ENTRY4__A",
	0x420079 => "SIO_HI_RA_RAM_VB_OFFSET4__A",
	0x42007A => "SIO_HI_RA_RAM_VB_ENTRY5__A",
	0x42007B => "SIO_HI_RA_RAM_VB_OFFSET5__A",
	0x42007C => "SIO_HI_RA_RAM_VB_ENTRY6__A",
	0x42007D => "SIO_HI_RA_RAM_VB_OFFSET6__A",
	0x42007E => "SIO_HI_RA_RAM_VB_ENTRY7__A",
	0x42007F => "SIO_HI_RA_RAM_VB_OFFSET7__A",
	0x430000 => "SIO_HI_IF_RAM_TRP_BPT_0__A",
	0x430001 => "SIO_HI_IF_RAM_TRP_BPT_1__A",
	0x430002 => "SIO_HI_IF_RAM_TRP_STK_0__A",
	0x430003 => "SIO_HI_IF_RAM_TRP_STK_1__A",
	0x430300 => "SIO_HI_IF_RAM_FUN_BASE__A",
	0x440000 => "SIO_HI_IF_COMM_EXEC__A",
	0x440001 => "SIO_HI_IF_COMM_STATE__A",
	0x440003 => "SIO_HI_IF_COMM_INT_REQ__A",
	0x440005 => "SIO_HI_IF_COMM_INT_STA__A",
	0x440006 => "SIO_HI_IF_COMM_INT_MSK__A",
	0x440007 => "SIO_HI_IF_COMM_INT_STM__A",
	0x440010 => "SIO_HI_IF_STK_0__A",
	0x440011 => "SIO_HI_IF_STK_1__A",
	0x440012 => "SIO_HI_IF_STK_2__A",
	0x440013 => "SIO_HI_IF_STK_3__A",
	0x44001F => "SIO_HI_IF_BPT_IDX__A",
	0x440020 => "SIO_HI_IF_BPT__A",
	0x450000 => "SIO_CC_COMM_EXEC__A",
	0x450010 => "SIO_CC_PLL_MODE__A",
	0x450011 => "SIO_CC_PLL_TEST__A",
	0x450012 => "SIO_CC_PLL_LOCK__A",
	0x450014 => "SIO_CC_CLK_MODE__A",
	0x450015 => "SIO_CC_PWD_MODE__A",
	0x450016 => "SIO_CC_SOFT_RST__A",
	0x450017 => "SIO_CC_UPDATE__A",
	0x460000 => "SIO_SA_COMM_EXEC__A",
	0x460003 => "SIO_SA_COMM_INT_REQ__A",
	0x460005 => "SIO_SA_COMM_INT_STA__A",
	0x460006 => "SIO_SA_COMM_INT_MSK__A",
	0x460007 => "SIO_SA_COMM_INT_STM__A",
	0x460010 => "SIO_SA_PRESCALER__A",
	0x460011 => "SIO_SA_TX_DATA0__A",
	0x460012 => "SIO_SA_TX_DATA1__A",
	0x460013 => "SIO_SA_TX_DATA2__A",
	0x460014 => "SIO_SA_TX_DATA3__A",
	0x460015 => "SIO_SA_TX_LENGTH__A",
	0x460016 => "SIO_SA_TX_COMMAND__A",
	0x460017 => "SIO_SA_TX_STATUS__A",
	0x460018 => "SIO_SA_RX_DATA0__A",
	0x460019 => "SIO_SA_RX_DATA1__A",
	0x46001A => "SIO_SA_RX_LENGTH__A",
	0x46001B => "SIO_SA_RX_COMMAND__A",
	0x46001C => "SIO_SA_RX_STATUS__A",
	0x7F0000 => "SIO_PDR_COMM_EXEC__A",
	0x7F0010 => "SIO_PDR_MON_CFG__A",
	0x7F0011 => "SIO_PDR_FDB_CFG__A",
	0x7F0012 => "SIO_PDR_SMA_RX_SEL__A",
	0x7F0013 => "SIO_PDR_SMA_TX_SILENT__A",
	0x7F0014 => "SIO_PDR_UIO_IN_LO__A",
	0x7F0015 => "SIO_PDR_UIO_IN_HI__A",
	0x7F0016 => "SIO_PDR_UIO_OUT_LO__A",
	0x7F0017 => "SIO_PDR_UIO_OUT_HI__A",
	0x7F0018 => "SIO_PDR_PWM1_MODE__A",
	0x7F0019 => "SIO_PDR_PWM1_PRESCALE__A",
	0x7F001A => "SIO_PDR_PWM1_VALUE__A",
	0x7F001C => "SIO_PDR_PWM2_MODE__A",
	0x7F001D => "SIO_PDR_PWM2_PRESCALE__A",
	0x7F001E => "SIO_PDR_PWM2_VALUE__A",
	0x7F001F => "SIO_PDR_OHW_CFG__A",
	0x7F0020 => "SIO_PDR_I2S_WS_CFG__A",
	0x7F0021 => "SIO_PDR_GPIO_CFG__A",
	0x7F0022 => "SIO_PDR_IRQN_CFG__A",
	0x7F0023 => "SIO_PDR_OOB_CRX_CFG__A",
	0x7F0024 => "SIO_PDR_OOB_DRX_CFG__A",
	0x7F0025 => "SIO_PDR_MSTRT_CFG__A",
	0x7F0026 => "SIO_PDR_MERR_CFG__A",
	0x7F0028 => "SIO_PDR_MCLK_CFG__A",
	0x7F0029 => "SIO_PDR_MVAL_CFG__A",
	0x7F002A => "SIO_PDR_MD0_CFG__A",
	0x7F002B => "SIO_PDR_MD1_CFG__A",
	0x7F002C => "SIO_PDR_MD2_CFG__A",
	0x7F002D => "SIO_PDR_MD3_CFG__A",
	0x7F002F => "SIO_PDR_MD4_CFG__A",
	0x7F0030 => "SIO_PDR_MD5_CFG__A",
	0x7F0031 => "SIO_PDR_MD6_CFG__A",
	0x7F0032 => "SIO_PDR_MD7_CFG__A",
	0x7F0033 => "SIO_PDR_I2C_SCL1_CFG__A",
	0x7F0034 => "SIO_PDR_I2C_SDA1_CFG__A",
	0x7F0036 => "SIO_PDR_VSYNC_CFG__A",
	0x7F0037 => "SIO_PDR_SMA_RX_CFG__A",
	0x7F0038 => "SIO_PDR_SMA_TX_CFG__A",
	0x7F003F => "SIO_PDR_I2C_SDA2_CFG__A",
	0x7F0040 => "SIO_PDR_I2C_SCL2_CFG__A",
	0x7F0041 => "SIO_PDR_I2S_CL_CFG__A",
	0x7F0042 => "SIO_PDR_I2S_DA_CFG__A",
	0x7F0050 => "SIO_PDR_GPIO_GPIO_FNC__A",
	0x7F0051 => "SIO_PDR_IRQN_GPIO_FNC__A",
	0x7F0052 => "SIO_PDR_MSTRT_GPIO_FNC__A",
	0x7F0053 => "SIO_PDR_MERR_GPIO_FNC__A",
	0x7F0054 => "SIO_PDR_MCLK_GPIO_FNC__A",
	0x7F0055 => "SIO_PDR_MVAL_GPIO_FNC__A",
	0x7F0056 => "SIO_PDR_MD0_GPIO_FNC__A",
	0x7F0057 => "SIO_PDR_MD1_GPIO_FNC__A",
	0x7F0058 => "SIO_PDR_MD2_GPIO_FNC__A",
	0x7F0059 => "SIO_PDR_MD3_GPIO_FNC__A",
	0x7F005A => "SIO_PDR_MD4_GPIO_FNC__A",
	0x7F005B => "SIO_PDR_MD5_GPIO_FNC__A",
	0x7F005C => "SIO_PDR_MD6_GPIO_FNC__A",
	0x7F005D => "SIO_PDR_MD7_GPIO_FNC__A",
	0x7F005E => "SIO_PDR_SMA_RX_GPIO_FNC__A",
	0x7F005F => "SIO_PDR_SMA_TX_GPIO_FNC__A",
	0x1C00000 => "VSB_COMM_EXEC__A",
	0x1C00002 => "VSB_COMM_MB__A",
	0x1C00003 => "VSB_COMM_INT_REQ__A",
	0x1C00005 => "VSB_COMM_INT_STA__A",
	0x1C00006 => "VSB_COMM_INT_MSK__A",
	0x1C00007 => "VSB_COMM_INT_STM__A",
	0x1C10000 => "VSB_TOP_COMM_EXEC__A",
	0x1C10002 => "VSB_TOP_COMM_MB__A",
	0x1C10003 => "VSB_TOP_COMM_INT_REQ__A",
	0x1C10005 => "VSB_TOP_COMM_INT_STA__A",
	0x1C10006 => "VSB_TOP_COMM_INT_MSK__A",
	0x1C10007 => "VSB_TOP_COMM_INT_STM__A",
	0x1C10010 => "VSB_TOP_CKGN1ACQ__A",
	0x1C10011 => "VSB_TOP_CKGN1TRK__A",
	0x1C10012 => "VSB_TOP_CKGN2ACQ__A",
	0x1C10013 => "VSB_TOP_CKGN2TRK__A",
	0x1C10014 => "VSB_TOP_CKGN3__A",
	0x1C10015 => "VSB_TOP_CYGN1ACQ__A",
	0x1C10016 => "VSB_TOP_CYGN1TRK__A",
	0x1C10017 => "VSB_TOP_CYGN2ACQ__A",
	0x1C10018 => "VSB_TOP_CYGN2TRK__A",
	0x1C10019 => "VSB_TOP_CYGN3__A",
	0x1C1001A => "VSB_TOP_SYNCCTRLWORD__A",
	0x1C1001B => "VSB_TOP_MAINSMUP__A",
	0x1C1001C => "VSB_TOP_EQSMUP__A",
	0x1C1001D => "VSB_TOP_SYSMUXCTRL__A",
	0x1C1001E => "VSB_TOP_SNRTH_RCA1__A",
	0x1C1001F => "VSB_TOP_SNRTH_RCA2__A",
	0x1C10020 => "VSB_TOP_SNRTH_DDM1__A",
	0x1C10021 => "VSB_TOP_SNRTH_DDM2__A",
	0x1C10022 => "VSB_TOP_SNRTH_PT__A",
	0x1C10023 => "VSB_TOP_CYSMSTATES__A",
	0x1C10024 => "VSB_TOP_SMALL_NOTCH_CONTROL__A",
	0x1C10025 => "VSB_TOP_TAPREADCYC__A",
	0x1C10026 => "VSB_TOP_VALIDPKLVL__A",
	0x1C10027 => "VSB_TOP_CENTROID_FINE_DELAY__A",
	0x1C10028 => "VSB_TOP_CENTROID_SMACH_DELAY__A",
	0x1C10029 => "VSB_TOP_SNR__A",
	0x1C1002A => "VSB_TOP_LOCKSTATUS__A",
	0x1C1002B => "VSB_TOP_CTST__A",
	0x1C1002C => "VSB_TOP_EQSMRSTCTRL__A",
	0x1C1002D => "VSB_TOP_EQSMTRNCTRL__A",
	0x1C1002E => "VSB_TOP_EQSMRCA1CTRL__A",
	0x1C1002F => "VSB_TOP_EQSMRCA2CTRL__A",
	0x1C10030 => "VSB_TOP_EQSMDDM1CTRL__A",
	0x1C10031 => "VSB_TOP_EQSMDDM2CTRL__A",
	0x1C10032 => "VSB_TOP_SYSSMRSTCTRL__A",
	0x1C10033 => "VSB_TOP_SYSSMCYCTRL__A",
	0x1C10034 => "VSB_TOP_SYSSMTRNCTRL__A",
	0x1C10035 => "VSB_TOP_SYSSMEQCTRL__A",
	0x1C10036 => "VSB_TOP_SYSSMAGCCTRL__A",
	0x1C10037 => "VSB_TOP_SYSSMCTCTRL__A",
	0x1C10038 => "VSB_TOP_EQCTRL__A",
	0x1C10039 => "VSB_TOP_PREEQAGCCTRL__A",
	0x1C1003A => "VSB_TOP_PREEQAGCPWRREFLVLHI__A",
	0x1C1003B => "VSB_TOP_PREEQAGCPWRREFLVLLO__A",
	0x1C1003C => "VSB_TOP_CORINGSEL__A",
	0x1C1003D => "VSB_TOP_BEDETCTRL__A",
	0x1C1003E => "VSB_TOP_LBAGCREFLVL__A",
	0x1C1003F => "VSB_TOP_UBAGCREFLVL__A",
	0x1C10040 => "VSB_TOP_NOTCH1_BIN_NUM__A",
	0x1C10041 => "VSB_TOP_NOTCH2_BIN_NUM__A",
	0x1C10042 => "VSB_TOP_NOTCH_START_BIN_NUM__A",
	0x1C10043 => "VSB_TOP_NOTCH_STOP_BIN_NUM__A",
	0x1C10044 => "VSB_TOP_NOTCH_TEST_DURATION__A",
	0x1C10045 => "VSB_TOP_RESULT_LARGE_PEAK_BIN__A",
	0x1C10046 => "VSB_TOP_RESULT_LARGE_PEAK_VALUE__A",
	0x1C10047 => "VSB_TOP_RESULT_SMALL_PEAK_BIN__A",
	0x1C10048 => "VSB_TOP_RESULT_SMALL_PEAK_VALUE__A",
	0x1C10049 => "VSB_TOP_NOTCH_SWEEP_RUNNING__A",
	0x1C1004A => "VSB_TOP_PREEQDAGCRATIO__A",
	0x1C1004B => "VSB_TOP_AGC_TRUNCCTRL__A",
	0x1C1004C => "VSB_TOP_BEAGC_DEADZONEINIT__A",
	0x1C1004D => "VSB_TOP_BEAGC_REFLEVEL__A",
	0x1C1004E => "VSB_TOP_BEAGC_GAINSHIFT__A",
	0x1C1004F => "VSB_TOP_BEAGC_REGINIT__A",
	0x1C10050 => "VSB_TOP_BEAGC_SCALE__A",
	0x1C10051 => "VSB_TOP_CFAGC_DEADZONEINIT__A",
	0x1C10052 => "VSB_TOP_CFAGC_REFLEVEL__A",
	0x1C10053 => "VSB_TOP_CFAGC_GAINSHIFT__A",
	0x1C10054 => "VSB_TOP_CFAGC_REGINIT__A",
	0x1C10055 => "VSB_TOP_CFAGC_SCALE__A",
	0x1C10056 => "VSB_TOP_CKTRKONCTL__A",
	0x1C10057 => "VSB_TOP_CYTRKONCTL__A",
	0x1C10058 => "VSB_TOP_PTONCTL__A",
	0x1C10059 => "VSB_TOP_NOTCH_SCALE_1__A",
	0x1C1005A => "VSB_TOP_NOTCH_SCALE_2__A",
	0x1C1005B => "VSB_TOP_FIRSTLARGFFETAP__A",
	0x1C1005C => "VSB_TOP_FIRSTLARGFFETAPADDR__A",
	0x1C1005D => "VSB_TOP_SECONDLARGFFETAP__A",
	0x1C1005E => "VSB_TOP_SECONDLARGFFETAPADDR__A",
	0x1C1005F => "VSB_TOP_FIRSTLARGDFETAP__A",
	0x1C10060 => "VSB_TOP_FIRSTLARGDFETAPADDR__A",
	0x1C10061 => "VSB_TOP_SECONDLARGDFETAP__A",
	0x1C10062 => "VSB_TOP_SECONDLARGDFETAPADDR__A",
	0x1C10063 => "VSB_TOP_PARAOWDBUS__A",
	0x1C10064 => "VSB_TOP_PARAOWCTRL__A",
	0x1C10065 => "VSB_TOP_CURRENTSEGLOCAT__A",
	0x1C10066 => "VSB_TOP_MEASUREMENT_PERIOD__A",
	0x1C10067 => "VSB_TOP_NR_SYM_ERRS__A",
	0x1C10068 => "VSB_TOP_ERR_ENERGY_L__A",
	0x1C10069 => "VSB_TOP_ERR_ENERGY_H__A",
	0x1C1006A => "VSB_TOP_SLICER_SEL_8LEV__A",
	0x1C1006B => "VSB_TOP_BNFIELD__A",
	0x1C1006C => "VSB_TOP_CLPLASTNUM__A",
	0x1C1006D => "VSB_TOP_BNSQERR__A",
	0x1C1006E => "VSB_TOP_BNTHRESH__A",
	0x1C1006F => "VSB_TOP_BNCLPNUM__A",
	0x1C10070 => "VSB_TOP_PHASELOCKCTRL__A",
	0x1C10071 => "VSB_TOP_DLOCKACCUM__A",
	0x1C10072 => "VSB_TOP_PLOCKACCUM__A",
	0x1C10073 => "VSB_TOP_CLOCKACCUM__A",
	0x1C10074 => "VSB_TOP_DCRMVACUMI__A",
	0x1C10075 => "VSB_TOP_DCRMVACUMQ__A",
	0x1C20000 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO1__A",
	0x1C20001 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO2__A",
	0x1C20002 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO3__A",
	0x1C20003 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO4__A",
	0x1C20004 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO5__A",
	0x1C20005 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO6__A",
	0x1C20006 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO7__A",
	0x1C20007 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO8__A",
	0x1C20008 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO9__A",
	0x1C20009 => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO10__A",
	0x1C2000A => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO11__A",
	0x1C2000B => "VSB_SYSCTRL_RAM0_FFETRAINLKRATIO12__A",
	0x1C2000C => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO1__A",
	0x1C2000D => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO2__A",
	0x1C2000E => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO3__A",
	0x1C2000F => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO4__A",
	0x1C20010 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO5__A",
	0x1C20011 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO6__A",
	0x1C20012 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO7__A",
	0x1C20013 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO8__A",
	0x1C20014 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO9__A",
	0x1C20015 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO10__A",
	0x1C20016 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO11__A",
	0x1C20017 => "VSB_SYSCTRL_RAM0_FFERCA1TRAINLKRATIO12__A",
	0x1C20018 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO1__A",
	0x1C20019 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO2__A",
	0x1C2001A => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO3__A",
	0x1C2001B => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO4__A",
	0x1C2001C => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO5__A",
	0x1C2001D => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO6__A",
	0x1C2001E => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO7__A",
	0x1C2001F => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO8__A",
	0x1C20020 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO9__A",
	0x1C20021 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO10__A",
	0x1C20022 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO11__A",
	0x1C20023 => "VSB_SYSCTRL_RAM0_FFERCA1DATALKRATIO12__A",
	0x1C20024 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO1__A",
	0x1C20025 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO2__A",
	0x1C20026 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO3__A",
	0x1C20027 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO4__A",
	0x1C20028 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO5__A",
	0x1C20029 => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO6__A",
	0x1C2002A => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO7__A",
	0x1C2002B => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO8__A",
	0x1C2002C => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO9__A",
	0x1C2002D => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO10__A",
	0x1C2002E => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO11__A",
	0x1C2002F => "VSB_SYSCTRL_RAM0_FFERCA2TRAINLKRATIO12__A",
	0x1C20030 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO1__A",
	0x1C20031 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO2__A",
	0x1C20032 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO3__A",
	0x1C20033 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO4__A",
	0x1C20034 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO5__A",
	0x1C20035 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO6__A",
	0x1C20036 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO7__A",
	0x1C20037 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO8__A",
	0x1C20038 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO9__A",
	0x1C20039 => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO10__A",
	0x1C2003A => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO11__A",
	0x1C2003B => "VSB_SYSCTRL_RAM0_FFERCA2DATALKRATIO12__A",
	0x1C2003C => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO1__A",
	0x1C2003D => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO2__A",
	0x1C2003E => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO3__A",
	0x1C2003F => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO4__A",
	0x1C20040 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO5__A",
	0x1C20041 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO6__A",
	0x1C20042 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO7__A",
	0x1C20043 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO8__A",
	0x1C20044 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO9__A",
	0x1C20045 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO10__A",
	0x1C20046 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO11__A",
	0x1C20047 => "VSB_SYSCTRL_RAM0_FFEDDM1TRAINLKRATIO12__A",
	0x1C20048 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO1__A",
	0x1C20049 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO2__A",
	0x1C2004A => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO3__A",
	0x1C2004B => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO4__A",
	0x1C2004C => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO5__A",
	0x1C2004D => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO6__A",
	0x1C2004E => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO7__A",
	0x1C2004F => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO8__A",
	0x1C20050 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO9__A",
	0x1C20051 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO10__A",
	0x1C20052 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO11__A",
	0x1C20053 => "VSB_SYSCTRL_RAM0_FFEDDM1DATALKRATIO12__A",
	0x1C20054 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO1__A",
	0x1C20055 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO2__A",
	0x1C20056 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO3__A",
	0x1C20057 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO4__A",
	0x1C20058 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO5__A",
	0x1C20059 => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO6__A",
	0x1C2005A => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO7__A",
	0x1C2005B => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO8__A",
	0x1C2005C => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO9__A",
	0x1C2005D => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO10__A",
	0x1C2005E => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO11__A",
	0x1C2005F => "VSB_SYSCTRL_RAM0_FFEDDM2TRAINLKRATIO12__A",
	0x1C20060 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO1__A",
	0x1C20061 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO2__A",
	0x1C20062 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO3__A",
	0x1C20063 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO4__A",
	0x1C20064 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO5__A",
	0x1C20065 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO6__A",
	0x1C20066 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO7__A",
	0x1C20067 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO8__A",
	0x1C20068 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO9__A",
	0x1C20069 => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO10__A",
	0x1C2006A => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO11__A",
	0x1C2006B => "VSB_SYSCTRL_RAM0_FFEDDM2DATALKRATIO12__A",
	0x1C2006C => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN1__A",
	0x1C2006D => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN2__A",
	0x1C2006E => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN3__A",
	0x1C2006F => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN4__A",
	0x1C20070 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN5__A",
	0x1C20071 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN6__A",
	0x1C20072 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN7__A",
	0x1C20073 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN8__A",
	0x1C20074 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN9__A",
	0x1C20075 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN10__A",
	0x1C20076 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN11__A",
	0x1C20077 => "VSB_SYSCTRL_RAM0_FIRTRAINGAIN12__A",
	0x1C20078 => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN1__A",
	0x1C20079 => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN2__A",
	0x1C2007A => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN3__A",
	0x1C2007B => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN4__A",
	0x1C2007C => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN5__A",
	0x1C2007D => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN6__A",
	0x1C2007E => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN7__A",
	0x1C2007F => "VSB_SYSCTRL_RAM0_FIRRCA1GAIN8__A",
	0x1C30000 => "VSB_SYSCTRL_RAM1_FIRRCA1GAIN9__A",
	0x1C30001 => "VSB_SYSCTRL_RAM1_FIRRCA1GAIN10__A",
	0x1C30002 => "VSB_SYSCTRL_RAM1_FIRRCA1GAIN11__A",
	0x1C30003 => "VSB_SYSCTRL_RAM1_FIRRCA1GAIN12__A",
	0x1C30004 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN1__A",
	0x1C30005 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN2__A",
	0x1C30006 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN3__A",
	0x1C30007 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN4__A",
	0x1C30008 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN5__A",
	0x1C30009 => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN6__A",
	0x1C3000A => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN7__A",
	0x1C3000B => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN8__A",
	0x1C3000C => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN9__A",
	0x1C3000D => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN10__A",
	0x1C3000E => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN11__A",
	0x1C3000F => "VSB_SYSCTRL_RAM1_FIRRCA2GAIN12__A",
	0x1C30010 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN1__A",
	0x1C30011 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN2__A",
	0x1C30012 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN3__A",
	0x1C30013 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN4__A",
	0x1C30014 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN5__A",
	0x1C30015 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN6__A",
	0x1C30016 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN7__A",
	0x1C30017 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN8__A",
	0x1C30018 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN9__A",
	0x1C30019 => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN10__A",
	0x1C3001A => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN11__A",
	0x1C3001B => "VSB_SYSCTRL_RAM1_FIRDDM1GAIN12__A",
	0x1C3001C => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN1__A",
	0x1C3001D => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN2__A",
	0x1C3001E => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN3__A",
	0x1C3001F => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN4__A",
	0x1C30020 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN5__A",
	0x1C30021 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN6__A",
	0x1C30022 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN7__A",
	0x1C30023 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN8__A",
	0x1C30024 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN9__A",
	0x1C30025 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN10__A",
	0x1C30026 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN11__A",
	0x1C30027 => "VSB_SYSCTRL_RAM1_FIRDDM2GAIN12__A",
	0x1C30028 => "VSB_SYSCTRL_RAM1_DFETRAINLKRATIO__A",
	0x1C30029 => "VSB_SYSCTRL_RAM1_DFERCA1TRAINLKRATIO__A",
	0x1C3002A => "VSB_SYSCTRL_RAM1_DFERCA1DATALKRATIO__A",
	0x1C3002B => "VSB_SYSCTRL_RAM1_DFERCA2TRAINLKRATIO__A",
	0x1C3002C => "VSB_SYSCTRL_RAM1_DFERCA2DATALKRATIO__A",
	0x1C3002D => "VSB_SYSCTRL_RAM1_DFEDDM1TRAINLKRATIO__A",
	0x1C3002E => "VSB_SYSCTRL_RAM1_DFEDDM1DATALKRATIO__A",
	0x1C3002F => "VSB_SYSCTRL_RAM1_DFEDDM2TRAINLKRATIO__A",
	0x1C30030 => "VSB_SYSCTRL_RAM1_DFEDDM2DATALKRATIO__A",
	0x1C30031 => "VSB_SYSCTRL_RAM1_DFETRAINGAIN__A",
	0x1C30032 => "VSB_SYSCTRL_RAM1_DFERCA1GAIN__A",
	0x1C30033 => "VSB_SYSCTRL_RAM1_DFERCA2GAIN__A",
	0x1C30034 => "VSB_SYSCTRL_RAM1_DFEDDM1GAIN__A",
	0x1C30035 => "VSB_SYSCTRL_RAM1_DFEDDM2GAIN__A",
	0x1C40000 => "VSB_TCMEQ_RAM__A",
	0x1C50000 => "VSB_FCPRE_RAM__A",
	0x1C60000 => "VSB_EQTAP_RAM__A",
);

my $old_flags = -1;
my $old_reg;
my $old_timestamp;
sub parse_drxj_addr($$$$$)
{
	my $timestamp = shift;
	my $addr = shift;
	my $n = shift;
	my $app_data = shift;
	my $write = shift;
	my $j = 0;

	my ($reg, $flags, $cmd, $bits);

	$n = hex($n);

	goto parse_error if ($n < 2);
	if ($old_flags < 0) {
		my $r0 = hex(substr($app_data, $j, 2));
		my $r1 = hex(substr($app_data, $j + 3, 2));
		$n -= 2;
		$j += 6;

		if ($r0 & 1) {
			goto parse_error if ($n < 2);
			# Decode a Long format
			my $r2 = hex(substr($app_data, $j, 2));
			my $r3 = hex(substr($app_data, $j + 3, 2));
			$n -= 2;
			$j += 6;

			$reg = ($r0 >> 1) |
			       ($r1 << 16) |
			       (($r2 & 0x0f) << 24) |
			       ($r3 << 7);

			$flags = $r2 & 0xf0;
		} else {
			$flags = 0;
			$reg = ($r0 >> 1) |
			       ($r1 & 0x0f << 16) |
			       ($r1 & 0xf0 << 18);
		}
	} else {
		goto parse_error if ($write);	# Parse error!!! Should be a read!

		$flags = $old_flags;
		$reg = $old_reg;
		$old_flags = -1;
	}

	if ($n == 0) {
		goto parse_error if (!$write);	# Parse error!!! Should be a write!

		$old_flags = $flags;
		$old_reg = $reg;
		$old_timestamp = $timestamp;
		return;
	}

# Hack: may be useful for firmware dumps
#goto parse_block if ($n > 1);

	goto parse_block if ($n > 4 || $n == 3);
	goto parse_error if ($n != 2 && $n != 4);

	if (defined($drxj_map{$reg})) {
		$reg = $drxj_map{$reg};
	} else {
		$reg = sprintf "0x%08x", $reg;
	}

	my $data;
	if ($n == 2) {
		$bits = 16;
		$data = hex(substr($app_data, $j, 2)) |
			hex(substr($app_data, $j + 3, 2)) << 8;
	} else {
		$bits = 32;
		$data = hex(substr($app_data, $j, 2)) |
			hex(substr($app_data, $j + 3, 2)) << 16 |
			hex(substr($app_data, $j + 6, 2)) << 24 |
			hex(substr($app_data, $j + 9, 2)) << 8 if ($n > 2);
	}

	if ($flags) {
		my $descr;

		# That _seems_ to be the flags. Not sure through
		$descr .= "R/W/Modify " if ($flags & 0x10);
		$descr .= "Broadcast " if ($flags & 0x20);
		$descr .= "SingleMaster " if (($flags & 0xc0) == 0xc0);
		$descr .= "MultiMaster " if (($flags & 0xc0) == 0x40);
		$descr .= "ClearCRC " if (($flags & 0xc0) == 0x80);

		printf "$timestamp " if ($timestamp);

		if ($write) {
			printf "drxj_write%d_flags(state, 0x%s, %s, 0x%08x, 0x%02x); /* Flags = %s */\n", $bits, $addr, $reg, $data, $flags, $descr;
		} else {
			printf "drxj_read%d_flags(state, 0x%s, %s, 0x%02x) /* Read = 0x%08x Flags = %s */\n", $bits, $addr, $reg, $flags, $data, $descr;
		}

	} else {
		printf "$timestamp " if ($timestamp);
		if ($write) {
			printf "drxj_write%d(state, 0x%s, %s, 0x%08x, 0x%08x);\n", $bits, $addr, $reg, $data;
		} else {
			printf "drxj_read%d(state, 0x%s, %s, 0x%08x); /* Read = 0x%08x */\n", $bits, $addr, $reg, $data;
		}
	}

	return;

parse_block:
	# On several cases, we don't want to get the reg name
	# (e. g. during microcode load)
	$reg = sprintf "0x%08x", $reg;

	my $data = add_hex_mark(substr($app_data, $j));

	if ($flags) {
		my $descr;

		# That _seems_ to be the flags. Not sure through
		$descr .= "R/W/Modify " if ($flags & 0x10);
		$descr .= "Broadcast " if ($flags & 0x20);
		$descr .= "SingleMaster " if (($flags & 0xc0) == 0xc0);
		$descr .= "MultiMaster " if (($flags & 0xc0) == 0x40);
		$descr .= "ClearCRC " if (($flags & 0xc0) == 0x80);

		printf "$timestamp " if ($timestamp);
		if ($write) {
			printf "drxj_write_block_flags(state, 0x%s, %s, %d, %s, %d); /* Flags = %s */\n", $addr, $reg, $n, $data, $flags, $descr;
		} else {
			printf "drxj_read_block_flags(state, 0x%s, %s, %d, %d); /* Read = %s Flags = %s */\n", $addr, $reg, $n, $flags, $data, $descr;
		}

	} else {
		printf "$timestamp " if ($timestamp);
		if ($write) {
			printf "drxj_write_block(state, 0x%s, %s, %d, %s);\n", $addr, $reg, $n, $data;
		} else {
			printf "drxj_read_block(state, 0x%s, %s, %d); /* Read = %s */\n", $addr, $reg, $n, $data;
		}
	}

	return;

parse_error:
	# Fallback: Couldn't parse it
	if ($write) {
		if ($old_flags > 0) {
			if (defined($drxj_map{$old_reg})) {
				$old_reg = $drxj_map{$old_reg};
			} else {
				$old_reg = sprintf "0x%08x", $old_reg;
			}
			printf "$old_timestamp " if ($old_timestamp);
			printf "ERR: DRX-K write(state, 0x%s, %s, 0x%08x) without data. Probably an read ops + read error\n", $bits, $addr, $old_reg, $old_flags;
		}
		printf "$timestamp " if ($timestamp);
		my $data = add_hex_mark($app_data);
		printf "i2c_master_send(0x%s>>1, %s, %d);\n", $addr, $data, $n;
	} else {
		printf "$timestamp " if ($timestamp);
		my $data = add_hex_mark($app_data);
		printf "i2c_master_recv(0x%s>>1, &buf, 0x%d); /* %s */\n", $addr, $n, $data;
	}
}

##############
# Main program
##############

while (<>) {
	my $org_line = $_;

	my $timestamp;

	if ($show_timestamp) {
		$timestamp = $1 if (m/(.*)[4c]0 [0-9a-f]. [0-9a-f]. [0-9a-f]. [0-9a-f]. [0-9a-f]. [0-9a-f]. [0-9a-f]./);
		$timestamp =~ s/\s+$//;
	}

	tr/A-F/a-f/;
	if (m/c0 00 00 00 ([0-9a-f].) 00 01 00\s+[\<]+\s+([0-9a-f].)/) {
		if ($1 eq "43" && $2 eq "00") {
			$r43 = 1;
			$r40 = -1;
			$r42 = -1;
			$dir = 0;
			next;
		}

		my $reg = "0x$1";
		$reg = $reg_map{$reg} if defined($reg_map{$reg});

		printf "$timestamp " if ($show_timestamp && $show_em28xx);
		printf "em28xx_read_reg(dev, %s);\t\t/* read 0x%s */\n",
			$reg, $2 if ($show_em28xx);
		next;
	}
	if (m/40 00 00 00 ([0-9a-f].) 00 01 00\s+[\>]+\s+([0-9a-f].)/) {
		if ($r43 == 1) {
			if ($1 eq "42") {
				$r42 = "0x$2";
				if ($r40 >= 0) {
					output_ac97($timestamp) if ($show_ac97);
					next;
				}
				next;
			}
			$r43 = 0;
		}

		my $reg = "0x$1";
		$reg = $reg_map{$reg} if defined($reg_map{$reg});

		printf "$timestamp " if ($timestamp && $show_em28xx);
		printf "em28xx_write_reg(dev, %s, 0x%s);\n",
			$reg, $2  if ($show_em28xx);
		next;
	}
	if (m/c0 00 00 00 ([0-9a-f].) 00 02 00\s+[\<]+\s+([0-9a-f].) ([0-9a-f].)/) {
		if ($r43 == 1) {
			if ($1 eq "40") {
				$r40 = "0x$3$2";
				$dir = -1;

				if ($r42 >= 0) {
					output_ac97($timestamp) if ($show_ac97);
					next;
				}
				next;
			}
			$r43 = 0;
		}
		my $reg = "0x$1";
		$reg = $reg_map{$reg} if defined($reg_map{$reg});

		printf "$timestamp " if ($timestamp && $show_em28xx);
		printf "em28xx_read_reg16(dev, %s);\t\t/*read 0x%s%s */\n",
			$reg, $3, $2  if ($show_em28xx);
		next;
	}
	if (m/40 00 00 00 ([0-9a-f].) 00 02 00\s+[\>]+\s+([0-9a-f].) ([0-9a-f].)/) {
		if ($r43 == 1) {
			if ($1 eq "40") {
				$r40 = "0x$3$2";
				$dir = 1;

				if ($r42 >= 0) {
					output_ac97($timestamp) if ($show_ac97);
					next;
				}
				next;
			}
			$r43 = 0;
		}
		my $reg = "0x$1";
		$reg = $reg_map{$reg} if defined($reg_map{$reg});

		printf "$timestamp " if ($timestamp && $show_em28xx);
		printf "em28xx_write_reg16(dev, %s,0x%s%s);\n",
			$reg, $3, $2 if ($show_em28xx);
		next;
	}

	if (m/40 0[23] 00 00 ([0-9a-f].) 00 ([0-9a-f].) 00\s+[\>]+\s+([0-9a-f ]+)/) {
		if ($1 eq "52") {
			parse_drxj_addr($timestamp, $1, $2, $3, 1);
		} else {
			printf "$timestamp " if ($timestamp && $show_other_xfer);
			printf "i2c_master_send(0x$1>>1, %s, 0x$2);\n", add_hex_mark($3) if ($show_other_xfer);
		}
		next;
	}
	if (m/c0 0[23] 00 00 ([0-9a-f].) 00 ([0-9a-f].) 00\s+[\<]+\s+([0-9a-f ]+)/) {
		if ($1 eq "52") {
			parse_drxj_addr($timestamp, $1, $2, $3, 0);
		} else {
			printf "$timestamp " if ($timestamp && $show_other_xfer);
			printf "i2c_master_recv(0x$1>>1, &buf, 0x$2); /* %s */\n", add_hex_mark($3) if ($show_other_xfer);
		}
		next;
	}
	if (m/c0 0[23] 00 00 ([0-9a-f].) 00 ([0-9a-f].) 00\s+[\<]+/) {
		printf "$timestamp " if ($timestamp && $show_other_xfer);
		printf "i2c_master_recv(0x$1>>1, &buf, 0x$2); /* nothing returned */\n" if ($show_other_xfer);
		next;
	}
	print $org_line if ($show_other_lines);
}
