(*  :Title:	Data Type Association and Analysis  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*
    :Summary:	To provide the ability to determine data type of expressions
		To allow the adding of new data types in a tree hierarchy
 *)

(*  :Context:	SignalProcessing`Support`DataType`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	data type, number theory  *)

(*  :Source:	*)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0  *)

(*  :Limitation:  *)

(*
    :Discussion:  The initial recognized data types are (listed as a hierarchy)
    
		  Integer
		  Rational
		  Real			Imaginary
		  ComplexSymmetric	ComplexAntiSymmetric
				Complex


		  Three levels of abstraction of data type analysis:
		  numeric, associated, and expression.

		    --  primitives for numbers (like ComplexQ and RealQ)
			are added by "SupCode.m"
		    --  the user can specify new data types (heads)
			and associate symbols with them.
		    --  expressions can now be evaluated to determine
			their data type

		   The data type rule base is actually initialized outside
		   of the package so its display by the front end would
		   be free of contextual information which would make
		   the rules unreadable.

		   This file is still under development.  A tree structure
		   will be used in the future to maintain relationships
		   between data types.  Until then, adding new data types
		   will work, but this package will not "see" its
		   relationship to existing data types.
 *)

(*
    :Functions:	AddDataType
		AssociateDataType
		CombineDataType
		DataType
		DataTypes
		DataTypeQ
		DefConstant
		FTDataType
		GetPosition
		IrrationalQ
		IsComplexQ
		IsImaginaryQ
		IsIntegerQ
		IsIrrationalQ
		IsRationalQ
		IsRealQ
		SubDataTypeQ
		SymbolDataType
		SymbolDataTypeQ
		UnAssociateDataType
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ]


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Support`DataType`",
	      "SignalProcessing`Support`SigProc`",
	      "SignalProcessing`Support`Tree`",
	      "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

AddDataType::usage =
	"AddDataType[data_type], AddDataType[data_type, parent_data_type], \
	and AddDataType[data_type, parent_data_type, Fourier_data_type] \
	will update the tree data type structure to include the \
	new data_type and its corresponding Fourier domain \
	data type (which defaults to Complex). \
	If data_type is a symbol, a predicate will be created by \
	appending a Q to the data_type. \
	For example, positive integers are a subset of integers. \
	To add this data type, use \
	AddDataType[PositiveInteger, Integer, Complex]. \
	This will also defined a new query function called IsPositiveIntegerQ."

AllDataTypes::usage =
	"AllDataTypes is the universal set of all data types. \
	Symbols cannot be associated with this data type."

AssociateDataType::usage =
	"AssociateDataType[symbol, datatype] will indicate to all Signal \
	Processing Packages that the symbol is of type datatype. \
	This in no way restricts assignments to that symbol. \
	For example, AssociateDataType[n, Integer] means that \
	IsIntegerQ[n] will always be True, as well as \
	IsRationalQ[n], IsRealQ[n], and IsComplexQ[n]. \
	Also see DefConstant and UnAssociateDataType."

CombineDataType::usage =
	"CombineDataType[ datatype1, datatype2, ...] resolves multiple \
	data types."

ComplexAntiSymmetric::usage =
	"ComplexAntiSymmetric is an abstract data type. \
	It indicates that the real and imaginary components of a \
	complex-valued sequence (or \ function) are anti-symmetric. \
	See AssociateDataType."

ComplexSymmetric::usage =
	"ComplexSymmteric is an abstract data type. \
	It indicates that the real and imaginary components of a \
	complex-valued sequence (or function) are symmetric. \
	See AssociateDataType."

DataType::usage =
	"DataType[expr1, expr2, ...] returns the data type which results \
	in the combination of data types in expr1, expr2, .... \
	To associate a data type with a new function aa, use \
	aa/: DataType[ aa[args_] ] := newdatatype. \
	Use AssociateDataType to associate data types with symbols. \
	Use AddDataType to add more data types to the data type rules base."

DataTypeQ::usage =
	"DataTypeQ[datatype1, datatype2, ...] returns True if every \
	passed data type is valid and recognized by the DataType rules base. \
	Use AddDataType to add more data types to the rules base."

DataTypes::usage =
	"DataTypes[] returns a listing of all recognized abstracted \
	data types. \
	DataTypes[datatype] returns a list of all symbols \
	associated with datatype. \
	DataTypes[datatype1, datatype2, ...] returns a sorted table of \
	symbols associated with the data types datatype1, datatype2, .... \
	All symbols having an associated data type can therefore be found \
	by the code fragment DataTypes[ DataTypes[] ]."

DefConstant::usage =
	"DefConstant[symbol] simply sets the Constant attribute of symbol. \
	DefConstant[symbol, datatype] establishes the data type of the \
	symbol as datatype by calling AssociateDataType as well as \
	setting the symbol's Constant attribute."

FTDataType::usage =
	"FTDataType[datatype] returns the data type of datatype after \
	it has been sent through a Fourier transform."

GetPosition::usage =
	"GetPosition[list, element] returns the position which element \
	occurs in the list as an integer."

Imaginary::usage =
	"Imaginary is an abstract data type.  See AssociateDataType."

Irrational::usage =
	"Irrational is an abstract data type.  See AssociateDataType."

IrrationalQ::usage =
	"IrrationalQ[atom] returns True if the atom represents a number \
	which is irrational.  IrrationalQ[Pi] is True."

IsComplexQ::usage =
	"IsComplexQ[expression] returns True if the expression could \
	be considered complex-valued.  IsComplexQ[1.0] is True."

IsImaginaryQ::usage =
	"IsImaginaryQ[expression] returns True if the expression could \
	be considered imaginary."

IsIntegerQ::usage =
	"IsIntegerQ[expression] returns True if the expression could \
	be considered integer-valued."

IsIrrationalQ::usage =
	"IsIrrationalQ[expression] returns True if the expression could \
	be considered irrational."

IsRationalQ::usage =
	"IsRationalQ[expression] returns True if the expression could \
	be considered rational.  IsRationalQ[1] is True."

IsRealQ::usage =
	"IsRealQ[expression] returns True if the expression could \
	be considered real-valued."

SubDataTypeQ::usage =
	"SubDataTypeQ[expression, datatype] returns True if the data type \
	of expression could be coerced to datatype. \
	For example, SubDataTypeQ[5, Rational] is True, but \
	SubDataTypeQ[5.0, Rational] is False."

SymbolDataType::usage =
	"SymbolDataType[symbol] returns the data type associated with \
	symbol or AllDataTypes if none exists."

SymbolDataTypeQ::usage =
	"SymbolDataTypeQ[symbol, data-type] returns True if the symbol \
	is associated with the data type data-type. \
	Note that if a symbol is associated with Real, \
	then it is also associated with Integer and Rational as well."

UnAssociateDataType::usage =
	"UnAssociateDataType[symbol] removes any association of that \
	that symbol with any data type. \
	UnAssociateDataType[symbol, datatype] removes the association \
	of symbol with datatype if one exists. \
	The function returns symbol. \
	See also AssociateDataType."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin["`Private`"]


context = $Context


(*  M E S S A G E S  *)

AddDataType::defined = "`` is already defined as a data type."
AddDataType::parent = "Invalid parent data type: ``"

AssociateDataType::datatype = "Unrecognized data type ``.  Use AddDataType."
AssociateDataType::notvalid = "Can not associate symbols with AllDataTypes."
AssociateDataType::symbol = "The first argument is not a symbol."

SymbolDataType::noinfo = "The symbol `` has no data type associated with it."

(*  P A C K A G E     D A T A  *)

PossibleDataTypesTree =
	{ AllDataTypes, { Number, { Complex, Imaginary,
					     ComplexAntiSymmetric,
					     ComplexSymmetric },
	 			  { Real,    Irrational,
					     { Rational, Integer }}}}

FourierDataTypesTree =
	{ AllDataTypes, { Number, { Complex,		ComplexAntiSymmetric,
							Imaginary,
							Real },
				  { ComplexSymmetric,	ComplexSymmetric,
							{ ComplexSymmetric,
							  ComplexSymmetric }}}}

PossibleDataTypes = Flatten[ PossibleDataTypesTree ]
FourierDataTypes = Flatten[ FourierDataTypesTree ]
NumDataTypes = Length[ PossibleDataTypes ]

AllDataTypesList = {}
ComplexList = {}
ComplexSymmetricList = {}
ComplexAntiSymmetricList = {}
ImaginaryList = {}
IntegerList = {}
IrrationalList = {Catalan, Degree, E, EulerGamma, GoldenRatio, Pi}
NumberList = {}
RationalList = {}
RealList = {}


(*  S U P P O R T I N G     R O U T I N E S  *)

(*  GetPosition  *)
GetPosition[list_, element_] := Last[Last[Position[list, element]]]


(*  S P E C I F Y I N G     N E W     D A T A     T Y P E S  *)

(*  AddDataType  *)
newusagestring =
	"`1`::usage = \"An abstract data type.  See AssociateDataType.\""

newquerystring = "Is`1`Q[x_] := SubDataTypeQ[x, `1`]"

newliststring = "````List = {}"

AddDataType[ dtype_Symbol, parent_:AllDataTypes, ftype_:Complex ] :=
	Block [	{context, position},
		position = GetPosition[PossibleDataTypes, parent] + 1;
		FourierDataTypes = Insert[FourierDataTypes, ftype, position];

		context = "SignalProcessing`Support`DataType`Private`";
		PossibleDataTypesTree =
		    AddChildToTree[PossibleDataTypesTree, parent, dtype];
		PossibleDataTypes = Flatten[PossibleDataTypesTree];
		NumDataTypes = Length[PossibleDataTypes];

		GenerateCode[StringForm[newquerystring, dtype]];
		GenerateCode[StringForm[newliststring, context, dtype]];
		GenerateCode[StringForm[newusagestring, dtype]];

		PossibleDataTypesTree ] /;
	! DataTypeQ[dtype] && DataTypeQ[parent]

AddDataType[ dtype_?DataTypeQ, parent_:AllDataTypes, ftype_:Complex ] :=
	Message[AddDataType::defined, dtype]

AddDataType[ dtype_, parent_:AllDataTypes, ftype_:Complex ] :=
	Message[AddDataType::parent, parent] /;
	! DataTypeQ[parent]


(*  D A T A     T Y P E S     A T T A C H E D     T O     S Y M B O L S   *)

(*  AssociateDataType  *)
SetAttributes[AssociateDataType, Listable]

	(* code to update the list corresponding to the data type *)
associatestring =
	"Block [ {}, \
		 `1` = Sort[PrependTo[`1`, `2`]]; \
		 `2` ]"

AssociateDataType[symbol_, AllDataTypes] :=
	Message[AssociateDataType::notvalid]

AssociateDataType[symbol_, ComplexSymmetric] :=
	Message[AssociateDataType::notvalid]

AssociateDataType[symbol_, ComplexAntiSymmetric] :=
	Message[AssociateDataType::notvalid]

AssociateDataType[symbol_, datatype_] :=
	Message[AssociateDataType::datatype, datatype] /;
	! DataTypeQ[datatype]

AssociateDataType[symbol_, datatype_] :=
	Message[AssociateDataType::symbol, symbol] /;
	! SameQ[Head[symbol], Symbol]

AssociateDataType[symbol_Symbol, datatype_?DataTypeQ] :=
	Block [ {datatypelist},
		datatypelist = StringJoin[ToString[context],
					  ToString[datatype],
					  "List"];
		list = ToExpression[ datatypelist ];
		If [ ! MemberQ[list, symbol],
		     GenerateCode[ StringForm [	associatestring,
						datatypelist,
						symbol ] ] ];
		symbol ]

(*  DefConstant  *)
SetAttributes[DefConstant, {HoldAll}]

DefConstant[symbol_] := DefConstant[symbol, Null] /; ! ValueQ[symbol]
DefConstant[symbol_?ValueQ] := DefConstant[symbol, Head[symbol]]

DefConstant[symbol_, datatype_] :=
	Block [	{valueflag, value},
		valueflag = ValueQ[symbol];
		value = symbol;
		Clear[symbol];
		SetAttributes[symbol, {Constant}];
		AssociateDataType[symbol, datatype];
		If [ valueflag, symbol = value ];
		symbol ] /;
	Implies[ValueQ[symbol], SameQ[datatype, Head[symbol]]]

(*  UnAssociateDataType  *)
SetAttributes[UnAssociateDataType, { Listable }]

UnAssociateDataType[symbol_Symbol, datatype_Symbol] :=
	Block [	{exprstr, list, liststr, headstr, pos},
		headstr = "SignalProcessing`Support`DataType`Private`";
		liststr = ToString[StringForm["````List", headstr, datatype]];
		list = ToExpression[liststr];
		If [ MemberQ[list, symbol],
		     pos = GetPosition[list, symbol];
		     exprstr = StringForm[ "`` = ``", liststr,
					   Drop[list, {pos, pos}] ];
		     ToExpression[ ToString[ exprstr ] ] ];
		symbol ]

UnAssociateDataType[symbol_Symbol] :=
	Block [	{},
		Map[ removesymbol[symbol, #1]&, PossibleDataTypes ];
		symbol ]


(*  D A T A     T Y P E S     O F     E X P R E S S I O N S  *)

(*  CombineTwoDataTypes  *)
CombineTwoDataTypes[ComplexSymmetric, ComplexAntiSymmetric] := Complex
CombineTwoDataTypes[ComplexAntiSymmetric, ComplexSymmetric] := Complex
CombineTwoDataTypes[Real, Imaginary] := Complex
CombineTwoDataTypes[Imaginary, Real] := Complex
CombineTwoDataTypes[d1_, d2_] :=
	If [ DataTypeQ[d1] && DataTypeQ[d2],
	     TableLookup[ Min[GetPosition[PossibleDataTypes, d1],
			      GetPosition[PossibleDataTypes, d2]],
		          PossibleDataTypes, NumDataTypes, Complex ],
	     Complex ]

(*  CombineDataType  *)
CombineDataType[x_] := DataType[x]
CombineDataType[x_, y_] := CombineTwoDataTypes[ DataType[x], DataType[y] ]
CombineDataType[x_, y_, rest__] := CombineDataType[CombineDataType[x, y], rest]

(*  DataTypeQ  *)
DataTypeQ[d_] := AtomQ[d] && MemberQ[PossibleDataTypes, d]
DataTypeQ[d1_, d__] := DataTypeQ[d1] && DataTypeQ[d]

(*  FTDataType --  Fourier transform data type  *)
FTDataType[d_] :=
	TableLookup[ GetPosition[PossibleDataTypes, d],
		     FourierDataTypes, NumDataTypes, AllDataTypes ]

(*  IrrationalQ  *)
IrrationalQ[a_] := AtomQ[a] && MemberQ[IrrationalList, a]

(*  IsComplex --  Complex data type checking  *)
IsComplexQ[a_] := ComplexQ[a] || SubDataTypeQ[a, Complex]

(*  IsImaginaryQ --  Imaginary data type checking  *)
IsImaginaryQ[a_] := ImaginaryQ[a] || SubDataTypeQ[a, Imaginary]

(*  IsIntegerQ --  Integer data type checking  *)
IsIntegerQ[a_] := IntegerQ[a] || SubDataTypeQ[a, Integer]

(*  IsIrrationalQ --  Irrational data type checking  *)
IsIrrationalQ[a_] := IrrationalQ[a] || SubDataTypeQ[a, Irrational]

(*  IsRationalQ --  Rational data type checking  *)
IsRationalQ[a_] := RationalQ[a] || SubDataTypeQ[a, Rational]

(*  IsRealQ --  Real data type checking  *)
IsRealQ[a_] := ( AtomQ[a] && RealValuedQ[N[a]] ) || SubDataTypeQ[a, Real]

(*  SubDataTypeQ  *)
SubDataTypeQ[expr_, datatype_] := 
	Block [	{exprdatatype},
		exprdatatype = DataType[expr];
		If [ SameQ[exprdatatype, datatype],
		     True,
		     MemberQ[Flatten[ToList[SubTree[PossibleDataTypesTree,
						    datatype]]],
		    	     exprdatatype] ] ]

(*  SymbolDataType  *)
SymbolDataType[symbol_Symbol] :=
	Block [	{curlist, datatype, i, result},
		For [ i = 2, i <= NumDataTypes, i++,
		      datatype = PossibleDataTypes[[i]];
		      curlist = DataTypes[datatype];
		      If [ MemberQ[curlist, symbol],
			   result = datatype; Break[] ] ];
		If [ ! ValueQ[result],
		     result = AllDataTypes;
		     Message[ SymbolDataType::noinfo, symbol ] ];
		result ]

(*  SymbolDataTypeQ  *)
SymbolDataTypeQ[symbol_Symbol, datatype_?DataTypeQ] :=
	Block [	{completelist, possibletypes},
		possibletypes = Flatten[ToList[SubTree[ PossibleDataTypesTree,
							datatype ] ] ];
		completelist = Sort[Flatten[Map[DataTypes, possibletypes]]];
		MemberQ[completelist, datatype] ]


(*  D A T A     T Y P E     R U L E S  *)

DataType[x_, rest__] := CombineDataType[x, rest]

DataType[a_?PatternQ] := AllDataTypes		(* Pattern		*)

DataType[x_?DataTypeQ] := x			(* Identity		*)

DataType[x_Integer] := Integer			(* Numbers		*)
DataType[x_Rational] := Rational
DataType[x_Real] := Real
DataType[x_?ImaginaryQ] := Imaginary
DataType[x_Complex] := Complex

DataType[x_Symbol] := SymbolDataType[x]		(* Symbols 		*)

DataType[a_ ^ n_] := CombineDataType[a, n]	(* Standard functions	*)
DataType[x_?IsIntegerQ / y_?IsIntegerQ] := Rational
DataType[x_ / y_] := CombineDataType[x, y]
DataType[Abs[x_]] := Real /; SameQ[DataType[x], Complex]
DataType[Ceiling[x_?IsRealQ]] := Integer
DataType[Floor[x_?IsRealQ]] := Integer
DataType[Im[x_]] := Real
DataType[Re[x_]] := Real
DataType[Round[x_?IsRealQ]] := Integer

DataType[h_[a1_, a___]] := CombineDataType[a1, a] /;
	SameQ[Context[h], "System`"]

  (*  Signal primitives  *)

Unprotect[CStep, Delta, FIR, IIR, Impulse, LineImpulse]
Unprotect[Pulse, Sinc, Step, Unit]

CStep/: DataType[CStep[n_]] := Integer
Delta/: DataType[Delta[x_]] := Integer /; ! SameQ[N[x], 0.]
FIR/:	DataType[FIR[n_, h__]] := DataType[h]
IIR/:	DataType[IIR[n_, a__]] := DataType[a]
Impulse/: DataType[Impulse[x_]] := Integer
LineImpulse/: DataType[LineImpulse[a__]] := Integer
Pulse/:	DataType[Pulse[a__]] := Integer
Sinc/:  DataType[Sinc[x_]] := DataType[x]
Step/:	DataType[Step[n_]] := Integer
Unit/:	DataType[Unit[n_][t_]] := Integer /; N[n < 0]

Protect[CStep, Delta, FIR, IIR, Impulse, LineImpulse]
Protect[Pulse, Sinc, Step, Unit]

  (* System primitives which are not Mathematica primitives  *)

Unprotect[Aliasby, CConvolve, Convolve, DFT, FT, Interleave]
Unprotect[InvDFT, InvFT, InvL, InvZ]
Unprotect[L, PolyphaseDownsample, PolyphaseUpsample, Reciprocal]
Unprotect[Rev, ScaleAxis, Shift, Upsample, Z]

Aliasby/:    DataType[Aliasby[m_, w_][x_]] := DataType[x]
CConvolve/:  DataType[CConvolve[t_][x_, y__]] := CombineDataType[x, y]
Convolve/:   DataType[Convolve[n_][x_, y__]] := CombineDataType[x, y]
DFT/:	     DataType[DFT[l_, n_, k_][x_]] := FTDataType[ DataType[x] ]
FT/:	     DataType[FT[t_, w_][x_]] := FTDataType[ DataType[x] ]
Interleave/: DataType[Interleave[n_][x_, y__]] := CombineDataType[x, y]
InvDFT/:     DataType[InvDFT[l_, k_, n_][x_]] := FTDataType[ DataType[x] ]
InvFT/:	     DataType[InvFT[w_, t_]] := FTDataType[ DataType[x] ]
InvL/:	     DataType[InvL[s_, t_][x_]] := Complex
InvZ/:	     DataType[InvZ[z_, n_][x_]] := Complex
L/:	     DataType[L[t_, s_][f_]] := Complex
PolyphaseDownsample/: DataType[PolyphaseDownsample[m_, n_][x_, h_]] :=
		CombineDataType[x, h]
PolyphaseUpsample/: DataType[PolyphaseUpsample[l_, n_][x_, h_]] :=
		CombineDataType[x, h]
Reciprocal/: DataType[Reciprocal[x_]] := DataType[x]
Rev/:	     DataType[Rev[n_][x_]] := DataType[x]
ScaleAxis/:  DataType[ScaleAxis[l_, w_][x_]] := DataType[x]
Shift/:      DataType[Shift[l_, n_][x_]] := DataType[x]
Upsample/:   DataType[Upsample[l_, n_][x_]] := DataType[x]
Z/:	     DataType[Z[n_, z_][f_]] := Complex

Protect[Aliasby, CConvolve, Convolve, DFT, FT, Interleave]
Protect[InvDFT, InvFT, InvL, InvZ]
Protect[L, PolyphaseDownsample, PolyphaseUpsample, Reciprocal]
Protect[Rev, ScaleAxis, Shift, Upsample, Z]


(*  D I S P L A Y I N G     A S S O C I A T E D     D A T A     T Y P E S  *)

(*  DataTypes  *)
process[datatype_] :=
	Map[ Function[symbol, {symbol, datatype}], DataTypes[datatype] ]

tableoutput[list_] := TableForm[Sort[Apply[Join, Map[process, list]]]]

DataTypes[] := PossibleDataTypes

DataTypes[datatype_?DataTypeQ] :=
	GenerateSymbol[	datatype, "List",
			"SignalProcessing`Support`DataType`Private`" ]

DataTypes[typelist_List] := tableoutput[ typelist ]

DataTypes[datatype_, rest__] := tableoutput[ {datatype, rest} ]


(*  E N D     O F     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ]


(*  H E L P     I N F O R M A T I O N  *)

Block [ {newfuns},
	newfuns =
      {	AddDataType,		AssociateDataType,	CombineDataType,
	CombineTwoDataTypes,	DataType,		DataTypeQ,
	DataTypes,		DataTypeQ,		DefConstant,
	FTDataType,		GetPosition,		IrrationalQ,
	IsComplexQ,		IsImaginaryQ,		IsIntegerQ,
	IsIrrationalQ,		IsRationalQ,		IsRealQ,
	SubDataTypeQ,		SymbolDataType,		SymbolDataTypeQ };

	Combine[ SPfunctions, newfuns ];
	Apply[ Protect, newfuns ] ]


(*  E N D I N G     M E S S A G E  *)

Print["Rules analyzing the data type of expressions have been loaded."]
Null
