/**********************************************************************/
/* geo.c :                                                            */
/*                                                                    */
/* Geometric operations                                               */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          *
/**********************************************************************/
#include <stdio.h>
#ifdef IRIS
#include <gl/gl.h>
#endif /* IRIS */
#include <math.h>
#include "misc.h"
#include "geo.h"

#ifdef IRIS
/*
 * ResetWindMat() - Reset matrices
 */
void ResetWindMat()
{
  ResetMat(Mat);
  loadmatrix(IdentityMat);
}
#endif IRIS

/**********************************************************************/
/* find dot product of vectors v0 and v1 */
/**********************************************************************/
double dot(v0,v1)
     Vector *v0, *v1;
{  return (v0->x * v1->x) + (v0->y * v1->y) + (v0->z * v1->z); }

/**********************************************************************/
/* Find cross product of vectors v0 and v1 */
/**********************************************************************/
Vector cross(v0, v1)
     Vector *v0, *v1;
{
  Vector v2;
  v2.x = (v0->y * v1->z) - (v0->z * v1->y);
  v2.y = (v0->z * v1->x) - (v0->x * v1->z);
  v2.z = (v0->x * v1->y) - (v0->y * v1->x);
  return v2;
}

/**********************************************************************/
/* Normalized vector v0 and return length, return negative length if */
/* close to zero, or less than zero */
/**********************************************************************/
double norm(v0)
     Vector *v0;
{
  double len;

  if ((len = dot(v0,v0)) <= 0.0) return FALSE;
  len = sqrt((double)len);
  if (len < MIN_VECTOR_LENGTH) return FALSE;
  v0->x = v0->x / len;
  v0->y = v0->y / len;
  v0->z = v0->z / len;
  return TRUE;
}

/**********************************************************************/
/* Return the distance between two points */
/**********************************************************************/
double vdist(a, b)
     Vector *a, *b;
{
  double dx, dy, dz;

  dx = a->x - b->x; dy = a->y - b->y;  dz = a->z - b->z;
  return(sqrt((dx*dx)+(dy*dy)+(dz*dz)));
}

/**********************************************************************/
/* Create vector from 2 points, and return length */
/**********************************************************************/
double geo_line(p0, p1, v)
     point3 *p0, *p1;
     Line *v;
{
  v->start = *p0;
  v->dir.x = p1->x - p0->x;
  v->dir.y = p1->y - p0->y;
  v->dir.z = p1->z - p0->z;
  return norm(&v->dir);
}

/**********************************************************************/
/* Add 2 vectors */
/**********************************************************************/
Vector *vadd(v0, v1)
     Vector *v0, *v1;
{
  static Vector *add;
  
  add->x = v0->x + v1->x;
  add->y = v0->y + v1->y;
  add->z = v0->z + v1->z;
  return add;
}

/**********************************************************************/
/* Subtract vector v1 from v0 */
/**********************************************************************/
Vector *vsub(v0, v1)
     Vector *v0, *v1;
{
  static Vector sub;
  
  sub.x = v0->x - v1->x;
  sub.y = v0->y - v1->y;
  sub.z = v0->z - v1->z;
  return &sub;
}

/**********************************************************************/
/* Vector linear combination: a*v0 + b*v1 */
/**********************************************************************/
Vector *vcomb(v0,v1,a,b)
     Vector *v0, *v1;
     double a,b;
{
  static Vector comb;
  
  comb.x = a * v0->x + b * v1->x;
  comb.y = a * v0->y + b * v1->y;
  comb.z = a * v0->z + b * v1->z;
  return &comb;
}

/**********************************************************************/
/* This function is similar to pow() but works right only for */
/* positive integer b (but a lot faster) */
/**********************************************************************/
double Power(a,b)
     double a,b;
{
  register int power;
  double result;
  
  result= 1.0;
  power= (int) b;
  while(power > 0) {
    if(power & 1)
      result *= a;
    a *= a;
    power >>= 1;
  }
  return(result);
}

/**********************************************************************/
/* Calculate normal for a polygon from its vertices */
/**********************************************************************/
Vector polynorm(vert,size)
     register Vector vert[];
     register int size;
{
  Vector n;
  register int i,j;
  
  n.x = n.y = n.z= 0.0;
  for(i=0;i<size;i++) {
    j= (i+1)%size;
    n.x += (vert[i].y-vert[j].y) * (vert[i].z+vert[j].z);
    n.y += (vert[i].z-vert[j].z) * (vert[i].x+vert[j].x);
    n.z += (vert[i].x-vert[j].x) * (vert[i].y+vert[j].y);
  }
  if(norm(&(n)) < 0.) {
    n.x = n.y = 0.;
    n.z= 1.;
  }
  return(n);
}

/**********************************************************************/
/* Matrix * vector */
/**********************************************************************/
Vector vtransform(v,M)
     Vector v;
     register Matrix M;
{
  Vector vPrime;
  
  vPrime.x= v.x * M[0][0] + v.y * M[1][0] + v.z * M[2][0] + M[3][0];
  vPrime.y= v.x * M[0][1] + v.y * M[1][1] + v.z * M[2][1] + M[3][1];
  vPrime.z= v.x * M[0][2] + v.y * M[1][2] + v.z * M[2][2] + M[3][2];
  return(vPrime);
}

/**********************************************************************/
/* Invert matrix a -> b (using determinant method) */
/**********************************************************************/
int invert_matrix(a,b)
     Matrix a,b;
{
  double det;

  b[0][0]= a[1][1]*a[2][2] - a[1][2]*a[2][1];
  b[1][0]= a[1][0]*a[2][2] - a[1][2]*a[2][0];
  b[2][0]= a[1][0]*a[2][1] - a[1][1]*a[2][0];
  
  b[0][1]= a[0][1]*a[2][2] - a[0][2]*a[2][1];
  b[1][1]= a[0][0]*a[2][2] - a[0][2]*a[2][0];
  b[2][1]= a[0][0]*a[2][1] - a[0][1]*a[2][0];
  
  b[0][2]= a[0][1]*a[1][2] - a[0][2]*a[1][1];
  b[1][2]= a[0][0]*a[1][2] - a[0][2]*a[1][0];
  b[2][2]= a[0][0]*a[1][1] - a[0][1]*a[1][0];
  
  det= a[0][0]*b[0][0] - a[0][1]*b[1][0] + a[0][2]*b[2][0];
  
  /* matrix is singular */
  if(det < DAMN_SMALL && det > -DAMN_SMALL) return(ERROR);

  b[0][0] /= det;
  b[0][1] /= -det;
  b[0][2] /= det;
  b[1][0] /= -det;
  b[1][1] /= det;
  b[1][2] /= -det;
  b[2][0] /= det;
  b[2][1] /= -det;
  b[2][2] /= det;
  
  b[3][0]= -(b[0][0]*a[3][0] + b[1][0]*a[3][1] + b[2][0]*a[3][2]);
  b[3][1]= -(b[0][1]*a[3][0] + b[1][1]*a[3][1] + b[2][1]*a[3][2]);
  b[3][2]= -(b[0][2]*a[3][0] + b[1][2]*a[3][1] + b[2][2]*a[3][2]);
  b[3][3]= 1.;
  b[0][3]= 0.;
  b[1][3]= 0.;
  b[2][3]= 0.;
  return(OK);
}

/*====================================================================*/
/* Output routines */
/*====================================================================*/

/**********************************************************************/
/* Print a vector */
/**********************************************************************/
void print_vector(label,v)
     char *label;
     Vector *v;
{
  printf("\tVector: %s = %e,%e,%e\n", label, v->x, v->y, v->z);
}

/**********************************************************************/
/* Print a matrix */
/**********************************************************************/
/* originally printmatrix from trans.c, label parm added !! */
void print_matrix(label, a)
     char *label;
     register Matrix a;
{
  int i,j;

  printf("\tMatrix: %s\n", label);
  for(i=0;i<4;i++) {
    printf("\t");
    for(j=0;j<4;j++)
      printf("%10.6f  ",a[i][j]);
    printf("\n");
  }
  printf("\n");
}


