/**********************************************************************/
/* Geometric primitives (general)                                     */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#ifndef GEO_H
#define GEO_H

#ifndef IRIS4D
typedef double Matrix[4][4];              /* 4x4 transformation matrix */
static Matrix Identity = {
  1.0, 0.0, 0.0, 0.0,
  0.0, 1.0, 0.0, 0.0,
  0.0, 0.0, 1.0, 0.0,
  0.0, 0.0, 0.0, 1.
  };  
#endif /* IRIS4D */

#define Point_On_Ray(v1,s,v2,r)	(r).x = (v1).x + (s)*(v2).x, \
				(r).y = (v1).y + (s)*(v2).y, \
				(r).z = (v1).z + (s)*(v2).z
/* Two components are "equal"         */
#define MIN_EQUAL (0.000001)
#define equal(a, b) (fabs((a) - (b)) < MIN_EQUAL)

/**********************************************************************/
/* 2D / 3D  geometry */
/**********************************************************************/
typedef struct { double x,y; } Vector2;   /* 2D Vector */
typedef Vector2 point2;                  /* 2D Point  */
typedef struct { int x,y;}  Intpoint2; /* 2d integer point */
typedef struct { double element[3][3]; } Matrix3; 
typedef struct Box2dStruct { point2 min, max; } Box2; /* 2d box */

typedef struct { int x,y,z; } Intpoint3; /* 3d integer point */
typedef struct { double element[4][4]; } Matrix4; 
typedef struct { double x,y,z; } point3; /* 3D Point  */
typedef struct { point3 min, max; } Box;

/**********************************************************************/
typedef struct { double x,y,z; } Vector; /* 3D Vector */
typedef struct {double x,y,z,h;} point4; /* 4D Point  */

typedef struct {  
  Vector n;                      /* normal */
  Vector p;                      /* point on plane */
  double d;                      /* distance from origin to plane */
} Plane;                         /* A plane */

typedef struct {               
  point3 start;                  /* Start point */
  Vector dir;                    /* vector direction */
} line;                          /* A line */

typedef struct {
  Vector min, max;             /* Upper and lower bounds of box */
} BoundingBoxType;               /* bounding box (axis aligned) */

typedef struct {
  double r;                    /* Radius of sphere */
  Vector pos;                  /* Position of sphere */
} BoundingSphereType;          /* bounding sphere (axis aligned) */

/**********************************************************************/
/* 2-D library */
/**********************************************************************/
double vdot2(); 
double vlen2();
Vector2 *vnegate2();
Vector2 *vnorm2();
Vector2 *vscale2();
Vector2 *vadd2();
Vector2 *vsub2();
Vector2 *vinterp2();
Vector2 *vcomb2();
Vector2 *vmult2();
double vdist2();
Vector2 *vperp2();
Vector2 *vnew2();
Vector2 *vdup2();
point2 *vtransform2();
Matrix3 *multmatrix2();

/**********************************************************************/
/* 3-D library */
/**********************************************************************/
double dot();                    /* dot product */
Vector cross();                  /* cross product */
double norm();                   /* vector normalization */
double geo_line();               /* vector from 2 pts */
Vector *vadd();                  /* vector add */
Vector *vsub();                  /* vector subtract */
Vector *vnegate();
Vector *vmult();
Vector *vcomb();                 /* linear combination of 2 vectors */
Vector *vscalar();
double vdist();
double vlen();
Vector *vscale();
Vector *vmiddle();
Vector *vinterp();
Vector *vnew();
Vector *vdup();
Vector polynorm();               /* polygon normal from vertices */
double Power();                  /* fast power function for positive ints */

Vector *reflection();            /* ideal reflection vector */
Vector *refraction();            /* ideal simple refraction vector */
Vector *geo_h();                 /* 'half-way' reflection vector */
Vector *geo_ht();                /* 'half-way' refraction vector */

/**********************************************************************/
/* Matrix library */
/**********************************************************************/
void copymatrix();               /* copy matrices */
int same_matrix();               /* check if 2 matrices are the same */
void mult_matrix();               /* multiply 2 matrices */
void smult_matrix();
Vector vtransform();             /* Matrix * vector */
point3 *ptransform();
Vector vrotate();                /* Rotate a vector */
void transpose_matrix();
int invert_matrix();             /* Invert matrix */
void cr_rotatez();
void rotatex_matrix();
void rotatey_matrix();
void rotatez_matrix();
void scale_matrix();
void translate_matrix();
void subtract_matrix();
void ZPerspective_matrix();
void reorthogonalize_matrix();
void adjoint();
double det4x4();
double det3x3();
double det2x2();

void print_vector();             /* print 3D vector */
void print_vector2();
void print_matrix();             /* print a 4x4 matrix */

#endif /* GEO_H */

