/**********************************************************************/
/* draw.c :                                                           */
/*                                                                    */
/* Scene drawing, and hemicube computations on SGI framebuffer        */
/*                                                                    */
/* Copyright (C) 1992, Bernard Kwok                                   */
/* All rights reserved.                                               */
/* Revision 1.0                                                       */
/* May, 1992                                                          */
/**********************************************************************/
#include <stdio.h>
#ifdef IRIS4D
#include <gl/gl.h>
#endif
#include "geo.h"
#include "io.h"
#include "struct.h"
#include "rad.h"

extern RadParams ReadLog;
extern Hemicube hemicube;
extern OptionType Option;
int showmesh = 0;
long HCWid; /* window to use of hc */
long DisplWid; /* window to use for display */
long CurWid;
float zmax;
#ifdef IRIS4D
Matrix Identity = {
  1,0,0,0,
  0,1,0,0,
  0,0,1,0,
  0,0,0,1
};
#endif

void Reset_Buffers();
void Begin_DrawHC();
void End_DrawHC();
void Draw_Polygon();
void Draw_PolygonRGB();
void Begin_DrawDispl();
void End_DrawDispl();
void Create_Display_Buffer();
void Create_Item_Buffer();
void CleanUpBuffers();

#ifndef IRIS4D
void Reset_Buffers() {}
void Begin_DrawHC() {}
void End_DrawHC() {}
void Draw_Polygon() {}
void Draw_PolygonRGB() {}
void Begin_DrawDispl() {}
void End_DrawDispl() {}
void Create_Display_Buffer() {}
void Create_Item_Buffer() {}
void CleanUpBuffers() {}
#else

/**********************************************************************/
/* Clear z and frame buffers                                          */
/**********************************************************************/
void Reset_Buffers(fvalue, zvalue)
     unsigned long fvalue;
     float zvalue;
{  czclear(fvalue, zvalue); }

/**********************************************************************/
/* Set up item buffer, and view */
/**********************************************************************/
void Begin_DrawHC(view, background)
     Camera view;
     unsigned long background;
{
  
  winset(HCWid);
  if (Option.debug) {
    CurWid = winget();
    printf("+ Current window: %d\n", CurWid);
    printf("+ Eye: view.fovy = %d, near = %g, far = %g\n",
	   view.fovy, view.near, view.far);
    printf("  lookfrom: %g,%g,%g; lookat: %g,%g,%g\n",
	   view.lookfrom.x, view.lookfrom.y, view.lookfrom.z, 
	   view.lookat.x, view.lookat.y, view.lookat.z);
    printf("  bank: %g\n", view.bank);
  }

  pushmatrix();
  perspective((int)view.fovy*10,1,view.near, view.far);
  lookat(view.lookfrom.x, view.lookfrom.y, view.lookfrom.z,
	 view.lookat.x, view.lookat.y, view.lookat.z, 0);
  Reset_Buffers(background,zmax);
}

/**********************************************************************/
/* Read the frame buffer to hemicube buffer */
/**********************************************************************/
void End_DrawHC()
{
  long pixels_read;
  unsigned long temp[13000];
  unsigned long *pb;
  int i;
  int hsize = ReadLog.hemicubeRes;

  winset(HCWid);
  popmatrix();
  for (i=0;i<13000;i++) temp[i] = kBackgroundItem;

  readsource(SRC_FRONT);
  pixels_read = lrectread(1,1,hemicube.view.xRes,hemicube.view.yRes,temp);
  pb = hemicube.view.buffer;
  for(i=0;i<(hsize*hsize);i++, pb++) 
    *pb = temp[i];  
}

/**********************************************************************/
/* Draw patch */
/**********************************************************************/
void Draw_Polygon(nPts,pts, normal, p_colour)
     int nPts;
     Vector pts[MAX_PATCH_VTX];
     Vector *normal;
     unsigned long p_colour;
{
  int i;
  float fpoint[3];

  shademodel(FLAT);
  cpack(p_colour);

  bgnpolygon();
  for(i=0;i<nPts;i++) {
    fpoint[0] = pts[i].x; fpoint[1] = pts[i].y; fpoint[2] = pts[i].z;
    v3f(fpoint); 
  }
  endpolygon();
}

/**********************************************************************/
void Draw_PolygonRGB(nPts,pts, normal, p_colour)
     int nPts;
     Vector pts[MAX_PATCH_VTX];
     Vector *normal;
     Colour p_colour[4];
{
  int i;
  float fpoint[3];
  float fnorm[3];
  float fcolour[3];

  fnorm[0] = normal->x; fnorm[1] = normal->y; fnorm[2] = normal->z;

  shademodel(GOURAUD);
  
  if (showmesh)
    bgnline();
  else
    bgnpolygon(); 

  for(i=0;i<nPts;i++) {
    fpoint[0] = pts[i].x; 
    fpoint[1] = pts[i].y; 
    fpoint[2] = pts[i].z;
    fcolour[0] = p_colour[i].r; 
    fcolour[1] = p_colour[i].g; 
    fcolour[2] = p_colour[i].b;

    c3f(fcolour);
    /* n3f(fnorm); */
    v3f(fpoint); 
  }

  if (showmesh)
    endline();
  else
    endpolygon();
}

/**********************************************************************/
void Begin_DrawDispl(view, background)
     Camera view;
     unsigned char background;
{
  winset(DisplWid);
  pushmatrix();
  perspective(view.fovy*10,1,view.near, view.far);
  lookat(view.lookfrom.x, view.lookfrom.y, view.lookfrom.z,
	 view.lookat.x, view.lookat.y, view.lookat.z, view.bank);
  Reset_Buffers(0x000000,zmax);
}

/**********************************************************************/
void End_DrawDispl()
{
  winset(DisplWid); 
  popmatrix();
  swapbuffers();
}

/**********************************************************************/
void Create_Display_Buffer(sizex, sizey)
     int sizex, sizey;
{
  prefsize(sizex,sizey);
  DisplWid = winopen("PR Radiosity");
  
  zbuffer(TRUE);
  zmax = getgdesc(GD_ZMAX);
  doublebuffer();
  RGBmode();
  mmode(MVIEWING);
  loadmatrix(Identity);
  Reset_Buffers(0xff000f, zmax);
  gconfig();
}

/**********************************************************************/
/* Set up zbuffer, and item_buffer (part of frame-buffer)             */
/**********************************************************************/
void Create_Item_Buffer(buff_sizex, buff_sizey)
     int buff_sizex, buff_sizey;
{
  /* Setup frame buffer area of size buff_size as item buffer */
  prefsize(buff_sizex,buff_sizey);
  keepaspect(1,1);
  HCWid = winopen("FF");

  zmax = getgdesc(GD_ZMAX);
  RGBmode();
  mmode(MVIEWING);
    zbuffer(TRUE);
  Reset_Buffers(kBackgroundItem,zmax);
  readsource(SRC_AUTO);
  loadmatrix(Identity);
  gconfig();
}


/**********************************************************************/
void CleanUpBuffers()
{
  sleep(10);
  if (Option.ff_raytrace == 0)
    winclose(HCWid);
  winclose(DisplWid);
}

#endif /* IRIS4D */
