/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: dmallog.c,v 1.1 1997/08/18 03:31:36 timj Exp $")


#define		__dmallog_c__

#include	"dmallog.h"
#include	<stdlib.h>
#include	<stdarg.h>
#include	<string.h>
#include	<stdio.h>
#include	<time.h>
#include	<errno.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<sys/mman.h>



/* --- defines --- */
#undef	NULL
#define	NULL			((void*)0)
#define	dmg_BUFFER_SIZE		(512)
#define	dmg_initialized()	(dmg_block_id!=0)
#define	dmg_FMTstr_block	"block(id%05lx)[start:0x%09lx, length:%8lu]"
#define	dmg_STR_malloc		"malloc : "
#define	dmg_STR_realloc		"realloc: "
#define	dmg_STR_free		"free   : "
#define	dmg_STR_calloc		"calloc : "
#define	dmg_STR_spaced		"         "
#define	dmg_STR_info		"INFO: "
#define	D_MALLOG_LOG		"./d_mallog.log"


/* --- ident-string --- */
#ifdef	D_MALLOG
RCS_KEY(dmallog, "$dmallog: D_MALLOG defined upon compilation (\"" D_MALLOG_LOG "\") $")
#endif	/*D_MALLOG*/



/* --- typedefs --- */
typedef	unsigned long		dmg_ulong;
typedef	struct	dmg_List	dmg_List;



/* --- structures --- */
struct	dmg_List {
	dmg_ulong	block_id;
	void		*block;
	dmg_ulong	length;
	dmg_List	*next;
	dmg_List	*prev;
};



/* --- global variables --- */
static	char		dmg_logfile[dmg_BUFFER_SIZE]=D_MALLOG_LOG;
static	char		*dmg_PREFIX_malloc=dmg_STR_malloc;
static	FILE		*dmg_LOG=NULL;
static	dmg_List	*dmg_LISTS_free=NULL;
static	dmg_List	*dmg_BLOCKS_used=NULL;
static	dmg_ulong	dmg_block_id=0;



/* --- prototypes --- */
static	void	dmg_fat		(const char	*format,
				 ...)
				 #ifdef		__GNUC__
				 	__attribute__
				 	((format (printf, 1, 2)))
				 	__attribute__
				 	((noreturn))
				 #endif         /*__GNUC__*/
				 ;
static	void	dmg_log		(const char	*format,
				 ...)
				 #ifdef		__GNUC__
				 	__attribute__
				 	((format (printf, 1, 2)))
				 #endif         /*__GNUC__*/
				 ;
static	dmg_List*	dmg_List_prepend(dmg_List	*list,
					 dmg_List	*prepender);
static	dmg_List*	dmg_List_new	(void		*mem_block,
					 unsigned int	length);
static	dmg_List*	dmg_List_remove	(dmg_List	*list,
					 dmg_List	*link);
static	dmg_List*	dmg_List_lookup	(dmg_List	*list,
					 void		*block);



/* --- functions --- */
dmg_List*
dmg_List_prepend(dmg_List	*list,
		 dmg_List	*prepender)
{
	prepender->prev=NULL;
	prepender->next=list;
	if (list)
		list->prev=prepender;
	return prepender;
}


dmg_List*
dmg_List_new	(void		*mem_block,
		 unsigned int	length)
{
	register dmg_List	*list;
	
	list=dmg_LISTS_free;
	
	if (!list)
		dmg_fat("ran out of linked lists");
	
	dmg_LISTS_free=list->next;
	
	list->next=NULL;
	list->prev=NULL;
	list->block_id=0;
	list->block=mem_block;
	list->length=length;
	
	return list;
}


dmg_List*
dmg_List_remove	(dmg_List		*list,
		 dmg_List		*link)
{
	if (link) {
		if (link->next)
			link->next->prev=link->prev;
		if (link->prev)
			link->prev->next=link->next;
		
		if (link==list)
			list=list->next;
		
		link->prev=NULL;
		link->next=dmg_LISTS_free;
		dmg_LISTS_free=link;
	}
	
	return list;
}


dmg_List*
dmg_List_lookup	(dmg_List	*list,
		 void	*block)
{
	if (list) {
		while (list->prev)
			list=list->prev;
		
		while (list)
			if (list->block==block)
				break;
			else
				list=list->next;
	}
	
	return list;
}


void
dmg_fat		(const char	*format,
		 ...)
{
	va_list	ap;
	
	fprintf(stderr, "d_mallog: fatal: ");
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	fprintf(stderr, "\n");
	fflush(stderr);
	
	if (dmg_LOG) {
		fprintf(dmg_LOG, "d_mallog: fatal: ");
		va_start(ap, format);
		vfprintf(dmg_LOG, format, ap);
		va_end(ap);
		fprintf(dmg_LOG, "\n");
		fflush(dmg_LOG);
	}
	
	_exit(-1);
}


void
dmg_log		(const char	*format,
		 ...)
{
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	if (dmg_LOG) {
		static	 va_list	ap;
		
		va_start(ap, format);
		vfprintf(dmg_LOG, format, ap);
		va_end(ap);
		fprintf(dmg_LOG, "\n");
		fflush(dmg_LOG);
	}
}


void
dmg_log_pos	(const char		*file,
		 const int		line,
		 const char		*function)
{
	static	 long long	o=0;
	static	 int		l=-1;
	register long long	n;
	static	 char		buffer[32];
	
	n=(long)function;
	n<<=sizeof(line)*8;
	n+=(long)file;
	
	if (n!=o) {
		dmg_log("IN:%s:%s():",
			file,
			function);
		o=n;
		l=~line;
	}
	sprintf(buffer, "%5u", line);
	if (l==line) {
		register int	i;
		
		i=0;
		while (buffer[i])
			buffer[i++]=' ';
	} else
		l=line;
	fprintf(dmg_LOG, "  %s: ", buffer);
	fflush(dmg_LOG);
}


void
dmg_init	(const	char	*log_file)
{
	if (!dmg_initialized()) {
		register unsigned int	lists=65536;
		register unsigned int	i;
		static	 time_t		tt;
		
		atexit(dmg_log_blocks);
		
		if (log_file && log_file!=dmg_logfile && log_file[0]!=0) {
			register unsigned int	l;
			
			l=strlen(log_file);
			if (l<(dmg_BUFFER_SIZE-2))
				for (i=0; i<=l; i++)
					dmg_logfile[i]=log_file[i];
		}
		
		dmg_LOG=fopen(dmg_logfile, "w");
		if (!dmg_LOG)
			dmg_fat("failed to open log file \"%s\": %s",
				dmg_logfile,
				strerror(errno));
		
		tt=time(NULL);
		fprintf(dmg_LOG, "#d_mallog logged on %s",
			ctime(&tt));
		fflush(dmg_LOG);
		
		dmg_LISTS_free=mmap(0, lists*sizeof(dmg_List), PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
		if ((int)dmg_LISTS_free<=0)
			dmg_fat("failed to allocate %8u bytes for %8u links: %s",
				lists*sizeof(dmg_List),
				lists,
				strerror(errno));
		for (i=0; i<lists-1; i++) {
			dmg_LISTS_free[i].prev=NULL;
			dmg_LISTS_free[i].next=&dmg_LISTS_free[i+1];
		}
		dmg_LISTS_free[i].prev=NULL;
		dmg_LISTS_free[i].next=NULL;
		
		dmg_block_id++;
	}
}


void*
dmg_malloc	(unsigned long int	size)
{
	register void	*ptr;
	
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	if (size>0) {
		ptr=mmap(0, size, PROT_READ | PROT_WRITE, MAP_PRIVATE | MAP_ANONYMOUS, -1, 0);
		if ((int)ptr<=0)
			dmg_fat("failed to map %8lu bytes: %s",
				size,
				strerror(errno));
		
		dmg_BLOCKS_used=dmg_List_prepend(dmg_BLOCKS_used, dmg_List_new(ptr, size));
		dmg_BLOCKS_used->block_id=dmg_block_id++;
		
		dmg_log("%s" dmg_FMTstr_block,
			dmg_PREFIX_malloc,
			dmg_BLOCKS_used->block_id,
			(dmg_ulong)dmg_BLOCKS_used->block,
			dmg_BLOCKS_used->length);
	} else {
		dmg_log("%s" "zero length block requested", dmg_PREFIX_malloc);
		ptr=NULL;
	}
	
	return ptr;
}


void*
dmg_realloc	(void			*ptr,
		 unsigned long int	size)
{
	register dmg_List	*list;
	
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	if (!ptr)
		return dmg_malloc(size);
	
	list=dmg_List_lookup(dmg_BLOCKS_used, ptr);
	if (!list)
		dmg_fat("realloc called for unknown memory block %08lx",
			(unsigned long)ptr);
	
	if (size<=list->length) {
		dmg_log(dmg_STR_realloc dmg_FMTstr_block " changed to " dmg_FMTstr_block,
			list->block_id,
			(dmg_ulong)list->block,
			list->length,
			list->block_id,
			(dmg_ulong)list->block,
			size);
		list->length=size;
	} else {
		if ((int)mremap(ptr, list->length, size, MAP_PRIVATE | MAP_ANONYMOUS)<0)
			dmg_fat("failed to reallocate block\n" dmg_STR_spaced dmg_FMTstr_block " with %8lu bytes: %s",
				list->block_id,
				(dmg_ulong)list->block,
				list->length,
				size,
				strerror(errno));
		dmg_log(dmg_STR_realloc dmg_FMTstr_block " changed to " dmg_FMTstr_block,
			list->block_id,
			(dmg_ulong)list->block,
			list->length,
			list->block_id,
			(dmg_ulong)list->block,
			size);
		list->length=size;
	}
	
	return ptr;
}


char*
dmg_strdup	(const char	*str)
{
	
	if (str)
		return strcpy(dmg_malloc(strlen(str)+1), str);
	else
		return NULL;
}


void
dmg_free	(void		*ptr)
{
	register dmg_List	*list;
	
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	if (!ptr) {
		dmg_log(dmg_STR_free "zero block pointer 0x%01lx",
			(dmg_ulong)ptr);
		return;
	}

	list=dmg_List_lookup(dmg_BLOCKS_used, ptr);
	if (!list) {
		dmg_log(dmg_STR_free "unknown/foreign block pointer 0x%09lx",
			(dmg_ulong)ptr);
		return;
	}
	
	if (1 /* do protecting? */) {
		if ((int)mprotect(list->block, list->length, PROT_NONE)<0)
			dmg_fat("failed to protect block\n" dmg_STR_spaced dmg_FMTstr_block ": %s",
			list->block_id,
			(dmg_ulong)list->block,
			list->length,
			strerror(errno));
		dmg_log(dmg_STR_free dmg_FMTstr_block " (protecting only)",
			list->block_id,
			(dmg_ulong)list->block,
			list->length);
	} else {
		if ((int)munmap(list->block, list->length)<0)
			dmg_fat("failed to unmap block\n" dmg_STR_spaced dmg_FMTstr_block ": %s",
			list->block_id,
			(dmg_ulong)list->block,
			list->length,
			strerror(errno));
		dmg_log(dmg_STR_free dmg_FMTstr_block " (unmapping)",
			list->block_id,
			(dmg_ulong)list->block,
			list->length);
	}
	dmg_BLOCKS_used=dmg_List_remove(dmg_BLOCKS_used, list);
}


void*
dmg_calloc	(unsigned long int	n_mem_blocks,
		 unsigned long int	block_size)
{
	register dmg_List	*dlist;
	register void		*ptr;
	register char		*orgstr;
	
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	orgstr=dmg_PREFIX_malloc;
	dmg_PREFIX_malloc=dmg_STR_calloc;
	ptr=dmg_malloc(n_mem_blocks*block_size);
	dmg_PREFIX_malloc=orgstr;
	
	if (ptr) {
		dlist=dmg_List_lookup(dmg_BLOCKS_used, ptr);
	
		if (!dlist)
			dmg_fat("failed to lookup new block at 0x%09lx",
				(dmg_ulong)ptr);
	
		memset(ptr, 0, dlist->length);
	}
	
	return ptr;
}


void
dmg_log_blocks	(void)
{
	register dmg_List	*list;
	
	if (!dmg_initialized())	dmg_init(dmg_logfile);
	
	list=dmg_BLOCKS_used;
	
	dmg_log(dmg_STR_info "still allocated blocks out of %lu (0x%05lx):",
		dmg_block_id-1,
		dmg_block_id-1);
	while (list) {
		dmg_log(dmg_STR_spaced dmg_FMTstr_block,
			list->block_id,
			(dmg_ulong)list->block,
			list->length);
		list=list->next;
	}
}
