/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __MEMORY_H__
#define __MEMORY_H__

#include <glib.h>

/* ----------------- === user interface === ------------------

    These macros are the user interface to the COW memory manager.
    Instead of using a raw pointer, you create a GimpMemPtr object
    (which is just a smart pointer) and operate through that.  The
    smart pointer has a copy of the real data pointer.  This copy is
    NULL if the readwrite/release balance is zero.

    The smart pointer never moves (once you have a pointer to it, the
    pointer is always valid).  However, the copy of the real data
    pointer (inside the smart pointer) can change after ANY operation
    using the smart pointer.  Always use the macros to read it so you
    get the newest value.

    The first set (d_*) are commands that operate on the smart
    pointer.  They must appear on the RHS.

    The second set (*_d) are different casts of the real data pointer.
    These are okay on either the LHS or RHS.

    q    - GimpMemPtr *
    new  - GimpMemPtr *
    size - int */

/* construct and destruct a smart pointer */
#define   d_init(q)              (gimp_memptr_init((q)))
#define   d_uninit(q)            (gimp_memptr_uninit((q)))

/* malloc/free equivalents */
#define   d_alloc(q,size)        (gimp_memptr_alloc((q),(size)))
#define   d_unalloc(q)           (gimp_memptr_unalloc((q)))

/* grab and release the data */
#define   d_read(q)              (gimp_memptr_use((q),FALSE))
#define   d_write(q)             (gimp_memptr_use((q),TRUE))
#define   d_release(q)           (gimp_memptr_unuse((q)))

/* make 'new' share 'q's data if 'q' allows it, copy it if not  */
#define   d_join(q,new)          (gimp_memptr_join((q),(new)))

/* ensure 'q's data is unshared, copy if needed */
#define   d_split(q)             (gimp_memptr_split((q)))

/* have we done a d_alloc? */
#define   d_is_alloced(q)         ((q)->id != InvalidGimpMemId)

/* is our memory shared?   nomeansno, yesmeansmaybe */
#define   d_is_joined(q)          ((q)->joined == TRUE)

/* what's our readwrite/release balance? */
#define   d_usecount(q)          ((q)->usecount)



/* LHS expressions to cast the real buffer to different types */
#define   void_d(q)             ((q)->data)
#define   gchar_d(q)            ((gchar*)(q)->data)
#define   guchar_d(q)           ((guchar*)(q)->data)

/* ------------ === end of user interface === -----------------*/




typedef struct _GimpMemPtr GimpMemPtr;
typedef struct _GimpMemMgr GimpMemMgr;
typedef struct _GimpMem GimpMem;


/* a unique identifier for a piece of memory */
typedef GimpMem * GimpMemId;
#define InvalidGimpMemId NULL


/* a pointer to a piece of sharable swappable memory.  'id' and
   'data' are volatile, and may change after any operation on this
   object.  'data' will be NULL if usecount == 0 */
struct _GimpMemPtr
{
  /* the GimpMem we are associated with */
  GimpMemId id;

  /* the number of local refs */
  guint usecount;

  /* might we be sharing our GimpMem? */
  gboolean joined;

  /* a local copy of the data pointer in the GimpMem */
  void * data;
};

#define   gmp_is_alloced(gmp)    ((gmp)->id != InvalidGimpMemId)
#define   gmp_is_joined(gmp)     ((gmp)->joined == TRUE)
#define   gmp_usecount(gmp)      ((gmp)->usecount)
#define   gmp_data(gmp)          ((gmp)->data)


/* create and destroy a pointer */
void      gimp_memptr_init      (GimpMemPtr  *ptr);
void      gimp_memptr_uninit    (GimpMemPtr  *ptr);

/* reserve new private storage space and mark it as sharable */
gboolean  gimp_memptr_alloc     (GimpMemPtr  *ptr,
                                 gint         size);

/* unreserve storage space.  you cannot unalloc a ptr with outstanding
   uses.  storage lives on until a private (ie: last) alloc is dropped */
gboolean  gimp_memptr_unalloc   (GimpMemPtr  *ptr);

/* assert your intention to use ptr. you cannot use an unalloced ptr.
   if dirty is TRUE, make sure storage is not shared, copy if needed,
   and mark storage as unsharable. */
gboolean  gimp_memptr_use       (GimpMemPtr  *ptr,
                                 gboolean     dirty);

/* negate your intention to use ptr.  the final unuse on an unsharable
   piece of storage will mark it as sharable */
gboolean  gimp_memptr_unuse     (GimpMemPtr  *ptr);

/* attach newptr to storage of ptr.  ptr must be alloced, newptr must
   be unalloced.  if ptr is marked as unsharable, make a copy of the
   storage instead of sharing.  the copy will be sharable. */
gboolean  gimp_memptr_join      (GimpMemPtr  *ptr,
                                 GimpMemPtr  *newptr);

/* if ptr is shared, make a private copy and detach from the shared
   one.  the private copy will be sharable.  all outstanding uses for
   ptr are moved from old storage to new storage.  no effect if ptr is
   unalloced or unshared */
gboolean  gimp_memptr_split     (GimpMemPtr  *ptr);



#endif /* __MEMORY_H__ */
