
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************
*
*   File:    trirevise.c
*
*   Purpose: Contains routines for modifying surface
*            triangulations. 
*
*/

#include "include.h"

/****************************************************************
*
*  Function: refine()
*
*  Purpose:  Refines current triangulation by subdividing each
*            triangulation into four congruent triangles.
*/

void refine()    /* REAL the resolution of triangulation */
{
  vertex_id v_id;
  edge_id e_id;
  facet_id f_id;

  web.vol_flag = 0;

  /* clean out NEW flags */
  FOR_ALL_VERTICES(v_id)
    unset_attr(v_id,NEWVERTEX);
  FOR_ALL_EDGES(e_id)
    unset_attr(e_id,NEWEDGE);
  FOR_ALL_FACETS(f_id)
    unset_attr(f_id,NEWFACET);

  if ( web.simplex_flag ) 
    { refine_all_simplices(); 
      if ( web.dimension > 2 )
         web.maxscale *= 1 << (web.dimension-2);
      return;
    }

  /* first, subdivide all edges */
  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    { if ( get_eattr(e_id) & NEWEDGE ) continue;
      edge_divide(e_id);
    }

  if ( web.dimension == STRING ) goto windup; 
  /* don't want to subdivide facets */
    
  /* now subdivide each facet */
  f_id = NULLFACET;
  while ( generate_all(FACET,&f_id) )
    { facetedge_id fe;
      facetedge_id next_fe;
      int i,j,k,n;
      REAL *x[FACET_VERTS],*xmid;
      vertex_id vv_id[12];

      if ( get_fattr(f_id) & NEWFACET ) continue;
      fe = get_facet_fe(f_id);

      /* walk around the 6 edges twice, gathering vertices */
      for ( i = 0 ; i < 12 ; i++ )
        {
          vv_id[i] = get_fe_tailv(fe);
          fe = get_next_edge(fe);
        }

      /* walk around subdivided edges, cutting off points as new facets */
      fe = get_facet_fe(f_id);
      for ( i = 0, n = 0 ; i < FACET_EDGES ; n++  )
        { vertex_id headv,midv;
          edge_id next_e;

          next_fe = get_next_edge(fe);
          headv = get_fe_headv(fe);
          if ( get_vattr(headv) & NEWVERTEX )
             { cross_cut(get_prev_edge(fe),fe);
               i++;
               if ( web.modeltype == QUADRATIC )
                 {
                   /* add quadratic correction to linear interpolation */
                   for ( k = 0 ; k < FACET_VERTS ; k++ )
                     x[k] = get_coord(vv_id[n+k+2]);
                   next_e = get_next_edge(fe);
                   midv = get_fe_midv(next_e);
                   xmid = get_coord(midv);
                   for ( j = 0 ; j < web.sdim ; j++ )
                     xmid[j] -= (x[0][j] - 2*x[1][j] + x[2][j])/8.0;
                 }
             }
          fe = next_fe;
        }
    }

windup:
  if ( level < MAXLEVEL-1 ) level++;  /* for extrapolation */
  autochop_size /= 2;   /* keep same relative scale */

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();
}



/***********************************************************
*
*  Function:  areaweed()
*
*  Purpose:   Subdivide long edges of skinny triangles.
*             Should be called after edgeweed() gets rid
*             of short triangles.  Newly created trianges
*             are marked as NEWFACET, and are not tested
*             again in this pass.
*
*  Input:     Area cutoff for skinny triangles.      
*
*  Output:    Skinny triangles have long edges subdivided.
*
*  Return:    Number of facets created.
*/


int areaweed(min_area)
REAL min_area;   /* criterion for weeding out small triangles */
{
  facet_id f_id;  /* facet being worked on */
  int weedcount = 0; /* number of facets created */

  web.vol_flag = 0;

  /* first, unmark all NEWFACET attributes */
  FOR_ALL_FACETS(f_id)
     unset_attr(f_id,NEWFACET);

  /* main loop sweeping over all triangles */
  f_id = NULLFACET;
  while ( generate_all(FACET,&f_id) )
    {
      REAL side[MAXCOORD];  /* side vector */
      REAL sside[FACET_EDGES]; /* squares of side lengths */
      REAL temp;     /* temporary variable in area calculation */
      REAL area;     /* area of triangle */
      facetedge_id fe[FACET_EDGES]; /* edges of triangle */
      edge_id e_id;
      int i;           /* side number */
      int elimcount = 0;


      /* skip already modified triangles */
      if ( get_fattr(f_id) & NEWFACET )
          continue;

      /* find sides and area */
      generate_facet_fe_init();
      for ( i = 0 ; i < FACET_EDGES ; i++ ) 
        { generate_facet_fe(f_id,fe+i);
          get_fe_side(fe[i],side);
          sside[i] = dot(side,side,web.sdim);
        }

      temp = sside[0] + sside[1] - sside[2];
      temp = 4*sside[0]*sside[1] - temp*temp;
      area = sqrt(temp)/4;
      if ( area > min_area )  /* skip big triangles */
          continue;

      /* find longest side */
/*      i = (sside[0] > sside[1]) ? 0 : 1; */
/*      i = (sside[i] > sside[2]) ? i : 2; */
/*      weedcount += edge_refine(get_fe_edge(fe[i])); */
/*  this doesn't really work too well */

  /* weed by eliminating shortest side. eliminate obtuse vertex,
     the one between the two shortest sides, if possible.
     Follow with equiangulation  */
      i = (sside[0] < sside[1]) ? 0 : 1;
      i = (sside[i] < sside[2]) ? i : 2;
      if ( sside[(i+1)%3] < sside[(i+2)%3] )
         e_id = inverse_id(get_fe_edge(fe[i]));
      else e_id = get_fe_edge(fe[i]);
      elimcount = eliminate_edge(e_id);
      if ( elimcount ) free_element(e_id);
      weedcount += elimcount;

   }  /* end main sweep loop */

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

  return weedcount;
}


/***********************************************************************
*
*  Function: area_histogram()
*
*  Purpose: Give user a histogram of area lengths to guide areaweeding.
*           Reports in bins of width powers of two.
*/

void area_histogram()
{
  facet_id f_id;  /* facet being worked on */
  int bincount[HISTO_BINS];
  int n;
  double ref_area = web.total_area/1000;

  for ( n = 0 ; n < HISTO_BINS ; n++ ) bincount[n] = 0;

  /* main loop sweeping over all triangles */
  FOR_ALL_FACETS(f_id)
    {
      REAL side[MAXCOORD];  /* side vector */
      REAL sside[FACET_EDGES]; /* squares of side lengths */
      REAL temp;     /* temporary variable in area calculation */
      REAL area;     /* area of triangle */
      facetedge_id fe[FACET_EDGES]; /* edges of triangle */
      int i;           /* side number */

      /* find sides and area */
      generate_facet_fe_init();
      for ( i = 0 ; i < FACET_EDGES ; i++ ) 
        { generate_facet_fe(f_id,fe+i);
          get_fe_side(fe[i],side);
          sside[i] = dot(side,side,web.sdim);
        }
      temp = sside[0] + sside[1] - sside[2];
      temp = 4*sside[0]*sside[1] - temp*temp;
      area = sqrt(temp)/4;
      if ( area == 0.0 ) n = 0;
      else
        n = HISTO_BINS/2 + 1 + (int)floor(log(area/ref_area)*HISTO_BINSIZE);
      if ( n < 0 ) n = 0;
      if ( n >= HISTO_BINS ) n = HISTO_BINS - 1;
      bincount[n]++;
    }
  outstring("       area             number\n");
  if ( bincount[0] )
    {
      sprintf(msg,"%f - %g     %6d \n",0.0,
          ref_area*exp((-HISTO_BINS/2)/HISTO_BINSIZE), bincount[0]);
      outstring(msg);
    }
  for ( n = 1 ; n < HISTO_BINS ; n++ )
   if ( bincount[n] )
    {
      sprintf(msg,"%g - %g     %6d\n",
            ref_area*exp((n-HISTO_BINS/2-1)/HISTO_BINSIZE),
              exp((n-HISTO_BINS/2)/HISTO_BINSIZE)*ref_area,bincount[n]);
      outstring(msg);
    }
}


/***********************************************************
*
*  Function:  skinny()
*
*  Purpose:   Subdivide long edges of skinny triangles.
*             Newly created trianges
*             are marked as NEWFACET, and are not tested
*             again in this pass.
*
*  Input:     Acutest angle cutoff for skinny triangles.      
*
*  Output:    Skinny triangles have long edges subdivided.
*
*  Return:    Number of facets created.
*/


int skinny(min_angle)
REAL min_angle;   /* criterion for weeding out small triangles */
{
  facet_id f_id;  /* facet being worked on */
  int weedcount = 0; /* number of facets created */

  web.vol_flag = 0;

  /* first, unmark all NEWFACET attributes */
  FOR_ALL_FACETS(f_id)
     unset_attr(f_id,NEWFACET);

  /* main loop sweeping over all triangles */
  f_id = NULLFACET;
  while ( generate_all(FACET,&f_id) )
    {
      REAL side[MAXCOORD];  /* side vector */
      REAL sside[FACET_EDGES]; /* squares of side lengths */
      REAL angle;     /* area of triangle */
      facetedge_id fe[FACET_EDGES]; /* edges of triangle */
      int i;           /* side number */
      int small,mid,big;

      /* skip already modified triangles */
      if ( get_fattr(f_id) & NEWFACET )
          continue;

      /* find sides */
      generate_facet_fe_init();
      for ( i = 0 ; i < FACET_EDGES ; i++ ) 
        { generate_facet_fe(f_id,fe+i);
          get_fe_side(fe[i],side);
          sside[i] = dot(side,side,web.sdim);
        }

      /* find shortest side */
      small = (sside[0] < sside[1]) ? 0 : 1;
      small = (sside[small] < sside[2]) ? small : 2;
      /* find longest side */
      big = (sside[0] > sside[1]) ? 0 : 1;
      big = (sside[big] > sside[2]) ? big : 2;
      /* find middle side */
      mid = 3 - (small+big);
      angle = acos( (sside[mid]+sside[big]-sside[small])/2/
			 sqrt(sside[mid]*sside[big]) );
      if ( angle > min_angle )  /* skip fat triangles */
          continue;
      edge_refine(get_fe_edge(fe[big]));

      weedcount++;

   }  /* end main sweep loop */

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

  return weedcount;
}


/***********************************************************************
*
*  Function: skinny_histogram()
*
*  Purpose: Give user a histogram of acute angles to guide skinnyweeding.
*           Reports in bins of width powers of two.
*/

void skinny_histogram()
{
  facet_id f_id;  /* facet being worked on */ 
  int bincount[HISTO_BINS]; 
  int n; 
 
  for ( n = 0 ; n < HISTO_BINS ; n++ ) bincount[n] = 0;
  
  /* main loop sweeping over all triangles */ 
  FOR_ALL_FACETS(f_id)
    { REAL side[MAXCOORD];  /* side vector */ 
      REAL sside[FACET_EDGES]; /* squares of side lengths */ 
      REAL angle;     /* area of triangle */
      facetedge_id fe[FACET_EDGES]; /* edges of triangle */
      int i;           /* side number */
      int small,mid,big;

      /* find sides and area */
      generate_facet_fe_init();
      for ( i = 0 ; i < FACET_EDGES ; i++ ) 
        { generate_facet_fe(f_id,fe+i);
          get_fe_side(fe[i],side);
          sside[i] = dot(side,side,web.sdim);
        }
      /* find shortest side */
      small = (sside[0] < sside[1]) ? 0 : 1;
      small = (sside[small] < sside[2]) ? small : 2;
      /* find longest side */
      big = (sside[0] > sside[1]) ? 0 : 1;
      big = (sside[big] > sside[2]) ? big : 2;
      /* find middle side */
      mid = 3 - (small+big);
      angle = acos((sside[mid]+sside[big]-sside[small])/2/
			 sqrt(sside[mid]*sside[big]) );
      if ( angle == 0.0 ) n = 0;
      else
        n = HISTO_BINS + (int)floor(log(angle/M_PI)*HISTO_BINSIZE); 
	
      if ( n < 0 ) n = 0; if ( n >= HISTO_BINS ) n = HISTO_BINS - 1;
      bincount[n]++;
    }
  outstring("       angle             number\n");
  if ( bincount[0] )
    {
      sprintf(msg,"%f - %g     %6d \n",0.0,
          M_PI*exp((-HISTO_BINS/2)/HISTO_BINSIZE), bincount[0]);
      outstring(msg);
    }
  for ( n = 1 ; n < HISTO_BINS ; n++ )
   if ( bincount[n] )
    {
      sprintf(msg,"%g - %g     %6d\n",
            M_PI*exp((n-HISTO_BINS)/HISTO_BINSIZE),
              exp((n-HISTO_BINS+1)/HISTO_BINSIZE)*M_PI,bincount[n]);
      outstring(msg);
    }
}


/******************************************************************
*
*  Function:  edgeweed()
*
*  Purpose: eliminate all unfixed edges whose length is less
*           than min_length.  Will not remove fixed edges.
*           Only removes free boundary edges with both ends
*           on same boundary.
*
*  Input:   none
*
*  Output:  number of facets removed
*/

int edgeweed(min_length)
REAL min_length; /* minimum allowed edge length */
{
  edge_id e_id;
  int weedcount = 0;

  web.vol_flag = 0;

  /* main loop over all edges */

  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    {
      REAL side_len;  /* actual side length */
      REAL side[MAXCOORD];   /* side vector */
      int elimcount = 0;

      get_edge_side(e_id,side);
      side_len = sqrt(dot(side,side,web.sdim));

      if ( side_len < min_length ) elimcount = eliminate_edge(e_id);
      if ( elimcount ) free_element(e_id);
      weedcount += elimcount;
     }

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

   return weedcount;
} 

/***********************************************************************
*
*  Function: edge_histogram()
*
*  Purpose: Give user a histogram of edge lengths to guide edgeweeding.
*           Reports in bins of width powers of two.
*/

void edge_histogram()
{
  edge_id e_id;
  int bincount[HISTO_BINS];
  int n;

  for ( n = 0 ; n < HISTO_BINS ; n++ ) bincount[n] = 0;

  /* main loop over all edges */

  FOR_ALL_EDGES(e_id)
    {
      REAL side_len;  /* actual side length */
      REAL side[MAXCOORD];   /* side vector */

      get_edge_side(e_id,side);
      side_len = sqrt(dot(side,side,web.sdim));
      n = HISTO_BINS/2 + 1 + (int)floor(log(side_len/overall_size)*HISTO_BINSIZE);
      if ( n < 0 ) n = 0;
      if ( n >= HISTO_BINS ) n = HISTO_BINS - 1;
      bincount[n]++;
    }
  outstring("    side length         number\n");
  if ( bincount[0] )
    {
      sprintf(msg,"%f - %g     %6d \n",0.0,
         overall_size*exp((-HISTO_BINS/2)/HISTO_BINSIZE),
                                                  bincount[0]);
      outstring(msg);
    }
  for ( n = 1 ; n < HISTO_BINS ; n++ )
   if ( bincount[n] )
    {
      sprintf(msg,"%g - %g     %6d\n",
         overall_size*exp((n-HISTO_BINS/2-1)/HISTO_BINSIZE),
         overall_size*exp((n-HISTO_BINS/2)/HISTO_BINSIZE),bincount[n]);
      outstring(msg);
    }
}


/*******************************************************************
*
*  Function: eliminate_edge()
*
*  Purpose: delete an edge and adjacent facets (if triangles in STRING);
*           merges head of edge to tail, unless head fixed.
*           If both ends fixed, will do nothing.
*           Does not deallocate edge itself; caller must do that
*           due to use in chaining to next edge.
*
*  Input:   edge id of edge to eliminate
*
*  Output:  returns number of edges eliminated
*/

int eliminate_edge(short_edge)
edge_id short_edge;
{
  facetedge_id base_fe;  /* along edge to be eliminated */
  vertex_id headv,tailv;
  vertex_id elim_v;  /* vertex to eliminate */
  vertex_id keep_v;  /* vertex to keep */
  WRAPTYPE wrap = 0;
  int edge_head_same,edge_tail_same;  /* whether same constraints and stuff */
  facetedge_id fe_id;
  vertex_id v_id;

  if ( !valid_element(short_edge) ) return 0;
  if ( get_eattr(short_edge) & FIXED ) return 0;
  web.vol_flag = 0;

  headv = get_edge_headv(short_edge);
  tailv = get_edge_tailv(short_edge);

  /* Figure out which vertex to eliminate.  Keep fixed vertices. */
  edge_head_same =  (get_boundary(headv) == get_edge_boundary(short_edge) ) 
       && equal_constr(headv,short_edge) && !( get_vattr(headv) & FIXED); 
  edge_tail_same =  (get_boundary(tailv) == get_edge_boundary(short_edge) ) 
       && equal_constr(tailv,short_edge) && !( get_vattr(tailv) & FIXED); 
  if ( !edge_head_same )
    { if ( edge_tail_same ) short_edge = edge_inverse(short_edge);
      else return 0; /* can't safely eliminate edge */
    }
 
  elim_v = get_edge_headv(short_edge);
  keep_v = get_edge_tailv(short_edge);

  /* put keep_v at middle of old edge if possible */
  if ( web.modeltype == LINEAR )
    if ( edge_head_same && edge_tail_same )
     {
      if ( web.dimension == STRING )
       {
        /* could eliminate either end, so keep high valence vertex */
        facetedge_id fe = get_edge_fe(short_edge);
        facetedge_id ffe = get_next_facet(fe);
	edge_id e1,e2;
        e1 = get_fe_edge(get_prev_edge(fe));
	e2 = get_fe_edge(get_prev_edge(ffe));
        edge_tail_same = equal_id(e1,e2);
        e1 = get_fe_edge(get_next_edge(fe));
	e2 = get_fe_edge(get_next_edge(ffe));
        edge_head_same = equal_id(e1,e2);
        if ( edge_tail_same && !edge_head_same )
          { short_edge = edge_inverse(short_edge);
            elim_v = get_edge_headv(short_edge);
            keep_v = get_edge_tailv(short_edge);
          }
       }
      if ( edge_tail_same == edge_head_same )
        { REAL *tailx = get_coord(keep_v);
  	  REAL side[MAXCOORD];
	  int i;
	  get_edge_side(short_edge,side);
	  for ( i = 0 ; i < web.sdim ; i++ )
	     tailx[i] += side[i]/2;
        }
      }

  if ( web.symmetry_flag )
    wrap = get_edge_wrap(short_edge);

  /* change all references to the eliminated vertex to the
     kept vertex. */
  if ( !equal_id(elim_v,keep_v) )
    { facetedge_id short_fe;

      short_fe  = fe_inverse(get_edge_fe(short_edge));
      if ( !valid_id(short_fe) )
	{ error("Eliminated edge must be on facet.\n",WARNING);
	  return 0;
        }
      change_vertex(short_fe,elim_v,keep_v,wrap); /* recursive */
      /* note: if elim_v == keep_v, then we had a loop
         shrinking down to a point, and this is the last edge to go. */
    }

  /* Go through facets around edge, adjusting facet loops
     of the merged edges, deleting facetedges, and facets */
  generate_edge_fe_init();
  generate_edge_fe(short_edge,&base_fe);
  while ( valid_id(base_fe) )
   {
     facetedge_id next_base;  /* to get before freeing base_fe */

     if ( (web.dimension == STRING) && (!valid_id(get_fe_facet(base_fe)) ||
         !equal_id(get_next_edge(get_next_edge(base_fe)),
                            get_prev_edge(base_fe))) ) 
       {
         /* do not eliminate facets with more than 3 edges */
	 /* but might have free end */
	 facetedge_id next_fe,prev_fe;
	 next_fe = get_next_edge(base_fe);
	 if ( equal_element(short_edge,get_fe_edge(next_fe)) )
	   next_fe = get_next_edge(base_fe);
	 prev_fe = get_prev_edge(base_fe);
	 if ( equal_element(short_edge,get_fe_edge(prev_fe)) )
	   prev_fe = get_prev_edge(base_fe);
         set_next_edge(get_prev_edge(base_fe),get_next_edge(base_fe));
         set_prev_edge(get_next_edge(base_fe),get_prev_edge(base_fe));
	 if ( valid_id(get_fe_facet(base_fe)) )
  	   set_facet_fe(get_fe_facet(base_fe),get_next_edge(base_fe));
         set_vertex_fe(keep_v,inverse_id(prev_fe));
       }
      else
       { /* eliminate whole facet */
         facetedge_id a;  /* for edge to be merged */
         facetedge_id b;  /* for edge to be merged with */
         facetedge_id a_next,a_prev,b_next,b_prev;
                      /* facet chain links around edges a,b */
         facetedge_id next_fe;  /* next around facet loop of a */
         edge_id  a_edge; /* edge of side a */
         edge_id  b_edge; /* edge of side b */
         facet_id  facet;  /* facet to be eliminated */ 
         edge_id keep_edge,throw_edge;

         facet = get_fe_facet(base_fe);
         a = get_next_edge(base_fe);
         a_edge = get_fe_edge(a);
         a_next = get_next_facet(a);
         a_prev = get_prev_facet(a);
         b = get_prev_edge(base_fe);
         b_edge = get_fe_edge(b);
         b_next = get_next_facet(b);
         b_prev = get_prev_facet(b);

         /* keep edge with constraints or boundaries */
         if ( get_eattr(a_edge) & (CONSTRAINT|BOUNDARY) )
           { keep_edge = inverse_id(a_edge); throw_edge = b_edge; }
         else
           { keep_edge = b_edge; throw_edge = a_edge; }

         if ( equal_id(a_next,a) )  /* single facet on edge a */
           {  
             if ( equal_id(b_next,b) ) /* and on edge b */
               set_edge_fe(keep_edge,NULLID);
             else  /* have more around b */
               {
                 set_next_facet(b_prev,b_next);
                 set_prev_facet(b_next,b_prev);
                 set_edge_fe(keep_edge,b_next);
		 set_vertex_fe(keep_v,b_next);
               }
           }
         else if ( equal_id(b_next,b) )  /* single facet on edge b */
           { set_next_facet(a_prev,a_next);
             set_prev_facet(a_next,a_prev);
             set_edge_fe(keep_edge,fe_inverse(a_next));
	     set_vertex_fe(keep_v,fe_inverse(a_next));
           }
         else if ( equal_element(a_edge,b_edge) )
           { /* excise a,b from loop */
             set_prev_facet(a_next,a_prev);
             set_next_facet(a_prev,a_next);
             b_next = get_next_facet(b); /* in case of change */
             b_prev = get_prev_facet(b);
             if ( equal_id(b,b_next) )
               set_edge_fe(keep_edge,NULLEDGE); /* dropped only facet */
             else
               { /* excise b */
                 set_prev_facet(b_next,b_prev);
                 set_next_facet(b_prev,b_next);
                 set_edge_fe(keep_edge,b_next);
		 set_vertex_fe(keep_v,fe_inverse(b_next));
               }
           }
         else  /* have to join chains */
           { set_prev_facet(a_next,fe_inverse(b_next));
             set_next_facet(a_prev,fe_inverse(b_prev));
             set_next_facet(b_prev,fe_inverse(a_prev));
             set_prev_facet(b_next,fe_inverse(a_next));
             set_edge_fe(keep_edge,fe_inverse(a_next));
             set_vertex_fe(keep_v,a_next);
           }
           
         /* fix edge references around deleted edge to refer to kept edge */
         next_fe = get_edge_fe(keep_edge);
         if ( valid_id(next_fe) )
           do
             {
               set_fe_edge(next_fe,keep_edge);
               next_fe = get_next_facet(next_fe);
             }
           while ( next_fe != get_edge_fe(keep_edge) );

         /* fix any body references to facet */
	 if ( valid_id(facet) )
	  {
            set_body_fe(get_facet_body(facet),fe_inverse(b_next));
            set_body_fe(get_facet_body(facet_inverse(facet)),b_next);
          }

         v_id = get_fe_tailv(b);
         set_vertex_fe(v_id,b_next);

         /* free structures  */
         if ( !equal_element(a_edge,b_edge) ) free_element(throw_edge);
         if ( !valid_id(get_edge_fe(keep_edge)) ) free_element(keep_edge);
         free_element(a);
         free_element(b);
	 check_vertex_fe(v_id);
	 if ( valid_id(facet) ) free_element(facet);
       }
     next_base = base_fe;
     generate_edge_fe(short_edge,&next_base); /* before freeing! */
     free_element(base_fe);
     base_fe = next_base;

   }  /* end facets around short edge loop */

  if ( web.modeltype == QUADRATIC )
        free_element(get_edge_midv(short_edge));
  if ( !equal_id(elim_v,keep_v) )   /* see note above */
       free_element(elim_v);

     /* note: short_edge cannot be eliminated here since caller
        has to use it to continue edge generation.  Caller 
        cannot save next edge before calling, since this routine
        frees other edges, which might include the saved one. 
     */

  check_vertex_fe(keep_v);

  return 1;
}

/******************************************************************
*
* function: check_vertex_fe()
*
* purpose: make sure vertex has legal facetedge
*
*/

void check_vertex_fe(v_id)
vertex_id v_id;
{ facetedge_id fe_id;
  /* check vertex facetedge */
  fe_id = get_vertex_fe(v_id);
  if ( !valid_id(fe_id) || !equal_id(v_id,get_fe_tailv(fe_id)) )
   { sprintf(errmsg,"Vertex %d has bad facetedge link.\n",ordinal(v_id)+1);
     error(errmsg,WARNING);
     /* try to find one */
     set_vertex_fe(v_id,NULLID);
     FOR_ALL_FACETEDGES(fe_id)
       { if ( equal_id(v_id,get_fe_tailv(fe_id)) )
	   set_vertex_fe(v_id,fe_id);
       }
   }
}

/************************************************************
*
*  Function: change_vertex()
*
*  Purpose:  Recursively goes through facetedges with tail
*            at old vertex and put tail at new vertex.
*            To seek new edges, looks at facet loop around
*            edge predecessor.
*/

void change_vertex(fe,old_v,new_v,wrap)
facetedge_id fe;  /* known to have tail at old vertex */
vertex_id  old_v; /* old vertex */
vertex_id  new_v; /* new vertex */
WRAPTYPE wrap;  /* wraps to add to edges */
{
  facetedge_id next_fe;  /* looping around current edge */
  facetedge_id pre_fe;   /* going back around facet to new edge */
  edge_id e_id;

  if ( equal_id(old_v,new_v) )
     error("change_vertex: looping edge! ",WARNING);

  set_vertex_fe(new_v,fe);
  e_id = get_fe_edge(fe);
  set_edge_tailv(e_id,new_v);
  if ( web.symmetry_flag )
      set_edge_wrap(e_id,(*sym_compose)(wrap,get_edge_wrap(e_id)));

  next_fe = fe;
  do
    { pre_fe = fe_inverse(get_prev_edge(next_fe));
      if ( equal_id(get_fe_tailv(pre_fe),old_v) )
        change_vertex(pre_fe,old_v,new_v,wrap);
      next_fe = get_next_facet(next_fe);
    }
  while ( !equal_id(next_fe,fe) );
}


/*******************************************************************
*
*  Function: articulate()
*
*  Purpose: Subdivides all existing edges longer than a given length, and
*           re-triangulates accordingly.
*
*  Input:   REAL max_len - upper bound for final edges
*
*  Output:  all edges less than max_len
*
*  Return value: number of facets created
*/

int articulate(max_len)
REAL max_len;  /* maximum allowed edge length */
{
  edge_id e_id;
  int new_edge_count = 0;
  edge_id last_edge = web.skel[EDGE].last;  /* so know when old edges done */

  web.vol_flag = 0;

  /* first, a little error check */
  if ( max_len <= 0.0 ) 
    error("Must have positive minimum length.\n",RECOVERABLE);

  /* main loop over all edges */
  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    {
      REAL side_len;  /* actual side length */
      REAL side[MAXCOORD];   /* side vector */

      get_edge_side(e_id,side);
      side_len = sqrt(dot(side,side,web.sdim));

      if ( side_len > max_len ) 
        {
          new_edge_count += edge_refine(e_id);
        }
      if ( equal_id(e_id,last_edge) )
        break;  /* done with old edges */
     }

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

   return new_edge_count;
} 

/*************************************************************
*
*  Function: edge_refine()
*
*  Purpose: subdivide an edge of a triangulation, and also
*           subdivide adjacent facets.
*
*  Input: edge id of edge to refine
*
*  Output:  number of new facets created.
*/

int edge_refine(e_id)
edge_id e_id;
{
  facetedge_id fe; /* for facet being subdvided */
  int new_facet_count = 0;

  edge_divide(e_id);  /* e_id now tail half of old edge */

  if ( web.dimension == SOAPFILM )
    {
      /* now go around facet loop of edge dividing facets */
      generate_edge_fe_init();
      while ( generate_edge_fe(e_id,&fe) )
        { cross_cut(get_prev_edge(fe),fe);
          new_facet_count++;
        }
    }
  else new_facet_count++;  /* edges for soapfilm */

  return new_facet_count;
}

/*************************************************************
*
*  Function: equiangulate()
*
*  Purpose:  Switch diagonals of quadrilaterals around so
*            that the triangulation becomes more equiangular.
*            The criterion used is that for a Delauney triangulation:
*            the sum of the angles opposite a common base of
*            two triangles should be less than pi.  Only 
*            quadrilaterals within a face are examined, and
*            fixed edges are not affected.  Angles next to
*            diagonal must be acute.
*
*  Input:    Triangulation.
*
*  Output:   One pass made through alll edges, and diagonals
*            switched if appropriate.
*
*  Return value: Number of edges switched.
*/

int equiangulate()
{
  int switchcount = 0;
  edge_id e_id;  /* edge being examined */
  vertex_id v_id;

  if ( web.simplex_flag ) { simplex_equiangulate(); return 0; }
  
  web.vol_flag = 0;

  /* first, calculate all edge lengths */
  FOR_ALL_EDGES(e_id)
     calc_edge(e_id);

  /* main loop through edges */
  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    {
      facetedge_id fe_a; /* for edge under test */
      facetedge_id fe_ai; /* other facetedge of e_id */
      REAL a;  /* length of e_id */
      facetedge_id fe_b,fe_c; /* other sides of one triangle */
      REAL b,c;  /* lengths of other sides of one triangle */
      facetedge_id fe_d,fe_e; /* other sides of other triangle */
      REAL d,e;  /* lengths of other sides of other triangle */
      facet_id f1,f2;

      if ( get_eattr(e_id) & FIXED ) continue;

     
      /* test to be sure edge has at most two adjacent facets */
      fe_a = get_edge_fe(e_id);
      if ( !equal_id(get_next_facet(fe_a),get_prev_facet(fe_a)) ) continue;

      f1 = get_fe_facet(fe_a);
      f2 = get_fe_facet(get_next_facet(fe_a));

      /* test for equal density */
      if ( (get_fattr(f1)&DENSITY) || (get_fattr(f2)&DENSITY) )
        if ( fabs(get_facet_density(f1) - get_facet_density(f2)) > 1e-10 )
           continue;

      /* test for equal constraints */
      if ( (get_e_constraint_map(e_id) != get_f_constraint_map(f1)) 
          || (get_e_constraint_map(e_id) != get_f_constraint_map(f2)) )
        continue;
 
      /* test for equal boundary */
      if ( ( get_edge_boundary(e_id) != get_facet_boundary(f1))
         ||( get_edge_boundary(e_id) != get_facet_boundary(f2)) )
        continue;

      /* test equiangularity */
      a = get_edge_length(e_id);
      fe_b = get_next_edge(fe_a);
      b = get_edge_length(MSCBUG get_fe_edge(fe_b));
      fe_c = get_prev_edge(fe_a);
      c = get_edge_length(MSCBUG get_fe_edge(fe_c));
      fe_ai = fe_inverse(get_next_facet(fe_a));
      fe_d = get_next_edge(fe_ai); 
      d = get_edge_length(MSCBUG get_fe_edge(fe_d));
      fe_e = get_prev_edge(fe_ai); 
      e = get_edge_length(MSCBUG get_fe_edge(fe_e));
      if ( (b*b + c*c - a*a)/b/c + (d*d + e*e - a*a)/d/e > -0.001 )
         continue;
                               /* -0.01 prevents cycling */

      /* test acuteness ??? (fix 0.0001 to make scale invariant)*/
/*      if ( a*a + d*d <= e*e + 0.0001 ) continue;
      if ( a*a + e*e <= d*d + 0.0001 ) continue;
      if ( a*a + b*b <= c*c + 0.0001 ) continue;
      if ( a*a + c*c <= b*b + 0.0001 ) continue;
      if ( a*a + d*d <= e*e + 0.0001 ) continue;
*/
      /* may want to switch, but test that opposite vertices are different */
      if ( equal_id(get_fe_tailv(fe_c),get_fe_headv(fe_d)) ) continue;

      /* if we are here, we want to switch diagonals */
      set_vertex_fe(get_edge_headv(e_id),fe_b);
      set_vertex_fe(get_edge_tailv(e_id),fe_d);
      set_facet_fe(get_fe_facet(fe_a),fe_a);
      set_facet_fe(get_fe_facet(fe_ai),fe_ai);
      set_fe_facet(fe_b,get_fe_facet(fe_ai));
      set_fe_facet(fe_d,get_fe_facet(fe_a));
      v_id = get_fe_tailv(fe_c);
      set_edge_headv(e_id,v_id);
      v_id = get_fe_headv(fe_d);
      set_edge_tailv(e_id,v_id);
      set_next_edge(fe_a,fe_c);
      set_prev_edge(fe_a,fe_d);
      set_next_edge(fe_ai,fe_e);
      set_prev_edge(fe_ai,fe_b);
      set_next_edge(fe_b,fe_ai);
      set_prev_edge(fe_b,fe_e);
      set_next_edge(fe_c,fe_d);
      set_prev_edge(fe_c,fe_a);
      set_next_edge(fe_d,fe_a);
      set_prev_edge(fe_d,fe_c);
      set_next_edge(fe_e,fe_b);
      set_prev_edge(fe_e,fe_ai);
      calc_edge(e_id);

      if ( web.symmetry_flag )
          { 
            edge_id ed,ec;
            WRAPTYPE w;

            ed = get_fe_edge(fe_d);
            ec = get_fe_edge(fe_c);
            w = (*sym_compose)(get_edge_wrap(ec),get_edge_wrap(ed));
            set_edge_wrap(e_id,(*sym_inverse)(w));
          }

      switchcount++;
    }

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

  return switchcount;
}


/******************************************************************
*
*  Function:  ridge_notcher()
*
*  Purpose:   Subdivide internal edges in a face whose adjacent
*             faces are non-parallel to a certain degree.
*
*  Input:     REAL max_angle - maximum allowed angle of deviation
*
*  Output:    offending edges subdivided, with middle vertex set
*             to the average of the four adjacent vertices
*
*/

int ridge_notcher(max_angle)
REAL max_angle;
{
  int notchcount = 0;  /* number of edges notched */
  edge_id e_id;        /* edge to notch */
  REAL max_cos;      /* cosine of max_angle */
  facet_id *todo;    /* for list of facets to refine */
  int i;
  int maxfacet = web.skel[FACET].max_ord;

  web.vol_flag = 0;

  max_cos = cos(max_angle);

  todo = (facet_id *)temp_calloc(web.skel[FACET].max_ord,sizeof(facet_id));

  /* first, unmark all NEWEDGE attributes */
  FOR_ALL_EDGES(e_id)
     unset_attr(e_id,NEWEDGE);

  /* main loop over all edges */
  e_id = NULLEDGE;
  while ( generate_all(EDGE,&e_id) )
    {
      facetedge_id fe_a; /* for edge under test */
      facetedge_id fe_ai; /* other facetedge of e_id */
      facetedge_id fe_b; /* other side of one triangle */
      facetedge_id fe_d; /* other side of other triangle */
      REAL a[MAXCOORD],b[MAXCOORD],d[MAXCOORD];  /* edge vectors */
      REAL normb[MAXCOORD];         /* normal to one facet */
      REAL normd[MAXCOORD];         /* normal to other facet */
      REAL denom;           /* denominator of cosine formula */

      if ( get_eattr(e_id) & (FIXED | BOUNDARY | HIT_WALL | NEWEDGE) )
         continue;

      /* test to be sure edge has at most two adjacent facets */
      fe_a = get_edge_fe(e_id);
      if ( !equal_id(get_next_facet(fe_a),get_prev_facet(fe_a)) ) continue;
      if ( equal_id(get_next_facet(fe_a),fe_a) ) continue;

      /* test parallelism */
      get_fe_side(fe_a,a);
      fe_b = get_next_edge(fe_a);
      get_fe_side(fe_b,b);
      fe_ai = fe_inverse(get_next_facet(fe_a));
      fe_d = get_next_edge(fe_ai); 
      get_fe_side(fe_d,d);
      cross_prod(a,b,normb);
      cross_prod(a,d,normd);
      denom = sqrt(dot(normb,normb,web.sdim)*dot(normd,normd,web.sdim));
      if ( -dot(normb,normd,web.sdim)/denom > max_cos )
        continue;

#ifdef OLDNOTCHER
      /* subdivide edge */
      edge_refine(e_id);

      /* set coordinates of new vertex to average of old */
      x = get_coord(get_edge_headv(e_id));
      for ( i = 0 ; i < web.sdim ; i++ ) x[i] = 0.0;

      y = get_coord(get_fe_tailv(fe_d));
      for ( i = 0 ; i < web.sdim ; i++ ) x[i] += y[i]/4;

      y = get_coord(get_fe_headv(fe_d));
      for ( i = 0 ; i < web.sdim ; i++ ) x[i] += y[i]/4;

      y = get_coord(get_fe_tailv(fe_b));
      for ( i = 0 ; i < web.sdim ; i++ ) x[i] += y[i]/4;

      y = get_coord(get_fe_headv(fe_b));
      for ( i = 0 ; i < web.sdim ; i++ ) x[i] += y[i]/4;

      set_attr(e_id,NEWEDGE);
      set_attr(get_fe_edge(get_next_edge(fe_a)),NEWEDGE);
      notchcount++;
#else
      /* record neighboring facets for subdivision */
      todo[ordinal(get_fe_facet(fe_a))] = get_fe_facet(fe_a);
      todo[ordinal(get_fe_facet(fe_ai))] = get_fe_facet(fe_ai);
#endif
    }

#ifndef OLDNOTCHER
  for ( i = 0 ; i < maxfacet ; i++ )
    if ( valid_id(todo[i]) )
      { face_triangulate(todo[i],FACET_EDGES);
	notchcount++;
      }
#endif
  temp_free((char*)todo);

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

  return notchcount;
}

/******************************************************************
*
*  Function:  ridge_histogram()
*
*  Purpose:   Calculate histogram of ridge angles.
*
*
*/

void ridge_histogram()
{
  edge_id e_id;        /* edge to notch */
  int bincount[HISTO_BINS];
  int n;

  for ( n = 0 ; n < HISTO_BINS ; n++ ) bincount[n] = 0;


  /* main loop over all edges */
  FOR_ALL_EDGES(e_id)
    {
      facetedge_id fe_a; /* for edge under test */
      facetedge_id fe_ai; /* other facetedge of e_id */
      facetedge_id fe_b; /* other side of one triangle */
      facetedge_id fe_d; /* other side of other triangle */
      REAL a[MAXCOORD],b[MAXCOORD],d[MAXCOORD];  /* edge vectors */
      REAL normb[MAXCOORD];         /* normal to one facet */
      REAL normd[MAXCOORD];         /* normal to other facet */
      REAL denom;           /* denominator of cosine formula */
      REAL angle;

      if ( get_eattr(e_id) & (FIXED | BOUNDARY | CONSTRAINT) ) continue;

      /* test to be sure edge has exactly two adjacent facets */
      fe_a = get_edge_fe(e_id);
      if ( !equal_id(get_next_facet(fe_a),get_prev_facet(fe_a)) ) continue;
      if ( equal_id(get_next_facet(fe_a),fe_a) ) continue;

      /* test parallelism */
      get_fe_side(fe_a,a);
      fe_b = get_next_edge(fe_a);
      get_fe_side(fe_b,b);
      fe_ai = fe_inverse(get_next_facet(fe_a));
      fe_d = get_next_edge(fe_ai); 
      get_fe_side(fe_d,d);
      cross_prod(a,b,normb);
      cross_prod(a,d,normd);
      denom = sqrt(dot(normb,normb,web.sdim)*dot(normd,normd,web.sdim));
      angle = acos( -dot(normb,normd,web.sdim)/denom );
      if ( angle == 0.0 ) n = 0;
      else
        n = HISTO_BINS/2 + 1 + (int)floor(log(angle)*HISTO_BINSIZE);
      if ( n < 0 ) n = 0;
      if ( n >= HISTO_BINS ) n = HISTO_BINS - 1;
      bincount[n]++;

    }

  /* print histogram */
  outstring("       angle             number\n");
  if ( bincount[0] )
    {
      sprintf(msg,"%f - %f     %6d \n",0.0,exp((-HISTO_BINS/2)/HISTO_BINSIZE),
                                                  bincount[0]);
      outstring(msg);
    }
  for ( n = 1 ; n < HISTO_BINS ; n++ )
   if ( bincount[n] )
    {
      sprintf(msg,"%f - %f     %6d\n",exp((n-HISTO_BINS/2-1)/HISTO_BINSIZE),
              exp((n-HISTO_BINS/2)/HISTO_BINSIZE),bincount[n]);
      outstring(msg);
    }

}


#ifdef UNDER_DEVELOPMENT
/*******************************************************************
*
*  function: merge_collapsed_facets()
*
*  purpose: Merge facets that have the same three vertices.
*
*/

void merge_collapsed_facets()
{
  facetedge_id fe,next_fe;
  vertex_id v1,v2;

  fe = NULLFACETEDGE;
  while ( generate_all(FACETEDGE,&fe) )
    { next_fe = get_next_facet(fe);
      if ( equal_id(fe,next_fe) ) continue;  /* only one facet on edge */
      v1 = get_fe_headv(get_next_edge(fe));
      v2 = get_fe_headv(get_next_edge(next_fe));
      if ( equal_id(v1,v2) )
        { /* merge */
          
        }
    }
}
#endif

