/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: psgraph.c
*
*  Purpose:  PostScript output.
*/

#include "include.h"

static FILE *fd;
static int gridflag;

/*****************************************************************
*
*  Function: ps_init()
*
*  Purpose:  Get file name from user, write file header.
*/

void ps_init()
{
  char filename[100];

  if ( web.dimension >= SOAPFILM )
    {
      prompt("Show grid lines? ",filename);
      gridflag = (toupper(filename[0]) == 'Y');
    }
  else gridflag = 0;
  prompt("Enter file name(.ps will be added): ",filename);
  strcat(filename,".ps");

  fd = fopen(filename,"w");
  if ( fd == NULL )
    { sprintf(errmsg,"Cannot open %s.\n",filename);
      error(errmsg,RECOVERABLE);
    }

  fputs("%!PS-Adobe-1.0\n",fd);
  fputs("144 144 scale\n",fd);
  fputs("2.1 3 translate\n",fd);
  fputs("/edgewidth 0.004 def\n",fd);
  fputs("/fixwidth 0.004 def\n",fd);
  fputs("/triplewidth 0.002 def\n",fd);
  fputs("/wallwidth 0.004 def\n",fd);
  fputs("/barewidth 0.005 def\n",fd);
  fputs("/gridwidth 0 def\n",fd);
}

/************************************************************
*
*  Function: ps_edge()
*
*  Purpose: Graphs one edge, already transformed.
*/

void ps_edge(t)
struct tsort *t;
{
  fputs("newpath \n",fd);
  fputs("edgewidth setlinewidth\n",fd);
  fputs("1 setlinecap\n",fd);
  fprintf(fd,"  %f %f  moveto\n",t->x[0][0],t->x[0][1]);
  fprintf(fd,"  %f %f  lineto\n",t->x[1][0],t->x[1][1]);
  fprintf(fd,"  stroke\n");
}


/************************************************************
*
*  Function: ps_facet()
*
*  Purpose: Graphs one facet, already transformed.
*/

void ps_facet(t)
struct tsort *t;
{ edge_id e_id;
  facetedge_id fe_id;
  int i;

  /* paint facet */
/*  if ( web.hide_flag && valid_id(t->f_id)
	 && (get_facet_color(t->f_id) != CLEAR) )
	 */
if ( web.hide_flag )
    { double cosine,gray;
      double denom;
      denom = sqrt(dotf(t->normal,t->normal,3));
      if ( denom == 0.0 ) return;
      fputs("newpath \n",fd);
      fprintf(fd,"  %f %f  moveto\n",t->x[0][0],t->x[0][1]);
      fprintf(fd,"  %f %f  lineto\n",t->x[1][0],t->x[1][1]);
      fprintf(fd,"  %f %f  lineto\n",t->x[2][0],t->x[2][1]);
      fprintf(fd,"  closepath\n");
      cosine = t->normal[1]/denom;
      if ( t->normal[2] < (float)0.0 ) cosine = -cosine;
      gray = .5 + .4*cosine;
      fprintf(fd,"%f setgray\n",gray);
      fprintf(fd,"  fill\n");
    }

  /* see what types edges are and plot */
  if ( valid_id(t->f_id) ) 
    { fe_id = get_facet_fe(t->f_id);
      for ( i = 0 ; i < 3 ; i++, fe_id = get_next_edge(fe_id) )
        { 
          e_id = get_fe_edge(fe_id);
/* don't want this to plot interior edges
          if ( get_eattr(e_id) & (FIXED|BOUNDARY) ) 
            fputs("fixwidth setlinewidth\n",fd);
          else
*/
	  if ( equal_id(get_next_facet(fe_id),fe_id) )
    	  /* edge of some sort */
            fputs("edgewidth setlinewidth\n",fd);
          else if ( !equal_id(get_next_facet(fe_id),get_prev_facet(fe_id)) )
	      /* triple line at least */
            fputs("triplewidth setlinewidth\n",fd);
          else if ( get_eattr(e_id) & HIT_WALL ) 
            fputs("wallwidth setlinewidth\n",fd);
          else if ( gridflag ) /* ordinary internal grid line */
            fputs("gridwidth setlinewidth\n",fd);
          else continue;  
          fputs("newpath \n",fd);
          fprintf(fd,"  %f %f  moveto",t->x[i][0],t->x[i][1]);
          fprintf(fd,"  %f %f  lineto\n",t->x[(i+1)%3][0],t->x[(i+1)%3][1]);
          fprintf(fd,"  closepath 0 setgray stroke\n");
        }
    }
  else  /* bare edges */
    {
      fputs("barewidth setlinewidth\n",fd);
      fputs("newpath \n",fd);
      fprintf(fd,"  %f %f  moveto",t->x[0][0],t->x[0][1]);
      fprintf(fd,"  %f %f  lineto\n",t->x[1][0],t->x[1][1]);
      fprintf(fd,"  %f %f  lineto\n",t->x[2][0],t->x[2][1]);
      fprintf(fd,"  closepath 0 setgray stroke %%  edge\n");
    }
}

/*************************************************************
*
*  Function:  ps_finish()
*
*  Purpose:   End PostScript output.
*/

void ps_finish()
{
  fputs("showpage\n\n\n",fd);
  fclose(fd);
}
