/*
**	BBRIFFS -- Riff generator for the Song of the Grid (BallBlazer)
**	This version to generate 1, 2, & 3 voices with separate channels
**	and riffs at strange tempos.
**	Also allows infinite number of bars specification... (3/88)
**	Also allows random choice of time value... (3/88)
**	Also allows choice of scale at execution time (2/89)
*/
#include	<stdio.h>
#include	"midi.h"
#undef	loop

#define	PERIOD	MPU_CLOCK_PERIOD

extern	char	Majriffs[];		/* in riffs.c */
extern	int	Nmajriffs;		/* in riffs.c */
extern	char	Minriffs[];		/* in riffs.c */
extern	int	Nminriffs;		/* in riffs.c */
char	*Riffs	= Minriffs;		/* points to Majriffs or Minriffs */
int	Numriffs;			/* set to Nmajriffs or Nminriffs */

#define	NPV		8	/* notes per voice (in a riff) */
#define	VPR		3	/* voices per riff */
#define	NPR		(NPV*VPR)/* notes per riff (counting all voices) */
#define	FIRST		12	/* index in Riffs[] of first to play */

#define	NUMTRIES	3	/* how many attempts to match dist */

#define	MAXCHAN		16	/* number of channels possible */

#define	DEFT	16			/* default tempo */
#define	DEFB	4			/* default number of bars */
#define	DEFV	80			/* default key velocity */

#define	KEYON(T,V,K)	cp = key(1,T,V,K,cp)
#define	KEYOFF(T,V,K)	cp = key(0,T,V,K,cp)

#define	NDURS	5
int	Durs[NDURS]	= {	/* choices for randomly chosen durations */
	20, 60, 30, 40, 45,
};
int	Biv[]	= {		/* beat importance values */
	96, 0, 12, 0, 48, 0, 24, 6,
};
int	Trace	= 0;
int	Fflg	= 1;		/* always make FIRST first */
int	Rflg	= 0;		/* if != 0, add some rests */
int	Last;			/* last note played in voice 0 */
int	Notedur	= 480/DEFT;	/* how long each note lasts */
int	Voice[VPR];		/* != 0 when voice being used */
int	Vel[VPR][MAXCHAN];	/* voice/channel map */
int	Shift	= 0;		/* used to offset times */
int	Lt;			/* last time output */

char	*key(), *dtime();

main(argc, argv)
char	*argv[];
{
	register int i, r, cn;
	char buf[64], *cp;
	int numbars, loop, randur, pause;
	int numn, numr, resid, rn, energy, effort;
	long now;

	numbars = DEFB;
	Numriffs = Nminriffs;
	cn = loop = randur = pause = 0;
	while (--argc > 0) {
	    if (argv[argc][0] == '-') {
		switch (argv[argc][1]) {
		case 'c':		/* voice channel */
		    cp = &argv[argc][2];
		    i = atoi(cp) - 1;		/* channel */
		    if (i < 0 || 15 < i)
			goto syntax;
		    for (; *cp && *cp != '='; cp++);
		    if (*cp++ != '=')
			goto syntax;
		    r = atoi(cp) - 1;		/* voice */
		    if (r < 0 || 2 < r)
			goto syntax;
		    for (; *cp && *cp != ','; cp++);
		    if (*cp++ == ',')
			Vel[r][i] = atoi(cp);
		    else
			Vel[r][i] = DEFV;
		    Voice[r] = 1;
		    cn++;
		    break;
		case 'f':		/* no particular one first */
		    Fflg = 0;
		    break;
		case 'l':		/* infinite play (numbars is -1) */
		    loop++;
		    pause = atoi(&argv[argc][2]);
		    break;
		case 'M':		/* C major riffs */
		    Riffs = Majriffs;
		    break;
		case 'r':		/* add rests */
		    Rflg++;
		    break;
		case 'T':		/* trace */
		    Trace++;
		    break;
		case 't':		/* tempo */
		    if (argv[argc][2] == '\0') {
			randur = 1;
			break;
		    }
		    Notedur = 480 / atoi(&argv[argc][2]);
		    for (cp = &argv[argc][2]; '0' <= *cp && *cp <= '9'; cp++);
		    for (resid = Notedur; *cp++ == '.'; resid >>= 1)
			Notedur += resid;
		    break;
		default:
		    goto syntax;
		}
	    } else if (numbars == DEFB)
		numbars = atoi(argv[argc]);
	    else {
syntax:
		fprintf(stderr,
		 "Usage: %s [bars] [-c#=#[,#]] [-f] [-l] [-M] [-r] [-t[#]]\n",
		 argv[0]);
		fprintf(stderr, "-c associates voices with channels\n");
		fprintf(stderr, "   -c1=3,8 puts voice 3 on chan 1 quietly\n");
		fprintf(stderr, "   -c16=1 puts voice 1 on chan 16\n");
		fprintf(stderr, "   there many be several -c arguments.\n");
		fprintf(stderr, "-f defeats automatic choice of first riff\n");
		fprintf(stderr, "-l causes the program to loop forever\n");
		fprintf(stderr, "-l# adds a #-clock-long pause every loop\n");
		fprintf(stderr, "-M uses C major riffs instead of A minor\n");
		fprintf(stderr, "-r adds rests (randomly in the middle)\n");
		fprintf(stderr, "-t# sets time value, # must be >= %d\n", NPV);
		fprintf(stderr, "   1=whole, 2=half, 4.=dotted quarter\n");
		fprintf(stderr, "-t chooses time value randomly\n");
		fprintf(stderr, "defaults are %d -t%d -c1=1,%d\n",
		 DEFB, DEFT, DEFV);
		exit(2);
	    }
	}
	if (cn == 0)
	    Vel[0][0] = DEFV;		/* default is voice 1 on chan 1 */
	time(&now);
	srand((int) now);
	do {
	    if (randur)
		Notedur = Durs[(rand() % NDURS + rand() % NDURS) >> 1];
	    numn = (numbars * 480) / Notedur;
	    resid = numbars * 480 - numn * Notedur;
	    numr = (numn + NPV - 1) / NPV;
	    if (numn <= 0) {
		fprintf(stderr, "Too slow, can't play a note!\n");
		exit(1);
	    }
	    effort = (Notedur < 30? 130 - 4 * Notedur : 0);
	    for (r = 0; r < numr; r++) {
		rn = (r == 0 && Fflg)? FIRST : pickriff();
		if (Rflg == 0)		/* no added rests */
		    energy = 256;
		else if (r < numr / 2)	/* first half, increasing rests */
		    energy = 300 - 20 * Rflg - (200 * r) / numr - effort;
		else			/* second half, decreasing rests */
		    energy = 100 - 20 * Rflg + (200 * r) / numr - effort;
		play(rn, numn > NPV? NPV : numn, energy);
		numn -= NPV;
	    }
	    if (resid) {
		buf[0] = resid;
		buf[1] = 0xF9;
		write(1, buf, 2);
	    }
	    if (pause) {
		Lt = 0;
		cp = dtime(pause, buf, 1);
		*cp++ = RT_TCWME;
		write(1, buf, cp - buf);
	    }
	    if (Trace)
		fprintf(stderr, "\n");
	} while (loop);

}

pickriff()
{
	register int i, j, best, riff, bestr;

if (Trace > 1) fprintf(stderr, "\nLast=%d ", Last);
	best = 999;
	for (i = NUMTRIES; --i >= 0; ) {
	    riff = (rand() % 32767) % Numriffs;
	    if (Last == 0)
		return(riff);
	    j = Last - Riffs[riff * NPR];
if (Trace > 1) fprintf(stderr, "  r:%d j:%d ", riff, j);
	    if (j < 0)
		j = -j;
	    else if (j == 0)
		j = 6;
	    if (j < best) {
		bestr = riff;
		best = j;
	    }
	}
if (Trace > 1) fprintf(stderr, "  %d chosen\n", bestr);
	return(bestr);
}

play(riff, numnotes, energy)
{
	register char *cp;
	register int i, note, v, q;
	char buf[1024];
	int nip[VPR];

	if (Trace)
	    fprintf(stderr, " %d", riff);
	cp = buf;
	Lt = 0;
	nip[0] = nip[1] = nip[2] = 0;
	for (i = 0; i < numnotes; i++) {
	    q = (energy + Biv[i]) > (rand() & 0xFF);
	    for (v = 0; v < VPR; v++) {
		if (Voice[v] == -1)
		    continue;
	        note = Riffs[riff * NPR + v * NPV + i];
		if (note && q) {			/* new note */
		    if (nip[v]) {			/* previous note-off */
			KEYOFF(i, v, nip[v]);
			nip[v] = 0;
		    }
		    KEYON(i, v, note);
		    nip[v] = note;
		}
	    }
	    if (cp - buf > (sizeof buf >> 1)) {
		write(1, buf, cp - buf);
		cp = buf;
	    }
	}
	Last = nip[0];
	Shift = -1;				/* skimp by 1 for note offs */
	for (v = 0; v < VPR; v++) {
	    if (nip[v]) {
		KEYOFF(numnotes, v, nip[v]);
		nip[v] = 0;
	    }
	}
	Shift++;		/* in case we skimped by 1 on note-offs */
	cp = dtime(numnotes, cp, Notedur);
	*cp++ = RT_TCWME;
	write(1, buf, cp - buf);
}

char	*
key(onoff, when, voice, key, cp)
register char *cp;
{
	register int chan, dt;

/****/if (key == 0) fprintf(stderr, "key(%d, %d, %d, %d, 0x%x);\n", onoff, when, voice, key, cp);
	for (chan = 0; chan < MAXCHAN; chan++) {
	    if (Vel[voice][chan]) {
		cp = dtime(when, cp, Notedur);
		*cp++ = CH_KEY_ON | chan;
		*cp++ = key;
		*cp++ = onoff? Vel[voice][chan] : 0;
	    }
	}
	return(cp);
}

char	*
dtime(when, cp, notedur)
char	*cp;
{
	int dt;

	if (Lt > when)
	    dt = 0;
	else {
	    dt = (when - Lt) * notedur;
	    if (Shift && dt + Shift >= 0) {
		dt += Shift;
		Shift = 0;
	    }
	    while (dt >= PERIOD) {
		*cp++ = RT_TCIP;
		dt -= PERIOD;
	    }
	    Lt = when;
	}
	*cp++ = dt;
	return(cp);
}
