/*
 * This file implements POSIX.1e audit for Linux.
 *
 * This file is based on the PSSG Draft 15 (Dec 1995)
 * pp. 85-164 and pp. 286-312
 *
 * Copyright (c) 1997 Andrew G Morgan <morgan@parc.power.net>
 */

#ifndef _SYS_AUDIT_H
#define _SYS_AUDIT_H

#include <unistd.h>

/* misc p109 */

#define AUD_STD_96_1		0xAD5D199601         /* p296 par 3 */
#define AUD_SYSTEM_LOG		-1	/* file descriptor for system
					   audit log - cannot be 'opened'. */
#define AUD_NATIVE		0	/* default format of records */

/* macros (types of numeric argument) */

#define AUD_AET_AUD_SWITCH	1	/* aud_switch() */
#define AUD_AET_AUD_WRITE	2	/* aud_write() */
#define AUD_AET_CHDIR		3	/* chdir() */
#define AUD_AET_CHMOD		4	/* chmod() */
#define AUD_AET_CHOWN		5	/* chown() */
#define AUD_AET_CREAT		6	/* creat() */
#define AUD_AET_DUP		7	/* dup(), dup2(), fcntl(F_DUPFD) */
#define AUD_AET_EXEC		8	/* exec(), execl(), execlp(), execv(),
					execvp(), execle(), execve() */
#define AUD_AET_EXIT		9	/* _exit() */
#define AUD_AET_FORK		10	/* fork() */
#define AUD_AET_KILL		11	/* kill() */
#define AUD_AET_LINK		12	/* link() */
#define AUD_AET_MKDIR		13	/* mkdir() */
#define AUD_AET_MKFIFO		14	/* mkfifo() */
#define AUD_AET_OPEN		15	/* open, opendir() */
#define AUD_AET_PIPE		16	/* pipe() */
#define AUD_AET_RENAME		17	/* rename() */
#define AUD_AET_RMDIR		18	/* rmdir() */
#define AUD_AET_SETGID		19	/* setgid() */
#define AUD_AET_SETUID		20	/* setuid() */
#define AUD_AET_UNLINK		21	/* unlink() */
#define AUD_AET_UTIME		22	/* utime() */

/*
 * conditional audit event types
 */

#if defined(_POSIX_ACL)
# define AUD_AET_ACL_DELETE_DEF_FILE	50	/* acl_delete_def_file() */
# define AUD_AET_ACL_SET_FD             51	/* acl_set_fd() */
# define AUD_AET_ACL_SET_FILE           52	/* acl_set_file() */
#endif

#if defined(_POSIX_CAP)
# define AUD_AET_CAP_SET_FD		60	/* cap_set_fd */
# define AUD_AET_CAP_SET_FILE		61	/* cap_set_file */
# define AUD_AET_CAP_SET_PROC		62	/* cap_set_proc */
#endif

#if defined(_POSIX_INF)
# define AUD_AET_INF_SET_FD		70	/* inf_set_fd */
# define AUD_AET_INF_SET_FILE		71	/* inf_set_file */
# define AUD_AET_INF_SET_PROC		72	/* inf_set_proc */
#endif

#if defined(_POSIX_MAC)
# define AUD_AET_MAC_SET_FD		80	/* mac_set_fd */
# define AUD_AET_MAC_SET_FILE		81	/* mac_set_file */
# define AUD_AET_MAC_SET_PROC		82	/* mac_set_proc */
#endif

/* XXX - need more for networking */
/* XXX - need more for kernel activity? */

/* Structures/typedef's (NE=non-exportable=OS-dependent) */

/* event specific data within a specific record (NE). this type is
   opaque and represents a descriptor for the event specific data in an
   audit record - defined in libaudit */

typedef const struct __aud_evinfo_s *aud_evinfo_t;

/* header of an audit record (NE). this is opaque and defined in libaudit */

typedef const struct __aud_hdr_s *aud_hdr_t;

/* identifier of responsible individual */

typedef __u32 aud_id_t;

/* item_id's - these are used in aud_*_info() functions */

/* position values */

#define AUD_NEXT_ITEM           -2      /* p148 */
#define AUD_FIRST_ITEM          -1      /* first item in of *_info (p134) */
#define AUD_LAST_ITEM		0	/* position value argument for
					   appending - see aud_put_*_info() */

/* proper item_id's */

#define AUD_ACL			1 /* ACL */
#define AUD_ACL_TYPE		2 /* ACL type */
#define AUD_ATIME		3 /* access time */
#define AUD_AUDIT_STATE		4 /* Audit state */
#define AUD_CAP			5 /* capability state */
#define AUD_CMD_ARGS		6 /* command arguments (argv[0]..argv[n]) */
#define AUD_ENVP		7 /* environment variables (envp) */
#define AUD_EXIT_CODE		8 /* exit code */
#define AUD_FILE_ID		9 /* file descriptor */
#define AUD_GID			10 /* group identifier */
#define AUD_INF_LBL		11 /* information label */
#define AUD_LINKNAME		12 /* name of link-file (will
				      _point_to_ other) */
#define AUD_MAC_LBL		13 /* MAC label */
#define AUD_MODE		14 /* file access mode */
#define AUD_MTIME		15 /* modification time */
#define AUD_NEW_PATHNAME	16 /* new pathname (used by rename) */
#define AUD_OLD_PATHNAME	17 /* old pathname (used by rename) */
#define AUD_OFLAG		18 /* Open flag */
#define AUD_PID			19 /* process identifier */
#define AUD_PATHNAME		20 /* file(path)name */
#define AUD_RETURN_ID		21 /* return value (may be file descriptor) */
#define AUD_RD_FILE_ID		22 /* read file descriptor */
#define AUD_SIG			23 /* signal value */
#define AUD_STATUS		24 /* audit status of event */
#define AUD_UID			25 /* owner's identifier */
#define AUD_WR_FILE_ID		26 /* write file descriptor */
#define AUD_EVENT_TYPE_ID       27 /* the event type see next table */
#define AUD_STATUS_ID           28 /* the audit status? (aud_status_t) */

/* p111: member flavor types (in aud_info_t below) */

#define AUD_TYPE_ACL	       1	/* acl_t */
#define AUD_TYPE_ACL_TYPE      2	/* acl_type_t */
#define AUD_TYPE_AUD_ID	       3	/* aud_id_t */
#define AUD_TYPE_AUD_OBJ_TYPE  4	/* aud_obj_type_t */
#define AUD_TYPE_AUD_STATE     5	/* aud_state_t */
#define AUD_TYPE_AUD_STATUS    6	/* aud_status_t */
#define AUD_TYPE_AUD_TIME      7	/* aud_time_t */
#define AUD_TYPE_CAP	       8	/* cap_t */
#define AUD_TYPE_CHAR	       9	/* char */
#define AUD_TYPE_GID	       10	/* gid_t */
#define AUD_TYPE_INF	       11	/* inf_t */
#define AUD_TYPE_INT	       12	/* int */
#define AUD_TYPE_LONG	       13	/* long */
#define AUD_TYPE_MAC	       14	/* mac_t */
#define AUD_TYPE_MODE	       15	/* mode_t */
#define AUD_TYPE_OPAQUE	       16	/* void */
#define AUD_TYPE_PID	       17	/* pid_t */
#define AUD_TYPE_SHORT	       18	/* short */
#define AUD_TYPE_STRING	       19	/* char (nul termintated) */
#define AUD_TYPE_STRING_ARRAY  20	/* char * (array of null..) */
#define AUD_TYPE_TIME	       21	/* time_t */
#define AUD_TYPE_UID	       22	/* uid_t */

/* p110: Misc. info within record (NE).  This is _non_opaque_: */
typedef struct {
    short aud_info_type;	/* type of member data stored (see above) */
    ssize_t aud_info_length;	/* precise length of data stored */
    void *aud_info_p;		/* pointer to zero [or more] data items of
				   indicated type (in the case of void
				   the assumed type is __u8) */
} aud_info_t;

/* "object attribute set" (NE) - this type is opaque and defined in
   libaudit */

typedef const struct __aud_obj_s *aud_obj_t;

/* p112: possible objects - see aud_rec_t below (XXX - need network
   objects too.) */

#define AUD_OBJ_BLOCK_DEV	1	/* block device */
#define AUD_OBJ_CHAR_DEV	2	/* char device */
#define AUD_OBJ_DIR		3	/* directory */
#define AUD_OBJ_FIFO		4	/* FIFO object (pipes..) */
#define AUD_OBJ_FILE		5	/* regular file */
#define AUD_OBJ_PROC		6	/* process */

/* type identifying object (see above) */

typedef short aud_obj_type_t;

/* pointer to "opaque" audit record (NE): defined in libaudit */

typedef const struct __aud_rec_s *aud_rec_t;

/* p112: possible audit states (see aud_state_t and aud_switch()) */

#define AUD_STATE_ON		1	/* turn auditing on */
#define AUD_STATE_OFF		0	/* turn auditing off */
#define AUD_STATE_QUERY		2	/* find out current state */

/* controls whether system is requested to not audit a process */

typedef short aud_state_t;

/* p113: possible audit statuses - why did an event succeed/fail? 
	 local(Linux) convention SUCCESS(>= 0) FAIL(<0) */

#define AUD_SUCCESS		0	/* event completed successfully */
#define AUD_FAIL_DAC		-1	/* failed due to DAC check */
#define AUD_FAIL_MAC		-2	/* failed due to MAC check */
#define AUD_FAIL_PRIV		-3	/* fail for lack of privilege */
#define AUD_PRIV_USED		3	/* succeeded but used privilege */
#define AUD_FAIL_OTHER		-4	/* "miscellaneous.." */

/* success failure status of audit event */

typedef short aud_status_t;

/* subject attribute set of an audit record (NE) - opaque but defined
   within libaudit */

typedef const struct __aud_subj_s *aud_subj_t;

/* time an event was audited - to conform to standard, we require at least
 * 1/50 second resolution. */

typedef struct {
    time_t sec;	/* whole seconds */
    long nsec;  /* whole nanoseconds (1E-9 s) */
} aud_time_t;

/* **************************************************************** */

/*
 * Function prototypes  (actual functions provided in 'libaudit')
 */

/* RECORD related prototypes */

aud_rec_t aud_init_record(void);
aud_rec_t aud_dup_record(aud_rec_t original_d);
int aud_valid(aud_rec_t ar);

/* EVENT related functions */

int aud_delete_event(aud_evinfo_t aud_event_d);
int aud_delete_event_info(aud_evinfo_t aud_event_d, int item_id);
int aud_get_event(aud_rec_t ar, int index, aud_evinfo_t *aud_event_p);
int aud_get_event_info(aud_evinfo_t aud_event_d, int item_id,
		       const aud_info_t **aud_event_info_p);
int aud_put_event(aud_rec_t ar, const aud_evinfo_t *next_p,
		  aud_evinfo_t *new_p);
int aud_put_event_info(aud_evinfo_t aud_evinfo_d, int position, int item_id,
		       const aud_info_t *aud_event_info_p);

/* HEADER related functions */

int aud_delete_hdr(aud_hdr_t aud_hdr_d);
int aud_delete_hdr_info(aud_hdr_t aud_hdr_d, int item_id);
int aud_get_hdr(aud_rec_t ar, int index, aud_hdr_t *aud_hdr_p);
int aud_get_hdr_info(aud_hdr_t aud_hdr_d, int item_id,
		     const aud_info_t **aud_hdr_info_p);
int aud_put_hdr(aud_rec_t ar, const aud_hdr_t *next_p, aud_hdr_t *new_p);
int aud_put_hdr_info(aud_hdr_t aud_hdr_d, int position, int item_id,
		     const aud_info_t *aud_hdr_info_p);

/* OBJECT related functions */

int aud_delete_obj(aud_obj_t aud_obj_d);
int aud_delete_obj_info(aud_obj_t aud_obj_d, int item_id);
int aud_get_obj(aud_rec_t ar, int index, aud_obj_t *aud_obj_p);
int aud_get_obj_info(aud_obj_t aud_obj_d, int item_id,
		     const aud_info_t **aud_obj_info_p);
int aud_put_obj(aud_rec_t ar, const aud_obj_t *next_p, aud_obj_t *new_p);
int aud_put_obj_info(aud_obj_t aud_obj_d, int position, int item_id,
		     const aud_info_t *aud_obj_info_p);

/* SUBJECT related functions */

int aud_delete_subj(aud_subj_t aud_subj_d);
int aud_delete_subj_info(aud_subj_t aud_subj_d, int item_id);
int aud_get_subj(aud_rec_t ar, int index, aud_subj_t *aud_subj_p);
int aud_get_subj_info(aud_subj_t aud_subj_d, int item_id,
		      const aud_info_t **aud_subj_info_p);
int aud_put_subj(aud_rec_t ar, const aud_subj_t *next_p, aud_subj_t *new_p);
int aud_put_subj_info(aud_subj_t aud_subj_d, int position, int item_id,
		      const aud_info_t *aud_subj_info_p);

/* SYSTEM calls */

aud_id_t aud_get_id(pid_t pid);
aud_id_t aud_set_id(pid_t pid);  /* NON-POSIX! Set audit id of process */
int aud_write(int audit_fd, aud_rec_t ar);
aud_state_t aud_switch(aud_state_t aud_state);
aud_rec_t aud_read(int audit_fd);

/* RECORD REPRESENTATION calls */

/* text representaions */

const char *aud_rec_to_text(aud_rec_t ar, ssize_t *len_p);
const char *aud_evid_to_text(int event_type, ssize_t *aud_info_length);
int aud_evid_from_text(const char *text);
const char *aud_id_to_text(aud_id_t audit_id, ssize_t *len_p);
aud_id_t aud_id_from_text(const char *name);

/* INTERNAL vs. EXTERNAL representations */

const int *aud_get_all_evid(void);
ssize_t aud_copy_ext(void *aud_rec_ext_p, aud_rec_t aud_rec_int, ssize_t size);
aud_rec_t aud_copy_int(const void *aud_rec_ext_p);
ssize_t aud_size(aud_rec_t ar);

/* MEMORY management */

int aud_free(const void *obj_p);

#endif

