
##############################################################################
#
# Copyright (c) 2003-2018 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2018 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
test for util operations for overloaded operations without tagged data

:remark: use see `test_util`
:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement 
:var __url__: url entry point on documentation 
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
from esys.escript import *
from test_util_base  import Test_util_base

class Test_util_overloaded_binary_no_tagged_data(Test_util_base):
   """
   test for the overloaded operation +,*,-,** without tagged data
   """
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_float_rank0(self):
      arg0=Data(1.47096003563,self.functionspace)
      arg1=1.93723512751
      res=arg0+arg1
      ref=Data(3.40819516314,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_array_rank0(self):
      arg0=Data(3.55709506753,self.functionspace)
      arg1=numpy.array(-1.02066716344)
      res=arg0+arg1
      ref=Data(2.53642790408,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_array_rank1(self):
      arg0=Data(3.90207149481,self.functionspace)
      arg1=numpy.array([4.4657818195431265, 3.32119617284539])
      res=arg0+arg1
      ref=Data(numpy.array([8.3678533143528444, 7.223267667655108]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_array_rank2(self):
      arg0=Data(-4.68442009374,self.functionspace)
      arg1=numpy.array([[0.74091492726922503, -1.2315080259370914, 2.2164556519528764, -0.91669294301144344, 
-3.4400314935339358], [-3.8524492819121772, -1.1591829602127337, -0.53289141633881343, 3.3546171819556196, 
0.85900205093947601], [-2.3179538915218068, -1.1584770451135307, 3.1239046969883812, 3.6090769299869336, -0.46475232270720745], 
[-2.6806636898905376, -4.4489053479159066, -4.0827675375972996, -2.5414777260259811, -4.3148597795733092]])
      res=arg0+arg1
      ref=Data(numpy.array([[-3.9435051664670882, -5.9159281196734046, -2.4679644417834368, -5.6011130367477566, 
-8.124451587270249], [-8.5368693756484895, -5.8436030539490469, -5.2173115100751266, -1.3298029117806935, -3.8254180427968372], 
[-7.0023739852581199, -5.8428971388498443, -1.5605153967479319, -1.0753431637493795, -5.1491724164435206], 
[-7.3650837836268508, -9.1333254416522198, -8.7671876313336128, -7.2258978197622943, -8.9992798733096215]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_array_rank3(self):
      arg0=Data(2.86593608481,self.functionspace)
      arg1=numpy.array([[[4.2067166346735707, -3.4609658779855259], [-3.6295755931428006, -1.6145496628914024]], 
[[-0.30689360722551307, -3.5655741472014202], [4.6344461667856578, -0.97647709177833253]], [[4.7165703625301258, 
-3.3102895783063202], [-1.2504979838581298, 0.50152744957891482]], [[-2.1394588300623232, 4.9923148177575083], 
[-0.80242407585098352, 2.6762541229248686]], [[3.9122442911586237, -3.0792469908730791], [-4.8790034395863069, 
2.1324974233290197]], [[-3.9291649380081859, -4.3912591131059973], [-3.9474124521037979, -2.6386161030917634]]])
      res=arg0+arg1
      ref=Data(numpy.array([[[7.0726527194793647, -0.59502979317973193], [-0.76363950833700667, 1.2513864219143915]], 
[[2.5590424775802809, -0.6996380623956262], [7.5003822515914518, 1.8894589930274615]], [[7.5825064473359198, 
-0.44435349350052622], [1.6154381009476642, 3.3674635343847088]], [[0.7264772547434708, 7.8582509025633023], 
[2.0635120089548105, 5.5421902077306626]], [[6.7781803759644177, -0.21331090606728509], [-2.0130673547805129, 
4.9984335081348137]], [[-1.0632288532023919, -1.5253230283002033], [-1.0814763672980039, 
0.22731998171403056]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_array_rank4(self):
      arg0=Data(-3.85755594628,self.functionspace)
      arg1=numpy.array([[[[0.88276063932111803, -2.7408746573326259, -3.5567283218012524, -2.448299837516986], 
[-0.4875571118716806, -3.1147667323656147, -2.480712655160823, -2.911902697324813], [-1.1333278403602831, 1.7312558164203633, 
-0.11864967921777936, 0.14011362164844421]], [[-1.7789861927781425, 2.0572394075179385, 4.6380214601010099, 
-1.5634940719366677], [-3.208971551632346, -3.2880620561642546, 2.8971312238355518, 3.7798211760847735], [-3.4074898662220932, 
3.7819019835604841, 2.7140235050721673, -3.3185012870474315]]], [[[4.4998810797740827, -0.41883112780607856, 
0.59900334755132079, -4.5665886301371685], [3.1467582175434607, -4.0206699299317696, -0.88560207445334793, 
-1.9408859182025715], [-3.6137345592479719, -3.5643310585296692, -4.439536822899246, -0.68544441859978278]], 
[[-4.8485595511269501, 1.8256731060584155, -1.5750604251355158, 0.51320244300914997], [4.8877398127831668, 1.5023482512746442, 
-1.5699594680618398, 2.2967849912948051], [-1.2698656553328513, 2.8651428586056724, 1.2609807349009419, -2.131560546883926]]], 
[[[-3.0641844430003538, 4.4111756598317662, 4.8130825708663956, 1.9495656905534116], [1.9870942111554726, -1.1118481945778589, 
-1.5984361257323121, 4.8792709427297467], [-1.6815745512393212, 2.9840496331913275, -1.2525173072129316, 2.1654649120089147]], 
[[4.7337623788305088, 4.6572512028315369, -2.0049629024389679, 4.1271327196890013], [-3.8854431680204451, -3.3278266940992127, 
2.6595014097696641, -4.0976536048761059], [-0.094491255148859921, -4.8167572108255952, -3.0751101857814733, 
-3.1471801490818638]]]])
      res=arg0+arg1
      ref=Data(numpy.array([[[[-2.9747953069573354, -6.5984306036110798, -7.4142842680797063, -6.3058557837954394], 
[-4.345113058150134, -6.9723226786440682, -6.3382686014392764, -6.7694586436032669], [-4.9908837866387366, -2.1263001298580901, 
-3.9762056254962328, -3.7174423246300092]], [[-5.636542139056596, -1.8003165387605149, 0.78046551382255647, 
-5.4210500182151211], [-7.0665274979107995, -7.1456180024427081, -0.96042472244290167, -0.077734770193679914], 
[-7.2650458125005466, -0.075653962717969314, -1.1435324412062862, -7.176057233325885]]], [[[0.64232513349562925, 
-4.276387074084532, -3.2585525987271327, -8.4241445764156211], [-0.7107977287349927, -7.8782258762102231, -4.7431580207318014, 
-5.7984418644810249], [-7.4712905055264258, -7.4218870048081227, -8.2970927691776986, -4.5430003648782362]], 
[[-8.7061154974054027, -2.031882840220038, -5.4326163714139692, -3.3443535032693035], [1.0301838665047134, -2.3552076950038092, 
-5.4275154143402933, -1.5607709549836484], [-5.1274216016113048, -0.99241308767278102, -2.5965752113775116, 
-5.9891164931623795]]], [[[-6.9217403892788072, 0.55361971355331274, 0.95552662458794213, -1.9079902557250419], 
[-1.8704617351229809, -4.9694041408563123, -5.4559920720107655, 1.0217149964512933], [-5.5391304975177746, -0.8735063130871259, 
-5.1100732534913851, -1.6920910342695388]], [[0.87620643255205533, 0.79969525655308349, -5.8625188487174213, 
0.26957677341054787], [-7.7429991142988985, -7.1853826403776662, -1.1980545365087893, -7.9552095511545593], 
[-3.9520472014273134, -8.6743131571040486, -6.9326661320599268, -7.0047360953603173]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_constData_rank0(self):
      arg0=Data(-3.08260537353,self.functionspace)
      arg1=Data(-2.89221025803,self.functionspace)
      res=arg0+arg1
      ref=Data(-5.97481563156,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_constData_rank1(self):
      arg0=Data(-2.16878537375,self.functionspace)
      arg1=Data(numpy.array([-4.4631173640080659, -3.455603168817527]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([-6.6319027377583328, -5.624388542567794]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_constData_rank2(self):
      arg0=Data(2.46013464027,self.functionspace)
      arg1=Data(numpy.array([[-2.8803835589152369, 1.1402030152156826, -3.6920021346728884, -4.6108380393676001, 
-3.6408034714016546], [1.2229535091156363, -2.727989045200844, 1.3854081477700486, -2.375039972557432, -4.720266104694721], 
[-2.4097576510855321, -3.8453262611311478, -3.5501882558646072, 0.91381517923812794, -2.1791752438547594], 
[-3.0497711846745656, -0.34792816936237614, 3.1817028038712323, 4.9803469176762203, -2.6079556426881014]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[-0.42024891865018432, 3.6003376554807351, -1.2318674944078358, -2.1507033991025475, 
-1.180668831136602], [3.6830881493806888, -0.26785440493579138, 3.8455427880351012, 0.08509466770762053, -2.2601314644296684], 
[0.050376989179520493, -1.3851916208660953, -1.0900536155995546, 3.3739498195031805, 0.28095939641029322], 
[-0.58963654440951307, 2.1122064709026764, 5.6418374441362849, 7.4404815579412729, -0.14782100242304885]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_constData_rank3(self):
      arg0=Data(-3.08238890389,self.functionspace)
      arg1=Data(numpy.array([[[4.6136565443570525, 4.0613526022516648], [3.5187221781315738, -0.77445406376516779]], 
[[-2.290203036153561, -1.1723045223090178], [-3.7261995704102446, -4.244005967655113]], [[0.42146429026212928, 
3.0210516216318446], [-4.3705283118525955, 2.3214483704359843]], [[0.96145621640409473, 1.581388771594046], 
[-3.3406899661089309, -4.8329582809014271]], [[-2.6388421600019707, -2.6215418372616996], [-1.0449129359840512, 
3.4100692317293557]], [[1.1230064551496888, 3.1030922826594569], [-2.6069343560195932, 
-1.7189444842472792]]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[1.5312676404671381, 0.97896369836175046], [0.43633327424165946, -3.8568429676550822]], 
[[-5.3725919400434758, -4.2546934261989318], [-6.8085884743001586, -7.3263948715450269]], [[-2.6609246136277851, 
-0.061337282258069781], [-7.4529172157425094, -0.76094053345393009]], [[-2.1209326874858196, -1.5010001322958684], 
[-6.4230788699988448, -7.9153471847913419]], [[-5.7212310638918851, -5.7039307411516145], [-4.1273018398739652, 
0.32768032783944134]], [[-1.9593824487402256, 0.020703378769542535], [-5.6893232599095072, 
-4.8013333881371931]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_constData_rank4(self):
      arg0=Data(-1.2725832394,self.functionspace)
      arg1=Data(numpy.array([[[[2.4625170194961825, 2.8298110009249564, -0.081494177646101562, -2.0378113620220262], 
[-2.2971572367730495, -1.7928559625827658, 1.6877655649303716, -1.2115878509367906], [-0.07455583129694876, 
-2.8487125560829796, -2.6333542052113499, -3.9194078321261605]], [[-0.68998401013279587, -2.647782290583903, 
3.5172020356314171, 3.4246962385577771], [2.254129010134637, -2.4922756952047385, -0.28424553113606166, -4.6807709617369184], 
[0.91963012815892942, 1.5636836096973949, 4.5963927598834822, 4.711133119793482]]], [[[2.3339558908012688, -1.3910420414037663, 
-2.4142154412956041, -3.924049981298976], [3.1956918641212955, 4.0423239734863188, -0.03422833818486648, 1.0471406302746411], 
[4.2795298023027044, -4.7306087498575451, 2.1851881497442305, -2.7122425334753988]], [[1.5866225032096271, 0.14373303300084928, 
-4.8798451436973966, 0.51159066875757997], [-0.70469527741449234, 3.6590598355952046, -4.0022476472158974, 1.1815984440047647], 
[-1.8531063602244391, 4.2305867413317966, -1.385452166221417, 4.2535126707687851]]], [[[-2.8545169963430772, 
-0.94911263671032842, -2.5360488620035424, 3.4065220725947167], [-1.9820988376444193, 2.4595710949611425, 3.120398575567414, 
4.6073445235237109], [-4.5433746690165489, 2.3719746298648428, 2.2530092552567593, -4.5975525332386127]], [[3.6866300002961392, 
-1.3815871766503243, -3.2511258989060909, -2.7171983799900348], [-2.1702028219211602, 1.3519398966343665, -2.4652876404141519, 
-0.94036289863853284], [-1.0133477540674187, 2.7449932403437982, 3.4846229725649867, 4.4470528372290499]]]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[[1.1899337800918879, 1.5572277615206618, -1.3540774170503962, -3.3103946014263208], 
[-3.5697404761773441, -3.0654392019870604, 0.41518232552607692, -2.4841710903410852], [-1.3471390707012434, 
-4.1212957954872742, -3.9059374446156445, -5.1919910715304551]], [[-1.9625672495370905, -3.9203655299881977, 
2.2446187962271225, 2.1521129991534824], [0.98154577073034233, -3.7648589346090331, -1.5568287705403563, -5.9533542011412131], 
[-0.35295311124536521, 0.29110037029310032, 3.3238095204791875, 3.4385498803891874]]], [[[1.0613726513969741, 
-2.6636252808080609, -3.6867986806998987, -5.196633220703271], [1.9231086247170008, 2.7697407340820241, -1.3068115775891611, 
-0.22544260912965353], [3.0069465628984098, -6.0031919892618397, 0.91260491033993585, -3.9848257728796934]], 
[[0.31403926380533242, -1.1288502064034454, -6.1524283831016913, -0.76099257064671466], [-1.977278516818787, 2.38647659619091, 
-5.2748308866201921, -0.090984795399529972], [-3.1256895996287337, 2.9580035019275019, -2.6580354056257116, 
2.9809294313644905]]], [[[-4.1271002357473723, -2.221695876114623, -3.808632101407837, 2.1339388331904221], 
[-3.2546820770487139, 1.1869878555568478, 1.8478153361631193, 3.3347612841194163], [-5.8159579084208435, 1.0993913904605481, 
0.98042601585246469, -5.8701357726429073]], [[2.4140467608918446, -2.6541704160546189, -4.5237091383103856, 
-3.9897816193943294], [-3.4427860613254548, 0.079356657230071903, -3.7378708798184466, -2.2129461380428275], 
[-2.2859309934717134, 1.4724100009395036, 2.212039733160692, 3.1744695978247552]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_expandedData_rank0(self):
      arg0=Data(4.04085770183,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.66323579369)+(1.-msk_arg1)*(-1.02399166403)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(2.37762190814)+(1.-msk_ref)*(3.0168660378)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_expandedData_rank1(self):
      arg0=Data(1.30655790569,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-2.1480298944514562, 3.9552819358413434])+(1.-msk_arg1)*numpy.array([-1.831123192784665, 
0.59952615048914115])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.8414719887644182, 5.2618398415283814])+(1.-msk_ref)*numpy.array([-0.52456528709762695, 
1.9060840561761792])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_expandedData_rank2(self):
      arg0=Data(-1.39878419945,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[3.6391005454608187, -3.0445657453499853, 1.0613086746042342, 2.7433661687434103, 
0.038214678549561398], [-4.6904979029020941, -3.103946648746386, -2.6519408931438804, 4.9610631947491832, -1.033000201572285], 
[4.5088987590560805, 2.6335114645935125, 3.0765600884594555, -2.5687608325162445, -1.84219426303153], [-2.8513082989188998, 
-2.7894932867591127, -4.0497598836260664, -2.3704526696472628, 
-4.12783330957748]])+(1.-msk_arg1)*numpy.array([[2.7387301489588118, -2.1280211311779462, 0.063276015231268801, 
0.32730733832031866, 0.53556675469091708], [-3.830296514114865, -4.947136618590493, -3.6493165367971736, 3.3257353111778869, 
1.0109604170682109], [0.7774202513323214, -2.0659119744762808, -0.90013912472597646, 3.1485792367098124, -3.6487235937492954], 
[2.0780004435904225, 3.6954708894027863, 0.20297736577356584, 1.5888051008674324, -0.53731309944974015]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[2.2403163460150672, -4.4433499447957363, -0.33747552484151733, 1.3445819692976588, 
-1.3605695208961901], [-6.0892821023478456, -4.5027308481921375, -4.0507250925896319, 3.5622789953034317, -2.4317844010180365], 
[3.110114559610329, 1.234727265147761, 1.677775889013704, -3.967545031961996, -3.2409784624772815], [-4.2500924983646513, 
-4.1882774862048642, -5.4485440830718179, -3.7692368690930143, 
-5.5266175090232315]])+(1.-msk_ref)*numpy.array([[1.3399459495130603, -3.5268053306236977, -1.3355081842144827, 
-1.0714768611254328, -0.86321744475483442], [-5.2290807135606165, -6.3459208180362445, -5.0481007362429251, 1.9269511117321354, 
-0.38782378237754056], [-0.6213639481134301, -3.4646961739220323, -2.298923324171728, 1.7497950372640609, -5.0475077931950469], 
[0.67921624414467097, 2.2966866899570348, -1.1958068336721857, 0.19002090142168093, -1.9360972988954916]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_expandedData_rank3(self):
      arg0=Data(1.18948229685,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-4.3960049632901939, -3.8860281576119302], [-3.0282849116046071, 1.461176317687495]], 
[[-2.7691243927207543, 1.7457161979792435], [-1.193857039539413, 3.4790351732516793]], [[-3.1223311678243815, 
4.6240444239361871], [-1.083025345227048, 4.4725275113353984]], [[1.1533941473025777, -0.78466137428150251], 
[0.90623647316947942, 4.3407149202513988]], [[0.51365913802042762, -3.6702247393702203], [3.5727929380437846, 
2.407798803889742]], [[-1.8583271856988146, 3.4437955004455141], [-0.58837365078751613, 
0.37218673760292642]]])+(1.-msk_arg1)*numpy.array([[[-0.35362859064088337, 2.6654899568068373], [3.2253110394119915, 
4.9704490095600296]], [[-4.2560817372702218, 2.1398967238277757], [1.8675656551418651, 3.4827592071145688]], 
[[-1.8210720472269113, 0.0138818456397658], [3.2154420692271461, 3.0854375859140877]], [[4.4222292768447087, 
0.79035407784716849], [-4.4364194506082058, 3.7550500479776545]], [[-0.88018912445173036, -3.931446103396059], 
[-1.8704594689743104, 4.3370841394356372]], [[-0.54569324503429772, -1.7615739731138955], [0.37505663996364902, 
-4.9234184958339942]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-3.2065226664405007, -2.696545860762237], [-1.8388026147549139, 2.6506586145371882]], 
[[-1.5796420958710611, 2.9351984948289367], [-0.0043747426897198238, 4.6685174701013725]], [[-1.9328488709746883, 
5.8135267207858803], [0.10645695162264524, 5.6620098081850916]], [[2.3428764441522709, 0.40482092256819069], 
[2.0957187700191726, 5.530197217101092]], [[1.7031414348701208, -2.4807424425205271], [4.7622752348934778, 
3.5972811007394352]], [[-0.66884488884912141, 4.6332777972952073], [0.60110864606217707, 
1.5616690344526196]]])+(1.-msk_ref)*numpy.array([[[0.83585370620880983, 3.8549722536565305], [4.4147933362616847, 
6.1599313064097228]], [[-3.0665994404205286, 3.3293790206774689], [3.0570479519915583, 4.672241503964262]], 
[[-0.6315897503772181, 1.203364142489459], [4.4049243660768393, 4.2749198827637809]], [[5.6117115736944019, 
1.9798363746968617], [-3.2469371537585126, 4.9445323448273477]], [[0.30929317239796283, -2.7419638065463658], 
[-0.68097717212461717, 5.5265664362853304]], [[0.64378905181539547, -0.57209167626420232], [1.5645389368133422, 
-3.733936198984301]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_expandedData_rank4(self):
      arg0=Data(-4.49750723771,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[2.5091822483859039, 4.9777028607992708, -2.3403200811970581, -1.9640188544346628], 
[4.2137434619762946, -4.2042220310089924, 2.3140839079480644, -1.7136883632497604], [-4.4219980396877459, -1.6200905430645021, 
4.1693218090701212, -1.6823294456436768]], [[-4.5025748868263467, 1.0567983315532716, -2.5009026496953002, -1.105235212503739], 
[-2.5646665108042264, -4.0298893768679269, 1.7474261659509525, -3.2496689533000787], [-2.0009402160179146, -1.3297253784527587, 
-4.0221300238883302, 3.1736766551244404]]], [[[-3.6051571034710648, -4.6866399142398221, -4.1793749270207936, 
3.3721956878961521], [-3.3129805464955551, 4.5971802231089605, 0.96099132115811692, -1.7127765907309431], [2.3871423211482909, 
2.032086587656277, 0.28225563624463312, -1.640280901508504]], [[1.1494037199920495, 2.8138432451588171, -3.9496485296955197, 
1.3488702698514432], [-1.5820192064113838, 1.397308590740872, 1.8777027877606756, 1.5213751386164232], [2.5894945535645411, 
-3.4932720262402608, -1.2200462747409491, -1.8608881170726423]]], [[[0.1040770040704464, 1.3410750384945169, 
3.7067724422584707, 2.7814902320709036], [-1.6041279209503423, -0.99968182938290351, 3.972553146512837, -1.0539293272340622], 
[4.7334590780659873, -3.9997940225775652, -1.7341675447997229, 1.9627188531211726]], [[0.71583275404061197, 2.1414755256544211, 
1.728882735995267, -1.8909184794723268], [-3.4164097070297572, 1.4096714343045615, 3.8903713479271254, 1.8579973853076641], 
[-3.677389227562041, -0.57230355243241604, 3.8831129988306543, 
2.9444822100635388]]]])+(1.-msk_arg1)*numpy.array([[[[2.2623155672334692, 2.6941548113820213, 2.6278688598815068, 
0.0031906247536621635], [1.7110024809857594, 0.24572110565971084, 0.35699495938663528, -2.6430152992108482], 
[4.3131876846800097, -4.1100898061232609, -2.3717802628533144, 2.7926793195899888]], [[4.4952215622516896, -4.6577782381162312, 
0.59934443651978242, -4.3196997055193878], [-1.864942732046011, 3.8152995556077425, -3.3050894242086279, -0.57891882876180301], 
[0.090474505243525627, 3.4417276199459224, -3.5708071810966526, -3.3859675282826385]]], [[[-3.5864405443650007, 
-2.7027562922114825, -2.7632191124911163, -0.34925341009343924], [-1.9950814456442623, -4.9795659063956119, 
-4.3424857220121682, 0.17058211128912504], [-2.8837503879731563, -3.5022499336961332, 4.8424760102210147, 3.6978850304152644]], 
[[3.1310356154523582, -0.89953412465328775, 4.7467830867300869, 1.58301525932022], [4.788954781829176, 0.23308525129814495, 
-3.0234624882490557, -4.7324359077685596], [0.92415280057223637, -4.4999575080800458, -0.59761258532224204, 
0.70587048027670996]]], [[[-1.0551726634368319, -1.1908058914548558, 2.5265249795823603, -0.4588564680535292], 
[4.4193805521732656, 4.8272726359189164, 3.7355920347235312, 0.79694185157383135], [-4.8267607865345337, 3.3898605975413325, 
-1.5333301258963461, -4.8905752130407629]], [[-0.5678878331216497, 1.8221910633200524, 2.2583366328175778, 
-4.4853648345873403], [-1.3862534615836841, -4.7701438997154764, 4.282229579143543, -1.2921742146697368], [1.2253630457091793, 
-1.6865544581450487, -1.07753302084192, -1.0924106892025041]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.9883249893265464, 0.48019562308682051, -6.8378273189095085, -6.4615260921471132], 
[-0.28376377573615574, -8.7017292687214436, -2.1834233297643859, -6.2111956009622107], [-8.9195052774001962, 
-6.1175977807769524, -0.32818542864232914, -6.1798366833561271]], [[-9.0000821245387961, -3.4407089061591787, 
-6.9984098874077505, -5.6027424502161889], [-7.0621737485166767, -8.5273966145803772, -2.7500810717614979, -7.747176191012529], 
[-6.4984474537303649, -5.827232616165209, -8.5196372616007814, -1.3238305825880099]]], [[[-8.102664341183516, 
-9.1841471519522724, -8.676882164733243, -1.1253115498162982], [-7.8104877842080054, 0.09967298539651015, -3.5365159165543334, 
-6.2102838284433934], [-2.1103649165641594, -2.4654206500561733, -4.2152516014678172, -6.1377881392209543]], 
[[-3.3481035177204008, -1.6836639925536332, -8.4471557674079705, -3.1486369678610071], [-6.0795264441238341, 
-3.1001986469715783, -2.6198044499517747, -2.9761320990960272], [-1.9080126841479093, -7.9907792639527111, -5.7175535124533994, 
-6.3583953547850927]]], [[[-4.3934302336420039, -3.1564321992179334, -0.79073479545397962, -1.7160170056415467], 
[-6.101635158662793, -5.4971890670953538, -0.52495409119961334, -5.5514365649465125], [0.235951840353537, -8.4973012602900155, 
-6.2316747825121732, -2.5347883845912778]], [[-3.7816744836718383, -2.3560317120580292, -2.7686245017171833, 
-6.3884257171847771], [-7.9139169447422075, -3.0878358034078888, -0.60713588978532496, -2.6395098524047862], 
[-8.1748964652744913, -5.0698107901448664, -0.61439423888179601, 
-1.5530250276489115]]]])+(1.-msk_ref)*numpy.array([[[[-2.2351916704789812, -1.803352426330429, -1.8696383778309436, 
-4.4943166129587881], [-2.786504756726691, -4.2517861320527395, -4.140512278325815, -7.1405225369232985], 
[-0.18431955303244063, -8.6075970438357103, -6.8692875005657648, -1.7048279181224615]], [[-0.0022856754607607499, 
-9.1552854758286806, -3.8981628011926679, -8.8172069432318381], [-6.3624499697584618, -0.68220768210470784, 
-7.8025966619210783, -5.0764260664742533], [-4.4070327324689247, -1.055779617766528, -8.0683144188091021, 
-7.8834747659950892]]], [[[-8.0839477820774519, -7.2002635299239328, -7.2607263502035666, -4.8467606478058896], 
[-6.492588683356713, -9.4770731441080613, -8.8399929597246185, -4.3269251264233253], [-7.3812576256856062, -7.999757171408584, 
0.34496877250856439, -0.79962220729718592]], [[-1.3664716222600921, -5.3970413623657381, 0.24927584901763655, 
-2.9144919783922303], [0.29144754411672569, -4.2644219864143054, -7.520969725961506, -9.229943145481009], [-3.5733544371402139, 
-8.9974647457924952, -5.0951198230346924, -3.7916367574357404]]], [[[-5.5526799011492827, -5.6883131291673061, 
-1.97098225813009, -4.9563637057659795], [-0.07812668553918467, 0.32976539820646611, -0.76191520298891913, -3.700565386138619], 
[-9.3242680242469831, -1.1076466401711178, -6.0308373636087964, -9.3880824507532132]], [[-5.0653950708341, -2.6753161743923979, 
-2.2391706048948725, -8.9828720722997915], [-5.8837606992961344, -9.2676511374279258, -0.21527765856890735, 
-5.7896814523821867], [-3.272144192003271, -6.1840616958574994, -5.5750402585543704, -5.5899179269149544]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([3.380801117646401, 0.68276102374943193]),self.functionspace)
      arg1=-3.13461677558
      res=arg0+arg1
      ref=Data(numpy.array([0.24618434206335982, -2.4518557518336093]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([-0.10529216435566191, 0.11892996878134365]),self.functionspace)
      arg1=numpy.array(-2.70701928771)
      res=arg0+arg1
      ref=Data(numpy.array([-2.8123114520675214, -2.5880893189305159]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([4.4157908698607287, -2.1849199527121166]),self.functionspace)
      arg1=numpy.array([0.39055423217990715, -3.652006707491867])
      res=arg0+arg1
      ref=Data(numpy.array([4.8063451020406358, -5.836926660203984]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([-4.0781401396320272, -3.8342176518094631]),self.functionspace)
      arg1=Data(-4.09703006703,self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([-8.1751702066609404, -7.9312477188383772]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([0.88837527824544615, -2.8254004186216775]),self.functionspace)
      arg1=Data(numpy.array([-4.5587803867724732, -1.2103602386595069]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([-3.6704051085270271, -4.0357606572811839]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([-4.3870739207874001, -0.81079869650812686]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.35025254145)+(1.-msk_arg1)*(-2.54174527848)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-2.0368213793324044, 1.5394538449468689])+(1.-msk_ref)*numpy.array([-6.9288191992642147, 
-3.3525439749849415])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([2.0300026906658211, 3.8523515021299222]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-2.0440061592530121, -3.4355138530088505])+(1.-msk_arg1)*numpy.array([0.99302130973079805, 
2.3442922249368658])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.014003468587191037, 0.41683764912107168])+(1.-msk_ref)*numpy.array([3.0230240003966191, 
6.196643727066788])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[-4.3817793917205874, -0.44646179867020486, 1.646091317876377, 0.93543149086059252, 
1.9112427814867692], [-3.5194530728090188, -0.19161678468272569, -4.7905929905102651, -2.8734377213110553, 1.2784951165366953], 
[-1.3872853149970066, 1.0175959879721796, -1.161525727635595, -0.10617350105968004, -4.3915817823298129], [4.0908125781680553, 
1.2482416212215561, 1.7228486273053223, 1.9169381988907972, -2.0651031539630038]]),self.functionspace)
      arg1=0.874725892666
      res=arg0+arg1
      ref=Data(numpy.array([[-3.507053499055087, 0.42826409399529553, 2.5208172105418774, 1.8101573835260929, 
2.7859686741522696], [-2.6447271801435184, 0.68310910798277469, -3.9158670978447647, -1.998711828645555, 2.1532210092021957], 
[-0.5125594223315062, 1.89232188063768, -0.28679983497009465, 0.76855239160582034, -3.5168558896643125], [4.9655384708335557, 
2.1229675138870565, 2.5975745199708227, 2.7916640915562976, -1.1903772612975034]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[-0.549632572896809, -1.7169197856801777, 3.4398299042166052, 3.6973541899774851, 
3.1702432164972389], [-0.66564461893191851, 2.656464654950609, -1.4932686989336084, 4.492622667412455, 3.204350286191838], 
[-1.4253803853055267, 0.41776619450403008, 1.2117354277769898, 2.6058064856763803, 0.68882841450227961], [0.10661662801144534, 
-3.5372523877376971, -0.234917486563476, -4.6378324912436213, 3.8765424724683868]]),self.functionspace)
      arg1=numpy.array(-0.802981045734)
      res=arg0+arg1
      ref=Data(numpy.array([[-1.3526136186305386, -2.5199008314139073, 2.6368488584828755, 2.8943731442437555, 
2.3672621707635093], [-1.4686256646656481, 1.8534836092168794, -2.296249744667338, 3.6896416216787253, 2.4013692404581084], 
[-2.2283614310392563, -0.38521485122969956, 0.4087543820432602, 1.8028254399426507, -0.11415263123145003], 
[-0.6963644177222843, -4.3402334334714272, -1.0378985322972056, -5.4408135369773509, 3.0735614267346572]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[2.7310237299815281, 4.3187359006236701, 0.75148360487339971, -3.6176847203788753, 
-0.46298700387238956], [3.4016211251259687, 1.2277731195467485, -3.4772192405876368, -2.0773475241311576, 
-0.47678727519771691], [0.94889125884478709, 1.8000210534260885, 4.5423354377728735, 2.0134113884291294, 3.1430303445418648], 
[3.3342413374529212, 1.4832445233692919, -4.4861669321317379, 4.6497627965912756, 2.7747940373987623]]),self.functionspace)
      arg1=numpy.array([[2.6251016195406951, 1.7039813348581898, 3.9760810773080539, 4.8998965789677928, -3.5751600849606477], 
[-3.3910899156532226, -0.14987856299683777, -1.7088795545073534, -2.3744591336294087, -2.3648467835010631], 
[-1.6265256306918676, 3.3843361262855218, -1.5473868349374156, -4.7988730921652865, 0.86867254086105383], [2.1032731764939747, 
-1.5385767093223879, -0.93028530946946209, 1.5025621825610802, 4.8910842603937557]])
      res=arg0+arg1
      ref=Data(numpy.array([[5.3561253495222232, 6.0227172354818599, 4.7275646821814536, 1.2822118585889175, 
-4.0381470888330373], [0.010531209472746106, 1.0778945565499107, -5.1860987950949902, -4.4518066577605664, 
-2.8416340586987801], [-0.67763437184708053, 5.1843571797116104, 2.9949486028354579, -2.7854617037361571, 4.0117028854029186], 
[5.4375145139468959, -0.055332185953095969, -5.4164522416012, 6.1523249791523558, 7.6658782977925179]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[-0.76362016394606602, -1.4809276008107961, -2.2646029172902993, 0.96494832759073645, 
4.2129306622053146], [-0.58460481913347095, -3.4606222254010999, -2.7388603028962901, -1.7060327463402167, 
0.43720020581359709], [-4.861640611089717, -1.4969707017526135, -0.77353705648168347, 2.3352679805014525, -1.4830627453342893], 
[4.3906468705423407, -4.8568146825430087, 3.8434480919934924, -2.8456803224425906, -1.321169773273235]]),self.functionspace)
      arg1=Data(0.550260154046,self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[-0.21336000989960713, -0.93066744676433721, -1.7143427632438404, 1.5152084816371953, 
4.7631908162517735], [-0.034344665087012061, -2.910362071354641, -2.1886001488498312, -1.1557725922937578, 
0.98746035986005598], [-4.3113804570432581, -0.94671054770615459, -0.22327690243522458, 2.8855281345479113, 
-0.93280259128783038], [4.9409070245887996, -4.3065545284965498, 4.3937082460399512, -2.2954201683961317, 
-0.77090961922677614]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-2.7757203238086072, 4.6163980348217386, -3.1156165812405554, 0.52683941172689508, 
4.2194500034262905], [-4.9156959162649851, -1.233657563257208, 1.3311820569612358, 4.2009281469037347, -4.4253147822752457], 
[4.809888557518299, -4.9102796110213633, 2.1148024719074581, -3.9914397633329868, 0.072394748678142662], [0.51335766355364143, 
2.0793947135417383, -0.35667881407990798, -0.054149551513233973, 2.0177886211623255]]),self.functionspace)
      arg1=Data(numpy.array([[-3.1939111137579488, -0.62539324358752069, 2.3178721471402142, -2.6224610360814982, 
2.8767599942930655], [1.9091205710899208, -2.2361373415039343, 2.8545535673177582, 1.4622839566947867, 1.026368469955691], 
[3.3737150528330471, 4.0555539162551248, 3.7977364134543592, -2.2164238734412125, 3.7207220439970747], [1.132952399659974, 
-4.1576382137820804, -4.3051344247589114, -4.8399964923696341, 4.3908847920124643]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[-5.969631437566556, 3.9910047912342179, -0.79774443410034124, -2.0956216243546031, 
7.096209997719356], [-3.0065753451750643, -3.4697949047611423, 4.185735624278994, 5.6632121035985215, -3.3989463123195547], 
[8.1836036103513461, -0.8547256947662385, 5.9125388853618173, -6.2078636367741993, 3.7931167926752174], [1.6463100632136154, 
-2.0782435002403421, -4.6618132388388194, -4.8941460438828681, 6.4086734131747898]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[0.025380065310302236, 4.8838529524898249, -1.4480638213800834, -0.29806270573543348, 
-3.3408695294631627], [-2.5253542670242379, 4.451047103181935, 4.144477204534839, 0.54198269500550644, 4.3731306660358591], 
[-2.7087650891497503, -4.8237390182006914, -3.5014429450783471, -4.0891574117611267, -1.8742732518428862], 
[-1.7389993255930749, -3.873191701119346, 0.50681498575260697, -0.17228214070813408, 3.2869506274343117]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.38536662602)+(1.-msk_arg1)*(1.19352775481)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.410746691334098, 6.2692195785136207, -0.062697195356287594, 1.0873039202883623, 
-1.9555029034393669], [-1.1399876410004421, 5.8364137292057308, 5.5298438305586348, 1.9273493210293022, 5.7584972920596549], 
[-1.3233984631259545, -3.4383723921768956, -2.1160763190545513, -2.7037907857373309, -0.48890662581909039], 
[-0.35363269956927912, -2.4878250750955502, 1.8921816117764028, 1.2130844853156617, 
4.6723172534581074]])+(1.-msk_ref)*numpy.array([[1.218907820116784, 6.0773807072963066, -0.25453606657360162, 
0.89546504907104829, -2.147341774656681], [-1.3318265122177562, 5.6445748579884167, 5.3380049593413208, 1.7355104498119882, 
5.5666584208423409], [-1.5152373343432686, -3.6302112633942096, -2.3079151902718653, -2.8956296569546449, 
-0.68074549703640441], [-0.54547157078659314, -2.6796639463128642, 1.7003427405590887, 1.0212456140983477, 4.4804783822407934]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[2.3621062714229604, 2.2868646385962608, 4.0393312137192332, -2.7780709454208354, 
1.5876169977415753], [-0.60842762148557572, -0.68697905313115282, 2.5942782899794423, -0.08375220954863849, 
-0.3579115739891412], [3.1946085907401773, 3.1828107365688911, -2.810877993995641, -4.8710409513645958, 0.7112402372829072], 
[-4.7632676265067877, -2.2281276721453782, 3.225065915868278, -1.9754686040810068, 1.4122977220286828]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-4.9063812974942964, 1.8991559238311382, 3.3303581963793984, 2.77355546918157, 
1.9239698510859569], [0.42104435455712075, 2.0984640013926921, -0.17898023356983295, -0.49189288145345955, 4.3993872647206782], 
[4.8166355978041295, -1.863170343088616, -2.4397056960418695, -3.0111900231024737, -3.3329909434132885], [-4.3010716804293612, 
0.94301902070756061, 2.6954675398958052, 3.0970847894217872, 
1.7121537860415366]])+(1.-msk_arg1)*numpy.array([[1.4524591019417867, -0.60352228968704758, 1.3146650350427631, 
0.2256371789734013, 2.5470717063868094], [-4.8911598957736846, -2.5953346585470136, 1.8446476404244496, 1.0747610605771074, 
2.2735879352742652], [3.0242449122683865, 0.46706464261334713, -4.1657889067607634, -4.5329761232176473, -1.9737069481237199], 
[-3.8894284143530831, -1.1834844481858098, -3.3434995137643506, -3.3816250672018477, -2.4392456677388088]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.544275026071336, 4.1860205624273989, 7.3696894100986317, -0.0045154762392654035, 
3.5115868488275321], [-0.18738326692845497, 1.4114849482615393, 2.4152980564096094, -0.57564509100209804, 4.041475690731537], 
[8.0112441885443069, 1.3196403934802752, -5.2505836900375105, -7.8822309744670696, -2.6217507061303813], [-9.064339306936148, 
-1.2851086514378176, 5.9205334557640832, 1.1216161853407804, 
3.1244515080702193]])+(1.-msk_ref)*numpy.array([[3.8145653733647471, 1.6833423489092132, 5.3539962487619963, 
-2.5524337664474341, 4.1346887041283846], [-5.4995875172592603, -3.2823137116781664, 4.4389259304038919, 0.99100885102846892, 
1.915676361285124], [6.2188535030085639, 3.6498753791822383, -6.9766669007564044, -9.4040170745822422, -1.2624667108408127], 
[-8.6526960408598708, -3.411612120331188, -0.11843359789607266, -5.3570936712828541, -1.026947945710126]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-1.7101419802817661, 4.6197481808498999], [-2.0707313157893212, 2.7142584125107314]], 
[[0.46097972932011455, 2.1462178917049606], [2.7892289627314728, 0.53388645219176833]], [[-0.23727125444570252, 
3.1206761706227386], [-3.2011473909004771, 1.709216570474946]], [[3.3895162053657355, 1.6923092242186488], 
[-0.28790135680243623, 4.6172083955420113]], [[-1.4664122533670607, 0.3043404608895397], [-1.1192320270979494, 
4.4236851656943106]], [[-2.6230252331344794, -1.9019295347784748], [-0.44983618994491081, 
4.263089914043185]]]),self.functionspace)
      arg1=-0.221449683979
      res=arg0+arg1
      ref=Data(numpy.array([[[-1.9315916642608428, 4.3982984968708232], [-2.292180999768398, 2.4928087285316547]], 
[[0.23953004534103783, 1.9247682077258839], [2.5677792787523961, 0.31243676821269162]], [[-0.45872093842477923, 
2.8992264866436619], [-3.4225970748795538, 1.4877668864958693]], [[3.1680665213866588, 1.4708595402395721], 
[-0.50935104078151294, 4.3957587115629346]], [[-1.6878619373461374, 0.082890776910462982], [-1.3406817110770262, 
4.2022354817152339]], [[-2.8444749171135562, -2.1233792187575515], [-0.67128587392398753, 
4.0416402300641083]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[-0.86417204131083647, -2.466331901741643], [-4.8934938324937871, -3.5625913426588318]], 
[[3.832370994799895, 3.1674824422304386], [2.7958316645160739, 4.2521619257746117]], [[-1.6695279250496577, 
-4.3392691039327822], [-3.4138647380372191, -2.5795445399595907]], [[1.5656183524532574, -0.028794330019772829], 
[1.5741267702945301, 0.44400167622904885]], [[-1.1827189933381965, -0.60156944634290532], [-0.92280468101449209, 
-2.0698533890657265]], [[-4.6720545487817411, 2.0987889897781233], [3.5379814311187463, 
3.0664385091454189]]]),self.functionspace)
      arg1=numpy.array(-3.41752342296)
      res=arg0+arg1
      ref=Data(numpy.array([[[-4.2816954642734535, -5.8838553247042604], [-8.3110172554564041, -6.9801147656214493]], 
[[0.41484757183727794, -0.2500409807321784], [-0.62169175844654312, 0.8346385028119947]], [[-5.0870513480122748, 
-7.7567925268953992], [-6.8313881609998361, -5.9970679629222072]], [[-1.8519050705093596, -3.4463177529823898], 
[-1.8433966526680869, -2.9735217467335682]], [[-4.600242416300814, -4.0190928693055223], [-4.3403281039771091, 
-5.4873768120283435]], [[-8.0895779717443581, -1.3187344331844937], [0.12045800815612928, 
-0.35108491381719809]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[2.8828239066513852, 2.1625489684310981], [0.23759919726026979, -0.26643504351603475]], 
[[2.1306376319039364, 3.6745195841120104], [-1.807004012304001, 2.3727052665435764]], [[-2.1727288753793661, 
-1.3975218430655989], [-2.3049312744925832, -0.48734813338164873]], [[2.8177426658924318, -2.1281803084783935], 
[3.1961866139659723, 1.6508742174208777]], [[-0.84938786668461308, -1.5551224018777141], [4.6976623390165102, 
-0.74008403505175657]], [[-4.2499108200327083, -1.2380267771891873], [-0.47581886784526262, 
3.6853524700593603]]]),self.functionspace)
      arg1=numpy.array([[[3.7956943478210068, 3.5008130226768284], [1.7504519504516223, -3.9461101733487847]], 
[[-0.041655260721947585, -0.67462259417828108], [-0.51606346253996538, -1.0921904243858895]], [[0.58858718696888701, 
2.0668147462921915], [1.0181305210772864, -2.3413226869592974]], [[2.048222318580625, 2.1965958153940726], 
[-4.6019065974244011, 3.8529199397148268]], [[-4.9217274100696153, -4.745358414075401], [-0.33542070965880466, 
0.31252609971592715]], [[-4.7582569933445829, -2.1408750920403188], [-0.68883181376959257, -4.8930343741979518]]])
      res=arg0+arg1
      ref=Data(numpy.array([[[6.678518254472392, 5.6633619911079265], [1.9880511477118921, -4.2125452168648199]], 
[[2.0889823711819888, 2.9998969899337293], [-2.3230674748439664, 1.2805148421576869]], [[-1.5841416884104791, 
0.66929290322659263], [-1.2868007534152968, -2.8286708203409461]], [[4.8659649844730568, 0.068415506915679103], 
[-1.4057199834584289, 5.5037941571357045]], [[-5.7711152767542284, -6.3004808159531152], [4.3622416293577055, 
-0.42755793533582942]], [[-9.0081678133772911, -3.3789018692295061], [-1.1646506816148552, 
-1.2076819041385916]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-3.1418400257252923, 3.0052055797812081], [-3.2499776925753898, -0.68827458926063567]], 
[[-3.3457417565856451, 2.1721888216900966], [-3.0721723912019225, -3.7716996808205105]], [[-0.7944277148556349, 
-2.7997131143311984], [-0.13765045131191656, 4.9230194539371102]], [[-2.5964514873157118, -0.84644136154569782], 
[-3.5542900418419929, 1.6459301033633018]], [[4.0992281064060663, -4.0656977810777093], [-1.2298422974046632, 
-0.53582695140009839]], [[2.6935065941584533, 1.3537291046906716], [1.9026193957283963, 
-4.6632134267165739]]]),self.functionspace)
      arg1=Data(3.97329151654,self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[0.83145149081874337, 6.9784970963252437], [0.72331382396864585, 3.2850169272834]], 
[[0.62754975995839057, 6.1454803382341323], [0.90111912534211314, 0.20159183572352513]], [[3.1788638016884008, 
1.1735784022128373], [3.8356410652321191, 8.8963109704811458]], [[1.3768400292283238, 3.1268501549983378], 
[0.41900147470204274, 5.6192216199073375]], [[8.0725196229501019, -0.092406264533673621], [2.7434492191393725, 
3.4374645651439373]], [[6.666798110702489, 5.3270206212347073], [5.875910912272432, -0.68992191017253823]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[2.19993627642849, 0.19762612972600913], [1.9619237142176518, -1.9932062137370199]], 
[[-4.1045013488419997, 4.832389567986926], [1.9831524987395275, 0.81510485305062552]], [[0.23641081680056519, 
1.76764611855174], [0.10954533154500368, -2.1979111445364796]], [[3.7430865116035896, 1.9690028258095351], [1.1108200977211613, 
-4.684458303311704]], [[-1.5261434871651502, 0.95415756967539345], [-0.48002405486379995, 4.1010396960348157]], 
[[3.6464218098287855, 4.6015948886112295], [2.222185693984132, -0.11134852975327014]]]),self.functionspace)
      arg1=Data(numpy.array([[[-0.16046933035215716, 1.2734271820090628], [-0.37720292369231778, -0.87701417113956559]], 
[[-0.48401575819680787, -3.8990467008015406], [4.8533373696909088, -4.2325274746188519]], [[-2.7301603972075661, 
-4.4037838947945849], [-1.5549226791811268, 3.6179125509173922]], [[-0.086311157399629579, -1.3131562387296567], 
[0.47037892588558972, 3.855254078415145]], [[1.5984976362580472, -4.1193186033437259], [1.5406997244839857, 
-2.755374082252362]], [[-1.779204548373853, -1.7532929276961671], [4.3821985516730848, 
-1.2493342505772631]]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[2.0394669460763328, 1.4710533117350719], [1.584720790525334, -2.8702203848765855]], 
[[-4.5885171070388076, 0.93334286718538539], [6.8364898684304363, -3.4174226215682264]], [[-2.4937495804070009, 
-2.6361377762428448], [-1.4453773476361231, 1.4200014063809125]], [[3.65677535420396, 0.65584658707987842], [1.581199023606751, 
-0.82920422489655898]], [[0.072354149092896947, -3.1651610336683325], [1.0606756696201858, 1.3456656137824536]], 
[[1.8672172614549325, 2.8483019609150624], [6.6043842456572168, -1.3606827803305332]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[-3.6682873346278191, -3.6410540119487655], [4.5196115327392956, 2.4356074468434032]], 
[[-0.094205678316717112, -0.30515240231588514], [-4.7472131686650467, 1.0281702541208819]], [[0.62244119605194737, 
-0.90269835362511319], [-0.21970528047607019, 0.19511034478432077]], [[3.4687009665879014, 3.7800756009788223], 
[0.2517321923764646, 3.634562996797257]], [[-4.4803503983984658, 2.0321951538745786], [-2.8070649412095605, 
-3.7844326911501716]], [[3.5616038956442519, -1.3619807506538106], [-4.3674285456660558, 
4.2464367420087825]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-4.46338710797)+(1.-msk_arg1)*(-2.74533283779)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-8.1316744426004455, -8.1044411199213915], [0.056224424766669223, -2.0277796611292231]], 
[[-4.5575927862893435, -4.7685395102885115], [-9.210600276637674, -3.4352168538517445]], [[-3.840945911920679, 
-5.3660854615977396], [-4.6830923884486966, -4.2682767631883056]], [[-0.994686141384725, -0.68331150699380405], 
[-4.2116549155961618, -0.82882411117536936]], [[-8.9437375063710931, -2.4311919540980478], [-7.2704520491821869, 
-8.2478197991227979]], [[-0.90178321232837444, -5.8253678586264375], [-8.8308156536386822, 
-0.21695036596384387]]])+(1.-msk_ref)*numpy.array([[[-6.413620172422382, -6.386386849743328], [1.7742786949447327, 
-0.3097253909511597]], [[-2.8395385161112801, -3.0504852401104481], [-7.4925460064596097, -1.717162583673681]], 
[[-2.1228916417426156, -3.6480311914196761], [-2.9650381182706331, -2.5502224930102422]], [[0.72336812879333845, 
1.0347427631842594], [-2.4936006454180983, 0.88923015900269409]], [[-7.2256832361930288, -0.71313768391998433], 
[-5.5523977790041235, -6.5297655289447345]], [[0.81627105784968901, -4.107313588448374], [-7.1127613834606187, 
1.5011039042142196]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[2.6962830832473195, -4.5333105355686385], [0.54278369187188069, 2.4436212417810319]], 
[[-2.8723521887990255, -0.31285282169968021], [-2.9359926891838359, 2.1789258771370781]], [[1.6716177510980721, 
1.2980820771748842], [-3.5209665323909931, 2.3064325513547779]], [[2.5997810220508013, 1.4087671772659593], 
[2.5340871516523524, 3.857081159090189]], [[-4.0452883741302381, -3.6616632268330198], [2.8635206025560827, 
4.0098698767655474]], [[-4.347149357339898, 0.98058277476708078], [3.113355671893526, 
-0.38233554076781395]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-3.6247673807478362, -3.8091422444830405], [-2.9959875873682229, 0.78981899834612079]], 
[[-1.4336988312884946, -4.8937332581287931], [-0.35521490594312688, 0.014476610331607631]], [[1.5232177600310255, 
-1.4639400638235078], [-0.73935725789784446, -3.0776071167731258]], [[-1.5794714584774905, 1.6556901941074162], 
[2.8681474673214904, 3.6302877813823127]], [[-2.8481763285294539, 3.302707248032732], [2.0913649861990802, 3.976602080423115]], 
[[-0.40380229712017446, 1.5278136698297686], [0.9782109311541749, 
-4.0072358292931209]]])+(1.-msk_arg1)*numpy.array([[[-2.8680000738981715, -2.2415906068444711], [2.9013634425571944, 
-3.3252094687942968]], [[-1.6851367595243749, 3.4154174759422755], [-3.6508767157908317, 3.8118084158940633]], 
[[-1.6201559047050349, -2.3532895557104236], [2.4448744648885175, -2.009442885922661]], [[1.5931973028552635, 
-3.9604447613816052], [3.4478074847662299, 0.76969687788214891]], [[-4.8123077521381505, -0.20830375070611851], 
[0.32304715665930939, -4.9330997566113535]], [[-2.0272623422556366, 0.54485368842913928], [-4.7041469960439848, 
3.4454225345792153]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.92848429750051675, -8.3424527800516799], [-2.4532038954963422, 3.2334402401271527]], 
[[-4.3060510200875202, -5.2065860798284733], [-3.2912075951269628, 2.1934024874686857]], [[3.1948355111290976, 
-0.16585798664862361], [-4.2603237902888376, -0.77117456541834795]], [[1.0203095635733108, 3.0644573713733756], 
[5.4022346189738428, 7.4873689404725017]], [[-6.893464702659692, -0.35895597880028784], [4.9548855887551628, 
7.9864719571886624]], [[-4.7509516544600725, 2.5083964445968494], [4.0915666030477009, 
-4.3895713700609349]]])+(1.-msk_ref)*numpy.array([[[-0.171716990650852, -6.7749011424131096], [3.4441471344290751, 
-0.88158822701326489]], [[-4.5574889483234005, 3.1025646542425953], [-6.5868694049746672, 5.9907342930311414]], 
[[0.051461846393037192, -1.0552074785355394], [-1.0760920675024757, 0.29698966543211691]], [[4.1929783249060648, 
-2.5516775841156458], [5.9818946364185823, 4.6267780369723379]], [[-8.8575961262683887, -3.8699669775391383], 
[3.186567759215392, -0.92322987984580607]], [[-6.3744116995955347, 1.5254364631962201], [-1.5907913241504588, 
3.0630869938114014]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[3.4708671377203792, 2.9520155024055006, 1.8968288460108971, -4.9421908927442857], 
[0.19074592100859356, -3.7146529577716847, 1.7423738851561037, -2.5054404215089479], [3.4555785212292438, -2.4582242473308167, 
1.5050298070766051, 3.9992618072838688]], [[-1.8127591514154018, 0.75169379572550454, 4.5239560569712296, 3.1148254469141374], 
[-3.9499470049909045, 0.72884073561107243, 4.4462038790801284, -1.5525294463475383], [3.5338633149613123, 2.0643403757057097, 
4.9128477205630627, 1.0075230976557261]]], [[[-0.45405825931907895, 4.8450076935466431, 2.895472545700879, 3.7790965738529927], 
[-2.6614681537455063, -4.4065216601082096, -4.374446233118289, -4.1179447764753965], [4.5361211872887441, 3.2115972716318186, 
-1.4234123090521966, 4.0762596609284234]], [[-3.9002692005961048, 1.8360245789760334, 4.1486946675252092, 0.30922239653696781], 
[4.1029273943560121, 4.8910604885901758, -2.7800911837007103, -1.941291645754716], [-1.5942160434165764, 1.5529953312196829, 
1.3782142268121902, -3.5542757570964154]]], [[[-2.3549942112759359, -3.0573639001463535, 1.2795360601153103, 
2.8072509791028608], [4.5622266895986776, -1.5921203645011364, -3.6057841037537761, -2.6923432300783503], [-2.8182321373328967, 
-0.61879424838684383, 3.0751516706689834, -1.0729189488324273]], [[-3.1816052913220174, -0.71357363675829699, 
4.963998418648309, 4.2394912175381201], [4.2720651865524779, -2.0568595293258332, -1.4172263018083306, -1.0055346532937257], 
[-1.5045322816575202, 0.47801634651457281, -3.8888642424075339, -0.97472481814279455]]]]),self.functionspace)
      arg1=1.1084616407
      res=arg0+arg1
      ref=Data(numpy.array([[[[4.5793287784168788, 4.0604771431020001, 3.0052904867073966, -3.8337292520477861], 
[1.2992075617050931, -2.6061913170751851, 2.8508355258526032, -1.3969787808124483], [4.5640401619257434, -1.3497626066343171, 
2.6134914477731046, 5.1077234479803684]], [[-0.70429751071890223, 1.8601554364220041, 5.6324176976677292, 4.223287087610637], 
[-2.8414853642944049, 1.837302376307572, 5.5546655197766279, -0.44406780565103876], [4.6423249556578119, 3.1728020164022093, 
6.0213093612595623, 2.1159847383522257]]], [[[0.6544033813774206, 5.9534693342431426, 4.0039341863973785, 4.8875582145494922], 
[-1.5530065130490067, -3.29806001941171, -3.2659845924217894, -3.0094831357788969], [5.6445828279852437, 4.3200589123283182, 
-0.31495066835569707, 5.184721301624923]], [[-2.7918075598996053, 2.944486219672533, 5.2571563082217088, 1.4176840372334674], 
[5.2113890350525116, 5.9995221292866754, -1.6716295430042107, -0.83283000505821647], [-0.48575440272007686, 2.6614569719161825, 
2.4866758675086897, -2.4458141163999159]]], [[[-1.2465325705794363, -1.948902259449854, 2.3879977008118098, 
3.9157126197993604], [5.6706883302951772, -0.48365872380463681, -2.4973224630572766, -1.5838815893818508], 
[-1.7097704966363971, 0.48966739230965572, 4.183613311365483, 0.035542691864072218]], [[-2.0731436506255179, 
0.39488800393820256, 6.0724600593448086, 5.3479528582346196], [5.3805268272489775, -0.94839788862933361, -0.3087646611118311, 
0.10292698740277384], [-0.39607064096102063, 1.5864779872110724, -2.7804026017110344, 0.133736822553705]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[-1.4320069558586948, 0.52071481861502456, -3.1999482284900904, 1.1834805882708155], 
[1.6916520467919085, 3.2653081321066963, -3.8931793639497658, 3.2884057459513336], [-0.65802588333201761, 3.3275579640870063, 
4.0415679303185623, 1.7387959988961077]], [[2.4470734416409767, -0.22107852066048572, -2.9513708077694845, 
0.18779944131567472], [-3.9786580013370507, -3.5726965058867832, -3.350048817268132, 3.6504798861618326], 
[-0.19005118157258138, -3.2147526490117673, 3.6557476735018462, -4.630335617931987]]], [[[2.9253486482940296, 
-0.65486363156351146, 2.2594561891285423, -0.67636419600633246], [4.9872850030513938, -4.9001069958216359, -3.5993453999765004, 
3.911069564515703], [0.80494053442070523, -1.3119156132237029, 2.2930562137871826, -4.5645660989389016]], 
[[-4.1725735042167873, 3.0425380298181732, 0.015108242610695655, -4.9019267292272186], [-2.3351132373751926, 
-4.8310274713379364, -4.7769927275183708, -4.3868254626333991], [-4.2679408456781189, 1.0238313491180326, -3.2389830837475051, 
-2.7895670178447753]]], [[[-0.81533824956314671, 3.9263545041168655, -0.0045661031343282588, 3.3354410479030712], 
[4.3402223107545126, 3.4297457917765488, 2.4509565290701838, -3.9432469299661066], [4.9994837660085132, -1.7454911915284841, 
-2.6823773942475637, 3.7889227893400044]], [[1.7307850681446313, -2.9434417624280895, 4.8280779110008964, -2.9863979144425823], 
[2.1011339651961674, -3.1058482669354781, -2.5520811589622494, 0.96441665566635137], [2.1539485024501657, -0.45752228549093132, 
-2.1305174854496167, 0.84990681315838401]]]]),self.functionspace)
      arg1=numpy.array(3.86989734015)
      res=arg0+arg1
      ref=Data(numpy.array([[[[2.4378903842912294, 4.3906121587649487, 0.66994911165983373, 5.0533779284207396], 
[5.5615493869418327, 7.1352054722566205, -0.023282023799841678, 7.1583030861012578], [3.2118714568179065, 7.1974553042369305, 
7.9114652704684865, 5.6086933390460318]], [[6.3169707817909009, 3.6488188194894384, 0.91852653238043969, 4.0576967814655989], 
[-0.10876066118712657, 0.29720083426314092, 0.51984852288179217, 7.5203772263117568], [3.6798461585773428, 0.65514469113815688, 
7.5256450136517703, -0.76043827778206285]]], [[[6.7952459884439538, 3.2150337085864127, 6.1293535292784664, 
3.1935331441435917], [8.8571823432013179, -1.0302096556717117, 0.27055194017342377, 7.7809669046656271], [4.6748378745706294, 
2.5579817269262213, 6.1629535539371068, -0.69466875878897749]], [[-0.30267616406686315, 6.9124353699680974, 3.8850055827606198, 
-1.0320293890772945], [1.5347841027747315, -0.96113013118801227, -0.90709538736844664, -0.51692812248347497], 
[-0.39804350552819479, 4.8937286892679568, 0.63091425640241905, 1.0803303223051488]]], [[[3.0545590905867774, 
7.7962518442667896, 3.8653312370155959, 7.2053383880529953], [8.2101196509044367, 7.299643131926473, 6.3208538692201079, 
-0.073349589816182448], [8.8693811061584373, 2.1244061486214401, 1.1875199459023604, 7.6588201294899285]], 
[[5.6006824082945554, 0.92645557772183462, 8.6979752511508206, 0.88349942570734186], [5.9710313053460915, 0.76404907321444604, 
1.3178161811876747, 4.8343139958162755], [6.0238458426000898, 3.4123750546589928, 1.7393798547003074, 
4.7198041533083082]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[2.4900941733892772, -0.44346990257986896, 0.36244838280853031, 1.657555655289733], 
[0.12650313429849191, 4.0602446330183835, 3.2769880996006879, -1.8580799877983023], [-2.7497241879092726, 1.1213442346257789, 
1.0898926388672612, 0.19430427879212697]], [[4.1922208921718127, -4.0655166796641664, -4.4892002607107973, 4.4331929590332066], 
[3.5717568834498596, 0.001356319040109355, -3.0671665832119244, -2.0178485348236208], [-1.7025450268621456, 
-0.53647250522980805, -0.93182286464645081, 1.0722678659664506]]], [[[0.68810228655050576, 0.67130203789249077, 
-2.8659843991121692, 2.2798871145289858], [-3.7811324511409894, 4.8959496520676886, 2.6432152144954957, 4.513187357021863], 
[-1.6906947306586484, -1.434445224980232, 3.907667630635018, 2.4925783774666774]], [[3.8720058986575872, -4.7010811788089546, 
-2.6176918711328412, 4.6686510548906917], [2.5316727613369441, -3.2196941741282714, 1.9417666127351669, 2.6243698171233101], 
[4.7594054579116509, -0.049200327658393661, -0.041836851525931529, -1.617246677361126]]], [[[3.9629424468881496, 
-2.6122748205427957, -3.2112578161812388, 4.429389690107266], [3.1468495119206743, -1.8695941113039938, 3.012826435249135, 
1.8846867675774961], [-2.0908542946249362, -1.8498194276081428, -3.6864780677334408, -0.024930094946352455]], 
[[-1.8103842657372038, -2.1914322279843192, -2.143102177316397, 0.085572239637520475], [-0.43960320303560962, 
-0.59358092241016269, -3.0187711363857908, -3.6027608091593324], [-0.4459095327752971, -3.0645213796201922, 3.7969668070779541, 
-3.6702175009274232]]]]),self.functionspace)
      arg1=numpy.array([[[[3.2402174307245293, -1.5847630466786455, -4.096208135308089, 0.38044129148200323], 
[4.3369842081355277, -1.7967945401431127, -1.8646687068362744, -4.4816713525794194], [0.51402245092356491, -1.5160566590067734, 
1.8758053019544594, -3.2805945983092446]], [[4.9770141176342619, 1.2506612887781312, 2.4375870885669997, 0.46308684574113546], 
[-3.3587855772451283, 3.0305740348513943, 0.81757821058310842, -2.9054402614585184], [4.0533152602568308, 2.2627949461684533, 
1.515805714266091, -2.6755937688517584]]], [[[4.9643085921673187, 0.69122681399653807, -4.4215820556943122, 
-1.4588791541095247], [2.1718000742391688, -1.6485481978743657, -3.064893448493895, 4.9457258915380446], [2.9210185614522457, 
-4.5934825705712958, 4.2367319838836792, -2.2544333729614188]], [[0.52691793892191452, -1.3015421733223329, 
-1.0381319394929633, -0.33229994437857702], [-3.5858405895472654, 1.6146596785984606, 0.79451318611488464, 
-1.1801019722111796], [2.3751962652461991, -0.62513524505240259, -1.7993243659238134, 3.4849151201365665]]], 
[[[-2.3154175127431706, 4.0114159659409978, 1.9647872275272258, -2.6685450464037883], [4.4603052996083488, 
-0.87852331903080838, 3.2743406364222398, -2.5798332865479026], [-1.3136550609744302, 0.84062139801237645, 1.9173750416435, 
0.5499713448902197]], [[-1.4532305547459412, 4.215047906246733, 1.247372457933996, -1.486586394687607], [1.9163723337465992, 
3.7523875738986163, 1.7778415654264021, 0.56362936471089053], [-4.4562656344850282, -0.86085520873970722, 3.346405843989757, 
-4.7078642900420391]]]])
      res=arg0+arg1
      ref=Data(numpy.array([[[[5.7303116041138065, -2.0282329492585145, -3.7337597524995587, 2.0379969467717363], 
[4.4634873424340196, 2.2634500928752708, 1.4123193927644135, -6.3397513403777221], [-2.2357017369857077, -0.39471242438099452, 
2.9656979408217206, -3.0862903195171176]], [[9.1692350098060746, -2.8148553908860352, -2.0516131721437976, 4.8962798047743421], 
[0.21297130620473137, 3.0319303538915037, -2.2495883726288159, -4.9232887962821392], [2.3507702333946852, 1.7263224409386453, 
0.5839828496196402, -1.6033259028853077]]], [[[5.6524108787178244, 1.3625288518890288, -7.2875664548064814, 
0.82100796041946111], [-1.6093323769018206, 3.2474014541933229, -0.42167823399839932, 9.4589132485599077], [1.2303238307935973, 
-6.0279277955515278, 8.1443996145186972, 0.23814500450525866]], [[4.3989238375795017, -6.0026233521312875, -3.6558238106258045, 
4.3363511105121146], [-1.0541678282103213, -1.6050344955298108, 2.7362797988500516, 1.4442678449121304], [7.13460172315785, 
-0.67433557271079625, -1.8411612174497449, 1.8676684427754404]]], [[[1.6475249341449789, 1.3991411453982021, 
-1.2464705886540131, 1.7608446437034777], [7.6071548115290231, -2.7481174303348022, 6.2871670716713748, -0.69514651897040647], 
[-3.4045093555993664, -1.0091980295957663, -1.7691030260899407, 0.52504124994386725]], [[-3.263614820483145, 
2.0236156782624137, -0.89572971938240098, -1.4010141550500865], [1.4767691307109896, 3.1588066514884536, -1.2409295709593886, 
-3.0391314444484419], [-4.9021751672603253, -3.9253765883598994, 7.1433726510677111, 
-8.3780817909694623]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[1.9102921294445139, -4.4466950988165843, -2.2217834148394267, -4.5698256143245555], 
[2.541724304794589, 1.3996747876411657, 1.1887004562695491, 4.426376001203753], [-1.7470892976014927, -2.5401549086033071, 
3.673376846125004, -2.1734346947360486]], [[3.8861472707573306, -2.6384847912108613, -0.18457946540359593, 
-3.8381644223779308], [0.062751052548578379, 3.4799410039417182, -3.9277798921703488, 2.4363929859578608], 
[-3.4727401917081813, 0.099094996894030452, -4.5752434096729502, 4.1598074740048716]]], [[[-4.3886211304738154, 
3.0382601651037557, -1.7839337822933587, -1.0966703665242497], [-3.7287641002147565, -1.5037198978436006, 1.3655700710080163, 
2.1639660126250782], [3.0024824591931374, 2.066232629920405, -1.3828006316610764, 0.34413193398217246]], [[2.3471951224298326, 
4.3814649326288375, -1.2989745575208991, -3.4111840494149104], [-0.12362401510263687, 0.44822573173059688, 3.8283011007681793, 
4.3552054386520851], [-0.71082063814024643, -0.47001547060213866, -1.4794564291479375, 0.55678914445402405]]], 
[[[0.93946484655154094, 1.5285543092786842, 3.3883843329386547, -1.9071358884240541], [0.55636141674973416, 
-0.15292728499392894, 0.10014465627087166, -2.4172827791656437], [4.5679145085340114, -3.9806140540609003, -1.4075703486595614, 
-2.9250402895948846]], [[-4.0407186146497942, 0.73784702953270198, -0.25255202406894561, -0.31020930688899995], 
[-0.72405208821797107, 1.7722728066723779, 2.2431330433004142, 1.7211123527396985], [1.5632891433006844, 2.3697223000301282, 
-3.9258077347781883, -0.77972256162334741]]]]),self.functionspace)
      arg1=Data(0.324211049611,self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[[2.2345031790556744, -4.1224840492054238, -1.8975723652282661, -4.2456145647133949], 
[2.8659353544057495, 1.7238858372523262, 1.5129115058807097, 4.7505870508149135], [-1.4228782479903321, -2.2159438589921465, 
3.9975878957361646, -1.849223645124888]], [[4.2103583203684911, -2.3142737415997008, 0.13963158420756461, -3.5139533727667702], 
[0.38696210215973892, 3.8041520535528788, -3.6035688425591883, 2.7606040355690213], [-3.1485291420970207, 0.42330604650519099, 
-4.2510323600617896, 4.4840185236160321]]], [[[-4.0644100808626549, 3.3624712147149163, -1.4597227326821982, 
-0.77245931691308911], [-3.4045530506035959, -1.1795088482324401, 1.6897811206191768, 2.4881770622362387], [3.3266935088042979, 
2.3904436795315656, -1.0585895820499158, 0.668342983593333]], [[2.6714061720409932, 4.7056759822399981, -0.97476350790973854, 
-3.0869729998037498], [0.20058703450852367, 0.77243678134175742, 4.1525121503793399, 4.6794164882632456], 
[-0.38660958852908589, -0.14580442099097812, -1.155245379536777, 0.88100019406518459]]], [[[1.2636758961627015, 
1.8527653588898447, 3.7125953825498152, -1.5829248388128936], [0.8805724663608947, 0.1712837646172316, 0.4243557058820322, 
-2.0930717295544832], [4.8921255581451719, -3.6564030044497398, -1.0833592990484009, -2.6008292399837241]], 
[[-3.7165075650386337, 1.0620580791438625, 0.071659025542214927, 0.014001742722160593], [-0.39984103860681053, 
2.0964838562835384, 2.5673440929115747, 2.0453234023508591], [1.8875001929118449, 2.6939333496412887, -3.6015966851670278, 
-0.45551151201218687]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[0.55138226756411779, 1.7029821737072606, -0.042731190744240699, -1.5621000959042441], 
[4.0795901058166084, -1.7346187216183875, 4.5502886269350213, -0.19722031862314271], [1.3125703649199414, 1.2855027948527473, 
-3.9418647394062067, -2.7936068078690166]], [[-3.2767056189450319, -1.3746539885335798, 0.59229306956365058, 
2.2588233557588167], [-2.7066386025275211, 1.6821061960477115, 4.081223000928226, 1.274718022190914], [2.2870064791655684, 
-1.8269875923579146, -0.2543128413238458, 3.3760228954854359]]], [[[4.315652150310882, 2.5404963812791674, -4.1078932797993586, 
2.8511392803759348], [4.076189348384764, -1.2090952983073411, -3.749215438853736, -3.6203289878277136], [-2.0242933789998965, 
4.7548698981134017, -3.0403166553452863, -0.68755154900135018]], [[-3.133290650039211, -3.6618792355211727, 
-3.3874675439497128, 1.8903992831290308], [-0.57484407930222137, -1.6791989897255744, 0.67930993929600447, 4.2873999065131567], 
[3.7343677070603718, -3.9185499674660154, -3.7776136475378852, 1.0148363035326602]]], [[[-2.2181170501574421, 
1.3373957377930896, 3.9833912125239319, 4.906092283714937], [-2.5720015746874116, 0.29076714320211039, 3.5781250989248417, 
-0.45210852934443846], [0.014438410062123097, 4.8265275114597923, -4.6891485681942626, 1.4864105203593656]], 
[[-2.6990868104914556, -0.770462805546134, -3.4281088987752373, -0.56583350393111331], [3.7548152771984142, 
-0.52786814011509176, -3.7601684531664512, -4.8132959123938388], [-1.687784732096743, -3.3693203800519402, 2.4133183928602655, 
-4.3049714187586128]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[4.3781101174572292, -2.2026155599541042, -4.6583769325041304, 0.2243696875375587], 
[-4.7972382043984236, 4.7524275121077473, 2.3908293248633408, -1.5485924008833249], [-2.8946217255545159, 1.6745263427573107, 
-1.6676206115076986, 1.7387068288870422]], [[-0.14258801579577352, 0.68135950825390257, -1.1869970506093597, 
4.5918362661520256], [2.2453799217772872, 1.8727068455251006, -4.5433975192002931, 2.1141859167813246], [1.6375668495858653, 
-1.916100439160596, 0.32970441143984086, -3.5690582686064598]]], [[[-3.3096917045379346, -3.8878864793186576, 
-1.3783861088899938, -2.0401587943757029], [2.2098084151118593, 0.89600674188485563, 3.8406016506979981, 0.33299167644640981], 
[-1.5946812086593676, 3.6822386285009241, 1.2176627889221514, -4.0240002164659181]], [[-4.199883082400417, -1.5121337743094196, 
-2.537929589459921, 0.62423896455482364], [-2.4231872169221322, 1.2910816028980809, 4.3728192011594853, 0.42538083987324171], 
[-1.6503264597449396, 2.1003571304215569, 1.4585280131471476, 4.9018748009475281]]], [[[-4.3794884391986404, 
1.1228890278700803, 2.1405066566411159, -1.5078057058610139], [-3.5623894362150521, 4.6139981777245627, -4.8618825986142165, 
2.3032167304714024], [-4.9332205016841506, 2.3219972670213256, -3.4182939818304559, 3.9788883441492491]], [[-3.668526524323247, 
3.94967569062457, 4.2649598352516467, 1.7341028201998281], [-2.4372342453766382, 0.42935160087469004, 2.989827155979059, 
0.49033888335964004], [2.2335752783212239, -1.6707659103274466, 2.8338095825173593, -4.1472857187837526]]]]),self.functionspace)
      res=arg0+arg1
      ref=Data(numpy.array([[[[4.929492385021347, -0.49963338624684361, -4.7011081232483711, -1.3377304083666854], 
[-0.71764809858181522, 3.0178087904893598, 6.9411179517983621, -1.7458127195064677], [-1.5820513606345745, 2.960029137610058, 
-5.6094853509139053, -1.0548999789819744]], [[-3.4192936347408054, -0.69329448027967722, -0.5947039810457091, 
6.8506596219108422], [-0.46125868075023391, 3.5548130415728121, -0.46217451827206713, 3.3889039389722386], [3.9245733287514337, 
-3.7430880315185107, 0.075391570115995066, -0.19303537312102392]]], [[[1.0059604457729474, -1.3473900980394902, 
-5.4862793886893524, 0.81098048600023187], [6.2859977634966233, -0.31308855642248545, 0.0913862118442621, -3.2873373113813038], 
[-3.6189745876592641, 8.4371085266143258, -1.8226538664231349, -4.7115517654672683]], [[-7.3331737324396276, 
-5.1740130098305919, -5.9253971334096338, 2.5146382476838545], [-2.9980312962243536, -0.38811738682749342, 5.0521291404554898, 
4.7127807463863984], [2.0840412473154322, -1.8181928370444584, -2.3190856343907376, 5.9167111044801883]]], 
[[[-6.5976054893560825, 2.4602847656631699, 6.1238978691650479, 3.3982865778539231], [-6.1343910109024637, 4.9047653209266731, 
-1.2837574996893748, 1.8511082011269639], [-4.9187820916220275, 7.1485247784811179, -8.107442550024718, 5.4652988645086147]], 
[[-6.3676133348147026, 3.179212885078436, 0.8368509364764094, 1.1682693162687148], [1.317581031821776, -0.098516539240401713, 
-0.77034129718739219, -4.3229570290341988], [0.54579054622448098, -5.0400862903793868, 5.2471279753776248, 
-8.4522571375423645]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[0.66124563506526712, 4.3848265050612749, -4.9138584058469723, 2.1449382225489275], 
[-1.0441869759744007, 0.869004194005222, 3.7318264955462812, 2.8599163048900698], [1.9952479544179837, 2.3827626368948849, 
3.3464724168720998, -1.6560337206186135]], [[4.0300618632394958, 1.9097873862189552, -2.8027809597116269, 4.7067012401543575], 
[-1.1246112967235256, -2.5165541578706332, -4.7141935581629006, 1.3049699982906766], [2.369369371806064, -3.3925460376205105, 
-2.6065356917168292, 3.5035546462643623]]], [[[4.3494796605887736, 2.2862718121678727, 0.27029705349310618, 
1.3550149210936144], [-0.34661098564884618, 1.8574245558265323, -2.077945027586694, -3.6244743199725802], [-4.3479059635227113, 
-3.1102719298773063, 3.4246005095181857, 0.1996317149973077]], [[1.3136731977156852, 0.16043545846047191, 2.2879311899469537, 
-1.0291715041939953], [-4.6352146089292194, 3.7610402189451211, -2.9588103575494196, 0.52266235169291875], 
[0.16960994402405039, 1.4247574350508003, 0.29921930670456653, -4.1964798679032054]]], [[[-1.3250196189250705, 
3.7238070953648137, -2.4783783409923732, 2.6958771833999675], [-3.9470992925312842, 3.3073662949356919, 3.4680310320724779, 
-1.3906324194488673], [0.26610742261248266, 2.733534084618455, -4.5750891894739709, 2.7275422279691721]], [[2.2110254435632468, 
-4.5139448750594804, -4.6261359610405535, -0.29107021383263909], [3.7477903656674485, -3.2863593885053843, 3.8331387617974269, 
-2.2782203513306531], [-3.6805161103946613, 2.4223557593453702, -4.7314963117783959, 1.7824005863369932]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.298549506664)+(1.-msk_arg1)*(3.90098406353)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.95979514172949543, 4.6833760117255032, -4.615308899182744, 2.4434877292131558], 
[-0.74563746931017238, 1.1675537006694503, 4.0303760022105095, 3.1584658115542981], [2.2937974610822121, 2.6813121435591132, 
3.6450219235363281, -1.3574842139543852]], [[4.3286113699037241, 2.2083368928831835, -2.5042314530473986, 5.0052507468185858], 
[-0.82606179005929725, -2.2180046512064049, -4.4156440514986723, 1.6035195049549049], [2.6679188784702923, -3.0939965309562822, 
-2.3079861850526009, 3.8021041529285906]]], [[[4.6480291672530019, 2.584821318832101, 0.56884656015733448, 1.6535644277578427], 
[-0.048061478984617878, 2.1559740624907606, -1.7793955209224657, -3.3259248133083519], [-4.049356456858483, -2.811722423213078, 
3.723150016182414, 0.498181221661536]], [[1.6122227043799136, 0.45898496512470022, 2.586480696611182, -0.73062199752976698], 
[-4.3366651022649911, 4.0595897256093494, -2.6602608508851913, 0.82121185835714705], [0.46815945068827869, 1.7233069417150286, 
0.59776881336879484, -3.8979303612389771]]], [[[-1.0264701122608422, 4.022356602029042, -2.1798288343281449, 
2.9944266900641958], [-3.6485497858670559, 3.6059158015999202, 3.7665805387367062, -1.092082912784639], [0.56465692927671096, 
3.0320835912826833, -4.2765396828097426, 3.0260917346334004]], [[2.5095749502274751, -4.2153953683952521, -4.3275864543763252, 
0.0074792928315892127], [4.0463398723316768, -2.987809881841156, 4.1316882684616552, -1.9796708446664248], [-3.381966603730433, 
2.7209052660095985, -4.4329468051141676, 2.0809500930012215]]]])+(1.-msk_ref)*numpy.array([[[[4.5622296985980633, 
8.2858105685940711, -1.0128743423141762, 6.0459222860817237], [2.8567970875583955, 4.7699882575380181, 7.6328105590790774, 
6.7609003684228659], [5.8962320179507799, 6.283746700427681, 7.247456480404896, 2.2449503429141826]], [[7.9310459267722919, 
5.8107714497517513, 1.0982031038211693, 8.6076853036871537], [2.7763727668092706, 1.384429905662163, -0.8132094946301045, 
5.2059540618234728], [6.2703534353388601, 0.5084380259122856, 1.2944483718159669, 7.4045387097971584]]], [[[8.2504637241215697, 
6.1872558757006688, 4.1712811170259023, 5.2559989846264106], [3.55437307788395, 5.7584086193593285, 1.8230390359461022, 
0.27650974356021596], [-0.44692189998991516, 0.79071213365548987, 7.3255845730509819, 4.1006157785301038]], 
[[5.2146572612484814, 4.0614195219932681, 6.1889152534797498, 2.8718125593388009], [-0.7342305453964233, 7.6620242824779172, 
0.94217370598337657, 4.4236464152257149], [4.0705940075568465, 5.3257414985835965, 4.2002033702373627, -0.29549580437040923]]], 
[[[2.5759644446077257, 7.6247911588976098, 1.4226057225404229, 6.5968612469327637], [-0.046115228998488078, 7.208350358468488, 
7.369015095605274, 2.5103516440839289], [4.1670914861452788, 6.6345181481512512, -0.67410512594117478, 6.6285262915019683]], 
[[6.112009507096043, -0.61296081152668425, -0.72515189750775733, 3.6099138497001571], [7.6487744292002446, 0.6146246750274118, 
7.734122825330223, 1.622763712202143], [0.22046795313813483, 6.3233398228781663, -0.83051224824559977, 5.6833846498697893]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[-0.015008800086557805, -1.2606091874203109, -0.19388627208058118, -0.21136737573221609], 
[-1.7983586844819808, -3.4422912032265893, -1.3631467124127905, 2.6437209483167665], [-1.1199106650420463, 4.5552954683369968, 
3.8342833923406197, 4.0833901040649696]], [[0.94222459455252938, 1.81764226876414, -0.56446438762617746, -4.6580360138913326], 
[2.933539014962343, -2.8878194974832394, -4.063526950099881, -3.3683657583114748], [-3.0461356186104602, 3.2860457281544058, 
-4.7845146542855366, -2.4154215513646671]]], [[[-0.82893671039571348, 3.932454335603575, -4.4670298557850643, 
-3.5968765122945054], [2.3424661471160215, 1.9634531644571283, -0.97085293547054086, 1.6019706188098777], [-2.6852585452179625, 
4.0277843733562584, -3.8338854520852959, -2.6778139131607883]], [[2.3280611266095521, 2.3851996205176409, 0.59335296037785756, 
-2.430547140808156], [2.7147882265942345, -2.8232383032001342, 4.2687255573703649, 1.227555169907224], [3.5576851022153324, 
4.1326475422408784, -2.8143218059926922, -1.4697680299614868]]], [[[2.0692006511042926, 0.58522194285438456, 
-2.7222000046739225, -4.2870853706271692], [-3.9430161652185469, -1.6740077633010086, -1.0065395953542722, 
-2.7888383913558989], [-4.332131270452547, 3.7153826734518631, 1.0653152388774521, -0.65159903673115771]], 
[[-2.6314896294401158, 0.18777408529104012, -0.32591467842877542, -4.7568869500131248], [3.1148017812340729, 
0.99069142651924924, 3.5919889888923269, -3.4331509839159748], [-3.9672809944935237, 0.022741919092831964, -1.572074853997699, 
2.9933812819790706]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-2.2926345908483228, -3.9846886777702983, -3.4574608219835623, -4.1753016331132518], 
[1.8290533042811159, 2.8613047636606384, -3.4253852510544904, 2.3158327148388436], [-0.66454489957970875, -4.9982361109462818, 
-0.39769546160133817, -0.85844101319581423]], [[-4.2079126670036384, 0.55018038785953127, 1.9138309040317916, 
-1.9344723164914801], [3.7417360277325198, 2.0885791855638978, -4.5733863147875473, -2.3534309469357373], [-1.3880775543171699, 
-2.6997780398560245, 3.736929376710707, 2.3010114899153997]]], [[[1.2235729909157333, -0.46188034399582811, 
-4.0125599440580286, 2.0132697899849266], [2.0591610353631857, -1.8206264300217478, -4.3807959480254741, 0.70390188849187751], 
[0.31712361359081154, -2.3991916741735944, 4.6733364801473005, 4.3100756283129087]], [[-0.79600966678685126, 
4.0761318531881141, -2.8578750181616197, 0.41969691400273312], [-4.3468783873016363, -3.7718501694329443, 3.2715681836782906, 
-1.8203374557862197], [-2.065462333589354, 3.07050916732506, 1.9965925588408728, -4.3448288158455082]]], [[[1.6596382424436049, 
2.5030018790609301, -3.1884651312699006, -2.4477749302968577], [1.2791984687560101, 4.9895676633397112, 0.37507186129938841, 
4.5846305341004872], [1.6172658108298963, -1.3128230194556867, 4.1963313024877085, -1.0268534721539782]], [[3.5996577705612829, 
4.2176226503239089, -4.8003313229349498, 4.8850372967337101], [0.21445619099817925, 0.00097779279357634863, 
-2.7576716520470224, -4.5252335084656341], [-2.7379709244997388, 2.0657994008819314, -1.2528533538139519, 
-2.403613612102208]]]])+(1.-msk_arg1)*numpy.array([[[[2.2619066174593172, 0.54565051955918165, -4.187413066749782, 
-1.8215689731816864], [3.9165128272470948, 2.5500505058503373, 3.8231983267508447, 1.1683735036743128], [-2.1106185855873036, 
1.8430547130034487, 2.7474354799692735, -1.2427191020543704]], [[4.5858793696654594, -1.8147712379431793, 2.2742374639180563, 
1.0994852370601658], [4.2228126621594022, -2.3503512068206858, -4.7395015347292135, 4.3532122740012138], [-2.2753199374295328, 
-4.4375736117074824, 4.2106313681064194, 3.7369939929638409]]], [[[3.593017013469062, -0.092335592851274129, 
-2.5862449116352626, -4.8343069906100498], [-0.71195551368315257, -2.6793558063914591, 2.8216380317604939, 3.5587292010352218], 
[-1.9656011022530695, 1.8216449646078834, -1.8016350967395156, 4.9528199334735437]], [[-3.4251965694520159, 
-2.6717063360862259, -2.7768539460987443, -3.018174062702351], [-1.8155128396464439, 4.8919053423136578, 1.0039568443391991, 
-3.290818879361439], [3.1917074338447229, -1.1801825439048006, 4.8210352252086821, -2.2892018250189072]]], 
[[[-1.3117441737339708, 2.2851761957654846, -3.3691798212323611, -1.9715370562438403], [-2.5444105769868486, 
3.6722272344535796, 0.015857974003360198, 2.63168404287822], [2.3785849431316031, -3.0578758120077643, 3.1376773388137753, 
-1.4332363174841101]], [[0.60634815052195723, -3.7193202812875912, 3.8799288975215767, 0.68029670057697089], 
[-4.143005311489917, -1.1899369775980375, 2.5022236776127755, 2.7397778637119741], [-4.2063880054798712, 1.300936137825321, 
-1.3288778016203739, -0.20874744775707299]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-2.3076433909348806, -5.2452978651906097, -3.6513470940641435, -4.3866690088454678], 
[0.030694619799135126, -0.58098643956595097, -4.7885319634672809, 4.9595536631556101], [-1.784455564621755, 
-0.44294064260928501, 3.4365879307392815, 3.2249490908691554]], [[-3.265688072451109, 2.3678226566236713, 1.3493665164056141, 
-6.5925083303828131], [6.6752750426948628, -0.79924031191934164, -8.6369132648874292, -5.7217967052472121], 
[-4.4342131729276302, 0.58626768829838127, -1.0475852775748296, -0.11441006144926735]]], [[[0.39463628052001987, 
3.4705739916077469, -8.4795897998430938, -1.5836067223095789], [4.4016271824792073, 0.14282673443538041, -5.351648883496015, 
2.3058725073017552], [-2.3681349316271509, 1.628592699182664, 0.83945102806200467, 1.6322617151521204]], [[1.5320514598227009, 
6.461331473705755, -2.2645220577837621, -2.0108502268054229], [-1.6320901607074019, -6.5950884726330781, 7.5402937410486555, 
-0.59278228587899573], [1.4922227686259784, 7.2031567095659383, -0.81772924715181938, -5.814596845806995]]], 
[[[3.7288388935478975, 3.0882238219153146, -5.9106651359438231, -6.7348603009240264], [-2.6638176964625369, 3.3155599000387026, 
-0.63146773405488377, 1.7957921427445882], [-2.7148654596226507, 2.4025596539961764, 5.2616465413651605, -1.6784525088851359]], 
[[0.96816814112116711, 4.405396735614949, -5.1262460013637252, 0.12815034672058534], [3.3292579722322522, 0.99166921931282559, 
0.83431733684530451, -7.9583844923816089], [-6.7052519189932624, 2.0885413199747633, -2.8249282078116509, 
0.58976766987686258]]]])+(1.-msk_ref)*numpy.array([[[[2.2468978173727594, -0.71495866786112927, -4.3812993388303632, 
-2.0329363489139025], [2.118154142765114, -0.892240697376252, 2.4600516143380542, 3.8120944519910793], [-3.2305292506293499, 
6.3983501813404455, 6.5817188723098932, 2.8406710020105992]], [[5.5281039642179888, 0.0028710308209607405, 1.7097730762918788, 
-3.5585507768311668], [7.1563516771217452, -5.2381707043039256, -8.8030284848290954, 0.98484651568973902], [-5.321455556039993, 
-1.1515278835530767, -0.57388328617911721, 1.3215724415991739]]], [[[2.7640803030733485, 3.8401187427523009, 
-7.0532747674203264, -8.4311835029045561], [1.6305106334328689, -0.71590264193433084, 1.850785096289953, 5.1606998198450995], 
[-4.650859647471032, 5.8494293379641418, -5.635520548824811, 2.2750060203127553]], [[-1.0971354428424638, -0.28650671556858498, 
-2.1835009857208867, -5.448721203510507], [0.89927538694779052, 2.0686670391135236, 5.2726824017095639, -2.063263709454215], 
[6.7493925360600553, 2.9524649983360778, 2.0067134192159899, -3.758969854980394]]], [[[0.75745647737032185, 2.8703981386198691, 
-6.0913798259062837, -6.2586224268710096], [-6.4874267422053951, 1.9982194711525709, -0.99068162135091198, 
-0.15715434847767895], [-1.9535463273209439, 0.65750686144409887, 4.2029925776912274, -2.0848353542152678]], 
[[-2.0251414789181585, -3.5315461959965511, 3.5540142190928012, -4.0765902494361539], [-1.0282035302558441, 
-0.19924555107878827, 6.0942126665051024, -0.69337312020400077], [-8.173668999973394, 1.323678056918153, -2.9009526556180729, 
2.7846338342219976]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.77357600284)+(1.-msk_arg0)*(1.64266330921)
      arg1=3.14052684391
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.633049158926)+(1.-msk_ref)*(4.78319015313)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.51938883119)+(1.-msk_arg0)*(-4.22626937896)
      arg1=numpy.array(-0.292086247699)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.81147507889)+(1.-msk_ref)*(-4.51835562666)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.67617377123)+(1.-msk_arg0)*(2.09842240224)
      arg1=numpy.array([-0.86399908641029555, -3.169295486266944])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-4.540172857635473, -6.8454692574921214])+(1.-msk_ref)*numpy.array([1.2344233158267448, 
-1.0708730840299037])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.11419409954)+(1.-msk_arg0)*(-3.3640733455)
      arg1=numpy.array([[1.1579114853662054, -1.0884901054029736, -1.9022931162131362, -4.7597757306364263, 
1.2618309946275597], [-3.2779316882230325, 0.0047554775087865764, -1.3708313331987321, -2.5640457946333903, 
4.6864431720082926], [-0.57783435752669554, 2.6053428148098421, -1.8483046469067386, 4.8755935678342688, -4.7653471495810766], 
[-0.31875048455720467, -4.1797368149653362, -3.2865198355289893, -4.5303311141056515, 2.7067080231330252]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.0437173858266382, -1.2026842049425408, -2.0164872157527034, -4.8739698301759935, 
1.1476368950879925], [-3.3921257877625997, -0.10943862203078059, -1.4850254327382992, -2.6782398941729575, 4.5722490724687255], 
[-0.6920284570662627, 2.4911487152702749, -1.9624987464463057, 4.7613994682947016, -4.8795412491206438], [-0.43294458409677183, 
-4.2939309145049034, -3.4007139350685565, -4.6445252136452186, 
2.592513923593458]])+(1.-msk_ref)*numpy.array([[-2.2061618601297459, -4.4525634508989249, -5.2663664617090875, 
-8.1238490761323767, -2.1022423508683916], [-6.6420050337189842, -3.3593178679871647, -4.7349046786946829, -5.9281191401293416, 
1.3223698265123414], [-3.9419077030226468, -0.75873053068610918, -5.2123779924026898, 1.5115202223383175, -8.1294204950770279], 
[-3.6828238300531559, -7.5438101604612875, -6.6505931810249406, -7.8944044596016028, -0.65736532236292611]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.2687652828)+(1.-msk_arg0)*(2.40982742064)
      arg1=numpy.array([[[-3.306981346160728, 0.85724429957170312], [-3.5111131381950331, -1.6497655723959612]], 
[[-3.1839541208302649, -0.79205895539857352], [2.6961636177277049, 2.5096710566943479]], [[4.4670298105676789, 
0.29785980961611092], [0.94052972920734845, -4.2570894172914322]], [[-3.0977082766668316, 0.16750597372400389], 
[0.3899609646458515, 0.44647567950377276]], [[1.3413221538067379, 2.1866311951787747], [2.6253698117638073, 
-2.4366310448294781]], [[-2.3920102290756184, 1.6159988541426662], [1.813365787026048, -0.73152685047475607]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-4.5757466289557787, -0.41152098322334707], [-4.7798784209900838, -2.9185308551910114]], 
[[-4.4527194036253146, -2.0608242381936237], [1.4273983349326547, 1.2409057738992977]], [[3.1982645277726287, 
-0.97090547317893927], [-0.32823555358770173, -5.5258547000864819]], [[-4.3664735594618822, -1.1012593090710463], 
[-0.87880431814919868, -0.82228960329127743]], [[0.072556871011687729, 0.9178659123837245], [1.3566045289687572, 
-3.7053963276245283]], [[-3.6607755118706686, 0.34723357134761601], [0.54460050423099782, 
-2.0002921332698063]]])+(1.-msk_ref)*numpy.array([[[-0.89715392552129725, 3.2670717202111339], [-1.1012857175556023, 
0.76006184824346956]], [[-0.77412670019083407, 1.6177684652408573], [5.1059910383671356, 4.9194984773337787]], 
[[6.8768572312071097, 2.7076872302555417], [3.3503571498467792, -1.8472619966520014]], [[-0.68788085602740079, 
2.5773333943634347], [2.7997883852852823, 2.8563031001432035]], [[3.7511495744461687, 4.5964586158182055], [5.0351972324032381, 
-0.026803624190047337]], [[0.01781719156381234, 4.025826274782097], [4.2231932076654788, 1.6783005701646747]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.579600629075)+(1.-msk_arg0)*(3.47733619986)
      arg1=numpy.array([[[[-4.168052935664095, 4.9295859575445817, -4.3097515243734739, -0.35165534192921211], 
[3.6996392042717048, -4.3525868862522596, 4.7014355031229922, 3.6822631378875634], [3.1270636470969428, -0.8758160445315184, 
3.5722139138728952, -1.958279329803283]], [[-3.3321814038160369, -0.34537302289869309, -3.7215544288800184, 
1.3312309125115442], [-0.21281207303718475, 0.48563075469750494, 4.7580881843265068, 3.4824148082836803], [4.8432716229297643, 
-0.12941832349017446, 3.968560495395872, 1.7361914615252854]]], [[[1.5808847452204216, 2.5894308066316229, 3.7512506484950574, 
-2.2870829792964531], [-3.5159093053932331, 3.5715269719881331, 2.6761233276128626, 4.2071038961770011], [2.8853188392500186, 
0.36196410790691758, 1.2703542310020595, 0.59954313550411342]], [[0.71981695057950024, -0.093403558168155598, 
3.630558153946648, 0.15707343333088009], [-2.0138126086161376, 3.6417786309694016, -3.1520967300766545, -0.26489295124037415], 
[4.3132496203346395, 3.8828759363172818, -1.0038774912603401, -3.8765663692259302]]], [[[0.46848243479783758, 
-4.3563804727762481, -4.6085956104080017, 1.6474781649681667], [0.39587769792179373, -4.434117448911544, 4.728714395238903, 
-2.2001223307328588], [-3.1834433094395855, 4.9594024168641671, -0.84684780700851636, -2.986520669787005]], 
[[1.3065567906057263, 4.625637766562491, 2.774065423329076, 0.61085863694888509], [-1.8252905829010926, -3.4835018411071927, 
-0.90493700281441125, -1.6951688521246187], [1.7886922045538691, -2.9877619505249942, 1.7784773687485433, 
-2.8233966003944011]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-4.7476535647388518, 4.3499853284698249, -4.8893521534482307, -0.93125597100396895], 
[3.1200385751969479, -4.9321875153270165, 4.1218348740482353, 3.1026625088128066], [2.547463018022186, -1.4554166736062752, 
2.9926132847981384, -2.5378799588780399]], [[-3.9117820328907937, -0.92497365197344994, -4.3011550579547748, 
0.75163028343678739], [-0.79241270211194159, -0.093969874377251905, 4.17848755525175, 2.9028141792089235], [4.2636709938550075, 
-0.7090189525649313, 3.3889598663211151, 1.1565908324505285]]], [[[1.0012841161456647, 2.0098301775568661, 3.1716500194203006, 
-2.8666836083712099], [-4.0955099344679899, 2.9919263429133762, 2.0965226985381058, 3.6275032671022442], [2.3057182101752618, 
-0.21763652116783927, 0.69075360192730262, 0.019942506429356577]], [[0.14021632150474339, -0.67300418724291244, 
3.0509575248718912, -0.42252719574387676], [-2.5934132376908945, 3.0621780018946447, -3.7316973591514113, -0.844493580315131], 
[3.7336489912598827, 3.3032753072425249, -1.5834781203350969, -4.4561669983006871]]], [[[-0.11111819427691927, 
-4.9359811018510049, -5.1881962394827585, 1.0678775358934098], [-0.18372293115296312, -5.0137180779863009, 4.1491137661641462, 
-2.7797229598076156], [-3.7630439385143424, 4.3798017877894102, -1.4264484360832732, -3.5661212988617619]], 
[[0.72695616153096942, 4.0460371374877342, 2.1944647942543192, 0.031258007874128246], [-2.4048912119758494, 
-4.0631024701819491, -1.4845376318891681, -2.2747694811993755], [1.2090915754791123, -3.567362579599751, 1.1988767396737865, 
-3.4029972294691579]]]])+(1.-msk_ref)*numpy.array([[[[-0.69071673580211446, 8.4069221574065622, -0.83241532451149336, 
3.1256808579327684], [7.1769754041336853, -0.87525068639027914, 8.1787717029849727, 7.1595993377495439], [6.6043998469589233, 
2.6015201553304621, 7.0495501137348757, 1.5190568700586975]], [[0.14515479604594361, 3.1319631769632874, -0.24421822901803791, 
4.8085671123735247], [3.2645241268247958, 3.9629669545594854, 8.2354243841884873, 6.9597510081456608], [8.3206078227917448, 
3.347917876371806, 7.4458966952578525, 5.2135276613872659]]], [[[5.0582209450824021, 6.0667670064936035, 7.2285868483570379, 
1.1902532205655274], [-0.038573105531252594, 7.0488631718501136, 6.1534595274748431, 7.6844400960389816], [6.3626550391119991, 
3.8393003077688981, 4.74769043086404, 4.0768793353660939]], [[4.1971531504414807, 3.3839326416938249, 7.1078943538086286, 
3.6344096331928606], [1.4635235912458429, 7.1191148308313821, 0.32523946978532603, 3.2124432486216064], [7.79058582019662, 
7.3602121361792623, 2.4734587086016404, -0.39923016936394973]]], [[[3.9458186346598181, -0.87904427291426757, 
-1.1312594105460212, 5.1248143648301472], [3.8732138977837742, -0.95678124904956352, 8.2060505951008835, 1.2772138691291217], 
[0.29389289042239497, 8.4367386167261476, 2.6304883928534641, 0.49081553007497547]], [[4.7838929904677068, 8.1029739664244715, 
6.2514016231910565, 4.0881948368108656], [1.6520456169608879, -0.0061656412452122389, 2.5723991970475693, 1.7821673477373619], 
[5.2660284044158496, 0.48957424933698634, 5.2558135686105238, 0.6539395994675794]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.86080697215)+(1.-msk_arg0)*(-3.7356334521)
      arg1=Data(4.78440444774,self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(9.64521141989)+(1.-msk_ref)*(1.04877099563)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.80033980486)+(1.-msk_arg0)*(-4.17470688541)
      arg1=Data(numpy.array([-3.9104380200434519, 3.1474219081694006]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.11009821518718921, 6.9477617130256633])+(1.-msk_ref)*numpy.array([-8.0851449054500577, 
-1.0272849772372057])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.63210372443)+(1.-msk_arg0)*(4.67720600719)
      arg1=Data(numpy.array([[1.9539318654143933, 0.3350256540175387, 1.7919865479622166, -0.98899191186229629, 
-3.0501137540753795], [1.3641925916303013, -2.3751333425125498, 4.454861198127535, 4.2309807043067096, -4.8887577391658512], 
[0.71572042134636948, 1.0194706334819088, -4.5946249060295239, 2.0203153511404102, 0.91851062251624871], [-2.8764004749273897, 
-0.8627027813531809, -4.027969614199213, 4.8789790112056366, 4.5319704112175625]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.6781718590139914, -3.297078070410846, -1.8401171764661681, -4.621095636290681, 
-6.6822174785037642], [-2.2679111327980834, -6.0072370669409345, 0.82275747369915031, 0.59887697987832489, 
-8.5208614635942368], [-2.9163833030820152, -2.6126330909464759, -8.2267286304579095, -1.6117883732879745, -2.713593101912136], 
[-6.5085041993557748, -4.4948065057815656, -7.6600733386275976, 1.2468752867772519, 
0.89986668678917781]])+(1.-msk_ref)*numpy.array([[6.6311378726064625, 5.0122316612096078, 6.4691925551542857, 
3.6882140953297728, 1.6270922531166896], [6.0413985988223704, 2.3020726646795193, 9.1320672053196041, 8.9081867114987787, 
-0.21155173197378208], [5.3929264285384386, 5.6966766406739779, 0.082581101162545245, 6.6975213583324793, 5.5957166297083178], 
[1.8008055322646794, 3.8145032258388882, 0.64923639299285618, 9.5561850183977057, 9.2091764184096316]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.19331043729)+(1.-msk_arg0)*(-2.77403359787)
      arg1=Data(numpy.array([[[-2.3129772201067378, 1.5704821103792002], [4.3844769261648651, -1.2340201505119119]], 
[[0.85970321105234326, 4.9301342659092846], [-2.4319720428853522, -4.9197633261178408]], [[-3.165767075274577, 
-1.8522948676017483], [4.2441194321368894, -0.72474616260043589]], [[-4.9145940600242009, 3.1398456044724696], 
[-4.6731627416017254, -2.9580967888448537]], [[-3.6914709195700732, -2.6817286915842189], [3.7732332807365268, 
-4.7964923160656925]], [[2.4551005332702278, 3.5416315709546513], [2.8793883559733828, 
3.8467981532926991]]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-6.5062876574000974, -2.622828326914159], [0.19116648887150589, -5.4273305878052707]], 
[[-3.3336072262410159, 0.73682382861592544], [-6.6252824801787114, -9.1130737634112009]], [[-7.3590775125679357, 
-6.0456053048951075], [0.050808994843530186, -4.9180565998937951]], [[-9.1079044973175591, -1.0534648328208895], 
[-8.8664731788950846, -7.1514072261382129]], [[-7.8847813568634324, -6.8750391288775781], [-0.42007715655683242, 
-8.9898027533590508]], [[-1.7382099040231314, -0.65167886633870786], [-1.3139220813199763, 
-0.34651228400066003]]])+(1.-msk_ref)*numpy.array([[[-5.0870108179765392, -1.2035514874906017], [1.6104433282950632, 
-4.0080537483817142]], [[-1.9143303868174586, 2.1561006680394827], [-5.206005640755154, -7.6937969239876427]], 
[[-5.9398006731443793, -4.6263284654715502], [1.4700858342670875, -3.4987797604702378]], [[-7.6886276578940027, 
0.36581200660266777], [-7.4471963394715273, -5.7321303867146556]], [[-6.4655045174398751, -5.4557622894540208], 
[0.99919968286672489, -7.5705259139354943]], [[-0.31893306459957405, 0.76759797308484945], [0.10535475810358097, 
1.0727645554228973]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.9031628484)+(1.-msk_arg0)*(-2.31696311171)
      arg1=Data(numpy.array([[[[-3.2233203179411984, -0.85963384119128783, -3.2815208607384916, -0.14102216288746483], 
[-4.8121404231409883, -4.7701007766754087, -1.9614218925506965, -3.4808732805052167], [-0.7384548504952706, 
0.52009333172349592, -3.1127087707970191, 2.7588234020765814]], [[3.9507955572007898, -1.4076271034062593, 0.42731205080210444, 
-1.2041617560257256], [2.2699722137386624, 1.1941172106651132, 1.7871173164507459, 1.8471076975583367], [-2.1803163224886779, 
2.8975118754765008, 3.1772540173933699, -2.712124250078698]]], [[[-3.3210813583241827, -0.40807244784997199, 
-3.2401637279286923, -3.6305822801271193], [4.5306948420195354, -3.4663677598150278, -4.8046936700651397, -2.7524229373010867], 
[-3.2456724065561557, 3.0761238311910049, -0.34322047334638306, 4.4975050412651854]], [[-4.1433694894124855, 
3.5679200958326369, 4.0219026386946766, -3.9333732216322614], [3.4092640708334194, -2.5195593266261085, 4.9881482741240539, 
1.3235494732447286], [-1.0474121558121574, -1.045053379361498, 0.571704943101496, 2.8368591342875966]]], [[[0.9228872367218468, 
0.96659659984305435, 0.69831487639853584, -4.96957392328593], [2.8239871973058346, 3.7344756903784493, 2.530827411086932, 
1.0969657955512044], [-4.5737398611310738, -1.1605526103818864, 0.23279120987725932, -2.3615187721980178]], 
[[-0.12299824541890025, -0.11877561932407321, 2.4017504880073384, 3.0898790287499942], [2.9280564862321565, 1.7494174904661666, 
1.8750453688537121, 4.7673162682165309], [-4.642024380226637, 2.4691539229314827, 1.0886338263264781, 
-4.2254970510176619]]]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-5.1264831663446522, -2.7627966895947411, -5.1846837091419449, -2.0441850112909181], 
[-6.7153032715444416, -6.6732636250788619, -3.8645847409541498, -5.3840361289086704], [-2.6416176988987239, 
-1.3830695166799574, -5.0158716192004729, 0.8556605536731281]], [[2.0476327087973365, -3.3107899518097126, -1.4758507976013489, 
-3.1073246044291789], [0.36680936533520914, -0.70904563773834006, -0.11604553195270739, -0.056055150845116586], 
[-4.0834791708921312, 0.99434902707304751, 1.2740911689899166, -4.6152870984821508]]], [[[-5.2242442067276365, 
-2.3112352962534253, -5.1433265763321456, -5.5337451285305725], [2.6275319936160821, -5.369530608218481, -6.707856518468593, 
-4.65558578570454], [-5.148835254959609, 1.1729609827875516, -2.2463833217498363, 2.5943421928617321]], [[-6.0465323378159388, 
1.6647572474291836, 2.1187397902912233, -5.8365360700357147], [1.5061012224299661, -4.4227221750295618, 3.0849854257206006, 
-0.57961337515872469], [-2.9505750042156107, -2.9482162277649513, -1.3314579053019573, 0.93369628588414333]]], 
[[[-0.98027561168160648, -0.93656624856039894, -1.2048479720049174, -6.8727367716893832], [0.9208243489023813, 
1.831312841974996, 0.6276645626834787, -0.80619705285224885], [-6.4769027095345271, -3.0637154587853397, -1.670371638526194, 
-4.264681620601471]], [[-2.0261610938223535, -2.0219384677275265, 0.49858763960388508, 1.1867161803465409], 
[1.0248936378287032, -0.15374535793728672, -0.028117479549741198, 2.8641534198130776], [-6.5451872286300903, 
0.56599107452802944, -0.8145290220769752, -6.1286598994211152]]]])+(1.-msk_ref)*numpy.array([[[[-5.5402834296526624, 
-3.1765969529027522, -5.598483972449956, -2.4579852745989292], [-7.1291035348524527, -7.0870638883868731, -4.2783850042621605, 
-5.7978363922166807], [-3.055417962206735, -1.7968697799879685, -5.4296718825084831, 0.44186029036511698]], 
[[1.6338324454893254, -3.7245902151177237, -1.88965106090936, -3.52112486773719], [-0.046990897972801982, -1.1228459010463512, 
-0.52984579526071851, -0.4698554141531277], [-4.4972794342001423, 0.58054876376503639, 0.86029090568190547, 
-5.0290873617901628]]], [[[-5.6380444700356467, -2.7250355595614364, -5.5571268396401567, -5.9475453918385837], 
[2.213731730308071, -5.7833308715264922, -7.1216567817766041, -5.0693860490125511], [-5.5626355182676201, 0.75916071947954045, 
-2.6601835850578475, 2.1805419295537209]], [[-6.4603326011239499, 1.2509569841211725, 1.7049395269832122, -6.2503363333437258], 
[1.092300959121955, -4.8365224383375729, 2.6711851624125895, -0.9934136384667358], [-3.3643752675236218, -3.3620164910729624, 
-1.7452581686099684, 0.51989602257613221]]], [[[-1.3940758749896176, -1.3503665118684101, -1.6186482353129286, 
-7.2865370349973944], [0.50702408559437018, 1.4175125786669849, 0.21386429937546758, -1.21999731616026], [-6.8907029728425382, 
-3.4775157220933508, -2.0841719018342051, -4.6784818839094822]], [[-2.4399613571303647, -2.4357387310355376, 
0.084787376295873962, 0.7729159170385298], [0.61109337452069212, -0.56754562124529784, -0.44191774285775232, 
2.4503531565050665], [-6.9589874919381014, 0.15219081122001832, -1.2283292853849863, -6.5424601627291263]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.34396045723)+(1.-msk_arg0)*(-0.694063060297)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.74760439837)+(1.-msk_arg1)*(-4.17782147122)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.403643941142)+(1.-msk_ref)*(-4.87188453152)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.75531961719)+(1.-msk_arg0)*(2.28522919221)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-3.0841091513861496, 0.12611014013223087])+(1.-msk_arg1)*numpy.array([-0.58249378665109219, 
-2.9462670293600288])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-4.8394287685729847, -1.6292094770546037])+(1.-msk_ref)*numpy.array([1.7027354055565347, 
-0.66103783715240194])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.12355180069)+(1.-msk_arg0)*(3.46474217121)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-1.7867175720369097, 2.9011879024487381, 2.4812058829058294, -0.31469400164402828, 
0.01609889096894257], [2.6588970294982461, -2.6481293577046392, -4.7434823094539134, 0.45184389660972357, -3.5777880424896766], 
[-1.9686574573718874, 4.7694358710993452, 3.3811106079883935, -0.42739282241222476, 1.2206840078706138], [-3.0985373142217254, 
0.27387954019956062, -3.6007529146941564, -3.1495843853332506, 
3.3950043721069125]])+(1.-msk_arg1)*numpy.array([[1.3857343960467494, 0.33969457637713596, -3.981328266461599, 
-4.0150259589699964, 0.59323318192365893], [-2.063882154187191, -2.4385095849589025, -2.9840882295104132, 4.5740383392009711, 
2.4104287933554041], [0.80991838175166553, -4.3088469516604686, -4.5823411275040957, 4.5662726225032237, 0.031749739447684], 
[-1.7676347736421216, 3.1445151580996153, 1.0993372559397665, -2.9105535803366953, -3.8349518714373412]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-5.9102693727274831, -1.2223638982418352, -1.6423459177847439, -4.4382458023346016, 
-4.1074529097216308], [-1.4646547711923272, -6.7716811583952126, -8.8670341101444876, -3.6717079040808498, -7.70133984318025], 
[-6.0922092580624607, 0.64588407040877183, -0.7424411927021799, -4.5509446231027981, -2.9028677928199595], 
[-7.2220891149122988, -3.8496722604910127, -7.7243047153847293, -7.2731361860238239, 
-0.72854742858366084]])+(1.-msk_ref)*numpy.array([[4.8504765672551793, 3.8044367475855658, -0.51658609525316912, 
-0.5502837877615665, 4.0579753531320888], [1.4008600170212389, 1.0262325862495274, 0.48065394169801667, 8.038780510409401, 
5.8751709645638339], [4.2746605529600954, -0.84410478045203874, -1.1175989562956659, 8.0310147937116536, 3.4964919106561139], 
[1.6971073975663082, 6.6092573293080452, 4.5640794271481964, 0.55418859087173455, -0.37020970022891131]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.86813402078)+(1.-msk_arg0)*(2.04694658899)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.4095149133618845, -2.6225288794272905], [-1.744034730923202, 3.7093205647220095]], 
[[0.88997306850102742, 2.4718984891257705], [-3.4160671336836024, 3.6320404517891767]], [[2.7547734667771531, 
-2.1596077043885984], [2.4005015450082237, 1.6065442866730804]], [[-2.1032549679704893, 4.7571514482344792], 
[-0.31427847633099137, 4.9308729759382626]], [[1.3072216115631585, -0.74820656593096224], [4.0400771555583574, 
2.5104018492513287]], [[-0.21763838195356744, -1.3392508121332183], [4.008229592490963, 
-0.55548277152962733]]])+(1.-msk_arg1)*numpy.array([[[4.1603180817239824, -1.65696389153997], [-0.17153489037859604, 
4.4749431711276095]], [[-3.9295971506372362, -1.9512605527965219], [-1.7112162790933683, -2.4036820382274038]], 
[[2.951964008045068, 4.4428214435118409], [-1.2137846472904501, -0.75400643888698937]], [[0.0093184204710272667, 
4.2592561509030293], [-3.4919630084516906, 3.2504256594032572]], [[2.9788843751369933, -1.7131492699598172], 
[-0.38493612776273611, 2.2667335585293618]], [[2.0096227343710646, -3.5338438628723576], [3.8535442207224939, 
3.3784750095271292]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.4586191074204349, -4.4906629002096103], [-3.6121687517055214, 1.8411865439396902]], 
[[-0.97816095228129196, 0.6037644683434511], [-5.2842011544659222, 1.7639064310068573]], [[0.88663944599483369, 
-4.0277417251709178], [0.53236752422590428, -0.26158973410923902]], [[-3.9713889887528087, 2.8890174274521598], 
[-2.1824124971133108, 3.0627389551559432]], [[-0.56091240921916086, -2.6163405867132816], [2.1719431347760381, 
0.64226782846900932]], [[-2.0857724027358868, -3.2073848329155377], [2.1400955717086436, 
-2.4236167923119467]]])+(1.-msk_ref)*numpy.array([[[6.2072646707122026, 0.38998269744825009], [1.8754116986096241, 
6.5218897601158297]], [[-1.8826505616490161, 0.095686036191698243], [0.33573030989485186, -0.35673544923918366]], 
[[4.9989105970332881, 6.4897680325000611], [0.83316194169777003, 1.2929401501012308]], [[2.0562650094592474, 
6.3062027398912495], [-1.4450164194634705, 5.2973722483914774]], [[5.0258309641252135, 0.33379731902840293], 
[1.662010461225484, 4.3136801475175819]], [[4.0565693233592848, -1.4868972738841375], [5.900490809710714, 5.4254215985153493]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.09113529158)+(1.-msk_arg0)*(-4.89355395288)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-2.2539933137624182, 4.8371547315647518, -1.1815763976754887, 1.7751664153234836], 
[-3.3508578165991931, -0.069774937972759687, -0.18238089587099271, -1.1937037523379965], [-1.8534586264952404, 
-4.9415773313400502, 4.3167582336769641, -0.54038120257833633]], [[4.778133922084475, -0.41848369162930155, 
-3.2735292503547408, 4.5589773754004845], [-0.51179437764416225, -0.793116896474098, -4.1956371512703541, -2.9769713473712365], 
[1.9430874158630376, -2.2139418526304109, 0.011495402517391007, -4.2495574018292768]]], [[[0.82734159106964711, 
-0.2963012847689086, 3.1106231633446253, 4.5307917521996615], [-2.0648894244965876, 4.8802534767732038, 1.3044139417896581, 
0.70246545786822967], [1.4264174856753797, 4.2736085557563559, -3.5829553819750126, 2.8926343729879704]], 
[[0.64544087976508369, -0.5426297721559914, -2.9568275026783564, 4.5531580617801666], [4.5426290789502204, -4.4972081776459927, 
-4.1013609224948304, 1.1763606168861527], [4.5589847937387589, 3.8952892022733447, 2.6782022668570695, -0.75970112765390585]]], 
[[[-0.26462875478974013, 0.3870773162815162, 0.995805879541801, 2.4669081638625903], [3.7881340064871765, -2.5121040604531428, 
3.5099906566656855, 3.6286174128357889], [3.9756952143731326, 0.054217225376059552, 3.0287172454706699, -0.63466066854998004]], 
[[3.6334907224594897, -3.2205686293284219, 0.80482185198078682, 0.21074147313334635], [-0.013754424932663767, 
1.5157048359899612, -3.5972864920742955, 3.0962475373021476], [1.7108462453659801, -2.9061712877224153, 2.8552316648868352, 
-3.86378615737206]]]])+(1.-msk_arg1)*numpy.array([[[[2.2721241315547509, 3.7590738716949268, -0.59524590237071884, 
1.9076096133160299], [-0.68870890758921632, 0.59018215214565206, -0.58784011080067788, 1.1160621017630223], 
[4.9675670882575371, 0.25436070172188519, -4.1511254648376861, -3.1255474453817489]], [[-4.9730774034318381, 
2.3127687669707626, -0.29387729188960066, -0.24937207747443502], [-2.1088020399286487, -2.1627157475427539, 
-2.4485838347519562, 0.56471540043464419], [-2.3367044762709046, 0.037251998721154145, -0.78869679422456329, 
-3.41402655127541]]], [[[-1.8079154286245815, 0.84402055626579742, -3.0958252781103326, -0.65474888212519033], 
[-4.766970621467113, 0.53702391321521326, -3.1209034938881963, 2.8667575561631784], [2.7244598938839948, 0.72815318451032951, 
2.3158739631788752, -0.43374009527182089]], [[0.95421484092870834, 3.1840382710671769, 4.879386352000278, 
-0.76634418058467624], [-4.7148841533909733, 1.2584296194206255, 0.81416136170276232, 4.4346978638231889], [2.066476427941998, 
0.4973490684407631, 2.6247053957713105, -4.2077994416762419]]], [[[0.0038169593221013542, -0.19309834029819051, 
-0.46116729111804666, 2.1211925432498235], [-1.9423389496891605, 3.9503278202496954, -1.0942453055974113, 
-0.34523990072589417], [3.4193772673568361, 1.735381184811164, 3.4795888835696527, 1.7418798685920152]], [[-1.7336466816826324, 
0.7931684660649454, 3.4663615897367563, 0.68654744670036116], [1.7619613438045327, 4.4072154543160131, 0.51645658843172093, 
2.5729429023395856], [1.7617842093850165, -4.7035539233518806, 4.0337071131218245, 3.6239156852601955]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.83714197782072475, 7.9282900231478948, 1.9095588939076542, 4.8663017069066266], 
[-0.2597225250160502, 3.0213603536103832, 2.9087543957121502, 1.8974315392451464], [1.2376766650879025, -1.8504420397569072, 
7.407893525260107, 2.5507540890048066]], [[7.8692692136676179, 2.6726515999538414, -0.18239395877159792, 7.6501126669836275], 
[2.5793409139389807, 2.2980183951090449, -1.1045018596872112, 0.11416394421190645], [5.0342227074461805, 0.87719343895273205, 
3.1026306941005339, -1.1584221102461338]]], [[[3.91847688265279, 2.7948340068142343, 6.2017584549277682, 7.6219270437828044], 
[1.0262458670865553, 7.9713887683563467, 4.395549233372801, 3.7936007494513726], [4.5175527772585227, 7.3647438473394988, 
-0.49182009039186969, 5.9837696645711134]], [[3.7365761713482266, 2.5485055194271515, 0.13430778890478656, 7.6442933533633095], 
[7.6337643705333633, -1.4060728860628497, -1.0102256309116875, 4.2674959084692956], [7.6501200853219018, 6.9864244938564877, 
5.7693375584402125, 2.3314341639292371]]], [[[2.8265065367934028, 3.4782126078646591, 4.0869411711249439, 5.5580434554457332], 
[6.8792692980703194, 0.5790312311300001, 6.6011259482488285, 6.7197527044189318], [7.0668305059562755, 3.1453525169592025, 
6.1198525370538128, 2.4564746230331629]], [[6.7246260140426326, -0.12943333774527899, 3.8959571435639297, 3.3018767647164893], 
[3.0773808666504792, 4.6068401275731041, -0.50615120049115259, 6.1873828288852906], [4.801981536949123, 0.18496400386072764, 
5.9463669564699781, -0.77265086578891706]]]])+(1.-msk_ref)*numpy.array([[[[-2.6214298213298974, -1.1344800811897215, 
-5.4887998552553672, -2.9859443395686185], [-5.5822628604738647, -4.3033718007389963, -5.4813940636853262, -3.777491851121626], 
[0.074013135372888783, -4.6391932511627632, -9.0446794177223353, -8.0191013982663968]], [[-9.8666313563164856, 
-2.5807851859138857, -5.187431244774249, -5.1429260303590834], [-7.002355992813297, -7.0562697004274018, -7.3421377876366041, 
-4.3288385524500042], [-7.2302584291555529, -4.8563019541634942, -5.6822507471092116, -8.3075805041600574]]], 
[[[-6.7014693815092299, -4.0495333966188509, -7.9893792309949809, -5.5483028350098387], [-9.6605245743517614, 
-4.3565300396694351, -8.0144574467728447, -2.0267963967214699], [-2.1690940590006536, -4.1654007683743188, -2.5776799897057732, 
-5.3272940481564692]], [[-3.93933911195594, -1.7095156818174715, -0.014167600884370302, -5.6598981334693246], 
[-9.6084381062756208, -3.6351243334640229, -4.079392591181886, -0.45885608906145947], [-2.8270775249426503, 
-4.3962048844438852, -2.2688485571133379, -9.1013533945608902]]], [[[-4.889736993562547, -5.0866522931828388, 
-5.354721244002695, -2.7723614096348248], [-6.8358929025738089, -0.94322613263495292, -5.9877992584820596, 
-5.2387938536105425], [-1.4741766855278122, -3.1581727680734843, -1.4139650693149957, -3.1516740842926332]], 
[[-6.6272006345672807, -4.1003854868197029, -1.427192363147892, -4.2070065061842872], [-3.1315926090801156, 
-0.4863384985686352, -4.3770973644529274, -2.3206110505450628], [-3.1317697434996319, -9.5971078762365281, 
-0.85984683976282383, -1.2696382676244529]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-3.1873260068906992, 1.1633553412205764])+(1.-msk_arg0)*numpy.array([-1.8329419918289691, 
-1.4193826797300324])
      arg1=-3.77489703645
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-6.9622230433418526, -2.611541695230577])+(1.-msk_ref)*numpy.array([-5.6078390282801225, 
-5.1942797161811853])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([4.1195321956694784, -0.85145901815053548])+(1.-msk_arg0)*numpy.array([-0.11028941818547899, 
3.1927298150410213])
      arg1=numpy.array(-4.82301253105)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.70348033537578392, -5.6744715491957978])+(1.-msk_ref)*numpy.array([-4.9333019492307413, 
-1.630282716004241])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([2.7249092636013827, -1.6973455806257363])+(1.-msk_arg0)*numpy.array([-3.4087370471785752, 
4.2892237297073859])
      arg1=numpy.array([3.9032332867472306, -1.2176629471895306])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([6.6281425503486133, -2.9150085278152669])+(1.-msk_ref)*numpy.array([0.4944962395686554, 
3.0715607825178552])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.67144579499171986, 3.6896414234755213])+(1.-msk_arg0)*numpy.array([-0.55184443462797894, 
-3.4755434457663248])
      arg1=Data(-1.80985343497,self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-2.4812992299567673, 1.8797879885104738])+(1.-msk_ref)*numpy.array([-2.3616978695930264, 
-5.2853968807313727])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.94050717497608716, -3.5101871565874565])+(1.-msk_arg0)*numpy.array([-4.1824270460905328, 
-4.400177987347746])
      arg1=Data(numpy.array([3.6443795706846611, -4.5258951283033975]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.703872395708574, -8.0360822848908544])+(1.-msk_ref)*numpy.array([-0.53804747540587172, 
-8.9260731156511426])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-2.0156565783128402, -0.091897704700659943])+(1.-msk_arg0)*numpy.array([-1.7072243218490533, 
-2.5672213663365042])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.410820185641)+(1.-msk_arg1)*(-3.03186970823)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.6048363926714053, 0.31892248094077491])+(1.-msk_ref)*numpy.array([-4.7390940300810991, 
-5.59909107456855])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.7000394438671051, 2.2359973232602757])+(1.-msk_arg0)*numpy.array([-4.5993964819793467, 
2.0728228841238492])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([4.0446212927483245, -3.5330676956204932])+(1.-msk_arg1)*numpy.array([2.7544059138309409, 
3.0830130374550215])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([7.7446607366154296, -1.2970703723602175])+(1.-msk_ref)*numpy.array([-1.8449905681484058, 
5.1558359215788707])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.32806933693237195, 1.5130331996337842, -2.3046908456563364, -0.80343261518357068, 
1.2744470164366293], [3.8188846343298053, -2.0036349716085389, -1.43270804409494, -3.2073053192958936, -3.7552852965662931], 
[-0.39531209654608368, 3.3441828247929237, -0.0092233823264979975, 1.9300400139553195, -4.1332403801477611], 
[4.7655807355294417, 3.483515155049151, 0.40755960867025376, 1.2867857831083178, 
-3.9384010928996482]])+(1.-msk_arg0)*numpy.array([[-1.9640180128306985, 4.9593327029761589, 4.9700465143748058, 
2.5169865433158716, -2.9518154439650734], [2.071112293886948, -1.1747464968274524, 0.84567948756957367, -4.9814179281616324, 
-4.6436847922594815], [2.0822147162599913, 0.56770638343125679, -3.7865506053934181, 3.6270169687200458, 1.5371621085129474], 
[-0.96334278552165031, 2.0791565098285734, -0.49433221743792366, 1.8200243142652184, -3.4946427433763541]])
      arg1=-4.14868473119
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-3.8206153942566168, -2.6356515315552045, -6.4533755768453247, -4.9521173463725594, 
-2.8742377147523595], [-0.32980009685918343, -6.1523197027975272, -5.5813927752839287, -7.3559900504848823, 
-7.9039700277552818], [-4.5439968277350724, -0.804501906396065, -4.1579081135154867, -2.2186447172336692, -8.2819251113367507], 
[0.616896004340453, -0.66516957613983774, -3.741125122518735, -2.8618989480806709, 
-8.0870858240886374]])+(1.-msk_ref)*numpy.array([[-6.1127027440196873, 0.81064797178717019, 0.8213617831858171, 
-1.6316981878731172, -7.1005001751540622], [-2.0775724373020408, -5.3234312280164406, -3.3030052436194151, -9.1301026593506212, 
-8.7923695234484711], [-2.0664700149289974, -3.5809783477577319, -7.9352353365824069, -0.52166776246894297, 
-2.6115226226760413], [-5.112027516710639, -2.0695282213604154, -4.6430169486269124, -2.3286604169237703, -7.6433274745653428]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-4.2337356711704555, 2.6098263244128255, -2.4219191806152707, 0.74439392857480513, 
1.100310477037393], [3.407858147281976, -3.4244465381809586, -3.8536385771982848, 2.1145963597672512, -3.6806135786692531], 
[-1.3169779181011929, -2.3054321599411356, 1.5486304281396581, 3.398112005867695, -1.7785844225702929], [-3.4608580016661437, 
0.056586253359318661, -0.92921030080123757, 2.3858251261161687, 
4.0373794823604907]])+(1.-msk_arg0)*numpy.array([[-0.30546964940917221, 4.3302969435506249, -3.3276424385452472, 
-1.5840821446270636, 3.3522504073921855], [3.281061694144821, 1.0217320599288824, -2.1987942168731922, 0.018628773988359271, 
-1.8640885017852993], [-0.27557827953350067, 4.8886368681971852, -1.2069407011518676, -3.03219519630726, -2.1039034131710932], 
[3.0176051619248412, 2.1302206202986369, 4.0435352582056634, -1.8845822805194592, -3.6993349031858336]])
      arg1=numpy.array(3.35060187078)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.88313380039459943, 5.9604281951886815, 0.92868269016058536, 4.0949957993506612, 
4.4509123478132491], [6.758460018057832, -0.073844667405102538, -0.50303670642242881, 5.4651982305431073, 
-0.33001170789339707], [2.0336239526746631, 1.0451697108347204, 4.8992322989155141, 6.748713876643551, 1.5720174482055631], 
[-0.11025613089028763, 3.4071881241351747, 2.4213915699746185, 5.7364269968920247, 
7.3879813531363467]])+(1.-msk_ref)*numpy.array([[3.0451322213666838, 7.6808988143264809, 0.022959432230608812, 
1.7665197261487924, 6.7028522781680415], [6.631663564920677, 4.3723339307047384, 1.1518076539026638, 3.3692306447642153, 
1.4865133689905567], [3.0750235912423554, 8.2392387389730413, 2.1436611696239884, 0.31840667446859605, 1.2466984576047628], 
[6.3682070327006972, 5.4808224910744929, 7.3941371289815194, 1.4660195902563968, -0.34873303240997755]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.8277851379613761, -1.109322667318998, -1.9650446658777487, 2.5823778564934656, 
-2.7298622131416228], [-2.4173631158005326, -4.451977440854912, -1.8728286847237752, -1.1564292268189815, -1.7098587624369985], 
[-3.6601591106826903, -3.7732500561535764, 4.2708675917654997, -1.4317565424382206, -3.6570942022729502], [-1.2212576145342799, 
-0.98345043656539932, -3.0020190615926792, 2.5664250391844678, 
3.7796322232995649]])+(1.-msk_arg0)*numpy.array([[2.1937899638139751, -1.063049509610817, -3.1982984984559462, 
-2.761991825184432, -1.6119818838873412], [-0.12347419050207797, -1.1748510753194106, -3.1665704141784321, -2.8039866178048936, 
0.99580850900628537], [-1.0125867316065449, -2.2071997922428621, -0.71826502139348936, -1.008462546942873, 4.3993222102913467], 
[1.5487003664652033, 0.3205593390948156, 2.3145043436530157, 2.3581065566014949, 4.0912430719279183]])
      arg1=numpy.array([[-2.8949983309031846, -0.71303585744768938, 4.5470060161606582, 4.2600985359160077, 
0.32412342505920222], [-2.4151384546544854, -3.0516461165970723, -0.81630204895407932, 4.6065905514615384, 
-3.9974212107427665], [-0.9696604186677602, 0.62259652990791992, -1.3921441432508299, -2.0724202648555536, 
-3.5846162478281562], [-1.2917502158898375, 3.4925102007050555, 1.5786913526000648, 3.6407249762901355, 4.0130959577500782]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.067213192941808497, -1.8223585247666874, 2.5819613502829095, 6.8424763924094734, 
-2.4057387880824206], [-4.832501570455018, -7.5036235574519843, -2.6891307336778545, 3.4501613246425569, -5.707279973179765], 
[-4.6298195293504509, -3.1506535262456565, 2.8787234485146698, -3.5041768072937742, -7.241710450101106], [-2.5130078304241175, 
2.5090597641396561, -1.4233277089926144, 6.2071500154746033, 
7.7927281810496432]])+(1.-msk_ref)*numpy.array([[-0.70120836708920953, -1.7760853670585064, 1.3487075177047121, 
1.4981067107315758, -1.287858458828139], [-2.5386126451565634, -4.2264971919164829, -3.9828724631325114, 1.8026039336566448, 
-3.0016127017364811], [-1.9822471502743051, -1.5846032623349422, -2.1104091646443193, -3.0808828117984266, 
0.81470596246319049], [0.25695015057536574, 3.8130695397998711, 3.8931956962530805, 5.9988315328916304, 8.1043390296779965]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-4.3165389516160833, -0.46772960119889184, 0.85545188683622175, 0.79972503404946238, 
0.42255165049544985], [2.6937264740429505, -3.4476640533593486, 2.7648660285526407, 1.2220382109828698, -0.88625223090414984], 
[-4.8124424103595551, 2.5789795649916556, 1.8318767735101762, -4.6727177756191915, -0.16892281438321177], [4.3771498577317374, 
-0.91012963038350847, 4.1304518405116255, 3.0399310296303863, 
1.7006429897662123]])+(1.-msk_arg0)*numpy.array([[-0.43441945421797268, 2.2490714383065473, 0.83490064697900301, 
-2.166931160906743, -4.5986038928028208], [-0.42028565010161856, -2.2764595459612904, 1.0004492188790106, -1.0941356548789738, 
-0.03337521844491409], [-2.5347128309303866, 2.9558329101956238, -1.0843101084448561, 1.6710971232952758, 
-0.95141170836013522], [0.45569461751809737, 1.0910230009031885, -2.9791360961313473, -2.426385026511432, -4.9960353780460487]])
      arg1=Data(-3.02462883447,self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-7.3411677860854763, -3.4923584356682849, -2.1691769476331713, -2.2249038004199306, 
-2.6020771839739432], [-0.3309023604264425, -6.4722928878287416, -0.25976280591675227, -1.8025906234865232, 
-3.9108810653735429], [-7.8370712448289481, -0.44564926947773742, -1.1927520609592168, -7.6973466100885846, 
-3.1935516488526048], [1.3525210232623444, -3.9347584648529015, 1.1058230060422325, 0.015302195160993293, 
-1.3239858447031807]])+(1.-msk_ref)*numpy.array([[-3.4590482886873657, -0.77555739616284569, -2.18972818749039, 
-5.191559995376136, -7.6232327272722138], [-3.4449144845710116, -5.3010883804306834, -2.0241796155903824, -4.1187644893483668, 
-3.0580040529143071], [-5.5593416653997796, -0.068795924273769238, -4.1089389429142491, -1.3535317111741172, 
-3.9760405428295282], [-2.5689342169512956, -1.9336058335662045, -6.0037649306007399, -5.451013860980825, -8.0206642125154417]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-1.3009022796022229, 2.9683968641618543, 4.0222212364328573, 3.1934271350045655, 
3.737630079833643], [-3.8211711025113972, 3.4500132537012824, 2.4733456126328228, -2.963403700283699, 4.3155337207407101], 
[-1.9471688450861659, -2.2473458164693341, 4.9728407656257918, 0.74461353319177004, -2.0929558749748578], 
[0.016524066466243603, 1.1748735455969666, -0.1468539387455472, 2.6638438842471572, 
3.0569980737473905]])+(1.-msk_arg0)*numpy.array([[2.4641657383337892, -0.4784747560405993, -2.0528142976684496, 
3.5814967912408129, 0.013116744870716346], [1.7166097467120922, -3.3876839311247675, 1.3782775389661737, 0.55294354062860851, 
4.21560999032166], [-1.3932501698703703, 4.9110378397695325, -3.3134461993163864, 2.1618626687510805, 2.81887410027546], 
[-3.0816621319156479, 4.4496924465881769, 3.9190689132849386, -0.16278818875940271, 3.2492001872903735]])
      arg1=Data(numpy.array([[-1.384952095920732, 3.840511723749561, 4.6926399983066176, -2.8118459478177882, 
1.3072162878441702], [0.086856129579560459, -3.7377909166420298, 1.7867557976472721, 1.9982041616533506, -3.1695988449949919], 
[2.4572311328532166, 4.1637175965424564, -1.6547423844152167, -2.3678159678571919, -2.0718723949542426], [0.81444736279038921, 
-1.0538206930005236, -2.0555414337736799, -4.2532276139997123, -3.5041130049619804]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.6858543755229549, 6.8089085879114153, 8.7148612347394749, 0.38158118718677736, 
5.0448463676778132], [-3.7343149729318368, -0.28777766294074736, 4.2601014102800949, -0.96519953863034846, 1.1459348757457182], 
[0.51006228776705065, 1.9163717800731224, 3.3180983812105751, -1.6232024346654219, -4.1648282699291004], [0.83097142925663281, 
0.12105285259644294, -2.2023953725192271, -1.589383729752555, 
-0.44711493121458989]])+(1.-msk_ref)*numpy.array([[1.0792136424130572, 3.3620369677089617, 2.639825700638168, 
0.7696508434230247, 1.3203330327148866], [1.8034658762916527, -7.1254748477667977, 3.1650333366134458, 2.5511477022819591, 
1.0460111453266681], [1.0639809629828463, 9.0747554363119889, -4.9681885837316031, -0.20595329910611149, 0.74700170532121746], 
[-2.2672147691252587, 3.3958717535876533, 1.8635274795112586, -4.416015802759115, -0.25491281767160689]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.4974791969899446, 2.6595703978273839, -1.9871160084701343, 2.3694482857400958, 
-1.0196243003460781], [4.8929327773405067, -0.93867214051109737, 4.1507151547874539, 2.7985377155220883, 2.9497877694922199], 
[2.9854578851825053, 3.8837554455506424, 3.7839856257787332, -2.7691554206067659, -0.78909944411218547], [3.7156660208830274, 
-2.9606484499674091, 4.8775147941253945, -1.4895380751804699, 
-2.3126228526724812]])+(1.-msk_arg0)*numpy.array([[-3.5441437546346153, -3.7013589173746455, 0.87890677123764149, 
4.1378898408373566, 0.73369580055581274], [-3.2008002856860518, 0.76734030170569145, -1.2539100475273468, 2.6023594964040209, 
1.8761136161627974], [-1.9143625874822079, 4.5128849304489247, 1.0203673057888141, 1.0854045946282067, 4.6658196092828614], 
[4.8076547348254763, 1.9233294789388689, 0.097342037022688288, 0.73293267687869701, -4.3034701645543896]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.03831555999)+(1.-msk_arg1)*(3.08859603023)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[4.5357947569817645, 4.6978859578192038, 0.051199551521685649, 4.4077638457319157, 
1.0186912596457418], [6.9312483373323266, 1.0996434194807225, 6.1890307147792738, 4.8368532755139082, 4.9881033294840398], 
[5.0237734451743252, 5.9220710055424624, 5.8223011857705531, -0.73083986061494599, 1.2492161158796344], [5.7539815808748473, 
-0.92233288997558915, 6.9158303541172144, 0.54877748481135002, 
-0.27430729268066134]])+(1.-msk_ref)*numpy.array([[-0.45554772440700386, -0.61276288714703409, 3.9675028014652529, 
7.226485871064968, 3.8222918307834242], [-0.11220425545844037, 3.8559363319333029, 1.8346859827002646, 5.6909555266316323, 
4.9647096463904088], [1.1742334427454035, 7.6014809606765361, 4.1089633360164255, 4.1740006248558181, 7.7544156395104729], 
[7.8962507650530878, 5.0119255091664803, 3.1859380672502997, 3.8215287071063084, -1.2148741343267782]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[4.3276607610539415, 4.5484775581744206, -0.46799888469443474, 4.2055347386375725, 
-3.4210548185235021], [0.60551066431539091, 2.9749053176466678, -3.1432797079721486, 4.0801756545701657, -1.8013770809908944], 
[-0.055437294996240283, 2.1646703029790455, 4.4894960107812736, 1.1799193792106148, -4.2900388135393355], [2.6658591775547684, 
-1.0523474760652398, -2.7445839757089718, 3.8596929934355035, 
4.4433141629734223]])+(1.-msk_arg0)*numpy.array([[-2.046523423215898, 3.4773392483326031, 3.2515445714304203, 
-3.2008049698837535, -4.53106369748925], [4.8450140567807107, 4.0529431182657678, -0.58161008356276689, -0.23139522140941438, 
1.6522673651478499], [1.7531399563694103, 0.75036247433896719, 2.4873806574188242, -3.7896384621056303, 3.7207014790226758], 
[4.9977514370767224, -0.33939591596100627, 2.9222226984008053, 2.0617896938901428, 0.94369973525117068]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[3.0040611064810179, 4.8975632756395449, -3.5703981932079412, -2.9809297729897501, 
0.027918974392060036], [-4.8672452261744965, 2.5924160813148163, -2.94948932726275, -3.1494936740525636, -1.9821190704367853], 
[-2.5068801625276449, -3.4808500707620924, 1.9371186033872947, 1.6551569427883344, 1.6550025314915038], [4.8075388259313847, 
3.1516483818152423, 4.7420292337655798, -4.3870160472815511, 
3.9644319554159502]])+(1.-msk_arg1)*numpy.array([[2.3314474802412422, -0.57998508091612244, -0.82496002496911824, 
-0.74317691548838383, 2.0993623071028713], [-0.61698282910434443, 0.98224209171217236, -1.119701987745426, 4.5094706687451431, 
2.3905297820721376], [-1.0116088171098516, -3.6666073762333653, -3.928544223512922, -2.5001390145688349, 3.4355783667328605], 
[3.8663708040197342, -2.3759650672739054, 1.9350845682187687, 2.4676993791377599, -3.5408917720942599]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[7.3317218675349594, 9.4460408338139654, -4.0383970779023759, 1.2246049656478224, 
-3.3931358441314421], [-4.2617345618591056, 5.567321398961484, -6.0927690352348982, 0.93068198051760209, -3.7834961514276797], 
[-2.5623174575238852, -1.3161797677830469, 6.4266146141685683, 2.8350763219989492, -2.6350362820478317], [7.4733980034861531, 
2.0993009057500025, 1.997445258056608, -0.52732305384604761, 
8.4077461183893725]])+(1.-msk_ref)*numpy.array([[0.28492405702534418, 2.8973541674164807, 2.4265845464613021, 
-3.9439818853721373, -2.4317013903863787], [4.2280312276763663, 5.0351852099779402, -1.7013120713081928, 4.2780754473357288, 
4.0427971472199875], [0.74153113925955871, -2.9162449018943981, -1.4411635660940978, -6.2897774766744652, 7.1562798457555363], 
[8.8641222410964566, -2.7153609832349117, 4.857307266619574, 4.5294890730279027, -2.5971920368430892]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.86244483507971381, -2.2267140307718556], [-2.8159614601160587, 4.7395636816015578]], 
[[4.6446274549636151, 0.45351194514949267], [1.243736758186972, 3.3877009554942514]], [[0.69716518812614225, 
-4.8571199502314517], [-4.0184751281732147, 0.78694148653011986]], [[-0.72963931752676814, 3.8096168102916934], 
[-4.6022311665471962, -3.4734850074671089]], [[0.49954101209058521, -1.847582783241287], [-0.93916371772736351, 
-3.6692037460946136]], [[-3.5105224820170977, 3.7585721804932071], [-3.5723951282155628, 
-2.5203550180117351]]])+(1.-msk_arg0)*numpy.array([[[-3.519806162014393, 2.4721315319549664], [3.5583458425894481, 
1.5013862667399529]], [[0.74887976610109774, 2.6234922095771429], [-4.0726234063172573, 0.96083252184489609]], 
[[3.8458346816690909, 1.3295334243176491], [1.8670689306636916, 4.2273966528483449]], [[4.6281662522267926, 4.342383214924455], 
[2.6967660507673337, 4.6659368987128254]], [[2.3174285879487089, -4.7088742917740536], [-1.1859835949059971, 
0.55969574412768974]], [[3.5050740541354912, -1.4731945597925988], [-0.0082405762754387268, -2.7984629629119162]]])
      arg1=0.306468069186
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.55597676589418832, -1.9202459615863301], [-2.5094933909305333, 5.0460317507870833]], 
[[4.9510955241491406, 0.75998001433501816], [1.5502048273724975, 3.6941690246797769]], [[1.0036332573116677, 
-4.5506518810459262], [-3.7120070589876892, 1.0934095557156454]], [[-0.42317124834124265, 4.1160848794772189], 
[-4.2957630973616707, -3.1670169382815834]], [[0.80600908127611071, -1.5411147140557615], [-0.63269564854183802, 
-3.3627356769090881]], [[-3.2040544128315722, 4.0650402496787326], [-3.2659270590300373, 
-2.2138869488262096]]])+(1.-msk_ref)*numpy.array([[[-3.2133380928288675, 2.7785996011404919], [3.8648139117749736, 
1.8078543359254784]], [[1.0553478352866232, 2.9299602787626684], [-3.7661553371317318, 1.2673005910304216]], 
[[4.1523027508546164, 1.6360014935031746], [2.1735369998492171, 4.5338647220338704]], [[4.9346343214123181, 
4.6488512841099805], [3.0032341199528592, 4.9724049678983508]], [[2.6238966571342344, -4.4024062225885281], 
[-0.87951552572047165, 0.86616381331321524]], [[3.8115421233210167, -1.1667264906070733], [0.29822749291008677, 
-2.4919948937263907]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[1.4688427513388982, 3.9404388213485646], [-0.077802390014852563, -2.3494331127543866]], 
[[2.6291387160897806, 3.3616369471105827], [-1.5142654971566474, 1.415295423010785]], [[3.4484995629568633, 
-3.6402602102363826], [4.1608825074427038, -0.3231760856613155]], [[-4.9940460031975897, -2.8609355349328167], 
[2.4993564202628793, -2.4637135687896694]], [[-2.0041978820576989, -2.3171161836015197], [2.4569361472796025, 
-2.6544331313451353]], [[1.0446740012511757, 2.2216686368085936], [-4.4343566164472792, 
-3.8463684819223651]]])+(1.-msk_arg0)*numpy.array([[[-1.1843124387929951, 4.5485788632371147], [-2.3641990169273983, 
-1.1396835827486642]], [[2.5026587924456987, -0.30673142985863322], [-0.5947284963246684, 0.44812313039136598]], 
[[-3.2836376228330244, -0.08281084532108185], [-1.9909932588363111, -3.8656939050940253]], [[4.2474852071772879, 
-4.2355952144305515], [1.7985896564684287, 0.36898523191013766]], [[4.219699841715979, -2.5919848200814046], 
[0.3235360405598815, -0.20462565070506589]], [[4.5322563814063681, 3.8617970313972467], [-0.83191526776574953, 
-0.33306806817667578]]])
      arg1=numpy.array(-2.12697695954)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.65813420819788426, 1.8134618618117821], [-2.204779349551635, -4.4764100722911691]], 
[[0.50216175655299811, 1.2346599875738002], [-3.6412424566934298, -0.71168153652599742]], [[1.3215226034200809, 
-5.767237169773165], [2.0339055479059214, -2.450153045198098]], [[-7.1210229627343722, -4.9879124944695992], 
[0.37237946072609684, -4.5906905283264514]], [[-4.1311748415944809, -4.4440931431383017], [0.32995918774282007, 
-4.7814100908819182]], [[-1.0823029582856067, 0.094691677271811159], [-6.5613335759840616, 
-5.9733454414591476]]])+(1.-msk_ref)*numpy.array([[[-3.3112893983297775, 2.4216019037003322], [-4.4911759764641808, 
-3.2666605422854467]], [[0.37568183290891621, -2.4337083893954157], [-2.7217054558614509, -1.6788538291454165]], 
[[-5.4106145823698064, -2.2097878048578643], [-4.1179702183730935, -5.9926708646308082]], [[2.1205082476405055, 
-6.362572173967334], [-0.32838730306835373, -1.7579917276266448]], [[2.0927228821791966, -4.7189617796181871], 
[-1.803440918976901, -2.3316026102418483]], [[2.4052794218695857, 1.7348200718604643], [-2.958892227302532, 
-2.4600450277134582]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.19057032028804954, 4.4591537357012623], [-3.4450862953866612, -3.4286282497314078]], 
[[4.7360859248888882, 3.8601644530587844], [1.1706068280768376, -0.52909676758131141]], [[3.0951656560827132, 
0.30177660537621254], [-3.4332552977213018, 1.7633774660626269]], [[-2.3537543014188254, 3.1118988709114763], 
[4.6998295832750259, -0.26468636684630376]], [[2.2093626012064433, 1.4774442768919265], [-1.2218989053799598, 
2.2303385124443516]], [[-4.0058861374169759, -3.9815694332594012], [1.9771219573439982, 
-3.8781878804268102]]])+(1.-msk_arg0)*numpy.array([[[-2.9683259910653437, 1.919624128003238], [0.28233834079128606, 
1.4466477091653314]], [[-0.29074499980647683, -3.385176651893044], [2.077734230437482, 1.3978926420268136]], 
[[3.0570607917255632, 3.4681923212286829], [-3.7353386810235589, -4.7221816572526247]], [[-4.0406185186338419, 
-2.699050775631413], [-4.9582929700851572, -1.3427330589973954]], [[-4.5827587032177188, -4.3388198809201697], 
[1.1307869715622418, 0.83359975706952838]], [[-2.9842719563150211, -3.2649789611769622], [-3.0388219140082473, 
-3.1316927245230994]]])
      arg1=numpy.array([[[0.07658579762323825, 0.36569874685786097], [1.2271461808337634, 3.6116003051906702]], 
[[0.45921667324872661, 0.063795078538618988], [-3.7912926403986535, 1.28076202248561]], [[-1.5991558335278788, 
-3.3339043005618461], [2.1716759210094336, -2.70519241547033]], [[2.7994001214711925, -3.07859075389716], 
[-0.45352737111122643, -1.8727120731532612]], [[-3.3519663915794498, 0.50327439248916761], [-1.6789754748333396, 
0.40466153919677428]], [[-3.7147396155455983, -4.6080595623487621], [0.310830348269163, -1.8240209302830355]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.11398452266481129, 4.8248524825591232], [-2.2179401145528979, 0.18297205545926243]], 
[[5.1953025981376149, 3.9239595315974034], [-2.620685812321816, 0.75166525490429859]], [[1.4960098225548344, 
-3.0321276951856335], [-1.2615793767118682, -0.94181494940770305]], [[0.44564582005236719, 0.033308117014316352], 
[4.2463022121637994, -2.1373984399995649]], [[-1.1426037903730064, 1.9807186693810941], [-2.9008743802132995, 
2.6350000516411258]], [[-7.7206257529625741, -8.5896289956081624], [2.2879523056131612, 
-5.7022088107098456]]])+(1.-msk_ref)*numpy.array([[[-2.8917401934421054, 2.285322874861099], [1.5094845216250494, 
5.0582480143560016]], [[0.16847167344224978, -3.321381573354425], [-1.7135584099611716, 2.6786546645124236]], 
[[1.4579049581976844, 0.13428802066683687], [-1.5636627600141253, -7.4273740727229551]], [[-1.2412183971626494, 
-5.777641529528573], [-5.4118203411963837, -3.2154451321506565]], [[-7.9347250947971686, -3.8355454884310021], 
[-0.5481885032710978, 1.2382612962663027]], [[-6.6990115718606198, -7.8730385235257243], [-2.7279915657390843, 
-4.9557136548061349]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-4.896390335512546, -3.6494156481985849], [3.1209884510239654, -1.6436189181930816]], 
[[2.0939949433105332, 2.0006031029708975], [2.2470346410657891, 3.438801365932191]], [[-2.0690503811860204, 
-0.099744349075501226], [1.0137685530306335, 1.2171755934164921]], [[-0.87827846761489781, 4.0872724520618267], 
[2.9354643866583068, 4.1143096083900605]], [[-3.7113940069035944, -1.6181605987660062], [1.3372720300991645, 
-3.6218038147837675]], [[-1.4851423642261374, -0.40761343035845599], [-1.4854015089650927, 
-2.9440832348636183]]])+(1.-msk_arg0)*numpy.array([[[-0.21225775608570174, -2.0655760111707124], [1.6483788125806678, 
1.3141851129402351]], [[-4.4968900670344647, 2.1126840631632318], [-1.0966691945425677, -1.0977642798057663]], 
[[-0.74754446898864924, 0.6235426239687305], [-1.4079998510575309, 1.2602378656653173]], [[-0.38695209359615479, 
0.44155220987930566], [-3.5420846208848844, 2.0906995356531635]], [[0.39534852998466619, 2.0192284311167725], 
[0.29906114679711582, -1.5849018103391979]], [[-0.60116052951303622, 4.4016766097054987], [-3.1438941072029749, 
-0.32123963826199287]]])
      arg1=Data(4.73709214991,self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.15929818560002929, 1.0876765017139318], [7.8580806009364821, 3.0934732317194351]], 
[[6.8310870932230499, 6.7376952528834142], [6.9841267909783058, 8.1758935158447077]], [[2.6680417687264963, 
4.6373478008370155], [5.7508607029431502, 5.9542677433290088]], [[3.8588136822976189, 8.8243646019743434], [7.6725565365708235, 
8.8514017583025772]], [[1.0256981430089223, 3.1189315511465106], [6.0743641800116812, 1.1152883351287493]], 
[[3.2519497856863793, 4.3294787195540607], [3.251690640947424, 
1.7930089150488984]]])+(1.-msk_ref)*numpy.array([[[4.524834393826815, 2.6715161387418043], [6.3854709624931845, 
6.0512772628527518]], [[0.240202082878052, 6.8497762130757485], [3.640422955369949, 3.6393278701067504]], [[3.9895476809238675, 
5.3606347738812472], [3.3290922988549858, 5.997330015577834]], [[4.3501400563163619, 5.1786443597918224], [1.1950075290276323, 
6.8277916855656802]], [[5.1324406798971829, 6.7563205810292892], [5.0361532967096325, 3.1521903395733188]], 
[[4.1359316203994805, 9.1387687596180154], [1.5931980427095418, 4.4158525116505238]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[2.7446408715793105, -2.7283411080261768], [-4.7155693170541024, 0.38699453140687101]], 
[[-3.5837675840571173, 3.5031910095978134], [0.46933587676578092, -0.28572064176866085]], [[2.5256823386435503, 
-3.8073541917236953], [3.1555359877456297, 0.92703026726678672]], [[-4.5580231071648853, 2.5545992873439989], 
[0.074179517566524744, -2.8679075121748854]], [[-0.69826044456594882, -1.8342091048285383], [2.6260243844870725, 
4.9519111269844291]], [[-3.0085492565545078, -4.5957667435961538], [0.14032227268657049, 
3.8701832457689527]]])+(1.-msk_arg0)*numpy.array([[[-2.9436008429263847, -0.26982328326691274], [-0.14744721049544829, 
2.6262027551321765]], [[-2.8907419060540218, 2.8148204110277888], [1.7839773225502986, -1.651320589475195]], 
[[-2.1753098121468528, -2.5770045874202085], [-3.9060924845729472, 2.3499443927998254]], [[3.4626574323499799, 
1.3564717334254937], [4.1997433613065489, 4.711002189259105]], [[1.6288151654445713, -3.9869817451307354], 
[-4.1550096810205037, -3.2848036686383764]], [[3.6430987867852611, 2.415872489030586], [-2.9066079204346371, 
-2.1213735830685589]]])
      arg1=Data(numpy.array([[[-2.732895811729529, 2.5602955131348883], [-4.1066349609006298, 4.0864237695612378]], 
[[-0.49573224916341641, 4.8832786369429488], [3.6441038195080271, -1.4234743529395599]], [[-1.1098132827774165, 
3.3541478790890196], [-1.3033381865817795, 3.9711105221846434]], [[3.4153703168751246, -4.3065092558004618], 
[3.4115229392782567, -2.5850865084074224]], [[-4.1100782297535936, -1.3958034196086144], [4.8146713215176931, 
-2.248223461081813]], [[-0.32919890575824517, 2.694242925756793], [-0.99421043495718475, 
-0.40846961144262561]]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.01174505984978147, -0.16804559489128845], [-8.8222042779547323, 4.4734183009681088]], 
[[-4.0794998332205337, 8.3864696465407622], [4.113439696273808, -1.7091949947082208]], [[1.4158690558661338, 
-0.4532063126346757], [1.8521978011638502, 4.8981407894514302]], [[-1.1426527902897607, -1.751909968456463], 
[3.4857024568447814, -5.4529940205823078]], [[-4.8083386743195424, -3.2300125244371527], [7.4406957060047656, 
2.7036876659026161]], [[-3.337748162312753, -1.9015238178393608], [-0.85388816227061426, 
3.4617136343263271]]])+(1.-msk_ref)*numpy.array([[[-5.6764966546559137, 2.2904722298679756], [-4.2540821713960781, 
6.7126265246934143]], [[-3.3864741552174382, 7.6980990479707376], [5.4280811420583257, -3.0747949424147549]], 
[[-3.2851230949242693, 0.77714329166881102], [-5.2094306711547267, 6.3210549149844688]], [[6.8780277492251045, 
-2.9500375223749682], [7.6112663005848056, 2.1259156808516826]], [[-2.4812630643090223, -5.3827851647393494], 
[0.65966164049718934, -5.533027129720189]], [[3.3138998810270159, 5.1101154147873791], [-3.9008183553918219, 
-2.5298431945111846]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[3.3222663399916605, -0.65702496140374755], [4.8968289893443018, 4.3780581269361392]], 
[[1.0031977760195767, 1.5228628790548315], [-1.9700235420848311, 4.1477033202907148]], [[4.9594952685935159, 
3.0567907501479858], [1.6413837637635957, -4.4678433800303576]], [[2.8361500302464746, 4.4755801913208675], [4.85016468467105, 
-3.3220260848859731]], [[1.3247663183831921, -3.0089712275094116], [-2.025645560629159, -4.7585979270134864]], 
[[-2.3550487484763738, -4.9954090357828882], [4.050757337306635, 
-2.7384347838456948]]])+(1.-msk_arg0)*numpy.array([[[-0.74362938822235769, -0.27994863266717473], [4.242727542726497, 
3.1568853422693319]], [[2.1640064554813119, -4.2293971652828546], [1.6373222902162681, 1.247635106777043]], 
[[-2.3778012300594762, -3.7784276579017453], [-1.2830118252967146, -2.8735094216579671]], [[-0.34799976874213812, 
0.38868104495158651], [-4.214463653540812, 2.3217021169261525]], [[1.8909464127076756, 3.2918070884787802], 
[-1.3799479772509891, 2.9776360738793395]], [[2.270604358768983, -2.7391453540533872], [-4.2328114233990561, 
2.6194651469841324]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.61683740516)+(1.-msk_arg1)*(4.29962081324)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.7054289348282401, -2.273862366567168], [3.2799915841808813, 2.7612207217727187]], 
[[-0.61363962914384373, -0.093974526108588918], [-3.5868609472482516, 2.5308659151272943]], [[3.3426578634300954, 
1.4399533449845654], [0.024546358600175289, -6.0846807851937781]], [[1.2193126250830542, 2.858742786157447], 
[3.2333272795076295, -4.9388634900493935]], [[-0.29207108678022831, -4.6258086326728325], [-3.6424829657925795, 
-6.3754353321769068]], [[-3.9718861536397942, -6.6122464409463086], [2.4339199321432146, 
-4.3552721890091153]]])+(1.-msk_ref)*numpy.array([[[3.5559914250156481, 4.0196721805708311], [8.5423483559645028, 
7.4565061555073378]], [[6.4636272687193177, 0.070223647955151236], [5.9369431034542739, 5.5472559200150489]], 
[[1.9218195831785296, 0.52119315533626054], [3.0166089879412912, 1.4261113915800387]], [[3.9516210444958677, 
4.6883018581895923], [0.08515715969719384, 6.6213229301641583]], [[6.1905672259456814, 7.5914279017167861], 
[2.9196728359870168, 7.2772568871173453]], [[6.5702251720069889, 1.5604754591846186], [0.066809389838949684, 
6.9190859602221382]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-4.9379462049086431, -0.50115771077774518], [-1.3730395916442948, -1.1862176534734004]], 
[[-2.6227338990678275, -3.6277131094374324], [-2.7528039114318603, 1.5502243210228084]], [[3.9824931320323991, 
-0.65990411796340176], [4.5890285282532233, -3.156647479246887]], [[0.26028353343059862, -0.82892474823903761], 
[-2.8255759563216234, 2.9916519115863993]], [[-2.7205582735309108, -4.1108251324566929], [-3.8114521781144148, 
-4.1323718466414885]], [[4.6524437414485575, -2.6366197145798953], [-4.2063520354185426, 
-3.2607694482803398]]])+(1.-msk_arg0)*numpy.array([[[-1.2154312948200872, -0.29489687970913536], [-2.0474762012076875, 
-1.4583329816853272]], [[1.2850141893373443, -2.0756879460950084], [1.5675080033551634, -3.43391713798023]], 
[[-0.68097042151458265, 2.1488253217882649], [-4.120421816025635, 4.9765207439709833]], [[3.4894146187000565, 
-3.3222552788962734], [0.06714844639972295, -0.6590312201373596]], [[-3.6104722020936721, -0.17336938695444193], 
[-1.6681976397937301, 3.059319779350183]], [[-3.4896385084769621, -0.86146031518236832], [-1.0085279988793463, 
4.6405653687833173]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[4.0684917026216922, 3.3559978079715727], [-3.835651513748195, -4.6955037791660805]], 
[[1.409577670413519, -3.7155824518003344], [4.3074043832932993, 0.80629918291511338]], [[-2.5834209356069482, 
1.7549236457747064], [4.7783267376501879, 1.3345815221429982]], [[-0.31856808753177468, -0.46345086911731759], 
[-1.7071506443877684, -1.2292875928994054]], [[3.1889403072436888, -0.78190389310360242], [4.7671289066906688, 
-4.4530230668545121]], [[-4.2562735394491726, 2.4325184936103517], [3.1512195062731294, 
0.57226010139736871]]])+(1.-msk_arg1)*numpy.array([[[4.7865366334155652, -0.68943380654272435], [-3.2062629492845094, 
4.664147136777089]], [[3.2382884085626031, -3.6992690785029456], [1.2822780874639665, 2.4910236122627438]], 
[[-3.8779664464436703, -4.9649386337045271], [4.1022769520898645, 2.3740306775308264]], [[-0.42163030020139658, 
2.4609912672875538], [-3.4920117200142187, -3.9824769003688312]], [[-3.6909018499851709, 4.2519964903684464], 
[2.7604796465832173, 3.1358386769233331]], [[4.8725740344391557, 3.4931853917442108], [-1.8138426091824655, 
0.062225045977380766]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.86945450228695087, 2.8548400971938275], [-5.2086911053924894, -5.8817214326394804]], 
[[-1.2131562286543085, -7.3432955612377668], [1.554600471861439, 2.3565235039379218]], [[1.3990721964254509, 
1.0950195278113046], [9.3673552659034112, -1.8220659571038889]], [[-0.058284554101176056, -1.2923756173563552], 
[-4.5327266007093918, 1.7623643186869939]], [[0.46838203371277798, -4.8927290255602953], [0.95567672857625396, 
-8.5853949134959997]], [[0.39617020199938491, -0.20410122096954364], [-1.0551325291454132, 
-2.6885093468829711]]])+(1.-msk_ref)*numpy.array([[[3.5711053385954781, -0.98433068625185971], [-5.2537391504921969, 
3.2058141550917618]], [[4.5233025978999475, -5.7749570245979545], [2.8497860908191299, -0.94289352571748619]], 
[[-4.5589368679582529, -2.8161133119162622], [-0.018144863935770417, 7.3505514215018097]], [[3.06778431849866, 
-0.86126401160871957], [-3.4248632736144957, -4.6415081205061908]], [[-7.3013740520788435, 4.0786271034140045], 
[1.0922820067894872, 6.195158456273516]], [[1.3829355259621936, 2.6317250765618425], [-2.8223706080618118, 4.702790414760698]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.2681594830366842, -1.5376956290564681, -3.3624269118616201, -4.0716218633868344], 
[2.4508099121664522, -4.2583734580525796, 1.5870206089578263, -4.2039370171751314], [-0.055203310259175353, 
0.69635944703200359, -4.9432910169815205, 1.3796414018797147]], [[1.230574765082916, 1.2885063238999592, -3.144288594261452, 
-2.4376473574473834], [3.6006896378173856, 2.1510376549629004, 2.8590917221870686, -3.6392190921274556], [1.514316542494619, 
-4.5981557571111731, 0.38154357292178176, -4.4572595454325938]]], [[[-0.40112004877855245, -2.2936256002026423, 
2.9677744815975879, -3.9822392685130161], [-2.7215482012313008, -2.4717925556357812, 4.5494106948237434, -2.7163993436629461], 
[0.45236659654278988, -4.7602597240579101, -1.9660586216028229, -0.39237656126935327]], [[1.492684370977841, 
-0.91294617060333927, -1.2488502831083159, -0.15528501244716253], [1.1426381764952627, 4.1797906985753972, 
-0.68303670356068658, 2.6374607281674392], [-1.2974423501060386, -4.1304496465858129, 4.637531953058124, 
-1.3900795473657892]]], [[[1.8454046020774229, -0.68793705341374967, 2.3494391559216279, -0.26380732598176415], 
[3.9946819080782809, 1.7802551573050032, 0.35161831865094406, -1.3489903203727369], [2.0135068513354764, 1.2559355869687092, 
0.62043002174170159, -4.400772779763015]], [[-1.5609140205489549, 0.4306363321023472, -1.3825868931640004, 4.7762410424293105], 
[4.6404401899084409, -2.3876621930830142, 2.1344793227886889, -0.28084914624671597], [2.6720517798522572, 2.6530350599357924, 
-1.3953479702723581, 3.3027322306168951]]]])+(1.-msk_arg0)*numpy.array([[[[-0.63729719481573532, 1.3050424186166794, 
3.9730370583881029, 1.8431579187668659], [4.339379593114046, 3.3680737149920539, -2.5727829028440476, -3.5724070891965143], 
[3.853640225683554, 1.5302165093952889, -4.2058305672117324, -4.2766475009936231]], [[-2.1522504350422098, 0.92704044118854512, 
2.4982603056054744, 1.7176169924991207], [-3.0521887611716636, -3.4582025348526049, 2.4141234647975915, -2.5267883278676981], 
[2.6840936936902366, -2.3036538969096352, 4.0048506424888437, 4.4548463341403934]]], [[[1.8877134646616742, 
-1.7054087221046288, 0.37123455056738663, -2.9383555224376332], [3.093788564271879, 1.8235867798741321, 0.095692153656293577, 
2.0015773137372594], [3.0856422975346867, 0.87468265575748028, 3.5199430400249536, 2.0797324670582995]], [[-2.8220335098363392, 
1.1503230307234622, 1.817646672560759, 4.6359388901945877], [4.5306553152772171, -1.8000850941409285, 2.0561373930988101, 
2.947169979403216], [2.2738255966148992, 0.2608839371772671, 1.1482657414535637, -1.3606879071862377]]], 
[[[-0.60669684991151307, 2.8501451282123558, 2.9029737215521676, -1.6619807020917845], [0.48048666290602071, 
-4.1911984266763245, -3.3249730583261563, 4.6660538693394624], [-4.5958309393932542, 0.91497356108001249, -4.2986666436684953, 
1.1097291373503673]], [[2.4307018578893853, 3.0517808484120863, -1.4927443128914719, 2.8552273805993709], [-4.1067506675754242, 
0.15572055388090789, -1.952986340790007, 1.0094162712936967], [1.5111741037687576, 4.6992712975628308, -0.98666653898764967, 
-3.4166201765940496]]]])
      arg1=-4.41847302496
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-8.6866325080006597, -5.956168654020443, -7.7808999368255947, -8.4900948883508089], 
[-1.9676631127975224, -8.6768464830165541, -2.8314524160061483, -8.622410042139105], [-4.4736763352231499, -3.722113577931971, 
-9.3617640419454951, -3.0388316230842598]], [[-3.1878982598810586, -3.1299667010640153, -7.5627616192254266, 
-6.8561203824113583], [-0.81778338714658894, -2.2674353700010741, -1.5593813027769059, -8.0576921170914311], 
[-2.9041564824693555, -9.0166287820751485, -4.0369294520421928, -8.8757325703965684]]], [[[-4.819593073742527, 
-6.7120986251666164, -1.4506985433663866, -8.4007122934769907], [-7.1400212261952749, -6.8902655805997561, 0.13093766985976885, 
-7.1348723686269206], [-3.9661064284211847, -9.1787327490218846, -6.3845316465667974, -4.8108495862333278]], 
[[-2.9257886539861335, -5.3314191955673138, -5.6673233080722909, -4.5737580374111371], [-3.2758348484687119, 
-0.23868232638857734, -5.1015097285246611, -1.7810122967965354], [-5.7159153750700131, -8.5489226715497875, 
0.21905892809414951, -5.8085525723297637]]], [[[-2.5730684228865517, -5.1064100783777242, -2.0690338690423467, 
-4.6822803509457387], [-0.4237911168856936, -2.6382178676589714, -4.0668547063130305, -5.7674633453367115], 
[-2.4049661736284982, -3.1625374379952653, -3.798043003222273, -8.8192458047269895]], [[-5.9793870455129294, 
-3.9878366928616273, -5.8010599181279749, 0.35776801746533593], [0.22196716494446633, -6.8061352180469887, -2.2839937021752856, 
-4.6993221712106905], [-1.7464212451117174, -1.7654379650281822, -5.8138209952363322, 
-1.1157407943470794]]]])+(1.-msk_ref)*numpy.array([[[[-5.0557702197797099, -3.1134306063472952, -0.44543596657587159, 
-2.5753151061971087], [-0.079093431849928564, -1.0503993099719207, -6.9912559278080222, -7.9908801141604888], 
[-0.56483279928042052, -2.8882565155686857, -8.624303592175707, -8.6951205259575985]], [[-6.5707234600061843, 
-3.4914325837754294, -1.9202127193585001, -2.7008560324648538], [-7.4706617861356381, -7.8766755598165794, -2.004349560166383, 
-6.9452613528316727], [-1.7343793312737379, -6.7221269218736097, -0.41362238247513083, 0.0363733091764189]]], 
[[[-2.5307595603023003, -6.1238817470686033, -4.0472384743965879, -7.3568285474016077], [-1.3246844606920956, 
-2.5948862450898424, -4.322780871307681, -2.4168957112267151], [-1.3328307274292879, -3.5437903692064943, -0.89852998493902092, 
-2.338740557905675]], [[-7.2405065348003141, -3.2681499942405123, -2.6008263524032156, 0.21746586523061318], 
[0.11218229031324256, -6.2185581191049035, -2.3623356318651645, -1.4713030455607585], [-2.1446474283490753, 
-4.1575890877867074, -3.2702072835104108, -5.7791609321502122]]], [[[-5.0251698748754876, -1.5683278967516188, 
-1.5154993034118069, -6.080453727055759], [-3.9379863620579538, -8.609671451640299, -7.7434460832901308, 0.24758084437548789], 
[-9.0143039643572287, -3.5034994638839621, -8.7171396686324698, -3.3087438876136073]], [[-1.9877711670745892, 
-1.3666921765518882, -5.9112173378554465, -1.5632456443646037], [-8.5252236925393987, -4.2627524710830667, -6.3714593657539815, 
-3.4090567536702778], [-2.9072989211952169, 0.28079827259885626, -5.4051395639516242, -7.8350932015580241]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-1.7029874502278188, -4.1806522421568175, 3.3403213933721094, -1.8015377022800072], 
[-0.91530609735667401, -2.6420850866613996, -3.0333226841436209, -2.6301759295168594], [3.0679901880583422, 
-3.4530988050036027, -0.074754696621921468, -2.5903969495353971]], [[-4.3633217867198688, -4.3460778703503022, 
-0.78147030366203118, -2.0486491129289277], [-3.6460145779921396, -1.1751145856592027, 0.79569274412989088, 
-0.97603271966770411], [-3.1255669829302457, -3.5763178555997808, 1.9392798905582893, -3.9482022628921918]]], 
[[[4.0926922292234629, 2.2534266429183987, -0.70293291790992818, 4.6157950694470475], [-3.0576258013507838, 2.4098094024391417, 
2.6655825779126685, -0.63720288337193765], [-1.5780424425258932, -1.1810704836845076, -4.4155344953251259, 
-1.162987659875736]], [[-3.9387922210570991, -0.69626119272413156, 4.1902623701218307, 0.7217908837505238], 
[-0.97790463375530479, -0.048565360859394779, -4.7800772693834848, -3.7056571441538946], [3.8525373076691363, 
3.2670419394770818, -2.8091195036501295, -0.24596150639200065]]], [[[4.5291617111680473, -1.2189624480453434, 
1.8088753772224049, -2.431555950039578], [1.3991393905591352, 2.971220906335537, 1.5053222211216246, 0.89136135966454422], 
[3.9229723251237658, -1.089286539137829, 2.0372682609969139, -4.0168107215486293]], [[-4.3908511717245613, -4.006020483444205, 
-2.2863227282118426, -1.5038784930940441], [-2.6594774788068554, -0.5418701640207404, -3.5463693235911231, 
-1.7841135461147486], [-0.35737593897669662, 0.92047166488821297, -1.6876600473812853, 
0.2515335498640292]]]])+(1.-msk_arg0)*numpy.array([[[[-1.2906578413734895, 2.6646706726836058, -4.5059197747011979, 
-0.13626404180184792], [-1.2045182263849838, -0.34865310803085237, -1.7821481414885199, -2.7857581187431926], 
[4.8713493470903124, -4.6421823857317692, 3.0405167350323552, 0.18948653813210559]], [[-2.5009849069988932, -4.393937000006181, 
2.1384406404935348, -1.98024536710804], [-1.8275259442981397, -2.0104694188355965, -1.5727125445682533, -0.20152341616551528], 
[4.2854698155967235, 3.3200183807433135, -1.1302861549275445, 1.6475899351670886]]], [[[-1.3774205333701959, 
0.93931418937473143, -3.1208936859226757, 0.38630602069300668], [-1.6211954043498586, -2.5395759587629998, -2.1677050850514967, 
3.9489828402258098], [-3.2134202518623711, 1.5372659284736088, -3.0851613333681716, -2.3769593493705496]], 
[[4.4933189221238816, 1.2741803106282852, -3.3613627859466035, -1.7917992945073569], [0.31566943460378916, 
-0.96158503872613821, 2.6638620054789621, -3.6067612200060895], [0.94070204144957348, 0.97528308755009707, 2.6215430055246038, 
-1.2877783634173898]]], [[[-4.9391314507220505, -2.7621198824701878, -4.6279277085973911, -3.1753384415636243], 
[3.244831083489025, -1.2889368845574136, 0.4443845296965323, 0.6366474659194532], [-2.1964269355668509, 0.37593720944557152, 
-0.54130744081315019, -4.8406905967766249]], [[4.8022016197352553, 0.4286233948994127, -0.48970401830843446, 
-2.2456163392608683], [0.13263383394337414, -3.0570072940352411, 0.63934574518503773, -0.71410276507906989], 
[4.0400296835900686, 1.4389282981230389, -1.1467498321258383, -0.75021141353363596]]]])
      arg1=numpy.array(0.836074032549)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.86691341767867502, -3.3445782096076737, 4.1763954259212532, -0.96546366973086339], 
[-0.0792320648075302, -1.8060110541122558, -2.1972486515944771, -1.7941018969677156], [3.904064220607486, -2.6170247724544589, 
0.76131933592722234, -1.7543229169862533]], [[-3.527247754170725, -3.5100038378011584, 0.054603728887112624, 
-1.2125750803797839], [-2.8099405454429958, -0.33904055311005887, 1.6317667766790347, -0.1399586871185603], 
[-2.2894929503811019, -2.740243823050637, 2.7753539231074331, -3.112128230343048]]], [[[4.9287662617726067, 3.0895006754675425, 
0.13314111463921563, 5.4518691019961913], [-2.22155176880164, 3.2458834349882855, 3.5016566104618123, 0.19887114917720616], 
[-0.7419684099767494, -0.34499645113536381, -3.5794604627759821, -0.32691362732659224]], [[-3.1027181885079553, 
0.13981283982501225, 5.0263364026709745, 1.5578649162996676], [-0.14183060120616098, 0.78750867168974903, -3.944003236834341, 
-2.8695831116047508], [4.6886113402182801, 4.1031159720262256, -1.9730454711009857, 0.59011252615714316]]], 
[[[5.3652357437171911, -0.38288841549619956, 2.6449494097715487, -1.5954819174904342], [2.235213423108279, 3.8072949388846808, 
2.3413962536707684, 1.727435392213688], [4.7590463576729096, -0.25321250658868522, 2.8733422935460577, -3.1807366889994855]], 
[[-3.5547771391754175, -3.1699464508950612, -1.4502486956626988, -0.66780446054490028], [-1.8234034462577116, 
0.29420386852840341, -2.7102952910419793, -0.94803951356560479], [0.47869809357244719, 1.7565456974373568, 
-0.85158601483214147, 1.087607582413173]]]])+(1.-msk_ref)*numpy.array([[[[-0.45458380882434568, 3.5007447052327496, 
-3.6698457421520541, 0.69980999074729588], [-0.36844419383584004, 0.48742092451829144, -0.94607410893937605, 
-1.9496840861940488], [5.7074233796394562, -3.8061083531826254, 3.876590767581499, 1.0255605706812494]], [[-1.6649108744497494, 
-3.5578629674570372, 2.9745146730426786, -1.1441713345588962], [-0.99145191174899594, -1.1743953862864527, 
-0.73663851201910946, 0.63455061638362853], [5.1215438481458673, 4.1560924132924573, -0.29421212237840066, 
2.4836639677162324]]], [[[-0.54134650082105207, 1.7753882219238752, -2.2848196533735319, 1.2223800532421505], 
[-0.78512137180071484, -1.703501926213856, -1.3316310525023529, 4.7850568727749536], [-2.3773462193132273, 2.3733399610227526, 
-2.2490873008190277, -1.5408853168214058]], [[5.3293929546730254, 2.110254343177429, -2.5252887533974597, 
-0.95572526195821306], [1.151743467152933, -0.1255110061769944, 3.4999360380281059, -2.7706871874569456], [1.7767760739987173, 
1.8113571200992409, 3.4576170380737477, -0.45170433086824602]]], [[[-4.1030574181729067, -1.926045849921044, 
-3.7918536760482473, -2.3392644090144805], [4.0809051160381689, -0.45286285200826981, 1.2804585622456761, 1.472721498468597], 
[-1.3603529030177071, 1.2120112419947153, 0.29476659173599362, -4.0046165642274811]], [[5.6382756522843991, 1.2646974274485565, 
0.34637001424070935, -1.4095423067117245], [0.96870786649251794, -2.2209332614860973, 1.4754197777341815, 0.12197126747007392], 
[4.8761037161392125, 2.2750023306721827, -0.31067579957669444, 0.085862619015507846]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[1.2090022638403708, 2.4906868206324217, -0.96860224299082098, 3.0649032918276884], 
[4.9599370937979934, -2.5779549502245516, -0.77271245738056571, -2.6665187999851825], [3.6711129250705525, -4.0662029983468839, 
-2.7336622678111167, -0.5113794022988305]], [[1.2903730974580405, 1.3699817386085966, 2.1276484160745515, -2.7944649742874805], 
[3.9633580246524076, 0.30072429848755, -1.441152870738176, 3.6087864864246715], [2.9109167552781665, -0.013944769805290136, 
2.0405950828890438, 3.8171970329675879]]], [[[4.6354722939385518, -0.88327609977932653, -0.67541176028094796, 
-3.336310597983474], [-1.9938398122826251, -2.4595011656191037, 0.73233505905593788, 0.35027746348343047], [1.8602158246610863, 
0.67971315243455344, -3.0825194791694366, -2.7128372055271086]], [[3.1277326779619532, 2.4057709666936944, -4.7501462849586806, 
0.66875822407768837], [-1.5005014266863803, 0.037063413995949368, 3.28368234781931, 1.3876585068197294], [2.9257601438352818, 
-2.3624676588848903, 0.56113823971392751, 1.5078411220223771]]], [[[3.9500050529735002, -4.1442074509228313, 
1.0787984557554156, 4.8848307938989848], [-1.5851688533330597, -1.9072563412898305, 3.0578203582468415, 2.8061120078616426], 
[2.6016405612252154, 2.2456066946594397, -3.6686241955399912, -1.2981304275790908]], [[0.35239551240801337, 1.4608779649136636, 
2.3099366815158442, 3.6048318602458984], [-3.3575698839201982, -1.3386220514928815, 3.7710486349403389, -0.23650273614853923], 
[-4.5752410302331086, 4.288803784809966, 3.7459984213335069, 
2.9619005224184107]]]])+(1.-msk_arg0)*numpy.array([[[[3.2816404096294907, -3.0839181331484999, 0.82059949133192411, 
-3.8163984277195118], [-1.5166817274130109, 4.9694905910782499, -3.00504053913465, 4.0159560063625435], [-3.4986332740358872, 
-3.0459609658716791, -3.5206751999880179, -3.4146569721141673]], [[-3.0945587313603329, 0.30896225216545226, 
-3.199041177068441, -0.054320729529350409], [4.3299466268002842, 0.94540117212833685, 1.1072202801538991, 1.171219349115745], 
[4.8019287822533077, 4.297611929301743, 4.2640685827068872, 4.9338759518716504]]], [[[-3.166725315717378, 3.7366003230972957, 
4.6848134808132205, -4.7059757633982731], [-2.6590912932843871, -4.5004394485043111, -4.8795290578445352, 4.3033985552093448], 
[-4.6305476597191202, -1.5442343712502806, 3.8249142127809179, 2.1508580059372306]], [[-4.6562462360256385, 4.11977450859656, 
-2.2684137286617299, -3.3726636185259506], [2.2873380277711961, -1.9537415691573976, 1.6862399802469694, -1.3605730557826803], 
[3.9322388898143785, 2.9336495032245544, -3.1539909429216282, -1.7899400333141982]]], [[[1.9971775291512275, 
3.5038043792718341, 0.31696506927510004, 2.2643596597256108], [-3.16658080120098, 0.3811251549436232, -1.914765094647517, 
1.7306981890034487], [-4.3596565875377795, -1.9625591101482587, 1.7749412977122798, 3.8092951766550183]], 
[[-3.7148786713302693, -4.7389979469095458, -0.42101324493263892, -2.8515126199827314], [-4.5687630528207794, 
-0.36977529163428979, 4.798981791917571, -0.24570027365999358], [-1.7919969000454272, -2.7403748620844635, -2.3317025932899984, 
4.6391557034546178]]]])
      arg1=numpy.array([[[[-1.650341636631667, -1.3747255993415699, 2.4550305881518568, 1.6673134361744646], 
[2.2945371694443883, -3.719200633650944, 1.84238932050802, -0.30355744207369284], [1.9875665364463089, -2.9831220370766243, 
0.51619132760193764, 1.1082169966076751]], [[-1.5656917776074519, -0.59446776744913077, -3.5676813730931158, 
-1.9022063034961931], [-2.177566085543059, -2.1596439880238769, -1.2128477271801752, -4.905855241561687], [4.077733226884547, 
4.273252489354439, -3.3662038367753375, 1.5563484400157677]]], [[[-4.1452520588566735, -2.1954463050723061, 
-2.9574378430115322, 0.42149379839281131], [-1.2808443750129639, 3.3968936307771642, 3.4350743179153014, 4.2176176272986048], 
[-1.6210127696841137, 0.91053228807847475, 0.61421568711536345, 4.0194320695143997]], [[2.9557553460741159, 4.8777946247319033, 
1.0734225717927517, -2.5105871614410868], [0.24381276857611933, -1.5430702011981943, 0.33127576250817548, 2.3851225735817225], 
[-1.8025119540925969, 1.2966706480373045, -3.5311092445895542, 1.130834536418349]]], [[[0.74913717796009305, 
-2.8235591581192865, -4.2582639143226784, 4.6809545477192493], [-0.30600013412045879, -2.3040343368361302, -1.4399038913680906, 
0.73135991636374964], [0.055534857551258199, 1.7820455278381981, -4.9386135782798846, 0.36725448792931559]], 
[[-4.3007803007417227, -3.2895391565420793, 2.944736163949595, 3.8612986064148771], [4.1534653234933057, 4.2766501636617864, 
4.7701557510122417, -0.84675005142144233], [3.4521054958119013, 2.4267016604095915, 1.6554850588013634, -1.0021658959204096]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.4413393727912962, 1.1159612212908518, 1.4864283451610358, 4.732216728002153], 
[7.2544742632423818, -6.2971555838754956, 1.0696768631274542, -2.9700762420588753], [5.6586794615168614, -7.0493250354235082, 
-2.217470940209179, 0.59683759430884464]], [[-0.27531868014941141, 0.77551397115946585, -1.4400329570185644, 
-4.6966712777836737], [1.7857919391093486, -1.8589196895363269, -2.6540005979183512, -1.2970687551370155], [6.9886499821627135, 
4.2593077195491489, -1.3256087538862937, 5.3735454729833556]]], [[[0.49022023508187829, -3.0787224048516326, 
-3.6328496032924802, -2.9148167995906626], [-3.274684187295589, 0.93739246515806052, 4.1674093769712393, 4.5678950907820353], 
[0.23920305497697258, 1.5902454405130282, -2.4683037920540731, 1.3065948639872911]], [[6.0834880240360691, 7.2835655914255977, 
-3.6767237131659289, -1.8418289373633985], [-1.256688658110261, -1.5060067872022449, 3.6149581103274855, 3.7727810804014519], 
[1.1232481897426849, -1.0657970108475858, -2.9699710048756267, 2.638675658440726]]], [[[4.6991422309335933, 
-6.9677666090421173, -3.1794654585672628, 9.5657853416182341], [-1.8911689874535185, -4.2112906781259607, 1.6179164668787509, 
3.5374719242253923], [2.6571754187764736, 4.0276522224976379, -8.6072377738198753, -0.93087593964977522]], 
[[-3.9483847883337093, -1.8286611916284157, 5.2546728454654392, 7.4661304666607755], [0.79589543957310749, 2.9380281121689049, 
8.5412043859525806, -1.0832527875699816], [-1.1231355344212073, 6.7155054452195575, 5.4014834801348703, 
1.9597346264980011]]]])+(1.-msk_ref)*numpy.array([[[[1.6312987729978237, -4.4586437324900698, 3.2756300794837809, 
-2.1490849915450472], [0.77785544203137746, 1.2502899574273059, -1.16265121862663, 3.7123985642888506], [-1.5110667375895783, 
-6.0290830029483029, -3.0044838723860803, -2.3064399755064922]], [[-4.6602505089677848, -0.28550551528367851, 
-6.7667225501615569, -1.9565270330255435], [2.1523805412572252, -1.2142428158955401, -0.10562744702627613, -3.734635892445942], 
[8.8796620091378546, 8.570864418656182, 0.89786474593154963, 6.4902243918874181]]], [[[-7.3119773745740515, 1.5411540180249896, 
1.7273756378016882, -4.2844819650054617], [-3.939935668297351, -1.1035458177271469, -1.4444547399292338, 8.5210161825079496], 
[-6.2515604294032343, -0.63370208317180587, 4.4391298998962814, 6.1702900754516303]], [[-1.7004908899515225, 
8.9975691333284633, -1.1949911568689782, -5.883250779967037], [2.5311507963473154, -3.4968117703555919, 2.0175157427551449, 
1.0245495177990422], [2.1297269357217816, 4.2303201512618589, -6.685100187511182, -0.65910549689584919]]], 
[[[2.7463147071113205, 0.6802452211525476, -3.9412988450475783, 6.9453142074448602], [-3.4725809353214387, -1.922909181892507, 
-3.3546689860156076, 2.4620581053671984], [-4.3041217299865213, -0.18051358231006054, -3.1636722805676047, 
4.1765496645843339]], [[-8.0156589720719928, -8.028537103451626, 2.5237229190169561, 1.0097859864321457], 
[-0.41529772932747377, 3.9068748720274966, 9.5691375429298127, -1.0924503250814359], [1.6601085957664741, -0.313673201674872, 
-0.67621753448863497, 3.6369898075342082]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.2332451265218474, -2.5788114009031524, -3.0673624112749232, 1.8536246542909289], 
[-4.5002425286561722, 2.9051818811381933, -2.8868393758891719, 4.0366925832934619], [1.6699988489924502, 3.0926757408397858, 
-2.8080800579374374, 3.4722759445550508]], [[0.98488483320557485, -1.1164036323028945, 3.4765965179793223, 
0.65619193197041525], [1.3232536907616836, 2.616755099770117, -3.4341641667865694, -0.39842241977521375], [4.0794140284458678, 
-4.859147851691743, -2.1168223700912083, -4.0439664570792768]]], [[[3.5247676204351048, 3.4042951802977157, 
-3.6406836613192004, -4.5611649976444211], [-1.5079693145508335, 4.7524307689900045, -4.559898474230577, -3.415058166852325], 
[0.29918476219128998, -4.1500805945133497, 4.7024446009239611, 2.0152115457838136]], [[-2.1113440189598442, 4.2818271864471793, 
2.1402268793610002, -0.87535159995537448], [3.7623582504442226, -0.67222470325801886, -0.71915667043210174, 
-1.2319528484330688], [-1.93482422552432, -0.32753641535790212, -0.64681342159539579, -0.51986409125805633]]], 
[[[4.857862820076269, -1.1546218272919773, -4.2394893417096107, 3.8958449040136109], [0.62585262776198114, -1.0835900160473888, 
-3.7918837104695005, -0.54090894211884422], [3.8289587844521442, -4.8671314761091491, -2.0493376432524215, 
2.6027016439465829]], [[4.5878077438794165, -3.7518840059364211, 1.9865261839358386, -0.039850396755517714], 
[0.596818266156113, 3.5011391868773991, -0.50279331295165264, -1.0596356306056309], [0.15238870440435814, -1.6314940060358385, 
1.529375412388382, 0.39767765466914273]]]])+(1.-msk_arg0)*numpy.array([[[[0.16036105992749849, 3.4482140119429534, 
-1.7152033279974912, 3.0949254815727603], [1.5317115391572376, 4.1052472523148893, 0.66532863624585481, 2.0854945911259879], 
[0.89129748519741092, -3.1447239426635951, 0.24644023267216042, 1.9856685080864303]], [[1.2132760231992528, 4.5677501267621423, 
3.9952977734496393, 1.4725459690968457], [3.3507695021442832, 1.6512281067741981, -1.8622844447669329, -4.2927469067908373], 
[-4.8931575771597284, 2.2840358593392738, 0.81185823492076636, 3.7343242997166346]]], [[[1.6330186586921034, 
-1.818450529865665, -4.5914160354533164, 4.6559957608292386], [1.8794873134194248, -4.2077423378983383, 1.9184078938660178, 
-3.1406995088015934], [2.0700094369867807, -3.1590020038195412, -1.2435541932070415, 4.171131082565406]], 
[[-2.6609659360020563, 4.8813151542909914, -0.076751859669404787, 1.500996823606763], [3.8025432868604589, 
-0.45094519148818435, -2.4972834107276851, 4.8088526782866232], [-3.7599433204705313, 4.2460190886347338, 4.6152477597629122, 
-1.7760269761975964]]], [[[0.87978812167692411, -4.0032489944516358, -4.4852874022873905, 2.2105712339694747], 
[1.8100008427473142, -0.074678983944628463, -1.0947912359737866, 3.4902921654836128], [1.9121971262375048, 2.8422194453424279, 
-4.0684268034095359, -4.5389619846344234]], [[1.6785644672629569, 3.8888619124510697, 4.1708665043328388, 1.5175797834205964], 
[4.5463846910238601, 3.5403244275010035, 0.87302855876207985, -3.0464488457156569], [4.6931345562648978, 1.7631194072202288, 
2.6860925581737725, -0.23048194800596367]]]])
      arg1=Data(2.87469365998,self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.3585514665443279, 0.29588225907436705, -0.19266875129740368, 4.7283183142684484], 
[-1.6255488686786528, 5.7798755411157128, -0.012145715911652477, 6.9113862432709814], [4.5446925089699697, 5.9673694008173053, 
0.066613602040082043, 6.3469696045325703]], [[3.8595784931830943, 1.7582900276746249, 6.3512901779568418, 3.5308855919479347], 
[4.1979473507392031, 5.4914487597476365, -0.55947050680904997, 2.4762712402023057], [6.9541076884233872, -1.9844541917142235, 
0.75787128988631114, -1.1692727971017574]]], [[[6.3994612804126243, 6.2789888402752352, -0.76599000134168094, 
-1.6864713376669016], [1.366724345426686, 7.627124428967524, -1.6852048142530576, -0.54036450687480553], [3.1738784221688094, 
-1.2753869345358302, 7.5771382609014806, 4.889905205761333]], [[0.76334964101767522, 7.1565208464246988, 5.0149205393385197, 
1.999342060022145], [6.637051910421742, 2.2024689567195006, 2.1555369895454177, 1.6427408115444506], [0.93986943445319948, 
2.5471572446196173, 2.2278802383821237, 2.3548295687194631]]], [[[7.7325564800537885, 1.7200718326855422, -1.3647956817320912, 
6.7705385639911304], [3.5005462877395006, 1.7911036439301307, -0.91719005049198099, 2.3337847178586753], [6.7036524444296637, 
-1.9924378161316296, 0.82535601672509795, 5.4773953039241023]], [[7.4625014038569359, -0.87719034595890166, 4.861219843913358, 
2.8348432632220018], [3.4715119261336325, 6.3758328468549186, 2.3719003470258668, 1.8150580293718885], [3.0270823643818776, 
1.243199653941681, 4.4040690723659015, 3.2723713146466622]]]])+(1.-msk_ref)*numpy.array([[[[3.035054719905018, 
6.3229076719204729, 1.1594903319800283, 5.9696191415502797], [4.406405199134757, 6.9799409122924088, 3.5400222962233743, 
4.9601882511035074], [3.7659911451749304, -0.27003028268607565, 3.1211338926496799, 4.8603621680639497]], [[4.0879696831767722, 
7.4424437867396618, 6.8699914334271588, 4.3472396290743651], [6.2254631621218026, 4.5259217667517175, 1.0124092152105866, 
-1.4180532468133178], [-2.018463917182209, 5.1587295193167932, 3.6865518948982858, 6.6090179596941541]]], 
[[[4.5077123186696229, 1.0562431301118544, -1.716722375475797, 7.5306894208067581], [4.7541809733969442, -1.3330486779208188, 
4.7931015538435373, -0.26600584882407396], [4.9447030969643002, -0.28430834384202175, 1.6311394667704779, 7.0458247425429255]], 
[[0.21372772397546314, 7.7560088142685109, 2.7979418003081147, 4.3756904835842825], [6.6772369468379784, 2.4237484684893351, 
0.37741024924983435, 7.6835463382641427], [-0.88524966049301179, 7.1207127486122532, 7.4899414197404317, 1.098666683779923]]], 
[[[3.7544817816544436, -1.1285553344741164, -1.6105937423098711, 5.0852648939469942], [4.6846945027248337, 2.800014676032891, 
1.7799024240037329, 6.3649858254611322], [4.7868907862150243, 5.7169131053199473, -1.1937331434320164, -1.6642683246569039]], 
[[4.5532581272404764, 6.7635555724285892, 7.0455601643103583, 4.3922734433981159], [7.4210783510013796, 6.415018087478523, 
3.7477222187395993, -0.17175518573813742], [7.5678282162424173, 4.6378130671977482, 5.5607862181512919, 2.6442117119715558]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.9499484379170049, 0.010046085232921165, 2.4350075893888024, -2.9661124801363159], 
[-4.7650501048745166, 3.3468226866285065, -4.6712258748266251, -4.3040761669148768], [-1.2466592386624975, 1.603790853874516, 
3.1218749269268677, 2.3279442330831923]], [[-2.873813935319772, 4.0002612667871968, 1.9460428730930577, 3.3127249661061207], 
[-4.2007454167107348, -2.1588518059933759, -0.3522643678131665, 2.9031175964039786], [4.6868000202029929, -2.1523617181809449, 
4.7713011145953086, -3.522058358849911]]], [[[-3.4571944900388507, -3.1808876049914971, -3.264421861886917, 
-1.0003100908338802], [1.520596010800106, -3.7110787932470979, -0.11604840014859974, 0.13944802614639684], 
[0.86617871211817032, -1.9029822388688133, -4.8025257032250179, -3.9535340495780402]], [[2.0456234381126892, 
-4.6822488317724815, 4.5539988141489616, 2.5598877225680594], [-2.3639944047350294, 2.4316293499744273, -4.1453411205910742, 
3.4518258750859552], [-4.3375485145970583, 0.68743703674176704, 4.925525371440191, -3.4011259505455085]]], [[[3.83067737835019, 
2.9373902316225742, -1.1406444081332179, 3.0485977546579015], [-2.5906409017099206, -2.2008874814946822, 0.62119068940979094, 
3.4124669804751537], [-4.7043555940654134, -0.44300956405102454, 4.9531539200217782, -0.58517953067235062]], 
[[4.5079694107435895, -1.1765034365461124, 1.8063595482777197, -2.9662153057469345], [2.0584506859801941, 2.3848159518458658, 
-3.3489893565590445, -2.2861973937927149], [3.9454259837285548, -0.39255119371264779, 4.2652367940872296, 
-2.5317013309567504]]]])+(1.-msk_arg0)*numpy.array([[[[-0.22796876323601634, -4.6691358122296931, -3.8086235090829623, 
-2.8960569241069556], [1.3695805721972327, -2.3911045889780178, -1.3185857131725207, 2.074284110204295], [0.94989961142865376, 
2.861663409552091, -2.4237359728889762, 1.2030308083599053]], [[-4.8288430630840926, 1.5813618471264936, 1.2441739315934397, 
1.329358393158576], [-0.13409368927678855, 0.47423788003105471, 0.79408048198977088, 3.1382827488211777], [-3.4382524787131876, 
2.4582084316831754, -2.3424094813203498, -0.24012421371545667]]], [[[-1.3112005455713871, 1.4614593904830295, 
3.2397574461579204, -4.7146056787068176], [-4.6652620784687917, 2.8699242701826266, 0.35722698178513479, 0.18406910755273564], 
[-3.606261469154294, 3.7101894643553823, -4.709887546162034, -3.1231163217080651]], [[2.1751768243695127, -0.78272376039286851, 
-3.3432926250258674, 3.1824967053025492], [-4.5899549635505785, -4.4705847310881284, 3.7228063397361062, -0.38285915130476944], 
[2.272222289221526, 1.7145072977243681, 4.8872478156402668, 3.0106638504718291]]], [[[0.82182870402402486, 
-0.82427541716658581, -0.52644654302249982, -2.7310288389884452], [2.417597023316123, 0.85216261667072413, -4.9566890503732894, 
2.0034203678358855], [4.9385237750544739, -0.28298805451007158, 4.178062769824221, 1.2851011850615981]], [[0.95239153808474164, 
0.19239257134254117, -3.2615145963431891, 4.759539695436132], [-0.36521698020421045, 4.6169540946793788, 3.3250708872503481, 
-2.363840183455399], [-0.017896783980214437, -4.3835323228345864, 1.3094640180709334, -2.2566298314990698]]]])
      arg1=Data(numpy.array([[[[-2.1910787487497041, -1.0752239364745941, 0.085027920388016653, 0.64270497247428615], 
[-1.8936972290472687, -1.0911510827381177, 0.20005569021996994, 0.49956808767567118], [2.920336309414453, -3.9477546609064476, 
0.631753650062997, -3.9551903773606769]], [[-3.2268984668541911, -4.1954261295292508, -3.7802595583856222, 4.7002336499977346], 
[-0.1692017412324649, 4.4715911506928467, 2.6311577458717581, 2.2656355013822314], [-4.7680285447022586, -2.243702154146725, 
4.9884126661579504, 0.4251158497110108]]], [[[3.7604896989769259, 1.0244204795539344, -1.0390208687064941, 4.6486955695699876], 
[-2.209138566150334, 1.5994451550799837, -4.5181293701777676, 4.2978202868817732], [3.3172031576948324, -4.2476732791777208, 
2.9759870022115287, -2.1624359788820104]], [[2.5294969397153899, -2.1730373230846514, -1.004259202582225, 0.30113062931356271], 
[0.49931947691679301, 1.5772650839677373, -4.4426347553444812, -2.2221330310329899], [-4.7439333562437209, 3.7781951682542516, 
1.2370627044546243, 0.58974423001285547]]], [[[1.1711351794187834, 3.8523462786446103, 3.4199597478075567, 1.6229624286536923], 
[-4.2032438069344336, -1.6134212508944099, -0.52064431718850024, -1.0968998009897013], [-0.58556022329740465, 
3.0246606343918074, -1.3108288052176209, -3.912834235315449]], [[2.3296373932631411, -1.6481716540076716, 1.1221094044619075, 
-3.5781090281955406], [4.4541728820829949, -2.0741961771865802, 1.5463754938322509, 0.29475315883426312], [-4.3758386197352062, 
-1.5881307804806024, 3.7848663887033531, 4.6278958766043949]]]]),self.functionspace)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-7.141027186666709, -1.065177851241673, 2.520035509776819, -2.3234075076620297], 
[-6.6587473339217853, 2.2556716038903888, -4.4711701846066552, -3.8045080792392056], [1.6736770707519555, -2.3439638070319315, 
3.7536285769898647, -1.6272461442774846]], [[-6.1007124021739632, -0.19516486274205391, -1.8342166852925645, 
8.0129586161038553], [-4.3699471579431997, 2.3127393446994708, 2.2788933780585916, 5.1687530977862099], [-0.081228524499265653, 
-4.3960638723276695, 9.759713780753259, -3.0969425091389002]]], [[[0.30329520893807516, -2.1564671254375627, 
-4.3034427305934111, 3.6483854787361074], [-0.68854255535022801, -2.1116336381671141, -4.6341777703263674, 4.43726831302817], 
[4.1833818698130028, -6.1506555180465341, -1.8265387010134893, -6.1159700284600511]], [[4.5751203778280791, 
-6.8552861548571329, 3.5497396115667366, 2.8610183518816221], [-1.8646749278182364, 4.0088944339421646, -8.5879758759355553, 
1.2296928440529653], [-9.0814818708407792, 4.4656322049960187, 6.1625880758948153, -2.811381720532653]]], 
[[[5.0018125577689734, 6.7897365102671845, 2.2793153396743389, 4.6715601833115938], [-6.7938847086443541, -3.8143087323890921, 
0.1005463722212907, 2.3155671794854524], [-5.289915817362818, 2.5816510703407829, 3.6423251148041573, -4.4980137659877997]], 
[[6.8376068040067306, -2.8246750905537841, 2.9284689527396273, -6.544324333942475], [6.512623568063189, 0.31061977465928559, 
-1.8026138627267936, -1.9914442349584518], [-0.43041263600665136, -1.9806819741932502, 8.0501031827905827, 
2.0961945456476445]]]])+(1.-msk_ref)*numpy.array([[[[-2.4190475119857204, -5.7443597487042872, -3.7235955886949457, 
-2.2533519516326694], [-0.52411665685003594, -3.4822556717161355, -1.1185300229525508, 2.5738521978799662], 
[3.8702359208431067, -1.0860912513543566, -1.7919823228259792, -2.7521595690007716]], [[-8.0557415299382846, 
-2.6140642824027571, -2.5360856267921825, 6.0295920431563106], [-0.30329543050925345, 4.9458290307239015, 3.425238227861529, 
5.4039182502034091], [-8.2062810234154462, 0.21450627753645035, 2.6460031848376007, 0.18499163599555413]]], 
[[[2.4492891534055388, 2.485879870036964, 2.2007365774514263, -0.065910109136829931], [-6.8744006446191257, 4.4693694252626104, 
-4.1609023883926328, 4.4818893944345088], [-0.28905831145946159, -0.53748381482233842, -1.7339005439505053, 
-5.2855523005900755]], [[4.7046737640849026, -2.9557610834775199, -4.3475518276080924, 3.483627334616112], 
[-4.0906354866337855, -2.893319647120391, -0.71982841560837496, -2.6049921823377593], [-2.4717110670221949, 5.4927024659786197, 
6.1243105200948911, 3.6004080804846845]]], [[[1.9929638834428083, 3.0280708614780245, 2.8935132047850569, -1.1080664103347528], 
[-1.7856467836183105, -0.76125863422368578, -5.4773333675617897, 0.90652056684618421], [4.3529635517570693, 2.7416725798817358, 
2.8672339646066001, -2.627733050253851]], [[3.2820289313478828, -1.4557790826651305, -2.1394051918812815, 1.1814306672405914], 
[4.0889559018787844, 2.5427579174927986, 4.871446381082599, -2.0690870246211359], [-4.3937354037154206, -5.9716631033151888, 
5.0943304067742865, 2.3712660451053251]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[1.2741274601283292, 2.4420109240018704, 4.3168961020338177, -1.9302074879801823], 
[-2.0454719197065421, 0.90405110934699362, 1.4343952561392967, -0.5497030906333169], [3.6823167312387834, 1.6050223744188781, 
2.5014956184919912, -3.7415810331896813]], [[4.9736727889898251, 3.008266729900912, 4.2102584626037025, -1.4023346242092005], 
[-0.093498513416303553, 4.023040763654997, -4.0085085812651711, -2.0526029150885305], [-4.4566084359436964, 
-3.5056420266082036, -3.6900539053587811, -0.98941151228587998]]], [[[3.1563166599133119, -2.3618573528178777, 
-1.8795051365047533, -0.59823778607333189], [1.8128589736128387, 3.4809740790896733, 0.95134379807439995, 
-0.85451111326402085], [2.7058454560487641, -2.4479896226014408, -4.0668570505167869, -3.1707364951148911]], 
[[3.6371506892861145, 3.8490905290959692, 4.1829301602304376, 4.5282864539058725], [2.4494531339133676, 3.0608371186247378, 
-3.9217382400514533, 0.24817164727456742], [-4.3509291265664487, 1.7371063171518406, -3.6135835196294188, 
4.6779536569492652]]], [[[1.0475720803660682, 4.9974471941967824, -0.21426069120806357, 0.29429139684864225], 
[3.9488700308849083, -1.965946856775167, 1.7617298343965881, 4.5709687768262164], [1.2127547477636602, 2.5405075922701368, 
1.9269288677113403, 4.9489306797895143]], [[-1.1444406354007186, 2.4975776008253616, -3.8317044672809599, -1.8299522982438257], 
[2.1347243499975601, -0.5465175004244216, -1.8916758458343388, -1.5337808011934664], [-3.108089264483116, 2.1471662644827836, 
0.42386586192520248, -2.1628008189551484]]]])+(1.-msk_arg0)*numpy.array([[[[1.4285424445714332, -0.73406422991392439, 
0.80007271057191076, 4.4012140683865013], [3.6171351222474062, 0.060998819868251175, 1.8633585204533176, 2.0333541381547535], 
[3.6943547105978425, -1.767782864794083, 1.8228465038283259, 0.13681492948203644]], [[-0.78444128508515032, 3.32864794554723, 
2.71994794514071, -1.353882815929639], [-0.98052817557386707, -2.4250548224775437, -2.6753182610686221, 1.9439693086881995], 
[1.5401254526222523, 0.029574514480290581, 4.9775232957412623, -4.7422939518893417]]], [[[0.008007601774315809, 
-2.2771120401472422, -3.2015581127160377, -4.3897047776594142], [1.5006143753817405, 3.0041424449241845, -0.62648237120416539, 
-1.9925844977714604], [1.2400694882836572, -0.94599323132174185, -2.2875979144439071, 0.06717404211857847]], 
[[1.2292517491139678, 2.3380170265533327, 4.5993353949094047, -4.1277464881549992], [1.5428328619390976, 1.3173978892583964, 
2.3285807870582715, 1.0932107206888872], [4.3207248850660491, -1.4073622327507982, 1.7812581589656231, -1.973035244019564]]], 
[[[4.5989252749762652, -4.6250418719784472, -3.9528919535158167, -4.276794895070509], [4.4192281902442154, 
-0.49328173107990647, 0.14824006920428801, 1.1406939438029635], [0.23240795817943649, 0.82114899044633205, 0.24819549255305162, 
4.7673383066852644]], [[-1.4538423270310719, 1.5522498396456719, -1.1936127420572284, -4.078730093787474], 
[-1.5110869103076641, 4.9342197190052666, -1.5820685808168533, -1.2330490301442043], [0.47878996891634085, 4.438122190981229, 
-3.1906094044554911, -3.3420155804495066]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.20588182578)+(1.-msk_arg1)*(1.97997606244)
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[2.4800092859057079, 3.647892749779249, 5.5227779278111964, -0.72432566220280359], 
[-0.83959009392916339, 2.1099329351243723, 2.6402770819166754, 0.65617873514406178], [4.8881985570161621, 2.8109042001962568, 
3.7073774442693699, -2.5356992074123026]], [[6.1795546147672038, 4.2141485556782907, 5.4161402883810812, -0.19645279843182184], 
[1.1123833123610751, 5.2289225894323756, -2.8026267554877924, -0.84672108931115186], [-3.2507266101663177, -2.2997602008308249, 
-2.4841720795814024, 0.2164703134914987]]], [[[4.3621984856906906, -1.155975527040499, -0.67362331072737458, 
0.60764403970404679], [3.0187407993902173, 4.6868559048670519, 2.1572256238517786, 0.35137071251335783], [3.9117272818261428, 
-1.2421077968240621, -2.8609752247394082, -1.9648546693375124]], [[4.8430325150634932, 5.0549723548733478, 5.3888119860078163, 
5.7341682796832512], [3.6553349596907463, 4.2667189444021165, -2.7158564142740746, 1.4540534730519461], [-3.14504730078907, 
2.9429881429292193, -2.4077016938520401, 5.8838354827266439]]], [[[2.2534539061434469, 6.2033290199741611, 0.99162113456931511, 
1.5001732226260209], [5.154751856662287, -0.76006503099778833, 2.9676116601739668, 5.7768506026035951], [2.4186365735410389, 
3.7463894180475155, 3.132810693488719, 6.154812505566893]], [[0.061441190376660071, 3.7034594266027403, -2.6258226415035812, 
-0.62407047246644698], [3.3406061757749388, 0.65936432535295708, -0.68579402005696011, -0.3278989754160877], 
[-1.9022074387057373, 3.3530480902601623, 1.6297476877025812, 
-0.95691899317776974]]]])+(1.-msk_ref)*numpy.array([[[[3.4085185070153026, 1.245911832529945, 2.7800487730157801, 
6.3811901308303707], [5.5971111846912756, 2.0409748823121205, 3.843334582897187, 4.0133302005986229], [5.6743307730417119, 
0.21219319764978639, 3.8028225662721953, 2.1167909919259058]], [[1.195534777358719, 5.3086240079910993, 4.6999240075845794, 
0.62609324651423037], [0.9994478868700023, -0.4450787600336743, -0.69534219862475277, 3.9239453711320689], [3.5201015150661217, 
2.0095505769241599, 6.9574993581851317, -2.7623178894454723]]], [[[1.9879836642181852, -0.29713597770337286, 
-1.2215820502721684, -2.4097287152155449], [3.4805904378256098, 4.9841185073680538, 1.353493691239704, -0.012608435327591039], 
[3.2200455507275265, 1.0339828311221275, -0.30762185200003778, 2.0471501045624478]], [[3.2092278115578372, 4.317993088997202, 
6.5793114573532741, -2.1477704257111299], [3.522808924382967, 3.2973739517022658, 4.3085568495021409, 3.0731867831327566], 
[6.3007009475099185, 0.57261382969307117, 3.7612342214094925, 0.0069408184243053839]]], [[[6.5789013374201346, 
-2.6450658095345778, -1.9729158910719473, -2.2968188326266397], [6.3992042526880848, 1.4866943313639629, 2.1282161316481574, 
3.1206700062468329], [2.2123840206233059, 2.8011250528902014, 2.228171554996921, 6.7473143691291337]], [[0.52613373541279751, 
3.5322259020895412, 0.78636332038664092, -2.0987540313436046], [0.4688891521362053, 6.9141957814491359, 0.39790748162701606, 
0.74692703229966506], [2.4587660313602102, 6.4180982534250983, -1.2106333420116218, -1.3620395180056373]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-2.5424743209543124, -2.240775038265955, -2.8942850917693708, -0.55816606120069245], 
[-2.4090922895815226, -1.7839232051257969, 0.18774876502316484, -0.58599814475932543], [4.5121687114968445, 
0.023730546611671066, 2.8120728119534943, 0.51612927302434564]], [[-3.0180418285653987, -4.2250128922634378, 
3.3820178252026647, -3.4052568671305483], [-4.0176233597796527, 2.1552294038593711, -0.13254235414775106, -2.0187361982360263], 
[-0.3329275378119636, 1.8464673537324945, -2.4676055868060418, -0.90550582103086175]]], [[[-3.5444158970008877, 
-3.389497974858291, 0.65025235966348927, 3.1530185318312967], [0.21874945015708658, -3.7072438792512843, 4.4801474102015089, 
0.17745631515143945], [4.5977765744146044, -4.8415393178062258, 0.31616720923711483, 4.6851168163693622]], 
[[1.0102309663736744, -0.85875464397337975, 2.2498864631045628, 1.388586864605351], [-2.543612390496869, -0.25668778023582828, 
1.0891170314932168, 4.8909566521270165], [-4.1549781213487673, 1.3699502533386374, -3.078557589329546, 1.7761769949594228]]], 
[[[2.4009368700492653, -4.0521592485669498, -2.3373889279837403, 4.499346456777273], [-3.8206584901560636, -4.0233647180331955, 
-2.5575729342917963, 0.97027474440004013], [-3.7218477397844874, -4.160907723585261, 3.716698454586453, 1.8277961998748484]], 
[[-2.2920423989172312, 0.72329531559669125, 4.2753717756098766, 1.9748720907747934], [3.4748361692668013, -0.10051914846652377, 
4.7398308122788997, -4.2959375258694514], [1.2074270586029554, -3.367554099792299, 3.1795042684062835, 
-2.5377702687034951]]]])+(1.-msk_arg0)*numpy.array([[[[3.4830982281014506, -2.2761401911964043, -2.7498487849279742, 
-3.4130600559722279], [-0.49977492006118496, -0.70247314609590283, 2.4657314699595734, -4.1824463976263448], 
[0.17501706732272027, 3.472915748543965, -4.2974180450913178, 0.52719392693488132]], [[-3.418814946827593, 
-0.10744295091129619, 0.37071350973561579, -0.54590028070249552], [0.540745645655619, -0.73680315484965675, 
-2.9543507311359987, -3.4400150527800442], [-2.7652656242137121, 4.8829273075447439, -3.539769978578863, 
-2.3650384175523955]]], [[[1.8788673146249115, -1.4854659506775505, -4.9829258288210134, -3.4805055311488275], 
[3.4861632010287806, -2.7597224620022889, -2.1308621497948597, 4.5683622295190993], [-4.5588550861373198, -0.39508913930874101, 
-3.1913667754968555, -4.2064932924709542]], [[-1.5441134541283041, -0.063276425877631048, 2.0829953853515164, 
4.4595500364007172], [3.3100790449675106, 0.87008030693235128, 4.6038912671527967, -2.0655956309509973], [1.440003627936397, 
-4.8927882541478915, 2.8715950451266128, 2.92753997977022]]], [[[1.0693172073376278, -1.7298628346761733, 0.072478640233073932, 
-4.3507471028241191], [-4.018912995816506, -0.59275034530572679, 2.3705197652038272, -2.9624096881868898], 
[-4.2880997367574567, 4.4776185660484096, -0.39349078823759598, -3.8491343873930841]], [[3.6342636075260373, 
-1.0704138720010858, -4.9855848100635383, 2.6160822940859818], [2.1085171173339408, 2.8905094710039849, -3.6986826821442742, 
3.3872191279929194], [3.9641907258200106, 2.1653002364179974, 2.8746367439988418, 2.3517880229425128]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[1.9343111611753248, 1.2617568145285425, -3.75770843270816, -3.0622952501741283], 
[2.0537210681331883, 0.47436329939258304, -1.4166767625275654, -1.6059403862683466], [0.82852283503851254, -2.877749451326407, 
3.903754834807998, 4.1210410229142447]], [[4.9407624695140093, 2.91138337584289, -0.29608507106283355, -3.86920703957496], 
[-4.8859980921992516, 4.38537180805708, -3.6505088657674669, 3.8781420005010734], [1.3720141131868768, -0.94833711861769832, 
0.70715947571459914, 3.4096858818736351]]], [[[1.636457268166204, 4.8966517112493051, 3.189851920726877, -2.2527945596596299], 
[0.89666756070707176, 4.4682832127178944, -1.5560420869433234, 2.7336811976698847], [2.5193643965620813, 2.9127209321943148, 
-2.9766329297788729, -4.0120357590907121]], [[1.6937300042236183, -4.3665779897086043, 0.93918441766315652, 
2.6124824043890928], [-3.8927331433021695, 2.933268392078741, 1.1244810171308064, -4.6818667151742153], [3.0691671857638188, 
-0.90963694310961785, -3.3907498680137826, -2.3771279420312386]]], [[[2.2289311141733359, 2.8200759539297522, 
2.2615576544969365, 2.7095659038479329], [-1.0348377910883255, -2.4743384731036357, 2.0010147943053234, 4.333143919480376], 
[-2.9548462877006765, -2.2883361637452229, -0.14918710570680815, -2.8226666925957011]], [[-3.2192870145883878, 
1.7501853863902195, -2.559045042985967, 1.2253382943168436], [0.93729091131790554, 4.0987604346917834, 2.009972518973874, 
-3.3706033584635331], [-4.0526136490191833, 4.9728174302886732, 0.58777739581224964, 
-1.1371768791387735]]]])+(1.-msk_arg1)*numpy.array([[[[-0.4023329405977556, 4.228058186826912, -0.311936890697039, 
4.1657988488978503], [-2.616997788426688, 2.0034718232832782, 3.0991751166991328, -0.3315863264626282], [3.3346855557934703, 
4.7347234015447537, -0.96598531390660636, 1.4821487976278025]], [[-3.3298327652370405, 4.6144630711899026, 1.6025374622245936, 
-3.7370962686171438], [0.88751064169954041, 4.8978889509393984, 1.8267404329001833, 4.900923239559674], [-1.7413183615520014, 
3.9399481706481936, 4.9134028737563078, -4.4996927089116561]]], [[[-1.9420201011551064, -1.4353640332378803, 
4.8443180116291789, 2.4364103927750289], [-1.0357872187761963, 1.9001790276055877, 1.9727832806384988, -2.6996053963134994], 
[2.2893496560411286, -0.11482776804507111, -3.3763412122413881, -0.48185993529149584]], [[-0.51389765208460858, 
1.8212087928337448, -1.8208411970210658, 2.0055111753374373], [0.89870815921910818, -4.302352211104882, -3.9537025745349785, 
-2.3134882210540231], [3.0823772228919317, 0.27336115342283129, 3.2916667836247822, -2.3647621692521756]]], 
[[[-2.4195604255745327, 4.3946308585591218, -0.50181708104574252, -1.539510435885644], [2.9118540797695358, 4.860782198643772, 
3.4768241143903786, 3.7863278166443042], [-0.083698893003277597, -1.6731592958597918, -1.553361452126063, 
-2.4634732124054537]], [[2.033179422266695, -3.8899455535678715, -2.416024871071174, -4.5353503084299973], [2.8399620879996359, 
-2.1829328965912542, 4.3754728386833932, -1.6579523511277836], [4.2559308471669492, -4.8795735288647633, -2.0012616380139936, 
0.56202324224949329]]]])
      res=arg0+arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.60816315977898761, -0.97901822373741254, -6.6519935244775308, -3.6204613113748207], 
[-0.35537122144833422, -1.3095599057332139, -1.2289279975044005, -2.191938531027672], [5.340691546535357, -2.854018904714736, 
6.7158276467614924, 4.6371702959385903]], [[1.9227206409486106, -1.3136295164205478, 3.0859327541398311, -7.2744639067055079], 
[-8.9036214519789034, 6.5406012119164512, -3.783051219915218, 1.859405802265047], [1.0390865753749132, 0.89813023511479617, 
-1.7604461110914427, 2.5041800608427733]]], [[[-1.9079586288346837, 1.5071537363910141, 3.8401042803903662, 
0.90022397217166672], [1.1154170108641583, 0.76103933346661012, 2.9241053232581855, 2.9111375128213242], [7.1171409709766857, 
-1.928818385611911, -2.6604657205417581, 0.67308105727865009]], [[2.7039609705972927, -5.2253326336819841, 3.1890708807677193, 
4.0010692689944438], [-6.4363455337990381, 2.6765806118429127, 2.2135980486240232, 0.2090899369528012], [-1.0858109355849486, 
0.46031331022901956, -6.4693074573433282, -0.60095094707181573]]], [[[4.6298679842226012, -1.2320832946371976, 
-0.075831273486803852, 7.2089123606252059], [-4.8554962812443891, -6.4977031911368313, -0.55655813998647297, 
5.3034186638804162], [-6.6766940274851638, -6.4492438873304838, 3.5675113488796448, -0.99487049272085271]], 
[[-5.511329413505619, 2.4734807019869107, 1.7163267326239096, 3.200210385091637], [4.4121270805847068, 3.9982412862252596, 
6.7498033312527737, -7.666540884332985], [-2.8451865904162279, 1.6052633304963742, 3.7672816642185332, 
-3.6749471478422686]]]])+(1.-msk_ref)*numpy.array([[[[3.080765287503695, 1.9519179956305077, -3.0617856756250132, 
0.7527387929256224], [-3.1167727084878729, 1.3009986771873754, 5.5649065866587062, -4.514032724088973], [3.5097026231161905, 
8.2076391500887187, -5.2634033589979241, 2.0093427245626838]], [[-6.7486477120646331, 4.5070201202786064, 1.9732509719602094, 
-4.2829965493196394], [1.4282562873551594, 4.1610857960897416, -1.1276102982358154, 1.4609081867796299], [-4.5065839857657135, 
8.8228754781929375, 1.3736328951774448, -6.8647311264640516]]], [[[-0.063152786530194849, -2.9208299839154308, 
-0.13860781719183457, -1.0440951383737986], [2.4503759822525844, -0.85954343439670122, -0.15807886915636082, 
1.8687568332055999], [-2.2695054300961912, -0.50991690735381212, -6.5677079877382436, -4.68835322776245]], 
[[-2.0580111062129127, 1.7579323669561138, 0.26215418833045057, 6.4650612117381545], [4.2087872041866188, -3.4322719041725307, 
0.6501886926178182, -4.3790838520050208], [4.5223808508283287, -4.6194271007250602, 6.1632618287513949, 0.56277781051804432]]], 
[[[-1.3502432182369049, 2.6647680238829485, -0.42933844081266859, -5.8902575387097631], [-1.1070589160469702, 
4.2680318533380452, 5.8473438795942059, 0.82391812845741441], [-4.3717986297607343, 2.8044592701886177, -1.946852240363659, 
-6.3126075997985378]], [[5.6674430297927323, -4.9603594255689574, -7.4016096811347118, -1.9192680143440155], 
[4.9484792053335767, 0.7075765744127307, 0.676790156539119, 1.7292667768651357], [8.2201215729869599, -2.7142732924467659, 
0.87337510598484824, 2.9138112651920061]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_constData_rank0(self):
      arg0=2.90218798914
      arg1=Data(1.45689271181,self.functionspace)
      res=arg0-arg1
      ref=Data(1.44529527733,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_constData_rank1(self):
      arg0=1.64373899131
      arg1=Data(numpy.array([-1.8194731196281455, -3.5286564150010138]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([3.4632121109336427, 5.172395406306511]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_constData_rank2(self):
      arg0=-2.44144081472
      arg1=Data(numpy.array([[-1.298774764818782, -2.6032344931607208, -4.139122830137314, -3.6232165967981942, 
-4.6179941842686549], [2.2891811471442693, 0.16662362669696229, -2.8894288433212614, 0.013867308882426599, 1.1075543015816933], 
[-4.0405920783978457, -3.3592620886355218, 0.13984059709823971, -4.5267187173921624, 0.2173165151747849], [3.1979826293178686, 
-4.0655740999189263, -2.6174635377699618, 3.4117027459523523, 3.2329240978894056]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[-1.1426660498981147, 0.16179367844382408, 1.6976820154204173, 1.1817757820812975, 
2.1765533695517583], [-4.730621961861166, -2.608064441413859, 0.44798802860436471, -2.4553081235993233, -3.54899511629859], 
[1.5991512636809491, 0.91782127391862511, -2.5812814118151364, 2.0852779026752657, -2.6587573298916816], [-5.6394234440347653, 
1.6241332852020296, 0.1760227230530651, -5.853143560669249, -5.6743649126063023]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_constData_rank3(self):
      arg0=-2.84332204179
      arg1=Data(numpy.array([[[4.777846845241422, 0.99122240581566157], [1.1186757270642778, -1.6222246032165422]], 
[[0.48837787296932778, -3.3141506878054416], [4.0273997786848206, -1.8607991957562953]], [[3.6448910844159101, 
-1.4581470978929745], [4.3948974413336668, -3.8400836560460681]], [[-4.3822779335248292, -0.75515656774557183], 
[3.8489952361427058, 2.2786123607841349]], [[3.8238453467262765, -0.58458007163440051], [-0.58459665410840689, 
2.897113720603306]], [[1.2070711566859318, -2.5215257220590259], [-4.2778794577395249, 
-2.0145493417191607]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[-7.6211688870352008, -3.8345444476094404], [-3.9619977688580565, -1.2210974385772366]], 
[[-3.3316999147631066, 0.47082864601166285], [-6.8707218204785994, -0.98252284603748352]], [[-6.4882131262096889, 
-1.3851749439008043], [-7.2382194831274456, 0.99676161425228926]], [[1.5389558917310504, -2.088165474048207], 
[-6.6923172779364846, -5.1219344025779137]], [[-6.6671673885200553, -2.2587419701593783], [-2.2587253876853719, 
-5.7404357623970848]], [[-4.0503931984797106, -0.32179631973475287], [1.4345574159457462, 
-0.82877270007461812]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_constData_rank4(self):
      arg0=-4.47292133193
      arg1=Data(numpy.array([[[[2.4974904778000164, 1.799682830032765, 3.1133037721991936, 3.669990517825882], 
[-2.5157588637131099, -3.9788373211961439, 2.3526988085959877, 2.3694821204400185], [-3.6732623237882822, 
-0.032207726870451481, 2.7629371931716085, 4.4172565924096379]], [[-2.1694941338343976, 0.42242395372966257, 
1.6941391502994438, 4.6897099239149824], [-2.3285753443498391, -2.7239414460802558, 4.9045586536212706, -3.7126025393925777], 
[1.9306379612208495, 0.22659890897778823, 4.374393711351825, -0.58781606865313751]]], [[[0.10644047934004774, 
2.5280547208373791, 0.85576719982359606, 3.5552601761167608], [0.33809324345913261, -4.5711115062099044, 3.5469982569338026, 
-2.450039713651849], [-4.7722784599974304, 2.7342164353932041, 0.55570301645330833, 3.0694659342754189]], 
[[-0.057336695334692322, 0.56892286106246992, -4.4331880296734916, 4.9392845133670278], [-2.4861437677060128, 
0.032976366581300987, -3.2705096880323179, 4.3561896413317598], [4.6103118460791528, 4.2184820406196835, 3.233604623860284, 
1.4113772946287479]]], [[[-3.8711194544824989, -3.8791260159761123, 1.2386256810163534, 0.0056978588581841905], 
[-2.7200005528296045, -3.5013183787084756, -3.5404729085941824, -0.27587087095474327], [-2.5385378222478403, 
4.7569667905422985, 4.0564450594532246, 3.7571113072719715]], [[-3.5065730652285332, -2.0981902983242673, -2.9450671727986641, 
-2.7366818470347232], [-0.41202621176505172, -0.69718576824130718, -2.1373678853216207, -2.1494931348022552], 
[3.6303609743266794, -2.6508321328256965, 2.4269213440931159, -2.695758146471996]]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[[-6.9704118097333501, -6.2726041619660986, -7.5862251041325273, -8.1429118497592157], 
[-1.9571624682202238, -0.49408401073718977, -6.8256201405293213, -6.8424034523733521], [-0.79965900814505142, 
-4.4407136050628822, -7.2358585251049421, -8.8901779243429715]], [[-2.303427198098936, -4.8953452856629962, 
-6.1670604822327775, -9.1626312558483161], [-2.1443459875834945, -1.7489798858530778, -9.3774799855546043, 
-0.76031879254075596], [-6.4035592931541832, -4.6995202409111219, -8.8473150432851586, -3.8851052632801961]]], 
[[[-4.5793618112733814, -7.0009760527707128, -5.3286885317569297, -8.0281815080500945], [-4.8110145753924662, 
0.09819017427657073, -8.0199195888671362, -2.0228816182814846], [0.29935712806409676, -7.2071377673265378, -5.028624348386642, 
-7.5423872662087525]], [[-4.4155846365986413, -5.0418441929958036, -0.039733302259842063, -9.4122058453003614], 
[-1.9867775642273209, -4.5058976985146346, -1.2024116439010157, -8.8291109732650934], [-9.0832331780124864, 
-8.6914033725530171, -7.7065259557936177, -5.8842986265620816]]], [[[-0.60180187745083469, -0.59379531595722135, 
-5.711547012949687, -4.4786191907915178], [-1.7529207791037291, -0.97160295322485801, -0.93244842333915123, 
-4.1970504609785904], [-1.9343835096854933, -9.2298881224756322, -8.5293663913865583, -8.2300326392053051]], 
[[-0.96634826670480045, -2.3747310336090663, -1.5278541591346695, -1.7362394848986105], [-4.0608951201682819, 
-3.7757355636920265, -2.3355534466117129, -2.3234281971310784], [-8.103282306260013, -1.8220891991076371, -6.8998426760264495, 
-1.7771631854613377]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_expandedData_rank0(self):
      arg0=-0.771334291858
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.27875083825)+(1.-msk_arg1)*(-2.6811194061)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-3.0500851301)+(1.-msk_ref)*(1.90978511424)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_expandedData_rank1(self):
      arg0=4.66809552948
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([3.4294739197862825, 3.9274817030272331])+(1.-msk_arg1)*numpy.array([-0.86678300414400145, 
4.7539604129938642])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.2386216096923679, 0.74061382645141727])+(1.-msk_ref)*numpy.array([5.5348785336226518, 
-0.085864883515213819])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_expandedData_rank2(self):
      arg0=3.35656462337
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-3.2986443829283529, -3.6187111459019117, -2.5708695030442898, -4.2823693388837274, 
-3.4936390578980037], [3.7406624654488603, 2.2911606945716745, 2.7175673456170095, -4.7861630187263691, -1.4233370724976102], 
[4.3534339301738569, -1.9097610099197526, 3.1111745977158218, -3.0691966515510902, 1.267898100900692], [-1.8327720187814056, 
1.793311082787616, -1.1663260854018622, 3.0413349528336227, 
-3.5511047270749674]])+(1.-msk_arg1)*numpy.array([[-4.9346894567675754, -3.6310479180985613, -1.5193790152510314, 
-3.6594761620904412, 2.6688158009386962], [-3.7039767549070959, -3.749323776405987, 0.22377964399257877, 1.5013477518626726, 
1.6742669924812947], [4.024609435423983, -1.1515915544114605, -2.658384651906307, -2.7039367749890686, -0.61403623194649271], 
[-3.49221615113537, -0.46935514534542833, -3.7527718210916317, 2.1988300618415648, -3.9348376779204552]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[6.6552090062938856, 6.9752757692674443, 5.927434126409822, 7.6389339622492596, 
6.8502036812635358], [-0.38409784208332809, 1.0654039287938577, 0.63899727774852266, 8.1427276420919021, 4.779901695863142], 
[-0.99686930680832475, 5.2663256332852848, 0.24539002564971035, 6.4257612749166224, 2.0886665224648402], [5.1893366421469374, 
1.5632535405779162, 4.5228907087673944, 0.31522967053190953, 6.9076693504405]])+(1.-msk_ref)*numpy.array([[8.2912540801331076, 
6.9876125414640935, 4.8759436386165635, 7.0160407854559734, 0.68774882242683599], [7.0605413782726281, 7.1058883997715192, 
3.1327849793729534, 1.8552168715028596, 1.6822976308842374], [-0.66804481205845079, 4.5081561777769927, 6.0149492752718388, 
6.0605013983546012, 3.9706008553120249], [6.8487807745009022, 3.8259197687109605, 7.1093364444571634, 1.1577345615239674, 
7.2914023012859879]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_expandedData_rank3(self):
      arg0=-3.47740305947
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-2.4385844166051673, 4.6468974471865092], [2.6945798004059291, -3.9792855272549241]], 
[[3.615167755877998, -1.5587667459916679], [1.9778411027409524, 4.6400125804031553]], [[-4.7821855990875815, 
2.3797611502543203], [-1.0518835110726821, -1.1822250034309678]], [[3.3205834818133777, -0.67615375037752745], 
[2.1942460825612731, 2.9892869208465944]], [[-1.92696377446509, -3.4155417896711482], [3.1847801134757461, 
3.0048882196193389]], [[2.7802440248296589, 1.8547414280965437], [-4.1425042018312883, 
-2.4363123921100382]]])+(1.-msk_arg1)*numpy.array([[[-4.5702855202538286, -4.8903461196721629], [1.2979777369490453, 
4.5887578154749828]], [[-2.0568765937548141, -1.9444417113219012], [-0.094265854149097272, -1.7295258197663212]], 
[[-2.9712160803723418, -2.7755423277631195], [-1.8813990407453316, -2.2507723269093951]], [[3.8100714463767655, 
4.3948316690857929], [0.55207567274196379, 4.6614552384627164]], [[2.7488203066141867, 0.33495471239277919], 
[-0.17891662283854259, 2.6237690653951118]], [[0.8960835211807483, 1.9555522116718453], [0.52209724959174952, 
-4.6998873253780316]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.038818642867859, -8.1243005066595355], [-6.1719828598789555, 0.50188246778189782]], 
[[-7.0925708153510243, -1.9186363134813584], [-5.4552441622139787, -8.1174156398761816]], [[1.3047825396145551, 
-5.8571642097273466], [-2.4255195484003442, -2.2951780560420585]], [[-6.797986541286404, -2.8012493090954989], 
[-5.6716491420342994, -6.4666899803196207]], [[-1.5504392850079363, -0.061861269801878116], [-6.6621831729487724, 
-6.4822912790923652]], [[-6.2576470843026852, -5.33214448756957], [0.66510114235826201, 
-1.0410906673629881]]])+(1.-msk_ref)*numpy.array([[[1.0928824607808023, 1.4129430601991366], [-4.7753807964220716, 
-8.0661608749480092]], [[-1.4205264657182122, -1.5329613481511251], [-3.383137205323929, -1.7478772397067051]], 
[[-0.50618697910068455, -0.70186073170990682], [-1.5960040187276947, -1.2266307325636312]], [[-7.2874745058497918, 
-7.8722347285588192], [-4.0294787322149901, -8.1388582979357427]], [[-6.2262233660872131, -3.8123577718658055], 
[-3.2984864366344837, -6.1011721248681381]], [[-4.3734865806537746, -5.4329552711448716], [-3.9995003090647758, 
1.2224842659050053]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_float_rank0_expandedData_rank4(self):
      arg0=4.70609709674
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.65717690520202243, -3.0832044891606825, -0.74481896598197928, -2.0522135913781092], 
[-0.81751260922484636, 4.4580503457704452, 1.9212110737036596, 3.2252615874961172], [1.7089914553114784, 0.61805223066829473, 
-3.2617883483889578, -0.21276379302143589]], [[-1.2459154080717836, 1.2292587212892236, 4.7371515885461264, 
2.7048806253654734], [-4.0329469431140925, 0.083051822887910021, -0.36228007471570489, -1.0129340086460448], 
[2.9335125143239038, -2.8189966942032543, -1.4541698839648962, 3.2610931613959799]]], [[[1.1075130235510686, 
-4.2371433952851429, 1.2885227253133724, 3.4054094146837492], [1.2702008308001291, 4.7079663934056235, -0.6768767013640602, 
3.5550321584023497], [1.5766635807529976, -4.4364297680052811, 2.1424749064288662, -3.2601821219127567]], [[2.8126268131608425, 
-3.5702847983207784, 4.0113024066723302, 4.979971375921151], [-4.0091045955981457, -3.2004182984798266, -3.4921388997093672, 
-1.6803906372718358], [-2.276774604527577, 3.9917731707909923, -4.592532737262685, 1.6545271573040043]]], 
[[[-3.2661186200415147, 3.9093198000777711, 0.017693097708867178, -3.7407523156117248], [3.3785978973470048, 
4.2321535750053272, 3.2616548774213943, -0.46013555424037555], [0.58099002530795474, -3.6421696171076743, 3.6840255708075826, 
-1.7797254586118894]], [[3.0825986513700823, -4.4030639083490444, 1.2799683302202567, -3.8856552243570261], [3.439619360868388, 
4.4841599266029384, 0.047317644654895297, -2.4279010492457243], [-4.9285434826252832, -0.95692509815120452, 
-1.5210977139024129, -1.7003209369760839]]]])+(1.-msk_arg1)*numpy.array([[[[2.6548775349677589, -1.494364599242346, 
1.6729556628880964, 2.3410723155705337], [1.8927979312037202, 1.5356075037119528, 1.6035806718493806, 4.8516069627723244], 
[-2.7908972493563242, -0.659590186650588, 1.6043563245911692, -0.67283510908663668]], [[2.3879053133171615, 2.6562523889747371, 
1.6800427147252854, 3.9009168363020628], [-2.1482695515587746, 2.5183023091116405, -1.6086209050931766, 4.7444498612283788], 
[3.1820477211852083, -2.6199943751390387, 2.5876893701182322, 4.1109405627814546]]], [[[-0.35991551150180179, 
1.5768346278373491, 1.5352827775223856, -2.8388889856799882], [1.839500154401037, 2.9949990063493139, 4.8826217018080929, 
-4.7212588171537195], [-3.9449856886725931, 2.9440430846544601, 2.2914054396761632, -0.45528771085746289]], 
[[1.8277460070494769, -1.2702351566081371, -0.83628291616938455, 2.501812616972928], [2.3366938495220522, 3.3455529990481097, 
-2.948334807761912, 0.20216371639421116], [-2.3520925478061407, -3.3376498865943094, 2.6697184601861377, 2.2655379672548817]]], 
[[[0.48668777718253509, 0.60624114268257756, 4.4953053334335493, 4.19129616487146], [-1.6756772418478691, -1.3084701477711489, 
-2.5071743683862655, 2.5645466750946619], [-2.7307196172213857, -1.6796176671897856, 1.4198924817690184, 4.7651545836859075]], 
[[3.8407765530050355, 4.0382837006054508, -2.2355224309798594, 0.028597864854493338], [0.92721535942160571, 4.6025176172094753, 
-4.8889895605076497, -2.8138963090103877], [-2.0230251374174077, -1.0930241018577194, 0.26909593228275952, 
-3.4090940609713192]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[4.0489201915388913, 7.7893015859015957, 5.450916062722893, 6.7583106881190229], 
[5.5236097059657601, 0.24804675097046847, 2.7848860230372541, 1.4808355092447965], [2.9971056414294353, 4.088044866072619, 
7.9678854451298715, 4.9188608897623496]], [[5.9520125048126973, 3.4768383754516901, -0.031054491805212692, 2.0012164713754403], 
[8.7390440398550062, 4.6230452738530037, 5.0683771714566186, 5.7190311053869589], [1.7725845824170099, 7.525093790944168, 
6.1602669807058099, 1.4450039353449338]]], [[[3.5985840731898451, 8.9432404920260566, 3.4175743714275413, 1.3006876820571645], 
[3.4358962659407846, -0.0018692966647098075, 5.3829737981049739, 1.151064938338564], [3.1294335159879161, 9.1425268647461948, 
2.5636221903120475, 7.9662792186536704]], [[1.8934702835800712, 8.276381895061693, 0.69479469006858352, -0.27387427918023732], 
[8.7152016923390594, 7.9065153952207403, 8.1982359964502809, 6.3864877340127499], [6.9828717012684907, 0.71432392594992145, 
9.2986298340035987, 3.0515699394369094]]], [[[7.9722157167824284, 0.79677729666314256, 4.6884039990320465, 8.4468494123526376], 
[1.3274991993939089, 0.47394352173558651, 1.4444422193195194, 5.1662326509812893], [4.125107071432959, 8.348266713848588, 
1.0220715259333311, 6.4858225553528026]], [[1.6234984453708314, 9.109161005089959, 3.426128766520657, 8.591752321097939], 
[1.2664777358725257, 0.22193717013797531, 4.6587794520860184, 7.133998145986638], [9.6346405793661969, 5.6630221948921182, 
6.2271948106433266, 6.4064180337169976]]]])+(1.-msk_ref)*numpy.array([[[[2.0512195617731548, 6.2004616959832592, 
3.0331414338528173, 2.36502478117038], [2.8132991655371935, 3.1704895930289609, 3.1025164248915331, -0.1455098660314107], 
[7.4969943460972379, 5.3656872833915017, 3.1017407721497445, 5.3789322058275504]], [[2.3181917834237522, 2.0498447077661766, 
3.0260543820156283, 0.80518026043885094], [6.8543666482996883, 2.1877947876292732, 6.3147180018340903, -0.038352764487465052], 
[1.5240493755557054, 7.3260914718799519, 2.1184077266226815, 0.59515653395945911]]], [[[5.0660126082427155, 3.1292624689035646, 
3.1708143192185281, 7.5449860824209019], [2.8665969423398767, 1.7110980903915998, -0.17652460506717915, 9.4273559138946332], 
[8.6510827854135073, 1.7620540120864536, 2.4146916570647505, 5.1613848075983766]], [[2.8783510896914368, 5.9763322533490513, 
5.5423800129102982, 2.2042844797679857], [2.3694032472188615, 1.360544097692804, 7.6544319045028253, 4.5039333803467025], 
[7.0581896445470544, 8.0437469833352235, 2.036378636554776, 2.440559129486032]]], [[[4.2194093195583786, 4.0998559540583361, 
0.21079176330736438, 0.51480093186945375], [6.3817743385887828, 6.0145672445120626, 7.2132714651271792, 2.1415504216462518], 
[7.4368167139622994, 6.3857147639306993, 3.2862046149718953, -0.059057486944993798]], [[0.86532054373587819, 
0.66781339613546287, 6.9416195277207731, 4.6774992318864204], [3.778881737319308, 0.10357947953143842, 9.5950866572485634, 
7.5199934057513014], [6.7291222341583214, 5.7991211985986331, 4.4370011644581542, 8.115191157712232]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_float_rank0(self):
      arg0=Data(-1.71503107185,self.functionspace)
      arg1=1.12087823217
      res=arg0-arg1
      ref=Data(-2.83590930402,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_array_rank0(self):
      arg0=Data(1.8624638034,self.functionspace)
      arg1=numpy.array(-3.54267680565)
      res=arg0-arg1
      ref=Data(5.40514060905,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_array_rank1(self):
      arg0=Data(-1.48075996283,self.functionspace)
      arg1=numpy.array([0.85109202237095083, 3.6576817935890773])
      res=arg0-arg1
      ref=Data(numpy.array([-2.3318519851995503, -5.1384417564176772]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_array_rank2(self):
      arg0=Data(2.28159918662,self.functionspace)
      arg1=numpy.array([[0.23401101560112902, -1.4346807696676089, 2.1738467120575011, 4.9988436139857821, 1.8333893134121695], 
[2.165547502020968, -0.91977904138086242, 4.2012367738129068, -1.3347405763270146, -3.8870880700865982], [2.5516106084178052, 
1.4376942518164268, -0.86451569411581719, 0.17754376364930291, -4.3367088808026386], [-3.677360364986233, 2.6548742342264262, 
4.0262382941103674, -0.56210120506773986, 3.7310292944968193]])
      res=arg0-arg1
      ref=Data(numpy.array([[2.0475881710152128, 3.7162799562839508, 0.10775247455884074, -2.7172444273694403, 
0.44820987320417238], [0.11605168459537385, 3.2013782279972043, -1.919637587196565, 3.6163397629433565, 6.16868725670294], 
[-0.27001142180146331, 0.84390493479991502, 3.1461148807321591, 2.104055422967039, 6.6183080674189805], [5.9589595516025753, 
-0.37327504761008434, -1.7446391074940255, 2.8437003916840817, -1.4494301078804774]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_array_rank3(self):
      arg0=Data(2.88177606978,self.functionspace)
      arg1=numpy.array([[[-2.2010612015226716, -2.7677786223762024], [-4.5332674708171332, -4.7147600694763092]], 
[[2.2439105227643052, 2.3948224438035703], [-3.0179267745478522, 4.7979312568443113]], [[4.1839212478865626, 
-0.86662835044707176], [-1.8580062393766141, 1.0658371072778987]], [[4.8031103120316985, 1.483893938774477], 
[1.3012357748827599, -3.8926550740602037]], [[-4.8606402089231446, 4.0920460166151322], [0.27280537694079499, 
2.164898352512016]], [[-3.4086220235183395, -0.9570314787135592], [-1.2985041133443573, -4.9061626580500644]]])
      res=arg0-arg1
      ref=Data(numpy.array([[[5.0828372712999981, 5.6495546921535293], [7.4150435405944597, 7.5965361392536357]], 
[[0.63786554701302123, 0.48695362597375613], [5.8997028443251782, -1.9161551870669848]], [[-1.3021451781092361, 
3.7484044202243982], [4.7397823091539406, 1.8159389624994278]], [[-1.921334242254372, 1.3978821310028495], [1.5805402948945666, 
6.7744311438375302]], [[7.7424162787004711, -1.2102699468378058], [2.6089706928365315, 0.71687771726531047]], 
[[6.290398093295666, 3.8388075484908857], [4.1802801831216838, 7.7879387278273908]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_array_rank4(self):
      arg0=Data(-1.79772982697,self.functionspace)
      arg1=numpy.array([[[[3.7980426742620068, 2.5970164257708559, -2.2105581764404256, -2.2840025195778555], 
[-0.9899260864977224, 0.6452966076040294, 4.7927627776412685, -2.5295371205933446], [-3.0966642993131313, 2.7661250464270415, 
3.8449855823440444, -1.0741276139917844]], [[4.4391744703399496, -4.1270150086076454, -4.6930226795350096, 2.0252856547564502], 
[0.72529038124795964, 0.69967571213256718, 1.5860459396195523, 3.2323355247046486], [-4.8333386621087406, -4.0251398397320761, 
4.0156431943430171, 0.020512705604436299]]], [[[3.7017997151727897, 2.2095584570298223, -1.3813567501107094, 
-3.0173272231346884], [-0.37818934407957361, 3.3012689422797568, 1.8594061223449732, -2.6809232723795793], 
[-1.0609886930646626, 4.55401590833754, -1.7767931776744139, -4.2078119148862472]], [[-0.77920493007627289, 
-0.15399849119821418, -3.4917431482167904, 2.0694150586321749], [-1.0964185687026107, -0.87426921689970882, 
-1.7697798118999843, 3.6382813630153024], [2.4244878962317902, 1.7725790738779992, 3.103213116080326, 2.7839361272118879]]], 
[[[3.3584017965594821, 4.5451922495650887, 4.3159476335649334, 1.748518729095859], [0.89409626022649036, 0.69585848538395023, 
-4.6541856081713497, 3.1960066922617294], [-2.7799634144533547, -4.4283179527839023, -2.9667711388101004, 1.1176323952926515]], 
[[-4.6615795086663256, 2.4752867328596126, 3.1499384715848127, 4.0561473473736545], [1.740495573000004, -2.0276184332531799, 
3.1239224119640632, -3.2954201173978772], [0.80593575299516118, -0.64067885175735562, 2.0781184694444761, 
-0.3872702665989971]]]])
      res=arg0-arg1
      ref=Data(numpy.array([[[[-5.595772501235956, -4.3947462527448051, 0.4128283494664764, 0.48627269260390626], 
[-0.80780374047622683, -2.4430264345779786, -6.5904926046152177, 0.73180729361939534], [1.298934472339182, -4.5638548734009907, 
-5.6427154093179936, -0.72360221298216487]], [[-6.2369042973138988, 2.3292851816336961, 2.8952928525610604, 
-3.8230154817303994], [-2.5230202082219089, -2.4974055391065164, -3.3837757665935015, -5.0300653516785978], 
[3.0356088351347914, 2.2274100127581269, -5.8133730213169663, -1.8182425325783855]]], [[[-5.4995295421467389, 
-4.0072882840037716, -0.41637307686323988, 1.2195973961607391], [-1.4195404828943756, -5.098998769253706, -3.6571359493189224, 
0.88319344540563005], [-0.73674113390928664, -6.3517457353114892, -0.020936649299535315, 2.4100820879122979]], 
[[-1.0185248968976763, -1.643731335775735, 1.6940133212428412, -3.8671448856061241], [-0.70131125827133856, 
-0.92346061007424041, -0.027950015073964884, -5.4360111899892516], [-4.2222177232057394, -3.5703089008519484, 
-4.9009429430542752, -4.5816659541858371]]], [[[-5.1561316235334314, -6.3429220765390379, -6.1136774605388826, 
-3.5462485560698083], [-2.6918260872004396, -2.4935883123578995, 2.8564557811974005, -4.9937365192356786], 
[0.98223358747940548, 2.630588125809953, 1.1690413118361511, -2.9153622222666007]], [[2.8638496816923764, -4.2730165598335619, 
-4.9476682985587619, -5.8538771743476037], [-3.5382253999739532, 0.22988860627923069, -4.9216522389380124, 1.497690290423928], 
[-2.6036655799691104, -1.1570509752165936, -3.8758482964184253, -1.4104595603749521]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_constData_rank0(self):
      arg0=Data(-1.24399288417,self.functionspace)
      arg1=Data(-2.75107501128,self.functionspace)
      res=arg0-arg1
      ref=Data(1.5070821271,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_constData_rank1(self):
      arg0=Data(1.43305688056,self.functionspace)
      arg1=Data(numpy.array([1.7619071185902948, 1.5567508613887533]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([-0.32885023802744318, -0.12369398082590166]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_constData_rank2(self):
      arg0=Data(0.88884371782,self.functionspace)
      arg1=Data(numpy.array([[-2.3859920337850271, -0.21516198988785096, -1.8301742070811966, -2.0254011466168387, 
-0.20121168833003544], [2.8766362736356186, -4.0675468597525466, -0.12346589795057739, -2.8322866540875391, 
-1.1440147219685004], [4.3581272011519623, 3.1283211942631493, 3.8682210754717357, -2.9984964652589765, 3.3141426872945381], 
[0.87561639605202934, -0.42804570976353062, -0.978826018319916, -0.22724560097534585, 
-0.17792568617300919]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[3.2748357516050683, 1.1040057077078922, 2.7190179249012378, 2.9142448644368799, 
1.0900554061500767], [-1.9877925558155773, 4.9563905775725878, 1.0123096157706186, 3.7211303719075803, 2.0328584397885416], 
[-3.469283483331921, -2.239477476443108, -2.9793773576516944, 3.8873401830790177, -2.4252989694744969], [0.01322732176801189, 
1.3168894275835719, 1.8676697361399572, 1.1160893187953871, 1.0667694039930504]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_constData_rank3(self):
      arg0=Data(2.73268805242,self.functionspace)
      arg1=Data(numpy.array([[[4.1479731791053069, 2.8755071263809313], [-1.8478426966866355, -1.5543879669155825]], 
[[2.0174842460711009, -4.612348996889426], [-2.8763217384615802, 2.256501202629158]], [[-4.1428647789103268, 
3.8671819302288153], [0.89132930750180339, -0.34952844831469854]], [[-4.9734577778270053, 0.36997929933990648], 
[-0.92892270186049153, -1.5339798369067745]], [[-2.5800067544997161, 2.7532254930307518], [-4.0767421873733198, 
-0.49633705468318112]], [[4.2909955069137862, 0.79906312050637052], [-2.0409124103818366, 
-1.7363630981963962]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[-1.4152851266858395, -0.14281907396146387], [4.5805307491061029, 4.2870760193350499]], 
[[0.71520380634836656, 7.3450370493088935], [5.6090097908810481, 0.47618684979030945]], [[6.8755528313297942, 
-1.1344938778093479], [1.841358744917664, 3.082216500734166]], [[7.7061458302464727, 2.362708753079561], [3.661610754279959, 
4.266667889326242]], [[5.3126948069191835, -0.020537440611284374], [6.8094302397927873, 3.2290251071026486]], 
[[-1.5583074544943187, 1.9336249319130969], [4.773600462801304, 4.4690511506158632]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_constData_rank4(self):
      arg0=Data(-3.59659310422,self.functionspace)
      arg1=Data(numpy.array([[[[0.9007732121571097, 3.8739642437223889, 1.6724155476348281, 3.4671178409268073], 
[-1.0804281557963167, 0.45862818725878185, 1.4422456122351255, -4.7846370739922186], [-1.1565820602334709, -4.5747141817191492, 
-3.7053786326038773, -4.1966124891138818]], [[-2.384075210032556, -3.0712828355637543, -4.8061728254824354, 
-3.2724485479323464], [3.9864334906678671, -3.0706813155634838, 1.941848417191081, 3.5694765915909645], [3.8696442444316546, 
2.2034349457379658, 3.5191562623483481, -3.5356299543736269]]], [[[3.7802522664272438, 4.6180685418268759, -4.8913836901950178, 
-2.5255361350849226], [1.4383214878391311, 3.4529966848956661, 3.4032616791738999, -2.9680382830655869], [-2.5365672824275198, 
-3.0230370710541399, -4.9888222738539767, 0.55392746066918441]], [[-3.8835960694831364, -2.9025446559714183, 
1.2656858266805981, -0.29978640527173805], [0.070800286741653728, -3.163496959037353, 0.29703647272514377, 
-3.0496091872973121], [1.6041309003634572, -1.4363355254692012, 2.4707080432778605, -4.4347388698478909]]], 
[[[4.1416129956171073, -0.7391228739742246, -4.0639497640986955, -1.8464641091533074], [-2.8504437193270435, 
-0.179311301344109, -2.7690084047475585, -4.3812311028647786], [2.5518589954989634, 0.53600136433418299, -2.5201179600023016, 
4.4132978603519852]], [[3.1639410439760915, 2.4809353145376516, 0.78501365835196246, 0.83595575201890338], 
[0.89883085907645022, -4.0125820988371821, 1.3409364764638498, 4.0358851337650012], [3.3926109171960466, -2.8810651429659062, 
4.2173194512334007, 1.7854597602207969]]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[[-4.497366316377108, -7.4705573479423872, -5.2690086518548265, -7.0637109451468056], 
[-2.5161649484236817, -4.0552212914787802, -5.0388387164551238, 1.1880439697722203], [-2.4400110439865275, 0.97812107749915089, 
0.108785528383879, 0.60001938489388351]], [[-1.2125178941874424, -0.52531026865624408, 1.2095797212624371, 
-0.32414455628765193], [-7.5830265948878655, -0.52591178865651456, -5.5384415214110794, -7.1660696958109629], 
[-7.4662373486516529, -5.8000280499579642, -7.1157493665683464, -0.060963149846371412]]], [[[-7.3768453706472421, 
-8.2146616460468742, 1.2947905859750195, -1.0710569691350758], [-5.0349145920591294, -7.0495897891156645, -6.9998547833938982, 
-0.62855482115441141], [-1.0600258217924785, -0.57355603316585846, 1.3922291696339784, -4.1505205648891828]], 
[[0.28700296526313807, -0.69404844824858003, -4.8622789309005965, -3.2968066989482603], [-3.6673933909616521, 
-0.43309614518264539, -3.8936295769451421, -0.54698391692268622], [-5.2007240045834555, -2.1602575787507972, 
-6.0673011474978589, 0.83814576562789256]]], [[[-7.7382060998371056, -2.8574702302457737, 0.46735665987869712, 
-1.750128995066691], [-0.74614938489295479, -3.4172818028758893, -0.82758469947243984, 0.78463799864478023], 
[-6.1484520997189618, -4.1325944685541813, -1.0764751442176967, -8.0098909645719836]], [[-6.7605341481960899, 
-6.0775284187576499, -4.3816067625719608, -4.4325488562389017], [-4.4954239632964486, 0.41598899461718375, -4.9375295806838482, 
-7.6324782379849996], [-6.9892040214160449, -0.71552796125409213, -7.813912555453399, 
-5.3820528644407952]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_expandedData_rank0(self):
      arg0=Data(-2.03972866583,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.0812408809)+(1.-msk_arg1)*(-3.88795230637)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-3.12096954673)+(1.-msk_ref)*(1.84822364053)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_expandedData_rank1(self):
      arg0=Data(0.978076223248,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-3.4046943663034734, -4.7983762357268835])+(1.-msk_arg1)*numpy.array([2.1580113049552088, 
-4.1647951013525386])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.38277058955142, 5.7764524589748305])+(1.-msk_ref)*numpy.array([-1.1799350817072618, 
5.1428713246004856])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_expandedData_rank2(self):
      arg0=Data(-2.37174052924,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.61076119206543389, 4.0235551557254983, -4.0901200910655824, -3.5804428927622469, 
-1.7645459061750834], [1.4350053526214168, -0.76885555844455133, -3.6954427691951364, -1.0928478067187983, 
-1.0687545987479741], [-1.3752840052849624, -0.99308642796703506, -1.6620711018716183, -0.65665546152848897, 
-0.62735693078813526], [-3.7401615515162776, -2.6859804518536787, 0.28366793927157197, -0.47046947519513438, 
-2.8362829627912278]])+(1.-msk_arg1)*numpy.array([[4.0901805723398752, -2.952860422248559, 0.50670067614373338, 
-4.5071767678831662, -0.36438312961050912], [-4.6024104088291811, -0.54723168087245355, -4.8412530670411229, 
-2.6213712239525835, -4.7146732035120751], [1.7967120379357171, -1.5674686902898873, -0.39091245176399259, -2.2101442627190471, 
-2.0905526382238069], [-1.4689861719969546, 4.1703057395912833, -3.0049299372062865, 4.9442642983674663, 3.8490915021474059]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.760979337173902, -6.3952956849648341, 1.7183795618262465, 1.208702363522911, 
-0.60719462306425243], [-3.8067458818607527, -1.6028849707947845, 1.3237022399558005, -1.2788927225205375, 
-1.3029859304913618], [-0.99645652395437345, -1.3786541012723008, -0.70966942736771754, -1.7150850677108469, 
-1.7443835984512006], [1.3684210222769417, 0.31423992261434286, -2.6554084685109078, -1.9012710540442015, 
0.46454243355189195]])+(1.-msk_ref)*numpy.array([[-6.4619211015792111, 0.58111989300922318, -2.8784412053830692, 
2.1354362386438304, -2.0073573996288268], [2.2306698795898452, -1.8245088483668823, 2.469512537801787, 0.24963069471324761, 
2.3429326742727392], [-4.168452567175053, -0.80427183894944854, -1.9808280774753433, -0.1615962665202888, 
-0.28118789101552899], [-0.90275435724238129, -6.5420462688306191, 0.63318940796695067, -7.3160048276068022, 
-6.2208320313867418]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_expandedData_rank3(self):
      arg0=Data(2.0446268931,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.24184794937337273, -4.3947963657619482], [-3.5006385436576268, 0.14971440474993436]], 
[[-3.4947428306264658, 3.6603693917048261], [-0.53871068114081844, 3.3023683480183248]], [[-4.7345347156960651, 
3.1474834410467452], [-0.94401032931714468, 2.9768630818946793]], [[-1.7612231238401055, -4.189803064412807], 
[-4.6914075454295467, -3.9049360898641119]], [[1.5734124054896022, -4.8585804822070502], [-3.981844674201894, 
-1.7993661422894225]], [[-1.454816650344426, 4.6224695263737274], [2.3857982762159482, 
-0.98544942310946126]]])+(1.-msk_arg1)*numpy.array([[[3.1185335919513175, 1.9972324172484885], [-0.78923890032659294, 
3.9488016780358226]], [[3.3938041510645274, -3.9381940339348365], [4.1454409436258413, 0.97048032592596201]], 
[[4.8880828775295839, -1.4705169172307473], [0.59856779066986388, -3.9995793781201439]], [[-3.8128682241616643, 
-1.1924751759486139], [0.396073030749581, 0.81805855538736516]], [[-1.6426407559620815, 4.0216573611625943], 
[-2.3998893535620027, 0.79038519045998701]], [[4.1590574403334681, 0.49527003179762552], [-0.79809790313990803, 
4.401207872800514]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.8027789437307185, 6.4394232588660394], [5.545265436761718, 1.8949124883541568]], 
[[5.539369723730557, -1.6157424986007349], [2.5833375742449096, -1.2577414549142336]], [[6.7791616088001563, 
-1.102856547942654], [2.9886372224212359, -0.93223618879058812]], [[3.8058500169441967, 6.2344299575168982], 
[6.7360344385336379, 5.9495629829682031]], [[0.47121448761448903, 6.9032073753111414], [6.0264715673059852, 
3.8439930353935137]], [[3.4994435434485172, -2.5778426332696363], [-0.34117138311185702, 
3.0300763162135524]]])+(1.-msk_ref)*numpy.array([[[-1.0739066988472263, 0.047394475855602636], [2.8338657934306841, 
-1.9041747849317314]], [[-1.3491772579604362, 5.9828209270389276], [-2.1008140505217501, 1.0741465671781292]], 
[[-2.8434559844254927, 3.5151438103348385], [1.4460591024342273, 6.044206271224235]], [[5.8574951172657554, 3.237102069052705], 
[1.6485538623545102, 1.226568337716726]], [[3.6872676490661727, -1.9770304680585031], [4.4445162466660939, 
1.2542417026441042]], [[-2.114430547229377, 1.5493568613064657], [2.8427247962439992, -2.3565809796964228]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_expandedData_rank4(self):
      arg0=Data(2.91658915836,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[2.7053756826411046, 2.4039746407181681, 0.30215041387276109, 4.8917289303552298], 
[1.5725027972239438, 0.026595650147548611, 1.9596899610821055, 4.5067794038717111], [-3.3097080565892556, 3.9438905852367352, 
2.926098540104114, -2.3137147487763219]], [[1.4355896959378258, -2.7238011548235064, -0.4008873857026316, 1.2269163719814209], 
[0.43886301291069429, 1.7386808769212578, -4.3223887486176702, 2.7307903277451429], [2.3784527541347895, 0.21898949644034627, 
1.6503223943032355, 4.4034789818417064]]], [[[3.744310763315827, -2.0765096215039658, -0.096420479801349046, 
0.85779777496205245], [0.0066189558411613447, -4.3002292886702804, -3.4509577342717037, 3.546437638835533], 
[4.1340416122208392, -4.0315065661174785, -3.6240929877991777, 2.8372301626571357]], [[0.92191239385611645, 3.9162256451776614, 
2.7065742677447133, 2.6099411075538512], [2.3142428942001789, -2.5245868704989025, -1.8889437310237445, -1.0847316908476277], 
[-1.9640075969477677, 4.5482213295519198, -4.6674780646261205, -4.9083380544927699]]], [[[-3.2291550671077895, 
-2.3886186730291783, 2.2868862846122253, 1.2988882676729023], [-3.9261534136459852, -4.1938015330806309, -1.5106596945473805, 
-1.7553575892235296], [-4.4753471756210592, 2.9163963229264986, 4.952728588992608, 3.278251619630927]], [[-4.3216112614048434, 
1.2446748035429493, 2.7366375528989604, -0.39659797368744876], [-3.124316876479234, 2.2128757660863565, -1.3245714280974239, 
1.0481735743423002], [4.7223764748463726, -1.2770600136731169, 1.3925516015207071, 
-3.8035611312735131]]]])+(1.-msk_arg1)*numpy.array([[[[0.25029324008670173, -2.8065225522656734, 1.4459348471132696, 
-2.4521548606576116], [1.8830973304101644, 0.69620239772981574, 0.86538962141579034, -4.5196142008990723], 
[0.34348077830155699, -2.7763269070623431, -0.29159867337391354, -2.1870389165951853]], [[3.0844454057615742, 
1.6688659966643336, 4.2129108051100879, -3.5220084697478402], [3.3499505117719881, -2.0189582162395867, -3.4596271171609416, 
2.5883206701918002], [3.0007584732421471, 2.1306588283496097, 2.2302229394613224, -3.5408833287337602]]], 
[[[0.23910705184024739, -4.0546092742834094, -0.76508579473423932, -2.9847378535109694], [-3.2957203732490159, 
-2.5766678233297915, -4.4728866774590941, -4.2236344181941465], [-4.8006374699373291, 2.0740376710185346, -2.1091173452357435, 
4.942006231484946]], [[1.020891650266468, 3.4651596043407, 4.9921430512202978, 3.1462024524797627], [4.0735560934794428, 
-3.0449968326822807, 1.07869995073046, 2.720376489102537], [4.4820876040983979, -2.4753402983674921, -0.13050262191349393, 
2.2309666421146641]]], [[[4.1259187497747156, 2.0143414606611119, 3.1939917316513249, -0.2219556846987869], 
[-1.8430365385826986, 4.4448669920458155, -3.2022856323129725, 4.1306249211401322], [2.0060965949377785, -0.49180461844588041, 
4.7966263049597284, -2.6834031120211712]], [[-2.4780507394530247, -3.5378025203109029, -1.4800968406411243, 
1.4451393954365424], [3.6302182383044865, 2.1276261135907362, 0.57971992676409023, -1.8388091073347903], [-4.7957320059532647, 
-0.77853118668687493, -4.4140258336498048, 1.5273107859654411]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.21121347572188753, 0.51261451764482402, 2.6144387444902311, -1.9751397719922377], 
[1.3440863611390483, 2.8899935082154435, 0.95689919728088668, -1.5901902455087189], [6.2262972149522477, -1.027301426873743, 
-0.009509381741121814, 5.230303907139314]], [[1.4809994624251663, 5.640390313186499, 3.3174765440656238, 1.6896727863815713], 
[2.4777261454522979, 1.1779082814417343, 7.2389779069806623, 0.1857988306178493], [0.53813640422820264, 2.6975996619226459, 
1.2662667640597567, -1.4868898234787142]]], [[[-0.82772160495283487, 4.9930987798669584, 3.0130096381643412, 
2.0587913834009397], [2.9099702025218308, 7.2168184470332726, 6.3675468926346959, -0.62984848047254083], [-1.2174524538578471, 
6.9480957244804706, 6.5406821461621698, 0.079358995705856472]], [[1.9946767645068757, -0.99963648681466921, 
0.21001489061827883, 0.30664805080914093], [0.60234626416281323, 5.4411760288618947, 4.8055328893867362, 4.0013208492106198], 
[4.8805967553107603, -1.6316321711889277, 7.5840672229891126, 7.8249272128557621]]], [[[6.1457442254707821, 5.3052078313921704, 
0.62970287375076683, 1.6177008906900898], [6.8427425720089774, 7.110390691443623, 4.4272488529103722, 4.6719467475865217], 
[7.3919363339840514, 0.00019283543649351031, -2.0361394306296159, -0.36166246126793489]], [[7.2382004197678356, 
1.6719143548200428, 0.17995160546403177, 3.3131871320504409], [6.0409060348422265, 0.70371339227663565, 4.241160586460416, 
1.868415584020692], [-1.8057873164833804, 4.1936491720361087, 1.524037556842285, 
6.7201502896365053]]]])+(1.-msk_ref)*numpy.array([[[[2.6662959182762904, 5.7231117106286655, 1.4706543112497226, 
5.3687440190206033], [1.0334918279528278, 2.2203867606331764, 2.0511995369472018, 7.4362033592620644], [2.5731083800614352, 
5.6929160654253348, 3.2081878317369057, 5.1036280749581771]], [[-0.16785624739858207, 1.2477231616986586, -1.2963216467470957, 
6.4385976281108324], [-0.43336135340899595, 4.9355473746025789, 6.3762162755239338, 0.32826848817119192], 
[-0.084169314879154911, 0.78593033001338242, 0.68636621890166971, 6.4574724870967524]]], [[[2.6774821065227448, 
6.9711984326464016, 3.6816749530972315, 5.9013270118739616], [6.2123095316120081, 5.4932569816927836, 7.3894758358220862, 
7.1402235765571387], [7.7172266283003212, 0.84255148734445751, 5.0257065035987356, -2.0254170731219538]], [[1.8956975080965242, 
-0.54857044597770788, -2.0755538928573056, -0.22961329411677056], [-1.1569669351164507, 5.9615859910452729, 1.8378892076325322, 
0.19621266926045511], [-1.5654984457354058, 5.3919294567304838, 3.0470917802764861, 0.68562251624832804]]], 
[[[-1.2093295914117235, 0.90224769770188029, -0.27740257328833273, 3.1385448430617791], [4.7596256969456903, 
-1.5282778336828233, 6.1188747906759646, -1.2140357627771401], [0.91049256342521367, 3.4083937768088726, -1.8800371465967363, 
5.5999922703841634]], [[5.3946398978160168, 6.4543916786738951, 4.3966859990041165, 1.4714497629264498], [-0.71362907994149438, 
0.78896304477225598, 2.3368692315989019, 4.7553982656977825], [7.7123211643162568, 3.6951203450498671, 7.330614992012797, 
1.389278372397551]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([-0.086403026197459099, -0.046407186588146132]),self.functionspace)
      arg1=-0.669086668937
      res=arg0-arg1
      ref=Data(numpy.array([0.58268364273971773, 0.6226794823490307]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([0.63975693104119635, 4.1160294564197226]),self.functionspace)
      arg1=numpy.array(-3.63363344461)
      res=arg0-arg1
      ref=Data(numpy.array([4.2733903756487717, 7.7496629010272979]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.86270901893347052, -1.6324218890648923]),self.functionspace)
      arg1=numpy.array([1.2697947562346616, -0.56703741994267975])
      res=arg0-arg1
      ref=Data(numpy.array([-2.1325037751681322, -1.0653844691222125]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([3.1718857767106652, -2.4973304899388502]),self.functionspace)
      arg1=Data(2.76214450165,self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([0.40974127506176306, -5.2594749915877523]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([4.4670182233378188, 1.5185099838854565]),self.functionspace)
      arg1=Data(numpy.array([-3.7509936338953165, 1.5445133407669696]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([8.2180118572331349, -0.026003356881513007]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([0.73740211715578852, 3.864411217092611]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(4.36771107785)+(1.-msk_arg1)*(-3.33170383215)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-3.6303089606916696, -0.50329986075484712])+(1.-msk_ref)*numpy.array([4.0691059493016768, 
7.1961150492384993])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([3.1679362943636455, -3.9298811460962666]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([4.7647888496509445, 1.6938962420920722])+(1.-msk_arg1)*numpy.array([0.51118622984383766, 
4.8775859928314382])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.596852555287299, -5.6237773881883388])+(1.-msk_ref)*numpy.array([2.6567500645198079, 
-8.8074671389277057])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[-2.8519917111860558, 1.9717885392089185, 2.6731928702453098, 0.83866963052838805, 
-1.8409114500669954], [-1.9725774585763558, 0.026861238295489365, -3.5717518923527969, -0.97673802026110934, 
-1.950944898729472], [3.9556554930307435, 4.9216153565426737, -0.94172609506491334, -0.23284596053724371, -3.9077224398938704], 
[4.3647043822005234, -3.2971013400413343, -3.7861457278822552, -1.121420764415384, 0.03446352676207276]]),self.functionspace)
      arg1=-2.67401041754
      res=arg0-arg1
      ref=Data(numpy.array([[-0.17798129364674065, 4.6457989567482336, 5.3472032877846249, 3.5126800480677032, 
0.83309896747231971], [0.70143295896295932, 2.7008716558348045, -0.89774147481348177, 1.6972723972782058, 0.72306551880984316], 
[6.6296659105700586, 7.5956257740819888, 1.7322843224744018, 2.4411644570020714, -1.2337120223545552], [7.0387147997398385, 
-0.62309092250201914, -1.1121353103429401, 1.5525896531239312, 2.7084739443013879]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[2.5450646536920445, -3.3031771527992482, -1.0120238117053382, 2.02117861477742, 
2.8913479474219415], [0.39005415595631376, -4.0884238157042896, 1.2869635536961272, -3.1811449172720971, 4.07003523419373], 
[-2.6335807831220048, 4.6258922315105195, -4.1310155510620241, 3.4789296988862954, 2.6641555650546822], [-3.0128286359108838, 
-4.5323841835933134, 1.9671272563924802, 2.4191249517525151, 2.1147043240374828]]),self.functionspace)
      arg1=numpy.array(-2.63971047346)
      res=arg0-arg1
      ref=Data(numpy.array([[5.1847751271487201, -0.66346667934257253, 1.6276866617513375, 4.6608890882340956, 
5.5310584208786171], [3.0297646294129894, -1.4487133422476139, 3.9266740271528029, -0.5414344438154215, 6.7097457076504057], 
[0.006129690334670812, 7.2656027049671952, -1.4913050776053485, 6.1186401723429711, 5.3038660385113578], [-0.37311816245420815, 
-1.8926737101366378, 4.6068377298491558, 5.0588354252091907, 4.7544147974941584]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[-0.67912975708699541, -3.6114162491759805, -2.3357625116182335, 2.9913623711654678, 
-2.001298194680925], [-0.68996625280241908, 1.5632495621078295, 4.8548972206710221, 4.3483474533648643, -0.048486465210599405], 
[1.636887971247674, 4.9790750365521159, 4.8099177647207352, -4.7289265839445607, -4.243072093038788], [-2.4217588539054971, 
2.9587796497495109, 1.949953609377828, -0.86638864177258323, -1.8713672454198793]]),self.functionspace)
      arg1=numpy.array([[3.3871100163688102, -4.5488039858677887, 1.390128107035796, 0.40817967105332009, 4.1043607979907222], 
[0.79686982515870586, -2.6988881037033208, 0.32569294535415594, 3.0901948504805823, 0.67561029667729677], 
[-0.51560890294926143, 3.5411051200986563, 0.14012877435148585, 0.96077405916870617, 2.3398310140352105], [-4.5019413027789525, 
4.7066711935506405, 3.4172622792148015, 3.3884571285025054, 4.2351501221589896]])
      res=arg0-arg1
      ref=Data(numpy.array([[-4.0662397734558056, 0.93738773669180819, -3.7258906186540295, 2.5831827001121477, 
-6.1056589926716471], [-1.4868360779611249, 4.2621376658111503, 4.5292042753168662, 1.258152602884282, -0.72409676188789618], 
[2.1524968741969355, 1.4379699164534596, 4.6697889903692493, -5.6897006431132668, -6.5829031070739985], [2.0801824488734555, 
-1.7478915438011295, -1.4673086698369735, -4.2548457702750886, -6.1065173675788689]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[-0.63194117706069974, 2.7937135140804958, 2.2707579668200566, -4.189217205495714, 
-1.2082162678111299], [-3.5043637989127818, -4.7217160078561315, 2.7164310309348414, -2.281051023020173, 3.5711893220173465], 
[0.94201080791589309, -3.9965293203377392, -4.7392679857415079, 3.4715184440517159, -0.42482704830201534], 
[-1.5154689658128042, -2.9724138409524956, 0.57321326941998052, -0.49143898738968872, -2.9709920412129498]]),self.functionspace)
      arg1=Data(3.36228071212,self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[-3.9942218891771866, -0.56856719803599098, -1.0915227452964302, -7.5514979176122008, 
-4.5704969799276167], [-6.8666445110292686, -8.0839967199726175, -0.64584968118164543, -5.6433317351366599, 
0.20890860990085969], [-2.4202699042005937, -7.358810032454226, -8.1015486978579947, 0.10923773193522912, -3.7871077604185022], 
[-4.877749677929291, -6.3346945530689824, -2.7890674426965063, -3.8537196995061755, -6.3332727533294371]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-0.21041578589924192, -4.3566775381829981, 0.46024508456702407, 4.3404468278056179, 
1.7663085893943373], [-1.4414803309489308, -1.4828707580856006, 0.63813036363780995, 4.7220888558616121, 0.3506537339428375], 
[-4.5110064539468269, 0.55320778191007847, 0.17258513039839851, 1.8502365095906281, 0.95269915801584926], [1.9183602440477374, 
3.6313257495434605, 2.1132396987109363, -0.35560756677464589, -1.7171535131310875]]),self.functionspace)
      arg1=Data(numpy.array([[3.9509777374983166, 2.9179724586501665, -2.5472402574107367, 2.089990285519705, 
-1.8815705750995448], [2.2944024492279116, 2.9925762035848011, 1.2211025046085657, 2.8330214698682212, 0.55985371127935579], 
[4.6415764235597976, 3.983668054452302, -0.93393925058415395, 3.1811876865506648, 0.3151352615131966], [-1.8272908595996409, 
4.6640401730549979, -2.6524606897167624, 2.0753771898714728, -3.0765528720214474]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[-4.1613935233975585, -7.2746499968331646, 3.0074853419777607, 2.2504565422859129, 
3.6478791644938822], [-3.7358827801768424, -4.4754469616704018, -0.58297214097075578, 1.8890673859933909, 
-0.20919997733651829], [-9.1525828775066245, -3.4304602725422235, 1.1065243809825525, -1.3309511769600366, 
0.63756389650265266], [3.7456511036473783, -1.0327144235115373, 4.7657003884276987, -2.4309847566461187, 
1.3593993588903599]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[-3.3787586283425108, 1.1525972996568168, 0.71473388568655771, -2.4888600180458886, 
2.4332795938656835], [-3.2290182143491828, -3.7465920611123957, 4.9748621752275408, -4.8389587416142987, 2.1909630308827115], 
[-1.6014573890214723, -3.3455203847460302, 4.5310692671494728, -0.58393012393523769, -4.1033654627596494], 
[-1.8849921939847727, 3.2708353683635227, 3.6925895619451978, 4.9844453836188922, 2.8997952336135384]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.10303055489)+(1.-msk_arg1)*(-0.884384178641)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-5.481789183232137, -0.95043325523280942, -1.3882966692030685, -4.5918905729355153, 
0.33024903897605729], [-5.332048769238809, -5.8496226160020219, 2.8718316203379146, -6.9419892965039249, 0.087932475993085291], 
[-3.7044879439110985, -5.4485509396356564, 2.4280387122598466, -2.6869606788248639, -6.2063960176492756], [-3.9880227488743989, 
1.1678048134738965, 1.5895590070555716, 2.881414828729266, 
0.79676467872391221]])+(1.-msk_ref)*numpy.array([[-2.4943744497019678, 2.0369814782973599, 1.5991180643271008, 
-1.6044758394053455, 3.3176637725062266], [-2.3446340357086397, -2.8622078824718526, 5.8592463538680839, -3.9545745629737556, 
3.0753472095232546], [-0.71707321038092919, -2.4611362061054871, 5.4154534457900159, 0.3004540547053054, -3.2189812841191063], 
[-1.0006080153442296, 4.1552195470040658, 4.5769737405857409, 5.8688295622594353, 3.7841794122540815]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[0.64084916094540123, 1.1214452337283891, 3.1817758168162023, -4.1471220989317681, 
-4.1739364785132418], [-0.033643458999028653, 3.9976266223642618, 1.638031544850608, 4.7335250345571005, -3.2292189463909002], 
[4.0969849430904191, -4.1802334586875967, -3.6624231568127175, 3.3353521349232125, 3.0354939704930555], [-0.77560171826169011, 
-0.92069071764868315, 2.3749160052400189, 2.964749971420817, 4.4894840790184141]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-1.1940911653527042, 0.60954389847544732, -0.93436987655924497, -1.2147444060536881, 
3.3491044264600927], [-0.25339464598762351, 4.0417405119260774, 0.23680806823690226, -2.9790292711586286, 1.0108686788275492], 
[2.8686863488898027, 0.83732083527521972, -4.3659877534693674, -3.3939208578842797, -3.3331873683776312], [2.649740153395336, 
2.5659320547427207, -3.9442235262270451, -4.7575444090554235, 
3.689777828373213]])+(1.-msk_arg1)*numpy.array([[4.3472601336810062, -3.8577291004840406, 4.2185030364496541, 
0.13159351790032581, -1.9218707379161293], [-3.2451084844740063, -1.5756293921468134, -2.9265135677377963, -2.5033746470828575, 
4.4979506475830036], [-0.82597986065992046, -2.7212566530196414, -2.389751558465194, 1.0509905511510942, 0.16515626070319378], 
[2.8689815361724662, 0.79804907801623237, -1.4146397326543489, -1.0881160178063167, -2.8795996138116307]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.8349403262981054, 0.51190133525294179, 4.1161456933754472, -2.93237769287808, 
-7.5230409049733344], [0.21975118698859486, -0.044113889561815611, 1.4012234766137057, 7.7125543057157291, 
-4.2400876252184494], [1.2282985942006164, -5.0175542939628164, 0.70356459665664994, 6.7292729928074921, 6.3686813388706867], 
[-3.4253418716570261, -3.4866227723914038, 6.319139531467064, 7.7222943804762405, 
0.79970625064520107]])+(1.-msk_ref)*numpy.array([[-3.7064109727356049, 4.9791743342124297, -1.0367272196334518, 
-4.2787156168320939, -2.2520657405971125], [3.2114650254749777, 5.5732560145110757, 4.5645451125884042, 7.236899681639958, 
-7.7271695939739038], [4.9229648037503395, -1.4589768056679553, -1.2726715983475234, 2.2843615837721183, 2.8703377097898617], 
[-3.6445832544341563, -1.7187397956649155, 3.7895557378943678, 4.0528659892271337, 7.3690836928300447]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-1.0252014732841106, -0.48754604628999232], [-0.24202959835504245, -3.9457066263925276]], 
[[0.34844882567872126, 3.6698772296327657], [4.1230053178597608, 3.9657299049218366]], [[-3.5127089098137487, 
3.8659691418125828], [-1.2678481923172411, 2.1862994848455592]], [[4.2338268373854362, -0.81906112517914043], 
[3.904572239467818, -2.0569447415019049]], [[-4.7410756860220307, 0.66385705836994102], [-0.40481658299998724, 
-4.2646835312082292]], [[3.9359305786384162, 0.48535655874110617], [-0.59898411844761501, 
-2.1041653526979545]]]),self.functionspace)
      arg1=-2.73962694157
      res=arg0-arg1
      ref=Data(numpy.array([[[1.7144254682906794, 2.2520808952847977], [2.4975973432197476, -1.2060796848177375]], 
[[3.0880757672535113, 6.4095041712075558], [6.8626322594345508, 6.7053568464966267]], [[-0.77308196823895869, 
6.6055960833873728], [1.4717787492575489, 4.9259264264203493]], [[6.9734537789602262, 1.9205658163956496], [6.644199181042608, 
0.68268220007288516]], [[-2.0014487444472406, 3.4034839999447311], [2.3348103585748028, -1.5250565896334392]], 
[[6.6755575202132063, 3.2249835003158962], [2.140642823127175, 0.63546158887683557]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[4.2674861027850284, 3.7061042385367688], [-2.3958002229620856, -0.65099822165416299]], 
[[0.65532200872971202, -2.8226382826367558], [-4.1549667665372247, -4.0135799599899151]], [[0.15262340371198135, 
-0.7746711472327803], [1.3398274408293611, 4.7898665321871157]], [[-4.6496971028630432, -0.45913290911347104], 
[3.3769509765632737, 0.041344001444816314]], [[-2.8433820348215919, -1.7189622378463767], [3.7194562031895426, 
2.6456907914698551]], [[-1.3269497506555505, 2.5033030507359193], [0.041413723285725901, 
0.065183592719888495]]]),self.functionspace)
      arg1=numpy.array(-4.86304024517)
      res=arg0-arg1
      ref=Data(numpy.array([[[9.13052634795892, 8.5691444837106587], [2.4672400222118052, 4.2120420235197278]], 
[[5.5183622539036028, 2.040401962537135], [0.70807347863666603, 0.84946028518397565]], [[5.0156636488858721, 
4.0883690979411105], [6.2028676860032519, 9.6529067773610073]], [[0.21334314231084761, 4.4039073360604197], 
[8.2399912217371636, 4.9043842466187071]], [[2.0196582103522989, 3.1440780073275141], [8.5824964483634325, 
7.5087310366437459]], [[3.5360904945183402, 7.3663432959098101], [4.9044539684596167, 4.9282238378937793]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[2.4412950201387922, -0.86219034579998954], [-1.6171724745909186, 0.81044318546214189]], 
[[3.1578534709312773, 1.1689220711298098], [1.5012997496330618, -0.59506503959930956]], [[0.047587097292952407, 
0.27795358374026247], [-1.2527349706099464, 3.4725047687219242]], [[1.4396452295302478, -1.9143791431825976], 
[2.6562178084181074, -0.30282037214252]], [[0.15016512242942959, 2.197954300967], [-0.062279424776018999, 2.4226169502767707]], 
[[-3.3337218928393195, 2.9370065699234171], [0.0078242060953357395, 2.1059145871847287]]]),self.functionspace)
      arg1=numpy.array([[[0.9001025939166869, 0.52964638425110433], [-1.7602371131003447, 0.39150169763954068]], 
[[4.3431063758895867, 1.8695639406449169], [-4.4848350396507843, 4.0574465291245865]], [[4.2524115151155364, 
3.6563926843394601], [1.2504194055158866, 1.1783599369170252]], [[-1.4742828366720007, -2.3695778958630189], 
[1.8747770790741072, -2.9701055688534783]], [[3.9609613913092936, -2.332931325170553], [-4.9651936758190605, 
1.5508633732879806]], [[1.125030176050811, -3.1883808301266789], [1.5899790092283244, -3.1390621331692503]]])
      res=arg0-arg1
      ref=Data(numpy.array([[[1.5411924262221053, -1.3918367300510939], [0.14306463850942608, 0.41894148782260121]], 
[[-1.1852529049583094, -0.7006418695151071], [5.9861347892838461, -4.652511568723896]], [[-4.204824417822584, 
-3.3784391005991976], [-2.503154376125833, 2.2941448318048989]], [[2.9139280662022484, 0.45519875268042131], 
[0.78144072934400022, 2.6672851967109583]], [[-3.810796268879864, 4.530885626137553], [4.9029142510430415, 
0.87175357698879008]], [[-4.4587520688901305, 6.1253874000500961], [-1.5821548031329886, 
5.2449767203539785]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[3.483761537043419, -2.6968346520293585], [0.46008274824379747, 3.8311610063404871]], 
[[2.3822895796785222, 3.7155557845261846], [2.1012065427444959, 2.9903329876538081]], [[-3.6638630188776142, 
-1.012167960685002], [0.41885281141625264, -1.8915016547282426]], [[4.5464765184217057, 1.3810654946107785], 
[1.7706565881606897, 1.1215310794722182]], [[-0.9740550715664833, 0.59717064383601937], [2.4262060133865386, 
1.0008516922130761]], [[2.7027597940485339, -4.2193457673911716], [-2.715704649182864, 
-3.7960532134033063]]]),self.functionspace)
      arg1=Data(2.68309197361,self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[0.80066956342941253, -5.3799266256433649], [-2.223009225370209, 1.1480690327264806]], 
[[-0.30080239393548425, 1.0324638109121782], [-0.5818854308695105, 0.30724101403980164]], [[-6.3469549924916207, 
-3.6952599342990085], [-2.2642391621977538, -4.5745936283422495]], [[1.8633845448076993, -1.3020264790032279], 
[-0.91243538545331671, -1.5615608941417882]], [[-3.6571470451804897, -2.0859213297779871], [-0.25688596022746779, 
-1.6822402814009303]], [[0.019667820434527528, -6.902437741005178], [-5.3987966227968709, 
-6.4791451870173127]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[-1.0902537352946551, 0.32957080494293667], [2.430566413238731, -1.3707106984737747]], 
[[4.4694407338953805, 2.6173375046323759], [-4.8436995282905464, -0.31227367000174233]], [[-1.2023060136621098, 
3.6073795612737705], [1.115295038202091, 1.7439443926545497]], [[-3.3343435172364257, -1.7024944770277295], 
[-1.0258875761193464, -3.0393634235114799]], [[4.2695316088435593, 4.4666634534357428], [4.1008847206797565, 
0.83885747314880721]], [[-4.2863076482567495, -1.019791980279928], [-2.8921846496224091, 
2.3097169218406757]]]),self.functionspace)
      arg1=Data(numpy.array([[[3.1798125917444011, 3.5009117560784269], [-3.1667439143689844, 0.30269240876540415]], 
[[-2.6449978270402488, -3.6867608538774146], [0.67424403702830649, 4.2661340891525761]], [[-3.4442016992291702, 
1.2211719717141234], [2.631805503182834, -3.6244253633131218]], [[-0.72467851055743271, -1.5962138357120139], 
[0.46362174952146518, 1.5539937924967848]], [[-2.2390967730046274, 2.2227028121585093], [1.2638450073446892, 
-1.0665410636902561]], [[-0.033712599207014193, 3.0563795134078031], [-4.4259975720113056, 
1.187432499381317]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[-4.2700663270390562, -3.1713409511354902], [5.5973103276077154, -1.6734031072391788]], 
[[7.1144385609356293, 6.3040983585097905], [-5.5179435653188529, -4.5784077591543184]], [[2.2418956855670604, 
2.3862075895596471], [-1.516510464980743, 5.368369755967672]], [[-2.609665006678993, -0.10628064131571557], 
[-1.4895093256408116, -4.5933572160082647]], [[6.5086283818481867, 2.2439606412772335], [2.8370397133350673, 
1.9053985368390634]], [[-4.2525950490497353, -4.0761714936877311], [1.5338129223888965, 
1.1222844224593587]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[-0.077078012804054907, -0.36784598543783442], [-1.9497383298095849, 2.4340198974820559]], 
[[2.0071959266311161, -1.5847112780675268], [4.433251942554767, -3.1307183985312759]], [[0.74155801267383481, 
4.9068635625345074], [-3.7053906981735238, 2.0098843086987515]], [[4.1122069215251162, 0.18950020324926697], 
[-4.0363539604431233, 1.6861081871315147]], [[-2.3671014065812002, -0.73253014075991629], [-0.94991752614271174, 
4.1711814282984125]], [[0.20361922003143107, 0.52240524963441359], [-0.25944381698955432, 
-3.242321958891341]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.96013555409)+(1.-msk_arg1)*(-1.65000800524)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-4.0372135668891325, -4.327981539522912], [-5.9098738838946625, -1.5261156566030216]], 
[[-1.9529396274539614, -5.5448468321526043], [0.4731163884696894, -7.090853952616353]], [[-3.2185775414112427, 
0.94672800844942984], [-7.6655262522586014, -1.9502512453863261]], [[0.15207136744003869, -3.7706353508358106], 
[-7.9964895145282009, -2.2740273669535629]], [[-6.3272369606662782, -4.6926656948449939], [-4.9100530802277893, 
0.21104587421333498]], [[-3.7565163340536465, -3.437730304450664], [-4.2195793710746319, 
-7.2024575129764186]]])+(1.-msk_ref)*numpy.array([[[1.5729299924389419, 1.2821620198051624], [-0.2997303245665881, 
4.0840279027250528]], [[3.657203931874113, 0.065296727175470082], [6.0832599477977638, -1.480710393288279]], 
[[2.3915660179168317, 6.5568715677775042], [-2.055382692930527, 3.6598923139417483]], [[5.7622149267681131, 
1.8395082084922638], [-2.3863459552001265, 3.3361161923745115]], [[-0.71709340133820332, 0.91747786448308055], 
[0.7000904791002851, 5.8211894335414094]], [[1.8536272252744279, 2.1724132548774104], [1.3905641882534425, 
-1.5923139536483442]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[-2.9568168833965105, 2.4643014595819226], [-1.9303442660744952, -0.74426815341594921]], 
[[-0.54929473822355934, -3.1264138600290559], [1.4466770595374232, 2.1404054856310806]], [[-1.6725207401055187, 
4.1940162721922523], [-2.6424640935481722, 2.5368200117643749]], [[0.42681188949625515, -2.3713384598164877], 
[4.2591969617066923, -4.1250072824567594]], [[-0.55543045417245285, -1.2532084492568361], [2.9320501785010675, 
1.3286773160158392]], [[2.5622706916273366, -4.8554739322530738], [0.47710235106384413, 
1.4459922847524656]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-3.3037111991945665, -1.4088006899712791], [-4.6788253103055588, -4.2010269873040276]], 
[[-3.221913466382571, 2.3900529000676753], [1.7488647147988647, 3.7900712859611225]], [[1.80601038649045, -0.4250096465690163], 
[4.6740514226171577, 0.45188442262084827]], [[-0.22766090704474973, -1.4496508194820077], [3.5775007014516174, 
-3.4269823474554082]], [[-2.6420251714249576, 0.21398711302169549], [-0.46027391355567193, 3.2414406709873482]], 
[[4.8557357315404275, -3.5003599386768789], [3.4808267058482159, 
-2.1098275309792216]]])+(1.-msk_arg1)*numpy.array([[[0.172748066346184, -3.8637537350601558], [-0.92559041735971537, 
0.70234586266824461]], [[2.0032169620015523, 2.1065167779332921], [0.15771260848019786, 0.34736382318325187]], 
[[-2.1244542713343595, -3.8282383971017362], [-3.6891501116953132, -4.6540976934927141]], [[-0.17055442756279326, 
-1.0548676915216659], [3.5791429873583951, 2.1394240644663558]], [[-4.3341948308209002, -0.004534135575516629], 
[2.8747421078694559, 2.3529198032835117]], [[0.97063076336306331, -0.3779113025679397], [-2.1181213952804336, 
-0.029555688232303368]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.34689431579805596, 3.8731021495532016], [2.7484810442310637, 3.4567588338880784]], 
[[2.6726187281590117, -5.5164667600967316], [-0.30218765526144153, -1.649665800330042]], [[-3.4785311265959686, 
4.6190259187612686], [-7.3165155161653299, 2.0849355891435266]], [[0.65447279654100488, -0.92168764033448003], 
[0.6816962602550749, -0.69802493500135121]], [[2.0865947172525048, -1.4671955622785315], [3.3923240920567395, 
-1.9127633549715091]], [[-2.2934650399130909, -1.3551139935761949], [-3.0037243547843717, 
3.5558198157316872]]])+(1.-msk_ref)*numpy.array([[[-3.1295649497426945, 6.3280551946420784], [-1.0047538487147798, 
-1.4466140160841938]], [[-2.5525117002251116, -5.2329306379623475], [1.2889644510572253, 1.7930416624478287]], 
[[0.45193353122884083, 8.0222546692939893], [1.0466860181471409, 7.190917705257089]], [[0.59736631705904841, 
-1.3164707682948218], [0.68005397434829717, -6.2644313469231152]], [[3.7787643766484473, -1.2486743136813194], 
[0.057308070631611585, -1.0242424872676725]], [[1.5916399282642733, -4.4775626296851341], [2.5952237463442778, 
1.4755479729847689]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[1.2023708812360212, -2.9181047245454916, 0.95706407608196464, -3.9498595228789455], 
[-3.9706736251400465, 1.7100026075683701, -0.52047970726534487, 1.8292980297931205], [-4.1668637570037292, -1.2157781951784075, 
-4.0520503706717594, -3.8122556865277701]], [[-3.0202890585852193, -3.9910007223884882, 0.33756512088859303, 
0.10198352234378305], [-4.7713449299371335, 0.51233731483926448, -1.6597131204722415, 0.33646081838867392], 
[1.5282964564001533, -2.0300282520390622, 1.7003459204910616, -2.2068236216634531]]], [[[2.6505164145746409, 
-4.1628773984666925, 2.8318655174412921, 4.6302337613245115], [1.7354935460010346, 2.8137039315710144, 3.2557282615413357, 
3.7760733949977805], [-4.0952664950596525, 0.97348451762758881, -3.5273666622659805, -4.6012628922441214]], 
[[4.8491783336207348, -2.2817742891418757, -3.540443683061989, -1.8349670221068228], [-0.039514793479882293, 2.617317628927597, 
1.5261453151379438, 3.5364527034661428], [-4.8524879260199008, -0.0084624114311493415, -2.9664081685623565, 
-4.579321883655898]]], [[[-0.8485407813004926, 1.0106762716392641, 0.43190767397623819, 4.0421505495134546], 
[3.5162980736028224, 3.6678947241426094, -4.1179973035086785, -0.19201106915449628], [1.9432542105123494, 1.9808261622587429, 
0.25647408955513118, -0.76626003327833025]], [[1.7389373428679082, -1.4733993521485478, -2.029523085640581, 
-0.86560138245690954], [-1.7637760832931915, -3.0748932909447557, 4.4933030839619619, 4.729889024085459], [-3.970662326940908, 
2.5307699021535557, 0.94086798135918048, -4.1294675776104466]]]]),self.functionspace)
      arg1=-1.46348388508
      res=arg0-arg1
      ref=Data(numpy.array([[[[2.6658547663181134, -1.4546208394633995, 2.4205479611640568, -2.4863756377968533], 
[-2.5071897400579544, 3.1734864926504622, 0.94300417781674728, 3.2927819148752127], [-2.7033798719216371, 0.24770568990368469, 
-2.5885664855896673, -2.348771801445678]], [[-1.5568051735031272, -2.5275168373063961, 1.8010490059706852, 1.5654674074258752], 
[-3.3078610448550414, 1.9758211999213566, -0.19622923539014936, 1.7999447034707661], [2.9917803414822455, -0.56654436695697008, 
3.1638298055731537, -0.74333973658136099]]], [[[4.1140002996567331, -2.6993935133846003, 4.2953494025233843, 
6.0937176464066036], [3.1989774310831267, 4.2771878166531065, 4.7192121466234278, 5.2395572800798726], [-2.6317826099775603, 
2.436968402709681, -2.0638827771838884, -3.1377790071620293]], [[6.312662218702827, -0.81829040405978359, -2.0769597979798968, 
-0.37148313702473068], [1.4239690916022099, 4.0808015140096892, 2.989629200220036, 4.999936588548235], [-3.3890040409378086, 
1.4550214736509428, -1.5029242834802643, -3.1158379985738058]]], [[[0.61494310378159955, 2.4741601567213563, 
1.8953915590583303, 5.5056344345955468], [4.9797819586849146, 5.1313786092247016, -2.6545134184265864, 1.2714728159275959], 
[3.4067380955944415, 3.444310047340835, 1.7199579746372233, 0.6972238518037619]], [[3.2024212279500004, -0.0099154670664556299, 
-0.56603920055848889, 0.59788250262518261], [-0.30029219821109931, -1.6114094058626636, 5.956786969044054, 6.1933729091675511], 
[-2.5071784418588159, 3.9942537872356478, 2.4043518664412726, -2.6659836925283544]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[3.5141625618646124, 4.0681083960052238, -0.9117326116800486, -2.8192491299103963], 
[3.6395257114596173, 4.7668471277878535, 1.7253809801821784, 0.12890022641919252], [-2.7831093840377754, 4.3563197887246989, 
4.3005342849541481, 4.2449289932028815]], [[0.26202163857124461, -3.7918288776529852, -2.5680732484332305, 2.1032765192660214], 
[-1.0688291399924275, -3.0129196664116344, 0.029923456768207402, 2.917625187153031], [3.4262950817274955, 4.1374787818026331, 
-2.4172850195025664, -4.6643822899945944]]], [[[0.59817037690321051, -0.67413594027248891, 4.7613914914029998, 
4.5248455274049491], [1.255045143387771, 2.9236228649905316, -1.8522183833258277, -2.3833764517172904], [1.7576515846274603, 
-1.8322551814530739, -3.6462120204083179, -2.1790819574210998]], [[-2.8199959264187644, 0.48696117546027651, 
3.2813763765148298, 3.2719260051098065], [-2.0438580895290173, 2.5852087369746748, -1.8942977891967594, 1.0532499446405019], 
[3.026411479821693, 2.6248506554879691, 1.3365932332494967, -2.3898309603767229]]], [[[4.6552312311221442, 2.6043503222098296, 
3.0117585496307377, -3.1200779841280477], [-4.5765339325626844, -0.91619359011837265, -3.2937833881963821, 3.3034340200245502], 
[0.28625629451562773, -3.9701966348910647, -1.6327277199832171, -2.5692693984255666]], [[-2.7561446795417566, 
-1.9056950602661358, 3.0400634687518995, 3.97459581182971], [-4.9190352201798557, 0.28301874461475407, -1.0338912252275891, 
3.7667167912827253], [-0.35870209852947976, -2.2276335337595397, 4.9523471199261895, 2.1176548995896116]]]]),self.functionspace)
      arg1=numpy.array(-0.960073248342)
      res=arg0-arg1
      ref=Data(numpy.array([[[[4.4742358102062596, 5.028181644346871, 0.048340636661598602, -1.8591758815687491], 
[4.5995989598012645, 5.7269203761295007, 2.6854542285238256, 1.0889734747608397], [-1.8230361356961282, 5.3163930370663461, 
5.2606075332957953, 5.2050022415445287]], [[1.2220948869128918, -2.831755629311338, -1.6080000000915833, 3.0633497676076686], 
[-0.10875589165078026, -2.0528464180699872, 0.9899967051098546, 3.8776984354946782], [4.3863683300691427, 5.0975520301442803, 
-1.4572117711609192, -3.7043090416529472]]], [[[1.5582436252448577, 0.28593730806915829, 5.721464739744647, 
5.4849187757465963], [2.2151183917294182, 3.8836961133321788, -0.89214513498418047, -1.4233032033756432], [2.7177248329691075, 
-0.87218193311142667, -2.6861387720666707, -1.2190087090794526]], [[-1.8599226780771172, 1.4470344238019237, 4.241449624856477, 
4.2319992534514537], [-1.0837848411873701, 3.545281985316322, -0.9342245408551122, 2.0133231929821491], [3.9864847281633402, 
3.5849239038296163, 2.2966664815911439, -1.4297577120350757]]], [[[5.6153044794637914, 3.5644235705514768, 3.9718317979723849, 
-2.1600047357864005], [-3.6164606842210372, 0.043879658223274554, -2.3337101398547349, 4.2635072683661974], 
[1.2463295428572749, -3.0101233865494175, -0.67265447164156988, -1.6091961500839194]], [[-1.7960714312001094, 
-0.94562181192448858, 4.0001367170935467, 4.9346690601713572], [-3.9589619718382085, 1.2430919929564013, -0.073817976885941938, 
4.7267900396243725], [0.60137114981216744, -1.2675602854178925, 5.9124203682678367, 3.0777281479312588]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[-0.70393776097946947, -0.070882204936550508, 3.6283232384386803, -3.4861030597994902], 
[-2.7397865655253639, -1.0912577383325086, -2.4238898655818151, 0.49872642006038159], [3.9295536578653465, 
-0.46634345526823129, 0.47174940158354772, 3.7582555613428976]], [[-0.50410758085798335, -4.7469568247892653, 
-3.10887091012816, 3.1199513561152017], [-3.0490364333719224, -4.5948560611925267, -2.2195838229113916, 4.6590313172691395], 
[-4.8308929397816112, -2.3327010878465781, -0.04976805006086682, 3.1481984462237023]]], [[[-0.66068956016861868, 
-3.8021911531263441, 1.4083100795839023, -4.5072723843251259], [-0.37537669622971848, -4.3576653909655549, -4.4945818399382622, 
4.2553136644232552], [1.0560354608270561, -0.31464734483628209, 4.3031299877213005, 3.1743221825510215]], 
[[-4.4496097637426892, 4.9204570130583107, -1.7194977048231586, 1.0468964287885862], [0.040825650370090116, 4.7072668800382278, 
3.5728153152502937, 3.0987438056293044], [0.93142287030157256, 1.7933387956653721, -0.60581149083485464, 2.9568562109141103]]], 
[[[2.3507942384336946, -2.917680546212329, 3.3171152637032595, -3.8364898099017442], [-1.0423756888867883, 
-0.12233305532847272, 4.1151430650030427, 4.4019778112764527], [-3.7782039320819552, 2.427719546565374, 2.0145799580540347, 
-0.12946511761921098]], [[-1.5452433092117559, -4.8514863586864889, 0.890499226375848, -2.2366937037335113], 
[-0.31694344459170498, -1.8017234320445716, -2.4019594092241592, 0.96235624145101184], [-0.52081165883334801, 
1.739504427203471, 4.3031590560876758, -2.9342096768959736]]]]),self.functionspace)
      arg1=numpy.array([[[[1.6448267670054584, -1.9999054425499905, -0.61085969649318805, -3.3077201464057202], 
[-3.6377633133521678, -2.0122286257119204, 0.60464637410542554, -4.3408554681277325], [-1.9092166709399319, 2.8880992269274417, 
-0.91811917332855053, -3.4624107709703891]], [[0.50297343378937764, -4.2713095163261974, -4.0412600150846849, 
-3.9348631932064713], [4.5496887272202056, -4.2880417752335624, 4.0048175885667145, -2.3259355943029347], [-2.2623248979474528, 
0.59718959000773175, -2.0631607287306473, -0.097650311624960295]]], [[[-1.0845041789062271, 3.8460131058729523, 
-1.122849358440269, -2.723488239763455], [-1.2037645737267058, -2.1229445684496531, -1.7475819130524251, 0.93261156369880194], 
[-2.5034348435509193, -0.14324979113733338, -1.231265283957208, 0.74512073928912681]], [[-2.8691630068793517, 
-3.4918205865103937, -4.2527970631559375, 4.7576632371597967], [-2.1828136955786972, -4.3496543937050136, -4.5338743162621897, 
-0.14885012624863503], [1.4790029970441552, 1.537172381792594, 0.38936360772544365, -1.9935298083789998]]], 
[[[0.2443804404475225, -2.5177009557353736, -3.5436302880385395, 3.2452976878765014], [2.4268106389906308, 3.7732278205282768, 
2.7641193768954695, 2.6948989250510529], [-3.958273414333525, -4.6893264614318442, -3.6055602493053862, -1.4585328507110908]], 
[[0.00031549025685340126, 1.5012699809364163, -4.4986499412133387, -2.1925652007517362], [-3.4824488548093671, 
2.0604501138733236, 1.6238222313684094, -2.4537240108819605], [0.063035093527022745, 4.20858365259671, -4.4952911179569499, 
1.553012206539135]]]])
      res=arg0-arg1
      ref=Data(numpy.array([[[[-2.3487645279849279, 1.92902323761344, 4.2391829349318684, -0.17838291339376999], 
[0.89797674782680392, 0.9209708873794118, -3.0285362396872406, 4.8395818881881141], [5.8387703288052784, -3.354442682195673, 
1.3898685749120983, 7.2206663323132867]], [[-1.007081014647361, -0.47564730846306791, 0.93238910495652494, 7.054814549321673], 
[-7.598725160592128, -0.30681428595896421, -6.2244014114781061, 6.9849669115720747], [-2.5685680418341583, -2.9298906778543099, 
2.0133926786697804, 3.2458487578486626]]], [[[0.4238146187376084, -7.6482042589992965, 2.5311594380241713, 
-1.7837841445616709], [0.82838787749698728, -2.2347208225159019, -2.7469999268858372, 3.3227021007244533], [3.5594703043779754, 
-0.17139755369894871, 5.534395271678509, 2.4292014432618947]], [[-1.5804467568633376, 8.4122775995687036, 2.533299358332779, 
-3.7107668083712104], [2.2236393459487873, 9.0569212737432423, 8.1066896315124843, 3.2475939318779394], [-0.54758012674258261, 
0.25616641387277816, -0.99517509856029829, 4.9503860192931102]]], [[[2.1064137979861721, -0.39997959047695542, 
6.8607455517417986, -7.0817874977782456], [-3.4691863278774191, -3.8955608758567495, 1.3510236881075732, 1.7070788862253998], 
[0.18006948225156982, 7.1170460079972182, 5.6201402073594213, 1.3290677330918799]], [[-1.5455587994686093, -6.3527563396229052, 
5.3891491675891867, -0.044128502981775064], [3.1655054102176621, -3.8621735459178952, -4.0257816405925686, 3.4160802523329723], 
[-0.58384675236037076, -2.469079225393239, 8.7984501740446248, -4.4872218834351081]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[-3.507144230514684, -2.561374415909591, 0.43802899096726211, 1.0717264300005827], 
[-1.9719537831458767, -1.3650490569340743, -0.32641721284699088, 4.2401301134669165], [4.3644354872500948, -1.7180691994614761, 
-3.519502126037092, 4.8291221677703451]], [[-3.6741225016844004, -1.1065258644347664, 3.6917688474416046, -3.1619256041899146], 
[-3.1511390708316203, -2.6281172277709741, -3.7301356086722368, -1.6708744130460609], [-4.1377390142218875, 
0.26179664937023706, 3.9203202409987234, -4.8757030443464791]]], [[[2.4837378127866643, -4.5637224915272192, 
-4.4434598142604509, 4.2466964194202514], [-2.0445707790294176, -2.0235856234790384, 0.13788348967139719, -4.5670195570486776], 
[4.9736893826710027, -2.3048935860430841, -0.33410925473265252, 3.1706474111152332]], [[0.48867680805471636, 
-2.4839322637609729, 2.7576647449857372, 2.4333807373824436], [-4.6727984545524057, 0.11685208540538383, 1.5788871088256968, 
-2.8330584943747841], [-3.9792475916357359, 0.64294281893958605, -1.7461816522175533, 3.7322888773174121]]], 
[[[-2.4775885418029411, 3.026498540525667, -2.6840732431608072, 1.2371895257742507], [4.3896948948784758, -1.8093607605524986, 
0.26412420534981251, 0.29906379028825558], [-4.0427081087415377, 3.428733759095854, -0.15932634960236491, 4.6461191033866314]], 
[[-0.7611479206396119, 2.52447210022427, -3.0306977694121762, 2.3308773546258017], [1.4614651694583642, 1.3457659173182588, 
-2.4602979791189075, 3.7930329651106618], [2.3180359779520563, -0.32836213365212696, -2.8357917830027191, 
-4.5576334259559452]]]]),self.functionspace)
      arg1=Data(3.24741427653,self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[[-6.7545585070422751, -5.8087886924371821, -2.809385285560329, -2.1756878465270084], 
[-5.2193680596734673, -4.6124633334616654, -3.5738314893745819, 0.99271583693932541], [1.1170212107225037, -4.9654834759890676, 
-6.7669164025646831, 1.581707891242754]], [[-6.9215367782119914, -4.3539401409623579, 0.44435457091401354, 
-6.4093398807175053], [-6.3985533473592113, -5.8755315042985652, -6.9775498851998279, -4.918288689573652], 
[-7.3851532907494786, -2.985617627157354, 0.67290596447113238, -8.1231173208740692]]], [[[-0.76367646374092679, 
-7.8111367680548103, -7.690874090788042, 0.99928214289266037], [-5.2919850555570083, -5.270999900006629, -3.1095307868561939, 
-7.8144338335762686], [1.7262751061434116, -5.5523078625706752, -3.5815235312602436, -0.076766865412357888]], 
[[-2.7587374684728747, -5.7313465402885644, -0.48974953154185386, -0.81403353914514742], [-7.9202127310799968, 
-3.1305621911222072, -1.6685271677018942, -6.0804727709023751], [-7.226661868163327, -2.604471457588005, -4.9935959287451439, 
0.48487460078982103]]], [[[-5.7250028183305322, -0.22091573600192405, -5.9314875196883978, -2.0102247507533404], 
[1.1422806183508847, -5.0567750370800901, -2.9832900711777786, -2.9483504862393355], [-7.2901223852691288, 0.18131948256826291, 
-3.406740626129956, 1.3987048268590403]], [[-4.008562197167203, -0.72294217630332103, -6.2781120459397677, 
-0.91653692190178937], [-1.7859491070692268, -1.9016483592093323, -5.7077122556464985, 0.54561868858307072], 
[-0.92937829857553478, -3.575776410179718, -6.0832060595303101, -7.8050477024835363]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[-1.4855746206648766, -1.4109799995385517, -0.013939334971384554, 3.1119043276398273], 
[-3.4919675654491389, -0.91596810143057983, -0.2320441770289392, 2.2926598102656639], [-4.8445236707938033, 
-4.9104059387628727, -0.33669913529667994, 2.8918136356417454]], [[4.5779828314714521, -1.8374195228812096, 1.5523559799992306, 
4.4715442152460163], [0.0028835527947590478, 4.4616272220713213, -2.3302997116453694, -3.4998138135914001], 
[3.5607216544048654, 4.2771634130452281, -1.3436122590895083, 2.9370770638932786]]], [[[-1.7542263936211224, 
3.9731480968914052, 0.50108471485408312, -1.4528353601467039], [-4.8996400093122983, -4.4334849883323777, 0.1141683415830208, 
-0.10278657136031466], [2.2609420416818473, 0.90555754208380446, 1.0685839274431537, -2.2650570344295682]], 
[[0.92263032526557787, 2.0682986378763291, 0.86623196669682123, 2.5694473946786651], [-1.570890033877006, 2.9391007540001377, 
-4.0316153669160144, -0.19536845570003081], [0.32216773729721293, 0.76942871333356955, 2.1517946273639108, 
-0.35592039845743528]]], [[[-1.7427113695890482, 3.2933059523962314, 3.7414721792857453, 0.42833256670039788], 
[1.0141499588142491, 1.5106355535346125, -2.9523240621654514, -0.99114369354606513], [-1.6280589363037343, -1.8912094730432449, 
-0.047804572055046179, -2.1584025215153968]], [[-0.10709907784480333, -3.3386998035672888, 0.42589472120197236, 
-4.147915995336616], [-1.3564814429334158, -0.62080159992230488, -2.8339883757219821, 0.0042265027446939385], 
[0.62633588446333466, 3.2843099727525917, -1.9297784612110092, -2.6739788922814247]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[2.0219337422553352, 2.6210634248060494, 1.5970488984185351, 3.8848885131792468], 
[-0.25742905433896013, 3.4982646209244788, -1.6245850752082749, 3.4321431523944703], [-2.8166566775867863, -3.0865848068133248, 
3.2631887796519639, 2.3293862709445925]], [[2.3091790864573181, 2.1550771556633208, 4.0078102346811022, 2.9732635888133929], 
[-0.36969054961323078, 1.7076266049459701, -2.9415068948766088, 2.3648067704719997], [2.6608383329933671, 2.8157175167760764, 
-4.4478409895147388, 4.0617409356101035]]], [[[3.9609093225017471, -3.5275587479897275, -3.1198373157204795, 
1.3779984136559005], [-1.3670012874587858, -0.94558092692382978, -4.1864988198969204, 4.9571425296465392], 
[-4.0999540484127603, 0.6340890518349207, -0.72486758492613923, -0.65160684916962985]], [[0.28310785296799867, 
-4.4577405659097868, 1.1856938015317482, 3.193973968271159], [-4.6826100947814187, -4.7835904024536191, -1.8375693942134044, 
-1.6066022164339246], [4.6601724225738437, 2.969768990919782, 4.1269428267144974, -4.7700464997938443]]], 
[[[-2.1716064763306022, -2.3135310282320698, -4.0725079031968647, -3.6614033972379478], [2.3281566090821437, 
2.4474842135811601, 0.46979006919700694, -4.1016053771359697], [-2.7401553913834773, 4.6020664509247382, 4.2967672492260398, 
0.58054706890251317]], [[-0.72874907808758405, -4.5831241573957371, 2.7557629655440072, 0.40137380411929335], 
[0.66103526247437205, 0.92761308985415702, -2.400238350393372, 4.8774445356301737], [-0.86366789592547644, 3.2254118283344724, 
2.4349677392639668, 4.0162389993599046]]]]),self.functionspace)
      res=arg0-arg1
      ref=Data(numpy.array([[[[-3.5075083629202117, -4.0320434243446011, -1.6109882333899197, -0.7729841855394195], 
[-3.2345385111101788, -4.4142327223550586, 1.3925408981793357, -1.1394833421288064], [-2.027866993207017, -1.8238211319495479, 
-3.5998879149486438, 0.56242736469715293]], [[2.2688037450141341, -3.9924966785445304, -2.4554542546818716, 
1.4982806264326234], [0.37257410240798983, 2.7540006171253513, 0.61120718323123935, -5.8646205840633998], [0.89988332141149829, 
1.4614458962691517, 3.1042287304252305, -1.1246638717168249]]], [[[-5.7151357161228695, 7.5007068448811332, 3.6209220305745626, 
-2.8308337738026044], [-3.5326387218535125, -3.4879040614085479, 4.3006671614799412, -5.0599291010068539], [6.3608960900946077, 
0.27146849024888375, 1.7934515123692929, -1.6134501852599383]], [[0.63952247229757919, 6.5260392037861159, 
-0.31946183483492696, -0.62452657359249386], [3.1117200609044127, 7.7226911564537568, -2.19404597270261, 1.4112337607338938], 
[-4.3380046852766307, -2.2003402775862124, -1.9751481993505866, 4.414126101336409]]], [[[0.42889510674155407, 
5.6068369806283016, 7.81398008248261, 4.0897359639383453], [-1.3140066502678946, -0.93684866004654754, -3.4221141313624583, 
3.1104616835899046], [1.112096455079743, -6.4932759239679836, -4.344571821281086, -2.73894959041791]], [[0.62165000024278072, 
1.2444243538284483, -2.3298682443420349, -4.5492897994559094], [-2.0175167054077878, -1.5484146897764619, -0.43375002532861018, 
-4.8732180328854797], [1.4900037803888111, 0.058898144418119358, -4.3647462004749755, 
-6.6902178916413293]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[-4.6059615424260469, -1.7325057192074711, 3.1314040044400695, -1.7736517097727713], 
[-3.8792882564803732, -3.8801353265473635, 3.5246868447961806, 0.22748752437957176], [3.4257188665863456, -1.7500052709775247, 
3.6424246584119686, -3.6305565521802463]], [[4.8368795479275786, -1.9641676695735955, -0.27066725745386133, 
2.5665482095372667], [-1.0706383955974963, 1.709401660063091, -1.6206966008612964, 2.3854725319389214], [2.1968432399343829, 
2.29883017220574, 3.7565418391072782, -4.2735310637500401]]], [[[-3.1320383580297726, -4.4385631852306808, 2.2981375629627943, 
0.90469141462380875], [3.1280437269801968, 0.025440963990820187, 1.6103891041484353, 0.30609883307511865], [1.817577235251953, 
0.31559180849475954, -2.5980015493304229, -0.88810450950030173]], [[2.2025313380653593, 1.7743281430944213, 1.2808190048982668, 
3.3529901399815962], [-0.86441444648566268, 3.9587116213010596, -3.8056625734128033, 4.5375358587016255], [2.403690357334364, 
-2.6521134243052567, -0.63759637886833076, 1.068779338210514]]], [[[4.6668568761531599, -1.9855741769759616, 
3.3332623467064177, -3.8643775781335465], [-3.0584979711341411, 4.4845240645708806, 4.8417184091897241, 0.30031271550930771], 
[3.2010829100997782, 2.5562443003976023, 4.8447915826480852, -4.9485425101217029]], [[-4.4052940459531875, -4.9305449772661847, 
-3.6337661026174475, 2.0768223361501894], [-3.3705110293967033, -3.9080805129151042, -1.3015914234271198, -2.7912994155767623], 
[3.393355967108814, -4.7074870172460219, -2.6333445190551052, -0.30543725141328437]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-3.00820936522)+(1.-msk_arg1)*(-2.48091671067)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.5977521772107472, 1.2757036460078286, 6.1396133696553692, 1.2345576554425284], 
[-0.87107889126507354, -0.87192596133206379, 6.5328962100114802, 3.2356968895948715], [6.4339282318016453, 1.258204094237775, 
6.6506340236272683, -0.62234718696494662]], [[7.8450889131428783, 1.0440416956417042, 2.7375421077614384, 5.5747575747525664], 
[1.9375709696178034, 4.7176110252783907, 1.3875127643540033, 5.3936818971542211], [5.2050526051496826, 5.3070395374210397, 
6.7647512043225779, -1.2653216985347404]]], [[[-0.1238289928144729, -1.4303538200153811, 5.3063469281780939, 
3.9129007798391084], [6.1362530921954965, 3.0336503292061199, 4.618598469363735, 3.3143081982904183], [4.8257866004672527, 
3.3238011737100592, 0.41020781588487676, 2.120104855714998]], [[5.210740703280659, 4.782537508309721, 4.2890283701135665, 
6.3611995051968959], [2.143794918729637, 6.9669209865163593, -0.79745320819750365, 7.5457452239169251], [5.4118997225496637, 
0.35609594091004304, 2.3706129863469689, 4.0769887034258137]]], [[[7.6750662413684596, 1.0226351882393381, 6.3414717119217174, 
-0.85616821291824685], [-0.050288605918841434, 7.4927334297861803, 7.8499277744050238, 3.3085220807246074], 
[6.2092922753150779, 5.564453665612902, 7.8530009478633849, -1.9403331449064032]], [[-1.3970846807378878, -1.922335612050885, 
-0.62555673740214779, 5.0850317013654891], [-0.36230166418140364, -0.89987114769980447, 1.7066179417881799, 
0.21690994963853738], [6.4015653323241137, -1.6992776520307222, 0.37486484616019444, 
2.7027721138020153]]]])+(1.-msk_ref)*numpy.array([[[[-2.125044831755778, 0.74841099146279788, 5.6123207151103385, 
0.70726500089749766], [-1.3983715458101043, -1.3992186158770945, 6.0056035554664495, 2.7084042350498407], [5.9066355772566146, 
0.7309114396927443, 6.1233413690822376, -1.1496398415099773]], [[7.3177962585978475, 0.51674904109667352, 2.2102494532164076, 
5.0474649202075357], [1.4102783150727727, 4.19031837073336, 0.86022010980897257, 4.8663892426091904], [4.6777599506046519, 
4.779746882876009, 6.2374585497775472, -1.7926143530797711]]], [[[-0.65112164735950362, -1.9576464745604119, 
4.7790542736330632, 3.3856081252940777], [5.6089604376504658, 2.5063576746610892, 4.0913058148187043, 2.7870155437453876], 
[4.298493945922222, 2.7965085191650285, -0.11708483866015396, 1.5928122011699672]], [[4.6834480487356283, 4.2552448537646903, 
3.7617357155685358, 5.8339068506518652], [1.6165022641846063, 6.4396283319713286, -1.3247458627425344, 7.0184525693718944], 
[4.884607068004633, -0.17119671363498767, 1.8433203318019382, 3.5496960488807829]]], [[[7.1477735868234289, 
0.49534253369430736, 5.8141790573766867, -1.3834608674632776], [-0.57758126046387215, 6.9654407752411496, 7.322635119859993, 
2.7812294261795767], [5.6819996207700472, 5.0371610110678713, 7.3257082933183542, -2.4676257994514339]], [[-1.9243773352829185, 
-2.4496282665959157, -1.1528493919471785, 4.5577390468204584], [-0.88959431872643435, -1.4271638022448352, 1.1793252872431492, 
-0.31038270490649333], [5.874272677779083, -2.2265703065757529, -0.15242780838483627, 2.1754794592569846]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[2.4394779691925006, -1.2013608697025502, 1.7179856950102037, -3.2170087912032566], 
[-1.7177578281377848, -2.6126790376971676, -4.2678439895805633, -2.0737302922189658], [2.6945046817998284, -2.3618364357814317, 
3.652191609373455, -0.99184041558004665]], [[-4.4131148625297261, 3.6497802814037641, -1.5408618319934266, 
-0.84353047034570583], [0.84610434258342782, -4.8769326587145292, -0.089234016720443954, -3.3046429893391749], 
[-4.8462055065778156, -0.72334003442802519, -0.24440945115219304, -0.59915047491768458]]], [[[0.84766378042896307, 
-3.5978660294659326, -1.6192256678036507, 3.6029046594225775], [-3.2840724721707848, 2.7147235415590369, 0.19379291382998964, 
-2.6888870424883526], [-3.3813501391012499, 3.2324423119914751, 1.8772833083288685, 3.7292630714252724]], 
[[-3.3587130498388316, -0.37643502206925206, -2.1633965940823296, -3.2624859725379816], [-2.5037802757132264, 
4.290458332341009, 4.8085582691232638, -2.7492126066589484], [1.8720653805881602, 3.0713905566018056, 4.5229482176814066, 
3.2200624282001904]]], [[[-2.0380640509732819, 0.3995455270257775, -1.7140265458353565, 1.4591502123701314], 
[-0.17835458977943031, -2.2839318951282972, 0.49889446591153597, 2.8107525200454253], [-2.6769994241779225, 
-0.22481144760654015, -1.8411772390888981, 0.10072439696559332]], [[-4.6613354691091633, 2.353025115808026, 
-3.9478677834642362, 0.5884014049777857], [2.8361554351233984, -2.3386268199397833, 4.3230358098391655, 1.5638626070279846], 
[3.8287787573513885, 3.2342853516190608, -0.098609773887146979, 2.610651559103065]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-3.5909289306484005, -2.7638228834813017, 0.31334723458808611, -4.3717750820226318], 
[2.336327081753077, 1.1976327449868034, 4.3466640582515019, 3.464759312347006], [3.4431632025429106, -0.58048150755161476, 
-0.27388358300874405, 3.7115929611456036]], [[1.450429530519167, 3.585515195124735, -4.2570227791364754, 2.3930296247805138], 
[-0.041870520093686636, -4.0376637477524602, -1.3179501242510314, 2.8796465198084276], [-3.0767888334584459, 
4.8319566228739923, 2.1850195828946806, -1.6414102821813059]]], [[[3.9297573267182653, 1.8269203433276999, -4.8660749743808864, 
-4.6900307345630399], [-0.53958932122048697, -3.5908444401718054, 1.3113724336483283, -2.0144808558738472], 
[0.45409447504393885, -1.7018623537605491, -1.3907939561692784, -1.0672062497485513]], [[-2.4732644544532487, 
3.0431976418377502, 1.2120250098232352, 0.63052615993147132], [3.7418862261388917, -1.8555201141367803, 1.1698884491584671, 
1.7027240217937676], [2.086407238349187, -1.838031068476238, 2.7621213575145287, -2.2437656248053162]]], [[[1.3702340767839418, 
-4.3938252802386613, -1.6637741183340058, 1.4145680107623821], [4.1772622936136496, -1.9868280867662835, 3.7800627422642386, 
-3.871686626195916], [-0.5968273908298185, 2.4301670517847374, 3.4877537725209322, 1.049741508501433]], [[1.8730398061515015, 
3.9605558282346998, 4.1020471857301182, 0.36333248630592951], [-2.7679211496016531, -4.795972586389321, 0.9479860434264733, 
0.68245712281981774], [-4.4883737965571839, -4.8999972871474267, -0.63676472835300935, 
-3.7124002399773817]]]])+(1.-msk_arg1)*numpy.array([[[[-1.1811574440862103, 0.096079936123435239, 0.66611945285748675, 
3.1124729834789591], [1.6583398995929244, -0.56364522941862649, -0.28898088589691806, 4.8000112008973908], 
[0.32426776385665956, 1.3386317113670003, 0.59996360119036307, 3.4596850807237303]], [[2.0142651638423779, 3.7553407356533377, 
2.5256706077951758, -2.2602306745286227], [4.8051332098350326, 3.6859653962064893, -2.2504159770551659, 3.2482203269559342], 
[4.0479938701259606, -1.2959960083148525, 4.3114906730464924, 1.920709948269713]]], [[[-0.23641968999129492, 
-3.5776383218263774, 0.74816917822164175, -4.5711458993927199], [2.6758606180916118, 3.61961806118574, 2.5735691806581951, 
-0.067041217104551798], [-4.1782022674753394, 4.5970442160057665, 2.3727498658638035, 3.2199527103916719]], 
[[2.6837994714765472, -0.69544812065891382, 3.9454749018856852, -3.021806644170967], [3.4486510005786482, 4.8258816329773691, 
1.9819246471608469, -0.34285772802746273], [-2.2865699904888039, 2.5006875562688178, -0.96933797444933223, 
2.477455558346068]]], [[[-3.4066652568233757, -0.18711993240428626, 0.66355435137871677, 2.7907048887669816], 
[0.45987979322352235, -4.9816119071608247, -3.4633655184370102, 0.48718223847286612], [2.9054098493715976, 4.0809413396980574, 
1.8885565367349155, -0.064018008697745543]], [[1.986024326348236, 0.78644638696903257, 0.28181260174655254, 
-1.4375537879227638], [-0.90193005673794513, 3.3131746704113691, 4.1405583388550902, -4.2415769468585038], 
[-2.5407758675946592, 3.2890140441893081, 1.0693935965844217, 1.2271033668603959]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[6.0304068998409015, 1.5624620137787515, 1.4046384604221176, 1.1547662908193752], 
[-4.0540849098908618, -3.810311782683971, -8.6145080478320644, -5.5384896045659717], [-0.74865852074308226, 
-1.7813549282298169, 3.926075192382199, -4.7034333767256502]], [[-5.8635443930488931, 0.064265086279029049, 2.7161609471430488, 
-3.2365600951262197], [0.88797486267711445, -0.83926891096206901, 1.2287161075305875, -6.184289509147602], 
[-1.7694166731193697, -5.5552966573020175, -2.4294290340468736, 1.0422598072636213]]], [[[-3.0820935462893022, 
-5.4247863727936325, 3.2468493065772357, 8.2929353939856174], [-2.7444831509502978, 6.3055679817308423, -1.1175795198183387, 
-0.67440618661450547], [-3.8354446141451888, 4.9343046657520242, 3.2680772644981468, 4.7964693211738236]], 
[[-0.88544859538558285, -3.4196326639070023, -3.3754216039055649, -3.8930121324694529], [-6.245666501852118, 
6.1459784464777893, 3.6386698199647967, -4.4519366284527155], [-0.21434185776102677, 4.9094216250780436, 1.7608268601668779, 
5.463828053005507]]], [[[-3.4082981277572237, 4.7933708072644388, -0.050252427501350727, 0.044582201607749283], 
[-4.35561688339308, -0.29710380836201367, -3.2811682763527026, 6.6824391462413413], [-2.080172033348104, -2.6549784993912775, 
-5.3289310116098303, -0.94901711153583967]], [[-6.5343752752606648, -1.6075307124266738, -8.0499149691943543, 
0.22506891867185619], [5.6040765847250515, 2.4573457664495377, 3.3750497664126922, 0.88140548420816689], [8.3171525539085724, 
8.1342826387664875, 0.53815495446586237, 6.3230517990804467]]]])+(1.-msk_ref)*numpy.array([[[[3.6206354132787109, 
-1.2974408058259854, 1.051866242152717, -6.3294817746822156], [-3.3760977277307092, -2.0490338082785411, -3.9788631036836453, 
-6.8737414931163565], [2.3702369179431688, -3.700468147148432, 3.0522280081830919, -4.4515254963037769]], 
[[-6.4273800263721039, -0.10556045424957361, -4.0665324397886025, 1.4167002041829169], [-3.9590288672516047, 
-8.5628980549210176, 2.161181960334722, -6.5528633162951095], [-8.8941993767037761, 0.57265597388682732, -4.5559001241986854, 
-2.5198604231873976]]], [[[1.084083470420258, -0.02022770763955517, -2.3673948460252925, 8.1740505588152974], 
[-5.9599330902623961, -0.90489451962670309, -2.3797762668282054, -2.6218458253838008], [0.79685212837408947, 
-1.3646019040142914, -0.49546655753493507, 0.5093103610336005]], [[-6.0425125213153787, 0.31901309858966176, 
-6.1088714959680148, -0.24067932836701456], [-5.9524312762918745, -0.53542330063636001, 2.8266336219624169, 
-2.4063548786314857], [4.1586353710769641, 0.57070300033298782, 5.4922861921307389, 0.74260686985412239]]], 
[[[1.3686012058500938, 0.58666545943006376, -2.3775808972140733, -1.3315546763968502], [-0.63823438300295265, 
2.6976800120325275, 3.9622599843485462, 2.3235702815725592], [-5.5824092735495201, -4.3057527873045975, -3.7297337758238136, 
0.16474240566333886]], [[-6.6473597954573993, 1.5665787288389934, -4.2296803852107887, 2.0259551929005495], 
[3.7380854918613435, -5.6518014903511524, 0.1824774709840753, 5.8054395538864885], [6.3695546249460477, -0.054728692570247262, 
-1.1680033704715687, 1.3835481922426691]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.07873015689)+(1.-msk_arg0)*(1.02364660145)
      arg1=0.402031418727
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(2.67669873816)+(1.-msk_ref)*(0.621615182718)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.28892430022)+(1.-msk_arg0)*(-0.0661129219287)
      arg1=numpy.array(3.88804115649)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.400883143728)+(1.-msk_ref)*(-3.95415407842)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.235338843637)+(1.-msk_arg0)*(2.13994445483)
      arg1=numpy.array([3.195501679500893, -0.1542612189839323])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-2.9601628358636169, 0.38960006262120839])+(1.-msk_ref)*numpy.array([-1.0555572246693874, 
2.2942056738154379])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.83261824771)+(1.-msk_arg0)*(-2.89017265755)
      arg1=numpy.array([[-1.6455104803061982, 0.10671642587973285, 4.592760565241921, 2.3903620799748824, -4.1241118682789359], 
[3.872707382607409, -4.7103021664713136, -0.44304104496978347, 1.5485580236829897, -3.2820723997450587], 
[-0.023847569420301973, -2.5375959082299082, -4.1843918781096692, -2.8773442900523873, 1.9102074153550195], 
[4.1846835397083293, -1.4106097160448461, -4.5419594016414031, 1.6463452615829883, 1.6882520943674342]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[3.4781287280171211, 1.7259018218311901, -2.760142317530998, -0.5577438322639594, 
5.9567301159898589], [-2.040089134896486, 6.5429204141822366, 2.2756592926807064, 0.28406022402793329, 5.1146906474559817], 
[1.8564658171312249, 4.3702141559408307, 6.0170101258205921, 4.7099625377633103, -0.077589167644096513], [-2.3520652919974063, 
3.243227963755769, 6.3745776493523261, 0.18627298612793464, 
0.14436615334348879]])+(1.-msk_ref)*numpy.array([[-1.2446621772431743, -2.9968890834291053, -7.4829332227912939, 
-5.2805347375242544, 1.2339392107295635], [-6.7628800401567819, 1.8201295089219411, -2.447131612579589, -4.4387306812323626, 
0.39189974219568624], [-2.8663250881290705, -0.35257674931946426, 1.2942192205602967, -0.012828367496985127, 
-4.8003800729043924], [-7.0748561972577022, -1.4795629415045264, 1.6517867440920306, -4.5365179191323612, -4.5784247519168062]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.11329216156)+(1.-msk_arg0)*(-0.608861097249)
      arg1=numpy.array([[[-1.5034674314915075, -3.2403304982099557], [1.8365596904258457, -4.3340325165602787]], 
[[-0.94916014322429909, -2.5730288448051497], [2.6030202761423222, 3.3166274616516471]], [[2.7595170356471135, 
-0.62845205968192452], [4.9334116882663839, 2.9157350209853856]], [[4.6133390863029007, -2.1606071464420573], 
[-1.5090338104787659, 1.4801625572298605]], [[-1.6972698746774273, -1.0082629491827753], [-4.9521312294894804, 
-4.9068538498756284]], [[-3.7555799159774916, -4.2750053916642816], [3.8995700424036457, -3.119588596537505]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-2.6098247300714577, -0.87296166335300951], [-5.9498518519888108, 0.22074035499731348]], 
[[-3.1641320183386661, -1.5402633167578155], [-6.7163124377052874, -7.4299196232146123]], [[-6.8728091972100787, 
-3.4848401018810407], [-9.0467038498293491, -7.0290271825483508]], [[-8.7266312478658659, -1.9526850151209079], 
[-2.6042583510841992, -5.5934547187928256]], [[-2.4160222868855379, -3.1050292123801899], [0.83883906792651519, 
0.79356168831266327]], [[-0.35771224558547354, 0.16171323010131644], [-8.0128622039666109, 
-0.9937035650254602]]])+(1.-msk_ref)*numpy.array([[[0.89460633424231117, 2.6314694009607593], [-2.445420787675042, 
3.7251714193110823]], [[0.34029904597510274, 1.9641677475559534], [-3.2118813733915186, -3.9254885589008435]], 
[[-3.3683781328963098, 0.019590962432728176], [-5.5422727855155802, -3.524596118234582]], [[-5.222200183552097, 
1.551746049192861], [0.9001727132295696, -2.0890236544790568]], [[1.0884087774282309, 0.39940185193357891], [4.343270132240284, 
4.2979927526264321]], [[3.1467188187282953, 3.6661442944150853], [-4.508431139652842, 2.5107274992883086]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.82428828897)+(1.-msk_arg0)*(-4.16999105728)
      arg1=numpy.array([[[[-1.4104134688482803, 1.7174949134896558, 2.8200102494765673, 0.65470566370657846], 
[4.8764991837527347, 2.411182455995899, 0.80212410929585687, 0.42746281808792475], [-3.5079458367234206, 1.0386952370097093, 
2.850430363285418, -0.99525025012919599]], [[-4.9997791158649783, -2.4456981726341906, -2.2051672616097271, 
-2.6135377834871063], [-2.450231804219162, 3.7635098283357511, 2.6673877194644247, 4.6826479418666409], [-2.3917012508899296, 
0.29447606762690892, -2.7649130524491259, 2.1216379553208222]]], [[[0.0095201762571672788, -4.3302546337442847, 
1.9235904384996951, -0.24335786422564709], [-1.0403284783718139, 2.1273689843639643, 2.8166921636539524, 1.7100073436620979], 
[0.34165853373029087, -2.1162811589074271, -3.8503078145146308, 3.8830299560166086]], [[-1.0891550280268927, 
-1.4851824068343058, -0.54513244891187451, -2.9963823274328227], [0.21066973106279541, -3.0445496822819837, 
-1.9106294720971295, 4.7670176146641907], [3.0293803115162277, -1.7231503429315409, -3.2849280154817881, 
0.73897415491014407]]], [[[-1.6566733988375648, 3.0197381273083916, -2.8211094072214369, -1.4716765451769289], 
[1.1037486398538645, 3.8788182005240106, 4.6747042962703969, 1.8301996288144515], [-0.19204367196061689, 4.3301913470254103, 
-1.7365976588541043, 1.4096650199909462]], [[-4.2736126183814749, 4.6564288270373257, -3.9081836961946728, 
-2.2674765773760353], [3.6122716616964343, -1.17924415881276, -2.785140216920694, -4.0126414309485856], [-1.944985021559277, 
-0.22600779446457242, 3.885662143878033, 0.34086771753927803]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.4138748201266873, -3.5417832024646234, -4.644298538451535, -2.4789939526815461], 
[-6.7007874727277024, -4.2354707449708666, -2.6264123982708245, -2.2517511070628924], [1.683657547748453, -2.862983525984677, 
-4.6747186522603856, -0.82903803884577165]], [[3.1754908268900106, 0.62140988365922301, 0.38087897263475945, 
0.78924949451213866], [0.62594351524419434, -5.5877981173107187, -4.4916760084393923, -6.5069362308416085], 
[0.56741296191496193, -2.1187643566018766, 0.94062476347415824, -3.9459262442957899]]], [[[-1.8338084652321349, 
2.505966344769317, -3.7478787274746628, -1.5809304247493206], [-0.78395981060315378, -3.9516572733389319, -4.64098045262892, 
-3.5342956326370656], [-2.1659468227052585, 0.29199286993245943, 2.0260195255396631, -5.7073182449915763]], 
[[-0.73513326094807496, -0.33910588214066184, -1.2791558400630931, 1.1720940384578551], [-2.034958020037763, 
1.2202613933070161, 0.08634118312216188, -6.5913059036391584], [-4.8536686004911953, -0.10113794604342674, 1.4606397265068205, 
-2.5632624438851117]]], [[[-0.16761489013740283, -4.8440264162833593, 0.99682111824646924, -0.35261174379803872], 
[-2.9280369288288322, -5.7031064894989782, -6.4989925852453645, -3.6544879177894192], [-1.6322446170143508, -6.154479636000378, 
-0.087690630120863311, -3.2339533089659138]], [[2.4493243294065072, -6.4807171160122934, 2.0838954072197051, 
0.44318828840106761], [-5.436559950671402, -0.64504413016220763, 0.96085192794572638, 2.188353141973618], [0.12069673258430935, 
-1.5982804945103952, -5.7099504328530006, -2.1651560065142457]]]])+(1.-msk_ref)*numpy.array([[[[-2.7595775884344533, 
-5.8874859707723894, -6.990001306759301, -4.8246967209893121], [-9.0464902410354675, -6.5811735132786326, -4.9721151665785905, 
-4.5974538753706584], [-0.66204522055931303, -5.208686294292443, -7.0204214205681517, -3.1747408071535377]], 
[[0.82978805858224458, -1.724292884648543, -1.9648237956730066, -1.5564532737956274], [-1.7197592530635717, 
-7.9335008856184848, -6.8373787767471583, -8.8526389991493737], [-1.7782898063928041, -4.4644671249096426, -1.4050780048336078, 
-6.2916290126035559]]], [[[-4.1795112335399009, 0.16026357646155098, -6.0935814957824288, -3.9266331930570866], 
[-3.1296625789109198, -6.2973600416466979, -6.986683220936686, -5.8799984009448316], [-4.5116495910130245, -2.0537098983753066, 
-0.31968324276810289, -8.0530210132993432]], [[-3.080836029255841, -2.6848086504484279, -3.6248586083708592, 
-1.1736087298499109], [-4.3806607883455291, -1.1254413750007499, -2.2593615851856041, -8.9370086719469235], 
[-7.1993713687989613, -2.4468407143511928, -0.88506304180094553, -4.9089652121928777]]], [[[-2.5133176584451689, 
-7.1897291845911253, -1.3488816500612968, -2.6983145121058048], [-5.2737396971365982, -8.0488092578067452, -8.8446953535531314, 
-6.0001906860971852], [-3.9779473853221168, -8.5001824043081449, -2.4333933984286293, -5.5796560772736798]], 
[[0.10362156109874121, -8.8264198843200603, -0.26180736108806091, -1.9025144799066984], [-7.782262718979168, 
-2.9907468984699737, -1.3848508403620396, -0.15734962633414806], [-2.2250060357234567, -3.9439832628181613, 
-8.0556532011607658, -4.5108587748220117]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.93756455938)+(1.-msk_arg0)*(-3.47766317105)
      arg1=Data(-1.40516165711,self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(4.34272621649)+(1.-msk_ref)*(-2.07250151393)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.575948663227)+(1.-msk_arg0)*(-4.82854958918)
      arg1=Data(numpy.array([-2.9592946982053139, 1.972668332415167]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([3.5352433614321974, -1.3967196691882835])+(1.-msk_ref)*numpy.array([-1.8692548909703497, 
-6.8012179215908306])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.04742819684)+(1.-msk_arg0)*(1.70318073382)
      arg1=Data(numpy.array([[2.6959161945313319, -2.671488911628237, 0.83536603487748717, 1.2057086257062632, 
-1.4011731792179551], [0.46566716347976378, 4.4326245861755531, -4.3241055172162293, 0.72257785351607318, 3.4992355799938579], 
[4.2532482947589543, -0.87000099930823271, 3.1358377735000005, -1.3394089777760101, -2.716158078866818], [3.6303096475657242, 
-4.5839718184547982, -1.9999384822118618, -4.6629963256082281, 1.7302200145844537]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.6484879976893705, 3.7189171084701984, 0.21206216196447425, -0.15828042886430183, 
2.4486013760599166], [0.58176103336219764, -3.3851963893335917, 5.3715337140581907, 0.32485034332588825, -2.4518073831518965], 
[-3.2058200979169929, 1.9174291961501941, -2.0884095766580391, 2.3868371746179715, 3.7635862757087795], [-2.5828814507237627, 
5.6314000152967596, 3.0473666790538232, 5.7104245224501895, 
-0.68279181774249231]])+(1.-msk_ref)*numpy.array([[-0.99273546071184349, 4.3746696454477254, 0.86781469894200125, 
0.49747210811322518, 3.1043539130374436], [1.2375135703397246, -2.7294438523560647, 6.0272862510357177, 0.98060288030341525, 
-1.7960548461743695], [-2.5500675609394658, 2.5731817331277211, -1.4326570396805121, 3.0425897115954985, 4.4193388126863065], 
[-1.9271289137462357, 6.2871525522742866, 3.7031192160313502, 6.3661770594277165, -0.027039280764965312]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.45341826652)+(1.-msk_arg0)*(-3.8530907759)
      arg1=Data(numpy.array([[[2.4044936754106905, -2.2444126979186074], [-1.1907065599600966, -4.4362098204273916]], 
[[0.59747021609652595, -3.9689207882178366], [-0.24565623973755812, 0.5952998711610471]], [[3.5562498126183044, 
2.0306302987328939], [-1.9326647988205692, -2.4576852412299219]], [[-4.6451932239670164, 4.7713034593863508], 
[-1.0312692678710245, -3.6604872453180706]], [[-0.25641551786268657, 1.5574439985420421], [2.3393595598252404, 
-1.1739153887761979]], [[-2.9102058244868378, 2.0075573184652225], [4.8459037602976842, 
-3.3283383176978898]]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-3.8579119419277683, 0.79099443140152959], [-0.26271170655698128, 2.9827915539103138]], 
[[-2.0508884826136038, 2.5155025217007587], [-1.2077620267795197, -2.0487181376781249]], [[-5.0096680791353823, 
-3.4840485652499718], [0.47924653230349135, 1.0042669747128441]], [[3.1917749574499386, -6.2247217259034286], 
[-0.42214899864605337, 2.2070689788009927]], [[-1.1970027486543913, -3.0108622650591199], [-3.7927778263423182, 
-0.2795028777408799]], [[1.45678755796976, -3.4609755849823003], [-6.2993220268147621, 
1.874920051180812]]])+(1.-msk_ref)*numpy.array([[[-6.2575844513067924, -1.6086780779774945], [-2.6623842159360054, 
0.58311904453128971]], [[-4.4505609919926279, 0.11583001232173462], [-3.6074345361585438, -4.448390647057149]], 
[[-7.4093405885144064, -5.8837210746289959], [-1.9204259770755328, -1.39540553466618]], [[0.79210244807091446, 
-8.6243942352824519], [-2.8218215080250775, -0.19260353057803137]], [[-3.5966752580334154, -5.410534774438144], 
[-6.1924503357213423, -2.679175387119904]], [[-0.94288495140926409, -5.8606480943613244], [-8.698994536193787, 
-0.52475245819821215]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.85349916161)+(1.-msk_arg0)*(1.14846536956)
      arg1=Data(numpy.array([[[[-1.9879384597158936, 3.429838404923407, 1.6256026219556672, -3.3716639860896089], 
[2.0277494419367903, -3.884147517888612, -0.42146980062691597, 2.3309648284447206], [0.97209056175715247, 2.6384457057947506, 
1.5933291365205555, -2.5475149595564384]], [[-2.11081010127781, -3.1930513597169972, -4.9659283201428748, 2.0078161819951346], 
[-1.1983667363284289, 1.3036821885018268, 4.9453520612509276, -2.7575398928383166], [3.0166124390562725, -0.226573956469867, 
-4.8353072772455885, -3.414911875852249]]], [[[-1.4105744318455793, -3.8363636632800824, 2.0195087063815311, 
0.79728250246035692], [-2.2161562259194412, 1.8724874112208356, -0.88519460709482356, 4.4261529358409035], [4.7441975131384773, 
-2.2053577091429655, -0.36050377467677475, -2.1875643143045407]], [[-2.3819924795758118, 1.2756278622042547, 
-3.5776051420667354, 4.8964791694596137], [4.5957244037927918, -2.0816433307011959, -0.66126184361634444, -2.3087854304061226], 
[-4.6444834566040436, -2.289068006960544, -2.2511189324408387, 1.0364750934242126]]], [[[3.2526197837733744, 
4.9886698098748816, 2.5816941744651576, 2.6316027037808798], [-3.392754238279708, -2.868469377464816, -1.24421648755097, 
0.77981221238849674], [0.50094590328228961, 2.4214860437483496, 1.3244382576910283, -2.780266543784534]], 
[[-2.7658068147045789, -4.7769328018453709, -3.75322814068896, -2.8311865005150083], [-4.3628468362829445, -1.0734486701446944, 
3.7713392420808471, -1.7623482218473563], [-3.9834961804798494, -3.658558117436467, 2.0909851771917705, 
4.8734008138435509]]]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.86556070189779266, -6.2833375665370932, -4.4791017835693534, 0.51816482447592271], 
[-4.8812486035504765, 1.0306483562749258, -2.4320293609867703, -5.1844639900584069], [-3.8255897233708387, -5.4919448674084368, 
-4.4468282981342417, -0.30598420205724786]], [[-0.74268906033587623, 0.33955219810331094, 2.1124291585291886, 
-4.8613153436088208], [-1.6551324252852573, -4.157181350115513, -7.7988512228646139, -0.095959268775369644], 
[-5.8701116006699587, -2.6269252051438192, 1.9818081156319023, 0.56141271423856276]]], [[[-1.4429247297681069, 
0.98286450166639616, -4.8730078679952173, -3.6507816640740431], [-0.63734293569424505, -4.7259865728345218, 
-1.9683045545188627, -7.2796520974545897], [-7.5976966747521635, -0.64814145247072075, -2.4929953869369115, 
-0.66593484730914554]], [[-0.47150668203787438, -4.1291270238179409, 0.72410598045304919, -7.7499783310732999], 
[-7.449223565406478, -0.77185583091249033, -2.1922373179973418, -0.54471373120756361], [1.7909842949903574, 
-0.5644311546531422, -0.6023802291728475, -3.8899742550378988]]], [[[-6.1061189453870606, -7.8421689714885678, 
-5.4351933360788438, -5.485101865394566], [0.53925507666602179, 0.014970215851129787, -1.6092826740627162, -3.633311374002183], 
[-3.3544450648959758, -5.2749852053620359, -4.1779374193047145, -0.073232617829152247]], [[-0.087692346909107322, 
1.9234336402316847, 0.89972897907527383, -0.022312661098677911], [1.5093476746692582, -1.7800504914689919, -6.6248384036945334, 
-1.0911509397663299], [1.1299970188661632, 0.80505895582278075, -4.9444843388054567, 
-7.7268999754572372]]]])+(1.-msk_ref)*numpy.array([[[[3.1364038292743874, -2.2813730353649131, -0.47713725239717331, 
4.5201293556481028], [-0.87928407237829642, 5.0326128874471063, 1.5699351701854098, -1.1824994588862268], [0.1763748078013414, 
-1.4899803362362567, -0.44486376696206165, 3.6959803291149322]], [[3.2592754708363039, 4.341516729275491, 6.1143936897013687, 
-0.85935081243664069], [2.3468321058869228, -0.15521681894333295, -3.7968866916924338, 3.9060052623968105], 
[-1.8681470694977786, 1.3750393260283609, 5.9837726468040824, 4.5633772454107433]]], [[[2.5590398014040732, 4.9848290328385758, 
-0.87104333682303725, 0.35118286709813695], [3.364621595477935, -0.72402204166234174, 2.0336599766533174, -3.2776875662824096], 
[-3.5957321435799834, 3.3538230787014593, 1.5089691442352686, 3.3360296838630346]], [[3.5304578491343057, -0.12716249264576085, 
4.7260705116252293, -3.7480137999011198], [-3.4472590342342979, 3.2301087002596898, 1.8097272131748383, 3.4572507999646165], 
[5.7929488261625375, 3.4375333765190379, 3.3995843019993326, 0.11199027613428125]]], [[[-2.1041544142148805, 
-3.8402044403163877, -1.4332288049066637, -1.4831373342223859], [4.5412196078382019, 4.0169347470233099, 2.3926818571094639, 
0.36865315716999714], [0.64751946627620427, -1.2730206741898558, -0.17597288813253442, 3.9287319133430278]], 
[[3.9142721842630728, 5.9253981714038648, 4.9016935102474539, 3.9796518700735022], [5.5113122058414383, 2.2219140397031882, 
-2.6228738725223533, 2.9108135914058502], [5.1319615500383433, 4.8070234869949608, -0.94251980763327659, 
-3.7249354442850571]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.99245861047)+(1.-msk_arg0)*(2.45471909085)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.7547133102)+(1.-msk_arg1)*(2.02182027353)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-7.74717192067)+(1.-msk_ref)*(0.432898817325)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.51149607416)+(1.-msk_arg0)*(-4.19578936482)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([1.6282002830279261, 4.2262804326347574])+(1.-msk_arg1)*numpy.array([4.2760331889416197, 
0.76290651710170465])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.8832957911324826, -0.71478435847434874])+(1.-msk_ref)*numpy.array([-8.4718225537641132, 
-4.9586958819241982])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.96518287936)+(1.-msk_arg0)*(-1.60131775412)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[1.533249429665454, 3.820609556746728, 4.3402480399976522, 1.880589881988298, 
1.0855529608265035], [1.4307398192970204, 0.61499069990847399, 3.375092205609544, -4.0547659631848934, 0.0076807541626244813], 
[-4.8773514611406386, -0.81648771544033671, -3.9199163320245045, 2.8221873258217958, 3.9683026503617942], [2.7855826611250523, 
-0.74257765596743486, -4.7277531116368063, 3.8648495235535254, 
-2.5265889426456178]])+(1.-msk_arg1)*numpy.array([[4.2299281845302179, 3.0624698494321443, -1.965609290337361, 
-4.4667954356182094, -1.668858320914357], [1.6602534265101401, -4.6093088781877354, 2.1856435105070924, -1.0159546123764915, 
-3.7869081782609593], [-0.98840110229719613, 0.064148817031540517, -3.4134380160360078, 0.44440438318535591, 
-3.1352194320758198], [3.1474937228528503, -0.85540566328911538, 4.6663936606194873, 3.0184574177872872, 0.46552240954993174]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-6.4984323090246816, -8.7857924361059556, -9.3054309193568798, -6.8457727613475257, 
-6.0507358401857312], [-6.3959226986562481, -5.5801735792677016, -8.3402750849687717, -0.91041691617433429, 
-4.9728636335218521], [-0.087831418218589086, -4.1486951639188909, -1.0452665473347231, -7.7873702051810234, 
-8.9334855297210218], [-7.7507655404842799, -4.2226052233917928, -0.2374297677224213, -8.830032402912753, 
-2.4385939367136098]])+(1.-msk_ref)*numpy.array([[-5.8312459386521933, -4.6637876035541197, 0.36429153621538513, 
2.8654776814962335, 0.067540566792381185], [-3.2615711806321159, 3.0079911240657595, -3.7869612646290682, -0.58536314174548432, 
2.1855904241389834], [-0.61291665182477972, -1.6654665711535164, 1.812120261914032, -2.0457221373073318, 1.5339016779538439], 
[-4.7488114769748258, -0.74591209083286047, -6.2677114147414628, -4.6197751719092626, -2.0668401636719076]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.19303411741)+(1.-msk_arg0)*(-3.50311563113)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[2.3857016042533177, -3.5092629279301413], [4.6930472111961077, -1.9361467796319776]], 
[[-1.2771014863240948, 2.7914179088425373], [-4.3877608257566267, -2.39782217716076]], [[3.5462860979457549, 
4.7573080120449323], [4.3731992198184084, -0.10441714493497312]], [[-1.0621915814365512, 0.69358030564855255], 
[-0.43222952620169952, -2.3662106358997717]], [[-0.82446994614782643, 4.8956455063098936], [3.3152331213628905, 
3.8375061107111623]], [[1.4620775416324001, 3.0121072301770333], [2.7677247785947774, 
-3.2337373155062288]]])+(1.-msk_arg1)*numpy.array([[[-1.093195197218356, -1.0011980648304375], [2.7828247880235493, 
1.3040151044469024]], [[0.010676630955916089, -0.53367817690650465], [3.422869646195263, 2.71659044048286]], 
[[-0.60546726917637628, -4.2470008046040286], [0.85513865445730275, 0.32430811421774841]], [[-1.8738395033612343, 
-3.4249797146559824], [4.7685672658272775, -3.7441122023370421]], [[-0.76415626373217727, -2.0040424023302803], 
[0.63861730451298282, -2.9792912783229033]], [[-4.0622456167427234, 1.3287651937830507], [-0.1076071775210643, 
0.75563322963524637]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.1926674868389435, 4.7022970453445154], [-3.5000130937817335, 3.1291808970463517]], 
[[2.470135603738469, -1.5983837914281631], [5.5807949431710009, 3.5908562945751341]], [[-2.3532519805313807, 
-3.5642738946305581], [-3.1801651024040343, 1.2974512623493473]], [[2.2552256988509254, 0.4994538117658216], 
[1.6252636436160737, 3.5592447533141458]], [[2.0175040635622006, -3.7026113888955194], [-2.1221990039485163, 
-2.6444719932967882]], [[-0.26904342421802596, -1.8190731127626592], [-1.5746906611804032, 
4.4267714329206029]]])+(1.-msk_ref)*numpy.array([[[-2.4099204339096092, -2.5019175662975277], [-6.2859404191515145, 
-4.8071307355748676]], [[-3.5137922620838813, -2.9694374542214605], [-6.9259852773232282, -6.2197060716108252]], 
[[-2.8976483619515889, 0.74388517347606342], [-4.358254285585268, -3.8274237453457136]], [[-1.6292761277667309, 
-0.078135916471982814], [-8.2716828969552427, 0.24099657120907692]], [[-2.7389593673957879, -1.4990732287976849], 
[-4.141732935640948, -0.5238243528050619]], [[0.55912998561475824, -4.8318808249110159], [-3.3955084536069009, 
-4.2587488607632116]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.81961594194)+(1.-msk_arg0)*(4.79519153011)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[2.5532743323170486, -2.985464393768702, -1.2247613283970491, 3.4296850286735463], 
[-3.9084957015280142, -4.9986276450969811, -0.49273111748694376, -2.1443006639489317], [2.8476768891643029, 1.854078462160337, 
-2.8259525114312698, -4.1657030622045692]], [[2.5527080672069049, -4.948532418474139, 0.27977525619566812, 4.42577724222566], 
[4.8592412108183165, 0.50298043893769062, 4.2888038344468065, 1.5515444476783564], [-4.5008382987930116, 2.0433035785754257, 
-2.3916816300725383, 2.9027787783178214]]], [[[-4.3158512824227291, -4.0995197837785202, 0.40342618437239963, 
-4.3889934738157415], [-0.98691733629906597, -3.4595262718847239, -2.6141613748967307, 2.3972122301495595], 
[-3.4451088488755746, -4.2673118410115043, 1.8480969859770138, -2.2820170840793343]], [[2.701174256500539, -1.4065536281248403, 
1.3346294128507905, 2.2982790761274421], [-2.891245180677732, -4.3631500034060071, -2.9453876452216186, -1.337575646276179], 
[3.8316487633119873, -0.82612299857228955, -3.3813419495616182, -1.4882242009617572]]], [[[0.48470976312898451, 
2.2322000283407428, -0.29264252186122874, -1.1152238127385203], [-1.6757805770635334, -0.56401398191834229, 4.2006133489234685, 
1.4723363519631523], [-0.64337148957632806, 2.0303811445036306, -4.0598723382802131, -3.3646221494101445]], 
[[4.6333082679499515, -4.8080737752307137, -1.7506324142461072, 3.2165387606459106], [4.3646074871163822, 0.1050120689937506, 
-4.8611674296662821, 4.0765844806204115], [-0.31598756293286812, -2.0428330226546434, -1.0425381793097364, 
-2.8176823612127899]]]])+(1.-msk_arg1)*numpy.array([[[[2.1659183656927263, 1.8451950556552479, -1.8691056008200544, 
-3.7566767935726455], [-0.93467662553730069, 3.2182415583067794, 2.6487563143317141, -1.4444930047481694], [-3.863516057621573, 
-0.64654855692897417, 2.7877904585738076, -2.1636919670222667]], [[0.79062852833509645, 1.4781286539625063, 
-1.9875850386130702, -0.97055360237439015], [-2.5123864649943961, -1.5257073128148448, 0.71707385556423375, 
-1.900743120403857], [3.611364595329178, 4.5620539009057968, 3.3531149401091973, 4.1233112247224444]]], 
[[[-0.19158256530483975, -0.48246634062905258, -3.1413987510428654, 4.4078359877411728], [-0.51881895904608033, 
4.8714614940727721, 2.1430955221487213, 1.9119875044574002], [3.9152324559145786, 4.2830413040281652, -3.5558073672258583, 
-0.93004872907982961]], [[3.8375283296696097, 2.3849758515499744, -2.3759413717654256, -1.4056867551015739], 
[2.2932293186338253, -0.90074339743110343, -2.9730041413958572, 1.3140890127809044], [-0.38034348106393878, 3.0928585968860709, 
2.8274115505828066, 3.7514692150439508]]], [[[1.6934310208172629, 4.4758559501768751, -3.153595220962202, 3.0752414709316156], 
[1.8702160170336635, 1.505345216691591, -0.30100675561584822, 0.97057016084214887], [4.3832196357236182, 0.11895215872443377, 
2.9093627467844696, -4.8110542817620416]], [[4.8174465554359251, -0.33151954577439557, 2.6923606215912272, 3.9678402545693565], 
[-3.9922230669884007, -1.5098358661912514, 4.3643874666059972, 1.5538230688286268], [0.42208110233939244, -4.9305140408583279, 
2.5992527682149902, 4.0774423706695853]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-5.3728902742536189, 0.16584845183213126, -1.5948546135395216, -6.2493009706101166], 
[1.0888797595914435, 2.1790117031604104, -2.326884824449627, -0.67531527798763902], [-5.6672928311008732, -4.6736944040969082, 
0.0063365694946990736, 1.3460871202679985]], [[-5.3723240091434761, 2.1289164765375683, -3.0993911981322388, 
-7.2453931841622303], [-7.6788571527548868, -3.3225963808742613, -7.1084197763833767, -4.3711603896149267], 
[1.6812223568564408, -4.8629195205119959, -0.42793431186403241, -5.7223947202543926]]], [[[1.4962353404861584, 
1.2799038418419495, -3.2230421263089704, 1.5693775318791707], [-1.8326986056375048, 0.63991032994815322, -0.20545456703984, 
-5.2168281720861298], [0.62549290693900383, 1.4476958990749336, -4.6677129279135841, -0.53759885785723638]], 
[[-5.5207901984371102, -1.4130623138117304, -4.1542453547873617, -5.1178950180640133], [0.071629238741161227, 
1.5435340614694364, 0.12577170328504783, -1.4820402956603917], [-6.6512647052485576, -1.9934929433642812, 0.56172600762504743, 
-1.3313917409748135]]], [[[-3.3043257050655552, -5.051815970277314, -2.526973420075342, -1.7043921291980504], 
[-1.1438353648730373, -2.2556019600182284, -7.0202292908600388, -4.2919522938997225], [-2.1762444523602427, 
-4.8499970864402009, 1.2402563963436424, 0.54500620747357376]], [[-7.4529242098865218, 1.9884578332941429, -1.0689835276904636, 
-6.0361547025824809], [-7.1842234290529525, -2.9246280109303213, 2.0415514877297114, -6.8962004225569817], 
[-2.5036283790037026, -0.77678291928192733, -1.7770777626268344, 
-0.0019335807237808034]]]])+(1.-msk_ref)*numpy.array([[[[2.6292731644202041, 2.9499964744576825, 6.6642971309329848, 
8.551868323685575], [5.7298681556502311, 1.576949971806151, 2.1464352157812163, 6.2396845348610999], [8.6587075877345043, 
5.4417400870419046, 2.0074010715391228, 6.9588834971351972]], [[4.004563001777834, 3.3170628761504242, 6.7827765687260007, 
5.7657451324873206], [7.3075779951073265, 6.3208988429277753, 4.0781176745486967, 6.6959346505167874], [1.1838269347837524, 
0.23313762920713366, 1.4420765900037331, 0.67188030539048604]]], [[[4.9867740954177702, 5.277657870741983, 7.9365902811557962, 
0.38735554237175762], [5.3140104891590108, -0.076269963959841647, 2.6520960079642091, 2.8832040256555302], 
[0.87995907419835184, 0.51215022608476524, 8.3509988973387888, 5.72524025919276]], [[0.95766320044332076, 2.410215678562956, 
7.1711329018783561, 6.2008782852145039], [2.5019622114791051, 5.6959349275440339, 7.7681956715087876, 3.481102517332026], 
[5.1755350111768692, 1.7023329332268595, 1.9677799795301238, 1.0437223150689796]]], [[[3.1017605092956675, 0.31933557993605532, 
7.9487867510751329, 1.7199500591813148], [2.924975513079267, 3.2898463134213394, 5.0961982857287786, 3.8246213692707816], 
[0.41197189438931225, 4.6762393713884967, 1.8858287833284608, 9.6062458118749721]], [[-0.022255025322994726, 5.126711075887326, 
2.1028309085217032, 0.82735127554357391], [8.787414597101332, 6.3050273963041814, 0.43080406350693323, 3.2413684612843037], 
[4.373110427773538, 9.7257055709712574, 2.1959387618979402, 0.71774915944334516]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.2410608567771995, -4.2946457693389881])+(1.-msk_arg0)*numpy.array([-3.9859849629345678, 
-0.14161636055770721])
      arg1=-3.64200388185
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.8830647386229984, -0.65264188749318919])+(1.-msk_ref)*numpy.array([-0.3439810810887689, 
3.5003875212880917])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.4517771017402294, -1.3308496687293614])+(1.-msk_arg0)*numpy.array([4.0273883686804304, 
-4.8217145995464197])
      arg1=numpy.array(0.705033610995)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.74674349074503255, -2.0358832797245583])+(1.-msk_ref)*numpy.array([3.3223547576852335, 
-5.5267482105416166])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.61974323809240595, -2.1185373937578911])+(1.-msk_arg0)*numpy.array([-0.89740390311791796, 
0.64712704604309312])
      arg1=numpy.array([-1.4051199926636349, -0.75199398716893651])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.0248632307560408, -1.3665434065889546])+(1.-msk_ref)*numpy.array([0.5077160895457169, 
1.3991210332120296])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-2.2661822579435462, -2.6505799297422392])+(1.-msk_arg0)*numpy.array([1.5399403542252132, 
-1.5214375712457873])
      arg1=Data(3.94423278472,self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-6.2104150426588429, -6.5948127144575359])+(1.-msk_ref)*numpy.array([-2.4042924304900835, 
-5.465670355961084])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([4.6043571528030984, 2.003482903786149])+(1.-msk_arg0)*numpy.array([-0.48108140534421384, 
-1.4562128834438823])
      arg1=Data(numpy.array([-0.36895575042725959, 1.0598703930353377]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.973312903230358, 0.94361251075081132])+(1.-msk_ref)*numpy.array([-0.11212565491695425, 
-2.51608327647922])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.9286577934612232, -4.5521750534981251])+(1.-msk_arg0)*numpy.array([-4.1920014184447405, 
-1.5502046610899156])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.88247614299)+(1.-msk_arg1)*(2.84805890189)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.0461816504692774, -6.434651196490071])+(1.-msk_ref)*numpy.array([-7.0400603203378909, 
-4.3982635629830664])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.5467568069001025, -3.2274899350712314])+(1.-msk_arg0)*numpy.array([0.93283262764873331, 
-2.7099475433178255])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([1.0907694109259687, 2.9403700897105631])+(1.-msk_arg1)*numpy.array([0.029422968487883949, 
4.7765138715923232])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.4559873959741338, -6.1678600247817945])+(1.-msk_ref)*numpy.array([0.90340965916084937, 
-7.4864614149101492])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-3.3749133835200764, 2.6574459024133867, -4.8432690484335819, -1.0388893598885107, 
4.6149323247742693], [3.076641107663681, -2.8098580960526198, -1.2745605698774485, 0.22428662514845321, 2.6121859520282431], 
[4.4417299469034646, -3.553809095489048, 3.3178455301365002, -2.4086651593116457, 4.6270710443018324], [1.1998806388868903, 
-1.7661581138908344, 4.9846636071446397, 4.2852574645305825, 
-4.0235871913740802]])+(1.-msk_arg0)*numpy.array([[2.6941529108264834, -3.7398085356158361, 4.6268225828503748, 
-3.1838506144861212, -3.0691721749005607], [2.0734387470485673, -4.8863136484039469, -3.3114586342351373, 0.15569626527856073, 
4.1944117728466246], [4.1754366841223174, 0.22187357098980787, 1.8795536570835347, -2.1002280380897442, 4.3855691495409008], 
[4.841913910281086, 2.0621306107580466, -0.25869691069306455, 4.6207193572826739, -3.2448060091935185]])
      arg1=3.06314292325
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-6.4380563067721921, -0.40569702083872894, -7.9064119716856975, -4.1020322831406268, 
1.5517894015221536], [0.013498184411565362, -5.8730010193047359, -4.3377034931295642, -2.8388562981036625, 
-0.45095697122387257], [1.3785870236513489, -6.6169520187411637, 0.25470260688438451, -5.4718080825637614, 1.5639281210497167], 
[-1.8632622843652253, -4.8293010371429501, 1.9215206838925241, 1.2221145412784669, 
-7.0867301146261958]])+(1.-msk_ref)*numpy.array([[-0.36899001242563223, -6.8029514588679518, 1.5636796595982592, 
-6.2469935377382368, -6.1323150981526764], [-0.98970417620354834, -7.9494565716560626, -6.374601557487253, -2.9074466579735549, 
1.1312688495945089], [1.1122937608702017, -2.8412693522623078, -1.183589266168581, -5.1633709613418599, 1.3224262262887851], 
[1.7787709870289703, -1.0010123124940691, -3.3218398339451802, 1.5575764340305582, -6.3079489324456341]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[1.188012756982328, 2.7703562776249369, 2.5570618426080642, -3.2904165375218075, 
1.947212411235113], [-0.071494651588838387, -2.9266284635376039, 3.7512503395863934, -3.9722001304544885, -1.2087009142603113], 
[0.76734709496842513, 1.2753469464410525, 2.8602956531316668, 3.3052656177547419, -4.9064452920501465], [1.3033708579730554, 
0.32750013342182083, 2.7639482377539402, 0.83687610333136231, 
0.26775248279713448]])+(1.-msk_arg0)*numpy.array([[-4.792634706222346, 0.37058628954601147, -3.5062473542684747, 
-1.6986936352375759, -2.5484018055573001], [3.4878469670726346, 2.9401880232994317, -0.08578801131115199, 2.5293371015326915, 
0.50434220989305878], [-3.6444042809459685, -1.7355905994169096, 0.85365930952585067, -1.1951215307554808, 
-0.051197882933893268], [3.2700652184128476, -3.300143876117879, 4.6927307726690799, 4.8945170106956475, 2.5395638601420787]])
      arg1=numpy.array(4.52473375084)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-3.3367209938527314, -1.7543774732101225, -1.9676719082269951, -7.8151502883568664, 
-2.5775213395999463], [-4.5962284024238977, -7.4513622143726632, -0.77348341124866593, -8.4969338812895483, 
-5.7334346650953707], [-3.7573866558666342, -3.2493868043940068, -1.6644380977033926, -1.2194681330803174, 
-9.4311790428852049], [-3.2213628928620039, -4.1972336174132385, -1.7607855130811192, -3.687857647503697, 
-4.2569812680379249]])+(1.-msk_ref)*numpy.array([[-9.3173684570574054, -4.1541474612890479, -8.0309811051035336, 
-6.2234273860726352, -7.0731355563923595], [-1.0368867837624247, -1.5845457275356276, -4.6105217621462113, -1.9953966493023678, 
-4.0203915409420006], [-8.1691380317810278, -6.2603243502519685, -3.6710744413092087, -5.7198552815905401, 
-4.5759316337689526], [-1.2546685324222118, -7.8248776269529383, 0.16799702183402054, 0.36978325986058813, 
-1.9851698906929807]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.3899620557496135, 1.7953240624718738, 0.53153223455809862, -0.8264112573535618, 
-0.16101836024612659], [4.455659230642917, -3.9414332874246294, 3.2648666407996156, -4.8012703055463817, -4.0823751097858461], 
[0.49014112719274294, 1.1824084596349183, -0.0013745763208072859, 3.0563027388091655, 2.1537721032555037], 
[-2.3500746824426342, -3.7373595920283051, 3.2804525461429854, -3.3504426913270313, 
-0.93004051758304307]])+(1.-msk_arg0)*numpy.array([[4.4538404130942979, -2.3837147403235792, -0.82902218378244008, 
3.63024455430158, 2.7498413868572777], [3.2320601590516311, -3.6823456602003981, 0.49724254468041984, -4.3631262433514015, 
3.9510427241154673], [2.5241351404367007, -1.8113593704193107, 2.0804841858128311, 4.1495947040160548, 1.9599257807132267], 
[-4.7686041560289247, -3.2283284084112926, -3.4245675766933492, -1.4362507568373473, -3.8907532193201977]])
      arg1=numpy.array([[4.9700616049854744, 4.6556991574274882, 3.8251353392322578, -1.8184636437102677, 1.8875343126153821], 
[3.2446829490353206, -3.4303051727074862, -3.4535132977383833, 0.67208798562271355, -2.5262900075776438], [-3.4505881822514253, 
-2.9864392865551324, 2.6807674033979385, 4.2750861280412327, -2.4101982148154564], [2.4757901723259135, -4.7046600984368157, 
-0.66423438951137026, 3.9108571339280083, 3.9834615517867871]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.5800995492358609, -2.8603750949556144, -3.2936031046741592, 0.9920523863567059, 
-2.0485526728615087], [1.2109762816075964, -0.51112811471714314, 6.718379938537999, -5.4733582911690952, -1.5560851022082023], 
[3.9407293094441682, 4.1688477461900506, -2.6821419797187458, -1.2187833892320672, 4.5639703180709601], [-4.8258648547685477, 
0.96730050640851051, 3.9446869356543557, -7.2612998252550396, 
-4.9135020693698301]])+(1.-msk_ref)*numpy.array([[-0.51622119189117655, -7.0394138977510679, -4.6541575230146979, 
5.4487081980118477, 0.86230707424189568], [-0.012622789983689486, -0.25204048749291186, 3.9507558424188032, 
-5.0352142289741151, 6.4773327316931111], [5.974723322688126, 1.1750799161358216, -0.60028321758510739, -0.12549142402517788, 
4.370123995528683], [-7.2443943283548382, 1.476331690025523, -2.760333187181979, -5.347107890765356, -7.8742147711069848]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-3.6532443935784675, -1.179343263723549, -0.20671879830116868, 1.2600027730052163, 
1.5544275299874064], [4.2485252253709405, -0.4547813573750048, 1.3726598603770928, 0.83603858710271339, 0.50594842201175716], 
[-3.2872737346169831, 3.3399242573354364, 3.2316469534659227, -0.45021221045277748, 1.987934290996983], [-3.9740339444128847, 
3.6602056839859856, 2.6170931316504635, 3.470854140899327, 
-4.1427037568158465]])+(1.-msk_arg0)*numpy.array([[1.4938081366364484, -0.1520068744366494, -4.9785044366910842, 
-4.0738917695221826, -4.551385446453402], [-4.7002880839159165, 2.6339806888327457, 0.99764627227290159, -3.4092744977233025, 
1.6214454869592974], [2.2741726544216432, -3.6761706265088776, -2.2243011679138434, 4.9549611481827469, -4.459716683979396], 
[-0.2803610852127516, -1.8075102486981875, -1.2428411893867484, -1.4247489795172918, 0.96560758340281883]])
      arg1=Data(3.48574995539,self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-7.1389943489663183, -4.6650932191113998, -3.6924687536890195, -2.2257471823826345, 
-1.9313224254004444], [0.76277526998308964, -3.9405313127628556, -2.1130900950107581, -2.6497113682851374, 
-2.9798015333760937], [-6.773023690004834, -0.14582569805241441, -0.25410300192192814, -3.9359621658406283, 
-1.4978156643908678], [-7.4597838998007351, 0.17445572859813474, -0.86865682373738728, -0.01489581448852384, 
-7.6284537122036973]])+(1.-msk_ref)*numpy.array([[-1.9919418187514024, -3.6377568298245002, -8.464254392078935, 
-7.5596417249100334, -8.0371354018412529], [-8.1860380393037673, -0.8517692665551051, -2.4881036831149492, -6.8950244531111533, 
-1.8643044684285535], [-1.2115773009662076, -7.1619205818967284, -5.7100511233016942, 1.4692111927948961, -7.9454666393672468], 
[-3.7661110406006024, -5.2932602040860388, -4.7285911447745992, -4.9104989349051422, -2.520142371985032]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[3.4374247441806141, -2.4394879021624485, -2.6970526425508776, 1.685989427603829, 
-3.1788995581337822], [-4.0994780403062547, 0.50784683571141187, -4.8647116136369659, 4.8841320859979511, 3.2823091573150478], 
[-1.8894606738264983, 1.1656867885194897, 1.2561773626276196, -3.7423490126872139, -1.9036688626941469], [-0.83757986901817905, 
1.5273540039453142, -1.9141360566913757, -1.4021596320589569, 
3.6107865429020443]])+(1.-msk_arg0)*numpy.array([[-0.90578943420634594, -4.4539214261777325, -3.9230954902760216, 
-0.43389927157711039, 0.082663634886262827], [-2.5511558349774957, 0.4266688005322381, -1.8423229309950786, 
-1.3293557784597745, 4.9840394528954288], [4.3572027366939317, -1.0634305048054427, 0.41638622606777353, 0.32554769636754344, 
1.684693458803598], [1.3812915390835157, -1.5638683291733049, 1.4037748087019661, -1.830805766140815, 3.1474131096763571]])
      arg1=Data(numpy.array([[2.6149330843909935, 2.5807327569225826, -2.6350561955990037, -3.6811508610782182, 
-2.0678109153495914], [3.9489379824706177, -3.328235898143812, -0.8558707690822267, -2.7072989872821642, 1.0762273220620706], 
[0.3004856362457371, 4.7271000916259887, -3.4611632796404681, 1.7084432207530771, -0.062424965658093079], [-3.2043354390876644, 
2.3465689036261921, -4.8167709237678764, 0.39270347430182984, 3.5299667059909581]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.82249165978962058, -5.0202206590850311, -0.061996446951873896, 5.3671402886820472, 
-1.1110886427841908], [-8.0484160227768733, 3.8360827338552239, -4.0088408445547392, 7.5914310732801153, 2.2060818352529772], 
[-2.1899463100722354, -3.561413303106499, 4.7173406422680877, -5.450792233440291, -1.8412438970360538], [2.3667555700694853, 
-0.81921489968087791, 2.9026348670765008, -1.7948631063607867, 
0.080819836911086185]])+(1.-msk_ref)*numpy.array([[-3.5207225185973394, -7.0346541831003151, -1.2880392946770178, 
3.2472515895011078, 2.1504745502358542], [-6.5000938174481133, 3.7549046986760501, -0.9864521619128519, 1.3779432088223897, 
3.9078121308333582], [4.0567171004481946, -5.7905305964314309, 3.8775495057082416, -1.3828955243855336, 1.7471184244616911], 
[4.58562697817118, -3.910437232799497, 6.2205457324698425, -2.2235092404426449, -0.38255359631460095]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-1.1577008237262776, -3.2201348791618423, -1.8037638884040899, 3.8225705743826239, 
4.9928596967847607], [-0.10828190254409797, 0.40607847685814491, 2.4432367689797188, 1.7853236966873265, 3.072772372118834], 
[-3.0978633034380545, 0.55626374907247644, -3.1830960401135044, 3.4052978336689623, -3.1437458083167282], [-2.2510578517879898, 
4.3142539752046769, -0.55148681479776229, 3.2213924132140015, 
-0.73769855404420959]])+(1.-msk_arg0)*numpy.array([[0.57688622637114229, -2.4746800461653784, -3.4741012441148333, 
0.67901021369161896, 0.67960579551246436], [-0.027847304954182306, 1.1302634876818987, 1.8198829682756754, -1.7686784429930382, 
-4.0356228206299942], [1.3678265264758274, -2.3967381051110492, 3.2107825414271556, -4.8616906250820158, -3.9624032194085466], 
[-0.68811877891062068, 3.3223280513983582, 2.5021703384996306, -4.2546260372252931, -4.1220658932246979]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.941710593011)+(1.-msk_arg1)*(-1.30842522086)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.0994114167368583, -4.1618454721724234, -2.7454744814146705, 2.8808599813720432, 
4.05114910377418], [-1.0499924955546787, -0.53563211615243578, 1.5015261759691381, 0.8436131036767458, 2.1310617791082533], 
[-4.0395738964486352, -0.38544684393810424, -4.1248066331240851, 2.4635872406583816, -4.0854564013273089], 
[-3.1927684447985705, 3.3725433821940962, -1.493197407808343, 2.2796818202034208, 
-1.6794091470547903]])+(1.-msk_ref)*numpy.array([[1.8853114472280148, -1.1662548253085059, -2.1656760232579608, 
1.9874354345484915, 1.9880310163693369], [1.2805779159026902, 2.4386887085387712, 3.1283081891325479, -0.46025322213616571, 
-2.7271975997731217], [2.6762517473326999, -1.0883128842541767, 4.5192077622840277, -3.5532654042251433, -2.653977998551674], 
[0.62030644194625184, 4.6307532722552303, 3.8105955593565031, -2.9462008163684206, -2.8136406723678253]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.2616200659226768, 3.6836651166560088, 1.3998680171368587, -4.9585878121330609, 
-1.1952994761628313], [-4.176268020877921, 1.4776176763762061, 3.8774188838101153, -2.4008983745901835, -1.2609569532081077], 
[2.8793793124111531, 3.388080580863523, -1.8720048049288507, 3.305494728364426, -1.3497040196594314], [-4.8199826723393935, 
3.578841905490096, -0.45069615350587888, 1.9709589818773798, 
-1.3963633396507813]])+(1.-msk_arg0)*numpy.array([[-2.6313126509811138, 0.38058815566588056, -2.2702494661891603, 
-0.47829348560524121, 4.5081867607726949], [-2.8493169455641065, -4.2563250407916486, 2.1825136910837806, 1.1074441054304218, 
-4.2464324897516903], [-3.6221449807478088, -0.52039287749746777, 0.34997513366946809, -0.59633211158088972, 
3.8196404460216318], [3.006942803351464, 3.8644735453985763, -1.7604332326480732, -3.208779560093201, -0.32752172279377412]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[2.2347412740725634, 0.20463017127506156, 4.085710574112122, 3.3258515583623058, 
1.8626223879919674], [3.6967619268095557, 0.16545779247669223, -2.1655386900312257, -3.9911806946622685, -4.6130259840988508], 
[-2.9779290264802372, -4.2694813169588208, -3.8419761197907167, -3.8273459879517446, -4.2161790286312186], [3.3949980186303925, 
2.6809961171063188, 2.9838248450799298, -3.7524619382372251, 
1.6144773005272093]])+(1.-msk_arg1)*numpy.array([[-0.62889998794144031, 0.19715285159436124, 4.1781958854848096, 
-0.10922925063658528, 4.5222295245454021], [3.7319700520568393, -3.9586554800415961, 2.7653335729425708, 2.3388878242176503, 
4.7197980665620278], [1.7960629294638739, 0.18938829682479597, -2.5106577564286114, -1.0775750861273545, 0.30327621747664324], 
[4.1561510990219332, -1.2326984046427136, 0.96449672781292595, -3.4531839237133077, 2.2671838655671905]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.026878791850113437, 3.4790349453809473, -2.6858425569752633, -8.2844393704953667, 
-3.0579218641547987], [-7.8730299476874768, 1.3121598838995139, 6.042957573841341, 1.590282320072085, 3.352069030890743], 
[5.8573083388913902, 7.6575618978223439, 1.9699713148618661, 7.1328407163161707, 2.8664750089717872], [-8.2149806909697851, 
0.89784578838377715, -3.4345209985858087, 5.7234209201146049, 
-3.0108406401779906]])+(1.-msk_ref)*numpy.array([[-2.0024126630396735, 0.18343530407151931, -6.4484453516739695, 
-0.36906423496865592, -0.01404276377270719], [-6.5812869976209463, -0.29766956075005258, -0.58281988185879019, 
-1.2314437187872285, -8.9662305563137181], [-5.4182079102116827, -0.70978117432226373, 2.8606328900980795, 0.48124297454646481, 
3.5163642285449885], [-1.1492082956704692, 5.0971719500412895, -2.7249299604609991, 0.24440436362010676, -2.5947055883609647]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.2413072869553563, 2.449255866561745], [0.6606257769246513, -2.1697880316496487]], 
[[3.2721678663465035, -1.0871480907431086], [4.2665577255269618, 2.3941327118265976]], [[1.8726358828641452, 
-0.73281457677845641], [1.2246648290863096, -4.2342416446592344]], [[-3.5591281455755643, -2.1897551984011834], 
[4.8570042629599346, 4.8524166385913219]], [[1.4290174139696097, -3.731455825063934], [-2.698185837208309, 
-3.3552323418816368]], [[1.62035207228275, 0.28323113732563687], [0.95230909141283071, 
-3.0242158797460803]]])+(1.-msk_arg0)*numpy.array([[[3.2439127035563846, -3.3928500484155499], [-1.7033470654091722, 
3.1931234483224884]], [[-2.2266875497831338, -1.4922570484684594], [-2.7793075545885939, -1.8804084547467537]], 
[[2.2946510197660857, 1.2167441851120433], [-1.2910185767087889, -0.33422285548338682]], [[0.10654554024145835, 
-2.5155398479131916], [-0.084824927625967383, 4.0800446768278036]], [[4.4599036861468235, 2.0013261534931628], 
[2.8473459803164802, 4.1231818477895317]], [[-0.71785864138721234, -3.1760689578428059], [-4.0717224921303874, 
3.4430233881547299]]])
      arg1=-4.78074731201
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[5.0220545989623906, 7.2300031785687793], [5.4413730889316856, 2.6109592803573856]], 
[[8.0529151783535369, 3.6935992212639257], [9.0473050375339952, 7.1748800238336319]], [[6.6533831948711795, 
4.0479327352285779], [6.0054121410933439, 0.54650566734779993]], [[1.2216191664314699, 2.5909921136058509], 
[9.6377515749669698, 9.6331639505983553]], [[6.209764725976644, 1.0492914869431003], [2.0825614747987253, 1.4255149701253975]], 
[[6.4010993842897843, 5.0639784493326712], [5.733056403419865, 
1.756531432260954]]])+(1.-msk_ref)*numpy.array([[[8.0246600155634198, 1.3878972635914844], [3.077400246597862, 
7.9738707603295227]], [[2.5540597622239005, 3.2884902635385749], [2.0014397574184404, 2.9003388572602806]], 
[[7.0753983317731199, 5.9974914971190776], [3.4897287352982453, 4.4465244565236475]], [[4.8872928522484926, 
2.2652074640938427], [4.6959223843810669, 8.860791988834837]], [[9.2406509981538569, 6.7820734655001971], [7.6280932923235145, 
8.9039291597965651]], [[4.062888670619822, 1.6046783541642284], [0.70902481987664689, 8.223770700161765]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-2.0690730602155627, 1.2960654406678396], [4.5743223350750721, -2.8357620771887735]], 
[[4.7805663519092931, 4.094547348525662], [1.6103136708396644, 2.7867432158910148]], [[-4.6958306077303167, 
2.2064348095906077], [-1.1405665570605672, -1.9028814552894824]], [[4.5683894624972012, 3.6930544250192519], 
[-4.4089825254179402, 3.4957033416287953]], [[4.4782108815130393, 2.2238553581229548], [-3.0583213262365119, 
-4.2714201258705975]], [[-2.7863799523312949, 4.2645853896106942], [-4.0029327528537939, 
2.4171339161309788]]])+(1.-msk_arg0)*numpy.array([[[-2.129534005538555, -2.3639442454300674], [-4.149210300288213, 
-2.0183500583777958]], [[3.5536591146654271, 4.9442564809750618], [0.40862930100882089, -2.1431583695434342]], 
[[-3.0332312511594575, 2.2265821838654531], [0.75141664128530739, 3.9818033314438086]], [[-2.5217179211451035, 
-4.7111062915536746], [-1.7803199924926103, -2.7055122891939098]], [[-4.9839065146275496, 3.9975298273373454], 
[0.97975327546517654, 4.34312574041898]], [[-2.0928023515871583, -1.3185390335010396], [-2.5720259477365639, 
2.0961052961732207]]])
      arg1=numpy.array(-0.125205824353)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.9438672358621858, 1.4212712650212165], [4.699528159428449, -2.7105562528353966]], 
[[4.90577217626267, 4.2197531728790389], [1.7355194951930413, 2.9119490402443917]], [[-4.5706247833769398, 2.3316406339439846], 
[-1.0153607327071903, -1.7776756309361055]], [[4.6935952868505781, 3.8182602493726288], [-4.2837767010645633, 
3.6209091659821722]], [[4.6034167058664162, 2.3490611824763317], [-2.933115501883135, -4.1462143015172206]], 
[[-2.661174127977918, 4.3897912139640711], [-3.877726928500417, 
2.5423397404843557]]])+(1.-msk_ref)*numpy.array([[[-2.0043281811851781, -2.2387384210766905], [-4.0240044759348361, 
-1.8931442340244189]], [[3.678864939018804, 5.0694623053284387], [0.53383512536219779, -2.0179525451900573]], 
[[-2.9080254268060806, 2.3517880082188301], [0.87662246563868429, 4.1070091557971855]], [[-2.3965120967917266, 
-4.5859004672002976], [-1.6551141681392334, -2.5803064648405329]], [[-4.8587006902741727, 4.1227356516907223], 
[1.1049590998185534, 4.4683315647723569]], [[-1.9675965272337814, -1.1933332091476627], [-2.446820123383187, 
2.2213111205265976]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[2.7216899699833865, -4.6990937787361], [-3.8136004095959453, -2.2488131383236709]], 
[[0.3981817013182738, -4.2431195468530643], [1.006954563089856, -4.8330278226567582]], [[-2.6280218516586942, 
2.5551983845635062], [4.9160091160648474, -4.6084191027006884]], [[-1.3609506137934915, -2.072851188065953], 
[-3.6704228280804219, 2.5092298138108688]], [[-2.4937209642436873, 3.8581321227283354], [-1.2403794863234472, 
0.094844820071759095]], [[0.5280278868537005, 1.9392594520663504], [3.0146535779663104, 
1.6648027305170245]]])+(1.-msk_arg0)*numpy.array([[[-3.9808910520294072, -4.3013675442426766], [1.612577597694437, 
-0.28314712364172845]], [[-2.3549727136447194, 2.0640539370494739], [0.2702899448263949, -2.9955722137627641]], 
[[-2.6873971778407513, -4.0028411336881584], [-0.66755743205844631, 3.4696902885940588]], [[0.89713666478704734, 
3.878748940714452], [3.3906737916064138, 4.4604088894599414]], [[0.34192950439670167, 1.0177974315203731], 
[-2.6989085991051374, 2.3757118980449565]], [[3.0979143153420381, -3.311146358228406], [-3.9470426984650739, 
-1.0240048661447618]]])
      arg1=numpy.array([[[2.1530318813246279, 3.817956825033642], [-2.8654897817320943, -1.1949642883157519]], 
[[2.6327545472172353, -4.9179530645883851], [2.7409970253928879, 3.3125394548747344]], [[-1.3516245344229372, 
1.5532148113516637], [4.5035505846873338, 0.087971982443754371]], [[2.2281715600105994, -0.27619411332240329], 
[-2.4506523274447867, -2.1586508385000314]], [[2.5657326133502991, -1.1951384203157609], [-3.9217646349755686, 
2.0295145643842307]], [[-1.9470468799884566, -3.3522607411077709], [-4.2878101385056198, 2.6652223283440799]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.56865808865875866, -8.5170506037697429], [-0.94811062786385092, -1.0538488500079191]], 
[[-2.2345728458989615, 0.67483351773532085], [-1.7340424623030319, -8.1455672775314927]], [[-1.276397317235757, 
1.0019835732118425], [0.41245853137751354, -4.6963910851444428]], [[-3.5891221738040908, -1.7966570747435497], 
[-1.2197705006356352, 4.6678806523109007]], [[-5.0594535775939864, 5.0532705430440963], [2.6813851486521214, 
-1.9346697443124716]], [[2.4750747668421571, 5.2915201931741214], [7.3024637164719302, 
-1.0004195978270554]]])+(1.-msk_ref)*numpy.array([[[-6.133922933354035, -8.1193243692763186], [4.4780673794265313, 
0.91181716467402341]], [[-4.9877272608619547, 6.982007001637859], [-2.470707080566493, -6.3081116686374985]], 
[[-1.3357726434178141, -5.5560559450398221], [-5.1711080167457801, 3.3817183061503044]], [[-1.331034895223552, 
4.1549430540368553], [5.8413261190512005, 6.6190597279599732]], [[-2.2238031089535975, 2.2129358518361339], 
[1.2228560358704312, 0.3461973336607258]], [[5.0449611953304947, 0.041114382879364975], [0.34076744004054582, 
-3.6892271944888417]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-1.3613644554184177, 2.6614299720415726], [-0.76218731834160725, 2.3409479644656708]], 
[[-2.6315182114023719, -2.2973672378015952], [4.2828736852866029, 1.8993140552545915]], [[4.6709123081754385, 
-3.545241222040465], [0.10505428781007442, 2.672725157274515]], [[-4.5534519364359376, 3.5265289587406112], 
[-4.9377901250772336, -0.26159639883028607]], [[-0.088929427733107858, -2.8296978511989379], [-0.74126311198423966, 
-4.5693583100845894]], [[-3.2365182581030649, 3.4623062662015744], [3.0416552488037052, 
1.8797671711986252]]])+(1.-msk_arg0)*numpy.array([[[-0.52456634696367832, 1.888037107198965], [0.18663501759310197, 
-4.3749479249157677]], [[4.6702794739454774, 4.9797638347784243], [-2.7625419545800067, 0.37663262831736866]], 
[[-3.4814085996783395, -4.5074380550293549], [-4.6239614601765746, 2.5183075331766389]], [[0.32103803853632584, 
-1.3289297859949913], [0.60822708248310065, -3.8513954049033448]], [[1.9141819443016352, -1.7155479396149365], 
[-2.9110096829350751, -4.1674642786770981]], [[1.4020006617880654, -4.6020389565596247], [4.2242898697195201, 
-0.95007019052097608]]])
      arg1=Data(-0.233628509893,self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.127735945525532, 2.8950584819344582], [-0.52855880844872161, 2.5745764743585564]], 
[[-2.3978897015094862, -2.0637387279087096], [4.5165021951794886, 2.1329425651474772]], [[4.9045408180683241, 
-3.3116127121475794], [0.33868279770296006, 2.9063536671674006]], [[-4.319823426543052, 3.7601574686334969], 
[-4.7041616151843479, -0.027967888937400431]], [[0.14469908215977778, -2.5960693413060523], [-0.50763460209135403, 
-4.3357298001917037]], [[-3.0028897482101793, 3.6959347760944601], [3.2752837586965908, 
2.1133956810915109]]])+(1.-msk_ref)*numpy.array([[[-0.29093783707079268, 2.1216656170918506], [0.42026352748598761, 
-4.141319415022882]], [[4.903907983838363, 5.21339234467131], [-2.5289134446871211, 0.61026113821025429]], 
[[-3.2477800897854538, -4.2738095451364693], [-4.390332950283689, 2.7519360430695246]], [[0.55466654842921148, 
-1.0953012761021057], [0.84185559237598628, -3.6177668950104591]], [[2.1478104541945209, -1.4819194297220508], 
[-2.6773811730421895, -3.9338357687842125]], [[1.635629171680951, -4.3684104466667391], [4.4579183796124058, 
-0.71644168062809044]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[4.5917153081560809, -3.1278973200325879], [-1.2430699191485219, -1.9509128069409987]], 
[[-3.2701312342573861, 4.5473531404890792], [3.636767555274778, -1.2490246336494626]], [[2.2895926734949086, 
4.1831038532651661], [-4.8181101257856014, 3.6335184168748338]], [[0.27651904432325214, -3.9663534144037382], 
[2.3302360210464226, -1.7524041423621748]], [[-3.6708025750866238, -4.0480386732488274], [-2.8698838144437042, 
-3.4599185324154447]], [[-1.2448316374750679, -4.5111309920668212], [-4.1511192193964632, 
-4.3533295651664847]]])+(1.-msk_arg0)*numpy.array([[[1.5433959047962702, 0.14937503892347692], [-4.7967652594031698, 
0.3507602662234941]], [[-0.55857173574205987, -4.9534534373425378], [3.5674879188016497, -0.27247721547511183]], 
[[-0.44837884069669443, 1.5664350718948929], [4.5305320991680151, 2.1973784418485405]], [[4.9752387533916576, 
-1.2735793192512288], [-1.3988123778849779, -4.1740256692104598]], [[4.5426525043828132, 1.8208203114463473], 
[-4.2443936977023489, 0.09130063985344794]], [[-3.0613971071272839, 4.4620962971217715], [-2.8537502558092576, 
-2.7716000701712895]]])
      arg1=Data(numpy.array([[[-0.3988190563245233, -4.4571202103884433], [1.4318826561562084, -2.773783777588247]], 
[[0.29698411313541939, -2.0439174369743327], [4.8737710135039496, 4.1586672929822015]], [[4.8803313435939071, 
1.4838900634279302], [2.0765412697230055, -1.3986494435143513]], [[-3.733562149842812, 1.4798225547559927], 
[-4.1280232925517009, -0.56441846327487255]], [[-1.7534446675156303, -2.4329125775372908], [-1.5750435664774476, 
3.8693519327493302]], [[2.3516326146154292, -4.1516407175357752], [2.6628867044134061, 
-3.8033882660743834]]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[4.9905343644806042, 1.3292228903558554], [-2.6749525753047303, 0.82287097064724835]], 
[[-3.5671153473928054, 6.5912705774634119], [-1.2370034582291716, -5.4076919266316636]], [[-2.5907386700989985, 
2.6992137898372359], [-6.8946513955086068, 5.0321678603891851]], [[4.0100811941660641, -5.4461759691597305], 
[6.4582593135981234, -1.1879856790873022]], [[-1.9173579075709934, -1.6151260957115365], [-1.2948402479662566, 
-7.3292704651647753]], [[-3.596464252090497, -0.35949027453104598], [-6.8140059238098694, 
-0.54994129909210132]]])+(1.-msk_ref)*numpy.array([[[1.9422149611207935, 4.6064952493119202], [-6.2286479155593781, 
3.1245440438117411]], [[-0.85555584887747926, -2.9095360003682051], [-1.3062830947022999, -4.4311445084573133]], 
[[-5.3287101842906015, 0.082545008466962777], [2.4539908294450097, 3.5960278853628918]], [[8.7088009032344686, 
-2.7534018740072215], [2.7292109146667229, -3.6096072059355873]], [[6.2960971718984435, 4.2537328889836381], 
[-2.6693501312249013, -3.7780512928958823]], [[-5.4130297217427135, 8.6137370146575467], [-5.5166369602226641, 
1.0317881959030939]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-2.0909675922137594, 3.372610306620766], [-1.1389035272738788, 1.22764723509004]], 
[[0.417806801069732, -1.3121169446251493], [-2.2579221734922559, -0.81412245944588069]], [[1.2360045329073817, 
1.1823793661084601], [-0.81378467398821908, 3.6350154477857437]], [[-4.6923472235495165, 2.8337843571566204], 
[0.24609255134181218, -0.42280265808776996]], [[0.96990554802589735, -2.8205803929389472], [-0.6749522346248682, 
-3.4514730403824148]], [[4.7022709586182572, 4.3214364007807298], [-1.5984970804204868, 
-0.16682083745886889]]])+(1.-msk_arg0)*numpy.array([[[-3.4649537778950479, 0.63372192516748882], [-4.9405502850286069, 
-4.4506400832459514]], [[2.7484008289460329, 2.8872006278151439], [3.8386686817514644, -4.5690361668675896]], 
[[-0.21303447857576341, 0.14220723073494845], [0.38200519486715567, -4.2214567806704988]], [[3.5498442408513178, 
3.1063584695347579], [-0.88586534205485812, 4.1655936274608543]], [[-1.7638710636590926, -4.4430818788085116], 
[0.88014839498571007, -2.0036558234781521]], [[-0.9618435232573086, 1.0424565700455783], [2.9789025391995247, 
-4.6985723492686091]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-3.62651434878)+(1.-msk_arg1)*(-0.0336198131999)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.5355467565675549, 6.9991246554020803], [2.4876108215074355, 4.8541615838713543]], 
[[4.0443211498510463, 2.314397404156165], [1.3685921752890584, 2.8123918893354336]], [[4.8625188816886959, 4.8088937148897744], 
[2.8127296747930952, 7.261529796567058]], [[-1.0658328747682022, 6.4602987059379346], [3.8726069001231265, 
3.2037116906935443]], [[4.5964198968072116, 0.80593395584236704], [2.9515621141564461, 0.17504130839889953]], 
[[8.3287853073995706, 7.9479507495620441], [2.0280172683608275, 
3.4596935113224454]]])+(1.-msk_ref)*numpy.array([[[-3.4313339646951033, 0.66734173836743338], [-4.9069304718286624, 
-4.4170202700460068]], [[2.7820206421459774, 2.9208204410150884], [3.8722884949514089, -4.5354163536676451]], 
[[-0.17941466537581885, 0.17582704393489301], [0.41562500806710023, -4.1878369674705542]], [[3.5834640540512623, 
3.1399782827347025], [-0.85224552885491356, 4.1992134406607988]], [[-1.7302512504591481, -4.409462065608567], 
[0.91376820818565463, -1.9700360102782075]], [[-0.92822371005736404, 1.0760763832455229], [3.0125223523994693, 
-4.6649525360686646]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-1.2285478760013699, -1.0889557421060916], [-4.0602492073092789, -0.43783646087775452]], 
[[0.21370498131530358, 3.8006441303415279], [1.4107669666154967, -4.5655252353832623]], [[-4.8019623929283286, 
-3.4287648641294322], [1.6488419659497842, -2.1890416082328823]], [[1.2734833509816283, -4.0932206088291174], 
[1.8893608340059753, 3.1322449017124363]], [[0.74571306260915105, -1.9626681599145099], [-0.89278402304071136, 
2.7524904356853757]], [[1.1227135941448951, -3.5060612342593069], [0.34646320989398927, 
2.2984450370907252]]])+(1.-msk_arg0)*numpy.array([[[-0.034433853303207584, -2.7226348853657067], [-1.4761983718484428, 
3.2170380581743139]], [[-2.5306193390194531, -2.4442351640220128], [-2.9000985447734271, -0.29368456927640452]], 
[[1.7756984048721485, -4.8391105150251228], [-4.0119834577497668, 1.2670132311648494]], [[3.6957787754186704, 
4.4497271358531414], [1.4624220259822689, 1.5817778131731073]], [[3.289945976779924, -0.8150007322677677], 
[-0.74082273274408283, -2.1112444325316702]], [[-0.60599987207344608, 0.15801551177233808], [4.5742988681767009, 
-4.9989635559138144]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-4.4830579941897692, 0.055325710987234267], [-3.1188351172971163, 0.057066017547253267]], 
[[-0.37263116330640145, 0.40743577638798278], [0.56134929911241116, -4.340199918448465]], [[-4.3330479977361627, 
4.0362499769056317], [1.3151769386126277, 1.2798191215787833]], [[0.011774111788679598, 2.1626566007204033], 
[-3.8571598927404094, -4.6757404292691049]], [[3.7623562032069966, -1.8846097753835043], [4.2179868034277117, 
4.6070790988666381]], [[1.1616148830660968, 3.2732281165371404], [-3.7487077529706605, 
2.7183274366441514]]])+(1.-msk_arg1)*numpy.array([[[0.29353812963369563, 3.4927398594817589], [-1.4652161763446991, 
4.3722355377748556]], [[-3.4717885039310978, -4.746086530214038], [1.4840269322953272, -4.8668268321381642]], 
[[2.0574913251778764, -4.2255245824157797], [2.877468903403388, -3.6731210495792013]], [[0.10034303525085253, 
4.0971852042863635], [-1.6092357023764481, -3.3542285573344133]], [[-3.7017478644851587, -2.7629505478222018], 
[-3.9664483774830139, -0.37499262054498939]], [[2.5453965731691657, -4.4267994678562701], [4.8518811856124984, 
0.94889171672399097]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[3.2545101181883993, -1.1442814530933259], [-0.94141409001216259, -0.49490247842500779]], 
[[0.58633614462170502, 3.3932083539535451], [0.84941766750308556, -0.22532531693479729]], [[-0.46891439519216593, 
-7.4650148410350639], [0.33366502733715642, -3.4688607298116656]], [[1.2617092391929488, -6.2558772095495208], 
[5.7465207267463843, 7.8079853309815412]], [[-3.0166431405978456, -0.078058384531005665], [-5.1107708264684231, 
-1.8545886631812625]], [[-0.038901288921201704, -6.7792893507964473], [4.0951709628646498, 
-0.41988239955342621]]])+(1.-msk_ref)*numpy.array([[[-0.32797198293690322, -6.2153747448474661], [-0.010982195503743686, 
-1.1551974796005418]], [[0.94116916491164471, 2.3018513661920252], [-4.3841254770687543, 4.5731422628617597]], 
[[-0.28179292030572789, -0.61358593260934313], [-6.8894523611531548, 4.9401342807440507]], [[3.5954357401678179, 
0.35254193156677793], [3.071657728358717, 4.9360063705075206]], [[6.9916938412650822, 1.9479498155544341], [3.2256256447389311, 
-1.7362518119866808]], [[-3.1513964452426118, 4.5848149796286082], [-0.27758231743579742, -5.9478552726378053]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.8772711279337102, -0.84005405430613411, 4.8092847770698004, 3.4111751088826416], 
[-3.8120114979270192, -4.980861959095054, 2.0747464419214969, 1.6930860256284586], [3.2980983427804169, -1.9436404862273493, 
-2.555427955280436, 1.0767777474020939]], [[2.433058703022958, -4.7685311051586048, 4.3390589384026299, -4.7045509949307949], 
[3.4802688493430374, -0.61008984155819057, -3.6640479241909163, -1.0089552345256281], [2.8801503159418749, 
-0.033400384239956615, -0.92612788694113668, 4.0024341917621076]]], [[[2.5439128926368868, 2.3352151076648973, 
-1.1677888188981287, 3.5527282020748636], [-3.8024744263683763, 2.1531840095537564, 0.73917500651035883, 2.8798640188974627], 
[4.5996233624850902, 3.0555387181865719, -3.5804439855182055, 1.5755515747709978]], [[-1.034205812871086, -1.567941970832718, 
2.8517531993553122, 3.513982982828697], [-0.15118256052258339, 4.5424424654813365, -0.73941883861525604, 0.26580866292286753], 
[2.4152732831357486, -0.0017438618125522609, 1.6701893518142565, 2.2779057853355393]]], [[[3.0261572111810491, 
0.077006194576844322, 1.448140307046744, -1.6445378967168178], [3.1736269974053002, -0.88871539042199998, 1.6688542594699731, 
-0.12218641478048742], [-2.065637702694636, 0.25429530544865742, 2.047259468102645, 2.9793221154653171]], [[1.1973262214495639, 
-1.4497223240335364, -4.0702192623014373, 3.7587450663104089], [0.75948704931145095, -1.5464775283168541, -2.8396991865937702, 
0.68009378605926685], [-3.4312963436472987, 1.4329727619208743, 4.7410276012918189, 
-0.30265454747468468]]]])+(1.-msk_arg0)*numpy.array([[[[-3.0432134330149676, 1.5882573598216236, 2.8747428543105276, 
0.71868472981826326], [1.0865123755325543, 3.3651443637884526, -1.2547163109688753, 1.7952552392679806], [1.959811210192429, 
3.8276637251462624, -3.8759668228134836, -4.461571426593439]], [[4.2098367431321435, 3.7025895819323011, 2.8554245363209336, 
0.71495031469744141], [-4.845745187411743, 3.0693744672211949, 0.75203664457745489, 1.0622211556137975], [-3.1805373830897299, 
-1.4328298404368445, 4.6988211131298065, 4.2081121188395656]]], [[[-0.93242826499923748, 2.2004318825960425, 
-0.57044760157241559, -1.2228839613766249], [0.54976906657994995, 3.9747306141740104, -4.5433231598398596, 
-0.1945915248535961], [2.2709087860790174, 4.4288826629519082, -3.5519559817327839, 0.45318536608693005]], 
[[-4.8730618109395962, 2.1076746533529427, 2.8581419748010752, 4.2039407675603684], [-0.066710635749545055, 
-3.2889575749248667, 0.55155906672074195, -1.1761579642055962], [2.4379561351466181, 0.66936696573978605, -0.54558972611023915, 
3.5423306267370798]]], [[[2.9288746173585629, -2.2454638385135661, 2.4103454804678055, -4.3927780069472053], 
[3.5935586405227316, 3.775613613613988, 0.16295946187750232, 2.0438288584202162], [4.3898718259218672, 1.9849230829044107, 
-3.1697502986831116, -1.0842300256719826]], [[-0.28318299654336965, 1.6616795070086079, -0.29904236658320116, 
1.3414657374599788], [-1.1866047352712927, 1.8048172454080742, 3.8841133403870405, 4.2502239873892975], [-3.0983347301939643, 
-3.2876989745496479, 0.67737524080566747, 1.3854702635975293]]]])
      arg1=2.06231633893
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-6.939587466860937, -2.9023703932333609, 2.7469684381425736, 1.3488587699554149], 
[-5.8743278368542455, -7.0431782980222808, 0.012430102994270165, -0.36923031329876821], [1.2357820038531901, 
-4.0059568251545761, -4.6177442942076627, -0.9855385915251329]], [[0.37074236409573125, -6.8308474440858316, 
2.2767425994754031, -6.7668673338580216], [1.4179525104158106, -2.6724061804854173, -5.7263642631181426, -3.0712715734528548], 
[0.81783397701464811, -2.0957167231671834, -2.9884442258683634, 1.9401178528348808]]], [[[0.48159655370966004, 
0.27289876873767049, -3.2301051578253555, 1.4904118631476369], [-5.8647907652956031, 0.090867670626529673, -1.3231413324168679, 
0.81754767997023592], [2.5373070235578634, 0.99322237925934509, -5.6427603244454323, -0.48676476415622894]], 
[[-3.0965221517983128, -3.6302583097599448, 0.78943686042808547, 1.4516666439014703], [-2.2134988994498102, 2.4801261265541097, 
-2.8017351775424828, -1.7965076760043592], [0.35295694420852186, -2.064060200739779, -0.39212698711297023, 
0.21558944640831257]]], [[[0.96384087225382231, -1.9853101443503824, -0.61417603188048275, -3.7068542356440446], 
[1.1113106584780734, -2.9510317293492268, -0.39346207945725364, -2.1845027537077142], [-4.1279540416218623, 
-1.8080210334785694, -0.015056870824581736, 0.91700577653809034]], [[-0.86499011747766286, -3.5120386629607632, 
-6.132535601228664, 1.6964287273831822], [-1.3028292896157758, -3.6087938672440809, -4.902015525520997, -1.3822225528679599], 
[-5.4936126825745255, -0.62934357700635246, 2.6787112623645921, 
-2.3649708864019114]]]])+(1.-msk_ref)*numpy.array([[[[-5.1055297719421944, -0.47405897910560313, 0.8124265153833008, 
-1.3436316091089635], [-0.97580396339467246, 1.3028280248612258, -3.317032649896102, -0.26706109965924618], 
[-0.10250512873479778, 1.7653473862190356, -5.9382831617407099, -6.5238877655206657]], [[2.1475204042049167, 
1.6402732430050744, 0.79310819739370686, -1.3473660242297854], [-6.9080615263389697, 1.0070581282939681, -1.3102796943497719, 
-1.0000951833134293], [-5.2428537220169567, -3.4951461793640712, 2.6365047742025798, 2.1457957799123388]]], 
[[[-2.9947446039264642, 0.13811554366881573, -2.6327639404996424, -3.2852003003038517], [-1.5125472723472768, 
1.9124142752467836, -6.6056394987670863, -2.2569078637808229], [0.20859244715179059, 2.3665663240246815, -5.6142723206600103, 
-1.6091309728402967]], [[-6.935378149866823, 0.045358314425715918, 0.79582563587384847, 2.1416244286331416], 
[-2.1290269746767718, -5.3512739138520935, -1.5107572722064848, -3.2384743031328229], [0.37563979621939136, 
-1.3929493731874407, -2.6079060650374659, 1.480014287809853]]], [[[0.86655827843133615, -4.3077801774407929, 
0.34802914154057873, -6.4550943458744321], [1.5312423015955048, 1.7132972746867612, -1.8993568770497244, 
-0.018487480507010545], [2.3275554869946404, -0.077393256022816104, -5.2320666376103384, -3.1465463645992093]], 
[[-2.3454993354705964, -0.40063683191861887, -2.3613587055104279, -0.720850601467248], [-3.2489210741985195, 
-0.25749909351915257, 1.8217970014598137, 2.1879076484620708], [-5.1606510691211911, -5.3500153134768746, -1.3849410981215593, 
-0.67684607532969743]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-2.4980315488503013, -1.2677131696872013, 1.1495021057344559, -4.8224935261463111], 
[-4.0847371430528634, -0.21038345634151501, -1.350990608093392, 1.7066845581044845], [3.8713499841987762, 1.703892977837155, 
-0.67145659125622892, 3.1403056562610239]], [[0.40581574471696769, -4.0016366190750272, 3.3766718616486724, 
-3.3578056764875743], [-2.0566223847776319, 1.7918780417988955, -3.3867112463091709, -2.9671566074233136], 
[-3.5789124044395617, 3.2979695668157998, -1.6838293185396944, -4.2686583879826445]]], [[[-0.6173458125042437, 
-2.7836511711223757, -0.019017847932218501, -3.9647719520781899], [-3.6136058488817566, 1.6456891220514702, 3.2560592021474406, 
-4.082357763706959], [-0.24475741127733386, -1.5641858031205089, -4.9618728805419599, 4.9569364451539517]], 
[[4.9352769126442748, -3.4664939882164347, -0.43354861088301977, -3.4951565476698123], [4.6626937880486619, 
-2.2718185626583431, -2.121748058581149, 3.4693210322602255], [-2.3785758027756843, -0.59024969226780044, -3.9397579221105139, 
4.2176155895072007]]], [[[1.3447655933565636, 2.1964897807684967, 1.3295300771019436, -2.2187112419752575], 
[3.8875723865649316, -4.1375175692317736, 2.7935699414379407, 0.97462433556883177], [-4.330956956674715, -3.2038882093334111, 
-1.6102451503535544, -3.7978171080496481]], [[-3.3109844880314867, 0.71639744937036287, -1.9537169799044163, 
-0.11033694309958708], [-3.536599314601828, 3.8168494462725757, -0.87502942848260723, -1.3419297268692612], 
[3.7416143230349341, 3.892264483809619, -0.63545782881015977, 
-1.0238261709935506]]]])+(1.-msk_arg0)*numpy.array([[[[-1.5046640230113653, 4.4916597850947309, 4.0194645834226392, 
4.2912203087368255], [-3.9291832660155412, 0.13180606879613332, 0.053977118029672866, 1.4475494322147604], 
[-4.2330176670101602, -3.8228520456354409, -4.9172100411562223, 2.8121413472694998]], [[1.0286717358473147, 
-1.7519153634147457, 3.6706630762879442, 0.49897232393169322], [3.6103196528652646, 4.1072332900220108, 3.4406782023312168, 
-1.6790513206210345], [-1.6379550504738249, -0.36380467695478202, -2.4694960962740851, -0.8123735013881328]]], 
[[[-2.4371581077955629, 0.85523421754017548, 2.2069446415816243, 2.9609645764012544], [0.67955058470744056, 1.3044846613950103, 
-4.9402068922769447, -1.4129905250814758], [4.444327825405205, 1.304849227203956, 4.2095168739110242, 4.3001543779802418]], 
[[1.0374696446115994, 1.5934061203052705, -2.0781657945642995, 2.3297066314718959], [-2.1296598312191142, -2.4017805426938734, 
-0.9037900996631496, -3.1821025729743324], [-1.8034748616213578, 2.9388331888122838, 0.83415301723636048, 
-3.7815580693562612]]], [[[3.0525465680082711, -0.55752615394515281, 3.0204456232327832, 1.757598188841671], 
[-3.0006426232681358, -4.9946301761928886, 1.2520586234049746, 1.3678485979297488], [-2.5944349681705514, 4.7541945608505927, 
0.70274435189571793, -2.811593132990482]], [[1.2409894091315827, 0.29741719553318369, 2.5720083081398517, -2.633225751035325], 
[0.26339631824781939, 3.2707224667975066, -0.076018921794711147, -4.9320589450127148], [2.6268601253279344, 
-2.5112755581075472, 1.2425142567901082, -0.20838467856030185]]]])
      arg1=numpy.array(1.09813374663)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-3.5961652954804357, -2.3658469163173357, 0.05136835910432147, -5.9206272727764455], 
[-5.1828708896829978, -1.3085172029716494, -2.4491243547235264, 0.60855081147435008], [2.7732162375686418, 0.60575923120702058, 
-1.7695903378863633, 2.0421719096308895]], [[-0.69231800191316673, -5.0997703657051616, 2.278538115018538, 
-4.4559394231177087], [-3.1547561314077663, 0.69374429516876113, -4.4848449929393048, -4.065290354053448], 
[-4.6770461510696961, 2.1998358201856654, -2.7819630651698288, -5.3667921346127789]]], [[[-1.7154795591343781, 
-3.8817849177525101, -1.1171515945623529, -5.0629056987083239], [-4.711739595511891, 0.54755537542133581, 2.1579254555173062, 
-5.1804915103370934], [-1.3428911579074683, -2.6623195497506433, -6.0600066271720943, 3.8588026985238173]], 
[[3.8371431660141404, -4.5646277348465691, -1.5316823575131542, -4.5932902942999467], [3.5645600414185274, -3.3699523092884776, 
-3.2198818052112834, 2.3711872856300911], [-3.4767095494058187, -1.6883834388979349, -5.0378916687406488, 
3.1194818428770663]]], [[[0.24663184672642924, 1.0983560341383622, 0.23139633047180919, -3.3168449886053919], 
[2.7894386399347972, -5.235651315861908, 1.6954361948078063, -0.12350941106130264], [-5.4290907033048494, -4.3020219559635455, 
-2.7083788969836888, -4.8959508546797821]], [[-4.4091182346616211, -0.38173629725977154, -3.0518507265345507, 
-1.2084706897297215], [-4.6347330612319624, 2.7187156996424413, -1.9731631751127416, -2.4400634734993956], [2.6434805764047997, 
2.7941307371794846, -1.7335915754402942, -2.121959917623685]]]])+(1.-msk_ref)*numpy.array([[[[-2.6027977696414997, 
3.3935260384645964, 2.9213308367925048, 3.1930865621066911], [-5.0273170126456757, -0.96632767783400109, -1.0441566286004615, 
0.34941568558462599], [-5.3311514136402947, -4.9209857922655758, -6.0153437877863567, 1.7140076006393654]], 
[[-0.069462010782819661, -2.8500491100448802, 2.5725293296578098, -0.59916142269844119], [2.5121859062351302, 
3.0090995433918764, 2.3425444557010824, -2.7771850672511689], [-2.7360887971039594, -1.4619384235849164, -3.5676298429042195, 
-1.9105072480182672]]], [[[-3.5352918544256973, -0.24289952908995893, 1.1088108949514899, 1.86283082977112], 
[-0.41858316192269385, 0.20635091476487588, -6.0383406389070791, -2.5111242717116102], [3.3461940787750706, 
0.20671548057382161, 3.1113831272808898, 3.2020206313501074]], [[-0.060664102018535004, 0.4952723736751361, 
-3.1762995411944339, 1.2315728848417615], [-3.2277935778492486, -3.4999142893240078, -2.001923846293284, -4.2802363196044668], 
[-2.9016086082514922, 1.8406994421821494, -0.26398072939377393, -4.879691815986396]]], [[[1.9544128213781367, 
-1.6556599005752872, 1.9223118766026488, 0.65946444221153655], [-4.0987763698982702, -6.092763922823023, 0.15392487677484024, 
0.26971485129961437], [-3.6925687148006858, 3.6560608142204583, -0.39538939473441648, -3.9097268796206164]], 
[[0.14285566250144832, -0.80071655109695072, 1.4738745615097173, -3.7313594976654594], [-0.83473742838231502, 
2.1725887201673721, -1.1741526684248456, -6.0301926916428492], [1.5287263786978, -3.6094093047376816, 0.14438051015997377, 
-1.3065184251904363]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.21364854845134307, -3.4010523904005927, -3.3040184543496287, -0.4538839998735531], 
[0.069296117702625004, -0.073912933966381544, -1.8352549268700145, -1.2031337036841485], [-3.4132310324256498, 
3.570134009774117, 1.3416940654567533, -1.2458207009365232]], [[3.669909210927651, -3.0260532243164429, -1.0486695327762598, 
1.4910549211874304], [3.384059575328763, -4.7695153724427017, -3.7097370788105688, -4.1783774893487706], [0.62840499490082635, 
4.8455351799559665, 0.40415795996634696, 4.2390775579151079]]], [[[-0.97150368712317636, 4.6143245330813034, 
-0.82871316666533623, 0.19348585961377029], [-1.8991202906195168, 4.309781360516773, -0.85376384868565758, 1.9668200098810127], 
[4.7769313136479479, -4.0154515475358465, 1.8786418438848855, 2.7195548362023541]], [[3.378225160772871, -0.14872542635385511, 
1.3955046097251236, 0.944065785958351], [-1.175076092915277, -4.164279788408221, 0.90542157670098877, -4.4757637444604805], 
[1.7756937354547837, -1.3269424625903548, 0.64181659166977667, -3.5387471123118952]]], [[[4.1827360281806332, 
-0.65503010892171254, 3.2371185554680295, 2.2444096455777078], [4.7056306223439428, 2.5477987675647835, 4.3949300044009831, 
-4.8255320375623434], [-4.1174392437500016, -3.9513234737288916, -3.7522827124486691, -3.6809487837399004]], 
[[4.5874456708050069, -4.3450104333857062, 2.529294682983255, -4.054717355278421], [3.911130729805242, 0.1056735748567359, 
2.3050737683051024, -4.1141343381355107], [3.4764113511551091, 4.8070384685306102, 4.8836472268075344, 
1.4757815111032224]]]])+(1.-msk_arg0)*numpy.array([[[[-1.1526809701426677, 0.50597754653380989, 1.5000298265824732, 
4.3596817930168132], [1.9942473224962685, -2.9139310314922962, 0.38506801450545325, 3.8894513305256702], [3.1424307164813197, 
3.0519153630500764, -0.89402167492357965, 2.6290218125557683]], [[0.29327856295338162, -4.1755992986598978, 
-3.4113040157588692, -0.91261890220694042], [-4.981957083143163, -4.3943085770123052, -4.9744531197010646, 1.1961960698600125], 
[1.9432578313058837, -4.959883040940408, -3.2789505533460863, 0.45609326266383565]]], [[[-3.7950940222583531, 
1.8942597993779788, -0.37810757818038887, -2.4929672969207326], [3.8895814914300999, -0.50563827931041239, 1.5353415891942479, 
0.77760889852865844], [-0.76506880398210431, -1.0332425583795111, 1.104742196604203, 2.6136020869949839]], 
[[-4.1770113128499196, -0.80828799117572103, -2.1054649650412394, 2.6344827711011645], [-2.2138660947841151, 
-2.6744060723728449, 1.6819419963134505, -1.9113070926169149], [-4.3853847182895809, 0.00019513840526652615, 
3.2387367567023357, 3.4835703838223466]]], [[[0.69426355769979509, 1.8457290243539495, 1.3926848653177224, 1.827441494451115], 
[-2.1334353451781265, -1.222772392187105, 3.2881667971289925, -4.3728113973620841], [4.8728074292878016, 0.59394126329814512, 
-3.0479072831961815, 3.9754376253146599]], [[-2.6632958040022214, 2.9705563466977747, 4.195448653535756, 0.92110804594499562], 
[3.7621355492148751, -4.3313153825859096, -3.1989528229173358, -4.0179081395934437], [0.54859067252020211, 2.8512113791991052, 
0.11458425333918143, 4.9235406051202393]]]])
      arg1=numpy.array([[[[4.2744418992475737, -0.2576082017633361, 4.6049677862762479, -1.1999950609564323], 
[-2.5233568082864855, -4.2454389073192624, -3.5959781698687632, 1.9912502387287478], [-3.5481550861482303, -1.183109095614391, 
2.5093034990564487, 2.5442206884475045]], [[-1.1754003504984012, 3.4013099945343086, 2.6239291701835024, -0.26763172035476313], 
[0.83759889660992659, 1.2105597562292463, 3.8655903607446245, 3.8758095427304458], [4.1806203546335432, -3.1372588901176544, 
-1.4234679430476902, 1.4210061921219275]]], [[[1.8020142017175509, -4.2297803000566585, -2.734128057627724, 
2.3797038867815843], [-2.8206539338026393, 0.92162349052762682, -0.99912071282754056, 2.2336104850198364], 
[-3.1197836008861337, -2.1664719976036606, 0.31778106099310577, 3.1747852218545916]], [[-3.6781746755386702, 
-0.87679617669803456, 0.85226605862254523, 2.3493881692346577], [-3.1600990835334928, -4.0403203541168056, 1.3651845197655499, 
-1.6825646955735927], [1.3604405899455738, 1.1797293107468834, -2.0485733490623117, 1.3925160948757176]]], 
[[[-2.1438971459743272, -3.4107936839961575, -2.4374778690691135, 2.9442940030222076], [-3.2752483215667185, 
1.7197054211812368, -1.4547186454939434, -1.3845453618088763], [2.7476242002094633, -4.6885062994772886, 3.4674169407914555, 
-1.5305689815399717]], [[-3.6732620074955999, 1.1741454207482231, -2.9583852743837769, -3.7583581010812916], 
[-4.663560553275361, 4.3455373721256745, 0.74161433818828648, -1.1603490614318068], [3.0561340280140872, -2.8133043055941256, 
0.69911098851125431, 1.1820664895431072]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-4.4880904476989167, -3.1434441886372566, -7.9089862406258771, 0.74611106108287917], 
[2.5926529259891105, 4.1715259733528809, 1.7607232429987487, -3.1943839424128964], [0.13492405372258043, 4.753243105388508, 
-1.1676094335996954, -3.7900413893840277]], [[4.8453095614260526, -6.4273632188507515, -3.6725987029597622, 
1.7586866415421936], [2.5464606787188364, -5.980075128671948, -7.5753274395551937, -8.0541870320792164], [-3.5522153597327168, 
7.9827940700736209, 1.8276259030140372, 2.8180713657931804]]], [[[-2.7735178888407273, 8.8441048331379619, 1.9054148909623878, 
-2.186218027167814], [0.92153364318312248, 3.3881578699891461, 0.14535686414188298, -0.2667904751388237], [7.8967149145340816, 
-1.8489795499321859, 1.5608607828917798, -0.4552303856522375]], [[7.0563998363115417, 0.72807075034417945, 0.54323855110257835, 
-1.4053223832763067], [1.9850229906182157, -0.12395943429141543, -0.45976294306456111, -2.7931990488868879], 
[0.41525314550920989, -2.5066717733372381, 2.6903899407320884, -4.9312632071876124]]], [[[6.32663317415496, 2.7557635750744449, 
5.6745964245371425, -0.69988435744449973], [7.9808789439106613, 0.8280933463835467, 5.8496486498949265, -3.4409866757534671], 
[-6.8650634439594649, 0.737182825748397, -7.2196996532401245, -2.1503798021999287]], [[8.2607076783006068, -5.5191558541339294, 
5.4876799573670318, -0.29635925419712938], [8.5746912830806039, -4.2398637972689386, 1.5634594301168159, -2.9537852767037038], 
[0.42027732314102195, 7.6203427741247358, 4.1845362382962801, 
0.2937150215601152]]]])+(1.-msk_ref)*numpy.array([[[[-5.4271228693902414, 0.76358574829714598, -3.1049379596937747, 
5.5596768539732455], [4.5176041307827539, 1.3315078758269663, 3.9810461843742164, 1.8982010917969223], [6.6905858026295499, 
4.2350244586644674, -3.4033251739800283, 0.084801124108263792]], [[1.4686789134517828, -7.5769092931942064, 
-6.0352331859423716, -0.64498718185217729], [-5.8195559797530896, -5.6048683332415514, -8.84004348044569, -2.6796134728704333], 
[-2.2373625233276595, -1.8226241508227536, -1.8554826102983961, -0.96491292945809182]]], [[[-5.597108223975904, 
6.1240400994346373, 2.3560204794473352, -4.8726711837023169], [6.7102354252327387, -1.4272617698380392, 2.5344623020217885, 
-1.4560015864911779], [2.3547147969040294, 1.1332294392241495, 0.78696113561109726, -0.56118313485960769]], 
[[-0.49883663731124939, 0.068508185522313525, -2.9577310236637846, 0.28509460186650681], [0.94623298874937767, 
1.3659142817439607, 0.31675747654790065, -0.22874239704332222], [-5.7458253082351547, -1.1795341723416168, 5.2873101057646474, 
2.091054288946629]]], [[[2.8381607036741223, 5.2565227083501069, 3.8301627343868359, -1.1168525085710925], [1.141812976388592, 
-2.9424778133683418, 4.7428854426229359, -2.9882660355532078], [2.1251832290783383, 5.2824475627754337, -6.515324223987637, 
5.5060066068546316]], [[1.0099662034933785, 1.7964109259495515, 7.1538339279195329, 4.6794661470262877], [8.4256961024902353, 
-8.6768527547115841, -3.9405671611056223, -2.8575590781616369], [-2.5075433554938851, 5.6645156847932308, -0.58452673517207288, 
3.7414741155771321]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[2.4443843044994482, 2.7680360464907681, -2.446778359582841, 4.3769432717485444], 
[4.9910395859797934, -2.0438369260388702, -3.9751272747992248, -0.46571003515160214], [2.126741049599187, 1.4881658104798809, 
-3.160131072637399, 1.1217597015580703]], [[3.1117815755843274, -1.2834256380909839, 0.50111268033742817, -2.2399793730920168], 
[4.7541373428141327, 2.2202621883834484, -2.8137876669626882, 0.70011696342604157], [-2.0662860336399591, -3.978678062931138, 
4.7226835578864481, -3.7657665128358375]]], [[[2.1483851419212741, -0.81397533389786414, 0.075247991215151799, 
3.1632123235357845], [3.9854012174994793, -1.2763725867067057, 4.8095014036857116, 2.8255027527605101], [-2.2421005116542414, 
1.8769313135695089, 4.1219003533622995, -0.0040903432288308395]], [[-2.7630838814528245, 1.9545543270516781, 
4.9022292694905563, -2.8648409410532727], [-3.4791652896110534, 2.9604159453379282, 2.8521387589020764, -2.0925416013780018], 
[3.7214946244352891, -1.2952366840075702, -4.9907523183575275, 1.0952415012621488]]], [[[-4.7465135885061738, 
4.8799248339825603, -2.9547653298921119, -3.059498424475656], [-3.687378344603772, 0.03575706950253732, -2.475262939605114, 
0.66687189730840757], [2.9484956172839336, 3.0933080593648707, -4.4241017578921982, -0.90463911618311155]], 
[[-4.4373446856185907, -2.7363926841363551, -1.6468361127139, 3.8426598139310144], [0.84085895739763927, -3.2181980933674659, 
-4.0060626138590507, 3.5350765528550792], [1.6126180364852836, 3.5230847835559196, -0.32597969025770013, 
0.089822052952914788]]]])+(1.-msk_arg0)*numpy.array([[[[-4.1702370054221873, -2.4402253955088793, 1.7871974639275248, 
-1.2575321759633828], [-2.728693285173851, -1.8286786550835754, -2.499999632840626, 4.2457710433360543], [-2.151967339474953, 
-4.7155980783375675, -4.5896399407145463, -3.7535677337788087]], [[1.4686614099226496, -1.3662632137177178, 
-4.0899279986723034, 2.3634512556095268], [-2.0731667107772878, 3.6173435807185328, 1.2200658014328738, -0.18606439356196702], 
[1.2165434754457944, -4.9144254607559557, 4.458963735620145, 2.5709427507549094]]], [[[3.3423380615402802, -3.0570912302816566, 
-4.2619772231330089, 0.86466839918171079], [4.2978125234221149, -1.6225141495095654, 2.6195188420857276, 
0.0042445730346987176], [-4.3819702143953911, -0.75639436612416056, -4.9536408256609823, -4.2226971461361433]], 
[[0.76991808314904553, 4.9077675865196326, -1.9533619534407887, -1.061111620069477], [-1.9239986303583767, -1.5476493885169198, 
-4.7831470409220245, 2.0632368896290938], [-0.15429839121408051, 0.94861852627211807, -1.0519608940181757, 
1.5797399132386349]]], [[[3.3676516217577088, -1.8438500065378784, 0.51445597498835482, -0.42965692690179136], 
[3.1151998172879232, 1.7978722951722403, -1.4955402854479316, -4.3559889969947605], [3.6468767439898926, -4.7119363972802848, 
-4.8360970739713869, 3.615563221111751]], [[-4.5355200320142108, -4.2617559284299329, -2.6857331637885871, 2.9926200385175852], 
[-0.57593940263453458, -2.9518948339802051, -1.1580940641123947, 3.1387469060138447], [1.0412430770659888, -2.0702719172479744, 
0.89136368532893684, -3.220294893719251]]]])
      arg1=Data(3.94017884111,self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.4957945366125882, -1.1721427946212684, -6.3869572006948774, 0.43676443063650794], 
[1.0508607448677569, -5.9840157671509067, -7.9153061159112612, -4.4058888762636386], [-1.8134377915128495, -2.4520130306321555, 
-7.1003099137494354, -2.8184191395539662]], [[-0.82839726552770898, -5.2236044792030203, -3.4390661607746082, 
-6.1801582142040532], [0.81395850170209627, -1.7199166527285881, -6.7539665080747247, -3.2400618776859949], 
[-6.0064648747519955, -7.9188569040431744, 0.78250471677441169, -7.7059453539478735]]], [[[-1.7917936991907624, 
-4.7541541750099006, -3.8649308498968846, -0.7769665175762519], [0.045222376387442864, -5.2165514278187421, 
0.86932256257367513, -1.1146760883515263], [-6.1822793527662778, -2.0632475275425275, 0.18172151225026312, 
-3.9442691843408673]], [[-6.7032627225648609, -1.9856245140603583, 0.96205042837851984, -6.8050197821653091], 
[-7.4193441307230898, -0.97976289577410824, -1.0880400822099601, -6.0327204424900387], [-0.21868421667674731, 
-5.2354155251196062, -8.930931159469564, -2.8449373398498876]]], [[[-8.6866924296182102, 0.93974599287052385, 
-6.8949441710041484, -6.9996772655876924], [-7.6275571857158084, -3.9044217716094991, -6.4154417807171509, 
-3.2733069438036289], [-0.99168322382810281, -0.84687078174716568, -8.3642805990042355, -4.844817957295148]], 
[[-8.3775235267306272, -6.6765715252483915, -5.5870149538259364, -0.097519027181022011], [-3.0993198837143972, 
-7.1583769344795023, -7.9462414549710871, -0.40510228825695727], [-2.3275608046267529, -0.41709405755611684, 
-4.2661585313697366, -3.8503567881591216]]]])+(1.-msk_ref)*numpy.array([[[[-8.1104158465342238, -6.3804042366209153, 
-2.1529813771845117, -5.1977110170754193], [-6.668872126285887, -5.7688574961956114, -6.4401784739526624, 0.30559220222401784], 
[-6.0921461805869894, -8.6557769194496039, -8.5298187818265827, -7.6937465748908451]], [[-2.4715174311893868, 
-5.3064420548297537, -8.0301068397843398, -1.5767275855025096], [-6.0133455518893246, -0.32283526039350363, 
-2.7201130396791626, -4.1262432346740034], [-2.723635365666242, -8.8546043018679921, 0.51878489450810861, 
-1.369236090357127]]], [[[-0.59784077957175619, -6.9972700713936931, -8.2021560642450453, -3.0755104419303256], 
[0.35763368231007853, -5.5626929906216018, -1.3206599990263088, -3.9359342680773377], [-8.3221490555074276, -4.696573207236197, 
-8.8938196667730196, -8.1628759872481798]], [[-3.1702607579629909, 0.96758874540759621, -5.8935407945528251, 
-5.0012904611815134], [-5.8641774714704127, -5.4878282296289562, -8.7233258820340609, -1.8769419514829426], 
[-4.0944772323261169, -2.9915603148399184, -4.9921397351302126, -2.3604389278734015]]], [[[-0.5725272193543276, 
-5.7840288476499149, -3.4257228661236816, -4.3698357680138278], [-0.82497902382411326, -2.1423065459397961, 
-5.4357191265599685, -8.2961678381067969], [-0.29330209712214383, -8.6521152383923212, -8.7762759150834242, 
-0.32461562000028543]], [[-8.4756988731262481, -8.2019347695419693, -6.6259120049006235, -0.94755880259445124], 
[-4.516118243746571, -6.8920736750922416, -5.0982729052244311, -0.80143193509819177], [-2.8989357640460476, 
-6.0104507583600109, -3.0488151557830996, -7.160473734831287]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.524275381237441, 1.6878927842138616, -2.9817330556262789, 2.917233058805266], 
[-1.8365730970237601, -4.170683824677611, 2.4597809270720008, -0.44984000347854991], [4.2286190411598117, 3.2895206237217138, 
0.10085692647238886, -2.509654151610905]], [[-3.5955088842627725, -1.1573206303611761, -3.6709994572684392, 
2.7243263552027299], [0.92287907644437972, -2.7152866696259705, 4.7570131693598654, -2.0525824510420487], [1.332996630240725, 
-2.7709157401559148, 0.64855885528016088, -2.2809347226992207]]], [[[3.0016047693304149, -1.5002545893448618, 
4.0867280142730387, 1.020668019153554], [-2.4873585436311538, 3.4074235216566642, 3.7392016003287445, -3.1157056381944823], 
[1.177930263080663, 3.6851361653929295, -3.5554493655272856, 1.9247601150669267]], [[1.193350534398542, -4.4675258066038293, 
3.0783115485959875, 2.7175757796282554], [-4.0369226523030193, 1.4564028320078446, 1.6764646335509532, 4.6108809544709644], 
[-4.5619564158305481, 0.30238212599601244, 4.4319416155467763, 0.62923830372351652]]], [[[4.7707466601714437, 
-4.3617022413754167, -3.2187682302481191, 0.27807378971963281], [-4.3014362722305641, -3.6286481479972799, 2.42704228608559, 
3.3314527163772709], [-4.216495250532807, -3.4811706762228214, -2.1211201533334578, -0.55907826296634955]], 
[[-0.68743926724529558, 0.45885226714093985, 2.1941537829740909, -1.8569211370377103], [-4.7540158821732419, 
-4.0170888747062889, 4.6151446809077754, -3.2925171138178833], [-3.3272572807660761, -1.7436328561531376, 1.3984121304308337, 
-4.5806156228185184]]]])+(1.-msk_arg0)*numpy.array([[[[0.72875871217808808, 0.66511686759824507, 4.6501036547427628, 
1.8028061871005647], [0.54444855537358272, 2.9709484352122466, 4.1131245744977605, -4.2402621579999913], [1.1207996696833176, 
-3.4957746188135754, 4.9406874831679186, 3.4003760654786745]], [[-3.9079786295853989, -4.0460065398202127, -3.5264567074691824, 
1.433078373904868], [2.3687329605908225, 1.6963508689285201, 3.9037728465658752, -2.5326230266635585], [2.53956992645334, 
0.90893944517885394, 0.39899387922175045, 3.6919803988555771]]], [[[3.3209563104439273, -2.9702827457074399, 
-0.6033354873539416, 1.9618589405162119], [-0.66637949645470407, 0.45452418765609703, 0.99837590932159248, 
-4.4883588627911388], [1.5160499448735596, 2.3213035230166579, -1.1227723214366425, 2.6858808680896651]], 
[[-3.3668133392619772, 2.0225393362696771, 4.0418374909834967, 1.2975416713402161], [4.9224316209412606, 1.2466819209389799, 
-4.0367299071473237, -0.057922747549286946], [-4.575337482503393, -3.4311970267956662, 0.49812668560608131, 
3.9260252291247149]]], [[[-1.3282591144581146, -2.3622951968863859, -2.9993970237737244, -4.052338233040107], 
[-0.44809670597860318, -0.53068791404105387, -0.016291009480854157, 4.6939508092701185], [2.0895640879718425, 
-0.53446349013750982, -3.6291795854591946, 0.42222396722715949]], [[3.9112558634254739, -3.1459201217118671, 
1.0391785017831214, 4.7178892879564867], [0.74148180190297808, -2.9477942379635227, 2.6811966506900164, 1.9648560746721335], 
[1.1209741437139789, 4.2155772384556869, -3.2383477139789116, -4.9437629586848377]]]])
      arg1=Data(numpy.array([[[[-3.1387642787367454, -2.1400423912860598, -3.0102090450891783, 2.6382029490405126], 
[-2.4932220717589706, 1.5934808687950834, -0.49459342982172316, 0.23671057408431651], [-0.73580596525951947, 
-1.7687241433206369, 3.8924752783260281, 0.058019190102798213]], [[0.91854180312587452, 1.8255658227866558, 
0.87339357935739059, -3.0107549707847103], [-0.65102830026264247, -0.060535864666867667, -4.2505371960966212, 
-0.30982589396937765], [1.2760929503450011, -0.88901057327943089, -3.4803613238126352, -4.2718757836922272]]], 
[[[1.5342888142381783, 4.2525292633027352, 4.9906900860061576, 0.11466023723401086], [4.7490102643325987, 0.91045943338861424, 
-0.74257846899522217, -4.9233076527774609], [3.8146106947734548, 4.8917687998613442, -2.1813021783363471, 
-4.7754285564799979]], [[-1.1351393258850928, -1.5472457542253295, 0.90968949620414374, 1.3207811744140008], 
[-3.6423024749998154, 3.4435693334804238, 0.56238864445228565, 1.6675572087977812], [-3.8856361845814389, 1.5104775345173396, 
-4.8336467673208015, -2.770455330074638]]], [[[-3.8561107167720046, -3.0042239654893645, -3.5202387113845326, 
-4.2255490309735908], [3.8017518843214919, -2.4329364896640069, 2.2953810512923321, 1.6523676197101871], [-1.3514501656150122, 
0.63489758033723032, -1.5172977409591182, -2.2880989647585759]], [[-0.84836656099736629, 3.5678022924611614, 
-3.1060939179059597, -2.900947459981853], [-0.49665912714546234, 4.3833893808167712, -2.8587416159764754, -3.3281389387625735], 
[1.3135639710739699, 1.1781817241791313, 4.8020589471179402, -1.2822428011142506]]]]),self.functionspace)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.3855111025006956, 3.8279351754999213, 0.02847598946289942, 0.2790301097647534], 
[0.65664897473521044, -5.7641646934726944, 2.954374356893724, -0.68655057756286642], [4.9644250064193312, 5.0582447670423507, 
-3.7916183518536393, -2.5676733417137032]], [[-4.5140506873886466, -2.9828864531478319, -4.5443930366258298, 
5.7350813259874407], [1.5739073767070222, -2.6547508049591029, 9.0075503654564866, -1.7427565570726711], [0.056903679895723869, 
-1.8819051668764839, 4.1289201790927965, 1.9909410609930065]]], [[[1.4673159550922366, -5.752783852647597, 
-0.90396207173311893, 0.90600778191954312], [-7.2363688079637525, 2.4969640882680499, 4.4817800693239667, 1.8076020145829785], 
[-2.6366804316927919, -1.2066326344684146, -1.3741471871909385, 6.7001886715469245]], [[2.3284898602836348, 
-2.9202800523784997, 2.1686220523918438, 1.3967946052142546], [-0.39462017730320387, -1.9871665014725792, 1.1140759890986676, 
2.9433237456731831], [-0.67632023124910923, -1.2080954085213271, 9.2655883828675769, 3.3996936337981545]]], 
[[[8.6268573769434482, -1.3574782758860522, 0.3014704811364135, 4.5036228206932236], [-8.103188156552056, -1.195711658333273, 
0.13166123479325798, 1.6790850966670838], [-2.8650450849177949, -4.1160682565600517, -0.60382241237433965, 
1.7290207017922263]], [[0.16092729375207071, -3.1089500253202216, 5.3002477008800506, 1.0440263229441427], 
[-4.2573567550277795, -8.4004782555230602, 7.4738862968842508, 0.035621824944690239], [-4.6408212518400465, 
-2.9218145803322688, -3.4036468166871066, -3.2983728217042678]]]])+(1.-msk_ref)*numpy.array([[[[3.8675229909148334, 
2.8051592588843048, 7.6603126998319411, -0.83539676193994783], [3.0376706271325533, 1.3774675664171632, 4.6077180043194836, 
-4.4769727320843078], [1.856605634942837, -1.7270504754929386, 1.0482122048418905, 3.3423568753758763]], [[-4.8265204327112734, 
-5.8715723626068685, -4.399850286826573, 4.4438333446895779], [3.019761260853465, 1.7568867335953877, 8.1543100426624964, 
-2.2227971326941809], [1.2634769761083389, 1.7979500184582848, 3.8793552030343856, 7.9638561825478043]]], 
[[[1.7866674962057489, -7.2228120090101751, -5.5940255733600992, 1.847198703282201], [-5.4153897607873027, -0.4559352457325172, 
1.7409543783168147, 0.43494878998632203], [-2.2985607498998952, -2.5704652768446863, 1.0585298568997046, 7.461309424569663]], 
[[-2.2316740133768844, 3.5697850904950066, 3.1321479947793529, -0.02323950307378464], [8.5647340959410769, -2.1968874125414439, 
-4.5991185515996094, -1.7254799563470682], [-0.68970129792195412, -4.9416745613130058, 5.3317734529268828, 
6.6964805591993528]]], [[[2.5278516023138899, 0.64192876860297865, 0.52084168761080818, 0.17321079793348382], 
[-4.2498485903000951, 1.902248575622953, -2.3116720607731862, 3.0415831895599315], [3.4410142535868546, -1.1693610704747401, 
-2.1118818445000764, 2.7103229319857354]], [[4.7596224244228402, -6.713722414173029, 4.1452724196890811, 7.6188367479383396], 
[1.2381409290484404, -7.3311836187802939, 5.5399382666664918, 5.292995013434707], [-0.19258982735999108, 3.0373955142765556, 
-8.0404066610968528, -3.6615201575705871]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[1.7730469052633238, -2.4057713665823655, 3.4289951090253563, 2.2846105483714663], 
[-4.4909300964111853, 1.6599951176639092, 2.1849081769430132, 2.4071421253508509], [-3.6508012864526784, 1.8284488691493461, 
4.0652214106535602, 4.8421140695908633]], [[-1.853290224454506, -1.043872071110703, 1.8938777342159874, 3.5031373062740627], 
[-3.925027542882523, -2.2875362569277771, 4.1565643931434675, 3.6348671847669376], [0.37789516728475103, -2.9757188125059586, 
4.9011747720831771, 3.0792302305146659]]], [[[-2.3849528035661938, -1.0992411999139806, 4.0074909504207508, 
-0.021122204890426488], [-3.0607154807987613, 1.0840906016961815, 4.4151295912405999, 2.2223678740297848], 
[0.85745062085975654, -4.2251252650008411, -1.4479356133101726, -0.14257727516749608]], [[4.0134127889680258, 
2.7026015827193195, -3.2956001329069928, 4.5374346272062773], [-3.1085413620717861, 2.2010301153667147, 4.7943364366929089, 
0.66867068454100398], [3.1649532863184415, -3.7496036972065969, 3.7672859081059258, 4.8187579759986772]]], 
[[[-4.116663336999693, -3.7006802172696194, -2.9679367057100805, 1.0284530495145514], [3.0649430596733289, -4.6890572233874011, 
0.38905986627723355, -0.47775435516842713], [4.7035945461200139, 2.2399968859053585, -1.2746724808073431, 4.2073074617418325]], 
[[2.4051109466004394, 0.7854007456859593, 2.1538123750220848, 1.2163724730751158], [0.063098992273047827, -4.5466178613809145, 
-1.587752952616488, -0.60928273874146477], [-4.5516712209314445, -0.40379692548405011, 2.182673785054452, 
-0.12422157365814002]]]])+(1.-msk_arg0)*numpy.array([[[[1.8859269173400186, -0.23192849569474916, -2.941462409450657, 
0.53956751082766985], [-3.7609543035358097, 3.5868332798407057, 2.4203921618412725, -2.2886727244571259], [-3.8503173661295662, 
-4.1823854741258071, -4.4162609350701301, 0.62532208971066083]], [[1.4772780516039772, -3.2410356648190164, 
-3.4790856839962716, -1.6456864888432432], [-1.7553902185240924, -0.57955631478025182, 1.270025344079885, 3.8372649502242329], 
[0.93141809925156327, -3.1946285418595197, 3.4241646371103673, 3.2654814566491055]]], [[[0.56009395924929262, 
-3.5723907532073529, 3.794644514568331, 1.2073175984741624], [3.1283211763089618, -2.4327050941674999, -4.9811328204974581, 
-1.7959006164833502], [-3.5974552588701849, 2.8399924706278501, 3.4712424155715844, 1.1389401554159964]], 
[[-3.8593766063929538, -3.3636354346624984, 0.096237039033836069, 4.4203834665814092], [-1.8320898196261703, 
-3.7633590306602049, 4.2588574527051328, -2.2991573582159841], [-2.5600057286238407, -2.0537012232065277, -0.74782071201744493, 
-0.80586019208029658]]], [[[-2.1520205845138349, 1.5714874477304939, 0.09027676466975354, -2.5451292725388095], 
[3.4046497757613423, 1.6768620997260335, 3.7086321556490667, 3.6290117587054613], [-4.3330437848843086, -2.3799163733346607, 
-3.8892223955127059, 3.8815684489240567]], [[3.7741059737980827, -0.58499709464619087, 2.0757153050666668, 
-2.8291094039427191], [-2.8664347347631991, -0.92430987121235209, -4.4994759391000141, -1.5411468677037843], 
[-1.2530750073855481, -4.3657941191460958, 3.2146459776995151, 0.35695743038147132]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.22832879921)+(1.-msk_arg1)*(1.05020154637)
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.4552818939459442, -4.634100165791633, 1.2006663098160884, 0.056281749162198302], 
[-6.7192588956204533, -0.56833368154535879, -0.04342062226625476, 0.17881332614158296], [-5.8791300856619468, 
-0.39987993005992184, 1.8368926114442923, 2.6137852703815954]], [[-4.0816190236637739, -3.272200870319971, 
-0.33445106499328059, 1.2748085070647948], [-6.1533563420917909, -4.515865056137045, 1.9282355939341995, 1.4065383855576696], 
[-1.8504336319245169, -5.204047611715227, 2.6728459728739091, 0.85090143130539797]]], [[[-4.6132816027754622, 
-3.3275699991232486, 1.7791621512114828, -2.2494510040996945], [-5.2890442800080297, -1.1442381975130864, 2.1868007920313319, 
-0.0059609251794832119], [-1.3708781783495114, -6.453454064210109, -3.6762644125194406, -2.370906074376764]], 
[[1.7850839897587578, 0.47427278351005153, -5.5239289321162612, 2.3091058279970094], [-5.3368701612810536, 
-0.027298683842553295, 2.5660076374836409, -1.559658114668264], [0.93662448710917356, -5.9779324964158649, 1.5389571088966578, 
2.5904291767894092]]], [[[-6.344992136208961, -5.9290090164788873, -5.1962655049193485, -1.1998757496947166], 
[0.83661426046406095, -6.917386022596669, -1.8392689329320344, -2.7060831543776951], [2.475265746910746, 0.01166808669609054, 
-3.503001280016611, 1.9789786625325645]], [[0.17678214739117148, -1.4429280535233087, -0.074516424187183183, 
-1.0119563261341522], [-2.1652298069362201, -6.7749466605901825, -3.8160817518257559, -2.8376115379507327], 
[-6.7800000201407125, -2.6321257246933181, -0.045655014154815987, 
-2.352550372867408]]]])+(1.-msk_ref)*numpy.array([[[[0.83572537096734845, -1.2821300420674193, -3.9916639558233271, 
-0.51063403554500031], [-4.8111558499084799, 2.5366317334680355, 1.3701906154686023, -3.338874270829796], [-4.9005189125022364, 
-5.2325870204984772, -5.4664624814428002, -0.42487945666200932]], [[0.42707650523130702, -4.291237211191687, 
-4.5292872303689418, -2.6958880352159134], [-2.8055917648967625, -1.629757861152922, 0.21982379770721483, 2.7870634038515627], 
[-0.11878344712110689, -4.2448300882321899, 2.3739630907376972, 2.2152799102764353]]], [[[-0.49010758712337754, 
-4.622592299580023, 2.7444429681956608, 0.15711605210149227], [2.0781196299362916, -3.4829066405401701, -6.0313343668701282, 
-2.8461021628560204], [-4.6476568052428551, 1.7897909242551799, 2.4210408691989143, 0.0887386090433262]], 
[[-4.9095781527656239, -4.4138369810351685, -0.95396450733883409, 3.3701819202087391], [-2.8822913659988405, 
-4.8135605770328755, 3.2086559063324627, -3.3493589045886543], [-3.6102072749965108, -3.1039027695791979, -1.7980222583901151, 
-1.8560617384529667]]], [[[-3.2022221308865051, 0.52128590135782371, -0.95992478170291662, -3.5953308189114797], 
[2.3544482293886722, 0.62666055335336335, 2.6584306092763965, 2.5788102123327912], [-5.3832453312569788, -3.4301179197073308, 
-4.9394239418853765, 2.8313669025513866]], [[2.7239044274254125, -1.635198641018861, 1.0255137586939966, -3.8793109503153893], 
[-3.9166362811358693, -1.9745114175850222, -5.5496774854726842, -2.5913484140764544], [-2.3032765537582183, -5.415995665518766, 
2.1644444313268449, -0.69324411599119884]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.83551316057212244, 3.1909951243292589, 1.3987846538178363, 4.4620819074946709], 
[-0.91569203602093108, -1.6939779911128916, -2.4210006263440964, -2.0242822655685999], [3.0975969618604999, 4.5389354159488775, 
-1.3163188960961216, -4.5924341681692651]], [[-3.0369259365194412, 4.1277299193664199, 2.3327525197908212, 2.9254275235820817], 
[-0.079781950139418356, -2.2191591262735679, 1.588861066681801, 3.3666600520214267], [0.17766661030500686, -4.1620537977918151, 
3.8900251425411874, -0.64739115567485239]]], [[[-3.781109410011755, -2.3783769216928574, -1.5046683858710033, 
-4.0821639903363582], [2.5064628512217819, -1.8909880628928311, 1.8904714319741567, -0.094008452879382531], 
[0.23250693071946049, 0.53472399708506568, 3.7598903704593951, -1.5400640539197399]], [[4.8051033091099118, 
-3.6369352988689627, -4.5625832455450563, 1.7920995484494142], [3.9921072834449483, 3.1663401526245227, -0.97403161515480452, 
-0.90226942631777796], [-0.79215840707813534, -1.2361767354936379, -3.1573146504326441, 1.8251996190995712]]], 
[[[-2.4819147868433236, 3.5329211403932401, -2.2502564537430461, 2.4631457829202787], [-2.1014137059802316, 
-1.0749775673858508, 0.13189437616806998, 3.4169906539372548], [-2.3981493656412542, 4.6913000557964217, 0.65624898661828368, 
-0.7018512314730927]], [[-4.3599345911235261, 2.9049361363938599, -4.7312584962264692, -2.8412025243694172], 
[4.7674838601776628, -1.3530645216913184, -3.0231368390220514, -0.92838154825118657], [-3.4161144744837566, 
-1.5334550728341934, -4.9302542448683706, -3.595652147602709]]]])+(1.-msk_arg0)*numpy.array([[[[3.5835575177362884, 
-2.9319637706158028, -1.542606412823746, 3.6832109288979886], [0.64436239842884913, -4.3456436433993604, -0.63070383234858873, 
1.7149585486872514], [-2.0687077997851699, -1.968432583015737, 4.076213728406163, -1.3734673759088443]], [[-4.3669091383217431, 
2.7207645213463767, 0.25315471100807141, 2.3353267078790738], [4.2375574183621438, -0.62124874922970275, 4.2555801119286514, 
-2.1792019302677126], [2.5555144590273171, -3.2777855744217845, 1.1377002830038201, -4.2266190873173173]]], 
[[[2.7690083361193985, -3.7938354231183133, 1.1493744385158404, -0.59693109872967831], [-4.5259972364884931, 
-4.6168223255698866, -4.9342274047211498, -1.3864865395111936], [2.3361009955897494, 3.247452271155268, 2.726652450428098, 
-3.8510037957724954]], [[2.107765388077194, 2.6605872798764612, -2.0993552517047256, -3.8289285255032524], 
[-0.18680040545395382, -1.4343493346629432, -0.33880833450390391, 4.153228121581968], [-2.1498885530290979, 
-3.1884085956892747, -1.896507756192074, -4.1313712836940706]]], [[[1.1120058208289976, -3.1229298259853868, 3.05806015666459, 
1.683153023227125], [1.0943749461379451, 3.5852171701000586, 2.1568785485813757, 0.095207983227095561], [0.58587669742543458, 
0.078245505210615462, -3.5631302656239585, 4.3819635038809981]], [[0.70533864030468774, -4.413787097285363, 
-2.5445612468746583, -2.5524492593571644], [0.35167648933647833, -3.4445727739269048, 1.0502453568105761, 0.56146489334248351], 
[1.5114164024924994, 1.0441015457873757, 2.8018071200773873, 2.5819038179832301]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[2.0399172151932188, -1.8137471346728939, 1.0524632476635425, -4.3334330506286847], 
[-3.459264856491572, 0.3354464191904647, 1.9264035743047874, 4.2522482664996435], [-3.5569738160007924, 1.0496482698373359, 
1.7497419142686654, 4.4960349967220683]], [[-4.6295484684775001, 4.9826341802964969, 4.4953200198289807, 3.284630431281192], 
[0.18279993241714454, -4.7385676045802425, 4.5911231846059408, -0.69440930822443381], [-4.5476665997945602, 
0.44895811995060697, -2.7286346016553233, -3.144709708164386]]], [[[-4.3457346161043606, -1.4512036312483101, 
-4.7825463818696869, 1.085812551897881], [-0.74347862986729574, -2.4905426494888849, 1.5250196559465143, -1.8396899969462321], 
[-2.3337167360135904, -1.172680828895829, -1.7985053992004953, -2.6832266923442294]], [[2.1562687817590103, 
-0.54797613529270262, -2.1750839032646461, -3.1757226878400271], [0.58348747644363375, -2.5790381235375435, 
-4.5664504930948286, 1.6497486722904053], [-0.67627559917383895, -2.0127418663881436, -3.7759812717408767, 
0.54814975603275062]]], [[[4.9132032773109913, 3.2863706918381688, -4.9640201380147175, -1.8156832263974074], 
[-0.12223796638562234, 0.944303151991984, 4.2448881669120624, 2.0750633554744615], [3.9306692352271693, -4.7413814552097531, 
3.3826107926123132, 4.3242084573724178]], [[-2.736373499194209, 4.3976077605457302, 3.0234040433168268, -4.9928641905108844], 
[-2.0099186347783493, 3.2770134356075769, -2.5490064484099473, -1.9656524259471428], [-4.5095229299478632, -3.5443444802607504, 
-2.6996543138085904, -4.8578594703816513]]]])+(1.-msk_arg1)*numpy.array([[[[1.5725223418668879, 4.1823766968455924, 
-4.6773914612828627, -4.7006199769130843], [4.7116980386774188, -3.2439949524895129, -3.1453969648465554, 3.370511582942326], 
[3.9151462284497445, -4.1279513315286458, -0.91187797736255405, 1.3585173041119001]], [[4.3553489731481836, 2.3899782926201532, 
1.5928175306267391, -1.6404615238822395], [1.6278098426168004, 2.3447685147465824, 3.7517047885799713, -3.3092143587509391], 
[-3.6592165664081691, -1.5706009925409283, 3.0149591451481719, 0.45023406060006099]]], [[[-2.0705773708315647, 
0.6990013472791734, -0.16635085767710134, 0.75785986001714889], [4.0847047225459328, 3.9227592362625074, 2.945594254088042, 
1.6099598764028977], [1.4093678032650656, 2.6090168691414162, 4.3408761970183072, -1.8967908483873974]], [[-1.0001014389496685, 
1.2901461673308408, 4.0447583602325246, 1.2264258375564179], [-2.007401528170579, 2.7345770520006081, -4.3025758607416122, 
0.0035844474287474171], [0.025554356231838327, 0.76548484763302227, -4.2784240661240913, 4.6650158907697801]]], 
[[[0.00030457780689019387, 2.1189689773849043, -3.3665076053812926, 4.6909882275729551], [0.017071384437236325, 
-0.63923607335437094, 1.0445447613537766, -0.37159442722213143], [-4.0989537357149155, 2.4936552708334672, 3.409916197350233, 
-3.190243231281169]], [[1.6112061901722887, -3.656406759816627, -4.4859317360923123, 0.55636595224890328], 
[-3.6955638462830045, 4.2721549959599194, 1.1019884056273153, -3.0891001891764223], [4.2651813080085716, -3.5229859725027648, 
-4.3927831166243809, 4.1348107196208357]]]])
      res=arg0-arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.2044040546210963, 5.0047422590021533, 0.34632140615429385, 8.7955149581233556], 
[2.5435728204706409, -2.0294244103033563, -4.3474042006488833, -6.276530532068243], [6.6545707778612924, 3.4892871461115416, 
-3.0660608103647871, -9.0884691648913325]], [[1.5926225319580589, -0.85490426093007699, -2.1625675000381595, 
-0.35920290769911034], [-0.2625818825565629, 2.5194084783066746, -3.0022621179241398, 4.0610693602458605], [4.7253332100995671, 
-4.611011917742422, 6.6186597441965107, 2.4973185524895336]]], [[[0.56462520609260558, -0.92717329044454733, 
3.2778779959986837, -5.1679765422342392], [3.2499414810890777, 0.59955458659605387, 0.36545177602764234, 1.7456815440668496], 
[2.5662236667330509, 1.7074048259808947, 5.5583957696598905, 1.1431626384244895]], [[2.6488345273509015, -3.0889591635762601, 
-2.3874993422804103, 4.9678222362894413], [3.4086198070013145, 5.7453782761620662, 3.5924188779400241, -2.5520180986081833], 
[-0.11588280790429639, 0.77656513089450563, 0.61866662130823258, 1.2770498630668206]]], [[[-7.3951180641543148, 
0.24655044855507136, 2.7137636842716715, 4.2788290093176862], [-1.9791757395946092, -2.0192807193778348, -4.1129937907439924, 
1.3419272984627932], [-6.3288186008684235, 9.4326815110061748, -2.7263618059940296, -5.0260596888455105]], 
[[-1.6235610919293171, -1.4926716241518703, -7.754662539543296, 2.1516616661414671], [6.7774024949560125, -4.6300779572988953, 
-0.47413039061210416, 1.0372708776959563], [1.0934084554641066, 2.0108894074265571, -2.2305999310597802, 
1.2622073227789423]]]])+(1.-msk_ref)*numpy.array([[[[2.0110351758694005, -7.1143404674613953, 3.1347850484591167, 
8.3838309058110738], [-4.0673356402485696, -1.1016486909098475, 2.5146931324979667, -1.6555530342550746], [-5.983854028234914, 
2.1595187485129088, 4.9880917057687171, -2.7319846800207443]], [[-8.7222581114699267, 0.33078622872622354, -1.3396628196186677, 
3.9757882317613134], [2.6097475757453434, -2.9660172639762852, 0.50387532334868013, 1.1300124284832265], [6.2147310254354862, 
-1.7071845818808562, -1.8772588621443518, -4.6768531479173783]]], [[[4.8395857069509631, -4.4928367703974867, 
1.3157252961929418, -1.3547909587468272], [-8.6107019590344258, -8.539581561832394, -7.8798216588091918, -2.9964464159140913], 
[0.92673319232468376, 0.63843540201385185, -1.6142237465902092, -1.954212947385098]], [[3.1078668270268626, 1.3704411125456204, 
-6.1441136119372501, -5.0553543630596707], [1.8206011227166252, -4.1689263866635518, 3.9637675262377083, 4.1496436741532206], 
[-2.1754429092609362, -3.953893443322297, 2.3819163099320173, -8.7963871744638507]]], [[[1.1117012430221074, 
-5.2418988033702911, 6.4245677620458821, -3.0078352043458301], [1.0773035617007087, 4.2244532434544295, 1.1123337872275991, 
0.46680241044922699], [4.6848304331403501, -2.4154097656228517, -6.9730464629741915, 7.5722067351621671]], 
[[-0.90586754986760099, -0.75738033746873601, 1.941370489217654, -3.1088152116060677], [4.0472403356194828, 
-7.7167277698868242, -0.05174304881673919, 3.6505650825189058], [-2.7537649055160722, 4.5670875182901405, 7.1945902367017682, 
-1.5529069016376056]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_constData_rank0(self):
      arg0=-3.23475699401
      arg1=Data(3.49463105039,self.functionspace)
      res=arg0*arg1
      ref=Data(-11.3042822317,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_constData_rank1(self):
      arg0=0.234054691712
      arg1=Data(numpy.array([4.5632450024976787, 3.7652996189775418]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([1.0680489022643145, 0.8812860415217757]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_constData_rank2(self):
      arg0=1.00155446833
      arg1=Data(numpy.array([[-1.6944680673466039, 2.7756433358534691, 1.9453337159916329, -2.9240692732449016, 
-3.9071189755918789], [3.9465466526576147, 1.2663545198089405, 2.3673991427357475, 4.1119434449137504, -0.5464515133351755], 
[2.0851152997207318, 0.6093443773099807, -1.3292736489861898, -4.7901613392360272, -1.5095734935246008], [-2.0091399700623014, 
-0.20369782009324489, -3.7338694965279773, 3.8811435000018193, 4.6998119137951253]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[-1.6971020642912031, 2.779957985510682, 1.9483576756417971, -2.9286146463209399, 
-3.9131924682957044], [3.9526814344367112, 1.2683230278028266, 2.3710791897244037, 4.118335330768069, -0.547300954905798], 
[2.0883565454157313, 0.61029158384575011, -1.331339962773648, -4.7976074933269937, -1.5119200777100545], [-2.0122631145135883, 
-0.20401446190319492, -3.7396736784036428, 3.8871766146515183, 4.7071176225657325]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_constData_rank3(self):
      arg0=-4.72385170298
      arg1=Data(numpy.array([[[-1.4442884386020358, 0.45808240723052673], [-1.7492385861238171, 4.0856966990326331]], 
[[1.3434482679952513, -4.9172989473411013], [-3.259538150172796, -0.27010095510759768]], [[0.30329987966600847, 
4.7720159068001244], [-0.403027574411011, -2.8851561760137754]], [[1.4195119130432285, -1.5793356352231882], 
[-4.5920578555814826, 1.9376516860436768]], [[4.5954332550874692, -2.7829406956601055], [4.4472213908571394, 
-2.1042036075640338]], [[2.9891670201186944, -1.9220964988460665], [-3.8430431265172018, 
-3.6857735405661907]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[6.8226044002782986, -2.1639133594991184], [8.2631436739717472, -19.300225309567377]], 
[[-6.3462503886290822, 23.228591006437732], [15.397574841607929, 1.2759168567603805]], [[-1.4327436530725899, 
-22.542295467964756], [1.9038424937276079, 13.629049915413447]], [[-6.7055637678235067, 7.46054733001922], [21.692200321251391, 
-9.1531792168911021]], [[-21.708145207955969, 13.146199144474286], [-21.008014340710329, 9.939945794998911]], 
[[-14.120381718466405, 9.0796988193575654], [18.153965817807229, 17.411047616386266]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_constData_rank4(self):
      arg0=0.856652753582
      arg1=Data(numpy.array([[[[1.6890221885771268, -1.4889692704967716, -3.1993457630382318, 4.5351410560517582], 
[1.7801266997191467, 0.93550648573497952, 3.085521471695321, 0.5949338071944732], [-4.8820991453681923, -2.5088611723149867, 
-2.0503499788191801, 2.6873639363589614]], [[0.573068339216384, -3.4431743923871694, -0.012030408420637073, 
4.4237543603824427], [-2.1345022124570381, -0.61972380511868064, -0.89980308274234133, 1.9298812231176923], 
[-2.097412030746717, 3.0115553073755912, -1.9914727777356855, -4.9563071536489502]]], [[[-1.777756736094751, 
-1.2540216301241713, 2.4736351395004217, 1.6933868899265168], [2.0344484195872736, 0.40972542300757642, -3.9906065441258951, 
4.8307270956596753], [2.3536182844440319, -4.5838866447205344, -4.4235338504097683, -0.56334016968101253]], 
[[-2.6054566981240965, -1.035448831633119, 4.1787710981669086, 3.9976815216785315], [-3.4307831542090037, -0.57472270006044468, 
1.6605291602059999, -2.5348670068135659], [-2.3089657835567778, -0.31523736239473443, -2.5560605950919255, 
1.3795792136566378]]], [[[3.9481955756458671, 4.4302982057986, -1.4236120983463927, -3.0529917067667234], [0.19252099875043616, 
1.5545793006587267, -0.93574728952217523, 3.9992035784848508], [1.9031962272061289, 1.1314466917287902, 3.5737195723726671, 
0.73086082818570475]], [[1.8259501708466521, -0.70883363888823503, 1.5476841693662822, 0.63134602175176191], 
[-4.3584120023353137, 0.086972629168103666, 3.9962247229630385, 4.7008394894832932], [-4.3844154391235799, -2.9100649596443495, 
-1.9179846966121463, 2.6024530798533139]]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[[1.4469055087050089, -1.2755296255694393, -2.7407283575663128, 3.8850410735476846], 
[1.5249504390385256, 0.80140420699831205, 2.6432204649629347, 0.50965168413192763], [-4.1822636761380174, -2.1492228316175837, 
-1.7564379551614171, 2.3021377159577807]], [[0.49092057078014734, -2.9496048243001067, -0.010305882500249965, 
3.7896213539902104], [-1.8285271978273301, -0.530888104114982, -0.77081878851243513, 1.653238063869189], [-1.7967537915343417, 
2.5798571466265692, -1.7060006387300644, -4.2458341707695348]]], [[[-1.5229202031737987, -1.0742610824967527, 
2.1190463536092312, 1.4506445421345251], [1.7428158406591636, 0.35099241183182445, -3.4185640844861838, 4.1382556682980853], 
[2.0162335842489716, -3.9267991163057476, -3.7894324535149266, -0.48258690756036277]], [[-2.2319716547856197, 
-0.88702009281135752, 3.579755767831871, 3.4246248834881778], [-2.9389898359944957, -0.49233778355262947, 1.4224968774930047, 
-2.1715008013499788], [-1.9779818964096005, -0.27004895452724864, -2.1896563471069102, 1.1818203321628915]]], 
[[[3.3822326115557058, 3.7952271571849741, -1.2195412240805108, -2.6153537522634895], [0.16492364370184009, 1.3317346385702498, 
-0.80161049222568614, 3.4259287576424189], [1.6303782886422347, 0.96925692400025509, 3.0614367122014881, 0.62609394095020954]], 
[[1.5642052417585697, -0.60722428858486899, 1.3258279053622704, 0.5408443079964329], [-3.7336456430436229, 
0.074505342263087013, 3.4233769128571367, 4.0269870928109661], [-3.7559215587708761, -2.4929151607806475, -1.6430468716801567, 
2.2293985969232457]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_expandedData_rank0(self):
      arg0=-0.94732903411
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.48526234558)+(1.-msk_arg1)*(-1.45187062932)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.40703214323)+(1.-msk_ref)*(1.37539920093)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_expandedData_rank1(self):
      arg0=3.02549130194
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([2.5291822795556111, -2.3865264047014954])+(1.-msk_arg1)*numpy.array([0.47446829939657675, 
1.9957694934578427])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([7.6520189878239853, -7.2204148792817824])+(1.-msk_ref)*numpy.array([1.4354997128720517, 
6.0381832431399811])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_expandedData_rank2(self):
      arg0=0.988383851962
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-1.612763396646443, 0.046297288511465418, 2.5170497378583647, 2.174972605644677, 
-1.9471118063375403], [-3.6465874984640214, 4.3759259252328526, 3.2523008036321031, -1.9368846644186934, 2.642040840035035], 
[-4.1688497922376566, -0.82080176550561124, 0.541993731579268, 4.7309699407240799, 3.1477670544489804], [-4.9169083079513234, 
2.8517788953278416, 0.47679874501600583, -1.098488791157096, 
-4.4404974855801989]])+(1.-msk_arg1)*numpy.array([[1.4499430612195745, 0.72827312868459781, -2.8990239761189516, 
0.031519293842543483, -2.4180750278100902], [-4.4883562457664103, 3.0437282558352692, -3.3014220232218827, -2.7951712124092176, 
2.4627092680946125], [4.1012261966819317, 3.4043865167081737, -3.6184992142807126, -2.0925126825052542, 2.4269163080788054], 
[-1.6206458272382953, 4.083778339842473, 2.8847547387314201, -0.66728491119786426, 3.8286486425269501]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.5940292982801849, 0.045759492354342582, 2.4878113154835417, 2.1497078018781783, 
-1.9244938673479275], [-3.6042281982471103, 4.3250945218805459, 3.2145215960319065, -1.9143855254236191, 2.611350502513853], 
[-4.1204238159014288, -0.81126721068736873, 0.53569785215739196, 4.6760142935277011, 3.1112021263542977], [-4.8597927731552266, 
2.8186522095071052, 0.47126018020940608, -1.0857285827405601, 
-4.3889160094238315]])+(1.-msk_ref)*numpy.array([[1.4331003079732867, 0.71981340020945384, -2.8653484844456623, 
0.031153161059204619, -2.3899863103192396], [-4.4362188351667884, 3.0083718578270142, -3.2630722162631076, -2.762702089813371, 
2.4341020726610383], [4.0535857460425637, 3.3648406589503694, -3.5764661917310177, -2.0682057454131733, 2.398724888967505], 
[-1.6018201653913804, 4.0363405660911038, 2.8512450006320185, -0.65953363088564054, 3.784174493108575]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_expandedData_rank3(self):
      arg0=-2.35754675158
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.017828377633262171, 3.8652525940975107], [-2.109814138304861, -1.9694693717882594]], 
[[-1.8481071301971994, 4.5919685293189012], [2.9654656205360821, -1.6123354938903955]], [[1.2627745711908727, 
-2.1065624679942285], [0.38761785944965244, 0.88875085767776607]], [[-1.3889361982534911, -4.39624898959066], 
[4.7204053396719718, -3.0898040567824427]], [[-2.9401335165404818, 4.5491022547383189], [1.8057553856721595, 
-2.8747323771039146]], [[-4.1521193438149631, -3.2951968442900901], [2.3847639811237329, 
0.54955939294976375]]])+(1.-msk_arg1)*numpy.array([[[-4.5658343273709274, 4.8014409667002127], [-4.4533993643088312, 
3.9674298037320135]], [[4.075749511090871, -0.44903490311140271], [3.0147483088550384, -0.54564447092806923]], 
[[-1.6533191149707283, -2.9660961945416684], [3.3214378085748049, 3.1549527898016638]], [[-1.3844399661408402, 
3.6816720713943507], [4.7642773364933149, 3.2404948965201488]], [[3.4444186699311494, -4.1650430067619224], 
[-4.3138902060796411, -2.6507960285333052]], [[-4.5598482364565562, 2.5341132892158571], [-4.9308876642909807, 
-4.9991707166613439]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.042031233775285511, -9.1125136972608889], [4.9739854682037139, 4.6431161198008786]], 
[[4.3569989613730895, -10.825780489665405], [-6.9912238406247846, 3.8011563060826647]], [[-2.9770500882921804, 
4.9663195034256651], [-0.91382722540093742, -2.0952716974844865]], [[3.2744820223480349, 10.364362524557844], 
[-11.12857627469692, 7.2843575170942554]], [[6.9315022211392048, -10.724721243275505], [-4.2571527436442249, 
6.7773159773107228]], [[9.7888154711943347, 7.768580616081409], [-5.622192576989498, 
-1.2956119616504334]]])+(1.-msk_ref)*numpy.array([[[10.764167886757756, -11.319621553959811], [10.499097204826398, 
-9.3534012459204874]], [[-9.6087700201367436, 1.0586207771775049], [-7.1074100823803992, 1.2863823499554881]], 
[[3.8977771088286959, 6.9927104483232876], [-7.8304449161892338, -7.4379487009934433]], [[3.2638819449364931, 
-8.6797140323082154], [-11.232006558288521, -7.6396182168111419]], [[-8.1203780463867172, 9.8192836107934873], 
[10.170197842027147, 6.2493755661768091]], [[10.750055397567902, -5.9742905531331978], [11.624798195368024, 
11.78577868367192]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_float_rank0_expandedData_rank4(self):
      arg0=-4.66360593674
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[4.0471093329733847, -2.6137392139614866, 2.5656672599836137, -3.9960331537796123], 
[0.55112695365778563, 3.6954653684741281, 3.032850164405442, 4.0290745307542064], [4.7428247466148736, 2.5947179817041013, 
4.6351579242573244, -2.4464398261659079]], [[-2.6356848870257545, 4.9859318661628489, 4.9573004763083421, -2.7347298642964568], 
[-1.0468530871963679, 1.1455643138031508, 1.1680708417525212, 1.5490279033673202], [3.1884359445844854, -0.065033236539646211, 
-3.2266729900529634, 1.4803969150796874]]], [[[2.6962804651832233, -3.6852553493086173, -1.7632417973290271, 
-1.9781149909384519], [4.6805450498709789, 0.21182871197546493, -3.2831860106499922, -0.51154538861267618], 
[1.7146641490828118, 3.1763283266410482, 3.896437004197109, -2.4492846611006582]], [[0.553372570721546, 3.8601281364538558, 
2.7001519246985142, -3.6213633506162002], [4.9170776896990951, 2.8823357603613697, -0.34335788112885091, 4.268105408638748], 
[-1.2137609026816278, 2.134631283232431, -4.2843063700352602, 1.0305719600396612]]], [[[-2.8969015073328244, 
-2.221778037499039, 1.0555711727106312, -1.3723812242926572], [-2.2564386905514611, -3.2368272811816698, -3.8579316628804272, 
3.8310826918478877], [-0.71081261914128557, -0.52596899898970051, 4.4402447990921594, 2.3862372043805919]], 
[[3.2195223371002832, -3.6339799819335896, -2.6220312776303469, -0.89205245360850416], [4.4976039865878885, 
0.71512306989943042, -4.1106006923661775, 0.11710846458617041], [-1.0577724640955601, -1.7642175680906211, 3.244295968837088, 
4.4015326233453855]]]])+(1.-msk_arg1)*numpy.array([[[[-3.083271427811324, -3.7716581836726917, -3.2778240512371095, 
0.336439261617417], [-1.9140146353862777, 1.2958174940215201, -0.96796492469265782, 2.091773051589449], [-3.468202980334131, 
2.0418910461367297, -1.6656118010300371, 2.9887720867357874]], [[-2.740022744742785, 2.9401920581109682, 0.98230180100037945, 
0.60133482571629759], [-4.3896212984910488, 4.4922400074822875, -0.23678804536230658, 1.5026099707408509], 
[-1.2539723894030175, -3.507912676009477, 2.3890760824821653, -3.2010101632591548]]], [[[4.0070684780840473, 
-3.9876608718874405, 2.8026513887011966, -3.0394237076314212], [-2.9241223151690257, -2.3351554842193867, -3.0926155032457938, 
4.4010605205879099], [3.437171217736525, 1.8373248428576119, 4.2673992945910815, 2.8634599455455989]], [[1.0606364599956786, 
-3.8641693026059687, 0.28472729166020283, -3.693133169182258], [2.452201512254657, -2.627843738083814, -4.6609122099670763, 
-2.9656005454262981], [-3.5429710330241457, 1.5286824384597333, 1.8645508739718197, -0.032146236819356133]]], 
[[[0.35750071554841867, 4.2302752968227395, 1.3603054360980051, -0.06802653059371444], [0.1300416596143803, 1.4717375471788321, 
-1.3372789664822538, -3.5839303597761685], [-2.6030159239285444, 2.099391913821659, 0.75691926230141782, 1.8105526558755338]], 
[[-1.4058921992134832, -0.4720568114102397, -4.0175255006498745, -4.5462764994678571], [-4.2506419663111199, 
1.2746604014025689, -2.2497883176324054, 1.0510235988148677], [4.6972172138313919, 1.3407224074525548, -1.9359374842755575, 
-3.9124422333595179]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-18.874123111897841, 12.189449715325647, -11.965261065363659, 18.635923939383677], 
[-2.5702389329768747, -17.234194231439684, -14.144018031969578, -18.790015901200519], [-22.118665645239069, 
-12.100742183645959, -21.616550013302277, 11.409231297188917]], [[12.291795686513961, -23.252421451227207, -23.118895931524559, 
12.753702430518066], [4.8821102723454679, -5.3424605347719254, -5.4474221121320543, -7.2240557263225451], [-14.869608800085169, 
0.30328938801182814, 15.047931312335431, -6.903987841899883]]], [[[-12.574389584549435, 17.186578725445159, 8.223064913934941, 
9.2251488152985246], [-21.828217681765764, -0.9878856387411481, 15.311485770694945, 2.3856461112469702], [-7.9965179051809363, 
-14.813143641164354, -18.171446744914089, 11.422498486279668]], [[-2.5807116060470761, -18.002116493750279, 
-12.592444546128801, 16.888611621032904], [-22.931312705101377, -13.442078163704487, 1.6012858528595959, -19.904761722367471], 
[5.6605025515311311, -9.9550791252375426, 19.980316622117169, -4.8061815110806023]]], [[[13.510007067753643, 
10.361497245803074, -4.9227679877068082, 6.4002452250842223], [10.523140873149698, 15.09528692472667, 17.991873006553341, 
-17.866659985850582], [3.3149499505382907, 2.4529121462305117, -20.707552005633115, -11.128469992823495]], 
[[-15.014583484773731, 16.947450617746366, 12.228120632679584, 4.1601811185337132], [-20.975052652964884, -3.3350521942840081, 
19.170221792493876, -0.54614773068678168], [4.9330339432780619, 8.2276155242516094, -15.130117940816149, 
-20.527013672996269]]]])+(1.-msk_ref)*numpy.array([[[[14.379162935327271, 17.589527496736775, 15.286479704944457, 
-1.569020137832015], [8.9262100165981444, -6.043182158052649, 4.5142069693545128, -9.7552052217090761], [16.174332008911872, 
-9.5225752049431858, 7.7677570835908902, -13.938455247269211]], [[12.778386339190027, -13.711897137367416, -4.581068510817536, 
-2.804388663180124], [20.471463947691124, -20.950037168163444, 1.1042861341011407, -7.0075807801544618], [5.848033079730218, 
16.359522381409626, -11.14170940159168, 14.928250000946248]]], [[[-18.687388343323711, 18.596878915847267, -13.070461654964562, 
14.174674447183683], [13.636954188781457, 10.890244979420714, 14.422740020996828, -20.524811971773754], [-16.029612096634114, 
-8.5685590448739628, -19.901468684702756, -13.354048801668819]], [[-4.946390491560658, 18.020962900208634, -1.3278558877389373, 
17.223317772976454], [-11.43610153063805, 12.25522765775745, 21.736657853034821, 13.830392309654817], [16.523020743315652, 
-7.1291724953937505, -8.6955305252120993, 0.14991738087465725]]], [[[-1.6672424594210484, -19.728336988314727, 
-6.3439285075688066, 0.31724893193279469], [-0.60646305580138093, -6.863603962349023, 6.2365421271665831, 16.714038902721331], 
[12.139440516266614, -9.7907365928464287, -3.5299731653031192, -8.4437041147247811]], [[6.5565272066709923, 2.2014869481722004, 
18.736155775842345, 21.202042072988046], [19.823319109052395, -5.944513815310712, 10.492126154522843, -4.901559895088754], 
[-21.905970084589878, -6.2526009789184993, 9.0284495448284829, 18.246088826654812]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_float_rank0(self):
      arg0=Data(-1.85600455227,self.functionspace)
      arg1=-3.60193121332
      res=arg0*arg1
      ref=Data(6.68520072888,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_array_rank0(self):
      arg0=Data(-0.472794301953,self.functionspace)
      arg1=numpy.array(1.53185161265)
      res=arg0*arg1
      ref=Data(-0.724250713899,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_array_rank1(self):
      arg0=Data(3.12426331257,self.functionspace)
      arg1=numpy.array([1.9316991041329254, -3.6149244607002453])
      res=arg0*arg1
      ref=Data(numpy.array([6.0351366419756145, -11.2939758702941]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_array_rank2(self):
      arg0=Data(3.70068549842,self.functionspace)
      arg1=numpy.array([[-4.1531612468858681, 0.07577669309031787, 1.9333130464563464, -2.8373080062982692, 
2.3363606239811432], [-1.4163475172368125, -4.3901155588107423, 1.0984667919909743, -4.4483648040718897, 4.7193733435445093], 
[3.2562957418505309, -1.3132849130598001, 1.88797879137823, -3.7902043523743068, -1.2683716109639578], [1.5083165387415667, 
0.1378801471225497, -1.3209317454794069, 0.23086948352312753, -3.1040775784468533]])
      res=arg0*arg1
      ref=Data(numpy.array([[-15.369543598963864, 0.28042570923780696, 7.154583554933434, -10.499984593468126, 
8.6461358802540609], [-5.241456717766015, -16.246436984893101, 4.0650801276204831, -16.461999122125125, 17.464916494100308], 
[12.050526430443567, -4.8600544330584121, 6.9868157345840283, -14.0263542828922, -4.6938444273060265], [5.5818051419528327, 
0.51025106097688089, -4.8883529549025235, 0.8543753497024984, -11.487214880538961]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_array_rank3(self):
      arg0=Data(0.985797855893,self.functionspace)
      arg1=numpy.array([[[3.3065713637059098, -4.260473843035772], [2.2778284373303741, -2.8140317170410603]], 
[[2.4359420731917778, 1.4135654357688301], [-0.77540165337194722, 2.5675490972427371]], [[1.0720097875432764, 
0.63139049429458538], [-2.0009865558216431, -3.4809951990423271]], [[0.46545211942698028, 1.1123472964482026], 
[-1.3772022082798498, 4.3613589763126495]], [[3.370491727087515, 0.84286601892008584], [0.43081324466248017, 
4.429663520686498]], [[1.8581514482212933, 2.4683868200796324], [2.8781315164599679, 2.9068688818449715]]])
      res=arg0*arg1
      ref=Data(numpy.array([[[3.2596109606993156, -4.1999659795539515], [2.2454783896129618, -2.7740664330746863]], 
[[2.4013464728326199, 1.3934897757457245], [-0.76438928735014888, 2.5310843949625474]], [[1.0567849500567434, 
0.62242339550698345], [-1.9725682564002007, -3.4315576035906337]], [[0.45884170135208746, 1.0965495798472948], 
[-1.3576429840537292, 4.2994183276298026]], [[3.3226235178688195, 0.83089551425670261], [0.42469477287868834, 
4.3667528010213079]], [[1.8317617135814939, 2.4333304347496663], [2.837255877905033, 2.8655851110855908]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_array_rank4(self):
      arg0=Data(-0.392631257247,self.functionspace)
      arg1=numpy.array([[[[-1.7309889769654752, 3.7571508009665333, -3.3079333738532632, -2.2024171948041529], 
[-3.6496873714050215, 3.0909773289557556, -2.4126447316942148, -1.255339025475096], [0.61443257259309192, 0.14241652906163083, 
-4.2424834640669786, 1.9544853973360024]], [[-0.66978950613107457, -4.7852601479415355, -4.7957251445571405, 
-0.43772276625960505], [-0.41051891370808669, 2.8772768319138091, 2.4545229222345188, -0.42961311034789595], 
[-2.3110368164884418, 2.574149733569592, -4.0201757026683378, 3.1450991506927934]]], [[[-3.0448334096756255, 
-3.0995379800076197, 2.9637646434768623, 4.1374867541647049], [4.2191900624872787, 0.95074175466305011, 4.5806157999925485, 
-3.4427308261539635], [-2.1914830373213201, 0.63009024588470375, 1.4673055746586865, 1.0566315696942805]], 
[[-3.2892381736385268, 3.5877267555551757, 3.0533295740552777, -3.9198468794495165], [-3.0245306364948856, 3.2866846477126153, 
0.11278937441595804, 1.5133391805686287], [1.3221947631244371, -2.5164345747849639, 2.4741525592257272, -4.6888878586411629]]], 
[[[3.0820246726184877, -1.4520421381256234, -0.40797019270510049, -4.1692064792321561], [4.0087453682694871, 
-4.4363651695973596, 3.1653941351113453, -0.020071491100415173], [-3.7237932985658331, 3.31598017341096, 1.1700371657239597, 
-0.10707109615529564]], [[-3.3302617075418519, 1.0268702450027254, -4.2486012882694988, -3.4750855243807957], 
[-1.2937812462626583, 3.8554081771911797, 2.6501077245596161, 0.43819792445841443], [-0.64782521420781425, -4.3153248680435388, 
1.9111356535528294, -2.4977949274381617]]]])
      res=arg0*arg1
      ref=Data(numpy.array([[[[0.67964037830664104, -1.4751748426500375, 1.2987980394652947, 0.86473783217835054], 
[1.4329813411932275, -1.2136143147898513, 0.94727973429543422, 0.49288533984350219], [-0.2412454334707301, 
-0.055917180858221061, 1.6657316163461968, -0.76739205882692374]], [[0.26298029588308647, 1.8788427081402179, 
1.8829515929184883, 0.17186364004214047], [0.161182557212876, -1.1297088199619645, -0.96372242089850291, 0.16867953564568561], 
[0.90738529080194563, -1.0106916462334414, 1.5784466404924837, -1.2348642337029621]]], [[[1.1954967697485901, 
1.2169754939751976, -1.1636666381525071, -1.6245066261304693], [-1.6565858987984001, -0.37329093045056572, -1.798492940516516, 
1.3517237326358102], [0.86044474017892925, -0.24739312542077832, -0.57611003254376192, -0.41486658165592927]], 
[[1.2914577195004986, -1.4086536666923046, -1.1988326294507499, 1.5390544084939668], [1.1875252663890357, -1.2904551254057952, 
-0.04428463388103946, -0.59418426510779521], [-0.51913499217093806, 0.9880308708776232, -0.97142962994966309, 
1.8410039350284415]]], [[[-1.2100992220764495, 0.57011713026787569, 0.16018184968110172, 1.6369607816632317], 
[-1.5739587339267096, 1.7418556341457816, -1.2428326789510262, 0.0078806947855778447], [1.462077644543831, -1.3019574644924479, 
-0.45939316340390685, 0.04203945909826743]], [[1.3075648411936758, -0.40318135532494803, 1.6681336653544483, 
1.3644271984784584], [0.50797895732268927, -1.5137537598109112, -1.0405151277338105, -0.1720502020031302], 
[0.25435642833071681, 1.6943314283691497, -0.75037159442400148, 0.98071236270520756]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_constData_rank0(self):
      arg0=Data(-2.57166056386,self.functionspace)
      arg1=Data(-3.8564380433,self.functionspace)
      res=arg0*arg1
      ref=Data(9.91744963292,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_constData_rank1(self):
      arg0=Data(-4.37427685746,self.functionspace)
      arg1=Data(numpy.array([-0.27663628575409582, 3.0507545403780796]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([1.2100837027065965, -13.344844983753218]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_constData_rank2(self):
      arg0=Data(-3.54498316858,self.functionspace)
      arg1=Data(numpy.array([[-3.819238106636119, -4.9584086748306468, -0.39669126285551926, -2.5804813343806887, 
0.70667686858357648], [-1.2079047850529778, -4.6090467213850541, 2.4422041581119842, 1.5295705307649223, -3.5768637283210944], 
[4.0985354650729366, 3.2031742225227013, 0.9948082971836758, 1.2518900470649896, 2.9692927306252725], [4.1400063001575944, 
-3.127514904341282, 0.33951850460632826, 2.4451922899000955, -4.7939657971829552]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[13.539134804830343, 17.577475295223437, 1.4062638499461788, 9.1477628972184242, 
-2.5051576047547011], [4.2820021322619324, 16.338993050516034, -8.6575726347468809, -5.4223017867200118, 12.679921713208163], 
[-14.529239239518153, -11.355198704877298, -3.5265786694814123, -4.4379291457601644, -10.526092752658169], 
[-14.676252651880286, 11.086987695377809, -1.2035873842514142, -8.6681655116412397, 16.994528061769252]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_constData_rank3(self):
      arg0=Data(-0.506306286663,self.functionspace)
      arg1=Data(numpy.array([[[0.85613351332977761, 4.0517670557268399], [-1.7202086151490317, -2.836484095163827]], 
[[1.8151658022585355, 4.0559769147050311], [1.5022276298826185, 2.5778138051243573]], [[-4.1233258894853595, 
0.80920155736608379], [1.7558222860434149, 1.9687673809647332]], [[-0.90261252926173974, -0.0064011052240839916], 
[-4.103281689086117, -1.7827646882910289]], [[2.132876963042496, -1.6107353845071848], [1.669574161335067, 
0.84542621459204526]], [[-3.0680079229440227, -0.27458591329510895], [0.53930834851233023, 
3.7274168166622985]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[-0.43346578002137698, -2.0514351324067781], [0.87095243622106289, 1.4361297293998285]], 
[[-0.91902985701839834, -2.0535666104733994], [-0.7605872930077775, -1.3051633353800154]], [[2.087665819804958, 
-0.40970383567158802], [-0.88898386168602084, -0.99679930195864141]], [[0.45699839798561898, 0.0032409198165423243], 
[2.0775173151316975, 0.90262496932177949]], [[-1.0798890150661791, 0.81552545132583465], [-0.84531589393332718, 
-0.42804460735728889]], [[1.553351698917123, 0.13902457413029617], [-0.27305520730139943, 
-1.8872145672878944]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_constData_rank4(self):
      arg0=Data(4.9347766888,self.functionspace)
      arg1=Data(numpy.array([[[[-4.9584416227488406, -4.8370957112526938, -0.99759502000755518, -4.9245754179759071], 
[0.0050606428856569963, -2.8346356443470411, 2.0353287990860744, 0.85457373035679396], [3.6648810317565701, 
-2.2889211506602622, 2.0457844449499207, 1.3168675455996901]], [[-3.1500268467704826, 4.4851140718278018, 1.0766745787271823, 
2.3900600143250381], [4.48104441144101, 3.0595292948555315, -0.6181205938595884, 3.6596968521387971], [3.7657418411335488, 
2.922737258298965, 3.6584418249013257, -1.4474777527117286]]], [[[-0.43168146556013287, 1.7962375912503301, 
-2.1287664194494438, 0.8540310705223062], [2.540031553856716, -2.825389892659885, 4.3939087797307668, 1.0989176293619547], 
[-0.25625570394937114, 3.5864281194215959, 3.1439867459121196, -3.0535144242831236]], [[3.4930591057661946, 
-4.8663504165532432, -0.031744420896294478, 2.9814192560758741], [2.4418180812340857, 4.0040872084885937, 0.23636792170289134, 
0.35623734523111672], [2.5845362544003692, -2.3792280426931911, -4.706893929064897, -4.935207025484428]]], 
[[[2.4649989531762628, -4.2282958800401813, 0.96371432414231428, 1.5568856614892423], [3.4372928381540788, -0.2530813923182782, 
1.4938593012845915, -3.8246012862764536], [-0.17256193128881314, 1.3168030255804215, 0.94132820394504346, 1.8879389914127396]], 
[[-0.25992862481036472, 3.9025442200475986, 3.7893775548410034, 4.130464516124615], [1.3188795771956769, -3.2116554048809531, 
-4.7707687435125292, 0.42432208736361154], [3.7853258808671306, -3.0381514695294323, 2.1786078024080178, 
-3.0212787543006714]]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[[-24.468802132707399, -23.869987157375249, -4.9229086495943974, -24.301679974855862], 
[0.024973142542472293, -13.988293898960073, 10.043893111769473, 4.2171305234239735], [18.085369482730798, -11.295314736775277, 
10.09548938924471, 6.4984472662601718]], [[-15.544679052531286, 22.133036368256342, 5.3131486125244569, 11.794412443519747], 
[22.112953503048274, 15.098093842948087, -3.0502870974443592, 18.059786714002467], [18.583095053657622, 14.42305568973552, 
18.053593434847187, -7.1429794716357566]]], [[[-2.1302516332323607, 8.8640313928450496, -10.504986902595398, 
4.2144526183227971], [12.534488500783839, -13.94266817906388, 21.682958618920868, 5.4229131002846884], [-1.264564674220914, 
17.698221879771843, 15.514872503617447, -15.068411799861231]], [[17.237466647729093, -24.01435259513006, -0.15665162823843054, 
14.712638244417114], [12.049826945559769, 19.759276216364327, 1.1664229099990922, 1.7579517469258499], [12.754109259568601, 
-11.740959082417184, -23.227470437794938, -24.354144583753364]]], [[[12.164219372046039, -20.865695942163502, 
4.7557149814383468, 7.6828830694411847], [16.962272570295546, -1.2489001551808159, 7.3718620563234794, -18.873553271464424], 
[-0.8515545958980214, 6.4981288743731245, 4.6452444773362211, 9.3165573246966584]], [[-1.2826897184655455, 19.258184244094807, 
18.699732022684277, 20.382920008079637], [6.5083761928771731, -15.848802224459078, -23.542678383132419, 2.0939347452641179], 
[18.679737916407401, -14.992599048871654, 10.75094299735683, -14.909335967084035]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_expandedData_rank0(self):
      arg0=Data(0.963635327224,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.60595527547)+(1.-msk_arg1)*(-3.22520610761)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.54755523738)+(1.-msk_ref)*(-3.10792254287)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_expandedData_rank1(self):
      arg0=Data(-4.99123372751,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([2.5517225723895507, 4.0197440152949628])+(1.-msk_arg1)*numpy.array([4.9605322558538045, 
-2.6592711130092352])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-12.73624376656195, -20.063481905100861])+(1.-msk_ref)*numpy.array([-24.759175901823919, 
13.27304366984751])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_expandedData_rank2(self):
      arg0=Data(4.74781144317,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[3.781116052258735, -3.4748805631405322, -4.0849306694269512, 0.27053076235374007, 
3.985125710787651], [-2.216177995384867, 0.74858525141029642, -1.0997624749622181, 2.2462462029437349, 2.426137855022434], 
[2.8653080426050339, -3.912181831423843, -1.9736623061513638, -2.2056493778532706, -4.3222781445892693], [-4.6511246676309188, 
-1.3148104039135822, 4.579412974527779, 1.8804689508379022, 
-1.9373078278102884]])+(1.-msk_arg1)*numpy.array([[-3.9987678454816686, -4.8805509609695621, 3.5583629610759449, 
-0.88110464588550919, 1.7700237954376643], [3.4812061255492601, -0.89647002262449149, -3.5545124751293211, 
-0.90440583080880099, 3.2123280910083114], [-1.8106652939167378, -4.9146098690498299, -1.9860574908312714, -4.0696493615704963, 
4.1914747301353152], [-0.56407164168932944, 1.8198172999302233, -1.8643092203829914, -3.6550858024212274, 4.1998044757839086]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[17.952026060858262, -16.49807770131887, -19.394480576851066, 1.2844290492319088, 
18.920625452138541], [-10.521995246584234, 3.5541416228322089, -5.2214648633918062, 10.664753426507762, 11.518845070777312], 
[13.603942312879989, -18.574301666986024, -9.3705764820937603, -10.472007355786987, -20.521361635433628], [-22.08266292057661, 
-6.2424718812965594, 21.74218932345261, 8.9281120033093337, 
-9.1979722738156173]])+(1.-msk_ref)*numpy.array([[-18.985395735348028, -23.171935701453318, 16.894436385539684, 
-4.1833187203632489, 8.4037392306576741], [16.528110278907498, -4.2562706318731687, -16.876155004300546, -4.293948352781416, 
15.251528069697601], [-8.596697402204093, -23.333640974978604, -9.4294264817571989, -19.321927808543624, 19.900331687483767], 
[-2.678105795178864, 8.6401494010828568, -8.8513886501370074, -17.35365819849449, 19.939879749192833]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_expandedData_rank3(self):
      arg0=Data(4.36858925116,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[2.6280363666917497, 2.1582653282592528], [3.3938411955509338, 2.3382728905664063]], 
[[-1.735563229005944, -0.26480694999604903], [-3.0250166984583018, 4.5833471747319621]], [[-3.1279683001138814, 
1.5651507885141624], [4.1180115604160594, -0.15647795882544457]], [[0.78290110174876126, 4.6350793343221639], 
[4.4719535509891113, -3.3797132905920648]], [[4.5360952340349066, 3.6917973484817423], [2.252306002876078, 
-1.5147183737226975]], [[1.4696748353551436, -4.6710202394200362], [4.6966007555858731, 
-4.9654865070671335]]])+(1.-msk_arg1)*numpy.array([[[2.2063575086311538, -1.1406359837870617], [4.9613470059630593, 
4.7226215902896644]], [[-1.403550179106853, -1.7028254504991747], [3.7808203628089636, -3.0542239824229167]], 
[[-4.6455026114732014, 2.7688476274559983], [3.0866691773018324, -2.29103819833702]], [[-2.6362472464594431, 
-0.68479707199360362], [-4.4037715265919539, -4.5370447570592756]], [[2.3681263433035635, -2.5505321872015143], 
[-3.8785171693416487, 2.2253649155419186]], [[-0.031956037931431425, -4.983634623565818], [2.8213714151020568, 
-0.74556985950566812]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[11.480811423189968, 9.428574714186988], [14.826298167031441, 10.214953816009725]], 
[[-7.581962866945763, -1.1568327953854864], [-13.215055433467681, 20.022761201873504]], [[-13.664808693850063, 
6.8375009111492417], [17.989901038990617, -0.68358792896846143]], [[3.4201733378217969, 20.248757758198607], 
[19.536128214542604, -14.7645791532867]], [[19.816336881647846, 16.127946214042275], [9.8393997944899851, 
-6.6171824059811506]], [[6.4204056884343945, -20.405768809886172], [20.517519577847398, 
-21.6921709815588]]])+(1.-msk_ref)*numpy.array([[[9.6386696964245733, -4.9829700982596892], [21.674087201530373, 
20.631193916640623]], [[-6.1315342259113912, -7.438944959654199], [16.516851197538131, -13.342650060251108]], 
[[-20.294292774722503, 12.095957983407102], [13.484389789850965, -10.008604847254526]], [[-11.516681384285688, 
-2.9915971279378293], [-19.23826895563878, -19.820484957725835]], [[10.345371288747314, -11.142227497748866], 
[-16.943648416429582, 9.7217052499473855]], [[-0.13960280381694673, -21.771452648223772], [12.32541283754794, 
-3.2570884742261299]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_expandedData_rank4(self):
      arg0=Data(0.90909100865,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.087022777439285548, -2.5052632411116118, -3.9598545916170091, -4.156881249040695], 
[3.9976472331963091, 1.3541443579398269, -1.8384947933967144, -0.18854904356102509], [-1.310072560237642, 4.6350425982504042, 
3.9858059337594138, 3.5536866291373119]], [[-2.5892303284753879, 0.39373722896478291, -3.8870638333710894, 0.5338380851976634], 
[3.9679590931708884, 2.5638218765078191, 4.9631068477208409, -4.6474561763879105], [-4.8281417325519369, -0.41063957504264526, 
-3.0661751771463788, 4.8235263216914479]]], [[[-4.8149861655246209, -4.0522360645217761, -0.20867776371943414, 
0.13781763392141233], [-0.60915612331868196, -1.7302584694878931, 3.8299523983419235, 0.57700149296203573], 
[-4.226639402913543, 2.3137116023523072, 2.5605089068356062, -1.1759747529166198]], [[-1.8360074854759647, -4.6699980168592594, 
-4.9451809301657423, -1.8584297582084153], [0.9480155107903645, 0.05264870437493574, -2.6475552121017145, 3.141649333754998], 
[0.074911466633671431, 2.3984513589958665, -0.083872551334799539, -1.630912694745319]]], [[[0.61412297291533768, 
-0.58238148008190738, -3.6824916323454273, -1.4150916586702533], [-2.1132175796021011, -0.83705232126437146, 
-4.0395341158757176, -2.7527134837613088], [2.0163127950289113, 1.5758394028696543, 2.0612802398054715, 3.0697411527712326]], 
[[-1.1408495397702243, 2.7971877170952197, 1.4972057942195836, 4.4564133426220831], [-3.7312724226462493, -1.3256851827900218, 
-0.6089949029056152, -0.62671668180219342], [-0.62061267432050471, -1.0940918517834106, -1.5008569305028274, 
-0.9633358368786693]]]])+(1.-msk_arg1)*numpy.array([[[[3.5523200317055448, 3.7283176945299612, -0.75818264058281137, 
0.37002616110643682], [1.0264216692294683, 3.5431090339612528, -0.85389389874074872, 2.8903606280955385], [4.98151420971252, 
1.0394529606484229, 1.3489102702500331, -3.7915067066547126]], [[-0.33061333781284841, -3.9859845419024822, 0.5203816432514845, 
-3.2894528148548483], [-1.098265361607087, 0.55480644104083598, 1.5769754282134132, 1.7447524276864534], [1.39766618591596, 
3.2880916601356258, 3.2548703672391088, 3.3632855224483258]]], [[[0.79798028019371969, 2.1903887602493057, 1.7426126332149003, 
-2.1076310107028706], [4.5258728145518017, 1.7709703924315718, -3.2117548315982312, 1.4796092284113538], [1.2324822956198123, 
-2.4215351561138365, 4.891314400674327, 2.1926550098560105]], [[-4.7417465042273435, 4.8688570065742205, 1.2018931752635726, 
-2.2475663350852226], [0.10145701360847781, 4.7758053287581994, 0.90698137384416899, -3.299462132030162], [2.9893215127182673, 
-4.7546672631647144, -3.4035389122213866, 2.6801675545356343]]], [[[2.6953027177751778, 2.4137726274619906, 
-3.6892849314246847, -0.39197259080169555], [4.5097286615390377, -2.6931058471380496, -1.6194200012470503, 2.412974825838905], 
[-3.0298129658097195, -1.6357850353491599, 1.3480643903252121, -2.7349757934514587]], [[2.0438483148546984, 2.5309156314554802, 
3.242786870776925, 3.7566535054598535], [0.65349041619499459, -4.3280555238855953, -2.3085108294051224, -4.066411788264932], 
[2.9853327042907516, 0.73715409438654689, -3.3795250620505701, 4.7053611817296535]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.079111624517841039, -2.2775122867969735, -3.5998682048021005, -3.7789833675304196], 
[3.63422515545499, 1.2310404602177913, -1.6713590861275631, -0.17140824019096412], [-1.190975185191675, 4.2136755507811197, 
3.6234603366061711, 3.2306245621099468]], [[-2.3538460109419463, 0.35794297462281555, -3.5336947809678882, 0.4853074033283522], 
[3.6072359342943252, 2.3307474157145038, 4.5119158102343411, -4.2249606230511061], [-4.3892202375528226, -0.37330874546729786, 
-2.7874322844908792, 4.3850244090383246]]], [[[-4.3772606298545913, -3.6838513711857064, -0.18970707870261422, 
0.12528877183143095], [-0.55377835457335967, -1.5729624172526793, 3.4817752888917513, 0.52454686922965477], 
[-3.8423998779962782, 2.1033744143086364, 2.3277356247735632, -1.0690680742763974]], [[-1.6690978968610646, 
-4.2454532075420408, -4.495619519763193, -1.6894817833956428], [0.86183237692065473, 0.047862463764348065, -2.406868638227222, 
2.8560451616492486], [0.068101340761486578, 2.1804105651485206, -0.076247782291036972, -1.4826480666867952]]], 
[[[0.55829367288299836, -0.52943776714698521, -3.3477200323956331, -1.2864471033133353], [-1.9211071009382714, 
-0.76095673903140215, -3.6723041438792352, -2.5024670774781774], [1.8330118325875788, 1.4325814322258483, 1.8738913323159339, 
2.7906740808685004]], [[-1.0371360588280796, 2.5428982031186567, 1.3610963256243331, 4.0512853006074954], [-3.3920662102529717, 
-1.2051684799754963, -0.55363179054542977, -0.56974250039759977], [-0.56419340207926172, -0.99462906509398563, 
-1.3644155407907874, -0.87575994761712517]]]])+(1.-msk_ref)*numpy.array([[[[3.2293822006722825, 3.3893800934894478, 
-0.68925702146866619, 0.33638745602729314], [0.93311071058046424, 3.2210085654422476, -0.77626726568666615, 
2.6276008587588322], [4.5286497775139507, 0.94495734044053914, 1.2262821981605121, -3.4468246562575615]], 
[[-0.30055761274556414, -3.6236227076631065, 0.47307427294665461, -2.9904119773643547], [-0.99842316534920406, 
0.5043695470915629, 1.4336141826514583, 1.5861387443307455], [1.2706057627109244, 2.9891745638477274, 2.9589733851797617, 
3.0575326279819004]]], [[[0.72543669780445275, 1.9912627273915824, 1.5841934764162966, -1.9160284013827751], 
[4.1144302820044087, 1.6099732603456463, -2.9197774393954932, 1.3450994458649461], [1.1204385732687991, -2.2013958375539779, 
4.4466499421353447, 1.9933229545323954]], [[-4.3106791122926342, 4.4262341270812184, 1.0926302789904163, -2.043242346571351], 
[0.092233658835990398, 4.3416416834388381, 0.82452861197513849, -2.9995113576111625], [2.7175653091774463, -4.322425258067538, 
-3.0941266226922908, 2.4365162255049269]]], [[[2.4502754663204525, 2.1943389925521934, -3.3538957595076591, 
-0.3563387579352314], [4.0997537776582282, -2.4482783109770709, -1.472200162362344, 2.1936137182699595], [-2.7543757251100756, 
-1.4870774677208294, 1.2255132163264595, -2.4863419027032672]], [[1.8580441260797171, 2.3008326442089748, 2.9479883871929311, 
3.4151399244286309], [0.59408226160208988, -3.9345963617041742, -2.0986464383843186, -3.6967383941817218], [2.7139391193007629, 
0.67014015919665215, -3.072295847418923, 4.2776015427631391]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([-3.3809527968866284, 2.2382952522749786]),self.functionspace)
      arg1=1.44992404885
      res=arg0*arg1
      ref=Data(numpy.array([-4.9021247682252946, 3.245358114695438]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([1.7159488191529544, -0.32130328540983744]),self.functionspace)
      arg1=numpy.array(-2.33412641228)
      res=arg0*arg1
      ref=Data(numpy.array([-4.0052414609056122, 0.74996248482744521]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([0.65529406175334515, 3.9128865330900471]),self.functionspace)
      arg1=numpy.array([4.1905362530917039, 4.8620550986874633])
      res=arg0*arg1
      ref=Data(numpy.array([2.7460335222131067, 19.024669918795976]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([-1.4281402331228432, 2.5308314151749176]),self.functionspace)
      arg1=Data(-1.0207493852,self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([1.4577732649454369, -2.5833446110951162]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([1.0935242136409604, 1.1907158666826163]),self.functionspace)
      arg1=Data(numpy.array([0.17233966857505578, -3.115262301921542]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([0.1884576005576816, -3.7093922517761913]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([-4.6804981186201058, -1.7687879981864976]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-4.94997714701)+(1.-msk_arg1)*(0.17237435164)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([23.168358723802008, 8.7554601689321991])+(1.-msk_ref)*numpy.array([-0.80679782854759829, 
-0.30489368437533748])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([1.631784294227022, 2.0377383502254363]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([1.0435590423192664, -4.2108896405963012])+(1.-msk_arg1)*numpy.array([0.060469608821676957, 
3.8399434766022971])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.7028632553551712, -8.5806913092100867])+(1.-msk_ref)*numpy.array([0.098673357953264235, 
7.8248000849704908])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[-2.6080308734370226, -4.9217599554362126, 3.6918765555841286, -1.8206701004818662, 
-2.2227157448584856], [-3.865243783645381, -0.085849408641266223, 1.5268154809088745, -0.72480785939820969, 
-1.9208971652766396], [-4.2995056116368762, -1.7897855595137626, -2.1743369594057405, -2.3925559808574581, 
-4.8985438201448854], [-0.85362053684628236, -3.2181366588996796, -0.64827398696228045, -1.3126442438360373, 
1.5321712270145937]]),self.functionspace)
      arg1=1.30774127976
      res=arg0*arg1
      ref=Data(numpy.array([[-3.4106296320786851, -6.4363886627871869, 4.8280193715106634, -2.3809654472225237, 
-2.9067371327210081], [-5.0547388522037009, -0.11226881552305555, 1.9966796309591393, -0.9478611576285656, 
-2.5120365172036974], [-5.6226409708916432, -2.3405764580921367, -2.8434701979198644, -3.1288442203008211, 
-6.4060279643102556], [-1.1163148132836505, -4.208490152747796, -0.84777465334431579, -1.716599063302007, 
2.0036835612254951]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[-4.2330660421639461, 4.056504007746911, -3.7210692002938703, -0.75696341448108662, 
2.2648177049538631], [3.6692333682814091, 2.0720588933037041, -0.21374680353184772, 0.22793755145966976, 2.8088394462036446], 
[-2.0454840115873907, 0.7999756524279098, 2.621521547014293, 0.52719438608287383, 3.319438084414946], [-4.4899288705111911, 
2.8540951371997636, -3.5469043085339713, 4.1656244555337913, 2.7980872663366094]]),self.functionspace)
      arg1=numpy.array(-4.24823373402)
      res=arg0*arg1
      ref=Data(numpy.array([[17.983053958639271, -17.232977167882296, 15.807971703297063, 3.2157575128146307, 
-9.6214749755821263], [-15.58776097311093, -8.8025904894010463, 0.90804638130212623, -0.96833199536002013, 
-11.932606488797674], [8.6896941804163141, -3.3984835530358564, -11.136836270476426, -2.2396449753412195, -14.101748848189652], 
[19.074267291038847, -12.124863241943599, 15.068078534841382, -17.696546335241472, -11.886928715572326]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[-4.5838752322798886, 0.78240678256477292, -4.2201222680178105, -4.966078370247403, 
-0.2938183719125762], [-0.38033068029637107, 2.8658421643336283, -4.1020900321822147, -3.9096981725136457, 0.1986057974872768], 
[1.6854669762252108, 2.1615586132931259, -3.6027983029539845, -3.9793296999853576, -3.8355120744992712], [0.55965309059870716, 
3.4979818537050615, -3.7332699661685966, 3.7873702293996807, -2.3487295741255743]]),self.functionspace)
      arg1=numpy.array([[-3.9218923253460467, -1.0891040193770873, 3.436568442148193, 1.8099906122248726, -3.9706676835907695], 
[4.5270728476475135, -2.1381393302450933, -2.8381856668570302, -1.7960839555372701, -2.0474050857407677], [1.6715238517932693, 
3.6242141743895679, -3.3684639033952579, -3.6854558418379426, -3.7169848186347041], [-1.864205175681235, -1.2865392913885043, 
1.7603783125482364, -0.49682763998244006, -2.9426956847600652]])
      res=arg0*arg1
      ref=Data(numpy.array([[17.977465093822321, -0.85212237167918903, -14.502739008276865, -8.9885552297207933, 
1.1666551141985202], [-1.7217846958970087, -6.1275698458364527, 11.642493133496655, 7.0221461586451452, -0.40662651983305154], 
[2.8172982521703189, 7.8339513650708055, 12.13589603471419, 14.665643889410264, 14.256540152603892], [-1.0433081880801089, 
-4.500291095355557, -6.5719674833308854, -1.8816702128123961, 6.9115963824476729]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[-2.3062888729093354, 3.7301589869234029, 3.9953834954852585, -1.5820133422197746, 
-2.8116781173730154], [3.7175115930347538, -3.1155386715472577, 4.1796888000714674, 1.4264160483460362, -0.6740262863062938], 
[-4.1183953926153771, -3.0599789148870835, 3.5418908302385628, -0.013145956482913945, 0.73007331593485425], 
[-1.451976305523436, 1.5695393326416527, -1.2951813271673238, -4.1298062859324016, -3.2811559515699464]]),self.functionspace)
      arg1=Data(-1.82996218094,self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[4.2204214157389108, -6.8260498749504963, -7.3114006950761796, 2.8950245859992507, 
5.1452646197595522], [-6.8029056224468603, 5.7013179421768436, -7.6486724322149371, -2.6102874227542361, 1.2334426128976421], 
[7.5365078146295561, 5.5996456887066861, -6.4815262683425985, 0.024056603195970441, -1.3360065574717392], [2.6570617267238901, 
-2.8721976202266473, 2.3701328461714368, 7.5573893178504132, 6.0043913011279448]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-3.798549390021809, -0.056562649484224181, 3.8249828263522758, -1.6912861542415989, 
3.4071209506631348], [1.5996222592036453, 3.4180286359976115, -2.1824732497059141, -0.58011504337826736, 1.6049802047517012], 
[-3.3239902268002872, -4.7439622033210762, 2.8103610180536789, 2.7492954884059584, 1.1262180345134896], [4.2361905041154841, 
0.47841118412905104, -1.7130188200780472, 0.76887137888783741, 2.8800216580548348]]),self.functionspace)
      arg1=Data(numpy.array([[-2.051663721715463, -4.0523705418269707, 3.4748895130512736, -1.8380739630991636, 
0.49845942288662481], [-4.4966529488796692, -0.32632754949408316, -3.9979069633130795, -2.4994709759920672, 
4.8926548732844317], [-4.1535759929480065, -4.9770958332542881, -2.5694575189062241, -4.8235996322468342, 4.604385071476699], 
[-0.5712558094861544, -2.6834156482198046, 4.8421998775593007, -0.13484482926113461, 3.0489581558452787]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[7.7933459786521464, 0.22921281453755457, 13.291392710892744, 3.108709044261599, 
1.6983115427724746], [-7.1929461489416298, -1.1153969088857041, 8.7253250022437996, 1.4499807136603584, 7.8526142203034563], 
[13.806446006831472, 23.611154515265159, -7.2211032486789755, -13.261500706812861, 5.1855415053417406], [-2.4199484355660514, 
-1.2837760577752617, -8.2947795208386985, -0.10367832980990357, 8.7810655233373307]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[-0.26772004113564662, 2.9011741422402402, 3.8293513454756347, 3.3582823345338433, 
-1.7409439726863352], [1.5979539074300062, 4.3223317398990897, 3.0146935410733118, -1.8323573672778091, 0.0031101440361469557], 
[-3.9077738055678681, 4.1209590774490739, 0.45330041783262942, -3.0098539357716536, -4.8255206004448779], [-4.8858896026140908, 
0.38661274635902299, 0.87886624492448462, -0.42341626499197105, -4.6585337640933311]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-4.94992898332)+(1.-msk_arg1)*(-3.67846975279)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[1.325195191034189, -14.360605972346825, -18.955017212302863, -16.623259061896043, 
8.6175490287443459], [-7.9097583604045685, -21.395235154870313, -14.92251893480023, 9.0700388400967711, -0.015394992106837941], 
[19.343202820457027, -20.398454776559742, -2.24380487638288, 14.898563232249701, 23.885984279771986], [24.184806553303758, 
-1.9137056385252467, -4.3503254982173551, 2.0958804420948032, 
23.059411298681784]])+(1.-msk_ref)*numpy.array([[0.98480007353387899, -10.671881329814873, -14.086153097147944, 
-12.353339988920617, 6.4040097448333499], [-5.8780251148380973, -15.899566566754411, -11.089459004777531, 6.7402711518381855, 
-0.011440570763795014], [14.374627744536813, -15.158823318892704, -1.6674518759255954, 11.071656663059926, 17.750531570214289], 
[17.972597118700012, -1.4221432935257612, -3.2328828987051699, 1.5575239236134011, 17.136275543580588]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-1.334930766150888, -0.90271606808551219, -2.6710291315641053, -3.7925253230358402, 
1.2044870082149197], [4.6397913684472343, 4.9974708258186844, 4.0102701849378519, 1.1517447928121296, 2.5293019932660812], 
[4.3877975180748692, 0.91069788038889232, 3.9700675717862275, 3.4655388333217729, -3.4032415450777909], [0.67852259706501972, 
-4.4486138278632197, -3.5386535761588056, -4.6009383287485681, 3.3573412257313056]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-4.6280617233374901, -4.7444849805688873, -1.0489540765987426, -0.6968203430660056, 
4.3248949779478814], [1.9052619927635384, -2.3455322527876787, 1.9698296751173583, 1.9526307470657809, 1.1563328876632148], 
[1.8855938445497245, -0.33614335686843155, -3.5605695093083503, 2.7404890255742567, 2.978107525024587], [-4.3061192583513304, 
1.9818724410077024, 4.3457956400226401, 0.067141990647288985, 
3.2075092907904299]])+(1.-msk_arg1)*numpy.array([[-4.763843048437419, -1.3250859961736658, -2.6120849791617982, 
1.7746981458966715, -2.1560640618799312], [-3.497939927649989, -2.2728519391733726, -4.5998114791774682, -4.0410425901655147, 
4.3556909459309185], [1.0729703933718593, 4.5973695352921222, -0.76860792001985256, 4.790182274884943, 4.051480713724402], 
[-1.909016400590108, -2.7318752780860542, -3.9994788851631591, 2.261147364580097, -3.0325622530713625]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[6.1781419821285146, 4.282922826749914, 2.8017868962681676, 2.6427087966843481, 
5.2092798128321744], [8.8400181486548419, -11.721729004323199, 7.8995492155289568, 2.2489322952178719, 2.9247150776456929], 
[8.2736039912125321, -0.30612504260688761, -14.135701545995882, 9.4972711404197323, -10.135219254872471], [-2.9217992224482412, 
-8.816585146127899, -15.378265282821461, -0.30891615823758978, 
10.768703173886893]])+(1.-msk_ref)*numpy.array([[6.3594006504731455, 1.1961764203410656, 6.9769550734621815, 
-6.730587659057881, -2.596951151413466], [-16.229711483657361, -11.358511257424352, -18.446486831280279, -4.6542497607551727, 
11.016857791594095], [4.7079768290048598, 4.1868146911550026, -3.051425378688879, 16.600562692303402, -13.788167484028305], 
[-1.2953107659681162, 12.1530581380913, 14.152770259754245, -10.403399576645381, -10.181346271833098]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[1.2435108444530085, 1.2561476447142859], [-1.3698888796920561, 2.1328864444169557]], 
[[-0.71051288960470327, -1.0493989092522202], [-4.966276551416728, 0.32128513568838457]], [[-0.34836999129637647, 
3.4060650595444315], [1.7907946529154568, 1.5838141069302578]], [[2.3745767863395839, -0.044445475736337947], 
[1.4555723476389106, 0.44397034531019663]], [[3.938468676619669, 2.4604502668612227], [1.4997706269249713, 
1.1950059000132649]], [[-0.92669503150260368, -1.8540370149942009], [-1.2553967354270643, 
-2.7789380280926048]]]),self.functionspace)
      arg1=0.0912995398586
      res=arg0*arg1
      ref=Data(numpy.array([[[0.11353196790772756, 0.1146857019568671], [-0.12507022437328538, 0.1947315509458942]], 
[[-0.064869499884507242, -0.095809637542834961], [-0.4534187639548572, 0.029333185051754473]], [[-0.031806019905900512, 
0.31097217266483079], [0.16349872779240632, 0.14460149918427773]], [[0.21679776795169567, -0.0040578514835238238], 
[0.13289308557032153, 0.040534288237680689]], [[0.35958037792284947, 0.22463797720937698], [0.13692836813168038, 
0.10910348879951244]], [[-0.084606829965430175, -0.16927272634976606], [-0.11461714428446823, 
-0.25371576326039497]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[-0.59731997407524684, -2.3175888749902795], [1.1181030510620147, -0.90290543557081637]], 
[[0.89678741558722663, -2.8954013215089249], [-0.019228264187455935, -4.3099468115559851]], [[-1.8431136042383498, 
-4.4807244637194259], [-3.4719568883197338, 3.0915325840981271]], [[-2.5714202410235898, -0.94934759785817224], 
[-1.2325522579139445, -4.1001852307316149]], [[-1.147282329029192, 1.2681992896553727], [-4.1449918338281675, 
0.17412155149786557]], [[-2.4149110382817907, -4.818924194777475], [3.2644916431885118, 
-4.4214396216885294]]]),self.functionspace)
      arg1=numpy.array(-3.2080536049)
      res=arg0*arg1
      ref=Data(numpy.array([[[1.9162344961091857, 7.434949345082166], [-3.5869345236060317, 2.8965690374642161]], 
[[-2.8769421014010277, 9.2886026470907659], [0.061685302242483357, 13.826540405727284]], [[5.9128072423118727, 
14.374404268386087], [11.138223811621717, -9.9178022510730912]], [[8.2492539739213022, 3.0455579836093878], 
[3.9540937142249883, 13.153614010194733]], [[3.6805432114769321, -4.0684513029069613], [13.297355994781825, 
-0.55859127097301764]], [[7.7471640618658899, 15.459367134782118], [-10.472664184087623, 
14.184215317193111]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[2.8472111898698085, 0.42294360276680187], [-2.6380739355006302, 3.8982746808303101]], 
[[2.6933673384846148, 3.7423173757067154], [-3.4871833095127669, 4.9973556978098888]], [[2.5768765639587778, 
1.0937632328380333], [2.864874149849558, -0.17930750341904744]], [[-0.35212867338956233, 4.4564282749802846], 
[4.7092446034000126, 2.618127835923203]], [[3.5871011673473632, 4.846290199377453], [4.9424997381524953, -4.9359004820788162]], 
[[2.3596241253558938, 3.9013064244580686], [-2.0365909568578298, 2.9947964319276501]]]),self.functionspace)
      arg1=numpy.array([[[1.9611017136673947, -3.0143518226768817], [-4.2609006343671405, -0.72192881250176733]], 
[[0.31499002659341802, 4.3447182792728967], [3.1165642258927395, -2.7708218280368357]], [[-3.4864145058178639, 
-0.79900384198602126], [-2.4954362289560206, -2.2440347174570139]], [[-4.4673669439580719, -0.49812556774025651], 
[-2.8758427422571406, 2.6266252419169209]], [[-1.8675109754020216, 3.9415572256218283], [-2.3689295484423911, 
0.49653596928805399]], [[2.5168607812498731, 1.2678444284813297], [-3.9500402931720755, -0.63209538429622203]]])
      res=arg0*arg1
      ref=Data(numpy.array([[[5.583670743626663, -1.2749008198896363], [11.240570905282054, -2.8142768111375318]], 
[[0.84838384957511237, 16.259314709073543], [-10.868030751557738, -13.846782249955893]], [[-8.9840598322879774, 
-0.87392102526063975], [-7.1491107449341662, 0.40237226277288468]], [[1.5730879955203392, -2.219860864568286], 
[-13.543046914201533, 6.876840660401208]], [[-6.6989507998986042, 19.10193015281645], [-11.708433672878227, 
-2.4508521301783781]], [[5.9388454195992839, 4.9462496140475798], [8.0446163402982993, 
-1.8929970015282624]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-0.85422679858122219, -1.214136293164084], [-3.2877063893783642, -2.3586097037754614]], 
[[0.0071580270379989486, -1.2993981273988142], [-0.25819335370279717, -1.6795200202648966]], [[4.3310115452404876, 
-2.1673247391718533], [1.5275354323075172, -0.057001027912387592]], [[3.4770047487386648, 1.598358111042808], 
[4.2279518919500436, 1.5296618283718288]], [[1.7484300523405931, -3.1713826231417839], [2.8589859378118039, 
0.59716134573183055]], [[0.40301989261961424, 2.5946246072630874], [0.07607320859171196, 
-3.6435296578921115]]]),self.functionspace)
      arg1=Data(-2.06862529886,self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[1.7670751665072508, 2.5115930523004106], [6.8010326122838567, 4.8790797033609472]], 
[[-0.01480727582071135, 2.6879678396253768], [0.53410530346649454, 3.4742976038577926]], [[-8.9592400521288162, 
4.4833827862908429], [-3.1598984401727095, 0.11791376840045313]], [[-7.1926199874888113, -3.3064040251373963], 
[-8.7460482460407096, -3.164297156866736]], [[-3.6168466395546668, 6.560402326588469], [-5.914170640035616, 
-1.2353030672807139]], [[-0.83369714581580701, -5.3673061036228846], [-0.15736696385808663, 
7.537097627453595]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[-3.5738179852227936, -2.9310167491632386], [-0.45126910514282947, -0.11251013045603298]], 
[[0.82483684734873286, 1.3598075416545186], [3.3355470912758101, -4.4432841090096158]], [[-2.2940322229451815, 
4.1111599530644121], [-0.95394219305364203, 2.4735876256007572]], [[-3.6776818844385795, -2.2795160389990174], 
[-2.722441151642041, 2.9494590216267262]], [[1.3547782915486897, 2.7404085437372858], [-0.54056327457897257, 
4.5371161989882101]], [[1.1900905297817719, 1.4275128489407098], [2.582875261271564, 
-0.77676573885301003]]]),self.functionspace)
      arg1=Data(numpy.array([[[4.3437118425284531, -4.105902155708292], [3.9743886156804766, -0.75942048100498649]], 
[[-2.3583136955498443, 3.330514086845481], [1.7796316910635213, 2.4933825866446577]], [[1.2725997284098423, 2.113983031896808], 
[3.0233499827270212, -4.3345672363362926]], [[-1.9039344485400256, -0.077971799478543602], [4.1480606768559358, 
0.3847928847049209]], [[0.043485722372976099, -2.2936571067250657], [0.21664132845511741, 0.80688733624898124]], 
[[1.7914164061879543, -0.93553786958889518], [3.8370711224853906, -2.2524479407561451]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[-15.523635505453425, 12.034467988806451], [-1.7935187940879775, 0.085442497388854344]], 
[[-1.9452240336966728, 4.5288581728790973], [5.9360453106691793, -11.078807224919499]], [[-2.9193847838834648, 
8.6909223821918449], [-2.8841011128913054, -10.721931878135926]], [[7.0020652305542086, 0.17773796750095536], 
[-11.292851086180738, 1.1349308452507016]], [[0.058913512663221188, -6.2855575316731134], [-0.11710834591883702, 
3.6609416040536993]], [[2.1319476998999805, -1.335492329508766], [9.9106760780070271, 1.7496243889293881]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[-0.013460237718527779, 3.4351175277317232], [4.3358123712664103, 2.7329793971804293]], 
[[3.8680708860143724, 2.1844476248555624], [-1.8949983059187869, 0.49645175093273686]], [[-2.1951461554359355, 
-4.7613868377036521], [3.702391276597405, -3.7984656127103191]], [[4.3662944143643205, -2.5702614275697577], 
[-0.1860143944719459, 2.6204024253935918]], [[-4.694078796857843, 2.8520950107954226], [1.1358648212646072, 
-1.8709182758402845]], [[4.9726681920435816, 4.8300882729729473], [-0.78417789728664999, 
-0.87644261913405686]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.17838984187)+(1.-msk_arg1)*(-2.9205117552)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.015861407396629449, 4.0479076002986698], [5.1092772545420164, 3.220515159669473]], 
[[4.5580954397011881, 2.5741308912204963], [-2.2330467540500512, 0.58501370027626987]], [[-2.5867379309793122, 
-5.610769882749671], [4.3628602709597262, -4.4760732926993088]], [[5.1451969844879493, -3.0287699571910189], 
[-0.21919747288679964, 3.0878555996877051]], [[-5.5314547711409947, 3.3608797887611468], [1.3384915671123958, 
-2.2046710912136898]], [[5.8597416844797658, 5.6917269561926984], [-0.9240672683792861, 
-1.0327910793669632]]])+(1.-msk_ref)*numpy.array([[[0.039310782484801701, -10.03230112024807], [-12.662790998642821, 
-7.9816984561959989]], [[-11.296746492567628, -6.3797049670182986], [5.5343648285276315, -1.4498931744907053]], 
[[6.4109501513416882, 13.905686230587488], [-10.812877245667755, 11.093463473658948]], [[-12.751814163832904, 
7.5064787131650927], [0.54325722569248647, -7.6529160867272621]], [[13.709112306077547, -8.3295770059869341], 
[-3.3173065628260647, 5.4640388176176966]], [[-14.52273590959269, -14.106329579890858], [2.2902007671968776, 
2.559660971942864]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[-2.2375959653073152, -2.1685972371959039], [0.17083460520765836, 3.0477119840998323]], 
[[-1.3048613310884862, 1.3699152068824096], [-4.3155307168922041, 1.6925060708279949]], [[0.7817496238282402, 
-0.75047330609129936], [-1.775665806028667, 1.3679337026947644]], [[-1.1613647547681536, -0.6631665599781833], 
[-3.7390816730175693, 0.52327737238700145]], [[-2.1612777146720852, -0.29325897187941763], [0.35759444733202805, 
3.5092443403032405]], [[-1.6133880284184587, -1.0109015009806956], [3.5511757268317741, 
4.7579445601606629]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-3.334030830619521, -1.4787595063801007], [3.5831236197296317, -3.9612995130914053]], 
[[-0.57658783822266191, -1.6310751958854963], [1.5472193848821902, -4.4976998512501538]], [[-4.2439073496401312, 
-3.3979835820762174], [-1.1394404570917684, 1.1429977526712687]], [[-3.3370368846434575, -3.1236021275552028], 
[0.10941043140337925, 2.8750604874768735]], [[1.5672253201486672, -1.1960899415186299], [2.7519415577111204, 
2.4031935435279728]], [[-1.9526772482111268, -4.9846716170170158], [3.074341644550902, 
4.145041038391561]]])+(1.-msk_arg1)*numpy.array([[[-0.49067355988012906, 2.3813006264580983], [4.4759749301000067, 
2.4418012137073353]], [[3.0636149520377796, 3.0742241920607434], [-4.4578778046796499, 0.65692069970931133]], 
[[4.0524016750491985, 0.49986279974124415], [0.89158806381093392, 1.7103354959703401]], [[-1.6828249058425317, 
3.6488954640895717], [-3.0142363747314329, -0.63580463677963195]], [[-3.8321526841944156, -2.41672926297825], 
[4.405301887616158, 2.4468803495794234]], [[-4.0070389888152835, 3.8753219730436523], [3.4867046454082899, 
-3.6720136734468012]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[7.460213934804437, 3.2068337800130648], [0.61212150898674744, -12.072899998657507]], 
[[0.75236717407265541, -2.2344347144122465], [-6.6770727812301534, -7.6123843030030551]], [[-3.3176729741430764, 
2.5500959728846948], [2.0232654576635278, 1.563545147983403]], [[3.8755170231862324, 2.0714684776713184], 
[-0.40909453889732128, 1.5044540973405898]], [[-3.3872091583071384, 0.35076410652506618], [0.98407902041974848, 
8.433393341278828]], [[3.1504260956289314, 5.0390120195383723], [10.917527424117241, 
19.721875460257834]]])+(1.-msk_ref)*numpy.array([[[1.0979291778707541, -5.1640819594699074], [0.76465141010301085, 
7.4419068218053619]], [[-3.9975926842586058, 4.2114264700698021], [19.238108598247013, 1.1118422723105836]], 
[[3.1679634850706413, -0.37513368791386459], [-1.5831624379723805, 2.3396255678529938]], [[1.9543735340915531, 
-2.419825452640278], [11.27047598690122, -0.33270217968551768]], [[8.282346195570204, 0.70872753897190433], 
[1.5753114938328401, 8.5867010181608059]], [[6.4649087339605851, -3.9175687993332993], [12.381900903405507, 
-17.47123748241178]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-3.3781067806545262, 3.7294901421598112, -2.0534751175127952, -1.9472611098628292], 
[-1.2781449049086548, 1.7353702540102169, -2.9383689617786835, 1.3452789148415603], [-4.7923078794106253, -1.9045600330242851, 
-2.4961201185036108, 3.4522973027014121]], [[-3.8909405017577905, -0.60580201830319869, -2.0627078901394613, 
-0.20830395652566658], [-0.069728814420745522, -0.0065691468792463681, 3.7466795036787612, 2.5567140037907157], 
[1.5651139683055151, 3.6241111603366214, -1.2799374349771853, -3.6563623735056372]]], [[[-1.4657691760235703, 
-1.4879644280703355, -0.76778769068237374, -1.1545443463984126], [3.8275469209039272, 4.0696673746222025, -2.036610626794749, 
0.087842922892712672], [-3.1558081397534044, 3.0250776323158526, 2.9110076624908077, 0.3755914260385298]], 
[[3.5402566617217879, 2.5093520955669, -0.96888187811112392, 1.959745995378892], [-3.2838648583218255, -0.71981038939357234, 
0.84286730633247942, -2.373676441188465], [0.74400863425084918, -1.6096149607546231, 3.0820883038719575, 
-3.9147420302979077]]], [[[1.0241990210434295, -3.7238826060666517, -2.4460410564984847, -0.58987849319115249], 
[1.725874378720369, 0.96967144559891505, -3.3154546433560697, 0.27095043635282146], [-0.14008572081085635, 4.7240151480336916, 
-4.077862946817481, 1.9073391583413377]], [[1.0302012506955149, -1.753811475451589, 4.5321748186829556, -1.3805713707716425], 
[-2.7291977982706692, 3.9344302476977617, 4.2586916638592918, 3.9030484602518154], [1.4857825028129632, 2.1190758682317243, 
4.1012687290103109, -2.388359833150945]]]]),self.functionspace)
      arg1=4.13558512292
      res=arg0*arg1
      ref=Data(numpy.array([[[[-13.970448145707845, 15.423623947990494, -8.4923211462709833, -8.0530640763881411], 
[-5.2858770536754029, 7.1767714052414302, -12.151874963979905, 5.5635154663958462], [-19.81899717053977, -7.8764701382820226, 
-10.322917227103222, 14.277269364946566]], [[-16.091315653233877, -2.5053458143292255, -8.5305040633891203, 
-0.86145874365278619], [-0.28836944755723953, -0.027167266104083369, 15.494712015760745, 10.573508397636456], 
[6.4726620429975581, 14.987820198494122, -5.2932902143592022, -15.121197835872003]]], [[[-6.0618131979968339, 
-6.1536035521608818, -3.1752513511466298, -4.7747164227159224], [15.829146103366055, 16.830455849717836, -8.42257660935182, 
0.36328188506885428], [-13.051113193551974, 12.510466051881536, 12.038719981701222, 1.5532903138210079]], [[14.641032781532754, 
10.377639194592971, -4.0068934809825247, 8.1046963831897223], [-13.580702653753407, -2.9768371376988432, 3.48574949266371, 
-9.816540976803168], [3.0769110391313572, -6.6566996853252354, 12.746238537016605, -16.189748900567125]]], 
[[[4.2356622343357717, -15.40043350514739, -10.115811003305067, -2.4394927207714145], [7.137500404663637, 4.010158804538575, 
-13.711344898777247, 1.1205385936292351], [-0.57933642291881138, 19.536566766653795, -16.864349336162441, 7.8879634475979543]], 
[[4.2604849659892814, -7.2530366462828297, 18.743194754614944, -5.7094704220915808], [-11.28682981203243, 16.271171199542763, 
17.612181888157149, 16.141389146250692], [6.144580014527171, 8.7636186349965275, 16.9611459407894, 
-9.8772653941571935]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[2.4128204758779912, -3.9627903547107559, 2.8975206353688119, -3.8451938839783195], 
[1.0765616767032533, -1.4889235070955076, -0.27660397985663288, -0.37338080335057011], [-3.6198411565654878, 
-3.2803248224587334, -4.9986158901829238, 4.4929731229991408]], [[2.2297928384146539, -3.6485199130514112, 
-0.98929153385105106, -3.0512761469794611], [-4.9759816334677467, 3.5895045146401134, 4.9720024166580092, 1.8327389417116429], 
[2.3047226714454734, -2.6308520299784002, -3.546097781023211, 4.5716055495853052]]], [[[2.7872260206268873, 
-2.4896827529710364, -4.2309309450553751, -3.6049812347499675], [0.63550056858115767, 0.14898263313006233, 4.2236946839117397, 
-2.8848173047058969], [-2.106681421575388, 3.362269248926756, -1.3821526862434808, -2.6816033384684967]], 
[[-4.5378358426757694, 3.3042571718796054, -0.48589990866177679, 0.43582230335550953], [0.78035590004330135, 
1.1946023321841102, 1.1962867438795852, -2.9821124717899505], [-3.1054696320959341, 0.49871195550319491, -2.3707609045341216, 
-2.7379039594107422]]], [[[3.7180391047071808, 0.51438937285260256, 2.4219566654743554, -3.1629445794823652], 
[1.8026295457773598, 3.3669267464107051, -4.4747766868623131, -1.634277882265879], [2.9241273298507977, 1.6096432012375743, 
1.1624032235338451, 2.1816205343517829]], [[-2.9960506418113431, -2.6237509165639841, -3.6398118315527341, 1.8585353307023258], 
[-0.99960793028969697, -1.8359522199455256, -0.29634563497398769, 3.2122699634626901], [-3.2844943791083092, 
1.1157826519560885, -2.7990494565914368, -4.9450474568872025]]]]),self.functionspace)
      arg1=numpy.array(3.36865090515)
      res=arg0*arg1
      ref=Data(numpy.array([[[[8.1279498800285133, -13.349257315312242, 9.760735511023146, -12.953115857737087], 
[3.6265604666751736, -5.0156635198749528, -0.93178224711187096, -1.2577895811721707], [-12.193981188560048, -11.05026918235845, 
-16.838591942957041, 15.135257977601327]], [[7.5113936634203533, -12.290589907554903, -3.3325778209636172, 
-10.278684154382011], [-16.762345033586083, 12.091787632278955, 16.748940441278176, 6.1738576948988042], [7.7638061132822891, 
-8.8624220720999052, -11.945565499790813, 15.400143172594875]]], [[[9.3891914572396988, -8.3868720593298143, 
-14.252529357683839, -12.143923299485753], [2.140779565573641, 0.50187048194507056, 14.228152920032436, -9.7179424246871093], 
[-7.09667427765058, 11.326311348751872, -4.6559898975682676, -9.0333855133825676]], [[-15.28638481884745, 11.130888912897413, 
-1.6368271671253263, 1.4681331966826732], [2.6287466090192546, 4.0241982276051473, 4.0298724225877525, -10.045695877351434], 
[-10.4612430870728, 1.6799864803144813, -7.9862658669508066, -9.2230426510801156]]], [[[12.52477579545134, 1.7327982264589625, 
8.158726513381918, -10.654856120609493], [6.0724296510312916, 11.342000831866907, -15.073960536538516, -5.5053116673599964], 
[9.8503641764729117, 5.4223260268159397, 3.9157306711054396, 7.3491179877358483]], [[-10.092648706410118, -8.8385008999688672, 
-12.261255420932274, 6.2607567240218449], [-3.3673301591645379, -6.184682107529869, -0.99828499149208827, 10.821016120001639], 
[-11.064314963140113, 3.7586822404614648, -9.4290204855036492, -16.658138591647994]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[1.716980176149975, -2.0655591299060063, -4.0384421404980264, 4.3563199745288603], 
[1.8888675690497836, 0.63456448972028312, -2.9203661556330927, 1.554015532261297], [4.958902254936131, 1.6709715436079779, 
4.5028066715101325, -1.1685510890627104]], [[-3.483737975715929, -3.655825214754044, -2.0852129340877736, -3.2463730850482522], 
[-3.8377019610366405, -2.4606628253242291, 0.33015726496961761, -1.3857872492735392], [0.8887663215633177, -3.0578663543103723, 
1.4965381395678055, 3.4785219470494138]]], [[[2.0370265649996169, 4.2985784266381657, -1.2231433839744774, 
-1.2172140334766612], [3.0272457299802049, -4.0565244975978345, -3.0553554261443794, 3.1218562010453947], [-1.0490371192137049, 
0.71966149425637305, -4.0324849751588232, 0.12743580993255321]], [[3.7616161216022022, -1.6788777817196801, 
-3.6518229348271278, -0.075016302303524185], [-1.7208486873815665, 2.3421185037699699, -3.0753942172888316, 
0.42454911848383325], [-0.71667665114964674, 3.2034835282628986, 2.5088318502725002, -4.3105729599354436]]], 
[[[2.3695977949857872, -3.915260040087861, -1.7293427954801475, -1.8915903070651732], [-2.8396661631569442, 
-1.8054890692941683, 4.3763585070120463, -0.33031978675914342], [4.4704773830677382, 3.121051794558058, -4.2139826560695095, 
-2.5894125883378205]], [[2.2971139283697077, 4.3020924388383293, -1.1131300846897787, 0.55958395761253943], 
[-0.909268190493985, 2.4981263415548804, 0.96325653438015735, -2.4307982772483827], [4.2737679543777034, 1.349365491229781, 
1.076921874500762, -0.62204012870677428]]]]),self.functionspace)
      arg1=numpy.array([[[[-2.9886891624902789, -3.0102484888051526, -3.4989596484727867, -4.8920697812221778], 
[0.84539010281810612, 2.2416831667060677, -0.42238999276978006, -2.8531808901103362], [3.650319605817625, 1.1328992217918294, 
1.8765161398359904, -0.51761875528120882]], [[-3.7425752967809545, -2.363712775395248, -3.0265170228691476, 
-1.3842205907877747], [-1.4146156000683963, 3.1197738956759018, -3.3097466405136275, 1.7741955214119001], [-2.133337584754905, 
-4.4308801742344155, -0.26037044404706933, -3.9112571017911324]]], [[[1.9859242418333887, 2.023036505089129, 
3.4065556380122661, -2.6859619180638603], [4.7741734411825636, -1.1193855240652284, -4.8606832811103517, -3.355085788853053], 
[4.0259408338333387, -3.6742132295328069, -2.2327870168491026, 1.2316655057255224]], [[3.5104808818706186, 
-0.038047212531669317, 3.8703557857628574, 2.5719686859450306], [2.9616880433488699, 0.30347518557501196, -3.6289681862063441, 
-0.51277192051759091], [-2.5419727841977346, -0.77947400880359474, 2.8653340567924452, -2.9099636739864931]]], 
[[[0.10624553810858028, -1.2635884514203455, 3.1693364994091624, -4.9233952975615001], [0.76853184910960692, 
3.2212861027279729, -1.0628426857614137, -0.13180534827878265], [-0.92813591456169409, -3.5778061996008459, 4.3112206391508412, 
-2.6680171912724306]], [[-2.4475470387563156, -2.5082306683733258, -3.8661676956790316, 1.3607477033207473], 
[-0.75994863479333397, 2.022192420267789, 1.4301626177707858, -1.7925402479780095], [-0.822903625416183, 1.7737095013539772, 
1.0449651388487382, -3.9838025918555942]]]])
      res=arg0*arg1
      ref=Data(numpy.array([[[[-5.1315200446700802, 6.2178462493372413, 14.130346092294664, -21.311421304727205], 
[1.5968299484087827, 1.4224925347953843, 1.2335334393629724, -4.4338874195825753], [18.10157812452659, 1.8930423613897702, 
8.4495893936499389, 0.60486396020314115]], [[13.038151688372125, 8.6413207647262098, 6.310932441323569, 4.4936964697030222], 
[5.4288730624955086, -7.6767116485066413, -1.0927368985743595, -2.4586575312908296], [-1.8960385978553893, 13.5490394047723, 
-0.3896542999326445, -13.605393669133337]]], [[[4.0453804366913362, 8.6961810770776022, -4.1667059907756583, 
3.2693905400512211], [14.452596164004817, 4.5408148006269897, 14.851115037709778, -10.474095374970183], [-4.2233613744493468, 
-2.6441897829821142, 9.003680098173696, 0.15695829128811969]], [[13.205081479820835, 0.063876619775786195, -14.133854024389672, 
-0.19293958046005028], [-5.0966169818305831, 0.71077484757026099, 11.160507774584131, -0.21769686683900544], 
[1.8217725422923761, -2.4970321479113653, 7.1886413433513994, 12.543610727480576]]], [[[0.25175919282917025, 
4.9472773709625804, -5.4808692417055056, 9.3130468227175882], [-2.1823738872249892, -5.8159968475445663, -4.6513806294474938, 
0.043537914537162117], [-4.1492106144609444, -11.166518459845166, -18.167408999870549, 6.9085973009825468]], 
[[-5.6222943930671656, -10.790640193271294, 4.3035475745160872, 0.76145258513639746], [0.69099712002690905, 5.0516921527635814, 
1.3776134867939407, 4.3573037466833338], [-3.5168991438449164, 2.3933823925934394, 1.1253458161169323, 
2.4780850769802347]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[4.9561610245272973, -1.6187251559071489, -2.0034359493652332, 4.4714283672556814], 
[-2.8123676938036168, 4.3226939655626602, -4.9302565170788384, -4.4869486373454128], [-3.6696645708585693, 
-0.34089116715665302, 4.987568902517598, -2.9494539392864327]], [[2.6447183903979123, 3.1256748914252377, -1.0679120212021909, 
-0.59681699433394275], [-2.3129265369386101, -4.474479507725329, 0.62512637336610855, -1.0536692650206003], 
[0.62713664742925346, 3.707554447756543, -1.6885086426396798, -4.1708195405054056]]], [[[-0.70992823628553658, 
-1.4349269196143553, -3.755012042397782, -3.8732706862231083], [-1.0586402244578852, -1.1200166871051485, -0.38947577238073539, 
-0.35689742791315737], [2.8798346505933505, 1.4134759384775926, -1.1416396757513914, -4.3231427646561116]], 
[[4.3275247121065359, -2.45972678682688, -1.5704496548756284, -3.7488715732574516], [3.3941346786123088, -3.4097690257402102, 
1.3613120811391211, 0.58481815909558232], [-2.5146690683254427, -4.4783065328327574, 4.0507645023654995, 
-0.60996115103388338]]], [[[3.4253569862054505, 3.0986343741157096, 2.0537293589677788, -2.7896592776330809], 
[2.7897851455986045, 1.3208850366649925, 1.5481455139807112, -1.9476656449735765], [-1.1324975841971154, -1.966882670693455, 
-4.0843563672210266, -4.0967205749373345]], [[1.1515910770165751, 1.1017304943269091, 1.1176850301988051, -3.5223710286572563], 
[-2.2847929931861568, 0.0024177397657325628, -3.4444042155497847, 3.3810802387678827], [0.81519941042823874, 
4.1016836106188741, 0.50785099830697344, -4.2300908037952523]]]]),self.functionspace)
      arg1=Data(4.89510738828,self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[[24.260940448668965, -7.9238334702757847, -9.8070341176835143, 21.888122036718077], 
[-13.766841876498079, 21.160051168099162, -24.134135102868257, -21.964095425502421], [-17.963402153319151, -1.6686988709479258, 
24.414685384269482, -14.437893769592577]], [[12.946180532756818, 15.300514254376978, -5.2275440250198759, -2.9214832784151481], 
[-11.322023779517071, -21.903057696973729, 3.0600607288731219, -5.1578242040059017], [3.0699012362920901, 18.148877169663439, 
-8.2654311317601366, -20.416609547910618]]], [[[-3.4751749545899218, -7.0241213658460966, -18.381187191821766, 
-18.960075952939096], [-5.1821575842741945, -5.4826019600453044, -1.9065257309369985, -1.7470512362358264], 
[14.097099875144272, 6.9191165095976741, -5.5884488115242226, -21.162248087857371]], [[21.183698191196996, -12.040626767346492, 
-7.6875197085034692, -18.351128935965431], [16.614653742092486, -16.691185550229211, 6.6637688261389387, 2.8627476913890955], 
[-12.309575135439067, -21.921791395852235, 19.828927243711728, -2.9858253369897372]]], [[[16.767490290670825, 
15.168148018312193, 10.053225758610729, -13.65568174072555], [13.656297877933532, 6.4658741020473078, 7.5783385436195223, 
-9.5340324886092915], [-5.5436972916125544, -9.6281018931914364, -19.993363029552121, -20.053887154094447]], 
[[5.6371619893811635, 5.3930890826730327, 5.4711882490961292, -17.24238444664357], [-11.184307061635939, 0.011835095790175831, 
-16.860728523760539, 16.550750857160178], [3.9904886569087741, 20.078181746727449, 2.4859851739578409, 
-20.706748746753437]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[1.1256866879042384, -0.30740512255976871, 2.9240552038604992, -2.2207522634075518], 
[-2.2177281849057771, -3.4949929099880372, -1.3416631092939637, -2.8284104828918935], [-4.6569722837655236, 
-1.0851890283450274, 1.2768834837915275, -0.76561211648324523]], [[-4.0767060541697493, -4.8580262299633645, 
0.77101909907139188, 0.25030825860412165], [-0.6180758063928824, -1.8305022222443066, -1.0409195609497202, 
-4.1059863422512333], [-3.7780515962731265, -1.2555275725347794, -4.0305342535490549, -3.2196693053405068]]], 
[[[-2.3274226192821148, 1.1978635781714804, 3.7448911392471018, 2.5671641460157426], [2.5417757800972005, -1.1965338954738245, 
-1.5515889853545959, 0.025600817783150376], [-4.8001520666785957, 3.5766974955648028, 0.85246354056670359, 
-2.6361051710815797]], [[1.935008418295566, -4.0320132532706392, -2.3449109731655637, -0.45991664548240774], 
[3.9601688599317182, 4.6979639352231395, -1.2864851189647708, -0.30721207900106595], [3.554788292559337, -4.4779157754975927, 
4.5374674812906797, -4.2530505122172286]]], [[[-3.5651683863512251, -1.0021813465423337, 4.4189386076161572, 
-1.4073116563591515], [0.53666300253736487, 3.3503541562934451, -2.8625608509583023, -4.2829373883567365], [3.8070943383084135, 
-3.6752171401173737, 2.0828488140482158, 3.4712239618997422]], [[-0.49996163464808596, 0.39076854271950179, 
-0.95051601707701749, -3.2096074694750723], [2.0317217417995534, 3.7152280383418361, 0.52671755097332262, -3.687737952881156], 
[3.2926661634575733, 2.8069839861530363, -0.8216100317073689, -4.1947805364675173]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[-2.9032919727696473, 1.1240440295750602, 4.958998642852853, -2.4974295038029082], 
[-4.3168437215698869, -1.4084516752196929, 3.3183228595441712, 3.820085368417967], [3.7147359014730679, 1.8741621545583467, 
-3.5111142400038498, -3.455118308852275]], [[-3.6228477399489001, 4.9381889768494993, 2.1176525257560073, -1.5720350756410562], 
[0.55821313079533397, -3.519312979375111, 2.3530798672186828, -1.2518770214395758], [3.2785924001070335, -2.1779224339017986, 
4.3196067783028376, 4.0727874149657488]]], [[[-3.5491130933269899, -3.7713629346577582, 3.1392669533081854, 
1.8353499618628302], [3.3965430212491299, 2.8091364068341349, -1.2954539942458387, 2.9934412445106737], [3.6800537622827392, 
-1.0580726809088938, 2.0847147501884811, 3.1787742357166309]], [[3.4031375370968533, -1.9708348652856591, -1.9969974946283511, 
-1.2282126250074352], [3.7606280919785267, -1.0779322083861476, 4.877845173111476, -3.6637708572247742], [0.31741572530303497, 
0.28593766823763467, 2.8503147651290668, -1.4144246752674059]]], [[[-0.6957210987618323, 0.38163856113793138, 
-0.20381672978332954, -3.2282257634541045], [-1.7955337979213759, 3.7869152619038786, -0.92895056055230274, 3.170577048539819], 
[1.8930404417516762, 2.7876732137606206, -1.0716727822959826, 1.9369096665775141]], [[4.4319155394566128, -3.091218549422817, 
0.63140366899809042, 0.95221610380590072], [-2.4480519813960431, -3.0483841352072627, 3.9382295642661429, 0.31254268880067659], 
[3.2885328396748186, -3.2866059152163194, 1.5415163427748046, -0.055461901123829094]]]]),self.functionspace)
      res=arg0*arg1
      ref=Data(numpy.array([[[[-3.2681971248460266, -0.34553689267409765, 14.500385787571037, 5.5461722232711077], 
[9.573585991159085, 4.9225286189536002, -4.4520713653772699, -10.804769501575318], [-17.299422134668813, -2.0338202074661953, 
-4.4832837827661569, 2.6452804411404012]], [[14.769285314784874, -23.989851578050818, 1.6327505425546542, 
-0.39349336224831144], [-0.34501803095542155, 6.442110229519372, -2.4493668622648972, 5.1401899522090524], 
[-12.386691250753319, 2.7344416666057638, -17.410323081812265, -13.113028627142331]]], [[[8.2602860917995518, 
-4.5175782994924374, 11.756212997175069, 4.7116446174856179], [8.6332507874692102, -3.3612269277865896, 2.0100121485054596, 
0.076634543845284647], [-17.664817672509834, -3.7844059079323773, 1.7771433170173037, -8.3795832004735065]], 
[[6.5850997829000502, 7.9464322968396326, 4.6827813385381596, 0.56487543043256194], [14.892722263637795, -5.0640866396135547, 
-6.2752752278220498, 1.1255546620315404], [1.1283457041814593, -1.2804047954103013, 12.933210558215821, 6.0156195896387281]]], 
[[[2.480362867023223, -0.38247104709369079, -0.90065361611762484, 4.5431197462678821], [-0.9635965591498038, 
12.687507287250741, 2.6591775071127914, -13.579382983856943], [7.2069835479816646, -10.245304376259115, -2.2321323836529392, 
6.7234472466591075]], [[-2.2157877377289816, -1.2079509677854465, -0.60015930062388045, -3.0562399193298697], 
[-4.9737604356578169, -11.325442210758453, 2.0743346312609985, -1.1525755353857794], [10.828040808616324, -9.2254501728080527, 
-1.2665252912646345, 0.2326505033497242]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[-4.2888085111850804, 2.5151868398439099, -2.4967639288030021, -4.5511268117767099], 
[3.5292317896798444, -1.1404201136359795, 1.7845253418065603, 3.874764651053189], [2.4557544576455399, -4.224921158025249, 
-2.2998706380428438, -1.5720951366793221]], [[-3.0370271907688462, -0.62946615229281999, -4.360867748972705, 
-0.38538611957684576], [-0.0044413709152628655, 2.9881044928758316, -0.38642648045972727, -2.1256432236313003], 
[-4.8812630549416456, -0.18276059814620993, -1.5285255939842726, -4.61611703961335]]], [[[-1.6864763576428565, 
-2.6185456729661305, -4.0262929508328344, 1.5354666502061658], [-2.5055321096903693, 4.1643034616106984, 4.8740869382484391, 
3.9549535436515377], [4.3202472248445414, -2.6009984896451366, -0.49574383258217924, 3.1333793953961795]], 
[[3.8217305219230759, 3.4458702708688538, 2.4031861018409959, 1.7187784572846976], [-2.315287044273393, -1.1622998725576372, 
-0.7763908102266539, -4.1175403217425472], [2.6746046790807902, 0.44323343503583246, -0.026587787281692421, 
-0.85580979850894145]]], [[[2.9776214505665983, 2.5019086173847036, -2.8884259123458831, 2.9772062920859614], 
[-2.054814362354259, -2.6012515867309549, -0.222051477590818, -2.3233068056427761], [4.3041853420017482, 4.3714976426437619, 
4.797009702819885, 0.13885935952158146]], [[0.63328040769233773, 1.0744081071133404, -2.9410180584489396, -4.1294183534165709], 
[1.7920601846048339, 2.1742338487296937, -3.4330977810779029, 2.5527088987492537], [2.223500247696613, -2.4828701397105735, 
-3.4855072358360917, 1.438964584079681]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.14422325626)+(1.-msk_arg1)*(2.24072240633)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-13.484971462512974, 7.9083089556757411, -7.8503832103329145, -14.309758763775717], 
[11.096692669842659, -3.5857354432006576, 5.610946081093104, 12.183125128374714], [7.7214402773927073, -13.284095360926951, 
-7.2313067465233072, -4.9430180898000078]], [[-9.5490915231086859, -1.9791821150674394, -13.711541793793176, 
-1.2117399998132274], [-0.013964661721445247, 9.3952676386344987, -1.2150111266960864, -6.6834968582525232], 
[-15.347780817269138, -0.5746401230192596, -4.8060257203936301, -14.514102549569301]]], [[[-5.3026581848329402, 
-8.2332922025185002, -12.659563932523376, 4.8278499507895134], [-7.8779523285940662, 13.093499790119425, 15.325217504272723, 
12.435256909376157], [13.583821797147943, -8.1781199406387781, -1.5587292875522383, 9.8520443656898475]], [[12.016373986188327, 
10.834585443720007, 7.5561536305287218, 5.4042231977528372], [-7.2797793695213482, -3.6545302900434908, -2.4411460414610118, 
-12.946466038210257], [8.4095542332670217, 1.3936248743915687, -0.083597939103585067, -2.6908570714068025]]], 
[[[9.3623066132094532, 7.8665592598177145, -9.0818559275812731, 9.3610012622596006], [-6.4607951054108534, -8.1789157343820982, 
-0.69817941992789534, -7.3049952897286161], [13.533319651574313, 13.744964552685282, 15.082869468110054, 0.43660482755709307]], 
[[1.9911749855999175, 3.378178957099804, -9.2472173764551151, -12.983813221638313], [5.6346373090516968, 6.8362766317230914, 
-10.794425884278958, 8.0262866859086728], [6.9911811891070519, -7.8066980355509319, -10.95921291077755, 
4.5244259101975013]]]])+(1.-msk_ref)*numpy.array([[[[-9.6100293274739297, 5.6358355081461839, -5.5945548785869859, 
-10.197811821200167], [7.9080287482699845, -2.555364901254265, 3.9986259180507897, 8.6822719728727744], [5.502664037692691, 
-9.4668755037675361, -5.1533716703245274, -3.5226287976407749]], [[-6.8051348749911291, -1.4104589114692518, 
-9.771494076167766, -0.86354331322465461], [-0.0099518793246546906, 6.6955126896441062, -0.86587447316559707, 
-4.7629763990555292], [-10.937555498401657, -0.40951576726060118, -3.4250015470903978, -10.343436880906259]]], 
[[[-3.7789253623172212, -5.8674339614153324, -9.0218048293822104, 3.440554527290395], [-5.6142019379640695, 9.3310480731913046, 
10.921475812936746, 8.8619530212567348], [9.6804747575968957, -5.8281155945799901, -1.1108243134671107, 7.0210334187989485]], 
[[8.5634372114306974, 7.7212387252444454, 5.3848729449774879, 3.8513054007562193], [-5.1879155571904141, -2.6043913673151358, 
-1.739676284544057, -9.2262648578983661], [5.9930466324930762, 0.99316308911968243, -0.059575850696840821, 
-1.9176321910762888]]], [[[6.672023101855296, 5.606082697565598, -6.4721606608194184, 6.6710928469455544], 
[-4.6042685825771787, -5.8286827148911602, -0.49755572119657016, -5.205885616184216], [9.6444845368231924, 9.7953127170934167, 
10.748767124493963, 0.31114527820872839]], [[1.4190055990064188, 2.4074503191521437, -6.5900050609895517, -9.2528802296134547], 
[4.01550940913706, 4.8718545014511108, -7.6926191211852322, 5.7199120262670453], [4.982246825495511, -5.5634227540587489, 
-7.8100541607654774, 3.2243201854635801]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[-4.2640949513516198, 2.7258474397086232, -1.3013855969698715, -1.5171018042908124], 
[0.24829539405478229, 1.2543706090155959, -3.3696745356776701, -0.78427968112267443], [0.060548837202238914, 
-4.2359755864084283, 2.1963647579200511, 2.4826268389986055]], [[-1.0225892930604816, 1.1942835350114365, 0.54380691359011113, 
-2.5696904893502213], [4.2753256682077065, 3.3832401376388361, -4.5551333197076573, -1.7176169250200091], [-0.0529960624677841, 
-4.2152540037586297, -2.0748154811817376, 4.1738220608704957]]], [[[-1.3691758840450676, 3.0196255525339097, 
1.5558101252162757, 0.64843754930816377], [-2.9698103898096715, 3.2609770424443951, -3.6291910159835696, -0.1213500234225906], 
[3.4927671729823633, -3.3207906191318335, 1.5354294273257709, 1.9994670711579241]], [[-3.0115629354864035, -1.7918808939683117, 
-0.74825011697827115, 1.8522393599139715], [4.0531448996589177, 4.7343489496274209, 4.9970290112486779, 3.7467988437056636], 
[-3.8922240585132495, 3.0591821914223711, -1.8934444947071238, -3.7138644481936725]]], [[[-0.60718632045923648, 
-3.942849663784501, 3.5075635664688427, 2.0765930373476387], [-0.85703387159873667, -1.3864448574732933, -0.065968554454190809, 
-0.7363535181419758], [3.1610679455045325, 3.2596307061884637, 1.7941983091170588, -1.941604747389325]], [[3.9937269730021221, 
-1.0324114606464398, -3.7933232758544633, 4.306527545444121], [-0.38688918105478454, 0.72391531871721249, 1.4228848724327232, 
-0.63969888287971433], [4.8891344582448113, -0.36939220066352796, 1.4768378310050521, 
1.2771271747256776]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.4190862102991586, 1.1118290461154885, 1.7379222641638101, -2.8343235705991399], 
[-0.20519350706433759, -1.5984639522165214, -1.4269500208982899, 1.0970259985217563], [4.3764047594961166, 0.94218827968076901, 
-2.2198854743873464, 1.115342683539148]], [[-2.0715913373656791, 4.9577046671701623, -4.6929822285989342, 2.5466299378751796], 
[0.77238065093561037, 4.5825945404805406, -0.6806646878575604, -3.0311063849443221], [3.4238871056974816, 2.877452790700028, 
-4.0816362788214153, -1.033634640139419]]], [[[-4.9772032872343193, -0.40913543832239352, 2.6309523566337649, 
-2.3241667422024115], [3.9576099652953047, -2.496298228747047, 1.553627052213705, -1.1020992319326028], [2.3001509349546447, 
3.9163497610181892, -2.7593836531652691, 1.0224479634875241]], [[1.9385857510467899, -4.614282847481002, -1.7067924496043476, 
0.32966983649145476], [2.4433902039818527, -1.4287282705646542, 2.1408479368798563, 3.8448214881981961], [3.5831171798172754, 
1.9084051523277745, 4.8837742514853684, 3.1178061675645274]]], [[[-2.5976297803115846, 4.5942466501524208, 2.8898855753045991, 
0.1968088665122778], [-0.89431510417347582, 0.88753654548890193, -3.0814641340191917, 3.3181252461269146], 
[-2.0273075277158648, 3.9852456810526053, -1.5358538877608385, -3.3627923605726053]], [[1.4700574494426704, 
-0.085516138275087528, -0.28821653227604305, 4.3030099159733908], [2.5253337014537376, -3.0889090239950292, 
-1.1230195299768351, -0.67667987398122875], [4.487099739315676, 4.7700051980409377, -2.7576471973180281, 
-2.3307755335169533]]]])+(1.-msk_arg1)*numpy.array([[[[-2.8956611280288982, -3.9393343677724566, -4.1890727094442459, 
-3.9169837118581907], [-2.9810816809142668, -1.0721683301979223, -4.3680575752738715, -0.95912995947965562], 
[-0.28391245540793442, -4.3087224151401671, -1.0996144785416471, -1.7712173206963922]], [[1.8939032986650908, 
-1.4960265132859862, -0.68343848700696341, -0.89263204493523141], [1.5696085465652763, -1.5682491116209887, 3.2563249363809561, 
-0.34613159713130859], [-4.5408967598370467, -4.4069845310858744, -1.3707116182536918, -3.9546560803066058]]], 
[[[-4.8773355181590201, -0.83090957225869122, 4.9250935421386828, -2.6575689452351225], [4.5983560924538001, 
0.65269651851343724, -0.09456466082257009, 4.2481638158032791], [1.181503592842895, 1.0103384162573867, -0.51885214238466038, 
0.46248882380778156]], [[-3.1792233330546429, -4.8717719994267119, 4.2984205315960846, -3.1764490570278414], 
[-4.7854639876055005, 1.7237768181852626, -2.4383425547561552, 4.1800019675075735], [2.7872261457267502, 0.31628645318222404, 
3.0255102325592969, -4.1262836993980798]]], [[[-0.31101549533723816, -2.990189804288522, 1.0479179867040127, 
2.3957443525139945], [-3.281943500600625, -2.4732940113552369, -0.78474254287939083, -1.6131825458243956], [4.3612463317081183, 
4.6028737402879365, -0.9102917374527344, -3.5783447417936278]], [[-1.8480098536845411, -2.3335092261088732, 
-0.85391988520108786, -2.2935941414058481], [4.7848029491430051, -4.2713947018203164, 1.6001961881873248, -2.850052603114138], 
[4.374461642176394, -3.0275353912085032, 0.74451378569035853, -4.1715560070662852]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.7870233935177253, 3.0306763587475851, -2.2617070032360509, 4.299957402899933], 
[-0.050948602694022453, -2.0050662012313145, 4.8083571491056869, -0.86037520030392656], [0.26498621931383393, 
-3.9910865505278936, -4.8756782225630015, 2.7689796808350171]], [[2.1183871211869874, 5.9209050554506781, -2.5520761812676276, 
-6.5440507312523941], [3.3021788225719919, 15.504017783878362, 3.1005183992083856, 5.2062796283165822], [-0.18145253493618424, 
-12.129194396624735, 8.468642139851692, -4.3142070638938428]]], [[[6.8146667108710659, -1.2354358240054608, 4.0932623154124332, 
-1.5070769864972706], [-11.753351193748289, -8.1403712150387282, -5.6384093400830144, 0.13373976760904047], 
[8.0338916785142747, -13.005377547628401, -4.2368388623516431, 2.0443510349657839]], [[-5.8381729951145847, 8.2682452737669045, 
1.277107650074083, 0.61062744692587578], [9.9034145431456082, -6.7640981870507728, 10.69787924926052, 14.405772706235689], 
[-13.946294891756944, 5.8381590560198253, -9.2471554698673746, -11.579109482076863]]], [[[1.5772452682227258, 
-18.114423859896544, 10.136457355202262, 0.40869192188767695], [0.76645833615902148, -1.2305204793126998, 0.20327973452368098, 
-2.4433131986212628], [-6.4084568415426615, 12.990429193664029, -2.7556264484713573, 6.5292136117723256]], 
[[5.8710080877018962, 0.088287841225426031, 1.0932984803687733, 18.5310307314586], [-0.97702428764548432, -2.2361085605938351, 
-1.5979275006505458, 0.43287135945297789], [21.938033953069581, -1.7620027172808095, -4.0725977055643172, 
-2.9766967720402406]]]])+(1.-msk_ref)*numpy.array([[[[12.347373996853161, -10.738024500548738, 5.4515988887302971, 
5.9424630566377852], [-0.74018885067210061, -1.3448964413176023, 14.718932381774312, 0.75222613877590794], 
[-0.017190569042182937, 18.251642959144508, -2.415154487967508, -4.3972716580600633]], [[-1.9366852353068493, 
-1.7866798327580213, -0.37165857424795207, 2.2937880763593035], [6.7105877081687169, -5.3057633402525761, -14.83299421750381, 
0.59452148951694284], [0.24064964834408253, 18.576559189162079, 2.8439736857884319, -16.506030791139352]]], 
[[[6.6779301698597839, -2.509035776237365, 7.6625104004966547, -1.7232674939657446], [-13.656245699413898, 2.1284283625557019, 
0.34319321748680481, -0.5155147785507298], [4.1267169638423837, -3.3551223348560435, -0.79666084784842839, 
0.92473117398221816]], [[9.5744311534609086, 8.7296351655425255, -3.216293665588573, -5.8835439681885875], 
[-19.396178953864659, 8.1609609685674958, -12.184468485478726, 15.661626538544775], [-10.848508660914813, 0.96757788496320529, 
-5.7286356935194709, 15.324458334355596]]], [[[0.18884435421962445, 11.789868864490842, 3.6756389508103759, 
4.9749860416954883], [2.8127367446880642, 3.429085763062961, 0.051768331172459256, 1.1878726430230229], [13.78619598161176, 
15.003668580551102, -1.6332438961409257, 6.9477311384621361]], [[-7.3804467990336571, 2.4091416685590055, 3.2391941762482577, 
-9.8774263480335431], [-1.851188494502455, -3.0921280569352674, 2.2768949490962518, 1.823175466360536], [21.387331151034793, 
1.118347960745224, 1.0995261244123093, -5.3276075375144938]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.03211649917)+(1.-msk_arg0)*(4.61815129983)
      arg1=-1.07755489321
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-3.26727197045)+(1.-msk_ref)*(-4.9763115307)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.07896743132)+(1.-msk_arg0)*(0.475379415707)
      arg1=numpy.array(-0.760934700419)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(3.10382786037)+(1.-msk_ref)*(-0.361732693277)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.27108713311)+(1.-msk_arg0)*(2.85473839237)
      arg1=numpy.array([0.74694597065855106, 4.9946728872706885])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-3.1902713244116936, -21.33268310293926])+(1.-msk_ref)*numpy.array([2.1323353394662328, 
14.258484448629117])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.00349452085)+(1.-msk_arg0)*(1.01723691582)
      arg1=numpy.array([[2.4669065148289082, -0.97937040768199868, 2.0703503488568664, 1.3172883311167425, 4.4054576881707437], 
[-1.8214150407855776, -1.4986576850694053, -1.9517440169084512, 3.7712498978218267, -1.4898571588466423], [3.2926184043760625, 
-1.8385046331857682, -3.9638027111850183, 1.518407595716309, -1.9576508796768257], [-0.25792609089103546, 1.7984516603925558, 
0.48664694579955103, 4.9206937681536029, -2.4282279373630979]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[7.4093402007363389, -2.9415336533549357, 6.2182859290302623, 3.9564682848879991, 
13.231768028254736], [-5.4706100951921863, -4.5012101457348193, -5.8620524608851508, 11.326928404861754, -4.4747778134441587], 
[9.8893613367914295, -5.5219385923297084, -11.905259724772238, 4.5605288941500595, -5.8797936908453732], [-0.77467960077533182, 
5.4016397080015643, 1.4616414352970513, 14.779276771427181, -7.293169305243528]])+(1.-msk_ref)*numpy.array([[2.509428374769938, 
-0.99625173295943081, 2.1060368035456691, 1.3399943191957371, 4.4813941915065918], [-1.85281061852361, -1.5244899214354799, 
-1.985386064237302, 3.8362546148606986, -1.5155377012830107], [3.3493729906518404, -1.870194782789464, -4.0321264448594469, 
1.5445802596297296, -1.9913947431019969], [-0.26237194120845886, 1.8294514202757262, 0.49503523824015599, 5.0055113524294459, 
-2.4700830979201696]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.20667430699)+(1.-msk_arg0)*(-3.29653578339)
      arg1=numpy.array([[[4.1483919982121051, 4.1361015346100203], [0.032231091097703946, -4.5760757212460392]], 
[[-4.0117136604095833, 0.91477963029775111], [-2.2804998852285561, 4.4137103522313481]], [[1.8609433425832735, 
-3.7702921297048633], [-4.433347296091366, 3.1956556383666825]], [[-2.6639274709231122, 2.2267711251101634], 
[2.9215043090427839, -1.771313752684367]], [[-2.6891298732055677, 3.5476246328640269], [0.98300470777792714, 
-0.75356874842756216]], [[-2.7939802849124939, 0.21842156082553554], [3.8147018151339651, 3.1570405239298367]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-17.450934034185018, -17.399232056729183], [-0.13558570280683516, 19.250060163387971]], 
[[16.875972782229603, -3.8481799673276673], [9.5933202742754027, -18.567041937209574]], [[-7.8283825460016327, 
15.860391031860834], [18.649648164413236, -13.443082467891948]], [[11.206275247606351, -9.3673008795391492], 
[-12.289817114599057, 7.4513400530282139]], [[11.31229354576228, -14.923701393899609], [-4.1351806478556501, 
3.1700182925577942]], [[11.753365078766707, -0.91882836801655199], [-16.047208114536765, 
-13.280641258129146]]])+(1.-msk_ref)*numpy.array([[[-13.675322665641978, -13.634806712583231], [-0.10625094514133854, 
15.085197362597524]], [[13.224757634261467, -3.0156037851943611], [7.5177494756765864, -14.549954113656998]], 
[[-6.13466631969031, 12.42890291941216], [14.614688001768, -10.534593163273197]], [[8.781732232258177, -7.340630695349037], 
[-9.6308434960925631, 5.8391991693378413]], [[8.8648128532097239, -11.694871548278087], [-3.2405101944324324, 
2.4841663444371522]], [[9.2104559873049574, -0.72003449112564344], [-12.575301036558363, -10.40729705675237]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.87018209538)+(1.-msk_arg0)*(-1.41356460416)
      arg1=numpy.array([[[[0.89499864308825483, -1.2859761355796797, 4.4586059690517885, 1.2447980556398717], 
[0.76161929710261145, 0.63331100647822325, 4.7147516365445856, -2.1446820363307806], [-3.3799664803692462, -1.0145493441184241, 
1.0623053505725268, -3.3532142311256683]], [[1.6279151079136032, -0.27623482467522109, -0.0463211991047352, 
-2.5440185174292096], [-3.0940023490777593, -4.6086642296571387, -4.3655803260414237, -0.19626610250057475], 
[-3.4692290597744169, -4.649052583497264, -4.6411992349497808, 2.4619506118873247]]], [[[-4.7926551515171845, 
-4.5424729524915355, -0.71928231952849497, -1.1582344568723588], [2.0322006232336278, -0.88834076795382, -2.3420049683828625, 
4.2833546879931852], [0.83147162047224299, 0.14560854238999443, 2.4601426577367906, 3.2053851596547105]], 
[[-4.8897030143218245, -3.8492384785788625, 1.1837721646911659, 4.3854885223510127], [-3.4899504149719895, 
-0.14367989080200871, -2.852004683889616, 4.7328137540950515], [-3.6990870559221491, -4.6091482708784257, -2.3932944403511511, 
-0.9156742445011119]]], [[[0.44009446178047007, -2.4279928119836844, 4.0924464920338828, 0.82758222356797617], 
[0.36334035692032618, -3.6279853768370929, 0.67808433338727081, -3.7633188563549869], [-2.8779381421220327, 
-4.8951747555337155, -4.7117751543216944, 0.3551604778193056]], [[-2.3582028685461829, -0.86226558046877777, 
3.0671608027450006, -1.9764855547429825], [-4.0242579798343154, -1.1635507197786521, 1.3099641294458229, -0.66691841080900716], 
[3.9113774391563378, 3.7394546578063768, -2.0184399903177672, -3.0554668360146477]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[1.6738104376949996, -2.4050095438498831, 8.3384050536847667, 2.327999036024238], 
[1.4243667729388645, 1.1844069051240405, 8.8174440948395123, -4.010945944633618], [-6.3211527945784862, -1.8973920182520101, 
1.9867044464694288, -6.2711212170319452]], [[3.0444976876221688, -0.5166094232286339, -0.086629077202308807, 
-4.7577778816168248], [-5.7863477963156287, -8.6190413259330843, -8.1644301617153676, -0.36705335082701845], 
[-6.4880900723696664, -8.6945749021468473, -8.6798877103045484, 4.6042959540668749]]], [[[-8.9631378537086039, 
-8.4952515845074945, -1.345188915507155, -2.1661093434973884], [3.8005852197960368, -1.6613589988252895, -4.3799757591657364, 
8.0106532456561759], [1.5550033374255956, 0.27231448891246468, 4.6009147505852743, 5.9946539343899881]], [[-9.1446350291209484, 
-7.1987768834943289, 2.2138695074172228, 8.2016621140049146], [-6.5268427798522213, -0.26870755924438339, -5.3337680957564721, 
8.8512235436870821], [-6.9179663812455816, -8.6199465711585628, -4.4758964113224362, -1.7124775772685834]]], 
[[[0.82305678269869198, -4.5407886846885166, 7.6536201557113763, 1.5477294569734008], [0.67951263004216456, 
-6.7849932940690998, 1.2681411794600344, -7.0380915443692356], [-5.3822683850140791, -9.1548681815659911, -8.8118775310790376, 
0.66421476660504497]], [[-4.4102687820339659, -1.6125936500570299, 5.7361492169517323, -3.6963878962618399], 
[-7.5260952210849954, -2.1760517231990826, 2.4498714604824814, -1.247258870975742], [7.3149880549919821, 6.9934611475229795, 
-3.7748503304956675, -5.7142793697486312]]]])+(1.-msk_ref)*numpy.array([[[[-1.2651384026394668, 1.8178103470480007, 
-6.3025275817415327, -1.7596024707778779], [-1.0765980802283477, -0.89522602218162728, -6.6646060308179091, 
3.0316466137318212], [4.7778009798922394, 1.4341310420180522, -1.5016372423775379, 4.7399849472798907]], [[-2.3011631751215766, 
0.3904757705968287, 0.065478007476633279, 3.5961345285617807], [4.3735722058396522, 6.5146446274948513, 6.1710298255029938, 
0.27743481549096161], [4.9039794026152785, 6.5717361749034824, 6.5606349593726403, -3.4801262421503467]]], 
[[[6.7747276821227089, 6.4210789809895079, 1.0167520272825235, 1.6372392315515412], [-2.8726468695519527, 1.2557270660105224, 
3.3105753260694222, -6.0547985740036534], [-1.1753388520618941, -0.20582708158561241, -3.4775705821572105, 
-4.5310190043829239]], [[6.9119111058925808, 5.4411472662840961, -1.6733384313955493, -6.1991713471388676], 
[4.9332703768727635, 0.20310080796708169, 4.0314928720406868, -6.6901380008633984], [5.2288985299525201, 6.5153288510322156, 
3.3830763082097759, 1.2943647009663715]]], [[[-0.62210195365906973, 3.4321246981714633, -5.7849375055518237, 
-1.1698409382664989], [-0.51360506780489845, 5.1283917131016459, -0.9585160123106754, 5.3196943295057535], [4.0681514906614549, 
6.9196457655928256, 6.6603985809027231, -0.50204228024139963]], [[3.3334721044019853, 1.2188681039348694, -4.3356299460227836, 
2.7938900208153084], [5.6885486382962833, 1.6447541126222782, -1.8517189261019527, 0.94273225938126737], [-5.5289847014956175, 
-5.2859607431308273, 2.8531953259312735, 4.319099768570549]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.10471082212)+(1.-msk_arg0)*(-1.8238676005)
      arg1=Data(-1.92491980085,self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-7.90123913828)+(1.-msk_ref)*(3.51079885834)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.12682778151)+(1.-msk_arg0)*(-2.89933204639)
      arg1=Data(numpy.array([2.0742757674077001, -3.1986839505197464]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.3373515612176328, -3.6043659396998655])+(1.-msk_ref)*numpy.array([-6.0140142054902741, 
9.274046884007431])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.882624974542)+(1.-msk_arg0)*(-2.29849201387)
      arg1=Data(numpy.array([[-0.70176265409067184, 4.2595548573977577, -1.471689653279673, -1.150960755966155, 
1.2668618571901034], [-1.2321223622093944, -0.79297650758108951, -0.94012120667600385, 0.84091786864605478, 
-1.5185122919823604], [-3.4281146694986218, -1.7375109349523745, -4.1277622977599204, 2.0006064022722807, 0.68807783753198404], 
[-3.2140493378568058, 0.42898789251285319, -2.7215813554660806, -1.1139453320091341, -4.2151680276636156]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.61939324470103374, 3.7595894975692983, -1.2989500427591261, -1.0158667079330228, 
1.1181639144501549], [-1.0875019685772183, -0.69990086981585597, -0.82977445610843803, 0.74221511240531124, 
-1.3402768730520567], [-3.0257396228919498, -1.5335705447281132, -3.6432660929741782, 1.7657851748733588, 0.60731468383431531], 
[-2.8368002150013498, 0.37863542770781711, -2.402135674581177, -0.98319597030531025, 
-3.720412573105218]])+(1.-msk_ref)*numpy.array([[1.6129958560586719, -9.7905528223641305, 3.3826669149564403, 
2.6454741058644231, -2.9118718614262495], [2.8320234096472601, 1.8226501698605815, 2.1608610856133463, -1.9328430054023971, 
3.4902883760828236], [7.8794941904685238, 3.9936550079974715, 9.4876286765492548, -4.5983778385173144, -1.5815414144872708], 
[7.3874667352436667, -0.98602524498713273, 6.2555330106325817, 2.5603944495092485, 9.688530048699258]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.31475938537)+(1.-msk_arg0)*(-4.27264812715)
      arg1=Data(numpy.array([[[4.6425729224605377, -2.3065501262848862], [2.4606140265714416, -0.96729394909452537]], 
[[-0.50857249140448957, 4.9744390191366463], [-2.3049587110781919, -1.659171001048513]], [[1.0047079356798161, 
1.9194603953883282], [-2.0822389148894507, -1.4299065934135733]], [[-1.9569935273318952, 4.3800867574996971], 
[1.6844019344815937, -0.38325185855751087]], [[1.6269014616233282, 3.2894002741135981], [-0.53531454557512603, 
-3.171626981574327]], [[2.8304759957509926, -4.161984688733118], [-1.5816985128629204, 
-2.2673268865926111]]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[20.031585089465995, -9.9522088052214048], [10.616957464930017, -4.1736406452702894]], 
[[-2.194367930430142, 21.463507444786398], [-9.9453422315222948, -7.1589236487130847]], [[4.335072995033399, 
8.2820097558539025], [-8.9843599006085224, -6.1697028941382062]], [[-8.4439561891698585, 18.899020445670594], 
[7.2677890555451956, -1.6536395536727342]], [[7.0196883506165983, 14.192970704980755], [-2.3097534596470535, 
-13.684807285650635]], [[12.212822867740091, -17.957962497490691], [-6.8246485032060855, 
-9.7829699636344181]]])+(1.-msk_ref)*numpy.array([[[-19.836080502312356, 9.855077077250721], [-10.513337912271631, 
4.1329066800030922]], [[2.1729513029198442, -21.2540275587404], [9.8482775200483186, 7.0890538702529629]], 
[[-4.2927634797159824, -8.2011788634962084], [8.896674199983071, 6.1094877283491842]], [[8.3615447294009968, 
-18.714569481189407], [-7.196856770732083, 1.6375003356928388]], [[-6.9511774830639252, -14.054449920641023], 
[2.2872106905881813, 13.551246082844713]], [[-12.093627962190972, 17.782696085546153], [6.7580411887010721, 
9.6874899756387318]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.23397530867)+(1.-msk_arg0)*(-3.32670658823)
      arg1=Data(numpy.array([[[[1.8115416952932986, 3.3547469195433592, 2.9243888543798704, 2.6223587501061942], 
[-0.72992404036320124, -0.72275712866591935, -4.2390212491952139, -0.43282973529711999], [0.61667573552299082, 
0.80131403573768356, -1.4452908316590007, -3.1228748449258381]], [[3.3293118989593964, -0.24988545630770265, 
-0.1027575334862183, 2.6518905793868086], [-1.1265139981569017, 2.5532462554239927, 0.16304891918515096, -1.7229626671701972], 
[0.052536875670695871, 2.8449245864494994, 4.4322838555877855, -0.33570931500169188]]], [[[1.2003508226768966, 
-3.079691570195048, -4.2034981555726336, -0.4980083388698322], [-1.1327319810056888, -1.0956277451205865, -2.1839013514374264, 
0.021789545450503489], [1.6477381221077003, -3.5664094979017125, 2.3810468777870017, 2.3857843464347663]], 
[[-4.8456039907044035, 2.0234705342065435, 1.1145427164574748, -3.1013853935096058], [-2.095726434729265, 3.0091835286935567, 
-4.8898954090416034, -3.4604856292073261], [-1.6676650027961646, -4.2903150313393725, 3.2743396319778952, 
0.61393236087808756]]], [[[-3.3079357498939101, -4.3908696298722454, -4.5691024495231405, -1.3542802764443209], 
[2.0184254995461171, -2.0207553868188777, -3.8574924805283084, -2.6484032612719832], [-1.673902155109559, -4.5492454276939709, 
-3.1422987786612877, -1.9920805424025021]], [[-2.4124561206632968, -4.5902505232303756, -2.2681746130151001, 
4.1193435009532529], [1.3987438003097799, -4.8241835005454909, 2.3596370002904088, 4.0267548194355545], [1.8166959057506968, 
3.5255319186153695, 1.9708965239423222, 2.5267964263390059]]]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-7.6700228084947977, -14.203915624177361, -12.381790202388919, -11.103002198419686], 
[3.0904803641011407, 3.0601358369354434, 17.947911302012457, 1.8325904121054082], [-2.6109898376591576, -3.3927438418026372, 
6.1193256950887687, 13.2221749854771]], [[-14.096224375049394, 1.0580088520021047, 0.43507285956029623, -11.228039234413613], 
[4.7696324530654399, -10.810381602414781, -0.69034509793496945, 7.2949813905557583], [-0.22243983438429854, 
-12.045340454050333, -18.766180405567471, 1.4213849506070857]]], [[[-5.0822557449535672, 13.039338066519498, 
17.797507400726943, 2.108555010285746], [4.7959592389169279, 4.6388608203324031, 9.2465843985532246, -0.092256397424535758], 
[-6.9764825241553465, 15.100089754715681, -10.081293689331726, -10.101352014611951]], [[20.516167652226645, 
-8.5673242796482025, -4.7189463419369551, 13.131189178783947], [8.873253978366991, -12.740808759739631, 20.703696423852243, 
14.651610710065034], [7.0608524449690817, 18.165087909099032, -13.863473153988201, -2.5993744571502106]]], 
[[[14.005718287711714, 18.590833596460261, 19.345466954056466, 5.7339892514816286], [-8.5459637274645797, 8.5558284126494293, 
16.332527915930189, 11.21327401562197], [7.0872603938603964, 19.261392813928076, 13.304415441310201, 8.4344198294105919]], 
[[10.214279648133923, 19.435007375958797, 9.6033953072540328, -17.441198670958983], [-5.9222467136643617, 20.425473825794239, 
-9.9906447966495406, -17.049180479550902], [-7.6918456083071014, -14.927015093339177, -8.3447272183118191, 
-10.698393679150453]]]])+(1.-msk_ref)*numpy.array([[[[-6.0264676925928891, -11.160258679102764, -9.7285836684237292, 
-8.723818130691475], [2.4282431139866754, 2.4044009016260359, 14.101979917361835, 1.4398975319965273], [-2.0514992321684096, 
-2.6657366819329638, 4.808058531594261, 10.38888832084516]], [[-11.075643828554227, 0.83129559380270523, 0.34184416363928305, 
-8.8220618617218971], [3.7475815394064407, -8.4939011393029045, -0.54241591365768227, 5.7317912561563986], 
[-0.17477477041893691, -9.4642293647705689, -14.744907903307285, 1.116806389947667]]], [[[-3.9932149899913894, 
10.24523023629672, 13.983805007773141, 1.6567276219137641], [3.7682669439150276, 3.644832037944667, 7.2651990138801228, 
-0.072487424404815148], [-5.4815412665000816, 11.864397973010103, -7.9210443352281246, -7.9368045033901948]], 
[[16.119902719849524, -6.7314927572423731, -3.7077565977073519, 10.317399221241246], [6.9718669375500948, -10.01067067011023, 
16.267247273034119, 11.512020341173249], [5.5478321517693496, 14.272619280356249, -10.892767225816707, -2.0423728296632158]]], 
[[[11.004531652627001, 14.607134925772787, 15.20006322114495, 4.505293117962748], [-6.7147094071996634, 6.7224602585397983, 
12.832745649036816, 8.810460577574041], [5.5685813274621383, 15.134004735803144, 10.453506049172303, 6.6270674647032557]], 
[[8.025533670436138, 15.270416657275268, 7.5455514283825416, -13.703847163820287], [-4.6532102157420718, 16.048643034114669, 
-7.8498199547069243, -13.395831787019455], [-6.0436142384786606, -11.728410260687166, -6.5565944509265037, 
-8.405910318628214]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.78622176037)+(1.-msk_arg0)*(-1.8404333108)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.92383574318)+(1.-msk_arg1)*(1.32803226522)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-14.856512275)+(1.-msk_ref)*(-2.44415481872)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.06237191006)+(1.-msk_arg0)*(0.129235215427)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-4.6325223491525751, -2.7255311523759951])+(1.-msk_arg1)*numpy.array([-0.52643202664626543, 
0.036696645870019751])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-9.5539839656400218, -5.6210588886670028])+(1.-msk_ref)*numpy.array([-0.068033556371348253, 
0.0047424989344635347])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.550030305573)+(1.-msk_arg0)*(3.77185989194)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-3.9047943468212121, 1.5895480541185645, 3.6595958617769426, -0.19612728648362676, 
-4.1057200443373221], [-4.6575168128656221, -1.9102281496353135, 2.4679862534040167, -2.9496316873322801, 2.0362019418950306], 
[-2.6054255566601747, -2.1834805923067622, -1.6709589083716061, -3.4670153153052272, -3.731998678364655], [-2.3424169469364973, 
-1.392360353761497, 2.9019005453132927, -3.665378055088182, 
4.1753451727594122]])+(1.-msk_arg1)*numpy.array([[2.3654916204781342, 3.4116933294690188, 4.8966533869911828, 
-4.8425740418831431, 2.9819569993294763], [-3.7990170737496842, 0.81584742936161359, 0.72985294183543825, 0.7057533343476754, 
3.6040557711039618], [4.0480913229558837, 2.4722926972925841, 1.9830828869985222, -2.0131418770192044, 4.4871415644661479], 
[-4.4353120986434158, -1.1559101404476158, 4.4585856754803004, -2.5164708566238714, 2.6991795617789087]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[2.1477552277813823, -0.87429960192963396, -2.0128886301264721, 0.10787595131577186, 
2.2582704505836153], [2.561775395791372, 1.0506833728578564, -1.3574672331095146, 1.6223868183108665, -1.1199727763086449], 
[1.4330630150772248, 1.2009804973989733, 0.91907803897138485, 1.9069634933032396, 2.05271237345855], [1.2884003091026084, 
0.76584039084701983, -1.5961332436808198, 2.0160690116803348, 
-2.2965663812451695]])+(1.-msk_ref)*numpy.array([[8.9223029679925041, 12.86842923301027, 18.469490515105306, 
-18.265510802310118, 11.247524005249101], [-14.329360129257042, 3.0772621967482752, 2.7529030383206914, 2.6620026954261946, 
13.593993411328018], [15.268833299952012, 9.3251416660445194, 7.4799108038546587, -7.5932891027057785, 16.924869296449454], 
[-16.729375813092219, -4.3599310974366343, 16.817160484105155, -9.4917754933257665, 10.180927130207637]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.28914330832)+(1.-msk_arg0)*(2.53082722502)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[4.1671511588981076, 1.0631414819541449], [-2.5926279378838037, -1.941950795330436]], 
[[1.0739226167943485, 2.6070978477588538], [1.0858057364332971, -2.5365379693016141]], [[2.2956131717178909, 
-0.65913301518900713], [-3.3470668021681349, 2.9461614215126763]], [[-2.9404317947906544, 2.5742156733479016], 
[-0.90434572077453979, 1.6875930181964529]], [[3.2642778482849533, 2.1885849412690987], [-1.6075707950009068, 
-1.7975179414975573]], [[2.0412388544209712, -3.8753782007067974], [-2.6837366310998876, 
-0.79592815263301286]]])+(1.-msk_arg1)*numpy.array([[[-3.9963529969071043, 0.86904080582745458], [3.4476341726748885, 
1.9815017641344701]], [[-4.4447614169611391, -2.5595274648194897], [-2.5976189242299617, 2.0015016384572624]], 
[[-1.5206809959179877, 4.7015006292421067], [2.0586590585627018, 0.48804793774688449]], [[0.66847863605865054, 
-0.15285035082758913], [-0.78317665826360638, -3.1868565463842757]], [[-1.7641487995608074, 0.85656514001687256], 
[0.5331451872130426, -1.2704585180218086]], [[-0.94499681078649544, 2.4800569150089444], [3.2078159726388336, 
3.0268651186145963]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[5.3720550312513247, 1.3705417272585674], [-3.3422689570863215, -2.5034528728868852]], 
[[1.3844401550939112, 3.3609227445737355], [1.3997591992584275, -3.2699609494247142]], [[2.9593743588113131, 
-0.84971691582367703], [-4.3148487705149892, 3.7980242817735319]], [[-3.790637971825666, 3.3185329094688458], 
[-1.1658312343443027, 2.1755492465754673]], [[4.2081219446136746, 2.8213996317269241], [-2.0723891330260416, 
-2.3172582258666723]], [[2.6314494098595267, -4.9959178746502726], [-3.4597211192756134, 
-1.0260654518703283]]])+(1.-msk_ref)*numpy.array([[[-10.114078965365072, 2.1993921310419426], [8.7253664261168993, 
5.0148386110978187]], [[-11.248923202766287, -6.4777217911530611], [-6.5741246936698499, 5.0654548375309307]], 
[[-3.8485808650406477, 11.898685790937297], [5.2101103924457153, 1.235165007964963]], [[1.6918039314818547, 
-0.38683782922840904], [-1.9820848087341716, -8.0653833098243748]], [[-4.46475581091586, 2.1678183763582632], 
[1.3492983546874604, -3.2153110056688883]], [[-2.3916236562960815, 6.2765955601051795], [8.1184279964102224, 
7.6604726486549577]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.34546876107)+(1.-msk_arg0)*(2.60519005115)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[2.0599017657329552, 4.2879962998125194, 4.0771055275232087, 3.4807384844531732], 
[-1.2812030968701462, 1.0358299866614988, 4.3497954102075518, 0.29305688035031618], [3.9297703804265858, -3.1558499129556736, 
-1.5316103055758647, -2.3544065015944184]], [[-4.048367590717838, -0.11909756198259203, -2.4493952554430445, 
3.2740537050610836], [0.27308708250737901, -0.031130671568472046, 3.2788783280867637, -0.24067783478777027], 
[1.8983934544921555, 4.4435207956056963, 3.0633865832706064, -2.8936233862274063]]], [[[2.1710780259949729, 
-3.8743411636633738, 4.9952417083602025, 3.8899726909120069], [3.0684974418789235, -2.8714635264001345, -3.9791827140246996, 
4.512165339136665], [-4.265748554593598, -2.3285440630235863, 1.566656176457454, -1.0999148998542818]], [[2.9275178859975526, 
-1.4539278175349568, -3.3171669314847363, -2.0461371189841326], [-1.9853404225765292, 0.22482928919262424, 
-0.45729686003390579, 4.6747588858596618], [-0.45165731568318623, -1.7805223034206596, 2.8337243745257465, 
-2.9245600073247302]]], [[[2.6376326711020139, 4.7975116001519229, 2.9152629524246185, -0.865140798198059], 
[-1.4745045029067083, -2.973615350306178, 4.9182955125965382, -4.9841425789286653], [-2.116800343388987, -3.5461945846998844, 
0.62100373084889959, 0.3376680720658376]], [[2.0748021222380642, -4.2739021142108227, 1.7842313729437977, 2.0978666766277874], 
[-0.52732757927811491, -0.32432906880336976, -1.225625010081334, -0.51221867997549708], [1.5501413486502535, 
3.3801658690853973, -0.80751771328439581, -2.2386677449870773]]]])+(1.-msk_arg1)*numpy.array([[[[1.2421239825010968, 
3.8204100901244669, -2.7101586614646855, -2.6927085606400638], [1.9554223470219467, -1.9445161738465622, -0.19087802995434089, 
3.9261277155747223], [-1.1561124576515946, -4.9858553763959348, -2.262338984960397, -3.1534374751826055]], 
[[2.8757616226419556, -0.74846403902468062, 3.8563559120584063, 4.6162951012029456], [2.6368476859490713, -2.6751218249345063, 
-3.7987666378731988, -4.0288498243854534], [1.9658780796681725, -4.5524317897174154, 0.98830801784276012, 
2.5493925718784389]]], [[[3.6867913592802104, 2.627549328910713, -2.6341778350556542, 2.3191100945128005], 
[-2.6220695086324231, 2.6410316604966253, -4.5865698580442791, -4.0069655296704703], [-4.0834385129214983, 3.5858236029739814, 
0.68811028946692332, 2.1313597954464285]], [[-1.780143235614029, 0.13672385750237837, 3.3808158859735009, 4.5178967581346399], 
[-4.000237417484974, 1.1896729162091377, 0.54052848922000774, -0.28807425604258619], [2.7019904172018148, -3.3846544311547291, 
-0.92236569342477903, -2.8419804175767327]]], [[[4.3033769539120197, 2.6820323119017875, 1.1057289154875125, 
0.19917072544626802], [-2.3903668555648263, 3.2209570375527754, 0.84704276138530865, -2.6955580508436983], 
[-4.0811290982231725, 3.8326717635992331, 2.1156702705188204, 1.9424579478095358]], [[0.51380194737956852, -3.5237944305475377, 
3.8580610051057906, -2.7800489796506067], [4.1425203729071711, 0.62303402381811868, -2.7432080954367821, 0.99872789514656635], 
[4.6522833351124468, 2.2765214476542202, -0.7196904590997395, 4.3887142251968552]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[2.7715334766706281, 5.7693650689898286, 5.4856181228762244, 4.6832248962926455], 
[-1.723818743427413, 1.3936768888346147, 5.8525138414883813, 0.39429887772854871], [5.2873832850497786, -4.2460974725134708, 
-2.0607338202881795, -3.1677803987599713]], [[-5.4469521266469387, -0.16024204916740711, -3.2955847997164498, 
4.4051369822315429], [0.36743013856595486, -0.04188534610656966, 4.4116283617965433, -0.32382450818937919], 
[2.5542290892426474, 5.9786184196610135, 4.1216909508775172, -3.8932798724761906]]], [[[2.9211176618259773, 
-5.2128050054441921, 6.7209416726023008, 5.2338367370450722], [4.128567451477263, -3.8634644733288632, -5.3538660363177062, 
6.0709775085999746], [-5.7394314227934826, -3.1329832955777746, 2.1078869447639188, -1.4799011375915114]], 
[[3.9388838630890826, -1.9562144593467932, -4.4631444815735888, -2.753013574462897], [-2.6712135186700916, 0.30250078518268586, 
-0.61527863971190921, 6.2897420464676586], [-0.60769080896133632, -2.3956371376443579, 3.8126876234125247, 
-3.934904129735759]]], [[[3.5488523621505079, 6.4549019888846857, 3.9223952327976881, -1.1640199179043349], 
[-1.9838997467208908, -4.0009065612809671, 6.6174129699189637, -6.7060081406770733], [-2.848088735456245, -4.7712940343961892, 
0.83554112036632366, 0.4543218425759743]], [[2.7915814408770876, -5.7504017825499956, 2.4006275748203842, 2.8226140782965046], 
[-0.70950278477039241, -0.43637463038248703, -1.6490401638529208, -0.68917423274453837], [2.0856667598548482, 
4.5479075840959986, -1.086489857236405, -3.0120575172994846]]]])+(1.-msk_ref)*numpy.array([[[[3.235969041503294, 
9.9528943580949392, -7.0604783818784247, -7.0150175528186036], [5.0942468442526376, -5.0658341904000359, -0.49727354461964113, 
10.228308864148859], [-3.011892672681364, -12.98910082304586, -5.8938230159414591, -8.2153039372607175]], [[7.4919055687779776, 
-1.9498910681086066, 10.04654005577755, 12.026326070813834], [6.8694893578252447, -6.9692007639263283, -9.8965090516174552, 
-10.495919480055445], [5.1214860149200403, -11.85995000709841, 2.5747302155530458, 6.6416521647264828]]], 
[[[9.6047921698525567, 6.8452653705768984, -6.8625338888396676, 6.0417225457399732], [-6.8309893973058227, 6.880389406690786, 
-11.948886163068943, -10.438906733187595], [-10.638133388334726, 9.3417519756369067, 1.7926580802113028, 5.5525973345123347]], 
[[-4.6376114470387941, 0.35619163331967435, 8.8076679108988376, 11.769979686402907], [-10.421378722258938, 3.0993240454274158, 
1.4081794424776333, -0.75048818583379939], [7.0391985531894532, -8.8176680506158522, -2.4029379280297953, 
-7.4038991094262929]]], [[[11.211114826668075, 6.9872038958220717, 2.8806339698939376, 0.51887759241240361], 
[-6.2273599507096895, 8.391205229405303, 2.2067073748573245, -7.0224410163479529], [-10.632116924138675, 9.9848383478418175, 
5.5117231402637028, 5.0604721204053629]], [[1.3385517215733496, -9.1801541927506349, 10.050982147220877, -7.2425559434879041], 
[10.792052862172678, 1.6231220403772195, -7.1465784384585795, 2.6018759762390973], [12.120082259753229, 5.9307710266521747, 
-1.8749304239522588, 11.433434636811384]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-3.811943307495167, -4.2282720665807254])+(1.-msk_arg0)*numpy.array([0.61750636253069313, 
4.9857329849992382])
      arg1=0.198997849028
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.75856851880758591, -0.84141704635403936])+(1.-msk_ref)*numpy.array([0.12288243790460988, 
0.9921501398419712])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-3.8160949918139266, -0.95509775879377035])+(1.-msk_arg0)*numpy.array([-0.61013882624405902, 
-3.7544911159648042])
      arg1=numpy.array(-0.954410927741)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([3.6421227614866787, 0.91155573805415391])+(1.-msk_ref)*numpy.array([0.58232316320667932, 
3.5833273491850477])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-4.0556137827612044, 3.8846549838096127])+(1.-msk_arg0)*numpy.array([-4.7616896375013447, 
1.8422861647136397])
      arg1=numpy.array([-2.8601644735049589, 4.7936763893261265])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([11.599722459710655, 18.621778876566207])+(1.-msk_ref)*numpy.array([13.619215535038052, 
8.8313236901699579])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-3.1870400192964263, 3.9001332515976852])+(1.-msk_arg0)*numpy.array([-2.8456252462654863, 
-0.51112439900515394])
      arg1=Data(0.448000224264,self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.4277946433821107, 1.7472605713739846])+(1.-msk_ref)*numpy.array([-1.2748407484973174, 
-0.22898384538094582])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.1146570468422556, -3.9935466693281674])+(1.-msk_arg0)*numpy.array([4.4751218065877083, 
-3.714089737151185])
      arg1=Data(numpy.array([2.8025627286269348, -0.1915914795389515]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([8.7290217519353419, 0.76512951498443549])+(1.-msk_ref)*numpy.array([12.541809581208346, 
0.71158794788123103])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([4.251276776498031, -3.0997289319507271])+(1.-msk_arg0)*numpy.array([0.38135865163194183, 
-0.038739457494182616])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.16929957634)+(1.-msk_arg1)*(3.24881440421)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-4.9710161336427703, 3.6245117268839091])+(1.-msk_ref)*numpy.array([1.2389634805913661, 
-0.12585730751832158])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([2.3607020229570832, -2.5901690405834721])+(1.-msk_arg0)*numpy.array([0.81723149805925388, 
-0.10967384394074742])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.74328040902913362, 3.2187743216207352])+(1.-msk_arg1)*numpy.array([1.4743794126559129, 
-1.0771454408349701])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.754663565219444, -8.3371695964870955])+(1.-msk_ref)*numpy.array([1.2049092961125145, 
0.1181346809796221])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-1.0111981489954012, -3.214286562526667, 1.0426141355728422, -4.3208792868235051, 
3.0214128409911059], [-1.5255616803189964, 4.0936021111022765, -0.35671740115169115, 4.8707864908617218, 0.32960144492456767], 
[-1.7562492226844837, -0.2790348115097725, -0.72185789490097907, 4.9520119947096646, -1.4204467642329819], 
[-3.4279047979548327, 2.9496162322323025, 3.4340103192601195, -2.393976811540405, 
2.1042767742262285]])+(1.-msk_arg0)*numpy.array([[3.9577447367699179, -0.17334519030467632, 4.7678203257600114, 
-4.2555842276025491, 4.272717515896181], [0.36796568383303452, -0.15794016747876949, 1.5050100368263557, 0.37196805129140564, 
1.8079941811584881], [-0.21097689092216676, 1.1275685438761194, -0.88451283829271787, -4.6289829151753592, -1.295027978330745], 
[0.53534816667539253, 2.9171542684963816, -3.658620476278942, -4.943906560405229, 3.3645813978517332]])
      arg1=-3.42410316426
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[3.4624467816726585, 11.006048789597562, -3.5700183607209373, 14.795136438413454, 
-10.345629269384348], [5.2236805768595858, -14.016915941861498, 1.2214371820313972, -16.678075435812065, -1.1285893505120701], 
[6.0135785206294461, 0.95544398103031192, 2.4717159020791106, -16.956199940556715, 4.8637562600781594], [11.737499665471553, 
-10.099790274149933, -11.758405600292464, 8.1972235755692093, 
-7.2052607611144506]])+(1.-msk_ref)*numpy.array([[-13.551726276521524, 0.59355181463211981, -16.325508664075212, 
14.571559419524199, -14.630225566184665], [-1.2599524623531166, 0.54080342722837915, -5.1533096293456166, -1.2736569814318708, 
-6.1907585966749732], [0.72240663979308972, -3.8609110190103313, 3.0286632084298821, 15.850115047174137, 4.4343093984122097], 
[-1.833087351495934, -9.9886371614035561, 12.527493949666361, 16.928446097307166, -11.520673810806599]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[1.4713077736305351, -4.5746645867174029, -3.8851187078536178, -4.9810117376598928, 
-4.3201756327614858], [-4.7870203226103056, -0.49771010498633039, 4.3224883922504169, -1.7442359662608578, 
-3.8780481851763229], [-3.5633691290808533, 0.56040719896380864, 4.4017624343361526, 0.548274403834621, 3.7136899807657464], 
[4.8705606283944896, -2.9578934961890546, 4.1651850390761798, 2.780302488385086, 
1.5055199912376214]])+(1.-msk_arg0)*numpy.array([[-2.7657827410715861, 2.4334279773710286, -1.0212438733805627, 
-3.6674892306577855, -2.7937101709247814], [-4.4831383591073672, 2.6827452939438112, 2.8683669889223937, -1.2389414452035652, 
-4.8104083280359724], [4.6383079783247556, 3.6529538057366278, -4.25885579570377, -1.0810716397338216, 4.4033559260148678], 
[4.3223359586148966, -0.24707783367596292, -3.3805885939360634, 0.20398704962739433, -1.0277505462244596]])
      arg1=numpy.array(-3.62925801016)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-5.3397555228634337, 16.602638095152287, 14.100098190911138, 18.077376747617155, 
15.679032020509819], [17.373331850645624, 1.8063183852606139, -15.687425621410602, 6.3302823521662912, 14.074437439848527], 
[12.932385954883571, -2.0338623157923075, -15.975131573648243, -1.9898292718840389, -13.477939109955466], [-17.676521174584323, 
10.734958664252613, -15.116531166877532, -10.090435076647173, 
-5.4639204876593848]])+(1.-msk_ref)*numpy.array([[10.037739167404114, -8.8315379790280968, 3.706357507796104, 
13.310264667550619, 10.139095015902083], [16.270465800458581, -9.736374847272165, -10.410043870633185, 4.4964381641277305, 
17.458212956678455], [-16.833616383937201, -13.257511860224387, 15.456486510686226, 3.9234879080638181, -15.980914766087347], 
[-15.686872400417872, 0.8967092070021635, 12.269028233607497, -0.74032163382970007, 3.7299719023343259]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-1.7594239098469391, 3.8578236185200563, 1.187458378475144, 3.0932397201418596, 
0.066007679953576037], [2.6417153684602601, -1.6630190012600021, 1.6393890843953498, -4.5631955023778845, -1.4363549238531972], 
[3.1980281591837443, -1.6604194262821315, 1.8774175197873806, -0.80575558109706602, -0.016334833031037554], 
[0.50162872812491965, 2.4960891195754389, 2.3814574021895254, 2.6981880501473796, 
-4.5362726141421561]])+(1.-msk_arg0)*numpy.array([[-2.6166541303666579, -0.45739938531573632, -3.3898789946731158, 
-3.9291802033166934, 0.72758048552464238], [-3.2244569825991687, -1.993482667205412, 4.4305892490248002, 1.1152532845917849, 
1.0627403405242291], [-2.5003580180870335, -3.7115176440087936, 2.5272399306432085, 2.8170216302632696, 0.91674945575742761], 
[0.75226667824247073, 3.7361042084071752, -0.050562342758314394, -2.5844182260095181, -4.0139922527195973]])
      arg1=numpy.array([[-0.093152165316139346, -0.60709397758729633, -4.5727701028017211, -1.0129784709750078, 
3.8710239655702257], [-4.7649442509563773, 4.8844703018182969, 0.071577918988588785, -2.5232967881442092, 4.1344567800299163], 
[-0.67407717202566442, 4.5341322936935171, -2.5397049895524129, 0.89556897278839287, 2.0048368811179653], [-2.9206373271529937, 
0.78173055650187884, 0.40734651782869502, 0.1497485909078744, -4.7457305703024621]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.16389414691123033, -2.3420614853975574, -5.4299741714125496, -3.1333852420684618, 
0.25551731101198222], [-12.587626457607824, -8.1229669230140047, 0.11734405907362709, 11.514296554824217, -5.9385473534542053], 
[-2.1557177776010197, -7.5285613417818746, -4.768086642477126, -0.72160969808161401, -0.032748675707528048], 
[-1.4650755877339212, 1.9512691365239927, 0.97007838013927328, 0.40404985851403524, 
21.527927620160295]])+(1.-msk_ref)*numpy.array([[0.24374699812707376, 0.27768441217731477, 15.501137318956779, 
3.9801749545410146, 2.8164814963471114], [15.364357761692057, -9.737106885154363, 0.31713235833840958, -2.8141150309777303, 
4.3938540062917006], [1.6854342618838025, -16.828512008313549, -6.4184438616506503, 2.5228371677375603, 1.8379331196473132], 
[-2.1970981404483507, 2.9206268219871525, -0.020596394255860302, -0.38701298766155379, 19.049325742688637]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.23514088351356044, -0.60464165284421245, -4.6656977317505319, 2.7007820985737343, 
0.33950420585459185], [-0.87842966912336617, 3.2192366374767616, 0.27474806851699984, -3.0821814718315563, 
-3.8740990492337901], [3.7762676974949532, -2.5797828128074385, -2.3976534690118037, 2.827248398417912, -0.27966575382066949], 
[-1.2241221944286895, 4.1642497053198895, 2.5695048898859643, -1.4955174910468969, 
4.4220209977965634]])+(1.-msk_arg0)*numpy.array([[-0.34121455328992845, -4.981851239750009, -2.5853886496042899, 
-2.7959014467047427, 1.690906634092455], [3.735992029392742, -4.7207467328043986, 4.8219848075631866, 4.1305566963466838, 
-4.7596307449550341], [-3.9108570339359403, 0.47015899617000567, -4.0368232606830858, 1.2200179263480972, 4.3699241336936723], 
[3.2724071012720248, 2.1171055818860163, -4.4900090325697644, -1.1030663589843792, -1.0967446828975058]])
      arg1=Data(-0.696743590053,self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.16383290334751893, 0.42128019589837601, 3.2507949877229283, -1.8817526153116737, 
-0.23654737922528826], [0.61204024127425694, -2.2429824920263441, -0.19142895561871648, 2.1474901838793681, 
2.6992536797848317], [-2.6310903125545573, 1.797447138552992, 1.6705496857027882, -1.9698671990858503, 0.19485532133194713], 
[0.85289929241004392, -2.9014142895625517, -1.7902860616383938, 1.0419922256993663, 
-3.0810147852954022]])+(1.-msk_ref)*numpy.array([[0.23773905283762298, 3.4710729178943964, 1.8013529694080825, 
1.9480264114119923, -1.1781283586823463], [-2.6030284989692318, 3.2891500263460416, -3.3596870060035537, -2.8779389015308654, 
3.3162422125675475], [2.7248645700093284, -0.32758026688729752, 2.8126307310585901, -0.85003966993303137, -3.0447166291698404], 
[-2.2800286718558493, -1.4750797436449277, 3.1283850127239439, 0.768554415025686, 0.76414982773376439]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.4472003013156991, -0.20080652090848528, -2.2584630595905208, -2.3747453496859472, 
1.5783920786782932], [-1.483146511244251, -2.6400333269724721, 3.5416471803278533, -0.89124804005201419, -1.6710703107359004], 
[0.70645959582005524, -0.27455817966345819, 4.0425246705561975, 1.7388497198330359, -1.4757436897846246], [-2.3827787835869927, 
-1.3366370693308349, 4.5906907989581818, 3.1804617689200541, 
0.6981651353495657]])+(1.-msk_arg0)*numpy.array([[2.0422629491199427, 4.341477479836076, -2.6564184070192463, 
2.2752614216563041, -4.9215023244791176], [1.0949036085946231, -1.8544475697367946, -0.20490715178143226, 2.0793096635481643, 
0.95438382970927904], [-3.5145531618136716, -2.3133485748807106, -4.5129558421120652, -4.8134584085027932, 
-3.8942809566813996], [-1.1846204767685595, 0.31457465931217321, 4.1873578718614617, -2.4251229902754901, 2.5959008098694341]])
      arg1=Data(numpy.array([[2.34508054805735, 4.2532460596452974, -1.8881862993645648, -3.632902697370739, 
0.34837284256905399], [0.72606402739670806, 4.34185173037711, 1.6904921296344222, 0.47759776218528138, 2.39265779685425], 
[3.605473905456142, 2.401197004233703, 1.0181306401224397, -0.92265854365056743, 0.19218522321530607], [2.8548316550058512, 
-2.1367919199054732, -0.23313547118098477, 2.9194936901085011, 3.9104412758747227]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.0487207277008317, -0.85407954380509599, 4.2643990067397981, 8.6272187864426968, 
0.54986893513763491], [-1.0768593291733779, -11.462633268968666, 5.9871266842861788, -0.42565806948085999, 
-3.9982994080739065], [2.5471216379883019, -0.65926827849575464, 4.1158182305441358, -1.6043645501283461, 
-0.28361613042983747], [-6.8024322982604835, 2.8561152895922599, -1.0702528624613272, 9.2853380659934199, 
2.7301337626476041]])+(1.-msk_ref)*numpy.array([[4.7892711159994148, 18.465371984151588, 5.0158128415135828, -8.26580335595877, 
-1.7145177544889969], [0.79497012366740094, -8.0517363895553284, -0.34639392739231722, 0.99307364220083361, 
2.2835139113455254], [-12.671629714257572, -5.554805667751868, -4.5947786203738605, 4.4411785251117655, -0.74842325492293049], 
[-3.3818920362470073, -0.67218059022526877, -0.97622165045982745, -7.0801312678463528, 10.151117674990056]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[3.6021167322017664, -3.5660932907532175, -0.63106811619973779, -3.8831932451613937, 
-1.4493847221133196], [4.5737908342624607, 2.9735998891020632, -2.3519756408948309, -0.22265310455729193, -2.4324256283853884], 
[-4.7001261588221492, 4.036395702874028, -2.2617305178147884, -3.685286893399109, -2.4642630780781962], [-1.2993436955276625, 
3.6391802695349131, -2.2211108580869467, 0.84934006610267332, 
-4.6288905574079955]])+(1.-msk_arg0)*numpy.array([[3.8741699397257765, -0.79778461557563851, 3.2734275653522769, 
-3.4724856401543072, -0.5361126549667139], [3.364054231842351, 4.82101537082875, -0.48344236236669946, 0.053373965732863127, 
-4.4040434117932374], [-4.4083261684991086, 4.925235506476179, -4.2032084343265428, -3.0654746233125052, -1.7847721822966722], 
[4.9758243293363371, 2.2215462672477404, 1.5359190509238774, -0.89295389951767135, -4.9868340002858478]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(1.94729619402)+(1.-msk_arg1)*(0.508056393381)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[7.0143882030353781, -6.9442398926070847, -1.2288765408436668, -7.5617274269503163, 
-2.8223813530432573], [8.90652548380681, 5.7904797465893161, -4.5799932139442907, -0.43357154309134444, -4.7366531683936799], 
[-9.1525377804922829, 7.8600579897687721, -4.4042592292415792, -7.1763451413910646, -4.798650113007815], [-2.5302070330260236, 
7.0865618882211647, -4.3251607204511302, 1.6539166781511663, 
-9.013820964979713]])+(1.-msk_ref)*numpy.array([[1.9682968069240381, -0.40531957448459233, 1.6630858028484088, 
-1.7642185304057898, -0.27237546192856044], [1.7091292601695423, 2.4493476817399507, -0.24561598303184964, 
0.027116984530705338, -2.2375024120911564], [-2.2396782940168718, 2.5022973879747128, -2.135466917774576, -1.5574339811226137, 
-0.90676491794524716], [2.5279993628624591, 1.1286707842679846, 0.78033349353829651, -0.45367093764487987, 
-2.5335928965773844]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[4.7831110561810046, -3.9672702759088576, 2.7829436697266079, 1.5847476149920752, 
-2.3891146334183477], [0.97541176887684422, -2.5446067225897262, -2.3127338689933028, -3.8920483441765272, 4.8889135284163352], 
[2.2400065275766998, 1.3008004133502151, 0.3744886390624993, -1.3577281900108993, -4.8337670378333861], [-3.8979391664294329, 
4.1135665004411255, 1.5121436065257843, 3.6386240671438603, 
4.0588825972680151]])+(1.-msk_arg0)*numpy.array([[1.173969646497067, -1.8012556549284797, 3.6554163809402738, 
4.879456187664724, -1.8724423640533194], [-1.644787755807374, -1.5510376620673072, -4.204332703733785, 1.8937473524853967, 
-3.8604008956328073], [-2.5970315708966698, -1.7073526456055719, -3.6337895554855058, 4.5676591545410812, -3.194328878744451], 
[2.7236909000324836, -3.8295570142048971, 4.7153873344011448, 0.4735417364360881, -4.5317014940667475]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[3.8508050161726857, 0.60990907874867339, -2.6095542135561112, -4.4592892915310758, 
4.1596060132089807], [-0.16505454761371041, -0.26533100268792253, -1.9006047369625714, -2.6352112393358551, 
4.0519687643372997], [1.7973323480371679, -4.0954095293028843, 0.56655178206654444, -4.2804677572583767, 3.0806422193932654], 
[-1.4761764965316915, -2.0489453074644146, 2.5093767146865433, -1.5971896537357044, 
0.90790773068629615]])+(1.-msk_arg1)*numpy.array([[0.10314789696349536, -2.5451098321329999, -3.1923298130929165, 
4.6951395995678418, -3.2725656062792274], [-1.823080794418297, 1.8890052989006847, 0.29966529677206388, 3.5739294374297064, 
-2.1680427730942444], [2.6900857612694757, 4.3295659602681305, 3.1532888108179904, 2.4934727120937072, 2.073662928195418], 
[-2.6285361005355643, 4.183359149636706, 2.6362441527820568, 4.4653229552941198, 3.9256017055079937]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[18.418828048052845, -2.4196741591265667, -7.262242379424376, -7.0668480693135729, 
-9.9377755954125284], [-0.16099614824905659, 0.67516305315116043, 4.3955929467424459, 10.256369540612489, 19.809724908689045], 
[4.0260361918280134, -5.3273104085556016, 0.21216720582453391, 5.811711740462429, -14.891106815461054], [5.7540461823934619, 
-8.4284727780216571, 3.7945379554779337, -5.8115727138759032, 
3.6850908880077036]])+(1.-msk_ref)*numpy.array([[0.12109250013515054, 4.5843934775436397, -11.669294692143849, 
22.909727971060981, 6.1276904803410615], [2.9985809685067952, -2.9299183624396727, -1.2598926073928784, 6.7681194101021296, 
8.3695142630232571], [-6.9862376504364301, -7.3920958965876205, -11.458387946179725, 11.3893334599932, -6.6239613763164042], 
[-7.1593198574355856, -16.02041237442948, 12.430912288417588, 2.1145167859979019, -17.789655113961548]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[3.989847329431889, 4.5691531628158693], [2.4655767959246013, -0.28794635408316971]], 
[[-3.251183575943386, -2.7873251603726112], [-2.6388553635517931, -0.29180267569557383]], [[-2.4132990863907802, 
-3.8954849590407647], [-2.0534174591291987, 2.1758359306347952]], [[3.6541704964038342, 1.2858912151980189], 
[2.8684314117714793, -1.2737524605406945]], [[2.2500555904335586, 0.97849480530932009], [-1.8445919048396386, 
3.152005439498927]], [[-2.1121805349470146, -3.0461818750262513], [3.818928875589048, 
-3.5391654105711057]]])+(1.-msk_arg0)*numpy.array([[[1.7071127724740585, 4.3217649473296706], [-3.821066521437908, 
-0.72145674429873541]], [[-1.0034229725430777, 1.0930235922751628], [-2.7963411594457566, -1.4945443447064131]], 
[[3.448507731453164, -2.9569659394377834], [2.2775815792534226, 4.093111255975229]], [[-1.7876565612823336, 
-3.9859571640581493], [-4.8230154251346704, 0.5810641958169942]], [[0.58501799446937675, 4.8825044884557691], 
[4.5853501111260506, 4.8037319075704623]], [[1.9753359401118207, 1.3950858525140184], [1.892903122610333, -4.172161134925485]]])
      arg1=2.23442056913
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[8.9149969405620322, 10.209409810490891], [5.5091355075780211, -0.64339325636877653]], 
[[-7.2645114560982122, -6.2280566711838654], [-5.8963127032732112, -0.65200990070070419]], [[-5.392325118088765, 
-8.7041517192084541], [-4.5881982076843215, 4.8617325584576054]], [[8.1649537202644851, 2.8732217808991307], 
[6.4092821475943422, -2.8460986978092095]], [[5.0275704929456264, 2.1863689197677973], [-4.1215940938202245, 
7.0429057880189543]], [[-4.7194996329968628, -6.8064514388627915], [8.5330932316520762, 
-7.9079839909255352]]])+(1.-msk_ref)*numpy.array([[[3.8144078926367357, 9.6566404932487213], [-8.5378696315062808, 
-1.6120377891970337]], [[-2.2420689293855616, 2.4422743971215262], [-6.2482022049641381, -3.3394406252855626]], 
[[7.7054166079550228, -6.6071055172899422], [5.0890751285503111, 9.1457319820793685]], [[-3.9943765910654276, 
-8.9063046750336401], [-10.776644871141325, 1.2983417911171662]], [[1.3071762401522393, 10.909568457864131], 
[10.245600604952259, 10.733557382850721]], [[4.4137312555231523, 3.1172085245564447], [4.2295416725266746, 
-9.3223626575928797]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[4.399518373292068, -0.0050701407094111417], [-1.8680375980528527, -2.3435738396476724]], 
[[3.169307564857208, 4.6941436537702526], [-2.6523495226459382, -4.7659144877298445]], [[-1.1593904752159712, 
-4.1889594445416032], [-0.56322172169446283, -1.7332143950392211]], [[-1.1987163223201125, -4.5668410947657385], 
[-0.71725064662965821, 2.3029586565887818]], [[-2.6744178411938471, -3.8361030040224877], [1.5070623148457774, 
-1.2333238569337635]], [[-0.20792827388576285, -2.5473000331013673], [3.6617594218979335, 
1.7518396980392659]]])+(1.-msk_arg0)*numpy.array([[[4.7139313992685352, -0.68841029375747809], [-1.5916362808866413, 
4.0399154348681172]], [[2.3814535954465601, -2.5372613624708129], [0.3934280258123497, -3.8395756274998072]], 
[[-1.5031994204728294, -1.9367144781874037], [3.7689133617163719, 3.2382653732562527]], [[4.2273064743255286, 
-4.3847861897098674], [-4.823333539468404, 2.1667852827018823]], [[-1.7122113517588144, 3.9572359381043949], 
[-0.1229151702841218, -3.3690045114345724]], [[-4.1907375113304504, -2.7523694268233312], [0.25918342804129413, 
-3.3320524523800299]]])
      arg1=numpy.array(-4.5840981218)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-20.167823911822641, 0.023242022503261673], [8.5632676446815683, 10.743172436623119]], 
[[-14.528416855461169, -21.518415106696807], [12.158630465112271, 21.847419651850828]], [[5.3147596998676763, 
19.202601122009899], [2.5818636365752541, 7.9452248529720304]], [[5.4950332417158743, 20.934847685064174], [3.2879473420732022, 
-10.556988452246388]], [[12.259793802718974, 17.585072575762002], [-6.9085215269166218, 5.6536775761383602]], 
[[0.95316360978836323, 11.677073297395197], [-16.785864488397348, 
-8.0306050694724487]]])+(1.-msk_ref)*numpy.array([[[-21.609124073670095, 3.1557403346398583], [7.2962168857975289, 
-18.519368757200475]], [[-10.916816954034955, 11.631055046212326], [-1.8035126741889693, 17.600991422522092]], 
[[6.890813640076888, 8.8780892019172892], [-17.277068662662277, -14.844526215426516]], [[-19.378387669218913, 
20.100290136733495], [22.110634219080964, -9.9327563447725975]], [[7.8489448417182821, -18.140357831374715], 
[0.56345520123988724, 15.443847253195202]], [[19.210751954637086, 12.617131519994244], [-1.1881222656851858, 
15.274455388686714]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-3.272507496762429, -2.5281628592389715], [2.0405641530717755, 2.8253338982516283]], 
[[-4.6128156012885757, -1.822430243787454], [3.0468010790072384, 3.0047460752883595]], [[4.3620845179355161, 
-1.0691635543925173], [2.4522958035006504, 4.25327344430913]], [[-3.4686597740848937, 0.90004605802960214], 
[-4.1779937750220677, 2.2432119547725868]], [[-2.815546503911416, 0.69806900720275955], [-0.33054235664646114, 
-4.9258384610127761]], [[-0.52186599107445364, 3.5620448172064485], [2.4615999587036965, 
2.8596104913430924]]])+(1.-msk_arg0)*numpy.array([[[-3.7438451320454691, 3.9382490391706373], [1.5986025161292838, 
2.6940769756137364]], [[-1.8672642293945185, -1.4543751005263603], [3.6989088067658784, 4.6081404196355287]], 
[[-0.35550733881442653, 0.041069296121913013], [-2.7797438217723336, 0.92389549582121866]], [[-1.7046737092973041, 
1.1286682618732167], [-1.3711957369093275, -0.55446733506489654]], [[1.7453063003927607, -0.68317790423665059], 
[3.6271185765466836, -1.5245412947361716]], [[0.90482176143903637, -0.065810505366766137], [-0.93697839637859914, 
-1.7781604840127319]]])
      arg1=numpy.array([[[2.0629380493765659, -1.9462520408178197], [-2.4999481888199124, 0.3032496361633088]], 
[[1.0185354276245198, -2.6501430999265638], [-4.1328533793659794, -4.535739799531525]], [[-0.38413301815485212, 
-1.4627352114753487], [-1.2930544530561261, 4.5774206959026174]], [[4.5682892521468457, 4.8293384583394978], 
[2.7623185216246382, 2.5861116026581019]], [[-0.15060709879289114, -0.42956808979188565], [-2.8361131742311665, 
1.2689331367312704]], [[-2.4547889683232658, 1.9821522381546206], [4.0691651903205219, 4.2211254603220087]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-6.7509802319412735, 4.9204421243136629], [-5.1013046586426238, 0.8567814766846692]], 
[[-4.6983161110115157, 4.8297009356708065], [-12.591982135630978, -13.62874636117156]], [[-1.6756206913211229, 
1.5639031778360744], [-3.1709520089273666, 19.46902188931362]], [[-15.845841165306126, 4.3466270423192208], 
[-11.540949587975899, 5.8011964634587478]], [[0.42404129047056588, -0.29986816996700749], [0.93745553232644518, 
-6.2505596493644759]], [[1.2810708778326567, 7.0605151068328285], [10.016656864451516, 
12.070774651612256]]])+(1.-msk_ref)*numpy.array([[[-7.7233205738698318, -7.66482522973467], [-3.9964234648403578, 
0.81697786265081296]], [[-1.9018747703743153, 3.8543021373649364], [-15.287047762008942, -20.901325903170768]], 
[[0.13656210703498528, -0.060073505548030148], [3.5943601270979704, 4.2290583634232561]], [[-7.7874425846001705, 
5.4507210437715212], [-3.7876793808373801, -1.4339144085062463]], [[-0.26285551840710786, 0.29347142731096176], 
[-10.286918779442646, -1.9345409672059224]], [[-2.2211464782793722, -0.13044644050682216], [-3.8127198746261399, 
-7.5058384916046483]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[1.9087810368027149, -1.2211826307379514], [-3.712334087187358, -3.0587761169129459]], 
[[2.1194645685954683, -1.0645493952040819], [-3.2094959615867791, -2.2082154226130593]], [[-1.8965112625549976, 
-3.4945618887511842], [4.3019058532817969, 2.7882537356797563]], [[-2.5510419799825246, -2.5124453808362213], 
[-1.2388090957995979, 3.6636166446895988]], [[-0.82857213905159188, 2.6940552459729545], [1.2184714139218045, 
3.4719771276586613]], [[2.2264361624328792, -0.9144631492375801], [0.99724783740711054, 
2.260180510188067]]])+(1.-msk_arg0)*numpy.array([[[2.5880613694051, 2.998165314828622], [-2.8684394306558412, 
-0.69452240247736086]], [[1.7014217816950339, 3.347695079087682], [2.0494811826636443, -2.3420239673811807]], 
[[-2.7324768505676476, 2.680013232989622], [-4.1492502774932918, -2.215104799421963]], [[-4.6659225570154073, 
-1.8038182324804017], [4.1835384404059273, -4.7843230393218983]], [[-1.5110462635107047, -0.49512225189348857], 
[-2.1830470569954272, 1.1930904669523565]], [[3.4404622789168524, -1.2109796733770448], [1.3366342300532672, 
-0.022299011966195437]]])
      arg1=Data(4.25396072866,self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[8.1198795701763249, -5.1948629536852113], [-15.792123418573839, -13.011913479121469]], 
[[9.0161190405988236, -4.5285513209206316], [-13.653069779394217, -9.3936616882250021]], [[-8.0676844323769696, 
-14.865729038631642], [18.300138558268358, 11.861121893130948]], [[-10.852032400017608, -10.687843982989266], 
[-5.2698452438426138, 15.584881331387448]], [[-3.5247133403901745, 11.460405217218803], [5.1833295438224294, 
14.769654351877803]], [[9.4711719998657102, -3.8900903246666041], [4.2422531370744396, 
9.6147191300306432]]])+(1.-msk_ref)*numpy.array([[[11.009511428820344, 12.754077507321961], [-12.202228690559803, 
-2.9544710253157103]], [[7.2377814422233362, 14.240963397979009], [8.7184124651859438, -9.9628779828281999]], 
[[-11.623849214296863, 11.400671045436322], [-17.650747733852544, -9.422968826615044]], [[-19.848651320528667, 
-7.6733719226188146], [17.796608232340912, -20.352322322515295]], [[-6.4279314641682381, -2.1062306154423318], 
[-9.2865964492829498, 5.0753599921581074]], [[14.635591422960378, -5.1514599737556859], [5.6859895232339568, 
-0.094859121192192572]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.45647684561897606, 1.6929656758573035], [-0.026956939588139939, 4.7420301400465927]], 
[[-3.0946818092082351, -2.9128041725223541], [-0.91764848859665182, -0.60050401884054061]], [[4.2031155666270941, 
1.1380080561411878], [-2.8489046391139237, -3.9330096337568419]], [[0.62535760601320334, -2.7627525375983031], 
[0.90905262704097023, 0.49913341813736434]], [[-1.4655518656278952, 4.3225382490587929], [2.2318763360688596, 
-3.3319187083736637]], [[-3.3897692100332835, -3.830736902882089], [3.4419355544417805, 
-3.8326140547809606]]])+(1.-msk_arg0)*numpy.array([[[-0.78229279375876626, -4.8608091883683908], [0.72164598859131956, 
-3.278366875103802]], [[-4.470728851913166, 1.5573384379005919], [1.6009497278575076, 4.7881669824531876]], 
[[-3.5252817895950916, 3.8411092308106838], [-1.4761621427874694, -4.4715265901703258]], [[-4.9226770593661184, 
1.8245329937687904], [-1.3906134358729716, -3.2317798668526301]], [[-3.4904784764250785, 4.2036701353800137], 
[-0.53177029864648784, 0.22362198399806488]], [[1.5087157601789078, 0.69716139958736889], [3.3571581509925448, 
-3.2999368990481912]]])
      arg1=Data(numpy.array([[[-0.65339614528175183, -4.2422905668412136], [-3.5902522453192764, 4.3546969213975242]], 
[[-0.98993237305464632, -2.9019079521560931], [0.77038151189692883, -1.6753968416837886]], [[1.7594430633575673, 
2.3371376507886819], [2.8524979621691235, -1.0391330558898169]], [[-3.4926694868339392, -1.9894174242439111], 
[-3.0598825289165124, 0.90456005102767101]], [[-2.2026743128144641, 1.7050617389014828], [2.8046921079665035, 
4.337753440710479]], [[4.3465523221413687, -2.8180367673721838], [3.1073564586575415, 
-1.5628589782573767]]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.29826021133781228, -7.1820523166753985], [0.096782212883255503, 20.650104052035168]], 
[[3.0635257072385542, 8.4526895913160676], [-0.70693943003502024, 1.0060825365838639]], [[7.3951425281922516, 
2.6596814749084099], [-8.1264946774866296, 4.0869203195698365]], [[-2.1841674288818358, 5.4962680371721451], 
[-2.7815942513483232, 0.45149615017995015]], [[3.2281334485158797, 7.370194583408356], [6.2597259457295262, 
-14.453041841415475]], [[-14.733809231393481, 10.795157438451174], [10.695320675377692, 
5.9898352857098338]]])+(1.-msk_ref)*numpy.array([[[0.51114709592367036, 20.620964967030321], [-2.5908911308656339, 
-14.276294138226149]], [[4.4257192216582748, -4.5192527971420757], [1.2333420718178434, -8.0220798398566675]], 
[[-6.2025325910838349, 8.9772010041196015], [-4.2107495041324627, 4.6465110901362632]], [[17.193283958785464, 
-3.6297577289115388], [4.2551137569042687, -2.923338961270415]], [[7.6883872794532877, 7.1675171107992774], 
[-1.4914519598647951, 0.97001703050610955]], [[6.5577119908569124, -1.9646264568298564], [10.431887063221494, 
5.1573360103602726]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-1.4683792089546777, -4.7876622163778446], [-0.81581710334416968, 3.2825673454978901]], 
[[-3.3010127861672025, 3.9470160919861037], [4.371997679025263, 1.5513960719177868]], [[3.0499112336904375, 
-2.0878037063155839], [2.5929358407046168, 1.3634179710274994]], [[3.5859041898936077, 0.72777930128184565], 
[-4.0873157342160438, 0.97987327404801938]], [[-4.2927668123594875, 2.382100180175768], [-3.270261704065125, 
1.3554846370187814]], [[-0.0023274183456631903, -3.5114241528664616], [4.6443705329506741, 
-4.3600125864930996]]])+(1.-msk_arg0)*numpy.array([[[-0.010578750341324827, -0.65082711598339849], [1.3746119546934734, 
-4.8991503817130475]], [[4.1127100155491885, -0.0049689082090189629], [-4.1767011449236389, 4.3211787089266842]], 
[[3.4671305334736893, -1.6895130001769756], [4.3844330222678671, 2.3483971189700101]], [[3.2628480012743299, 
4.5426831907152252], [4.847759960619646, -1.119962404654522]], [[2.1644726395400813, 3.2325555696791159], [-1.6979500994151522, 
0.21593171272347966]], [[-2.0735585651249746, 1.2192383783314114], [4.8080065844490214, 4.1063063820704961]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.08277067374)+(1.-msk_arg1)*(-4.6563933156)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-4.5266763632982787, -14.759264676433295], [-2.5149770413267318, 10.119402347284739]], 
[[-10.176265410844348, 12.167745457163383], [13.477866230568129, 4.7825983138670285]], [[9.4021769087378342, 
-6.4362200383599948], [7.9934265686193191, 4.2031049371366942]], [[11.054520275453365, 2.2435766869482769], 
[-12.600257079766367, 3.0207245932190072]], [[-13.233615638355698, 7.3434685773619144], [-10.081466876754224, 
4.178648287709616]], [[-0.0071748970215401147, -10.824915401526845], [14.317529276972859, 
-13.44091893878789]]])+(1.-msk_ref)*numpy.array([[[0.049258822376753629, 3.0305070324767831], [-6.4007339173795126, 
22.81237108953129]], [[-19.150395425407325, 0.023137190970309383], [19.448363292484256, -20.121107655762309]], 
[[-16.144323440382003, 7.8670370406445667], [-20.415644617587105, -10.935060647147916]], [[-15.19310362295492, 
-21.152519644138071], [-22.57307707626607, 5.2149854547574117]], [[-10.078635930555055, -15.052050146961674], 
[7.9063234931402722, -1.005462983751823]], [[9.6553042421545268, -5.6772534349862305], [-22.387969721192615, 
-19.120577589281584]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[2.2825591144708763, -1.6406692868297332], [-3.9692464525679974, 0.4929674573547862]], 
[[2.0610999921278257, -4.8473768326878606], [2.5723894084835468, 3.1027533911262264]], [[-4.1699141124661114, 
2.8672896951026496], [2.8999386024511402, -0.74750653497685704]], [[3.5595918928143746, 4.6480554810166641], 
[-4.1751019637235878, 0.68303514491325679]], [[-2.9789692354222685, 0.70984025449916821], [4.0831158285024927, 
1.7569574005376793]], [[-1.8434548397827144, -2.9347331820047771], [-1.5051542245615614, 
1.3215890300096795]]])+(1.-msk_arg0)*numpy.array([[[-3.5547857929892954, 2.3234934619530438], [0.87119614333688844, 
-3.9598516860251829]], [[1.6643650471425184, -4.1842460749840829], [1.6590749837977983, -3.9379409503679552]], 
[[3.7166927510401528, 3.987381354407642], [-1.1626283798171699, -2.8062605907252944]], [[4.5721265700447873, 
1.6261754559178767], [-1.1889575493305369, -2.6995576956849296]], [[3.6498466599955357, -1.1758983572556336], 
[1.2318709663484073, -4.8048303234811787]], [[-4.14193930109683, -4.1007554935570409], [-2.8406892702913988, 
-1.1533425459303235]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.19542829837474152, 4.7209948236316652], [-4.4804284601859221, 4.5005331491361193]], 
[[2.7218395508511017, -2.601512101520397], [-4.0239108242979951, -2.8272924292405244]], [[-1.2242182076422301, 
4.0417658086553416], [-3.2425056531451659, -2.381643065661827]], [[3.7178569065028864, -0.35675100896916145], 
[-0.67107186079772863, -2.7889342568486395]], [[-2.4796194825977325, 2.0817502040703584], [-0.48727518255226876, 
4.3905867703094827]], [[-1.5947855934345156, 2.6010103218010725], [-1.4782918941276746, 
4.8558435127261834]]])+(1.-msk_arg1)*numpy.array([[[4.9799193607340868, -0.83047960430503753], [-2.1058434302041507, 
3.1360423504288555]], [[4.0661827423351902, 2.3078280399164077], [1.3148855972496678, 0.95927264157051084]], 
[[-2.7433881505446478, -1.7435275457503896], [-0.56835169996165646, 1.2987982985456092]], [[-2.6025190344851667, 
0.36417082056186345], [2.0944645529015968, -2.5905659006358093]], [[-0.78287059509307788, -1.2555636246673174], 
[-0.38140944285351175, -2.5293035850346524]], [[-3.0930004904426989, 4.5736584963478037], [2.1559599779615883, 
1.2621053757855734]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.44607664368080019, -7.7455912104146263], [17.783924771577666, 2.2186163832705614]], 
[[5.6099834768324106, 12.610509490867083], [-10.351065585106461, -8.7723911725315435]], [[5.1048847807853033, 
11.588913453175689], [-9.4030673122217134, 1.7802937555645315]], [[13.234053303031605, -1.6581984825973359], 
[2.8017934438162388, -1.9049401142801565]], [[7.3867101542123281, 1.4777100946609985], [-1.9896010107156101, 
7.714073918798074]], [[2.939915220632606, -7.6332712981265303], [2.225057289581382, 
6.4174295178625913]]])+(1.-msk_ref)*numpy.array([[[-17.702546593769867, -1.9296139308881057], [-1.83460267486518, 
-12.41826258879208]], [[6.7676124316368034, -9.6565204177584381], [2.1814938009529508, -3.7775590178081564]], 
[[-10.196330852418743, -6.9521092268212206], [0.66078181609275499, -3.6447664805096087]], [[-11.899046426616938, 
0.59220565015917559], [-2.490229441977561, 6.9933821132403597]], [[-2.8573576267091876, 1.4764152036762275], 
[-0.46984721894236314, 12.152874562664154]], [[12.811020289676385, -18.755455204552092], [-6.1244123765731651, 
-1.4556398273408808]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-2.2558543609702331, 3.9648886091445732, 0.31090602259785349, 1.1175893725005608], 
[1.7479896187664874, -4.5187798766651532, 1.1757910449637832, -2.8432234993189995], [2.496381177634575, 3.0300829191304839, 
2.5988984154665378, 2.8394168038313827]], [[0.98157986864442925, 4.6616680598294131, -0.83670857853612546, 
-1.6670253811190214], [4.6227015346244045, -0.30517330290470301, -2.782739204881862, 1.2706463935650278], [4.5757315506012901, 
1.8405261244624151, 4.8243887200063469, 1.3990591217093353]]], [[[1.9315140909866813, 3.4333213444122919, 4.978290132334843, 
1.4240363207499573], [1.7554258145826198, -0.96332635421361967, -3.0889444941641608, -0.96738320979379822], 
[4.2056199610184457, 2.3236669006866784, -3.7871638506993088, 3.439004579967289]], [[-3.1957487837808198, 3.088631188802367, 
-4.382186554455247, 0.34716696268649905], [-0.69381964785732464, 0.7518356249331859, -2.1291809451377142, 3.8871819062870419], 
[0.94807246141196622, 1.2274370948161986, 2.1095008152008248, -1.1838939760075675]]], [[[-0.61887238796764876, 
3.5193786740109338, 1.9215213269505673, 3.9053704346229594], [-2.13825813201723, -1.8795310923861406, -3.9207800912081794, 
0.32589269745880145], [-1.7873658334639417, 0.49188460466482908, 3.9204695860458578, 3.5585978874372675]], [[4.206280789254599, 
2.9076932168338576, -3.6630203519852245, -4.0806264760269428], [-2.7351288364838378, -0.17579666334200628, -3.5975068233950669, 
4.7860443546561502], [-1.6057334351546926, 2.5830887040178583, -4.9056664783715167, 
3.1684052919060672]]]])+(1.-msk_arg0)*numpy.array([[[[0.30324540066302585, 2.9503993828206632, 2.6334023350278901, 
-2.2752055147380523], [-3.2691577238922589, 4.4186892799653386, -0.416366755789217, -2.994866899487941], [-0.80325677981231003, 
0.38878048483294059, -4.7425196102840639, 0.61728155151775077]], [[1.2971943214795898, 0.13382529707250335, 
-2.8123819430773835, 0.59737290230525097], [-2.0457172281677405, 4.9991562277006896, 1.2331337503686139, -2.8099948025636392], 
[-3.7428656302312566, -4.4553897042735846, -0.12936789600193421, 0.27822517484437448]]], [[[-3.4529656750383109, 
-1.5115068904974427, -0.44583645655190463, 3.2507016977866154], [3.1075676213256145, -4.8638700374541486, 4.4643182298089794, 
-4.3799081017156682], [-0.59710480464845972, 1.7645892751200822, -3.9603655873381696, 0.73993015603214474]], 
[[-4.027831785222828, -4.7968225200295267, -0.86640842949085251, 0.87964983536565544], [-3.6406809404904661, 
-2.8948244727860706, 2.5509340562377725, 1.7640463769561654], [1.2928594069067012, -1.9419220228066747, -0.33065189456712574, 
3.5726028812012665]]], [[[1.6748427314535128, -2.029980923766229, -3.5595780903421739, -4.7374272904208787], 
[-4.1151085084315762, 4.8446367834805066, 3.1387260066961389, 1.2274886537242482], [-1.2878519740398975, -3.0002518767256992, 
2.4618107296462801, 0.50439339404393824]], [[0.66916196621683888, -2.1223130023852144, -3.0095870285848889, 
4.7134300599991121], [-0.83238149247905113, -4.3518222086805149, 3.8879604344238743, -0.59612426020536269], 
[-4.6058101795874196, 4.8547593830713218, 3.8299181766916508, 4.7411148400140561]]]])
      arg1=0.206884040831
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.46670026572370199, 0.82027217690453924, 0.064321494273730809, 0.23121140537267137], 
[0.36163115566100895, -0.93486344051018821, 0.24325240255498379, -0.58821756652470358], [0.51646142548343243, 
0.62687579836263529, 0.53767060590093918, 0.58743002198001237]], [[0.20307320962349859, 0.96442472523020706, 
-0.17310165172549605, -0.3448809470137017], [0.95636317303865281, -0.063135486058660525, -0.57570433128473808, 
0.26287646036804019], [0.94664583294618443, 0.38077548188376115, 0.99808903293429529, 0.28944300446066418]]], 
[[[0.39959944006529491, 0.71029939320325552, 1.0299287790064089, 0.29461038832682762], [0.36316958589986081, 
-0.19929684879868617, -0.63905331885527816, -0.20013614747418121], [0.87007565173490964, 0.48072959795925124, 
-0.78350376072167383, 0.71147516393986765]], [[-0.66114942186925441, 0.63898850097601623, -0.9066044620608753, 
0.071823304083599743], [-0.14354021235664846, 0.15554279212685992, -0.44049355759040787, 0.80419590021772125], 
[0.19614106181747756, 0.25393714604140954, 0.43642205278498553, -0.24492876967189664]]], [[[-0.12803482038146294, 
0.7281032812937458, 0.39753209666243322, 0.8079588164566246], [-0.44237148269142001, -0.38884498726030409, 
-0.81114682847879249, 0.067421998127583732], [-0.36977746607024636, 0.1017630746356072, 0.8110825899161126, 
0.73621711064559814]], [[0.87021236655070033, 0.60155532219540908, -0.75782045206480886, -0.84421649448232161], 
[-0.56585450588510344, -0.036369524076797048, -0.7442667485409814, 0.99015619568754731], [-0.33220062156220748, 
0.53439982891206461, -1.0149041040145652, 0.6554924897797767]]]])+(1.-msk_ref)*numpy.array([[[[0.062736633852575246, 
0.61039054638315782, 0.54480891620429872, -0.47070371060992999], [-0.6763365600326281, 0.914156293415747, 
-0.086139636905357572, -0.619590165917003], [-0.1661810084324486, 0.080432477698464899, -0.98115162069571482, 
0.12770570170840684]], [[0.26836880297069426, 0.027686318223765334, -0.58183694074392234, 0.12358691991183844], 
[-0.42322624656088675, 1.0342456411320597, 0.25511569316131544, -0.58134307946840758], [-0.7743391658696217, 
-0.92174902549684834, -0.026764153078681671, 0.05756034843270913]]], [[[-0.71436349170258606, -0.31270665324997515, 
-0.092236447681222092, 0.67251830277421065], [0.64290614665534873, -1.006257087425227, 0.92359619493827338, 
-0.9061330865512689], [-0.12353145478526416, 0.36506535964384623, -0.81933643587646388, 0.153079740612625]], 
[[-0.83329411551434429, -0.99238602609273596, -0.17924607690308772, 0.18198551245674996], [-0.75319878434498733, 
-0.59889298442638328, 0.52774754544782365, 0.36495304267793538], [0.26747197832719799, -0.40175267505690848, 
-0.068406600056464956, 0.73911452034730685]]], [[[0.34649823203949265, -0.41997065631855579, -0.73641989898339943, 
-0.98009810098521521], [-0.85135027668225682, 1.0022780341248316, 0.64935231932657167, 0.25394781275664768], 
[-0.26643602038152375, -0.62070423176773326, 0.5093093515102769, 0.10435094352826091]], [[0.13843893154134099, 
-0.43907268984157488, -0.62263552570613323, 0.97513345698680809], [-0.17220644667698512, -0.90032256350980977, 
0.80435696526456968, -0.12332859578866188], [-0.9528686212534907, 1.0043722384318934, 0.79234894844597437, 
0.98086099614581634]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-1.9246773623150828, -3.1966581223806689, 0.34956243519964403, -0.99145005267955888], 
[-1.9352716747749255, -1.4654699552786599, -1.8895331656075687, 0.72253493321303175], [-3.8759854150642958, 
-4.6662253313388815, 4.9555192753384816, 4.0380580688455296]], [[0.53400931832665943, -4.60955361867485, 3.8747207141129678, 
4.0991131041707547], [-0.91787656191600675, 2.9992998337690313, 3.7762612810154721, 0.21260386646796103], [3.067011548440723, 
3.9440309438689312, 2.1504200706904637, -0.861106910786825]]], [[[3.010140529995887, 4.4330861543637816, 3.8082067692261674, 
-4.0317157298201769], [-0.81043682330020594, -4.2366015454517711, -0.5950693051726148, 2.8778192471150676], 
[-1.7485235628683626, 2.1335479886125777, 3.9967589873996161, -2.7772902789407459]], [[0.75332766678173702, 
0.66359994886118656, 1.9679649005084121, 1.0209923718247076], [-2.2209376791021818, -2.4251206224840693, -3.0989033698376089, 
1.9581087239349078], [3.2287326925109721, 2.9129889511283906, -1.3149632908914297, -0.85319939733925487]]], 
[[[3.8907810237155811, 4.8147716073554996, 1.8256331912915762, -3.9964098002265205], [1.4205986141684424, 0.8097156847051945, 
4.036649994895491, -0.96955830976869084], [-3.8929385361701287, 1.0239058632476095, -0.014477786582698826, 
2.7945961158953834]], [[3.7936300858256491, 1.9494508289482742, -2.4003567042158434, -0.44587962520673585], 
[0.96426470458958491, -0.57751758672905495, 3.3404327942556264, -2.4574035697636631], [2.2354249694785278, 
-0.83293152253380676, 3.6303563686693217, 0.53501699595292607]]]])+(1.-msk_arg0)*numpy.array([[[[-1.2332116625543055, 
-3.9031800796229299, 0.73545227781183531, 2.0676955094220002], [-3.2361178448278025, -2.5040865643599086, 2.4838319567558429, 
3.7651830988422859], [-3.372471906796334, 4.6832378689253833, 4.8283981599434966, -3.0806063261635721]], [[3.323762406986873, 
3.3681674422533874, 0.6031174156633039, -2.2961203971633806], [1.8093998846126933, 3.802436913582337, 1.7547736347315777, 
2.3180337320293845], [1.8260165871841538, -1.2085075918408759, 2.9383007648967654, 1.0383914393011953]]], 
[[[-0.782985247609985, -3.7574203392029304, 4.0484177982315526, 4.637502961860152], [-3.9678940812924015, -2.1404115041907379, 
0.15928508188156876, -2.3771417409465592], [0.55160832408443294, 2.8448884048107912, 2.4465705356288669, -4.6663713959408675]], 
[[-2.0025756305325837, -1.3659537175098713, -3.7997188560101569, 2.8187578281131289], [-4.2651910093870651, 
-4.7168105728055494, -1.1985837230629026, -4.7486262345692234], [3.8901046840351565, -4.9053013611808272, 1.6346519169439571, 
-1.6498450371658659]]], [[[0.34555242204710623, -0.73972603555837857, -2.5915606182675379, -2.0356721820694892], 
[4.2911852429841488, 1.3195780898086653, -2.208996221981149, -0.62275150151432079], [4.084728560157739, 1.6551645795809637, 
-3.1324567064485818, -3.4133037185610924]], [[-1.2822307812565139, 1.2348872558697375, 4.8683376157025577, 3.7988627013630598], 
[3.5900711732933459, 4.4913476054744432, 1.4169462328013367, 3.4201082836266128], [-0.64378296552757774, -3.8621840866514101, 
-2.6570757487604659, 4.2008833449046552]]]])
      arg1=numpy.array(1.14762993396)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-2.2088173542077914, -3.668580549880136, 0.40116831442303269, -1.137817758481191], 
[-2.2209757043164315, -1.6818171879966801, -2.1684848220612731, 0.82920271768699938], [-4.4481968859198107, 
-5.3550998688464979, 5.687102258693761, 4.6341963148754752]], [[0.61284507872520055, -5.2900617149844811, 4.446745477250559, 
4.7042649010336834], [-1.0533826181350157, 3.4420862701543213, 4.3337504845471502, 0.24399056123424756], [3.5197942607913073, 
4.5262879716481415, 2.4678864437125609, -0.98823206715870582]]], [[[3.4545273776492271, 5.0875423705710965, 4.3704120830727078, 
-4.6269176567586587], [-0.93008155800269432, -4.8620507518212674, -0.68291934739681726, 3.302671512515222], 
[-2.0066579809819651, 2.4485235372717504, 4.5868002527630969, -3.1873014594082671]], [[0.86454138047889761, 
0.76156716548736292, 2.2584954288058889, 1.1717214082507608], [-2.5488145619971116, -2.7831410198262074, -3.5563942696748767, 
2.2471841855357413], [3.7053902866805681, 3.3430333176094216, -1.509091234685437, -0.97915716802308372]]], 
[[[4.4651767692991822, 5.5255760217814398, 2.0951512987569707, -4.5863995151106973], [1.6303214937616686, 0.92925395776452535, 
4.6325803670611823, -1.1126941390101244], [-4.4676527951749119, 1.1750650182200184, -0.016615141259788319, 
3.2071621559296388]], [[4.3536834448644157, 2.2372481260839989, -2.7547212059394548, -0.51170480483007552], 
[1.1066190392480171, -0.66277646991855177, 3.8335806670691008, -2.820189896480719], [2.565440610094976, -0.95589714819859961, 
4.1663056396269109, 0.61400151973288575]]]])+(1.-msk_ref)*numpy.array([[[[-1.415270618855788, -4.4794062970112982, 
0.84402704901586167, 2.3729492609271716], [-3.7138657085462161, -2.873764698486259, 2.8505199044992211, 4.3210368310713401], 
[-3.8703497116783274, 5.3746239662333855, 5.5412142614281041, -3.5353960346515803]], [[3.8144492316287755, 3.8654097793191728, 
0.69215559990774878, -2.6350964997606119], [2.0765214700851335, 4.3637904240212197, 2.0138307505415911, 2.6602448988057255], 
[2.0955912953598501, -1.3869194878143938, 3.3720819127728268, 1.1916890989097662]]], [[[-0.89857730800623059, 
-4.3121280557390804, 4.6460854504265994, 5.3221372178584518], [-4.5536740224735155, -2.4564003132014474, 0.18280032800054355, 
-2.7280790191758442], [0.63304222454075421, 3.2648790921363209, 2.8077575822320173, -5.3552674969560288]], 
[[-2.2982157386178335, -1.5676093746181465, -4.36067109978916, 3.2348908601264483], [-4.894860876429278, -5.4131530061702362, 
-1.3755305589441016, -5.4496656119789719], [4.4644005816364016, -5.6294706771854077, 1.8759754714898331, 
-1.8934115510467473]]], [[[0.39656630329360737, -0.84893174133628779, -2.974152541195477, -2.3361983318724331], 
[4.9246926370156379, 1.5143873160620622, -2.5351101883499161, -0.71468826455631451], [4.6877567677379828, 1.8995164171572829, 
-3.5949010831538621, -3.9172095211173805]], [[-1.4715264268147763, 1.4171935799017208, 5.5870499764032706, 4.3596885510880519], 
[4.1200731435180176, 5.1544049558616338, 1.6261299115745407, 3.9250186436741497], [-0.73882460221292878, -4.4323580683047714, 
-3.0493396660764511, 4.821059475686214]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-3.3436568840311818, 4.6351189364507537, -4.6891784356461317, -0.097647938657718036], 
[-3.9459517255028964, 2.8839562693107901, -2.7133043791419365, -4.444350213304503], [-2.3260279974394971, -1.7977473657188483, 
2.2987119075856723, 1.390446487938676]], [[-3.9540197673578872, -4.6092655909341884, 1.9893980986684054, -3.3040482202407242], 
[-2.0047097305590178, -1.3286140946304292, -4.3239061861982639, 2.6984658090339941], [2.9882655532001419, 3.9948966075361696, 
-0.35802232355277042, -4.5041054835031433]]], [[[-3.4874888473381604, 4.8302557591895763, -0.36905477414743615, 
1.136170792407504], [2.5472478867585568, -3.5012697269571156, 0.94086474298157086, -0.44070791671505294], [-4.0677287580084105, 
-2.9601397433796395, -3.1246473169692148, 0.92595433116909565]], [[-2.9832718881454312, -2.1873902775524789, 
1.8486527268143904, 3.5694349598666353], [-4.1655091791270591, -2.8715827536293115, -4.9946150754391123, -3.1617204853511738], 
[0.68114123457240083, 1.7041728294031699, -4.0238676705775962, -3.0544130553890039]]], [[[-4.4428359742756696, 
2.7391109402924743, -2.827503718140536, 0.76333863336732932], [-1.8691894865536307, 2.0405408164899992, -2.2095419799090044, 
-3.28839896332628], [0.46564891314370627, -1.3042459229753947, -0.079180944264970243, 2.6424067969100697]], 
[[2.1073790057446331, -1.3195344970362943, 0.33179571821869036, -1.3991560944949799], [-1.5741885022513689, 
-2.4172077200334341, -4.1949951508869079, -2.0985834760368069], [2.6329904694625617, 1.0722754664000504, -3.4549862131359954, 
-0.049431918163450383]]]])+(1.-msk_arg0)*numpy.array([[[[-4.4504813668458922, 1.4629801549779122, 0.79546605125113068, 
-0.0059114286910455505], [2.553345702537352, -4.1570208602155088, 1.5233759288217907, 3.7441212204076972], 
[0.61913950081627789, -0.27907044064479081, 3.1132329198822024, -1.5007067058072012]], [[4.2049018309249568, 
0.51732029161989068, -1.5929470407765955, 3.5649799143839811], [-3.7893554556680611, -0.3516749385232707, -2.5749601461997829, 
3.9672464270465344], [3.2792809305626651, 1.9035938196914701, -3.6805555110095978, 0.8733570745926027]]], 
[[[-2.3068120447638476, -0.41357798030830395, -0.052871290023299267, -0.78103375873681102], [4.4612728239256718, 
2.0769996817023006, -1.6768528784707746, -4.1253625593870247], [-2.2471585715230233, 2.5710364192819144, -4.4483312640839774, 
-1.0354777497755761]], [[-4.4286672960509152, -0.34223074628650796, -1.5376113213519349, -0.15197501787761158], 
[4.3506105619537969, 1.5288255199472438, 2.0060422041029957, 3.6442220013879147], [3.3011731233138306, -0.42019960549269708, 
-3.9114781628796171, -3.5903708076028362]]], [[[1.0649860521193393, 2.5773063627091828, -1.495010820882936, 
-1.3937900011435298], [-3.6646896166163598, 4.5962829902296978, -0.90412233683262322, -0.18323241521251976], 
[-4.9690517394481359, 2.4359248635698734, -2.5575265916667664, 1.5128821610263881]], [[-0.67294714471146477, 
4.1783541330663496, 4.0660656685937475, 3.5432451917690457], [1.3474987054972276, 2.8004649974525844, 1.9401774598202737, 
1.9868105305548456], [-4.6845913830123784, 0.22818320877480325, 3.5069968050926583, 0.40920941156702728]]]])
      arg1=numpy.array([[[[-4.6464591048250341, 2.6741097373009044, -2.6173934130536014, 3.3918401846478403], 
[4.0996398311803031, -2.0627904940870656, 2.1080745312060447, -0.35549334502966978], [-2.1959481263791303, 3.3539689095846832, 
-3.2837611328329244, -1.4449666124497829]], [[0.032386700153001158, 3.9057812009048973, 1.932437104768252, 
0.43106672608878238], [-2.2262359921766031, 0.045863804565256849, -0.66129269962445036, 4.2439021377850459], 
[-0.56159674706138141, -4.1272584127639211, -4.2139082847174638, -0.49651966983478513]]], [[[4.964396313017005, 
-3.3402953996881779, -0.20258333666623685, -3.4814311862153913], [4.8963126086539539, 0.88190160348098257, 4.9240995935378802, 
2.9531837409355868], [-2.1111608030156126, -3.938499145651261, 4.2247914003653246, -4.333640237231303]], 
[[-0.074514528815567971, -0.9949307250536279, -1.7960486583625404, 1.0021465043805158], [-3.1536837559744413, 
1.5948319012039853, 4.7056499358701771, 0.73040582944499466], [-2.8652246335632015, -1.7798627226775832, -3.8449565861346713, 
3.659085386301566]]], [[[-4.2976032586444148, -4.1886886179818319, -3.2468056966726442, 1.0602415070370883], 
[-2.3964274439769162, -0.029474867086793921, 2.0716349837310908, 0.019114157088874428], [-1.5258976307945673, 
0.94612350598780992, 4.7025437264345591, 2.4483552178938606]], [[-2.5102805707804854, 1.7891051556152409, 4.7872047200606787, 
-3.3917262055277551], [-1.2749133943041038, -2.1857393807531187, 0.72725808438265993, 3.5698041731415859], 
[-1.5382972931802739, 0.82938008519413309, -1.9521215941051882, 4.5811387238375048]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[15.536164972217588, 12.394816681510772, 12.273424750093175, -0.33120620228727532], 
[-16.176980865786319, -5.9489975776970949, -5.7198478570789462, 1.5799369238109442], [5.1078368228826641, -6.0295887719087826, 
-7.5484208177100598, -2.0091487514694464]], [[-0.12805765260445928, -18.002782895048554, 3.8443867020222386, 
-1.4242652491386372], [4.4629569560371456, -0.060935297178775674, 2.8593675947939112, 11.452024815699222], [-1.678200214032779, 
-16.487970631575706, 1.5086732353328156, 2.2363769675700258]]], [[[-17.313276775413488, -16.134481091738269, 
0.074764347559392055, -3.9555004295545375], [12.47212194530306, -3.0877753864229023, 4.632911698489675, -1.3014914541444891], 
[8.5876295112067371, 11.658507850309054, -13.200983113906123, -4.0127529473929924]], [[0.22229709907388659, 2.1763017948205441, 
-3.3202702497732379, 3.5770967676439551], [13.136698633575435, -4.5796917824352104, -23.502910109436279, -2.309339073576155], 
[-1.9516226442324938, -3.0331936920546867, 15.471596501721706, -11.176358174702619]]], [[[19.093546360669752, 
-11.473282818992599, 9.1803551794217757, 0.80932330302100863], [4.4793769835702415, -0.060144669351220668, -4.5773644636019526, 
-0.062854974355910337], [-0.71053257334804665, -1.2339777253157875, -0.37235185270640031, 6.4695504690129724]], 
[[-5.2901125733914496, -2.3607859716597979, 1.5883740283524375, 4.7455543913224911], [2.006954006679786, 5.2833861051375361, 
-3.0508441374285602, -7.491532050442169], [-4.050322112143717, 0.88932391767445262, 6.7445531939984864, 
-0.22645447449214906]]]])+(1.-msk_ref)*numpy.array([[[[20.678979667835257, 3.9121694779044214, -2.0820476028524681, 
-0.020050621382968481], [10.467797744895183, 8.5750631141741884, 3.2113899970015694, -1.3310101768393019], 
[-1.3595982267848155, -0.93599358150672607, -10.223113259765134, 2.1684710849709044]], [[0.13618289477097215, 
2.0205398698556083, -3.0782699675274792, 1.5367442202657704], [8.4359995025590102, -0.016129150650930006, 1.7028023465058237, 
16.836605592862874], [-1.8416335033044124, -7.8566236068070268, 15.509523360205863, -0.43363896632459292]]], 
[[[-11.451929209848863, 1.3814726250361555, 0.010710842346768284, 2.7191152851533618], [21.843786378432497, 1.8317093497227492, 
-8.2569905773007655, -12.182953635846181], [4.744113094359963, -10.126024740780098, -18.793271670478202, 4.487388041185163]], 
[[0.33000005684614941, 0.34049588453847951, 2.7616247507971954, -0.15230123291921485], [-13.720449857804525, 
2.4382197105866341, 9.4397323690901302, 2.6617609936054385], [-9.4586025525755595, 0.7478976139002782, 15.039463723885929, 
-13.137473353503289]]], [[[-4.576887527998923, -10.795533826532109, 4.8540096498299627, -1.4777540113056411], 
[8.7821627709166883, -0.13547483023031207, -1.8730114625551673, -0.003502333168145967], [7.5822642765195338, 2.304685772243606, 
-12.026880628832112, 3.7040729330074971]], [[1.6892861425313936, 7.4755149214552565, 19.465088760768666, -12.017717569533287], 
[-1.7179441484458564, -6.1210866293527957, 1.4110097427913073, 7.0925245232163361], [7.2062942441435771, 0.18925060913351696, 
-6.8460841936792818, 1.8746450814884676]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.9880853400774674, -0.98803142696469148, 4.9564403436066389, -1.6935652850268545], 
[2.6398260650475862, 1.4275346686164667, -4.0053418725584491, 3.4570567869071596], [0.66587629063995823, -4.1495847688375278, 
2.6753852761239507, -4.0937204929835893]], [[3.6104197884711127, -2.9008533003409207, -2.3572453400016222, 3.6713920207750643], 
[-1.6929621102545722, -0.17498188806579229, -1.9357187303621712, 0.40640438301811965], [4.8677708215127513, 
-1.3752670859364233, -3.8413599382081132, 0.96522512220276369]]], [[[3.0211101035056647, -1.5301881624106981, 
2.4289315975716814, -3.5251852140955577], [4.4380324803915148, 3.9198258978519647, -3.5371282790185599, 0.29249752683248165], 
[-4.7811535622986394, -0.1805805739385411, 4.290024808560176, -4.5134724344427291]], [[-4.8313596987392433, 4.5564496271380541, 
-2.9196769238210161, 0.30189485403705163], [-0.69621819467006052, 1.1137449290447616, -4.6560373426648258, 
-0.33861542573666625], [-1.684092035400039, -1.5858449781210604, 2.2461129113436504, 1.6171014369498025]]], 
[[[3.7201886977051792, 4.2597378446439667, 1.9548011508190104, 4.9310413426570392], [4.9518903391992986, 0.5393098886473382, 
-4.8300616279865949, 4.2735606834167932], [3.2869559906285435, 3.079686901472062, 2.7593629376491089, 1.8241494283492958]], 
[[-0.67448753292407204, 0.57569086526296154, -4.5727563231624204, 0.45561431559156684], [-1.104361640433642, 3.76103618349706, 
0.26903849652292955, -3.7599240807927958], [1.0120358374874101, -0.68141529400753953, 1.6954653395724595, 
-3.9658357299859528]]]])+(1.-msk_arg0)*numpy.array([[[[3.7064608552569496, -1.4990736361949777, 4.7394346691449467, 
4.9548176582996906], [1.1596965087204572, 3.3500132320467788, -1.2561581641401141, 0.77396475077453619], [-1.5423825469625063, 
0.072295522366633058, 3.8000908216763261, -1.8130565269067356]], [[1.4367955420160028, -4.1442981087784982, 
-0.4975360177232151, 3.6805637246724494], [-4.209077387033525, 2.381703945566418, 1.3088454174460331, 2.8684784611144778], 
[-0.2406198228743861, 1.6742310545992298, 3.7672088999175752, 1.0336421288227102]]], [[[3.6917099482514697, 
-3.7290058330322626, 1.4906409235327782, -0.40665919501422199], [-1.8479342068281457, -1.560333268298991, 3.9450205244724952, 
-4.4424874108360735], [4.357756694745138, 3.2809267033768705, -0.91962590152930712, 3.1538768270618256]], 
[[-0.11351324899917881, -1.3442639650127775, 0.092637515235574064, -4.3749800364475959], [-3.8473969806959851, 
-0.42513842707085203, 1.1745440845096811, 2.070713020397803], [-2.942004528564496, -3.3462529748989156, 3.2472383989169735, 
2.5624902140144199]]], [[[0.40280371781229096, 1.3741844275134207, -4.9635371406315345, -0.96791551930849895], 
[-3.2650569560908593, 2.0475844458956649, 1.2257259617473837, -4.0193195002456932], [2.5952681010277248, 2.0516570232773592, 
2.0902200902515444, -2.9929872722545503]], [[3.6490687448725456, -2.1618987023526239, -1.4355126976769581, -4.190657987131682], 
[3.688120489790407, 0.88672164245780394, -0.46284442872251663, -0.52825517453619764], [2.1678369837513074, 0.63038927136342604, 
-1.7242781085847203, 3.538938713288637]]]])
      arg1=Data(-0.892512468076,self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[4.4519283578456212, 0.88183036741678356, -4.42368480394319, 1.5115281323869376], 
[-2.3560776766066418, -1.2740924903507564, 3.5748175601647807, -3.0854662851609569], [-0.59430289159227656, 3.7035561435252333, 
-2.3878147158472385, 3.6536965808055633]], [[-3.2223446761983245, 2.5890477386133171, 2.1038708562651984, -3.2767631537360198], 
[1.5109897913822468, 0.15617351678617644, 1.7276531015362371, -0.36272097892435445], [-4.3445461499360665, 1.227443021132631, 
3.4284616392179066, -0.86147545606602483]]], [[[-2.6963784348087971, 1.3657120134536571, -2.1678517349361748, 
3.1462717558570015], [-3.9609993224751201, -3.4984934865195823, 3.1569310902078205, -0.26105768957934711], [4.2672391661369113, 
0.16117041373244501, -3.8289006299947679, 4.0283304220569001]], [[4.3120487688840692, -4.0666881023803763, 2.605848057263668, 
-0.26944492127601444], [0.6213834192443044, -0.9940312354287284, 4.1555713801552132, 0.30221848935279455], [1.5030731389818095, 
1.4153864154085567, -2.0046837780804063, -1.4432831946211091]]], [[[-3.320314796296818, -3.8018691370793869, 
-1.7446843997150321, -4.4010158789190017], [-4.4196238682798388, -0.48134079977436012, 4.3108902245528853, 
-3.8142061930283373], [-2.9336492036526578, -2.7486589573337685, -2.4627658257982978, -1.6280761084352231]], 
[[0.60198853269647024, -0.51381127500458079, 4.0812420318952469, -0.4066414572993291], [0.98565653335174985, 
-3.3567716866556236, -0.24012021253910207, 3.3557791211262868], [-0.90325460309712169, 0.60817164583931582, 
-1.5132239547589141, 3.5395578353532446]]]])+(1.-msk_ref)*numpy.array([[[[-3.3080625257519918, 1.3379419108678532, 
-4.2300045338429157, -4.4222365370749754], [-1.0350435932170685, -2.9899285778209035, 1.1211368233703509, 
-0.69077318991750936], [1.3765956537066579, -0.064524655098278164, -3.3916284381668107, 1.6181755555906012]], 
[[-1.2823579353251144, 3.6988377335080709, 0.44405709913478808, -3.2849490138179362], [3.7566540470236385, -2.1257004666835289, 
-1.1681608538545556, -2.560152790951765], [0.21475619198159779, -1.4942720906696305, -3.3622809130228299, 
-0.92253848750275669]]], [[[-3.294897157334173, 3.3281841995089518, -1.3304156096771387, 0.36294840180789106], 
[1.6493043197780193, 1.392616896310426, -3.5209800049069222, 3.9649754034412994], [-3.8893521829011424, -2.9282679896069288, 
0.82077758308042181, -2.8148743909282539]], [[0.10131199002356821, 1.1997723491590131, -0.082680137359318506, 
3.9047242301125173], [3.4338497749086363, 0.37944134681890074, -1.0482952397296523, -1.848137188512089], [2.625775722879494, 
2.986572501433264, -2.8982007578481346, -2.2870544653302822]]], [[[-0.35950734033478537, -1.2264767349914338, 
4.4300187837713132, 0.8638766690269688], [2.9141040422889506, -1.827494647400109, -1.0939757033038306, 3.5872927671497687], 
[-2.3163091381668393, -1.8311294734904751, -1.8655474915721801, 2.671278457279584]], [[-3.2568393516647243, 1.9295215465667679, 
1.2812129807579167, 3.7402145029567682], [-3.2916935209040341, -0.79141012160630653, 0.41309442341430086, 0.47147432959915286], 
[-1.9348215367540358, -0.56263028443312268, 1.5389397103421472, -3.1585469253664962]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[3.3093728066104973, -0.70779914682090972, 0.26473501363256702, -0.15845457388648043], 
[-0.76645897590977441, 2.484103247814522, 2.2648547327672377, -2.3879651006088363], [-4.7490160986187142, 2.6094807960208186, 
-4.2001999671317591, -1.5365380754094495]], [[-4.5739338964008862, -2.1084317219138993, -3.6907777532115196, 
-3.7131005947182061], [4.0117974035203972, -0.96047601855128395, 3.2641957842165912, -3.9848723880878678], [4.2561921987773612, 
4.568034420073289, 4.8766125141070891, -0.70902896790789249]]], [[[3.5359588098507686, 4.8597882144676809, -1.0926044856258743, 
-4.4636860491204242], [-2.144447004978721, -4.0746572969049089, 1.3192588573847344, 3.2205658802732167], [1.9876946083187219, 
-2.9977434030805385, -4.7903350751611962, 3.0500007068426331]], [[4.9282036998195569, -1.6622192686421613, -4.5536351323970656, 
1.4023594365254386], [-4.8297392359963274, 2.4394694611597068, -4.7457878402170195, 2.9466338291326171], [3.9873920395379283, 
-1.3724275113816633, -0.38567142427339807, -3.0268716639617734]]], [[[3.2837087677437857, -2.7353970755395975, 
-1.0289998455580465, 4.8601652968281996], [2.5468857641003115, 0.66180611099570363, -1.5102063284179437, -4.7055156542221361], 
[1.0323400485428937, 3.5658127055887796, 2.0809446174783783, 1.1567312650957531]], [[2.2325517508871107, 0.30951665300276865, 
4.8286673103506832, -2.3653632873316388], [-3.6152783116468443, 4.5202437280816561, 4.6094950875798162, 0.1485899897833276], 
[4.0044594970067617, 2.2680368990732083, -4.5101971409420338, 
-3.8654257275911554]]]])+(1.-msk_arg0)*numpy.array([[[[-3.8277439863098826, 3.1964500021974178, -2.1801895614523605, 
-4.8260940657995386], [-2.018744490763499, 1.1552491577493758, 0.851919123230922, 3.0684506263592333], [-0.88805430726396573, 
4.8472774153362792, -4.49017361506868, 0.10247405702471291]], [[-2.9385033909864555, 4.4413330048773467, -3.6984541337992494, 
2.2625082012535058], [-3.8516327687432907, -0.63868043967361032, -1.3077398220490979, -0.6595484940349019], 
[-2.3757171271308284, 2.5096141555713256, 4.7399309706375377, -2.1424735372245984]]], [[[4.1294076635829313, 
-2.3111941344493947, -4.8218247267130501, -2.8243359287476277], [-2.2395367871438756, 3.1236789358602586, 4.3566643004942183, 
2.4580400278250725], [4.2935827646603038, -4.0390309524045351, -2.2808655274502829, -0.91160872716972996]], 
[[-3.3943682014686161, 2.5342654235199005, 2.4802864529512725, -3.9116156332552423], [2.6984773791564001, -3.3605013722212149, 
-3.7902707989065676, 1.3629487708895685], [2.7890369474475722, 3.8155343282912941, 0.40421084557801024, -2.4099465349895857]]], 
[[[-0.62110709856626833, -0.39580738271959781, -1.0500031539717058, -1.526396779377488], [-4.3278279632123606, 
-3.9825657301466197, -4.400144881062344, -3.3921813861533536], [-1.9681682272508016, -0.46976504144274767, 1.7617210779778114, 
1.9975764530877136]], [[0.33746267125346474, 1.9966074859931506, -1.0222513882531006, -3.3252164216002464], 
[-1.2872698924229731, -1.2423407784257146, 3.0576803632578748, 3.1878606723777523], [-2.9838451226205995, 4.3727872973924917, 
-1.762993890803275, -0.57159923384814437]]]])
      arg1=Data(numpy.array([[[[4.3846439555186034, -2.1695401006425841, -3.9303331763571272, -3.1916840571575325], 
[-1.5291514171772991, 0.39362065126497292, -4.0001888839372377, -3.2431739593477618], [-3.2037090062767826, 
-0.036556510082657923, 2.8238683885109817, -1.4240805552558422]], [[3.5581261527685069, -2.9187532949155228, 
4.2358802931248611, 3.9477789520456668], [-1.2836807201281433, -3.4535474114115186, 3.7368830481177344, 4.7022401400678806], 
[4.9514931618285267, 1.7814543100779705, -1.2919982217005135, 4.2106677756376669]]], [[[-3.1556520639520471, 
3.9189849397538836, -4.2824111241763116, -4.1655481710780426], [-2.3150051842505337, -1.1005838429053929, 0.26579081570373653, 
4.0374965709341364], [-0.54286282083228077, -4.9773645141259424, 3.6564273957029325, -2.0715541652719671]], 
[[0.90148422311360399, -3.9484962525032152, -4.350518094742104, -4.7470423191690632], [-2.5533346640726151, 4.4404656705653487, 
2.312538405398648, 4.6629822423480878], [-3.1512088216721668, -0.06586075720023743, -3.2739219286159882, 1.2240540269717028]]], 
[[[-2.6332045345823509, 3.2956962454070844, -0.82658753856363454, 1.0133324226263474], [-1.8572725360218225, 
-1.2585568355579326, -0.95240713908141128, -3.4071890321508325], [3.0473289954090141, -0.54824765275651188, 
-0.067552720247705977, 3.2342067159107852]], [[0.8654577588899599, 4.1447253426868986, 2.3662624045888139, 
-4.1957877474794163], [-1.1564630526295638, -0.28273974341574881, -1.3386838211006005, 1.1894479163767855], 
[-0.59040040770174684, 0.12980168013298954, 2.171101311479644, -2.0383592116853322]]]]),self.functionspace)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[14.510421473062353, 1.5355986322285715, -1.0404968070234344, 0.50573693725716984], 
[1.1720318292206928, 0.9777943382141866, -9.0598467257481481, 7.7445862301258357], [15.214465646098203, -0.095393511030237271, 
-11.86081191260824, 2.1881539956008322]], [[-16.274633817818351, 6.153992035440603, -15.633692751132328, -14.658500374656782], 
[-5.1498669799592784, 3.3170494675906284, 12.197917891776354, -18.737826896314925], [21.074506567674025, 8.1377446062240839, 
-6.3005746961488294, -2.9854854271633964]]], [[[-11.158255716355002, 19.04543682289226, 4.6789816035691816, 
18.593699258180155], [4.9644059338762689, 4.4845019863501054, 0.35064688782866799, 13.00302369807059], [-1.0790455020250169, 
14.920861636948214, -17.515512403416064, -6.3182416683423002]], [[4.4426978836774218, 6.563266553072209, 19.810672040346791, 
-6.657059591872339], [12.331940609701011, 10.832380396672226, -10.97481664437576, 13.740101219947542], [-12.565104970457293, 
0.090389115102033818, 1.2626581331692384, -3.7050544493989479]]], [[[-8.6466768174707607, -9.0150378715533712, 
0.85055844952218573, 4.92496307459942], [-4.7302609820484625, -0.83292060480765473, 1.4383312886711759, 16.032581327679711], 
[3.1458797630467092, -1.9549484460083955, -0.14057346959548642, 3.7411080260766636]], [[1.9321792349286149, 1.2828615156842023, 
11.425893920749807, 9.9245622993237248], [4.1809357923925656, -1.2780525518544552, -6.1706564971857949, 0.17674005374222687], 
[-2.3642345196579244, 0.29439500010331804, -9.7920949277309912, 
7.8791261389209089]]]])+(1.-msk_ref)*numpy.array([[[[-16.783294532846309, -6.9348264594663735, 8.5688713641237086, 
15.403367488154963], [3.0869659989698697, 0.4547299258466207, -3.407837406761892, -9.9515191669525933], [2.845067582244456, 
-0.17719954570718072, -12.679659330518522, -0.14593131202707199]], [[-10.455565765467849, -12.963155341802816, 
-15.666208980386418, 8.931882255739291], [4.9442667262495421, 2.2057131791539675, -4.8868707723637765, -3.1013554029722368], 
[-11.76334710942721, 4.4707629540752247, -6.1239823850468875, -9.0212442833480644]]], [[[-13.030973816484877, 
-9.0575350057546906, 20.649035848504369, 11.764907362504685], [5.1845392725578563, -3.4378705672317116, 1.1579613581757069, 
9.9243281835625794], [-2.3308264511003554, 20.103729333954643, -8.3398192004836336, 1.8884468558667304]], 
[[-3.0599693810624564, -10.0065375276168, -10.790531093708221, 18.568604947385928], [-6.8901158324158578, -14.922190979236051, 
-8.7651467893324533, 6.3554059158882099], [-8.7888378327664007, -0.25129397998476394, -1.3233547511222588, 
-2.949904760940504]]], [[[1.6355020284059849, -1.3044609051333833, 0.86791952252552529, -1.5467473463356443], 
[8.03795601670158, 5.0122853227347974, 4.1907293977163036, 11.557803213967913], [-5.9976561067441256, 0.25754758131805194, 
-0.1190090511351221, 6.4605751801215288]], [[0.29205968717204289, 8.2753896465941885, -2.4189150280620351, 13.951902319467663], 
[1.4886800693496016, 0.35125911292700818, -4.0932672323903239, 3.7917942344592159], [1.7616633769140708, 0.56759513806574002, 
-3.8276383484535907, 1.1651245637066434]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[3.3975009262662574, -1.7019314995757107, -0.35505034937690638, 0.35760319001719143], 
[-2.6296357167738469, -4.3967654316996638, -0.78530488412095423, -1.2075211971052902], [-4.221011097300682, 2.5640229633943843, 
0.41690862500460213, 1.2700688636135684]], [[1.7637836476149999, 2.5293169897794385, 1.8934420450116995, -4.6095154597182759], 
[4.9940940070994433, 4.5692714681630857, 0.32458322250436389, -2.2258580142234461], [2.5396541753475512, 4.8654497873102169, 
-3.5396036291937039, 1.6133507710838302]]], [[[4.8026832532645294, 4.339420294544901, 2.5193514377698323, -4.0044072368556431], 
[-1.2544745463845466, -1.2252763835268277, -2.2127207250281575, -3.6409285717537729], [4.1371195929485953, 0.45698783711993318, 
-3.8781137137148161, 2.5680174926365265]], [[3.7461313553239002, -1.1182344813603038, 1.0391304990400316, 2.1958641810348585], 
[4.8028950194133575, 4.8546405082987558, -0.92819799086101007, 0.93658953583587312], [-3.8888486686936963, 3.9263563448743781, 
2.9164362788399565, 3.1530959935339595]]], [[[4.8494173463986741, 2.7499653045595718, -4.3329319131935096, 
-3.4628096707067302], [-1.59007224784951, -2.3682952526145584, -1.7945456886241598, 4.6535897287305392], [2.1558397805851746, 
2.4537930866433619, -1.7367562850643816, 1.3431108459799859]], [[1.7445859225312237, -4.7531533181579224, -4.6800804635552185, 
-3.3183569325851558], [-1.2892595652882388, -3.8367892573085536, 2.9642856957407542, 3.4573319247520864], [-1.1997578681634979, 
0.82967627552134537, 3.0009995739998114, 3.0720682763631757]]]])+(1.-msk_arg0)*numpy.array([[[[2.7678734781880721, 
1.4564746979235688, 4.4954345765429231, 1.9518517038922125], [3.2196699355342258, -0.71797897510565623, 4.4065651256913991, 
3.8961340755681277], [-3.2637577682047558, -0.11580061094080296, 3.6607874898236581, -4.6725621174644889]], 
[[2.521571617740971, 2.6069326791715675, -0.65269332601842667, -4.9154648276708324], [-0.59408471234724836, -1.887250997286519, 
0.80667914955906284, 3.3156479313913003], [-0.084265791716406468, -3.6568754773685375, 0.28507777109917853, 
-4.6501008439026883]]], [[[3.4275712276660126, 2.9983039557652456, -3.8017960124273942, -0.40582796747362782], 
[2.3738499028828137, 3.0349846441217316, 1.905499831869168, 2.2307533608913532], [4.2034146828743584, -0.98098245099426506, 
-0.94327580302107972, 2.1074951325043756]], [[-1.9507530020895167, -2.3935285980908025, 2.1087765914154302, 2.687168120793622], 
[-1.0202396613782594, 3.5075897696376899, -3.0234488795962466, 3.2058961226605938], [3.2849921813668743, 0.74348287997797247, 
4.1710470438078353, -4.9318675773233158]]], [[[3.0000608419240198, 1.0729779313982792, 2.716309465003885, 2.8083686137952055], 
[2.0027450690721249, -1.2518016613738316, -2.1579884931924886, -3.1301321212199298], [0.49428177582113175, 3.1723239195218138, 
-0.98637290109191955, -0.89797222838082025]], [[-1.5073069129645789, -3.6302676182514135, 3.070580929042551, 
-1.1586156675522377], [2.4987621680402512, -3.8054121881022276, -3.4201139108732415, -3.883750920530149], [-3.6644230141184311, 
-4.9678486655611689, 4.6578446229622266, -0.83873420812397903]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(2.42008330076)+(1.-msk_arg1)*(-2.26116506549)
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[8.2222352559859981, -4.1188160011668122, -0.85925142145735023, 0.86542950846041489], 
[-6.3639374852560344, -10.640538598631231, -1.9005032360692535, -2.9223018844326432], [-10.215198468915425, 6.2051491564852732, 
1.0089536013179718, 3.0736724476510608]], [[4.2685033517530568, 6.121157809302999, 4.5822874740965869, -11.155411388676072], 
[12.086123509025176, 11.058017576757283, 0.78551843649086273, -5.3867618100930974], [6.1461746594732842, 11.774793780973502, 
-8.5661356343340884, 3.9044432593741334]]], [[[11.622893540082721, 10.501758589822987, 6.0970403433016598, 
-9.6909990833714463], [-3.0359329009382945, -2.9652709145933516, -5.3549684758942773, -8.811350435774548], [10.012174040156999, 
1.1059486332660484, -9.3853582370237234, 6.2148162499985933]], [[9.065949935486465, -2.7062205946781708, 2.5147823680409771, 
5.3141742352675063], [11.623406031803167, 11.748634425344569, -2.2463164574851002, 2.2666246953463736], [-9.4113377223025605, 
9.5021094230778793, 7.0580187361618547, 7.6307549596563007]]], [[[11.735993938452998, 6.6551451112440407, -10.486056166465501, 
-8.3802878578002264], [-3.8481072940283165, -5.7314717921303195, -4.3429500534967316, 11.262074791106016], [5.2173118521161497, 
5.9383836725148891, -4.2030948829806212, 3.2504401294307006]], [[4.2220432578651588, -11.503026971243314, -11.32618457608018, 
-8.0307001985226201], [-3.1201155443038679, -9.2853496101617932, 7.1738183109547533, 8.3670312562895681], [-2.9035139817022766, 
2.007885699428988, 7.2626689546357674, 7.4346611344322868]]]])+(1.-msk_ref)*numpy.array([[[[-6.2586188145633797, 
-3.2933297057086723, -10.164919618655547, -4.4134588858498915], [-7.2802051806247219, 1.623468976262167, -9.9639711210011779, 
-8.8098022621232364], [7.3798950476723055, 0.26184429602124965, -8.2776447841564966, 10.565434226322788]], 
[[-5.7016896521562508, -5.8947051022158981, 1.4758473472685623, 11.114677348953178], [1.3433235974987439, 4.2673860248674034, 
-1.8240347120387008, -7.4972272719120738], [0.19053886444463611, 8.2687990782572331, -0.64460789695600329, 
10.514645579218525]]], [[[-7.7503043194624643, -6.77966016048408, 8.5964883294038206, 0.91764402264845113], 
[-5.3676664711053395, -6.8626012515737358, -4.3086496521115176, -5.0441015693624349], [-9.504614436665328, 2.2181632480428108, 
2.1329022929092756, -4.7653943693001386]], [[4.4109745397162499, 5.4121632492439851, -4.7682919594226707, -6.0761306798255079], 
[2.3069302807315228, -7.9312394511599251, 6.8365169838250397, -7.249060316136327], [-7.4279095609005781, -1.68113751499292, 
-9.4314258619558533, 11.151766673445282]]], [[[-6.7836327700903363, -2.4261802145149458, -6.1420240693150499, 
-6.3501850005203382], [-4.5285371852597178, 2.8305301856155203, 4.8795681925270706, 7.0777454028572864], [-1.1176526839929981, 
-7.1731480232275269, 2.2303519454908716, 2.0304634325910951]], [[3.4082697345606632, 8.2086343167542246, -6.94309032749777, 
2.6198212717935618], [-5.6501137213300288, 8.604665099510413, 7.7334420952483898, 8.7818019045508642], [8.2858653046865616, 
11.233125853186875, -10.532155541902794, 1.8965164906377889]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[2.2430741830152758, 3.6425182150943023, 3.2034851894755061, -4.9343929342602655], 
[2.6257785883314488, -1.9821426682791876, -1.9488026675285344, 0.36411003530200414], [-0.80079489794012915, 
-4.0055922675288063, -4.3094865313940485, -2.9841104792674056]], [[0.98980255987863863, -1.4332489295195749, 
4.1566158753975628, 2.8605571625499753], [-2.8573644462606209, -1.0154589754203149, 3.7648744783016888, 2.0214816361612966], 
[-2.3549798380600082, 2.5030410943637946, 1.4326328499225873, 3.4977861431556541]]], [[[3.8939492759918615, 
-4.1860738888668294, -4.9794664988868167, -1.7330079886988159], [-4.941158491847979, -0.018205858606985537, 
-2.2183297591600057, -1.6692110422407969], [-1.4852330601791075, -2.1121349187440885, -1.8352232108277367, 
4.9907258596779229]], [[4.4716537470444262, 4.2988840686620868, -3.3056013118553262, 2.9295647753930263], [-2.5557164962361245, 
-1.7391550409149135, -3.0012436648948748, -1.6087133614594684], [4.110815283820207, 4.0293204205091921, 2.7032060397216204, 
-2.2077351503644884]]], [[[-3.6198914158038908, -4.3392944360793155, 3.3735950706561262, 2.4052789556927454], 
[3.4992560985539978, -2.8441838267243238, 4.15549152275522, 4.8536969175416473], [-0.94720371932400838, 0.45425612269362237, 
-3.370001469271473, -2.5318355590146666]], [[1.9019761836068394, 4.0176408420270526, 1.3563058342429022, -3.607378085052658], 
[-3.3034600605664033, -2.4586389570990042, -1.2824094413763154, -1.4115316587359694], [-2.5905643686608393, 
-1.3106688734160112, 4.6533856982771997, 1.5974461727658316]]]])+(1.-msk_arg0)*numpy.array([[[[-4.8148307802211221, 
1.3895654856496931, 2.6008842515589361, 0.5077901160315843], [1.5398463966791942, 2.7820256791618192, 0.19408045677987484, 
1.7302993049359872], [-4.7212130926921532, -2.1862983769221547, 0.88346559966671734, -4.5043824885039641]], 
[[-0.26383905470717295, -3.6613447596466928, -3.7730082687343813, 4.7853177202538824], [1.9773663106508401, 
0.71644043486449149, 0.42617165002469282, 2.0487779783267808], [4.0825978773047051, -2.3466674567907275, -1.9231552917114758, 
-4.717898336558294]]], [[[-2.3203955052101453, 3.0382418110176186, 2.1992787172560302, 3.736193673140825], 
[-2.4398431145841801, 3.6403892105901789, 1.4626956643410391, 3.4609770186994062], [-1.7087774100108719, 4.6064557212311499, 
4.7867022542388042, 1.2117367821422462]], [[0.92608931933675898, 1.1684530221053198, 4.9999125478918334, -3.8378676955853575], 
[3.2750239269522368, -0.85922917677859978, -3.7792999837346009, -0.63595782088134989], [4.8448983010150464, 
0.44796645117554768, -3.8305049220691769, -3.1532331874586039]]], [[[-3.0251514504617716, 1.7402924665602875, 
-0.097327793071590563, 3.5510129064358438], [1.0538693828934607, -0.031758094644954937, -0.28445867971461869, 
3.9229251049041043], [4.1633767303728408, 2.9929091934483401, 3.1417577824136895, 2.2190934879946722]], [[1.2487600906668961, 
-2.1676888329240018, -4.7067571246936009, 4.0078437781545677], [2.8209567831012672, 1.6237787701824411, 3.5813332867414314, 
0.033645469738532441], [-1.9303411352426636, 4.8761601379051314, 0.64035586477481132, -0.3381815614225161]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.90075190123457016, 3.2474188990539172, -4.8347588957145291, 3.272926153183338], 
[1.0295640843061635, -2.8988753557672831, 0.69935515100097412, -3.8486248421551252], [-4.8205483491396599, 3.3582003185155571, 
1.2032105142136862, 0.98739696962804935]], [[-3.8358457994265338, 3.3097155678271832, 2.9250034815626611, -1.5877255946668178], 
[-0.23132668028251313, -2.3894386220760166, 1.1671536429923606, 2.9420925703808534], [4.5076665071615309, -2.2546188605815987, 
0.10528899857024143, -2.670667976063188]]], [[[3.4093213130349831, 1.8800070572666563, -2.7868061317536732, 
0.21616656854721228], [-1.2038156639730566, 1.9191499850474045, -0.87899315446991011, -3.793461955431856], 
[0.41484799416608897, -1.2706760930418062, 4.765694159975272, 2.6734441778759983]], [[-1.972600013919088, -1.8496321409825054, 
-0.30103266257816674, -3.5449643116271967], [-3.6432279250250454, 0.47000692572677139, -0.61225986158215129, 
-1.7702355108803935], [1.6169800769490035, -1.9365921680962153, -2.5837536375944281, 3.6670966927768269]]], 
[[[-2.725472367155064, 0.86514976442369473, -4.2083552975910985, 4.1939205121991527], [3.6442308227152047, -4.973037166449342, 
-3.0570655790908963, -1.4415905277459662], [-1.7608612220953015, 4.9357988966158519, -1.169150048647154, 2.1009134444450499]], 
[[-1.49848608885496, -1.5427198158064472, 1.1604480585837997, 0.34138561457573235], [3.2634720486049797, 0.98152911945546073, 
0.36105149193856967, -1.1934713235181169], [-3.360962541129251, -4.123944684679218, 1.7128671063735279, 
-3.6556407993443916]]]])+(1.-msk_arg1)*numpy.array([[[[2.8228168550607604, 2.5225647487557463, -3.5364737412383382, 
-4.5279694225406661], [-3.5029085146758741, -0.24072383403763098, -3.7759255672894443, 4.8311562562439203], 
[1.1725723149184288, -3.8900422239242474, 2.4828583364849326, 2.1945847084704804]], [[-0.25903516031541418, 
-1.6761661073794585, 2.7078007186132194, 0.14506502455555736], [-4.2183978206093373, -1.3862219275335708, 2.990070966030304, 
-2.2570854582254043], [-0.82404730368402035, 2.3515997296939428, -3.1190891961065379, 4.9234245571982065]]], 
[[[-4.6381583508376902, 4.9548694808587008, 1.4076718501418686, -3.0436368988251905], [-3.4090212455261382, 4.8105218389290751, 
-4.3939334730500086, 4.3614225906676189], [4.7848859150493848, 4.3152278153133885, 0.18633078188676855, 4.2945672359283709]], 
[[-2.5484652549035758, -1.8493433116843514, 3.0862116502340289, -1.8004201641918902], [4.647927450493297, 1.2030201474921451, 
1.4048887005469766, 4.2591471159267478], [2.2942053659751496, -4.0278100361202984, -0.027641053950217653, 
4.3066535868393423]]], [[[-3.4651648188339812, -1.9501992048234418, 0.35461131691443448, 3.3548551037873988], 
[3.1775242451213792, 2.4136227306261535, -3.0727457320363252, 4.9785212339467435], [-0.9274748011111722, -1.1923564874030723, 
3.2330482294685332, 2.7616330562391767]], [[1.9299804363822926, -1.6861272285310527, -3.0051459888944523, 3.379388795158226], 
[1.3021976801564374, -4.1697193202961733, -4.387820504554484, 4.426172217731084], [2.5917133928913882, -1.9506042197569453, 
-1.5415736015536421, -1.7717925310678337]]]])
      res=arg0*arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-2.0204533349611897, 11.828782491845379, -15.488078517106446, -16.149903684623496], 
[2.703407327886199, 5.7459845326893415, -1.3629051838205193, -1.4013229271412728], [3.8602705232647518, -13.451581228658689, 
-5.1852195054355876, -2.9465016442639422]], [[-3.7967299915721062, -4.7436462946025824, 12.1581159070565, -4.5417798219880847], 
[0.66098463171075084, 2.4263768950030409, 4.3941869627587788, 5.9473861029114818], [-10.615463741063785, -5.6434036601634165, 
0.15084047808718021, -9.3414254396433751]]], [[[13.275724258516195, -7.8698484533093156, 13.876807771959776, 
-0.37461839018192905], [5.9482439906600817, -0.034939773273371451, 1.9498966726585294, 6.3320885843272201], 
[-0.61614595588446486, 2.683839346526911, -8.7461125380928113, 13.342426992931129]], [[-8.8207842436611763, 
-7.9513541437550392, 0.99509396432968977, -10.385202577368423], [9.3110577075346157, -0.81741491414263578, 1.8375410308428446, 
2.8478015192833173], [6.647106413954738, -7.8031503691082502, -6.9844184382979648, -8.095978268428766]]], 
[[[9.8659140258753268, -3.7541395591390692, -14.197286687522924, 10.087548749840762], [12.752096930924633, 14.144231878514178, 
-12.703610098418997, -6.9970435008778331], [1.6678942987820884, 2.2421168691721762, 3.9400373817397232, -5.3191673650579618]], 
[[-2.8500848524682962, -6.1980941397884344, 1.5739224721930567, -1.2315069845727302], [-10.780749571341371, 
-2.4132257306202778, -0.46301584208502639, 1.6846225569393403], [8.7067898034532281, 5.4051259338984581, 7.9706312958480261, 
-5.8396894039193237]]]])+(1.-msk_ref)*numpy.array([[[[-13.591385480673535, 3.5052689101875747, -9.197958859638506, 
-2.2992581184593908], [-5.3939410542205133, -0.66969988787897738, -0.73283335886634338, 8.359346312216001], 
[-5.5359637653212328, 8.5047930003242307, 2.193519929130169, -9.8852489303730096]], [[0.068343591833539871, 6.1370219935511763, 
-10.216554501412576, 0.69418223259477319], [-8.3413177353958297, -0.99314544058084508, 1.2742834772840619, 
-4.6242669820138191], [-3.3642537728190471, -5.5184225570706467, 5.9984928928123811, -23.228216528575675]]], 
[[[10.762361789736676, 15.054091624880067, 3.0958627408974313, -11.371616924728638], [8.3174770133681335, 17.512171799745833, 
-6.4269874404332121, 15.094783355137057], [-8.1763049611155889, 19.877905858266153, 0.89190997369147396, 5.2038850831573651]], 
[[-2.3601064532670324, -2.1608707814478407, 15.430788355455084, 6.9097743866325407], [15.222073611103657, -1.0336700109777455, 
-5.309495843126113, -2.7086379186578604], [11.115191679792606, -1.8043237678900645, 0.10587919320748838, 
-13.579883016909449]]], [[[10.48264837778472, -3.3939169844460988, -0.034513536873492304, 11.913133772771216], 
[3.3486955153350775, -0.076652059116439963, 0.87406919403378258, 19.530365933947838], [-3.8614270049534332, 
-3.5686146930164249, 10.157454435851564, 6.1283219313311816]], [[2.4100825447220875, 3.6549991641758592, 14.144492293973361, 
13.544062356640158], [3.6734433787760365, -6.7707017099164846, -15.714247629207556, 0.14892064340920422], [-5.0028909730575775, 
-9.5114585412083574, -0.98715569673690295, 0.59918756467327183]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_constData_rank0(self):
      arg0=-1.58150799807
      arg1=Data(0.74133131456,self.functionspace)
      res=arg0/arg1
      ref=Data(-2.13333494351,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_constData_rank1(self):
      arg0=-4.68279391573
      arg1=Data(numpy.array([-1.7687202452841944, 1.4285453375269981]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([2.6475605332262906, -3.2780156098085738]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_constData_rank2(self):
      arg0=-4.77843722279
      arg1=Data(numpy.array([[-4.2420214566013108, -0.64663347400184445, 3.8393818897588492, -3.0159062836945472, 
3.9979206815937118], [-0.20402775090354908, 1.4825250658701208, 1.1096906890682874, -1.5931273330501092, 3.3809991987766725], 
[-3.0148072293977579, 4.1865340578382266, 3.9358322332993545, -4.5005469551233324, 1.5924335813443893], [1.6104615377729505, 
-0.037519182884611091, 1.470163415745942, 1.2091574349265422, -3.7089627015863114]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[1.1264528649070116, 7.389715217204996, -1.2445850295672707, 1.5844117068955728, 
-1.1952306219543876], [23.420525892306152, -3.2231746584203687, -4.3060983297945912, 2.9994069674500166, -1.4133210160222798], 
[1.5849893075054449, -1.1413826226587469, -1.2140855960163379, 1.0617458878738915, -3.0007136741938294], [-2.9671228469035591, 
127.35984249660964, -3.250276242495834, -3.9518734986530988, 1.2883486859390199]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_constData_rank3(self):
      arg0=-4.2129028658
      arg1=Data(numpy.array([[[-0.34983504304066049, 2.9375995895072595], [2.4204868715516161, 2.8585940351285579]], 
[[-3.610692322429724, -4.2697890316143061], [2.5671768371889243, 3.2747632991551399]], [[3.4337501262514269, 
0.033415884257991024], [-3.7341435992880267, 3.8307520273690958]], [[-2.1863103352604463, -3.3162842591493091], 
[-2.1987165743034707, -1.4609787925678863]], [[-2.4938212793616952, 2.9677530095095879], [-4.1096156115923979, 
4.3328888067101872]], [[-3.3131868659980777, -3.7918439831168937], [-4.1244152345380396, 
-1.6659751597934158]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[12.042541047858395, -1.4341310779196821], [-1.7405187837667737, -1.4737674584166678]], 
[[1.1667853390957155, 0.98667705467491518], [-1.6410645362512768, -1.286475534547292]], [[-1.2269101451466489, 
-126.07485809054999], [1.1282112628447707, -1.0997586989961921]], [[1.9269464164585746, 1.2703684414788836], 
[1.9160736381545962, 2.8836167145120992]], [[1.6893363211957086, -1.419559799045619], [1.0251330693587768, 
-0.97230809599177637]], [[1.2715560685791112, 1.1110433036154241], [1.021454588402599, 
2.5287909252613767]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_constData_rank4(self):
      arg0=-2.81537197661
      arg1=Data(numpy.array([[[[2.4656599858001593, -2.6183572965227442, -3.4600389481847538, -4.1033804259339153], 
[-4.0480770479440364, 0.87660914485012764, 1.4492322453986315, -4.5136546351664659], [4.3702321489493166, -2.8822384776108114, 
0.57679521785766319, -0.18799747713645232]], [[-2.6092598411939374, -3.2496622966944733, 1.6557680792463483, 
4.9458764253489811], [-4.7274359417244778, -1.0395248592281425, 3.9983528279221083, -3.1464876430053614], [2.8791780096831712, 
-2.4843808494241362, 3.1965197411147805, 4.1740923732684383]]], [[[-4.7711376037623578, -2.0383536544191347, 
4.3495804063749706, 1.9655905644825955], [-3.7337425699086637, -3.9582160802452271, 3.4377323248089233, 2.3259687219978806], 
[2.4716121356670753, 4.250010432638458, -1.3882735643087818, 1.5211776852089471]], [[-2.8158693410091828, 4.3570813741329459, 
3.8654197698083408, 2.0948916199578846], [0.49790776290527816, -3.0143313991693574, 1.4366832954788862, -4.6647383183831623], 
[2.8831260637664311, -3.2097974622488836, 2.9485636721176798, 1.0856326614842695]]], [[[1.7672953305412946, 4.3583411081866927, 
4.9676516467050913, -1.736091688787742], [-1.6980696348792526, 0.14419587037388837, -1.2686598463883927, 1.6624353796433375], 
[4.822200009798328, -1.0955211724610159, 4.1916007891119555, -1.5107997020025712]], [[3.134866680190342, 3.1066903159181969, 
1.7876995283081945, 0.85297450772985961], [1.2361713570719726, 0.062974898074552144, -0.6935962943236218, 2.7251275891835842], 
[2.66537594962805, -0.74287101588632609, 2.1662466700429741, 2.9867092735484775]]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[[-1.1418330154295588, 1.0752436194819921, 0.8136821633430551, 0.68611039786042505], 
[0.6954837922465158, -3.2116616546266337, -1.9426644594398961, 0.62374554638613078], [-0.64421565735062059, 
0.97680049672501768, -4.8810598448910056, 14.975583818961283]], [[1.078992567992749, 0.8663583226706264, -1.7003419814033769, 
-0.56923621507820965], [0.59553889493491596, 2.7083257813580723, -0.70413295118662933, 0.89476657658858694], 
[-0.97783880230455311, 1.133228819270115, -0.88076164223166764, -0.67448722377115944]]], [[[0.59008400310862807, 
1.381198974234179, -0.64727438363568224, -1.4323288010650848], [0.75403483874333355, 0.71127293698321181, -0.81896195241631664, 
-1.2104083558750298], [-1.1390832469150121, -0.66243883897065692, 2.0279662805593843, -1.8507844310268613]], 
[[0.99982337092430384, -0.64616006332230935, -0.72834831513002629, -1.3439224968911752], [-5.654404663591869, 
0.93399550473647841, -1.9596329862468109, 0.60354338967207721], [-0.9764997833401291, 0.8771182636046021, -0.95482827901361844, 
-2.5933007328289497]]], [[[-1.5930398999854174, -0.64597329734513309, -0.56674102309036889, 1.6216724005954344], 
[1.6579838180843487, -19.524636657833298, 2.2191700830013419, -1.6935226542244397], [-0.5838355876756639, 2.5698928029712582, 
-0.67166987465108097, 1.863497836859785]], [[-0.89808347972209657, -0.90622871619506851, -1.5748574813769469, 
-3.3006519551248283], [-2.2774932945208843, -44.706256979999424, 4.0590931636330057, -1.0331156558631631], 
[-1.0562757486436691, 3.7898530382843831, -1.299654381721149, -0.94263341984581106]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_expandedData_rank0(self):
      arg0=-1.09855931938
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.22111665439)+(1.-msk_arg1)*(1.33873726471)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.34104921903)+(1.-msk_ref)*(-0.820593665644)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_expandedData_rank1(self):
      arg0=4.05749907601
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-1.5198157487642607, -0.53049822656922885])+(1.-msk_arg1)*numpy.array([-0.4004567724153727, 
2.7845648875824285])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-2.6697309060727403, -7.6484686899938881])+(1.-msk_ref)*numpy.array([-10.132177442122019, 
1.4571393520424547])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_expandedData_rank2(self):
      arg0=3.75753686931
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-4.1776820931205609, 1.4185117832018772, -3.8311909374339894, 2.1235887191381497, 
-3.1198649568284829], [4.189110788624701, 1.2131461666764753, 3.737370596152175, 4.1336396509356916, -3.5434794559349694], 
[-4.3696168015662016, 1.4306924580120572, 2.1408826677900405, -4.0242957787223377, -1.3057397288185935], [1.9077970511384557, 
-4.8160719247544002, 4.9887386350091099, 0.35146741030783168, 
-4.5453847054766641]])+(1.-msk_arg1)*numpy.array([[-2.4190114342714719, 2.547143537651591, 1.4517444899618956, 
0.43179189729206513, 1.8454893180900598], [2.2347094373500385, 1.4216820881609653, 4.4010024533706513, -3.5356569423948603, 
-1.0123812081435322], [3.9515151776442163, -0.10159804523562155, -2.8243389583934517, -1.9751519542108054, 
0.88336630498884539], [1.8235012577276386, 3.9828819153925021, -3.8888339010456932, -2.5352576694289808, 0.63188990804257372]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.8994310207325551, 2.6489289083168548, -0.98077515077549826, 1.7694277782924583, 
-1.2043908698956285], [0.89697720086919819, 3.0973488376966634, 1.0053958451913039, 0.9090141344229139, -1.060408820211459], 
[-0.8599236592015872, 2.6263763733909871, 1.7551344246204392, -0.93371289684490177, -2.8777073917412976], [1.9695684439124577, 
-0.78020779756173653, 0.75320379443063001, 10.690996545086545, 
-0.82667081287622257]])+(1.-msk_ref)*numpy.array([[-1.5533357205660305, 1.4751963577113161, 2.5882907738194127, 
8.7021940265125828, 2.0360653581025909], [1.6814431471535303, 2.6430218827418388, 0.85379113261655004, -1.0627549365030857, 
-3.7115829877976281], [0.9509103977557497, -36.984342175060902, -1.3304128593151991, -1.9024039448210002, 4.2536565500525763], 
[2.0606165492827841, 0.94342161006328717, -0.96623742873182517, -1.4821124158783665, 5.946505588214551]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_expandedData_rank3(self):
      arg0=-3.06945082764
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-1.6137339566571915, -0.84262588150557427], [3.7621204549396037, 0.61628003127611208]], 
[[2.1849436350812335, 2.3797970726361495], [-3.8953762877132059, 2.9487919480824125]], [[-1.2128249881825868, 
-0.14667802293957788], [-1.0334758679170397, -1.4921762462422117]], [[-0.10374606218564875, -2.0362432878901551], 
[3.0232848823222778, -1.4527811885508224]], [[-2.435206080806589, 1.536841861163694], [-2.969599295127856, 
1.3287016008240737]], [[4.8502531633609287, 0.99959998670021033], [-0.34743328959997744, 
2.4262475015316447]]])+(1.-msk_arg1)*numpy.array([[[-1.6309831506598149, 3.6763289230752374], [4.1179501669380301, 
0.29058545171167971]], [[-4.9013625375547765, 0.91747922472024968], [0.12173580949657481, 4.2698882632001212]], 
[[1.553643442491194, -2.0964233784416364], [3.3522448009557397, -1.4030175826203295]], [[-1.0473609824727212, 
-2.3121371702033211], [-4.7438803022627152, 1.8368350225282972]], [[2.5756703248990469, 4.8151142135448008], 
[-4.7340716234034854, 3.632756703228976]], [[-4.3928481320680426, 4.0014047546319969], [-0.32360053292368463, 
-2.7852345214969763]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.902079840963592, 3.6427208029163221], [-0.81588318726103504, -4.9806105534204015]], 
[[-1.4048192266167729, -1.2897951942751009], [0.78797286858209803, -1.0409180714265414]], [[2.5308274957589871, 
20.92645350762929], [2.9700266091577991, 2.0570296808878585]], [[29.58619115724527, 1.5074086902535393], [-1.0152701273980902, 
2.1128101408704887]], [[1.2604480794575692, -1.9972457187702293], [1.0336245811588205, -2.3101129898032453]], 
[[-0.63284342574586716, -3.0706791401314981], [8.8346480303315715, 
-1.2651021075543085]]])+(1.-msk_ref)*numpy.array([[[1.8819635422930712, -0.83492279713330797], [-0.74538318901474221, 
-10.56298864776473]], [[0.62624439716866298, -3.3455262472803904], [-25.214033901195045, -0.71885975426804749]], 
[[-1.9756468850499824, 1.46413690059005], [-0.91564041706051569, 2.1877493665499435]], [[2.9306522574379015, 
1.3275383775635345], [0.64703378501603548, -1.6710541719806373]], [[-1.1917095126513777, -0.6374616865788495], 
[0.64837439561789445, -0.84493707627262171]], [[0.69873820704815459, -0.76709331243805945], [9.4853083210463502, 
1.1020439406252676]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_float_rank0_expandedData_rank4(self):
      arg0=0.594519685746
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-4.7938772326802503, -3.2760410004444251, -3.1048526721112912, 1.3176039278542202], 
[4.3260840959307441, -4.9313630616433386, -1.8160153943284039, -3.8609740831039052], [-0.046505631366281719, 
2.1155572000215042, 4.4495939012871446, -0.47171829831305612]], [[4.0804784497080693, 2.3509791338094486, 2.103236229296952, 
-2.264946851161457], [-0.74373810294692344, -0.30002212907597325, 4.3852436264576138, -2.0508647355642031], 
[3.7154273043919552, -4.5244967722374341, -1.078808241328848, 1.8333173164987553]]], [[[-1.302610777479126, 
-2.9485293423143002, -0.69889066197142569, 3.2131230942700082], [1.1948591973154148, 0.95447628500421366, 0.70356294607013758, 
-1.7141492802089986], [0.40713941490540861, 1.8348508464517295, -4.9343104569402128, -1.2255848473312336]], 
[[-4.7056592009385101, -3.1582090120942063, -0.26383447475615007, -4.1355554643713797], [-1.7855675686089914, 
-1.9901551784980951, 2.4635756425157336, -1.1586492112493394], [0.017394295162715068, -1.1806264993221149, 2.5742045973631136, 
-0.15257906118494269]]], [[[1.6214838899055675, -0.84878018431876256, -3.0283078207616878, 1.1271792167087069], 
[-1.1657680108064152, -0.92760806903406401, 2.8076628773777195, -4.0077697995085089], [1.7273991559019732, -0.7315107789497155, 
3.2662630323674069, 4.1004013199519225]], [[3.2419754496247197, 3.6112743258264715, 4.1671926918678164, 3.3752218280005692], 
[0.50370393742950448, -2.8472629926046187, 2.107195506305696, 4.1415643060431133], [-2.9290192818113105, 4.6544852435849027, 
3.4031890460138943, 3.8570841351526255]]]])+(1.-msk_arg1)*numpy.array([[[[-2.2040483139839608, 2.4697358567881968, 
2.7993234095871742, 2.2306288371926115], [-3.1869827895092162, 2.070519494055838, -0.57724871829210844, -2.3769643544658381], 
[4.4251129913774871, 1.1126853085734592, -2.8018679248703782, -2.2999611363908992]], [[3.7107052610725777, 4.1437974533112225, 
-0.77400941890756325, -0.67768650694315458], [-4.0249699283957714, 1.6172138768135653, -2.4553136709757983, 
-4.7069782564167308], [-2.3690757946017937, -2.0082833191429463, -0.91580176140453684, 4.4705312316373735]]], 
[[[3.4631308234045264, 0.88740392592166373, 4.8331247595447344, 3.9445823888025693], [-2.1427824293707043, 4.4803808270387719, 
-2.9232039615543792, 1.2416648951841047], [-0.048580790209538094, -2.3978954403102257, 3.6622331661519087, 
3.9535190128031594]], [[0.0085768436900446332, 0.31606611342030266, -0.18363822559980036, 1.0559798181457936], 
[4.9111849007168598, -3.8184858947259714, 4.6777467936412584, 1.4308325259695209], [4.7985728465154018, 2.0751684734846378, 
3.9070652967504778, 0.065876700270111499]]], [[[-0.72382633441175326, -0.47312341416877235, -2.6788002778241236, 
-3.7759038686685207], [4.2829560745644883, 1.4722843359088023, 4.446662541231035, -3.6812400812479429], [4.4414788402580001, 
-1.1637698257632767, -3.7766219209176475, 3.803210936489581]], [[0.70999860285818528, -0.96822186384663134, 
-3.5866553486626485, 1.8510339866640741], [-4.3379091894891797, -3.4195405840612603, -0.42598984537520046, 4.592390536322208], 
[-3.0064239094896794, 1.2764046499696491, 2.3923978686903329, -1.4239482513603461]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.12401646034934231, -0.18147504431893649, -0.19148080393200537, 0.45121274548301349], 
[0.13742675189914355, -0.12055889584981926, -0.3273759063954243, -0.15398178618913708], [-12.783821405709451, 
0.28102274225450485, 0.13361212257467517, -1.2603278013009191]], [[0.14569852361034746, 0.25288173646312423, 
0.28266900192425604, -0.26248725679432056], [-0.79936698602724776, -1.9815861169219207, 0.13557278372388998, 
-0.28988732188754018], [0.16001381188202271, -0.13140017899765366, -0.55108930667232503, 0.324286298065253]]], 
[[[-0.45640623893566096, -0.20163261637398755, -0.85066193912119004, 0.18502860559762477], [0.49756463948393231, 
0.62287528258879743, 0.845012786797669, -0.34683075307992894], [1.4602361352914721, 0.32401526636129518, -0.12048688280448598, 
-0.48509059739184812]], [[-0.12634142430623374, -0.18824583283424914, -2.2533813531980305, -0.14375812170050883], 
[-0.33295838040419445, -0.29873031619321749, 0.2413239015219352, -0.5131144784582824], [34.179004103636629, 
-0.50356288469525534, 0.23095277133577546, -3.8964696802379817]]], [[[0.36665161426977172, -0.70044011008983575, 
-0.19632075764238854, 0.52744024812865653], [-0.50998112852234512, -0.64091689754854098, 0.2117489569479831, 
-0.14834177497400519], [0.34417041580402569, -0.81272853778026521, 0.18201831262664528, 0.14499060929803717]], 
[[0.18338192098744138, 0.16462877978959523, 0.14266671347031709, 0.17614240368269057], [1.1802958872631617, 
-0.20880392408094711, 0.28213788609895446, 0.14354954838654055], [-0.20297568180522632, 0.127730491049619, 0.17469487522083363, 
0.15413707995840226]]]])+(1.-msk_ref)*numpy.array([[[[-0.26973986095235458, 0.24072197199236386, 0.21237977852440457, 
0.26652559844718315], [-0.18654624923087801, 0.2871355171748371, -1.0299194556989157, -0.25011720711306507], 
[0.13435130061192807, 0.53431071765342275, -0.21218690590985267, -0.2584911876725125]], [[0.16021743682613571, 
0.14347218763582104, -0.76810394192025477, -0.87727832803992123], [-0.14770785777852066, 0.36761970341096828, 
-0.24213594082666809, -0.12630601913995107], [-0.25095004858054148, -0.29603377176882689, -0.64917945214967554, 
0.13298636223341107]]], [[[0.17167116001752603, 0.66995386022079284, 0.12300938116116561, 0.15071802972957538], 
[-0.27745219374460922, 0.1326940072054639, -0.20337947456467467, 0.47880848371585327], [-12.237752477512814, 
-0.24793394897543555, 0.16233802130381464, 0.15037734327848179]], [[69.316838132001834, 1.8809978688089117, 
-3.2374506114099124, 0.56300288654212038], [0.12105422576525803, -0.15569513732326357, 0.12709531147655775, 
0.41550613014158538], [0.12389510480767833, 0.28649225031256542, 0.15216528022727688, 9.0247338331856444]]], 
[[[-0.82135680546835255, -1.2565847893848439, -0.2219350545346398, -0.15745095914106197], [0.13881059609203644, 
0.40380765538697644, 0.13370020329487731, -0.16149984044093166], [0.133856246337936, -0.51085676272486269, 
-0.15742102285998702, 0.15632046070393757]], [[0.83735331781357625, -0.61403249394136639, -0.165758799759777, 
0.32118247964627922], [-0.13705212805902792, -0.17385952034533725, -1.3956193843603668, 0.12945756268856504], 
[-0.19774978633908646, 0.46577680969780377, 0.24850368474528117, -0.41751495195021326]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_float_rank0(self):
      arg0=Data(4.25838984482,self.functionspace)
      arg1=0.815954760479
      res=arg0/arg1
      ref=Data(5.21890434504,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_array_rank0(self):
      arg0=Data(-1.77856744207,self.functionspace)
      arg1=numpy.array(0.389183376209)
      res=arg0/arg1
      ref=Data(-4.56999849119,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_array_rank1(self):
      arg0=Data(-2.8836086383,self.functionspace)
      arg1=numpy.array([-1.0381794048924178, 3.5434746401048152])
      res=arg0/arg1
      ref=Data(numpy.array([2.7775629382642761, -0.8137799564478172]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_array_rank2(self):
      arg0=Data(-4.79812755477,self.functionspace)
      arg1=numpy.array([[0.14991843556406437, -4.1068513257704664, -3.4357277338360772, -2.2871100738690053, 
3.8463011449165627], [-3.4117566103130512, 0.93353125196226738, -3.8611665206139345, 4.161512902010827, 4.6010139874998117], 
[-0.16619311206576803, 0.33601792694262222, 4.3070962317977575, 2.106870040761387, 2.1373058812732202], [-3.4598200606894203, 
1.385187182455061, -1.9179904761458877, 4.6592653996295024, -3.4126419969091937]])
      res=arg0/arg1
      ref=Data(numpy.array([[-32.004920120184643, 1.1683226818226189, 1.3965389362834342, 2.0978997074041756, 
-1.2474653892122671], [1.4063510686158438, -5.139761036050313, 1.2426626847494822, -1.1529767341229418, -1.0428413318904575], 
[28.87079672033672, -14.27937967009197, -1.1140051897025336, -2.2773723399839922, -2.2449419134675397], [1.3868141899306841, 
-3.4638838819363622, 2.5016430552942799, -1.0298034439404264, 1.4059862004618591]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_array_rank3(self):
      arg0=Data(-1.98387086714,self.functionspace)
      arg1=numpy.array([[[-3.5587691615859374, 0.25214505221229366], [-0.79731795595904487, 4.2154602414700157]], 
[[-1.7443498285164436, -1.7820096104704932], [2.8429099756379621, -2.1636012165743623]], [[0.65497487567676149, 
1.9256075763737375], [-3.7484737665916432, 0.39493195498054501]], [[-0.52169416325855877, -0.19517243295513875], 
[0.22067078161295051, 4.460717018343491]], [[3.8323792106938157, -4.5103398270823725], [1.6611678796847986, 
-2.6788113971836025]], [[-3.2342403770538111, 3.5806586497177726], [-3.7229810924621565, 2.4370598696258288]]])
      res=arg0/arg1
      ref=Data(numpy.array([[[0.55745983430223145, -7.8679746032345967], [2.4881803455075979, -0.47061785748114954]], 
[[1.1373125016011583, 1.1132773108972172], [-0.69783105484808683, 0.91693000167501193]], [[-3.0289266669774579, 
-1.0302570946846661], [0.52924763268156805, -5.0233232386457987]], [[3.8027469096953541, 10.164708391956099], 
[-8.9901837145673138, -0.44474259608476757]], [[-0.51766037703205969, 0.43984953311620095], [-1.1942627180547347, 
0.74057877655117943]], [[0.61339623400063081, -0.55405193882247239], [0.5328715934535071, 
-0.81404272905375086]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_array_rank4(self):
      arg0=Data(-0.531745584533,self.functionspace)
      arg1=numpy.array([[[[-3.9799891439902133, -1.5263398400147787, 2.660391253443124, -4.6977617554422304], 
[-4.7229197918274615, -4.6534607451579806, 2.1367144006494527, -4.126347196493926], [-1.24367456836017, -0.10588287863401735, 
-3.2974299575556412, 2.1225066644477488]], [[0.12519404186507366, 0.76014591679897769, 0.066277161774110027, 
3.6170961266510133], [-1.8161041805384381, -2.0490571213384112, 1.0938835505501832, -3.4705272542623655], 
[-0.38332840520168521, -0.10437305879362402, 3.9704468702648583, 3.8057956818694301]]], [[[1.5013768955961071, 
-0.55418213798327542, 1.2865145104494058, 0.26503790212629585], [-4.4010065376877963, -2.3385117591117801, 
0.010598815480228119, -2.5687419867258674], [-0.48221826207107998, 0.078959379963317033, 1.2835771836219534, 
-0.25521319754306226]], [[0.25086478183835137, 0.67169388307993394, 3.3196318872984172, 4.6754057826689639], 
[2.8015230792733847, -4.9605680553819482, -1.2039556066089609, 3.8300774356643341], [-4.7298394238060926, 4.8143184236572836, 
4.7355333476484383, 4.2677096005749231]]], [[[3.0352558385587862, 3.284763831752036, 1.0490723286320991, 0.18475058478273887], 
[-3.5320023449368829, -4.0891492620259617, -3.5572108934263227, -4.1996068428583282], [1.6538346735344316, 4.4678252251382755, 
2.0452410218890602, -2.7544697188494305]], [[-2.6938573646031005, -2.5319388216215777, 0.5491405692518363, 
-0.037448832966751944], [-1.9958301035222403, -4.2136408392081766, 0.12156847820469707, -0.50743887050711045], 
[3.7379483652671261, -1.696402052388982, 4.8430236758013159, -4.4068659599627669]]]])
      res=arg0/arg1
      ref=Data(numpy.array([[[[0.13360478265030593, 0.34837954863855286, -0.19987495592807838, 0.11319126260870438], 
[0.11258831569689622, 0.11426884498524707, -0.24886132857613819, 0.12886593376951003], [0.42756006921852191, 
5.0220166980098888, 0.16126061550298051, -0.2505271683901843]], [[-4.247371333425531, -0.6995309358137497, -8.0230590794717216, 
-0.1470089723674888], [0.29279464814388445, 0.25950744808229742, -0.48610803614876663, 0.153217521597135], [1.3871802280178425, 
5.0946632270766896, -0.13392587834777353, -0.13971995056555242]]], [[[-0.35417195115571909, 0.95951411654743202, 
-0.41332264829849452, -2.0063001565715766], [0.12082362977191996, 0.22738632057824618, -50.170284172354023, 
0.20700622611426828], [1.102707272531726, -6.7344194544134979, -0.4142684922405121, 2.0835348236396274]], 
[[-2.1196501981534746, -0.79164869284675321, -0.16018209325195235, -0.11373249922060062], [-0.18980589111238214, 
0.107194494379827, 0.44166544149505177, -0.13883416026578516], [0.11242360192124555, -0.11045085466728216, 
-0.11228842571606908, -0.12459741507757596]]], [[[-0.17518970815517002, -0.16188244019046136, -0.50687218604530393, 
-2.8781807925463752], [0.15055074504569502, 0.13003819387848348, 0.14948385138369169, 0.12661794411485289], 
[-0.32152281787452747, -0.11901664853439572, -0.25999164833993488, 0.19304825930542632]], [[0.19739188552461995, 
0.21001517887894056, -0.96832325693556742, 14.199256489661167], [0.26642828144287689, 0.12619622906287256, -4.3740416297556681, 
1.0479007727610385], [-0.14225600050400625, 0.31345492879152759, -0.10979619760900883, 
0.12066298121258998]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_constData_rank0(self):
      arg0=Data(4.96192666081,self.functionspace)
      arg1=Data(-0.645776129271,self.functionspace)
      res=arg0/arg1
      ref=Data(-7.68366378982,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_constData_rank1(self):
      arg0=Data(3.96933735761,self.functionspace)
      arg1=Data(numpy.array([4.8220319433945917, -4.4370897538008869]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([0.82316695621426328, -0.89458126336341792]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_constData_rank2(self):
      arg0=Data(4.50691106733,self.functionspace)
      arg1=Data(numpy.array([[2.886252860605202, -0.79139596289242053, 0.944668909632016, 2.7820772540590397, 
-0.12004220374048558], [1.7484762468684458, -2.9623165829666629, -0.66195492677624479, -0.27796216183514044, 
3.8075826347630066], [-4.8273722815534237, -0.14691394817340164, -1.2405247615428703, -2.0860562811086645, 4.4587992236040179], 
[-2.2200737650599986, 4.1576302186630478, 2.1935594125262536, 1.8672374024493426, -4.012419061336896]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[1.5615094328175132, -5.6948876146139842, 4.7708895903932786, 1.6199805597619508, 
-37.54438794771599], [2.5776221297850155, -1.5214143867155585, -6.8084863259201764, -16.214117193420758, 1.1836673027615263], 
[-0.9336158067927699, -30.67721699244019, -3.6330682039157374, -2.160493515034164, 1.0107903140093955], [-2.0300726661709607, 
1.0840095992906322, 2.0546108947835999, 2.4136786577966083, -1.1232403690729897]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_constData_rank3(self):
      arg0=Data(-0.566882706433,self.functionspace)
      arg1=Data(numpy.array([[[-2.3119681157581038, 2.3655353340300387], [-2.0819768571955488, 3.5477847807211766]], 
[[-0.53028219894214335, 2.8528706756765443], [-2.7687306675459258, 3.8961333110798044]], [[-2.8336317127714272, 
2.5747372415777425], [1.2506496252631569, 2.8368327531159618]], [[-1.4151569268536326, -3.8570875139593364], 
[-3.6981050221877574, 1.9560326358821998]], [[4.493895443221513, -3.17505489919786], [2.9010746877679008, 
-2.9102319414421265]], [[4.5646131732506134, -0.60178703417325607], [0.84814252772227849, 
4.4519872907615152]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[0.24519486344519292, -0.23964245990237645], [0.2722809835631636, -0.15978497611056189]], 
[[1.069020811114948, -0.19870606518064995], [0.20474461928628354, -0.1454987961579369]], [[0.20005518144011378, 
-0.22017109057915943], [-0.45327060031994726, -0.1998294421164897]], [[0.40057939559630196, 0.14697169934084225], 
[0.15329005072375873, -0.28981249905233597]], [[-0.12614505913528115, 0.17854264711333478], [-0.19540438197718729, 
0.19478952806491331]], [[-0.12419074408209976, 0.94199887043385722], [-0.66838141928259975, 
-0.12733250780147276]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_constData_rank4(self):
      arg0=Data(-4.56300388113,self.functionspace)
      arg1=Data(numpy.array([[[[1.4946807605936971, 2.0772167367453864, -4.4247892116920395, -2.847767949972825], 
[4.510886998762027, -1.3627771958956538, 3.9780703257559633, -3.3915549896273856], [-2.6906044768711412, 2.8756672928292568, 
-1.4096303422816825, 0.49961654737888317]], [[4.5877100474234478, 4.7180988422923544, -0.96139055207059165, 
3.9683702735457249], [1.8387365771993522, -3.1960432474938729, -2.9946664953421873, 1.1324438983090088], [-2.1077789681630521, 
4.1365242800107414, 4.71796394986435, 1.802685436555187]]], [[[0.90724226372239514, 0.13314682438763992, -1.1305640017125018, 
-1.835267121092051], [-2.5272765293843378, 3.6786560288153058, -1.6051432850511516, -4.0294718968855285], [-1.8829218142572337, 
4.9710533987003771, 2.2521195411977013, -3.3829400185551961]], [[-3.9122378359176055, -3.850625363429323, 0.99648794968039667, 
-0.41419187310360961], [1.3565054301622297, -1.0657769650293627, 4.645096722767784, 2.3031489157219154], [-3.5847499889653269, 
-1.4380909803387754, -4.8511616960284911, -4.1213430624736338]]], [[[-1.7658093308054648, -4.1510711144338206, 
-0.49751386525209806, 2.7195627641528466], [-0.15949557340524922, 4.1640932386883112, 4.7852837410922362, 2.0579219676092508], 
[-2.3582476026454655, 3.8897787918344857, -1.2225035508122293, 4.9978419193227506]], [[0.063444983809236355, 
-1.6712125407233449, -2.8444009422691217, -1.7614533428473766], [3.8367201134734401, -1.9516137460762795, -1.9552821753727088, 
-2.6768260567872026], [0.95776174468042363, -4.12359439823644, 0.81049880215928027, -4.7786736065022666]]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[[-3.0528284041860392, -2.1966912746332925, 1.031236441517615, 1.6023088823562237], 
[-1.0115535774633919, 3.3483124716743609, -1.1470395210430855, 1.3454017095657198], [1.6959028799495168, -1.5867634939928916, 
3.2370216107472376, -9.1330119169781767]], [[-0.99461470623966486, -0.96712765748541041, 4.74625413293617, 
-1.1498432773647917], [-2.4815973847003749, 1.4277040477186407, 1.5237101988578656, -4.0293421051095013], [2.1648398385469845, 
-1.1031009543884469, -0.96715530886199885, -2.5312257971365382]]], [[[-5.029531872125232, -34.270467223808737, 
4.0360420765379956, 2.4862886871834098], [1.805502416564764, -1.2403997126636288, 2.8427392891503893, 1.1324074215923139], 
[2.4233634379188622, -0.91791487943456307, -2.0260931081412212, 1.3488279000227805]], [[1.1663411255927876, 1.1850033307491312, 
-4.5790858610927732, 11.01664271425577], [-3.3637933027552456, 4.281387223456381, -0.98232698982680444, -1.9812022791850483], 
[1.2728931990171677, 3.1729591128204637, 0.94060024527041297, 1.1071642937659532]]], [[[2.5840864024936607, 1.0992352950217275, 
9.1716114862822682, -1.6778446672666603], [28.608968786469511, -1.0957977210347425, -0.95354928317987897, -2.2172871240748786], 
[1.9349129735198654, -1.1730754177357323, 3.7325076709177485, -0.9129948395304901]], [[-71.920640642776036, 2.7303552181075923, 
1.6042055862527924, 2.5904767217704268], [-1.1892980843473298, 2.3380670946311755, 2.3336804981925683, 1.704632196612532], 
[-4.7642369372910895, 1.1065598214708789, -5.6298712212459314, 0.95486828707489535]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_expandedData_rank0(self):
      arg0=Data(2.36273490703,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(4.23559942222)+(1.-msk_arg1)*(2.79375146456)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.557827752698)+(1.-msk_ref)*(0.845721223595)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_expandedData_rank1(self):
      arg0=Data(-0.863670625384,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-2.8559821849537936, 3.1726164740322993])+(1.-msk_arg1)*numpy.array([-0.68587967685299844, 
2.1592461254842608])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.30240756750309189, -0.27222660931541565])+(1.-msk_ref)*numpy.array([1.2592159449114995, 
-0.39998711364613154])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_expandedData_rank2(self):
      arg0=Data(-4.94692685712,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[2.3093529505907684, 1.8761997961936983, 0.068062276984583114, -4.9719118922598513, 
1.2628648970634604], [3.0414467848566833, -3.2422208345827839, -2.7975194681198747, 2.6539826066686221, -1.0942073549751918], 
[3.679393109473434, -3.6778001117031889, -2.8348517675800657, -1.1851476871509004, -3.8641737856996428], [-3.26555816112886, 
2.6860478388297651, -3.9606468126691485, -4.6727380164396752, 
-0.64963659615955827]])+(1.-msk_arg1)*numpy.array([[-0.85014886626487041, -1.6045848813429995, 4.4950316018795, 
3.9708446605819923, -2.5703533436314538], [-0.91973583674440551, 3.7833649005025922, -0.72677365544154426, 4.6564695148793547, 
0.16733868497418225], [0.88877528911703507, -4.5170260696191971, -1.0721588793006251, 3.4119771543249691, -3.2126965215006464], 
[-3.5067386418897426, -0.79474849313687734, 4.7829111553907211, -2.7684390813434332, 0.026890086209919062]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-2.1421268047652786, -2.6366738058282739, -72.682359102455621, 0.99497476309375543, 
-3.9172257211577892], [-1.6265044917948075, 1.5257834396591472, 1.7683261594776123, -1.8639635560135934, 4.5210140789415814], 
[-1.3444953311423242, 1.3450776841793475, 1.745038987116829, 4.1741015999589903, 1.2802029958982772], [1.5148794212301471, 
-1.8417121190512773, 1.2490199432325679, 1.0586784107561651, 
7.6149140709881564]])+(1.-msk_ref)*numpy.array([[5.818894847038492, 3.0829948073446949, -1.1005321642356909, 
-1.2458122339137834, 1.9246096531363037], [5.378638799848158, -1.3075468497545948, 6.8066953446720815, -1.0623771596306566, 
-29.562362450066487], [-5.5660040481535633, 1.0951734129666681, 4.6139867445309752, -1.4498710376338901, 1.5398052147214849], 
[1.4106916318287201, 6.2245186997445181, -1.0342920234988455, 1.7869011062810485, -183.96842682116343]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_expandedData_rank3(self):
      arg0=Data(4.7612239752,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-2.5370722145526168, -4.3629282017466053], [-2.4951529695545718, 3.1380299858963809]], 
[[3.5571207336459665, -1.9149375161776772], [4.4180746719296344, 1.9606798450726481]], [[1.0788639700645177, 
1.3830245180830563], [1.1469190526989461, 3.8864473021743269]], [[1.2960469921605853, -0.21271505581906247], 
[-2.517811863647792, -1.1176168970100187]], [[-4.5401428632876053, 1.7729769126759942], [-0.60525069244101726, 
-3.805613077469002]], [[1.9167480731514752, 4.8841920519773367], [-4.5025312413488763, 
-2.6130073152838973]]])+(1.-msk_arg1)*numpy.array([[[-0.015871483675765852, -1.6541480006313689], [-4.2529288013801292, 
2.7822014194945099]], [[1.5085403251991956, -1.655784855477874], [-0.6568543925994339, -2.3259851402753151]], 
[[0.22852158554616864, -2.4902260552757771], [-2.7241851636324754, 4.6904064479890746]], [[-0.51455341317367509, 
-0.82976993338497973], [-0.42466372680688025, -0.76699623671431016]], [[0.12297731672647849, -2.6539935942436319], 
[-1.5442481060755151, 4.9718225859580993]], [[3.6611920548160874, -1.4001386885330125], [0.068445415712168334, 
-2.84490792819591]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.8766608013329766, -1.0912909300904583], [-1.908189210560467, 1.5172652895609269]], 
[[1.3385050246303558, -2.4863599647395356], [1.0776694213550799, 2.4283536076362982]], [[4.4131828546625904, 
3.4426171864264901], [4.1513164891609078, 1.2250838889641473]], [[3.6736507271736403, -22.383107565511899], 
[-1.8910165783014099, -4.2601574725111391]], [[-1.0486947478463815, 2.6854404821412592], [-7.8665320579801454, 
-1.251105637457103]], [[2.4840113533405792, 0.9748232511197813], [-1.0574549558873394, 
-1.8221242425739812]]])+(1.-msk_ref)*numpy.array([[[-299.98606762087718, -2.8783542786888718], [-1.1195165020530948, 
1.7113153425344008]], [[3.1561794508696654, -2.8755088316276529], [-7.2485226997719963, -2.0469709340612474]], 
[[20.834898216825231, -1.9119645644678902], [-1.7477607758693046, 1.0150983775069709]], [[-9.2531190218631529, 
-5.7380049380421791], [-11.211751027106569, -6.2076236457147029]], [[38.716277944099872, -1.7939847275927041], 
[-3.0831988438062008, 0.95764156763136543]], [[1.3004573111478344, -3.4005373997560837], [69.562350168551689, 
-1.6735951023277087]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_expandedData_rank4(self):
      arg0=Data(-1.77757554352,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[3.4627739286113908, -0.65439868940868173, 1.5278620174733302, 0.024430472800522907], 
[1.7460020228800861, 2.244021966048714, -3.3020953020512023, -4.575544462315797], [-1.7262871273985656, 3.3280348525718377, 
-3.9999784861060981, -3.8229853205434794]], [[0.48161615829237903, -4.323922499526951, 3.1312057493306487, 
-4.6205034707182122], [-2.6299021394201705, -1.5605926511422172, -1.6849278391931986, -2.6226477344992851], 
[-0.68667538302815334, 1.8239538383256617, -2.3136335635504057, 3.2207568418004264]]], [[[0.38465013433963513, 
-4.5821954444295878, -0.51372354937130549, 1.6054285802860253], [-2.9495819623386064, 3.583566418230701, -0.85413525657389222, 
2.5189636100625101], [-2.4339409475882903, 3.5386135634581777, -0.013452086167511368, 2.8136635664492466]], 
[[0.34604490964475154, 0.81066807356521231, -1.7180435717279208, -2.8455925905152792], [3.7266456415946934, 3.7424945298055601, 
-4.6515421176151808, -3.9253718083679967], [3.7757693906827434, -3.0710322721455627, -1.6781364505415173, 
-3.702880482824674]]], [[[-4.9080435067971777, 3.9373474398620978, -2.9102078037640289, -2.3861455760154291], 
[-0.30535345886808329, -0.14666574100751362, 3.8462018981193378, -3.8704125343216766], [1.5325410387316545, 3.8896771871844216, 
4.0357198995365486, -4.3411322491065274]], [[0.97611372648148809, 0.77756420355764178, -4.2169527780144325, 
0.1269485374959709], [4.3673407265977904, -1.7920660163262001, -4.0888466409316484, -0.55343112299995667], [4.6102418626121331, 
2.430421016550314, 0.015091015036765576, 4.3297672834532595]]]])+(1.-msk_arg1)*numpy.array([[[[-2.568063125945689, 
3.6758075873914091, 4.6721205615377261, -0.18008733696011525], [-2.6720449257934287, 2.2793985653116184, -3.4102723809570401, 
-3.8149657351358499], [-2.5022187010978545, -4.1554950981967735, -1.3224493292631232, -0.81941703793439125]], 
[[4.868357652927962, 1.71865432208374, -4.2915752626314774, -2.7000097376814436], [-2.1229325293388648, 1.7316544743225322, 
-2.7217907106885475, -2.9069933430833683], [4.0058010415708747, -4.3839383358370689, -1.192112733705847, -2.74752496784338]]], 
[[[2.5728216667554964, -3.5435101896700107, 1.3802658999226551, -4.1247606443021159], [-2.5060149418618782, 
-3.9920472301675614, -3.0777263436961331, 0.82621258694888589], [4.7531074608373007, -1.4923410742800747, 3.7752886692664749, 
-1.8545576758622797]], [[-4.1797498847194925, -1.1101246546741739, 4.6532801525505096, -2.5657753602879718], 
[3.2557292075393391, 3.1740956157493638, 2.3272652226443435, -2.7393077256053164], [-1.6175524389352605, -0.058678892112294712, 
-2.2787242124967433, -0.15122155694843364]]], [[[-4.5899726769239333, 3.5612155119641589, -4.4225197073483375, 
0.015603834164149788], [-2.9086805431861364, 1.3213616898524698, -3.9186729266306006, -2.4652455809946785], 
[-0.19497802695558253, -2.8010352776294178, 0.36757395415538596, -4.649430270321079]], [[0.33706981618057696, 
4.4234696669302345, 0.9758276617181485, -1.5709392317965607], [4.2249854435209393, -4.200162865274546, -2.671784902114779, 
-4.837772872730759], [-2.1368273463166343, -4.6953021886637583, -1.5610020731140093, 2.3356311419080313]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.51333860660952746, 2.7163494858515036, -1.1634398415485181, -72.760587076281368], 
[-1.0180833242019978, -0.79213821005822649, 0.5383174563171973, 0.38849486835004882], [1.0297102465195893, -0.5341216730778815, 
0.44439627605287596, 0.46497053859062909]], [[-3.6908552857108292, 0.41110254490260689, -0.56769681899605584, 
0.38471468635015738], [0.67590938722498728, 1.1390387762086029, 1.0549861555901021, 0.67777899415704201], [2.58866938796944, 
-0.97457265977133212, 0.76830470110806015, -0.55191237054812647]]], [[[-4.6212788839107013, 0.38793097437127128, 
3.4601792066814268, -1.1072280419976548], [0.60265338146691183, -0.49603532795544092, 2.0811405802956577, 
-0.70567734143370908], [0.73032813112352213, -0.50233672358956338, 132.14125462638347, -0.63176549062701692]], 
[[-5.1368348268495643, -2.1927291840908762, 1.0346510256019732, 0.62467675430490477], [-0.47699076179309513, 
-0.47497077934527371, 0.38214757570087821, 0.45284259180946507], [-0.47078498700254179, 0.57882020962133207, 
1.0592556659761281, 0.48005209775530749]]], [[[0.36217599559893793, -0.45146524929977438, 0.61080708436630371, 
0.74495687161112101], [5.8213702576236628, 12.119909743790174, -0.46216386726509201, 0.45927288829140434], [-1.159887728023459, 
-0.4569982180973442, -0.44046058392736881, 0.4094727922382515]], [[-1.8210742204442119, -2.2860820179016854, 
0.42153081551796473, -14.00233179979397], [-0.407015539843607, 0.99191409653605533, 0.43473764110461915, 3.2119182851183989], 
[-0.38557099529481254, -0.73138585101614295, -117.79032352605451, 
-0.41054759462717433]]]])+(1.-msk_ref)*numpy.array([[[[0.69218529932459449, -0.48358775622927891, -0.3804643994315336, 
9.8706304036849524], [0.66524912300617201, -0.7798441091297944, 0.52124151532386442, 0.46594797094655205], 
[0.71039975152341994, 0.42776504399885534, 1.3441539907677569, 2.1693172843929767]], [[-0.36512838009099197, 
-1.0342833463811343, 0.41420118132268513, 0.65835893801020617], [0.83732079044017138, -1.0265186097316579, 0.65309045862217918, 
0.61148249539221333], [-0.44375033234803252, 0.40547457727362407, 1.4911136281476765, 0.64697339035009616]]], 
[[[-0.6909050737895932, 0.50164256580921729, -1.287850075566439, 0.43095241077144375], [0.70932360131759487, 
0.44527918659984367, 0.57756127251470546, -2.1514747797314073], [-0.37398177048663556, 1.1911322244981408, 
-0.47084493378953907, 0.95849030022249582]], [[0.42528275436185953, 1.6012395869533194, -0.38200484072356217, 
0.69280248420409762], [-0.54598384269822353, -0.56002583372004289, -0.7638044543532555, 0.64891414969611427], 
[1.0989291603351319, 30.293270365694642, 0.78007489180514911, 11.75477610062778]]], [[[0.38727366558278731, 
-0.49914854564271049, 0.40193728036160015, -113.91915120459279], [0.61112780077596074, -1.3452603909801844, 
0.45361671586242497, 0.72105414455299555], [9.116799319761892, 0.63461376502961242, -4.8359670847780718, 0.38232115338174871]], 
[[-5.2736123443484999, -0.40185096256142266, -1.8216080700022146, 1.1315367950193058], [-0.420729388841588, 
0.42321586103565972, 0.66531386643819224, 0.3674367503974591], [0.83187607392860929, 0.37858597212526868, 1.1387400274050397, 
-0.76106860866096682]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([-4.6919095076246728, -2.3314119572168543]),self.functionspace)
      arg1=2.33042634309
      res=arg0/arg1
      ref=Data(numpy.array([-2.0133266694034737, -1.0004229329664065]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([1.6623719770900465, -1.7721667711563969]),self.functionspace)
      arg1=numpy.array(0.948804041941)
      res=arg0/arg1
      ref=Data(numpy.array([1.7520709267735919, -1.8677900734181856]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-3.3230332672732099, -3.6300997874480667]),self.functionspace)
      arg1=numpy.array([-2.3704951151280431, 1.8278016313594083])
      res=arg0/arg1
      ref=Data(numpy.array([1.4018308859049116, -1.9860469129509519]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([-4.2117228251585122, -0.077373340375955557]),self.functionspace)
      arg1=Data(-4.02083378559,self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([1.0474749889564914, 0.01924310839538227]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([2.2838873194003204, 4.0954865061064982]),self.functionspace)
      arg1=Data(numpy.array([-0.89973305674607928, 2.1161222961580659]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([-2.5384054773535727, 1.9353732596372502]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([1.9512405076586496, -4.9230227324262259]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-4.72730635147)+(1.-msk_arg1)*(2.47365983572)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.41275947920177963, 1.0414012476468399])+(1.-msk_ref)*numpy.array([0.78880712678616149, 
-1.9901777363815558])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([1.5810907020408083, 0.56735398310493768]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.61197556118875429, 1.3234499790592817])+(1.-msk_arg1)*numpy.array([-4.5578811745449235, 
-2.9740367784300559])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.5835847087905943, 0.42869318227517533])+(1.-msk_ref)*numpy.array([-0.34689160192919477, 
-0.19076898685981764])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[2.6165681034632957, 4.4449598965285109, 4.3542894190831731, 4.3004700444691082, 
-1.4327735886729975], [4.9178986796821569, -1.3337584616613309, -3.47717708562443, 3.2561340732611423, 4.5504352295662116], 
[-3.6600099681024822, 1.6572854738258922, 3.34173080591356, 4.0312264840652219, 4.687281872718593], [3.1043165319352077, 
1.9564399745825689, 4.971730422412314, -0.5678000814562143, -1.1168096691507321]]),self.functionspace)
      arg1=1.29148999534
      res=arg0/arg1
      ref=Data(numpy.array([[2.0260072574309849, 3.441730026990971, 3.3715239256875722, 3.3298516132359417, 
-1.1093958093675316], [3.8079262692067268, -1.0327284504497316, -2.6923763235985798, 2.5212228395139604, 3.5233995199162598], 
[-2.8339437249237243, 1.283235239763866, 2.587500342987552, 3.1213764710553042, 3.6293598011773596], [2.4036705999507886, 
1.5148704067690544, 3.8496081582839112, -0.43964729382643086, -0.86474511856718439]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[-1.58381931538214, -2.5524449400651692, -3.3658141976033615, 1.4409777013213212, 
1.7962652893911892], [-1.7267444093615767, -1.9577340731760486, -3.588590449844661, 0.9631140259270401, 1.9962903486085253], 
[0.45086952873468533, 1.4278603595086947, 0.60088915029480194, 4.6942451574255308, 0.61157832014572033], [-0.75072956269803814, 
0.22344692951877487, -0.2131949758892473, -1.9106626823759862, -0.4887617148542045]]),self.functionspace)
      arg1=numpy.array(-3.83627804)
      res=arg0/arg1
      ref=Data(numpy.array([[0.41285310889064969, 0.66534409483689017, 0.87736450864886639, -0.37561868203954218, 
-0.46823125713590952], [0.45010929639563768, 0.51032121571068989, 0.93543544352860475, -0.25105428122918833, 
-0.52037165392983831], [-0.11752785487229953, -0.3721993934280648, -0.15663336807962464, -1.2236457077620648, 
-0.15941970674941608], [0.19569216695732816, -0.058245759871692213, 0.055573390058389349, 0.49805114813235407, 
0.12740518537974577]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[4.8823472974104334, 4.3613758390592245, 0.18436836430571812, -0.40555708035477167, 
4.1261796398443167], [-0.49590540079522683, -1.3410261268540156, -3.264579080153005, -2.3122838557368963, -2.7133740460154021], 
[1.3616532085187236, -3.2320824985513386, 0.95201425328335443, 1.2026768165933079, -3.6676581759429028], [3.8741975673552567, 
-2.210718583927803, 4.7663796900168407, 4.6523601686764877, 1.1033524543408699]]),self.functionspace)
      arg1=numpy.array([[2.6867537808736977, -4.0269443887579373, 2.013048595935361, 2.7917989124415836, -0.08466618422211436], 
[-4.6253938289713359, -0.41913549352135426, 1.6907899391852617, -0.2626275715741393, -3.4402058458855054], [3.0602064308057635, 
-2.7169216499902538, -2.7699342778096439, -1.0598153344495973, 4.17609424186141], [-0.99889915238741978, -4.3601765683209681, 
2.4823550172068902, -1.0727802199949932, -4.0014372796122153]])
      res=arg0/arg1
      ref=Data(numpy.array([[1.817191933316181, -1.083048440210628, 0.09158664360015191, -0.14526729649023662, 
-48.734682893227408], [0.10721365988104707, 3.199505046894084, -1.9308011033742623, 8.8044215688303797, 0.78872432859231489], 
[0.444954691556607, 1.1896119634377138, -0.34369561072624805, -1.134798466770538, -0.87825081608027067], [-3.8784671686783669, 
0.5070250136175366, 1.9201039565162206, -4.3367318691783865, -0.27573903506187092]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[4.3128088721702103, -3.1929184276928058, -4.4550921562343593, 1.4272841845822626, 
-3.5096241544694307], [3.4315971569689125, 4.0382135845403333, -3.6533532213362738, -3.0168456796036724, 4.4090065031170393], 
[-2.6722130648207019, -2.7166717077591143, 1.5307241928907933, 2.4384962007290403, 2.8705996084496803], [-4.0086696927881524, 
4.2006858101123719, -2.0209364140231534, -3.0080544425526288, 1.5424608208526127]]),self.functionspace)
      arg1=Data(3.73475844249,self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[1.1547758546056919, -0.85491966263969088, -1.1928729059291381, 0.38216238253720658, 
-0.93971918358692019], [0.91882707002668329, 1.0812516115098174, -0.97820335038917372, -0.80777531560882232, 
1.1805332449237491], [-0.71549823260868028, -0.72740225361111821, 0.40985895512721709, 0.65291938910630143, 
0.76861720848997039], [-1.0733410887257797, 1.1247543515327552, -0.54111569600649023, -0.80542141851308757, 
0.4130014951716543]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-4.223676127637912, -2.9976717232736725, 3.6768084516766919, 0.89891165125912309, 
3.7924525579392281], [2.9338611978838225, -4.3131052594043489, -3.8106757114515597, 4.9554656912002422, 0.53207878105600059], 
[-3.5825753607909192, -4.9549228172457127, 0.96267122254188209, -0.062998584519265322, -4.7294891052877759], 
[1.3328061776336426, 1.1214277593514721, -2.9332245473363141, -3.5634424847579371, 2.3790321411475999]]),self.functionspace)
      arg1=Data(numpy.array([[3.4678054860011844, 3.3592102864323365, 2.1593081371138219, -1.6722582817147638, 
-3.8019344391045484], [2.2197467934076549, -3.8410458048169938, -2.58498793492419, -1.7852093885654319, -3.7542650856223525], 
[2.0766192377017827, 3.376025170232305, -2.9124634805214997, 1.6830840686538124, -0.13646332896710156], [4.7771396109892628, 
-1.8526086377602602, -0.66589812280112781, 4.1062851627380965, 0.15639677189190326]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[-1.2179680044593104, -0.89237394139363702, 1.7027715444962821, -0.53754354879759536, 
-0.99750603769812674], [1.3217098484372136, 1.1228986787909045, 1.474156091782036, -2.7758456363387052, -0.14172648146069758], 
[-1.7251960762704841, -1.4676794654658227, -0.33053503639795279, -0.037430444320974247, 34.657582671370683], 
[0.27899669805916383, -0.60532361584324657, 4.4049148764642627, -0.86780200193934209, 15.211516915399732]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[-3.6427606067254024, 4.8475383574250976, 1.5860825634757125, -4.7323489938150107, 
4.5714223729389065], [1.6168990543222872, -0.2415898577030271, -0.2308876544095595, -4.2390138086503848, -4.4307064882055531], 
[-4.2305771648527735, 1.1944708800808552, 2.5316490927645798, -0.7372753725372041, -0.65054097279150991], [-3.2577077227900206, 
1.6627530460018356, 4.3956568549652193, 1.7369991278881081, 4.7939637942364381]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.13675315331)+(1.-msk_arg1)*(-0.268292932805)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[3.2045309011123724, -4.2643720347740173, -1.3952743907984688, 4.1630401288325469, 
-4.0214732280439041], [-1.4223836103830265, 0.2125262261196541, 0.20311151434842603, 3.7290539254714732, 3.8976856807445075], 
[3.7216322229030392, -1.0507741954360417, -2.2270878117983206, 0.6485800108748967, 0.57227989286512104], [2.8658004715433063, 
-1.4627212963141785, -3.8668525723091247, -1.5280354603171096, 
-4.2172425739649011]])+(1.-msk_ref)*numpy.array([[13.577549615784839, -18.068080686072069, -5.9117567760460981, 
17.638738912489927, -17.038922066054319], [-6.0266181349510068, 0.90047044913608199, 0.86058045583073106, 15.799945844008159, 
16.51443607508121], [15.768500200958961, -4.4521145883058395, -9.436137830003414, 2.7480238291362253, 2.4247413675415013], 
[12.142353839625018, -6.1975283083951282, -16.383796654676509, -6.4742634467754367, -17.868393863804624]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[3.5111799522701688, -3.1856883713044706, -1.5153564594240065, -4.7601454151211389, 
0.86501497059360766], [4.0914563279815717, -4.4864110679757108, -3.6156550411549171, 1.2208109336417872, 4.7552518035475728], 
[4.1501657340148963, -3.4606179523360767, -4.8509777680311679, 3.8464634654294869, -3.5282409796154592], [-4.262256056941931, 
3.0679692535919507, 2.2951515970165364, -0.48850109481454318, -3.7292535041982711]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[4.9875194926416615, -0.32073028318678354, 3.5013770949966059, 2.6542148978493207, 
2.2247944547526632], [1.0162771586802934, -2.5019645114401392, -0.57334992449153166, 0.20525610484781875, -3.4413856905623907], 
[4.1951658536587004, 2.6308295014958016, 3.3844204148166472, 3.8628396630156114, 0.74420170807912722], [3.4652672857860232, 
4.8128629505631153, -3.4619052828485852, -3.0628550716651217, 
4.9304633806788569]])+(1.-msk_arg1)*numpy.array([[2.2618750324153867, 2.5401658869774932, 0.041258040465745438, 
-3.2240673696769031, 2.0222595942932466], [0.80708619944110005, 4.2227685837001196, -4.4096524284029135, -0.85687143649362252, 
2.119479979012322], [-3.9360664422376779, -4.157006968699454, 4.5633453342153469, -4.0424332700275958, -3.5901311953989912], 
[-1.2166114261777183, -3.9862830489547951, 4.5030981874263372, 2.7181286549069474, 4.2925527854537489]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.70399322898895722, 9.9326086069933801, -0.43278870521813234, -1.7934287909310693, 
0.38880669121848105], [4.025925696583216, 1.7931553575047783, 6.3061925827607226, 5.9477448163937554, -1.3817840344336616], 
[0.98927333954995877, -1.3154094365934719, -1.4333259977968702, 0.99576058055349359, -4.7409740414628541], 
[-1.2299934479585541, 0.63745202909486376, -0.66297353898948952, 0.15949206977951114, 
-0.75636978033590108]])+(1.-msk_ref)*numpy.array([[1.5523315399616431, -1.2541261134307553, -36.728754984913401, 
1.4764410507954653, 0.42774675073104013], [5.0694167869737683, -1.0624335620221412, 0.81994104974491921, -1.4247305740957248, 
2.2435936411928368], [-1.0543942270587032, 0.83247826582757767, -1.06303104690745, -0.95152182076792302, 0.98276101556877682], 
[3.503383220995095, -0.76963156301617197, 0.50968277872001899, -0.17971963686585193, -0.86877289356479426]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-4.9383818938254818, 4.5942629742453605], [-3.3516736954458812, 1.7322123987737568]], 
[[3.1843962341774983, 4.6194843574059199], [1.2316456466250072, 3.868367500032253]], [[-0.26721363492245498, 
-0.9041137804010706], [-1.4995699109249738, 3.9838495856877252]], [[-2.4587368678608232, -2.1477736143374448], 
[3.4061121009005006, 4.0831341837149324]], [[-2.9092037512352054, 4.577754438191171], [0.13651497521491152, 
0.51887004432362716]], [[-0.13893821977804333, 0.84281606830615363], [0.31552297942309337, 
0.060481252078473879]]]),self.functionspace)
      arg1=-3.57625207886
      res=arg0/arg1
      ref=Data(numpy.array([[[1.3808819358741331, -1.2846585959103189], [0.93720286532973363, -0.48436529656699967]], 
[[-0.89042834899813827, -1.2917110582659992], [-0.34439564646650062, -1.0816819996841442]], [[0.074718903765834052, 
0.25281041729310033], [0.41931325808677056, -1.1139733715195412]], [[0.68751777381612533, 0.60056549901373102], 
[-0.95242506003364369, -1.1417355638477029]], [[0.81347838102209091, -1.2800424403121746], [-0.038172637779648827, 
-0.14508765961752651]], [[0.038850231111913741, -0.2356702071671648], [-0.088227276060463516, 
-0.016911909659849785]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[3.5415327327380641, 4.6801702844340412], [0.41618802137957189, -1.9685725208894569]], 
[[-3.0083520222289275, -2.291094513654226], [2.9626343194359457, -2.6651865801579087]], [[1.6348161979117908, 
-3.0705198176217232], [-3.396725228495312, -3.8416107532641455]], [[-2.8764388161670995, -3.1552172235219853], 
[3.8011814807881841, -3.7494368563476996]], [[-4.4119791696599533, -3.07944591384656], [-3.4486135722451383, 
3.818258360943231]], [[2.3539478430257468, -3.3555055033807948], [-2.0071206234414163, 
1.9654936360007804]]]),self.functionspace)
      arg1=numpy.array(-1.51915458184)
      res=arg0/arg1
      ref=Data(numpy.array([[[-2.3312523788328843, -3.0807729117032561], [-0.27396028445931025, 1.2958342386067734]], 
[[1.9802803863302914, 1.5081378426108762], [-1.9501862120182234, 1.7543880076531095]], [[-1.0761355147464262, 
2.0212030127277614], [2.2359312667002365, 2.5287819943917507]], [[1.8934470859944352, 2.0769559998908376], 
[-2.5021689867629324, 2.4681075258300464]], [[2.9042331981155862, 2.0270787124962495], [2.2700873324291444, 
-2.5134100285628533]], [[-1.5495117291940563, 2.2087979350385627], [1.321208945705491, 
-1.2938075291951874]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[1.5274558176793533, -0.040409249759264299], [-1.672136978160971, 2.8277175751749963]], 
[[2.0176985747516927, 4.8549565534372441], [-0.17908639207953314, -0.46788475519067152]], [[-3.1512871860347822, 
1.7761080757115408], [1.692838908990371, -2.0099542784875579]], [[4.4223926331368233, -4.4523472966026025], 
[-1.3629772489800116, -4.6297740158746548]], [[1.9980996691995569, -0.34473906898421092], [0.3070942216226813, 
0.63920298172092416]], [[1.9245858888065417, -1.2129631475867542], [-1.4086737634787228, 
1.6005656044174108]]]),self.functionspace)
      arg1=numpy.array([[[1.8190553556342071, 0.51676667944010113], [2.329124277183718, -4.7484591324786241]], 
[[4.4593799498557107, -3.7290040994503091], [3.2794364352465859, -2.284915832265825]], [[0.46298258411229298, 
4.1150723439686612], [3.2507743801674778, -4.4060486518839328]], [[-1.685596550770263, 1.5421457414863351], 
[4.0804498129726401, -1.4332617381395307]], [[4.0056358704983417, 4.0756074424272857], [3.0600420073637675, 
1.9833611266956508]], [[0.62285942635296276, 0.18506448511937457], [0.35034677100983025, 3.9509759418254031]]])
      res=arg0/arg1
      ref=Data(numpy.array([[[0.83969727086552093, -0.078196314443195775], [-0.71792518524724269, -0.59550214001714918]], 
[[0.45246168692510225, -1.301944547111898], [-0.054608892599580863, 0.20477111173355389]], [[-6.8064918512581913, 
0.43161041343895923], [0.52074943106422444, 0.45618068189695188]], [[-2.623636498968827, -2.8871118836740988], 
[-0.33402622540456439, 3.230236245533499]], [[0.49882209312025488, -0.084585935680521945], [0.10035621108588755, 
0.32228270137867365]], [[3.0899201447035898, -6.5542729433167075], [-4.0207984775152878, 
0.40510639092323308]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-3.8118030152837665, -2.7306637470293662], [-1.7770607237473146, 0.095002454584742324]], 
[[0.4561458719664655, 4.2977553503859465], [-0.20650218801172215, -4.4939983551137788]], [[-0.92331909664028622, 
2.7263886689755079], [-3.5792622802169349, 3.7094451044875534]], [[4.6926414896433606, -3.7639105760814928], 
[4.416227646940138, 1.4770417678976324]], [[-2.727795891029837, 1.8546569168537284], [2.7746485671501597, 1.2982754076682816]], 
[[2.3643061950471349, -1.1023279794389742], [3.9313146772468333, 0.058822333133246474]]]),self.functionspace)
      arg1=Data(-0.310252690032,self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[12.286123981370448, 8.8014184397560822], [5.7277850630899012, -0.30620993028302312]], 
[[-1.4702398613200951, -13.852435413037488], [0.66559354566946716, 14.484961773107887]], [[2.9760228559043265, 
-8.7876390973322636], [11.536603533882836, -11.956206098029167]], [[-15.125224181499107, 12.131758070159902], 
[-14.234292848478477, -4.760770221674731]], [[8.7921748261092976, -5.9778914943942461], [-8.9431893946389955, 
-4.1845742176656859]], [[-7.6205824188186311, 3.5530005536009019], [-12.671331477720718, 
-0.18959491737920781]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[-3.2741465831795313, -3.3221231152268196], [-3.0922470181619666, 0.29885127433512793]], 
[[3.6362496834447597, 3.4395749548826551], [-3.8639257756394816, -4.0063078258599729]], [[4.449475202433149, 
-0.74271115626113993], [-2.4609090738054249, -2.2657634445690658]], [[1.252506551886361, -4.6544142467290879], 
[1.4362022111052006, -1.982767592167888]], [[2.3213710189349488, -2.3052564260727868], [-3.0563945623115254, 
-2.2093265873740275]], [[-3.4301379220743589, 4.1930210016030074], [-4.7918501872311623, 
2.3408824720772756]]]),self.functionspace)
      arg1=Data(numpy.array([[[-1.9323369567692685, 4.610521223529954], [-2.6916167790530885, -4.4848600983385651]], 
[[3.1458900539557408, -0.51197897221840538], [-0.74140176126872959, -0.72199880229098135]], [[-4.6778461199810373, 
-1.3539056490770571], [-4.5318218759655915, -0.12545463565171922]], [[2.1372528804373943, 1.5431904098740068], 
[3.0641396522901818, -4.6001579118182772]], [[-3.1451013859919974, -2.3191823493033348], [-0.26832946189206375, 
-2.8569232785476197]], [[3.8338159296631265, -4.6480078385056274], [4.8308387942766053, 
-2.6551926091610492]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[1.6943973315366665, -0.72055261306948326], [1.1488437143900625, -0.066635584562791297]], 
[[1.1558730982579717, -6.7181957492882445], [5.2116490376652305, 5.5489120108614021]], [[-0.95118032707993094, 
0.54856936062526818], [0.54302864083356783, 18.060420269037831]], [[0.58603572995537723, -3.0160984781580491], 
[0.46871303990069857, 0.43102163668641785]], [[-0.73809099740762873, 0.99399533062385925], [11.39045463274908, 
0.77332373744988614]], [[-0.89470594963482286, -0.90211143080840805], [-0.99192922622637802, 
-0.88162435523534965]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[3.5216301703435313, -3.0568779074453634], [-0.82648524576659721, 3.7346917114418403]], 
[[0.42973301787825413, -2.6887047054313307], [-1.1620217936364865, 2.559575729625795]], [[2.7937050008080275, 
-3.2116279598049835], [2.913474612408649, 2.6290743621314245]], [[-3.6863352815298214, 4.8696194649328266], 
[-1.7862778429409909, 2.2156066485611028]], [[-1.5746683164964823, -3.6896583092070223], [0.77993345875836706, 
-1.3697546416182229]], [[-1.8199292652251486, 4.7716338700254362], [0.27305502006689242, 
-4.4744785663413822]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.77332450624)+(1.-msk_arg1)*(2.78973450911)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.93329639804965292, -0.81012854907987486], [-0.21903370473450434, 0.98976160286999959]], 
[[0.11388710861415009, -0.71255591746405], [-0.30795702614915327, 0.67833437738921842]], [[0.74038291596448069, 
-0.85114014299495655], [0.77212405336272205, 0.69675278598109891]], [[-0.97694626461004097, 1.2905382128898972], 
[-0.47339629549178469, 0.58717628046536952]], [[-0.41731590110886535, -0.97782692771540292], [0.20669662984701903, 
-0.36301002984347308]], [[-0.48231453780795286, 1.2645702382973523], [0.07236457389645401, 
-1.1858186485009075]]])+(1.-msk_ref)*numpy.array([[[1.2623531589980475, -1.0957594342618231], [-0.29625946234921291, 
1.3387265703038138]], [[0.15404082950353093, -0.96378515469813364], [-0.416534903174828, 0.91749796307335285]], 
[[1.0014232507369267, -1.151230681383784], [1.0443555122870558, 0.94241023780007904]], [[-1.3213928671309592, 
1.7455494237990028], [-0.64030388451176101, 0.79419982128201505]], [[-0.5644509580940934, -1.3225840298263147], 
[0.27957264614638849, -0.49099820687029405]], [[-0.65236647404285875, 1.710425796590987], [0.097878496744049803, 
-1.6039083832982681]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[3.3460560206471559, 3.9429065150274738], [-4.0690020023604889, -2.7992991831481442]], 
[[-0.8109479343188557, -1.6372129368702315], [-1.7551517698005368, -2.3944389587870574]], [[-3.1771766832387782, 
4.8037508914991083], [-2.4878231197516021, 3.5469179719945796]], [[1.8098313084545588, 2.2481862523086313], [4.367750406142596, 
0.72928066811825687]], [[2.1837570936425816, 0.26776752926477076], [-2.7866462514972143, -1.4416721112681499]], 
[[3.1768583574641376, -2.539056891011847], [-3.4990592624298045, 1.6010252799532134]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[4.7662547126075552, 4.9396338312307222], [2.626480529867222, -2.2230665271274654]], 
[[0.081150032801920347, -4.4960334572487701], [-2.7446288035599906, -1.9624659172973224]], [[-1.7263788861372653, 
-0.5386133879550723], [-1.5354726034474586, -4.4304723632205274]], [[0.88257731646088544, 1.1877322072279037], 
[2.7550656265900839, 4.1943964533755]], [[-3.2542087308268428, 2.9241889786618156], [-0.20999511218406219, 
0.70541446648579686]], [[4.9247938754555669, 1.0460421025250355], [-4.6631516801949635, 
-3.1150708876411715]]])+(1.-msk_arg1)*numpy.array([[[-3.7732499291966026, 1.8629648090071509], [0.28894370267904534, 
4.7822714216282112]], [[0.57301967216231287, -3.4489813465108154], [4.1254030543939901, 1.0575152392327141]], 
[[0.78687940599385087, -4.3411578826759412], [4.9320106212239665, 2.9921241993323608]], [[-4.7441202829031308, 
0.61647791997378043], [4.4843288294920054, 3.4404971636781596]], [[-4.1657904807541133, -2.7758042010781523], 
[1.7744409387628979, 4.7499407893724133]], [[-4.1969959078818144, 4.7291009420633738], [-0.87635153666952093, 
-1.0297909762117929]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.70203046677221592, 0.79821838009500568], [-1.5492222219390264, 1.2592062131245607]], 
[[-9.9931929331230691, 0.3641460750766034], [0.63948602722669545, 1.2201174744907883]], [[1.840370447502194, 
-8.9187365166270371], [1.6202328287498693, -0.80057332067777787]], [[2.0506206931670761, 1.8928393442792666], 
[1.5853525825257802, 0.17387022810668218]], [[-0.67105624570299882, 0.091569844226452188], [13.270052919396997, 
-2.0437234842236922]], [[0.64507438033033682, -2.4272989441656612], [0.75036359578239387, 
-0.51396110640857984]]])+(1.-msk_ref)*numpy.array([[[-0.88678356415144632, 2.1164685967035566], [-14.082334948411317, 
-0.58534929039955497]], [[-1.415218313986867, 0.47469463368554565], [-0.42544976737027301, -2.2642122495788861]], 
[[-4.0376920008802548, -1.1065598214405461], [-0.50442371495424809, 1.1854180293672338]], [[-0.381489338492709, 
3.6468236403410028], [0.97400315012968941, 0.21196955946291174]], [[-0.52421193618150153, -0.09646484761452806], 
[-1.5704361811219281, -0.30351370157997926]], [[-0.75693625326107816, -0.5369005487761952], [3.9927576047023323, 
-1.5547089816641946]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-4.899897955446189, 1.8545587121230733, 0.44602070968963226, 1.5653851170011732], 
[-3.1969104701030302, -1.3044859695579292, 2.8772779804663156, -3.0694704706488296], [-0.22738540534536256, 3.8762798001948582, 
0.52592214032583851, -4.0183453926649477]], [[3.6392604285809078, -3.1353258106573367, 3.3609466904147869, 
-1.3248033917634263], [-2.6289369338057145, 1.4492341024480639, -0.095002785387556798, -4.2478514742128981], 
[0.43439220274257728, 4.249455471256212, 0.47822670102636078, -0.65606939292081812]]], [[[-2.1939335545992398, 
1.4699312110403904, -3.7642367565730064, 1.2495966999821455], [1.9999965053150168, -3.3727848598603294, 4.1095796456054643, 
-3.0605666809094068], [-1.2917390084880118, 4.9142128240624174, 2.6206700267014522, -1.4659864122513544]], 
[[-1.2172227520451049, 0.37734384246795294, 0.68835894972758815, -0.89729267253853795], [-2.9219166637418672, 
2.9963555171504623, 1.7194253528414558, -2.7408016558182426], [-4.5680126546763731, -4.7007698604409187, 2.2014979229380689, 
-4.1498105357523478]]], [[[-4.1492453396954367, -0.12848076787493579, -4.8304265204404251, 3.96178916111303], 
[0.17935709119953991, -3.2086809525217475, -1.4732972541736289, 0.92614205796981608], [3.9032916754246152, -3.843963100519685, 
1.0949408053995384, -3.8808899218035888]], [[-3.3464503656179256, 3.6890860890206891, 4.755066059996448, 3.7424351052036045], 
[1.0482641469612624, -4.2096411298370873, 4.9055926600486206, 0.8816617997206837], [-3.21395810789322, -3.994464711396577, 
4.7453285414421966, -4.7039389255024942]]]]),self.functionspace)
      arg1=-2.28817037433
      res=arg0/arg1
      ref=Data(numpy.array([[[[2.1414043335336665, -0.81049852446801207, -0.19492460644278189, -0.68412087428669133], 
[1.3971470420090388, 0.57010001711165104, -1.25745792916018, 1.3414518888490783], [0.099374333264972872, -1.694052087941986, 
-0.2298439601468186, 1.7561390697786874]], [[-1.5904674186038423, 1.3702326740333504, -1.468835855985593, 0.57897934814113228], 
[1.1489253437168019, -0.63335935064477478, 0.041519104719428317, 1.8564402029998095], [-0.18984259547125293, 
-1.8571411984594619, -0.20899960352258087, 0.2867222652132782]]], [[[0.95881564555479004, -0.64240461616545008, 
1.6450858724542827, -0.54611173800811663], [-0.87405926051449179, 1.4740094958414218, -1.7960112112783646, 1.3375606621120644], 
[0.56452920769424952, -2.1476603662036298, -1.1453124540486994, 0.64068061919682529]], [[0.53196333878912527, 
-0.16491072810914045, -0.30083378294336582, 0.39214417012222452], [1.2769663904952282, -1.3094984319217782, 
-0.75144113923226852, 1.1978136272413418], [1.9963603698089383, 2.0543793037371514, -0.96212150442950584, 
1.8135933330461653]]], [[[1.8133463251907833, 0.056150000592832164, 2.1110432049283672, -1.7314222776261086], 
[-0.07838450021551796, 1.4022911005768726, 0.64387567932185041, -0.40475222840100078], [-1.7058570984135981, 
1.6799287079534286, -0.47852241147987407, 1.696066851204572]], [[1.4625005214495326, -1.612242746612119, -2.0781083932161244, 
-1.6355578881686639], [-0.45812329305670407, 1.8397411211462316, -2.1438930925292059, -0.38531300361765214], 
[1.4045973778675334, 1.7457024862373944, -2.0738528016467606, 2.0557642814893033]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[-3.16757205465822, -0.50479711248921788, 1.6788221631211204, 3.7770951129415167], 
[-1.5892122638067363, -2.6055467002131838, 4.6122507540887288, -1.2953855526057891], [1.1641170473798059, 4.5241918166931256, 
-2.5313540212143879, -0.24897855837369054]], [[0.72415331148364981, 2.1882186070814456, 4.8444270870950916, 
-2.5411609923764544], [-1.8251749655190208, 3.7884384834876101, -4.3017896892676024, -1.2725668231064335], [-3.50682036387574, 
1.3786262546295811, 1.5305012504844644, -4.6290512301253504]]], [[[-4.8365859954473951, -0.91172483721318009, 
2.1491640936693726, 2.4998516523363943], [2.1513221323901144, -4.4196321614120766, -0.038664617094972265, -4.9487877542133525], 
[-1.1935599101218139, 0.62673671373246975, -0.13842966546214441, -4.309383472684206]], [[0.95881131707193035, 
2.9825648114082925, -1.2109303249637016, -3.7232314920651643], [2.2468683421102673, -1.6916110844871204, 1.8718180796281603, 
-4.0615030110076464], [-0.65272316690494581, -2.8072415950523935, -4.0443513520455685, -4.3765191556200929]]], 
[[[1.1962558413249411, 4.0308682722625733, 2.1608496470347038, 0.79128773647734629], [-4.5584375624314397, -2.6095621451567999, 
2.1335279643128331, 3.7654654752431895], [-4.7296396223634485, 4.0922129916868517, -1.8738130663782968, 2.5869805427995898]], 
[[2.6284713209892381, -3.1612446534964378, -0.38126215239507211, 1.7678262783756358], [0.37994738386609139, 0.756123754023033, 
-4.7105102581178295, 3.7176099342507314], [3.3714519810083559, -2.6681110538782415, 2.2136248569758301, 
-0.91170117491244707]]]]),self.functionspace)
      arg1=numpy.array(-2.06081324758)
      res=arg0/arg1
      ref=Data(numpy.array([[[[1.5370495402116566, 0.24495044035721025, -0.81464061097763563, -1.8328177564721171], 
[0.77115782600450944, 1.2643293628232226, -2.2380731293823324, 0.62857978719110363], [-0.56488235833383349, 
-2.1953429414339976, 1.2283277119769642, 0.12081568219041967]], [[-0.35139201105838497, -1.061822855442522, 
-2.3507356102154526, 1.2330864989139041], [0.88565762456176778, -1.8383220740318971, 2.0874233481926123, 0.61750710531363229], 
[1.7016681972469532, -0.66897194893669343, -0.74266858109541589, 2.2462254818850802]]], [[[2.3469307571280145, 
0.44241021755991999, -1.042871835277112, -1.2130413346621085], [-1.0439190134837126, 2.144605857225264, 0.018761824799183389, 
2.401376136347892], [0.57916936991848444, -0.30412106214272044, 0.067172348403989404, 2.0911081961163691]], 
[[-0.46525871191718149, -1.4472756398026854, 0.58759828256365865, 1.8066806860997542], [-1.090282365346019, 
0.82084637531897253, -0.90829097776049339, 1.9708253602177805], [0.31673086713255388, 1.3622008681990003, 1.9625025978412278, 
2.12368547259431]]], [[[-0.58047755793905842, -1.9559599963712431, -1.0485421954512042, -0.38396867712575539], 
[2.2119605295529081, 1.266277838721428, -1.0352844765608944, -1.8271745291141934], [2.2950355292564826, -1.9857272348613155, 
0.90925903576015799, -1.2553202216817936]], [[-1.2754534279479741, 1.5339791983603481, 0.18500567814302912, 
-0.85782944206601286], [-0.18436769285711191, -0.36690551893052853, 2.2857530946311897, -1.8039528514349545], 
[-1.6359813219193582, 1.2946884231303424, -1.0741511195024471, 0.44239873553904269]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[-1.0904546884462487, 1.3334257852891209, -0.22603178304485283, 0.18289397212996583], 
[-3.7181686664659419, -2.8225796632316902, 1.2180712114642667, -4.6471962445069916], [-3.3219144210330684, -2.8630553803613612, 
1.3544128565543554, 3.3768354265741571]], [[-0.87688000999399751, 0.77379878214848752, 1.6088664211586536, 1.5108386468492343], 
[-1.8270660022336238, 2.3287052166425966, 1.7593570573994599, -1.0400913774856746], [0.3402514268753718, -3.4951840496256947, 
1.8602561947704181, -4.5379996687843622]]], [[[0.2812285717584615, -4.3769139128784227, -4.928625992431285, 
2.4616622772820138], [-1.1277071206794465, -1.9674422176660755, -1.4326046643359422, 3.9225603914081972], [-4.8841013697199713, 
-1.9686191337559555, -2.6185495979380025, 4.133006569069428]], [[1.9491771263403663, -2.7172692912122933, 1.6008146426801684, 
-0.60777456069960767], [-2.8710251130186695, 3.6587482867615417, 3.4776534670955357, 2.4279061967266156], 
[-0.20493995736069781, 1.0216254351691187, 3.1304908369408029, 4.1924440263025904]]], [[[2.4631234661710639, 
-4.8181363107981792, -4.011180971149261, 4.0124761594307614], [-1.706588794554631, -1.5514906834760778, -1.0794675603812962, 
4.8457140893577382], [4.5785254798766974, 1.207713492827752, -0.59841302556737652, -1.3697809737485143]], 
[[-3.8464579860536219, -1.260600185798384, 3.9990832690121785, -4.5486212512751347], [4.9974814121590931, 2.003862757961655, 
2.2521347013428841, 1.6448200689357311], [-0.72745330552622178, 4.1349895995354764, -4.6922278226722334, 
-2.0069683572955688]]]]),self.functionspace)
      arg1=numpy.array([[[[-0.60440032943502064, -3.875914104480751, -0.58280980430530427, -2.3163677708493715], 
[-1.6465671766437637, -3.7691849817545107, -3.2108357220825856, -1.7440636117701267], [-4.6628024270662998, 
-2.4458176860565017, 0.10683632199126691, 1.5802772529840965]], [[-2.2347207138306779, -4.4056047517802188, 4.5090496642095488, 
4.014975611172213], [-1.188897015979741, 3.6920042476050394, -2.5532449990863846, -4.0606938164677562], [4.4197042202445953, 
-4.5619348496559375, 0.82776640282486014, 4.1566732111698705]]], [[[1.5794789714458668, -3.0570641564181464, 
3.8501898573628814, -0.7953567867700384], [-3.7392729570887018, -4.0581342793228199, -3.9245962674980071, -1.9180517504817018], 
[-4.9091654496139983, -2.1942628059641658, -0.30462112262014696, -1.4571673345824845]], [[-2.048752511236116, 
-1.6918563022326905, -1.7606950089245386, 1.7532539023144835], [2.3911474287014753, 0.51197130208550679, -0.32269550477930231, 
2.6320236454679682], [-0.95441173678796432, -2.8880590488837141, -2.3748871562551175, 1.0995331217286575]]], 
[[[3.1458049432219291, 4.323466533283268, -0.56637031859309594, 2.8287450298684345], [4.0619469469584484, -2.764332094025046, 
4.7192476920935214, 0.59654188476619385], [2.3781695750773881, 2.5854483322542468, -2.3731765570370698, -1.537712244261634]], 
[[2.4750970686958755, 4.7571761939897321, 2.7621753185494011, 3.7118252942974852], [-1.989099631041189, -1.751821460678201, 
2.9677164677803027, 1.2221562891457172], [3.8582858774463862, -0.54255471478740525, -1.6536059202887965, 3.7165005964984879]]]])
      res=arg0/arg1
      ref=Data(numpy.array([[[[1.8041927433520435, -0.34402872441048521, 0.38783112668167524, -0.078957225373111548], 
[2.2581336001393955, 0.74885676264098155, -0.37936266968968796, 2.6645795561265957], [0.71242873207542723, 1.1705923122085153, 
12.677456798494699, 2.1368626424240134]], [[0.39238908225398839, -0.17563962855174661, 0.35680831682316216, 
0.37630082799136344], [1.5367739826716476, 0.63074283247458363, -0.68906707269729395, 0.25613637089988989], 
[0.076985112559532679, 0.76616264037381032, 2.2473202444820819, -1.0917383778425944]]], [[[0.17805148206628085, 
1.4317376701726461, -1.2800994691225589, -3.0950415187615095], [0.30158459508595198, 0.48481446946954704, 0.3650323668195431, 
-2.0450753690160242], [0.99489443161953373, 0.89716652372045214, 8.596086756607658, -2.8363294118541571]], 
[[-0.95139706511663003, 1.6060875191506496, -0.90919474103466236, -0.34665518776104243], [-1.200689291909447, 
7.1463933073156447, -10.776888477185233, 0.92244847454436107], [0.21472908333086441, -0.35374118668522203, -1.3181640351607999, 
3.8129310917995256]]], [[[0.78298671107317175, -1.1144150819040239, 7.0822584437569382, 1.418465120420338], 
[-0.42014059189830388, 0.56125336273074455, -0.22873721211747416, 8.1230073077885336], [1.9252308699339522, 
0.46711956211275329, 0.25215697660291236, 0.89079148511706385]], [[-1.5540634889444211, -0.26498917307099951, 
1.4478021152952589, -1.225440555691893], [-2.5124339345149718, -1.1438738495564946, 0.75887798777063209, 1.345834475953525], 
[-0.18854313252902041, -7.621331981523249, 2.8375731878443879, -0.54001561554609734]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[1.1477780587460726, -0.30048623118843398, 4.9271179045017472, -4.5738511373597976], 
[-0.62075020987274332, -4.7370346225608202, -3.910927137656961, -0.2928405904964233], [0.70813720290879623, 2.5684616881624498, 
-2.0182146484975272, 3.1945936071555963]], [[-4.8252297165345777, -2.4849710753779575, -2.9836045084236664, 
-3.267831134175494], [2.328833954176206, 2.3753653198546809, 3.7196084190795382, 4.9282586786384126], [4.9064344698572242, 
0.42034604823578725, 1.3590767941136948, -3.8361270324300834]]], [[[-0.58082301606929576, 4.209657064614289, 
-4.8777725238941008, -0.081999525262073547], [3.0991677864127016, 2.4825428269650178, -4.1249866067694789, 
-1.8546788009945612], [3.0090037191554053, 1.8912852666170741, -1.5822853918527668, 1.8366209311741386]], [[2.2468956977970036, 
-1.4835814386127422, 2.6411282262893003, 3.8260660649513714], [-4.2385065844551484, -3.9694000646480032, 2.2991693765568408, 
3.3167146728641637], [-0.69539213078418527, 3.3055535688576558, -2.8589174012095486, -0.092102681457321367]]], 
[[[1.7850397354272296, 2.2300043238192346, 2.2951969214382224, 4.2948598150342239], [2.859726116774949, 3.6847644001572242, 
2.4615450236096512, -3.2402937486518049], [-2.8921263782820636, 0.98100923455332723, 1.1629982675505612, -1.1066905548506298]], 
[[4.7889075216040169, 0.46758725898206865, -1.4759259159368576, 2.0305205089619953], [3.3735874148449128, 2.2326506941124986, 
-2.3679862103803484, -3.4301879259368673], [-4.5678038655756694, 0.26833562401225119, -4.0723463323392384, 
4.8072943763492209]]]]),self.functionspace)
      arg1=Data(1.25152854037,self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[[0.91710098628890635, -0.24009538855465778, 3.9368801793587549, -3.6546119323747353], 
[-0.49599365084159086, -3.7849992786770463, -3.1249204564579949, -0.23398634633519433], [0.56581786196977746, 
2.0522597809827774, -1.6125997797012113, 2.5525535408085331]], [[-3.8554691809833685, -1.9855488670155128, -2.3839684131622931, 
-2.6110720041573505], [1.8607917271143779, 1.8979713552070421, 2.9720524135764559, 3.9377916840507088], [3.9203536408282509, 
0.33586613063577503, 1.0859335207067089, -3.065153457293516]]], [[[-0.46409090750382942, 3.3636125176617795, 
-3.8974520888152626, -0.06551950084777404], [2.4763061220206661, 1.9836086408563769, -3.2959588804387074, -1.4819308878403479], 
[2.4042629649142979, 1.5111802932207175, -1.2642823082399914, 1.4675022358057561]], [[1.7953211815088492, -1.1854155864227667, 
2.1103220111138716, 3.0571145135905606], [-3.3866639455039516, -3.1716416658469155, 1.8370890494187571, 2.650131072418028], 
[-0.55563425711103542, 2.6412130944043835, -2.2843405555532148, -0.07359215430258495]]], [[[1.4262876777010896, 
1.7818245863990678, 1.8339149666947385, 3.4316914688572884], [2.284986737832972, 2.9442112435206051, 1.9668309145177627, 
-2.5890690017204654], [-2.3108752896812761, 0.78384887192413411, 0.92926228210716677, -0.88427112858294354]], 
[[3.8264469144038986, 0.37361294121362187, -1.1792986482716044, 1.6224324443728351], [2.6955736972931552, 1.7839390969418896, 
-1.8920752775417857, -2.7407988034461659], [-3.6497800235615974, 0.2144063162410812, -3.253898094182492, 
3.8411384329366709]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[-3.5878440070279183, -0.1700996809165245, -4.1412468054894536, -2.0146265772801897], 
[0.55744045313737622, -0.52187183646786295, 1.0643260641164831, 2.2673056698704137], [-1.9080389547964294, -4.9003327611729315, 
-3.0538031853017733, 2.1559834164422034]], [[1.8811125433971316, -3.6916344922332978, -4.2062395558468682, 3.7495181480655067], 
[2.5137081712456357, 1.4133284971486884, 3.8076582726700163, 2.6357723857970363], [4.269562544289716, 4.1635604068507739, 
3.7304106982614673, -3.5140079429955429]]], [[[1.8373054437788019, 3.0361519630331628, 2.9877077445004421, 3.0341858153696286], 
[-4.9042733402549432, -3.7608695768218303, 4.0341909392906281, 4.9801935191941098], [-3.6066047852790186, -4.6680763845436903, 
-4.8384904751126481, -1.9831234078462545]], [[-3.7070713376869637, -1.2906402026465114, -2.3043901476311057, 
-2.2864913724053171], [4.8747446391483269, -0.35138766428814527, -4.383573712165699, -1.2543890355891607], 
[-1.9360914310397925, -4.7050922462630069, -2.6840952778213301, -0.15611556547080241]]], [[[-2.6025641931532126, 
-3.4694171310094135, 4.2230178219007328, -2.168997481514181], [-4.8479803382062645, -1.7287262092540026, 0.10648488115811183, 
4.772773926420804], [-0.45440587882830119, 4.1337671129268099, -0.6610150140630191, -2.4754686490143141]], 
[[-2.9322114375478812, 3.3192757415210448, -3.6933238643237463, 3.520882954677603], [-3.3966174668671423, -4.8857477036463193, 
1.898625940301697, 2.5405227607451994], [2.7447507748804529, -1.1964276678737527, 0.081547685768406097, 
-3.2270714621528906]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[-1.4192236603533583, 1.334627778091539, -0.44475276835635658, 2.6354943100394488], 
[4.8757948458992537, -0.30154755289694268, 0.018393073492706691, -4.7503904846814313], [0.036159739130172674, 
-3.7073317173047835, 2.4133732168704487, 4.7030266195492985]], [[2.2052107720673995, -4.7574153968443564, -0.73430661365261329, 
4.3961081848547607], [-3.6100471425175451, -1.153028380090154, 0.4238877374824277, -1.4334288568505382], [-3.6688773031429012, 
-2.2503468718797546, -3.5254650928078468, 3.0765751150677261]]], [[[2.1998266217312974, 1.1748827369587067, 
-4.8630189420301884, -4.6282652208324171], [-4.6622508113493568, -1.0980110611292626, -3.8798174063293422, 2.9619355569286832], 
[-1.3838634843552611, -1.5458925504765042, 4.7102488188535361, 4.6423848950616016]], [[-1.6023691345389235, 2.823379237552766, 
1.5564304425869313, 3.6952029309612282], [-4.4137044453792731, -1.6529088178640428, 2.9544637916450798, 2.3508449135824456], 
[-2.6989124137760303, -2.1745251170743285, -1.6786743431844675, -0.79707132252102753]]], [[[-2.5880751009677572, 
-4.3938522846248231, -0.96036936099589543, -3.327649852368042], [2.6342745476495244, 2.2689105136464889, 4.2433705235112669, 
-4.3448259206320907], [1.2148929711476715, -1.6987181147931962, -3.1226858533780053, 1.3917325568999797]], 
[[2.0186152568464655, -3.5046259094065322, 4.6416937208399904, -2.0802257070892285], [-2.3193800420129715, 
-0.67871525107829633, -2.2781158494394047, -0.40136884218995128], [1.7033440478745678, 3.4923796518388617, 3.1365103869960596, 
1.7243239074935914]]]]),self.functionspace)
      res=arg0/arg1
      ref=Data(numpy.array([[[[2.5280328303817994, -0.12745102695206884, 9.3113457636115129, -0.76442076524537605], 
[0.11432811895402178, 1.7306452380537758, 57.865590790930881, -0.47728827286552272], [-52.766944693035953, 1.3217950630906736, 
-1.265367148335973, 0.45842466795325432]], [[0.85303072487423792, 0.77597480654768924, 5.7281787711594294, 
0.85291762404372762], [-0.69630895996905084, -1.2257534346536916, 8.9827044662452948, -1.8387884220415585], 
[-1.1637245379212449, -1.8501860574822753, -1.0581329271623541, -1.1421817480696184]]], [[[0.83520465914391662, 
2.5842170180255817, -0.61437304277765137, -0.65557734282650182], [1.0519110915947356, 3.4251654741564432, -1.0397888655041982, 
1.6813983368220937], [2.6061853832058643, 3.0196641953574379, -1.02722608957425, -0.42717772280274052]], [[2.3134939745039844, 
-0.45712605146349583, -1.4805609583175436, -0.61877288341794401], [-1.1044565170764251, 0.21258744613766595, 
-1.4837121120123371, -0.53359072235760585], [0.71735985990409368, 2.1637332258518032, 1.5989374524719104, 
0.19586147570462106]]], [[[1.0055984048453763, 0.78960713885393075, -4.3972850378332531, 0.65181061041343225], 
[-1.8403474089410901, -0.761919079160012, 0.025094410343878871, -1.098496007344399], [-0.37402955619953759, 
-2.4334626663059158, 0.21168156039390182, -1.7786956529408975]], [[-1.4525855918322246, -0.94711270969377892, 
-0.79568452518564259, -1.6925485261905666], [1.4644505882353134, 7.1985235279215809, -0.8334193982140583, -6.3296461849992705], 
[1.6113895359574315, -0.34258236135461134, 0.025999494886579041, -1.8714995762273188]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[4.412518352066181, -3.1098643012240967, -4.7558876935081082, -3.2711898204883525], 
[-2.2457923255657777, 2.0059445983496635, 2.0215102596882897, 0.79758981166266008], [-4.4412761328213834, 1.0033814164110115, 
-1.2982782934866943, -4.1204179174790845]], [[1.7840922235302417, -0.95709853273933554, -1.7382828390973151, 
3.8554075221477841], [-2.4258705986317142, -3.627825487593034, 2.6559428396800264, -0.19799215564643458], [2.1661067884872223, 
-1.4262141977830045, -0.26406212935679463, 2.5100575370537719]]], [[[4.1820747068718607, 4.9050722386866283, 
-4.0633395532013079, -3.9405371548508885], [2.973051377771049, -1.946374689712369, 4.6134680551383607, 2.8686358221337338], 
[-3.3582201017310731, 1.9546297287973102, -0.97203540600203375, 3.1269645246978808]], [[-0.5878623537812846, 
-0.95965858065134135, -2.5752176173435903, 4.6948697605076344], [-4.0551769138295892, -2.8062824354588445, -4.9919212689190449, 
2.1747373667203309], [-1.7481317881152125, 0.6577549434941421, -1.2016837202981021, 0.23394398668818361]]], 
[[[1.9087213429605656, 3.3707427949854321, -3.3020394873909318, 4.3600839601717176], [0.50010396094649412, 4.4875982311177438, 
-3.1575441040879504, 1.5273508651815515], [4.5557285198427362, 2.392652489014413, -3.8600978932523233, -2.6728652249183673]], 
[[-0.42126710644162557, -3.6243525224129005, 1.7204672019618918, -4.4918600764859029], [1.2848888405605674, 
-1.9434977296685707, 2.4648856620122537, -2.7689365425916099], [-0.92689824733479309, -0.33131118170618734, 4.6548439908136121, 
-4.6590430970540506]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(4.5163558224)+(1.-msk_arg1)*(3.30550390167)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.97700857186320367, -0.68857823066154211, -1.0530365366526755, -0.72429851613245821], 
[-0.49725761518361483, 0.44415114247686149, 0.44759765155425402, 0.17660030410078381], [-0.98337604641261578, 
0.22216615693456895, -0.28746147215602741, -0.91233243781314854]], [[0.39502915485152079, -0.21191831874561251, 
-0.3848861576576873, 0.85365451123856617], [-0.53713008762542547, -0.80326387695192514, 0.58807209708964059, 
-0.04383891868408004], [0.47961384657632583, -0.31578871414654203, -0.058467963938378421, 0.55577054505009671]]], 
[[[0.92598432703919553, 1.0860685985727527, -0.89969429181173755, -0.87250369762927049], [0.65828546170507951, 
-0.43096132507100476, 1.02150234316343, 0.63516603539249605], [-0.74356853928057454, 0.43278913479405706, -0.21522560316917916, 
0.69236451857733972]], [[-0.13016298469354895, -0.21248515803212539, -0.57019812402132919, 1.0395261013814154], 
[-0.89788694099778266, -0.62135990737094526, -1.1052984895839357, 0.48152480722081464], [-0.38706688685733442, 
0.145638423844347, -0.26607374785193894, 0.051799281519855862]]], [[[0.42262421696161334, 0.74634128211723449, 
-0.73112917078318185, 0.96539868239558535], [0.11073174493170634, 0.99363256740386374, -0.69913537113913804, 
0.33818213737883052], [1.0087178023596943, 0.52977501842282426, -0.85469304125900414, -0.59181900851642]], 
[[-0.093275889457717934, -0.80249490184971761, 0.3809414646715692, -0.99457621434749566], [0.28449681360106427, 
-0.43032431590765868, 0.54576870356154372, -0.61309087491738334], [-0.2052314484917728, -0.073358077781025294, 
1.0306636974283143, -1.0315934528341251]]]])+(1.-msk_ref)*numpy.array([[[[1.3349003611333146, -0.94081398592614518, 
-1.4387784237997974, -0.98961910734281511], [-0.67940997571701855, 0.60684986556407761, 0.61155887871355974, 
0.24129144462961505], [-1.343600329915889, 0.30354870127493733, -0.39276259599360369, -1.2465324622360943]], 
[[0.53973381263569586, -0.28954693783781049, -0.52587529490416629, 1.1663599973972376], [-0.73388828777557802, 
-1.0975105749416154, 0.80349106178292717, -0.05989772256703009], [0.65530304998073086, -0.43146649957444488, 
-0.079885589977197274, 0.75935700326538225]]], [[[1.2651852278134199, 1.483910588098007, -1.2292647880854304, 
-1.1921139021680378], [0.89942455559331502, -0.58882843512280425, 1.3956928179114436, 0.86783616279650755], 
[-1.0159480072115357, 0.59132579689592812, -0.29406572641196832, 0.94598724361467712]], [[-0.17784349111927439, 
-0.29032141821604052, -0.77906960449901208, 1.4203189287226154], [-1.2267953796035356, -0.84897265861386328, 
-1.510184654871473, 0.65791402201097104], [-0.5288548554525363, 0.19898779824822899, -0.36354025166667497, 
0.070774076705832065]]], [[[0.57743732869186459, 1.0197364442023478, -0.99895192552130774, 1.3190376081448878], 
[0.15129431875542784, 1.357613956786202, -0.95523835336972085, 0.46206294429425493], [1.3782251225121243, 0.72383895472214443, 
-1.1677789553667532, -0.8086105188284306]], [[-0.1274441413391946, -1.0964599135951323, 0.52048560617125483, 
-1.3589032746920156], [0.3887119418953392, -0.58795808066872379, 0.74569134853155972, -0.83767456489557346], 
[-0.28041057427481803, -0.10023015902018878, 1.4082101032954826, -1.4094804410005199]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[2.9081699054784629, -0.52767602583237405, -4.8676204514975998, 3.0656135751128506], 
[3.4195689117030028, 3.0074043214835662, -1.7270820460100356, -2.8987878505521438], [-2.3130559970750708, 4.8207960200663553, 
0.55243953125140433, 0.88076242650078562]], [[-2.7536605772035183, -3.7711545932324917, 2.9067630057185028, 
-4.1129896972084978], [-1.2784492727347119, 0.69695621752376535, 2.9477627539737341, 1.4610988782668102], [0.2943489197851159, 
3.5737592084072247, -4.9298691454020052, 0.94541500860994532]]], [[[-3.8340550369717041, 0.97696064003551886, 
2.4224341317795286, 1.9967042904317545], [1.2044795557381676, -4.056325871650575, 0.51617607026094703, -2.849666753344493], 
[-3.7868839846612734, 0.88475821351153261, 2.6016189509013348, -4.8073704855795221]], [[0.44234464671708729, 
-4.8880779253744722, 1.8546890727264795, 2.9180066219843015], [-1.0126086589542602, -0.29422835104697853, -0.42280031697502363, 
0.4309828152431372], [-2.2614242309620272, 2.0773912025446695, 0.18146231086733433, 1.4055222090281303]]], 
[[[-2.548696220235672, -2.4788729149718458, 1.072945903211231, 3.683370286122857], [1.8564047155739649, -4.7544085613107443, 
0.53982777126928561, 1.9397243984667289], [-2.8351844181475396, 1.121983175603698, 4.2013657561163296, -1.7204618771743876]], 
[[0.5438114487557888, 4.4370545390089866, 0.69153496717829643, -1.9573982644538046], [3.3766998595686069, 2.7245238536969456, 
-3.2964330075690218, -3.1677351247888486], [3.3857102320850547, -1.793879557866239, 0.25924077080415575, 
2.2426857570134198]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[4.3111723034689877, -0.51272641378774164, -0.19422078974034385, 0.19456255846028814], 
[1.4081517012501514, -3.9568678810431468, -2.1295217591713964, -2.0694582579589316], [-3.0991052505594805, 2.2122995383785771, 
2.4794941183633235, 0.66114960804785294]], [[-4.8969028341427894, 4.1182601392623504, -1.8254876882020152, 
-0.47086880030218747], [-0.94885734694597712, -2.9079017397530835, -1.5779513714845086, -0.074632193936196245], 
[-2.8604679121870591, 4.1759406093784541, -4.0059464567021283, 0.48354992486232007]]], [[[-3.5447370220150241, 
0.92669668189849563, -2.4237548316909239, 4.1324406056163454], [-2.8758659810779608, 1.1119323922745759, -0.91587219847868795, 
-1.5859956755970783], [1.3218766489656906, -3.089764040029499, 0.19065669827063036, 3.6524960462605129]], 
[[-4.8849856289871774, 4.7047194997272221, 4.0020345768083629, -1.140574999979421], [0.91393014960572039, 2.4978706097746093, 
1.4713661486195653, -2.7031516471614081], [0.93060178482629841, 0.78398582765172886, 2.3775120381350217, -1.565537352291233]]], 
[[[-4.6086028886293775, -0.57050783749372513, 1.306045318608037, 4.4571163176042834], [2.7802845840789034, 4.4462864530704174, 
-0.86316101288559466, 1.1460856455585366], [3.955680981374444, 2.589442849252551, -3.020080734817939, 0.90711684678397386]], 
[[0.25197096379710082, 4.1087769867077171, -2.4469877326495251, -2.3335119633375259], [2.8684943015556232, -3.3580128714258595, 
2.9744564999117573, -3.1389790237754136], [-2.4987563810487377, 1.7416315503173498, 0.056015176878369211, 
-1.4899741977944538]]]])+(1.-msk_arg1)*numpy.array([[[[-0.54239286579831969, -0.36223024249238556, -2.1946334214605745, 
-0.83400107629310938], [0.24057761946722511, 3.3191490451625896, 4.4537215458189969, 4.8945184143326337], [-2.6707783510318537, 
-0.32239193277991962, -1.0278479997665926, -1.6540744775933671]], [[-2.7352300996019139, -4.489579010319992, 
-1.2170699275684393, 2.2344251911937896], [2.3565588572995066, -0.05461916588921234, -2.5208522829137547, 4.8612708202340009], 
[-0.82441411131140185, -0.83511915144139337, -1.0749313746768063, 0.6632730486836893]]], [[[4.1429094618922484, 
3.0055637804292985, -3.5021203571723367, 1.6596153829707454], [-0.053993613093562765, -2.037801710975474, -1.4036493113720647, 
2.7096340147789419], [4.0858348715598627, -2.3143687330034144, 3.0408775659994305, -0.46825136507924547]], 
[[2.6517610085705412, 0.067569675318274847, -4.0884250728844975, 2.6783965222804706], [3.0792598195235321, -3.5363607583123393, 
-4.2240289065704131, 4.7714192116398184], [-1.1536339089833483, 0.95014194052652279, 3.8824165323332647, 2.0408532146055176]]], 
[[[-4.2043991537907068, 0.26240477542094531, -3.4172363277466653, -1.8882732134368609], [4.9124496572578504, 
3.0561992411513756, 3.6712710554851249, 2.2055477640088386], [1.4835890212755825, -0.79785588894802295, -2.2071301868338509, 
2.7130390219950842]], [[-0.70164687426071559, -3.1926096645916999, -2.4799121778723632, -3.0213532603186333], 
[-1.4051653504429487, -4.9463275725930034, -4.2735750605806864, -2.7012201386196857], [-3.0110725118480963, 1.3371871335969177, 
1.2026768066127405, 0.41108097305897484]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.67456591868026294, 1.0291570936129326, 25.06230387594027, 15.756441523863741], 
[2.4284094594830394, -0.76004668639346273, 0.81101873628285848, 1.4007471952641193], [0.74636251758713112, 2.1790882909099998, 
0.2228033239361186, 1.3321681141147064]], [[0.56232697900479189, -0.91571548802353431, -1.5923213421293858, 
8.7348953563475042], [1.3473566673110242, -0.23967667407597668, -1.8680948014263148, -19.577327172184127], 
[-0.10290236731236825, 0.85579742211399457, 1.2306378027479905, 1.9551549074878483]]], [[[1.0816190349692616, 
1.0542399245824956, -0.9994551016901021, 0.48317797664606721], [-0.41882325659928443, -3.6479968564931626, 
-0.56358962649848166, 1.7967682996813226], [-2.8647786369661201, -0.28635138542912342, 13.645568052418644, 
-1.316187731538105]], [[-0.090551882914914589, -1.0389732959973237, 0.46343654387054317, -2.5583645284500802], 
[-1.1079716096368095, -0.11779167019124649, -0.28735221166512132, -0.1594371576214432], [-2.4300665094728284, 
2.6497815767500223, 0.076324455126493398, -0.89778899683938329]]], [[[0.55303012253973294, 4.3450286780663383, 
0.82152272047861263, 0.82640210029400407], [0.66770312873888193, -1.069298753351249, -0.62540796353233297, 1.6924777009325669], 
[-0.71673737884757938, 0.43329134525118451, -1.3911435239725809, -1.8966265297287643]], [[2.1582306173726113, 
1.0798966586318213, -0.28260663425129762, 0.83882075395671796], [1.1771680556371951, -0.81135003289611407, -1.1082471731110597, 
1.009160972658826], [-1.3549581134692528, -1.0299994608729779, 4.6280452058032155, 
-1.5051842913341542]]]])+(1.-msk_ref)*numpy.array([[[[-5.3617407028355357, 1.4567420494810461, 2.2179651525848434, 
-3.6757909099333608], [14.213994299535685, 0.90607691325782191, -0.38778402022716524, -0.59225190410227369], 
[0.86606063591215754, -14.95321541856714, -0.53747201081955143, -0.53248051307959887]], [[1.0067381817728194, 
0.83997955811979463, -2.3883286735429152, -1.8407372569099281], [-0.54250682887663915, -12.760286726777331, 
-1.1693516410912146, 0.30055903739929452], [-0.35704012794843215, -4.2793405015787407, 4.5862175591295227, 
1.4253782970470246]]], [[[-0.92544987338934581, 0.32505070975268907, -0.69170499146849351, 1.2031126675010766], 
[-22.307815438299833, 1.9905400264429334, -0.3677386267916064, -1.0516795765781584], [-0.92683236197833474, 
-0.3822892181762928, 0.85554873369137874, 10.266644892249138]], [[0.16681165658874278, -72.341296629738963, 
-0.45364389457135013, 1.0894602788312386], [-0.32884807333697025, 0.083200886774739968, 0.10009408702610063, 
0.090325916907858336], [1.9602615815574722, 2.1864009091036225, 0.046739526621137339, 0.68869343418204021]]], 
[[[0.60619749148644819, -9.4467522970772144, -0.31398059727368471, -1.9506553712207386], [0.37789796233967266, 
-1.5556605398277616, 0.14704111004354928, 0.87947512636999303], [-1.9110308700652572, -1.4062479091092985, -1.9035423380001106, 
-0.63414564376933058]], [[-0.77505005538401528, -1.3897892336225943, -0.2788546196710876, 0.64785481729712613], 
[-2.4030622862314202, -0.55081751333922957, 0.77135254695189748, 1.1727052821424433], [-1.1244200260082804, 
-1.3415321706251078, 0.21555314726180694, 5.4555815131139083]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.51258209259)+(1.-msk_arg0)*(0.157852552457)
      arg1=-2.40412038934
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.629162374436)+(1.-msk_ref)*(-0.0656591712949)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.623891836675)+(1.-msk_arg0)*(-4.57535280742)
      arg1=numpy.array(-2.22578280867)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.280302208393)+(1.-msk_ref)*(2.05561512543)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.28867426406)+(1.-msk_arg0)*(0.40151348993)
      arg1=numpy.array([-1.005881815433102, -4.2992828989978946])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([2.2752914198719063, 0.5323386057227294])+(1.-msk_ref)*numpy.array([-0.39916567112548246, 
-0.093390804783714101])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.75989379122)+(1.-msk_arg0)*(-4.32413720595)
      arg1=numpy.array([[-2.2199560634475954, 4.5148770728023351, -2.7347978982286349, -4.7481396347998501, 
2.9486390257995385], [-1.8318806519791808, -1.9147910769823162, -3.4650377663122645, -4.6139600747695191, 3.0033523449265154], 
[-3.4506169616679063, 3.1254177230085674, -1.6437502834473205, 4.6901813032166846, 3.1245689691919214], [3.2471879107702115, 
1.3189272753338894, -2.1793619521676364, -1.7765589665564949, -1.0584302238348853]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[2.1441387375140795, -1.0542687462945339, 1.7404919735751956, 1.0024755288010603, 
-1.6142680570832451], [2.598364574719767, 2.4858554274854776, 1.3736917494792749, 1.0316287341205534, -1.5848602643169782], 
[1.3794326765601039, -1.5229624367251999, 2.8957523774442593, -1.0148634953520363, -1.5233761322440136], [-1.4658510446622839, 
-3.6089130009175641, 2.1840767599356177, 2.6792771198826899, 
4.4971257282991148]])+(1.-msk_ref)*numpy.array([[1.9478481025592125, -0.95775303208139673, 1.5811542084159194, 
0.91070135643420147, -1.4664857814457994], [2.3604906800447831, 2.2582814688932, 1.2479336438959701, 0.93718565741321913, 
-1.4397701998754389], [1.2531490032034973, -1.3835389663653495, 2.6306533598780661, -0.92195523507493793, -1.3839147890755406], 
[-1.331655981967967, -3.2785258799478174, 1.9841298971244061, 2.4339958804366031, 4.085424913778521]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.90749636577)+(1.-msk_arg0)*(-0.695987638783)
      arg1=numpy.array([[[-3.8683229421324006, -4.3724709414995626], [2.574406810861932, -1.7769580091610129]], 
[[4.2532182277437478, -4.8639669587055021], [3.8702275365114005, 0.24072631795984289]], [[1.5252558189640686, 
3.3664193859246776], [1.2634283615539674, -4.6659891986663036]], [[-1.2387373780826061, -2.5105056344199053], 
[-2.5926495789718809, 2.2950980695351273]], [[1.051826718104123, 2.9257868139581653], [-4.9232592241212885, 
-3.6766442871517988]], [[4.3459387050977121, -0.053718956670220308], [2.8519225074631986, -3.4135551887992444]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.2686366777508078, 1.1223622595627876], [-1.9062629670914402, 2.7617401989657275]], 
[[-1.1538313114907846, 1.0089493632333724], [-1.2680123634791938, -20.386206241864794]], [[-3.2174906692748619, 
-1.4577792613400433], [-3.8842695914614822, 1.0517590497587963]], [[3.9616923268838495, 1.9547840476797171], 
[1.8928498496584925, -2.1382512716626936]], [[-4.6656890163616032, -1.6773253411224953], [0.99679828795729863, 
1.3347759485255477]], [[-1.1292143536257584, 91.355020089111349], [-1.7207677813585913, 
1.4376496334018096]]])+(1.-msk_ref)*numpy.array([[[0.17991973503627323, 0.15917490318289129], [-0.27034874047360691, 
0.39167365531153303]], [[-0.16363788583507666, 0.14309053591277454], [-0.1798311939588331, -2.8911987882409296]], 
[[-0.45630879104326449, -0.20674418692251761], [-0.55087226150848867, 0.14916186239396678]], [[0.56185245645892345, 
0.27723006443043424], [0.26844647438208502, -0.30324962929543614]], [[-0.66169420000823831, -0.23788050293439805], 
[0.14136725431259808, 0.18929969407575245]], [[-0.16014667624437887, 12.956090027136115], [-0.2440415673854629, 
0.20388937640935573]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.21838220841)+(1.-msk_arg0)*(-2.06441878469)
      arg1=numpy.array([[[[0.68572386050981748, 4.4522487615469366, 0.50370622039161361, 4.6237429183096133], 
[3.6077194751918178, 2.3866732928339918, -4.7951839032112433, -3.8410389007948553], [-1.7379636204151891, -3.4748138199186784, 
-0.91755527149451765, 2.3325639120760506]], [[-2.1905580146630541, 2.6640861923839925, -2.571764756677025, 
-0.17914640273705373], [-3.6393714994207151, 2.5839423755141819, 0.32653082441321857, 3.3013812723933249], [2.2877332157853827, 
-4.7079721092727969, 0.076638987671123537, 0.67594190662723186]]], [[[-1.2230426561840115, -0.29512381002348409, 
0.25716636179938934, -1.7725809751060151], [-2.1528567448026079, -0.01364185360953396, -2.5881565610916746, 
3.0724970422712197], [4.7910105105906169, 2.8685090530487276, 2.1030344803323606, -0.76547439901606573]], [[2.4235029056926018, 
-3.2225789307144947, -3.6636296798516534, -4.6472969483344233], [0.83056294531959285, 3.1840754470489081, -2.3528911482118531, 
-0.74113583450782894], [0.017823643268661371, -1.7054164729100076, 1.4084960380582405, 4.466265989649802]]], 
[[[0.67152109386268055, 2.0285998041341742, -4.6668082478655926, -4.9007888691443071], [4.9987349495229125, 4.2689391436524584, 
4.8360684199074537, -3.8553463464419324], [-3.0023605526396557, -1.3977196274310835, -3.1648209588333254, 4.6379929354807476]], 
[[-2.9282629210649267, -0.85219636114336783, -2.0018266922599324, 1.8587851382376028], [3.2626073191814715, 
-0.42500075903174483, -0.27027270963002614, 2.7074728517624624], [-2.2593027117522215, 4.1692303175051286, -2.1201106016348579, 
1.1038704472054022]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-4.6934085187288312, -0.72286666374278896, -6.3894033429053563, -0.69605561236285196], 
[-0.89208216729247802, -1.3484804217131938, 0.67116971389921221, 0.83789367708470586], [1.8518121844481104, 
0.92620277666782114, 3.5075622236579651, -1.3797616398634682]], [[1.4692065614648608, -1.2080623433329634, 1.2514294707774387, 
17.965095359108343], [0.88432362811124765, -1.2455317266012824, -9.8562891089865268, -0.97485929157138618], 
[-1.4067996155345071, 0.68360264965747985, -41.994059501715441, -4.7613296007511341]]], [[[2.631455405205573, 10.9051933429441, 
-12.514786871398826, 1.8156474957201389], [1.4949356087822214, 235.91971447070424, 1.2435036801077044, -1.0474809785441936], 
[-0.67175436190300664, -1.1219703856230874, -1.5303516126391659, 4.2044282768297396]], [[-1.3279877654995553, 
0.99869771310731981, 0.87846821039584866, 0.69252777349763639], [-3.8749407574088135, -1.0107744813004624, 1.3678415216353292, 
4.3424997936439693], [-180.56814535057242, 1.8871532317972559, -2.2849778213428809, -0.72059797062484177]]], 
[[[-4.7926747764538957, -1.5865042488190688, 0.68963240773484846, 0.65670696990754829], [-0.643839339535209, 
-0.75390679044882913, -0.66549559041887729, 0.83478419815190319], [1.0719506041946953, 2.3025949877570029, 1.0169239430210351, 
-0.69391701392895699]], [[1.0990755595272719, 3.7765735165712107, 1.6077226969029239, -1.7314439104370556], 
[-0.98644485638554602, 7.5726504953653633, 11.907906694752912, -1.1887034089067054], [1.4245024323970976, -0.77193677569203734, 
1.5180256190081036, -2.9155434105153999]]]])+(1.-msk_ref)*numpy.array([[[[-3.0105686903700222, -0.46368001772956491, 
-4.0984579922112241, -0.44648217281190117], [-0.57222264615813967, -0.86497753625892282, 0.43051920976545133, 
0.53746364929116208], [1.1878377432304017, 0.59410917870092694, 2.2499121838491356, -0.88504275231332175]], 
[[0.94241685035128386, -0.77490690451090272, 0.80272458020536475, 11.523640738243655], [0.56724596129279847, 
-0.79894149507870338, -6.3222784201164242, -0.62531971146547505], [-0.9023861569373588, 0.43849426818490461, 
-26.93692658819344, -3.0541364049928652]]], [[[1.6879368632420921, 6.9950939726830024, -8.0275614985016244, 
1.1646400439150868], [0.95892064795969512, 151.32978580328296, 0.79764061252142837, -0.6719026109019387], 
[-0.43089422995985527, -0.71968355215626656, -0.98163810626835701, 2.696914210774525]], [[-0.85183260141399997, 
0.64061077449941739, 0.5634900263102216, 0.44421925425497033], [-2.4855657194002894, -0.64835737061566145, 0.87739663870954288, 
2.7854796497074648], [-115.82473647911846, 1.2105071209778642, -1.4656901609297635, -0.46222477332846607]]], 
[[[-3.07424264637124, -1.0176569969509737, 0.44236203311646621, 0.42124213872754956], [-0.41298824713391702, 
-0.48359058661216292, -0.426879565266784, 0.53546908609021648], [0.68759855736696696, 1.4769906239957311, 0.65230191898487078, 
-0.4451103771412992]], [[0.7049977547575359, 2.422468434294585, 1.0312674881759729, -1.1106279807293298], 
[-0.63275122707946407, 4.8574472887854725, 7.6382805630523372, -0.76248919110921398], [0.91374156015129082, 
-0.49515585071487894, 0.97373164546161128, -1.8701640123774246]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.47744969089)+(1.-msk_arg0)*(0.169407601567)
      arg1=Data(-2.14770628714,self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(1.15353282044)+(1.-msk_ref)*(-0.0788783841539)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.98800864656)+(1.-msk_arg0)*(0.118510821597)
      arg1=Data(numpy.array([2.9320942484300829, -3.244701255225606]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-1.3601229389874594, 1.229083460346386])+(1.-msk_ref)*numpy.array([0.040418489842332039, 
-0.036524416972462191])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.2641791644)+(1.-msk_arg0)*(-1.75488092016)
      arg1=Data(numpy.array([[-0.51534423126757911, -0.71314638845691913, 4.6263192662707979, 0.88193037150846365, 
-4.7246223038144519], [-3.0202439837234261, 2.1088255222134498, 2.7580495812572758, -4.6725986847161991, -0.45185454512724466], 
[2.9862734207023633, 4.8337758724530193, 1.0906631641560027, 2.1131951532361732, 2.8211709039591408], [1.5951889269749824, 
-2.4982410181843506, -0.098690469778995649, -3.2717538451007666, -3.2732314222313228]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-8.2744288296667783, -5.9793882902914222, 0.92172176604668421, 4.8350519521279445, 
-0.90254392630712721], [-1.411865792096022, 2.0220635227930965, 1.5460850281230494, -0.91259263894218101, -9.4370615729938141], 
[1.427926570568343, 0.88216319434747748, 3.9097122783116149, 2.0178823322932118, 1.5114926778872941], [2.6731499274441717, 
-1.7068726089134516, -43.207608332922007, -1.3033312914992807, 
-1.3027429516413722]])+(1.-msk_ref)*numpy.array([[3.4052596569198772, 2.4607583359694543, -0.37932551109393187, 
-1.9898179911417566, 0.37143306010833094], [0.58103945562650106, -0.83216031941795221, -0.63627605975157986, 
0.3755685087833433, 3.8837297070181962], [-0.58764911075996884, -0.36304557068164528, -1.6090035657524648, 
-0.83043959166498549, -0.62203991885040733], [-1.1001085140991556, 0.70244660438618167, 17.781665484941957, 0.5363731512961134, 
0.53613102582450245]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.45710785129)+(1.-msk_arg0)*(0.236739391979)
      arg1=Data(numpy.array([[[-3.4664289240522308, -1.8386012755254821], [3.0528594455198643, 3.5607791850373545]], 
[[-3.2780146322724635, -0.44044943950651927], [-4.2554154378230358, 1.6732791515427801]], [[1.8244246995849958, 
4.0050245279669952], [4.7737632229387987, 0.94466062736218515]], [[-4.6891980802200788, -2.7839067877101575], 
[1.7597267075772987, -3.3038863058494758]], [[-4.07625336200796, 1.4002672077441343], [-2.9142431076439768, 
2.0413553960743283]], [[4.1909160508089354, -2.993482731284459], [0.32849775345447974, 
-0.80221788940745142]]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-1.2857923669998232, -2.4241840308850557], [1.4599780732895085, 1.2517226201565237]], 
[[-1.3596973629748279, -10.119454020165229], [-1.0473966446796352, 2.6636965190083619]], [[2.4430209985140152, 
1.1128790398585291], [0.93366755809683988, 4.7182106697296513]], [[-0.95050534761896488, -1.6010262523767422], 
[2.5328409417790141, -1.3490500091975117]], [[-1.093432487007677, 3.1830409415030734], [-1.5294221129331667, 
2.1834061133426861]], [[1.0635163761949975, -1.4889372184155845], [13.568153219992704, 
-5.5559816231297932]]])+(1.-msk_ref)*numpy.array([[[-0.068294892861241879, -0.12876059378975233], [0.077546770889395891, 
0.066485277428618081]], [[-0.072220358520833985, -0.53749504652452396], [-0.055632498269162424, 0.14148230542463941]], 
[[0.12976111978373703, 0.059110597282516324], [0.049591775067871262, 0.25060787453406869]], [[-0.05048611466806413, 
-0.085038548353834065], [0.13453190825590625, -0.071654824065862655]], [[-0.058077693154637323, 0.16906729706300144], 
[-0.081235292744897089, 0.11597166884047912]], [[0.056488698200853102, -0.079084936587449667], [0.7206727884428662, 
-0.2951060991099585]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.885219639853)+(1.-msk_arg0)*(2.75497413394)
      arg1=Data(numpy.array([[[[1.467689808527135, 4.6829406890102963, -2.4921975976744317, 1.3781710158380056], 
[4.5659368432735281, 4.9692562597585823, -1.551567324228925, -2.787845987672962], [2.3966510474245979, 2.8979288457607471, 
3.7476369053946037, 2.530534811510857]], [[4.5082349960469301, 3.597141099902414, -2.5112915941016434, 1.1332118422904589], 
[-2.6548606308631904, -3.2724917971804626, -1.0985631523180892, 1.4818906888351187], [4.0487865949301671, -0.8652476351631595, 
-1.6358540766570231, 4.367138448376501]]], [[[-4.6762648412445742, -1.7928331112415217, -0.0027030165873807022, 
-3.8042820448108259], [4.1391118848640769, 0.90898456721033583, -4.0982367960832544, -0.81494306022968566], 
[3.6913212737025543, 0.0023887817239582887, 1.0685948582595195, 2.2191101539589999]], [[2.5810005499598176, 2.3740879852413919, 
0.7967523085283057, 3.5164384506592885], [2.1810683906035155, 0.22550149190801783, -0.89132213947725614, -1.2631676092836441], 
[-3.2493097908956181, 2.5539108743600405, -2.7057015342600566, -3.8273428273599643]]], [[[-1.0466483509815063, 
-2.255214496117024, -0.51492744202985019, 0.21355392655934935], [3.7352326573944126, 1.1141322726163985, 1.9371441856301654, 
0.56493885663453725], [-1.410687366257287, 3.3389146402400609, 3.8636824951032338, 2.585777645486047]], [[2.3394277377851829, 
2.6315560293776006, 2.9381376491249593, -1.6420904736994615], [-1.9436881516367945, 4.264006161459136, -0.52545692395622812, 
2.6127356184800128], [-2.9426977153149068, 4.4690388259354297, -0.090876374361887891, 
-2.8229842082142156]]]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.60313809819345277, 0.1890307177988311, -0.35519641005954045, 0.642314799600328], 
[0.19387470090767062, 0.17813926140648964, -0.57053253573311258, -0.31752817184562954], [0.36935691610349852, 
0.30546631300072052, 0.23620741875466206, 0.34981523898675326]], [[0.19635614395194098, 0.24608977386986125, 
-0.35249576032194541, 0.78115989157310106], [-0.33343356316416217, -0.27050324178525459, -0.80579768034733201, 
0.59735825761130901], [0.21863825595585315, -1.0230824146500481, -0.5411360661593817, 0.20270015487648496]]], 
[[[-0.18930057853980964, -0.49375462462312625, -327.49323255569487, -0.23269032879947471], [0.21386704792640282, 
0.97385552162860023, -0.21600012002698002, -1.08623495683671], [0.23981105252457627, 370.57368238154947, 0.82839593790931765, 
0.3989074802229598]], [[0.34297537823723073, 0.37286724222352774, 1.1110349231219334, 0.25173756124950841], 
[0.40586514557116804, 3.9255600145388421, -0.9931534297713438, -0.7007934919697526], [-0.27243313097855559, 
0.34661336413111421, -0.32716825142912709, -0.23128830621728716]]], [[[-0.84576604837990177, -0.39252126189199282, 
-1.7191152919786572, 4.1451808173939391], [0.23699183452483305, 0.79453729293214248, 0.45697147709477359, 1.5669299950908202], 
[-0.62750944045210866, 0.26512197382480063, 0.22911293590373472, 0.34234174829312319]], [[0.37839152949902061, 
0.33638639267822684, 0.30128596599842983, -0.53908091790987611], [-0.45543295569688935, 0.20760280504614662, 
-1.6846664293393345, 0.33880949667917609], [-0.30081908693711651, 0.19807830594706921, -9.7409216209247909, 
-0.31357583839016512]]]])+(1.-msk_ref)*numpy.array([[[[1.877082008703691, 0.58830002703414408, -1.1054396876535095, 
1.9990074542881537], [0.60337543608441235, 0.55440371555277634, -1.7756072140235208, -0.9882088702624674], [1.149509911718062, 
0.95067004076866179, 0.73512301311217021, 1.088692446123352]], [[0.61109816510447301, 0.76587880692775423, -1.0970347451545461, 
2.4311201411164833], [-1.0377095136057328, -0.84185822446299208, -2.5077976883993891, 1.8590940308219861], 
[0.68044439225169084, -3.1840296603925946, -1.6841197349179358, 0.63084194982831876]]], [[[-0.58913988567226272, 
-1.5366595566925101, -1019.2220598297166, -0.72417715129768878], [0.66559547327494384, 3.0308260814581676, 
-0.67223400477420436, -3.3805725435192753], [0.74633821595829086, 1153.2967228914365, 2.5781278214563295, 1.2414769627497053]], 
[[1.0674054811754849, 1.160434723173889, 3.4577548184740228, 0.78345580979173035], [1.2631305583140067, 12.217099366543463, 
-3.0908848910224438, -2.1810044159590394], [-0.8478644115939119, 1.0787275944523289, -1.0182106559278974, 
-0.7198137868000668]]], [[[-2.6321869531065922, -1.2216018204421357, -5.350218126041038, 12.900601634118903], 
[0.73756426617501214, 2.4727531924682, 1.4221833121047929, 4.8765881503637862], [-1.9529303230759825, 0.82511068139974175, 
0.71304361510958991, 1.0654335026653847]], [[1.1776273699106772, 1.0468992881734842, 0.93765999518931353, -1.6777237174620325], 
[-1.4173951369843283, 0.64609994207918842, -5.2430066259314279, 1.054440454846628], [-0.93620697756556093, 0.61645786515792855, 
-30.315625521911699, -0.97590844678752642]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.67484445855)+(1.-msk_arg0)*(3.27683623995)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.89475413806)+(1.-msk_arg1)*(3.09586078274)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.686781338112)+(1.-msk_ref)*(1.05845723368)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.05775920787)+(1.-msk_arg0)*(-1.66738590464)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-3.108558816232855, 2.1510413854922428])+(1.-msk_arg1)*numpy.array([-1.7831183589525725, 
-4.5413331451668517])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.3053506295849187, -1.8864161495166827])+(1.-msk_ref)*numpy.array([0.93509547264226633, 
0.36715780396256248])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.93644557337)+(1.-msk_arg0)*(0.938861363015)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-3.2099437999551803, 1.8020046736467554, -1.5740407620740826, -4.18260567028317, 
-1.6602663372749369], [0.4042098145242603, -3.4684605353487905, -1.7818423771733372, -4.5201465147788156, -2.4605961127594576], 
[-1.3109812494912987, 3.4787890722158608, 3.1455917150069954, -1.7773488095861478, -1.8515975096241477], [3.9877702770306591, 
0.85042879534655746, -3.9000527925378248, 4.8137299129814828, 
-1.6615557213740697]])+(1.-msk_arg1)*numpy.array([[1.766102659477534, 4.8692241737180666, -1.9721898136101732, 
0.16187299709706249, 1.5343341859650872], [2.8610805840890787, 4.8650426317766193, -2.1055930553799307, -0.016950061994571719, 
-4.9923948861273235], [-0.61642113909532004, -4.3191742537003064, -2.5447300979403167, 3.3958724109742704, 2.9623739924568255], 
[-4.1079486183726317, -2.2621693132075169, 3.0203237112118977, -2.5572941808971117, -2.5995457458213078]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-1.5378604365109876, 2.7394188514421671, -3.136161205171176, -1.1802321238283515, 
-2.9732853473842047], [12.212582168953077, -1.4232382127647301, -2.7704165287649323, -1.0920985762817397, -2.0061990457421275], 
[-3.7654585641787826, 1.4190126135559225, 1.5693217749220303, -2.7774208116887609, -2.666046777291708], [1.2378961751654729, 
5.8046547816659535, -1.2657381414990001, 1.0254928428914123, 
-2.9709780477855299]])+(1.-msk_ref)*numpy.array([[0.53160067336803396, 0.19281539101914713, -0.47605020395917619, 
5.7999875201691076, 0.61190148248231491], [0.32814922034579835, 0.19298111734582654, -0.44588927600063993, -55.389848327162717, 
-0.18805831358096387], [-1.5230843062800157, -0.21737056850878264, -0.368943395519721, 0.27647133030712856, 
0.31692870832851666], [-0.22854749419608456, -0.41502700860358083, 0.3108479265087174, -0.36713076267431044, 
-0.36116362426955717]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.61551355308)+(1.-msk_arg0)*(-0.418684456572)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[1.0337525916843315, 1.2090581946232923], [2.0062625320437277, -1.5300764816692203]], 
[[0.26961511165817686, 0.068033632247285425], [-4.4689132725022462, -0.3788199859133492]], [[-3.5238143677475078, 
4.8406420450330927], [3.2457706468752683, 2.1078008402856803]], [[-1.2143579925937811, 2.009296011162311], [4.1648598171971862, 
3.6479034482587558]], [[1.7032787698271781, -2.4019761983842192], [-2.3731430652433882, -2.7771246379714931]], 
[[-2.4129000700510526, -1.2426703725863897], [3.1980088111095935, 
4.8614314033323645]]])+(1.-msk_arg1)*numpy.array([[[-3.9521313849702966, -4.0017618134689981], [4.289107075944008, 
0.87008192809932439]], [[-3.9486710681785553, -2.6680315364838036], [3.434535938443906, 4.5451335890293603]], 
[[-4.8025784132684137, 0.21899942406941442], [-2.272627823527511, 1.4909700034883908]], [[2.5512179596569187, 
-3.456369994608175], [-3.4381390695735647, -2.8519608555507112]], [[-0.99028505438727166, -1.4687523906656619], 
[-1.4986552331998251, 4.7046173146102639]], [[-0.70949063056789186, 2.8305895460493549], [-2.6434007356788114, 
0.82294666292709895]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-3.4974650435331638, -2.9903552774845972], [-1.8021138785833433, 2.3629626338243277]], 
[[-13.409906925623583, -53.143032844935632], [0.80903641055744924, 9.544146791412345]], [[1.0260227060112901, 
-0.74690785219030675], [-1.1139152905207421, -1.7153013149893865]], [[2.977304530565394, -1.7993931869631825], 
[-0.868099698853942, -0.99112095601201977]], [[-2.1226786930741857, 1.5052245544772151], [1.5235126807270494, 
1.3018909931671832]], [[1.49840998305465, 2.9094711138503571], [-1.1305514670621901, 
-0.74371378573796931]]])+(1.-msk_ref)*numpy.array([[[0.10593890126341542, 0.10462503169559446], [-0.097615762245866391, 
-0.48120118698134651]], [[0.10603173810715624, 0.15692635219904386], [-0.12190422929803667, -0.092117084871387908]], 
[[0.087179098505853675, -1.9118061992697597], [0.18422922232930461, -0.28081346746938324]], [[-0.16411159814378115, 
0.12113415439479791], [0.12177647503489243, 0.14680582160078978]], [[0.42279185646346046, 0.28506129367568828], 
[0.27937343245949059, -0.088994370545757734]], [[0.5901197824660338, -0.14791422414343686], [0.1583885677723009, 
-0.50876256680258614]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-1.31276654553)+(1.-msk_arg0)*(-2.46388535412)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-2.9292800948896924, 1.5137416134814323, -0.029622463331879345, -1.2172103966404322], 
[0.22907145606852986, 4.2230308117684388, 1.3554684102119694, -4.5274629765282013], [-2.6240464513027186, 3.3702605355285726, 
-4.9795235054212261, 2.0955743836929299]], [[2.4117173797484455, 2.2048624200603308, -4.0976564980224159, 
-0.97409810804782726], [3.8649002600118045, -4.8354035371656279, 0.055722708628356266, 4.628967281131608], [-3.788087182201556, 
-4.267714354863327, 0.52049336693165049, 3.085222753745974]]], [[[3.4127805048274755, 4.6059617975127427, -4.205093499272631, 
-0.45302458175247118], [3.9286541221554714, -1.2983257615759278, 4.5610664362388462, -3.4146441577554487], [4.772271394666749, 
4.2410716510255035, 4.9975037497876844, 0.80843997522222466]], [[-2.5777829144157938, -1.5234087564575169, -2.2912594020910779, 
-4.9090515640840628], [-0.021976074138432722, 0.21205883279217108, 3.451380854052438, -1.9241545385052361], 
[0.080143154239510572, -1.534910007486129, 1.4529974291995638, -3.1530116826552099]]], [[[0.12477668044596602, 
-4.4566613158715658, 0.53272714362384477, 4.9302797697039011], [3.4723332122743766, -3.5957695833805738, 0.22455693402086574, 
-4.8566076132737042], [-2.7997202817870912, -2.1031438536813041, -1.6425064049295335, -0.73617434864789022]], 
[[-3.9259149126690449, 3.8783169066061625, -2.4028889106515772, -2.1808738418722227], [1.2164972159471708, 2.6617581732833182, 
2.2045535561740728, -0.56208395245160148], [-3.6553324434992671, 2.5020971683881523, 3.6770962738345254, 
-3.0406943785029075]]]])+(1.-msk_arg1)*numpy.array([[[[3.283379617603666, -4.4590093417480166, 0.46027608464767411, 
1.2908075144510898], [-3.5538017758416784, -3.8271222381688998, 4.0417391745915658, 2.6901331457404662], [1.3255868418747054, 
-0.91145122396314626, 4.79808885479998, 2.7412853483587565]], [[-3.723092680307488, -2.3569675568056891, -3.5363263386845869, 
0.95958528855548764], [-0.55312545388644629, 4.3700354715180882, 1.2092245231087837, 3.7927572155532836], 
[-0.95820533357627991, -1.2955151729076286, -2.6941374533590756, -1.6099134956514281]]], [[[1.023338965457131, 
-1.8254617830081719, -3.7844723292670226, 0.97970878495877578], [-3.2979062242810619, -2.4271037418202788, 4.6786489385602241, 
4.4301076651053233], [3.8568156261605111, -2.3541678994852244, -3.271257289689975, 1.639916579017406]], [[1.5650370123777577, 
-3.4044556234708798, 4.4659300203284751, 0.47543486738602336], [-1.3982751162222962, 0.64935505706512853, 2.6376814166838152, 
-4.6064000403618479], [2.7582701891062751, -0.84797114120135575, 2.828598516624905, -0.98488470728915178]]], 
[[[-2.9041630178414368, -2.0563213902900168, -4.2942871177631847, 0.98589623750687316], [-0.88645924867286485, 
3.0178255460629284, -1.5797263713553287, -0.78138106836802024], [-1.0383918894101853, -4.8353860492201894, 4.4355776952969048, 
-2.7660157910534822]], [[2.4408928008486068, 4.1130124358200995, -3.3818596651064325, 4.0557228121249018], [2.663060374079075, 
2.8134627797075265, -1.7317754512515329, -2.1242409893335559], [1.5801137792862665, 4.5194479995089374, 2.0430823903539252, 
1.4236820209206158]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.44815330149388943, -0.86723291071184649, 44.316589434760665, 1.0785042168128514], 
[-5.7308167855379475, -0.31085886038691118, -0.96849659913494979, 0.28995632925789694], [0.50028327237630299, 
-0.38951485550931225, 0.26363296490034349, -0.6264471238724173]], [[-0.54432851732489385, -0.59539612702421074, 
0.32037008132787803, 1.3476738479207617], [-0.3396637577191034, 0.27149058717334162, -23.558914809412308, 
-0.28359814744774509], [0.34655130211712742, 0.30760412632330231, -2.5221580694947408, -0.42550138200920251]]], 
[[[-0.38466187428936716, -0.28501464042408126, 0.3121848647960439, 2.8977821478182699], [-0.33415172339093441, 
1.0111226198975618, -0.28782008854220614, 0.38445193258089633], [-0.27508212273766142, -0.3095365166037029, 
-0.26268445433000853, -1.6238268588392044]], [[0.50926186925351058, 0.86172968348810874, 0.5729454047529825, 
0.26741755069954881], [59.736172041271267, -6.1905770593943155, -0.38035980410094844, 0.6822562945203775], 
[-16.380270504476297, 0.8552726473359793, -0.90348855348514578, 0.4163532132616915]]], [[[-10.520928596859386, 
0.2945627797315396, -2.4642381399887623, -0.26626613637464847], [-0.37806468022269324, 0.36508639251877423, 
-5.8460298776753321, 0.27030525215526935], [0.46889203684562247, 0.62419246464159217, 0.79924592171162312, 
1.7832277746925971]], [[0.33438486944503371, -0.3384887251707146, 0.54632843811708887, 0.60194520211135827], 
[-1.0791364980667242, -0.49319527171990696, -0.59547954362397559, 2.3355346470920386], [0.35913738786189081, 
-0.52466649261700404, -0.35701174181011552, 0.43173248676556991]]]])+(1.-msk_ref)*numpy.array([[[[-0.75041135691583183, 
0.55256339811790167, -5.3530596880821513, -1.9087937794997332], [0.6933097312475196, 0.64379583425446241, -0.60961018207334905, 
-0.91589717706614149], [-1.8587128932505192, 2.7032553024640813, -0.51351390703215005, -0.89880659654454165]], 
[[0.66178458762199321, 1.0453624391228213, 0.69673585470967803, -2.5676564485734912], [4.4544783408603985, 
-0.56381358233228107, -2.0375747489650418, -0.64962907301629724], [2.5713542471318434, 1.9018575819423686, 0.9145358753109486, 
1.5304458039335691]]], [[[-2.4076923065419691, 1.3497326413788431, 0.65105122715825436, -2.5149160566314892], 
[0.74710594739632874, 1.0151545282806445, -0.52662325950747046, -0.55616827860032936], [-0.6388392894395547, 
1.0466056200386213, 0.75319216311146853, -1.5024455424385343]], [[-1.5743304053700149, 0.7237237393048449, 
-0.55170711204617806, -5.1823825367760232], [1.7620891093109079, -3.7943576896924531, -0.93411029039793381, 
0.53488306107304695], [-0.89327193682722272, 2.9056240647825162, -0.87106223793671145, 2.5016992708694596]]], 
[[[0.84839774454089434, 1.1982005175602701, 0.57375887697959627, -2.4991325256970311], [2.7794682697534965, 
-0.81644393173427487, 1.5596912217157317, 3.1532442413300119], [2.3727894827021982, 0.50955297654322906, -0.55548240237753899, 
0.89077053069799772]], [[-1.0094197308702584, -0.59904641489966093, 0.72855931295369403, -0.60750832052668235], 
[-0.92520822212594389, -0.87574833827084975, 1.4227510572084461, 1.1598897519102829], [-1.5593088209310364, 
-0.54517395805493651, -1.2059647548961208, -1.7306430213415878]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.9690479287799807, -4.4245825909133449])+(1.-msk_arg0)*numpy.array([-3.3692645962102361, 
1.4560599455853751])
      arg1=3.46860219991
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.56767764514222241, -1.2756096940226485])+(1.-msk_ref)*numpy.array([-0.97136091198286212, 
0.41978291590289601])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([4.5251679036808099, -3.0184890479158835])+(1.-msk_arg0)*numpy.array([3.289587056950392, 
0.46276486014064222])
      arg1=numpy.array(4.46602446919)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([1.0132429714385984, -0.67587830490828327])+(1.-msk_ref)*numpy.array([0.73658061652898132, 
0.1036189710408027])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-4.2564806448962518, 4.577944864438205])+(1.-msk_arg0)*numpy.array([-4.2326086220341095, 
-2.4511207307076344])
      arg1=numpy.array([-4.3650056044380872, 1.6988631049839444])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.97513749823562801, 2.6947108634050125])+(1.-msk_ref)*numpy.array([0.96966854240247391, 
-1.4428006138439267])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-1.6094695972274486, -3.4781088983761443])+(1.-msk_arg0)*numpy.array([-0.73288005195922068, 
2.7527271610694051])
      arg1=Data(4.87764199074,self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.32996878415523972, -0.71307178857787989])+(1.-msk_ref)*numpy.array([-0.15025294052972293, 
0.56435613074883972])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.830219429346986, -0.17859791069816033])+(1.-msk_arg0)*numpy.array([-2.6284919253580608, 
-3.0237199865233846])
      arg1=Data(numpy.array([-4.8506778822214871, -4.3915626304941409]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.7896255992147726, 0.040668419358979836])+(1.-msk_ref)*numpy.array([0.54188135950892669, 
0.68852940079398439])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-2.6800640519143992, 2.0023541858171878])+(1.-msk_arg0)*numpy.array([1.1613681387518646, 
-3.0696609819889922])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.656073388874)+(1.-msk_arg1)*(4.79752421351)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([4.0850064906850649, -3.0520277453334819])+(1.-msk_ref)*numpy.array([0.2420765559622608, 
-0.63984272832750255])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.31230244240899463, 3.1025957246958527])+(1.-msk_arg0)*numpy.array([-1.6964268758615617, 
4.030579379172039])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-2.6976456744781308, -4.7352730038900184])+(1.-msk_arg1)*numpy.array([-0.14939469477309508, 
0.77918822352547323])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.11576851821706001, -0.65520947200870483])+(1.-msk_ref)*numpy.array([11.355335465145822, 
5.1727929882403716])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.410303391226897, -0.15514188464505452, -1.687491967901702, -1.579101932074364, 
0.26706094549991288], [-3.5303665372888982, 2.089473433559589, -3.0059121525916965, -1.692959306980697, 2.2717737490018557], 
[-0.90645790278991445, 0.40341832853528903, 0.68666742917210311, 2.0626268306763613, 4.4570192216006461], [-1.4392172230686651, 
-1.0411162972308405, -3.2093062231663749, -4.1064799831755749, 
-3.6513620151057657]])+(1.-msk_arg0)*numpy.array([[4.4564729933588172, 4.4437025598736231, -4.3476199896151977, 
-1.2442254623656734, -1.5090363536676032], [-0.83001159258585133, 0.041459403655911053, 0.1051555316842121, -1.096017044685663, 
1.0799848928163911], [4.0718159471940361, 0.75790507207492297, 1.0631276118915185, 2.8175439476881659, 3.2337369051060598], 
[1.0925612271107603, -1.0819538840584588, 2.3088842689166285, -4.903225267172143, -4.4031757991605307]])
      arg1=-2.8426976555
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.84789298171165806, 0.054575583986173105, 0.59362344237931508, 0.55549415500412935, 
-0.093946306594786519], [1.2419071477608687, -0.73503189110410183, 1.0574153557187431, 0.59554673487875354, 
-0.79916122792947109], [0.31887242775749208, -0.14191390623435407, -0.24155485823245276, -0.7255878326298274, 
-1.5678836660581195], [0.50628571782297604, 0.36624235968830321, 1.1289650226983226, 1.4445714883642251, 
1.2844707589781064]])+(1.-msk_ref)*numpy.array([[-1.5676915146906421, -1.5631991503829272, 1.5293993651428346, 
0.437691803051314, 0.53084658888988168], [0.29198025719684595, -0.014584528036490844, -0.036991458265267022, 
0.38555526387584332, -0.37991549707255651], [-1.4323774247730907, -0.26661473147116416, -0.3739854675837519, 
-0.99115146566379508, -1.1375592120565943], [-0.38433958145241853, 0.38060814591555181, -0.81221591204016708, 
1.7248493724561027, 1.5489427061090604]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.0494478966696068, 3.6795371717758698, 1.9211166322238125, 1.0774994034195249, 
4.1279713581047979], [-1.0338439734143701, -2.8804363541401767, -0.85827010664703174, -1.0825401631894849, 
0.78437434804826545], [-4.9903529668748341, -0.84642958448940497, -4.4157077114671175, -1.4467904237103433, 
-0.60995905178916665], [0.2377779461374967, -1.5696190566720833, 2.2413420543278075, 3.949820863051249, 
4.9569096567408053]])+(1.-msk_arg0)*numpy.array([[-4.261788465022688, 2.0459050602270441, 0.54672637931962598, 
0.96422214632448444, -2.3189328103688958], [-0.83263701746795249, -2.1676938095875986, -4.2090990732929869, 3.6982260508334637, 
0.83530638981434535], [4.0678210803312478, 4.5417452459052114, -4.0533484119091368, -0.91034350081337401, -4.0677837839073936], 
[-1.4655018985123878, -0.77428579694339028, 4.764548547536636, 4.4554623162974973, 2.7678638809913325]])
      arg1=numpy.array(4.74353824914)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.43205046297296157, 0.77569463521144122, 0.40499655137666718, 0.22715098874042988, 
0.87023043586836835], [-0.21794785223929664, -0.60723371518385016, -0.180934581228089, -0.22821364693041646, 
0.1653563873319589], [-1.0520317756018309, -0.17843844405458126, -0.93088902830507236, -0.30500237327559232, 
-0.12858735816036335], [0.050126705773006965, -0.33089625807436795, 0.472504265088526, 0.83267397786237285, 
1.0449814877412633]])+(1.-msk_ref)*numpy.array([[-0.89844083491853621, 0.43130358664215357, 0.1152570824993535, 
0.20327065909079581, -0.48886141284741758], [-0.17553079025333665, -0.45697825035604073, -0.88733322094719602, 
0.77963449572821764, 0.17609352891089433], [0.85754996938651951, 0.95745939156946114, -0.85449894130115644, 
-0.19191233484392961, -0.85754210681181364], [-0.30894699727128766, -0.16322958860598455, 1.0044292461230653, 
0.93926981976124158, 0.58350196322225822]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[1.148355942207524, -2.7805928894099829, -3.3037216174015995, 2.7688529686635022, 
2.2034295113738782], [-1.861309394420434, -1.1883171485664299, -4.0607836758499962, 3.8357441882091301, -0.99517736137248036], 
[1.2904656369956253, -2.3889894990490013, 4.2136078547587736, -3.6468381336658506, -1.002002246584798], [0.61785866146358615, 
2.9845956006859797, 2.5505931344716064, 4.0082468440941259, 
4.3015577136561074]])+(1.-msk_arg0)*numpy.array([[4.8510001597883008, 3.1416471544116682, -4.3211585528159526, 
1.0561657588326536, -3.5374998997334894], [2.1776714817312577, -1.9620691058196016, 2.242090679863896, -1.9881111881919145, 
-4.7066582986587218], [2.8785003842997767, 2.6215055637484923, -0.74696058667359111, 1.837652358856003, -2.0259552833671588], 
[-0.98160647743729612, -0.90472511923361498, 2.3662408256752414, 2.5322932653187626, 4.7462719335290178]])
      arg1=numpy.array([[1.274302883112659, 0.86108266998222138, 3.3405750198536861, -4.0519348648221074, -2.21015726425702], 
[2.8545653750159508, 3.5793557428106997, -3.2190468252005688, -3.458702610572526, 2.0813450603379371], [-3.4442727915771618, 
1.8039097553838808, 2.039111133551005, -1.096450295792577, 4.5608306210849907], [-4.25431468672405, -1.8199563772301919, 
-0.41220889342830525, 0.14531084119227078, 3.9808935033994928]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.90116404618226054, -3.2291822682569995, -0.98896794646638386, -0.68334093736352874, 
-0.9969559845392254], [-0.65204651142734238, -0.33199190970420289, 1.261486364242925, -1.1090124304078839, 
-0.47814145781809897], [-0.3746699855340756, -1.3243398079748245, 2.0663944134427812, 3.326040539785442, -0.21969731608810072], 
[-0.14523106703687588, -1.6399269993648216, -6.1876227687824654, 27.583949079136762, 
1.0805508135253512]])+(1.-msk_ref)*numpy.array([[3.8067874004483686, 3.6484849410295683, -1.2935373482512647, 
-0.26065714135782947, 1.6005647909958467], [0.76287322083807219, -0.54816264344791865, -0.69650763148628581, 
0.5748141462393086, -2.2613541542672055], [-0.83573530857922751, 1.4532354270630479, -0.36631676144732656, -1.6760015168107942, 
-0.44420752526985924], [0.2307319861646536, 0.49711362895990086, -5.7403924645958604, 17.426733232987832, 1.1922629755043506]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-2.8200591316844048, 2.3658193338369742, 0.30017455821027195, -1.4277240174526993, 
1.2184664241047365], [0.23174560453216486, 1.1396249018959779, 2.0511745146631153, 2.5633133816805822, 3.0867762739010054], 
[2.6216076115985913, -3.2646219781091732, 2.6788298311248528, -0.021539946894937323, -3.0740712762933855], 
[-2.2185088137959861, 1.9652732798010275, -1.3592420593723533, -1.796676524640973, 
-1.0475872021329371]])+(1.-msk_arg0)*numpy.array([[0.28487875859378864, 0.28056076675615493, 3.0236154442989722, 
-1.7603512428813328, 0.55640282944439701], [-1.7421584040315263, -2.4767261762081327, -1.3892859789274259, -2.4851217301297126, 
-1.2546403469584657], [-1.6727069155125807, -2.9014764458894615, 4.6446210676836426, -4.9922542523336961, -4.6414768174621717], 
[1.4395471551487002, -3.7088942820706272, 1.8414151490073616, -1.1336130474058392, -0.67907023089558116]])
      arg1=Data(-3.81752500929,self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.73871399003846749, -0.61972595545008691, -0.078630672354412054, 0.37399205348445302, 
-0.31917706397194939], [-0.06070572005899013, -0.29852454119426164, -0.53730480079932508, -0.67145948630135655, 
-0.80858049819952116], [-0.68672964950294557, 0.85516714891572299, -0.70171899977229735, 0.0056423852738406183, 
0.80525242632659266], [0.58113799081789774, -0.5148029875425576, 0.35605321669525308, 0.47063909739127824, 
0.27441528204361804]])+(1.-msk_ref)*numpy.array([[-0.074623940354160603, -0.07349284315712179, -0.79203553007270411, 
0.46112369626849836, -0.14574962261945976], [0.45635808535421934, 0.64877798316470814, 0.36392321610114003, 
0.65097719702739365, 0.32865281665613549], [0.43816528023835744, 0.76004124107334536, -1.216657665995807, 1.3077201171399555, 
1.2158340302072217], [-0.3770891223097334, 0.97154420024617583, -0.4823583721196727, 0.29694973697542659, 0.17788232670199092]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-3.0977497463423687, -2.3170938574689659, -0.20503941931962011, -4.5898934065456771, 
-0.42619269712573704], [-4.8825497775602251, 3.3133187115519878, 2.8612308565498079, 4.7591865480545579, 4.7920088021431013], 
[-0.82765120977030726, -2.4798895081033647, 4.175604275145492, 2.1654082427629637, 4.6578827486594037], [-2.4711651791276346, 
-0.20392147725246623, 1.609054826713173, -4.0329116169603898, 
-1.066489449083595]])+(1.-msk_arg0)*numpy.array([[-1.9633542044305399, 4.8470062073414546, 3.1263192163642923, 
-4.8033079553405802, -0.31098787621779422], [4.0457527926001209, 0.086714290376979974, -4.2659319774817561, 1.373882877352413, 
0.60707521624791649], [-3.5465947274121437, -4.8890742314183067, -3.0959330031655297, 0.45333801671696872, 2.8097039371601742], 
[3.9811025489770113, 4.9007324717614935, -4.688754488597942, -2.0786884286701799, -3.0295248113578177]])
      arg1=Data(numpy.array([[-4.7217054280507238, 4.9485621313262502, 1.6106443486333486, 1.9163551383297381, 
-2.3765526117867433], [1.3032603561283587, 2.849591837056157, 2.4293372163697411, -1.0337996283918538, -0.088125953033413751], 
[0.83089398521219238, -1.5952547947939588, -3.1609028844824536, 1.6286949658854235, 2.6107477472621614], [3.3563427481156012, 
0.086610522222403397, 4.9168461903062859, -3.8734999999523732, -4.5187013360367008]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.65606586296960545, -0.46823578162248275, -0.12730272793841704, -2.3951162885946853, 
0.17933232153666323], [-3.7464116472206577, 1.1627344900646879, 1.1777824985637291, -4.6035870175904385, -54.376816785472585], 
[-0.99609724525679777, 1.5545413285679321, -1.3210163132959332, 1.3295357867000968, 1.7841182678579466], [-0.73626723031044894, 
-2.3544653931173065, 0.32725343938670975, 1.0411544125493679, 
0.23601680433675226]])+(1.-msk_ref)*numpy.array([[0.41581463188419981, 0.97947769043013355, 1.941036343012045, 
-2.5064811105560842, 0.13085671854072142], [3.104331972944363, 0.030430424894311307, -1.7560065143432473, -1.3289643753205658, 
-6.8887222815932381], [-4.2684082332193318, 3.0647607187100001, 0.97944578378669112, 0.27834433470512759, 1.0762065925774156], 
[1.186143027619029, 56.583569132363799, -0.95361016129444243, 0.53664345648528167, 0.67044147998835835]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[3.8516610989820972, -4.8722250008467816, 4.1216830944727914, 0.90637015968298051, 
-0.36938032693578293], [-2.2176679623179085, 3.3276032274927676, 0.93738453045659309, 0.39222744995013414, 3.3014585857597503], 
[4.3446698532390808, 2.6981495971750205, -3.5664760490157055, -4.1190431253208786, 1.3560308127964404], [-0.74218714552620746, 
1.1132965058122322, -0.48743631887238514, -3.0021052995725683, 
-2.1846020700761315]])+(1.-msk_arg0)*numpy.array([[-1.9154942873785985, 3.0914740603099702, -4.6761268807469838, 
4.4372883733280801, 2.5488806173091962], [4.0520455603788221, -3.2142755854208502, 4.1766453431150143, 1.2127780387045188, 
-0.50420382432476885], [-3.4125604853641365, 3.9739743075634006, -2.1338677740783685, 1.3457419019008228, 4.0667471532283663], 
[0.5619045178706541, -1.7101692536711854, -2.7565489564755099, -0.70371734308194256, 3.5544469255022193]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(4.53292687102)+(1.-msk_arg1)*(0.343181089485)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.84970731021569679, -1.0748518869765786, 0.90927632669747094, 0.19995252195151053, 
-0.081488260774071203], [-0.48923532750860904, 0.73409594334381845, 0.20679454073012093, 0.086528519234978368, 
0.72832822582334755], [0.95846899296166888, 0.59523342730777695, -0.78679320238188133, -0.90869392834271501, 
0.29915126614212639], [-0.16373243307112439, 0.24560213246078288, -0.10753235883602338, -0.6622884915181767, 
-0.48194072665079535]])+(1.-msk_ref)*numpy.array([[-5.5815846096177086, 9.0082879128148363, -13.625829114800336, 
12.929874370389276, 7.4272175694116083], [11.807310147727927, -9.3661209312231009, 12.170383133256795, 3.5339302655813936, 
-1.4692063163565476], [-9.9439059724709349, 11.57981727236964, -6.217906054449605, 3.9213754578442499, 11.850149317192733], 
[1.6373411446262418, -4.9832852277495867, -8.0323451406255373, -2.0505714465181759, 10.357350781889799]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.73557422607972622, 1.7639407521821386, -1.8377992827507916, -1.2116077750298571, 
0.76561287677937706], [0.16292894637794397, 3.268071442843091, -0.61795058764401301, 4.7073196563072432, -0.42896706835255127], 
[-3.846982945573524, -2.3454172375679319, 4.0705901432430203, -0.96720212323389543, 2.594674289552894], [1.2120149131551186, 
-4.095086821149061, -2.578915984150715, -1.9013177980006701, 
3.1807342237083915]])+(1.-msk_arg0)*numpy.array([[-3.5207870549436846, -4.2464297310459918, 0.65068385498336134, 
-3.6962527711041382, -1.5570207427632199], [-4.6023880051474197, -4.5948026835763454, -2.2516428959582893, -2.7389440214352865, 
-2.7933105001611502], [1.148448621049222, 0.99370580770158767, -2.1221270744208418, 2.1778821624703184, 1.4240790173614286], 
[-2.7274067134293323, 1.0583953633611998, 2.6620336129940974, 3.9159695624080708, -2.6165191033865955]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[3.171361813446703, -2.8748367359882607, 3.7843035759242465, -2.0942099858236727, 
4.463856932165692], [-1.2995495742381733, 0.7684169668401255, 2.8417288120226072, 3.8152560911587088, -0.76747942850152384], 
[-0.47986842708536415, -3.7967235554955616, 4.2002254104540864, 4.0453684107902337, 4.8990476610969402], [2.8677041371763954, 
4.8611438415748296, -4.1548154777389321, -1.0477125394243405, 
-3.0488091258688463]])+(1.-msk_arg1)*numpy.array([[0.40313469732577545, -3.5453929695332254, 1.9534330657783077, 
1.2642079286965107, 0.88807572573990434], [4.5766838649789481, -2.4107641273397737, -1.2535925413188109, -3.8256918341529813, 
-1.8106125683770644], [2.1323543216425946, -4.3448325694860728, -4.4763809636730532, 1.0053858623290823, 2.6919786192054316], 
[-3.8013314773600584, 1.533567635804892, 4.3833545622591217, -0.76760476227752683, 1.2599606421776848]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.23194270138489453, -0.61357945308701578, -0.48563738238202597, 0.57855123566002875, 
0.17151375781390266], [-0.12537339829721908, 4.2529922995870493, -0.21745586173797676, 1.2338148590381022, 
0.55892972817537812], [8.0167452752401687, 0.61774769832084864, 0.96913611662640475, -0.23908876152146535, 
0.52962830105880687], [0.42264294194187518, -0.84241218828497066, 0.62070529918074047, 1.814732311064386, 
-1.0432710256342956]])+(1.-msk_ref)*numpy.array([[-8.7335252418089837, 1.1977317514693053, 0.3330975943750133, 
-2.9237696483324864, -1.7532522257220586], [-1.0056163241610725, 1.9059528186387156, 1.7961521162127403, 0.71593430421760462, 
1.5427433504810604], [0.53858245292205909, -0.22870980453433856, 0.47407204427916921, 2.1662152254906637, 0.52900829419728523], 
[0.71748720932999421, 0.69015238627261566, 0.60730510735187282, -5.1015441212078558, -2.0766673305479357]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[1.6823162016463789, -2.8801337615862241], [-4.6286990664293963, 2.6106963528491569]], 
[[2.2937651007949835, -1.0850173780241921], [-0.093255562464694464, -0.072341477360184925]], [[-4.8855388038147574, 
-0.64614674019247342], [0.99218391436770137, -2.240995142266752]], [[-1.8894667338613336, -3.1126067206510175], 
[4.6853914057357322, 4.5366790328623363]], [[-4.3099687403088609, -4.2960400484517347], [0.34841827378277568, 
4.8512029911197914]], [[-4.0813429241216239, -2.2259924111163123], [-0.1168323680784642, 
-4.100013436580717]]])+(1.-msk_arg0)*numpy.array([[[2.4814938904968926, -1.5379279369526144], [4.1878664874949081, 
-4.1677743309540372]], [[3.6280664499592437, -3.7571147437585761], [2.9408884474936192, -1.1890373686046498]], 
[[3.2821762099956224, -0.84622752859024519], [-1.8671307085857469, -3.3712537321841207]], [[-4.2601804720206848, 
-1.6564232635003426], [-3.7050275270887401, 2.0874669623508506]], [[-2.805111427506807, 4.4678421707239941], 
[-3.2108254464468997, -4.1013313672484948]], [[4.3610960125362936, 3.2719816025884239], [0.19806421790910811, 
-0.47935070791021772]]])
      arg1=1.10113103857
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[1.5278074477252204, -2.6156140011550715], [-4.2035860440795325, 2.370922498236999]], 
[[2.0830991230407196, -0.98536626434101116], [-0.084690703647704119, -0.065697428213657511]], [[-4.4368368819809474, 
-0.58680276693723465], [0.901058892734108, -2.03517570913275]], [[-1.7159326798358816, -2.8267359756723027], 
[4.255071596044389, 4.1200173948053394]], [[-3.9141288269485992, -3.9014793861742896], [0.31641853837461992, 
4.4056545689843007]], [[-3.7065006626526817, -2.021550871919966], [-0.10610214768838792, 
-3.7234564215997414]]])+(1.-msk_ref)*numpy.array([[[2.2535863612771059, -1.3966802161457059], [3.8032407958812997, 
-3.7849939607298801]], [[3.2948544023205355, -3.4120505300103856], [2.6707888021364128, -1.0798327600958537]], 
[[2.9807317158750744, -0.76850756081706217], [-1.6956480593018679, -3.0616281024707157]], [[-3.8689132562814588, 
-1.5042925914192011], [-3.3647471529867476, 1.8957479984077026]], [[-2.5474819338064414, 4.0575027078814001], 
[-2.9159340114697208, -3.724653309728053]], [[3.9605604235857252, 2.9714734105053053], [0.17987343101924985, 
-0.4353257615309632]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.86983763706075212, 2.7856665110504064], [-0.23254233202011676, -4.1573323765906913]], 
[[3.7662038506487363, -3.4842152731378997], [-3.4671738586723775, 3.7482955460199214]], [[-1.887344963626878, 
1.3231876055887168], [-0.43615920193491853, -2.9205950157408287]], [[-3.818354251913191, 1.8065750372406475], 
[-0.18015964361994108, 4.8672168649628293]], [[-3.3440161635812515, 4.9122369062019775], [0.064881081192954326, 
-2.3569569688582117]], [[3.3313622446213387, 0.70123445691401809], [-1.5052990415021883, 
-2.9105121208407594]]])+(1.-msk_arg0)*numpy.array([[[-1.8935193789138047, -2.389241040498514], [1.3864480833518993, 
3.533693894436249]], [[3.660976342036685, 0.91965798694228962], [1.1794375091242735, -1.4904952068586175]], 
[[4.5621384538519649, -1.9417127237359635], [1.8235327088597302, -2.0190282414775282]], [[-2.0065458271839898, 
-4.7114289654774977], [1.5822038001136498, -1.5352726622602386]], [[-2.1088110960915154, 1.0652821700219581], 
[4.381594091526221, -1.0915690355764784]], [[-4.8797218498299753, 4.8730981381075171], [-3.6338377496068821, 
1.006994825786208]]])
      arg1=numpy.array(-2.73036397044)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.31857937127746905, -1.0202546404843098], [0.085168986456711643, 1.5226293716159647]], 
[[-1.379377947931514, 1.2760991980783896], [1.269857753842371, -1.3728190038410151]], [[0.69124299326444749, 
-0.48461949392648712], [0.15974397796678028, 1.0696724126756061]], [[1.3984781125356094, -0.66166088360407238], 
[0.065983746332196322, -1.7826256563817251]], [[1.2247510587553512, -1.7991143156691272], [-0.023762795691502331, 
0.86323911184640389]], [[-1.2201165414906499, -0.25682819745147734], [0.55131808718556308, 
1.0659795369237874]]])+(1.-msk_ref)*numpy.array([[[0.6935043823516821, 0.87506320269650573], [-0.50778874112140859, 
-1.2942208191640994]], [[-1.3408382111962773, -0.33682615098191543], [-0.43197080019156731, 0.5458961599978861]], 
[[-1.6708902194897937, 0.71115526895286663], [-0.6678716568938925, 0.73947219613864601]], [[0.73490049272156033, 
1.7255680987915745], [-0.57948457320871705, 0.5622959718495053]], [[0.7723553046128574, -0.39016123181962309], 
[-1.6047655693403891, 0.39978883672473198]], [[1.7872056263060794, -1.784779681708885], [1.3308986600142234, 
-0.36881340242147792]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[4.3521998235062131, -1.6482419251955194], [-2.7727308653101068, -1.5594983844649537]], 
[[-3.7196090167147022, 2.7060589614220421], [-0.89431531970656941, -4.1619106884778194]], [[2.0843258764025965, 
-1.8818896027690011], [-2.2835436125943955, -0.8848109050935502]], [[2.4873532636783446, 0.53961325382071301], 
[-1.2922008971412069, 3.1656940656093528]], [[2.7051977908343225, -1.5650697638735664], [-3.2035996207393547, 
3.8643577775436952]], [[1.9698261383066757, -0.82321736370819032], [0.18155875651825504, 
-0.88515034326247299]]])+(1.-msk_arg0)*numpy.array([[[-1.3589957136764355, 4.6827423735971081], [4.2559517378357477, 
4.5654165974375136]], [[1.6372271617723015, -3.7894723392800889], [4.3089778110426682, -3.7331735154083381]], 
[[4.4397466457498176, -3.35994006938631], [1.3013159784008463, -1.5312909619239203]], [[4.5127306836400898, 
-3.3062807414110882], [1.751140701125963, -2.281024405190891]], [[0.39682968997257451, 2.7437436468521952], 
[-3.666593294208079, -2.2992231260858818]], [[-0.96968987786387295, 3.5155211340729426], [-3.2007657517314736, 
-4.3769808201417417]]])
      arg1=numpy.array([[[0.08995161936665852, 0.8567894986772151], [-0.16361816705443033, -2.1711050036733406]], 
[[2.905722724395571, -3.6332330275608959], [-1.7429297291569021, -4.1910374084549815]], [[-2.6595183327385885, 
3.2544075461525797], [2.6044693874980664, 3.1866149333988059]], [[-2.0892229397915631, 3.0427171332061711], 
[-3.2287828769581162, -1.809721469337775]], [[3.6170653932586223, -3.8368989971689107], [-2.0446630546643574, 
4.7611593122944385]], [[-0.26362602115112121, -4.3183431101648617], [-4.4822592189471324, 1.9945283509115583]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[48.38378512971385, -1.9237419783274845], [16.946350855939556, 0.71829707997835379]], 
[[-1.2800977138960947, -0.74480743208444988], [0.51311037085767752, 0.99305023622113175]], [[-0.78372307148426446, 
-0.57825873867389643], [-0.87677882625759629, -0.2776648335573515]], [[-1.190563829404679, 0.17734584918582683], 
[0.4002130048331427, -1.7492714316793534]], [[0.74789850243686185, 0.40789965152284874], [1.5668105380156356, 
0.81164219133877968]], [[-7.4720474470063438, 0.1906326900635652], [-0.040506081341922608, 
-0.44378930129417971]]])+(1.-msk_ref)*numpy.array([[[-15.10807390956389, 5.4654525771227664], [-26.011486465435919, 
-2.1028078281396727]], [[0.56344920595025683, 1.0430028326105143], [-2.4722613533748294, 0.89075165682774615]], 
[[-1.6693799742219009, -1.0324275683783037], [0.49964725431114804, -0.48053843778691624]], [[-2.1600043718121889, 
-1.0866211338965956], [-0.54235319247472547, 1.2604284382091009]], [[0.10971039968261943, -0.71509405091890355], 
[1.7932506218292139, -0.48291245372712993]], [[3.678278318770388, -0.81409027592037042], [0.71409652931302858, 
-2.1944941610588451]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.031114782671034291, -2.7399464480546234], [4.4239905687282075, 4.733421382482117]], 
[[-4.5402395256466379, -3.382978259917361], [4.7216155406282319, 3.0543160197254231]], [[-0.40353176213230402, 
4.2652019547517064], [2.2403135238781893, -1.3995283008362791]], [[0.34670920406598071, 4.665733401152842], 
[-0.97829673858727784, 4.506560380528791]], [[-4.094925547740413, -1.4206147622461396], [-4.3081581937833775, 
2.3042472572167769]], [[-1.6266055366885035, -0.88690802424438431], [2.6734369988483442, 
-4.1910272811079317]]])+(1.-msk_arg0)*numpy.array([[[1.6056714405189032, -0.45600497470922541], [-0.653485358801559, 
2.1473116072279712]], [[-0.78806816863821627, 1.3545632921491011], [-1.7598167011214838, -1.4784015635035006]], 
[[4.521618087420805, 3.9005591548785858], [-1.175011876497706, -2.0833829499314174]], [[3.4328433791651332, 
2.3231592185979499], [-4.4012332805149379, -1.0840881688675741]], [[-0.87760874285067292, 3.5622892466198408], 
[0.10569648070234905, -1.7872037827201739]], [[-0.49033666269604304, 0.95954164246023055], [-1.0844319111343537, 
4.596929121675025]]])
      arg1=Data(2.02779476587,self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.01534414783719867, -1.3511951476394484], [2.1816757016957098, 2.3342704410459123]], 
[[-2.2390034741482898, -1.6683040694520006], [2.3284484308244737, 1.5062254184349484]], [[-0.19900029772432079, 
2.1033696439777652], [1.1048028930654834, -0.69017255808656708]], [[0.17097844905271997, 2.3008903463398411], 
[-0.48244366493701452, 2.2223947198085141]], [[-2.0193984207173639, -0.70057127385640239], [-2.124553365207662, 
1.1363315933139533]], [[-0.80215491432595709, -0.43737563542979563], [1.3183962419880249, 
-2.0667906593131682]]])+(1.-msk_ref)*numpy.array([[[0.79183133694922958, -0.22487728165783158], [-0.32226405245730821, 
1.0589393184003051]], [[-0.38863310128893291, 0.66799821902427692], [-0.86784754095413097, -0.72906863573440162]], 
[[2.2298203760659612, 1.9235473039608482], [-0.57945305721930351, -1.0274131213844695]], [[1.6928948811490288, 
1.1456579618890503], [-2.1704530234462736, -0.53461434416983855]], [[-0.43278972686043726, 1.7567306645494543], 
[0.052123855175702859, -0.88135338585501566]], [[-0.24180783526445257, 0.47319465392135379], [-0.53478385948399398, 
2.2669597530491354]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.26649649085528448, 1.5093354012819393], [-1.9828456745053744, -2.2856564293834012]], 
[[-2.5851066924566068, 3.0612170614087937], [2.2307144526623732, 0.39178141254406196]], [[2.8921278628701703, 
2.9166920740046631], [-1.4537870053062338, 1.0003616467032437]], [[-1.551536846355984, -4.2912563334316642], 
[-2.3770310211576051, 0.85940548774519154]], [[-1.9031152572343233, -1.8540009351844899], [4.4498988425895885, 
0.2643159464134639]], [[1.6090189490934739, 2.7226080637232783], [-2.7878122168425454, 
3.0340820602060266]]])+(1.-msk_arg0)*numpy.array([[[0.65790016904571846, 2.2441974122805082], [2.9459864875506199, 
-2.6843186032707034]], [[-1.5542268002015014, 0.90650448212054435], [0.52824009989401066, -0.73154619910858543]], 
[[1.0218880794859686, -4.3975888600675663], [0.73500173162997839, 2.2318184332316697]], [[2.5721544299370329, 
2.7206293523978449], [-3.6452995752840858, -1.5054018904898001]], [[4.5454989412834621, -0.057354257803915232], 
[-2.7465293500359964, -0.61818185208107046]], [[-2.9044532538419823, 2.7292650084948527], [1.3619828819530326, 
-4.85677127465644]]])
      arg1=Data(numpy.array([[[3.1025320815170261, 2.8045235537162005], [-0.74073187983458588, 2.7124118434597113]], 
[[-4.3499804353917089, 4.9001260593223055], [3.3326887401303136, 0.54276379705855682]], [[3.5123074056179, -1.412227479202155], 
[4.2148557562470739, -3.6089397626027022]], [[4.0481781407778623, -2.1369531767207905], [-2.8532365262925552, 
-3.3847333647044628]], [[4.0797999942341452, 1.5211233510763336], [-4.9487185167503691, -2.890094796201085]], 
[[4.1756911899455389, 4.0893781777918896], [-2.9052028503497795, 2.9255489107836263]]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.085896449691175256, 0.53817890004238278], [2.6768736819430088, -0.84266570170554234]], 
[[0.59428007340539268, 0.62472210395178374], [0.66934377213251206, 0.7218267221713649]], [[0.82342674739808974, 
-2.0653132140244592], [-0.34491975274634123, -0.27718989855951515]], [[-0.38326792754674927, 2.0081190267428823], 
[0.83309988472854613, -0.25390640713592233]], [[-0.46647268491689226, -1.2188366800579422], [-0.89920225357890504, 
-0.091455805104004459]], [[0.38532996716034917, 0.66577556424321316], [0.95959296491358581, 
1.0370983882793225]]])+(1.-msk_ref)*numpy.array([[[0.21205265626907846, 0.80020629860882464], [-3.9771293335025533, 
-0.98964270847852709]], [[0.35729512426221882, 0.18499615543480838], [0.1585026808934325, -1.347816864487116]], 
[[0.29094494344414984, 3.1139380339433744], [0.17438360269876166, -0.61841387777060663]], [[0.6353856822720727, 
-1.273134751867947], [1.2776016084515514, 0.44476232786544589]], [[1.1141474944133229, -0.037705198439910784], 
[0.55499809511079956, 0.21389673892138278]], [[-0.69556227262099413, 0.66740342659346641], [-0.46880818727995299, 
-1.6601230821177841]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[1.4169387775852282, 1.8322447655949503], [3.8558811619368285, -2.4881967265222515]], 
[[1.9708523978157677, 4.8942502839083026], [-3.6158368629178437, 4.3500942785700509]], [[3.1281852996265656, 
-3.8418868417635044], [-3.2040770337961435, -4.3306282562981968]], [[2.070936345092024, 2.7082797210785134], 
[-2.5664594874154414, -4.0061091254755734]], [[3.8180671077391874, 2.7355704227097206], [1.6241485183295721, 
-1.2362835486085677]], [[-3.7670462041456352, 4.8050117877490326], [0.5633165318004183, 
2.8702086935696514]]])+(1.-msk_arg0)*numpy.array([[[-0.2936164944911237, 4.3957149920680987], [-2.0616284256566866, 
2.4742508432386927]], [[1.8711305488163923, 0.12358735660955222], [1.2737073828140684, 3.28123084953776]], 
[[1.0906335246896006, -2.794054677233798], [1.4393455405054318, -4.9956240139537709]], [[-4.2063765875342227, 
1.1161961734723356], [-3.0317919198335863, -1.6728193408844261]], [[4.0716375368357678, 2.7141676077356047], 
[2.0499235034099268, -0.024506696142401196]], [[-0.76312345231665279, -4.8145170445426686], [4.7609303949493409, 
4.3072486927384102]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(3.48039380559)+(1.-msk_arg1)*(-1.39247337961)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.40712024464278429, 0.52644754241656633], [1.1078864569127806, -0.71491815740089926]], 
[[0.56627281506191318, 1.4062346266812602], [-1.0389160149376779, 1.249885651325602]], [[0.89880211101451446, 
-1.10386555555649], [-0.92060761303778404, -1.2442925996885756]], [[0.5950293158680845, 0.77815324137408282], 
[-0.73740491185020274, -1.1510505273974012]], [[1.0970215788817654, 0.78599450967750606], [0.46665653631515402, 
-0.35521369639916944]], [[-1.0823620585965614, 1.3805942821846322], [0.16185425077345411, 
0.82467929030309617]]])+(1.-msk_ref)*numpy.array([[[0.2108596823396223, -3.15676770301818], [1.4805514100620338, 
-1.7768747894700632]], [[-1.3437460106701027, -0.088753837896772547], [-0.91470860517996333, -2.3564047238409396]], 
[[-0.7832347394636715, 2.0065408202019972], [-1.0336610822015737, 3.587590317417694]], [[3.0207949747016141, 
-0.80159246834885778], [2.1772710087146341, 1.2013294942499553]], [[-2.9240325857961187, -1.9491701941864974], 
[-1.472145560141922, 0.017599400104330927]], [[0.54803449997006159, 3.4575289661107727], [-3.4190458967911255, 
-3.0932359324112531]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-1.4794767507221529, -4.6259085944934162], [-4.714539904649647, -0.60729029544023039]], 
[[-0.41013362064693748, 0.23166217875705275], [0.23963351942977518, -2.0866210934145135]], [[-1.4345570955142106, 
2.9423035343361557], [-3.9731424252799452, -0.067318376357377474]], [[4.8958897780869055, 0.38951748769881522], 
[-4.3741360961544764, -4.4419013591832632]], [[-4.1070673133833875, -2.2048166888896237], [-0.74250273171900805, 
3.3075305424706851]], [[-2.5359068704175058, 2.7191573933612787], [0.31683580992296001, 
-3.8740638165389485]]])+(1.-msk_arg0)*numpy.array([[[-1.7348049813750488, 1.4265849763537686], [3.2440465059279493, 
-0.89205993029639252]], [[-0.39868691307929716, -3.7399398312288001], [-1.6603128099016731, -3.2218895772682821]], 
[[-1.2655351838140083, 4.088954192079699], [-2.5524229443856319, 1.748800612027658]], [[-4.2560490822723684, 
0.59793531158506497], [0.96504303166002092, 3.8629561428073504]], [[4.0190342316555334, 4.5365625248117816], 
[-1.299979243920486, -3.5011542463884049]], [[4.1200666234849876, 2.5999865773923156], [-4.5429404912610272, 
0.85946312806513347]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.12000502740195884, 3.2116467512656559], [4.0873682744459252, -0.082035687707444183]], 
[[-1.7756948880308734, -4.3327563020846256], [-2.8097713161839022, -2.8441144727581626]], [[3.6589419575442115, 
-3.5360957246077973], [2.0281330527446473, -4.2350281630836362]], [[-4.4588584980183334, 3.0932715066444185], 
[0.25088020854809479, -2.5152914329246698]], [[-3.9546750675754949, -1.8495163482017585], [0.59332683006717435, 
2.1512330662465295]], [[2.4903282692185824, -0.52333580617934494], [-1.1580060363778522, 
-3.9316605815590231]]])+(1.-msk_arg1)*numpy.array([[[-4.3659957160061422, -2.5446795184857218], [0.43627694546538276, 
4.4687148420096214]], [[2.6091301099752675, 2.2990440880563447], [-1.3340648005302946, -3.4290053978022392]], 
[[-1.0625887885298146, 1.039808027240726], [-3.0300256828592698, 2.2817065572724236]], [[3.8908685572268364, 
3.7092259317888541], [-1.1099588033724661, 0.4306965756804928]], [[1.727156730185877, -3.2617139966883557], 
[4.5803500404832409, 2.6690507036508127]], [[4.9100923905360876, 2.0233139773213367], [-3.6652940712146567, 
-4.1077313795048855]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-12.328456421801571, -1.4403541088914662], [-1.1534414293237962, 7.4027574146260609]], 
[[0.23097077285712533, -0.05346762259525023], [-0.085285773276109186, 0.73366283720322689]], [[-0.39206883086963445, 
-0.83207689030038867], [-1.9590146809663898, 0.015895614802325465]], [[-1.0980141622038024, 0.12592411848171836], 
[-17.435158083886623, 1.7659589266832656]], [[1.0385347072019548, 1.1921044607328373], [-1.2514228146988473, 
1.5375045104906568]], [[-1.0183022462388964, -5.1958176017282396], [-0.27360462723838336, 
0.98535052458744143]]])+(1.-msk_ref)*numpy.array([[[0.3973446366461364, -0.56061479097481604], [7.4357504783285746, 
-0.19962337312515227]], [[-0.15280453495018553, -1.626736890631191], [1.2445518457886708, 0.93959886424596928]], 
[[1.1909924116223622, 3.932412604017208], [0.84237666988256343, 0.76644413649676013]], [[-1.0938557855847511, 
0.16120218142028836], [-0.86944040511040821, 8.9690895190052284]], [[2.3269655621948124, -1.3908523339010685], 
[-0.28381657131674903, -1.3117601106638457]], [[0.83910164937551324, 1.2850138962784388], [1.239447750438077, 
-0.20923060654680067]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[2.4185387379138099, 1.2685311495071563, -4.7501512725206947, 0.17628918569836838], 
[-3.5054549087070139, -1.6083740488565255, 4.5221303013643954, -2.0211228577830909], [2.1459831270294751, -2.1645236298733419, 
-2.8203042164229419, -4.3470870611186072]], [[-2.249928788711709, -1.0141682903059088, -4.2748324811752738, 
-4.4784523748604208], [-0.38162030434898764, -1.6037004160376434, 3.1382813476108247, 1.9921570647651148], [-0.306563754538133, 
2.8464098659381909, 1.624901967890402, -1.7839407950412198]]], [[[2.890536303458358, -2.7011236020822516, 
-0.082730425616652781, 2.337117009185647], [-0.22328143409960433, -2.8879466100907925, -2.6908495506392081, 
2.5326672200999658], [-2.7653662614128418, -4.1055643611883808, -4.6029561663533682, -0.51950590414493103]], 
[[3.2115408240443006, -4.8776127386315506, -3.1205274819776352, 2.4936507153495615], [-1.2856411394585923, -2.7331345901782078, 
-4.2390421196319581, -4.0856833254920879], [0.30966211121511744, -1.18985642586214, 4.0651185250409085, 1.7118053143815093]]], 
[[[-0.097207692262607281, -2.2532705115550211, -1.9510996099010978, -3.3144535998672051], [-2.4395614903178786, 
4.3902274968106738, -2.673872565544626, 0.59761798268035626], [-3.6552588962986499, 1.6796745368663206, 2.9062110504796523, 
-0.89079130202148171]], [[-3.9205950996743089, -3.8528670066430228, 2.7362312840214029, -3.8975086411827831], 
[4.3748865367724274, 1.258900970863098, 4.1803633196942087, -0.064324217625283175], [-0.34470130716176772, 0.937245766154847, 
1.7198749192753846, -2.5213180909011932]]]])+(1.-msk_arg0)*numpy.array([[[[-4.3860797614854814, 2.4040746118035541, 
4.5943911504560369, 3.7059504574433362], [1.9007301003120158, -1.0368736549257185, 0.68846282175746687, 3.6371373860569456], 
[2.8010135010980717, -1.9971046939894355, -3.4883428479687639, -2.2346305725175375]], [[-0.1818849783315768, 
1.6034238580015483, -3.0263024805442598, 2.1255938225732729], [0.35178748702258478, 4.4675113098687351, 0.087260214312334661, 
3.6638887189675042], [-3.2805868578994413, 1.9610536636700164, -0.18088454253811115, 4.1617143930926783]]], 
[[[-3.4377372276986708, -0.316774190723911, -1.8183213612026607, 1.7508577518530952], [0.30282095049522439, 
-3.6342230336175216, -1.8075628843730418, -4.0210748230225342], [-4.7863479091143768, -2.6513233246583203, -1.5719139196922862, 
2.8156778647540506]], [[-0.54872406077354441, 1.2690466639162556, -2.3991255874489092, -2.7467086409972263], 
[0.74205627071120528, 4.197709455883242, -1.475738862141772, -1.5578459428213542], [-4.3801157065029219, 3.0459427212426444, 
-4.1481233542108198, 4.4063881655278454]]], [[[-3.2814108117364094, -0.79416059818488272, 2.990321353989053, 
-3.3141383402096238], [-2.1854020057976777, -0.95404511666949166, 2.7321308894570926, 2.3525180966587005], [1.8327521802742393, 
0.77162693539527982, -3.7128459235700149, 0.4905617684002781]], [[0.56325925350634876, 1.2168012328050484, -1.3204229102324536, 
-1.854242796754547], [-2.1900879985795383, -3.0591256981621284, 4.6314628664803283, 3.6607751136727362], [-0.73451000015987944, 
-0.63506234891406965, 1.0431443044635502, 4.1923050416544534]]]])
      arg1=-0.992479481144
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-2.4368652288166115, -1.2781434514343175, 4.7861455705313487, -0.1776250179954976], 
[3.532017513013288, 1.6205615122668062, -4.5563967691823652, 2.036437927616769], [-2.1622443262563253, 2.1809253198648735, 
2.8416750875010672, 4.3800271377981295]], [[2.2669776367751653, 1.0218531562354218, 4.307225048368486, 4.5123878729444025], 
[0.38451203435369652, 1.6158524649689268, -3.1620616921907225, -2.0072526461391234], [0.30888674311408526, -2.8679785527224917, 
-1.6372146717002134, 1.7974586164591477]]], [[[-2.9124393585713975, 2.7215913813842287, 0.083357315882531383, 
-2.3548265264806103], [0.22497335042355807, 2.9098300417877594, 2.7112394782585332, -2.5518585202190711], [2.7863208398275798, 
4.13667429824942, 4.6378350926183378, 0.52344246305838316]], [[-3.2358762927193205, 4.9145728766192622, 3.1441732965407438, 
-2.5125463676842594], [1.2953830924309391, 2.7538449329227026, 4.2711634851590778, 4.1166426138932097], [-0.31200857760623313, 
1.1988725696279463, -4.0959219835508325, -1.7247765287886188]]], [[[0.097944284098001932, 2.2703446815421366, 
1.9658840781797227, 3.3395688906804173], [2.4580472812453791, -4.4234944703845311, 2.6941338499638174, -0.60214643630867781], 
[3.6829566411642891, -1.692402280125914, -2.9282328810764939, 0.89754127812766271]], [[3.9503034311148575, 3.8820621280773207, 
-2.7569650919809288, 3.9270420348541362], [-4.4080372641352268, -1.2684403000573665, -4.2120400462844412, 
0.064811634746481092], [0.34731328325743821, -0.94434775122453507, -1.7329072811590089, 
2.5404233929298994]]]])+(1.-msk_ref)*numpy.array([[[[4.419315305571561, -2.4222915007093229, -4.6292051752648238, 
-3.7340323178989401], [-1.9151328933481515, 1.0447305708834682, -0.69367965266534748, -3.6646978150767051], 
[-2.8222381966722168, 2.0122377660517463, 3.5147757855384536, 2.2515634982616546]], [[0.18326321277888821, -1.615573811312756, 
3.0492343046285248, -2.1417005217308152], [-0.35445315868604543, -4.5013639019711444, -0.087921429077557289, 
-3.6916518563630021], [3.3054455232847242, -1.9759135588471162, 0.18225519617756517, -4.1932498073365947]]], 
[[[3.4637867008958567, 0.31917454893754527, 1.832099701554238, -1.764124886325009], [-0.30511557795246969, 3.6617613791152634, 
1.8212597023058728, 4.0515445401324497], [4.8226164873426054, 2.6714137420777448, 1.5838251062685562, -2.8370136796263585]], 
[[0.55288202043334667, -1.2786628721567543, 2.4173049750947677, 2.7675218411882234], [-0.74767920628026141, 
-4.2295176229181042, 1.4869212816779918, 1.5696505292236826], [4.4133060579287937, -3.0690233693613753, 4.179555782281092, 
-4.4397775966598916]]], [[[3.3062757206365307, 0.80017835458871556, -3.0129805308842741, 3.3392512421410054], 
[2.2019619018003205, 0.96127439891228394, -2.7528336266537226, -2.3703443157806983], [-1.846639870238576, -0.77747394284246096, 
3.7409800344590152, -0.49427900296217092]], [[-0.56752735367091633, -1.2260215509971624, 1.3304284222686591, 
1.8682933319862545], [2.2066834027188675, 3.0823062403626862, -4.6665578024260013, -3.6885146577065835], [0.74007575382134771, 
0.63987453743833722, -1.0510487363034762, -4.2240722567100706]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.2059102872332144, -0.71734981156018041, -0.84843643023961235, -0.71048410923699912], 
[4.7973431156782418, -3.3476380281973093, 4.3690523174717537, 4.5200028107663766], [-4.446953359503711, -3.0684536322605216, 
-3.9836098095740513, 4.5936827600883525]], [[-4.2025519589624807, 4.3503105189885751, -4.024979543315176, 4.2367393229172006], 
[2.825944314146839, 3.5103433547218312, -4.4383168064378964, -2.761115771683186], [2.1490229737022259, -4.9245377132839163, 
-4.6782497650435317, -2.6597782573883091]]], [[[2.0330340571428991, 4.1698521532898898, 1.432192410518228, 4.9449211293981605], 
[0.44631652198213967, -2.6843785916289633, 1.0207463820279656, -2.6431473283648046], [1.6899596087625577, -3.8196216736752331, 
4.785285365765354, -1.0461173749142452]], [[-3.0189551976712026, 3.9131509939516462, -1.2633569088227636, -4.6816466457746007], 
[-3.1275606196218861, 1.8343881727716234, 3.0482340223892841, 0.45078480233680196], [2.6684955823633754, -0.53766295833425026, 
2.756040796790769, -0.65752175719904304]]], [[[-3.9725523514167236, -3.7490951049543222, -2.3312596595991755, 
4.1687700972037298], [-3.4904005594516097, -0.54337574073166017, -4.106720246603059, -3.121948677955694], [3.3835340016702311, 
1.2456610615016626, -3.175765375128714, 0.69189596958275779]], [[3.4597693411436268, 4.2537109751910531, 1.5637399072750933, 
-4.389049186079836], [-1.8866612687801556, 0.91925776349873978, 0.19170832321077569, -2.5014727205642959], 
[-0.64459958292088793, 1.4093494933266388, 3.7079144402323614, 
-2.7971464916320476]]]])+(1.-msk_arg0)*numpy.array([[[[3.9925316225264993, -2.5169488075507584, -4.8805748944879346, 
4.7982242699922146], [-4.4784912357670876, -1.5579844934712828, -1.7542855731691676, -0.10563296685459544], 
[3.7469132828194915, -1.4587774039976931, 4.4526728287806723, -3.4074439804714149]], [[-3.0300099737072839, -1.850468030414143, 
4.3229931030301003, 2.4128481937562798], [-2.4259247098600922, -2.680601896125391, -4.6671594734101385, 3.830824182236972], 
[1.3300213574478104, 1.8646482980910806, 4.7365809196629751, -3.3846295680760208]]], [[[4.2245880702387755, 
-0.074543223867538977, 1.6067112318400421, 1.3394545706283321], [-2.1241620451546996, -1.3931043968900312, 1.635166150642914, 
-4.6051322273002349], [-1.6087229853537002, 4.3764208749412798, 2.3112308780550954, -2.6353122734798928]], [[1.625191395398204, 
4.4742027826108437, 1.8104978536182141, 0.61796506344500823], [3.6759728598983941, 2.818147103030217, 2.4822978775132096, 
-0.26436192428962002], [4.7307336533963689, 3.2191537015637053, 0.93822435390806902, 3.3923676235748506]]], 
[[[-1.9768427504401274, 3.3144912024043034, -0.18567599276828606, -3.7254510422553491], [-1.2958838723965114, 
-4.4848964018822937, 3.8268440783323232, 1.0005196433062853], [-3.1570924735975581, 0.21142558194465888, 0.29260438324785376, 
-2.6579555214793213]], [[1.0390532909873906, 4.465268305535794, 2.7276130899470665, 3.8353379022479483], [0.97476899470072009, 
-2.2407360317396163, 2.7174240901398496, 1.5949633712418656], [-3.0859609209548711, 4.1306756585329616, 2.6124219437165417, 
-4.3881731546070366]]]])
      arg1=numpy.array(-1.476925952)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[2.8477462133596756, 0.48570465607068891, 0.57446104802217912, 0.48105601250540825], 
[-3.2481947447449908, 2.2666255025541702, -2.9582067479701823, -3.060412612177362], [3.0109521425014609, 2.0775947691190302, 
2.6972305579488767, -3.1102999807498897]], [[2.8454723496861614, -2.9455170132832516, 2.725241260641845, -2.8686199989696766], 
[-1.913396071286847, -2.3767903529339103, 3.0051044877466011, 1.8695018311080009], [-1.4550648059137077, 3.3343159192243559, 
3.167558778890994, 1.8008880227059303]]], [[[-1.37653079653977, -2.8233318993621541, -0.96971172357999447, 
-3.3481171636852447], [-0.30219289015567702, 1.8175444665901126, -0.69112901743154731, 1.7896275197667451], 
[-1.144241257639911, 2.5861971404127702, -3.2400306591342507, 0.70830726042460124]], [[2.044080269274172, -2.6495241610731353, 
0.85539624184150354, 3.1698587457420464], [2.1176150472386928, -1.2420312408229228, -2.0639044349192712, -0.30521828242304028], 
[-1.8067903666678842, 0.36404191936945668, -1.8660656568803562, 0.44519615645366922]]], [[[2.6897437518958576, 
2.538444869132551, 1.5784539884584563, -2.8225992586466928], [2.363287444922066, 0.36790994158794149, 2.7805864207542719, 
2.1138153024657229], [-2.2909300206135934, -0.84341470187527501, 2.1502536202432547, -0.46847031744803974]], 
[[-2.3425475978996668, -2.8801111994938591, -1.0587801677889508, 2.9717462680665983], [1.2774244140132169, 
-0.62241289906996655, -0.12980225782517923, 1.6937021908034113], [0.43644678465163833, -0.9542451951730776, 
-2.5105621816721051, 1.8938975835829048]]]])+(1.-msk_ref)*numpy.array([[[[-2.7032713570426568, 1.7041807709688279, 
3.3045494852766568, -3.248791358483667], [3.0323058713194664, 1.0548832806119062, 1.1877952112550598, 0.071522182077737292], 
[-2.5369675966045211, 0.98771194454123168, -3.0148246922858282, 2.3071190372455606]], [[2.0515652593111322, 1.2529186232409084, 
-2.9270208822338915, -1.6336961175891525], [1.642549991465541, 1.8149873339874614, 3.1600497418818319, -2.5937821574859252], 
[-0.9005335410639681, -1.2625198274567955, -3.2070537546142379, 2.2916718089234966]]], [[[-2.8603926043180121, 
0.050471876241576133, -1.0878752788245651, -0.9069206000547857], [1.4382319183110583, 0.9432459325393886, -1.1071415925921946, 
3.1180522090835741], [1.0892374009478361, -2.9631958657113433, -1.5648928606870214, 1.7843225450154527]], 
[[-1.1003878652091983, -3.0294022368140099, -1.2258555353848903, -0.41841303052894008], [-2.488935112088408, 
-1.9081167198704541, -1.6807192494277363, 0.17899470446087015], [-3.2030946757865753, -2.1796310757454118, 
-0.63525483632727364, -2.2969111071349797]]], [[[1.3384846733563354, -2.2441823829461627, 0.1257178753723536, 
2.5224358995109633], [0.87741966388890114, 3.0366426941009426, -2.5910873007131485, -0.67743385641536391], [2.137610534443446, 
-0.14315245910448088, -0.19811716548872627, 1.7996538810033527]], [[-0.70352429624331148, -3.0233528630705773, 
-1.846817767840881, -2.5968383161273589], [-0.65999855536278407, 1.5171620680770015, -1.8399189793177402, -1.0799210136955431], 
[2.0894486394302594, -2.7968061993405926, -1.7688239144088977, 2.9711531229115482]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[3.3792000952713277, -4.5445885020409733, 1.7935186794947615, -4.2025730596973752], 
[-1.9109232593339454, 1.2501002877672782, -0.54422266784979634, 2.9275558402449606], [4.7399035604715909, 2.7186459336110929, 
-2.7234364893613794, 0.35800235248045542]], [[4.8049873205716924, 0.61426787565058039, 0.73139614145182819, 
-4.0637505927473425], [0.37154944454776917, -3.4440089976819266, 0.90467569274409687, 0.21054648717547675], 
[-2.7602327829182283, 3.851683380347998, -2.2717022344303919, 2.6661424595637087]]], [[[1.4930725592599758, 
0.04028701897898479, 3.0348835730949428, -2.4359770998125221], [-3.7868322278366682, -1.7364201866216868, -2.6826504957662092, 
4.2826261165974433], [1.9537533723665588, 3.1973248355086028, 4.4988710247063342, 3.5038224323967064]], [[-4.5943358065613786, 
1.1907796548530714, 2.8545305191119326, 1.6459993070052379], [-1.3976336088222432, 1.0884902414247497, -0.37941579586446927, 
2.2123805554991591], [1.4459617578099593, 2.1082754179799998, -0.3284052035859375, 1.1813543920410998]]], [[[2.610059886251511, 
-3.3037726703640704, -2.7784863549209904, -0.85553794516812509], [-2.966018907621085, -2.7438644107836008, 1.6325781914134492, 
-2.0202836177615477], [0.8091463870133575, 1.1051711756222833, 3.6531581288910431, -2.7355026914568192]], [[3.8519302543686038, 
-4.6786597285668732, -1.9147401776047532, 3.1203428459744647], [-2.9835884961231751, -3.4608878081333927, 0.69178613786853216, 
-0.89941302029266801], [-3.3449616569144713, 1.424554420259545, 4.6752896803041448, 
2.2277709623165016]]]])+(1.-msk_arg0)*numpy.array([[[[-2.581327832508995, -2.0885571453455709, 4.2367002563295166, 
0.64856111983822018], [0.20412431042582302, 0.74268852183423473, 3.7593362963191321, 0.55044211064954851], 
[-2.5436262130690359, -3.5557086482447122, 2.2496903771273002, -1.7930387795659652]], [[-1.0153346353332768, 
-0.65651188640657221, 3.6525015889751522, 1.0090649813408348], [2.1642900471036679, 0.0075294509616057326, 
-0.22132794012458579, 3.1230314534884496], [3.8650912404422062, -0.48640860483079784, -3.1787057933675023, 
-0.28069132716487832]]], [[[2.5922912439764279, 3.8385566080860389, -0.19176443344859706, -4.2339330577703898], 
[-3.2490225873694056, 3.3645742342682041, 0.14374287532059071, -0.039814977263720408], [0.019121734191891449, 
1.5781909152238827, 4.0179247588537699, 2.1697426653454972]], [[-1.2165164189996815, 2.6751465246934627, -2.9684563070817682, 
-2.5568004486701312], [4.9881784941533507, -2.2428203299975147, -1.3661270914025225, -2.5803088489560611], 
[-3.4631517014792501, -3.1871230459494013, 2.428962927803509, 1.8233057844272214]]], [[[-0.21122833699425225, 
-0.7314523666208208, 0.33235470228426323, -3.1857707034195482], [-4.4325501033687456, 4.4166842378089637, 2.353618866580355, 
-0.16201358393804099], [4.2846064002175588, -2.4581931196297502, 0.076708823613843791, 2.3756003748097925]], 
[[-1.9997520805008948, -4.4975698400383344, 3.2607739354592393, -0.47097439867482116], [4.5175377533863159, 2.8665210868107867, 
-2.8720243238841539, -4.4607396928754781], [-2.0615505431459855, 4.9146906925065608, -4.6486117012407675, 
3.7007749242174945]]]])
      arg1=numpy.array([[[[-2.7744609249357142, 3.5478422153671687, -0.064940259202338702, 0.94298542826741194], 
[-0.30974626529347304, 0.30231145677814908, -1.2769602665344868, -1.1853628172878183], [0.84451145611182277, 
-0.50915278016492493, -4.7724275819044992, -2.5682972901729064]], [[-1.6365237004006392, 0.81940819097714357, 
1.3731379975036138, 1.8614989361465977], [3.1904931615945191, -1.5976130686224232, -4.5548769931043047, -1.8656201055824315], 
[1.6718730312569265, 4.1265463986903583, -1.9604622114363268, -2.941039244290736]]], [[[-1.748289536679545, 
-1.0275252426263148, 2.4909950975917674, 4.77508418171778], [-2.2422989818060604, 3.8969487004067513, -0.10181703332621783, 
-4.2419628490715624], [2.2091118749627938, -1.2948003785424267, 2.8468007469624741, -2.7094255166235834]], 
[[-4.4023906662114163, -0.27975457704053674, -3.5778770506448954, -2.9113445762363996], [3.0665708428057297, 
-4.2195843812867455, -3.847939613615238, -0.6083003907233735], [-1.9220672367481582, -0.626300639784235, 3.3951740296326847, 
4.985985210990604]]], [[[0.91496132760287008, -4.3672373048629325, -0.55062957851863903, -3.2228366728210647], 
[3.2396579196220614, 2.328724902597906, -3.0531334202518066, 3.8897723096312227], [2.4307898639185037, 3.0391813572211195, 
-2.6668202021775067, 3.4579661811379854]], [[-4.0223713879102743, -0.88214896673051246, -0.61185225845521529, 
3.2645514180210888], [-2.1249634381493889, 3.2371004049270979, 4.0298159403119591, -4.095571663209375], [1.1403900544436141, 
-1.0795209014115859, 0.56139380772926195, 3.7726974334803938]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-1.2179663677727326, -1.2809443673555958, -27.61797845472984, -4.4566680817315971], 
[6.1693181595697908, 4.1351402989819963, 0.42618606241112716, -2.4697550805106112], [5.6125982971200505, -5.3395484410994829, 
0.57066062137595741, -0.13939287863997765]], [[-2.9360939407082087, 0.74964820026764256, 0.53264576669025088, 
-2.1830528687593684], [0.116455176591596, 2.155721598253824, -0.19861693172256872, -0.11285603459432371], [-1.6509823002785473, 
0.93339151149988431, 1.1587584913284485, -0.90653073220268376]]], [[[-0.85401904429154663, -0.039207814375452933, 
1.2183418490180866, -0.5101432785497424], [1.6888168163848352, -0.4455845637486695, 26.347757424547041, -1.0095859555994888], 
[0.88440671317266994, -2.4693573530677155, 1.5803252227984741, -1.2931975471918786]], [[1.0436002060932827, 
-4.2565153623224763, -0.79782800770010165, -0.56537426742288288], [-0.45576433106090963, -0.25796148223793047, 
0.098602325910202737, -3.6369869052167783], [-0.75229509673985384, -3.3662354531624232, -0.096727060445106802, 
0.23693499720718006]]], [[[2.8526450326481796, 0.75649030261884531, 5.046017256094312, 0.2654611548835461], 
[-0.91553459692654859, -1.1782690208373554, -0.53472219084313799, -0.51938351578040942], [0.33287385266161595, 
0.36364107492183301, -1.3698554277893102, -0.79107271389120126]], [[-0.95762670397518435, 5.3037070891861697, 
3.1294158861798222, 0.95582591493258184], [1.4040658029963824, -1.0691320549914591, 0.17166693171971004, 0.21960622209888747], 
[-2.9331732979260745, -1.319617265767427, 8.3280036507970401, 
0.59049817845088504]]]])+(1.-msk_ref)*numpy.array([[[[0.93038896648681613, -0.588683774125909, -65.239965290698123, 
0.68777427561086457], [-0.65900491239958237, 2.4566998874252253, -2.943972803884892, -0.46436593304739665], 
[-3.0119499204664804, 6.9835789703297815, -0.47139329796378643, 0.69814300175711042]], [[0.62042158942440717, 
-0.80120249423389622, 2.6599668755911328, 0.54207121032776584], [0.67835595861989451, -0.0047129377628953462, 
0.048591419803357461, -1.6739911004086572], [2.3118329969928411, -0.11787304875214036, 1.6214063065457573, 
0.095439504151387189]]], [[[-1.4827585417572542, -3.7357297405899601, -0.076983063368526977, -0.8866719196241023], 
[1.4489693897789131, 0.86338684261279097, -1.4117763072122138, 0.0093859797174873187], [0.0086558469078047481, 
-1.2188681293099963, 1.4113825012659846, -0.80081281143663796]], [[0.27633086457694683, -9.5624763426331043, 
0.82966973572966074, 0.87821979903711689], [1.6266307709328718, 0.5315263607345081, 0.35502820433271071, 4.2418332920806305], 
[1.8017848883051404, 5.0888069458900569, 0.71541632523216858, 0.36568615975998275]]], [[[-0.2308603988188819, 
0.16748628836961671, -0.6035903541150085, 0.98849896126784753], [-1.3682154762456671, 1.8966105583711275, -0.7708863461283787, 
-0.041651174166901561], [1.7626395698847654, -0.80883396898610982, -0.028764152735609869, 0.68699352462377294]], 
[[0.49715749433565298, 5.0984244267808529, -5.32934853209815, -0.14426925429170209], [-2.1259366972076461, 0.88552121597703204, 
-0.71269367296756059, 1.0891616750224191], [-1.8077591391761105, -4.5526591343253209, -8.2804826794288573, 
0.98093605158350872]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-4.5875330208559806, -4.6055899737507131, -3.2930471305979636, 2.4015916183060861], 
[0.65669353984742518, 1.5763779214874232, -1.5052567607100786, -4.1022082927050736], [-2.8992950704496545, 0.55729130669571081, 
1.3433891553823587, 1.7842171752410865]], [[3.7763444085916689, -1.7394905036006914, -4.0404937602395377, -4.094288158896652], 
[-1.075670136670098, 2.0972460496029051, -2.990253090995024, -2.2958162339573693], [4.9323675337652624, 2.5308002267177852, 
-2.0959515903747072, 2.8434569073551073]]], [[[3.6074951802702735, -3.7887488392008031, -4.3310741686734033, 
-1.0679935356894199], [2.9229191193537174, 3.4506785544639911, 0.18537294248145653, -2.4471303110276321], [0.17401061576460286, 
4.8527410554411183, 0.24824971703591814, -2.409306034171419]], [[-0.27793264573433873, -3.7692515185245332, 2.8606541013164737, 
-3.4090729847536885], [-3.9374360637694563, -2.2330671432015636, 0.41848143651316416, 1.6868561694868758], 
[-0.88402841147285649, 2.9588540220284463, -1.8424718164857157, 3.9146513622767412]]], [[[0.57757744342594641, 
1.8236907949632206, -4.2635331341559599, 0.17321638916328119], [1.7925241982029503, 3.8324301746936786, 4.3836339342806845, 
-4.7114573810914093], [0.55346416108290342, 0.85869826975652952, 1.4450014745073734, 1.9945262904299819]], 
[[-4.1847960139738483, 4.6535730912455779, -4.1489897896382768, -2.7617343195454058], [-2.2770942487021251, 1.9511791465535833, 
-3.020685591999912, 3.3752461023617286], [-2.3087696268227544, -4.9218301094154882, 3.9690490825967029, 
-4.1622633977484371]]]])+(1.-msk_arg0)*numpy.array([[[[4.6187119792047078, -0.031439626842652224, -0.45269370328937697, 
4.0919720207087824], [1.6245484160056485, 2.344290265834057, -2.9485539433091956, 0.41915953092474467], [-1.3992838553068365, 
2.9640685295547424, 1.3475012955024335, -1.5677010609640787]], [[2.5174506029482639, -3.7440402004138971, -0.18941277014536162, 
-0.55133582291501426], [-0.819750651699791, -2.882187702052883, -2.5000610289102267, 4.080250473178392], [2.4976931512143263, 
-0.41337333557102252, -3.0934450971498153, 1.0946452318588964]]], [[[1.3054257672423928, 3.2420044962083754, 
-4.3179001470272933, 1.2893412620515914], [-1.0098669694974127, -2.9499745058105109, -1.3569725632975773, 1.0488097864020318], 
[1.6747702233964459, -4.0794978405737794, -3.5980332960710983, -2.7501291370791572]], [[-1.1621213811632214, 
2.3068403078212008, -1.2054512154081722, -4.4290663655960429], [4.440364537417766, 1.7654316997296702, 0.16480279185141278, 
-1.4436256083700885], [-4.4556918726946915, 3.0456634999165573, -0.45637337592914484, -2.4189445144769071]]], 
[[[1.2928885896566014, 3.1625725809117284, 4.2782205317343074, -4.2005661078806922], [-0.076045925362638656, 
-2.4932070619143287, 4.1724265207496565, 2.8268008962115498], [2.9687672545317305, 1.7398903719563039, 0.13298114867756361, 
-4.3920278719040509]], [[2.8626928973001, 2.5599335659173246, -3.8529076297440779, 1.8576423550881982], [4.4738890391641313, 
1.3720837313104255, -1.5441903555059566, -0.45752289796504453], [4.9812315599359085, 4.7429136721842227, -1.3422538959085442, 
4.9999869992612069]]]])
      arg1=Data(-2.18905449681,self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[2.095668713380777, 2.1039174586363307, 1.5043239605924019, -1.097090831590894], 
[-0.29998958034327922, -0.72011817146746215, 0.68762872870534897, 1.8739635302263438], [1.324450841525878, 
-0.25458082816445343, -0.61368465579012543, -0.81506293143404873]], [[-1.7251029675551719, 0.79463097247390546, 
1.845771206757288, 1.8703454687208154], [0.49138572759895521, -0.95806022767203025, 1.3660021234504733, 1.0487707077645001], 
[-2.2531954051152034, -1.1561156793504295, 0.95746889509895705, -1.2989429507100727]]], [[[-1.6479695619833878, 
1.7307695375867518, 1.9785136345300771, 0.48787891632867614], [-1.335242737725109, -1.5763328685915119, -0.084681739422823143, 
1.117893736583826], [-0.079491221446501795, -2.2168205782481341, -0.11340499626545077, 1.1006149173898383]], 
[[0.12696469920644141, 1.721862806071087, -1.3067989424116981, 1.5573266858898569], [1.7986925722965503, 1.0201057792087387, 
-0.19116994899965742, -0.77058664914141151], [0.40384029395339238, -1.3516584563502276, 0.84167471352053747, 
-1.7882841052958935]]], [[[-0.26384790523346752, -0.83309520051625174, 1.9476596587081947, -0.079128404256487539], 
[-0.81885773095775116, -1.7507239679385203, -2.002523893609423, 2.1522796202430148], [-0.25283251828072068, 
-0.3922690234559208, -0.66010301553080863, -0.91113596912913764]], [[1.9116911068531577, -2.1258370214269, 1.8953341708391365, 
1.2616105828185065], [1.0402181636033092, -0.89133420359999627, 1.3799042446855534, -1.5418739493585318], [1.0546880537621648, 
2.2483817175779053, -1.8131339755933962, 1.9013977969980496]]]])+(1.-msk_ref)*numpy.array([[[[-2.1099118299382331, 
0.014362194677395253, 0.20679873614315963, -1.8692874145740404], [-0.74212333149813359, -1.0709145291939188, 
1.3469531926236975, -0.19147971488834031], [0.63921837366049372, -1.3540405384472647, -0.61556315636014869, 
0.71615442340363922]], [[-1.1500173278525758, 1.7103458163628162, 0.086527206344632096, 0.25186025460661476], 
[0.37447704152333483, 1.316635883779655, 1.142073453424892, -1.8639327979815996], [-1.1409917637276514, 0.18883647537002463, 
1.4131421130236146, -0.50005389699191738]]], [[[-0.59634228802575939, -1.4810067547101793, 1.9724955012833758, 
-0.58899459283853572], [0.46132564126099779, 1.3476021314707585, 0.61988980414755368, -0.47911542993977191], 
[-0.76506556864386022, 1.8635889816875291, 1.6436472007936818, 1.2563091239086686]], [[0.5308782320656078, -1.0538067056713052, 
0.55067209023955399, 2.0232782564548923], [-2.0284394673055921, -0.80648138376642875, -0.075284919626882121, 
0.65947449479763909], [2.0354412734731602, -1.3913146083622059, 0.20847967768441658, 1.1050179509002063]]], 
[[[-0.590615076755302, -1.4447208077809002, -1.9543691296691654, 1.9188951732342392], [0.034739165001767228, 
1.1389424363552496, -1.9060404968559184, -1.2913341811852352], [-1.3561870016732052, -0.79481363962823837, 
-0.060748212925344408, 2.0063583973349299]], [[-1.3077303015837034, -1.1694243197894671, 1.7600784427038649, 
-0.84860489210885981], [-2.0437540708456945, -0.62679286116822175, 0.7054143045567336, 0.20900480030588323], 
[-2.2755173830475903, -2.1666494274536681, 0.61316604856702173, -2.2840852096378401]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[1.0377322137304326, -1.609678202034861, 1.9422025900625792, -1.8130191803779558], 
[3.7559725236374977, -2.5730646055414885, -0.60495128618710226, 2.5925724572893314], [-4.8565109305778575, 2.9535892996820037, 
3.3429234498548954, 4.3025968794898581]], [[4.6298309197702956, 4.4253098591787179, 2.8991661867199889, -3.8266220997427269], 
[4.2602370964200063, 0.46938855997947471, 3.4644861757716328, 4.0153162375007714], [4.6911209792299164, 2.7859464332534962, 
4.742130413630683, 0.41860012138133307]]], [[[-0.56478548748022828, -0.91411155398952992, -0.87704710291543453, 
-0.96762506806391446], [-1.5483375569132507, 1.4026464899442637, -2.973878214149658, 3.3258746439473015], [-1.6898558925415252, 
-2.2665542934311644, 4.8066753484176488, -4.1168441986072546]], [[2.4783665733626155, -1.031566679533793, -0.7278666475873008, 
-0.5094265547431851], [3.8784271011746085, 0.0017778261546261831, 3.2434067700996447, 2.3882440939070797], 
[-1.8041339247396793, -0.50632322565062449, -2.0418371310694328, 4.7396662799925053]]], [[[0.81867670783924495, 
-3.3370439222198467, 1.0759187214669543, -0.78962926810881928], [-2.6764223184027038, -0.27448846396117332, 
-4.9442046006331042, -4.958882087228913], [3.633600362707373, 4.7310974322007873, -4.7959769532293341, -4.9985114855717079]], 
[[-4.6999456589679074, -3.2074975853782348, -3.3652982228133901, 4.9636709061761994], [-2.3824794877642397, 
-1.9058926014444233, 4.0631589675910575, -4.4439471442173222], [1.7873780001465223, -4.9780382950606157, -0.43499888569910272, 
0.45253773666197628]]]])+(1.-msk_arg0)*numpy.array([[[[-0.55288334969735953, -3.7992287138064174, 2.1844152624109743, 
3.0035641337493963], [4.2995973060574038, -4.147257186028682, -1.6104121775603577, 1.3696327647559414], [2.5427733971343356, 
-0.034500640283451567, 2.0444604093521761, -0.87309089681268226]], [[-2.5121451610530308, 2.3555904893582564, 
-3.3938010049826772, 0.87453175471058131], [2.8899427420587065, -4.0516676930740481, -0.42327240409374589, 
-4.4891219351505587], [1.7989284507829915, 3.4343454392519028, 3.8009505100499119, -4.0388076129814152]]], 
[[[-0.29043233187892703, -1.5985388181787306, 1.9264380068291906, 3.2326512583502325], [-1.3256399791794848, 
0.79840066746719884, -1.5993296114905196, -1.760739549771313], [3.6207712977086928, -3.3183602422919489, 1.9833009136435091, 
-2.1169565906827814]], [[-3.8276918074449529, 0.18315946530662153, 0.14550966551996503, 4.7165408758616572], 
[-1.0546542854928709, -4.5707090242950841, 1.2779161734011808, 0.091913897793411081], [3.6118770513808656, 1.606743391963291, 
-4.8173211112770247, -2.0274455285439528]]], [[[0.046982107151698926, -4.2881047339292246, 1.888933061110766, 
-4.5764919931275454], [-2.8473401101144002, 1.8376452445014841, 3.0916033335492425, -4.9147795554064588], [-4.6912016107548036, 
-3.5597721351812437, -0.20716462281744263, 1.9323095507706931]], [[3.0004965104152266, 1.2004760571734083, 1.754596165983596, 
-3.0872188149516666], [3.5389142776557847, -2.6344277089439827, -3.1289295012552918, -0.79080815872370991], 
[-1.3333673982871721, -1.8118651042528056, -4.4452310286201193, 1.4105953001168174]]]])
      arg1=Data(numpy.array([[[[-4.4565731616124351, 2.2089305711292484, -3.5722807441285589, -0.58300582305028037], 
[2.2804592767642609, 2.4432051930631706, -1.7506399706747313, -4.7525791262050525], [-2.9337861379783661, -0.44414297579310613, 
3.9902014071470902, 4.6446391384642762]], [[-4.0963813181379418, -0.20159878931745734, 4.0552914578678347, 
-3.8463822222942468], [2.7055223074693204, -4.5078664578052798, 1.4261042579792811, 4.0980526415524992], [-0.53501664574801122, 
-2.5185048066601046, 2.6174082032088233, -3.2978442128966101]]], [[[-0.96122832591535712, 2.8631064697220179, 
0.15803033702082736, -4.8504600319128963], [2.9661632129663005, -4.6783236722874353, 0.47340430365890551, 2.0051599394971529], 
[-0.60516180541016418, -0.5680719407265391, -1.2367913811771789, 3.6604783550306088]], [[-0.061251612739617656, 
-4.5771711782257327, 4.7061308296742599, -4.5160997044613591], [-3.7340961670463244, -4.2737724609781473, -3.3383608656230201, 
2.2657993514349926], [4.0073723808009074, -0.34751762974561728, 0.0049953602295573774, -4.7511117605728961]]], 
[[[-2.3499156052338366, 0.39861688236522586, 1.7519686786593276, -1.0045596641386991], [1.1506553497256089, 2.1268876237964349, 
-1.8981830986848158, -3.6877934311652574], [-4.3957296686175642, 1.6053745441265574, -4.2602126673504941, 
-0.46228488994530892]], [[-4.01119988174203, -2.7145886039549536, -4.1941101705293917, 1.6934869130834258], 
[-0.34799528639547184, -0.80946922474999372, 2.4246740218939937, -0.83617784943294549], [0.018284300512841689, 
3.0083202863304788, -4.6906579813556961, 1.3175478875547251]]]]),self.functionspace)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.23285429770774146, -0.72871380525643326, -0.54368699695700162, 3.109778854166942], 
[1.6470245980304632, -1.0531512510070866, 0.34556007878304185, -0.54550853093518248], [1.655373194286212, -6.6500867077944417, 
0.83778313642694424, 0.92635762461246618]], [[-1.1302245958574091, -21.951073586112607, 0.71490945024313834, 
0.9948626731797513], [1.5746449713825972, -0.10412654509024727, 2.4293358331884076, 0.97981080008274757], [-8.7681776193546668, 
-1.1061906357637876, 1.811765703116941, -0.12693144198393233]]], [[[0.58756642126873981, -0.31927263748535417, 
-5.5498654210921883, 0.19949140116557318], [-0.52200012128288842, -0.29981818022831436, -6.2818994064159996, 
1.6586580344216098], [2.7924034158041109, 3.989907141923505, -3.8864075393561133, -1.1246738265640774]], [[-40.46206234435364, 
0.22537209978973594, -0.15466349617774669, 0.11280232680424071], [-1.038652173825092, -0.00041598521466893639, 
-0.97155667126907364, 1.0540404173010904], [-0.4502037128825816, 1.4569713370261326, -408.74672440797195, 
-0.99759098898170084]]], [[[-0.34838557862071801, -8.3715569255853488, 0.61411983819841343, 0.78604516615331221], 
[-2.3259982400819994, -0.12905640189452938, 2.6047037317204906, 1.3446745810982208], [-0.82662052415296106, 2.9470365339415885, 
1.1257599861116891, 10.812621381942771]], [[1.1717056734970686, 1.1815777833536727, 0.80238670086928432, 2.931035880955565], 
[6.8462981566271033, 2.3544966790220618, 1.675754732760814, 5.314595629662981], [97.754792363600984, -1.6547567483689647, 
0.092737284924231278, 0.34346966887241875]]]])+(1.-msk_ref)*numpy.array([[[[0.1240601981943724, -1.7199403020911506, 
-0.61149036676395152, -5.151859578408291], [1.8854085007639749, -1.6974657707030554, 0.91989912519801142, 
-0.28818726177623832], [-0.86672077566176231, 0.077679130739023328, 0.512370229154403, -0.18797819825920101]], 
[[0.61325959815551445, -11.684546803745489, -0.836882142810778, -0.22736475580654864], [1.0681644479811694, 
0.89879940566089023, -0.29680326787152428, -1.0954280795792577], [-3.3623784700528234, -1.3636445839491302, 1.4521810183792194, 
1.2246811408456408]]], [[[0.30214707998992285, -0.55832321818403574, 12.190304995523098, -0.66646281735783275], 
[-0.44692078082034586, -0.17065956171365676, -3.3783588343608706, -0.87810429237523324], [-5.9831457724841384, 
5.8414436700533239, -1.6035856522187291, -0.57832785372803541]], [[62.491282045365551, -0.040015865296438483, 
0.030919171350371626, -1.0443836904668615], [0.28243897272927065, 1.0694787956139753, -0.38279749399191759, 
0.040565771075536543], [0.90130806627434046, -4.6234874274995095, -964.35910322805125, 0.42673075918119013]]], 
[[[-0.019993104027675838, -10.757458912641644, 1.0781774149959398, 4.555719442559333], [-2.4745377586723873, 
0.86400674109022213, -1.6287171325523369, 1.3327155241050208], [1.0672179511507958, -2.2174091075537907, 0.048627765558540106, 
-4.1799106845116585]], [[-0.74802966665229764, -0.44223130364004476, -0.41834765770163979, -1.8229953778211345], 
[-10.169431644640326, 3.2545124983073093, -1.2904536745979489, 0.94574157789517732], [-72.924167777197852, 
-0.60228464119520397, 0.94767749989210615, 1.0706216551527259]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-3.5029548965084203, -1.6855962091683132, 0.87497327960221405, 2.4967172632515711], 
[3.3933541198276433, 2.038275744473582, 0.28467243562026567, -3.5617428761798662], [-4.0791574717723922, -3.2664107450510258, 
-2.255823991707989, -3.6439883936956607]], [[-1.3019674130197667, -4.5207190704103404, 2.4066256229043326, 3.8453630843114066], 
[1.3881924320781511, 2.0436762218614559, -1.7912658255056293, 1.720926792337151], [-3.8079353687578954, 1.5536039125776266, 
-3.7712533527508043, -1.7105606782730556]]], [[[4.4221670011676597, -3.6537171294196344, -0.5072624415784226, 
0.26531466946333104], [-1.8883590342521552, 0.86867804759302381, -0.054632442777271528, -2.975805007074773], 
[0.98984553655062157, -0.18574976488629691, 2.1084190828676146, -0.42363702512359769]], [[-3.0508456800195907, 
3.80878362439082, -3.4055035183645033, -0.064860862959637444], [4.4937630355348386, -3.1290134500027689, -1.5525291623878967, 
0.74805124682546698], [-0.8174936521159708, -4.9589552443010554, 0.50516443837682168, -0.072761314833631019]]], 
[[[3.2935022057766741, -0.98823449085644022, -4.9889935707280006, -3.7495357437344854], [-3.8646450752152148, 
0.10638396139388018, 1.0961759040371346, -4.1959281860156139], [-4.0802106257421613, -2.3544506280535837, 0.54760324143895645, 
4.1962744982583171]], [[0.019563851472780058, -0.3412316514675231, 4.4613825193511634, -3.2304476424183273], 
[-2.7050699622795671, -1.3952874897518042, 0.715151971315934, 4.5758992406696581], [0.46778747101657725, 0.86830540255481203, 
2.7375293122193263, -2.6939533724117948]]]])+(1.-msk_arg0)*numpy.array([[[[-2.696476475302545, 3.9631603843312764, 
1.2241494562328228, -0.99547358075971015], [4.2934217608494691, 4.5823470043762757, 2.604119380276229, -3.238314291460175], 
[-2.0421032520954485, -4.5992289367699399, -1.5238455009432572, 3.0342069560142892]], [[1.6269453937782856, 
0.79411080950932611, -4.7384117269900008, 3.992361385844001], [-3.9103068590461798, 0.52327404818654344, 4.4502447293193157, 
-1.8463352164281943], [-3.2266551401268084, -2.5508016470859616, -4.2698080681126847, -3.4322589835670358]]], 
[[[0.32896849650567805, 2.5227980572455122, -2.1078393795944139, -2.4202922540552771], [0.79972331351475923, 
3.7270012669353036, 2.7980434760528192, -3.1848259797967793], [-4.5084295401943733, -1.0889031164008012, -0.20007923081749013, 
-3.0444400268285623]], [[-1.3931657864992406, -1.9243087827663343, -2.2085482490507768, -2.0586355733552155], 
[2.082179599647108, -1.0009144640976952, -2.0999629352407889, 2.6742186953751892], [3.8437318670642355, 0.43507497686430519, 
4.4463437730909945, 0.91418485714109998]]], [[[0.10003564451685154, -0.084297064975952019, 0.091123741151923454, 
0.63336553210606894], [1.2852603040686352, -1.1339130976834033, 2.4849604338515343, -0.61993815909937222], 
[-2.9221277578303706, -0.89050052992205142, -3.2166808657324788, -2.9394937855602024]], [[-3.4846017891887957, 
-3.4399955301176832, 2.621486312290128, -0.65105824905728049], [-2.1126144308401216, 1.3081205046416979, -1.6137123520436614, 
-2.0404353579982306], [-3.8248821200531804, -1.7435708063689903, 1.5118787223242407, -4.0266129310982537]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-1.37222465467)+(1.-msk_arg1)*(4.52718652996)
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[2.5527561282194666, 1.2283675296381511, -0.63763121921985833, -1.8194668451339744], 
[-2.472885258455102, -1.4853805005860499, -0.20745322906930211, 2.5955974949528806], [2.9726600946014119, 2.3803760804976943, 
1.6439174037772901, 2.6555333933757512]], [[0.94880048145668883, 3.2944453045774083, -1.753813134540386, -2.8022839199240761], 
[-1.0116364163488361, -1.4893160641757917, 1.3053735912750155, -1.2541144676840037], [2.7750087099749803, -1.1321789819821459, 
2.7482769238332625, 1.2465602279108001]]], [[[-3.2226261101713587, 2.6626231477325533, 0.36966428190257311, 
-0.19334637995314322], [1.3761296503619413, -0.63304360888378419, 0.039813045619939223, 2.1685989950279225], 
[-0.72134364674204454, 0.1353639611807316, -1.5364970128533406, 0.30872279089372756]], [[2.2232844087386541, 
-2.7756268708789085, 2.4817390554594723, 0.047266941851536196], [-3.2748012654007912, 2.2802486745511263, 1.1313957646087585, 
-0.54513759410928042], [0.59574330583007007, 3.6138071323937049, -0.36813537539702568, 0.053024345966885357]]], 
[[[-2.4001188104023097, 0.72016960742661906, 3.6356973719581003, 2.7324503542242895], [2.8163355483058834, 
-0.077526636058940182, -0.79883122658144756, 3.0577560108171951], [2.9734275738666933, 1.7157909384858991, 
-0.39906238353496881, -3.0580083836641077]], [[-0.014257032480918979, 0.24867039832411808, -3.2512041699299994, 
2.3541681979106492], [1.971302550984749, 1.0168068945569744, -0.52116245607520184, -3.334657503120436], [-0.34089714787139375, 
-0.63277204617942451, -1.9949570960544933, 1.9632014067370329]]]])+(1.-msk_ref)*numpy.array([[[[-0.59561859390062799, 
0.87541353953522449, 0.2703996064952891, -0.21988790922820708], [0.9483642285188475, 1.0121842725162737, 0.57521804393100551, 
-0.71530392441914636], [-0.45107557167785789, -1.0159132844052097, -0.33659878842139157, 0.67021911642747201]], 
[[0.35937229071750004, 0.17540934181825749, -1.0466570563480884, 0.88186368275761173], [-0.86373884379753285, 
0.11558482177026554, 0.98300449956399072, -0.4078328127653274], [-0.71272856083379366, -0.56344081035836968, 
-0.94314825330322583, -0.75814392909384609]]], [[[0.072665107639899265, 0.55725516069381364, -0.46559587630058341, 
-0.53461288551654851], [0.17664907514231798, 0.82324888587384948, 0.61805349912886476, -0.70348901215313231], 
[-0.99585681092557499, -0.24052534818123616, -0.044195049064858494, -0.67247947631027016]], [[-0.30773324166754951, 
-0.42505621759348539, -0.48784123084690506, -0.45472735875363834], [0.45992794550571192, -0.22108973365094173, 
-0.46385606631009713, 0.59070212320076998], [0.8490332442952111, 0.096102728258435965, 0.98214282615987913, 
0.20193222680147241]]], [[[0.022096647411094121, -0.01862018814953428, 0.020128117220004221, 0.13990268081823817], 
[0.28389824354754001, -0.2504675012126395, 0.54889729358505868, -0.1369367387440984], [-0.64546219566838192, 
-0.19670064929471545, -0.71052536590725046, -0.64929813828194438]], [[-0.76970581311946085, -0.75985283737483755, 
0.57905418628983174, -0.14381078507550971], [-0.4666506265774063, 0.28894778158220286, -0.35644927403880977, 
-0.45070715432059222], [-0.84486956628320564, -0.38513341450129357, 0.33395547374022821, -0.88942942917153978]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[3.6191965741311645, 3.5492355784370062, -2.8903080510126431, -4.838636822006289], 
[3.3498441523061491, -4.3396470193107737, 4.9220598282140458, -4.2904577746237571], [-1.3679431904750383, -0.20980143265635753, 
-2.756430518927405, -4.2966223021957042]], [[0.085588622185870378, -0.36846832693207787, 2.4862177172519493, 
4.6925221939230504], [1.6607162207566448, 0.94983516559056991, 3.9321883857589039, -0.017039434470115467], [3.085230067757788, 
-2.0370617748460038, 1.104425612129905, 0.64623997648643972]]], [[[-2.1654577474796879, -0.31226204964637816, 
2.5955142098034178, 0.85467304434968394], [2.0837940087341478, -2.691641756975296, 0.9111886822932993, -1.2186972005053089], 
[3.8668491340453173, 2.3762188834506714, -1.2091580334537855, 0.032278991245596522]], [[3.0382952152855633, 
-0.43864678946469748, 2.2076519941124619, -2.1274782296179016], [1.7141588554261133, 2.8432461375116249, 0.39710687180329529, 
-1.940418980061811], [-2.5496451452177791, -2.2218450491885711, 1.5856509928622113, 1.3196065048145664]]], 
[[[-4.884113896273039, -1.1217059432804657, -0.48071731376890359, -2.3188055691914156], [-4.6705698748319024, 
3.4486857250406064, -0.97008079669843461, 3.2084974048173684], [-0.058024642643647262, -3.2327608556749619, 1.3492041699410686, 
-0.89433430534941216]], [[-2.9958100017586431, 3.2100225025585054, -3.6662767959563212, 4.8291198678007596], 
[3.0713688512805088, 2.3963945505412996, 3.1102519957774923, 2.8648342290312536], [-2.6659654484283513, 0.86091593135659039, 
4.8946816523812782, -0.46566876474847874]]]])+(1.-msk_arg0)*numpy.array([[[[-0.74530912632481261, 3.4567658017463128, 
0.74317942138318305, -3.6809676669023119], [3.3257608061564206, -1.8868520671546154, -1.7732551259194573, -1.3831970246013725], 
[-2.0736432476487532, 3.0453094775224834, -3.7692900650290584, 3.0569678319374454]], [[-3.2301745328094364, 
-2.5583007616557643, 0.96647981346438883, 0.17877366990911003], [3.8316169726246763, -4.8606910362443472, -2.527901159767004, 
-3.889275167086196], [0.19803313922231691, 1.0989502831723676, 2.2963157302162687, 1.347336881504285]]], 
[[[-1.5456101988084194, 4.084965780390462, 4.1748117436173864, 3.4952602165304398], [-3.8972728332438269, 2.9427181800675397, 
2.7715976507601408, -1.280883877176259], [0.43234110096085665, 3.2113132138793663, -3.3103519655714884, -2.7791302902187729]], 
[[-3.6197348913009963, 3.5320334739474823, 0.34351708473935183, -2.373843503965384], [0.5556131184588331, -4.157856876542473, 
1.362434438116388, 4.8711856582449951], [3.0017662573399519, -0.94571602131206234, -3.2372827729115485, 
0.0059300841215348399]]], [[[1.0546191242539793, -4.61431067290206, 0.50700583760312501, 2.9783765518766812], 
[4.5485683367522771, 0.32274039855635195, 2.7969925040868144, -4.7594567137140267], [0.86087915423929573, 4.8846659592247335, 
-2.9990832953185595, -2.4506544191089406]], [[-4.6628302142893352, 4.2241313947670776, 1.1917972218456683, 3.8589741764414214], 
[0.060701607053042395, 3.2969498043048322, 3.993484007169851, -0.15548066627168922], [0.94278963725869769, 1.794736039846728, 
-3.5867960669259324, 3.7001291215499172]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.87559068568296716, -2.911321471422732, -4.1772942592384474, 4.9373077214423837], 
[1.2194443299646984, 2.1352817885735176, -1.2687974838468739, -1.757185846288456], [-0.3981891570089191, -0.18036256834886899, 
4.882833895744719, -1.6930248378184256]], [[3.8234393369918784, 3.1539593733283908, -4.9632618312855881, -0.73188984118704603], 
[-4.3995366448233444, -3.1564450496057663, 2.0977243420762424, -2.6425011437623915], [4.70284455994463, -2.9040009827203339, 
4.2347142437236585, -4.1533879370835214]]], [[[0.34705229029721707, 0.94583548613956214, -0.87410625247820928, 
-3.339489964065586], [-0.42924109648108644, -4.424885959862249, 1.7684674898977963, 1.3429577261210142], [-0.17708089708904851, 
0.61027127677129034, -3.1298799601694762, 3.0475127661335719]], [[0.075576919273934173, -1.102574953904969, 1.2670869464815473, 
2.3968534935222232], [3.9440561755676704, -1.8657108519292276, -0.35045402088916955, 1.0239844511735399], [2.0286693203279054, 
0.056598444787891999, -2.2148915736265682, -4.7308682041445307]]], [[[-0.45917854492227317, 3.8076305275317051, 
2.5757785622683436, -1.0093840610553673], [-3.9183986730230735, -4.6093378461436583, 4.9945478035035187, -3.5677744626494832], 
[3.2792649036063146, 1.7738850556634187, -3.1654721369143237, -0.40519680295052574]], [[0.27192336082373814, 
-1.6877885670843451, -0.073264371462387068, -4.3641485165548186], [3.8825079482566434, -3.2265278111720264, 
-0.77965750908901654, 4.6034389700610827], [0.38983741724071042, -2.5106359015136359, -4.3242589561254103, 
0.14643087311415481]]]])+(1.-msk_arg1)*numpy.array([[[[-0.20143683370838694, 1.928959224144176, -4.2713829451756649, 
3.5955079133751937], [4.6326973117084158, 4.5177868827002126, 2.575783233906046, -3.4593728261827552], [4.1181156808447472, 
1.529902492064469, 2.0770861756083745, -2.5016378071545562]], [[0.68933751736488436, 4.9158053190862496, 1.6931003581265545, 
-3.7911718172462816], [-1.4418996135757833, -4.5484953712539697, -3.9488137450413188, -3.4501720700977137], 
[-1.7521063117957194, -4.3344704323876373, 1.3195529828018602, -4.2424912074650027]]], [[[-3.6012971170963959, 
-2.969578948994732, 0.66146664748649009, 3.842990196504374], [-2.2365341778485917, -1.8618965290231815, -4.5025071479965106, 
0.010174783300996637], [1.0914570774482542, -0.76896767480167494, 4.937003968799873, 0.57590350909988963]], 
[[3.1151438688099446, 3.7812140181034319, -2.8011240777379198, 1.426079236881229], [-3.2933464205612895, 2.3465314738623952, 
-4.1374531010801983, -2.7043538392831676], [0.24865488377007861, -2.4043610266760815, 0.054267393290205135, 
-2.4799523030686812]]], [[[-3.7703490912436601, -0.94857512991298343, 1.7244042042377679, -2.8584691142084129], 
[-2.9068277529756994, -1.1836832606522418, 0.64537148208290507, 3.0812454817913011], [2.2589595723639579, -2.9975120975873795, 
-3.8074642322154317, 2.6382400207544077]], [[1.6761195661097039, -0.68142573928691963, -0.97958326274695118, 
1.3070611430186698], [-1.4614845118955113, 4.8515520912780925, 3.5074998973673583, -4.6434698728586277], [2.7171240384350703, 
1.6745397393693731, -3.9718388519944812, 4.3083474879168691]]]])
      res=arg0/arg1
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-4.133434301334721, -1.2191149666142957, 0.6919091334349925, -0.98001524211108537], 
[2.7470250752677847, -2.0323533139904173, -3.8793108363446831, 2.4416642005659797], [3.4354104485180583, 1.1632204762716951, 
-0.56451449665932174, 2.5378377246562938]], [[0.022385243923657466, -0.11682722677027739, -0.50092415064227369, 
-6.4115143152038945], [-0.37747525587966274, -0.30091927806859886, 1.8745019576151667, 0.0064482221740327162], 
[0.65603488025853707, 0.70146731594346035, 0.26080286615957449, -0.15559345437407532]]], [[[-6.2395719838793759, 
-0.33014414686520077, -2.9693349091655441, -0.25592921480416181], [-4.854600423438173, 0.60829629992522771, 
0.51524197504244706, -0.90747249656575701], [-21.836624941541821, 3.8937091976904563, 0.38632728693796686, 
0.010591913380743403]], [[40.201363650098465, 0.39783852146391535, 1.7423050566835052, -0.88761296231399212], 
[0.43461826584642738, -1.5239479014508506, -1.1331211746287244, -1.8949691841882845], [-1.2568066760164074, 
-39.256291537959122, -0.71590456695175131, -0.27893537673666541]]], [[[10.636633506253631, -0.29459421946793013, 
-0.18662990709324129, 2.297248053200855], [1.1919588241460186, -0.74819547626041427, -0.1942279531328048, 
-0.89929939193372943], [-0.017694405407698434, -1.8224184511582884, -0.42622525537573103, 2.2071603202126195]], 
[[-11.017111559240178, -1.9019103252392684, 50.041742292685029, -1.1065434298310732], [0.79107857400772119, 
-0.74271622337909327, -3.9892542039537808, 0.6223247984089687], [-6.8386597348663809, -0.34290752029697069, 
-1.1319122425468648, -3.1801269421199989]]]])+(1.-msk_ref)*numpy.array([[[[3.6999644633204003, 1.7920367410980296, 
-0.17399035181862371, -1.0237684787757553], [0.71788864723604573, -0.41764964044228503, -0.68843336759763163, 
0.3998404028997537], [-0.50354176724423338, 1.9905252088406664, -1.8147008580060675, -1.2219865814286439]], 
[[-4.6859114025265223, -0.52042353095693816, 0.5708343329002753, -0.047155253976054912], [-2.657339620975836, 
1.0686371293160861, 0.64016723071362613, 1.1272699123600656], [-0.11302575528042837, -0.25353738139748072, 1.7402224542287106, 
-0.31758153773743547]]], [[[0.42918152780867819, -1.3756043703681673, 6.3114470842653541, 0.90951577750829726